---
title: Documentation comment references
short-title: Comment references
description: Learn about doc comment references and their syntax.
---

Doc comments can contain references to various identifiers.
Elements, such as functions and classes, can be referenced by
wrapping their name in square brackets (`[...]`) in
a doc comment (a comment starting with `///`). Some examples:

```dart
/// Returns a [String].
String f() => 'Hello';

/// Wraps [object] with [Future.value].
Future<T> g<T>(T object) => Future.value(object);
```

These doc comments contain references to the `String` class,
the `object` parameter, and the `Future.value` constructor.

## Features of references

There are several benefits to referring to code elements with
doc comment references:

### Editor support

Doc comment references enable several IDE features:

- **Code completion**
  An element's name can be code-completed within square brackets.
- **Rename refactoring**
  When an element is renamed via an IDE command, the IDE can
  rewrite uses of that element, including references in doc comments.
- **Find references**
  When an IDE lists all "references" to an element, it can
  include references in doc comments.
- **Go to definition**
  An IDE can also provide Go-to-definition support at
  the location of a doc comment reference.

:::tip
The [`comment_references`][] lint rule can help to
ensure that doc comment references are valid, avoiding typos and mis-uses.
Keeping doc comment references valid ensures that these IDE features
are enabled for each reference.
:::

[`comment_references`]: /tools/linter-rules/comment_references

### API documentation

In API documentation generated by [`dart doc`](/tools/dart-doc), a doc comment
reference is linked, if possible, to the documentation page of the element
being referenced. If the element does not have a documentation page (for
example, a function parameter, a type parameter, or a private class), then no
link is created.

## What can be referenced

Most library members can be referenced in a doc comment, including classes,
constants, enums, named extensions, extension types, functions, mixins, and
type aliases. This includes all in-scope library members, either declared
locally, or imported. Library members that are imported with an import prefix
can be referenced with the prefix. For example:

```dart
import 'dart:math' as math;

/// [List] is in scope.
/// So is [math.max].
int x = 7;
```

Most members of a class, an enum, an extension, an extension type, and a mixin
can also be referenced. A reference to a member that is not in scope must be
qualified (prefixed) with its container's name. For example the `wait` static
method on the `Future` class can be referenced in a doc comment with
`[Future.wait]`. This is true for instance members as well; the `add` method
and the `length` property on the `List` class can be referenced with
`[List.add]` and `[List.length]`. When container members are in-scope, such as
in an instance method's doc comment, they can be referenced without the
qualifying container name:

```dart
abstract class MyList<E> implements List<E> {
  /// Refer to [add] and [contains], which is declared on [Iterable].
  void myMethod() {}
}
```

Unnamed constructors can be referenced by using the `new` name, similar to the
tear-off of an unnamed constructor. For example, `[DateTime.new]` is a
reference to the unnamed `DateTime` constructor.

Parameters of a function and parameters of a function type can be referenced in
a doc comment only when they are in scope. They can therefore only be
referenced within a doc comment on such a parameter's function or on a type
alias for such a parameter's enclosing function type.

Type parameters can be referenced in a doc comment only when they are in scope.
Therefore, a type parameter of a method, top-level function, or type alias can
only be referenced within a doc comment on that element, and a type parameter
of a class, enum, extension, extension type, and mixin can only be referenced
within a doc comment on that element or on one of its members.

The doc comment for a type alias that aliases a class, enum, extension type, or
mixin can't reference any of the aliased type's members as if they were in
scope.
