import * as A from 'fp-ts/Array';
import * as R from 'fp-ts/Record';
import { pipe } from 'fp-ts/function';
import * as S from 'fp-ts/string';
import { type Edge } from 'reactflow';
import { Observable, Subject, from, mergeMap, of, type Observer } from 'rxjs';
import invariant from 'tiny-invariant';

import { Connector, NodeConfig, type VariableValueRecords } from 'flow-models';
import { computeGraphs } from 'graph-util';
import {
  FlowRunEvent,
  FlowRunEventType,
  GetAccountLevelFieldValueFunction,
  RunNodeProgressEventType,
  getNodeAllLevelConfigOrValidationErrors,
  runFlow,
  type RunFlowResult,
  type RunNodeProgressEvent,
  type ValidationError,
} from 'run-flow';

type Params = {
  // canvas data
  edges: Edge[];
  nodeConfigs: Readonly<Record<string, Readonly<NodeConfig>>>;
  connectors: Readonly<Record<string, Readonly<Connector>>>;
  inputValueMap: VariableValueRecords;
  // run options
  startNodeIds: string[];
  preferStreaming: boolean;
  // callbacks
  progressObserver: Observer<FlowRunEvent>;
  getAccountLevelFieldValue: GetAccountLevelFieldValueFunction;
};

function runFlowForCanvasTester(params: Params): Observable<RunFlowResult> {
  // SECTION[id=pre-execute-validation]: Pre execute validation
  // Keep this section in sync with:
  // LINK ./flowRunBatch.ts#pre-execute-validation

  // ANCHOR: Step 1 - compile graphs

  const { errors } = computeGraphs({
    edges: params.edges,
    nodeConfigs: params.nodeConfigs,
    startNodeIds: params.startNodeIds,
  });

  // ANCHOR: Step 2 - validate graphs

  if (!R.isEmpty(errors)) {
    // TODO: Apply errors to specific nodes
    return of({
      errors: pipe(
        errors,
        R.collect(S.Ord)((_, list) => list),
        A.flatten,
        A.uniq(S.Eq),
      ),
      variableValues: {},
    });
  }

  const result = getNodeAllLevelConfigOrValidationErrors(
    params.nodeConfigs,
    params.getAccountLevelFieldValue,
  );

  const validationErrors: ValidationError[] = [];

  if (result.errors) {
    validationErrors.push(...result.errors);
  }

  if (validationErrors.length) {
    params.progressObserver.next({
      type: FlowRunEventType.ValidationErrors,
      errors: validationErrors,
    });

    return of({
      errors: validationErrors.map((error) => error.message),
      variableValues: {},
    });
  }

  invariant(
    result.nodeAllLevelConfigs != null,
    'nodeAllLevelConfigs is not null',
  );

  // !SECTION

  // ANCHOR: Step 3 - run flow

  const subject = new Subject<RunNodeProgressEvent>();

  subject.pipe(mergeMap(transformEvent)).subscribe(params.progressObserver);

  return runFlow({
    edges: params.edges,
    nodeConfigs: result.nodeAllLevelConfigs,
    connectors: params.connectors,
    inputVariableValues: params.inputValueMap,
    startNodeId: params.startNodeIds[0],
    preferStreaming: params.preferStreaming,
    progressObserver: subject,
  });
}

function transformEvent(event: RunNodeProgressEvent): Observable<FlowRunEvent> {
  switch (event.type) {
    case RunNodeProgressEventType.Started:
      return of({
        type: FlowRunEventType.NodeStart,
        nodeId: event.nodeId,
        runFlowStates: event.runFlowStates,
      });
    case RunNodeProgressEventType.Finished:
      return of({
        type: FlowRunEventType.NodeFinish,
        nodeId: event.nodeId,
        runFlowStates: event.runFlowStates,
      });
    case RunNodeProgressEventType.Updated: {
      const { errors, conditionResults, variableValues } = event.result;

      const flowRunEvents: FlowRunEvent[] = [];

      if (errors != null && errors.length > 0) {
        flowRunEvents.push({
          type: FlowRunEventType.NodeErrors,
          nodeId: event.nodeId,
          errorMessages: errors,
        });
      }

      if (conditionResults != null || variableValues != null) {
        flowRunEvents.push({
          type: FlowRunEventType.VariableValues,
          variableResults: variableValues ?? {},
          conditionResults: conditionResults ?? {},
        });
      }

      return from(flowRunEvents);
    }
  }
}

export default runFlowForCanvasTester;
