import {
  DndContext,
  DragEndEvent,
  KeyboardSensor,
  PointerSensor,
  closestCenter,
  useSensor,
  useSensors,
} from '@dnd-kit/core';
import {
  SortableContext,
  sortableKeyboardCoordinates,
  verticalListSortingStrategy,
} from '@dnd-kit/sortable';
import { css } from '@emotion/react';
import { A } from '@mobily/ts-belt';
import {
  getNodeDefinitionForNodeTypeName,
  type NodeInputVariable,
  type NodeOutputVariable,
} from 'flow-models';
import { produce } from 'immer';
import { useCallback, useContext, useMemo } from 'react';
import { useFieldArray, useForm } from 'react-hook-form';
import { useUpdateNodeInternals } from 'reactflow';
import RouteFlowContext from 'state-flow/context/FlowRouteContext';
import { useFlowStore } from 'state-flow/flow-store';
import invariant from 'tiny-invariant';
import NodeConfigPaneAddConnectorButton from 'view-left-side-pane/left-side-pane-base-ui/NodeConfigPaneAddConnectorButton';
import { VariableFormValue } from '../types';
import NodeRenamableVariableItem, {
  HandlePosition,
} from './NodeRenamableVariableItem';

type Props = {
  nodeId: string;
  variableIds: string[];
  onVariableIdsChange: (variableIds: string[]) => void;
  labelForAddVariableButton: string;
  onAddVariable: () => void;
  onRemoveVariable: (variableId: string) => void;
  isListSortable?: boolean;
  showConnectorHandle?: HandlePosition;
};

function NodeRenamableVariableList(props: Props) {
  const updateNodeInternals = useUpdateNodeInternals();

  const { isCurrentUserOwner } = useContext(RouteFlowContext);
  const isReadOnly = !isCurrentUserOwner;

  const nodeConfig = useFlowStore(
    (s) => s.getFlowContent().nodeConfigs[props.nodeId],
  );

  const nodeDefinition = useMemo(
    () => getNodeDefinitionForNodeTypeName(nodeConfig.type),
    [nodeConfig.type],
  );

  const connectors = useFlowStore((s) => s.getFlowContent().connectors);

  const variables = useMemo((): (NodeOutputVariable | NodeInputVariable)[] => {
    return props.variableIds.map(
      (variableId) => connectors[variableId] as NodeInputVariable,
    );
  }, [connectors, props.variableIds]);

  const updateVariable = useFlowStore((s) => s.updateConnector);

  const sensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    }),
  );

  const { control, getValues, setValue, handleSubmit } =
    useForm<VariableFormValue>({ values: { list: variables } });

  // NOTE: field will contain all properties of the variable except
  // the "id" is generated by react-hook-form.
  // But in SubmitHandler, the "id" will be the id from the original
  // variable object.
  const { remove, move } = useFieldArray({ control, name: 'list' });

  const submit = useCallback(() => {
    handleSubmit((data) => {
      // NOTE: We don't handle add variable here

      if (variables.length === data.list.length) {
        // This is an update

        // NOTE: Elements from the first array, not existing in the
        // second array.
        const updatedVariables = A.difference(data.list, variables);

        for (const changedVariable of updatedVariables) {
          invariant(!isReadOnly, 'Node should not be readonly');

          const index = data.list.indexOf(changedVariable);
          const prevVariable = variables[index];

          if (prevVariable.name !== changedVariable.name) {
            // If variable name has changed, make sure it's not readonly
            invariant(
              nodeDefinition.fixedIncomingVariables?.[index] == null,
              'Variable should not be readonly',
            );
          }

          updateVariable(changedVariable.id, {
            name: changedVariable.name,
            isGlobal: changedVariable.isGlobal,
            globalVariableId: changedVariable.globalVariableId,
          });
        }
      } else {
        // This is a remove

        // NOTE: Elements from the first array, not existing in the
        // second array. Note the order of the arguments is different from
        // above.
        const removedVariables = A.difference(variables, data.list);

        for (const removedVariable of removedVariables) {
          invariant(!isReadOnly, 'Node should not be readonly');

          const index = variables.indexOf(removedVariable);

          invariant(
            nodeDefinition.fixedIncomingVariables?.[index] == null,
            'Variable should not be readonly',
          );

          props.onRemoveVariable(removedVariable.id);
        }

        // NOTE: Removing a variable will affect edge and handle positions.
        updateNodeInternals(props.nodeId);
      }
    })();
  }, [
    handleSubmit,
    variables,
    isReadOnly,
    updateVariable,
    nodeDefinition.fixedIncomingVariables,
    updateNodeInternals,
    props,
  ]);

  const onDragEnd = useCallback(
    (event: DragEndEvent) => {
      const { active, over } = event;

      if (over == null || active.id === over.id) {
        return;
      }

      const variables = getValues().list;

      const oldIndex = variables.findIndex((v) => v.id === active.id);
      const newIndex = variables.findIndex((v) => v.id === over.id);

      move(oldIndex, newIndex);

      props.onVariableIdsChange(
        produce(props.variableIds, (draft) => {
          draft.splice(oldIndex, 1);
          draft.splice(newIndex, 0, active.id as string);
        }),
      );
    },
    [getValues, move, props],
  );

  return (
    <div
      css={css`
        position: relative;
        margin-bottom: 10px;
        border-bottom: 1px solid #e0e0e0;
      `}
    >
      <NodeConfigPaneAddConnectorButton
        label={props.labelForAddVariableButton}
        onClick={props.onAddVariable}
      />
      <DndContext
        sensors={sensors}
        collisionDetection={closestCenter}
        onDragEnd={onDragEnd}
      >
        <SortableContext
          disabled={!props.isListSortable}
          items={getValues().list}
          strategy={verticalListSortingStrategy}
        >
          <div>
            {getValues().list.map((variable, index) => {
              const incomingVariableConfig =
                nodeDefinition.fixedIncomingVariables?.[variable.name];

              return (
                <NodeRenamableVariableItem
                  // Must use the variable ID instead of field ID,
                  // because a new field ID is generated
                  // when `props.variableConfigs` updates.
                  // This is to prevent loss of focus when updating variable.
                  key={variable.id}
                  isListSortable={!!props.isListSortable}
                  connectorHandlePosition={props.showConnectorHandle ?? 'none'}
                  nodeId={props.nodeId}
                  isNodeReadOnly={isReadOnly}
                  variable={variable}
                  variableDefinition={{
                    isVariableFixed: incomingVariableConfig != null,
                    helperText: incomingVariableConfig?.helperMessage,
                  }}
                  value={variable}
                  onChange={(value) => {
                    setValue(
                      `list.${index}`,
                      produce(variable, (draft) => {
                        Object.assign(draft, value);
                      }),
                    );
                    submit();
                  }}
                  onRemove={() => {
                    remove(index);
                    submit();
                  }}
                />
              );
            })}
          </div>
        </SortableContext>
      </DndContext>
    </div>
  );
}

export default NodeRenamableVariableList;
