import { A } from '@mobily/ts-belt';
import { useCallback, useMemo } from 'react';
import { useFieldArray, useForm } from 'react-hook-form';
import invariant from 'tiny-invariant';

import { NodeOutputVariable } from 'flow-models';

import { useFlowStore } from 'state-flow/flow-store';

import {
  NodeOutputVariableProps,
  NodeOutputVariablePropsArrayFieldValues,
} from '../types';
import NodeOutputVariableItem from './NodeOutputVariableItem';

type Props = {
  nodeId: string;
  isNodeReadOnly: boolean;
  variables: NodeOutputVariable[];
};

function NodeOutputVariableList(props: Props) {
  const variableResults = useFlowStore(
    (s) => s.getFlowContent().variableResults,
  );

  const updateVariable = useFlowStore((s) => s.updateConnector);

  const variablePropsArray = useMemo(() => {
    return props.variables.map<NodeOutputVariableProps>((variable) => {
      return {
        id: variable.id,
        name: variable.name,
        value: variableResults[variable.id]?.value,
        isGlobal: variable.isGlobal,
        globalVariableId: variable.globalVariableId,
      };
    });
  }, [props.variables, variableResults]);

  const { control, handleSubmit } =
    useForm<NodeOutputVariablePropsArrayFieldValues>({
      values: { list: variablePropsArray },
    });

  // NOTE: field will contain all properties of a item, except that the "id" is
  // generated by react-hook-form.
  // But in SubmitHandler, the "id" will be the id from the original
  // variable object.
  const { fields } = useFieldArray({
    control,
    name: 'list',
  });

  const update = useCallback(() => {
    handleSubmit((data) => {
      // NOTE: Elements from the first array, not existing in the
      // second array.
      const updatedVariables = A.difference(data.list, variablePropsArray);

      for (const changedVariable of updatedVariables) {
        invariant(!props.isNodeReadOnly, 'Node should not be readonly');

        updateVariable(changedVariable.id, {
          isGlobal: changedVariable.isGlobal,
          globalVariableId: changedVariable.globalVariableId,
        });
      }
    })();
  }, [props.isNodeReadOnly, variablePropsArray, handleSubmit, updateVariable]);

  return fields.map((field, index) => {
    const variable = variablePropsArray[index];

    return (
      <NodeOutputVariableItem
        // Must use the variable ID instead of field ID,
        // because a new field ID is generated
        // when `variablePropsArray` updates.
        // This is to prevent re-render of when variable updates.
        key={variable.id}
        nodeId={props.nodeId}
        isNodeReadOnly={props.isNodeReadOnly}
        variableId={props.variables[index].id}
        control={control}
        formField={field}
        index={index}
        onUpdateTrigger={update}
      />
    );
  });
}

export default NodeOutputVariableList;
