import res from './res';
import uLib from './u';
import templatesLib from './templates';
import {settings as settingsLib} from '.';
import roomsLib from 'rooms';

import type * as pixiMod from 'pixi.js';
declare var PIXI: typeof pixiMod;

import type {ExportedTilemap, ExportedTile, TextureShape} from './../node_requires/exporter/_exporterContracts';

/**
 * @extends {PIXI.Sprite}
 */
export class Tile extends PIXI.Sprite {
    shape: TextureShape;
}

class TileChunk extends PIXI.Container {
    chunkX: number;
    chunkY: number;
    cacheAsBitmap: boolean; // pixi.js' Container is a jerk
}

/**
 * @extends {PIXI.Container}
 */
export class Tilemap extends PIXI.Container {
    pixiTiles: Tile[];
    tiles: (ExportedTile & {sprite: Tile})[];
    cells: TileChunk[];
    diamondCellMap: Record<string, TileChunk>;
    cached: boolean;
    /**
     * @param template A template object that contains data about depth
     * and tile placement. It is usually used by ct.IDE.
     */
    constructor(template?: ExportedTilemap) {
        super();
        this.pixiTiles = [];
        if (template) {
            this.zIndex = template.depth;
            this.tiles = template.tiles.map((tile: ExportedTile) => ({
                ...tile
            })) as (ExportedTile & {sprite: Tile})[];
            if (template.extends) {
                Object.assign(this, template.extends);
            }
            for (let i = 0, l = template.tiles.length; i < l; i++) {
                const tile = template.tiles[i];
                const textures = res.getTexture(tile.texture);
                const sprite = new Tile(textures[tile.frame]);
                sprite.anchor.x = textures[0].defaultAnchor.x;
                sprite.anchor.y = textures[0].defaultAnchor.y;
                sprite.shape = textures.shape;
                sprite.scale.set(tile.scale.x, tile.scale.y);
                sprite.rotation = tile.rotation;
                sprite.alpha = tile.opacity;
                sprite.tint = tile.tint;
                sprite.x = tile.x;
                sprite.y = tile.y;
                this.addChild(sprite);
                this.pixiTiles.push(sprite);
                this.tiles[i].sprite = sprite;
            }
            if (template.cache) {
                this.cache();
            }
        } else {
            this.tiles = [] as (ExportedTile & {sprite: Tile})[];
        }
        templatesLib.list.TILEMAP.push(this);
        this.on('destroyed', () => {
            templatesLib.list.TILEMAP.splice(templatesLib.list.TILEMAP.indexOf(this), 1);
        });
    }
    /**
     * Adds a tile to the tilemap. Will throw an error if a tilemap is cached.
     * @param textureName The name of the texture to use
     * @param x The horizontal location of the tile
     * @param y The vertical location of the tile
     * @param [frame] The frame to pick from the source texture. Defaults to 0.
     * @returns The created tile
     */
    addTile(
        textureName: string,
        x: number,
        y: number,
        frame = 0
    ): Tile {
        if (this.cached) {
            throw new Error('[ct.tiles] Adding tiles to cached tilemaps is forbidden. Create a new tilemap, or add tiles before caching the tilemap.');
        }
        const texture = res.getTexture(textureName, frame);
        const sprite = new Tile(texture);
        sprite.x = x;
        sprite.y = y;
        sprite.shape = texture.shape;
        this.tiles.push({
            texture: textureName,
            frame,
            x,
            y,
            width: sprite.width,
            height: sprite.height,
            sprite,
            opacity: 1,
            rotation: 1,
            scale: {
                x: 1,
                y: 1
            },
            tint: 0xffffff
        });
        this.addChild(sprite);
        this.pixiTiles.push(sprite);
        return sprite;
    }
    /**
     * Enables caching on this tileset, freezing it and turning it
     * into a series of bitmap textures. This proides great speed boost,
     * but prevents further editing.
     */
    cache(chunkSize = 1024): void {
        if (this.cached) {
            throw new Error('[ct.tiles] Attempt to cache an already cached tilemap.');
        }

        // Divide tiles into a grid of larger cells so that we can cache these cells as
        // separate bitmaps
        const bounds = this.getLocalBounds();
        const cols = Math.ceil(bounds.width / chunkSize),
              rows = Math.ceil(bounds.height / chunkSize);
        this.cells = [];
        for (let y = 0; y < rows; y++) {
            for (let x = 0; x < cols; x++) {
                const cell = new TileChunk();
                this.cells.push(cell);
            }
        }
        for (let i = 0, l = this.tiles.length; i < l; i++) {
            const [tile] = this.children as Tile[],
                  x = Math.floor((tile.x - bounds.x) / chunkSize),
                  y = Math.floor((tile.y - bounds.y) / chunkSize);
            this.cells[y * cols + x].addChild(tile);
        }
        this.removeChildren();

        // Filter out empty cells, cache the filled ones
        for (let i = 0, l = this.cells.length; i < l; i++) {
            if (this.cells[i].children.length === 0) {
                this.cells.splice(i, 1);
                i--;
                l--;
                continue;
            }
            this.addChild(this.cells[i]);
            if (settingsLib.pixelart) {
                this.cells[i].cacheAsBitmapResolution = 1;
            }
            this.cells[i].cacheAsBitmap = true;
        }

        this.cached = true;
    }
    /**
     * Enables caching on this tileset, freezing it and turning it
     * into a series of bitmap textures. This proides great speed boost,
     * but prevents further editing.
     *
     * This version packs tiles into rhombus-shaped chunks, and sorts them
     * from top to bottom. This fixes seam issues for isometric games.
     */
    cacheDiamond(chunkSize = 1024): void {
        if (this.cached) {
            throw new Error('[ct.tiles] Attempt to cache an already cached tilemap.');
        }

        this.cells = [];
        this.diamondCellMap = {};
        for (let i = 0, l = this.tiles.length; i < l; i++) {
            const [tile] = this.children as Tile[];
            const {x: xNormalized, y: yNormalized} = uLib.rotate(tile.x, tile.y * 2, -45);
            const x = Math.floor(xNormalized / chunkSize),
                  y = Math.floor(yNormalized / chunkSize),
                  key = `${x}:${y}`;
            if (!(key in this.diamondCellMap)) {
                const chunk = new TileChunk();
                chunk.chunkX = x;
                chunk.chunkY = y;
                this.diamondCellMap[key] = chunk;
                this.cells.push(chunk);
            }
            this.diamondCellMap[key].addChild(tile);
        }
        this.removeChildren();

        this.cells.sort((a, b) => {
            const maxA = Math.max(a.chunkY, a.chunkX),
                  maxB = Math.max(b.chunkY, b.chunkX);
            if (maxA === maxB) {
                return b.chunkX - a.chunkX;
            }
            return maxA - maxB;
        });

        for (let i = 0, l = this.cells.length; i < l; i++) {
            this.addChild(this.cells[i]);
            this.cells[i].cacheAsBitmap = true;
        }

        this.cached = true;
    }
}

const tilemapsLib = {
    /**
     * Creates a new tilemap at a specified depth, and adds it to the main room (ct.room).
     * @param [depth] The depth of a newly created tilemap. Defaults to 0.
     * @returns {Tilemap} The created tilemap.
     */
    create(depth = 0): Tilemap {
        if (!roomsLib.current) {
            throw new Error('[emitters.fire] An attempt to create a tilemap before the main room is created.');
        }
        const tilemap = new Tilemap();
        tilemap.zIndex = depth;
        roomsLib.current.addChild(tilemap);
        return tilemap;
    },
    /**
     * Adds a tile to the specified tilemap. It is the same as
     * calling `tilemap.addTile(textureName, x, y, frame).
     * @param tilemap The tilemap to modify.
     * @param textureName The name of the texture to use.
     * @catnipAsset textureName:texture
     * @param x The horizontal location of the tile.
     * @param y The vertical location of the tile.
     * @param frame The frame to pick from the source texture. Defaults to 0.
     * @returns {PIXI.Sprite} The created tile
     */
    addTile(
        tilemap: Tilemap,
        textureName: string,
        x: number,
        y: number,
        frame = 0
    ): pixiMod.Sprite {
        return tilemap.addTile(textureName, x, y, frame);
    },
    /**
     * Enables caching on this tileset, freezing it and turning it
     * into a series of bitmap textures. This proides great speed boost,
     * but prevents further editing.
     *
     * This is the same as calling `tilemap.cache();`
     *
     * @param tilemap The tilemap which needs to be cached.
     * @param chunkSize The size of one chunk.
     */
    cache(tilemap: Tilemap, chunkSize: number): void {
        tilemap.cache(chunkSize);
    },
    /**
     * Enables caching on this tileset, freezing it and turning it
     * into a series of bitmap textures. This proides great speed boost,
     * but prevents further editing.
     *
     * This version packs tiles into rhombus-shaped chunks, and sorts them
     * from top to bottom. This fixes seam issues for isometric games.
     * Note that tiles should be placed on a flat plane for the proper sorting.
     * If you need an effect of elevation, consider shifting each tile with
     * tile.pivot.y property.
     *
     * This is the same as calling `tilemap.cacheDiamond();`
     *
     * @param tilemap The tilemap which needs to be cached.
     * @param chunkSize The size of one chunk.
     */
    cacheDiamond(tilemap: Tilemap, chunkSize: number): void {
        tilemap.cacheDiamond(chunkSize);
    }
};
export default tilemapsLib;
