import {required, default as uLib} from './u';
import type {TextureShape, ExportedTiledTexture, ExportedFolder, ExportedAsset} from '../node_requires/exporter/_exporterContracts';
// eslint-disable-next-line @typescript-eslint/no-unused-vars
import type {sound as pixiSound, Sound, Options as SoundOptions} from 'node_modules/@pixi/sound';
import {pixiSoundPrefix, exportedSounds, soundMap, pixiSoundInstances} from './sounds.js';

type AssetType = 'template' | 'room' | 'sound' | 'style' | 'texture' | 'tandem' | 'font' | 'behavior' | 'script';

import * as pixiMod from 'pixi.js';
declare var PIXI: typeof pixiMod & {
    sound: typeof pixiSound
};

export type CtjsTexture = pixiMod.Texture & {
    shape: TextureShape,
    hitArea: pixiMod.Polygon | pixiMod.Circle | pixiMod.Rectangle;
    defaultAnchor: {
        x: number,
        y: number
    }
};
export type CtjsAnimation = CtjsTexture[] & {
    shape: TextureShape,
    hitArea: pixiMod.Polygon | pixiMod.Circle | pixiMod.Rectangle;
    defaultAnchor: {
        x: number,
        y: number
    }
};

export interface ITextureOptions {
    anchor?: {
        x: number,
        y: number
    }
    shape?: TextureShape
}

const loadingScreen = document.querySelector('.ct-aLoadingScreen') as HTMLDivElement,
      loadingBar = loadingScreen.querySelector('.ct-aLoadingBar') as HTMLDivElement;

export const textures: Record<string, CtjsAnimation> = {};
export const skeletons: Record<string, any> = {};

const normalizeAssetPath = (path: string): string[] => {
    path = path.replace(/\\/g, '/');
    if (path[0] === '/') {
        path = path.slice(1);
    }
    return path.split('/').filter(empty => empty);
};
const getEntriesByPath = (nPath: string[]): (ExportedAsset | ExportedFolder)[] => {
    if (!resLib.tree) {
        throw new Error('[res] Asset tree was not exported; check your project\'s export settings.');
    }
    let current = resLib.tree;
    for (const subpath of nPath) {
        const folder = current.find(i => i.name === subpath && i.type === 'folder');
        if (!folder) {
            throw new Error(`[res] Could not find folder ${subpath} in path ${nPath.join('/')}`);
        }
        current = (folder as ExportedFolder).entries;
    }
    return current;
};

/**
 * An object that manages and stores textures and other assets,
 * also exposing API for dynamic asset loading.
 */
const resLib = {
    sounds: soundMap,
    pixiSounds: pixiSoundInstances,
    textures: {} as Record<string, CtjsAnimation>,
    tree: [/*!@assetTree@*/][0] as (ExportedFolder | ExportedAsset)[] | false,
    /**
     * Loads and executes a script by its URL
     * @param {string} url The URL of the script file, with its extension.
     * Can be relative or absolute.
     * @returns {Promise<void>}
     * @async
     */
    loadScript(url: string = required('url', 'ct.res.loadScript')): Promise<void> {
        var script = document.createElement('script');
        script.src = url;
        const promise = new Promise<void>((resolve, reject) => {
            script.onload = () => {
                resolve();
            };
            script.onerror = () => {
                reject();
            };
        });
        document.getElementsByTagName('head')[0].appendChild(script);
        return promise;
    },
    /**
     * Loads an individual image as a named ct.js texture.
     * @param {string|boolean} url The path to the source image.
     * @param {string} name The name of the resulting ct.js texture
     * as it will be used in your code.
     * @param {ITextureOptions} textureOptions Information about texture's axis
     * and collision shape.
     * @returns {Promise<CtjsAnimation>} The imported animation, ready to be used.
     */
    async loadTexture(
        url: string = required('url', 'ct.res.loadTexture'),
        name: string = required('name', 'ct.res.loadTexture'),
        textureOptions: ITextureOptions = {}
    ): Promise<CtjsAnimation> {
        let texture: CtjsTexture;
        try {
            texture = await PIXI.Assets.load(url);
        } catch (e) {
            console.error(`[ct.res] Could not load image ${url}`);
            throw e;
        }
        const ctTexture = [texture] as CtjsAnimation;
        ctTexture.shape = texture.shape = textureOptions.shape || ({} as TextureShape);
        texture.defaultAnchor = ctTexture.defaultAnchor = new PIXI.Point(
            textureOptions.anchor ? textureOptions.anchor.x : 0,
            textureOptions.anchor ? textureOptions.anchor.y : 0
        );
        const hitArea = uLib.getHitArea(texture.shape);
        if (hitArea) {
            texture.hitArea = ctTexture.hitArea = hitArea;
        }
        resLib.textures[name] = ctTexture;
        return ctTexture;
    },
    /**
     * Loads a Texture Packer compatible .json file with its source image,
     * adding ct.js textures to the game.
     * @param {string} url The path to the JSON file that describes the atlas' textures.
     * @returns A promise that resolves into an array
     * of all the loaded textures' names.
     */
    async loadAtlas(url: string = required('url', 'ct.res.loadAtlas')): Promise<string[]> {
        const sheet = await PIXI.Assets.load<pixiMod.Spritesheet>(url);
        for (const animation in sheet.animations) {
            const tex = sheet.animations[animation];
            const animData = sheet.data.animations as pixiMod.utils.Dict<string[]>;
            for (let i = 0, l = animData[animation].length; i < l; i++) {
                const a = animData[animation],
                      f = a[i];
                (tex[i] as CtjsTexture).shape = (
                    sheet.data.frames[f] as pixiMod.ISpritesheetFrameData & {shape: TextureShape}
                ).shape;
            }
            (tex as CtjsAnimation).shape = (tex[0] as CtjsTexture).shape || ({} as TextureShape);
            resLib.textures[animation] = tex as CtjsAnimation;
            const hitArea = uLib.getHitArea(resLib.textures[animation].shape);
            if (hitArea) {
                resLib.textures[animation].hitArea = hitArea;
                for (const frame of resLib.textures[animation]) {
                    frame.hitArea = hitArea;
                }
            }
        }
        return Object.keys(sheet.animations);
    },
    /**
     * Unloads the specified atlas by its URL and removes all the textures
     * it has introduced to the game.
     * Will do nothing if the specified atlas was not loaded (or was already unloaded).
     */
    async unloadAtlas(url: string = required('url', 'ct.res.unloadAtlas')): Promise<void> {
        const {animations} = PIXI.Assets.get(url);
        if (!animations) {
            // eslint-disable-next-line no-console
            console.log(`[ct.res] Attempt to unload an atlas that was not loaded/was unloaded already: ${url}`);
            return;
        }
        for (const animation of animations) {
            delete resLib.textures[animation];
        }
        await PIXI.Assets.unload(url);
    },
    /**
     * Loads a bitmap font by its XML file.
     * @param url The path to the XML file that describes the bitmap fonts.
     * @returns A promise that resolves into the font's name (the one you've passed with `name`).
     */
    async loadBitmapFont(url: string = required('url', 'ct.res.loadBitmapFont')): Promise<void> {
        await PIXI.Assets.load(url);
    },
    async unloadBitmapFont(url: string = required('url', 'ct.res.unloadBitmapFont')): Promise<void> {
        await PIXI.Assets.unload(url);
    },
    /**
     * Loads a sound.
     * @param path Path to the sound
     * @param name The name of the sound as it will be used in ct.js game.
     * @param preload Whether to start loading now or postpone it.
     * Postponed sounds will load when a game tries to play them, or when you manually
     * trigger the download with `sounds.load(name)`.
     * @returns A promise with the name of the imported sound.
     */
    loadSound(
        path: string = required('path', 'ct.res.loadSound'),
        name: string = required('name', 'ct.res.loadSound'),
        preload = true
    ): Promise<string> {
        return new Promise<string>((resolve, reject) => {
            const opts: SoundOptions = {
                url: path,
                preload
            };
            if (preload) {
                opts.loaded = (err) => {
                    if (err) {
                        reject(err);
                    } else {
                        resLib.pixiSounds[name] = asset;
                        resolve(name);
                    }
                };
            }
            const asset = PIXI.sound.add(name, opts);
            if (!preload) {
                resolve(name);
            }
        });
    },

    async loadGame(): Promise<void> {
        // !! This method is intended to be filled by ct.IDE and be executed
        // exactly once at game startup.
        const changeProgress = (percents: number) => {
            loadingScreen.setAttribute('data-progress', String(percents));
            loadingBar.style.width = percents + '%';
        };

        /* eslint-disable prefer-destructuring */
        const atlases: string[] = [/*!@atlases@*/][0];
        const tiledImages: ExportedTiledTexture[] = [/*!@tiledImages@*/][0];
        const bitmapFonts: string[] = [/*!@bitmapFonts@*/][0];
        /* eslint-enable prefer-destructuring */

        const totalAssets = atlases.length;
        let assetsLoaded = 0;
        const loadingPromises: Promise<unknown>[] = [];

        loadingPromises.push(...atlases.map(atlas =>
            resLib.loadAtlas(atlas)
            .then(texturesNames => {
                assetsLoaded++;
                changeProgress(assetsLoaded / totalAssets * 100);
                return texturesNames;
            })));

        for (const name in tiledImages) {
            loadingPromises.push(resLib.loadTexture(
                tiledImages[name].source,
                name,
                {
                    anchor: tiledImages[name].anchor,
                    shape: tiledImages[name].shape
                }
            ));
        }
        for (const font in bitmapFonts) {
            loadingPromises.push(resLib.loadBitmapFont(bitmapFonts[font]));
        }
        for (const sound of exportedSounds) {
            for (const variant of sound.variants) {
                loadingPromises.push(resLib.loadSound(
                    variant.source,
                    `${pixiSoundPrefix}${variant.uid}`,
                    sound.preload
                ));
            }
        }

        /*!@res@*/
        /*!%res%*/

        await Promise.all(loadingPromises);
        loadingScreen.classList.add('hidden');
    },
    /**
     * Gets a pixi.js texture from a ct.js' texture name,
     * so that it can be used in pixi.js objects.
     * @param name The name of the ct.js texture, or -1 for an empty texture
     * @catnipAsset name:texture
     * @param [frame] The frame to extract
     * @returns {PIXI.Texture|PIXI.Texture[]} If `frame` was specified,
     * returns a single PIXI.Texture. Otherwise, returns an array
     * with all the frames of this ct.js' texture.
     */
    getTexture: ((
        name: string | -1,
        frame?: number
    ): CtjsAnimation | CtjsTexture | pixiMod.Texture | pixiMod.Texture[] => {
        if (frame === null) {
            frame = void 0;
        }
        if (name === -1) {
            if (frame !== void 0) {
                return PIXI.Texture.EMPTY;
            }
            return [PIXI.Texture.EMPTY];
        }
        if (!(name in resLib.textures)) {
            throw new Error(`Attempt to get a non-existent texture ${name}`);
        }
        const tex = resLib.textures[name];
        if (frame !== void 0) {
            return tex[frame];
        }
        return tex;
    }) as {
        (name: -1): [pixiMod.Texture];
        (name: -1, frame: 0): typeof PIXI.Texture.EMPTY;
        (name: -1, frame: number): never;
        (name: string): CtjsAnimation;
        (name: string, frame: number): CtjsTexture;
    },
    /**
     * Returns the collision shape of the given texture.
     * @param name The name of the ct.js texture, or -1 for an empty collision shape
     * @catnipAsset name:texture
     */
    getTextureShape(name: string | -1): TextureShape {
        if (name === -1) {
            return {
                type: 'point'
            };
        }
        if (!(name in resLib.textures)) {
            throw new Error(`Attempt to get a shape of a non-existent texture ${name}`);
        }
        return resLib.textures[name].shape;
    },
    /**
     * Gets direct children of a folder
     * @catnipIcon folder
     */
    getChildren(path?: string): ExportedAsset[] {
        return getEntriesByPath(normalizeAssetPath(path || ''))
            .filter(entry => entry.type !== 'folder') as ExportedAsset[];
    },
    /**
     * Gets direct children of a folder, filtered by asset type
     * @catnipIcon folder
     */
    getOfType(type: AssetType | 'folder', path?: string): (ExportedAsset | ExportedFolder)[] {
        return getEntriesByPath(normalizeAssetPath(path || ''))
            .filter(entry => entry.type === type);
    },
    /**
     * Gets all the assets inside of a folder, including in subfolders.
     * @catnipIcon folder
     */
    getAll(path?: string): ExportedAsset[] {
        const folderEntries = getEntriesByPath(normalizeAssetPath(path || '')),
              entries: ExportedAsset[] = [];
        const walker = (currentList: (ExportedFolder | ExportedAsset)[]) => {
            for (const entry of currentList) {
                if (entry.type === 'folder') {
                    walker(entry.entries);
                } else {
                    entries.push(entry);
                }
            }
        };
        walker(folderEntries);
        return entries;
    },
    /**
     * Get all the assets inside of a folder, including in subfolders, filtered by type.
     * @catnipIcon folder
     */
    getAllOfType(type: AssetType | 'folder', path?: string): (ExportedAsset | ExportedFolder)[] {
        const folderEntries = getEntriesByPath(normalizeAssetPath(path || '')),
              entries: (ExportedAsset | ExportedFolder)[] = [];
        const walker = (currentList: (ExportedFolder | ExportedAsset)[]) => {
            for (const entry of currentList) {
                if (entry.type === 'folder') {
                    walker(entry.entries);
                }
                // No `else` to allow querying for all subfolders
                if (entry.type === type) {
                    entries.push(entry);
                }
            }
        };
        walker(folderEntries);
        return entries;
    }
};


/*!@fonts@*/

export default resLib;
