"""
Schema definitions for `OpenAPI` generation and validation of data from received requests and returned responses.

This module should contain any and every definition in use to build the Swagger UI and the OpenAPI JSON schema
so that one can update the specification without touching any other files after the initial integration.

Schemas defined in this module are employed (through ``deserialize`` method calls) to validate that data conforms to
reported definitions. This makes the documentation of the API better aligned with resulting code execution under it.
It also provides a reference point for external users to understand expected data structures with complete schema
definitions generated on the exposed endpoints (JSON and Swagger UI).

The definitions are also employed to generate the `OpenAPI` definitions reported in the documentation published
on `Weaver`'s `ReadTheDocs` page.
"""

# pylint: disable=C0103,invalid-name
# pylint: disable=E0241,duplicate-bases

import datetime
import inspect
import os
import re
from copy import copy
from decimal import Decimal
from typing import TYPE_CHECKING

import colander
import duration
import jsonschema
import yaml
from babel.numbers import list_currencies
from colander import All, DateTime, Email as EmailRegex, Length, Money, OneOf, Range, Regex, drop, null, required
from dateutil import parser as date_parser
from pygeofilter.backends.cql2_json import to_cql2
from pygeofilter.parsers import cql2_json, cql2_text, cql_json, ecql, jfe

# FIXME: https://github.com/geopython/pygeofilter/pull/102
from pygeofilter.parsers.fes.parser import parse as fes_parse

from weaver import WEAVER_SCHEMA_DIR, __meta__
from weaver.compat import cache
from weaver.config import WeaverFeature
from weaver.execute import (
    ExecuteCollectionFormat,
    ExecuteControlOption,
    ExecuteMode,
    ExecuteResponse,
    ExecuteTransmissionMode
)
from weaver.formats import (
    EDAM_NAMESPACE,
    EDAM_NAMESPACE_URL,
    IANA_NAMESPACE,
    IANA_NAMESPACE_URL,
    OGC_NAMESPACE,
    OGC_NAMESPACE_URL,
    OPENGIS_NAMESPACE,
    OPENGIS_NAMESPACE_URL,
    AcceptLanguage,
    ContentType,
    OutputFormat
)
from weaver.owsexceptions import OWSMissingParameterValue
from weaver.processes.constants import (
    CWL_NAMESPACE_CWL_SPEC_ID,
    CWL_NAMESPACE_CWL_SPEC_URL,
    CWL_NAMESPACE_CWLTOOL_ID,
    CWL_NAMESPACE_CWLTOOL_URL,
    CWL_NAMESPACE_OGC_API_PROC_PART1_ID,
    CWL_NAMESPACE_OGC_API_PROC_PART1_URL,
    CWL_NAMESPACE_SCHEMA_ID,
    CWL_NAMESPACE_SCHEMA_METADATA_AUTHOR,
    CWL_NAMESPACE_SCHEMA_METADATA_CODE_REPOSITORY,
    CWL_NAMESPACE_SCHEMA_METADATA_CONTRIBUTOR,
    CWL_NAMESPACE_SCHEMA_METADATA_DATE_CREATED,
    CWL_NAMESPACE_SCHEMA_METADATA_EMAIL,
    CWL_NAMESPACE_SCHEMA_METADATA_IDENTIFIER,
    CWL_NAMESPACE_SCHEMA_METADATA_KEYWORDS,
    CWL_NAMESPACE_SCHEMA_METADATA_LICENSE,
    CWL_NAMESPACE_SCHEMA_METADATA_NAME,
    CWL_NAMESPACE_SCHEMA_METADATA_PERSON,
    CWL_NAMESPACE_SCHEMA_METADATA_SOFTWARE_VERSION,
    CWL_NAMESPACE_SCHEMA_METADATA_VERSION,
    CWL_NAMESPACE_SCHEMA_URL,
    CWL_NAMESPACE_WEAVER_ID,
    CWL_NAMESPACE_WEAVER_URL,
    CWL_REQUIREMENT_APP_BUILTIN,
    CWL_REQUIREMENT_APP_DOCKER,
    CWL_REQUIREMENT_APP_DOCKER_GPU,
    CWL_REQUIREMENT_APP_ESGF_CWT,
    CWL_REQUIREMENT_APP_OGC_API,
    CWL_REQUIREMENT_APP_WPS1,
    CWL_REQUIREMENT_CUDA,
    CWL_REQUIREMENT_INIT_WORKDIR,
    CWL_REQUIREMENT_INLINE_JAVASCRIPT,
    CWL_REQUIREMENT_INPLACE_UPDATE,
    CWL_REQUIREMENT_LOAD_LISTING,
    CWL_REQUIREMENT_MULTIPLE_INPUT,
    CWL_REQUIREMENT_NETWORK_ACCESS,
    CWL_REQUIREMENT_RESOURCE,
    CWL_REQUIREMENT_SCATTER,
    CWL_REQUIREMENT_SECRETS,
    CWL_REQUIREMENT_STEP_INPUT_EXPRESSION,
    CWL_REQUIREMENT_SUBWORKFLOW,
    CWL_REQUIREMENT_TIME_LIMIT,
    CWL_REQUIREMENT_WORK_REUSE,
    CWL_REQUIREMENTS_SUPPORTED,
    OAS_COMPLEX_TYPES,
    OAS_DATA_TYPES,
    PACKAGE_ARRAY_BASE,
    PACKAGE_ARRAY_ITEMS,
    PACKAGE_CUSTOM_TYPES,
    PACKAGE_ENUM_BASE,
    PACKAGE_TYPE_POSSIBLE_VALUES,
    WPS_LITERAL_DATA_TYPES,
    JobInputsOutputsSchema,
    JobStatusSchema,
    ProcessSchema
)
from weaver.provenance import ProvenanceFormat
from weaver.quotation.status import QuoteStatus
from weaver.sort import Sort, SortMethods
from weaver.status import JOB_STATUS_CODE_API, JOB_STATUS_SEARCH_API, Status
from weaver.utils import AWS_S3_BUCKET_REFERENCE_PATTERN, json_hashable, load_file, repr_json
from weaver.visibility import Visibility
from weaver.wps_restapi.colander_extras import (
    NO_DOUBLE_SLASH_PATTERN,
    AllOfKeywordSchema,
    AnyOfKeywordSchema,
    BoundedRange,
    CommaSeparated,
    EmptyMappingSchema,
    ExpandStringList,
    ExtendedBoolean as Boolean,
    ExtendedFloat as Float,
    ExtendedInteger as Integer,
    ExtendedMappingSchema,
    ExtendedSchemaNode,
    ExtendedSequenceSchema,
    ExtendedString as String,
    NoneType,
    NotKeywordSchema,
    OAS3DefinitionHandler,
    OneOfCaseInsensitive,
    OneOfKeywordSchema,
    PermissiveMappingSchema,
    PermissiveSequenceSchema,
    SchemeURL,
    SemanticVersion,
    StrictMappingSchema,
    StringOneOf,
    StringRange,
    XMLObject
)
from weaver.wps_restapi.constants import ConformanceCategory
from weaver.wps_restapi.patches import WeaverService as Service  # warning: don't use 'cornice.Service'

if TYPE_CHECKING:
    from typing import Any, Dict, List, Type, Union
    from typing_extensions import TypedDict

    from pygeofilter.ast import AstType as FilterAstType

    from weaver.typedefs import DatetimeIntervalType, JSON

    ViewInfo = TypedDict("ViewInfo", {"name": str, "pattern": str})


WEAVER_CONFIG_REMOTE_LIST = f"[{', '.join(WeaverFeature.REMOTE)}]"

API_TITLE = "Weaver REST API"
API_INFO = {
    "description": __meta__.__description__,
    "contact": {"name": __meta__.__authors__, "email": __meta__.__emails__, "url": __meta__.__source_repository__}
}
API_DOCS = {
    "description": f"{__meta__.__title__} documentation",
    "url": __meta__.__documentation_url__
}
DOC_URL = f"{__meta__.__documentation_url__}/en/latest"

CWL_VERSION = "v1.2"
CWL_REPO_URL = "https://github.com/common-workflow-language"
CWL_SCHEMA_BRANCH = "v1.2.1"
CWL_SCHEMA_PATH = "json-schema/cwl.yaml"
CWL_SCHEMA_REPO = f"https://raw.githubusercontent.com/common-workflow-language/cwl-{CWL_VERSION}"
CWL_SCHEMA_URL = f"{CWL_SCHEMA_REPO}/{CWL_SCHEMA_BRANCH}/{CWL_SCHEMA_PATH}"
CWL_BASE_URL = "https://www.commonwl.org"
CWL_SPEC_URL = f"{CWL_BASE_URL}/#Specification"
CWL_USER_GUIDE_URL = f"{CWL_BASE_URL}/user_guide"
CWL_DOC_BASE_URL = f"{CWL_BASE_URL}/{CWL_VERSION}"
CWL_CMD_TOOL_URL = f"{CWL_DOC_BASE_URL}/CommandLineTool.html"
CWL_WORKFLOW_URL = f"{CWL_DOC_BASE_URL}/Workflow.html"
CWL_DOC_MESSAGE = (
    "Note that multiple formats are supported and not all specification variants or parameters "
    f"are presented here. Please refer to official CWL documentation for more details ({CWL_BASE_URL})."
)

IO_INFO_IDS = (
    "Identifier of the {first} {what}. To merge details between corresponding {first} and {second} "
    "{what} specifications, this is the value that will be used to associate them together."
)

# development references
OGC_API_REPO_URL = "https://github.com/opengeospatial/ogcapi-processes"
OGC_API_SCHEMA_URL = "https://raw.githubusercontent.com/opengeospatial/ogcapi-processes"
OGC_API_SCHEMA_VERSION = "master"
OGC_API_SCHEMA_BASE = f"{OGC_API_SCHEMA_URL}/{OGC_API_SCHEMA_VERSION}"
OGC_API_SCHEMA_CORE = f"{OGC_API_SCHEMA_BASE}/openapi/schemas/processes-core"
OGC_API_EXAMPLES_CORE = f"{OGC_API_SCHEMA_BASE}/core/examples"
# FIXME: OGC OpenAPI schema restructure (https://github.com/opengeospatial/ogcapi-processes/issues/319)
# OGC_API_SCHEMA_EXT_DEPLOY = f"{OGC_API_SCHEMA_BASE}/openapi/schemas/processes-dru"
OGC_API_SCHEMA_EXT_DEPLOY = f"{OGC_API_SCHEMA_BASE}/openapi/schemas/processes-dru"
OGC_API_EXAMPLES_EXT_DEPLOY = f"{OGC_API_SCHEMA_BASE}/extensions/deploy_replace_undeploy/examples"
# not available yet:
OGC_API_SCHEMA_EXT_BILL = f"{OGC_API_SCHEMA_BASE}/openapi/schemas/processes-billing"
OGC_API_SCHEMA_EXT_QUOTE = f"{OGC_API_SCHEMA_BASE}/openapi/schemas/processes-quotation"
OGC_API_SCHEMA_EXT_WORKFLOW = f"{OGC_API_SCHEMA_BASE}/openapi/schemas/processes-workflows"

# official/published references
OGC_API_SCHEMAS_URL = "https://schemas.opengis.net"
OGC_API_COMMON_PART1_BASE = f"{OGC_API_SCHEMAS_URL}/ogcapi/common/part1/1.0"
OGC_API_COMMON_PART1_SCHEMAS = f"{OGC_API_COMMON_PART1_BASE}/openapi/schemas"
OGC_API_PROC_PART1_BASE = f"{OGC_API_SCHEMAS_URL}/ogcapi/processes/part1/1.0"
OGC_API_PROC_PART1_SCHEMAS = f"{OGC_API_PROC_PART1_BASE}/openapi/schemas"
OGC_API_PROC_PART1_RESPONSES = f"{OGC_API_PROC_PART1_BASE}/openapi/responses"
OGC_API_PROC_PART1_PARAMETERS = f"{OGC_API_PROC_PART1_BASE}/openapi/parameters"
OGC_API_PROC_PART1_EXAMPLES = f"{OGC_API_PROC_PART1_BASE}/examples"
OGC_WPS_1_SCHEMAS = f"{OGC_API_SCHEMAS_URL}/wps/1.0.0"
OGC_WPS_2_SCHEMAS = f"{OGC_API_SCHEMAS_URL}/wps/2.0"

# Because this type has special handling functionalities to distinguish it from any other usual 'complex' I/O
# or any generic JSON-object data, define common constants that can be reused across the code.
# If this changes later on, it will be easier to ensure backward compatibility with explicit references to it.
OGC_API_BBOX_SCHEMA = f"{OGC_API_PROC_PART1_SCHEMAS}/bbox.yaml"
OGC_API_BBOX_FORMAT = "ogc-bbox"  # equal CRS:84 and EPSG:4326, equivalent to WGS84 with swapped lat-lon order
OGC_API_BBOX_EPSG = "EPSG:4326"

OGC_API_SCHEMA_JOB_STATUS_URL = f"{OGC_API_PROC_PART1_SCHEMAS}/statusInfo.yaml"

OPENEO_API_SCHEMA_URL = "https://openeo.org/documentation/1.0/developers/api/openapi.yaml"
OPENEO_API_SCHEMA_JOB_STATUS_URL = f"{OPENEO_API_SCHEMA_URL}#/components/schemas/batch_job"

WEAVER_SCHEMA_VERSION = "master"
WEAVER_SCHEMA_URL = f"https://raw.githubusercontent.com/crim-ca/weaver/{WEAVER_SCHEMA_VERSION}/weaver/schemas"

DATETIME_INTERVAL_CLOSED_SYMBOL = "/"
DATETIME_INTERVAL_OPEN_START_SYMBOL = "../"
DATETIME_INTERVAL_OPEN_END_SYMBOL = "/.."

# fields ordering for generation of ProcessDescription body (shared for OGC/OLD schema format)
PROCESS_DESCRIPTION_FIELD_FIRST = [
    "id",
    "title",
    "version",
    "mutable",
    "abstract",  # backward compat for deployment
    "description",
    "keywords",
    "metadata",
    "inputs",
    "outputs"
]
PROCESS_DESCRIPTION_FIELD_AFTER = [
    "processDescriptionURL",
    "processEndpointWPS1",
    "executeEndpoint",
    "deploymentProfile",
    "links"
]
# fields ordering for nested process definition of OLD schema format of ProcessDescription
PROCESS_DESCRIPTION_FIELD_FIRST_OLD_SCHEMA = ["process"]
PROCESS_DESCRIPTION_FIELD_AFTER_OLD_SCHEMA = ["links"]

PROCESS_IO_FIELD_FIRST = ["id", "title", "description", "minOccurs", "maxOccurs"]
PROCESS_IO_FIELD_AFTER = ["literalDataDomains", "formats", "crs", "bbox"]

PROCESSES_LISTING_FIELD_FIRST = ["description", "processes", "providers"]
PROCESSES_LISTING_FIELD_AFTER = ["page", "limit", "count", "total", "links"]

PROVIDER_DESCRIPTION_FIELD_FIRST = [
    "id",
    "title",
    "version",
    "mutable",
    "description",
    "url",
    "type",
    "public",
    "keywords",
    "metadata",
]
PROVIDER_DESCRIPTION_FIELD_AFTER = ["links"]

JOBS_LISTING_FIELD_FIRST = ["description", "jobs", "groups"]
JOBS_LISTING_FIELD_AFTER = ["page", "limit", "count", "total", "links"]

QUOTES_LISTING_FIELD_FIRST = ["description", "quotations"]
QUOTES_LISTING_FIELD_AFTER = ["page", "limit", "count", "total", "links"]

#########################################################
# Examples
#########################################################

# load examples by file names as keys
SCHEMA_EXAMPLE_DIR = os.path.join(os.path.dirname(__file__), "examples")
EXAMPLES = {}
for _name in os.listdir(SCHEMA_EXAMPLE_DIR):
    _path = os.path.join(SCHEMA_EXAMPLE_DIR, _name)
    _ext = os.path.splitext(_name)[-1]
    with open(_path, "r", encoding="utf-8") as f:
        if _ext in [".json", ".yaml", ".yml"]:
            EXAMPLES[_name] = yaml.safe_load(f)  # both JSON/YAML
        else:
            EXAMPLES[_name] = f.read()


#########################################################
# API tags
#########################################################

TAG_API = "API"
TAG_JOBS = "Jobs"
TAG_VISIBILITY = "Visibility"
TAG_BILL_QUOTE = "Billing & Quoting"
TAG_PROVIDERS = "Providers"
TAG_PROCESSES = "Processes"
TAG_GETCAPABILITIES = "GetCapabilities"
TAG_DESCRIBEPROCESS = "DescribeProcess"
TAG_EXECUTE = "Execute"
TAG_DISMISS = "Dismiss"
TAG_STATUS = "Status"
TAG_DEPLOY = "Deploy"
TAG_RESULTS = "Results"
TAG_EXCEPTIONS = "Exceptions"
TAG_LOGS = "Logs"
TAG_STATISTICS = "Statistics"
TAG_PROVENANCE = "Provenance"
TAG_VAULT = "Vault"
TAG_WPS = "WPS"
TAG_DEPRECATED = "Deprecated Endpoints"

###############################################################################
# API endpoints
# These "services" are wrappers that allow Cornice to generate the JSON API
###############################################################################

api_frontpage_service = Service(name="api_frontpage", path="/")
api_openapi_ui_service = Service(name="api_openapi_ui", path="/api")  # idem to swagger
api_swagger_ui_service = Service(name="api_swagger_ui", path="/swagger")
api_redoc_ui_service = Service(name="api_redoc_ui", path="/redoc")
api_versions_service = Service(name="api_versions", path="/versions")
api_conformance_service = Service(name="api_conformance", path="/conformance")
openapi_json_service = Service(name="openapi_json", path="/json")

quotes_service = Service(name="quotes", path="/quotations")
quote_service = Service(name="quote", path=f"{quotes_service.path}/{{quote_id}}")
bills_service = Service(name="bills", path="/bills")
bill_service = Service(name="bill", path=f"{bills_service.path}/{{bill_id}}")

jobs_service = Service(name="jobs", path="/jobs")
job_service = Service(name="job", path=f"{jobs_service.path}/{{job_id}}")
job_results_service = Service(name="job_results", path=f"{job_service.path}/results")
job_outputs_service = Service(name="job_outputs", path=f"{job_service.path}/outputs")
job_inputs_service = Service(name="job_inputs", path=f"{job_service.path}/inputs")
job_exceptions_service = Service(name="job_exceptions", path=f"{job_service.path}/exceptions")
job_logs_service = Service(name="job_logs", path=f"{job_service.path}/logs")
job_stats_service = Service(name="job_stats", path=f"{job_service.path}/statistics")
job_prov_service = Service(name="job_prov", path=f"{job_service.path}/prov")
job_prov_info_service = Service(name="job_prov_info", path=f"{job_prov_service.path}/info")
job_prov_who_service = Service(name="job_prov_who", path=f"{job_prov_service.path}/who")
job_prov_inputs_service = Service(name="job_prov_inputs", path=f"{job_prov_service.path}/inputs")
job_prov_inputs_run_service = Service(name="job_prov_inputs_run", path=f"{job_prov_service.path}/inputs/{{run_id}}")
job_prov_outputs_service = Service(name="job_prov_outputs", path=f"{job_prov_service.path}/outputs")
job_prov_outputs_run_service = Service(name="job_prov_outputs_run", path=f"{job_prov_service.path}/outputs/{{run_id}}")
job_prov_run_service = Service(name="job_prov_run", path=f"{job_prov_service.path}/run")
job_prov_run_id_service = Service(name="job_prov_run_id", path=f"{job_prov_service.path}/run/{{run_id}}")
job_prov_runs_service = Service(name="job_prov_runs", path=f"{job_prov_service.path}/runs")

processes_service = Service(name="processes", path="/processes")
process_service = Service(name="process", path=f"{processes_service.path}/{{process_id}}")
process_quotes_service = Service(name="process_quotes", path=process_service.path + quotes_service.path)
process_quote_service = Service(name="process_quote", path=process_service.path + quote_service.path)
process_estimator_service = Service(name="process_estimator_service", path=f"{process_service.path}/estimator")
process_visibility_service = Service(name="process_visibility", path=f"{process_service.path}/visibility")
process_package_service = Service(name="process_package", path=f"{process_service.path}/package")
process_payload_service = Service(name="process_payload", path=f"{process_service.path}/payload")
process_execution_service = Service(name="process_execution", path=f"{process_service.path}/execution")
process_jobs_service = Service(name="process_jobs", path=process_service.path + jobs_service.path)
process_job_service = Service(name="process_job", path=process_service.path + job_service.path)
process_results_service = Service(name="process_results", path=process_service.path + job_results_service.path)
process_inputs_service = Service(name="process_inputs", path=process_service.path + job_inputs_service.path)
process_outputs_service = Service(name="process_outputs", path=process_service.path + job_outputs_service.path)
process_exceptions_service = Service(name="process_exceptions", path=process_service.path + job_exceptions_service.path)
process_logs_service = Service(name="process_logs", path=process_service.path + job_logs_service.path)
process_stats_service = Service(name="process_stats", path=process_service.path + job_stats_service.path)
process_prov_service = Service(name="process_prov", path=process_service.path + job_prov_service.path)
process_prov_info_service = Service(name="process_prov_info", path=process_service.path + job_prov_info_service.path)
process_prov_who_service = Service(name="process_prov_who", path=process_service.path + job_prov_who_service.path)
process_prov_inputs_service = Service(
    name="process_prov_inputs",
    path=process_service.path + job_prov_inputs_service.path,
)
process_prov_inputs_run_service = Service(
    name="process_prov_inputs_run",
    path=process_service.path + job_prov_inputs_run_service.path,
)
process_prov_outputs_service = Service(
    name="process_prov_outputs",
    path=process_service.path + job_prov_outputs_service.path,
)
process_prov_outputs_run_service = Service(
    name="process_prov_outputs_run",
    path=process_service.path + job_prov_outputs_run_service.path,
)
process_prov_run_service = Service(
    name="process_prov_run",
    path=process_service.path + job_prov_run_service.path,
)
process_prov_run_id_service = Service(
    name="process_prov_run_id",
    path=process_service.path + job_prov_run_id_service.path,
)
process_prov_runs_service = Service(
    name="process_prov_runs",
    path=process_service.path + job_prov_runs_service.path,
)

providers_service = Service(name="providers", path="/providers")
provider_service = Service(name="provider", path=f"{providers_service.path}/{{provider_id}}")
provider_processes_service = Service(name="provider_processes", path=provider_service.path + processes_service.path)
provider_process_service = Service(name="provider_process", path=provider_service.path + process_service.path)
provider_process_package_service = Service(name="provider_process_pkg", path=f"{provider_process_service.path}/package")
provider_execution_service = Service(name="provider_execution", path=f"{provider_process_service.path}/execution")
provider_jobs_service = Service(name="provider_jobs", path=provider_service.path + process_jobs_service.path)
provider_job_service = Service(name="provider_job", path=provider_service.path + process_job_service.path)
provider_results_service = Service(name="provider_results", path=provider_service.path + process_results_service.path)
provider_inputs_service = Service(name="provider_inputs", path=provider_service.path + process_inputs_service.path)
provider_outputs_service = Service(name="provider_outputs", path=provider_service.path + process_outputs_service.path)
provider_exceptions_service = Service(
    name="provider_exceptions",
    path=provider_service.path + process_exceptions_service.path,
)
provider_logs_service = Service(name="provider_logs", path=provider_service.path + process_logs_service.path)
provider_stats_service = Service(name="provider_stats", path=provider_service.path + process_stats_service.path)
provider_prov_service = Service(name="provider_prov", path=provider_service.path + process_prov_service.path)
provider_prov_info_service = Service(
    name="provider_prov_info",
    path=provider_service.path + process_prov_info_service.path,
)
provider_prov_who_service = Service(
    name="provider_prov_who",
    path=provider_service.path + process_prov_who_service.path,
)
provider_prov_inputs_service = Service(
    name="provider_prov_inputs",
    path=provider_service.path + process_prov_inputs_service.path,
)
provider_prov_inputs_run_service = Service(
    name="provider_prov_inputs_run",
    path=provider_service.path + process_prov_inputs_run_service.path,
)
provider_prov_outputs_service = Service(
    name="provider_prov_outputs",
    path=provider_service.path + process_prov_outputs_service.path,
)
provider_prov_outputs_run_service = Service(
    name="provider_prov_outputs_run",
    path=provider_service.path + process_prov_outputs_run_service.path,
)
provider_prov_run_service = Service(
    name="provider_prov_run",
    path=provider_service.path + process_prov_run_service.path,
)
provider_prov_run_id_service = Service(
    name="provider_prov_run_id",
    path=provider_service.path + process_prov_run_id_service.path,
)
provider_prov_runs_service = Service(
    name="provider_prov_runs",
    path=provider_service.path + process_prov_runs_service.path,
)

# backward compatibility deprecated routes
job_result_service = Service(name="job_result", path=f"{job_service.path}/result")
process_result_service = Service(name="process_result", path=process_service.path + job_result_service.path)
provider_result_service = Service(name="provider_result", path=provider_service.path + process_result_service.path)

vault_service = Service(name="vault", path="/vault")
vault_file_service = Service(name="vault_file", path=f"{vault_service.path}/{{file_id}}")

#########################################################
# Generic schemas
#########################################################


class SLUG(ExtendedSchemaNode):
    schema_type = String
    description = "Slug name pattern."
    example = "some-object-slug-name"
    pattern = re.compile(r"^[A-Za-z0-9]+(?:[-_][A-Za-z0-9]+)*$")


class Tag(ExtendedSchemaNode):
    schema_type = String
    description = "Identifier with optional tagged version forming a unique reference."
    # ranges used to remove starting/ending ^$ characters
    pattern = re.compile(
        rf"{SLUG.pattern.pattern[:-1]}"
        rf"(:{SemanticVersion(v_prefix=False, rc_suffix=False).pattern[1:-1]})?$"
    )


class URL(ExtendedSchemaNode):
    """
    String format that will be automatically mapped to a URL-pattern validator.

    .. seealso::
        - :data:`weaver.wps_restapi.colander_extras.URL`
        - :class:`weaver.wps_restapi.colander_extras.ExtendedSchemaBase`
    """
    schema_type = String
    description = "URL reference."
    format = "url"


class URI(ExtendedSchemaNode):
    """
    String format that will be automatically mapped to a URI-pattern validator.

    .. seealso::
        - :data:`weaver.wps_restapi.colander_extras.URI`
        - :class:`weaver.wps_restapi.colander_extras.ExtendedSchemaBase`
    """
    schema_type = String
    description = "URI reference."
    format = "uri"


class Email(ExtendedSchemaNode):
    schema_type = String
    description = "Email recipient."
    format = "email"
    validator = EmailRegex()


class MediaType(ExtendedSchemaNode):
    schema_type = String
    description = "IANA identifier of content and format."
    example = ContentType.APP_JSON
    pattern = re.compile(r"^\w+\/[-.\w]+(?:\+[-.\w]+)?(?:\;\s*.+)*$")


class QueryBoolean(Boolean):
    description = "Boolean query parameter that allows handles common truthy/falsy values."

    def __init__(self, *_, **__):
        # type: (*Any, **Any) -> None
        super(QueryBoolean, self).__init__(
            allow_string=True,
            false_choices=("False", "false", "0", "off", "no", "null", "Null", "none", "None", ""),
            true_choices=("True", "true", "1", "on", "yes")
        )


class DateTimeInterval(ExtendedSchemaNode):
    _schema = f"{OGC_API_PROC_PART1_PARAMETERS}/datetime.yaml"
    schema_type = String
    description = (
        "DateTime format against OGC API - Processes, "
        "to get values before a certain date-time use '../' before the date-time, "
        "to get values after a certain date-time use '/..' after the date-time like the example, "
        "to get values between two date-times use '/' between the date-times, "
        "to get values with a specific date-time just pass the datetime. "
    )
    example = "2022-03-02T03:32:38.487000+00:00/.."
    regex_datetime = re.compile(r"(\d{4}-\d\d-\d\dT\d\d:\d\d:\d\d(\.\d+)?(([+-]\d\d:\d\d)|Z)?)")
    regex_interval_closed = re.compile(rf"{regex_datetime.pattern}\/{regex_datetime.pattern}")
    regex_interval_open_start = re.compile(rf"\.\.\/{regex_datetime.pattern}")
    regex_interval_open_end = re.compile(rf"{regex_datetime.pattern}\/\.\.")
    pattern = re.compile(
        rf"^{regex_datetime.pattern}"
        rf"|{regex_interval_closed.pattern}"
        rf"|{regex_interval_open_start.pattern}"
        rf"|{regex_interval_open_end.pattern}"
        r"$"
    )


class S3BucketReference(ExtendedSchemaNode):
    schema_type = String
    description = "S3 bucket shorthand URL representation: 's3://{bucket}/[{dirs}/][{file-key}]'"
    pattern = AWS_S3_BUCKET_REFERENCE_PATTERN


class FileLocal(ExtendedSchemaNode):
    schema_type = String
    description = "Local file reference."
    format = "file"
    pattern = re.compile(rf"^(file://)?{NO_DOUBLE_SLASH_PATTERN}(?:/|[/?]\S+)$")


class FileURL(ExtendedSchemaNode):
    schema_type = String
    description = "URL file reference."
    format = "url"
    validator = SchemeURL(schemes=["http", "https"])


class VaultReference(ExtendedSchemaNode):
    schema_type = String
    description = "Vault file reference."
    example = "vault://399dc5ac-ff66-48d9-9c02-b144a975abe4"
    pattern = re.compile(r"^vault://[a-f0-9]{8}(?:-?[a-f0-9]{4}){3}-?[a-f0-9]{12}$")


class ProcessURL(ExtendedSchemaNode):
    schema_type = String
    description = "Process URL reference."
    format = "url"
    validator = SchemeURL(schemes=["http", "https"], path_pattern=r"(?:/processes/\S+/?)")


class ReferenceURL(AnyOfKeywordSchema):
    _any_of = [
        FileURL(),
        FileLocal(),
        S3BucketReference(),
    ]


class ExecuteReferenceURL(AnyOfKeywordSchema):
    _any_of = [
        FileURL(),
        FileLocal(),
        S3BucketReference(),
        VaultReference(),
    ]


class UUID(ExtendedSchemaNode):
    schema_type = String
    description = "Unique identifier."
    example = "a9d14bf4-84e0-449a-bac8-16e598efe807"
    format = "uuid"
    pattern = re.compile("^[a-f0-9]{8}(?:-?[a-f0-9]{4}){3}-?[a-f0-9]{12}$")
    title = "UUID"


class AnyIdentifier(SLUG):
    pass


class CWLFileName(SLUG):
    schema_type = String
    description = "File with a CWL extension."
    pattern = re.compile(
        f"{SLUG.pattern.pattern[:-1]}"  # remove '$'
        r"\.cwl$"
    )


class ProcessIdentifier(AnyOfKeywordSchema):
    description = "Process identifier."
    _any_of = [
        # UUID first because more strict than SLUG, and SLUG can be similar to UUID, but in the end any is valid
        UUID(description="Unique identifier."),
        SLUG(description="Generic identifier. This is a user-friendly slug-name. "
                         "Note that this will represent the latest process matching this name. "
                         "For specific process version, use the UUID instead.", title="ID"),
    ]


class ProcessIdentifierTag(AnyOfKeywordSchema):
    description = "Process identifier with optional revision tag."
    _schema = f"{OGC_API_PROC_PART1_PARAMETERS}/processIdPathParam.yaml"
    _any_of = [Tag] + ProcessIdentifier._any_of  # type: ignore  # noqa: W0212


class JobID(UUID):
    _schema = f"{OGC_API_PROC_PART1_PARAMETERS}/jobId.yaml"
    description = "ID of the job."
    example = "a9d14bf4-84e0-449a-bac8-16e598efe807"


class Version(ExtendedSchemaNode):
    schema_type = String
    description = "Version string."
    example = "1.2.3"
    validator = SemanticVersion()


class ContentTypeHeader(ExtendedSchemaNode):
    # ok to use 'name' in this case because target 'key' in the mapping must
    # be that specific value but cannot have a field named with this format
    name = "Content-Type"
    schema_type = String


class ContentLengthHeader(ExtendedSchemaNode):
    name = "Content-Length"
    schema_type = String
    example = "125"


class ContentDispositionHeader(ExtendedSchemaNode):
    name = "Content-Disposition"
    schema_type = String
    example = "attachment; filename=test.json"


class DateHeader(ExtendedSchemaNode):
    description = "Creation date and time of the contents."
    name = "Date"
    schema_type = String
    example = "Thu, 13 Jan 2022 12:37:19 GMT"


class LastModifiedHeader(ExtendedSchemaNode):
    description = "Modification date and time of the contents."
    name = "Last-Modified"
    schema_type = String
    example = "Thu, 13 Jan 2022 12:37:19 GMT"


class AcceptHeader(ExtendedSchemaNode):
    # ok to use 'name' in this case because target 'key' in the mapping must
    # be that specific value but cannot have a field named with this format
    name = "Accept"
    schema_type = String
    # FIXME: raise HTTPNotAcceptable in not one of those?
    validator = OneOf([
        ContentType.APP_JSON,
        ContentType.APP_YAML,
        ContentType.APP_XML,
        ContentType.TEXT_XML,
        ContentType.TEXT_HTML,
        ContentType.ANY,
    ])
    missing = drop
    default = ContentType.APP_JSON  # defaults to JSON for easy use within browsers


class AcceptLanguageHeader(ExtendedSchemaNode):
    # ok to use 'name' in this case because target 'key' in the mapping must
    # be that specific value but cannot have a field named with this format
    name = "Accept-Language"
    schema_type = String
    missing = drop
    default = AcceptLanguage.EN_CA
    # FIXME: oneOf validator for supported languages (?)


class JsonHeader(ExtendedMappingSchema):
    content_type = ContentTypeHeader(example=ContentType.APP_JSON, default=ContentType.APP_JSON)


class HtmlHeader(ExtendedMappingSchema):
    content_type = ContentTypeHeader(example=ContentType.TEXT_HTML, default=ContentType.TEXT_HTML)


class XmlHeader(ExtendedMappingSchema):
    content_type = ContentTypeHeader(example=ContentType.APP_XML, default=ContentType.APP_XML)


class XAuthDockerHeader(ExtendedSchemaNode):
    summary = "Authentication header for private Docker registry access."
    description = (
        "Authentication header for private registry access in order to retrieve the Docker image reference "
        "specified in an Application Package during Process deployment. When provided, this header should "
        "contain similar details as typical Authentication or X-Auth-Token headers "
        f"(see {DOC_URL}/package.html#dockerized-applications for more details)."
    )
    name = "X-Auth-Docker"
    example = "Basic {base64-auth-credentials}"
    schema_type = String
    missing = drop


class RequestContentTypeHeader(ContentTypeHeader):
    example = ContentType.APP_JSON
    default = ContentType.APP_JSON
    validator = OneOf([
        ContentType.APP_JSON,
        # ContentType.APP_XML,
    ])


class ResponseContentTypeHeader(ContentTypeHeader):
    example = ContentType.APP_JSON
    default = ContentType.APP_JSON
    validator = OneOf([
        ContentType.APP_JSON,
        ContentType.APP_XML,
        ContentType.TEXT_XML,
        ContentType.TEXT_HTML,
    ])


class ResponseContentTypePlainTextHeader(ContentTypeHeader):
    example = ContentType.TEXT_PLAIN
    default = ContentType.TEXT_PLAIN
    validator = OneOf([ContentType.TEXT_PLAIN])


class PreferHeader(ExtendedSchemaNode):
    summary = "Header that describes job execution parameters."
    description = (
        "Header that describes the desired execution mode of the process job and desired results. "
        "Parameter 'return' indicates the structure and contents how results should be returned. "
        "Parameter 'wait' and 'respond-async' indicate the execution mode of the process job. "
        f"For more details, see {DOC_URL}/processes.html#execution-mode and {DOC_URL}/processes.html#execution-results."
    )
    name = "Prefer"
    schema_type = String


class RequestHeaders(ExtendedMappingSchema):
    """
    Headers that can indicate how to adjust the behavior and/or result to be provided in the response.
    """
    accept = AcceptHeader()
    accept_language = AcceptLanguageHeader()
    content_type = RequestContentTypeHeader()


class ResponseHeaders(ExtendedMappingSchema):
    """
    Headers describing resulting response.
    """
    content_type = ResponseContentTypeHeader()


class ResponsePlainTextHeaders(ResponseHeaders):
    """
    Headers describing resulting response.
    """
    content_type = ResponseContentTypePlainTextHeader()


class RedirectHeaders(ResponseHeaders):
    Location = URL(example="https://job/123/result", description="Redirect resource location.")


class AcceptFormatHeaders(ExtendedMappingSchema):
    accept = AcceptHeader(description="Output format selector. Equivalent to 'f' or 'format' queries.")
    accept_language = AcceptLanguageHeader(description="Output content language if supported.")


class OutputFormatQuery(ExtendedSchemaNode):
    schema_type = String
    description = "Output format selector for requested contents."
    example = OutputFormat.JSON
    validator = OneOf(OutputFormat.values())


class FormatQueryValue(OneOfKeywordSchema):
    _one_of = [
        MediaType(),
        OutputFormatQuery()
    ]


class FormatQuery(ExtendedMappingSchema):
    f = FormatQueryValue(
        missing=drop,
        description="Output format selector. Equivalent to 'format' query or 'Accept' header."
    )
    format = FormatQueryValue(
        missing=drop,
        description="Output format selector. Equivalent to 'f' query or 'Accept' header."
    )


class FormatQueryJSON(ExtendedMappingSchema):
    f = OutputFormatQuery(
        title="OutputFormatShortQueryJSON",
        missing=drop,
        description="Output format selector. Equivalent to 'format' query or 'Accept' header.",
        validator=OneOf([OutputFormat.JSON]),
    )
    format = OutputFormatQuery(
        title="OutputFormatLongQueryJSON",
        missing=drop,
        description="Output format selector. Equivalent to 'f' query or 'Accept' header.",
        validator=OneOf([OutputFormat.JSON]),
    )


class NoContent(ExtendedMappingSchema):
    description = "Empty response body."
    default = {}


class FileUploadHeaders(RequestHeaders):
    # MUST be multipart for upload
    content_type = ContentTypeHeader(
        example=f"{ContentType.MULTIPART_FORM}; boundary=43003e2f205a180ace9cd34d98f911ff",
        default=ContentType.MULTIPART_FORM,
        description="Desired Content-Type of the file being uploaded.", missing=required)
    content_length = ContentLengthHeader(description="Uploaded file contents size in bytes.")
    content_disposition = ContentDispositionHeader(example="form-data; name=\"file\"; filename=\"desired-name.ext\"",
                                                   description="Expected ")


class FileUploadContent(ExtendedSchemaNode):
    schema_type = String()
    description = (
        "Contents of the file being uploaded with multipart. When prefixed with 'Content-Type: {media-type}', the "
        "specified format will be applied to the input that will be attributed the 'vault://{UUID}' during execution. "
        "Contents can also have 'Content-Disposition' definition to provide the desired file name."
    )


class FileResponseHeaders(NoContent):
    content_type = ContentTypeHeader(example=ContentType.APP_JSON)
    content_length = ContentLengthHeader()
    content_disposition = ContentDispositionHeader()
    date = DateHeader()
    last_modified = LastModifiedHeader()


class AccessToken(ExtendedSchemaNode):
    schema_type = String


class DescriptionSchema(ExtendedMappingSchema):
    description = ExtendedSchemaNode(String(), description="Description of the obtained contents.")


class KeywordList(ExtendedSequenceSchema):
    keyword = ExtendedSchemaNode(String(), validator=Length(min=1))


class Language(ExtendedSchemaNode):
    schema_type = String
    example = AcceptLanguage.EN_CA
    validator = OneOf(AcceptLanguage.values())


class ValueLanguage(ExtendedMappingSchema):
    lang = Language(missing=drop, description="Language of the value content.")


class LinkLanguage(ExtendedMappingSchema):
    hreflang = Language(missing=drop, description="Language of the content located at the link.")


class LinkHeader(ExtendedSchemaNode):
    schema_type = String
    example = "<http://example.com>; rel=\"relation\"; type=text/plain"


class MetadataBase(ExtendedMappingSchema):
    title = ExtendedSchemaNode(String(), missing=drop)


class MetadataRole(ExtendedMappingSchema):
    role = URL(missing=drop)


class LinkRelationshipType(OneOfKeywordSchema):
    description = (
        "Link relation as registered or extension type "
        "(see https://www.rfc-editor.org/rfc/rfc8288.html#section-2.1)."
    )
    _one_of = [
        SLUG(description=(
            "Relationship of the link to the current content. "
            "This should be one item amongst registered relations https://www.iana.org/assignments/link-relations/."
        )),
        URL(description="Fully qualified extension link relation to the current content.")
    ]


class LinkRelationship(ExtendedMappingSchema):
    rel = LinkRelationshipType()


class LinkBase(LinkLanguage, MetadataBase):
    href = URL(description="Hyperlink reference.")
    type = MediaType(description="IANA identifier of content-type located at the link.", missing=drop)


class Link(LinkRelationship, LinkBase):
    _schema = f"{OGC_API_COMMON_PART1_SCHEMAS}/link.json"
    _schema_include_deserialize = False  # only in OpenAPI otherwise too verbose


class MetadataValueField(OneOfKeywordSchema):
    _one_of = [
        # pointer to a file or JSON schema relative item (as in OpenAPI definitions)
        ExtendedSchemaNode(String(), description="Plain text value of the information."),
        # literal JSON schema, permissive since it can be anything
        PermissiveMappingSchema(description="Flexible schema definition for the metadata value.")
    ]


class MetadataValue(NotKeywordSchema, ValueLanguage, MetadataBase):
    _not = [
        # make sure value metadata does not allow 'rel' and 'hreflang' reserved for link reference
        # explicitly refuse them such that when a href/rel link is provided, only link details are possible
        LinkRelationship(description="Field 'rel' must refer to a link reference with 'href'."),
        LinkLanguage(description="Field 'hreflang' must refer to a link reference with 'href'."),
    ]
    value = MetadataValueField(description="Explicit schema definition of the metadata value.")


class MetadataLink(Link):
    pass


class MetadataContent(OneOfKeywordSchema):
    _one_of = [
        MetadataLink(),
        MetadataValue(),
    ]


class Metadata(MetadataContent, MetadataRole):
    pass


class MetadataList(ExtendedSequenceSchema):
    metadata = Metadata()


class LinkList(ExtendedSequenceSchema):
    description = "List of links relative to the applicable object."
    title = "Links"
    link = Link()


class LandingPage(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/landingPage.yaml"
    links = LinkList()


# sub-schema within:
#   https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/format.yaml
class FormatSchema(OneOfKeywordSchema):
    _one_of = [
        # pointer to a file or JSON schema relative item (as in OpenAPI definitions)
        ReferenceURL(description="Reference where the schema definition can be retrieved to describe referenced data."),
        # literal JSON schema, permissive since it can be anything
        PermissiveMappingSchema(description="Explicit schema definition of the formatted reference data.")
    ]

    # because some pre-existing processes + pywps default schema is ""
    # deserialization against the validator pattern of 'ReferenceURL' makes it always fail
    # this causes the whole 'Format' container (and others similar) fail and be dropped
    # to resolve this issue, preemptively detect the empty string and signal the parent OneOf to remove it
    def deserialize(self, cstruct):  # type: ignore
        if isinstance(cstruct, str) and cstruct == "":
            return drop  # field that refers to this schema will drop the field key entirely
        return super(FormatSchema, self).deserialize(cstruct)


class FormatMimeType(ExtendedMappingSchema):
    """
    Used to respect ``mimeType`` field to work with pre-existing processes.
    """
    mimeType = MediaType(default=ContentType.TEXT_PLAIN, example=ContentType.APP_JSON)
    encoding = ExtendedSchemaNode(String(), missing=drop)
    schema = FormatSchema(missing=drop)


class Format(ExtendedMappingSchema):
    """
    Used to respect ``mediaType`` field as suggested per `OGC-API`.
    """
    _schema_include_deserialize = False  # only in OpenAPI otherwise too verbose
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/format.yaml"

    mediaType = MediaType(default=ContentType.TEXT_PLAIN, example=ContentType.APP_JSON)
    encoding = ExtendedSchemaNode(String(), missing=drop)
    schema = FormatSchema(missing=drop)


class FormatDefaultMimeType(FormatMimeType):
    description = (
        "Format for process input are assumed plain/text if the media-type was omitted and is not one of the known "
        "formats by this instance. When executing a job, the best match against supported formats by the process "
        "definition will be used to run the process, and will fall back to the default as last resort."
    )
    # NOTE:
    # The default is overridden from FormatMimeType since the FormatSelection 'oneOf' always fails,
    # due to the 'default' value which is always generated, and it causes the presence of both Format and FormatMimeType
    mimeType = MediaType(example=ContentType.APP_JSON)


class FormatDefaultMediaType(Format):
    description = (
        "Format for process input are assumed plain/text if the media-type was omitted and is not one of the known "
        "formats by this instance. When executing a job, the best match against supported formats by the process "
        "definition will be used to run the process, and will fall back to the default as last resort."
    )
    # NOTE:
    # The default is overridden from Format since the FormatSelection 'oneOf' always fails,
    # due to the 'default' value which is always generated, and it causes the presence of both Format and FormatMimeType
    mediaType = MediaType(example=ContentType.APP_JSON)


class FormatSelection(OneOfKeywordSchema):
    """
    Validation against ``mimeType`` or ``mediaType`` format.

    .. seealso::
        - :class:`FormatDefaultMediaType`
        - :class:`FormatDefaultMimeType`

    .. note::
        Format are validated to be retro-compatible with pre-existing/deployed/remote processes.
    """
    _one_of = [
        FormatDefaultMediaType(),
        FormatDefaultMimeType()
    ]


# only extra portion from:
# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1538-L1547
class FormatDescription(ExtendedMappingSchema):
    maximumMegabytes = ExtendedSchemaNode(Integer(), missing=drop, validator=Range(min=1))


# although original schema defines 'default' in above 'FormatDescription', separate it in order to omit it
# from 'ResultFormat' employed for result reporting, which shouldn't have a default (applied vs supported format)
class FormatDefault(ExtendedMappingSchema):
    default = ExtendedSchemaNode(
        Boolean(), missing=drop,
        # don't insert "default" field if omitted in deploy body to avoid causing differing "inputs"/"outputs"
        # definitions between the submitted payload and the validated one (in 'weaver.processes.utils._check_deploy')
        # default=False,
        description=(
            "Indicates if this format should be considered as the default one in case none of the other "
            "allowed or supported formats was matched nor provided as input during job submission."
        )
    )


class DescriptionFormat(Format, FormatDescription, FormatDefault):
    pass


class DeploymentFormat(FormatSelection, FormatDescription, FormatDefault):
    # NOTE:
    #   The 'OGC-API' suggest to use 'mediaType' field for format representation, but retro-compatibility is
    #   supported during deployment only, where either old 'mimeType' or new 'mediaType', but only 'mediaType'
    #   is used for process description and result reporting. This support is added for deployment so that
    #   pre-existing deploy definitions remain valid without need to update them.
    pass


class ResultFormat(FormatDescription):
    """
    Format employed for reference results respecting 'OGC API - Processes' schemas.
    """
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/format.yaml"
    _ext_schema_fields = []  # exclude "$schema" added on each sub-deserialize (too verbose, only for reference)

    mediaType = MediaType(String(), missing=drop)
    encoding = ExtendedSchemaNode(String(), missing=drop)
    schema = FormatSchema(missing=drop)


class DescriptionFormatList(ExtendedSequenceSchema):
    format_item = DescriptionFormat()


class DeploymentFormatList(ExtendedSequenceSchema):
    format_item = DeploymentFormat()


class AdditionalParameterUnique(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(String(), title="InputParameterLiteral.String"),
        ExtendedSchemaNode(Boolean(), title="InputParameterLiteral.Boolean"),
        ExtendedSchemaNode(Integer(), title="InputParameterLiteral.Integer"),
        ExtendedSchemaNode(Float(), title="InputParameterLiteral.Float"),
        # PermissiveMappingSchema(title="InputParameterLiteral.object"),
    ]


class AdditionalParameterListing(ExtendedSequenceSchema):
    param = AdditionalParameterUnique()


class AdditionalParameterValues(OneOfKeywordSchema):
    _one_of = [
        AdditionalParameterUnique(),
        AdditionalParameterListing()
    ]


class AdditionalParameterDefinition(ExtendedMappingSchema):
    name = SLUG(title="AdditionalParameterName", example="EOImage")
    values = AdditionalParameterValues(example=["true"])


class AdditionalParameterList(ExtendedSequenceSchema):
    param = AdditionalParameterDefinition()


class AdditionalParametersMeta(OneOfKeywordSchema):
    _one_of = [
        LinkBase(title="AdditionalParameterLink"),
        MetadataRole(title="AdditionalParameterRole")
    ]


class AdditionalParameters(ExtendedMappingSchema):
    parameters = AdditionalParameterList()


class AdditionalParametersItem(AnyOfKeywordSchema):
    _any_of = [
        AdditionalParametersMeta(),
        AdditionalParameters()
    ]


class AdditionalParametersList(ExtendedSequenceSchema):
    additionalParameter = AdditionalParametersItem()


class Content(ExtendedMappingSchema):
    href = ReferenceURL(description="URL to CWL file.", title="OWSContentURL",
                        default=drop,       # if invalid, drop it completely,
                        missing=required,   # but still mark as 'required' for parent objects
                        example="http://some.host/applications/cwl/multisensor_ndvi.cwl")


class Offering(ExtendedMappingSchema):
    code = ExtendedSchemaNode(String(), missing=drop, description="Descriptor of represented information in 'content'.")
    content = Content()


class OWSContext(ExtendedMappingSchema):
    description = "OGC Web Service definition from an URL reference."
    title = "owsContext"
    offering = Offering()


class DescriptionBase(ExtendedMappingSchema):
    title = ExtendedSchemaNode(String(), missing=drop, description="Short human-readable name of the object.")
    description = ExtendedSchemaNode(String(), missing=drop, description="Detailed explanation of the object.")


class DescriptionLinks(ExtendedMappingSchema):
    links = LinkList(missing=drop, description="References to endpoints with information related to object.")


class ProcessContext(ExtendedMappingSchema):
    owsContext = OWSContext(missing=drop)


class DescriptionExtra(ExtendedMappingSchema):
    additionalParameters = AdditionalParametersList(missing=drop)


class DescriptionType(DescriptionBase, DescriptionLinks, DescriptionExtra):
    pass


class DeploymentType(DescriptionType):
    deprecated = True
    abstract = ExtendedSchemaNode(
        String(), missing=drop, deprecated=True,
        description="Description of the object. Will be replaced by 'description' field if not already provided. "
                    "Preserved for backward compatibility of pre-existing process deployment. "
                    "Consider using 'description' directly instead."
    )


class DescriptionMeta(ExtendedMappingSchema):
    # employ empty lists by default if nothing is provided for process description
    keywords = KeywordList(
        default=[],
        description="Keywords applied to the process for search and categorization purposes.")
    metadata = MetadataList(
        default=[],
        description="External references to documentation or metadata sources relevant to the process.")


class ProcessDeployMeta(ExtendedMappingSchema):
    # don't require fields at all for process deployment, default to empty if omitted
    keywords = KeywordList(
        missing=drop, default=[],
        description="Keywords applied to the process for search and categorization purposes.")
    metadata = MetadataList(
        missing=drop, default=[],
        description="External references to documentation or metadata sources relevant to the process.")


class InputOutputDescriptionMeta(ExtendedMappingSchema):
    # remove unnecessary empty lists by default if nothing is provided for inputs/outputs
    def __init__(self, *args, **kwargs):
        # type: (*Any, **Any) -> None
        super(InputOutputDescriptionMeta, self).__init__(*args, **kwargs)
        for child in self.children:
            if child.name in ["keywords", "metadata"]:
                child.missing = drop


class ReferenceOAS(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/reference.yaml"
    _ref = ReferenceURL(name="$ref", description="External OpenAPI schema reference.")


class TypeOAS(ExtendedSchemaNode):
    name = "type"
    schema_type = String
    validator = OneOf(OAS_DATA_TYPES)


class EnumItemOAS(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(Float()),
        ExtendedSchemaNode(Integer()),
        ExtendedSchemaNode(String()),
    ]


class EnumOAS(ExtendedSequenceSchema):
    enum = EnumItemOAS()


class RequiredOAS(ExtendedSequenceSchema):
    required_field = ExtendedSchemaNode(String(), description="Name of the field that is required under the object.")


class MultipleOfOAS(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(Float()),
        ExtendedSchemaNode(Integer()),
    ]


class PermissiveDefinitionOAS(NotKeywordSchema, PermissiveMappingSchema):
    _not = [
        ReferenceOAS
    ]


# cannot make recursive declarative schemas
# simulate it and assume it is sufficient for validation purposes
class PseudoObjectOAS(OneOfKeywordSchema):
    _one_of = [
        ReferenceOAS(),
        PermissiveDefinitionOAS(),
    ]


class KeywordObjectOAS(ExtendedSequenceSchema):
    item = PseudoObjectOAS()


class AdditionalPropertiesOAS(OneOfKeywordSchema):
    _one_of = [
        ReferenceOAS(),
        PermissiveDefinitionOAS(),
        ExtendedSchemaNode(Boolean())
    ]


class AnyValueOAS(AnyOfKeywordSchema):
    _any_of = [
        PermissiveMappingSchema(),
        PermissiveSequenceSchema(),
        ExtendedSchemaNode(Float()),
        ExtendedSchemaNode(Integer()),
        ExtendedSchemaNode(Boolean()),
        ExtendedSchemaNode(String()),
    ]


# reference:
#   https://raw.githubusercontent.com/opengeospatial/ogcapi-processes/master/core/openapi/schemas/schema.yaml
# note:
#   although reference definition provides multiple 'default: 0|false' entries, we omit them since the behaviour
#   of colander with extended schema nodes is to set this value by default in deserialize result if they were missing,
#   but reference 'default' correspond more to the default *interpretation* value if none was provided.
#   It is preferable in our case to omit (i.e.: drop) these defaults to keep obtained/resolved definitions succinct,
#   since those defaults can be defined (by default...) if needed. No reason to add them explicitly.
# WARNING:
#   cannot use any KeywordMapper derived instance here, otherwise conflicts with same OpenAPI keywords as children nodes
class PropertyOAS(PermissiveMappingSchema):
    _type = TypeOAS(name="type", missing=drop)  # not present if top-most schema is {allOf,anyOf,oneOf,not}
    _format = ExtendedSchemaNode(String(), name="format", missing=drop)
    default = AnyValueOAS(unknown="preserve", missing=drop)
    example = AnyValueOAS(unknown="preserve", missing=drop)
    title = ExtendedSchemaNode(String(), missing=drop)
    description = ExtendedSchemaNode(String(), missing=drop)
    enum = EnumOAS(missing=drop)
    items = PseudoObjectOAS(name="items", missing=drop)
    required = RequiredOAS(missing=drop)
    nullable = ExtendedSchemaNode(Boolean(), missing=drop)
    deprecated = ExtendedSchemaNode(Boolean(), missing=drop)
    read_only = ExtendedSchemaNode(Boolean(), name="readOnly", missing=drop)
    write_only = ExtendedSchemaNode(Boolean(), name="writeOnly", missing=drop)
    multiple_of = MultipleOfOAS(name="multipleOf", missing=drop, validator=BoundedRange(min=0, exclusive_min=True))
    minimum = ExtendedSchemaNode(Integer(), name="minimum", missing=drop, validator=Range(min=0))  # default=0
    maximum = ExtendedSchemaNode(Integer(), name="maximum", missing=drop, validator=Range(min=0))
    exclusive_min = ExtendedSchemaNode(Boolean(), name="exclusiveMinimum", missing=drop)  # default=False
    exclusive_max = ExtendedSchemaNode(Boolean(), name="exclusiveMaximum", missing=drop)  # default=False
    min_length = ExtendedSchemaNode(Integer(), name="minLength", missing=drop, validator=Range(min=0))  # default=0
    max_length = ExtendedSchemaNode(Integer(), name="maxLength", missing=drop, validator=Range(min=0))
    pattern = ExtendedSchemaNode(Integer(), missing=drop)
    min_items = ExtendedSchemaNode(Integer(), name="minItems", missing=drop, validator=Range(min=0))  # default=0
    max_items = ExtendedSchemaNode(Integer(), name="maxItems", missing=drop, validator=Range(min=0))
    unique_items = ExtendedSchemaNode(Boolean(), name="uniqueItems", missing=drop)  # default=False
    min_prop = ExtendedSchemaNode(Integer(), name="minProperties", missing=drop, validator=Range(min=0))  # default=0
    max_prop = ExtendedSchemaNode(Integer(), name="maxProperties", missing=drop, validator=Range(min=0))
    content_type = ExtendedSchemaNode(String(), name="contentMediaType", missing=drop)
    content_encode = ExtendedSchemaNode(String(), name="contentEncoding", missing=drop)
    content_schema = ExtendedSchemaNode(String(), name="contentSchema", missing=drop)
    _not_key = PseudoObjectOAS(name="not", title="not", missing=drop)
    _all_of = KeywordObjectOAS(name="allOf", missing=drop)
    _any_of = KeywordObjectOAS(name="anyOf", missing=drop)
    _one_of = KeywordObjectOAS(name="oneOf", missing=drop)
    x_props = AdditionalPropertiesOAS(name="additionalProperties", missing=drop)
    properties = PermissiveMappingSchema(missing=drop)  # cannot do real recursive definitions, simply check mapping


# this class is only to avoid conflicting names with keyword mappers
class AnyPropertyOAS(OneOfKeywordSchema):
    _one_of = [
        ReferenceOAS(),
        PropertyOAS(),
    ]


class ObjectPropertiesOAS(ExtendedMappingSchema):
    property_name = AnyPropertyOAS(
        variable="{property-name}",
        description="Named of the property being defined under the OpenAPI object.",
    )


# would not need this if we could do explicit recursive definitions but at the very least, validate that when an
# object type is specified, its properties are as well and are slightly more specific than permissive mapping
class ObjectOAS(NotKeywordSchema, ExtendedMappingSchema):
    _not = [ReferenceOAS]
    _type = TypeOAS(name="type", missing=drop, validator=OneOf(OAS_COMPLEX_TYPES))
    properties = ObjectPropertiesOAS()  # required and more specific contrary to 'properties' in 'PropertyOAS'


# since we redefine 'properties', do not cause validation error for 'oneOf'
class DefinitionOAS(AnyOfKeywordSchema):
    _any_of = [
        ObjectOAS(),
        PropertyOAS(),  # for top-level keyword schemas {allOf, anyOf, oneOf, not}
    ]


class OAS(OneOfKeywordSchema):
    description = "OpenAPI schema definition."
    # _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/schema.yaml"  # definition used by OAP, but JSON-schema is more accurate
    _schema = "http://json-schema.org/draft-07/schema#"
    _one_of = [
        ReferenceOAS(),
        DefinitionOAS(),
    ]


class InputOutputDescriptionSchema(ExtendedMappingSchema):
    # Validation is accomplished only for the first few levels of the OpenAPI definition.
    # This is sufficient to know if the I/O type is literal/bbox/complex. If 'schema' is explicitly provided, it
    # should minimally succeed those top-level validation for proper I/O interpretation. Pseudo-recursive schema
    # are defined for any more deeply nested definition to keep everything intact (eg: explicit object structure).
    schema = OAS(missing=drop)


class MinOccursDefinition(OneOfKeywordSchema):
    description = "Minimum amount of values required for this input."
    title = "MinOccurs"
    example = 1
    _one_of = [
        ExtendedSchemaNode(Integer(), validator=Range(min=0), title="MinOccurs.integer",
                           ddescription="Positive integer."),
        ExtendedSchemaNode(String(), validator=StringRange(min=0), pattern="^[0-9]+$", title="MinOccurs.string",
                           description="Numerical string representing a positive integer."),
    ]


class MaxOccursDefinition(OneOfKeywordSchema):
    description = "Maximum amount of values allowed for this input."
    title = "MaxOccurs"
    example = 1
    _one_of = [
        ExtendedSchemaNode(Integer(), validator=Range(min=0), title="MaxOccurs.integer",
                           description="Positive integer."),
        ExtendedSchemaNode(String(), validator=StringRange(min=0), pattern="^[0-9]+$", title="MaxOccurs.string",
                           description="Numerical string representing a positive integer."),
        ExtendedSchemaNode(String(), validator=OneOf(["unbounded"]), title="MaxOccurs.unbounded",
                           description="Special value indicating no limit to occurrences."),
    ]


class DescribeMinMaxOccurs(ExtendedMappingSchema):
    minOccurs = MinOccursDefinition()
    maxOccurs = MaxOccursDefinition()


class DeployMinMaxOccurs(ExtendedMappingSchema):
    # entirely omitted definitions are permitted to allow inference from fields in package (CWL) or using defaults
    # if explicitly provided though, schema format and values should be validated
    # - do not use 'missing=drop' to ensure we raise provided invalid value instead of ignoring it
    # - do not use any specific value (e.g.: 1) for 'default' such that we do not inject an erroneous value when it
    #   was originally omitted, since it could be resolved differently depending on matching CWL inputs definitions
    minOccurs = MinOccursDefinition(default=null, missing=null)
    maxOccurs = MaxOccursDefinition(default=null, missing=null)


# does not inherit from 'DescriptionLinks' because other 'ProcessDescription<>' schema depend on this without 'links'
class ProcessDescriptionType(DescriptionBase, DescriptionExtra):
    id = ProcessIdentifierTag()
    version = Version(missing=None, default=None, example="1.2.3")
    mutable = ExtendedSchemaNode(Boolean(), default=True, description=(
        "Indicates if the process is mutable (dynamically deployed), or immutable (builtin with this instance)."
    ))


class InputIdentifierType(ExtendedMappingSchema):
    id = AnyIdentifier(description=IO_INFO_IDS.format(first="WPS", second="CWL", what="input"))


class OutputIdentifierType(ExtendedMappingSchema):
    id = AnyIdentifier(description=IO_INFO_IDS.format(first="WPS", second="CWL", what="output"))


class DescribeWithFormats(ExtendedMappingSchema):
    formats = DescriptionFormatList()


class DeployWithFormats(ExtendedMappingSchema):
    formats = DeploymentFormatList()


class DescribeComplexInputType(DescribeWithFormats):
    pass


class DeployComplexInputType(DeployWithFormats):
    pass


class AnyCRS(AnyOfKeywordSchema):
    # note:
    #   other CRS exist (EGM, NAVD, NAD, etc.)
    #   however, only support EPSG (short form, normative from, or URI) that are supported by 'owslib.crs'
    #   handle equivalent representations of EPSG:4326 that are also supported by 'owslib.crs'
    _any_of = [
        ExtendedSchemaNode(String(), pattern=re.compile(r"^urn:ogc:def:crs:EPSG::?[0-9]{4,5}$")),
        ExtendedSchemaNode(String(), pattern=re.compile(r"^\[?EPSG::?[0-9]{4,5}\]?$")),
        ExtendedSchemaNode(String(), pattern=re.compile(r"^https?://www\.opengis\.net/def/crs/EPSG/0/[0-9]{4,5}$")),
        ExtendedSchemaNode(String(), validator=OneOf([
            # equivalent forms of EPSG:4326, 2D or 3D
            "https://www.opengis.net/def/crs/OGC/1.3/CRS84",
            "http://www.opengis.net/def/crs/OGC/1.3/CRS84",
            "https://www.opengis.net/def/crs/OGC/0/CRS84h",
            "http://www.opengis.net/def/crs/OGC/0/CRS84h",
            "https://www.opengis.net/def/crs/OGC/0/CRS84",
            "http://www.opengis.net/def/crs/OGC/0/CRS84",
            "urn:ogc:def:crs:OGC:2:84",
            "WGS84",
        ])),
    ]
    default = OGC_API_BBOX_EPSG


class AnyFilterExpression(AnyOfKeywordSchema):
    _any_of = [
        PermissiveMappingSchema(),
        PermissiveSequenceSchema(validator=Length(min=1)),
        ExtendedSchemaNode(String(), validator=Length(min=1)),
    ]


class AnyFilterLanguage(ExtendedSchemaNode):
    schema_type = String
    name = "filter-lang"
    default = "cql2-json"
    validator = OneOfCaseInsensitive([
        "cql2-json",
        "cql2-text",
        "cql",
        "cql-text",
        "cql-json",
        "ecql",
        "simple-cql",
        "fes",
        "jfe"
    ])
    summary = "Filter expression language to use for parsing."
    description = (
        "Filter expression language to use for parsing. "
        "Supports multiple variants of OGC Common Query Language (CQL), "
        "Filter Expression Standard (FES), or JSON Filter Expression (JFE). "
        "Values are case-insensitive."
    )

    def deserialize(self, cstruct):
        # type: (Any) -> Union[str, colander.null]
        if isinstance(cstruct, str):
            cstruct = cstruct.lower()
        return super().deserialize(cstruct)


class FilterSchema(ExtendedMappingSchema):
    # note:
    #   defer format and parsing to 'pygeofilter'
    #   to accommodate all filter expression representations, string, array and object must be allowed
    filter = AnyFilterExpression(
        description="Filter expression according to the specified parsing language.",
        missing=drop,  # optional since combined within other JSON schema definitions
    )
    filter_crs = AnyCRS(
        name="filter-crs",
        missing=drop,
        default=drop,  # override to avoid injecting it by default, remote server could use a different default
        description="Coordinate Reference System for provided spatial properties.",
    )
    filter_lang = AnyFilterLanguage(
        name="filter-lang",
        missing=drop,
        default=drop,  # override to avoid injecting it by default, remote server could use a different default
        description=AnyFilterLanguage.description + (
            " If unspecified, the filter language will default to CQL2-Text if a string was provided,"
            " or CQL2-JSON if a JSON object or array structure is detected as the filter."
            " For any other language, or to resolve ambiguous cases such as a CQL2-JSON encoded as literal string,"
            " the filter language must be specified explicitly."
        )
    )

    @staticmethod
    @json_hashable
    @cache
    def parse(filter_expr, filter_lang):
        # type: (Union[JSON, str], str) -> FilterAstType
        parsed_expr = None
        if filter_lang == "cql2-json":
            parsed_expr = cql2_json.parse(filter_expr)
        elif filter_lang == "cql2-text":
            parsed_expr = cql2_text.parse(filter_expr)
        elif filter_lang == "cql-json":
            parsed_expr = cql_json.parse(filter_expr)
        elif filter_lang in ["cql", "cql-text", "ecql", "simple-cql"]:
            parsed_expr = ecql.parse(filter_expr)
        elif filter_lang == "fes":
            parsed_expr = fes_parse(filter_expr)  # FIXME: https://github.com/geopython/pygeofilter/pull/102
        elif filter_lang == "jfe":
            parsed_expr = jfe.parse(filter_expr)
        if not parsed_expr:
            raise colander.Invalid(
                node=AnyFilterLanguage(),
                msg="Unresolved filter expression language.",
                value={"filter-lang": filter_lang},
            )
        return parsed_expr

    def validate(self, filter_expr, filter_lang):
        # type: (Union[JSON, str], str) -> FilterAstType
        try:
            return self.parse(filter_expr, filter_lang)
        except (TypeError, ValueError) as exc:
            raise colander.Invalid(
                node=self,
                msg="Invalid filter expression could not be parsed against specified language.",
                value={"filter": filter_expr, "filter-lang": filter_lang},
            ) from exc

    def convert(self, filter_expr, filter_lang):
        # type: (Union[JSON, str], str) -> JSON
        try:
            parsed_expr = self.validate(filter_expr, filter_lang)
            return to_cql2(parsed_expr)
        except (TypeError, ValueError, NotImplementedError) as exc:
            raise colander.Invalid(
                node=self,
                msg="Invalid filter expression could not be interpreted.",
                value={"filter": filter_expr, "filter-lang": filter_lang},
            ) from exc

    def deserialize(self, cstruct):
        # type: (JSON) -> Union[JSON, colander.null]
        result = super().deserialize(cstruct)
        if not result:
            return result
        filter_expr = result.get("filter")
        filter_lang = result.get("filter-lang")
        if filter_expr in [null, drop, None]:  # explicit "", {}, [] should be raised as invalid since dropped
            if "filter" in cstruct:
                raise colander.Invalid(
                    node=self,
                    msg="Invalid filter expression could not be interpreted.",
                    value={"filter": repr_json(cstruct["filter"]), "filter-lang": filter_lang},
                )
            filter_crs = cstruct.get("filter-crs")
            filter_lang = cstruct.get("filter-lang")
            if filter_crs or filter_lang:
                raise colander.Invalid(
                    node=self,
                    msg="Missing filter expression provided with CRS and/or language parameters.",
                    value={"filter-crs": filter_crs, "filter-lang": filter_lang},
                )
            return result
        if not filter_lang:
            filter_lang = "cql2-text" if isinstance(filter, str) else "cql2-json"
        # perform conversion to validate
        # but don't return the converted CQL2-JSON to preserve the original definition where called (storage/dispatch)
        # conversion can be done as needed to obtain a uniform representation locally
        self.convert(filter_expr, filter_lang)
        return result


class SortByExpression(ExpandStringList, ExtendedSchemaNode):
    schema_type = String
    default = None
    example = "arg1,prop2,+asc,-desc"
    missing = drop
    description = (
        "Comma-separated list of sorting fields. "
        "Each field can be prefixed by +/- for ascending or descending sort order."
    )


class SortBySchema(ExtendedMappingSchema):
    sort_by_lower = SortByExpression(name="sortby", missing=drop)
    sort_by_upper = SortByExpression(name="sortBy", missing=drop)

    def deserialize(self, cstruct):
        # type: (JSON) -> Union[JSON, colander.null]
        """
        Resolve the upper/lower variant representation.

        Consider that this schema could be integrated with another.
        Therefore, additional fields must be left untouched.
        """
        result = super().deserialize(cstruct)
        if not result:
            return result
        if result.get("sortby"):
            # keep only 'official' "sortBy" from OGC API Processes
            # others OGC APIs use "sortby", but their query parameters are usually case-insensitive
            if not result.get("sortBy"):
                result["sortBy"] = result["sortby"]
            del result["sortby"]
        return result


class SupportedCRS(ExtendedMappingSchema):
    crs = AnyCRS(title="CRS", description="Coordinate Reference System")
    default = ExtendedSchemaNode(Boolean(), missing=drop)


class SupportedCRSList(ExtendedSequenceSchema):
    crs = SupportedCRS(title="SupportedCRS")


class BoundingBoxInputType(ExtendedMappingSchema):
    supportedCRS = SupportedCRSList()


class AnyLiteralType(OneOfKeywordSchema):
    """
    Submitted values that correspond to literal data.

    .. seealso::
        - :class:`AnyLiteralDataType`
        - :class:`AnyLiteralValueType`
        - :class:`AnyLiteralDefaultType`
    """
    _one_of = [
        ExtendedSchemaNode(
            Float(),
            title="LiteralDataFloat",
            description="Literal data type representing a floating point number.",
        ),
        ExtendedSchemaNode(
            Integer(),
            title="LiteralDataInteger",
            description="Literal data type representing an integer number.",
        ),
        ExtendedSchemaNode(
            Boolean(),
            title="LiteralDataBoolean",
            description="Literal data type representing a boolean flag.",
        ),
        ExtendedSchemaNode(
            # pylint: disable=C0301,line-too-long
            # FIXME: support byte/binary type (string + format:byte) ?
            #   https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/binaryInputValue.yaml
            #   see if we can use 'encoding' parameter available for below 'String' schema-type to handle this?
            String(allow_empty=True),  # valid to submit a process with empty string
            title="LiteralDataString",
            description="Literal data type representing a generic string.",
        ),
        ExtendedSchemaNode(
            NoneType(),
            title="LiteralDataNoneType",
            description="Literal data type representing a null value.",
        )
    ]


class Number(OneOfKeywordSchema):
    """
    Represents a literal number, integer or float.
    """
    _one_of = [
        ExtendedSchemaNode(Float()),
        ExtendedSchemaNode(Integer()),
    ]


class NumericType(OneOfKeywordSchema):
    """
    Represents a numeric-like value.
    """
    _one_of = [
        ExtendedSchemaNode(Float()),
        ExtendedSchemaNode(Integer()),
        ExtendedSchemaNode(String(), pattern="^[0-9]+$"),
    ]


class DecimalType(ExtendedSchemaNode):
    schema_type = colander.Decimal
    format = "decimal"


class PositiveNumber(AnyOfKeywordSchema):
    """
    Represents a literal number, integer or float, of positive value.
    """
    _any_of = [
        DecimalType(validator=Range(min=0.0)),
        ExtendedSchemaNode(Float(), validator=Range(min=0.0)),
        ExtendedSchemaNode(Integer(), validator=Range(min=0)),
    ]


class LiteralReference(ExtendedMappingSchema):
    reference = ExecuteReferenceURL()


# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1707-L1716
class NameReferenceType(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/nameReferenceType.yaml"
    _schema_include_deserialize = False
    name = ExtendedSchemaNode(String(), description="Name of the entity definition.")
    reference = ReferenceURL(missing=drop, description="Reference URL to schema definition of the named entity.")


class DataTypeSchema(NameReferenceType):
    description = "Type of the literal data representation."
    title = "DataType"
    # any named type that can be converted by: 'weaver.processes.convert.any2wps_literal_datatype'
    name = ExtendedSchemaNode(String(), validator=OneOf(list(WPS_LITERAL_DATA_TYPES)))


class UomSchema(NameReferenceType):
    title = "UnitOfMeasure"
    name = ExtendedSchemaNode(
        String(),
        description="Name of the entity definition.",
        missing=drop,  # override to make optional in contrat to 'NameReferenceType'
    )
    uom = ExtendedSchemaNode(
        String(allow_empty=True),  # unit/dimension-less value
        description="Unit applicable for the corresponding measurement representation.",
    )


class SupportedUoM(ExtendedSequenceSchema):
    description = "List of supported units for the represented measurement."
    uom_item = UomSchema()
    validator = Length(min=1)


class MeasurementDataDomain(ExtendedMappingSchema):
    supported = SupportedUoM()
    default = UomSchema(missing=drop)


# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1423
# NOTE: Original is only 'string', but we allow any literal type
class AllowedValuesList(ExtendedSequenceSchema):
    value = AnyLiteralType()


# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1772-L1787
# NOTE:
#   Contrary to original schema where all fields are 'string', we allow any literal type as well since those make more
#   sense when parsing corresponding data values (eg: float, integer, bool).
class AllowedRange(ExtendedMappingSchema):
    minimumValue = NumericType(missing=drop)
    maximumValue = NumericType(missing=drop)
    spacing = NumericType(missing=drop)
    rangeClosure = ExtendedSchemaNode(String(), missing=drop,
                                      validator=OneOf(["closed", "open", "open-closed", "closed-open"]))


class AllowedRangesList(ExtendedSequenceSchema):
    range = AllowedRange()


class AllowedValues(OneOfKeywordSchema):
    _one_of = [
        AllowedRangesList(description="List of value ranges and constraints."),  # array of {range}
        AllowedValuesList(description="List of enumerated allowed values."),     # array of "value"
        ExtendedSchemaNode(String(), description="Single allowed value."),       # single "value"
    ]


# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1425-L1430
class AnyValue(ExtendedMappingSchema):
    anyValue = ExtendedSchemaNode(
        Boolean(), missing=drop, default=True,
        description="Explicitly indicate if any value is allowed. "
                    "This is the default behaviour if no other constrains are specified."
    )


# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1801-L1803
class ValuesReference(ExecuteReferenceURL):
    description = "URL where to retrieve applicable values."


class ArrayLiteralType(ExtendedSequenceSchema):
    value_item = AnyLiteralType()


class ArrayLiteralDataType(ExtendedMappingSchema):
    data = ArrayLiteralType()


class ArrayLiteralValueType(ExtendedMappingSchema):
    value = ArrayLiteralType()


class AnyLiteralDataType(ExtendedMappingSchema):
    data = AnyLiteralType()


class AnyLiteralValueType(ExtendedMappingSchema):
    value = AnyLiteralType()


class AnyLiteralDefaultType(ExtendedMappingSchema):
    default = AnyLiteralType()


class LiteralDataValueDefinition(OneOfKeywordSchema):
    _one_of = [
        AllowedValues(description="Constraints of allowed values."),
        ValuesReference(description="Reference URL where to retrieve allowed values."),
        # 'AnyValue' must be last because it's the most permissive (always valid, default)
        AnyValue(description="Permissive definition for any allowed value."),
    ]


# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1675-L1688
#  literalDataDomain:
#    valueDefinition: oneOf(<allowedValues, anyValue, valuesReference>)
#    defaultValue: <string>
#    dataType: <nameReferenceType>
#    uom: <nameReferenceType>
class LiteralDataDomain(ExtendedMappingSchema):
    default = ExtendedSchemaNode(Boolean(), default=True,
                                 description="Indicates if this literal data domain definition is the default one.")
    defaultValue = AnyLiteralType(missing=drop, description="Default value to employ if none was provided.")
    dataType = DataTypeSchema(missing=drop, description="Type name and reference of the literal data representation.")
    valueDefinition = LiteralDataValueDefinition(description="Literal data domain constraints.")
    uoms = MeasurementDataDomain(name="UOMs", missing=drop, description="Unit of measure applicable for the data.")


class LiteralDataDomainList(ExtendedSequenceSchema):
    """
    Constraints that apply to the literal data values.
    """
    literalDataDomain = LiteralDataDomain()


# https://github.com/opengeospatial/ogcapi-processes/blob/e6893b/extensions/workflows/openapi/workflows.yaml#L1689-L1697
class LiteralDataType(NotKeywordSchema, ExtendedMappingSchema):
    # NOTE:
    #   Apply 'missing=drop' although original schema of 'literalDataDomains' (see link above) requires it because
    #   we support omitting it for minimalistic literal input definition.
    #   This is because our schema validation allows us to do detection of 'basic' types using the literal parsing.
    #   Because there is not explicit requirement though (ie: missing would fail schema validation), we must check
    #   that 'format' is not present to avoid conflict with minimalistic literal data definition in case of ambiguity.
    literalDataDomains = LiteralDataDomainList(missing=drop)
    _not = [
        DescribeWithFormats,
    ]


class LiteralInputType(LiteralDataType):
    pass


class DescribeInputTypeDefinition(OneOfKeywordSchema):
    _one_of = [
        # NOTE:
        #   LiteralInputType could be used to represent a complex input if the 'format' is missing in
        #   process description definition but is instead provided in CWL definition.
        #   This use case is still valid because 'format' can be inferred from the combining Process/CWL contents.
        BoundingBoxInputType,
        DescribeComplexInputType,  # should be 2nd to last because very permissive, but requires format at least
        LiteralInputType,  # must be last because it's the most permissive (all can default if omitted)
    ]


class DeployInputTypeDefinition(OneOfKeywordSchema):
    _one_of = [
        # NOTE:
        #   LiteralInputType could be used to represent a complex input if the 'format' is missing in
        #   process deployment definition but is instead provided in CWL definition.
        #   This use case is still valid because 'format' can be inferred from the combining Process/CWL contents.
        BoundingBoxInputType,
        DeployComplexInputType,  # should be 2nd to last because very permissive, but requires formats at least
        LiteralInputType,  # must be last because it's the most permissive (all can default if omitted)
    ]


class DescribeInputType(AllOfKeywordSchema):
    _all_of = [
        DescriptionType(),
        InputOutputDescriptionMeta(),
        InputOutputDescriptionSchema(),
        DescribeInputTypeDefinition(),
        DescribeMinMaxOccurs(),
        DescriptionExtra(),
    ]

    _sort_first = PROCESS_IO_FIELD_FIRST
    _sort_after = PROCESS_IO_FIELD_AFTER


class DescribeInputTypeWithID(InputIdentifierType, DescribeInputType):
    title = "DescribeInputTypeWithID"


# Different definition than 'Describe' such that nested 'complex' type 'formats' can be validated and backward
# compatible with pre-existing/deployed/remote processes, with either ``mediaType`` and ``mimeType`` formats.
class DeployInputType(AllOfKeywordSchema):
    _all_of = [
        DeploymentType(),
        InputOutputDescriptionMeta(),
        InputOutputDescriptionSchema(),
        DeployInputTypeDefinition(),
        DeployMinMaxOccurs(),
        DescriptionExtra(),
    ]

    _sort_first = PROCESS_IO_FIELD_FIRST
    _sort_after = PROCESS_IO_FIELD_AFTER


class DeployInputTypeWithID(InputIdentifierType, DeployInputType):
    pass


# for [{id: "", ...}] representation within ProcessDescription (OLD schema)
class DescribeInputTypeList(ExtendedSequenceSchema):
    """
    Listing of process inputs descriptions.
    """
    input = DescribeInputTypeWithID()


# for {"<id>": {...}} representation within ProcessDescription (OGC schema)
class DescribeInputTypeMap(PermissiveMappingSchema):
    """
    Description of all process inputs under mapping.
    """
    input_id = DescribeInputType(
        variable="{input-id}",
        description="Input definition under mapping of process description.",
        missing=drop,  # allowed because process can have empty inputs (see schema: ProcessDescriptionOGC)
    )


# for [{id: "", ...}] representation within ProcessDeployment (OLD schema)
class DeployInputTypeList(ExtendedSequenceSchema):
    """
    Listing of process input definitions to deploy.
    """
    input_item = DeployInputTypeWithID()


# for {"<id>": {...}} representation within ProcessDeployment (OGC schema)
class DeployInputTypeMap(PermissiveMappingSchema):
    """
    Definition of all process inputs under mapping.
    """
    input_id = DeployInputType(
        variable="{input-id}",
        description="Input definition under mapping of process deployment."
    )


class DeployInputTypeAny(OneOfKeywordSchema):
    _one_of = [
        DeployInputTypeList(),
        DeployInputTypeMap(),
    ]


class LiteralOutputType(LiteralDataType):
    pass


class BoundingBoxOutputType(ExtendedMappingSchema):
    supportedCRS = SupportedCRSList()


class DescribeComplexOutputType(DescribeWithFormats):
    pass


class DeployComplexOutputType(DeployWithFormats):
    pass


class DescribeOutputTypeDefinition(OneOfKeywordSchema):
    _one_of = [
        BoundingBoxOutputType,
        DescribeComplexOutputType,  # should be 2nd to last because very permissive, but requires formats at least
        LiteralOutputType,  # must be last because it's the most permissive (all can default if omitted)
    ]


class DeployOutputTypeDefinition(OneOfKeywordSchema):
    _one_of = [
        BoundingBoxOutputType,
        DeployComplexOutputType,  # should be 2nd to last because very permissive, but requires formats at least
        LiteralOutputType,  # must be last because it's the most permissive (all can default if omitted)
    ]


class DescribeOutputType(AllOfKeywordSchema):
    _all_of = [
        DescriptionType(),
        InputOutputDescriptionMeta(),
        InputOutputDescriptionSchema(),
        DescribeOutputTypeDefinition(),
    ]

    _sort_first = PROCESS_IO_FIELD_FIRST
    _sort_after = PROCESS_IO_FIELD_AFTER


class DescribeOutputTypeWithID(OutputIdentifierType, DescribeOutputType):
    pass


class DescribeOutputTypeList(ExtendedSequenceSchema):
    """
    Listing of process outputs descriptions.
    """
    output = DescribeOutputTypeWithID()


# for {"<id>": {...}} representation within ProcessDescription (OGC schema)
class DescribeOutputTypeMap(PermissiveMappingSchema):
    """
    Definition of all process outputs under mapping.
    """
    output_id = DescribeOutputType(
        variable="{output-id}", title="ProcessOutputDefinition",
        description="Output definition under mapping of process description."
    )


# Different definition than 'Describe' such that nested 'complex' type 'formats' can be validated and backward
# compatible with pre-existing/deployed/remote processes, with either ``mediaType`` and ``mimeType`` formats.
class DeployOutputType(AllOfKeywordSchema):
    _all_of = [
        DeploymentType(),
        InputOutputDescriptionMeta(),
        InputOutputDescriptionSchema(),
        DeployOutputTypeDefinition(),
    ]

    _sort_first = PROCESS_IO_FIELD_FIRST
    _sort_after = PROCESS_IO_FIELD_AFTER


class DeployOutputTypeWithID(OutputIdentifierType, DeployOutputType):
    pass


# for [{id: "", ...}] representation within ProcessDeployment (OLD schema)
class DeployOutputTypeList(ExtendedSequenceSchema):
    """
    Listing of process output definitions to deploy.
    """
    input = DeployOutputTypeWithID()


# for {"<id>": {...}} representation within ProcessDeployment (OGC schema)
class DeployOutputTypeMap(PermissiveMappingSchema):
    """
    Definition of all process outputs under mapping.
    """
    input_id = DeployOutputType(
        variable="{input-id}",
        description="Output definition under mapping of process deployment."
    )


class DeployOutputTypeAny(OneOfKeywordSchema):
    _one_of = [
        DeployOutputTypeList,
        DeployOutputTypeMap,
    ]


class JobExecuteModeEnum(ExtendedSchemaNode):
    # _schema: none available by itself, legacy parameter that was directly embedded in 'execute.yaml'
    # (https://github.com/opengeospatial/ogcapi-processes/blob/1.0-draft.5/core/openapi/schemas/execute.yaml)
    schema_type = String
    title = "JobExecuteMode"
    # no default to enforce required input as per OGC-API schemas
    # default = EXECUTE_MODE_AUTO
    example = ExecuteMode.ASYNC
    description = (
        "Desired execution mode specified directly. This is intended for backward compatibility support. "
        "To obtain more control over execution mode selection, employ the official Prefer header instead "
        f"(see for more details: {DOC_URL}/processes.html#execution-mode)."
    )
    validator = OneOf(ExecuteMode.values())


class JobControlOptionsEnum(ExtendedSchemaNode):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/jobControlOptions.yaml"
    schema_type = String
    title = "JobControlOptions"
    default = ExecuteControlOption.ASYNC
    example = ExecuteControlOption.ASYNC
    validator = OneOf(ExecuteControlOption.values())


class JobResponseOptionsEnum(ExtendedSchemaNode):
    # _schema: none available by itself, legacy parameter that was directly embedded in 'execute.yaml'
    # (https://github.com/opengeospatial/ogcapi-processes/blob/1.0-draft.6/core/openapi/schemas/execute.yaml)
    schema_type = String
    title = "JobResponseOptions"
    # no default to enforce required input as per OGC-API schemas
    # default = ExecuteResponse.DOCUMENT
    example = ExecuteResponse.DOCUMENT
    description = (
        "Indicates the desired representation format of the response. "
        f"(see for more details: {DOC_URL}/processes.html#execution-body)."
    )
    validator = OneOf(ExecuteResponse.values())


class TransmissionModeEnum(ExtendedSchemaNode):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/transmissionMode.yaml"
    schema_type = String
    title = "TransmissionMode"
    # no default to allow auto-resolution as data/link if omitted
    # default = ExecuteTransmissionMode.VALUE
    example = ExecuteTransmissionMode.VALUE
    validator = OneOf(ExecuteTransmissionMode.values())


class JobStatusEnum(ExtendedSchemaNode):
    _schema = f"{OGC_API_PROC_PART1_PARAMETERS}/status.yaml"  # subset of this implementation
    schema_type = String
    title = "JobStatus"
    default = Status.ACCEPTED
    example = Status.ACCEPTED
    validator = OneOf(JOB_STATUS_CODE_API)


class JobStatusCreate(ExtendedSchemaNode):
    schema_type = String
    title = "JobStatus"
    validator = OneOf(["create"])


class JobStatusSearchEnum(ExtendedSchemaNode):
    schema_type = String
    title = "JobStatusSearch"
    default = Status.ACCEPTED
    example = Status.ACCEPTED
    validator = StringOneOf(JOB_STATUS_SEARCH_API, delimiter=",", case_sensitive=False)


class JobTypeEnum(ExtendedSchemaNode):
    _schema = f"{OGC_API_PROC_PART1_PARAMETERS}/type.yaml"  # subset of this implementation
    schema_type = String
    title = "JobType"
    default = null
    example = "process"
    validator = OneOf(["process", "provider", "service"])


class JobTitle(ExtendedSchemaNode):
    schema_type = String
    description = "Title assigned to the job for user-readable identification."
    validator = Length(min=1)


class JobSortEnum(ExtendedSchemaNode):
    schema_type = String
    title = "JobSortingMethod"
    default = Sort.CREATED
    example = Sort.CREATED
    validator = OneOf(SortMethods.JOB)


class ProcessSortEnum(ExtendedSchemaNode):
    schema_type = String
    title = "ProcessSortMethod"
    default = Sort.ID
    example = Sort.CREATED
    validator = OneOf(SortMethods.PROCESS)


class QuoteSortEnum(ExtendedSchemaNode):
    schema_type = String
    title = "QuoteSortingMethod"
    default = Sort.ID
    example = Sort.PROCESS
    validator = OneOf(SortMethods.QUOTE)


class JobTagsCommaSeparated(ExpandStringList, ExtendedSchemaNode):
    schema_type = String
    validator = CommaSeparated()
    default = None
    missing = drop
    description = (
        "Comma-separated tags that can be used to filter jobs. "
        f"Only {validator.allow_chars} characters are permitted."
    )


class JobGroupsCommaSeparated(ExpandStringList, ExtendedSchemaNode):
    schema_type = String
    default = None
    example = "process,service"
    missing = drop
    description = "Comma-separated list of grouping fields with which to list jobs."
    validator = StringOneOf(["process", "provider", "service", "status"], delimiter=",", case_sensitive=True)


class JobExecuteSubscribers(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/subscriber.yaml"
    description = "Optional URIs for callbacks for this job."
    # basic OGC subscribers
    success_uri = URL(
        name="successUri",
        description="Location where to POST the job results on successful completion.",
        # allow omitting against official schema to support partial use/update
        # (see https://github.com/opengeospatial/ogcapi-processes/issues/460)
        missing=drop,
    )
    failed_uri = URL(
        name="failedUri",
        description="Location where to POST the job status if it fails execution.",
        missing=drop,
    )
    in_progress_uri = URL(
        name="inProgressUri",
        description="Location where to POST the job status once it starts execution.",
        missing=drop,
    )
    # additional subscribers
    success_email = Email(
        name="successEmail",
        description="Email recipient to send a notification on successful job completion.",
        missing=drop,
    )
    failed_email = Email(
        name="failedEmail",
        description="Email recipient to send a notification on failed job completion.",
        missing=drop,
    )
    in_progress_email = Email(
        name="inProgressEmail",
        description="Email recipient to send a notification of job status once it starts execution.",
        missing=drop,
    )


class LaunchJobQuerystring(ExtendedMappingSchema):
    tags = JobTagsCommaSeparated()


class VisibilityValue(ExtendedSchemaNode):
    schema_type = String
    validator = OneOf(Visibility.values())
    example = Visibility.PUBLIC


class JobAccess(VisibilityValue):
    pass


class VisibilitySchema(ExtendedMappingSchema):
    value = VisibilityValue()


class QuoteEstimatorConfigurationSchema(ExtendedMappingSchema):
    _schema = f"{WEAVER_SCHEMA_URL}/quotation/quote-estimator.yaml#/definitions/Configuration"
    description = "Quote Estimator Configuration"

    def deserialize(self, cstruct):
        schema = ExtendedMappingSchema(_schema=self._schema)  # avoid recursion
        return validate_node_schema(schema, cstruct)


class QuoteEstimatorWeightedParameterSchema(ExtendedMappingSchema):
    # NOTE:
    #   value/size parameters omitted since they will be provided at runtime by the
    #   quote estimation job obtained from submitted body in 'QuoteProcessParametersSchema'
    weight = ExtendedSchemaNode(
        Float(),
        default=1.0,
        missing=drop,
        description="Weight attributed to this parameter when submitted for quote estimation.",
    )


class QuoteEstimatorInputParametersSchema(ExtendedMappingSchema):
    description = "Parametrization of inputs for quote estimation."
    input_id = QuoteEstimatorWeightedParameterSchema(
        variable="{input-id}",
        title="QuoteEstimatorInputParameters",
        description="Mapping of input definitions for quote estimation.",
        missing=drop,  # because only weight expected, if missing/invalid, ignore mapping (1.0 applied by default later)
    )


class QuoteEstimatorOutputParametersSchema(ExtendedMappingSchema):
    description = "Parametrization of outputs for quote estimation."
    output_id = QuoteEstimatorWeightedParameterSchema(
        variable="{output-id}",
        title="QuoteEstimatorOutputParameters",
        description="Mapping of output definitions for quote estimation.",
        missing=drop,  # because only weight expected, if missing/invalid, ignore mapping (1.0 applied by default later)
    )


class QuoteEstimatorSchema(ExtendedMappingSchema):
    _schema = f"{WEAVER_SCHEMA_URL}/quotation/quote-estimator.yaml"
    description = "Configuration of the quote estimation algorithm for a given process."
    config = QuoteEstimatorConfigurationSchema()
    inputs = QuoteEstimatorInputParametersSchema(missing=drop, default={})
    outputs = QuoteEstimatorOutputParametersSchema(missing=drop, default={})


#########################################################
# Path parameter definitions
#########################################################


class LocalProcessQuery(ExtendedMappingSchema):
    version = Version(example="1.2.3", missing=drop, description=(
        "Specific process version to locate. "
        "If process ID was requested with tagged 'id:version' revision format, this parameter is ignored."
    ))


class LocalProcessPath(ExtendedMappingSchema):
    process_id = ProcessIdentifierTag(
        example="jsonarray2netcdf[:1.0.0]",
        summary="Process identifier with optional tag version.",
        description=(
            "Process identifier with optional tag version. "
            "If tag is omitted, the latest version of that process is assumed. "
            "Otherwise, the specific process revision as 'id:version' must be matched. "
            "Alternatively, the plain process ID can be specified in combination to 'version' query parameter."
        ),
    )


class ProviderPath(ExtendedMappingSchema):
    provider_id = AnyIdentifier(description="Remote provider identifier.", example="hummingbird")


class ProviderProcessPath(ProviderPath):
    # note: Tag representation not allowed in this case
    process_id = ProcessIdentifier(example="provider-process", description=(
        "Identifier of a process that is offered by the remote provider."
    ))


class JobPath(ExtendedMappingSchema):
    job_id = UUID(description="Job ID", example="14c68477-c3ed-4784-9c0f-a4c9e1344db5")


class BillPath(ExtendedMappingSchema):
    bill_id = UUID(description="Bill ID")


class QuotePath(ExtendedMappingSchema):
    quote_id = UUID(description="Quote ID")


class ResultPath(ExtendedMappingSchema):
    result_id = UUID(description="Result ID")


#########################################################
# These classes define each of the endpoints parameters
#########################################################


class FrontpageEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()


class VersionsEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()


class ConformanceQueries(ExtendedMappingSchema):
    category = ExtendedSchemaNode(
        String(),
        missing=drop,
        default=ConformanceCategory.CONFORMANCE,
        validator=OneOf(ConformanceCategory.values()),
        description="Select the desired conformance item references to be returned."
    )


class ConformanceEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()
    querystring = ConformanceQueries()


# FIXME: support YAML (https://github.com/crim-ca/weaver/issues/456)
class OpenAPIAcceptHeader(AcceptHeader):
    default = ContentType.APP_OAS_JSON
    validator = OneOf([ContentType.APP_OAS_JSON, ContentType.APP_JSON])


class OpenAPIRequestHeaders(RequestHeaders):
    accept = OpenAPIAcceptHeader()


class OpenAPIEndpoint(ExtendedMappingSchema):
    header = OpenAPIRequestHeaders()


class OpenAPIFormatRedirect(OpenAPIEndpoint):
    header = OpenAPIAcceptHeader()
    querystring = FormatQueryJSON()


class SwaggerUIEndpoint(ExtendedMappingSchema):
    pass


class RedocUIEndpoint(ExtendedMappingSchema):
    pass


class OWSNamespace(XMLObject):
    prefix = "ows"
    namespace = "http://www.opengis.net/ows/1.1"


class WPSNamespace(XMLObject):
    prefix = "wps"
    namespace = "http://www.opengis.net/wps/1.0.0"


class XMLNamespace(XMLObject):
    prefix = "xml"


class XMLReferenceAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    attribute = True
    name = "href"
    prefix = "xlink"
    format = "url"


class MimeTypeAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    attribute = True
    name = "mimeType"
    prefix = drop
    example = ContentType.APP_JSON


class EncodingAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    attribute = True
    name = "encoding"
    prefix = drop
    example = "UTF-8"


class OWSVersion(ExtendedSchemaNode, OWSNamespace):
    schema_type = String
    name = "Version"
    default = "1.0.0"
    example = "1.0.0"


class OWSAcceptVersions(ExtendedSequenceSchema, OWSNamespace):
    description = "Accepted versions to produce the response."
    name = "AcceptVersions"
    item = OWSVersion()


class OWSLanguage(ExtendedSchemaNode, OWSNamespace):
    description = "Desired language to produce the response."
    schema_type = String
    name = "Language"
    default = AcceptLanguage.EN_US
    example = AcceptLanguage.EN_CA


class OWSLanguageAttribute(OWSLanguage):
    description = "RFC-4646 language code of the human-readable text."
    name = "language"
    attribute = True


class OWSService(ExtendedSchemaNode, OWSNamespace):
    description = "Desired service to produce the response (SHOULD be 'WPS')."
    schema_type = String
    name = "service"
    attribute = True
    default = AcceptLanguage.EN_US
    example = AcceptLanguage.EN_CA


class WPSServiceAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    name = "service"
    attribute = True
    default = "WPS"
    example = "WPS"


class WPSVersionAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    name = "version"
    attribute = True
    default = "1.0.0"
    example = "1.0.0"


class WPSLanguageAttribute(ExtendedSchemaNode, XMLNamespace):
    schema_type = String
    name = "lang"
    attribute = True
    default = AcceptLanguage.EN_US
    example = AcceptLanguage.EN_CA


class WPSParameters(ExtendedMappingSchema):
    service = ExtendedSchemaNode(String(), example="WPS", description="Service selection.",
                                 validator=OneOfCaseInsensitive(["WPS"]))
    request = ExtendedSchemaNode(String(), example="GetCapabilities", description="WPS operation to accomplish",
                                 validator=OneOfCaseInsensitive(["GetCapabilities", "DescribeProcess", "Execute"]))
    version = Version(exaple="1.0.0", default="1.0.0", validator=OneOf(["1.0.0", "2.0.0", "2.0"]))
    identifier = ExtendedSchemaNode(String(), exaple="hello", missing=drop,
                                    example="example-process,another-process",
                                    description="Single or comma-separated list of process identifiers to describe, "
                                                "and single one for execution.")
    data_inputs = ExtendedSchemaNode(String(), name="DataInputs", missing=drop,
                                     example="message=hi&names=user1,user2&value=1",
                                     description="Process execution inputs provided as Key-Value Pairs (KVP).")


class WPSOperationGetNoContent(ExtendedMappingSchema):
    description = "No content body provided (GET requests)."
    default = {}


class WPSOperationPost(ExtendedMappingSchema):
    _schema = f"{OGC_WPS_1_SCHEMAS}/common/RequestBaseType.xsd"
    accepted_versions = OWSAcceptVersions(missing=drop, default="1.0.0")
    language = OWSLanguageAttribute(missing=drop)
    service = OWSService()


class WPSGetCapabilitiesPost(WPSOperationPost, WPSNamespace):
    _schema = f"{OGC_WPS_1_SCHEMAS}/wpsGetCapabilities_request.xsd"
    name = "GetCapabilities"
    title = "GetCapabilities"


class OWSIdentifier(ExtendedSchemaNode, OWSNamespace):
    schema_type = String
    name = "Identifier"


class OWSProcessIdentifier(ProcessIdentifier, OWSNamespace):
    pass


class OWSProcessIdentifierList(ExtendedSequenceSchema, OWSNamespace):
    name = "Identifiers"
    item = OWSProcessIdentifier()


class OWSTitle(ExtendedSchemaNode, OWSNamespace):
    schema_type = String
    name = "Title"


class OWSAbstract(ExtendedSchemaNode, OWSNamespace):
    schema_type = String
    name = "Abstract"


class OWSMetadataLink(ExtendedSchemaNode, XMLObject):
    schema_name = "Metadata"
    schema_type = String
    attribute = True
    name = "Metadata"
    prefix = "xlink"
    example = "WPS"
    wrapped = False  # metadata xlink at same level as other items


class OWSMetadata(ExtendedSequenceSchema, OWSNamespace):
    schema_type = String
    name = "Metadata"
    title = OWSMetadataLink(missing=drop)


class WPSDescribeProcessPost(WPSOperationPost, WPSNamespace):
    _schema = f"{OGC_WPS_1_SCHEMAS}/wpsDescribeProcess_request.xsd"
    name = "DescribeProcess"
    title = "DescribeProcess"
    identifier = OWSProcessIdentifierList(
        description="Single or comma-separated list of process identifier to describe.",
        example="example"
    )


class WPSExecuteDataInputs(ExtendedMappingSchema, WPSNamespace):
    description = "XML data inputs provided for WPS POST request (Execute)."
    name = "DataInputs"
    title = "DataInputs"
    # FIXME: missing details about 'DataInputs'


class WPSExecutePost(WPSOperationPost, WPSNamespace):
    _schema = f"{OGC_WPS_1_SCHEMAS}/wpsExecute_request.xsd"
    name = "Execute"
    title = "Execute"
    identifier = OWSProcessIdentifier(description="Identifier of the process to execute with data inputs.")
    dataInputs = WPSExecuteDataInputs(description="Data inputs to be provided for process execution.")


class WPSRequestBody(OneOfKeywordSchema):
    _one_of = [
        WPSExecutePost(),
        WPSDescribeProcessPost(),
        WPSGetCapabilitiesPost(),
    ]
    examples = {
        "Execute": {
            "summary": "Execute request example.",
            "value": EXAMPLES["wps_execute_request.xml"]
        }
    }


class WPSHeaders(ExtendedMappingSchema):
    accept = AcceptHeader(missing=drop)


class WPSEndpointGet(ExtendedMappingSchema):
    header = WPSHeaders()
    querystring = WPSParameters()
    body = WPSOperationGetNoContent(missing=drop)


class WPSEndpointPost(ExtendedMappingSchema):
    header = WPSHeaders()
    body = WPSRequestBody()


class XMLBooleanAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = Boolean
    attribute = True


class XMLString(ExtendedSchemaNode, XMLObject):
    schema_type = String


class OWSString(ExtendedSchemaNode, OWSNamespace):
    schema_type = String


class OWSKeywordList(ExtendedSequenceSchema, OWSNamespace):
    title = "OWSKeywords"
    keyword = OWSString(name="Keyword", title="OWSKeyword", example="Weaver")


class OWSType(ExtendedMappingSchema, OWSNamespace):
    schema_type = String
    name = "Type"
    example = "theme"
    additionalProperties = {
        "codeSpace": {
            "type": "string",
            "example": "ISOTC211/19115",
            "xml": {"attribute": True}
        }
    }


class OWSPhone(ExtendedMappingSchema, OWSNamespace):
    name = "Phone"
    voice = OWSString(name="Voice", title="OWSVoice", example="1-234-567-8910", missing=drop)
    facsimile = OWSString(name="Facsimile", title="OWSFacsimile", missing=drop)


class OWSAddress(ExtendedMappingSchema, OWSNamespace):
    name = "Address"
    delivery_point = OWSString(name="DeliveryPoint", title="OWSDeliveryPoint",
                               example="123 Place Street", missing=drop)
    city = OWSString(name="City", title="OWSCity", example="Nowhere", missing=drop)
    country = OWSString(name="Country", title="OWSCountry", missing=drop)
    admin_area = OWSString(name="AdministrativeArea", title="AdministrativeArea", missing=drop)
    postal_code = OWSString(name="PostalCode", title="OWSPostalCode", example="A1B 2C3", missing=drop)
    email = OWSString(name="ElectronicMailAddress", title="OWSElectronicMailAddress",
                      example="mail@me.com", validator=EmailRegex, missing=drop)


class OWSContactInfo(ExtendedMappingSchema, OWSNamespace):
    name = "ContactInfo"
    phone = OWSPhone(missing=drop)
    address = OWSAddress(missing=drop)


class OWSServiceContact(ExtendedMappingSchema, OWSNamespace):
    name = "ServiceContact"
    individual = OWSString(name="IndividualName", title="OWSIndividualName", example="John Smith", missing=drop)
    position = OWSString(name="PositionName", title="OWSPositionName", example="One-Man Team", missing=drop)
    contact = OWSContactInfo(missing=drop, default={})


class OWSServiceProvider(ExtendedMappingSchema, OWSNamespace):
    description = "Details about the institution providing the service."
    name = "ServiceProvider"
    title = "ServiceProvider"
    provider_name = OWSString(name="ProviderName", title="OWSProviderName", example="EXAMPLE")
    provider_site = OWSString(name="ProviderName", title="OWSProviderName", example="http://schema-example.com")
    contact = OWSServiceContact(required=False, defalult={})


class WPSDescriptionType(ExtendedMappingSchema, OWSNamespace):
    _schema = f"{OGC_WPS_1_SCHEMAS}/common/DescriptionType.xsd"
    name = "DescriptionType"
    _title = OWSTitle(description="Title of the service.", example="Weaver")
    abstract = OWSAbstract(description="Detail about the service.", example="Weaver WPS example schema.", missing=drop)
    metadata = OWSMetadata(description="Metadata of the service.", example="Weaver WPS example schema.", missing=drop)


class OWSServiceIdentification(WPSDescriptionType, OWSNamespace):
    name = "ServiceIdentification"
    title = "ServiceIdentification"
    keywords = OWSKeywordList(name="Keywords")
    type = OWSType()
    svc_type = OWSString(name="ServiceType", title="ServiceType", example="WPS")
    svc_type_ver1 = OWSString(name="ServiceTypeVersion", title="ServiceTypeVersion", example="1.0.0")
    svc_type_ver2 = OWSString(name="ServiceTypeVersion", title="ServiceTypeVersion", example="2.0.0")
    fees = OWSString(name="Fees", title="Fees", example="NONE", missing=drop, default="NONE")
    access = OWSString(name="AccessConstraints", title="AccessConstraints",
                       example="NONE", missing=drop, default="NONE")
    provider = OWSServiceProvider()


class OWSOperationName(ExtendedSchemaNode, OWSNamespace):
    schema_type = String
    attribute = True
    name = "name"
    example = "GetCapabilities"
    validator = OneOf(["GetCapabilities", "DescribeProcess", "Execute"])


class OperationLink(ExtendedSchemaNode, XMLObject):
    schema_type = String
    attribute = True
    name = "href"
    prefix = "xlink"
    example = "http://schema-example.com/wps"


class OperationRequest(ExtendedMappingSchema, OWSNamespace):
    href = OperationLink()


class OWS_HTTP(ExtendedMappingSchema, OWSNamespace):  # noqa: N802
    get = OperationRequest(name="Get", title="OWSGet")
    post = OperationRequest(name="Post", title="OWSPost")


class OWS_DCP(ExtendedMappingSchema, OWSNamespace):  # noqa: N802
    http = OWS_HTTP(name="HTTP", missing=drop)
    https = OWS_HTTP(name="HTTPS", missing=drop)


class Operation(ExtendedMappingSchema, OWSNamespace):
    name = OWSOperationName()
    dcp = OWS_DCP()


class OperationsMetadata(ExtendedSequenceSchema, OWSNamespace):
    name = "OperationsMetadata"
    op = Operation()


class ProcessVersion(ExtendedSchemaNode, WPSNamespace):
    schema_type = String
    attribute = True


class OWSProcessSummary(ExtendedMappingSchema, WPSNamespace):
    version = ProcessVersion(name="processVersion", default="None", example="1.2",
                             description="Version of the corresponding process summary.")
    identifier = OWSProcessIdentifier(example="example", description="Identifier to refer to the process.")
    _title = OWSTitle(example="Example Process", description="Title of the process.")
    abstract = OWSAbstract(example="Process for example schema.", description="Detail about the process.")


class WPSProcessOfferings(ExtendedSequenceSchema, WPSNamespace):
    name = "ProcessOfferings"
    title = "ProcessOfferings"
    process = OWSProcessSummary(name="Process")


class WPSLanguagesType(ExtendedSequenceSchema, WPSNamespace):
    title = "LanguagesType"
    wrapped = False
    lang = OWSLanguage(name="Language")


class WPSLanguageSpecification(ExtendedMappingSchema, WPSNamespace):
    name = "Languages"
    title = "Languages"
    default = OWSLanguage(name="Default")
    supported = WPSLanguagesType(name="Supported")


class WPSResponseBaseType(PermissiveMappingSchema, WPSNamespace):
    _schema = f"{OGC_WPS_1_SCHEMAS}/common/ResponseBaseType.xsd"
    service = WPSServiceAttribute()
    version = WPSVersionAttribute()
    lang = WPSLanguageAttribute()


class WPSProcessVersion(ExtendedSchemaNode, WPSNamespace):
    _schema = f"{OGC_WPS_1_SCHEMAS}/common/ProcessVersion.xsd"
    schema_type = String
    description = "Release version of this Process."
    name = "processVersion"
    attribute = True


class WPSInputDescriptionType(WPSDescriptionType):
    identifier = OWSIdentifier(description="Unique identifier of the input.")
    # override below to have different examples/descriptions
    _title = OWSTitle(description="Human-readable representation of the process input.")
    abstract = OWSAbstract(missing=drop)
    metadata = OWSMetadata(missing=drop)


class WPSLiteralInputType(ExtendedMappingSchema, XMLObject):
    pass


class WPSLiteralData(WPSLiteralInputType):
    name = "LiteralData"


class XMLStringCRS(AnyCRS, XMLObject):
    pass


class WPSCRSsType(ExtendedMappingSchema, WPSNamespace):
    crs = XMLStringCRS(name="CRS", description="Coordinate Reference System")


class WPSSupportedCRS(ExtendedSequenceSchema):
    crs = WPSCRSsType(name="CRS")


class WPSSupportedCRSType(ExtendedMappingSchema, WPSNamespace):
    name = "SupportedCRSsType"
    default = WPSCRSsType(name="Default")
    supported = WPSSupportedCRS(name="Supported")


class WPSBoundingBoxData(ExtendedMappingSchema, XMLObject):
    data = WPSSupportedCRSType(name="BoundingBoxData")


class WPSFormatDefinition(ExtendedMappingSchema, XMLObject):
    mime_type = XMLString(name="MimeType", default=ContentType.TEXT_PLAIN, example=ContentType.TEXT_PLAIN)
    encoding = XMLString(name="Encoding", missing=drop, example="base64")
    schema = XMLString(name="Schema", missing=drop)


class WPSFileFormat(ExtendedMappingSchema, XMLObject):
    name = "Format"
    format_item = WPSFormatDefinition()


class WPSFormatList(ExtendedSequenceSchema):
    format_item = WPSFileFormat()


class WPSComplexInputType(ExtendedMappingSchema, WPSNamespace):
    max_mb = XMLString(name="maximumMegabytes", attribute=True)
    defaults = WPSFileFormat(name="Default")
    supported = WPSFormatList(name="Supported")


class WPSComplexData(ExtendedMappingSchema, XMLObject):
    data = WPSComplexInputType(name="ComplexData")


class WPSInputFormChoice(OneOfKeywordSchema):
    title = "InputFormChoice"
    _one_of = [
        WPSComplexData(),
        WPSLiteralData(),
        WPSBoundingBoxData(),
    ]


class WPSMinOccursAttribute(MinOccursDefinition, XMLObject):
    name = "minOccurs"
    attribute = True


class WPSMaxOccursAttribute(MinOccursDefinition, XMLObject):
    name = "maxOccurs"
    prefix = drop
    attribute = True


class WPSDataInputDescription(ExtendedMappingSchema):
    min_occurs = WPSMinOccursAttribute()
    max_occurs = WPSMaxOccursAttribute()


class WPSDataInputItem(AllOfKeywordSchema, WPSNamespace):
    _all_of = [
        WPSInputDescriptionType(),
        WPSInputFormChoice(),
        WPSDataInputDescription(),
    ]


class WPSDataInputs(ExtendedSequenceSchema, WPSNamespace):
    name = "DataInputs"
    title = "DataInputs"
    input = WPSDataInputItem()


class WPSOutputDescriptionType(WPSDescriptionType):
    name = "OutputDescriptionType"
    title = "OutputDescriptionType"
    identifier = OWSIdentifier(description="Unique identifier of the output.")
    # override below to have different examples/descriptions
    _title = OWSTitle(description="Human-readable representation of the process output.")
    abstract = OWSAbstract(missing=drop)
    metadata = OWSMetadata(missing=drop)


class ProcessOutputs(ExtendedSequenceSchema, WPSNamespace):
    name = "ProcessOutputs"
    title = "ProcessOutputs"
    output = WPSOutputDescriptionType()


class WPSGetCapabilities(WPSResponseBaseType):
    _schema = f"{OGC_WPS_1_SCHEMAS}/wpsGetCapabilities_response.xsd"
    name = "Capabilities"
    title = "Capabilities"  # not to be confused by 'GetCapabilities' used for request
    svc = OWSServiceIdentification()
    ops = OperationsMetadata()
    offering = WPSProcessOfferings()
    languages = WPSLanguageSpecification()


class WPSProcessDescriptionType(WPSResponseBaseType, WPSProcessVersion):
    name = "ProcessDescriptionType"
    description = "Description of the requested process by identifier."
    store = XMLBooleanAttribute(name="storeSupported", example=True, default=True)
    status = XMLBooleanAttribute(name="statusSupported", example=True, default=True)
    inputs = WPSDataInputs()
    outputs = ProcessOutputs()


class WPSProcessDescriptionList(ExtendedSequenceSchema, WPSNamespace):
    name = "ProcessDescriptions"
    title = "ProcessDescriptions"
    description = "Listing of process description for every requested identifier."
    wrapped = False
    process = WPSProcessDescriptionType()


class WPSDescribeProcess(WPSResponseBaseType):
    _schema = f"{OGC_WPS_1_SCHEMAS}/wpsDescribeProcess_response.xsd"
    name = "DescribeProcess"
    title = "DescribeProcess"
    process = WPSProcessDescriptionList()


class WPSStatusLocationAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    name = "statusLocation"
    prefix = drop
    attribute = True
    format = "file"


class WPSServiceInstanceAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    name = "serviceInstance"
    prefix = drop
    attribute = True
    format = "url"


class CreationTimeAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = DateTime
    name = "creationTime"
    title = "CreationTime"
    prefix = drop
    attribute = True


class WPSStatusSuccess(ExtendedSchemaNode, WPSNamespace):
    schema_type = String
    name = "ProcessSucceeded"
    title = "ProcessSucceeded"


class WPSStatusFailed(ExtendedSchemaNode, WPSNamespace):
    schema_type = String
    name = "ProcessFailed"
    title = "ProcessFailed"


class WPSStatus(ExtendedMappingSchema, WPSNamespace):
    name = "Status"
    title = "Status"
    creationTime = CreationTimeAttribute()
    status_success = WPSStatusSuccess(missing=drop)
    status_failed = WPSStatusFailed(missing=drop)


class WPSProcessSummary(ExtendedMappingSchema, WPSNamespace):
    name = "Process"
    title = "Process"
    identifier = OWSProcessIdentifier()
    _title = OWSTitle()
    abstract = OWSAbstract(missing=drop)


class WPSOutputBase(ExtendedMappingSchema):
    identifier = OWSIdentifier()
    _title = OWSTitle()
    abstract = OWSAbstract(missing=drop)


class WPSOutputDefinitionItem(WPSOutputBase, WPSNamespace):
    name = "Output"
    # use different title to avoid OpenAPI schema definition clash with 'Output' of 'WPSProcessOutputs'
    title = "OutputDefinition"


class WPSOutputDefinitions(ExtendedSequenceSchema, WPSNamespace):
    name = "OutputDefinitions"
    title = "OutputDefinitions"
    out_def = WPSOutputDefinitionItem()


class WPSOutputLiteral(ExtendedMappingSchema):
    data = ()


class WPSReference(ExtendedMappingSchema, WPSNamespace):
    href = XMLReferenceAttribute()
    mimeType = MimeTypeAttribute()
    encoding = EncodingAttribute()


class WPSOutputReference(ExtendedMappingSchema):
    title = "OutputReference"
    reference = WPSReference(name="Reference")


class WPSOutputData(OneOfKeywordSchema):
    _one_of = [
        WPSOutputLiteral(),
        WPSOutputReference(),
    ]


class WPSDataOutputItem(AllOfKeywordSchema, WPSNamespace):
    name = "Output"
    # use different title to avoid OpenAPI schema definition clash with 'Output' of 'WPSOutputDefinitions'
    title = "DataOutput"
    _all_of = [
        WPSOutputBase(),
        WPSOutputData(),
    ]


class WPSProcessOutputs(ExtendedSequenceSchema, WPSNamespace):
    name = "ProcessOutputs"
    title = "ProcessOutputs"
    output = WPSDataOutputItem()


class WPSExecuteResponse(WPSResponseBaseType, WPSProcessVersion):
    _schema = f"{OGC_WPS_1_SCHEMAS}/wpsExecute_response.xsd"
    name = "ExecuteResponse"
    title = "ExecuteResponse"  # not to be confused by 'Execute' used for request
    location = WPSStatusLocationAttribute()
    svc_loc = WPSServiceInstanceAttribute()
    process = WPSProcessSummary()
    status = WPSStatus()
    inputs = WPSDataInputs(missing=drop)          # when lineage is requested only
    out_def = WPSOutputDefinitions(missing=drop)  # when lineage is requested only
    outputs = WPSProcessOutputs()


class WPSXMLSuccessBodySchema(OneOfKeywordSchema):
    _one_of = [
        WPSGetCapabilities(),
        WPSDescribeProcess(),
        WPSExecuteResponse(),
    ]


class OWSExceptionCodeAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    name = "exceptionCode"
    title = "Exception"
    attribute = True


class OWSExceptionLocatorAttribute(ExtendedSchemaNode, XMLObject):
    schema_type = String
    name = "locator"
    attribute = True


class OWSExceptionText(ExtendedSchemaNode, OWSNamespace):
    schema_type = String
    name = "ExceptionText"


class OWSException(ExtendedMappingSchema, OWSNamespace):
    name = "Exception"
    title = "Exception"
    code = OWSExceptionCodeAttribute(example="MissingParameterValue")
    locator = OWSExceptionLocatorAttribute(default="None", example="service")
    text = OWSExceptionText(example="Missing service")


class OWSExceptionReport(ExtendedMappingSchema, OWSNamespace):
    name = "ExceptionReport"
    title = "ExceptionReport"
    exception = OWSException()


class WPSException(ExtendedMappingSchema):
    report = OWSExceptionReport()


class OkWPSResponse(ExtendedMappingSchema):
    description = "WPS operation successful"
    header = XmlHeader()
    body = WPSXMLSuccessBodySchema()


class ErrorWPSResponse(ExtendedMappingSchema):
    description = "Unhandled error occurred on WPS endpoint."
    header = XmlHeader()
    body = WPSException()


class ProviderEndpoint(ProviderPath):
    header = RequestHeaders()


class ProcessDescriptionQuery(ExtendedMappingSchema):
    # see: 'ProcessDescription' schema and 'Process.offering' method
    schema = ExtendedSchemaNode(
        String(), example=ProcessSchema.OGC, default=ProcessSchema.OGC,
        validator=OneOfCaseInsensitive(ProcessSchema.values()),
        summary="Selects the desired schema representation of the process description.",
        description=(
            "Selects the desired schema representation of the process description. "
            f"When '{ProcessSchema.OGC}' is used, inputs and outputs will be represented as mapping of objects. "
            "Process metadata are also directly provided at the root of the content. "
            f"When '{ProcessSchema.OLD}' is used, inputs and outputs will be represented as list of objects with ID. "
            "Process metadata are also reported nested under a 'process' field. "
            "See '#/definitions/ProcessDescription' schema for more details about each case. "
            "These schemas are all represented with JSON content. "
            f"For the XML definition, employ '{ProcessSchema.WPS}' or any format selector (f, format, Accept) with XML."
        )
    )


class ProviderProcessEndpoint(ProviderProcessPath):
    header = RequestHeaders()
    querystring = ProcessDescriptionQuery()


class LocalProcessDescriptionQuery(ProcessDescriptionQuery, LocalProcessQuery, FormatQuery):
    pass


class LocalProcessEndpointHeaders(AcceptFormatHeaders, RequestHeaders):  # order important for descriptions to appear
    pass


class ProcessEndpoint(LocalProcessPath):
    header = LocalProcessEndpointHeaders()
    querystring = LocalProcessDescriptionQuery()


class ProcessPackageEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProviderProcessPackageEndpoint(ProviderProcessPath, ProcessPackageEndpoint):
    pass


class ProcessPayloadEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProcessQuoteEstimatorGetEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProcessQuoteEstimatorPutEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()
    body = QuoteEstimatorSchema()


class ProcessQuoteEstimatorDeleteEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProcessVisibilityGetEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProcessVisibilityPutEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()
    body = VisibilitySchema()


class JobStatusQueryProfileSchema(ExtendedSchemaNode):
    summary = "Job status schema representation."
    description = "Selects the schema employed for representation of returned job status response."
    schema_type = String
    title = "JobStatusQuerySchema"
    example = JobStatusSchema.OGC
    default = JobStatusSchema.OGC
    validator = OneOfCaseInsensitive(JobStatusSchema.values())


class GetJobQuery(ExtendedMappingSchema):
    schema = JobStatusQueryProfileSchema(missing=drop)
    profile = JobStatusQueryProfileSchema(missing=drop)


class GetProviderJobEndpoint(ProviderProcessPath, JobPath):
    header = RequestHeaders()
    querystring = GetJobQuery()


class GetJobEndpoint(JobPath):
    header = RequestHeaders()
    querystring = GetJobQuery()


class ProcessInputsEndpoint(LocalProcessPath, JobPath):
    header = RequestHeaders()


class ProviderInputsEndpoint(ProviderProcessPath, JobPath):
    header = RequestHeaders()


class JobInputsOutputsQuery(ExtendedMappingSchema):
    schema = ExtendedSchemaNode(
        String(),
        title="JobInputsOutputsQuerySchema",
        example=JobInputsOutputsSchema.OGC,
        default=JobInputsOutputsSchema.OGC,
        validator=OneOfCaseInsensitive(JobInputsOutputsSchema.values()),
        summary="Selects the schema employed for representation of submitted job inputs and outputs.",
        description=(
            "Selects the schema employed for representing job inputs and outputs that were submitted for execution. "
            f"When '{JobInputsOutputsSchema.OLD}' is employed, listing of object with IDs is returned. "
            f"When '{JobInputsOutputsSchema.OGC}' is employed, mapping of object definitions is returned. "
            "If no schema is requested, the original formats from submission are employed, which could be a mix of "
            "both representations. Providing a schema forces their corresponding conversion as applicable."
        )
    )


class JobInputsEndpoint(JobPath):
    header = RequestHeaders()
    querystring = JobInputsOutputsQuery()


class JobResultsQuery(FormatQuery):
    schema = ExtendedSchemaNode(
        String(),
        title="JobOutputResultsSchema",
        example=JobInputsOutputsSchema.OGC,
        default=JobInputsOutputsSchema.OGC,
        validator=OneOfCaseInsensitive(JobInputsOutputsSchema.values()),
        summary="Selects the schema employed for representation of job outputs.",
        description=(
            "Selects the schema employed for representation of job results (produced outputs) "
            "for providing file Content-Type details. "
            f"When '{JobInputsOutputsSchema.OLD}' is employed, "
            "'format.mimeType' is used and 'type' is reported as well. "
            f"When '{JobInputsOutputsSchema.OGC}' is employed, "
            "'format.mediaType' is used and 'type' is reported as well. "
            "When the '+strict' value is added, only the 'format' or 'type' will be represented according to the "
            f"reference standard ({JobInputsOutputsSchema.OGC}, {JobInputsOutputsSchema.OLD}) representation."
        )
    )


class LocalProcessJobResultsQuery(LocalProcessQuery, JobResultsQuery):
    pass


class JobOutputsEndpoint(JobPath):
    header = RequestHeaders()
    querystring = LocalProcessJobResultsQuery()


class ProcessOutputsEndpoint(LocalProcessPath, JobPath):
    header = RequestHeaders()
    querystring = LocalProcessJobResultsQuery()


class ProviderOutputsEndpoint(ProviderProcessPath, JobPath):
    header = RequestHeaders()
    querystring = JobResultsQuery()


class ProcessResultEndpoint(ProcessOutputsEndpoint):
    deprecated = True
    header = RequestHeaders()


class ProviderResultEndpoint(ProviderOutputsEndpoint):
    deprecated = True
    header = RequestHeaders()


class JobResultEndpoint(JobPath):
    deprecated = True
    header = RequestHeaders()


class ProcessResultsEndpoint(LocalProcessPath, JobPath):
    header = RequestHeaders()


class ProviderResultsEndpoint(ProviderProcessPath, JobPath):
    header = RequestHeaders()


class JobResultsEndpoint(JobPath):
    header = RequestHeaders()


class JobResultsTriggerExecutionEndpoint(JobResultsEndpoint):
    header = RequestHeaders()
    body = NoContent()


class ProcessJobResultsTriggerExecutionEndpoint(JobResultsTriggerExecutionEndpoint, LocalProcessPath):
    pass


class ProviderExceptionsEndpoint(ProviderProcessPath, JobPath):
    header = RequestHeaders()


class JobExceptionsEndpoint(JobPath):
    header = RequestHeaders()


class ProcessExceptionsEndpoint(LocalProcessPath, JobPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProviderLogsEndpoint(ProviderProcessPath, JobPath):
    header = RequestHeaders()


class JobLogsEndpoint(JobPath):
    header = RequestHeaders()


class ProcessLogsEndpoint(LocalProcessPath, JobPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class JobStatisticsEndpoint(JobPath):
    header = RequestHeaders()


class ProcessJobStatisticsEndpoint(LocalProcessPath, JobPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProviderJobStatisticsEndpoint(ProviderProcessPath, JobPath):
    header = RequestHeaders()


##################################################################
# These classes define schemas for requests that feature a body
##################################################################


class ProviderPublic(ExtendedMappingSchema):
    public = ExtendedSchemaNode(
        Boolean(),
        default=False,
        description="Whether the service is defined as publicly visible. "
                    "This will not control allowance/denial of requests to the registered endpoint of the service. "
                    "It only indicates if it should appear during listing of providers."
    )


class CreateProviderRequestBody(ProviderPublic):
    id = AnyIdentifier()
    url = URL(description="Endpoint where to query the provider.")


class ExecuteInputDataType(InputIdentifierType):
    pass


class ExecuteOutputDataType(OutputIdentifierType):
    pass


class ExecuteOutputDefinition(ExtendedMappingSchema):
    transmissionMode = TransmissionModeEnum(missing=drop)
    format = Format(missing=drop)


class ExecuteOutputItem(ExecuteOutputDataType, ExecuteOutputDefinition):
    pass


class ExecuteOutputSpecList(ExtendedSequenceSchema):
    """
    Filter list of outputs to be obtained from execution and their reporting method.
    """
    output = ExecuteOutputItem()


class ExecuteOutputMapAdditionalProperties(ExtendedMappingSchema):
    output_id = ExecuteOutputDefinition(variable="{output-id}", title="ExecuteOutputSpecMap",
                                        description="Desired output reporting method.")


class ExecuteOutputSpecMap(AnyOfKeywordSchema):
    _any_of = [
        ExecuteOutputMapAdditionalProperties(),  # normal {"<output-id>": {...}}
        EmptyMappingSchema(),                    # allows explicitly provided {}
    ]


class ExecuteOutputSpec(OneOfKeywordSchema):
    """
    Filter list of outputs to be obtained from execution and define their reporting method.
    """
    _one_of = [
        # OLD format: {"outputs": [{"id": "<id>", "transmissionMode": "value|reference"}, ...]}
        ExecuteOutputSpecList(),
        # OGC-API:    {"inputs": {"<id>": {"transmissionMode": "value|reference"}, ...}}
        ExecuteOutputSpecMap(),
    ]


class ProviderNameSchema(AnyIdentifier):
    title = "ProviderName"
    description = "Identifier of the remote provider."


class ProviderSummarySchema(DescriptionType, ProviderPublic, DescriptionMeta, DescriptionLinks):
    """
    Service provider summary definition.
    """
    id = ProviderNameSchema()
    url = URL(description="Endpoint of the service provider.")
    type = ExtendedSchemaNode(String())

    _schema_meta_include = True
    _sort_first = PROVIDER_DESCRIPTION_FIELD_FIRST
    _sort_after = PROVIDER_DESCRIPTION_FIELD_AFTER


class ProviderCapabilitiesSchema(ProviderSummarySchema):
    """
    Service provider detailed capabilities.
    """


class TransmissionModeList(ExtendedSequenceSchema):
    transmissionMode = TransmissionModeEnum()


class JobControlOptionsList(ExtendedSequenceSchema):
    jobControlOption = JobControlOptionsEnum()


class ExceptionReportType(ExtendedMappingSchema):
    code = ExtendedSchemaNode(String())
    description = ExtendedSchemaNode(String(), missing=drop)


class ProcessControl(ExtendedMappingSchema):
    jobControlOptions = JobControlOptionsList(missing=ExecuteControlOption.values(),
                                              default=ExecuteControlOption.values())
    outputTransmission = TransmissionModeList(missing=ExecuteTransmissionMode.values(),
                                              default=ExecuteTransmissionMode.values())


class ProcessLocations(ExtendedMappingSchema):
    """
    Additional endpoint locations specific to the process.
    """
    processDescriptionURL = URL(description="Process description endpoint using OGC-API interface.",
                                missing=drop, title="processDescriptionURL")
    processEndpointWPS1 = URL(description="Process description endpoint using WPS-1 interface.",
                              missing=drop, title="processEndpointWPS1")
    executeEndpoint = URL(description="Endpoint where the process can be executed from.",
                          missing=drop, title="executeEndpoint")
    # 'links' already included via 'ProcessDescriptionType->DescriptionType'


class ProcessSummary(
    ProcessDescriptionType,
    DescriptionMeta,
    ProcessControl,
    ProcessLocations,
    DescriptionLinks
):
    """
    Summary process definition.
    """
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/processSummary.yaml"
    _sort_first = PROCESS_DESCRIPTION_FIELD_FIRST
    _sort_after = PROCESS_DESCRIPTION_FIELD_AFTER


class ProcessSummaryList(ExtendedSequenceSchema):
    summary = ProcessSummary()


class ProcessNamesList(ExtendedSequenceSchema):
    process_name = ProcessIdentifierTag(
        description="Process identifier or tagged representation if revision was requested."
    )


class ProcessListing(OneOfKeywordSchema):
    _one_of = [
        ProcessSummaryList(description="Listing of process summary details from existing definitions."),
        ProcessNamesList(description="Listing of process names when not requesting details.",
                         missing=drop),  # in case of empty list, both schema are valid, drop this one to resolve
    ]


class ProcessCollection(ExtendedMappingSchema):
    processes = ProcessListing()


class ProcessPagingQuery(ExtendedMappingSchema):
    sort = ProcessSortEnum(missing=drop)
    # if page is omitted but limit provided, use reasonable zero by default
    page = ExtendedSchemaNode(Integer(allow_string=True), missing=0, default=0, validator=Range(min=0))
    limit = ExtendedSchemaNode(Integer(allow_string=True), missing=None, default=None, validator=Range(min=1),
                               schema=f"{OGC_API_PROC_PART1_PARAMETERS}/limit.yaml")


class ProcessVisibility(ExtendedMappingSchema):
    visibility = VisibilityValue(missing=drop)


class ProcessDeploymentProfile(ExtendedMappingSchema):
    deploymentProfile = URL(missing=drop)


class Process(
    # following are like 'ProcessSummary',
    # except without 'ProcessControl' and 'DescriptionLinks' that are outside the nested 'process'
    ProcessDescriptionType, DescriptionMeta,
    # following are additional fields only in description, just like for OGC-API ProcessDescription
    ProcessContext, ProcessVisibility, ProcessLocations
):
    """
    Old nested process schema for process description.
    """
    # note: deprecated in favor of OGC-API schema
    inputs = DescribeInputTypeList(description="Inputs definition of the process.")
    outputs = DescribeOutputTypeList(description="Outputs definition of the process.")

    _sort_first = PROCESS_DESCRIPTION_FIELD_FIRST
    _sort_after = PROCESS_DESCRIPTION_FIELD_AFTER


class ProcessDescriptionOLD(ProcessControl, ProcessDeploymentProfile, DescriptionLinks):
    """
    Old schema for process description.
    """
    deprecated = True
    process = Process()

    _sort_first = PROCESS_DESCRIPTION_FIELD_FIRST_OLD_SCHEMA
    _sort_after = PROCESS_DESCRIPTION_FIELD_AFTER_OLD_SCHEMA


class ProcessDescriptionOGC(
    ProcessSummary,
    ProcessContext,
    ProcessVisibility,
    ProcessLocations,
    ProcessDeploymentProfile,
    DescriptionLinks
):
    """
    OGC-API schema for process description.
    """
    # technically, empty inputs are allowed for processes that should generate constant/randomized outputs
    # example:
    #   https://pavics.ouranos.ca/twitcher/ows/proxy/catalog
    #   ?service=WPS&request=DescribeProcess&version=1.0.0&identifier=pavicstestdocs
    inputs = DescribeInputTypeMap(description="Inputs definition of the process.", missing=drop, default={})
    outputs = DescribeOutputTypeMap(description="Outputs definition of the process.")

    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/process.yaml"
    _sort_first = PROCESS_DESCRIPTION_FIELD_FIRST
    _sort_after = PROCESS_DESCRIPTION_FIELD_AFTER


class ProcessDescription(OneOfKeywordSchema):
    """
    Supported schema representations of a process description (based on specified query parameters).
    """
    _one_of = [
        ProcessDescriptionOGC,
        ProcessDescriptionOLD,
    ]


class ProcessDeployment(ProcessSummary, ProcessContext, ProcessDeployMeta):
    # override ID to forbid deploy to contain a tagged version part
    # if version should be applied, it must be provided with its 'Version' field
    id = ProcessIdentifier()

    # explicit "abstract" handling for bw-compat, new versions should use "description"
    # only allowed in deploy to support older servers that report abstract (or parsed from WPS-1/2)
    # recent OGC-API v1+ will usually provide directly "description" as per the specification
    abstract = ExtendedSchemaNode(String(), missing=drop, deprecated=True,
                                  description="Detailed explanation of the process being deployed. "
                                              "[Deprecated] Consider using 'description' instead.")
    # allowed undefined I/O during deploy because of reference from owsContext or executionUnit
    inputs = DeployInputTypeAny(
        missing=drop, title="DeploymentInputs",
        description="Additional definitions for process inputs to extend generated details by the referred package. "
                    "These are optional as they can mostly be inferred from the 'executionUnit', but allow specific "
                    f"overrides (see '{DOC_URL}/package.html#correspondence-between-cwl-and-wps-fields')")
    outputs = DeployOutputTypeAny(
        missing=drop, title="DeploymentOutputs",
        description="Additional definitions for process outputs to extend generated details by the referred package. "
                    "These are optional as they can mostly be inferred from the 'executionUnit', but allow specific "
                    f"overrides (see '{DOC_URL}/package.html#correspondence-between-cwl-and-wps-fields')")
    visibility = VisibilityValue(missing=drop)

    _schema = f"{OGC_API_SCHEMA_EXT_DEPLOY}/processSummary.yaml"
    _sort_first = PROCESS_DESCRIPTION_FIELD_FIRST
    _sort_after = PROCESS_DESCRIPTION_FIELD_AFTER


class Duration(ExtendedSchemaNode):
    # note: using String instead of Time because timedelta object cannot be directly handled (missing parts at parsing)
    schema_type = String
    description = "Human-readable representation of the duration."
    example = "hh:mm:ss"


# FIXME: use ISO-8601 duration (?) - P[n]Y[n]M[n]DT[n]H[n]M[n]S
#       https://pypi.org/project/isodate/
#       https://en.wikipedia.org/wiki/ISO_8601#Durations
#   See:
#       'duration.to_iso8601' already employed for quotes, should apply for jobs as well
class DurationISO(ExtendedSchemaNode):
    """
    Duration represented using ISO-8601 format.

    .. seealso::
        - https://json-schema.org/draft/2019-09/json-schema-validation.html#rfc.section.7.3.1
        - :rfc:`3339#appendix-A`
    """
    schema_type = String
    description = "ISO-8601 representation of the duration."
    example = "P[n]Y[n]M[n]DT[n]H[n]M[n]S"
    format = "duration"

    def deserialize(self, cstruct):
        # type: (Union[datetime.timedelta, str]) -> str
        if isinstance(cstruct, datetime.timedelta) or isinstance(cstruct, str) and not cstruct.startswith("P"):
            return duration.to_iso8601(cstruct)
        return cstruct


class JobStatusInfo(ExtendedMappingSchema):
    _schema = OGC_API_SCHEMA_JOB_STATUS_URL
    jobID = JobID()
    processID = ProcessIdentifierTag(missing=None, default=None,
                                     description="Process identifier corresponding to the job execution.")
    providerID = ProcessIdentifier(missing=None, default=None,
                                   description="Provider identifier corresponding to the job execution.")
    type = JobTypeEnum(description="Type of the element associated to the creation of this job.")
    title = JobTitle(missing=drop)
    status = JobStatusEnum(description="Last updated status.")
    message = ExtendedSchemaNode(String(), missing=drop, description="Information about the last status update.")
    created = ExtendedSchemaNode(DateTime(), missing=drop, default=None,
                                 description="Timestamp when the process execution job was created.")
    started = ExtendedSchemaNode(DateTime(), missing=drop, default=None,
                                 description="Timestamp when the process started execution if applicable.")
    finished = ExtendedSchemaNode(DateTime(), missing=drop, default=None,
                                  description="Timestamp when the process completed execution if applicable.")
    updated = ExtendedSchemaNode(DateTime(), missing=drop, default=None,
                                 description="Timestamp of the last update of the job status. This can correspond to "
                                             "any of the other timestamps according to current execution status or "
                                             "even slightly after job finished execution according to the duration "
                                             "needed to deallocate job resources and store results.")
    duration = Duration(missing=drop, description="Duration since the start of the process execution.")
    runningDuration = DurationISO(missing=drop,
                                  description="Duration in ISO-8601 format since the start of the process execution.")
    runningSeconds = Number(missing=drop,
                            description="Duration in seconds since the start of the process execution.")
    expirationDate = ExtendedSchemaNode(DateTime(), missing=drop,
                                        description="Timestamp when the job will be canceled if not yet completed.")
    estimatedCompletion = ExtendedSchemaNode(DateTime(), missing=drop)
    nextPoll = ExtendedSchemaNode(DateTime(), missing=drop,
                                  description="Timestamp when the job will be prompted for updated status details.")
    percentCompleted = Number(example=0, validator=Range(min=0, max=100),
                              description="Completion percentage of the job as indicated by the process.")
    progress = ExtendedSchemaNode(Integer(), example=100, validator=Range(0, 100),
                                  description="Completion progress of the job (alias to 'percentCompleted').")
    links = LinkList(missing=drop)


class JobEntrySchema(OneOfKeywordSchema):
    # note:
    #   Since JobID is a simple string (not a dict), no additional mapping field can be added here.
    #   They will be discarded by `OneOfKeywordSchema.deserialize()`.
    _one_of = [
        JobStatusInfo,
        UUID(description="Job ID."),
    ]


class JobCollection(ExtendedSequenceSchema):
    item = JobEntrySchema()


class CreatedJobStatusSchema(DescriptionSchema):
    jobID = JobID(description="Unique identifier of the created job for execution.")
    processID = ProcessIdentifierTag(description="Identifier of the process that will be executed.")
    providerID = AnyIdentifier(description="Remote provider identifier if applicable.", missing=drop)
    status = ExtendedSchemaNode(String(), example=Status.ACCEPTED)
    location = ExtendedSchemaNode(String(), example="http://{host}/weaver/processes/{my-process-id}/jobs/{my-job-id}")


class PagingBodySchema(ExtendedMappingSchema):
    # don't use defaults if missing, otherwise we might report incorrect values compared to actual contents
    count = ExtendedSchemaNode(Integer(), missing=drop, validator=Range(min=0),
                               description="Number of items returned within this paged result.")
    limit = ExtendedSchemaNode(Integer(), missing=drop, validator=Range(min=1, max=1000),
                               schema=f"{OGC_API_PROC_PART1_PARAMETERS}/limit.yaml",
                               description="Maximum number of items returned per page.")
    page = ExtendedSchemaNode(Integer(), missing=drop, validator=Range(min=0),
                              description="Paging index.")
    total = ExtendedSchemaNode(Integer(), missing=drop, validator=Range(min=0),
                               description="Total number of items regardless of paging.")


class GetPagingJobsSchema(PagingBodySchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/jobList.yaml"  # technically, no 'links' yet, but added after by oneOf
    jobs = JobCollection()


class JobCategoryFilters(PermissiveMappingSchema):
    category = ExtendedSchemaNode(String(), title="CategoryFilter", variable="{category}", default=None, missing=None,
                                  description="Value of the corresponding parameter forming that category group.")


class GroupedJobsCategorySchema(ExtendedMappingSchema):
    category = JobCategoryFilters(description="Grouping values that compose the corresponding job list category.")
    jobs = JobCollection(description="List of jobs that matched the corresponding grouping values.")
    count = ExtendedSchemaNode(Integer(), description="Number of matching jobs for the corresponding group category.")


class GroupedCategoryJobsSchema(ExtendedSequenceSchema):
    job_group_category_item = GroupedJobsCategorySchema()


class GetGroupedJobsSchema(ExtendedMappingSchema):
    groups = GroupedCategoryJobsSchema()


class GetQueriedJobsSchema(OneOfKeywordSchema):
    _one_of = [
        GetPagingJobsSchema(description="Matched jobs according to filter queries."),
        GetGroupedJobsSchema(description="Matched jobs grouped by specified categories."),
    ]
    total = ExtendedSchemaNode(Integer(),
                               description="Total number of matched jobs regardless of grouping or paging result.")
    links = LinkList()  # required by OGC schema

    _sort_first = JOBS_LISTING_FIELD_FIRST
    _sort_after = JOBS_LISTING_FIELD_AFTER


class DismissedJobSchema(ExtendedMappingSchema):
    status = JobStatusEnum()
    jobID = JobID()
    message = ExtendedSchemaNode(String(), example="Job dismissed.")
    percentCompleted = ExtendedSchemaNode(Integer(), example=0)


# same as base Format, but for process/job responses instead of process submission
# (ie: 'Format' is for allowed/supported formats, this is the result format)
class DataEncodingAttributes(FormatSelection):
    pass


class ReferenceBase(ExtendedMappingSchema):
    format = DataEncodingAttributes(missing=drop)
    body = ExtendedSchemaNode(String(), missing=drop)
    bodyReference = ReferenceURL(missing=drop)


class Reference(ReferenceBase):
    title = "Reference"
    href = ReferenceURL(description="Endpoint of the reference.")


class ExecuteReference(ReferenceBase):
    title = "ExecuteReference"
    href = ExecuteReferenceURL(description="Endpoint of the reference.")


class ArrayReference(ExtendedSequenceSchema):
    item = ExecuteReference()


class ArrayReferenceValueType(ExtendedMappingSchema):
    value = ArrayReference()


class ExecuteCollectionFormatEnum(ExtendedSchemaNode):
    schema_type = String
    default = ExecuteCollectionFormat.GEOJSON
    example = ExecuteCollectionFormat.STAC
    validator = OneOf(ExecuteCollectionFormat.values())


class ExecuteCollectionInput(FilterSchema, SortBySchema, PermissiveMappingSchema):
    description = inspect.cleandoc("""
        Reference to a 'collection' that can optionally be filtered, sorted, or parametrized.

        If only the 'collection' is provided to read the contents as a static GeoJSON FeatureCollection document,
        any scheme can be employed (s3, file, http, etc.) to request the contents.
        Note that in this context, each of the respective Feature contained in the collection will be extracted
        to form an array of Features. If the entire 'FeatureCollection' should be provided as a whole to the process
        input, consider using the usual 'href' or 'value' input instead of 'collection'.

        When more 'collection' capabilities are specified with
        additional parameters (filter, sortBy, subsetting, scaling, etc.),
        the scheme must be 'http(s)' since an OGC API or STAC API data access mechanism is expected
        to perform the requested operations. The appropriate API to employ should be indicated by 'format'
        to ensure appropriate interpretation of the 'collection' reference.

        Supported additional parameters depend on each API implementation.
        Not all parameters are listed in this definition. Refer to respective APIs
        for supported parameters and their expected value formats.
    """)
    collection = ExecuteReferenceURL(description="Endpoint of the collection reference.")
    format = ExecuteCollectionFormatEnum(
        missing=drop,
        description="Collection API to employ for filtering and extracting relevant data for the execution.",
    )
    type = MediaType(
        missing=drop,
        title="CollectionMediaType",
        description=(
            "IANA identifier of content-type to extract from the link. "
            "If none specified, default from the collection is employed. "
        )
    )


class ExecuteNestedProcessReference(ExtendedMappingSchema):
    title = "ExecuteNestedProcessReference"
    # 'process' is required for a nested definition, otherwise it will not even be detected as one!
    process = ProcessURL(description="Process reference to be executed.")


class ExecuteNestedProcessParameters(ExtendedMappingSchema):
    """
    Dynamically defines the nested process parameters with recursive schema handling.

    This class must create the nested properties dynamically because the required classes are not yet defined, and
    those required definitions also depend on this class to define the nested process as a possible input value.

    .. seealso::
        - https://docs.pylonsproject.org/projects/colander/en/latest/binding.html
    """
    title = "ExecuteNestedProcessParameters"
    _sort_first = ["process", "inputs", "outputs", "properties", "mode", "response"]
    _schema_extra = {
        "type": null,
        "title": "ExecuteNestedProcessParameters",
        "$ref": f"{OAS3DefinitionHandler.json_pointer}ExecuteProcessParameters"
    }

    @colander.deferred
    def _children(self, __bindings):
        # type: (Dict[str, Any]) -> List[colander.SchemaNode]
        self.children = [node.clone() for node in ExecuteProcessParameters().children]
        for child in self.children:
            # avoid inserting nested default properties that were omitted (ie: mode/response)
            # they should be included explicitly only on the top-most process by 'Execute(ExecuteParameters)' schema
            child.default = null
        return self.children

    # calling 'bind' method will initialize this
    # schema node attribute from the deferred method
    children = _children
    children_bound = False  # only for optimization

    def deserialize(self, cstruct):
        """
        Defer deserialization validation to the class that contains the set of expected properties.

        Additional properties that are added dynamically should "align" to reflect the :term:`OpenAPI` definition,
        although correspondence is not explicitly ensured.
        """
        node = self
        if not self.children_bound:
            node = self.bind()          # ensure bindings are applied to generate children recursive references
            node.children_bound = True  # avoid doing the binding to resolve children on each recursive resolution
        return ExtendedMappingSchema.deserialize(node, cstruct)


class ExecuteNestedProcessInput(AllOfKeywordSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/execute.yaml"
    title = "ExecuteNestedProcessInput"
    description = "Nested process to execute, for which the selected output will become the input of the parent call."

    _all_of = [
        ExecuteNestedProcessReference(),
        ExecuteNestedProcessParameters(),
    ]


# Backward compatible data-input that allows values to be nested under 'data' or 'value' fields,
# both for literal values and link references, for inputs submitted as list-items.
# Also allows the explicit 'href' (+ optional format) reference for a link.
#
# Because this data-input structure applies only to list-items (see 'ExecuteInputItem' below), mapping is always needed.
# (i.e.: values cannot be submitted inline in the list, because field 'id' of each input must also be provided)
# For this reason, one of 'value', 'data', 'href' or 'reference' is mandatory.
class ExecuteInputAnyType(OneOfKeywordSchema):
    """
    Permissive variants that we attempt to parse automatically.
    """
    title = "ExecuteInputAnyType"
    _one_of = [
        # Array of literal data with 'data' key
        ArrayLiteralDataType(),
        # same with 'value' key (OGC specification)
        ArrayLiteralValueType(),
        # Array of HTTP references with various keywords
        ArrayReferenceValueType(),
        # literal data with 'data' key
        AnyLiteralDataType(),
        # same with 'value' key (OGC specification)
        AnyLiteralValueType(),
        # HTTP references with various keywords
        LiteralReference(),
        ExecuteReference(),
        # HTTP reference to a 'collection' with optional processing arguments
        ExecuteCollectionInput(),
        # Nested Process with its own inputs and outputs
        ExecuteNestedProcessInput(),
    ]


class ExecuteInputItem(AllOfKeywordSchema):
    description = (
        "Default value to be looked for uses key 'value' to conform to older drafts of OGC-API standard. "
        "Even older drafts that allowed other fields 'data' instead of 'value' and 'reference' instead of 'href' "
        "are also looked for to remain back-compatible."
    )
    _all_of = [
        ExecuteInputDataType(),
        ExecuteInputAnyType(),
    ]


# backward compatible definition:
#
#   inputs: [
#     {"id": "<id>", "value": <data>},
#     {"id": "<id>", "href": <link>}
#     ... (other variants) ...
#   ]
#
class ExecuteInputListValues(ExtendedSequenceSchema):
    input_item = ExecuteInputItem(summary="Received list input value definition during job submission.")


# same as 'ExecuteInputReference', but using 'OGC' schema with 'type' field
# Defined as:
#   https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/link.yaml
# But explicitly in the context of an execution input, rather than any other link (eg: metadata)
class ExecuteInputFileLink(Link):  # for other metadata (title, hreflang, etc.)
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/link.yaml"
    href = ExecuteReferenceURL(  # not just a plain 'URL' like 'Link' has (extended with s3, vault, etc.)
        description="Location of the file reference."
    )
    type = MediaType(
        default=ContentType.TEXT_PLAIN,  # as per OGC, not mandatory (ie: 'default' supported format)
        description="IANA identifier of content-type located at the link."
    )
    rel = LinkRelationshipType(missing=drop)  # optional opposite to normal 'Link'
    # schema is not official, but logical (same name as under 'format' of process description for a complex file)
    # this extra field is not prohibited from OGC Link definition (just make it explicit here)
    schema = FormatSchema(missing=drop)


# same as 'ExecuteInputLink', but using 'OLD' schema with 'format' field
class ExecuteInputReference(Reference):
    summary = "Execute input reference link definition with parameters."


class ExecuteInputFile(AnyOfKeywordSchema):
    title = "ExecuteInputFile"
    _any_of = [                   # 'href' required for both to provide file link/reference
        ExecuteInputFileLink(),   # 'OGC' schema with 'type: <MediaType>'
        ExecuteInputReference(),  # 'OLD' schema with 'format: {mimeType|mediaType: <MediaType>}'
    ]


# https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/inputValueNoObject.yaml
# Any literal value directly provided inline in input mapping.
#
#   {"inputs": {"<id>": <literal-data>}}
#
# Excludes objects to avoid conflict with later object mapping and {"value": <data>} definitions.
# Excludes array literals that will be defined separately with allowed array of any item within this schema.
# Note: Also supports 'binaryInputValue', since 'type: string' regardless of 'format' is valid.
class ExecuteInputInlineLiteral(AnyLiteralType):
    title = "ExecuteInputInlineLiteral"
    description = "Execute input literal value provided inline."


class BoundingBox2D(ExtendedSequenceSchema):
    description = "Bounding Box using 2D points."
    item = Number()
    validator = Length(min=4, max=4)


class BoundingBox3D(ExtendedSequenceSchema):
    description = "Bounding Box using 3D points."
    item = Number()
    validator = Length(min=6, max=6)


class BoundingBoxValue(OneOfKeywordSchema):
    _one_of = [
        BoundingBox2D,
        BoundingBox3D,
    ]


class BoundingBoxObject(StrictMappingSchema):
    _schema = OGC_API_BBOX_SCHEMA
    description = "Execute bounding box value provided inline."
    format = OGC_API_BBOX_FORMAT
    bbox = BoundingBoxValue(
        description="Point values of the bounding box."
    )
    crs = AnyCRS(
        default="http://www.opengis.net/def/crs/OGC/1.3/CRS84",
        description="Coordinate Reference System of the Bounding Box points.",
    )


class ExecuteInputInlineBoundingBox(BoundingBoxObject):
    _schema_include_deserialize = True


class ExecuteInputInlineValue(OneOfKeywordSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/inputValueNoObject.yaml"
    _one_of = [
        ExecuteInputInlineLiteral(),
        ExecuteInputInlineBoundingBox(),
    ]


class ExecuteInputMeasurement(StrictMappingSchema):
    description = "Execute measurement value with a unit of measure."
    measurement = Number()
    uom = ExtendedSchemaNode(
        String(allow_empty=True),
        title="UoM",
        description="Unit of Measure for the specified value.",
    )
    reference = ExecuteReferenceURL(
        missing=drop,
        description="Reference URL to schema definition of the named entity.",
    )


class ExecuteInputObjectData(NotKeywordSchema, PermissiveMappingSchema):
    summary = "Data provided as any object schema."
    description = (
        "Data provided as any object schema. "
        "This content can represent any JSON definition. "
        "It is recommended to provide either a 'mediaType' or a JSON 'schema' along with this value, "
        "although this is not strictly required."
    )
    _not = [
        ExecuteInputInlineBoundingBox(),
        ExecuteInputMeasurement(),
    ]


# extended 'object' part form:
# https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/inputValue.yaml
class ExecuteInputAnyObjectValue(OneOfKeywordSchema):
    summary = "Data provided as any object schema."
    _one_of = [
        # NOTE: any explicit object variation added here must be applied to 'not' of the generic object schema
        ExecuteInputMeasurement(),
        ExecuteInputObjectData(),
    ]


class ExecuteInputQualifiedLiteralValue(Format):  # default 'mediaType: text/plain'
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/qualifiedInputValue.yaml"
    value = ExecuteInputInlineValue()


class ExecuteInputQualifiedAnyObjectValue(Format):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/qualifiedInputValue.yaml"
    value = ExecuteInputAnyObjectValue()    # can be anything, including literal value, array of them, nested object
    mediaType = MediaType(default=ContentType.APP_JSON, example=ContentType.APP_JSON)  # override default for object


class ExecuteInputQualifiedValue(OneOfKeywordSchema):
    title = "ExecuteInputQualifiedValue"
    # not exactly the same schema, but equivalent, with alternate 'mediaType' defaults for literal vs object
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/inputValue.yaml"
    _schema_include_deserialize = False
    _one_of = [
        ExecuteInputQualifiedLiteralValue(),
        ExecuteInputQualifiedAnyObjectValue(),
    ]


# https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/inlineOrRefData.yaml
#
#   oneOf:
#     - $ref: "inputValueNoObject.yaml"     # in OGC-API spec, includes a generic array
#     - $ref: "qualifiedInputValue.yaml"
#     - $ref: "link.yaml"
#
class ExecuteInputInlineOrRefData(OneOfKeywordSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/inlineOrRefData.yaml"
    _one_of = [
        ExecuteInputInlineValue(),          # <inline-value> (literal, bbox, measurement)
        ExecuteInputQualifiedValue(),       # {"value": <anything>, "mediaType": "<>", "schema": <OAS link or object>}
        ExecuteInputFile(),                 # 'href' with either 'type' (OGC) or 'format' (OLD)
        ExecuteCollectionInput(),           # 'collection' with optional processing operations
        ExecuteNestedProcessInput(),        # 'process' with nested 'inputs', 'outputs', etc.
    ]


class ExecuteInputArrayValues(ExtendedSequenceSchema):
    item_value = ExecuteInputInlineOrRefData()


# combine 'inlineOrRefData' and its 'array[inlineOrRefData]' variants to simplify 'ExecuteInputAny' definition
class ExecuteInputData(OneOfKeywordSchema, StrictMappingSchema):
    description = "Execute data definition of the input."
    _one_of = [
        ExecuteInputInlineOrRefData(),
        ExecuteInputArrayValues(),
    ]


# https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/execute.yaml
#
#   inputs:
#     additionalProperties:           # this is the below 'variable=<input-id>'
#       oneOf:
#       - $ref: "inlineOrRefData.yaml"
#       - type: array
#         items:
#           $ref: "inlineOrRefData.yaml"
#
# depend on 'StrictMappingSchema' such that any unmapped "additionalProperties"
# caused by the nested schema to fail validation is refused in the final mapping
class ExecuteInputMapAdditionalProperties(StrictMappingSchema):
    input_id = ExecuteInputData(variable="{input-id}", title="ExecuteInputData",
                                description="Received mapping input value definition during job submission.")


class ExecuteInputMapValues(AnyOfKeywordSchema):
    _any_of = [
        ExecuteInputMapAdditionalProperties(),  # normal {"<input-id>": {...}}
        EmptyMappingSchema(),                   # allows explicitly provided {}
    ]


class ExecuteInputValues(OneOfKeywordSchema):
    _one_of = [
        # OLD format: {"inputs": [{"id": "<id>", "value": <data>}, ...]}
        ExecuteInputListValues(description="Process job execution inputs defined as item listing."),
        # OGC-API:    {"inputs": {"<id>": <data>, "<id>": {"value": <data>}, ...}}
        ExecuteInputMapValues(description="Process job execution inputs defined as mapping."),
    ]


class ExecuteInputOutputs(ExtendedMappingSchema):
    # Permit unspecified (optional) inputs for processes that could technically allow no-inputs definition (CWL).
    # This is very unusual in real world scenarios, but has some possible cases: constant endpoint fetcher, RNG output.
    #
    # NOTE:
    #   It is **VERY** important to use 'default={}' and not 'missing=drop' contrary to other optional fields.
    #   Using 'drop' causes and invalid input definition to be ignored/removed and not be validated for expected schema.
    #   We want to ensure format is validated if present to rapidly report the issue and not move on to full execution.
    #   If 'inputs' are indeed omitted, the default will match against an empty 'ExecuteInputMapValues' schema.
    #   If 'inputs' are explicitly provided as '{}' or '[]', it will also behave the right way for no-inputs process.
    #
    # See tests validating both cases (incorrect schema vs optionals inputs):
    #   - 'tests.wps_restapi.test_processes.WpsRestApiProcessesTest.test_execute_process_missing_required_params'
    #   - 'tests.wps_restapi.test_providers.WpsRestApiProcessesTest.test_execute_process_no_error_not_required_params'
    #   - 'tests.wps_restapi.test_providers.WpsRestApiProcessesTest.test_get_provider_process_no_inputs'
    #   - 'tests.wps_restapi.test_colander_extras.test_oneof_variable_dict_or_list'
    inputs = ExecuteInputValues(default={}, description="Values submitted for execution.")
    outputs = ExecuteOutputSpec(
        description=(
            "Defines which outputs to be obtained from the execution (filtered or all), "
            "as well as the reporting method for each output according to 'transmissionMode', "
            "the 'response' type, and the execution 'mode' provided "
            f"(see for more details: {DOC_URL}/processes.html#execution-body)."
        ),
        # NOTE:
        #   Explicitly submitted {} or [] means that *no outputs* are requested.
        #   This must be distinguished from 'all outputs' requested, which is done by omitting 'outputs' field entirely.
        missing=None,
        default=None,
    )


class ExecuteParameters(ExecuteInputOutputs):
    """
    Basic execution parameters that can be submitted to run a process.

    These parameters can be either for a top-level process job, or any nested process call.
    """
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/execute.yaml"
    mode = JobExecuteModeEnum(
        missing=drop,
        default=ExecuteMode.AUTO,
        deprecated=True,
    )
    response = JobResponseOptionsEnum(
        missing=drop,  # no default to ensure 'Prefer' header vs 'response' body resolution order can be performed
    )
    notification_email = Email(
        missing=drop,
        deprecated=True,
        description=(
            "Optionally send a notification email when the job is completed. "
            "This is equivalent to using subscribers for both failed and successful job status emails simultaneously."
        )
    )
    subscribers = JobExecuteSubscribers(missing=drop)


class ExecuteProcessParameters(ExecuteParameters):
    title = "ExecuteProcessParameters"
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/execute.yaml"
    _sort_first = [
        "title",
        "process",
        "inputs",
        "outputs",
        "properties",
        "mode",
        "response",
        "subscribers",
    ]
    _title = JobTitle(name="title", missing=drop)
    process = ProcessURL(
        missing=drop,
        description=(
            "Process reference to be executed. "
            "This parameter is required if the process cannot be inferred from the request endpoint."
        ),
        example="https://example.com/processes/example"
    )


class ExecuteJobParameters(ExtendedMappingSchema):
    _title = JobTitle(name="title", missing=drop)
    status = JobStatusCreate(
        description=(
            "Status to request creation of the job without submitting it to processing queue "
            "and leave it pending until triggered by another results request to start it "
            "(see *OGC API - Processes* - Part 4: Job Management)."
        ),
        missing=drop,
    )


class Execute(AllOfKeywordSchema):
    """
    Main execution parameters that can be submitted to run a process.

    Additional parameters are only applicable to the top-most process in a nested definition.
    """
    # OGC 'execute.yaml' does not enforce any required item
    description = "Process execution parameters."
    examples = {
        "ExecuteJSON": {
            "summary": "Execute a process job using REST JSON payload with OGC API schema.",
            "value": EXAMPLES["job_execute.json"],
        },
    }
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/execute.yaml"
    _sort_first = [
        "title",
        "status",
        "process",
        "inputs",
        "outputs",
        "properties",
        "mode",
        "response",
        "subscribers",
    ]
    _all_of = [
        ExecuteJobParameters(),
        ExecuteProcessParameters(),
    ]


class QuoteStatusSchema(ExtendedSchemaNode):
    schema_type = String
    validator = OneOf(QuoteStatus.values())


class PartialQuoteSchema(ExtendedMappingSchema):
    status = QuoteStatusSchema()
    quoteID = UUID(description="Quote ID.")
    processID = ProcessIdentifierTag(description="Process identifier corresponding to the quote definition.")


class DecimalRegex(Regex):
    # because the received value can be a Decimal,
    # the pattern match object cannot perform regex check directly on it
    def __call__(self, node, value):
        return super().__call__(node, str(value))


class PriceAmount(ExtendedSchemaNode):
    schema_type = Money()
    format = "decimal"  # https://github.com/OAI/OpenAPI-Specification/issues/845#issuecomment-378139730
    description = "Monetary value of the price."
    validator = All(
        Range(min=0),
        DecimalRegex(re.compile("^[0-9]+.[0-9]+$"), msg="Number must be formatted as currency decimal."),
    )


class PriceCurrency(ExtendedSchemaNode):
    schema_type = String()
    description = "Currency code in ISO-4217 format."
    default = "USD"  # most common online
    validator = All(
        Length(min=3, max=3),
        OneOf(sorted(list_currencies())),
    )


class PriceSchema(ExtendedMappingSchema):
    amount = PriceAmount()
    currency = PriceCurrency(description=(
        "Until processed by the quotation estimator for the process, corresponds to the user-requested currency. "
        "Once processed, the requested currency will be applied for the amount if exchange rates could be resolved. "
        "Otherwise, the estimator-specific or API-wide default currency value will be used for the estimated amount."
    ))

    def __json__(self, value):
        """
        Handler for :mod:`pyramid` and :mod:`webob` if the object reaches the JSON serializer.

        Combined with :mod:`simplejson` to automatically handle :class:`Decimal` conversion.
        """
        return super().deserialize(value)


class QuoteProcessParameters(PermissiveMappingSchema, ExecuteInputOutputs):
    description = (
        "Parameters passed for traditional process execution (inputs, outputs) "
        "with added metadata for quote evaluation."
    )


class QuoteEstimateValue(PermissiveMappingSchema):
    description = "Details of an estimated value, with it attributed rate and resulting cost."
    estimate = PositiveNumber(default=Decimal("0.0"), missing=None)
    rate = PositiveNumber(default=Decimal("1.0"), missing=None)
    cost = PositiveNumber(default=Decimal("0.0"), missing=Decimal("0.0"))


class QuoteStepChainedInputLiteral(StrictMappingSchema, QuoteEstimatorWeightedParameterSchema):
    value = AnyLiteralType()


class QuoteStepChainedInputComplex(StrictMappingSchema, QuoteEstimatorWeightedParameterSchema):
    size = PositiveNumber()


class QuoteStepChainedInput(OneOfKeywordSchema):
    _one_of = [
        QuoteStepChainedInputLiteral(),
        QuoteStepChainedInputComplex(),
    ]


class QuoteStepOutputParameters(ExtendedMappingSchema):
    description = "Outputs from a quote estimation to be chained as inputs for a following Workflow step."
    output_id = QuoteStepChainedInput(
        variable="{output-id}",
        description="Mapping of output to chain as input for quote estimation.",
    )


class QuoteProcessResults(PermissiveMappingSchema):
    _schema = f"{WEAVER_SCHEMA_URL}/quotation/quote-estimation-result.yaml"
    description = (
        "Results of the quote estimation. "
        "Will be empty until completed. "
        "Contents may vary according to the estimation methodology. "
        "Each category provides details about its contribution toward the total."
    )
    flat = QuoteEstimateValue(missing=drop)
    memory = QuoteEstimateValue(missing=drop)
    storage = QuoteEstimateValue(missing=drop)
    duration = QuoteEstimateValue(missing=drop)
    cpu = QuoteEstimateValue(missing=drop)
    gpu = QuoteEstimateValue(missing=drop)
    total = PositiveNumber(default=Decimal("0.0"))
    currency = PriceCurrency(missing=drop, description=(
        "Optional currency employed by the estimator to produce the quote. "
        "API-wide default currency employed if not specified."
    ))


class UserIdSchema(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(String(), missing=drop),
        ExtendedSchemaNode(Integer(), default=None),
    ]


class StepQuotation(PartialQuoteSchema):
    detail = ExtendedSchemaNode(String(), description="Detail about quote processing.", missing=None)
    price = PriceSchema(description="Estimated price for process execution.")
    expire = ExtendedSchemaNode(DateTime(), description="Expiration date and time of the quote in ISO-8601 format.")
    created = ExtendedSchemaNode(DateTime(), description="Creation date and time of the quote in ISO-8601 format.")
    userID = UserIdSchema(description="User ID that requested the quote.", missing=required, default=None)
    estimatedTime = Duration(missing=drop,
                             description="Estimated duration of process execution in human-readable format.")
    estimatedSeconds = ExtendedSchemaNode(Integer(), missing=drop,
                                          description="Estimated duration of process execution in seconds.")
    estimatedDuration = DurationISO(missing=drop,
                                    description="Estimated duration of process execution in ISO-8601 format.")
    processParameters = QuoteProcessParameters(title="QuoteProcessParameters")
    results = QuoteProcessResults(title="QuoteProcessResults", default={})
    outputs = QuoteStepOutputParameters(missing=drop)


class StepQuotationList(ExtendedSequenceSchema):
    description = "Detailed child processes and prices part of the complete quote."
    step = StepQuotation(description="Quote of a workflow step process.")


class Quotation(StepQuotation):
    paid = ExtendedSchemaNode(Boolean(), default=False, description=(
        "Indicates if the quote as been paid by the user. "
        "This is mandatory in order to execute the job corresponding to the produced quote."
    ))
    steps = StepQuotationList(missing=drop)


class QuoteStepReferenceList(ExtendedSequenceSchema):
    description = "Summary of child process quote references part of the complete quote."
    ref = ReferenceURL()


class QuoteBase(ExtendedMappingSchema):
    price = PriceSchema(description=(
        "Total price of the quote including all estimated costs and step processes if applicable."
    ))


class QuoteSummary(PartialQuoteSchema, QuoteBase):
    steps = QuoteStepReferenceList()


class QuoteSchema(Quotation, QuoteBase):
    pass


class QuotationList(ExtendedSequenceSchema):
    quote = UUID(description="Quote ID.")


class QuotationListSchema(PagingBodySchema):
    _sort_first = QUOTES_LISTING_FIELD_FIRST
    _sort_after = QUOTES_LISTING_FIELD_AFTER

    quotations = QuotationList()


class CreatedQuotedJobStatusSchema(PartialQuoteSchema, CreatedJobStatusSchema):
    billID = UUID(description="ID of the created bill.")


class BillSchema(ExtendedMappingSchema):
    billID = UUID(description="Bill ID.")
    quoteID = UUID(description="Original quote ID that produced this bill.", missing=drop)
    processID = ProcessIdentifierTag()
    jobID = JobID()
    title = ExtendedSchemaNode(String(), description="Name of the bill.")
    description = ExtendedSchemaNode(String(), missing=drop)
    price = PriceSchema(description="Price associated to the bill.")
    created = ExtendedSchemaNode(DateTime(), description="Creation date and time of the bill in ISO-8601 format.")
    userID = ExtendedSchemaNode(Integer(), description="User id that requested the quote.")


class BillList(ExtendedSequenceSchema):
    bill = UUID(description="Bill ID.")


class BillListSchema(ExtendedMappingSchema):
    bills = BillList()


class SupportedValues(ExtendedMappingSchema):
    pass


class DefaultValues(ExtendedMappingSchema):
    pass


class CWLClass(ExtendedSchemaNode):
    # in this case it is ok to use 'name' because target fields receiving it will
    # never be able to be named 'class' because of Python reserved keyword
    name = "class"
    title = "Class"
    schema_type = String
    example = "CommandLineTool"
    validator = OneOf(["CommandLineTool", "ExpressionTool", "Workflow"])
    description = (
        "CWL class specification. This is used to differentiate between single Application Package (AP)"
        "definitions and Workflow that chains multiple packages."
    )


class CWLExpression(ExtendedSchemaNode):
    schema_type = String
    title = "CWLExpression"
    description = (
        f"When combined with '{CWL_REQUIREMENT_INLINE_JAVASCRIPT}', "
        "this field allows runtime parameter references "
        f"(see also: {CWL_CMD_TOOL_URL}#Expression)."
    )


class RequirementClass(ExtendedSchemaNode):
    # in this case it is ok to use 'name' because target fields receiving it will
    # never be able to be named 'class' because of Python reserved keyword
    name = "class"
    title = "RequirementClass"
    schema_type = String
    description = "CWL requirement class specification."


class CUDAComputeCapability(ExtendedSchemaNode):
    schema_type = String
    example = "3.0"
    title = "CUDA compute capability"
    description = "The compute capability supported by the GPU hardware."
    validator = SemanticVersion(regex=r"^\d+\.\d+$")


class CUDAComputeCapabilityArray(ExtendedSequenceSchema):
    item = CUDAComputeCapability()
    validator = Length(min=1)


class CUDAComputeCapabilitySchema(OneOfKeywordSchema):
    # https://github.com/common-workflow-language/cwltool/blob/67a180/cwltool/extensions.yml#L178
    title = CUDAComputeCapability.title
    description = inspect.cleandoc("""
        The compute capability supported by the GPU hardware.

        * If this is a single value, it defines only the minimum
          compute capability.  GPUs with higher capability are also
          accepted.
        * If it is an array value, then only select GPUs with compute
          capabilities that explicitly appear in the array.

        See https://docs.nvidia.com/deploy/cuda-compatibility/#faq and
        https://docs.nvidia.com/cuda/cuda-c-best-practices-guide/index.html#cuda-compute-capability for details.
    """)
    _one_of = [
        CUDAComputeCapability,
        CUDAComputeCapabilityArray,
    ]


class CUDARequirementSpecification(PermissiveMappingSchema):
    # https://github.com/common-workflow-language/cwltool/blob/67a180/cwltool/extensions.yml#L178
    cudaVersionMin = ExtendedSchemaNode(
        String(),
        example="11.4",
        title="CUDA version minimum",
        description=inspect.cleandoc("""
            The minimum CUDA version required to run the software. This corresponds to a CUDA SDK release.

            When run in a container, the container image should provide the CUDA runtime, and the host
            driver is injected into the container.  In this case, because CUDA drivers are backwards compatible,
            it is possible to use an older SDK with a newer driver across major versions.

            See https://docs.nvidia.com/deploy/cuda-compatibility/ for details.
        """),
        validator=SemanticVersion(regex=r"^\d+\.\d+$"),
    )
    cudaComputeCapability = CUDAComputeCapabilitySchema()
    cudaDeviceCountMin = ExtendedSchemaNode(
        Integer(),
        example=1,
        default=1,
        validator=Range(min=1),
        title="CUDA device count minimum",
        description="The minimum amount of devices required.",
    )
    cudaDeviceCountMax = ExtendedSchemaNode(
        Integer(),
        example=8,
        default=1,
        validator=Range(min=1),
        title="CUDA device count maximum",
        description="The maximum amount of devices required.",
    )


class CUDARequirementMap(ExtendedMappingSchema):
    CUDARequirement = CUDARequirementSpecification(
        name=CWL_REQUIREMENT_CUDA,
        title=CWL_REQUIREMENT_CUDA,
    )


class CUDARequirementClass(CUDARequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_CUDA, validator=OneOf([CWL_REQUIREMENT_CUDA]))


class NetworkAccessRequirementSpecification(PermissiveMappingSchema):
    networkAccess = ExtendedSchemaNode(
        Boolean(),
        example=True,
        title="Network Access",
        description="Indicate whether a process requires outgoing IPv4/IPv6 network access.",
    )


class NetworkAccessRequirementMap(ExtendedMappingSchema):
    NetworkAccessRequirement = NetworkAccessRequirementSpecification(
        name=CWL_REQUIREMENT_NETWORK_ACCESS,
        title=CWL_REQUIREMENT_NETWORK_ACCESS,
    )


class NetworkAccessRequirementClass(NetworkAccessRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_NETWORK_ACCESS, validator=OneOf([CWL_REQUIREMENT_NETWORK_ACCESS]))


class ResourceRequirementValue(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(Float(), validator=BoundedRange(min=0.0, exclusive_min=True)),
        ExtendedSchemaNode(Integer(), validator=Range(min=1)),
        CWLExpression,
    ]


class ResourceRequirementSpecification(PermissiveMappingSchema):
    description = inspect.cleandoc(f"""
        Specify basic hardware resource requirements
        (see also: {CWL_CMD_TOOL_URL}#{CWL_REQUIREMENT_RESOURCE}).
    """)
    coresMin = ResourceRequirementValue(
        missing=drop,
        default=1,
        title="ResourceCoresMinimum",
        summary="Minimum reserved number of CPU cores.",
        description=inspect.cleandoc("""
            Minimum reserved number of CPU cores.

            May be a fractional value to indicate to a scheduling algorithm that one core can be allocated
            to multiple jobs. For example, a value of 0.25 indicates that up to 4 jobs may run in parallel
            on 1 core. A value of 1.25 means that up to 3 jobs can run on a 4 core system (4/1.25 ≈ 3).

            Processes can only share a core allocation if the sum of each of their 'ramMax', 'tmpdirMax',
            and 'outdirMax' requests also do not exceed the capacity of the node.

            Processes sharing a core must have the same level of isolation (typically a container or VM)
            that they would normally have.

            The reported number of CPU cores reserved for the process, which is available to expressions on the
            'CommandLineTool' as 'runtime.cores', must be a non-zero integer, and may be calculated by rounding up
            the cores request to the next whole number.

            Scheduling systems may allocate fractional CPU resources by setting quotas or scheduling weights.
            Scheduling systems that do not support fractional CPUs may round up the request to the next whole number.
        """),
    )
    coresMax = ResourceRequirementValue(
        missing=drop,
        title="ResourceCoresMaximum",
        summary="Maximum reserved number of CPU cores.",
        description=inspect.cleandoc("""
            Maximum reserved number of CPU cores.
            See 'coresMin' for discussion about fractional CPU requests.
        """),
    )
    ramMin = ResourceRequirementValue(
        missing=drop,
        default=256,
        title="ResourceRAMMinimum",
        summary="Minimum reserved RAM in mebibytes.",
        description=inspect.cleandoc("""
            Minimum reserved RAM in mebibytes (2**20).

            May be a fractional value. If so, the actual RAM request must be rounded up to the next whole number.
            The reported amount of RAM reserved for the process, which is available to expressions on the
            'CommandLineTool' as 'runtime.ram', must be a non-zero integer.
        """),
    )
    ramMax = ResourceRequirementValue(
        missing=drop,
        title="ResourceRAMMaximum",
        summary="Maximum reserved RAM in mebibytes.",
        description=inspect.cleandoc("""
            Maximum reserved RAM in mebibytes (2**20).
            See 'ramMin' for discussion about fractional RAM requests.
        """),
    )
    tmpdirMin = ResourceRequirementValue(
        missing=drop,
        default=1024,
        title="ResourceTmpDirMinimum",
        summary="Minimum reserved filesystem based storage for the designated temporary directory in mebibytes.",
        description=inspect.cleandoc("""
            Minimum reserved filesystem based storage for the designated temporary directory in mebibytes (2**20).

            May be a fractional value. If so, the actual storage request must be rounded up to the next whole number.
            The reported amount of storage reserved for the process, which is available to expressions on the
            'CommandLineTool' as 'runtime.tmpdirSize', must be a non-zero integer.
        """),
    )
    tmpdirMax = ResourceRequirementValue(
        missing=drop,
        title="ResourceTmpDirMaximum",
        summary="Maximum reserved filesystem based storage for the designated temporary directory in mebibytes.",
        description=inspect.cleandoc("""
            Maximum reserved filesystem based storage for the designated temporary directory in mebibytes (2**20).
            See 'tmpdirMin' for discussion about fractional storage requests.
        """),
    )
    outdirMin = ResourceRequirementValue(
        missing=drop,
        default=1024,
        title="ResourceOutDirMinimum",
        summary="Minimum reserved filesystem based storage for the designated output directory in mebibytes.",
        description=inspect.cleandoc("""
            Minimum reserved filesystem based storage for the designated output directory in mebibytes (2**20).

            May be a fractional value. If so, the actual storage request must be rounded up to the next whole number.
            The reported amount of storage reserved for the process, which is available to expressions on the
            'CommandLineTool' as runtime.outdirSize, must be a non-zero integer.
        """),
    )
    outdirMax = ResourceRequirementValue(
        missing=drop,
        default=1,
        title="ResourceOutDirMaximum",
        summary="Maximum reserved filesystem based storage for the designated output directory in mebibytes.",
        description=inspect.cleandoc("""
            Maximum reserved filesystem based storage for the designated output directory in mebibytes (2**20).
            See 'outdirMin' for discussion about fractional storage requests.
        """),
    )


class ResourceRequirementMap(ExtendedMappingSchema):
    ResourceRequirement = ResourceRequirementSpecification(
        name=CWL_REQUIREMENT_RESOURCE,
        title=CWL_REQUIREMENT_RESOURCE,
    )


class ResourceRequirementClass(ResourceRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_RESOURCE, validator=OneOf([CWL_REQUIREMENT_RESOURCE]))


class DockerRequirementSpecification(PermissiveMappingSchema):
    dockerPull = ExtendedSchemaNode(
        String(),
        example="docker-registry.host.com/namespace/image:1.2.3",
        title="Docker pull reference",
        description="Reference package that will be retrieved and executed by CWL."
    )


class DockerRequirementMap(ExtendedMappingSchema):
    DockerRequirement = DockerRequirementSpecification(
        name=CWL_REQUIREMENT_APP_DOCKER,
        title=CWL_REQUIREMENT_APP_DOCKER
    )


class DockerRequirementClass(DockerRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_APP_DOCKER, validator=OneOf([CWL_REQUIREMENT_APP_DOCKER]))


class DockerGpuRequirementSpecification(DockerRequirementSpecification):
    deprecated = True
    description = inspect.cleandoc(f"""
        Docker requirement with GPU-enabled support (https://github.com/NVIDIA/nvidia-docker).
        The instance must have the NVIDIA toolkit installed to use this feature.

        WARNING:
        This requirement is specific to Weaver and is preserved only for backward compatibility.
        Prefer the combined use of official '{CWL_REQUIREMENT_APP_DOCKER}' and '{CWL_REQUIREMENT_CUDA}'
        for better support of GPU capabilities and portability to other CWL-supported platforms.
    """)


class DockerGpuRequirementMap(ExtendedMappingSchema):
    deprecated = True
    req = DockerGpuRequirementSpecification(name=CWL_REQUIREMENT_APP_DOCKER_GPU)


class DockerGpuRequirementClass(DockerGpuRequirementSpecification):
    deprecated = True
    title = CWL_REQUIREMENT_APP_DOCKER_GPU
    _class = RequirementClass(example=CWL_REQUIREMENT_APP_DOCKER_GPU, validator=OneOf([CWL_REQUIREMENT_APP_DOCKER_GPU]))


class DirectoryListingItem(PermissiveMappingSchema):
    entry = ExtendedSchemaNode(String(), missing=drop)
    entryname = ExtendedSchemaNode(String(), missing=drop)
    writable = ExtendedSchemaNode(Boolean(), missing=drop)


class InitialWorkDirListing(ExtendedSequenceSchema):
    item = DirectoryListingItem()


class InitialWorkDirRequirementSpecification(PermissiveMappingSchema):
    listing = InitialWorkDirListing()


class InitialWorkDirRequirementMap(ExtendedMappingSchema):
    req = InitialWorkDirRequirementSpecification(name=CWL_REQUIREMENT_INIT_WORKDIR)


class InitialWorkDirRequirementClass(InitialWorkDirRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_INIT_WORKDIR,
                              validator=OneOf([CWL_REQUIREMENT_INIT_WORKDIR]))


class InlineJavascriptLibraries(ExtendedSequenceSchema):
    description = inspect.cleandoc("""
        Additional code fragments that will also be inserted before executing the expression code.
        Allows for function definitions that may be called from CWL expressions.
    """)
    exp_lib = ExtendedSchemaNode(String(), missing=drop)


class InlineJavascriptRequirementSpecification(PermissiveMappingSchema):
    description = inspect.cleandoc(f"""
        Indicates that the workflow platform must support inline Javascript expressions.
        If this requirement is not present, the workflow platform must not perform expression interpolation
        (see also: {CWL_CMD_TOOL_URL}#{CWL_REQUIREMENT_INLINE_JAVASCRIPT}).
    """)
    expressionLib = InlineJavascriptLibraries(missing=drop)


class InlineJavascriptRequirementMap(ExtendedMappingSchema):
    req = InlineJavascriptRequirementSpecification(name=CWL_REQUIREMENT_INLINE_JAVASCRIPT)


class InlineJavascriptRequirementClass(InlineJavascriptRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_INLINE_JAVASCRIPT,
                              validator=OneOf([CWL_REQUIREMENT_INLINE_JAVASCRIPT]))


class InplaceUpdateRequirementSpecification(PermissiveMappingSchema):
    description = inspect.cleandoc(f"""
        If 'inplaceUpdate' is true, then an implementation supporting this feature may permit tools to directly
        update files with 'writable: true' in '{CWL_REQUIREMENT_INIT_WORKDIR}'. That is, as an optimization,
        files may be destructively modified in place as opposed to copied and updated
        (see also: {CWL_CMD_TOOL_URL}#{CWL_REQUIREMENT_INPLACE_UPDATE}).
    """)
    inplaceUpdate = ExtendedSchemaNode(Boolean())


class InplaceUpdateRequirementMap(ExtendedMappingSchema):
    req = InplaceUpdateRequirementSpecification(name=CWL_REQUIREMENT_INPLACE_UPDATE)


class InplaceUpdateRequirementClass(InplaceUpdateRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_INPLACE_UPDATE,
                              validator=OneOf([CWL_REQUIREMENT_INPLACE_UPDATE]))


class LoadContents(ExtendedSchemaNode):
    schema_type = Boolean
    title = "LoadContents"
    description = "Indicates if  a 'File' type reference should be loaded under the 'contents' property."


class LoadListingEnum(ExtendedSchemaNode):
    schema_type = String
    title = "LoadListingEnum"
    validator = OneOf(["no_listing", "shallow_listing", "deep_listing"])


class LoadListingRequirementSpecification(PermissiveMappingSchema):
    description = (
        "Specify the desired behavior for loading the listing field of a 'Directory' object for use by expressions "
        f"(see also: {CWL_CMD_TOOL_URL}#{CWL_REQUIREMENT_LOAD_LISTING})."
    )
    loadListing = LoadListingEnum()


class LoadListingRequirementMap(ExtendedMappingSchema):
    req = LoadListingRequirementSpecification(name=CWL_REQUIREMENT_LOAD_LISTING)


class LoadListingRequirementClass(LoadListingRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_LOAD_LISTING,
                              validator=OneOf([CWL_REQUIREMENT_LOAD_LISTING]))


class IdentifierArray(ExtendedSequenceSchema):
    item = AnyIdentifier()


class MultipleInputRequirementSpecification(PermissiveMappingSchema):
    description = inspect.cleandoc(f"""
        Indicates that a Workflow Step Input must support multiple 'source' simultaneously
        (see also: {CWL_WORKFLOW_URL}#WorkflowStepInput).
    """)


class MultipleInputRequirementMap(ExtendedMappingSchema):
    req = MultipleInputRequirementSpecification(name=CWL_REQUIREMENT_MULTIPLE_INPUT)


class MultipleInputRequirementClass(MultipleInputRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_MULTIPLE_INPUT, validator=OneOf([CWL_REQUIREMENT_MULTIPLE_INPUT]))


class ScatterIdentifiersSchema(OneOfKeywordSchema):
    title = "Scatter"
    description = inspect.cleandoc("""
        The scatter field specifies one or more input parameters which will be scattered.
        An input parameter may be listed more than once. The declared type of each input parameter implicitly
        becomes an array of items of the input parameter type. If a parameter is listed more than once, it
        becomes a nested array. As a result, upstream parameters which are connected to scattered parameters
        must be arrays.

        All output parameter types are also implicitly wrapped in arrays. Each job in the scatter results in an
        entry in the output array.

        If any scattered parameter runtime value is an empty array, all outputs are set to empty arrays and
        no work is done for the step, according to applicable scattering rules.
    """)
    _one_of = [
        AnyIdentifier(),
        IdentifierArray(validator=Length(min=1)),
    ]


class ScatterFeatureReference(PermissiveMappingSchema):
    scatter = ScatterIdentifiersSchema(missing=drop)
    scatterMethod = ExtendedSchemaNode(
        String(),
        validator=OneOf(["dotproduct", "nested_crossproduct", "flat_crossproduct"]),
        missing=drop,
        description=inspect.cleandoc("""
            If 'scatter' declares more than one input parameter, 'scatterMethod' describes how to decompose the
            input into a discrete set of jobs.

            - dotproduct: specifies that each of the input arrays are aligned and one element taken from each array
              to construct each job. It is an error if all input arrays are not the same length.

            - nested_crossproduct: specifies the Cartesian product of the inputs, producing a job for every
              combination of the scattered inputs. The output must be nested arrays for each level of scattering,
              in the order that the input arrays are listed in the 'scatter' field.

            - flat_crossproduct: specifies the Cartesian product of the inputs, producing a job for every combination
              of the scattered inputs. The output arrays must be flattened to a single level, but otherwise listed in
              the order that the input arrays are listed in the 'scatter' field.
        """)
    )


class ScatterFeatureRequirementSpecification(StrictMappingSchema):
    description = inspect.cleandoc(f"""
        A 'scatter' operation specifies that the associated Workflow step should execute separately over a list of
        input elements. Each job making up a scatter operation is independent and may be executed concurrently
        (see also: {CWL_WORKFLOW_URL}#WorkflowStep).
    """)


class ScatterFeatureRequirementMap(ExtendedMappingSchema):
    req = ScatterFeatureRequirementSpecification(name=CWL_REQUIREMENT_SCATTER)


class ScatterFeatureRequirementClass(ScatterFeatureRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_SCATTER, validator=OneOf([CWL_REQUIREMENT_SCATTER]))


class SecretsRequirementSpecification(StrictMappingSchema):
    description = "Lists input parameters containing sensitive information to be masked."
    secrets = IdentifierArray(
        title="Secrets",
        description="Input parameter identifiers to consider as secrets.",
        validator=Length(min=1),
    )


class SecretsRequirementMap(ExtendedMappingSchema):
    req = SecretsRequirementSpecification(name=CWL_REQUIREMENT_SECRETS)


class SecretsRequirementClass(SecretsRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_SECRETS, validator=OneOf([CWL_REQUIREMENT_SECRETS]))


class StepInputExpressionSpecification(StrictMappingSchema):
    description = inspect.cleandoc(f"""
        Indicate that the workflow platform must support the 'valueFrom' field of {CWL_WORKFLOW_URL}#WorkflowStepInput.
    """)


class StepInputExpressionRequirementMap(ExtendedMappingSchema):
    req = StepInputExpressionSpecification(name=CWL_REQUIREMENT_STEP_INPUT_EXPRESSION)


class StepInputExpressionRequirementClass(StepInputExpressionSpecification):
    _class = RequirementClass(
        example=CWL_REQUIREMENT_STEP_INPUT_EXPRESSION,
        validator=OneOf([CWL_REQUIREMENT_STEP_INPUT_EXPRESSION]),
    )


class SubworkflowRequirementSpecification(StrictMappingSchema):
    description = "Indicates that a Workflow employs another Workflow as one of its steps."


class SubworkflowRequirementMap(ExtendedMappingSchema):
    req = SubworkflowRequirementSpecification(name=CWL_REQUIREMENT_SUBWORKFLOW)


class SubworkflowRequirementClass(SubworkflowRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_SUBWORKFLOW, validator=OneOf([CWL_REQUIREMENT_SUBWORKFLOW]))


class TimeLimitValue(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(Float(), validator=Range(min=0.0)),
        ExtendedSchemaNode(Integer(), validator=Range(min=0)),
        CWLExpression,
    ]


class ToolTimeLimitRequirementSpecification(PermissiveMappingSchema):
    description = inspect.cleandoc("""
        Set an upper limit on the execution time of a CommandLineTool.
        A CommandLineTool whose execution duration exceeds the time limit may be preemptively terminated
        and considered failed. May also be used by batch systems to make scheduling decisions.
        The execution duration excludes external operations, such as staging of files, pulling a docker image etc.,
        and only counts wall-time for the execution of the command line itself.
    """)
    timelimit = TimeLimitValue(
        description=inspect.cleandoc("""
            The time limit, in seconds.
            A time limit of zero means no time limit.
            Negative time limits are an error.
        """)
    )


class ToolTimeLimitRequirementMap(ExtendedMappingSchema):
    req = ToolTimeLimitRequirementSpecification(name=CWL_REQUIREMENT_TIME_LIMIT)


class ToolTimeLimitRequirementClass(ToolTimeLimitRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_TIME_LIMIT, validator=OneOf([CWL_REQUIREMENT_TIME_LIMIT]))


class EnableReuseValue(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(Boolean(allow_string=False)),
        CWLExpression,
    ]


class WorkReuseRequirementSpecification(PermissiveMappingSchema):
    description = inspect.cleandoc(f"""
        For implementations that support reusing output from past work
        (on the assumption that same code and same input produce same results),
        control whether to enable or disable the reuse behavior for a particular tool or step
        (to accommodate situations where that assumption is incorrect).
        A reused step is not executed but instead returns the same output as the original execution.

        If '{CWL_REQUIREMENT_WORK_REUSE}' is not specified, correct tools should assume it is enabled by default.
    """)
    enableReuse = EnableReuseValue(
        description=inspect.cleandoc(f"""
            Indicates if reuse is enabled for this tool.
            Can be an expression when combined with '{CWL_REQUIREMENT_INLINE_JAVASCRIPT}'
            (see also: {CWL_CMD_TOOL_URL}#Expression).
        """)
    )


class WorkReuseRequirementMap(ExtendedMappingSchema):
    req = WorkReuseRequirementSpecification(name=CWL_REQUIREMENT_WORK_REUSE)


class WorkReuseRequirementClass(WorkReuseRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_WORK_REUSE, validator=OneOf([CWL_REQUIREMENT_WORK_REUSE]))


class BuiltinRequirementSpecification(PermissiveMappingSchema):
    title = CWL_REQUIREMENT_APP_BUILTIN
    description = (
        "Hint indicating that the Application Package corresponds to a builtin process of "
        "this instance. (note: can only be an 'hint' as it is unofficial CWL specification)."
    )
    process = AnyIdentifier(description="Builtin process identifier.")


class BuiltinRequirementMap(ExtendedMappingSchema):
    req = BuiltinRequirementSpecification(name=CWL_REQUIREMENT_APP_BUILTIN)


class BuiltinRequirementClass(BuiltinRequirementSpecification):
    _class = RequirementClass(example=CWL_REQUIREMENT_APP_BUILTIN, validator=OneOf([CWL_REQUIREMENT_APP_BUILTIN]))


class ESGF_CWT_RequirementSpecification(PermissiveMappingSchema):  # noqa: N802
    title = CWL_REQUIREMENT_APP_ESGF_CWT
    description = (
        "Hint indicating that the Application Package corresponds to an ESGF-CWT provider process"
        "that should be remotely executed and monitored by this instance. "
        "(note: can only be an 'hint' as it is unofficial CWL specification)."
    )
    process = AnyIdentifier(description="Process identifier of the remote ESGF-CWT provider.")
    provider = URL(description="ESGF-CWT provider endpoint.")


class ESGF_CWT_RequirementMap(ExtendedMappingSchema):  # noqa: N802
    req = ESGF_CWT_RequirementSpecification(name=CWL_REQUIREMENT_APP_ESGF_CWT)


class WeaverESGF_CWT_RequirementMap(ExtendedMappingSchema):
    req = ESGF_CWT_RequirementSpecification(name=f"{CWL_NAMESPACE_WEAVER_ID}:{CWL_REQUIREMENT_APP_ESGF_CWT}")


class ESGF_CWT_RequirementClass(ESGF_CWT_RequirementSpecification):  # noqa: N802
    _class = RequirementClass(
        example=CWL_REQUIREMENT_APP_ESGF_CWT,
        validator=OneOf([CWL_REQUIREMENT_APP_ESGF_CWT, f"{CWL_NAMESPACE_WEAVER_ID}:{CWL_REQUIREMENT_APP_ESGF_CWT}"]),
    )


class OGCAPIRequirementSpecification(PermissiveMappingSchema):
    title = CWL_REQUIREMENT_APP_OGC_API
    description = (
        "Hint indicating that the Application Package corresponds to an OGC API - Processes provider"
        "that should be remotely executed and monitored by this instance. "
        "(note: can only be an 'hint' as it is unofficial CWL specification)."
    )
    process = ReferenceURL(description="Process URL of the remote OGC API Process.")


class OGCAPIRequirementMap(ExtendedMappingSchema):
    req = OGCAPIRequirementSpecification(name=CWL_REQUIREMENT_APP_OGC_API)


class WeaverOGCAPIRequirementMap(ExtendedMappingSchema):
    req = OGCAPIRequirementSpecification(name=f"{CWL_NAMESPACE_WEAVER_ID}:{CWL_REQUIREMENT_APP_OGC_API}")


class OGCAPIRequirementClass(OGCAPIRequirementSpecification):
    _class = RequirementClass(
        example=CWL_REQUIREMENT_APP_OGC_API,
        validator=OneOf([CWL_REQUIREMENT_APP_OGC_API, f"{CWL_NAMESPACE_WEAVER_ID}:{CWL_REQUIREMENT_APP_OGC_API}"]),
    )


class WPS1RequirementSpecification(PermissiveMappingSchema):
    title = CWL_REQUIREMENT_APP_WPS1
    description = (
        "Hint indicating that the Application Package corresponds to a WPS-1 provider process"
        "that should be remotely executed and monitored by this instance. "
        "(note: can only be an 'hint' as it is unofficial CWL specification)."
    )
    process = AnyIdentifier(description="Process identifier of the remote WPS provider.")
    provider = URL(description="WPS provider endpoint.")


class WPS1RequirementMap(ExtendedMappingSchema):
    req = WPS1RequirementSpecification(name=CWL_REQUIREMENT_APP_WPS1)


class WeaverWPS1RequirementMap(ExtendedMappingSchema):
    req = WPS1RequirementSpecification(name=f"{CWL_NAMESPACE_WEAVER_ID}:{CWL_REQUIREMENT_APP_WPS1}")


class WPS1RequirementClass(WPS1RequirementSpecification):
    _class = RequirementClass(
        example=CWL_REQUIREMENT_APP_WPS1,
        validator=OneOf([CWL_REQUIREMENT_APP_WPS1, f"{CWL_NAMESPACE_WEAVER_ID}:{CWL_REQUIREMENT_APP_WPS1}"]),
    )


class UnknownRequirementMap(PermissiveMappingSchema):
    description = "Generic schema to allow alternative CWL requirements/hints not explicitly defined in schemas."


class UnknownRequirementClass(PermissiveMappingSchema):
    _class = RequirementClass(example="UnknownRequirement")


class CWLRequirementsMapDefinitions(AnyOfKeywordSchema):
    _any_of = [
        DockerRequirementMap(missing=drop),
        DockerGpuRequirementMap(missing=drop),
        InitialWorkDirRequirementMap(missing=drop),
        InlineJavascriptRequirementMap(missing=drop),
        InplaceUpdateRequirementMap(missing=drop),
        LoadListingRequirementMap(missing=drop),
        MultipleInputRequirementMap(missing=drop),
        NetworkAccessRequirementMap(missing=drop),
        ResourceRequirementMap(missing=drop),
        ScatterFeatureRequirementMap(missing=drop),
        StepInputExpressionRequirementMap(missing=drop),
        SubworkflowRequirementMap(missing=drop),
        ToolTimeLimitRequirementMap(missing=drop),
        WorkReuseRequirementMap(missing=drop),
        # specific weaver-namespaced definitions
        # note:
        #   Do not allow 'builtin', since it is only an internal 'hint', not 'required' for CWL execution.
        #   Also, disallow its used explicitly from deployment.
        WeaverESGF_CWT_RequirementMap(missing=drop),
        WeaverOGCAPIRequirementMap(missing=drop),
        WeaverWPS1RequirementMap(missing=drop),
    ]


class CWLRequirementsMapSupported(StrictMappingSchema):
    description = "Schema that ensures only supported CWL requirements are permitted."

    def __init__(self, *_, **__):
        # type: (*Any, **Any) -> None
        """
        Initialize the mapping to allow only supported CWL requirements.

        Because :class:`StrictMappingSchema` is used, initialized sub-nodes are equivalent
        the following :term:`JSON` schema definition, with a key of every known requirement name.

        .. code-block::

            {
              "<supported-requirement>": {},
              "additionalProperties: false
            }

        The actual validation of nested fields under each requirement will be
        handled by their respective schema in :class:`CWLRequirementsMapDefinitions`.
        """
        super().__init__(*_, **__)
        self.children = [
            PermissiveMappingSchema(name=req, missing=drop)
            for req in CWL_REQUIREMENTS_SUPPORTED
        ]


class CWLRequirementsMap(AllOfKeywordSchema):
    _all_of = [
        CWLRequirementsMapDefinitions(),
        CWLRequirementsMapSupported(),
    ]


class CWLRequirementsItem(OneOfKeywordSchema):
    # in case there is any conflict between definitions,
    # the 'class' field can be used to discriminate which one is expected.
    # however, this **requires** that every mapping defined in '_one_of' should
    # provide a 'class' definition that can perform discrimination using a validator
    discriminator = "class"
    _one_of = [
        DockerRequirementClass(missing=drop),
        DockerGpuRequirementClass(missing=drop),
        InitialWorkDirRequirementClass(missing=drop),
        InlineJavascriptRequirementClass(missing=drop),
        InplaceUpdateRequirementClass(missing=drop),
        LoadListingRequirementClass(missing=drop),
        MultipleInputRequirementClass(missing=drop),
        NetworkAccessRequirementClass(missing=drop),
        ResourceRequirementClass(missing=drop),
        ScatterFeatureRequirementClass(missing=drop),
        StepInputExpressionRequirementClass(missing=drop),
        SubworkflowRequirementClass(missing=drop),
        ToolTimeLimitRequirementClass(missing=drop),
        WorkReuseRequirementClass(missing=drop),
        # specific weaver-namespaced definitions
        # note:
        #   Do not allow 'builtin', since it is only an internal 'hint', not 'required' for CWL execution.
        #   Also, disallow its used explicitly from deployment.
        WeaverESGF_CWT_RequirementMap(missing=drop),
        WeaverOGCAPIRequirementMap(missing=drop),
        WeaverWPS1RequirementMap(missing=drop),
    ]


class CWLRequirementsList(ExtendedSequenceSchema):
    requirement = CWLRequirementsItem()


class CWLRequirements(OneOfKeywordSchema):
    _one_of = [
        CWLRequirementsMap(),
        CWLRequirementsList(),
    ]


class CWLHintsMap(AnyOfKeywordSchema, PermissiveMappingSchema):
    _any_of = [
        BuiltinRequirementMap(missing=drop),
        CUDARequirementMap(missing=drop),
        DockerRequirementMap(missing=drop),
        DockerGpuRequirementMap(missing=drop),
        InitialWorkDirRequirementMap(missing=drop),
        InlineJavascriptRequirementMap(missing=drop),
        InplaceUpdateRequirementMap(missing=drop),
        LoadListingRequirementMap(missing=drop),
        NetworkAccessRequirementMap(missing=drop),
        ResourceRequirementMap(missing=drop),
        ScatterFeatureRequirementMap(missing=drop),
        SecretsRequirementMap(missing=drop),
        StepInputExpressionRequirementMap(missing=drop),
        ToolTimeLimitRequirementMap(missing=drop),
        WorkReuseRequirementMap(missing=drop),
        ESGF_CWT_RequirementMap(missing=drop),
        OGCAPIRequirementMap(missing=drop),
        WPS1RequirementMap(missing=drop),
        UnknownRequirementMap(missing=drop),  # allows anything, must be last
    ]


class CWLHintsItem(AnyOfKeywordSchema, PermissiveMappingSchema):
    _any_of = [
        BuiltinRequirementClass(missing=drop),
        CUDARequirementClass(missing=drop),
        DockerRequirementClass(missing=drop),
        DockerGpuRequirementClass(missing=drop),
        InitialWorkDirRequirementClass(missing=drop),
        InlineJavascriptRequirementClass(missing=drop),
        InplaceUpdateRequirementClass(missing=drop),
        LoadListingRequirementClass(missing=drop),
        NetworkAccessRequirementClass(missing=drop),
        ResourceRequirementClass(missing=drop),
        ScatterFeatureRequirementClass(missing=drop),
        SecretsRequirementClass(missing=drop),
        StepInputExpressionRequirementClass(missing=drop),
        ToolTimeLimitRequirementClass(missing=drop),
        WorkReuseRequirementClass(missing=drop),
        ESGF_CWT_RequirementClass(missing=drop),
        OGCAPIRequirementClass(missing=drop),
        WPS1RequirementClass(missing=drop),
        UnknownRequirementClass(missing=drop),  # allows anything, must be last
    ]


class CWLHintsList(ExtendedSequenceSchema):
    hint = CWLHintsItem()


class CWLHints(OneOfKeywordSchema):
    _one_of = [
        CWLHintsMap(),
        CWLHintsList(),
    ]


class CWLArguments(ExtendedSequenceSchema):
    argument = ExtendedSchemaNode(String())


class CWLTypeString(ExtendedSchemaNode):
    schema_type = String
    description = "Field type definition."
    example = "float"
    validator = OneOf(PACKAGE_TYPE_POSSIBLE_VALUES)


# NOTE: CWL Enum does not support non-string values
#   - https://github.com/common-workflow-language/cwl-v1.2/issues/267
#   - https://github.com/common-workflow-language/common-workflow-language/issues/764
#   - https://github.com/common-workflow-language/common-workflow-language/issues/907
# class CWLTypeSymbolValues(OneOfKeywordSchema):
#     _one_of = [
#         ExtendedSchemaNode(Float()),
#         ExtendedSchemaNode(Integer()),
#         ExtendedSchemaNode(String()),
#     ]
class CWLTypeSymbolValues(ExtendedSchemaNode):
    schema_type = String


class CWLTypeSymbols(ExtendedSequenceSchema):
    symbol = CWLTypeSymbolValues()


class CWLTypeEnum(ExtendedMappingSchema):
    type = ExtendedSchemaNode(String(), example=PACKAGE_ENUM_BASE, validator=OneOf(PACKAGE_CUSTOM_TYPES))
    symbols = CWLTypeSymbols(summary="Allowed values composing the enum.")


class CWLTypeArrayItemObject(ExtendedMappingSchema):
    type = CWLTypeString(validator=OneOf(PACKAGE_ARRAY_ITEMS - PACKAGE_CUSTOM_TYPES))


class CWLTypeArrayItems(OneOfKeywordSchema):
    _one_of = [
        CWLTypeString(title="CWLTypeArrayItemsString", validator=OneOf(PACKAGE_ARRAY_ITEMS)),
        CWLTypeEnum(summary="CWL type as enum of values."),
        CWLTypeArrayItemObject(summary="CWL type in nested object definition."),
    ]


class CWLTypeArray(ExtendedMappingSchema):
    type = ExtendedSchemaNode(String(), example=PACKAGE_ARRAY_BASE, validator=OneOf([PACKAGE_ARRAY_BASE]))
    items = CWLTypeArrayItems()


class CWLTypeBase(OneOfKeywordSchema):
    _one_of = [
        CWLTypeString(summary="CWL type as literal value."),
        CWLTypeArray(summary="CWL type as list of items."),
        CWLTypeEnum(summary="CWL type as enum of values."),
    ]


class CWLTypeList(ExtendedSequenceSchema):
    type = CWLTypeBase()


class CWLType(OneOfKeywordSchema):
    _one_of = [
        CWLTypeBase(summary="CWL type definition."),
        CWLTypeList(summary="Combination of allowed CWL types."),
    ]


class AnyLiteralList(ExtendedSequenceSchema):
    default = AnyLiteralType()


class CWLDefault(OneOfKeywordSchema):
    _one_of = [
        AnyLiteralType(),
        AnyLiteralList(),
    ]


class CWLInputObject(PermissiveMappingSchema):
    type = CWLType()
    default = CWLDefault(missing=drop, description="Default value of input if not provided for task execution.")
    inputBinding = PermissiveMappingSchema(missing=drop, title="Input Binding",
                                           description="Defines how to specify the input for the command.")
    loadContents = LoadContents(missing=drop)


class CWLTypeStringList(ExtendedSequenceSchema):
    description = "List of allowed direct CWL type specifications as strings."
    type = CWLType()


class CWLInputType(OneOfKeywordSchema):
    description = "CWL type definition of the input."
    _one_of = [
        CWLTypeString(summary="Direct CWL type string specification."),
        CWLTypeStringList(summary="List of allowed CWL type strings."),
        CWLInputObject(summary="CWL type definition with parameters."),
    ]


class CWLInputMap(PermissiveMappingSchema):
    input_id = CWLInputType(variable="{input-id}", title="CWLInputDefinition",
                            description=IO_INFO_IDS.format(first="CWL", second="WPS", what="input") +
                            " (Note: '{input-id}' is a variable corresponding for each identifier)")


class CWLInputItem(CWLInputObject):
    id = AnyIdentifier(description=IO_INFO_IDS.format(first="CWL", second="WPS", what="input"))


class CWLInputList(ExtendedSequenceSchema):
    input = CWLInputItem(description=f"Input specification. {CWL_DOC_MESSAGE}")


class CWLInputEmpty(EmptyMappingSchema):
    pass


class CWLInputsDefinition(OneOfKeywordSchema):
    _one_of = [
        CWLInputList(description="Package inputs defined as items."),
        CWLInputMap(description="Package inputs defined as mapping."),
        CWLInputEmpty(description="Package inputs as empty mapping when it takes no arguments."),
    ]


class OutputBinding(PermissiveMappingSchema):
    description = "Defines how to retrieve the output result from the command."
    glob = ExtendedSchemaNode(
        String(),
        missing=drop,
        description="Glob pattern to find the output on disk or mounted docker volume.",
    )


class CWLOutputObject(PermissiveMappingSchema):
    type = CWLType()
    # 'outputBinding' should usually be there most of the time (if not always) to retrieve file,
    # but can technically be omitted in some very specific use-cases such as output literal or output is std logs
    outputBinding = OutputBinding(missing=drop)
    loadContents = LoadContents(missing=drop)


class CWLOutputType(OneOfKeywordSchema):
    _one_of = [
        CWLTypeString(summary="Direct CWL type string specification."),
        CWLTypeStringList(summary="List of allowed CWL type strings."),
        CWLOutputObject(summary="CWL type definition with parameters."),
    ]


class CWLOutputMap(ExtendedMappingSchema):
    output_id = CWLOutputType(variable="{output-id}", title="CWLOutputDefinition",
                              description=IO_INFO_IDS.format(first="CWL", second="WPS", what="output") +
                              " (Note: '{output-id}' is a variable corresponding for each identifier)")


class CWLOutputItem(CWLOutputObject):
    id = AnyIdentifier(description=IO_INFO_IDS.format(first="CWL", second="WPS", what="output"))


class CWLOutputList(ExtendedSequenceSchema):
    input = CWLOutputItem(description=f"Output specification. {CWL_DOC_MESSAGE}")


class CWLOutputsDefinition(OneOfKeywordSchema):
    _one_of = [
        CWLOutputList(description="Package outputs defined as items."),
        CWLOutputMap(description="Package outputs defined as mapping."),
    ]


class CWLCommandParts(ExtendedSequenceSchema):
    cmd = ExtendedSchemaNode(String())


class CWLCommand(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(String(), title="String command."),
        CWLCommandParts(title="CommandParts", summary="Command Parts")
    ]


class CWLVersion(Version):
    description = "CWL version of the described application package."
    example = CWL_VERSION
    validator = SemanticVersion(v_prefix=True, rc_suffix=False)


class CWLIdentifier(ProcessIdentifier):
    description = (
        "Reference to the process identifier. If CWL is provided within a process deployment payload, this can be "
        "omitted. If used in a deployment with only CWL details, this information is required."
    )


class CWLIntentURL(URL):
    description = (
        "Identifier URL to a concept for the type of computational operation accomplished by this Process "
        "(see example operations: http://edamontology.org/operation_0004)."
    )


class CWLIntent(ExtendedSequenceSchema):
    item = CWLIntentURL()


class CWLBase(ExtendedMappingSchema):
    cwlVersion = CWLVersion()


class CWLNamespaces(StrictMappingSchema):
    """
    Mapping of :term:`CWL` namespace definitions for shorthand notation.

    .. note::
        Use a combination of `strict` mapping and ``variable`` (see ``var`` field) such that any additional namespace
        other than the ones explicitly listed are allowed, but if provided, they must succeed URI validation minimally.
        If no additional namespace is provided, including none at all, the mapping definition remains valid because
        of ``missing=drop`` under ``var``. If a URI is invalid for additional namespaces, the failing validation causes
        the property to be unmapped to the variable, which leads to an ``"unknown"`` property raised by the `strict`
        mapping. For explicit URI definitions, the specific URI combinations provided must be matched exactly to
        succeed. This ensures that no invalid mapping gets applied for commonly-known URI namespaces.
    """
    name = "$namespaces"
    title = "CWL Namespaces Mapping"
    description = "Mapping of CWL namespace definitions for shorthand notation."
    var = URI(variable="{namespace}", missing=drop)
    cwl = URI(
        missing=drop,
        name=CWL_NAMESPACE_CWL_SPEC_ID,
        validator=OneOf([CWL_NAMESPACE_CWL_SPEC_URL, CWL_NAMESPACE_CWL_SPEC_URL.rstrip("#")]),
    )
    cwltool = URI(
        missing=drop,
        name=CWL_NAMESPACE_CWLTOOL_ID,
        validator=OneOf([CWL_NAMESPACE_CWLTOOL_URL, CWL_NAMESPACE_CWLTOOL_URL.rstrip("#")]),
    )
    edam = URI(
        missing=drop,
        name=EDAM_NAMESPACE,
        validator=OneOf([EDAM_NAMESPACE_URL, EDAM_NAMESPACE_URL.rstrip("#")]),
    )
    iana = URI(
        missing=drop,
        name=IANA_NAMESPACE,
        validator=OneOf([IANA_NAMESPACE_URL, IANA_NAMESPACE_URL.rstrip("#")]),
    )
    ogc = URI(
        missing=drop,
        name=OGC_NAMESPACE,
        validator=OneOf([OGC_NAMESPACE_URL, OGC_NAMESPACE_URL.rstrip("#")]),
    )
    ogc_api_proc_part1 = URI(
        missing=drop,
        name=CWL_NAMESPACE_OGC_API_PROC_PART1_ID,
        validator=OneOf([CWL_NAMESPACE_OGC_API_PROC_PART1_URL])
    )
    opengis = URI(
        missing=drop,
        name=OPENGIS_NAMESPACE,
        validator=OneOf([OPENGIS_NAMESPACE_URL, OPENGIS_NAMESPACE_URL.rstrip("#")]),
    )
    s = URI(
        missing=drop,
        name=CWL_NAMESPACE_SCHEMA_ID,
        validator=OneOf([CWL_NAMESPACE_SCHEMA_URL, CWL_NAMESPACE_SCHEMA_URL.rstrip("#")]),
    )
    weaver = URI(
        missing=drop,
        name=CWL_NAMESPACE_WEAVER_ID,
        validator=OneOf([CWL_NAMESPACE_WEAVER_URL, CWL_NAMESPACE_WEAVER_URL.rstrip("#")]),
    )


class CWLSchemas(ExtendedSequenceSchema):
    name = "$schemas"
    url = URL(title="CWLSchemaURL", description="Schema reference for the CWL definition.")


class CWLPerson(PermissiveMappingSchema):
    _sort_first = [
        "class",
        CWL_NAMESPACE_SCHEMA_METADATA_IDENTIFIER,
        CWL_NAMESPACE_SCHEMA_METADATA_EMAIL,
        CWL_NAMESPACE_SCHEMA_METADATA_NAME,
    ]
    _class = ExtendedSchemaNode(
        String(),
        name="class",
        validator=OneOf([CWL_NAMESPACE_SCHEMA_METADATA_PERSON]),
        description="Author of the Application Package.",
    )
    _id = URI(
        name=CWL_NAMESPACE_SCHEMA_METADATA_IDENTIFIER,
        description="Reference identifier of the person. Typically, an ORCID URI.",
        missing=drop,
    )
    name = ExtendedSchemaNode(
        String(),
        name=CWL_NAMESPACE_SCHEMA_METADATA_NAME,
        description="Name of the person.",
        missing=drop,
    )
    email = ExtendedSchemaNode(
        String(),
        name=CWL_NAMESPACE_SCHEMA_METADATA_NAME,
        description="Email of the person.",
        missing=drop,
    )


class CWLAuthors(ExtendedSequenceSchema):
    item = CWLPerson()
    validator = Length(min=1)


class CWLDateCreated(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(
            DateTime(),
            name=CWL_NAMESPACE_SCHEMA_METADATA_DATE_CREATED,
            format="date-time",
            description="Date-time of creation in ISO-8601 format.",
        )
    ]


class CWLMetadata(PermissiveMappingSchema):
    _sort_first = [
        "cwlVersion",
        "class",
        "id",
        "label",
        "doc",
        "intent",
        "version",
        CWL_NAMESPACE_SCHEMA_METADATA_DATE_CREATED,
        CWL_NAMESPACE_SCHEMA_METADATA_VERSION,
        CWL_NAMESPACE_SCHEMA_METADATA_SOFTWARE_VERSION,
        CWL_NAMESPACE_SCHEMA_METADATA_CODE_REPOSITORY,
        CWL_NAMESPACE_SCHEMA_METADATA_LICENSE,
        CWL_NAMESPACE_SCHEMA_METADATA_AUTHOR,
        CWL_NAMESPACE_SCHEMA_METADATA_CONTRIBUTOR,
        CWL_NAMESPACE_SCHEMA_METADATA_KEYWORDS,
    ]
    _sort_after = ["$namespaces", "$schemas"]

    _id = ExtendedSchemaNode(String(), name="id", missing=drop)
    label = ExtendedSchemaNode(String(), missing=drop)
    doc = ExtendedSchemaNode(String(), missing=drop)
    intent = ExtendedSchemaNode(String(), missing=drop)
    version = Version(missing=drop)
    s_version = Version(missing=drop, name=CWL_NAMESPACE_SCHEMA_METADATA_VERSION)

    author = CWLAuthors(
        name=CWL_NAMESPACE_SCHEMA_METADATA_AUTHOR,
        missing=drop,
        description="Author(s) of the Application Package.",
    )
    contributor = CWLAuthors(
        name=CWL_NAMESPACE_SCHEMA_METADATA_CONTRIBUTOR,
        missing=drop,
        description="Contributor(s) of the Application Package.",
    )
    keywords = KeywordList(
        name=CWL_NAMESPACE_SCHEMA_METADATA_KEYWORDS,
        missing=drop,
        description="Keywords applied to the Application Package.",
    )
    license = URL(
        name=CWL_NAMESPACE_SCHEMA_METADATA_LICENSE,
        missing=drop,
        description=(
            "License related to the Application Package. "
            "Preferably an URL to the specific license, but generic license URL is allowed."
        ),
    )
    code_repo = URL(
        name=CWL_NAMESPACE_SCHEMA_METADATA_CODE_REPOSITORY,
        missing=drop,
        description="URL to the original code repository providing the Application Package.",
    )
    date_created = ExtendedSchemaNode(
        DateTime(),
        name=CWL_NAMESPACE_SCHEMA_METADATA_DATE_CREATED,
        format="date-time",
        missing=drop,
        description="Date-time of creation in ISO-8601 format.",
    )
    namespaces = CWLNamespaces(missing=drop)
    schemas = CWLSchemas(missing=drop)


class CWLScatterMulti(ExtendedSequenceSchema):
    id = CWLIdentifier()


class CWLScatter(OneOfKeywordSchema):
    _one_of = [
        CWLIdentifier(),
        CWLScatterMulti()
    ]


class CWLScatterMethod(ExtendedSchemaNode):
    schema_type = String
    description = (
        "Describes how to decompose the scattered input into a discrete set of jobs. "
        "When 'dotproduct', specifies that each of the input arrays are aligned and one element taken from each array"
        "to construct each job. It is an error if all input arrays are of different length. "
        "When 'nested_crossproduct', specifies the Cartesian product of the inputs, producing a job for every "
        "combination of the scattered inputs. The output must be nested arrays for each level of scattering, "
        "in the order that the input arrays are listed in the scatter field. "
        "When 'flat_crossproduct', specifies the Cartesian product of the inputs, producing a job for every "
        "combination of the scattered inputs. The output arrays must be flattened to a single level, but otherwise "
        "listed in the order that the input arrays are listed in the scatter field."
    )
    validator = OneOf(["dotproduct", "nested_crossproduct", "flat_crossproduct"])


class CWLScatterDefinition(PermissiveMappingSchema):
    scatter = CWLScatter(missing=drop, description=(
        "One or more input identifier of an application step within a Workflow were an array-based input to that "
        "Workflow should be scattered across multiple instances of the step application."
    ))
    scatterMethod = CWLScatterMethod(missing=drop)


class CWLTool(PermissiveMappingSchema):
    description = "Base definition of the type of CWL tool represented by the object."
    _class = CWLClass()


class CWLWorkflowStepRunDefinition(AnyOfKeywordSchema):
    _any_of = [
        AnyIdentifier(
            title="CWLWorkflowSteprun_id",
            description="Reference to local process ID, with or without '.cwl' extension."
        ),
        CWLFileName(),
        ProcessURL(),
        # do not perform deeper validation other than checking there is a CWL 'class'
        # cwltool should perform any relevant cross-reference check of other files when resolving the workflow
        # Weaver runtime requirements (application package type) should be checked when reaching that step
        CWLTool(),
    ]


class CWLWorkflowStepInputSourceValue(ExtendedSchemaNode):
    description = "Specifies the workflow parameter that will provide input to the underlying step parameter."
    schema_type = String


class CWLWorkflowStepInputSourceList(ExtendedSequenceSchema):
    item = CWLWorkflowStepInputSourceValue()
    validator = Length(min=1)


class CWLWorkflowStepInputSource(OneOfKeywordSchema):
    _one_of = [
        CWLWorkflowStepInputSourceValue(),
        CWLWorkflowStepInputSourceList(),
    ]


class LinkMergeMethod(ExtendedSchemaNode):
    schema_type = String
    title = "LinkMergeMethod"
    validator = OneOf(["merge_nested", "merge_flattened"])


class PickValueMethod(ExtendedSchemaNode):
    schema_type = String
    title = "PickValueMethod"
    validator = OneOf(["first_non_null", "the_only_non_null", "all_non_null"])


class CWLWorkflowStepInputObject(ScatterFeatureReference):
    id = AnyIdentifier(description="Identifier of the step input.", missing=drop)
    source = CWLWorkflowStepInputSource(missing=drop)
    linkMerge = LinkMergeMethod(missing=drop)
    pickValue = PickValueMethod(missing=drop)
    loadListing = LoadListingEnum(missing=drop)
    loadContents = LoadContents(missing=drop)
    valueFrom = CWLExpression(missing=drop)


class CWLWorkflowStepInputItem(CWLWorkflowStepInputObject):
    id = AnyIdentifier(description="Identifier of the step input.")


class CWLWorkflowStepInput(OneOfKeywordSchema):
    _one_of = [
        CWLWorkflowStepInputSourceValue(),
        CWLWorkflowStepInputObject()
    ]


class CWLWorkflowStepInputMap(ExtendedMappingSchema):
    step_in = CWLWorkflowStepInput(variable="{step-in}", title="CWLWorkflowStepInput")


class CWLWorkflowStepInputList(ExtendedSequenceSchema):
    step_in = CWLWorkflowStepInputItem()
    validator = Length(min=1)


class CWLWorkflowStepInputDefinition(OneOfKeywordSchema):
    description = "Defines the input parameters of the workflow step."
    _one_of = [
        CWLWorkflowStepInputMap(),
        CWLWorkflowStepInputList(),
    ]


class CWLWorkflowStepOutputID(ExtendedSchemaNode):
    schema_type = String()
    description = "Identifier of the tool's output to use as the step output."


class CWLWorkflowStepOutputObject(PermissiveMappingSchema):
    id = CWLWorkflowStepOutputID()


class CWLWorkflowStepOutputItem(OneOfKeywordSchema):
    _one_of = [
        CWLWorkflowStepOutputID(),
        CWLWorkflowStepOutputObject(),
    ]


class CWLWorkflowStepOutputList(ExtendedSequenceSchema):
    out = CWLWorkflowStepOutputItem()
    validator = Length(min=1)


class CWLWorkflowStepOutputDefinition(OneOfKeywordSchema):
    description = "Defines the parameters representing the output of the process."
    _one_of = [
        CWLWorkflowStepOutputList(),  # only list allowed
    ]


class CWLWorkflowStepObject(CWLScatterDefinition, PermissiveMappingSchema):
    id = AnyIdentifier(description="Identifier of the step.", missing=drop)
    _in = CWLWorkflowStepInputDefinition(name="in")
    out = CWLWorkflowStepOutputDefinition()
    run = CWLWorkflowStepRunDefinition()
    requirements = CWLRequirements(missing=drop)
    hints = CWLHints(missing=drop)


class CWLWorkflowStepItem(CWLWorkflowStepObject):
    id = AnyIdentifier(description="Identifier of the step.")


class CWLWorkflowStepsMap(ExtendedMappingSchema):
    step_id = CWLWorkflowStepObject(variable="{step-id}", title="CWLWorkflowStep")


class CWLWorkflowStepsList(ExtendedSequenceSchema):
    step = CWLWorkflowStepItem()


class CWLWorkflowStepsDefinition(OneOfKeywordSchema):
    _one_of = [
        CWLWorkflowStepsMap(),
        CWLWorkflowStepsList(),
    ]


class CWLApp(CWLTool, CWLScatterDefinition, PermissiveMappingSchema):
    id = CWLIdentifier(missing=drop)  # can be omitted only if within a process deployment that also includes it
    intent = CWLIntent(missing=drop)
    requirements = CWLRequirements(description="Explicit requirement to execute the application package.", missing=drop)
    hints = CWLHints(description="Non-failing additional hints that can help resolve extra requirements.", missing=drop)
    baseCommand = CWLCommand(description="Command called in the docker image or on shell according to requirements "
                                         "and hints specifications. Can be omitted if already defined in the "
                                         "docker image.", missing=drop)
    arguments = CWLArguments(description="Base arguments passed to the command.", missing=drop)
    inputs = CWLInputsDefinition(description="All inputs available to the Application Package.")
    outputs = CWLOutputsDefinition(description="All outputs produced by the Application Package.")
    steps = CWLWorkflowStepsDefinition(missing=drop)


class CWL(CWLBase, CWLMetadata, CWLApp):
    _sort_first = ["$schema"] + CWLMetadata._sort_first
    _sort_after = ["requirements", "hints", "inputs", "outputs", "steps", "$graph", "$namespaces", "$schemas"]
    _schema = CWL_SCHEMA_URL


class ExecutionUnitCWL(CWL):
    description = f"Execution unit definition as CWL package specification. {CWL_DOC_MESSAGE}"


# use 'strict' base schema such that it can contain only 'unit', nothing else permitted to reduce oneOf conflicts
class ExecutionUnitNested(StrictMappingSchema):
    unit = ExecutionUnitCWL()
    type = MediaType(
        description="IANA identifier of content-type represented by the unit definition.",
        missing=drop,
        validator=OneOf(ContentType.ANY_CWL),
    )


class ProviderSummaryList(ExtendedSequenceSchema):
    provider_service = ProviderSummarySchema()


class ProviderNamesList(ExtendedSequenceSchema):
    provider_name = ProviderNameSchema()


class ProviderListing(OneOfKeywordSchema):
    _one_of = [
        ProviderSummaryList(description="Listing of provider summary details retrieved from remote service."),
        ProviderNamesList(description="Listing of provider names, possibly unvalidated from remote service.",
                          missing=drop),  # in case of empty list, both schema are valid, drop this one to resolve
    ]


class ProvidersBodySchema(ExtendedMappingSchema):
    checked = ExtendedSchemaNode(
        Boolean(),
        description="Indicates if the listed providers have been validated and are accessible from registered URL. "
                    "In such case, provider metadata was partially retrieved from remote services and is accessible. "
                    "Otherwise, only local metadata is provided and service availability is not guaranteed."
    )
    providers = ProviderListing(description="Providers listing according to specified query parameters.")


class ProviderProcessesSchema(ExtendedSequenceSchema):
    provider_process = ProcessSummary()


class JobOutputReference(ExtendedMappingSchema):
    href = ReferenceURL(description="Output file reference.")
    # either with 'type', 'format.mediaType' or 'format.mimeType' according requested 'schema=OGC/OLD'
    # if 'schema=strict' as well, either 'type' or 'format' could be dropped altogether
    type = MediaType(missing=drop, description="IANA Content-Type of the file reference.")
    format = FormatSelection(missing=drop)


class JobOutputArrayReference(ExtendedSequenceSchema):
    item = JobOutputReference()


class JobOutputQualifiedValueLiteral(Format):
    value = AnyLiteralType()
    mediaType = MediaType(missing=drop, example=ContentType.APP_JSON)  # override for optional, others already optional
    format = FormatSelection(missing=drop)


class JobOutputQualifiedDataLiteral(Format):
    data = AnyLiteralType()
    mediaType = MediaType(missing=drop, example=ContentType.APP_JSON)  # override for optional, others already optional
    format = FormatSelection(missing=drop)


class JobOutputLiteral(OneOfKeywordSchema):
    _one_of = [
        # AnyLiteralType(),  # NOTE: purposely omit value inline, always embed in 'value' or 'data' for job outputs
        JobOutputQualifiedDataLiteral(),
        JobOutputQualifiedValueLiteral(),
    ]


class JobOutputArrayLiteral(ExtendedSequenceSchema):
    item = JobOutputLiteral()


class JobOutputValueObject(OneOfKeywordSchema):
    _one_of = [
        JobOutputReference(),
        JobOutputLiteral(),
        # array possible since nested object under 'id'
        JobOutputArrayReference(),
        JobOutputArrayLiteral(),
    ]


class JobOutputMap(ExtendedMappingSchema):
    output_id = JobOutputValueObject(
        variable="{output-id}", title="JobOutputData",
        description=(
            "Output data as literal value or file reference. "
            "(Note: '{output-id}' is a variable corresponding for each identifier)"
        )
    )


class JobOutputFields(OneOfKeywordSchema):
    _one_of = [
        JobOutputReference(),
        JobOutputQualifiedDataLiteral(),
        JobOutputQualifiedValueLiteral(),
    ]


class JobOutputItem(AllOfKeywordSchema):
    _all_of = [
        OutputIdentifierType(),
        JobOutputFields(),  # cannot be an array directly, since 'id' field needed in this representation
    ]


class JobOutputList(ExtendedSequenceSchema):
    title = "JobOutputList"
    output = JobOutputItem(description="Job output result with specific keyword according to represented format.")


class JobOutputs(OneOfKeywordSchema):
    description = "Job outputs with many alternate representations according to the specified 'schema' query parameter."
    _one_of = [
        JobOutputMap(),
        JobOutputList(),
    ]


# implement only literal parts from following schemas:
# https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/inputValueNoObject.yaml
# https://github.com/opengeospatial/ogcapi-processes/blob/master/openapi/schemas/processes-core/inlineOrRefData.yaml
# Other parts are implemented separately with:
#   - 'ValueFormatted' (qualifiedInputValue)
#   - 'ResultBoundingBox' (bbox)
#   - 'ResultReference' (link)
class ResultLiteral(AnyLiteralType):
    ...


class ResultLiteralList(ExtendedSequenceSchema):
    result = ResultLiteral()


class ValueFormatted(ResultFormat):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/qualifiedInputValue.yaml"
    _schema_include_deserialize = False
    value = AnyValueOAS(
        example="<xml><data>test</data></xml>",
        description="Formatted content value of the result."
    )
    format = ResultFormat(missing=drop, schema_include_deserialize=False)  # backward compatibility


class ValueFormattedList(ExtendedSequenceSchema):
    result = ValueFormatted()


class ResultBoundingBox(BoundingBoxObject):
    _schema_include_deserialize = False


class ResultReference(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/link.yaml"
    _schema_include_deserialize = False
    href = ReferenceURL(description="Result file reference.")
    type = MediaType(description="IANA Content-Type of the file reference.")
    format = ResultFormat(missing=drop, schema_include_deserialize=False)  # backward compatibility


class ResultReferenceList(ExtendedSequenceSchema):
    result = ResultReference()


class ResultData(OneOfKeywordSchema):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/inlineOrRefData.yaml"
    _one_of = [
        # must place formatted value first since both value/format fields are simultaneously required
        # other classes require only one of the two, and therefore are more permissive during schema validation
        ValueFormatted(description="Result formatted content value."),
        ValueFormattedList(description="Result formatted content of multiple values."),
        ResultBoundingBox(description="Result bounding box value."),
        ResultReference(description="Result reference location."),
        ResultReferenceList(description="Result locations for multiple references."),
        ResultLiteral(description="Result literal value."),
        ResultLiteralList(description="Result list of literal values."),
    ]


class ResultsDocument(ExtendedMappingSchema):
    description = "Results representation as JSON document."
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/results.yaml"
    output_id = ResultData(
        variable="{output-id}", title="ResultData",
        description=(
            "Resulting value of the output that conforms to 'OGC API - Processes' standard. "
            "(Note: '{output-id}' is a variable corresponding for each output identifier of the process)"
        )
    )


class ResultsContent(ExtendedSchemaNode):
    description = "Results representation as literal contents."
    schema_type = String()


class ResultsBody(OneOfKeywordSchema):
    description = "Results obtained from a successful process job execution."
    _one_of = [
        ResultsDocument(),
        ResultsContent(),
    ]


class WpsOutputContextHeader(ExtendedSchemaNode):
    # ok to use 'name' in this case because target 'key' in the mapping must
    # be that specific value but cannot have a field named with this format
    name = "X-WPS-Output-Context"
    description = (
        "Contextual location where to store WPS output results from job execution. ",
        "When provided, value must be a directory or sub-directories slug. ",
        "Resulting contextual location will be relative to server WPS outputs when no context is provided.",
    )
    schema_type = String
    missing = drop
    example = "my-directory/sub-project"
    default = None


class JobExecuteHeaders(ExtendedMappingSchema):
    description = "Indicates the relevant headers that were supplied for job execution or a null value if omitted."
    accept = AcceptHeader(missing=None)
    accept_language = AcceptLanguageHeader(missing=None)
    content_type = RequestContentTypeHeader(missing=None, default=None)
    prefer = PreferHeader(missing=None)
    x_wps_output_context = WpsOutputContextHeader(missing=None)


class JobInputsBody(ExecuteInputOutputs):
    # note:
    #  following definitions do not employ 'missing=drop' to explicitly indicate the fields
    #  this makes it easier to consider everything that could be implied when executing the job
    mode = JobExecuteModeEnum(default=ExecuteMode.AUTO)
    response = JobResponseOptionsEnum(default=None)
    headers = JobExecuteHeaders(missing={}, default={})
    links = LinkList(missing=drop)


class JobOutputsBody(ExtendedMappingSchema):
    outputs = JobOutputs()
    links = LinkList(missing=drop)


class JobExceptionPlain(ExtendedSchemaNode):
    schema_type = String
    description = "Generic exception description corresponding to any error message."


class JobExceptionDetailed(ExtendedMappingSchema):
    description = "Fields correspond exactly to 'owslib.wps.WPSException' represented as dictionary."
    Code = ExtendedSchemaNode(String())
    Locator = ExtendedSchemaNode(String(), default=None)
    Text = ExtendedSchemaNode(String())


class JobException(OneOfKeywordSchema):
    _one_of = [
        JobExceptionDetailed(),
        JobExceptionPlain()
    ]


class JobExceptionsSchema(ExtendedSequenceSchema):
    exceptions = JobException()


class JobLogsSchema(ExtendedSequenceSchema):
    log = ExtendedSchemaNode(String())


class ApplicationStatisticsSchema(ExtendedMappingSchema):
    mem = ExtendedSchemaNode(String(), name="usedMemory", example="10 MiB")
    mem_bytes = ExtendedSchemaNode(Integer(), name="usedMemoryBytes", example=10485760)


class ProcessStatisticsSchema(ExtendedMappingSchema):
    rss = ExtendedSchemaNode(String(), name="rss", example="140 MiB")
    rss_bytes = ExtendedSchemaNode(Integer(), name="rssBytes", example=146800640)
    uss = ExtendedSchemaNode(String(), name="uss", example="80 MiB")
    uss_bytes = ExtendedSchemaNode(Integer(), name="ussBytes", example=83886080)
    vms = ExtendedSchemaNode(String(), name="vms", example="1.4 GiB")
    vms_bytes = ExtendedSchemaNode(Integer(), name="vmsBytes", example=1503238554)
    used_threads = ExtendedSchemaNode(Integer(), name="usedThreads", example=10)
    used_cpu = ExtendedSchemaNode(Integer(), name="usedCPU", example=2)
    used_handles = ExtendedSchemaNode(Integer(), name="usedHandles", example=0)
    mem = ExtendedSchemaNode(String(), name="usedMemory", example="10 MiB",
                             description="RSS memory employed by the job execution omitting worker memory.")
    mem_bytes = ExtendedSchemaNode(Integer(), name="usedMemoryBytes", example=10485760,
                                   description="RSS memory employed by the job execution omitting worker memory.")
    total_size = ExtendedSchemaNode(String(), name="totalSize", example="10 MiB",
                                    description="Total size to store job output files.")
    total_size_bytes = ExtendedSchemaNode(Integer(), name="totalSizeBytes", example=10485760,
                                          description="Total size to store job output files.")


class OutputStatisticsSchema(ExtendedMappingSchema):
    size = ExtendedSchemaNode(String(), name="size", example="5 MiB")
    size_bytes = ExtendedSchemaNode(Integer(), name="sizeBytes", example=5242880)


class OutputStatisticsMap(ExtendedMappingSchema):
    output = OutputStatisticsSchema(variable="{output-id}", description="Spaced used by this output file.")


class JobStatisticsSchema(ExtendedMappingSchema):
    application = ApplicationStatisticsSchema(missing=drop)
    process = ProcessStatisticsSchema(missing=drop)
    outputs = OutputStatisticsMap(missing=drop)


class FrontpageParameterSchema(ExtendedMappingSchema):
    name = ExtendedSchemaNode(String(), example="api")
    enabled = ExtendedSchemaNode(Boolean(), example=True)
    url = URL(
        description="Referenced parameter endpoint. Root URL when the functionality implies multiple endpoints.",
        example="https://weaver-host",
        missing=drop,
    )
    doc = ExtendedSchemaNode(
        String(),
        description="Endpoint where additional documentation can be found about the functionality.",
        example="https://weaver-host/api",
        missing=drop
    )
    api = URL(
        String(),
        description="OpenAPI documentation endpoint about the functionality.",
        example="https://weaver-host/api",
        missing=drop,
    )


class FrontpageParameters(ExtendedSequenceSchema):
    parameter = FrontpageParameterSchema()


class FrontpageSchema(LandingPage, DescriptionSchema):
    _schema = f"{OGC_API_COMMON_PART1_SCHEMAS}/landingPage.json"
    _sort_first = ["title", "configuration", "message", "description", "attribution"]
    _sort_after = ["parameters", "links"]

    title = ExtendedSchemaNode(String(), default="Weaver", example="Weaver")
    message = ExtendedSchemaNode(String(), default="Weaver Information", example="Weaver Information")
    configuration = ExtendedSchemaNode(String(), default="default", example="default")
    attribution = ExtendedSchemaNode(String(), description="Short representation of the API maintainers.")
    parameters = FrontpageParameters()


class OpenAPISpecSchema(ExtendedMappingSchema):
    # "http://json-schema.org/draft-04/schema#"
    _schema = "https://spec.openapis.org/oas/3.0/schema/2021-09-28"


class SwaggerUISpecSchema(ExtendedMappingSchema):
    pass


class VersionsSpecSchema(ExtendedMappingSchema):
    name = ExtendedSchemaNode(String(), description="Identification name of the current item.", example="weaver")
    type = ExtendedSchemaNode(String(), description="Identification type of the current item.", example="api")
    version = Version(description="Version of the current item.", example="0.1.0")


class VersionsList(ExtendedSequenceSchema):
    version = VersionsSpecSchema()


class VersionsSchema(ExtendedMappingSchema):
    versions = VersionsList()


class ConformanceList(ExtendedSequenceSchema):
    conformance = URL(description="Conformance specification link.",
                      example="http://www.opengis.net/spec/ogcapi-common-1/1.0/conf/core")


class ConformanceSchema(ExtendedMappingSchema):
    _schema = f"{OGC_API_COMMON_PART1_SCHEMAS}/confClasses.json"
    conformsTo = ConformanceList()


#################################################################
# Local Processes schemas
#################################################################


class PackageBody(ExtendedMappingSchema):
    pass


class ExecutionUnit(OneOfKeywordSchema):
    title = "ExecutionUnit"
    description = "Definition of the Application Package to execute."
    _one_of = [
        Reference(name="Reference", title="Reference", description="Execution Unit reference."),
        ExecutionUnitCWL(name="Unit", title="Unit", description="Execution Unit definition directly provided."),
        ExecutionUnitNested(
            name="UnitNested",
            title="UnitNested",
            description="Execution Unit definition nested under a 'unit' property.",
        ),
    ]


class ExecutionUnitList(ExtendedSequenceSchema):
    item = ExecutionUnit(name="ExecutionUnit")
    validator = Length(min=1, max=1)


class ProcessDeploymentWithContext(ProcessDeployment):
    description = "Process deployment with OWS Context reference."
    owsContext = OWSContext(missing=required)


class ProcessVersionField(ExtendedMappingSchema):
    processVersion = Version(
        title="processVersion", missing=drop, deprecated=True,
        description="Old method of specifying the process version, prefer 'version' under 'process'."
    )


class DeployProcessOfferingContext(ProcessControl, ProcessVersionField):
    # alternative definition to make 'executionUnit' optional if instead provided through 'owsContext'
    # case where definition is nested under 'processDescription.process'
    process = ProcessDeploymentWithContext(
        description="Process definition nested under process field for backward compatibility."
    )


class DeployProcessDescriptionContext(NotKeywordSchema, ProcessDeploymentWithContext, ProcessControl):
    # alternative definition to make 'executionUnit' optional if instead provided through 'owsContext'
    # case where definition is directly under 'processDescription'
    _not = [
        Reference()  # avoid conflict with deploy by href
    ]


class DeployProcessContextChoiceType(OneOfKeywordSchema):
    _one_of = [
        DeployProcessOfferingContext(),
        DeployProcessDescriptionContext(),
    ]


class DeployProcessOffering(ProcessControl, ProcessVersionField):
    process = ProcessDeployment(description="Process definition nested under process field for backward compatibility.")


class DeployProcessDescription(NotKeywordSchema, ProcessDeployment, ProcessControl):
    _not = [
        Reference()  # avoid conflict with deploy by href
    ]
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/process.yaml"
    description = "Process description fields directly provided."


class DeployReference(Reference):
    id = ProcessIdentifier(missing=drop, description=(
        "Optional identifier of the specific process to obtain the description from in case the reference URL "
        "corresponds to an endpoint that can refer to multiple process definitions (e.g.: GetCapabilities)."
    ))


class ProcessDescriptionChoiceType(OneOfKeywordSchema):
    _one_of = [
        DeployReference(),
        DeployProcessOffering(),
        DeployProcessDescription(),
    ]


# combination of following 2 references:
#   https://github.com/opengeospatial/ogcapi-processes/blob/8c41db3f/openapi/schemas/processes-dru/ogcapppkg.yaml
#   https://github.com/opengeospatial/ogcapi-processes/blob/8c41db3f/openapi/schemas/processes-dru/ogcapppkg-array.yaml
# but omitting the unsupported generic properties (which can be mapped to equivalent CWL requirements):
#   https://github.com/opengeospatial/ogcapi-processes/blob/8c41db3f/openapi/schemas/processes-dru/executionUnit.yaml
class ExecutionUnitVariations(OneOfKeywordSchema):
    _one_of = [
        # each 'ExecutionUnit', either individually or within an array,
        # already combines the JSON unit (nested under 'unit' or directly) vs href link representations
        ExecutionUnit(),
        ExecutionUnitList(),
    ]


class ExecutionUnitDefinition(ExtendedMappingSchema):
    executionUnit = ExecutionUnitVariations()


class DeployParameters(ExtendedMappingSchema):
    immediateDeployment = ExtendedSchemaNode(Boolean(), missing=drop, default=True)
    deploymentProfileName = URL(missing=drop)


class DeployOGCAppPackage(NotKeywordSchema, ExecutionUnitDefinition, DeployParameters):
    description = "Deployment using standard OGC Application Package definition."
    _schema = f"{OGC_API_SCHEMA_EXT_DEPLOY}/ogcapppkg.yaml"
    _not = [
        CWLBase()
    ]
    processDescription = ProcessDescriptionChoiceType()


class DeployContextDefinition(NotKeywordSchema, DeployParameters):
    # alternative definition to make 'executionUnit' optional if instead provided in 'owsContext' (somewhere nested)
    _not = [
        CWLBase(),
        ExecutionUnitDefinition(),
    ]
    processDescription = DeployProcessContextChoiceType()


class CWLGraphItem(CWLApp):  # no 'cwlVersion', only one at the top
    id = CWLIdentifier()  # required in this case


class CWLGraphList(ExtendedSequenceSchema):
    cwl = CWLGraphItem()


# FIXME: supported nested and $graph multi-deployment (https://github.com/crim-ca/weaver/issues/56)
class CWLGraphBase(ExtendedMappingSchema):
    graph = CWLGraphList(
        name="$graph", description=(
            "Graph definition that defines *exactly one* CWL Application Package represented as list. "
            "Multiple definitions simultaneously deployed is NOT supported currently."
            # "Graph definition that combines one or many CWL Application Packages within a single payload. "
            # "If a single application is given (list of one item), it will be deployed as normal CWL by itself. "
            # "If multiple applications are defined, the first MUST be the top-most Workflow process. "
            # "Deployment of other items will be performed, and the full deployment will be persisted only if all are "
            # "valid. The resulting Workflow will be registered as a package by itself (i.e: not as a graph)."
        ),
        validator=Length(min=1, max=1)
    )


class UpdateVersion(ExtendedMappingSchema):
    version = Version(missing=drop, example="1.2.3", description=(
        "Explicit version to employ for initial or updated process definition. "
        "Must not already exist and must be greater than the latest available semantic version for the "
        "corresponding version level according to the applied update operation. "
        "For example, if only versions '1.2.3' and '1.3.1' exist, the submitted version can be anything before "
        "version '1.2.0' excluding it (i.e.: '1.1.X', '0.1.2', etc.), between '1.2.4' and '1.3.0' exclusively, or "
        "'1.3.2' and anything above. If no version is provided, the next *patch* level after the current process "
        "version is applied. If the current process did not define any version, it is assumed '0.0.0' and this patch"
        "will use '0.0.1'. The applicable update level (MAJOR, MINOR, PATCH) depends on the operation being applied. "
        "As a rule of thumb, if changes affect only metadata, PATCH is required. If changes affect parameters or "
        "execution method of the process, but not directly its entire definition, MINOR is required. If the process "
        "must be completely redeployed due to application redefinition, MAJOR is required."
    ))


class DeployCWLGraph(CWLBase, CWLGraphBase, UpdateVersion):
    pass


class DeployCWL(NotKeywordSchema, CWL, UpdateVersion):
    _not = [
        CWLGraphBase()
    ]
    id = CWLIdentifier()  # required in this case, cannot have a version directly as tag, use 'version' field instead


class DeployOGCRemoteProcess(ExtendedMappingSchema):
    id = ProcessIdentifier(missing=drop, description=(
        "Optional identifier for the new process to deploy. "
        "If not provided, the ID inferred from the specified OGC API - Processes endpoint is reused."
    ))
    process = ProcessURL()


class Deploy(OneOfKeywordSchema):
    _one_of = [
        DeployOGCRemoteProcess(),
        DeployOGCAppPackage(),
        DeployContextDefinition(),
        DeployCWL(),
        DeployCWLGraph(),
    ]


class DeployContentType(ContentTypeHeader):
    example = ContentType.APP_JSON
    default = ContentType.APP_JSON
    validator = OneOf([
        ContentType.APP_JSON,
        ContentType.APP_CWL,
        ContentType.APP_CWL_JSON,
        ContentType.APP_CWL_YAML,
        ContentType.APP_CWL_X,
        ContentType.APP_OGC_PKG_JSON,
        ContentType.APP_OGC_PKG_YAML,
        ContentType.APP_YAML,
    ])


class DeployHeaders(RequestHeaders):
    x_auth_docker = XAuthDockerHeader()
    content_type = DeployContentType()


class PostProcessesEndpoint(ExtendedMappingSchema):
    header = DeployHeaders(description="Headers employed for process deployment.")
    querystring = FormatQuery()
    body = Deploy(title="Deploy", examples={
        "DeployCWL": {
            "summary": "Deploy a process from a CWL definition.",
            "value": EXAMPLES["deploy_process_cwl.json"],
        },
        "DeployOGC": {
            "summary": "Deploy a process from an OGC Application Package definition.",
            "value": EXAMPLES["deploy_process_ogcapppkg.json"],
        },
        "DeployWPS": {
            "summary": "Deploy a process from a remote WPS-1 reference URL.",
            "value": EXAMPLES["deploy_process_wps1.json"],
        }
    })


class UpdateInputOutputBase(DescriptionType, InputOutputDescriptionMeta):
    pass


class UpdateInputOutputItem(InputIdentifierType, UpdateInputOutputBase):
    pass


class UpdateInputOutputList(ExtendedSequenceSchema):
    io_item = UpdateInputOutputItem()


class UpdateInputOutputMap(PermissiveMappingSchema):
    io_id = UpdateInputOutputBase(
        variable="{input-output-id}",
        description="Input/Output definition under mapping for process update."
    )


class UpdateInputOutputDefinition(OneOfKeywordSchema):
    _one_of = [
        UpdateInputOutputMap(),
        UpdateInputOutputList(),
    ]


class PatchProcessBodySchema(UpdateVersion):
    title = ExtendedSchemaNode(String(), missing=drop, description=(
        "New title to override current one. "
        "Minimum required change version level: PATCH."
    ))
    description = ExtendedSchemaNode(String(), missing=drop, description=(
        "New description to override current one. "
        "Minimum required change version level: PATCH."
    ))
    keywords = KeywordList(missing=drop, description=(
        "Keywords to add (append) to existing definitions. "
        "To remove all keywords, submit an empty list. "
        "To replace keywords, perform two requests, one with empty list and the following one with new definitions. "
        "Minimum required change version level: PATCH."
    ))
    metadata = MetadataList(missing=drop, description=(
        "Metadata to add (append) to existing definitions. "
        "To remove all metadata, submit an empty list. "
        "To replace metadata, perform two requests, one with empty list and the following one with new definitions. "
        "Relations must be unique across existing and new submitted metadata. "
        "Minimum required change version level: PATCH."
    ))
    links = LinkList(missing=drop, description=(
        "Links to add (append) to existing definitions. Relations must be unique. "
        "To remove all (additional) links, submit an empty list. "
        "To replace links, perform two requests, one with empty list and the following one with new definitions. "
        "Note that modifications to links only considers custom links. Other automatically generated links such as "
        "API endpoint and navigation references cannot be removed or modified. "
        "Relations must be unique across existing and new submitted links. "
        "Minimum required change version level: PATCH."
    ))
    inputs = UpdateInputOutputDefinition(missing=drop, description=(
        "Update details of individual input elements. "
        "Minimum required change version levels are the same as process-level fields of corresponding names."
    ))
    outputs = UpdateInputOutputDefinition(missing=drop, description=(
        "Update details of individual output elements. "
        "Minimum required change version levels are the same as process-level fields of corresponding names."
    ))
    jobControlOptions = JobControlOptionsList(missing=drop, description=(
        "New job control options supported by this process for its execution. "
        "All desired job control options must be provided (full override, not appending). "
        "Order is important to define the default behaviour (first item) to use when unspecified during job execution. "
        "Minimum required change version level: MINOR."
    ))
    outputTransmission = TransmissionModeList(missing=drop, description=(
        "New output transmission methods supported following this process execution. "
        "All desired output transmission modes must be provided (full override, not appending). "
        "Minimum required change version level: MINOR."
    ))
    visibility = VisibilityValue(missing=drop, description=(
        "New process visibility. "
        "Minimum required change version level: MINOR."
    ))


class PutProcessBodySchema(Deploy):
    description = "Process re-deployment using an updated version and definition."


class PatchProcessEndpoint(LocalProcessPath):
    headers = RequestHeaders()
    querystring = LocalProcessQuery()
    body = PatchProcessBodySchema()


class PutProcessEndpoint(LocalProcessPath):
    headers = RequestHeaders()
    querystring = LocalProcessQuery()
    body = PutProcessBodySchema()


class ExecuteHeadersBase(RequestHeaders):
    description = "Request headers supported for job execution."
    prefer = PreferHeader(missing=drop)
    x_wps_output_context = WpsOutputContextHeader()


class ExecuteHeadersJSON(ExecuteHeadersBase):
    content_type = ContentTypeHeader(
        missing=drop, default=ContentType.APP_JSON,
        validator=OneOf([ContentType.APP_JSON])
    )


class ExecuteHeadersXML(ExecuteHeadersBase):
    content_type = ContentTypeHeader(
        missing=drop, default=ContentType.APP_XML,
        validator=OneOf(ContentType.ANY_XML)
    )


class PostJobsEndpointJSON(ExtendedMappingSchema):
    header = ExecuteHeadersJSON()
    querystring = LocalProcessQuery()
    body = Execute()


class PostProcessJobsEndpointJSON(PostJobsEndpointJSON, LocalProcessPath):
    pass


class PostJobsEndpointXML(ExtendedMappingSchema):
    header = ExecuteHeadersXML()
    querystring = LocalProcessQuery()
    body = WPSExecutePost(
        # very important to override 'name' in this case
        # original schema uses it to specify the XML class name
        # in this context, it is used to define the 'in' location of this schema to form 'requestBody' in OpenAPI
        name="body",
        examples={
            "ExecuteXML": {
                "summary": "Execute a process job using WPS-like XML payload.",
                "value": EXAMPLES["wps_execute_request.xml"],
            }
        }
    )


class PostProcessJobsEndpointXML(PostJobsEndpointXML, LocalProcessPath):
    pass


class JobTitleNullable(OneOfKeywordSchema):
    description = "Job title to update, or unset if 'null'."
    _one_of = [
        JobTitle(),
        ExtendedSchemaNode(NoneType(), name="null"),  # allow explicit 'title: null' to unset a predefined title
    ]


class PatchJobBodySchema(ExecuteProcessParameters):
    description = "Execution request contents to be updated."
    # 'missing=null' ensures that, if a field is provided with an "empty" definition (JSON null, no-field dict, etc.),
    # contents are passed down as is rather than dropping them (what 'missing=drop' would do due to DropableSchemaNode)
    # this is to allow "unsetting" any values that could have been defined during job creation or previous updates
    title = JobTitleNullable(missing=null)
    mode = JobExecuteModeEnum(missing=drop, deprecated=True)  # override without default 'auto'
    subscribers = JobExecuteSubscribers(missing=null)
    # all parameters that are not 'missing=drop' in original 'Execute' definition must be added to allow partial update
    inputs = ExecuteInputValues(missing=drop, description="Input values or references to be updated.")
    outputs = ExecuteOutputSpec(missing=drop, description="Output format and transmission mode to be updated.")


class PatchJobEndpoint(JobPath):
    summary = "Execution request parameters to be updated."
    description = (
        "Execution request parameters to be updated. "
        "If parameters are omitted, they will be left unmodified. "
        "If provided, parameters will override existing definitions integrally. "
        "Therefore, if only a partial update of certain nested elements in a mapping or list is desired, "
        "all elements under the corresponding parameters must be resubmitted entirely with the applied changes. "
        "In the case of certain parameters, equivalent definitions can cause conflicting definitions between "
        "headers and contents "
        f"(see for more details: {DOC_URL}/processes.html#execution-body and {DOC_URL}/processes.html#execution-mode). "
        "To verify the resulting parameterization of any pending job, consider using the `GET /jobs/{jobId}/inputs`."
    )
    header = JobExecuteHeaders()
    querystring = LocalProcessQuery()
    body = PatchJobBodySchema()


class PatchProcessJobEndpoint(JobPath, ProcessEndpoint):
    body = PatchJobBodySchema()


class PatchProviderJobEndpoint(PatchProcessJobEndpoint):
    header = RequestHeaders()


class PagingQueries(ExtendedMappingSchema):
    page = ExtendedSchemaNode(Integer(allow_string=True), missing=0, default=0, validator=Range(min=0))
    limit = ExtendedSchemaNode(Integer(allow_string=True), missing=10, default=10, validator=Range(min=1, max=1000),
                               schema=f"{OGC_API_PROC_PART1_PARAMETERS}/limit.yaml")


class GetJobsQueries(PagingQueries):
    # note:
    #   This schema is also used to generate any missing defaults during filter parameter handling.
    #   Items with default value are added if omitted, except 'default=null' which are removed after handling by alias.
    detail = ExtendedSchemaNode(QueryBoolean(), default=False, example=True, missing=drop,
                                description="Provide job details instead of IDs.")
    groups = JobGroupsCommaSeparated()
    min_duration = ExtendedSchemaNode(
        Integer(allow_string=True), name="minDuration", missing=drop, default=null, validator=Range(min=0),
        schema=f"{OGC_API_PROC_PART1_PARAMETERS}/minDuration.yaml",
        description="Minimal duration (seconds) between started time and current/finished time of jobs to find.")
    max_duration = ExtendedSchemaNode(
        Integer(allow_string=True), name="maxDuration", missing=drop, default=null, validator=Range(min=0),
        schema=f"{OGC_API_PROC_PART1_PARAMETERS}/maxDuration.yaml",
        description="Maximum duration (seconds) between started time and current/finished time of jobs to find.")
    datetime = DateTimeInterval(missing=drop, default=None)
    status = JobStatusSearchEnum(description="One of more comma-separated statuses to filter jobs.",
                                 missing=drop, default=None)
    processID = ProcessIdentifierTag(missing=drop, default=null,
                                     schema=f"{OGC_API_PROC_PART1_PARAMETERS}/processIdQueryParam.yaml",
                                     description="Alias to 'process' for OGC-API compliance.")
    process = ProcessIdentifierTag(missing=drop, default=None,
                                   description="Identifier and optional version tag of the process to filter search.")
    version = Version(
        missing=drop, default=None, example="0.1.0", description=(
            "Version of the 'process' or 'processID' query parameters. "
            "If version is provided, those query parameters should specify the ID without tag."
        )
    )
    service = AnyIdentifier(missing=drop, default=null, description="Alias to 'provider' for backward compatibility.")
    provider = AnyIdentifier(missing=drop, default=None, description="Identifier of service provider to filter search.")
    type = JobTypeEnum(missing=drop, default=null,
                       description="Filter jobs only to matching type (note: 'service' and 'provider' are aliases).")
    sort = JobSortEnum(missing=drop)
    access = JobAccess(missing=drop, default=None)
    tags = JobTagsCommaSeparated()


class GetProcessJobsQuery(LocalProcessQuery, GetJobsQueries):
    pass


class GetProviderJobsQueries(GetJobsQueries):  # ':version' not allowed for process ID in this case
    pass


class GetJobsEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()
    querystring = GetProcessJobsQuery()  # allowed version in this case since can be either local or remote processes


class GetProcessJobsEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = GetProcessJobsQuery()


class GetProviderJobsEndpoint(ProviderProcessPath):
    header = RequestHeaders()
    querystring = GetProviderJobsQueries()


class JobIdentifierList(ExtendedSequenceSchema):
    job_id = UUID(description="ID of a job to dismiss. Identifiers not matching any known job are ignored.")


class DeleteJobsBodySchema(ExtendedMappingSchema):
    jobs = JobIdentifierList()


class DeleteJobsEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()
    body = DeleteJobsBodySchema()


class DeleteProcessJobsEndpoint(DeleteJobsEndpoint, LocalProcessPath):
    querystring = LocalProcessQuery()


class DeleteProviderJobsEndpoint(DeleteJobsEndpoint, ProviderProcessPath):
    pass


class GetProcessJobQuery(LocalProcessQuery, GetJobQuery):
    pass


class GetProcessJobEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = GetProcessJobQuery()


class DeleteJobEndpoint(JobPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class DeleteProcessJobEndpoint(LocalProcessPath):
    header = RequestHeaders()


class DeleteProviderJobEndpoint(DeleteProcessJobEndpoint, ProviderProcessPath):
    pass


class BillsEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()


class BillEndpoint(BillPath):
    header = RequestHeaders()


class ProcessQuotesEndpoint(LocalProcessPath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class ProcessQuoteEndpoint(LocalProcessPath, QuotePath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()


class GetQuotesQueries(PagingQueries):
    process = AnyIdentifier(missing=None)
    sort = QuoteSortEnum(missing=drop)


class QuotesEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()
    querystring = GetQuotesQueries()


class QuoteEndpoint(QuotePath):
    header = RequestHeaders()


class PostProcessQuote(LocalProcessPath, QuotePath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()
    body = NoContent()


class PostQuote(QuotePath):
    header = RequestHeaders()
    body = NoContent()


class QuoteProcessParametersSchema(ExecuteInputOutputs):
    pass


class PostProcessQuoteRequestEndpoint(LocalProcessPath, QuotePath):
    header = RequestHeaders()
    querystring = LocalProcessQuery()
    body = QuoteProcessParametersSchema()


# ################################################################
# Provider Processes schemas
# ################################################################


class ProvidersQuerySchema(ExtendedMappingSchema):
    detail = ExtendedSchemaNode(
        QueryBoolean(), example=True, default=True, missing=drop,
        description="Return summary details about each provider, or simply their IDs."
    )
    check = ExtendedSchemaNode(
        QueryBoolean(),
        example=True, default=True, missing=drop,
        description="List only reachable providers, dropping unresponsive ones that cannot be checked for listing. "
                    "Otherwise, all registered providers are listed regardless of their availability. When requesting "
                    "details, less metadata will be provided since it will not be fetched from remote services."
    )
    ignore = ExtendedSchemaNode(
        QueryBoolean(), example=True, default=True, missing=drop,
        description="When listing providers with check of reachable remote service definitions, unresponsive response "
                    "or unprocessable contents will be silently ignored and dropped from full listing in the response. "
                    "Disabling this option will raise an error immediately instead of ignoring invalid services."
    )


class GetProviders(ExtendedMappingSchema):
    querystring = ProvidersQuerySchema()
    header = RequestHeaders()


class PostProvider(ExtendedMappingSchema):
    header = RequestHeaders()
    body = CreateProviderRequestBody()


class ProcessDetailQuery(ExtendedMappingSchema):
    detail = ExtendedSchemaNode(
        QueryBoolean(), example=True, default=True, missing=drop,
        description="Return summary details about each process, or simply their IDs."
    )


class ProcessLinksQuery(ExtendedMappingSchema):
    links = ExtendedSchemaNode(
        QueryBoolean(), example=True, default=True, missing=True,
        description="Return summary details with included links for each process."
    )


class ProcessRevisionsQuery(ExtendedMappingSchema):
    process = ProcessIdentifier(missing=drop, description=(
        "Process ID (excluding version) for which to filter results. "
        "When combined with 'revisions=true', allows listing of all reversions of a given process. "
        "If omitted when 'revisions=true', all revisions of every process ID will be returned. "
        "If used without 'revisions' query, list should include a single process as if summary was requested directly."
    ))
    revisions = ExtendedSchemaNode(
        QueryBoolean(), example=True, default=False, missing=drop, description=(
            "Return all revisions of processes, or simply their latest version. When returning all revisions, "
            "IDs will be replaced by '{processID}:{version}' tag representation to avoid duplicates."
        )
    )


class ProviderProcessesQuery(ProcessPagingQuery, ProcessDetailQuery, ProcessLinksQuery):
    pass


class ProviderProcessesEndpoint(ProviderPath):
    header = RequestHeaders()
    querystring = ProviderProcessesQuery()


class GetProviderProcess(ExtendedMappingSchema):
    header = RequestHeaders()


class PostProviderProcessJobRequest(ExtendedMappingSchema):
    """
    Launching a new process request definition.
    """
    header = ExecuteHeadersJSON()
    querystring = LaunchJobQuerystring()
    body = Execute()


# ################################################################
# Responses schemas
# ################################################################


class GenericHTMLResponse(ExtendedMappingSchema):
    """
    Generic HTML response.
    """
    header = HtmlHeader()
    body = ExtendedMappingSchema()

    def __new__(cls, *, name, description, **kwargs):  # pylint: disable=W0221
        # type: (Type[GenericHTMLResponse], *Any, str, str, **Any) -> GenericHTMLResponse
        """
        Generates a derived HTML response schema with direct forwarding of custom parameters to the body's schema.

        This strategy allows the quicker definition of schema variants without duplicating class definitions only
        providing alternate documentation parameters.

        .. note::
            Method ``__new__`` is used instead of ``__init__`` because some :mod:`cornice_swagger` operations will
            look explicitly for ``schema_node.__class__.__name__``. If using ``__init__``, the first instance would
            set the name value for all following instances instead of the intended reusable meta-schema class.
        """
        if not isinstance(name, str) or not re.match(r"^[A-Z][A-Z0-9_-]*$", name, re.I):
            raise ValueError(
                "New schema name must be provided to avoid invalid mixed use of $ref pointers. "
                f"Name '{name}' is invalid."
            )
        obj = super().__new__(cls)  # type: ExtendedSchemaNode
        obj.__init__(name=name, description=description)
        obj.__class__.__name__ = name
        obj.children = [
            child
            if child.name != "body" else
            ExtendedMappingSchema(name="body", **kwargs)
            for child in obj.children
        ]
        return obj

    def __deepcopy__(self, *args, **kwargs):
        # type: (*Any, *Any) -> GenericHTMLResponse
        return GenericHTMLResponse(name=self.name, description=self.description, children=self.children)


class OWSErrorCode(ExtendedSchemaNode):
    schema_type = String
    example = "InvalidParameterValue"
    description = "OWS error code or URI reference that identifies the problem type."


class OWSExceptionResponse(ExtendedMappingSchema):
    """
    Error content in XML format.
    """
    description = "OWS formatted exception."
    code = OWSErrorCode(example="NoSuchProcess")
    locator = ExtendedSchemaNode(String(), example="identifier",
                                 description="Indication of the element that caused the error.")
    message = ExtendedSchemaNode(String(), example="Invalid process ID.",
                                 description="Specific description of the error.")


class ErrorDetail(ExtendedMappingSchema):
    code = ExtendedSchemaNode(Integer(), description="HTTP status code.", example=400)
    status = ExtendedSchemaNode(String(), description="HTTP status detail.", example="400 Bad Request")


class ErrorSource(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(String(), description="Error name or description."),
        ErrorDetail(description="Detailed error representation.")
    ]


class ErrorCause(OneOfKeywordSchema):
    _one_of = [
        ExtendedSchemaNode(String(), description="Error message from exception or cause of failure."),
        PermissiveMappingSchema(description="Relevant error fields with details about the cause."),
    ]


class ErrorJsonResponseBodySchema(ExtendedMappingSchema):
    _schema = f"{OGC_API_COMMON_PART1_SCHEMAS}/exception.json"
    description = "JSON schema for exceptions based on RFC 7807"
    type = OWSErrorCode()  # only this is required
    title = ExtendedSchemaNode(String(), description="Short description of the error.", missing=drop)
    detail = ExtendedSchemaNode(String(), description="Detail about the error cause.", missing=drop)
    status = ExtendedSchemaNode(Integer(), description="Error status code.", example=500, missing=drop)
    cause = ErrorCause(missing=drop)
    value = ErrorCause(missing=drop)
    error = ErrorSource(missing=drop)
    instance = URI(missing=drop)
    exception = OWSExceptionResponse(missing=drop)


class ServerErrorBaseResponseSchema(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/ServerError.yaml"


class BadRequestResponseSchema(ServerErrorBaseResponseSchema):
    description = "Incorrectly formed request contents."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class ConflictRequestResponseSchema(ServerErrorBaseResponseSchema):
    description = "Conflict between the affected entity and another existing definition."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class UnprocessableEntityResponseSchema(ServerErrorBaseResponseSchema):
    description = "Wrong format or schema of given parameters."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class UnsupportedMediaTypeResponseSchema(ServerErrorBaseResponseSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/NotSupported.yaml"
    description = "Media-Type not supported for this request."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class MethodNotAllowedErrorResponseSchema(ServerErrorBaseResponseSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/NotAllowed.yaml"
    description = "HTTP method not allowed for requested path."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class NotAcceptableErrorResponseSchema(ServerErrorBaseResponseSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/NotSupported.yaml"
    description = "Requested media-types are not supported at the path."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class NotFoundResponseSchema(ServerErrorBaseResponseSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/NotFound.yaml"
    description = "Requested resource could not be found."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class ForbiddenProcessAccessResponseSchema(ServerErrorBaseResponseSchema):
    description = "Referenced process is not accessible."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class ForbiddenProviderAccessResponseSchema(ServerErrorBaseResponseSchema):
    description = "Referenced provider is not accessible."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class ForbiddenProviderLocalResponseSchema(ServerErrorBaseResponseSchema):
    description = (
        "Provider operation is not allowed on local-only Weaver instance. "
        f"Applies only when application configuration is not within: {WEAVER_CONFIG_REMOTE_LIST}"
    )
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class InternalServerErrorResponseSchema(ServerErrorBaseResponseSchema):
    description = "Unhandled internal server error."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class OkGetFrontpageResponse(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/LandingPage.yaml"
    header = ResponseHeaders()
    body = FrontpageSchema()


class OpenAPIResponseContentTypeHeader(ContentTypeHeader):
    example = ContentType.APP_OAS_JSON
    default = ContentType.APP_OAS_JSON
    validator = OneOf([ContentType.APP_OAS_JSON])


class OpenAPIResponseHeaders(ResponseHeaders):
    content_type = OpenAPIResponseContentTypeHeader()


class OkGetSwaggerJSONResponse(ExtendedMappingSchema):
    header = OpenAPIResponseHeaders()
    body = OpenAPISpecSchema(description="OpenAPI JSON schema of Weaver API.")
    examples = {
        "OpenAPI Schema": {
            "summary": "OpenAPI specification of this API.",
            "value": {"$ref": OpenAPISpecSchema._schema},
        }
    }


class OkGetSwaggerUIResponse(ExtendedMappingSchema):
    header = HtmlHeader()
    body = SwaggerUISpecSchema(description="Swagger UI of Weaver API.")


class OkGetRedocUIResponse(ExtendedMappingSchema):
    header = HtmlHeader()
    body = SwaggerUISpecSchema(description="Redoc UI of Weaver API.")


class OkGetVersionsResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = VersionsSchema()


class OkGetConformanceResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ConformanceSchema()


class OkGetProvidersListResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ProvidersBodySchema()


class OkGetProviderCapabilitiesSchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ProviderCapabilitiesSchema()


class NoContentDeleteProviderSchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = NoContent()


class NotImplementedDeleteProviderResponse(ExtendedMappingSchema):
    description = "Provider removal not supported using referenced storage."


class OkGetProviderProcessesSchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ProviderProcessesSchema()


class GetProcessesQuery(ProcessPagingQuery, ProcessDetailQuery, ProcessLinksQuery, ProcessRevisionsQuery):
    providers = ExtendedSchemaNode(
        QueryBoolean(), example=True, default=False, missing=drop,
        description="List local processes as well as all sub-processes of all registered providers. "
                    "Paging and sorting query parameters are unavailable when providers are requested since lists are "
                    "populated dynamically and cannot ensure consistent process lists per page across providers. "
                    f"Applicable only for Weaver configurations {WEAVER_CONFIG_REMOTE_LIST}, ignored otherwise."
    )
    ignore = ExtendedSchemaNode(
        QueryBoolean(), example=True, default=True, missing=drop,
        description="Only when listing provider processes, any unreachable remote service definitions "
                    "or unprocessable contents will be silently ignored and dropped from full listing in the response. "
                    "Disabling this option will raise an error immediately instead of ignoring invalid providers."
    )


class GetProcessesEndpoint(ExtendedMappingSchema):
    querystring = GetProcessesQuery()


class ProviderProcessesListing(ProcessCollection):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/processList.yaml"
    _sort_first = ["id", "processes"]
    id = ProviderNameSchema()


class ProviderProcessesList(ExtendedSequenceSchema):
    item = ProviderProcessesListing(description="Processes offered by the identified remote provider.")


class ProvidersProcessesCollection(ExtendedMappingSchema):
    providers = ProviderProcessesList(missing=drop)


class ProcessListingLinks(ExtendedMappingSchema):
    links = LinkList(missing=drop)


class ProcessListingMetadata(PagingBodySchema):
    description = "Metadata relative to the listed processes."
    total = ExtendedSchemaNode(Integer(), description="Total number of local processes, or also including all "
                                                      "remote processes across providers if requested.")


class ProcessesListing(ProcessCollection, ProcessListingLinks):
    _schema = f"{OGC_API_PROC_PART1_SCHEMAS}/processList.yaml"
    _sort_first = PROCESSES_LISTING_FIELD_FIRST
    _sort_after = PROCESSES_LISTING_FIELD_AFTER


class MultiProcessesListing(DescriptionSchema, ProcessesListing, ProvidersProcessesCollection, ProcessListingMetadata):
    _schema_meta_include = True


class OkGetProcessesListResponse(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/ProcessList.yaml"
    description = "Listing of available processes successful."
    header = ResponseHeaders()
    querystring = FormatQuery()
    body = MultiProcessesListing()


class OkPostProcessDeployBodySchema(ExtendedMappingSchema):
    description = ExtendedSchemaNode(String(), description="Detail about the operation.")
    deploymentDone = ExtendedSchemaNode(Boolean(), default=False, example=True,
                                        description="Indicates if the process was successfully deployed.")
    processSummary = ProcessSummary(missing=drop, description="Deployed process summary if successful.")
    failureReason = ExtendedSchemaNode(String(), missing=drop,
                                       description="Description of deploy failure if applicable.")


class OkPostProcessesResponse(ExtendedMappingSchema):
    description = "Process successfully deployed."
    header = ResponseHeaders()
    body = OkPostProcessDeployBodySchema()


class OkPatchProcessUpdatedBodySchema(ExtendedMappingSchema):
    description = ExtendedSchemaNode(String(), description="Detail about the operation.")
    processSummary = ProcessSummary(missing=drop, description="Deployed process summary if successful.")


class OkPatchProcessResponse(ExtendedMappingSchema):
    description = "Process successfully updated."
    header = ResponseHeaders()
    body = OkPatchProcessUpdatedBodySchema()


class BadRequestGetProcessInfoResponse(ExtendedMappingSchema):
    description = "Missing process identifier."
    body = NoContent()


class NotFoundProcessResponse(NotFoundResponseSchema):
    description = "Process with specified reference identifier does not exist."
    examples = {
        "ProcessNotFound": {
            "summary": "Example response when specified process reference cannot be found.",
            "value": EXAMPLES["local_process_not_found.json"]
        }
    }
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class OkGetProcessInfoResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    querystring = FormatQuery()
    body = ProcessDescription()


class OkGetProcessPackageSchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    querystring = FormatQuery()
    body = CWL()


class OkGetProcessPayloadSchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = NoContent()


class ProcessVisibilityResponseBodySchema(ExtendedMappingSchema):
    value = VisibilityValue()


class OkGetProcessVisibilitySchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ProcessVisibilityResponseBodySchema()


class OkPutProcessVisibilitySchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ProcessVisibilityResponseBodySchema()


class ForbiddenVisibilityUpdateResponseSchema(ExtendedMappingSchema):
    description = "Visibility value modification not allowed."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class OkDeleteProcessUndeployBodySchema(ExtendedMappingSchema):
    deploymentDone = ExtendedSchemaNode(Boolean(), default=False, example=True,
                                        description="Indicates if the process was successfully undeployed.")
    identifier = ExtendedSchemaNode(String(), example="workflow")
    failureReason = ExtendedSchemaNode(String(), missing=drop,
                                       description="Description of undeploy failure if applicable.")


class OkDeleteProcessResponse(ExtendedMappingSchema):
    description = "Process successfully undeployed."
    header = ResponseHeaders()
    body = OkDeleteProcessUndeployBodySchema()


class OkGetProviderProcessDescriptionResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ProcessDescription()


class CreatedPostProvider(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = ProviderSummarySchema()


class NotImplementedPostProviderResponse(ExtendedMappingSchema):
    description = "Provider registration not supported using specified definition."


class PreferenceAppliedHeader(ExtendedSchemaNode):
    description = "Applied preferences from submitted 'Prefer' header after validation."
    name = "Preference-Applied"
    schema_type = String
    example = "wait=10, respond-async"


class LocationHeader(URL):
    name = "Location"


class CreatedJobLocationHeader(ResponseHeaders):
    location = LocationHeader(description="Status monitoring location of the job execution.")
    prefer_applied = PreferenceAppliedHeader(missing=drop)


class CreatedLaunchJobResponse(ExtendedMappingSchema):
    description = (
        "Job successfully submitted. "
        "Execution should begin when resources are available or when triggered, according to requested execution mode."
    )
    examples = {
        "JobAccepted": {
            "summary": "Job accepted for execution asynchronously.",
            "value": EXAMPLES["job_status_accepted.json"]
        },
        "JobCreated": {
            "summary": "Job created for later execution by trigger.",
            "value": EXAMPLES["job_status_created.json"]
        }
    }
    header = CreatedJobLocationHeader()
    body = CreatedJobStatusSchema()


class CompletedJobLocationHeader(ResponseHeaders):
    location = LocationHeader(description="Status location of the completed job execution.")
    prefer_applied = PreferenceAppliedHeader(missing=drop)


class CompletedJobStatusSchema(DescriptionSchema, JobStatusInfo):
    pass


class CompletedJobResponse(ExtendedMappingSchema):
    description = "Job submitted and completed execution synchronously."
    header = CompletedJobLocationHeader()
    body = CompletedJobStatusSchema()


class FailedSyncJobResponse(CompletedJobResponse):
    description = "Job submitted and failed synchronous execution. See server logs for more details."


class InvalidJobParametersResponse(ExtendedMappingSchema):
    description = "Job parameters failed validation."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class OkDeleteProcessJobResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = DismissedJobSchema()


class OkGetQueriedJobsResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    querystring = FormatQuery()
    body = GetQueriedJobsSchema()


class BatchDismissJobsBodySchema(DescriptionSchema):
    jobs = JobIdentifierList(description="Confirmation of jobs that have been dismissed.")


class OkBatchDismissJobsResponseSchema(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = BatchDismissJobsBodySchema()


class OkDismissJobResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = DismissedJobSchema()


class NoContentJobUpdatedResponse(ExtendedMappingSchema):
    description = "Job detail updated with provided parameters."
    header = ResponseHeaders()
    body = NoContent()


class OkGetJobStatusResponse(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/Status.yaml"
    header = ResponseHeaders()
    body = JobStatusInfo()


class InvalidJobResponseSchema(ExtendedMappingSchema):
    description = "Job reference is not a valid UUID."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class NotFoundJobResponseSchema(NotFoundResponseSchema):
    description = "Job reference UUID cannot be found."
    examples = {
        "JobNotFound": {
            "summary": "Example response when specified job reference cannot be found.",
            "value": EXAMPLES["job_not_found.json"]
        }
    }
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class GoneJobResponseSchema(ExtendedMappingSchema):
    description = "Job reference UUID cannot be dismissed again or its result artifacts were removed."
    examples = {
        "JobDismissed": {
            "summary": "Example response when specified job reference was already dismissed.",
            "value": EXAMPLES["job_dismissed_error.json"]
        }
    }
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class JobProvAcceptHeader(AcceptHeader):
    validator = OneOf(ProvenanceFormat.media_types())


class JobProvRequestHeaders(RequestHeaders):
    accept = JobProvAcceptHeader()


class JobProvEndpoint(JobPath):
    header = JobProvRequestHeaders()
    querystring = FormatQuery()


class ProcessJobProvEndpoint(JobProvEndpoint, LocalProcessPath):
    pass


class ProviderJobProvEndpoint(JobProvEndpoint, ProviderProcessPath):
    pass


class OkGetJobProvResponseHeaders(ResponseHeaders):
    content_type = JobProvAcceptHeader()


class OkGetJobProvResponse(ExtendedMappingSchema):
    description = "Job provenance details in the requested format."
    header = OkGetJobProvResponseHeaders()


class JobProvMetadataRequestHeaders(ExtendedMappingSchema):
    accept = ResponseContentTypePlainTextHeader()


class JobProvMetadataEndpoint(JobPath):
    header = JobProvMetadataRequestHeaders()


class ProcessJobProvMetadataEndpoint(JobProvMetadataEndpoint, LocalProcessPath):
    pass


class ProviderJobProvMetadataEndpoint(JobProvMetadataEndpoint, ProviderProcessPath):
    pass


class JobProvMetadataResponseBody(ExtendedSchemaNode):
    schema_type = String
    description = "Multipart file contents for upload to the vault."


class OkGetJobProvMetadataResponse(ExtendedMappingSchema):
    description = "Job execution provenance metadata relative to the contextual request path."
    header = ResponsePlainTextHeaders()
    body = JobProvMetadataResponseBody()


class NotFoundJobProvResponseSchema(NotFoundResponseSchema):
    description = (
        "Job reference UUID cannot be found, or a specified provenance "
        "run UUID cannot be retrieved from the Workflow execution steps."
    )
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class GoneJobProvResponseSchema(ExtendedMappingSchema):
    description = (
        "Job reference contents have been removed or does not contain PROV metadata. "
        "This could be because the job was created before provenance support was enabled, "
        "or because a job retention period deleted the contents."
    )
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class OkGetJobInputsResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = JobInputsBody()


class OkGetJobOutputsResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = JobOutputsBody()


class RedirectResultResponse(ExtendedMappingSchema):
    header = RedirectHeaders()


class OkGetJobResultsResponse(ExtendedMappingSchema):
    _schema = f"{OGC_API_PROC_PART1_RESPONSES}/Results.yaml"
    header = ResponseHeaders()
    body = ResultsBody()


class NoContentJobResultsHeaders(NoContent):
    content_length = ContentLengthHeader(example="0")
    link = LinkHeader(description=(
        "Link to a result requested by reference output transmission. "
        "Link relation indicates the result ID. "
        "Additional parameters indicate expected content-type of the resource. "
        "Literal data requested by reference are returned with contents dumped to plain text file."
    ))


class NoContentJobResultsResponse(ExtendedMappingSchema):
    description = "Job completed execution synchronously with results returned in Link headers."
    header = NoContentJobResultsHeaders()
    body = NoContent(default="")


class CreatedQuoteExecuteResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = CreatedQuotedJobStatusSchema()


class CreatedQuoteResponse(ExtendedMappingSchema):
    description = "Quote successfully obtained for process execution definition."
    header = ResponseHeaders()
    body = QuoteSchema()


class AcceptedQuoteResponse(ExtendedMappingSchema):
    summary = "Quote successfully submitted."
    description = (
        "Quote successfully submitted for evaluating process execution definition. "
        "Complete details will be available once evaluation has completed."
    )
    header = ResponseHeaders()
    body = PartialQuoteSchema()


class QuotePaymentRequiredResponse(ServerErrorBaseResponseSchema):
    description = "Quoted process execution refused due to missing payment."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class OkGetQuoteInfoResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = QuoteSchema()


class NotFoundQuoteResponse(NotFoundResponseSchema):
    description = "Quote with specified reference identifier does not exist."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class OkGetQuoteListResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = QuotationListSchema()


class OkGetEstimatorResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = QuoteEstimatorSchema()


class OkPutEstimatorResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = DescriptionSchema()


class OkDeleteEstimatorResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = DescriptionSchema()


class OkGetBillDetailResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = BillSchema()


class OkGetBillListResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = BillListSchema()


class OkGetJobExceptionsResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = JobExceptionsSchema()


class OkGetJobLogsResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = JobLogsSchema()


class OkGetJobStatsResponse(ExtendedMappingSchema):
    header = ResponseHeaders()
    body = JobStatisticsSchema()


class VaultFileID(UUID):
    description = "Vault file identifier."
    example = "78977deb-28af-46f3-876b-cdd272742678"


class VaultAccessToken(UUID):
    description = "Vault file access token."
    example = "30d889cfb7ae3a63229a8de5f91abc1ef5966bb664972f234a4db9d28f8148e0e"  # nosec


class VaultEndpoint(ExtendedMappingSchema):
    header = RequestHeaders()


class VaultUploadBody(ExtendedSchemaNode):
    schema_type = String
    description = "Multipart file contents for upload to the vault."
    examples = {
        ContentType.MULTIPART_FORM: {
            "summary": "Upload JSON file to vault as multipart content.",
            "value": EXAMPLES["vault_file_upload.txt"],
        }
    }


class VaultUploadEndpoint(ExtendedMappingSchema):
    header = FileUploadHeaders()
    body = VaultUploadBody()


class VaultFileUploadedBodySchema(ExtendedMappingSchema):
    access_token = AccessToken()
    file_id = VaultFileID()
    file_href = VaultReference()


class VaultFileUploadedHeaders(ResponseHeaders):
    location = URL(name="Location", description="File download location.",
                   example=f"https://localhost:4002{vault_file_service.path.format(file_id=VaultFileID.example)}")


class OkVaultFileUploadedResponse(ExtendedMappingSchema):
    description = "File successfully uploaded to vault."
    header = VaultFileUploadedHeaders()
    body = VaultFileUploadedBodySchema()


class BadRequestVaultFileUploadResponse(ExtendedMappingSchema):
    description = "Missing or incorrectly formed file contents."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class UnprocessableEntityVaultFileUploadResponse(ExtendedMappingSchema):
    description = (
        "Invalid filename refused for upload. "
        "Filename should include only alphanumeric, underscore, dash, and dot characters. "
        "Filename should include both the base name and the desired file extension."
    )
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class XAuthVaultFileHeader(ExtendedSchemaNode):
    summary = "Authorization header with token for Vault file access."
    description = (
        "For accessing a single file from the Vault, such as to obtain file metadata, requests can simply provide "
        "the 'token {access-token}' portion in the header without additional parameters. If multiple files require "
        "access such as during an Execute request, all applicable tokens should be provided using a comma separated "
        "list of access tokens, each with their indented input ID and array index if applicable "
        f"(see {DOC_URL}/processes.html#file-vault-inputs for more details)."
    )
    name = "X-Auth-Vault"
    example = "token {access-token}[; id={vault-id}]"
    schema_type = String


class VaultFileRequestHeaders(ExtendedMappingSchema):
    access_token = XAuthVaultFileHeader()


class VaultFileEndpoint(VaultEndpoint):
    header = VaultFileRequestHeaders()
    file_id = VaultFileID()


class OkVaultFileDetailResponse(ExtendedMappingSchema):
    header = FileResponseHeaders()
    body = NoContent(default="")


class OkVaultFileDownloadResponse(OkVaultFileDetailResponse):
    pass


class BadRequestVaultFileAccessResponse(ExtendedMappingSchema):
    description = "Invalid file vault reference."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class ForbiddenVaultFileDownloadResponse(ExtendedMappingSchema):
    description = "Forbidden access to vault file. Invalid authorization from provided token."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


class GoneVaultFileDownloadResponse(ExtendedMappingSchema):
    description = "Vault File resource corresponding to specified ID is not available anymore."
    header = ResponseHeaders()
    body = ErrorJsonResponseBodySchema()


get_api_frontpage_responses = {
    "200": OkGetFrontpageResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_openapi_json_responses = {
    "200": OkGetSwaggerJSONResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_api_swagger_ui_responses = {
    "200": OkGetSwaggerUIResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_api_redoc_ui_responses = {
    "200": OkGetRedocUIResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_api_versions_responses = {
    "200": OkGetVersionsResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_api_conformance_responses = {
    "200": OkGetConformanceResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_processes_responses = {
    "200": OkGetProcessesListResponse(examples={
        "ProcessesListing": {
            "summary": "Listing of identifiers of local processes registered in Weaver.",
            "value": EXAMPLES["local_process_listing.json"],
        },
        "ProcessesDetails": {
            "summary": "Detailed definitions of local processes registered in Weaver.",
            "value": EXAMPLES["local_process_listing.json"],
        },
        "ProvidersProcessesListing": {
            "summary": "List of identifiers combining all local and remote processes known by Weaver.",
            "value": EXAMPLES["providers_processes_listing.json"],
        },
        "ProvidersProcessesDetails": {
            "summary": "Detailed definitions Combining all local and remote processes known by Weaver.",
            "value": EXAMPLES["providers_processes_listing.json"],
        }
    }),
    "400": BadRequestResponseSchema(description="Error in case of invalid listing query parameters."),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
post_processes_responses = {
    "201": OkPostProcessesResponse(examples={
        "ProcessDeployed": {
            "summary": "Process successfully deployed.",
            "value": EXAMPLES["local_process_deploy_success.json"],
        }
    }),
    "400": BadRequestResponseSchema(description="Unable to parse process definition"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "409": ConflictRequestResponseSchema(description="Process with same ID already exists."),
    "415": UnsupportedMediaTypeResponseSchema(description="Unsupported Media-Type for process deployment."),
    "422": UnprocessableEntityResponseSchema(description="Invalid schema for process definition."),
    "500": InternalServerErrorResponseSchema(),
}
put_process_responses = copy(post_processes_responses)
put_process_responses.update({
    "404": NotFoundProcessResponse(description="Process to update could not be found."),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "409": ConflictRequestResponseSchema(description="Process with same ID or version already exists."),
})
patch_process_responses = {
    "200": OkPatchProcessResponse(),
    "400": BadRequestGetProcessInfoResponse(description="Unable to parse process definition"),
    "404": NotFoundProcessResponse(description="Process to update could not be found."),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "409": ConflictRequestResponseSchema(description="Process with same ID or version already exists."),
    "422": UnprocessableEntityResponseSchema(description="Invalid schema for process definition."),
    "500": InternalServerErrorResponseSchema(),
}
get_process_responses = {
    "200": OkGetProcessInfoResponse(description="success", examples={
        "ProcessDescriptionSchemaOGC": {
            "summary": "Description of a local process registered in Weaver (OGC Schema) "
                       "with fields on top-level and using inputs/outputs as mapping with keys as IDs.",
            "value": EXAMPLES["local_process_description_ogc_api.json"],
        },
        "ProcessDescriptionSchemaOld": {
            "summary": "Description of a local process registered in Weaver (Old Schema) "
                       "with fields nested under a process section and using inputs/outputs listed with IDs.",
            "value": EXAMPLES["local_process_description.json"],
        },
        "ProcessDescriptionSchemaWPS": {
            "Summary": "Description of a local process registered in Weaver (WPS Schema) when requesting XML format.",
            "value": EXAMPLES["wps_describeprocess_response.xml"],
        }
    }),
    "400": BadRequestGetProcessInfoResponse(),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_process_package_responses = {
    "200": OkGetProcessPackageSchema(description="success", examples={
        "PackageCWL": {
            "summary": "CWL Application Package definition of the local process.",
            "value": EXAMPLES["local_process_package.json"],
        }
    }),
    "403": ForbiddenProcessAccessResponseSchema(),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_process_payload_responses = {
    "200": OkGetProcessPayloadSchema(description="success", examples={
        "Payload": {
            "summary": "Payload employed during process deployment and registration.",
            "value": EXAMPLES["local_process_payload.json"],
        }
    }),
    "403": ForbiddenProcessAccessResponseSchema(),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_process_visibility_responses = {
    "200": OkGetProcessVisibilitySchema(description="success"),
    "403": ForbiddenProcessAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
put_process_visibility_responses = {
    "200": OkPutProcessVisibilitySchema(description="success"),
    "403": ForbiddenVisibilityUpdateResponseSchema(),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
delete_process_responses = {
    "200": OkDeleteProcessResponse(examples={
        "ProcessUndeployed": {
            "summary": "Process successfully undeployed.",
            "value": EXAMPLES["local_process_undeploy_success.json"],
        }
    }),
    "403": ForbiddenProcessAccessResponseSchema(),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_providers_list_responses = {
    "200": OkGetProvidersListResponse(description="success", examples={
        "ProviderList": {
            "summary": "Listing of registered remote providers.",
            "value": EXAMPLES["provider_listing.json"],
        },
        "ProviderNames": {
            "summary": "Listing of registered providers names without validation.",
            "value": EXAMPLES["provider_names.json"],
        }
    }),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_responses = {
    "200": OkGetProviderCapabilitiesSchema(description="success", examples={
        "ProviderDescription": {
            "summary": "Description of a registered remote WPS provider.",
            "value": EXAMPLES["provider_description.json"],
        }
    }),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
delete_provider_responses = {
    "204": NoContentDeleteProviderSchema(description="success"),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
    "501": NotImplementedDeleteProviderResponse(),
}
get_provider_processes_responses = {
    "200": OkGetProviderProcessesSchema(description="success"),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_process_responses = {
    "200": OkGetProviderProcessDescriptionResponse(description="success", examples={
        "ProviderProcessWPS": {
            "summary": "Description of a remote WPS provider process converted to OGC-API Processes format.",
            "value": EXAMPLES["provider_process_description.json"]
        }
    }),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_process_package_responses = copy(get_process_package_responses)
get_provider_process_package_responses.update({
    "403": ForbiddenProviderAccessResponseSchema(),
})
post_provider_responses = {
    "201": CreatedPostProvider(description="success"),
    "400": ExtendedMappingSchema(description=OWSMissingParameterValue.description),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
    "501": NotImplementedPostProviderResponse(),
}
post_provider_process_job_responses = {
    "200": CompletedJobResponse(),
    "201": CreatedLaunchJobResponse(),
    "204": NoContentJobResultsResponse(),
    "400": InvalidJobParametersResponse(),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "415": UnsupportedMediaTypeResponseSchema(),
    "422": UnprocessableEntityResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
post_process_jobs_responses = {
    "200": CompletedJobResponse(),
    "201": CreatedLaunchJobResponse(),
    "204": NoContentJobResultsResponse(),
    "400": InvalidJobParametersResponse(),
    "403": ForbiddenProviderAccessResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "415": UnsupportedMediaTypeResponseSchema(),
    "422": UnprocessableEntityResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
post_jobs_responses = copy(post_process_jobs_responses)
post_job_results_responses = copy(post_process_jobs_responses)
post_job_results_responses.pop("201")   # job already created, therefore invalid
post_job_results_responses.update({
    "202": CreatedLaunchJobResponse(),  # alternate to '201' for async case since job already exists
})
get_all_jobs_responses = {
    "200": OkGetQueriedJobsResponse(description="success", examples={
        "JobListing": {
            "summary": "Job ID listing with default queries.",
            "value": EXAMPLES["jobs_listing.json"]
        }
    }),
    "400": BadRequestResponseSchema(description="Error in case of invalid search query parameters."),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "422": UnprocessableEntityResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
delete_jobs_responses = {
    "200": OkBatchDismissJobsResponseSchema(description="success"),
    "400": BadRequestResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "422": UnprocessableEntityResponseSchema(),
}
get_provider_all_jobs_responses = copy(get_all_jobs_responses)
get_provider_all_jobs_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_single_job_status_responses = {
    "200": OkGetJobStatusResponse(description="success", examples={
        "JobStatusSuccess": {
            "summary": "Successful job status response.",
            "value": EXAMPLES["job_status_success.json"],
        },
        "JobStatusFailure": {
            "summary": "Failed job status response.",
            "value": EXAMPLES["job_status_failed.json"],
        }
    }),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_single_job_status_responses = copy(get_single_job_status_responses)
get_provider_single_job_status_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
patch_job_responses = {
    "204": NoContentJobUpdatedResponse(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "415": UnsupportedMediaTypeResponseSchema(),
    "422": UnprocessableEntityResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
patch_process_job_responses = copy(patch_job_responses)
patch_provider_job_responses = copy(patch_job_responses)
delete_job_responses = {
    "200": OkDismissJobResponse(description="success", examples={
        "JobDismissedSuccess": {
            "summary": "Successful job dismissed response.",
            "value": EXAMPLES["job_dismissed_success.json"]
        },
    }),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
delete_provider_job_responses = copy(delete_job_responses)
delete_provider_job_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_job_inputs_responses = {
    "200": OkGetJobInputsResponse(description="success", examples={
        "JobInputs": {
            "summary": "Submitted job input values at for process execution.",
            "value": EXAMPLES["job_inputs.json"],
        }
    }),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_inputs_responses = copy(get_job_inputs_responses)
get_provider_inputs_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_job_outputs_responses = {
    "200": OkGetJobOutputsResponse(description="success", examples={
        "JobOutputs": {
            "summary": "Obtained job outputs values following process execution.",
            "value": EXAMPLES["job_outputs.json"],
        }
    }),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_outputs_responses = copy(get_job_outputs_responses)
get_provider_outputs_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_result_redirect_responses = {
    "308": RedirectResultResponse(description="Redirects '/result' (without 's') to corresponding '/results' path."),
}
get_job_results_responses = {
    "200": OkGetJobResultsResponse(description="success", examples={
        "JobResults": {
            "summary": "Obtained job results.",
            "value": EXAMPLES["job_results.json"],
        }
    }),
    "204": NoContentJobResultsResponse(description="success"),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_results_responses = copy(get_job_results_responses)
get_provider_results_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_job_exceptions_responses = {
    "200": OkGetJobExceptionsResponse(description="success", examples={
        "JobExceptions": {
            "summary": "Job exceptions that occurred during failing process execution.",
            "value": EXAMPLES["job_exceptions.json"],
        }
    }),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_exceptions_responses = copy(get_job_exceptions_responses)
get_provider_exceptions_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_job_logs_responses = {
    "200": OkGetJobLogsResponse(description="success", examples={
        "JobLogs": {
            "summary": "Job logs registered and captured throughout process execution.",
            "value": EXAMPLES["job_logs.json"],
        }
    }),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_logs_responses = copy(get_job_logs_responses)
get_provider_logs_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_job_stats_responses = {
    "200": OkGetJobStatsResponse(description="success", examples={
        "JobStatistics": {
            "summary": "Job statistics collected following process execution.",
            "value": EXAMPLES["job_statistics.json"],
        }
    }),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobResponseSchema(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_provider_stats_responses = copy(get_job_stats_responses)
get_provider_stats_responses.update({
    "403": ForbiddenProviderLocalResponseSchema(),
})
get_job_prov_responses = {
    "200": OkGetJobProvResponse(
        description="Successful job PROV details.",
        examples={
            "PROV-JSON": {
                "summary": "Provenance details returned in PROV-JSON format.",
                "value": EXAMPLES["job_prov.json"],
            },
            "PROV-N": {
                "summary": "Provenance details returned in PROV-N format.",
                "value": EXAMPLES["job_prov.txt"],
            },
            "PROV-XML": {
                "summary": "Provenance details returned in PROV-XML format.",
                "value": EXAMPLES["job_prov.xml"],
            }
        }
    ),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobProvResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobProvResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_job_prov_metadata_responses = {
    "200": OkGetJobProvMetadataResponse(
        description="Successful job PROV metadata retrieval.",
        examples={
            "PROV run": {
                "summary": "Provenance metadata of the run execution.",
                "value": EXAMPLES["job_prov_run.txt"],
            },
            "PROV who": {
                "summary": "Provenance metadata of who ran the job.",
                "value": EXAMPLES["job_prov_who.txt"],
            },
            "PROV info": {
                "summary": "Provenance metadata about the Research Object packaging information.",
                "value": EXAMPLES["job_prov_info.txt"],
            }
        }
    ),
    "400": InvalidJobResponseSchema(),
    "404": NotFoundJobProvResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneJobProvResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_quote_list_responses = {
    "200": OkGetQuoteListResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_quote_responses = {
    "200": OkGetQuoteInfoResponse(description="success"),
    "404": NotFoundQuoteResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
post_quotes_responses = {
    "201": CreatedQuoteResponse(),
    "202": AcceptedQuoteResponse(),
    "400": InvalidJobParametersResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
post_quote_responses = {
    "201": CreatedQuoteExecuteResponse(description="success"),
    "400": InvalidJobParametersResponse(),
    "402": QuotePaymentRequiredResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_process_quote_estimator_responses = {
    "200": OkGetEstimatorResponse(description="success"),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
put_process_quote_estimator_responses = {
    "200": OkPutEstimatorResponse(description="success"),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
delete_process_quote_estimator_responses = {
    "204": OkDeleteEstimatorResponse(description="success"),
    "404": NotFoundProcessResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_bill_list_responses = {
    "200": OkGetBillListResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
get_bill_responses = {
    "200": OkGetBillDetailResponse(description="success"),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "500": InternalServerErrorResponseSchema(),
}
post_vault_responses = {
    "200": OkVaultFileUploadedResponse(description="success", examples={
        "VaultFileUploaded": {
            "summary": "File successfully uploaded to vault.",
            "value": EXAMPLES["vault_file_uploaded.json"],
        }
    }),
    "400": BadRequestVaultFileUploadResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "422": UnprocessableEntityVaultFileUploadResponse(),
    "500": InternalServerErrorResponseSchema(),
}
head_vault_file_responses = {
    "200": OkVaultFileDetailResponse(description="success", examples={
        "VaultFileDetails": {
            "summary": "Obtain vault file metadata.",
            "value": EXAMPLES["vault_file_head.json"],
        }
    }),
    "400": BadRequestVaultFileAccessResponse(),
    "403": ForbiddenVaultFileDownloadResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneVaultFileDownloadResponse(),
    "500": InternalServerErrorResponseSchema(),
}
get_vault_file_responses = {
    "200": OkVaultFileDownloadResponse(description="success"),
    "400": BadRequestVaultFileAccessResponse(),
    "403": ForbiddenVaultFileDownloadResponse(),
    "405": MethodNotAllowedErrorResponseSchema(),
    "406": NotAcceptableErrorResponseSchema(),
    "410": GoneVaultFileDownloadResponse(),
    "500": InternalServerErrorResponseSchema(),
}
wps_responses = {
    "200": OkWPSResponse(examples={
        "GetCapabilities": {
            "summary": "GetCapabilities example response.",
            "value": EXAMPLES["wps_getcapabilities_response.xml"]
        },
        "DescribeProcess": {
            "summary": "DescribeProcess example response.",
            "value": EXAMPLES["wps_describeprocess_response.xml"]
        },
        "ExecuteSuccess": {
            "summary": "Successful process execute example response.",
            "value": EXAMPLES["wps_execute_response.xml"]
        },
        "ExecuteFailed": {
            "summary": "Failed process execute example response.",
            "value": EXAMPLES["wps_execute_failed_response.xml"]
        }
    }),
    "400": ErrorWPSResponse(examples={
        "MissingParameterError": {
            "summary": "Error report in case of missing request parameter.",
            "value": EXAMPLES["wps_missing_parameter_response.xml"],
        },
        "AccessForbiddenError": {
            "summary": "Error report in case of forbidden access to the service.",
            "value": EXAMPLES["wps_access_forbidden_response.xml"],
        }
    }),
    "405": ErrorWPSResponse(),
    "406": ErrorWPSResponse(),
    "500": ErrorWPSResponse(),
}


#################################################################
# Utility methods
#################################################################


def derive_responses(responses, response_schema, status_code=200):
    # type: (Dict[str, ExtendedSchemaNode], ExtendedSchemaNode, int) -> Dict[str, ExtendedSchemaNode]
    """
    Generates a derived definition of the responses mapping using the specified schema and status code.

    :param responses: Mapping of status codes to response schemas.
    :param response_schema: Desired response schema to apply.
    :param status_code: Desired status code for which to apply the response schema.
    :return: Derived responses mapping.
    """
    responses = copy(responses)
    responses[str(status_code)] = response_schema
    return responses


def datetime_interval_parser(datetime_interval):
    # type: (str) -> DatetimeIntervalType
    """
    This function parses a given datetime or interval into a dictionary that will be easy for database process.
    """
    parsed_datetime = {}

    if datetime_interval.startswith(DATETIME_INTERVAL_OPEN_START_SYMBOL):
        datetime_interval = datetime_interval.replace(DATETIME_INTERVAL_OPEN_START_SYMBOL, "")
        parsed_datetime["before"] = date_parser.parse(datetime_interval)

    elif datetime_interval.endswith(DATETIME_INTERVAL_OPEN_END_SYMBOL):
        datetime_interval = datetime_interval.replace(DATETIME_INTERVAL_OPEN_END_SYMBOL, "")
        parsed_datetime["after"] = date_parser.parse(datetime_interval)

    elif DATETIME_INTERVAL_CLOSED_SYMBOL in datetime_interval:
        datetime_interval = datetime_interval.split(DATETIME_INTERVAL_CLOSED_SYMBOL)
        parsed_datetime["after"] = date_parser.parse(datetime_interval[0])
        parsed_datetime["before"] = date_parser.parse(datetime_interval[-1])
    else:
        parsed_datetime["match"] = date_parser.parse(datetime_interval)

    return parsed_datetime


def validate_node_schema(schema_node, cstruct):
    # type: (ExtendedMappingSchema, JSON) -> JSON
    """
    Validate a schema node defined against a reference schema within :data:`WEAVER_SCHEMA_DIR`.

    If the reference contains an anchor (e.g.: ``#/definitions/Def``), the sub-schema of that
    reference will be used for validation against the data structure.
    """
    schema_node.deserialize(cstruct)
    schema_file = schema_node._schema.replace(WEAVER_SCHEMA_URL, WEAVER_SCHEMA_DIR)
    schema_path = []
    if "#" in schema_file:
        schema_file, schema_ref = schema_file.split("#", 1)
        schema_path = [ref for ref in schema_ref.split("/") if ref]
    schema_base = schema = load_file(schema_file)
    if schema_path:
        for part in schema_path:
            schema = schema[part]

    # ensure local schema can find relative $ref, since the provided reference can be a sub-schema (with "#/...")
    scheme_uri = f"file://{schema_file}" if schema_file.startswith("/") else schema_file
    validator = jsonschema.validators.validator_for(schema_base)
    validator.resolver = jsonschema.RefResolver(base_uri=scheme_uri, referrer=schema_base)
    validator(schema_base).validate(cstruct, schema)  # raises if invalid
    return cstruct
