﻿#region Copyright © 2015 Couchcoding

// File:    ListViewEx.cs
// Package: GuiLibrary
// Project: Logbert
// 
// The MIT License (MIT)
// 
// Copyright (c) 2015 Couchcoding
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#endregion

using Couchcoding.Logbert.Gui.Helper;
using Couchcoding.Logbert.Gui.Interop;
using System;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace Couchcoding.Logbert.Gui.Controls
{
  /// <summary>
  /// Implements a <see cref="ListView"/> <see cref="Control"/> that is flicker free.
  /// </summary>
  public class ListViewEx : ListView
  {
    #region Private Consts

    /// <summary>
    /// Defines the minimum width of a column.
    /// </summary>
    private const int MIN_COLUMN_WIDTH = 100;

    #endregion

    #region Private Methods

    /// <summary>
    /// Creates the window handle of the <see cref="ListViewEx"/> <see cref="Control"/>.
    /// </summary>
    protected override void CreateHandle()
    {
      base.CreateHandle();

      if (OSHelper.IsWinVista)
      {
        Win32.SetWindowTheme(Handle, "Explorer", null);
      }
    }

    /// <summary>
    /// Notifies the control of Windows messages.
    /// </summary>
    /// <param name="m">A <see cref="T:System.Windows.Forms.Message"/> that represents the Windows message. </param>
    protected override void OnNotifyMessage(Message m)
    {
      // Filter out the WM_ERASEBKGND message.
      if (m.Msg != 0x14)
      {
        base.OnNotifyMessage(m);
      }
    }

    /// <summary>
    /// Raises the <see cref="E:System.Windows.Forms.Control.SizeChanged"/> event.
    /// </summary>
    /// <param name="e">An <see cref="T:System.EventArgs"/> that contains the event data. </param>
    protected override void OnSizeChanged(EventArgs e)
    {
      base.OnSizeChanged(e);

      // Ensure the last column uses all available space.
      StretchLastColumn();
    }

    #endregion

    #region Public Methods

    /// <summary>
    /// Strches the last column to all available space.
    /// </summary>
    public void StretchLastColumn()
    {
      if (Columns.Count > 0)
      {
        foreach (ColumnHeader clm in Columns)
        {
          // Ensure all columns uses a min space.
          clm.Width = clm.Width < MIN_COLUMN_WIDTH
            ? MIN_COLUMN_WIDTH
            : clm.Width;
        }

        // Stretch the last column to all available space.
        Columns[Columns.Count - 1].Width = -2;
      }
    }

    #endregion

    #region Constructor

    /// <summary>
    /// Creates a new instance of the <see cref="ListViewEx"/> <see cref="Control"/>.
    /// </summary>
    public ListViewEx()
    {
      SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
      SetStyle(ControlStyles.AllPaintingInWmPaint,  true);
      SetStyle(ControlStyles.EnableNotifyMessage,   true);
    }

    #endregion
  }
}
