//
//  Unicode.Scalar+Information.swift
//  CharacterInfo
//
//  CotEditor
//  https://coteditor.com
//
//  Created by 1024jp on 2016-07-26.
//
//  ---------------------------------------------------------------------------
//
//  © 2015-2024 1024jp
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  https://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.
//

public extension Unicode.Scalar {
    
    /// Code point string in format like `U+000F`.
    var codePoint: String {
        
        String(format: "U+%04tX", self.value)
    }
    
    
    /// Code point pair in UTF-16 surrogate pair.
    var surrogateCodePoints: (lead: String, trail: String)? {
        
        guard self.isSurrogatePair else { return nil }
        
        return (String(format: "U+%04X", UTF16.leadSurrogate(self)),
                String(format: "U+%04X", UTF16.trailSurrogate(self)))
    }
    
    
    /// Unicode name.
    var name: String? {
        
        self.properties.nameAlias
            ?? self.properties.name
            ?? self.controlCharacterName  // get control character name from special table
    }
    
    
    /// Localized and sanitized unicode block name.
    var localizedBlockName: String? {
        
        guard let blockName else { return nil }
        
        // -> This is actually a dirty workaround to make the block name the same as the Apple's block naming rule.
        //    Otherwise, we cannot localize block names correctly. (2015-11)
        let key = blockName
            .replacing(/\ ([A-Z])$/) { "-\($0.1)" }
            .replacing("Description", with: "Desc.")
        
        return String(localized: String.LocalizationValue(key), table: "UnicodeBlock", bundle: .module)
    }
}


extension Unicode.Scalar {
    
    /// Boolean value indicating whether character becomes a surrogate pair in UTF-16.
    var isSurrogatePair: Bool {
        
        (UTF16.width(self) == 2)
    }
    
    
    /// Unicode block name.
    var blockName: String? {
        
        self.value.blockName
    }
}
