/*
 * Copyright (c) 2001-2024 Convertigo SA.
 * 
 * This program  is free software; you  can redistribute it and/or
 * Modify  it  under the  terms of the  GNU  Affero General Public
 * License  as published by  the Free Software Foundation;  either
 * version  3  of  the  License,  or  (at your option)  any  later
 * version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY;  without even the implied warranty of
 * MERCHANTABILITY  or  FITNESS  FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program;
 * if not, see <http://www.gnu.org/licenses/>.
 */

package com.twinsoft.convertigo.eclipse.popup.actions;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import com.twinsoft.convertigo.beans.core.DatabaseObject;
import com.twinsoft.convertigo.beans.core.Sequence;
import com.twinsoft.convertigo.beans.core.StepEvent;
import com.twinsoft.convertigo.beans.core.StepWithExpressions;
import com.twinsoft.convertigo.beans.steps.SimpleSourceStep;
import com.twinsoft.convertigo.beans.steps.SourceStep;
import com.twinsoft.convertigo.eclipse.ConvertigoPlugin;
import com.twinsoft.convertigo.eclipse.views.projectexplorer.model.DatabaseObjectTreeObject;
import com.twinsoft.convertigo.eclipse.views.projectexplorer.ProjectExplorerView;
import com.twinsoft.convertigo.eclipse.views.projectexplorer.model.StepTreeObject;
import com.twinsoft.convertigo.eclipse.views.projectexplorer.model.TreeObject;
import com.twinsoft.convertigo.eclipse.views.projectexplorer.TreeParent;

public class ChangeToSourceStepAction extends MyAbstractAction {

	public ChangeToSourceStepAction() {
	}

	/* (non-Javadoc)
	 * @see com.twinsoft.convertigo.eclipse.popup.actions.MyAbstractAction#run()
	 */
	@Override
	public void run() {
		Display display = Display.getDefault();
		Cursor waitCursor = new Cursor(display, SWT.CURSOR_WAIT);		
		
		Shell shell = getParentShell();
		shell.setCursor(waitCursor);
		
        try {
    		ProjectExplorerView explorerView = getProjectExplorerView();
    		if (explorerView != null) {
    			TreeObject treeObject = explorerView.getFirstSelectedTreeObject();
    			Object databaseObject = treeObject.getObject();
    			if ((databaseObject != null) && (databaseObject instanceof SimpleSourceStep)) {
    				SimpleSourceStep jSimpleSourceStep = (SimpleSourceStep)databaseObject;    											
					TreeParent treeParent = treeObject.getParent();
					DatabaseObjectTreeObject parentTreeObject = null;
					if (treeParent instanceof DatabaseObjectTreeObject)
						parentTreeObject = (DatabaseObjectTreeObject)treeParent;
					else
						parentTreeObject = (DatabaseObjectTreeObject)treeParent.getParent();
					
	        		if (parentTreeObject != null) {
						// New jSource step
						SourceStep jSourceStep = new SourceStep();
						
						// Set properties
						jSourceStep.setSourceDefinition(jSimpleSourceStep.getSourceDefinition());
						jSourceStep.setComment(jSimpleSourceStep.getComment());
						jSourceStep.setVariableName(jSimpleSourceStep.getVariableName());
						jSourceStep.setEnabled(jSimpleSourceStep.isEnabled());
						jSourceStep.setOutput(jSimpleSourceStep.isOutput());
						jSourceStep.setName(jSimpleSourceStep.getName());
						
						jSourceStep.bNew = true;
						jSourceStep.hasChanged = true;
						
						// Add new jSource step to parent
						DatabaseObject parentDbo = jSimpleSourceStep.getParent();
						parentDbo.add(jSourceStep);
						
						// Set correct order
						if (parentDbo instanceof StepWithExpressions)
							((StepWithExpressions)parentDbo).insertAtOrder(jSourceStep,jSimpleSourceStep.priority);
						else if (parentDbo instanceof Sequence)
							((Sequence)parentDbo).insertAtOrder(jSourceStep,jSimpleSourceStep.priority);
						
						// Add new jSource step in Tree
						StepTreeObject stepTreeObject = new StepTreeObject(explorerView.viewer,jSourceStep);
						treeParent.addChild(stepTreeObject);
						
		   				// Delete jSimpleSource step
						long oldPriority = jSimpleSourceStep.priority;
						jSimpleSourceStep.delete();
		   				
		   				// Simulate move of jSimpleSource to jSource step
						jSourceStep.getSequence().fireStepMoved(new StepEvent(jSourceStep,String.valueOf(oldPriority)));
						
	        			parentTreeObject.hasBeenModified(true);
		                explorerView.reloadTreeObject(parentTreeObject);
		                explorerView.setSelectedTreeObject(explorerView.findTreeObjectByUserObject(jSourceStep));
	        		}
				}
    		}
        }
        catch (Throwable e) {
        	ConvertigoPlugin.logException(e, "Unable to change step to jSource step!");
        }
        finally {
			shell.setCursor(null);
			waitCursor.dispose();
        }
	}
}
