/*
 * Copyright (c) 2001-2024 Convertigo SA.
 * 
 * This program  is free software; you  can redistribute it and/or
 * Modify  it  under the  terms of the  GNU  Affero General Public
 * License  as published by  the Free Software Foundation;  either
 * version  3  of  the  License,  or  (at your option)  any  later
 * version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY;  without even the implied warranty of
 * MERCHANTABILITY  or  FITNESS  FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program;
 * if not, see <http://www.gnu.org/licenses/>.
 */

package com.twinsoft.convertigo.eclipse.editors.sequence;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.part.FileInPlaceEditorInput;

import com.twinsoft.convertigo.beans.core.Project;
import com.twinsoft.convertigo.beans.core.Sequence;
import com.twinsoft.convertigo.eclipse.ConvertigoPlugin;
import com.twinsoft.convertigo.eclipse.swt.SwtUtils;

public class SequenceEditorInput extends FileInPlaceEditorInput {

	Sequence sequence;
	private String qname;
	
	private static IFile getTmpFile(Sequence sequence, String filename) {
		try {
			return ConvertigoPlugin.getDefault().getProjectPluginResource(sequence.getProject().getName()).getFile("_private/editor/" + sequence.getShortQName() + "/" + filename);
		} catch (Exception e) {
		}
		return null;
	}
	
	public SequenceEditorInput(Sequence sequence, String filename) {
		super(getTmpFile(sequence, filename));
		this.sequence = sequence;
		qname = sequence.getQName();
	}
	
	public SequenceEditorInput(Sequence sequence) {
		this(sequence, "init.txt");
		if (!fileExists()) {
			fileWrite("Click on the generation button to view the response generated by Convertigo.");
		}
	}
	
	@Override
	public boolean exists() {
		return super.exists();
	}

	@Override
	public ImageDescriptor getImageDescriptor() {
		return super.getImageDescriptor();
	}

	@Override
	public String getName() {
		return sequence.getParentName() + " [S: " + sequence.getName()+"]." + getFile().getFileExtension();
	}

	@Override
	public IPersistableElement getPersistable() {
		return super.getPersistable();
	}

	@Override
	public String getToolTipText() {
		return sequence.getParentName() + "/" + sequence.getName();
	}

	@Override
	@SuppressWarnings("unchecked")
	public Object getAdapter(@SuppressWarnings("rawtypes") Class adapter) {
		return super.getAdapter(adapter);
	}
	
	public boolean is(Sequence sequence) {
		return sequence.equals(this.sequence) && sequence.getQName().equals(qname);
	}
	
	public boolean is(Project project) {
		return qname.startsWith(project.getQName());
	}
	
	public Sequence getSequence() {
		return sequence;
	}
	
	public void fileDelete() {
		try {
			getFile().delete(true, null);
		} catch (CoreException e) {
		}
	}
	
	public boolean fileExists() {
		return getFile().exists();
	}
	
	public void fileWrite(String str) {
		SwtUtils.fillFile(getFile(), str);
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof SequenceEditorInput) {
			return sequence.equals(((SequenceEditorInput) obj).sequence);
		}
		return false;
	}
}
