/*
 * Decompiled with CFR 0.152.
 */
package apoc.spatial;

import apoc.ApocConfig;
import apoc.util.JsonUtil;
import apoc.util.MapUtil;
import apoc.util.Util;
import java.util.AbstractMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;
import org.apache.commons.configuration2.Configuration;
import org.neo4j.graphdb.security.URLAccessChecker;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;
import org.neo4j.procedure.TerminationGuard;

public class Geocode {
    public static final int MAX_RESULTS = 100;
    public static final String PREFIX = "apoc.spatial.geocode";
    public static final String GEOCODE_PROVIDER_KEY = "provider";
    @Context
    public TerminationGuard terminationGuard;
    @Context
    public URLAccessChecker urlAccessChecker;

    private GeocodeSupplier getSupplier(Map<String, Object> configMap) {
        return Geocode.getSupplier(configMap, this.terminationGuard);
    }

    public static GeocodeSupplier getSupplier(Map<String, Object> configMap, TerminationGuard terminationGuard) {
        AbstractMap.SimpleEntry<GeocodeSupplier, String> results = Geocode.getSupplierEntry(terminationGuard, configMap);
        return results.getKey();
    }

    public static AbstractMap.SimpleEntry<GeocodeSupplier, String> getSupplierEntry(TerminationGuard terminationGuard, Map<String, Object> configMap) {
        Configuration activeConfig = ApocConfig.apocConfig().getConfig().subset(PREFIX);
        String provider = (String)configMap.getOrDefault(GEOCODE_PROVIDER_KEY, activeConfig.getString(GEOCODE_PROVIDER_KEY, "osm"));
        configMap.forEach((key, value) -> {
            String dotCase = key.replaceAll("[A-Z][a-z]", ".$0").toLowerCase();
            activeConfig.setProperty(provider + "." + dotCase, value);
        });
        String supplier = provider.toLowerCase();
        GeocodeSupplier geocodeSupplier = Geocode.getGeocodeSupplier(terminationGuard, activeConfig, supplier);
        return new AbstractMap.SimpleEntry<GeocodeSupplier, String>(geocodeSupplier, supplier);
    }

    public static GeocodeSupplier getGeocodeSupplier(TerminationGuard terminationGuard, Configuration activeConfig, String supplier) {
        return switch (supplier) {
            case "google" -> new GoogleSupplier(activeConfig, terminationGuard);
            case "osm" -> new OSMSupplier(activeConfig, terminationGuard);
            default -> new SupplierWithKey(activeConfig, terminationGuard, supplier);
        };
    }

    @Procedure(value="apoc.spatial.geocodeOnce")
    @Description(value="Returns the geographic location (latitude, longitude, and description) of the given address using a geocoding service (default: OpenStreetMap).\nThis procedure returns at most one result.")
    public Stream<GeoCodeResult> geocodeOnce(@Name(value="location") String address, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        return this.geocode(address, 1L, false, config);
    }

    @Procedure(value="apoc.spatial.geocode")
    @Description(value="Returns the geographic location (latitude, longitude, and description) of the given address using a geocoding service (default: OpenStreetMap).")
    public Stream<GeoCodeResult> geocode(@Name(value="location") String address, @Name(value="maxResults", defaultValue="100") long maxResults, @Name(value="quotaException", defaultValue="false") boolean quotaException, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        if (address == null || address.isEmpty()) {
            return Stream.empty();
        }
        try {
            return this.getSupplier(config).geocode(address, maxResults == 0L ? 100L : Math.min(Math.max(maxResults, 1L), 100L), this.urlAccessChecker);
        }
        catch (IllegalStateException re) {
            if (!quotaException && re.getMessage().startsWith("QUOTA_EXCEEDED")) {
                return Stream.empty();
            }
            throw re;
        }
    }

    @Procedure(value="apoc.spatial.reverseGeocode")
    @Description(value="Returns a textual address from the given geographic location (latitude, longitude) using a geocoding service (default: OpenStreetMap).\nThis procedure returns at most one result.")
    public Stream<GeoCodeResult> reverseGeocode(@Name(value="latitude") double latitude, @Name(value="longitude") double longitude, @Name(value="quotaException", defaultValue="false") boolean quotaException, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        try {
            return this.getSupplier(config).reverseGeocode(latitude, longitude, this.urlAccessChecker);
        }
        catch (IllegalStateException re) {
            if (!quotaException && re.getMessage().startsWith("QUOTA_EXCEEDED")) {
                return Stream.empty();
            }
            throw re;
        }
    }

    static interface GeocodeSupplier {
        public Stream<GeoCodeResult> geocode(String var1, long var2, URLAccessChecker var4);

        public Stream<GeoCodeResult> reverseGeocode(Double var1, Double var2, URLAccessChecker var3);
    }

    private static class GoogleSupplier
    implements GeocodeSupplier {
        private final Throttler throttler;
        private Configuration config;
        private static final String BASE_GOOGLE_API_URL = "https://maps.googleapis.com/maps/api/geocode/json";
        private static final String REVERSE_GEOCODE_URL = "https://maps.googleapis.com/maps/api/geocode/json?%s&latlng=";
        private static final String GEOCODE_URL = "https://maps.googleapis.com/maps/api/geocode/json?%s&address=";

        public GoogleSupplier(Configuration config, TerminationGuard terminationGuard) {
            this.throttler = new Throttler(terminationGuard, Util.toLong(config.getString("google.throttle", Long.toString(Throttler.DEFAULT_THROTTLE))));
            this.config = config;
        }

        private String credentials(Configuration config) {
            if (config.containsKey("google.client") && config.containsKey("google.signature")) {
                return "client=" + config.getString("google.client") + "&signature=" + config.getString("google.signature");
            }
            if (config.containsKey("google.key")) {
                return "key=" + config.getString("google.key");
            }
            return "auth=free";
        }

        @Override
        public Stream<GeoCodeResult> geocode(String address, long maxResults, URLAccessChecker urlAccessChecker) {
            if (address.isEmpty()) {
                return Stream.empty();
            }
            this.throttler.waitForThrottle();
            Object value = JsonUtil.loadJson(String.format(GEOCODE_URL, this.credentials(this.config)) + Util.encodeUrlComponent(address), urlAccessChecker).findFirst().orElse(null);
            if (value instanceof Map) {
                Map map = value;
                if (map.get("status").equals("OVER_QUERY_LIMIT")) {
                    throw new IllegalStateException("QUOTA_EXCEEDED from geocode API: " + map.get("status") + " message: " + map.get("error_message"));
                }
                Object results = map.get("results");
                if (results instanceof List) {
                    return ((List)results).stream().limit(maxResults).map(data -> {
                        Map location = (Map)((Map)data.get("geometry")).get("location");
                        return new GeoCodeResult(Util.toDouble(location.get("lat")), Util.toDouble(location.get("lng")), String.valueOf(data.get("formatted_address")), (Map<String, Object>)data);
                    });
                }
            }
            throw new RuntimeException("Can't parse geocoding results " + value);
        }

        @Override
        public Stream<GeoCodeResult> reverseGeocode(Double latitude, Double longitude, URLAccessChecker urlAccessChecker) {
            if (latitude == null || longitude == null) {
                return Stream.empty();
            }
            this.throttler.waitForThrottle();
            Object value = JsonUtil.loadJson(String.format(REVERSE_GEOCODE_URL, this.credentials(this.config)) + Util.encodeUrlComponent(latitude + "," + longitude), urlAccessChecker).findFirst().orElse(null);
            if (value instanceof Map) {
                Map map = value;
                if (map.get("status").equals("OVER_QUERY_LIMIT")) {
                    throw new IllegalStateException("QUOTA_EXCEEDED from geocode API: " + map.get("status") + " message: " + map.get("error_message"));
                }
                Object results = map.get("results");
                if (results instanceof List) {
                    return ((List)results).stream().limit(1L).map(data -> {
                        Map location = (Map)((Map)data.get("geometry")).get("location");
                        return new GeoCodeResult(Util.toDouble(location.get("lat")), Util.toDouble(location.get("lng")), String.valueOf(data.get("formatted_address")), (Map<String, Object>)data);
                    });
                }
            }
            throw new RuntimeException("Can't parse reverse-geocoding results " + value);
        }
    }

    private static class OSMSupplier
    implements GeocodeSupplier {
        public static final String OSM_URL = "https://nominatim.openstreetmap.org";
        private static final String OSM_URL_REVERSE_GEOCODE = "https://nominatim.openstreetmap.org/reverse?format=jsonv2&";
        private static final String OSM_URL_GEOCODE = "https://nominatim.openstreetmap.org/search.php?format=json&q=";
        private Throttler throttler;

        public OSMSupplier(Configuration config, TerminationGuard terminationGuard) {
            this.throttler = new Throttler(terminationGuard, Util.toLong(config.getString("osm.throttle", Long.toString(Throttler.DEFAULT_THROTTLE))));
        }

        @Override
        public Stream<GeoCodeResult> geocode(String address, long maxResults, URLAccessChecker urlAccessChecker) {
            if (address.isEmpty()) {
                return Stream.empty();
            }
            this.throttler.waitForThrottle();
            Object value = JsonUtil.loadJson(OSM_URL_GEOCODE + Util.encodeUrlComponent(address), urlAccessChecker).findFirst().orElse(null);
            if (value instanceof List) {
                return ((List)value).stream().limit(maxResults).map(data -> new GeoCodeResult(Util.toDouble(data.get("lat")), Util.toDouble(data.get("lon")), String.valueOf(data.get("display_name")), (Map<String, Object>)data));
            }
            throw new RuntimeException("Can't parse geocoding results " + value);
        }

        @Override
        public Stream<GeoCodeResult> reverseGeocode(Double latitude, Double longitude, URLAccessChecker urlAccessChecker) {
            if (latitude == null || longitude == null) {
                return Stream.empty();
            }
            this.throttler.waitForThrottle();
            Object value = JsonUtil.loadJson(OSM_URL_REVERSE_GEOCODE + String.format("lat=%s&lon=%s", latitude, longitude), urlAccessChecker).findFirst().orElse(null);
            if (value instanceof Map) {
                Map data = value;
                return Stream.of(new GeoCodeResult(Util.toDouble(data.get("lat")), Util.toDouble(data.get("lon")), String.valueOf(data.get("display_name")), (Map)data.get("address")));
            }
            throw new RuntimeException("Can't parse reverse-geocoding results " + value);
        }
    }

    private static class SupplierWithKey
    implements GeocodeSupplier {
        private static final String[] FORMATTED_KEYS = new String[]{"formatted", "formatted_address", "address", "description", "display_name"};
        private static final String[] LAT_KEYS = new String[]{"lat", "latitude"};
        private static final String[] LNG_KEYS = new String[]{"lng", "longitude", "lon"};
        private Throttler throttler;
        private String configBase;
        private String urlTemplate;
        private String urlTemplateReverse;

        public SupplierWithKey(Configuration config, TerminationGuard terminationGuard, String provider) {
            this.configBase = provider;
            if (!config.containsKey(this.configKey("url"))) {
                throw new IllegalArgumentException("Missing 'url' for geocode provider: " + provider);
            }
            if (!config.containsKey(this.configKey("reverse.url"))) {
                throw new IllegalArgumentException("Missing 'reverse.url' for reverse-geocode provider: " + provider);
            }
            this.urlTemplate = config.getString(this.configKey("url"));
            if (!this.urlTemplate.contains("PLACE")) {
                throw new IllegalArgumentException("Missing 'PLACE' in url template: " + this.urlTemplate);
            }
            this.urlTemplateReverse = config.getString(this.configKey("reverse.url"));
            if (!this.urlTemplateReverse.contains("LAT") || !this.urlTemplateReverse.contains("LNG")) {
                throw new IllegalArgumentException("Missing 'LAT' or 'LNG' in url template: " + this.urlTemplateReverse);
            }
            if (this.urlTemplate.contains("KEY") && !config.containsKey(this.configKey("key"))) {
                throw new IllegalArgumentException("Missing 'key' for geocode provider: " + provider);
            }
            if (this.urlTemplateReverse.contains("KEY") && !config.containsKey(this.configKey("key"))) {
                throw new IllegalArgumentException("Missing 'key' for reverse-geocode provider: " + provider);
            }
            String key = config.getString(this.configKey("key"));
            this.urlTemplate = this.urlTemplate.replace("KEY", key);
            this.urlTemplateReverse = this.urlTemplateReverse.replace("KEY", key);
            this.throttler = new Throttler(terminationGuard, config.getInt(this.configKey("throttle"), (int)Throttler.DEFAULT_THROTTLE));
        }

        @Override
        public Stream<GeoCodeResult> geocode(String address, long maxResults, URLAccessChecker urlAccessChecker) {
            Object results;
            if (address.isEmpty()) {
                return Stream.empty();
            }
            this.throttler.waitForThrottle();
            String url = this.urlTemplate.replace("PLACE", Util.encodeUrlComponent(address));
            Object value = JsonUtil.loadJson(url, urlAccessChecker).findFirst().orElse(null);
            if (value instanceof List) {
                return this.findResults(value, maxResults);
            }
            if (value instanceof Map && (results = ((Map)value).get("results")) instanceof List) {
                return this.findResults((List)results, maxResults);
            }
            throw new RuntimeException("Can't parse geocoding results " + value);
        }

        @Override
        public Stream<GeoCodeResult> reverseGeocode(Double latitude, Double longitude, URLAccessChecker urlAccessChecker) {
            Object results;
            if (latitude == null || longitude == null) {
                return Stream.empty();
            }
            this.throttler.waitForThrottle();
            String url = this.urlTemplateReverse.replace("LAT", latitude.toString()).replace("LNG", longitude.toString());
            Object value = JsonUtil.loadJson(url, urlAccessChecker).findFirst().orElse(null);
            if (value instanceof List) {
                return this.findResults(value, 1L);
            }
            if (value instanceof Map && (results = ((Map)value).get("results")) instanceof List) {
                return this.findResults((List)results, 1L);
            }
            throw new RuntimeException("Can't parse reverse-geocoding results " + value);
        }

        private Stream<GeoCodeResult> findResults(List<Map<String, Object>> results, long maxResults) {
            return results.stream().limit(maxResults).map(data -> {
                String description = this.findFirstEntry((Map<String, Object>)data, FORMATTED_KEYS);
                Map location = (Map)data.get("geometry");
                if (location.containsKey("location")) {
                    location = (Map)location.get("location");
                }
                String lat = this.findFirstEntry(location, LAT_KEYS);
                String lng = this.findFirstEntry(location, LNG_KEYS);
                return new GeoCodeResult(Util.toDouble(lat), Util.toDouble(lng), description, (Map<String, Object>)data);
            });
        }

        private String findFirstEntry(Map<String, Object> map, String[] keys) {
            for (String key : keys) {
                if (!map.containsKey(key)) continue;
                return String.valueOf(map.get(key));
            }
            return "";
        }

        private String configKey(String name) {
            return this.configBase + "." + name;
        }
    }

    public static class GeoCodeResult {
        public final Map<String, Object> location;
        public final Map<String, Object> data;
        public final Double latitude;
        public final Double longitude;
        public final String description;

        public GeoCodeResult(Double latitude, Double longitude, String description, Map<String, Object> data) {
            this.data = data;
            this.latitude = latitude;
            this.longitude = longitude;
            this.description = description;
            this.location = MapUtil.map("latitude", latitude, "longitude", longitude, "description", description);
        }
    }

    private static class Throttler {
        private final TerminationGuard terminationGuard;
        private long throttleInMs;
        private static long lastCallTime = 0L;
        private static long DEFAULT_THROTTLE = 5000L;
        private static long MAX_THROTTLE = 3600000L;

        public Throttler(TerminationGuard terminationGuard, long throttle) {
            this.terminationGuard = terminationGuard;
            throttle = Math.min(throttle, MAX_THROTTLE);
            if (throttle < 0L) {
                throttle = DEFAULT_THROTTLE;
            }
            this.throttleInMs = throttle;
        }

        private void waitForThrottle() {
            long msSinceLastCall = System.currentTimeMillis() - lastCallTime;
            while (msSinceLastCall < this.throttleInMs) {
                try {
                    this.terminationGuard.check();
                    long msToWait = this.throttleInMs - msSinceLastCall;
                    Thread.sleep(Math.min(msToWait, 1000L));
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                msSinceLastCall = System.currentTimeMillis() - lastCallTime;
            }
            lastCallTime = System.currentTimeMillis();
        }
    }
}

