/*
 * Decompiled with CFR 0.152.
 */
package apoc.refactor;

import apoc.Pools;
import apoc.algo.Cover;
import apoc.refactor.NodeRefactorResult;
import apoc.refactor.RelationshipRefactorResult;
import apoc.refactor.util.PropertiesManager;
import apoc.refactor.util.RefactorConfig;
import apoc.refactor.util.RefactorUtil;
import apoc.result.GraphResult;
import apoc.result.NodeResult;
import apoc.result.RelationshipResult;
import apoc.util.Util;
import apoc.util.collection.Iterables;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Entity;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Label;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Path;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ResourceIterable;
import org.neo4j.graphdb.Result;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.ConstraintType;
import org.neo4j.kernel.impl.coreapi.InternalTransaction;
import org.neo4j.logging.Log;
import org.neo4j.procedure.Context;
import org.neo4j.procedure.Description;
import org.neo4j.procedure.Mode;
import org.neo4j.procedure.Name;
import org.neo4j.procedure.Procedure;

public class GraphRefactoring {
    @Context
    public Transaction tx;
    @Context
    public GraphDatabaseService db;
    @Context
    public Log log;
    @Context
    public Pools pools;

    @Procedure(name="apoc.refactor.extractNode", mode=Mode.WRITE)
    @Description(value="Expands the given `RELATIONSHIP` VALUES into intermediate `NODE` VALUES.\nThe intermediate `NODE` values are connected by the given `outType` and `inType`.")
    public Stream<NodeRefactorResult> extractNode(@Name(value="rels") Object rels, @Name(value="labels") List<String> labels, @Name(value="outType") String outType, @Name(value="inType") String inType) {
        return Util.relsStream((InternalTransaction)this.tx, rels).map(rel -> {
            NodeRefactorResult result = new NodeRefactorResult(rel.getId());
            try {
                Node copy = Util.withTransactionAndRebind(this.db, this.tx, transaction -> {
                    Node copyNode = RefactorUtil.copyProperties((Entity)rel, transaction.createNode(Util.labels(labels)));
                    copyNode.createRelationshipTo(rel.getEndNode(), RelationshipType.withName((String)outType));
                    return copyNode;
                });
                rel.getStartNode().createRelationshipTo(copy, RelationshipType.withName((String)inType));
                rel.delete();
                return result.withOther(copy);
            }
            catch (Exception e) {
                return result.withError(e);
            }
        });
    }

    @Procedure(name="apoc.refactor.collapseNode", mode=Mode.WRITE)
    @Description(value="Collapses the given `NODE` and replaces it with a `RELATIONSHIP` of the given type.")
    public Stream<RelationshipRefactorResult> collapseNode(@Name(value="nodes") Object nodes, @Name(value="relType") String type) {
        return Util.nodeStream((InternalTransaction)this.tx, nodes).map(node -> {
            RelationshipRefactorResult result = new RelationshipRefactorResult(node.getId());
            try {
                ResourceIterable outRels = node.getRelationships(Direction.OUTGOING);
                ResourceIterable inRels = node.getRelationships(Direction.INCOMING);
                if (node.getDegree(Direction.OUTGOING) == 1 && node.getDegree(Direction.INCOMING) == 1) {
                    Relationship outRel = (Relationship)outRels.iterator().next();
                    Relationship inRel = (Relationship)inRels.iterator().next();
                    Relationship newRel = inRel.getStartNode().createRelationshipTo(outRel.getEndNode(), RelationshipType.withName((String)type));
                    newRel = RefactorUtil.copyProperties((Entity)node, RefactorUtil.copyProperties((Entity)inRel, RefactorUtil.copyProperties((Entity)outRel, newRel)));
                    for (Relationship r : inRels) {
                        r.delete();
                    }
                    for (Relationship r : outRels) {
                        r.delete();
                    }
                    node.delete();
                    return result.withOther(newRel);
                }
                return result.withError(String.format("Node %d has more that 1 outgoing %d or incoming %d relationships", node.getId(), node.getDegree(Direction.OUTGOING), node.getDegree(Direction.INCOMING)));
            }
            catch (Exception e) {
                return result.withError(e);
            }
        });
    }

    @Procedure(name="apoc.refactor.cloneNodes", mode=Mode.WRITE)
    @Description(value="Clones the given `NODE` values with their labels and properties.\nIt is possible to skip any `NODE` properties using skipProperties (note: this only skips properties on `NODE` values and not their `RELATIONSHIP` values).")
    public Stream<NodeRefactorResult> cloneNodes(@Name(value="nodes") List<Node> nodes, @Name(value="withRelationships", defaultValue="false") boolean withRelationships, @Name(value="skipProperties", defaultValue="[]") List<String> skipProperties) {
        if (nodes == null) {
            return Stream.empty();
        }
        return nodes.stream().map(node -> {
            NodeRefactorResult result = new NodeRefactorResult(node.getId());
            Node newNode = this.tx.createNode(Util.getLabelsArray(node));
            Map properties = node.getAllProperties();
            if (skipProperties != null && !skipProperties.isEmpty()) {
                for (String skip : skipProperties) {
                    properties.remove(skip);
                }
            }
            try {
                RefactorUtil.copyProperties(properties, newNode);
                if (withRelationships) {
                    this.copyRelationships((Node)node, newNode, false, true);
                }
            }
            catch (Exception e) {
                if (withRelationships) {
                    for (Relationship rel : newNode.getRelationships()) {
                        rel.delete();
                    }
                }
                newNode.delete();
                return result.withError(e);
            }
            return result.withOther(newNode);
        });
    }

    @Procedure(name="apoc.refactor.cloneSubgraphFromPaths", mode=Mode.WRITE)
    @Description(value="Clones a sub-graph defined by the given `LIST<PATH>` values.\nIt is possible to skip any `NODE` properties using the `skipProperties` `LIST<STRING>` via the config `MAP`.")
    public Stream<NodeRefactorResult> cloneSubgraphFromPaths(@Name(value="paths") List<Path> paths, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        if (paths == null || paths.isEmpty()) {
            return Stream.empty();
        }
        HashSet<Node> nodes = new HashSet<Node>();
        HashSet<Relationship> rels = new HashSet<Relationship>();
        for (Path path : paths) {
            for (Relationship rel : path.relationships()) {
                rels.add(rel);
            }
            for (Node node : path.nodes()) {
                nodes.add(node);
            }
        }
        ArrayList<Node> nodesList = new ArrayList<Node>(nodes);
        ArrayList<Relationship> relsList = new ArrayList<Relationship>(rels);
        return this.cloneSubgraph(nodesList, relsList, config);
    }

    @Procedure(name="apoc.refactor.cloneSubgraph", mode=Mode.WRITE)
    @Description(value="Clones the given `NODE` values with their labels and properties (optionally skipping any properties in the `skipProperties` `LIST<STRING>` via the config `MAP`), and clones the given `RELATIONSHIP` values.\nIf no `RELATIONSHIP` values are provided, all existing `RELATIONSHIP` values between the given `NODE` values will be cloned.")
    public Stream<NodeRefactorResult> cloneSubgraph(@Name(value="nodes") List<Node> nodes, @Name(value="rels", defaultValue="[]") List<Relationship> rels, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        if (nodes == null || nodes.isEmpty()) {
            return Stream.empty();
        }
        if (rels == null || rels.isEmpty()) {
            rels = Cover.coverNodes(nodes).collect(Collectors.toList());
        }
        HashMap<Node, Node> copyMap = new HashMap<Node, Node>(nodes.size());
        ArrayList<NodeRefactorResult> resultStream = new ArrayList<NodeRefactorResult>();
        Map<Node, Node> standinMap = this.generateStandinMap(config.getOrDefault("standinNodes", Collections.emptyList()));
        List skipProperties = config.getOrDefault("skipProperties", Collections.emptyList());
        for (Node node : nodes) {
            if (node == null || standinMap.containsKey(node)) continue;
            NodeRefactorResult result = new NodeRefactorResult(node.getId());
            try {
                Node copy = Util.withTransactionAndRebind(this.db, this.tx, transaction -> {
                    Node copyTemp = transaction.createNode();
                    Map properties = node.getAllProperties();
                    if (skipProperties != null && !skipProperties.isEmpty()) {
                        for (String skip : skipProperties) {
                            properties.remove(skip);
                        }
                    }
                    RefactorUtil.copyProperties(properties, copyTemp);
                    this.copyLabels(node, copyTemp);
                    return copyTemp;
                });
                resultStream.add(result.withOther(copy));
                copyMap.put(node, copy);
            }
            catch (Exception e) {
                resultStream.add(result.withError(e));
            }
        }
        for (Relationship relationship : rels) {
            if (relationship == null) continue;
            Node oldStart = relationship.getStartNode();
            Node newStart = standinMap.getOrDefault(oldStart, (Node)copyMap.get(oldStart));
            Node oldEnd = relationship.getEndNode();
            Node newEnd = standinMap.getOrDefault(oldEnd, (Node)copyMap.get(oldEnd));
            if (newStart == null || newEnd == null) continue;
            Relationship newrel = newStart.createRelationshipTo(newEnd, relationship.getType());
            Map properties = relationship.getAllProperties();
            if (skipProperties != null && !skipProperties.isEmpty()) {
                for (String skip : skipProperties) {
                    properties.remove(skip);
                }
            }
            RefactorUtil.copyProperties(properties, newrel);
        }
        return resultStream.stream();
    }

    private Map<Node, Node> generateStandinMap(List<List<Node>> standins) {
        HashMap<Node, Node> standinMap = standins.isEmpty() ? Collections.emptyMap() : new HashMap<Node, Node>(standins.size());
        for (List<Node> pairing : standins) {
            if (pairing == null) continue;
            if (pairing.size() != 2) {
                throw new IllegalArgumentException("'standinNodes' must be a list of node pairs");
            }
            Node from = pairing.get(0);
            Node to = pairing.get(1);
            if (from == null || to == null) {
                throw new IllegalArgumentException("'standinNodes' must be a list of node pairs");
            }
            standinMap.put(from, to);
        }
        return standinMap;
    }

    @Procedure(name="apoc.refactor.mergeNodes", mode=Mode.WRITE, eager=true)
    @Description(value="Merges the given `LIST<NODE>` onto the first `NODE` in the `LIST<NODE>`.\nAll `RELATIONSHIP` values are merged onto that `NODE` as well.")
    public Stream<NodeResult> mergeNodes(@Name(value="nodes") List<Node> nodes, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        if (nodes == null || nodes.isEmpty()) {
            return Stream.empty();
        }
        RefactorConfig conf = new RefactorConfig(config);
        LinkedHashSet<Node> nodesSet = new LinkedHashSet<Node>(nodes);
        nodesSet.stream().sorted(Comparator.comparing(Entity::getElementId)).forEach(arg_0 -> ((Transaction)this.tx).acquireWriteLock(arg_0));
        Node first = nodes.get(0);
        List existingSelfRelIds = conf.isPreservingExistingSelfRels() ? StreamSupport.stream(first.getRelationships().spliterator(), false).filter(Util::isSelfRel).map(Entity::getElementId).collect(Collectors.toList()) : Collections.emptyList();
        nodesSet.stream().skip(1L).forEach(node -> this.mergeNodes((Node)node, first, conf, existingSelfRelIds));
        return Stream.of(new NodeResult(first));
    }

    @Procedure(name="apoc.refactor.mergeRelationships", mode=Mode.WRITE)
    @Description(value="Merges the given `LIST<RELATIONSHIP>` onto the first `RELATIONSHIP` in the `LIST<RELATIONSHIP>`.")
    public Stream<RelationshipResult> mergeRelationships(@Name(value="rels") List<Relationship> relationships, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        if (relationships == null || relationships.isEmpty()) {
            return Stream.empty();
        }
        LinkedHashSet<Relationship> relationshipsSet = new LinkedHashSet<Relationship>(relationships);
        RefactorConfig conf = new RefactorConfig(config);
        Iterator it = relationshipsSet.iterator();
        Relationship first = (Relationship)it.next();
        while (it.hasNext()) {
            Relationship other = (Relationship)it.next();
            if (first.getStartNode().equals(other.getStartNode()) && first.getEndNode().equals(other.getEndNode())) {
                RefactorUtil.mergeRels(other, first, true, conf);
                continue;
            }
            throw new RuntimeException("All Relationships must have the same start and end nodes.");
        }
        return Stream.of(new RelationshipResult(first));
    }

    @Procedure(name="apoc.refactor.setType", mode=Mode.WRITE)
    @Description(value="Changes the type of the given `RELATIONSHIP`.")
    public Stream<RelationshipRefactorResult> setType(@Name(value="rel") Relationship rel, @Name(value="newType") String newType) {
        if (rel == null) {
            return Stream.empty();
        }
        RelationshipRefactorResult result = new RelationshipRefactorResult(rel.getId());
        try {
            Relationship newRel = rel.getStartNode().createRelationshipTo(rel.getEndNode(), RelationshipType.withName((String)newType));
            RefactorUtil.copyProperties((Entity)rel, newRel);
            rel.delete();
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.to", mode=Mode.WRITE, eager=true)
    @Description(value="Redirects the given `RELATIONSHIP` to the given end `NODE`.")
    public Stream<RelationshipRefactorResult> to(@Name(value="rel") Relationship rel, @Name(value="endNode") Node newNode) {
        if (rel == null || newNode == null) {
            return Stream.empty();
        }
        RelationshipRefactorResult result = new RelationshipRefactorResult(rel.getId());
        try {
            Relationship newRel = rel.getStartNode().createRelationshipTo(newNode, rel.getType());
            RefactorUtil.copyProperties((Entity)rel, newRel);
            rel.delete();
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.invert", mode=Mode.WRITE, eager=true)
    @Description(value="Inverts the direction of the given `RELATIONSHIP`.")
    public Stream<RelationshipRefactorResult> invert(@Name(value="rel") Relationship rel) {
        if (rel == null) {
            return Stream.empty();
        }
        RelationshipRefactorResult result = new RelationshipRefactorResult(rel.getId());
        try {
            Relationship newRel = rel.getEndNode().createRelationshipTo(rel.getStartNode(), rel.getType());
            RefactorUtil.copyProperties((Entity)rel, newRel);
            rel.delete();
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.from", mode=Mode.WRITE, eager=true)
    @Description(value="Redirects the given `RELATIONSHIP` to the given start `NODE`.")
    public Stream<RelationshipRefactorResult> from(@Name(value="rel") Relationship rel, @Name(value="newNode") Node newNode) {
        if (rel == null || newNode == null) {
            return Stream.empty();
        }
        RelationshipRefactorResult result = new RelationshipRefactorResult(rel.getId());
        try {
            Relationship newRel = newNode.createRelationshipTo(rel.getEndNode(), rel.getType());
            RefactorUtil.copyProperties((Entity)rel, newRel);
            rel.delete();
            return Stream.of(result.withOther(newRel));
        }
        catch (Exception e) {
            return Stream.of(result.withError(e));
        }
    }

    @Procedure(name="apoc.refactor.normalizeAsBoolean", mode=Mode.WRITE)
    @Description(value="Refactors the given property to a `BOOLEAN`.")
    public void normalizeAsBoolean(@Name(value="entity") Object entity, @Name(value="propertyKey") String propertyKey, @Name(value="trueValues") List<Object> trueValues, @Name(value="falseValues") List<Object> falseValues) {
        Entity pc;
        Object value;
        if (entity instanceof Entity && (value = (pc = (Entity)entity).getProperty(propertyKey, null)) != null) {
            boolean isTrue = trueValues.contains(value);
            boolean isFalse = falseValues.contains(value);
            if (isTrue && !isFalse) {
                pc.setProperty(propertyKey, (Object)true);
            }
            if (!isTrue && isFalse) {
                pc.setProperty(propertyKey, (Object)false);
            }
            if (!isTrue && !isFalse) {
                pc.removeProperty(propertyKey);
            }
        }
    }

    @Procedure(name="apoc.refactor.categorize", mode=Mode.WRITE)
    @Description(value="Creates new category `NODE` values from `NODE` values in the graph with the specified `sourceKey` as one of its property keys.\nThe new category `NODE` values are then connected to the original `NODE` values with a `RELATIONSHIP` of the given type.")
    public void categorize(@Name(value="sourceKey") String sourceKey, @Name(value="type") String relationshipType, @Name(value="outgoing") Boolean outgoing, @Name(value="label") String label, @Name(value="targetKey") String targetKey, @Name(value="copiedKeys") List<String> copiedKeys, @Name(value="batchSize") long batchSize) throws ExecutionException {
        if (sourceKey == null) {
            throw new IllegalArgumentException("Invalid (null) sourceKey");
        }
        if (targetKey == null) {
            throw new IllegalArgumentException("Invalid (null) targetKey");
        }
        copiedKeys.remove(targetKey);
        if (!this.isUniqueConstraintDefinedFor(label, targetKey)) {
            throw new IllegalArgumentException("Before execute this procedure you must define an unique constraint for the label and the targetKey:\n" + String.format("CREATE CONSTRAINT FOR (n:`%s`) REQUIRE n.`%s` IS UNIQUE", label, targetKey));
        }
        ArrayList<Node> batch = null;
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (Node node : this.tx.getAllNodes()) {
            if (batch == null) {
                batch = new ArrayList<Node>((int)batchSize);
            }
            batch.add(node);
            if ((long)batch.size() != batchSize) continue;
            futures.add(this.categorizeNodes(batch, sourceKey, relationshipType, outgoing, label, targetKey, copiedKeys));
            batch = null;
        }
        if (batch != null) {
            futures.add(this.categorizeNodes(batch, sourceKey, relationshipType, outgoing, label, targetKey, copiedKeys));
        }
        for (Future future : futures) {
            Pools.force(future);
        }
    }

    @Procedure(name="apoc.refactor.deleteAndReconnect", mode=Mode.WRITE)
    @Description(value="Removes the given `NODE` values from the `PATH` and reconnects the remaining `NODE` values.")
    public Stream<GraphResult> deleteAndReconnect(@Name(value="path") Path path, @Name(value="nodes") List<Node> nodesToRemove, @Name(value="config", defaultValue="{}") Map<String, Object> config) {
        RefactorConfig refactorConfig = new RefactorConfig(config);
        ArrayList<Node> nodes = new ArrayList<Node>();
        path.nodes().forEach(nodes::add);
        Set rels = Iterables.asSet(path.relationships());
        if (!nodes.containsAll(nodesToRemove)) {
            return Stream.empty();
        }
        BiFunction<Node, Direction, Relationship> filterRel = (node, direction) -> StreamSupport.stream(node.getRelationships(direction).spliterator(), false).filter(rels::contains).findFirst().orElse(null);
        nodesToRemove.forEach(node -> {
            Relationship relationshipIn = (Relationship)filterRel.apply((Node)node, Direction.INCOMING);
            Relationship relationshipOut = (Relationship)filterRel.apply((Node)node, Direction.OUTGOING);
            if (relationshipIn == null || relationshipOut == null) {
                rels.remove(relationshipIn == null ? relationshipOut : relationshipIn);
            } else {
                RelationshipType newRelType;
                Node nodeIncoming = relationshipIn.getStartNode();
                Node nodeOutgoing = relationshipOut.getEndNode();
                HashMap<String, Object> newRelProps = new HashMap<String, Object>();
                RefactorConfig.RelationshipSelectionStrategy strategy = refactorConfig.getRelationshipSelectionStrategy();
                switch (strategy) {
                    case INCOMING: {
                        newRelType = relationshipIn.getType();
                        newRelProps.putAll(relationshipIn.getAllProperties());
                        break;
                    }
                    case OUTGOING: {
                        newRelType = relationshipOut.getType();
                        newRelProps.putAll(relationshipOut.getAllProperties());
                        break;
                    }
                    default: {
                        newRelType = RelationshipType.withName((String)(relationshipIn.getType() + "_" + relationshipOut.getType()));
                        newRelProps.putAll(relationshipIn.getAllProperties());
                    }
                }
                Relationship relCreated = nodeIncoming.createRelationshipTo(nodeOutgoing, newRelType);
                newRelProps.forEach((arg_0, arg_1) -> ((Relationship)relCreated).setProperty(arg_0, arg_1));
                if (strategy == RefactorConfig.RelationshipSelectionStrategy.MERGE) {
                    PropertiesManager.mergeProperties(relationshipOut.getAllProperties(), (Entity)relCreated, refactorConfig);
                }
                rels.add(relCreated);
                rels.removeAll(List.of(relationshipIn, relationshipOut));
            }
            this.tx.execute("WITH $node as n DETACH DELETE n", Map.of("node", node));
            nodes.remove(node);
        });
        return Stream.of(new GraphResult(nodes, List.copyOf(rels)));
    }

    private boolean isUniqueConstraintDefinedFor(String label, String key) {
        return StreamSupport.stream(this.tx.schema().getConstraints(Label.label((String)label)).spliterator(), false).anyMatch(c -> {
            if (!c.isConstraintType(ConstraintType.UNIQUENESS)) {
                return false;
            }
            return StreamSupport.stream(c.getPropertyKeys().spliterator(), false).allMatch(k -> k.equals(key));
        });
    }

    private Future<Void> categorizeNodes(List<Node> batch, String sourceKey, String relationshipType, Boolean outgoing, String label, String targetKey, List<String> copiedKeys) {
        return this.pools.processBatch(batch, this.db, (innerTx, node) -> {
            Object value = (node = Util.rebind(innerTx, node)).getProperty(sourceKey, null);
            if (value != null) {
                String nodeLabel = Util.sanitize(label);
                String key = Util.sanitize(targetKey);
                String relType = Util.sanitize(relationshipType);
                String q = "WITH $node AS n MERGE (cat:`" + nodeLabel + "` {`" + key + "`: $value}) " + (outgoing != false ? "MERGE (n)-[:`" + relType + "`]->(cat) " : "MERGE (n)<-[:`" + relType + "`]-(cat) ") + "RETURN cat";
                HashMap<String, Object> params = new HashMap<String, Object>(2);
                params.put("node", node);
                params.put("value", value);
                Result result = innerTx.execute(q, params);
                if (result.hasNext()) {
                    Node cat = (Node)result.next().get("cat");
                    for (String copiedKey : copiedKeys) {
                        Object copiedValue = node.getProperty(copiedKey, null);
                        if (copiedValue == null) continue;
                        Object catValue = cat.getProperty(copiedKey, null);
                        if (catValue == null) {
                            cat.setProperty(copiedKey, copiedValue);
                            node.removeProperty(copiedKey);
                            continue;
                        }
                        if (!copiedValue.equals(catValue)) continue;
                        node.removeProperty(copiedKey);
                    }
                }
                assert (!result.hasNext());
                result.close();
                node.removeProperty(sourceKey);
            }
        });
    }

    private void mergeNodes(Node source, Node target, RefactorConfig conf, List<String> excludeRelIds) {
        try {
            Map properties = source.getAllProperties();
            Iterable labels = source.getLabels();
            this.copyRelationships(source, target, true, conf.isCreatingNewSelfRel());
            if (conf.getMergeRelsAllowed()) {
                RefactorUtil.mergeRelationshipsWithSameTypeAndDirection(target, conf, Direction.OUTGOING, excludeRelIds);
                RefactorUtil.mergeRelationshipsWithSameTypeAndDirection(target, conf, Direction.INCOMING, excludeRelIds);
            }
            source.delete();
            labels.forEach(arg_0 -> ((Node)target).addLabel(arg_0));
            PropertiesManager.mergeProperties(properties, (Entity)target, conf);
        }
        catch (NotFoundException e) {
            this.log.warn("skipping a node for merging: " + e.getCause().getMessage());
        }
    }

    private void copyRelationships(Node source, Node target, boolean delete, boolean createNewSelfRel) {
        for (Relationship rel : source.getRelationships()) {
            this.copyRelationship(rel, source, target, createNewSelfRel);
            if (!delete) continue;
            rel.delete();
        }
    }

    private Node copyLabels(Node source, Node target) {
        for (Label label : source.getLabels()) {
            if (target.hasLabel(label)) continue;
            target.addLabel(label);
        }
        return target;
    }

    private void copyRelationship(Relationship rel, Node source, Node target, boolean createNewSelfRelf) {
        Node startNode = rel.getStartNode();
        Node endNode = rel.getEndNode();
        if (startNode.getElementId().equals(endNode.getElementId()) && !createNewSelfRelf) {
            return;
        }
        if (startNode.getElementId().equals(source.getElementId())) {
            startNode = target;
        }
        if (endNode.getElementId().equals(source.getElementId())) {
            endNode = target;
        }
        Relationship newrel = startNode.createRelationshipTo(endNode, rel.getType());
        RefactorUtil.copyProperties((Entity)rel, newrel);
    }
}

