#include <kernel.h>

#define MAX_CIRCLE_ANGLE      512
#define HALF_MAX_CIRCLE_ANGLE (MAX_CIRCLE_ANGLE/2)
#define QUARTER_MAX_CIRCLE_ANGLE (MAX_CIRCLE_ANGLE/4)
#define MASK_MAX_CIRCLE_ANGLE (MAX_CIRCLE_ANGLE - 1)
#define PI 3.14159265358979323846

/**
 * @brief SIN/COS lookup table, precalculated using:
 * 
 *  for (i = 0 ; i < MAX_CIRCLE_ANGLE ; i++) {
 *  	lut[i] = sin((double)i * PI / HALF_MAX_CIRCLE_ANGLE);
 *  }
 */
const float lut[] = {
	0.000000,	0.0122715,	0.0245412,	0.0368072,	0.0490677,	0.0613207,	0.0735646,	0.0857973,
	0.0980171,	0.110222,	0.122411,	0.134581,	0.14673,	0.158858,	0.170962,	0.18304,
	0.19509,	0.207111,	0.219101,	0.231058,	0.24298,	0.254866,	0.266713,	0.27852,
	0.290285,	0.302006,	0.313682,	0.32531,	0.33689,	0.348419,	0.359895,	0.371317,
	0.382683,	0.393992,	0.405241,	0.41643,	0.427555,	0.438616,	0.449611,	0.460539,
	0.471397,	0.482184,	0.492898,	0.503538,	0.514103,	0.52459,	0.534998,	0.545325,
	0.55557,	0.565732,	0.575808,	0.585798,	0.595699,	0.605511,	0.615232,	0.62486,
	0.634393,	0.643832,	0.653173,	0.662416,	0.671559,	0.680601,	0.689541,	0.698376,
	0.707107,	0.715731,	0.724247,	0.732654,	0.740951,	0.749136,	0.757209,	0.765167,
	0.77301,	0.780737,	0.788346,	0.795837,	0.803208,	0.810457,	0.817585,	0.824589,
	0.83147,	0.838225,	0.844854,	0.851355,	0.857729,	0.863973,	0.870087,	0.87607,
	0.881921,	0.88764,	0.893224,	0.898674,	0.903989,	0.909168,	0.91421,	0.919114,
	0.92388,	0.928506,	0.932993,	0.937339,	0.941544,	0.945607,	0.949528,	0.953306,
	0.95694,	0.960431,	0.963776,	0.966976,	0.970031,	0.97294,	0.975702,	0.978317,
	0.980785,	0.983105,	0.985278,	0.987301,	0.989177,	0.990903,	0.99248,	0.993907,
	0.995185,	0.996313,	0.99729,	0.998118,	0.998795,	0.999322,	0.999699,	0.999925,
	1.00000,	0.999925,	0.999699,	0.999322,	0.998795,	0.998118,	0.99729,	0.996313,
	0.995185,	0.993907,	0.99248,	0.990903,	0.989177,	0.987301,	0.985278,	0.983105,
	0.980785,	0.978317,	0.975702,	0.97294,	0.970031,	0.966976,	0.963776,	0.960431,
	0.95694,	0.953306,	0.949528,	0.945607,	0.941544,	0.937339,	0.932993,	0.928506,
	0.92388,	0.919114,	0.91421,	0.909168,	0.903989,	0.898674,	0.893224,	0.88764,
	0.881921,	0.87607,	0.870087,	0.863973,	0.857729,	0.851355,	0.844854,	0.838225,
	0.83147,	0.824589,	0.817585,	0.810457,	0.803208,	0.795837,	0.788346,	0.780737,
	0.77301,	0.765167,	0.757209,	0.749136,	0.740951,	0.732654,	0.724247,	0.715731,
	0.707107,	0.698376,	0.689541,	0.680601,	0.671559,	0.662416,	0.653173,	0.643831,
	0.634393,	0.624859,	0.615232,	0.605511,	0.595699,	0.585798,	0.575808,	0.565732,
	0.55557,	0.545325,	0.534998,	0.52459,	0.514103,	0.503538,	0.492898,	0.482184,
	0.471397,	0.460539,	0.449611,	0.438616,	0.427555,	0.416429,	0.405241,	0.393992,
	0.382683,	0.371317,	0.359895,	0.348419,	0.33689,	0.32531,	0.313682,	0.302006,
	0.290285,	0.27852,	0.266713,	0.254866,	0.24298,	0.231058,	0.219101,	0.207111,
	0.19509,	0.18304,	0.170962,	0.158858,	0.14673,	0.134581,	0.122411,	0.110222,
	0.0980171,	0.0857972,	0.0735645,	0.0613207,	0.0490676,	0.0368071,	0.0245411,	0.0122715,
	-8.74228e-08,	-0.0122716,	-0.0245413,	-0.0368073,	-0.0490678,	-0.0613208,	-0.0735647,	-0.0857974,
	-0.0980172,	-0.110222,	-0.122411,	-0.134581,	-0.146731,	-0.158858,	-0.170962,	-0.18304,
	-0.19509,	-0.207111,	-0.219101,	-0.231058,	-0.24298,	-0.254866,	-0.266713,	-0.27852,
	-0.290285,	-0.302006,	-0.313682,	-0.32531,	-0.33689,	-0.348419,	-0.359895,	-0.371317,
	-0.382684,	-0.393992,	-0.405241,	-0.41643,	-0.427555,	-0.438616,	-0.449611,	-0.460539,
	-0.471397,	-0.482184,	-0.492898,	-0.503538,	-0.514103,	-0.52459,	-0.534998,	-0.545325,
	-0.55557,	-0.565732,	-0.575808,	-0.585798,	-0.595699,	-0.605511,	-0.615232,	-0.62486,
	-0.634393,	-0.643832,	-0.653173,	-0.662416,	-0.671559,	-0.680601,	-0.689541,	-0.698376,
	-0.707107,	-0.715731,	-0.724247,	-0.732654,	-0.740951,	-0.749136,	-0.757209,	-0.765167,
	-0.773011,	-0.780737,	-0.788346,	-0.795837,	-0.803208,	-0.810457,	-0.817585,	-0.824589,
	-0.83147,	-0.838225,	-0.844854,	-0.851355,	-0.857729,	-0.863973,	-0.870087,	-0.87607,
	-0.881921,	-0.88764,	-0.893224,	-0.898674,	-0.903989,	-0.909168,	-0.91421,	-0.919114,
	-0.92388,	-0.928506,	-0.932993,	-0.937339,	-0.941544,	-0.945607,	-0.949528,	-0.953306,
	-0.95694,	-0.960431,	-0.963776,	-0.966977,	-0.970031,	-0.97294,	-0.975702,	-0.978317,
	-0.980785,	-0.983106,	-0.985278,	-0.987301,	-0.989177,	-0.990903,	-0.99248,	-0.993907,
	-0.995185,	-0.996313,	-0.99729,	-0.998118,	-0.998795,	-0.999322,	-0.999699,	-0.999925,
	-1.00000,	-0.999925,	-0.999699,	-0.999322,	-0.998795,	-0.998118,	-0.99729,	-0.996313,
	-0.995185,	-0.993907,	-0.99248,	-0.990903,	-0.989177,	-0.987301,	-0.985278,	-0.983105,
	-0.980785,	-0.978317,	-0.975702,	-0.97294,	-0.970031,	-0.966976,	-0.963776,	-0.960431,
	-0.95694,	-0.953306,	-0.949528,	-0.945607,	-0.941544,	-0.937339,	-0.932993,	-0.928506,
	-0.92388,	-0.919114,	-0.91421,	-0.909168,	-0.903989,	-0.898674,	-0.893224,	-0.88764,
	-0.881921,	-0.87607,	-0.870087,	-0.863973,	-0.857729,	-0.851355,	-0.844853,	-0.838225,
	-0.83147,	-0.824589,	-0.817585,	-0.810457,	-0.803207,	-0.795837,	-0.788346,	-0.780737,
	-0.77301,	-0.765167,	-0.757209,	-0.749136,	-0.740951,	-0.732654,	-0.724247,	-0.715731,
	-0.707107,	-0.698376,	-0.68954,	-0.680601,	-0.671559,	-0.662416,	-0.653173,	-0.643831,
	-0.634393,	-0.624859,	-0.615231,	-0.605511,	-0.595699,	-0.585798,	-0.575808,	-0.565732,
	-0.55557,	-0.545325,	-0.534997,	-0.52459,	-0.514103,	-0.503538,	-0.492898,	-0.482184,
	-0.471397,	-0.460539,	-0.449611,	-0.438616,	-0.427555,	-0.416429,	-0.405241,	-0.393992,
	-0.382683,	-0.371317,	-0.359895,	-0.348419,	-0.33689,	-0.32531,	-0.313682,	-0.302006,
	-0.290285,	-0.27852,	-0.266713,	-0.254865,	-0.24298,	-0.231058,	-0.219101,	-0.207111,
	-0.19509,	-0.18304,	-0.170962,	-0.158858,	-0.14673,	-0.134581,	-0.122411,	-0.110222,
	-0.098017,	-0.0857971,	-0.0735644,	-0.0613206,	-0.0490675,	-0.036807,	-0.0245411,	-0.0122714,
};

double pow(double x, double y)
{
	if (y == 0) {
		return 1;
	} else if (x == 0 && y > 0) {
		return 0;
	}

	const double base = x;
	double value = base;
	double _pow = y;
	if (y < 0) {
		_pow = y * -1;
	}

	for (double i = 1; i < _pow; i++) {
		value *= base;
	}

	if (y < 0) {
		return 1 / value;
	}
	return value;
}

double factorial(int n)
{
	int f = 1;
	for (int i = n; i > 0; i--) {
		f *= i;
	}
	return f;
}

double sin(double rads)
{
	double f = rads * HALF_MAX_CIRCLE_ANGLE / PI;
	int64_t i = (int64_t)f;
	if (i < 0) {
		return lut[(-((-i)&MASK_MAX_CIRCLE_ANGLE)) + MAX_CIRCLE_ANGLE];
	} else {
		return lut[i&MASK_MAX_CIRCLE_ANGLE];
	}
}

double cos(double rads)
{
	double f = rads * HALF_MAX_CIRCLE_ANGLE / PI;
	int64_t i = (int64_t)f;
	if (i < 0) {
		return lut[((-i) + QUARTER_MAX_CIRCLE_ANGLE)&MASK_MAX_CIRCLE_ANGLE];
	} else {
		return lut[(i + QUARTER_MAX_CIRCLE_ANGLE)&MASK_MAX_CIRCLE_ANGLE];
	}
}

double tan(double rads)
{
	if (rads == 0) {
		return 0;
	}
	return sin(rads) / cos(rads);
}
