// reference https://github.com/noeldelgado/gemini-scrollbar/blob/master/index.js

import { addResizeListener, removeResizeListener } from 'element-ui/src/utils/resize-event'
import scrollbarWidth from '@/utils/scrollbar-width'
import { toObject } from 'element-ui/src/utils/util'
import Bar from './bar'
import { random } from '@/utils/core'

/* istanbul ignore next */
export default {
  name: 'ElScrollbarFix',

  components: { Bar },

  props: {
    native: Boolean,
    vertical: {
      type: Boolean,
      required: true
    },
    minWrapHeight: String,
    wrapStyle: {},
    wrapClass: {},
    viewClass: {},
    viewStyle: {},
    noresize: Boolean, // 如果 container 尺寸不会发生变化，最好设置它可以优化性能
    tag: {
      type: String,
      default: 'div'
    }
  },

  data() {
    return {
      sizeWidth: '0',
      sizeHeight: '0',
      moveX: 0,
      moveY: 0
    };
  },

  computed: {
    wrap() {
      return this.$refs.wrap;
    }
  },

  render(h) {
    let gutter = scrollbarWidth() + 1;
    let style = this.wrapStyle;

    if (!this.vertical && !this.minWrapHeight) {
      throw new Error("property 'minWrapHeight' is required when property 'vertical' is false")
    }

    if (gutter) {
      const gutterWith = `-${gutter}px`;
      let gutterStyle = `margin-bottom: ${gutterWith}; margin-right: ${gutterWith}; `;
      if (!this.vertical) {
        if (!/^\d{1,4}px$/.test(this.minWrapHeight)) {
          throw new Error("value of property 'min-wrap-height' is illegal: " + this.minWrapHeight)
        }
        const minWrapHeight = parseInt(this.minWrapHeight.slice(0, -2))
        gutterStyle += `height: ${gutter + minWrapHeight}px;`
      }

      // FixMe 当 wrapStyle 中也指定了 height参数时的处理
      if (Array.isArray(this.wrapStyle)) {
        style = toObject(this.wrapStyle);
        style.marginRight = style.marginBottom = gutterWith;
      } else if (typeof this.wrapStyle === 'string') {
        style += gutterStyle;
      } else {
        style = gutterStyle;
      }
    }
    const view = h(this.tag, {
      class: ['el-scrollbar__view', this.viewClass],
      style: this.viewStyle,
      ref: 'resize'
    }, this.$slots.default);
    const wrap = (
      <div
        ref="wrap"
        style={ style }
        onScroll={ this.handleScroll }
        class={ [this.wrapClass, 'el-scrollbar__wrap', gutter ? '' : 'el-scrollbar__wrap--hidden-default'] }>
        { [view] }
      </div>
    );
    let nodes;

    if (!this.native) {
      nodes = [wrap]

      if (!this.vertical && parseFloat(this.sizeWidth) < 100) {
        nodes.push(<Bar
          move={this.moveX}
          size={this.sizeWidth}></Bar>)
      }

      if (this.vertical && parseFloat(this.sizeHeight) < 100) {
        nodes.push(<Bar
          vertical
          move={this.moveY}
          size={this.sizeHeight}></Bar>)
      }
    } else {
      nodes = ([
        <div
          ref="wrap"
          class={ [this.wrapClass, 'el-scrollbar__wrap'] }
          style={ style }>
          { [view] }
        </div>
      ]);
    }
    return h('div', { class: 'el-scrollbar' }, nodes);
  },

  methods: {
    handleScroll() {
      const wrap = this.wrap;

      this.moveY = ((wrap.scrollTop * 100) / wrap.clientHeight);
      this.moveX = ((wrap.scrollLeft * 100) / wrap.clientWidth);
    },

    update() {
      let heightPercentage, widthPercentage;
      const wrap = this.wrap;
      if (!wrap) return;

      heightPercentage = (wrap.clientHeight * 100 / wrap.scrollHeight);
      widthPercentage = (wrap.clientWidth * 100 / wrap.scrollWidth);

      this.sizeHeight = (heightPercentage < 100 ? heightPercentage : (100 + random())) + '%'
      this.sizeWidth = (widthPercentage < 100 ? widthPercentage : (100 + random())) + '%'

    }
  },

  mounted() {
    if (this.native) return;
    this.$nextTick(this.update);
    !this.noresize && addResizeListener(this.$refs.resize, this.update);
  },

  beforeDestroy() {
    if (this.native) return;
    !this.noresize && removeResizeListener(this.$refs.resize, this.update);
  }
};
