/*
 *  QuakeSense_Node
 * 
 *  QuakeSense is an IoT project that aims to "sense" and monitor earthquakes through a LoRa network.
 * 
 *  The main elements of this project are:
 *  1) At least one Sensor Node (Mote) consisting of:
 *     - a 3-axis accelerometer used to measure strong ground motion activity.
 *     - a LoRa radio module used to transmit an alarm and the main parameters that
 *       characterize strong motion activity to a single channel LoRa gateway
 *     - A GPS module used to get location (altitude, longitude and latitude)
 *       and date relative to the seismic event.
 *     - a development board that is used to control the accelerometer,
 *       the LoRa module and the GPS module.
 *     - a LiPo battery used to power the development board.
 *     - a solar panel used to charge the LiPo battery.
 *
 *     In this implementation, the sensor node has been made with:
 *     > X-NUCLEO-IKS01A2 motion MEMS and environmental sensor expansion board including
 *       * LSM6DSL: MEMS 3D accelerometer and 3D gyroscope
 *       * LSM303AGR: MEMS 3D accelerometer and magnetometer
 *       * LPS22HB: MEMS pressure sensor
 *       * HTS221: capacitive digital relative humidity and temperature sensor
 *     > STM32 Nucleo F401RE as the development board
 *     > Dragino LoRa/GPS Shield including
 *       - RFM95W 137 MHz to 1020 MHz low-power, long-range LoRa RF transceiver
 *       - Quectel L80 GPS module based on MTK MT3339
 *     > Seed Studio Solar Charger shield v2.2 to which are connected
 *       a 2000 mAh LiPo battery and a 1.5 W solar panel
 *
 *     In presence of a seismic event, the STM32 Nucleo board reads acceleration samples from
 *     the LSM6DSL sensor to calculate some of the main parameters that characterize the 
 *     strong-motion activity: bracketed duration and peak ground acceleration 
 *     relative to the three components of the motion (x, y and z).
 *     Then the Sensor Node sends the calculated parameters to the LoRa gateway.
 *     Each sensor node uses periodically the HTS221 and LPS22HB environmental sensors to get
 *     temperature, relative humidity and pressure and sends these values to the gateway.
 *
 *  2) A LoRa gateway that receives environmental data and earthquake alert messages
 *     from the sensor nodes and sends them to a IoT Platform.
 *
 *     In this implementation, the LoRa gateway has been made with:
 *     > B-L475E-IOT01A2 STM32L4 Discovery kit IoT node featuring:
 *       - Wi-Fi module Inventek ISM43362-M3G-L44 (802.11 b/g/n)
 *       - SPSGRF-868: Sub-GHz (868 Mhz) low-power RF module
 *       - SPBTLE-RF: Bluetooth V4.1 module
 *       - HTS221: capacitive relative humidity and temperature sensor
 *       - LSM303AGR: MEMS 3D accelerometer and MEMS 3D magnetometer
 *       - LSM6DSL: MEMS 3D accelerometer and MEMS 3D gyroscope
 *       - LSP22HB: 260-1260 hPa absolute digital output barometer
 *     > Dragino LoRa Shield which includes:
 *       - the RFM95W low-power, long-range LoRa RF transceiver based on SX1276
 *
 *  3) A cloud platform that allows the user to visualize earthquakes' parameters,
 *     alarm messages and envirnomental data in real-time.
 *     In the following implementation, AdafruitIO has been choosen as the cloud platform
 *     and data coming from the LoRa gateway is sent to Adafruit IO using the MQTT protocol.
 *
 *  This sketch uses the STM32LowPower library in order to put the STM32 MCU in deep sleep 
 *  low-power mode (STM32 stop mode) to save battery energy.
 *  The node wakes up from low-power mode using interrupts generated by the LSM6DSL accelerometer when
 *  a seismic event occurs or using RTC when the Sensor Node has to send environmental data to the gateway.
 *  To optimize battery life and reduce current consumption the LoRa radio module is put in sleep mode
 *  instead the GPS module is put in AlwaysLocate mode.
 *  Each Sensor Node uses the Solar Charger Shield as an energy harvester in fact it is connected to
 *  the solar panel whic is used to charge the LiPo battery.
 *
 *  Copyright (C) Biagio Montaruli <biagio.hkr@gmail.com>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 */

// Include sensor's library
#include <HTS221Sensor.h>
#include <LPS22HBSensor.h>
#include <LSM6DSLSensor.h>
// Include STM32LowPower library
#include <STM32LowPower.h>
// Include Adafruit GPS library for MT3339 GPS module
#include <Adafruit_GPS.h>
// include SPI library and LoRa library
#include <SPI.h>
#include <LoRa.h>

// Change these values according to your implementation
#define ENABLE_ENV_SENSORS 1
#define PRINT_ENV_DATA 1
#define SEND_ENV_DATA 1
#define DEBUG_LORA_PACKET 1
#define WAIT_LORA_ACK_EAM 1
#define WAIT_LORA_ACK_EDP 1
#define USE_EAM_REDUCED_FORMAT 0
#define PRINT_ACC_DATA 0
#define ENABLE_GPS 1
#define PRINT_GPS_DATA 1
#define GPS_MODE_ALWAYSLOCATE
//#define GPS_MODE_STANDBY
#define SERIAL_DEBUG 1
#define PRINT_EARTHQUAKE_VALUES 1

#if (PRINT_ENV_DATA == 1)  || (DEBUG_LORA_PACKET == 1) || \
    (PRINT_ACC_DATA == 1)  || (PRINT_GPS_DATA == 1)    || \
    (SERIAL_DEBUG == 1)    || (PRINT_EARTHQUAKE_VALUES == 1)
#define ENABLE_SERIAL 1
#else
#define ENABLE_SERIAL 0
#endif

#define PGHAX_THRESHOLD 50
#define PGHAY_THRESHOLD 50
#define PGVA_THRESHOLD 1120

volatile bool earthquakeDetected = false;
volatile bool motionDetected = false;

#define DEV_I2C Wire

// Use Serial2 port (USART2 on PA3, PA2) to print data in the Serial Monitor
#define SerialPort Serial
// HardwareSerial SerialPort(PA_3, PA_2);

#if ENABLE_GPS == 1
// For GPS data use Serial6 port (USART6 on PA_12 (RX) and PA_11 (TX))
HardwareSerial GPSSerial(PA_12, PA_11);

Adafruit_GPS GPS(&GPSSerial);

#define GPS_READ_TIME 2000
bool isGPSDataValid = false;

#define GPS_TIME_OFFSET 1

#if defined(GPS_MODE_ALWAYSLOCATE)
#undef GPS_MODE_STANDBY
#elif defined(GPS_MODE_STANDBY)
#undef GPS_MODE_ALWAYSLOCATE
#endif
#endif

const int csPin = 10;         // LoRa radio chip select
const int resetPin = 9;       // LoRa radio reset
const int irqPin = 2;         // change for your board; must be a hardware interrupt pin

byte msgID = 0;               // message ID (progressive number for packets sent)
byte nodeAddress = 0xAA;      // address of this device (LoRa Node)
byte gatewayAddress = 0xBB;   // address of the receiver (LoRa gateway)

#define LORA_RX_INTERVAL 5000  // 5 seconds

bool loraInit = false;
bool ackReceived = false;

// Set default LoRa mode (3):
// BW = 125 kHz; CR = 4/5; SF = 10
#define LORA_BW 125E3
#define LORA_SF 10
#define LORA_CR 5
#define LORA_TX_POWER 14

#define LORA_HEADER_LEN 4

// Objects for environmental sensors
#if ENABLE_ENV_SENSORS == 1
HTS221Sensor *HTS221_HumTemp;
LPS22HBSensor *LPS22HB_Press;

bool getEnvData = true;
#endif

// time interval during which sensor node is in low-power mode
// when the node will wake up it sends environmental data
// (temperature, humidity and pressure) to the gateway.
#define SLEEP_TIME_MIN 15  // set sleep time to 15 min 

#define SLEEP_TIME_MILLIS (SLEEP_TIME_MIN * 60000)

LSM6DSLSensor *LSM6DSL_AccGyro;

// Output data rate of LSM6DSL Accelerometer
// Available values: 13 Hz, 26 Hz, 52 Hz, 104 Hz, 208 Hz, 416 Hz,
//                   833 Hz, 1660 Hz, 3330 Hz, 6660 Hz
#define LSM6DSL_ACC_ODR LSM6DSL_ACC_GYRO_ODR_XL_104Hz
// Full scale range of LSM&DSL Accelerometer
// Available values: 2g, 4g, 8g, 16g
#define LSM6DSL_ACC_FS  LSM6DSL_ACC_GYRO_FS_XL_2g

// Maximum number of samples read by the accelerometer
// each sample is characterized by 3 values which represent
// the 3 components of acceleration
#define MAX_SAMPLES 4096

#define TIME_WINDOW 60000

// time interval to get samples from the accelerometer in millis
#define SAMPLES_PERIOD 12

typedef enum {
  LSM6DSL_X_AXIS = 0,
  LSM6DSL_Y_AXIS = 1,
  LSM6DSL_Z_AXIS = 2,
  LSM6DSL_NUM_AXIS = 3
} LSM6DSL_ACC_AXIS;

volatile uint32_t counter = 0;

const int wakeupPin = PA0;

void setup() {

#if ENABLE_SERIAL == 1
  // Initialize Serial communication at 115200 bps.
  SerialPort.begin(115200);
  // wait for the Serial port to open
  while (!SerialPort);
#endif

  // Initialize I2C bus.
  DEV_I2C.begin();

  LowPower.begin();

  // Enable interrupts for LSM6DSL accelerometer sensor
  // Remember to connect the LSM6DSL interrupt pin to wake up pin (PA0)
  LowPower.attachInterruptWakeup(wakeupPin, motionEventCB, RISING);

  // Enable and initialize sensors.
#if ENABLE_ENV_SENSORS == 1
  // Create a new object that represents the HTS221 humidity and temperature sensor and enable it
  HTS221_HumTemp = new HTS221Sensor(&DEV_I2C);
  HTS221_HumTemp->Enable();
  // Create a new object that represents the LPS22HB pressure sensor and enable it
  LPS22HB_Press = new LPS22HBSensor(&DEV_I2C);
  LPS22HB_Press->Enable();
#endif

  // Create a new object that represents the LSM6DSL accelemometer and enable it
  LSM6DSL_AccGyro = new LSM6DSLSensor(&DEV_I2C);
  LSM6DSL_AccGyro->Enable_X();

  // Enable wake up detection for LSM6DSL
  LSM6DSL_AccGyro->Enable_Wake_Up_Detection();
  // Enable single tap and double tap event detection
  LSM6DSL_AccGyro->Enable_Single_Tap_Detection(LSM6DSL_INT2_PIN);
  LSM6DSL_AccGyro->Enable_Double_Tap_Detection(LSM6DSL_INT2_PIN);
  // Enable tilt event detection
  LSM6DSL_AccGyro->Enable_Tilt_Detection(LSM6DSL_INT2_PIN);

  // Set LSM6DSL Accelerometer full scale (FS)
  LSM6DSL_AccGyro->Set_X_FS(LSM6DSL_ACC_FS);
  // Set LSM6DSL Accelerometer output data rate (ODR)
  LSM6DSL_AccGyro->Set_X_ODR(LSM6DSL_ACC_ODR);

  // set CS, reset, IRQ pins for LoRa module
  LoRa.setPins(csPin, resetPin, irqPin);

  // Set LoRa frequency band to 868 MHz
  if (!LoRa.begin(868E6)) {
  #if SERIAL_DEBUG == 1
    SerialPort.println("LoRa init failed. Check your connections.");
  #endif
  }
  else {
  #if SERIAL_DEBUG == 1
    SerialPort.println("LoRa module successfully initialized.");
  #endif

    loraInit = true;
    
    LoRa.setSpreadingFactor(LORA_SF);
    LoRa.setSignalBandwidth(LORA_BW);
    LoRa.setCodingRate4(LORA_CR);
    // set TX power of LoRa module:
    LoRa.setTxPower(LORA_TX_POWER);

    delay(250);
    LoRa.sleep();
  }

#if ENABLE_GPS == 1
  // 9600 NMEA is the default baud rate for MTK GPS's
  GPS.begin(9600);

  // Turn on RMC and GGA (fix data) including altitude
  GPS.sendCommand(PMTK_SET_NMEA_OUTPUT_RMCGGA);
  // Set the update rate for the GPS Module to 1 Hz
  GPS.sendCommand(PMTK_SET_NMEA_UPDATE_1HZ);
  // Request updates on antenna status
  GPS.sendCommand(PGCMD_ANTENNA);

  delay(20000);

  updateGPSData();

  delay(2000);

#if defined(GPS_MODE_STANDBY)
  // switch to Standby mode to save energy:
  if (GPS.setStandbyMode()) {
  #if SERIAL_DEBUG == 1
    SerialPort.println("GPS module in Standby mode");
  #endif
  }
  else {
  #if SERIAL_DEBUG == 1
    SerialPort.println("Failed to set GPS module in Standby mode");
  #endif
  }
#elif defined(GPS_MODE_ALWAYSLOCATE)
  // switch to AlwaysLocate mode to save energy
  if (GPS.setAlwaysLocateMode()) {
  #if SERIAL_DEBUG == 1
    SerialPort.println("GPS module in AlwaysLocate mode");
  #endif
  }
  else {
  #if SERIAL_DEBUG == 1
    SerialPort.println("Failed to set GPS module in AlwaysLocate mode");
  #endif
  }
#endif
#endif // ENABLE_GPS
}

void loop() {

#if ENABLE_ENV_SENSORS == 1
  if (getEnvData == true) {
    // Read humidity and temperature from HTS221 sensor.
    float hts221_humidity, hts221_temperature;
    HTS221_HumTemp->GetHumidity(&hts221_humidity);
    HTS221_HumTemp->GetTemperature(&hts221_temperature);

    // Read pressure and temperature from LPS22HB sensor.
    float lps22hb_pressure;
    LPS22HB_Press->GetPressure(&lps22hb_pressure);

    // send environmental data to the Serial monitor if PRINT_ENV_DATA is 1
  #if PRINT_ENV_DATA == 1
    SerialPort.print("Humidity[%]: ");
    SerialPort.println(hts221_humidity, 2);
    SerialPort.print("Temperature[°C]: ");
    SerialPort.println(hts221_temperature, 2);
    SerialPort.print("Pressure[mbar]: ");
    SerialPort.println(lps22hb_pressure, 2);
  #endif
    // if SEND_ENV_DATA is 1, send humidity, temperature and pressure values to the LoRa gateway
  #if SEND_ENV_DATA == 1
    if (loraInit == true) {
      // Payload format of an EDP LoRa packet (max 27 byte):
      // |  Type  |  TEMPERATURE  |  HUMIDITY  |  PRESSURE  |
      //   5 byte      7 byte         7 byte       8 byte
      String msg = "#EDPB#" + String(hts221_temperature, 2) + 
                   "#" + String(hts221_humidity, 2) +
                   "#" + String(lps22hb_pressure, 2) + "#";

      sendLoraPacket(msg);

    #if DEBUG_LORA_PACKET == 1
      SerialPort.println("LoRa packet sent:");
      SerialPort.println(msg);
    #endif    // DEBUG_LORA_PACKET

    #if WAIT_LORA_ACK_EDP == 1
      delay(500);
      LoRa.receive();
      uint32_t loraTimer = millis();
      SerialPort.println("Waiting for ACK of EDP...");
      // wait for the ACK sent by the gateway:
      // when a new LoRa message arrives, an interruput is generated and
      // the parseLoRaPacket function is called
      while ((millis() - loraTimer) <= LORA_RX_INTERVAL) {
        if (readLoRaPacket(LoRa.parsePacket()))
          break;
      }
    #endif
    }
  #endif    // SEND_ENV_DATA
  }
#endif   // ENABLE_ENV_SENSORS

#if SERIAL_DEBUG == 1
  SerialPort.println("Starting Deep Sleep low-power mode (STM32 Stop mode)...");
  delay(100);
#endif

  LowPower.deepSleep(SLEEP_TIME_MILLIS);

  if (motionDetected == true) {
  #if ENABLE_ENV_SENSORS == 1
    getEnvData = false;
  #endif
    earthquakeDetected = false;
    motionDetected = false;
  }
  else {
  #if SERIAL_DEBUG == 1
    delay(100);
    SerialPort.println("Waking up the STM32 MCU from Deep Sleep mode...");
  #endif
  #if ENABLE_ENV_SENSORS == 1
    getEnvData = true;
  #endif
  }
}

void motionEventCB() {
  LSM6DSL_Event_Status_t status;
  LSM6DSL_AccGyro->Get_Event_Status(&status);

  motionDetected = true;

  if (status.WakeUpStatus && !status.TapStatus && !status.DoubleTapStatus && !status.TiltStatus) {
  #if SERIAL_DEBUG == 1
    SerialPort.println("WakeUP event -> Motion Detected!");
    SerialPort.println("Waking up from Deep Sleep mode...");
  #endif
    earthquakeDetection();
  }
}

void earthquakeDetection() {

  int32_t lsm6dsl_acc[MAX_SAMPLES][LSM6DSL_NUM_AXIS];
  uint32_t times[MAX_SAMPLES];

  uint32_t n;
  bool startFlags[LSM6DSL_NUM_AXIS] = {false, false, false};
  uint32_t timer = millis();

  for (n = 0; ((n < MAX_SAMPLES) && ((millis() - timer) < TIME_WINDOW)); n++) {
    LSM6DSL_AccGyro->Get_X_Axes(lsm6dsl_acc[n]);

    if (n == 0) {
      if ( (abs(lsm6dsl_acc[n][LSM6DSL_X_AXIS]) < PGHAX_THRESHOLD) && \
           (abs(lsm6dsl_acc[n][LSM6DSL_Y_AXIS]) < PGHAY_THRESHOLD) && \
           (abs(lsm6dsl_acc[n][LSM6DSL_Z_AXIS]) < PGVA_THRESHOLD))  {
        break;
      }
    }
    times[n] = millis();
    delay(SAMPLES_PERIOD);
  }

  uint32_t totalDuration = millis() - timer;

  if (n > 0) {

    earthquakeDetected = true;

    #if ENABLE_GPS == 1
    #if defined(GPS_MODE_STANDBY)
    // switch from Standby to Full On mode
    if (GPS.wakeupStandby()) {
    #elif defined(GPS_MODE_ALWAYSLOCATE)
    // switch from AlwaysLocate mode to Full On mode
    if (GPS.wakeupAlwaysLocate()) {
    #endif
    #if SERIAL_DEBUG == 1
      SerialPort.println("GPS module in Full On mode");
    #endif
    }
    else {
    #if SERIAL_DEBUG == 1
      SerialPort.println("Failed to set GPS module in Full On mode");
    #endif
    }
    #endif

    uint32_t durations[LSM6DSL_NUM_AXIS][2];

    uint32_t pghax = 0, pghay = 0, pgva = 0;

    // calculate peak ground acceleration (PGA) and bracketed duration:
    for (uint16_t i = 0; i < n; i++) {
      // find the time in milliseconds when the horizontal and vertical
      // components of the acceleration records excedeed the threshold
      if (abs(lsm6dsl_acc[i][LSM6DSL_X_AXIS]) >= PGHAX_THRESHOLD) {
        if (startFlags[LSM6DSL_X_AXIS] == false) {
          durations[LSM6DSL_X_AXIS][0] = times[i];
          startFlags[LSM6DSL_X_AXIS] = true;
        }
        durations[LSM6DSL_X_AXIS][1] = times[i];
      }

      if (abs(lsm6dsl_acc[i][LSM6DSL_Y_AXIS]) >= PGHAY_THRESHOLD) {
        if (startFlags[LSM6DSL_Y_AXIS] == false) {
          durations[LSM6DSL_Y_AXIS][0] = times[i];
          startFlags[LSM6DSL_Y_AXIS] = true;
        }
        durations[LSM6DSL_Y_AXIS][1] = times[i];
      }

      if (abs(lsm6dsl_acc[i][LSM6DSL_Z_AXIS]) >= PGVA_THRESHOLD) {
        if (startFlags[LSM6DSL_Z_AXIS] == false) {
          durations[LSM6DSL_Z_AXIS][0] = times[i];
          startFlags[LSM6DSL_Z_AXIS] = true;
        }
        durations[LSM6DSL_Z_AXIS][1] = times[i];
      }

      // find the peak ground horizontal acceleration (x and y axis)
      if (pghax < abs(lsm6dsl_acc[i][LSM6DSL_X_AXIS])) {
        pghax = abs(lsm6dsl_acc[i][LSM6DSL_X_AXIS]);
      }
      if (pghay < abs(lsm6dsl_acc[i][LSM6DSL_Y_AXIS])) {
        pghay = abs(lsm6dsl_acc[i][LSM6DSL_Y_AXIS]);
      }

      // find the peak ground vertical acceleration (z axis)
      if (pgva < abs(lsm6dsl_acc[i][LSM6DSL_Z_AXIS])) {
        pgva = abs(lsm6dsl_acc[i][LSM6DSL_Z_AXIS]);
      }
    }

    uint32_t bracketedDuration[LSM6DSL_NUM_AXIS];

    bracketedDuration[LSM6DSL_X_AXIS] = durations[LSM6DSL_X_AXIS][1] - durations[LSM6DSL_X_AXIS][0];
    bracketedDuration[LSM6DSL_Y_AXIS] = durations[LSM6DSL_Y_AXIS][1] - durations[LSM6DSL_Y_AXIS][0];
    bracketedDuration[LSM6DSL_Z_AXIS] = durations[LSM6DSL_Z_AXIS][1] - durations[LSM6DSL_Z_AXIS][0];

    uint32_t averageBracketedDuration = (bracketedDuration[LSM6DSL_X_AXIS] + bracketedDuration[LSM6DSL_Y_AXIS] + bracketedDuration[LSM6DSL_Z_AXIS]) / LSM6DSL_NUM_AXIS;

  #if PRINT_EARTHQUAKE_VALUES == 1
    SerialPort.print("Total duration (Time window): ");
    SerialPort.println(totalDuration);
    SerialPort.print("Bracketed duration X-AXIS: ");
    SerialPort.println(bracketedDuration[LSM6DSL_X_AXIS]);
    SerialPort.print("Bracketed duration Y-AXIS: ");
    SerialPort.println(bracketedDuration[LSM6DSL_Y_AXIS]);
    SerialPort.print("Bracketed duration Z-AXIS: ");
    SerialPort.println(bracketedDuration[LSM6DSL_Z_AXIS]);
    SerialPort.print("Average bracketed duration: ");
    SerialPort.println(averageBracketedDuration);
  #endif

  #if PRINT_ACC_DATA == 1
    for (uint16_t i = 0; i < n; i++) {
      SerialPort.print(lsm6dsl_acc[i][LSM6DSL_X_AXIS]);
      SerialPort.print(",");
      SerialPort.print(lsm6dsl_acc[i][LSM6DSL_Y_AXIS]);
      SerialPort.print(",");
      SerialPort.println(lsm6dsl_acc[i][LSM6DSL_Z_AXIS]);
      delay(20);
    }
  #endif

  #if PRINT_EARTHQUAKE_VALUES == 1
    SerialPort.print("Earthquake duration: ");
    SerialPort.print(averageBracketedDuration);
    SerialPort.println(" msecs");

    SerialPort.print("PGHA (x-axis): ");
    SerialPort.println(pghax);
    SerialPort.print("PGHA (y-axis): ");
    SerialPort.println(pghay);
    SerialPort.print("PGVA (z-axis): ");
    SerialPort.println(pgva);

    SerialPort.print("Got ");
    SerialPort.print(n);
    SerialPort.println(" samples.");
  #endif

  #if ENABLE_GPS == 1
    updateGPSData();
  #endif

    if (loraInit) {
      // Payload format of an EAM LoRa packet:
      // 1) EAMF: FULL FORMAT (max 73 byte)
      //    |  Type  |  PGHAX  |  PGHAY  |  PGVA  |  Bracketed duration  |  Latitude  |  Longitude |  Altitude  |  Date (DD/MM/YYYY)  |  Time (HH:MM:SS)  |
      //      5 byte    5 byte   5 byte    5 byte          6 byte            9 byte       9 byte       8 byte              11 byte            10 byte
      // 2) EAMR: REDUCED FORMAT FORMAT (LoRaWAN-compliant: max 45 byte)
      //    |  Type  |  PGHAX  |  PGHAY  |  PGVA  |  Bracketed duration  |  Latitude  |  Longitude |
      //      5 byte    5 byte   5 byte    5 byte          6 byte            9 byte       10 byte
      // 3) EAMB: BASE FORMAT (max 27 byte)
      //    |  Type  |  PGHAX  |  PGHAY  |  PGVA  |  Bracketed duration  |
      //      5 byte    5 byte   5 byte    5 byte          7 byte
    #if USE_EAM_REDUCED_FORMAT == 1
      String msg = "#EAMR#";
    #else
      String msg = "#EAMF#";
    #endif
      msg += (String(pghax) + "#");
      msg += (String(pghay) + "#");
      msg += (String(pgva) + "#");
      msg += (String(averageBracketedDuration) + "#");

    #if ENABLE_GPS == 1
      if (isGPSDataValid) {
        msg += (String(GPS.getLatitude(), 2) + String(GPS.getLatCardinalDir()) + "#");
        msg += (String(GPS.getLongitude(), 2) + String(GPS.getLonCardinalDir()) + "#");
      #if USE_EAM_REDUCED_FORMAT == 0
        msg += (String(GPS.getAltitude(), 2) + "#");
        if (GPS.getDay() < 10)
          msg += "0";
        msg += (String(GPS.getDay()) + "/");
        if (GPS.getMonth() < 10)
          msg += "0";
        msg += (String(GPS.getMonth()) + "/20");
        if (GPS.getYear() < 10)
          msg += "0";
        msg += String(GPS.getYear());
        msg += "#";
        if ((GPS.getHour() + GPS_TIME_OFFSET) < 10)
          msg += "0";
        msg += (String(GPS.getHour() + GPS_TIME_OFFSET) + ":");
        if (GPS.getMinutes() < 10)
          msg += "0";
        msg += (String(GPS.getMinutes()) +  ":");
        if (GPS.getSeconds() < 10)
          msg += "0";
        msg += String(GPS.getSeconds());
        msg += "#";
      #endif
      }
      else {
        msg.setCharAt(4, 'B');
      }
    #else
      msg.setCharAt(4, 'B');
    #endif

      sendLoraPacket(msg);

    #if DEBUG_LORA_PACKET == 1
      SerialPort.println("Lora packet sent:");
      SerialPort.println(msg);
    #endif

    #if WAIT_LORA_ACK_EAM == 1
      delay(500);
      LoRa.receive();
      uint32_t loraTimer = millis();
      SerialPort.println("Waiting for ACK of EAM...");
      // wait for the ACK sent by the gateway:
      // when a new LoRa message arrives, an interruput is generated and
      // the parseLoRaPacket function is called
      while ((millis() - loraTimer) <= LORA_RX_INTERVAL) {
        if (readLoRaPacket(LoRa.parsePacket()))
          break;
      }
    #endif

      LoRa.sleep();
    }

  #if ENABLE_GPS == 1
  #if defined(GPS_MODE_STANDBY)
    // switch to standby mode to save energy:
    if (GPS.setStandbyMode()) {
    #if SERIAL_DEBUG == 1
      SerialPort.println("GPS module in Standby mode");
    #endif
    }
    else {
    #if SERIAL_DEBUG == 1
      SerialPort.println("Failed to set GPS module in Standby mode");
    #endif
    }
  #elif defined(GPS_MODE_ALWAYSLOCATE)
    // switch to AlwaysLocate mode to save energy
    if (GPS.setAlwaysLocateMode()) {
    #if SERIAL_DEBUG == 1
      SerialPort.println("GPS module in AlwaysLocate mode");
    #endif
    }
    else {
    #if SERIAL_DEBUG == 1
      SerialPort.println("Failed to set GPS module in AlwaysLocate mode");
    #endif
    }
  #endif
  #endif // ENABLE_GPS
  }
}

// Function that builds a LoRa packet and sends it.
// The format of the LoRa packet is:
// | Destination |   Sender   |  Message ID  | Payload Length |  Payload data |  Checksum  |
//    1 byte         1 byte       1 byte          1 byte           N byte         1 byte
void sendLoraPacket(String data) {
  uint16_t payloadLen = data.length();
  
  LoRa.beginPacket();          // start packet
  LoRa.write(gatewayAddress);  // add destination address
  LoRa.write(nodeAddress);     // add sender address
  LoRa.write(msgID);        // add message ID
  LoRa.write(data.length());   // add payload length
  
  byte buf[payloadLen + LORA_HEADER_LEN + 1] = {0};
  buf[0] = gatewayAddress;
  buf[1] = nodeAddress;
  buf[2] = msgID;
  buf[3] = payloadLen;
  data.getBytes(buf + LORA_HEADER_LEN, payloadLen+1);
  byte checksum = getChecksum(buf, payloadLen + LORA_HEADER_LEN + 1);
  
  LoRa.print(data);            // add payload
  LoRa.write(checksum);        // add checksum
  LoRa.endPacket();            // finish packet and send it
}

// Function that computes the packet checksum.
byte getChecksum(byte *buffer, uint16_t length) {
  byte checksum = 0;
  
  for (uint32_t i = 0; i < length; i++) {
    if ((checksum + buffer[i]) > 255) {
      checksum += (255 - buffer[i]);
    }
    else {
      checksum += buffer[i];
    }
  }

  return checksum;
}


#if ENABLE_GPS == 1
// Function that reads data from the GPS module, parse it to get:
// latitude, longitude, altitude, time, date and other GPS values.
void updateGPSData() {
  uint32_t gpsTimer = millis();

  bool NMEAparsed = false;
  while ((millis() - gpsTimer) < GPS_READ_TIME) {
    // read data from the GPS
    char c = GPS.read();

  #if (PRINT_GPS_DATA == 1) && (SERIAL_DEBUG == 1)
    // print GPS data in the Serial monitor
    if (c) SerialPort.print(c);
  #endif

    if (GPS.newNMEAreceived()) {
      // if a sentence is received, we can check the checksum and parse it...
      // we can fail to parse a sentence in which case we should just wait for another
      // this also sets the newNMEAreceived() flag to false
    #if (PRINT_GPS_DATA == 1) && (SERIAL_DEBUG == 1)
      SerialPort.println(GPS.lastNMEA());
    #endif
      if (GPS.parse(GPS.lastNMEA())) {
        NMEAparsed = true;
        break;
      }
    }
  }

  if (NMEAparsed == true) {

  #if PRINT_GPS_DATA == 1
    SerialPort.print("\nTime: ");
  #if GPS_FIX_TIME == 1
    if ((GPS.getHour() + 1) < 10)
      SerialPort.print("0");
    SerialPort.print((GPS.getHour() + 1), DEC);
  #else
    if (GPS.getHour() < 10)
      SerialPort.print("0");
    SerialPort.print(GPS.getHour(), DEC);
  #endif
    SerialPort.print(':');
    if (GPS.getMinutes() < 10)
      Serial.print("0");
    SerialPort.print(GPS.getMinutes(), DEC);
    SerialPort.print(':');
    if (GPS.getSeconds() < 10)
      Serial.print("0");
    SerialPort.print(GPS.getSeconds(), DEC);
    SerialPort.print('.');
    if (GPS.getMilliseconds() < 10)
      Serial.print("0");
    SerialPort.println(GPS.getMilliseconds());
    SerialPort.print("Date: ");
    if (GPS.getDay() < 10)
      Serial.print("0");
    SerialPort.print(GPS.getDay(), DEC); SerialPort.print('/');
    if (GPS.getMonth() < 10)
      Serial.print("0");
    SerialPort.print(GPS.getMonth(), DEC); SerialPort.print("/20");
    if (GPS.getYear() < 10)
      Serial.print("0");
    SerialPort.println(GPS.getYear(), DEC);
    SerialPort.print("Fix: "); SerialPort.print(GPS.isFixed());
    SerialPort.print(" quality: "); SerialPort.println(GPS.getQuality());
  #endif

    if (GPS.isFixed() || GPS.getQuality()) {

    #if PRINT_GPS_DATA == 1
      SerialPort.print("Location: ");
      SerialPort.print(GPS.getLatitude(), 4); SerialPort.print(GPS.getLatCardinalDir());
      SerialPort.print(", ");
      SerialPort.print(GPS.getLongitude(), 4); SerialPort.println(GPS.getLonCardinalDir());
      SerialPort.print("Speed (knots): "); SerialPort.println(GPS.getSpeed());
      SerialPort.print("Angle: "); SerialPort.println(GPS.getAngle());
      SerialPort.print("Altitude: "); SerialPort.println(GPS.getAltitude());
      SerialPort.print("Satellites: "); SerialPort.println(GPS.getSatellites());
    #endif

      isGPSDataValid = true;
    }
    else {
      isGPSDataValid = false;
    }
  }
}
#endif

#if (WAIT_LORA_ACK_EAM == 1) || (WAIT_LORA_ACK_EDP == 1)
// Function that parse a LoRa packet sent by a LoRa Node
// The format of the LoRa packet is:
// |  Receiver Addr  |  Sender Addr  |  Message ID  |  Payload Length  |  Payload data  |  Checksum  |
//       1 byte           1 byte          1 byte          1 byte            N byte          1 byte
bool readLoRaPacket(int packetSize) {

  if (packetSize == 0)
    return false;

  ackReceived = false;

  // read bytes from the packet header:
  byte receiverAddr = LoRa.read();  // receiver address
  byte senderAddr = LoRa.read();    // sender address
  byte msgID = LoRa.read();         // incoming message ID
  byte payloadLen = LoRa.read();    // incoming message length

  String data = "";

  // read payload of the packet
  for (uint16_t i = 0; LoRa.available() && i < payloadLen; i++) {
    data += (char)LoRa.read();
  }

  byte checksum = LoRa.read();

  byte buf[payloadLen + LORA_HEADER_LEN + 1] = {0};
  buf[0] = receiverAddr;
  buf[1] = senderAddr;
  buf[2] = msgID;
  buf[3] = payloadLen;
  data.getBytes(buf + LORA_HEADER_LEN, payloadLen+1);
  byte computedChecksum = getChecksum(buf, payloadLen + LORA_HEADER_LEN + 1);

  // discard the packet if the checksum is not valid or if the receiver isn't this device
  if (checksum != computedChecksum) {
  #if DEBUG_LORA_PACKET == 1
    SerialPort.println("LoRa Error: bad checksum");
    SerialPort.print("Checksum received: ");
    SerialPort.println(checksum);
    SerialPort.print("Checksum computed: ");
    SerialPort.println(computedChecksum);
    SerialPort.println("Payload received:\n" + data);
  #endif
    return false;
  }
  else if (receiverAddr != nodeAddress) {
  #if DEBUG_LORA_PACKET == 1
    SerialPort.println("LoRa Error: unknown address");
  #endif
    return false;
  }
  else {
  #if DEBUG_LORA_PACKET == 1
    // if message is for this device, print details and get the msg fields:
    SerialPort.println("Packet Info:");
    SerialPort.println("Received from: 0x" + String(senderAddr, HEX));
    SerialPort.println("Sent to: 0x" + String(receiverAddr, HEX));
    SerialPort.println("Message ID: " + String(msgID));
    SerialPort.println("Message length: " + String(payloadLen));
    SerialPort.println("RSSI: " + String(LoRa.packetRssi()));
    SerialPort.println("SNR: " + String(LoRa.packetSnr()));
    SerialPort.println("Message Data:");
    SerialPort.println(data);
  #endif
    if (data.equals("#ACK")) {
    #if DEBUG_LORA_PACKET == 1
      SerialPort.println("ACK received: the LoRa gateway has successfully sent the message to Adafruit IO");
    #endif
      ackReceived = true;
      return true;
    }
    else {
      return false;
    }
  }
}
#endif
