/*
 * μlogger
 *
 * Copyright(C) 2019 Bartek Fabiszewski (www.fabiszewski.net)
 *
 * This is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */


import uPosition from '../../src/position.js';
import uPositionSet from '../../src/positionset.js';
import uTrack from '../../src/track.js';
import uUser from '../../src/user.js';

export default class TrackFactory {

  /**
   * @template T
   * @param {(number|uPosition[])=} p Positions array or number of positions to be autogenerated
   * @param {T} type
   * @param {Object} params
   * @return {T}
   */
  static getSet(p = 2, type, params) {
    let track;
    if (type === uTrack) {
      track = new uTrack(params.id, params.name, params.user);
    } else {
      track = new uPositionSet();
    }
    if (Array.isArray(p)) {
      track.fromJson(p, true);
    } else {
      const length = p;
      if (length) {
        const positions = [];
        let lat = 21.01;
        let lon = 52.23;
        for (let i = 0; i < length; i++) {
          positions.push(this.getPosition({ id: i + 1, latitude: lat, longitude: lon }));
          lat += 0.5;
          lon += 0.5;
        }
        track.fromJson(positions, true);
      }
    }
    return track;
  }

  /**
   * @param {(number|uPosition[])=} p Positions array or number of positions to be autogenerated
   * @param {{ id: number, name: string, user: uUser }=} params
   * @return {uTrack}
   */
  static getTrack(p = 2, params) {
    params = params || {};
    params.id = params.id || 1;
    params.name = params.name || 'test track';
    params.user = params.user || new uUser(1, 'testUser');
    return this.getSet(p, uTrack, params);
  }

  /**
   * @param {(number|uPosition[])=} p Positions array or number of positions to be autogenerated
   * @param {uPosition[]=} positions
   * @return {uPositionSet}
   */
  static getPositionSet(p = 2, positions) {
    return this.getSet(p, uPositionSet, positions);
  }

  /**
   * @param {Object=} params
   * @return {uPosition}
   */
  static getPosition(params) {
    params = params || {};
    const position = new uPosition();
    position.id = params.id || 1;
    position.latitude = params.latitude || 52.23;
    position.longitude = params.longitude || 21.01;
    position.altitude = params.altitude || null;
    position.speed = params.speed || null;
    position.bearing = params.bearing || null;
    position.timestamp = params.timestamp || 1;
    position.accuracy = params.accuracy || null;
    position.provider = params.provider || null;
    position.comment = params.comment || null;
    position.image = params.image || null;
    position.username = params.username || 'testUser';
    position.trackid = params.trackid || 1;
    position.trackname = params.trackname || 'test track';
    position.meters = params.meters || 0;
    position.seconds = params.seconds || 0;
    return position;
  }
}
