<?php

/**
 * PHP library used for interacting with Fortigate firewall (FortiOS) configuration API (Configuration).
 * @author Benjamin Clerc <contact@benjamin-clerc.com>
 * @copyright Copyright (c) 2021, Benjamin Clerc.
 * @license MIT
 * @link https://github.com/benclerc/Fortinet-FortiOSAPI
 */

namespace Fortinet\FortiOSAPI;

use stdClass;

/**
 * FortiOS API Configuration
 * @link https://fndn.fortinet.net/ Fortinet developer ressource center.
 */
class Configuration extends FortiOSAPI
{
	/**
	 * Select all entries in a CLI table.
	 *  Access Group: loggrp
	 * @route  GET /alertemail/setting
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAlertemailSetting(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/alertemail/setting', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: loggrp
	 * @route  PUT /alertemail/setting
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAlertemailSetting(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/alertemail/setting', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /antivirus/heuristic
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAntivirusHeuristic(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/antivirus/heuristic', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /antivirus/heuristic
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAntivirusHeuristic(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/antivirus/heuristic', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /antivirus/profile/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAntivirusProfile(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/antivirus/profile', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /antivirus/profile/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAntivirusProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/antivirus/profile', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /antivirus/profile/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteAntivirusProfile(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/antivirus/profile', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /antivirus/profile
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAntivirusProfile(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/antivirus/profile', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /antivirus/profile
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addAntivirusProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/antivirus/profile', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /antivirus/quarantine
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAntivirusQuarantine(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/antivirus/quarantine', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /antivirus/quarantine
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAntivirusQuarantine(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/antivirus/quarantine', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /antivirus/settings
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAntivirusSettings(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/antivirus/settings', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /antivirus/settings
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAntivirusSettings(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/antivirus/settings', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/custom/{tag}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $tag mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getApplicationCustom(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $tag
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/custom', $tag, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /application/custom/{tag}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $tag mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateApplicationCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $tag
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/application/custom', $tag, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /application/custom/{tag}
	 * @param  $tag mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteApplicationCustom(string $tag, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/application/custom', $tag, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/custom
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllApplicationCustom(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/custom', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /application/custom
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addApplicationCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/application/custom', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/group/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getApplicationGroup(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/group', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /application/group/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateApplicationGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/application/group', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /application/group/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteApplicationGroup(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/application/group', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/group
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllApplicationGroup(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/group', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /application/group
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addApplicationGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/application/group', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/list/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getApplicationList(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/list', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /application/list/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateApplicationList(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/application/list', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /application/list/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteApplicationList(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/application/list', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/list
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllApplicationList(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/list', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /application/list
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addApplicationList(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/application/list', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/name/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getApplicationName(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/name', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /application/name/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateApplicationName(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/application/name', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /application/name/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteApplicationName(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/application/name', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/name
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllApplicationName(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/name', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /application/name
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addApplicationName(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/application/name', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/rule-settings/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getApplicationRuleSettings(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/rule-settings', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /application/rule-settings/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateApplicationRuleSettings(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/application/rule-settings', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /application/rule-settings/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteApplicationRuleSettings(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/application/rule-settings', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /application/rule-settings
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllApplicationRuleSettings(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/application/rule-settings', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /application/rule-settings
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addApplicationRuleSettings(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/application/rule-settings', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: authgrp
	 * @route  GET /authentication/rule/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAuthenticationRule(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/authentication/rule', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: authgrp
	 * @route  PUT /authentication/rule/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAuthenticationRule(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/authentication/rule', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: authgrp
	 * @route  DELETE /authentication/rule/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteAuthenticationRule(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/authentication/rule', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: authgrp
	 * @route  GET /authentication/rule
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAuthenticationRule(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/authentication/rule', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: authgrp
	 * @route  POST /authentication/rule
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addAuthenticationRule(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/authentication/rule', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: authgrp
	 * @route  GET /authentication/scheme/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAuthenticationScheme(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/authentication/scheme', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: authgrp
	 * @route  PUT /authentication/scheme/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAuthenticationScheme(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/authentication/scheme', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: authgrp
	 * @route  DELETE /authentication/scheme/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteAuthenticationScheme(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/authentication/scheme', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: authgrp
	 * @route  GET /authentication/scheme
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAuthenticationScheme(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/authentication/scheme', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: authgrp
	 * @route  POST /authentication/scheme
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addAuthenticationScheme(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/authentication/scheme', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: authgrp
	 * @route  GET /authentication/setting
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllAuthenticationSetting(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/authentication/setting', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: authgrp
	 * @route  PUT /authentication/setting
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateAuthenticationSetting(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/authentication/setting', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/ca/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getCertificateCa(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/ca', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: vpngrp
	 * @route  PUT /certificate/ca/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateCertificateCa(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/certificate/ca', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: vpngrp
	 * @route  DELETE /certificate/ca/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteCertificateCa(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/certificate/ca', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/ca
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllCertificateCa(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/ca', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: vpngrp
	 * @route  POST /certificate/ca
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addCertificateCa(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/certificate/ca', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/crl/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getCertificateCrl(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/crl', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: vpngrp
	 * @route  PUT /certificate/crl/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateCertificateCrl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/certificate/crl', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: vpngrp
	 * @route  DELETE /certificate/crl/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteCertificateCrl(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/certificate/crl', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/crl
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllCertificateCrl(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/crl', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: vpngrp
	 * @route  POST /certificate/crl
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addCertificateCrl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/certificate/crl', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/local/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getCertificateLocal(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/local', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: vpngrp
	 * @route  PUT /certificate/local/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateCertificateLocal(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/certificate/local', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: vpngrp
	 * @route  DELETE /certificate/local/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteCertificateLocal(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/certificate/local', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/local
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllCertificateLocal(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/local', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: vpngrp
	 * @route  POST /certificate/local
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addCertificateLocal(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/certificate/local', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/remote/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getCertificateRemote(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/remote', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: vpngrp
	 * @route  PUT /certificate/remote/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateCertificateRemote(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/certificate/remote', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: vpngrp
	 * @route  DELETE /certificate/remote/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteCertificateRemote(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/certificate/remote', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: vpngrp
	 * @route  GET /certificate/remote
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllCertificateRemote(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/certificate/remote', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: vpngrp
	 * @route  POST /certificate/remote
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addCertificateRemote(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/certificate/remote', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: netgrp
	 * @route  GET /credential-store/domain-controller/{server-name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $serverName mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getCredentialStoreDomainController(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $serverName
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/credential-store/domain-controller', $serverName, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: netgrp
	 * @route  PUT /credential-store/domain-controller/{server-name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $serverName mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateCredentialStoreDomainController(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $serverName
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/credential-store/domain-controller', $serverName, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: netgrp
	 * @route  DELETE /credential-store/domain-controller/{server-name}
	 * @param  $serverName mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteCredentialStoreDomainController(string $serverName, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/credential-store/domain-controller', $serverName, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: netgrp
	 * @route  GET /credential-store/domain-controller
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllCredentialStoreDomainController(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/credential-store/domain-controller', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: netgrp
	 * @route  POST /credential-store/domain-controller
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addCredentialStoreDomainController(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/credential-store/domain-controller', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/filepattern/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getDlpFilepattern(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/filepattern', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /dlp/filepattern/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateDlpFilepattern(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/dlp/filepattern', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /dlp/filepattern/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteDlpFilepattern(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/dlp/filepattern', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/filepattern
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllDlpFilepattern(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/filepattern', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /dlp/filepattern
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addDlpFilepattern(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/dlp/filepattern', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/fp-doc-source/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getDlpFpDocSource(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/fp-doc-source', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /dlp/fp-doc-source/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateDlpFpDocSource(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/dlp/fp-doc-source', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /dlp/fp-doc-source/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteDlpFpDocSource(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/dlp/fp-doc-source', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/fp-doc-source
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllDlpFpDocSource(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/fp-doc-source', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /dlp/fp-doc-source
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addDlpFpDocSource(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/dlp/fp-doc-source', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/sensitivity/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getDlpSensitivity(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/sensitivity', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /dlp/sensitivity/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateDlpSensitivity(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/dlp/sensitivity', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /dlp/sensitivity/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteDlpSensitivity(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/dlp/sensitivity', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/sensitivity
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllDlpSensitivity(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/sensitivity', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /dlp/sensitivity
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addDlpSensitivity(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/dlp/sensitivity', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/sensor/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getDlpSensor(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/sensor', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /dlp/sensor/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateDlpSensor(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/dlp/sensor', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /dlp/sensor/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteDlpSensor(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/dlp/sensor', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/sensor
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllDlpSensor(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/sensor', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /dlp/sensor
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addDlpSensor(stdClass $body, array $vdom = null, string $action = null, string $nkey = null) : stdClass
	{
		return $this->curlRequest('POST', '/cmdb/dlp/sensor', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dlp/settings
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllDlpSettings(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dlp/settings', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /dlp/settings
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateDlpSettings(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/dlp/settings', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dnsfilter/domain-filter/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getDnsfilterDomainFilter(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dnsfilter/domain-filter', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /dnsfilter/domain-filter/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateDnsfilterDomainFilter(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/dnsfilter/domain-filter', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /dnsfilter/domain-filter/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteDnsfilterDomainFilter(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/dnsfilter/domain-filter', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dnsfilter/domain-filter
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllDnsfilterDomainFilter(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dnsfilter/domain-filter', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /dnsfilter/domain-filter
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addDnsfilterDomainFilter(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/dnsfilter/domain-filter', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dnsfilter/profile/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getDnsfilterProfile(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dnsfilter/profile', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /dnsfilter/profile/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateDnsfilterProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/dnsfilter/profile', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /dnsfilter/profile/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteDnsfilterProfile(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/dnsfilter/profile', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /dnsfilter/profile
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllDnsfilterProfile(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/dnsfilter/profile', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /dnsfilter/profile
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addDnsfilterProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/dnsfilter/profile', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/bwl/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getEmailfilterBwl(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/bwl', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/bwl/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterBwl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/bwl', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /emailfilter/bwl/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteEmailfilterBwl(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/emailfilter/bwl', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/bwl
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterBwl(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/bwl', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /emailfilter/bwl
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addEmailfilterBwl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/emailfilter/bwl', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/bword/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getEmailfilterBword(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/bword', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/bword/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterBword(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/bword', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /emailfilter/bword/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteEmailfilterBword(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/emailfilter/bword', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/bword
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterBword(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/bword', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /emailfilter/bword
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addEmailfilterBword(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/emailfilter/bword', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/dnsbl/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getEmailfilterDnsbl(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/dnsbl', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/dnsbl/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterDnsbl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/dnsbl', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /emailfilter/dnsbl/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteEmailfilterDnsbl(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/emailfilter/dnsbl', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/dnsbl
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterDnsbl(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/dnsbl', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /emailfilter/dnsbl
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addEmailfilterDnsbl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/emailfilter/dnsbl', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/fortishield
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterFortishield(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/fortishield', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/fortishield
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterFortishield(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/fortishield', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/iptrust/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getEmailfilterIptrust(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/iptrust', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/iptrust/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterIptrust(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/iptrust', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /emailfilter/iptrust/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteEmailfilterIptrust(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/emailfilter/iptrust', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/iptrust
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterIptrust(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/iptrust', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /emailfilter/iptrust
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addEmailfilterIptrust(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/emailfilter/iptrust', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/mheader/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getEmailfilterMheader(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/mheader', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/mheader/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterMheader(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/mheader', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /emailfilter/mheader/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteEmailfilterMheader(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/emailfilter/mheader', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/mheader
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterMheader(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/mheader', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /emailfilter/mheader
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addEmailfilterMheader(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/emailfilter/mheader', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/options
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterOptions(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/options', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/options
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterOptions(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/options', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/profile/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getEmailfilterProfile(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/profile', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /emailfilter/profile/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEmailfilterProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/emailfilter/profile', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /emailfilter/profile/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteEmailfilterProfile(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/emailfilter/profile', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /emailfilter/profile
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEmailfilterProfile(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/emailfilter/profile', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /emailfilter/profile
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addEmailfilterProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/emailfilter/profile', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /endpoint-control/fctems/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getEndpointControlFctems(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/endpoint-control/fctems', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /endpoint-control/fctems/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateEndpointControlFctems(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/endpoint-control/fctems', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /endpoint-control/fctems/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteEndpointControlFctems(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/endpoint-control/fctems', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /endpoint-control/fctems
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllEndpointControlFctems(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/endpoint-control/fctems', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /endpoint-control/fctems
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addEndpointControlFctems(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/endpoint-control/fctems', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: sysgrp
	 * @route  GET /extender-controller/dataplan/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getExtenderControllerDataplan(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/extender-controller/dataplan', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: sysgrp
	 * @route  PUT /extender-controller/dataplan/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateExtenderControllerDataplan(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/extender-controller/dataplan', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: sysgrp
	 * @route  DELETE /extender-controller/dataplan/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteExtenderControllerDataplan(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/extender-controller/dataplan', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: sysgrp
	 * @route  GET /extender-controller/dataplan
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllExtenderControllerDataplan(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/extender-controller/dataplan', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: sysgrp
	 * @route  POST /extender-controller/dataplan
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addExtenderControllerDataplan(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/extender-controller/dataplan', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: sysgrp
	 * @route  GET /extender-controller/extender/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getExtenderControllerExtender(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/extender-controller/extender', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: sysgrp
	 * @route  PUT /extender-controller/extender/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateExtenderControllerExtender(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/extender-controller/extender', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: sysgrp
	 * @route  DELETE /extender-controller/extender/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteExtenderControllerExtender(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/extender-controller/extender', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: sysgrp
	 * @route  GET /extender-controller/extender
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllExtenderControllerExtender(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/extender-controller/extender', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: sysgrp
	 * @route  POST /extender-controller/extender
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addExtenderControllerExtender(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/extender-controller/extender', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /file-filter/profile/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFileFilterProfile(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/file-filter/profile', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: utmgrp
	 * @route  PUT /file-filter/profile/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFileFilterProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/file-filter/profile', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: utmgrp
	 * @route  DELETE /file-filter/profile/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFileFilterProfile(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/file-filter/profile', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: utmgrp
	 * @route  GET /file-filter/profile
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFileFilterProfile(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/file-filter/profile', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: utmgrp
	 * @route  POST /file-filter/profile
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFileFilterProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/file-filter/profile', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ipmacbinding/setting
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallIpmacbindingSetting(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ipmacbinding/setting', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.ipmacbinding/setting
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallIpmacbindingSetting(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.ipmacbinding/setting', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ipmacbinding/table/{seq-num}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $seqNum mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallIpmacbindingTable(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $seqNum
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ipmacbinding/table', $seqNum, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.ipmacbinding/table/{seq-num}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $seqNum mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallIpmacbindingTable(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $seqNum
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.ipmacbinding/table', $seqNum, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.ipmacbinding/table/{seq-num}
	 * @param  $seqNum mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallIpmacbindingTable(int $seqNum, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.ipmacbinding/table', $seqNum, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ipmacbinding/table
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallIpmacbindingTable(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ipmacbinding/table', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.ipmacbinding/table
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallIpmacbindingTable(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.ipmacbinding/table', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.schedule/group/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallScheduleGroup(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.schedule/group', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.schedule/group/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallScheduleGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.schedule/group', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.schedule/group/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallScheduleGroup(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.schedule/group', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.schedule/group
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallScheduleGroup(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.schedule/group', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.schedule/group
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallScheduleGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.schedule/group', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.schedule/onetime/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallScheduleOnetime(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.schedule/onetime', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.schedule/onetime/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallScheduleOnetime(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.schedule/onetime', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.schedule/onetime/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallScheduleOnetime(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.schedule/onetime', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.schedule/onetime
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallScheduleOnetime(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.schedule/onetime', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.schedule/onetime
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallScheduleOnetime(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.schedule/onetime', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.schedule/recurring/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallScheduleRecurring(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.schedule/recurring', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.schedule/recurring/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallScheduleRecurring(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.schedule/recurring', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.schedule/recurring/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallScheduleRecurring(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.schedule/recurring', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.schedule/recurring
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallScheduleRecurring(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.schedule/recurring', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.schedule/recurring
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallScheduleRecurring(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.schedule/recurring', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.service/category/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallServiceCategory(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.service/category', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.service/category/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallServiceCategory(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.service/category', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.service/category/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallServiceCategory(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.service/category', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.service/category
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallServiceCategory(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.service/category', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.service/category
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallServiceCategory(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.service/category', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.service/custom/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallServiceCustom(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.service/custom', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.service/custom/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallServiceCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.service/custom', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.service/custom/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallServiceCustom(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.service/custom', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.service/custom
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallServiceCustom(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.service/custom', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.service/custom
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallServiceCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.service/custom', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.service/group/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallServiceGroup(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.service/group', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.service/group/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallServiceGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.service/group', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.service/group/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallServiceGroup(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.service/group', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.service/group
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallServiceGroup(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.service/group', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.service/group
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallServiceGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.service/group', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.shaper/per-ip-shaper/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallShaperPerIpShaper(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.shaper/per-ip-shaper', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.shaper/per-ip-shaper/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallShaperPerIpShaper(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.shaper/per-ip-shaper', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.shaper/per-ip-shaper/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallShaperPerIpShaper(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.shaper/per-ip-shaper', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.shaper/per-ip-shaper
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallShaperPerIpShaper(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.shaper/per-ip-shaper', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.shaper/per-ip-shaper
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallShaperPerIpShaper(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.shaper/per-ip-shaper', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.shaper/traffic-shaper/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallShaperTrafficShaper(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.shaper/traffic-shaper', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.shaper/traffic-shaper/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallShaperTrafficShaper(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.shaper/traffic-shaper', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.shaper/traffic-shaper/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallShaperTrafficShaper(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.shaper/traffic-shaper', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.shaper/traffic-shaper
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallShaperTrafficShaper(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.shaper/traffic-shaper', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.shaper/traffic-shaper
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallShaperTrafficShaper(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.shaper/traffic-shaper', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssh/host-key/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallSshHostKey(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssh/host-key', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.ssh/host-key/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSshHostKey(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.ssh/host-key', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.ssh/host-key/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallSshHostKey(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.ssh/host-key', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssh/host-key
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSshHostKey(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssh/host-key', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.ssh/host-key
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallSshHostKey(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.ssh/host-key', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssh/local-ca/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallSshLocalCa(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssh/local-ca', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.ssh/local-ca/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSshLocalCa(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.ssh/local-ca', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.ssh/local-ca/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallSshLocalCa(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.ssh/local-ca', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssh/local-ca
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSshLocalCa(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssh/local-ca', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.ssh/local-ca
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallSshLocalCa(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.ssh/local-ca', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssh/local-key/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallSshLocalKey(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssh/local-key', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.ssh/local-key/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSshLocalKey(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.ssh/local-key', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.ssh/local-key/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallSshLocalKey(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.ssh/local-key', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssh/local-key
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSshLocalKey(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssh/local-key', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.ssh/local-key
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallSshLocalKey(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.ssh/local-key', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssh/setting
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSshSetting(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssh/setting', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.ssh/setting
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSshSetting(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.ssh/setting', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.ssl/setting
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSslSetting(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.ssl/setting', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.ssl/setting
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSslSetting(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.ssl/setting', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.wildcard-fqdn/custom/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallWildcardfqdnCustom(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.wildcard-fqdn/custom', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.wildcard-fqdn/custom/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallWildcardfqdnCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.wildcard-fqdn/custom', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.wildcard-fqdn/custom/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallWildcardfqdnCustom(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.wildcard-fqdn/custom', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.wildcard-fqdn/custom
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallWildcardfqdnCustom(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.wildcard-fqdn/custom', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.wildcard-fqdn/custom
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallWildcardfqdnCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.wildcard-fqdn/custom', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.wildcard-fqdn/group/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallWildcardfqdnGroup(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.wildcard-fqdn/group', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall.wildcard-fqdn/group/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallWildcardfqdnGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall.wildcard-fqdn/group', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall.wildcard-fqdn/group/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallWildcardfqdnGroup(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall.wildcard-fqdn/group', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall.wildcard-fqdn/group
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallWildcardfqdnGroup(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall.wildcard-fqdn/group', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall.wildcard-fqdn/group
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallWildcardfqdnGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall.wildcard-fqdn/group', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/DoS-policy/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallDosPolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/DoS-policy', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/DoS-policy/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallDosPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/DoS-policy', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/DoS-policy/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallDosPolicy(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/DoS-policy', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/DoS-policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallDosPolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/DoS-policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/DoS-policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallDosPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/DoS-policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/DoS-policy6/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallDosPolicy6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/DoS-policy6', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/DoS-policy6/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallDosPolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/DoS-policy6', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/DoS-policy6/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallDosPolicy6(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/DoS-policy6', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/DoS-policy6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallDosPolicy6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/DoS-policy6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/DoS-policy6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallDosPolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/DoS-policy6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/acl/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallAcl(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/acl', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/acl/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAcl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/acl', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/acl/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallAcl(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/acl', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/acl
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAcl(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/acl', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/acl
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallAcl(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/acl', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/acl6/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallAcl6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/acl6', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/acl6/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAcl6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/acl6', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/acl6/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallAcl6(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/acl6', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/acl6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAcl6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/acl6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/acl6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallAcl6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/acl6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/address/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallAddress(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/address', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/address/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAddress(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/address', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/address/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallAddress(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/address', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/address
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAddress(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/address', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/address
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallAddress(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/address', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/address6/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallAddress6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/address6', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/address6/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAddress6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/address6', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/address6/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallAddress6(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/address6', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/address6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAddress6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/address6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/address6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallAddress6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/address6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/address6-template/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallAddress6Template(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/address6-template', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/address6-template/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAddress6Template(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/address6-template', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/address6-template/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallAddress6Template(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/address6-template', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/address6-template
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAddress6Template(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/address6-template', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/address6-template
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallAddress6Template(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/address6-template', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/addrgrp/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallAddrgrp(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/addrgrp', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/addrgrp/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAddrgrp(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/addrgrp', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/addrgrp/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallAddrgrp(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/addrgrp', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/addrgrp
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAddrgrp(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/addrgrp', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/addrgrp
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallAddrgrp(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/addrgrp', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/addrgrp6/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallAddrgrp6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/addrgrp6', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/addrgrp6/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAddrgrp6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/addrgrp6', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/addrgrp6/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallAddrgrp6(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/addrgrp6', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/addrgrp6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAddrgrp6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/addrgrp6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/addrgrp6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallAddrgrp6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/addrgrp6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/auth-portal
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallAuthPortal(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/auth-portal', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/auth-portal
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallAuthPortal(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/auth-portal', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/central-snat-map/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallCentralSnatMap(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/central-snat-map', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/central-snat-map/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallCentralSnatMap(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/central-snat-map', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/central-snat-map/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallCentralSnatMap(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/central-snat-map', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/central-snat-map
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallCentralSnatMap(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/central-snat-map', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/central-snat-map
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallCentralSnatMap(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/central-snat-map', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/city/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallCity(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/city', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/city/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallCity(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/city', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/city/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallCity(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/city', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/city
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallCity(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/city', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/city
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallCity(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/city', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/country/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallCountry(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/country', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/country/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallCountry(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/country', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/country/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallCountry(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/country', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/country
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallCountry(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/country', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/country
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallCountry(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/country', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/decrypted-traffic-mirror/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallDecryptedTrafficMirror(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/decrypted-traffic-mirror', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/decrypted-traffic-mirror/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallDecryptedTrafficMirror(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/decrypted-traffic-mirror', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/decrypted-traffic-mirror/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallDecryptedTrafficMirror(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/decrypted-traffic-mirror', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/decrypted-traffic-mirror
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallDecryptedTrafficMirror(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/decrypted-traffic-mirror', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/decrypted-traffic-mirror
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallDecryptedTrafficMirror(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/decrypted-traffic-mirror', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/dnstranslation/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallDnstranslation(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/dnstranslation', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/dnstranslation/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallDnstranslation(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/dnstranslation', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/dnstranslation/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallDnstranslation(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/dnstranslation', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/dnstranslation
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallDnstranslation(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/dnstranslation', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/dnstranslation
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallDnstranslation(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/dnstranslation', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/identity-based-route/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallIdentityBasedRoute(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/identity-based-route', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/identity-based-route/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallIdentityBasedRoute(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/identity-based-route', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/identity-based-route/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallIdentityBasedRoute(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/identity-based-route', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/identity-based-route
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallIdentityBasedRoute(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/identity-based-route', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/identity-based-route
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallIdentityBasedRoute(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/identity-based-route', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/interface-policy/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInterfacePolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/interface-policy', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/interface-policy/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInterfacePolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/interface-policy', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/interface-policy/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInterfacePolicy(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/interface-policy', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/interface-policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInterfacePolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/interface-policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/interface-policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInterfacePolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/interface-policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/interface-policy6/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInterfacePolicy6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/interface-policy6', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/interface-policy6/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInterfacePolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/interface-policy6', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/interface-policy6/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInterfacePolicy6(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/interface-policy6', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/interface-policy6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInterfacePolicy6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/interface-policy6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/interface-policy6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInterfacePolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/interface-policy6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetService(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetService(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetService(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetService(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetService(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-addition/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceAddition(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-addition', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-addition/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceAddition(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-addition', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-addition/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceAddition(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-addition', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-addition
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceAddition(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-addition', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-addition
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceAddition(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-addition', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-append
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceAppend(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-append', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-append
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceAppend(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-append', NULL, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-botnet/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceBotnet(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-botnet', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-botnet/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceBotnet(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-botnet', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-botnet/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceBotnet(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-botnet', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-botnet
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceBotnet(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-botnet', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-botnet
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceBotnet(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-botnet', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-custom/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceCustom(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-custom', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-custom/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-custom', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-custom/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceCustom(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-custom', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-custom
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceCustom(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-custom', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-custom
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceCustom(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-custom', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-custom-group/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceCustomGroup(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-custom-group', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-custom-group/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceCustomGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-custom-group', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-custom-group/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceCustomGroup(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-custom-group', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-custom-group
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceCustomGroup(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-custom-group', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-custom-group
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceCustomGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-custom-group', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-definition/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceDefinition(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-definition', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-definition/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceDefinition(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-definition', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-definition/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceDefinition(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-definition', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-definition
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceDefinition(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-definition', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-definition
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceDefinition(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-definition', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-extension/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceExtension(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-extension', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-extension/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceExtension(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-extension', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-extension/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceExtension(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-extension', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-extension
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceExtension(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-extension', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-extension
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceExtension(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-extension', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-group/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceGroup(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-group', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-group/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-group', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-group/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceGroup(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-group', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-group
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceGroup(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-group', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-group
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-group', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-ipbl-reason/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceIpblReason(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-ipbl-reason', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-ipbl-reason/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceIpblReason(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-ipbl-reason', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-ipbl-reason/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceIpblReason(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-ipbl-reason', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-ipbl-reason
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceIpblReason(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-ipbl-reason', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-ipbl-reason
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceIpblReason(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-ipbl-reason', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-ipbl-vendor/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceIpblVendor(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-ipbl-vendor', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-ipbl-vendor/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceIpblVendor(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-ipbl-vendor', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-ipbl-vendor/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceIpblVendor(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-ipbl-vendor', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-ipbl-vendor
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceIpblVendor(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-ipbl-vendor', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-ipbl-vendor
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceIpblVendor(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-ipbl-vendor', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-list/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceList(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-list', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-list/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceList(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-list', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-list/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceList(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-list', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-list
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceList(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-list', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-list
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceList(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-list', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-name/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceName(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-name', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-name/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceName(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-name', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-name/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceName(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-name', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-name
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceName(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-name', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-name
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceName(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-name', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-owner/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceOwner(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-owner', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-owner/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceOwner(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-owner', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-owner/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceOwner(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-owner', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-owner
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceOwner(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-owner', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-owner
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceOwner(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-owner', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-reputation/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceReputation(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-reputation', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-reputation/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceReputation(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-reputation', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-reputation/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceReputation(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-reputation', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-reputation
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceReputation(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-reputation', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-reputation
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceReputation(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-reputation', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-sld/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallInternetServiceSld(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-sld', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/internet-service-sld/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallInternetServiceSld(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/internet-service-sld', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/internet-service-sld/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallInternetServiceSld(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/internet-service-sld', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/internet-service-sld
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallInternetServiceSld(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/internet-service-sld', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/internet-service-sld
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallInternetServiceSld(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/internet-service-sld', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ip-translation/{transid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $transid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallIpTranslation(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $transid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ip-translation', $transid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/ip-translation/{transid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $transid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallIpTranslation(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $transid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/ip-translation', $transid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/ip-translation/{transid}
	 * @param  $transid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallIpTranslation(int $transid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/ip-translation', $transid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ip-translation
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallIpTranslation(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ip-translation', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/ip-translation
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallIpTranslation(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/ip-translation', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ippool/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallIppool(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ippool', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/ippool/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallIppool(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/ippool', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/ippool/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallIppool(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/ippool', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ippool
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallIppool(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ippool', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/ippool
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallIppool(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/ippool', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ippool6/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallIppool6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ippool6', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/ippool6/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallIppool6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/ippool6', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/ippool6/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallIppool6(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/ippool6', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ippool6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallIppool6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ippool6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/ippool6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallIppool6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/ippool6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ldb-monitor/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallLdbMonitor(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ldb-monitor', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/ldb-monitor/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallLdbMonitor(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/ldb-monitor', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/ldb-monitor/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallLdbMonitor(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/ldb-monitor', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ldb-monitor
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallLdbMonitor(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ldb-monitor', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/ldb-monitor
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallLdbMonitor(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/ldb-monitor', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/local-in-policy/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallLocalInPolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/local-in-policy', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/local-in-policy/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallLocalInPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/local-in-policy', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/local-in-policy/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallLocalInPolicy(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/local-in-policy', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/local-in-policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallLocalInPolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/local-in-policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/local-in-policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallLocalInPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/local-in-policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/local-in-policy6/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallLocalInPolicy6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/local-in-policy6', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/local-in-policy6/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallLocalInPolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/local-in-policy6', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/local-in-policy6/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallLocalInPolicy6(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/local-in-policy6', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/local-in-policy6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallLocalInPolicy6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/local-in-policy6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/local-in-policy6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallLocalInPolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/local-in-policy6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-address/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallMulticastAddress(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-address', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/multicast-address/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallMulticastAddress(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/multicast-address', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/multicast-address/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallMulticastAddress(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/multicast-address', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-address
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallMulticastAddress(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-address', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/multicast-address
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallMulticastAddress(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/multicast-address', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-address6/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallMulticastAddress6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-address6', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/multicast-address6/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallMulticastAddress6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/multicast-address6', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/multicast-address6/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallMulticastAddress6(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/multicast-address6', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-address6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallMulticastAddress6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-address6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/multicast-address6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallMulticastAddress6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/multicast-address6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-policy/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallMulticastPolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-policy', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/multicast-policy/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallMulticastPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/multicast-policy', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/multicast-policy/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallMulticastPolicy(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/multicast-policy', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallMulticastPolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/multicast-policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallMulticastPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/multicast-policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-policy6/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallMulticastPolicy6(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-policy6', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/multicast-policy6/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallMulticastPolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/multicast-policy6', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/multicast-policy6/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallMulticastPolicy6(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/multicast-policy6', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/multicast-policy6
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallMulticastPolicy6(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/multicast-policy6', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/multicast-policy6
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallMulticastPolicy6(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/multicast-policy6', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/policy/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallPolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/policy', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/policy/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/policy', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/policy/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallPolicy(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/policy', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallPolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/policy46/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallPolicy46(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/policy46', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/policy46/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallPolicy46(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/policy46', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/policy46/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallPolicy46(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/policy46', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/policy46
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallPolicy46(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/policy46', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/policy46
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallPolicy46(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/policy46', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/policy64/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallPolicy64(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/policy64', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/policy64/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallPolicy64(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/policy64', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/policy64/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallPolicy64(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/policy64', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/policy64
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallPolicy64(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/policy64', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/policy64
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallPolicy64(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/policy64', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/profile-group/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallProfileGroup(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/profile-group', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/profile-group/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallProfileGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/profile-group', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/profile-group/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallProfileGroup(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/profile-group', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/profile-group
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallProfileGroup(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/profile-group', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/profile-group
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallProfileGroup(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/profile-group', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/profile-protocol-options/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallProfileProtocolOptions(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/profile-protocol-options', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/profile-protocol-options/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallProfileProtocolOptions(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/profile-protocol-options', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/profile-protocol-options/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallProfileProtocolOptions(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/profile-protocol-options', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/profile-protocol-options
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallProfileProtocolOptions(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/profile-protocol-options', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/profile-protocol-options
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallProfileProtocolOptions(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/profile-protocol-options', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/proxy-address/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallProxyAddress(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/proxy-address', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/proxy-address/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallProxyAddress(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/proxy-address', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/proxy-address/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallProxyAddress(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/proxy-address', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/proxy-address
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallProxyAddress(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/proxy-address', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/proxy-address
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallProxyAddress(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/proxy-address', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/proxy-addrgrp/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallProxyAddrgrp(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/proxy-addrgrp', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/proxy-addrgrp/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallProxyAddrgrp(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/proxy-addrgrp', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/proxy-addrgrp/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallProxyAddrgrp(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/proxy-addrgrp', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/proxy-addrgrp
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallProxyAddrgrp(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/proxy-addrgrp', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/proxy-addrgrp
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallProxyAddrgrp(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/proxy-addrgrp', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/proxy-policy/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallProxyPolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/proxy-policy', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/proxy-policy/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallProxyPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/proxy-policy', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/proxy-policy/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallProxyPolicy(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/proxy-policy', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/proxy-policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallProxyPolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/proxy-policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/proxy-policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallProxyPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/proxy-policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/region/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallRegion(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/region', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/region/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallRegion(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/region', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/region/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallRegion(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/region', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/region
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallRegion(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/region', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/region
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallRegion(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/region', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/security-policy/{policyid}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallSecurityPolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/security-policy', $policyid, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/security-policy/{policyid}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $policyid mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSecurityPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $policyid
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/security-policy', $policyid, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/security-policy/{policyid}
	 * @param  $policyid mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallSecurityPolicy(int $policyid, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/security-policy', $policyid, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/security-policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSecurityPolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/security-policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/security-policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallSecurityPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/security-policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/shaping-policy/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallShapingPolicy(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/shaping-policy', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/shaping-policy/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallShapingPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/shaping-policy', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/shaping-policy/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallShapingPolicy(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/shaping-policy', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/shaping-policy
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallShapingPolicy(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/shaping-policy', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/shaping-policy
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallShapingPolicy(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/shaping-policy', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/shaping-profile/{profile-name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $profileName mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallShapingProfile(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $profileName
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/shaping-profile', $profileName, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/shaping-profile/{profile-name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $profileName mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallShapingProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $profileName
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/shaping-profile', $profileName, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/shaping-profile/{profile-name}
	 * @param  $profileName mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallShapingProfile(string $profileName, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/shaping-profile', $profileName, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/shaping-profile
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallShapingProfile(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/shaping-profile', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/shaping-profile
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallShapingProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/shaping-profile', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: netgrp
	 * @route  GET /firewall/sniffer/{id}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallSniffer(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		int $id
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/sniffer', $id, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: netgrp
	 * @route  PUT /firewall/sniffer/{id}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $id mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSniffer(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		int $id
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/sniffer', $id, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: netgrp
	 * @route  DELETE /firewall/sniffer/{id}
	 * @param  $id mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallSniffer(int $id, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/sniffer', $id, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: netgrp
	 * @route  GET /firewall/sniffer
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSniffer(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/sniffer', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: netgrp
	 * @route  POST /firewall/sniffer
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallSniffer(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/sniffer', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ssl-server/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallSslServer(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ssl-server', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/ssl-server/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSslServer(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/ssl-server', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/ssl-server/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallSslServer(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/ssl-server', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ssl-server
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If present, objects will be filtered on property with this value.
	 * @param  $scope Scope [global|vdom|both*]
	 * @param  $excludeDefaultValues Exclude properties/objects with default value
	 * @param  $action default: Return the CLI default values for entire CLI tree.
	 * meta: Return meta data for a specific object, table, or the entire CLI tree.
	 * schema: Return schema for entire CLI tree.
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getAllFirewallSslServer(
		bool $datasource = null,
		int $start = null,
		int $count = null,
		bool $with_meta = null,
		bool $with_contents_hash = null,
		bool $skip = null,
		array $format = null,
		array $filter = null,
		string $key = null,
		string $pattern = null,
		string $scope = null,
		bool $excludeDefaultValues = null,
		string $action = null,
		array $vdom = null
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ssl-server', NULL, ['datasource'=>$datasource, 'start'=>$start, 'count'=>$count, 'with_meta'=>$with_meta, 'with_contents_hash'=>$with_contents_hash, 'skip'=>$skip, 'format'=>$format, 'filter'=>$filter, 'key'=>$key, 'pattern'=>$pattern, 'scope'=>$scope, 'exclude-default-values'=>$excludeDefaultValues, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Create object(s) in this table.
	 *  Access Group: fwgrp
	 * @route  POST /firewall/ssl-server
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _clone_: Clone this specific resource.
	 * When *action=clone* is set, the extra parameters *nkey* must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $nkey  If *action=clone*, use *nkey* to specify the ID for the new resource to be created.
	 * For example, to clone `address1` to `address1_clone`, use:
	 * __action=clone&nkey=address1_clone__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *clone*.
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function addFirewallSslServer(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $nkey = null
	) : stdClass {
		return $this->curlRequest('POST', '/cmdb/firewall/ssl-server', NULL, ['vdom'=>$vdom, 'action'=>$action, 'nkey'=>$nkey], $body);
	}


	/**
	 * Select a specific entry from a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ssl-ssh-profile/{name}
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $action default: Return the CLI default values for this object type.
	 * schema: Return the CLI schema for this object type.
	 * revision: Return the CMDB revision for this object type.
	 * transaction-list: List all configuration transaction(s).
	 *
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function getFirewallSslSshProfile(
		bool $datasource = null,
		bool $with_meta = null,
		bool $skip = null,
		array $format = null,
		string $action = null,
		array $vdom = null,
		string $name
	) : stdClass {
		return $this->curlRequest('GET', '/cmdb/firewall/ssl-ssh-profile', $name, ['datasource'=>$datasource, 'with_meta'=>$with_meta, 'skip'=>$skip, 'format'=>$format, 'action'=>$action, 'vdom'=>$vdom], NULL);
	}


	/**
	 * Update this specific resource.
	 *  Access Group: fwgrp
	 * @route  PUT /firewall/ssl-ssh-profile/{name}
	 * @param  $body Possible parameters to go in the body for the request
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @param  $action If supported, an action can be specified.
	 * _move_: Move this specific resource.
	 * When *action=move* is set, one of the extra parameters (*before*, *after*) must be provided.
	 * __*Note:*__ If this parameter is provided when not supported, the action will be ignored and an “invalid request” error will be returned.
	 *
	 * @param  $before If *action=move*, use *before* to specify the ID of the resource that this resource will be moved before.
	 * For example, to move `object 1` to before `object 2`, use:
	 * __action=move&before=2__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $after If *action=move*, use *after* to specify the ID of the resource that this resource will be moved after.
	 * For example, to move `object 1` to after `object 3`, use:
	 * __action=move&after=3__
	 * __*Note:*__ This parameter can only be used when the *action* parameter is set to *move*.
	 *
	 * @param  $name mkey
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function updateFirewallSslSshProfile(
		stdClass $body,
		array $vdom = null,
		string $action = null,
		string $before = null,
		string $after = null,
		string $name
	) : stdClass {
		return $this->curlRequest('PUT', '/cmdb/firewall/ssl-ssh-profile', $name, ['vdom'=>$vdom, 'action'=>$action, 'before'=>$before, 'after'=>$after], $body);
	}


	/**
	 * Delete this specific resource.
	 *  Access Group: fwgrp
	 * @route  DELETE /firewall/ssl-ssh-profile/{name}
	 * @param  $name mkey
	 * @param  $vdom Specify the Virtual Domain(s) from which results are returned or changes are applied to. If this parameter is not provided, the management VDOM will be used. If the admin does not have access to the VDOM, a permission error will be returned.
	 * The URL parameter is one of:
	 * vdom=root (Single VDOM)
	 * vdom=vdom1,vdom2 (Multiple VDOMs)
	 * vdom=* (All VDOMs)
	 *
	 * @return stdClass Return the firewall's response as an object.
	 */
	public function deleteFirewallSslSshProfile(string $name, array $vdom = null) : stdClass
	{
		return $this->curlRequest('DELETE', '/cmdb/firewall/ssl-ssh-profile', $name, ['vdom'=>$vdom], NULL);
	}


	/**
	 * Select all entries in a CLI table.
	 *  Access Group: fwgrp
	 * @route  GET /firewall/ssl-ssh-profile
	 * @param  $datasource Enable to include datasource information for each linked object.
	 * @param  $start Starting entry index.
	 * @param  $count Maximum number of entries to return.
	 * @param  $with_meta Enable to include meta information about each object (type id, references, etc).
	 * @param  $with_contents_hash Enable to include a checksum of each object's contents.
	 * @param  $skip Enable to call CLI skip operator to hide skipped properties.
	 * @param  $format List of property names to include in results, separated by | (i.e. policyid|srcintf).
	 * @param  $filter Filtering multiple key/value pairs
	 * Operator     |   Description
	 * ==     |   Case insensitive match with pattern.
	 * !=     |    Does not match with pattern (case insensitive).
	 * =@     |    Pattern found in object value (case insensitive).
	 * !@     |    ﻿Pattern not﻿ found in object value (case insensitive).
	 * <=     |    Value must be less than or equal to ﻿pattern﻿.
	 * <     |    Value must be less than pattern﻿.
	 * .>=    |    Value must be greater than or equal to ﻿pattern﻿.
	 * .>     |    Value must be greater than ﻿pattern﻿.
	 * Logical OR    |    Separate filters using commas ','
	 * Logical AND    |    Filter strings can be combined to create logical AND queries by including multiple filters in the request.
	 * Combining AND and OR    |    You can combine AND and OR filters together to create more complex filters.
	 *
	 * @param  $key If present, objects will be filtered on property with this name.
	 * @param  $pattern If presen