import React from 'react';

import {
  IconSvg,
  ToggleButton,
  Menu,
  MenuGroup,
  MenuList,
  MenuItem,
} from 'reactackle';

import { RouteDemo } from '../../components/Route/RouteStructure';

import {
  TestBox,
  DemoSnippet,
  DemoPreview,
  DemoCode,
} from '../../components/DemoSnippet/DemoSnippet';

import SnippetDefault from './snippets/1.snippet';
import Snippet2 from './snippets/2.snippet';
import Snippet3 from './snippets/3.snippet';

const menuContent = (
  <MenuGroup>
    <MenuList>
      <MenuItem text="Item 1" />
      <MenuItem text="Item 2" />
      <MenuItem text="Item 3" />
    </MenuList>

    <MenuList bordered title="Second items group">
      <MenuItem text="Item 1" />

      <MenuItem text="Item 2">
        <MenuGroup nestingLevel={1}>
          <MenuList>
            <MenuItem text="Item 2-1" />
            <MenuItem text="Item 2-2">
              <MenuGroup nestingLevel={2}>
                <MenuList>
                  <MenuItem text="Item 2-2-1" />
                  <MenuItem text="Item 2-2-2" />
                </MenuList>
              </MenuGroup>
            </MenuItem>
          </MenuList>
        </MenuGroup>
      </MenuItem>
    </MenuList>
  </MenuGroup>
);

const menuContentComplex = (
  <MenuGroup>
    <MenuList>
      <MenuItem
        text="Item 1"
        textSecondary="Item 1 - text secondary"
      />

      <MenuItem
        text="Item 2"
        textSecondary="Item with iconLeft"
        iconLeft={<IconSvg />}
      />

      <MenuItem
        text="Item 3"
        textSecondary="Item with iconLeft & iconRight"
        iconLeft={<IconSvg />}
        iconRight={<IconSvg />}
      />

      <MenuItem
        text="Item 4"
        textSecondary="Item with icon offset"
        addIconOffset
      />

      <MenuItem
        text="Item 5"
        textSecondary={
          'Active item  with iconLeft & iconRight, addonRight & textRight'
        }
        iconLeft={<IconSvg />}
        iconRight={<IconSvg />}
        textRight="CTRL+F"
        addonRight={<ToggleButton />}
        active
      />

      <MenuItem
        text="Item 6"
        textSecondary="Item with iconLeft & image"
        iconLeft={<IconSvg />}
        //eslint-disable-next-line
        image="data:image/jpeg;base64,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"
      />

      <MenuItem
        text="Item 7"
        textSecondary="Item with icon & image offset"
        addIconOffset
        addImageOffset
      />

      <MenuItem
        text="Item 8 with iconRight & subtree"
        iconRight={<IconSvg />}
      >
        <MenuGroup nestingLevel={1}>
          <MenuList>
            <MenuItem text="Item 8-1" />
            <MenuItem text="Item 8-2" />
          </MenuList>
        </MenuGroup>
      </MenuItem>
    </MenuList>
  </MenuGroup>
);

const menuContentInline = (
  <MenuGroup>
    <MenuList>
      <MenuItem text="Item 1-1" />
      <MenuItem
        text="Item 1-2"
        textSecondary="Item with iconLeft"
        iconLeft={<IconSvg />}
      />
    </MenuList>

    <MenuList bordered>
      <MenuItem text="Item 2-1" />
      <MenuItem text="Item 2-2" />
    </MenuList>

  </MenuGroup>
);

export const MenuDemoRoute = () => (
  <RouteDemo>
    <DemoSnippet title="Default menu">
      <DemoPreview>
        <TestBox flex contentSpaced >
          <TestBox maxWidth="400px" title="colorScheme = 'dark'">
            <Menu>
              { menuContent }
            </Menu>
          </TestBox>

          <TestBox
            maxWidth="400px"
            bgColor="#666"
            title="colorScheme = 'light'"
          >
            <Menu colorScheme="light">
              { menuContent }
            </Menu>
          </TestBox>
        </TestBox>
      </DemoPreview>
      <DemoCode
        code={SnippetDefault}
      />
    </DemoSnippet>

    <DemoSnippet title="Complex Menu Items">
      <DemoPreview>
        <TestBox maxWidth="400px" >
          <Menu>
            {menuContentComplex}
          </Menu>
        </TestBox>
      </DemoPreview>
      <DemoCode
        code={Snippet3}
      />
    </DemoSnippet>

    <DemoSnippet title="Inline Menu">
      <DemoPreview>
        <TestBox>
          <Menu inline>
            { menuContentInline }
          </Menu>
        </TestBox>
      </DemoPreview>
      <DemoCode
        code={Snippet2}
      />
    </DemoSnippet>
  </RouteDemo>
);

MenuDemoRoute.displayName = 'MenuDemoRoute';
