#!/usr/bin/python

# Copyright: (c) 2018, Terry Jones <terry.jones@example.org>
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

DOCUMENTATION = r'''
---
module: openai_chat

short_description: This is a test module to interface with the OpenAI API

# If this is part of a collection, you need to use semantic versioning,
# i.e. the version is of the form "2.5.0" and not "2.4".
version_added: "1.0.0"

description: it wouldn't be a very good module if there weren't options.

options:
    model:
        description: ID of the model to use. See the model endpoint compatibility table for details on which models work with the Chat API.
        required: true
        type: str
    messages:
        description: A list of messages comprising the conversation so far.
        required: true
        type: list
    role:
        description: The role of the messages author. One of system, user, assistant, or function.
        required: true
        type: str
    content:
        description: The contents of the message. content is required for all messages except assistant messages with function calls.
        required: false
        type: str
    name:
        description: The name of the author of this message. name is required if role is function, and it should be the name of the function whose response is in the content. May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
        required: false
        type: str
    function_call:
        description: The name and arguments of a function that should be called, as generated by the model.
        required: false
        type: dict
    functions:
        description: A list of functions the model may generate JSON inputs for.
        required: false
        type: list
    temperature:
        description: What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random, while lower values like 0.2 will make it more focused and deterministic.
        required: false
        type: float
    top_p:
        description: An alternative to sampling with temperature, called nucleus sampling, where the model considers the results of the tokens with top_p probability mass. So 0.1 means only the tokens comprising the top 10% probability mass are considered.
        required: false
        type: float
    n:
        description: How many chat completion choices to generate for each input message.
        required: false
        type: int
    stream:
        description: If set, partial message deltas will be sent, like in ChatGPT. Tokens will be sent as data-only server-sent events as they become available, with the stream terminated by a data: [DONE] message.
        required: false
        type: bool
    stop:
        description: Up to 4 sequences where the API will stop generating further tokens.
        required: false
        type: list
    max_tokens:
        description: The maximum number of tokens to generate in the chat completion. The total length of input tokens and generated tokens is limited by the model's context length.
        required: false
        type: int
    presence_penalty:
        description: Number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the text so far, increasing the model's likelihood to talk about new topics.
        required: false
        type: float
    frequency_penalty:
        description: Number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in the text so far, decreasing the model's likelihood to repeat the same line verbatim.
        required: false
        type: float
    user:
        description: A unique identifier representing your end-user, which can help OpenAI to monitor and detect abuse
        required: false
        type: str
# Specify this value according to your collection
# in format of namespace.collection.doc_fragment_name
extends_documentation_fragment:
    - my_namespace.my_collection.my_doc_fragment_name

author:
    - Your Name (@yourGitHubHandle)
'''

EXAMPLES = r'''

- name: Generate chat completion
  openai_chat:
    api_key: "{{ lookup('env','OPENAI_API_KEY') }}"
    model: "gpt-3.5-turbo-16k"
    messages:
      - role: "system"
        content: "You are a helpful assistant."
      - role: "user"
        content: "Who won the world series in 2020?"
    temperature: 1.5
    max_tokens: 60
    frequency_penalty: 2.0
    presence_penalty: -2.0
  register: chat_result

'''

from ansible.module_utils.basic import AnsibleModule
import openai

def run_module():
    module_args = dict(
        api_key=dict(type='str', required=True, no_log=True),
        model=dict(type='str', required=True),
        messages=dict(type='list', required=True),
        max_tokens=dict(type='int', required=False, default=None),
        temperature=dict(type='float', required=False),
        top_p=dict(type='float', required=False),
        n=dict(type='int', required=False),
        stream=dict(type='bool', required=False),
        stop=dict(type='list', required=False),
        presence_penalty=dict(type='float', required=False),
        frequency_penalty=dict(type='float', required=False),
        user=dict(type='str', required=False, default="None")
    )

    module = AnsibleModule(
        argument_spec=module_args,
        supports_check_mode=True
    )

    result = dict(
        changed=False,
        original_message='',
        message=''
    )

    if module.check_mode:
        return result

    try:
        openai.api_key = module.params['api_key']
        response = openai.ChatCompletion.create(
          model=module.params.get('model'),
          messages=module.params.get('messages'),
          max_tokens=module.params.get('max_tokens'),
          temperature=module.params.get('temperature'),
          top_p=module.params.get('top_p'),
          n=module.params.get('n'),
          stream=module.params.get('stream'),
          stop=module.params.get('stop'),
          presence_penalty=module.params.get('presence_penalty'),
          frequency_penalty=module.params.get('frequency_penalty'),
          user=module.params.get('user')
        )
        result['original_message'] = module.params['messages']
        result['message'] = response['choices'][0]['message']['content']
        result['changed'] = True
    except Exception as e:
        module.fail_json(msg='Failed to generate chat completion: %s' % to_native(e), **result)

    module.exit_json(**result)

def main():
    run_module()

if __name__ == '__main__':
    main()
