#!/usr/bin/env python

# Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#     http://www.apache.org/licenses/LICENSE-2.0
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.

"""
Convert the Taurus Test suite XML to Junit XML
"""
# pylint: disable=redefined-builtin

import glob
import shutil
import os

from .reader import get_mon_metrics_list


def get_taurus_options(artifacts_dir, jmeter_path=None):
    """The options for Taurus BZT command"""
    options = []
    if jmeter_path:
        options.append('-o modules.jmeter.path={}'.format(jmeter_path))
    options.append('-o settings.artifacts-dir={}'.format(artifacts_dir))
    options.append('-o modules.console.disable=true')
    options.append('-o settings.env.BASEDIR={}'.format(artifacts_dir))
    options_str = ' '.join(options)

    return options_str


def update_taurus_metric_files(suite_artifacts_dir, test_file):
    """
    It renames the server and local metric monitoring log files to metrics.csv.
    The order of the columns in header of server metric monitoring SALogs file generated by taurus
    is not inline with data. So as a work around this function rewrites the header based on order
    defined in the test yaml.
    """
    metrics_new_file = os.path.join(suite_artifacts_dir, "metrics.csv")

    server_metric_file_pattern = os.path.join(suite_artifacts_dir, "SAlogs_*")
    metrics_log_file = glob.glob(server_metric_file_pattern)
    if metrics_log_file:
        metrics = get_mon_metrics_list(test_file)
        if metrics:
            with open(metrics_log_file[0]) as from_file:
                line = from_file.readline()
                with open(metrics_log_file[0], mode="w") as to_file:
                    to_file.write(','.join(line.split(',')[0:1] + metrics) + "\n")
                    shutil.copyfileobj(from_file, to_file)

        os.rename(metrics_log_file[0], metrics_new_file)

    else:
         metrics_log_file = os.path.join(suite_artifacts_dir, "local_monitoring_logs.csv")
         if os.path.exists(metrics_log_file):
             os.rename(metrics_log_file, metrics_new_file)
