import { Mask as mask } from '@core/mask'
const Compo = mask.Compo;
const DomLite = Compo.DomLite;
const $ = mask.$;

UTest({
    $before () {
        Compo.config.setDOMLibrary(DomLite);
    },
    $after () {
        Compo.config.setDOMLibrary($);
    },
    'filter' () {
        var div = mask.render('div { span; section; }');
        var set = DomLite(div);
        eq_(set.length, 1);
        eq_(set.filter('div').length, 1);
        eq_(set.filter('span').length, 0);
        eq_(set.filter('section').length, 0);

        var fragment = mask.render('div; span.foo; section.foo;');
        var set = DomLite(fragment.childNodes);

        eq_(set.length, 3);
        eq_(set.filter('div').length, 1);
        eq_(set.filter('span').length, 1);
        eq_(set.filter('section').length, 1);
        eq_(set.filter('.foo').length, 2);
        eq_(set.filter('span.foo').length, 1);
    },
    'event' () {
        var div = document.createElement('div');
        document.body.appendChild(div);

        var set = DomLite(div);
        eq_(set[0], div);

        set.on('click', assert.await());
        DomLite(div).trigger('click').remove();
    },
    'delegated event' () {
        var div = mask.render('div { span; section }');
        document.body.appendChild(div);

        var set = DomLite(div);
        eq_(set.length, 1);
        eq_(set[0].tagName, 'DIV');

        var count = 0;
        set.on('click', 'span', () => ++count);

        $(div).children('span').trigger('click');
        $(div).children('section').trigger('click');
        $(div).trigger('click');

        eq_(count, 1);
    },
    'component with delegated event' () {
        mask.registerHandler('foo', Compo({
            template: `
                h4 > em > 'Hello'
                section > p > 'World'
            `,
            events: {
                'foo: p': function(event) {
                    notEq_(expects.length, 0);
                    eq_(event.target.tagName, expects.shift());
                },
                'click: h4': function(event) {
                    notEq_(expects.length, 0);
                    eq_(event.target.tagName, expects.shift());
                },

            }
        }));
        var expects = [ 'P', 'EM' ];
        var dom = $(mask.render('foo').childNodes).appendTo('body');
        DomLite(dom)
            .find('p')
            .trigger('foo', {});

        DomLite(dom)
            .find('em')
            .trigger('click', {});

        eq_(expects.length, 0);
    },
    'children' () {
        var dom = mask.render('div > span; section > p > a');
        var children = DomLite(dom).children();
        eq_(children.length, 2);
        eq_(children[0].tagName, 'SPAN');
        eq_(children[1].tagName, 'P');

        children = children.children();
        eq_(children.length, 1);
        eq_(children[0].tagName, 'A');
    },
    'closest' () {
        var dom = mask.render('section > p name=baz > a > em');
        var $ = DomLite(dom).find('em');
        eq_($.length, 1);

        $ = $.closest('[name=baz]');
        eq_($.length, 1);
        eq_($[0].tagName, 'P');
    },
    'next' () {
        var dom = mask.render('h1;h2 a;h3 b;');
        var $ = DomLite(dom).filter('h1');
        eq_($.length, 1);
        eq_($[0].tagName, 'H1');

        '> next'
        var section = $.next();
        eq_(section.length, 1);
        eq_(section[0].tagName, 'H2');

        '> next by selector'
        var section = $.next('[a]');
        eq_(section.length, 1);
        eq_(section[0].tagName, 'H2');

        '> find next by selector'
        var section = $.next('[b]');
        eq_(section.length, 1);
        eq_(section[0].tagName, 'H3');
    },
    'classHandler' () {
        var dom = mask.render('.foo.bar');
        var $ = DomLite(dom);
        eq_($.hasClass('foo'), true);
        eq_($.hasClass('bar'), true);

        $.removeClass('foo');
        eq_($.hasClass('foo'), false);
        eq_($.hasClass('bar'), true);

        $.toggleClass('foo');
        eq_($.hasClass('foo'), true);
        eq_($.hasClass('bar'), true);

        $.toggleClass('baz');
        eq_($.hasClass('foo'), true);
        eq_($.hasClass('bar'), true);
        eq_($.hasClass('baz'), true);
    },
    'trigger' () {
        var div = mask.render('.foo > input type=text');
        document.body.appendChild(div);

        div.addEventListener('change', assert.await(function(event){
            eq_(event.target.value, 'Baz');
        }), false);

        var input = div.querySelector('input');
        input.value = 'Baz';
        DomLite(input).trigger('change');

        div.addEventListener('foo', assert.await(function(event){
            eq_(event.detail, 'Hello');
        }), false);
        DomLite(input).trigger('foo', 'Hello');

        DomLite(div).remove();
        eq_(div.parentNode, null);
    },
    'attr' () {
        var div = DomLite(mask.render('.foo name=baz'))

        eq_(div.attr('name'), 'baz');
        eq_(div.attr('class'), 'foo');

        div.attr('name', 'qux')
        eq_(div.attr('name'), 'qux');

        div.removeAttr('name')
        eq_(div.attr('name'), null);
    },

    'manip': {
        'should append elements' () {
            var div = DomLite(mask.render('.foo name=baz'));
            div.append(document.createElement('section'));
            div.append(DomLite(mask.render('.baz')));

            DomLite(div).append(document.createElement('span'));

            $(div)
                .has_('span')
                .has_('.baz')
                .has_('section')
                ;
        },
        'should prepend elements' () {
            var div = DomLite(mask.render('.foo name=baz'));
            div.prepend(document.createElement('section'));
            div.prepend(DomLite(mask.render('.baz')));

            DomLite(div).prepend(document.createElement('span'));

            $(div)
                .has_('span')
                .has_('.baz')
                .has_('section')
                ;
        },
        'should insert elements before' () {
            var frag = DomLite(mask.render('section > .foo name=baz'));
            var div = DomLite(frag).find('div');
            notEq_(div, null);

            div.before(document.createElement('section'));
            div.before(DomLite(mask.render('.baz')));

            DomLite(div).before(document.createElement('span'));

            $(frag)
                .has_('span')
                .has_('.baz')
                .has_('section')

                .children()
                .eq(0)
                .eq_('prop', 'tagName', 'SECTION')
                .end()
                .eq(2)
                .eq_('prop', 'tagName', 'SPAN')
                ;
        },
        'should insert elements after' () {
            var frag = DomLite(mask.render('section > .foo name=baz'));
            var div = DomLite(frag).find('div');
            notEq_(div, null);

            div.after(document.createElement('section'));
            div.after(DomLite(mask.render('.baz')));

            DomLite(div).after(document.createElement('span'));

            $(frag)
                .has_('span')
                .has_('.baz')
                .has_('section')

                .children()
                .eq(1)
                .eq_('prop', 'tagName', 'SPAN')
                .end()
                .eq(3)
                .eq_('prop', 'tagName', 'SECTION')
                ;
        }
    }

});
