package iam

func IsWildcardAllowed(actions ...string) bool {
	for _, action := range actions {
		var ok bool
		for _, allowed := range allowedActionsForResourceWildcards {
			if allowed == action {
				ok = true
				break
			}
		}
		if !ok {
			return false
		}
	}
	return true
}

//see https://docs.aws.amazon.com/service-authorization/latest/reference/list_identityandaccessmanagement.html
var allowedActionsForResourceWildcards = []string{
	"account:DisableRegion",
	"account:EnableRegion",
	"account:ListRegions",
	"activate:CreateForm",
	"activate:GetAccountContact",
	"activate:GetContentInfo",
	"activate:GetCosts",
	"activate:GetCredits",
	"activate:GetMemberInfo",
	"activate:GetProgram",
	"activate:PutMemberInfo",
	"a4b:ApproveSkill",
	"a4b:AssociateSkillWithUsers",
	"a4b:CompleteRegistration",
	"a4b:CreateAddressBook",
	"a4b:CreateBusinessReportSchedule",
	"a4b:CreateConferenceProvider",
	"a4b:CreateContact",
	"a4b:CreateGatewayGroup",
	"a4b:CreateNetworkProfile",
	"a4b:CreateProfile",
	"a4b:CreateSkillGroup",
	"a4b:GetConferencePreference",
	"a4b:GetInvitationConfiguration",
	"a4b:ListBusinessReportSchedules",
	"a4b:ListConferenceProviders",
	"a4b:ListGatewayGroups",
	"a4b:ListSkills",
	"a4b:ListSkillsStoreCategories",
	"a4b:ListSkillsStoreSkillsByCategory",
	"a4b:PutConferencePreference",
	"a4b:PutDeviceSetupEvents",
	"a4b:PutInvitationConfiguration",
	"a4b:RegisterAVSDevice",
	"a4b:RegisterDevice",
	"a4b:RejectSkill",
	"a4b:ResolveRoom",
	"a4b:SearchAddressBooks",
	"a4b:SearchContacts",
	"a4b:SearchDevices",
	"a4b:SearchNetworkProfiles",
	"a4b:SearchProfiles",
	"a4b:SearchRooms",
	"a4b:SearchSkillGroups",
	"a4b:SearchUsers",
	"a4b:SendAnnouncement",
	"a4b:StartDeviceSync",
	"amplify:ListApps",
	"appmesh:ListMeshes",
	"appmesh-preview:ListMeshes",
	"apprunner:ListAutoScalingConfigurations",
	"apprunner:ListConnections",
	"apprunner:ListServices",
	"appconfig:ListApplications",
	"appconfig:ListDeploymentStrategies",
	"appflow:CreateConnectorProfile",
	"appflow:CreateFlow",
	"appflow:DescribeConnectorProfiles",
	"appflow:DescribeConnectors",
	"appflow:DescribeFlow",
	"appflow:DescribeFlows",
	"app-integrations:ListEventIntegrationAssociations",
	"app-integrations:ListEventIntegrations",
	"application-autoscaling:DeleteScalingPolicy",
	"application-autoscaling:DeleteScheduledAction",
	"application-autoscaling:DeregisterScalableTarget",
	"application-autoscaling:DescribeScalableTargets",
	"application-autoscaling:DescribeScalingActivities",
	"application-autoscaling:DescribeScalingPolicies",
	"application-autoscaling:DescribeScheduledActions",
	"application-autoscaling:PutScalingPolicy",
	"application-autoscaling:PutScheduledAction",
	"application-autoscaling:RegisterScalableTarget",
	"application-cost-profiler:DeleteReportDefinition",
	"application-cost-profiler:GetReportDefinition",
	"application-cost-profiler:ImportApplicationUsage",
	"application-cost-profiler:ListReportDefinitions",
	"application-cost-profiler:PutReportDefinition",
	"application-cost-profiler:UpdateReportDefinition",
	"discovery:AssociateConfigurationItemsToApplication",
	"discovery:BatchDeleteImportData",
	"discovery:CreateApplication",
	"discovery:CreateTags",
	"discovery:DeleteApplications",
	"discovery:DeleteTags",
	"discovery:DescribeAgents",
	"discovery:DescribeConfigurations",
	"discovery:DescribeContinuousExports",
	"discovery:DescribeExportConfigurations",
	"discovery:DescribeExportTasks",
	"discovery:DescribeImportTasks",
	"discovery:DescribeTags",
	"discovery:DisassociateConfigurationItemsFromApplication",
	"discovery:ExportConfigurations",
	"discovery:GetDiscoverySummary",
	"discovery:ListConfigurations",
	"discovery:ListServerNeighbors",
	"discovery:StartContinuousExport",
	"discovery:StartDataCollectionByAgentIds",
	"discovery:StartExportTask",
	"discovery:StartImportTask",
	"discovery:StopContinuousExport",
	"discovery:StopDataCollectionByAgentIds",
	"discovery:UpdateApplication",
	"arsenal:RegisterOnPremisesAgent",
	"mgn:BatchDeleteSnapshotRequestForMgn",
	"mgn:CreateReplicationConfigurationTemplate",
	"mgn:DescribeJobs",
	"mgn:DescribeReplicationConfigurationTemplates",
	"mgn:DescribeReplicationServerAssociationsForMgn",
	"mgn:DescribeSnapshotRequestsForMgn",
	"mgn:DescribeSourceServers",
	"mgn:GetAgentInstallationAssetsForMgn",
	"mgn:GetChannelCommandsForMgn",
	"mgn:InitializeService",
	"mgn:ListTagsForResource",
	"mgn:RegisterAgentForMgn",
	"mgn:SendChannelCommandResultForMgn",
	"mgn:SendClientLogsForMgn",
	"mgn:SendClientMetricsForMgn",
	"mgn:TagResource",
	"mgn:UntagResource",
	"appstream:CreateDirectoryConfig",
	"appstream:CreateUsageReportSubscription",
	"appstream:CreateUser",
	"appstream:DeleteDirectoryConfig",
	"appstream:DeleteUsageReportSubscription",
	"appstream:DeleteUser",
	"appstream:DescribeDirectoryConfigs",
	"appstream:DescribeUsageReportSubscriptions",
	"appstream:DescribeUsers",
	"appstream:DisableUser",
	"appstream:EnableUser",
	"appstream:ExpireSession",
	"appstream:ListTagsForResource",
	"appstream:UpdateDirectoryConfig",
	"appsync:CreateApiCache",
	"appsync:CreateApiKey",
	"appsync:CreateDataSource",
	"appsync:CreateFunction",
	"appsync:CreateGraphqlApi",
	"appsync:CreateResolver",
	"appsync:CreateType",
	"appsync:DeleteApiCache",
	"appsync:DeleteApiKey",
	"appsync:DeleteDataSource",
	"appsync:DeleteFunction",
	"appsync:DeleteResolver",
	"appsync:DeleteType",
	"appsync:FlushApiCache",
	"appsync:GetApiCache",
	"appsync:GetDataSource",
	"appsync:GetFunction",
	"appsync:GetIntrospectionSchema",
	"appsync:GetResolver",
	"appsync:GetSchemaCreationStatus",
	"appsync:GetType",
	"appsync:ListApiKeys",
	"appsync:ListDataSources",
	"appsync:ListFunctions",
	"appsync:ListGraphqlApis",
	"appsync:ListResolvers",
	"appsync:ListResolversByFunction",
	"appsync:ListTypes",
	"appsync:SetWebACL",
	"appsync:StartSchemaCreation",
	"appsync:UpdateApiCache",
	"appsync:UpdateApiKey",
	"appsync:UpdateDataSource",
	"appsync:UpdateFunction",
	"appsync:UpdateResolver",
	"appsync:UpdateType",
	"athena:ListDataCatalogs",
	"athena:ListEngineVersions",
	"athena:ListWorkGroups",
	"auditmanager:CreateAssessment",
	"auditmanager:CreateAssessmentFramework",
	"auditmanager:CreateControl",
	"auditmanager:DeregisterAccount",
	"auditmanager:DeregisterOrganizationAdminAccount",
	"auditmanager:GetAccountStatus",
	"auditmanager:GetDelegations",
	"auditmanager:GetOrganizationAdminAccount",
	"auditmanager:GetServicesInScope",
	"auditmanager:GetSettings",
	"auditmanager:ListAssessmentFrameworks",
	"auditmanager:ListAssessmentReports",
	"auditmanager:ListAssessments",
	"auditmanager:ListControls",
	"auditmanager:ListKeywordsForDataSource",
	"auditmanager:ListNotifications",
	"auditmanager:RegisterAccount",
	"auditmanager:RegisterOrganizationAdminAccount",
	"auditmanager:UpdateSettings",
	"auditmanager:ValidateAssessmentReportIntegrity",
	"autoscaling-plans:CreateScalingPlan",
	"autoscaling-plans:DeleteScalingPlan",
	"autoscaling-plans:DescribeScalingPlanResources",
	"autoscaling-plans:DescribeScalingPlans",
	"autoscaling-plans:GetScalingPlanResourceForecastData",
	"autoscaling-plans:UpdateScalingPlan",
	"datasync:CreateAgent",
	"datasync:CreateLocationEfs",
	"datasync:CreateLocationFsxWindows",
	"datasync:CreateLocationNfs",
	"datasync:CreateLocationObjectStorage",
	"datasync:CreateLocationS3",
	"datasync:CreateLocationSmb",
	"datasync:CreateTask",
	"datasync:ListAgents",
	"datasync:ListLocations",
	"datasync:ListTaskExecutions",
	"datasync:ListTasks",
	"backup:CopyFromBackupVault",
	"backup:CopyIntoBackupVault",
	"backup:DescribeBackupJob",
	"backup:DescribeCopyJob",
	"backup:DescribeGlobalSettings",
	"backup:DescribeProtectedResource",
	"backup:DescribeRegionSettings",
	"backup:DescribeRestoreJob",
	"backup:ExportBackupPlanTemplate",
	"backup:GetBackupPlanFromJSON",
	"backup:GetBackupPlanFromTemplate",
	"backup:GetSupportedResourceTypes",
	"backup:ListBackupJobs",
	"backup:ListBackupPlanTemplates",
	"backup:ListBackupPlans",
	"backup:ListBackupVaults",
	"backup:ListCopyJobs",
	"backup:ListProtectedResources",
	"backup:ListRecoveryPointsByResource",
	"backup:ListRestoreJobs",
	"backup:StopBackupJob",
	"backup:UpdateGlobalSettings",
	"backup:UpdateRegionSettings",
	"backup-storage:MountCapsule",
	"batch:CreateComputeEnvironment",
	"batch:DescribeComputeEnvironments",
	"batch:DescribeJobDefinitions",
	"batch:DescribeJobQueues",
	"batch:DescribeJobs",
	"batch:ListJobs",
	"batch:RegisterJobDefinition",
	"aws-portal:ModifyAccount",
	"aws-portal:ModifyBilling",
	"aws-portal:ModifyPaymentMethods",
	"aws-portal:ViewAccount",
	"aws-portal:ViewBilling",
	"aws-portal:ViewPaymentMethods",
	"aws-portal:ViewUsage",
	"braket:GetDevice",
	"braket:SearchDevices",
	"braket:SearchQuantumTasks",
	"budgets:DescribeBudgetActionsForAccount",
	"bugbust:CreateEvent",
	"bugbust:ListEvents",
	"acm:GetAccountConfiguration",
	"acm:ListCertificates",
	"acm:ListTagsForCertificate",
	"acm:PutAccountConfiguration",
	"acm:RequestCertificate",
	"acm-pca:CreateCertificateAuthority",
	"acm-pca:ListCertificateAuthorities",
	"chatbot:CreateChimeWebhookConfiguration",
	"chatbot:CreateSlackChannelConfiguration",
	"chatbot:DeleteChimeWebhookConfiguration",
	"chatbot:DeleteSlackChannelConfiguration",
	"chatbot:DescribeChimeWebhookConfigurations",
	"chatbot:DescribeSlackChannelConfigurations",
	"chatbot:DescribeSlackChannels",
	"chatbot:DescribeSlackWorkspaces",
	"chatbot:GetSlackOauthParameters",
	"chatbot:RedeemSlackOauthCode",
	"chatbot:UpdateChimeWebhookConfiguration",
	"chatbot:UpdateSlackChannelConfiguration",
	"chime:AcceptDelegate",
	"chime:ActivateUsers",
	"chime:AddDomain",
	"chime:AddOrUpdateGroups",
	"chime:AssociatePhoneNumberWithUser",
	"chime:AssociatePhoneNumbersWithVoiceConnector",
	"chime:AssociatePhoneNumbersWithVoiceConnectorGroup",
	"chime:AssociateSigninDelegateGroupsWithAccount",
	"chime:AuthorizeDirectory",
	"chime:BatchCreateRoomMembership",
	"chime:BatchDeletePhoneNumber",
	"chime:BatchSuspendUser",
	"chime:BatchUnsuspendUser",
	"chime:BatchUpdatePhoneNumber",
	"chime:BatchUpdateUser",
	"chime:ConnectDirectory",
	"chime:CreateAccount",
	"chime:CreateApiKey",
	"chime:CreateAppInstance",
	"chime:CreateAppInstanceUser",
	"chime:CreateBot",
	"chime:CreateBotMembership",
	"chime:CreateCDRBucket",
	"chime:CreateMediaCapturePipeline",
	"chime:CreateMeeting",
	"chime:CreateMeetingWithAttendees",
	"chime:CreatePhoneNumberOrder",
	"chime:CreateProxySession",
	"chime:CreateRoom",
	"chime:CreateRoomMembership",
	"chime:CreateSipMediaApplication",
	"chime:CreateSipMediaApplicationCall",
	"chime:CreateSipRule",
	"chime:CreateUser",
	"chime:CreateVoiceConnector",
	"chime:CreateVoiceConnectorGroup",
	"chime:DeleteAccount",
	"chime:DeleteAccountOpenIdConfig",
	"chime:DeleteApiKey",
	"chime:DeleteCDRBucket",
	"chime:DeleteDelegate",
	"chime:DeleteDomain",
	"chime:DeleteEventsConfiguration",
	"chime:DeleteGroups",
	"chime:DeleteMediaCapturePipeline",
	"chime:DeletePhoneNumber",
	"chime:DeleteProxySession",
	"chime:DeleteRoom",
	"chime:DeleteRoomMembership",
	"chime:DeleteSipMediaApplication",
	"chime:DeleteSipRule",
	"chime:DeleteVoiceConnector",
	"chime:DeleteVoiceConnectorEmergencyCallingConfiguration",
	"chime:DeleteVoiceConnectorGroup",
	"chime:DeleteVoiceConnectorOrigination",
	"chime:DeleteVoiceConnectorProxy",
	"chime:DeleteVoiceConnectorStreamingConfiguration",
	"chime:DeleteVoiceConnectorTermination",
	"chime:DeleteVoiceConnectorTerminationCredentials",
	"chime:DisassociatePhoneNumberFromUser",
	"chime:DisassociatePhoneNumbersFromVoiceConnector",
	"chime:DisassociatePhoneNumbersFromVoiceConnectorGroup",
	"chime:DisassociateSigninDelegateGroupsFromAccount",
	"chime:DisconnectDirectory",
	"chime:GetAccount",
	"chime:GetAccountResource",
	"chime:GetAccountSettings",
	"chime:GetAccountWithOpenIdConfig",
	"chime:GetBot",
	"chime:GetCDRBucket",
	"chime:GetDomain",
	"chime:GetEventsConfiguration",
	"chime:GetGlobalSettings",
	"chime:GetMediaCapturePipeline",
	"chime:GetMeetingDetail",
	"chime:GetMessagingSessionEndpoint",
	"chime:GetPhoneNumber",
	"chime:GetPhoneNumberOrder",
	"chime:GetPhoneNumberSettings",
	"chime:GetProxySession",
	"chime:GetRetentionSettings",
	"chime:GetRoom",
	"chime:GetSipMediaApplication",
	"chime:GetSipMediaApplicationLoggingConfiguration",
	"chime:GetSipRule",
	"chime:GetTelephonyLimits",
	"chime:GetUser",
	"chime:GetUserActivityReportData",
	"chime:GetUserByEmail",
	"chime:GetUserSettings",
	"chime:GetVoiceConnector",
	"chime:GetVoiceConnectorEmergencyCallingConfiguration",
	"chime:GetVoiceConnectorGroup",
	"chime:GetVoiceConnectorLoggingConfiguration",
	"chime:GetVoiceConnectorOrigination",
	"chime:GetVoiceConnectorProxy",
	"chime:GetVoiceConnectorStreamingConfiguration",
	"chime:GetVoiceConnectorTermination",
	"chime:GetVoiceConnectorTerminationHealth",
	"chime:InviteDelegate",
	"chime:InviteUsers",
	"chime:InviteUsersFromProvider",
	"chime:ListAccountUsageReportData",
	"chime:ListAccounts",
	"chime:ListApiKeys",
	"chime:ListBots",
	"chime:ListCDRBucket",
	"chime:ListCallingRegions",
	"chime:ListDelegates",
	"chime:ListDirectories",
	"chime:ListDomains",
	"chime:ListGroups",
	"chime:ListMediaCapturePipelines",
	"chime:ListMeetingEvents",
	"chime:ListMeetings",
	"chime:ListMeetingsReportData",
	"chime:ListPhoneNumberOrders",
	"chime:ListPhoneNumbers",
	"chime:ListProxySessions",
	"chime:ListRoomMemberships",
	"chime:ListRooms",
	"chime:ListSipMediaApplications",
	"chime:ListSipRules",
	"chime:ListSupportedPhoneNumberCountries",
	"chime:ListUsers",
	"chime:ListVoiceConnectorGroups",
	"chime:ListVoiceConnectorTerminationCredentials",
	"chime:ListVoiceConnectors",
	"chime:LogoutUser",
	"chime:PutEventsConfiguration",
	"chime:PutRetentionSettings",
	"chime:PutSipMediaApplicationLoggingConfiguration",
	"chime:PutVoiceConnectorEmergencyCallingConfiguration",
	"chime:PutVoiceConnectorLoggingConfiguration",
	"chime:PutVoiceConnectorOrigination",
	"chime:PutVoiceConnectorProxy",
	"chime:PutVoiceConnectorStreamingConfiguration",
	"chime:PutVoiceConnectorTermination",
	"chime:PutVoiceConnectorTerminationCredentials",
	"chime:RedactConversationMessage",
	"chime:RedactRoomMessage",
	"chime:RegenerateSecurityToken",
	"chime:RenameAccount",
	"chime:RenewDelegate",
	"chime:ResetAccountResource",
	"chime:ResetPersonalPIN",
	"chime:RestorePhoneNumber",
	"chime:RetrieveDataExports",
	"chime:SearchAvailablePhoneNumbers",
	"chime:StartDataExport",
	"chime:SubmitSupportRequest",
	"chime:SuspendUsers",
	"chime:UnauthorizeDirectory",
	"chime:UpdateAccount",
	"chime:UpdateAccountOpenIdConfig",
	"chime:UpdateAccountResource",
	"chime:UpdateAccountSettings",
	"chime:UpdateBot",
	"chime:UpdateCDRSettings",
	"chime:UpdateGlobalSettings",
	"chime:UpdatePhoneNumber",
	"chime:UpdatePhoneNumberSettings",
	"chime:UpdateProxySession",
	"chime:UpdateRoom",
	"chime:UpdateRoomMembership",
	"chime:UpdateSipMediaApplication",
	"chime:UpdateSipMediaApplicationCall",
	"chime:UpdateSipRule",
	"chime:UpdateSupportedLicenses",
	"chime:UpdateUser",
	"chime:UpdateUserLicenses",
	"chime:UpdateUserSettings",
	"chime:UpdateVoiceConnector",
	"chime:UpdateVoiceConnectorGroup",
	"chime:ValidateAccountResource",
	"clouddirectory:CreateSchema",
	"clouddirectory:ListDevelopmentSchemaArns",
	"clouddirectory:ListDirectories",
	"clouddirectory:ListManagedSchemaArns",
	"clouddirectory:ListPublishedSchemaArns",
	"clouddirectory:PutSchemaFromJson",
	"servicediscovery:CreateHttpNamespace",
	"servicediscovery:CreatePrivateDnsNamespace",
	"servicediscovery:CreatePublicDnsNamespace",
	"servicediscovery:DiscoverInstances",
	"servicediscovery:GetInstance",
	"servicediscovery:GetInstancesHealthStatus",
	"servicediscovery:GetOperation",
	"servicediscovery:ListInstances",
	"servicediscovery:ListNamespaces",
	"servicediscovery:ListOperations",
	"servicediscovery:ListServices",
	"servicediscovery:ListTagsForResource",
	"servicediscovery:TagResource",
	"servicediscovery:UntagResource",
	"servicediscovery:UpdateInstanceCustomHealthStatus",
	"cloud9:CreateEnvironmentEC2",
	"cloud9:GetUserSettings",
	"cloud9:ListEnvironments",
	"cloud9:UpdateUserSettings",
	"cloudformation:CreateStackSet",
	"cloudformation:CreateUploadBucket",
	"cloudformation:DeregisterType",
	"cloudformation:DescribeAccountLimits",
	"cloudformation:DescribeStackDriftDetectionStatus",
	"cloudformation:DescribeType",
	"cloudformation:DescribeTypeRegistration",
	"cloudformation:EstimateTemplateCost",
	"cloudformation:ListExports",
	"cloudformation:ListImports",
	"cloudformation:ListStacks",
	"cloudformation:ListTypeRegistrations",
	"cloudformation:ListTypeVersions",
	"cloudformation:ListTypes",
	"cloudformation:RegisterType",
	"cloudformation:SetTypeDefaultVersion",
	"cloudformation:ValidateTemplate",
	"cloudfront:CreateCachePolicy",
	"cloudfront:CreateFieldLevelEncryptionConfig",
	"cloudfront:CreateFieldLevelEncryptionProfile",
	"cloudfront:CreateFunction",
	"cloudfront:CreateKeyGroup",
	"cloudfront:CreateMonitoringSubscription",
	"cloudfront:CreateOriginRequestPolicy",
	"cloudfront:CreatePublicKey",
	"cloudfront:CreateRealtimeLogConfig",
	"cloudfront:DeleteCachePolicy",
	"cloudfront:DeleteFieldLevelEncryptionConfig",
	"cloudfront:DeleteFieldLevelEncryptionProfile",
	"cloudfront:DeleteFunction",
	"cloudfront:DeleteKeyGroup",
	"cloudfront:DeleteMonitoringSubscription",
	"cloudfront:DeleteOriginRequestPolicy",
	"cloudfront:DeletePublicKey",
	"cloudfront:DeleteRealtimeLogConfig",
	"cloudfront:DescribeFunction",
	"cloudfront:GetCachePolicy",
	"cloudfront:GetCachePolicyConfig",
	"cloudfront:GetFieldLevelEncryption",
	"cloudfront:GetFieldLevelEncryptionConfig",
	"cloudfront:GetFieldLevelEncryptionProfile",
	"cloudfront:GetFieldLevelEncryptionProfileConfig",
	"cloudfront:GetFunction",
	"cloudfront:GetKeyGroup",
	"cloudfront:GetKeyGroupConfig",
	"cloudfront:GetMonitoringSubscription",
	"cloudfront:GetOriginRequestPolicy",
	"cloudfront:GetOriginRequestPolicyConfig",
	"cloudfront:GetPublicKey",
	"cloudfront:GetPublicKeyConfig",
	"cloudfront:GetRealtimeLogConfig",
	"cloudfront:ListCachePolicies",
	"cloudfront:ListCloudFrontOriginAccessIdentities",
	"cloudfront:ListDistributions",
	"cloudfront:ListDistributionsByCachePolicyId",
	"cloudfront:ListDistributionsByKeyGroup",
	"cloudfront:ListDistributionsByOriginRequestPolicyId",
	"cloudfront:ListDistributionsByRealtimeLogConfig",
	"cloudfront:ListDistributionsByWebACLId",
	"cloudfront:ListFieldLevelEncryptionConfigs",
	"cloudfront:ListFieldLevelEncryptionProfiles",
	"cloudfront:ListFunctions",
	"cloudfront:ListKeyGroups",
	"cloudfront:ListOriginRequestPolicies",
	"cloudfront:ListPublicKeys",
	"cloudfront:ListRealtimeLogConfigs",
	"cloudfront:ListStreamingDistributions",
	"cloudfront:PublishFunction",
	"cloudfront:TestFunction",
	"cloudfront:UpdateCachePolicy",
	"cloudfront:UpdateFieldLevelEncryptionConfig",
	"cloudfront:UpdateFieldLevelEncryptionProfile",
	"cloudfront:UpdateFunction",
	"cloudfront:UpdateKeyGroup",
	"cloudfront:UpdateOriginRequestPolicy",
	"cloudfront:UpdatePublicKey",
	"cloudfront:UpdateRealtimeLogConfig",
	"cloudhsm:AddTagsToResource",
	"cloudhsm:CreateHapg",
	"cloudhsm:CreateLunaClient",
	"cloudhsm:DeleteHapg",
	"cloudhsm:DeleteHsm",
	"cloudhsm:DeleteLunaClient",
	"cloudhsm:DescribeBackups",
	"cloudhsm:DescribeClusters",
	"cloudhsm:DescribeHapg",
	"cloudhsm:DescribeHsm",
	"cloudhsm:DescribeLunaClient",
	"cloudhsm:GetConfig",
	"cloudhsm:ListAvailableZones",
	"cloudhsm:ListHapgs",
	"cloudhsm:ListHsms",
	"cloudhsm:ListLunaClients",
	"cloudhsm:ListTagsForResource",
	"cloudhsm:ModifyHapg",
	"cloudhsm:ModifyHsm",
	"cloudhsm:ModifyLunaClient",
	"cloudhsm:RemoveTagsFromResource",
	"cloudshell:CreateEnvironment",
	"cloudtrail:DescribeTrails",
	"cloudtrail:GetTrail",
	"cloudtrail:ListPublicKeys",
	"cloudtrail:ListTrails",
	"cloudtrail:LookupEvents",
	"cloudwatch:DeleteAnomalyDetector",
	"cloudwatch:DescribeAlarmsForMetric",
	"cloudwatch:DescribeAnomalyDetectors",
	"cloudwatch:DescribeInsightRules",
	"cloudwatch:GetMetricData",
	"cloudwatch:GetMetricStatistics",
	"cloudwatch:GetMetricWidgetImage",
	"cloudwatch:ListDashboards",
	"cloudwatch:ListMetricStreams",
	"cloudwatch:ListMetrics",
	"cloudwatch:CloudWatch-Alarm",
	"cloudwatch:CloudWatch-InsightRule",
	"cloudwatch:PutAnomalyDetector",
	"cloudwatch:PutMetricData",
	"cloudwatch:CloudWatch-Alarm",
	"cloudwatch:CloudWatch-InsightRule",
	"cloudwatch:CloudWatch-Alarm",
	"cloudwatch:CloudWatch-InsightRule",
	"applicationinsights:CreateApplication",
	"applicationinsights:CreateComponent",
	"applicationinsights:CreateLogPattern",
	"applicationinsights:DeleteApplication",
	"applicationinsights:DeleteComponent",
	"applicationinsights:DeleteLogPattern",
	"applicationinsights:DescribeApplication",
	"applicationinsights:DescribeComponent",
	"applicationinsights:DescribeComponentConfiguration",
	"applicationinsights:DescribeComponentConfigurationRecommendation",
	"applicationinsights:DescribeLogPattern",
	"applicationinsights:DescribeObservation",
	"applicationinsights:DescribeProblem",
	"applicationinsights:DescribeProblemObservations",
	"applicationinsights:ListApplications",
	"applicationinsights:ListComponents",
	"applicationinsights:ListConfigurationHistory",
	"applicationinsights:ListLogPatternSets",
	"applicationinsights:ListLogPatterns",
	"applicationinsights:ListProblems",
	"applicationinsights:ListTagsForResource",
	"applicationinsights:TagResource",
	"applicationinsights:UntagResource",
	"applicationinsights:UpdateApplication",
	"applicationinsights:UpdateComponent",
	"applicationinsights:UpdateComponentConfiguration",
	"applicationinsights:UpdateLogPattern",
	"logs:CancelExportTask",
	"logs:CreateLogDelivery [permission only]",
	"logs:DeleteDestination",
	"logs:DeleteLogDelivery [permission only]",
	"logs:DeleteQueryDefinition",
	"logs:DeleteResourcePolicy",
	"logs:DescribeDestinations",
	"logs:DescribeExportTasks",
	"logs:DescribeQueries",
	"logs:DescribeQueryDefinitions",
	"logs:DescribeResourcePolicies",
	"logs:GetLogDelivery [permission only]",
	"logs:GetLogRecord",
	"logs:GetQueryResults",
	"logs:ListLogDeliveries [permission only]",
	"logs:PutDestination",
	"logs:PutDestinationPolicy",
	"logs:PutQueryDefinition",
	"logs:PutResourcePolicy",
	"logs:StopQuery",
	"logs:TestMetricFilter",
	"logs:UpdateLogDelivery [permission only]",
	"synthetics:CreateCanary",
	"synthetics:DescribeCanaries",
	"synthetics:DescribeCanariesLastRun",
	"synthetics:DescribeRuntimeVersions",
	"codeartifact:CreateDomain",
	"codeartifact:CreateRepository",
	"codeartifact:ListDomains",
	"codeartifact:ListRepositories",
	"codebuild:DeleteOAuthToken",
	"codebuild:DeleteSourceCredentials",
	"codebuild:ImportSourceCredentials",
	"codebuild:ListBuildBatches",
	"codebuild:ListBuilds",
	"codebuild:ListConnectedOAuthAccounts",
	"codebuild:ListCuratedEnvironmentImages",
	"codebuild:ListProjects",
	"codebuild:ListReportGroups",
	"codebuild:ListReports",
	"codebuild:ListRepositories",
	"codebuild:ListSharedProjects",
	"codebuild:ListSharedReportGroups",
	"codebuild:ListSourceCredentials",
	"codebuild:PersistOAuthToken",
	"codecommit:CreateApprovalRuleTemplate",
	"codecommit:DeleteApprovalRuleTemplate",
	"codecommit:GetApprovalRuleTemplate",
	"codecommit:ListApprovalRuleTemplates",
	"codecommit:ListRepositories",
	"codecommit:ListRepositoriesForApprovalRuleTemplate",
	"codecommit:UpdateApprovalRuleTemplateContent",
	"codecommit:UpdateApprovalRuleTemplateDescription",
	"codecommit:UpdateApprovalRuleTemplateName",
	"codedeploy:BatchGetDeploymentTargets",
	"codedeploy:ContinueDeployment",
	"codedeploy:CreateCloudFormationDeployment [permission only]",
	"codedeploy:DeleteGitHubAccountToken",
	"codedeploy:DeleteResourcesByExternalId",
	"codedeploy:GetDeploymentTarget",
	"codedeploy:ListApplications",
	"codedeploy:ListDeploymentConfigs",
	"codedeploy:ListDeploymentTargets",
	"codedeploy:ListGitHubAccountTokenNames",
	"codedeploy:ListOnPremisesInstances",
	"codedeploy:PutLifecycleEventHookExecutionStatus",
	"codedeploy:SkipWaitTimeForInstanceTermination",
	"codedeploy:StopDeployment",
	"codeguru:GetCodeGuruFreeTrialSummary",
	"codeguru-profiler:CreateProfilingGroup",
	"codeguru-profiler:GetFindingsReportAccountSummary",
	"codeguru-profiler:ListProfilingGroups",
	"codeguru-reviewer:CreateConnectionToken [permission only]",
	"codeguru-reviewer:GetMetricsData [permission only]",
	"codeguru-reviewer:ListCodeReviews",
	"codeguru-reviewer:ListRepositoryAssociations",
	"codeguru-reviewer:ListThirdPartyRepositories [permission only]",
	"codepipeline:AcknowledgeJob",
	"codepipeline:AcknowledgeThirdPartyJob",
	"codepipeline:GetJobDetails",
	"codepipeline:GetThirdPartyJobDetails",
	"codepipeline:ListActionTypes",
	"codepipeline:ListPipelines",
	"codepipeline:PollForThirdPartyJobs",
	"codepipeline:PutJobFailureResult",
	"codepipeline:PutJobSuccessResult",
	"codepipeline:PutThirdPartyJobFailureResult",
	"codepipeline:PutThirdPartyJobSuccessResult",
	"codestar:CreateProject",
	"codestar:DescribeUserProfile",
	"codestar:ListProjects",
	"codestar:ListUserProfiles",
	"codestar-connections:CreateConnection",
	"codestar-connections:CreateHost",
	"codestar-connections:GetIndividualAccessToken",
	"codestar-connections:GetInstallationUrl",
	"codestar-connections:ListConnections",
	"codestar-connections:ListHosts",
	"codestar-connections:ListInstallationTargets",
	"codestar-connections:RegisterAppCode",
	"codestar-connections:StartAppRegistrationHandshake",
	"codestar-connections:StartOAuthHandshake",
	"codestar-notifications:DeleteTarget",
	"codestar-notifications:ListEventTypes",
	"codestar-notifications:ListNotificationRules",
	"codestar-notifications:ListTargets",
	"cognito-identity:CreateIdentityPool",
	"cognito-identity:DeleteIdentities",
	"cognito-identity:DescribeIdentity",
	"cognito-identity:GetCredentialsForIdentity",
	"cognito-identity:GetId",
	"cognito-identity:GetOpenIdToken",
	"cognito-identity:ListIdentityPools",
	"cognito-identity:SetIdentityPoolRoles",
	"cognito-identity:SetPrincipalTagAttributeMap",
	"cognito-identity:UnlinkIdentity",
	"cognito-sync:QueryRecords [permission only]",
	"cognito-idp:AssociateSoftwareToken",
	"cognito-idp:ChangePassword",
	"cognito-idp:ConfirmDevice",
	"cognito-idp:ConfirmForgotPassword",
	"cognito-idp:ConfirmSignUp",
	"cognito-idp:CreateUserPool",
	"cognito-idp:DeleteUser",
	"cognito-idp:DeleteUserAttributes",
	"cognito-idp:DescribeUserPoolDomain",
	"cognito-idp:ForgetDevice",
	"cognito-idp:ForgotPassword",
	"cognito-idp:GetDevice",
	"cognito-idp:GetUser",
	"cognito-idp:GetUserAttributeVerificationCode",
	"cognito-idp:GlobalSignOut",
	"cognito-idp:InitiateAuth",
	"cognito-idp:ListDevices",
	"cognito-idp:ListUserPools",
	"cognito-idp:ResendConfirmationCode",
	"cognito-idp:RespondToAuthChallenge",
	"cognito-idp:SetUserMFAPreference",
	"cognito-idp:SetUserSettings",
	"cognito-idp:SignUp",
	"cognito-idp:UpdateDeviceStatus",
	"cognito-idp:UpdateUserAttributes",
	"cognito-idp:VerifySoftwareToken",
	"cognito-idp:VerifyUserAttribute",
	"comprehend:BatchDetectDominantLanguage",
	"comprehend:BatchDetectEntities",
	"comprehend:BatchDetectKeyPhrases",
	"comprehend:BatchDetectSentiment",
	"comprehend:BatchDetectSyntax",
	"comprehend:ContainsPiiEntities",
	"comprehend:CreateDocumentClassifier",
	"comprehend:CreateEntityRecognizer",
	"comprehend:DescribeDocumentClassificationJob",
	"comprehend:DescribeDominantLanguageDetectionJob",
	"comprehend:DescribeEntitiesDetectionJob",
	"comprehend:DescribeEventsDetectionJob",
	"comprehend:DescribeKeyPhrasesDetectionJob",
	"comprehend:DescribePiiEntitiesDetectionJob",
	"comprehend:DescribeSentimentDetectionJob",
	"comprehend:DescribeTopicsDetectionJob",
	"comprehend:DetectDominantLanguage",
	"comprehend:DetectKeyPhrases",
	"comprehend:DetectPiiEntities",
	"comprehend:DetectSentiment",
	"comprehend:DetectSyntax",
	"comprehend:ListDocumentClassificationJobs",
	"comprehend:ListDocumentClassifiers",
	"comprehend:ListDominantLanguageDetectionJobs",
	"comprehend:ListEndpoints",
	"comprehend:ListEntitiesDetectionJobs",
	"comprehend:ListEntityRecognizers",
	"comprehend:ListEventsDetectionJobs",
	"comprehend:ListKeyPhrasesDetectionJobs",
	"comprehend:ListPiiEntitiesDetectionJobs",
	"comprehend:ListSentimentDetectionJobs",
	"comprehend:ListTopicsDetectionJobs",
	"comprehend:StartDominantLanguageDetectionJob",
	"comprehend:StartEventsDetectionJob",
	"comprehend:StartKeyPhrasesDetectionJob",
	"comprehend:StartPiiEntitiesDetectionJob",
	"comprehend:StartSentimentDetectionJob",
	"comprehend:StartTopicsDetectionJob",
	"comprehend:StopDominantLanguageDetectionJob",
	"comprehend:StopEntitiesDetectionJob",
	"comprehend:StopEventsDetectionJob",
	"comprehend:StopKeyPhrasesDetectionJob",
	"comprehend:StopPiiEntitiesDetectionJob",
	"comprehend:StopSentimentDetectionJob",
	"comprehendmedical:DescribeEntitiesDetectionV2Job",
	"comprehendmedical:DescribeICD10CMInferenceJob",
	"comprehendmedical:DescribePHIDetectionJob",
	"comprehendmedical:DescribeRxNormInferenceJob",
	"comprehendmedical:DetectEntitiesV2",
	"comprehendmedical:DetectPHI",
	"comprehendmedical:InferICD10CM",
	"comprehendmedical:InferRxNorm",
	"comprehendmedical:ListEntitiesDetectionV2Jobs",
	"comprehendmedical:ListICD10CMInferenceJobs",
	"comprehendmedical:ListPHIDetectionJobs",
	"comprehendmedical:ListRxNormInferenceJobs",
	"comprehendmedical:StartEntitiesDetectionV2Job",
	"comprehendmedical:StartICD10CMInferenceJob",
	"comprehendmedical:StartPHIDetectionJob",
	"comprehendmedical:StartRxNormInferenceJob",
	"comprehendmedical:StopEntitiesDetectionV2Job",
	"comprehendmedical:StopICD10CMInferenceJob",
	"comprehendmedical:StopPHIDetectionJob",
	"comprehendmedical:StopRxNormInferenceJob",
	"compute-optimizer:DescribeRecommendationExportJobs",
	"compute-optimizer:ExportAutoScalingGroupRecommendations",
	"compute-optimizer:ExportEBSVolumeRecommendations",
	"compute-optimizer:ExportEC2InstanceRecommendations",
	"compute-optimizer:ExportLambdaFunctionRecommendations",
	"compute-optimizer:GetAutoScalingGroupRecommendations",
	"compute-optimizer:GetEBSVolumeRecommendations",
	"compute-optimizer:GetEC2InstanceRecommendations",
	"compute-optimizer:GetEC2RecommendationProjectedMetrics",
	"compute-optimizer:GetEnrollmentStatus",
	"compute-optimizer:GetLambdaFunctionRecommendations",
	"compute-optimizer:GetRecommendationSummaries",
	"compute-optimizer:UpdateEnrollmentStatus",
	"config:BatchGetResourceConfig",
	"config:DeleteConfigurationRecorder",
	"config:DeleteConformancePack",
	"config:DeleteDeliveryChannel",
	"config:DeleteOrganizationConfigRule",
	"config:DeleteOrganizationConformancePack",
	"config:DeletePendingAggregationRequest",
	"config:DeleteRemediationExceptions",
	"config:DeleteResourceConfig",
	"config:DeleteRetentionConfiguration",
	"config:DeliverConfigSnapshot",
	"config:DescribeAggregationAuthorizations",
	"config:DescribeComplianceByResource",
	"config:DescribeConfigurationAggregators",
	"config:DescribeConfigurationRecorderStatus",
	"config:DescribeConfigurationRecorders",
	"config:DescribeConformancePackCompliance",
	"config:DescribeConformancePackStatus",
	"config:DescribeConformancePacks",
	"config:DescribeDeliveryChannelStatus",
	"config:DescribeDeliveryChannels",
	"config:DescribeOrganizationConfigRuleStatuses",
	"config:DescribeOrganizationConfigRules",
	"config:DescribeOrganizationConformancePackStatuses",
	"config:DescribeOrganizationConformancePacks",
	"config:DescribePendingAggregationRequests",
	"config:DescribeRemediationExceptions",
	"config:DescribeRetentionConfigurations",
	"config:GetComplianceDetailsByResource",
	"config:GetComplianceSummaryByConfigRule",
	"config:GetComplianceSummaryByResourceType",
	"config:GetConformancePackComplianceDetails",
	"config:GetConformancePackComplianceSummary",
	"config:GetDiscoveredResourceCounts",
	"config:GetOrganizationConfigRuleDetailedStatus",
	"config:GetOrganizationConformancePackDetailedStatus",
	"config:GetResourceConfigHistory",
	"config:ListDiscoveredResources",
	"config:PutConfigurationRecorder",
	"config:PutConformancePack",
	"config:PutDeliveryChannel",
	"config:PutEvaluations",
	"config:PutExternalEvaluation",
	"config:PutOrganizationConfigRule",
	"config:PutOrganizationConformancePack",
	"config:PutRemediationExceptions",
	"config:PutResourceConfig",
	"config:PutRetentionConfiguration",
	"config:SelectAggregateResourceConfig",
	"config:SelectResourceConfig",
	"config:StartConfigurationRecorder",
	"config:StopConfigurationRecorder",
	"connect:CreateInstance",
	"connect:ListInstances",
	"profile:CreateDomain",
	"profile:GetProfileObjectTypeTemplate",
	"profile:ListAccountIntegrations",
	"profile:ListDomains",
	"profile:ListProfileObjectTypeTemplates",
	"profile:ListTagsForResource",
	"profile:TagResource",
	"profile:UntagResource",
	"awsconnector:GetConnectorHealth",
	"awsconnector:RegisterConnector",
	"awsconnector:ValidateConnectorId",
	"controltower:CreateManagedAccount",
	"controltower:DeregisterManagedAccount",
	"controltower:DeregisterOrganizationalUnit",
	"controltower:DescribeAccountFactoryConfig",
	"controltower:DescribeCoreService",
	"controltower:DescribeGuardrail",
	"controltower:DescribeGuardrailForTarget",
	"controltower:DescribeManagedAccount",
	"controltower:DescribeManagedOrganizationalUnit",
	"controltower:DescribeSingleSignOn",
	"controltower:DisableGuardrail",
	"controltower:EnableGuardrail",
	"controltower:GetAvailableUpdates",
	"controltower:GetGuardrailComplianceStatus",
	"controltower:GetHomeRegion",
	"controltower:GetLandingZoneStatus",
	"controltower:ListDirectoryGroups",
	"controltower:ListEnabledGuardrails",
	"controltower:ListGuardrailViolations",
	"controltower:ListGuardrails",
	"controltower:ListGuardrailsForTarget",
	"controltower:ListManagedAccounts",
	"controltower:ListManagedAccountsForGuardrail",
	"controltower:ListManagedAccountsForParent",
	"controltower:ListManagedOrganizationalUnits",
	"controltower:ListManagedOrganizationalUnitsForGuardrail",
	"controltower:ManageOrganizationalUnit",
	"controltower:SetupLandingZone",
	"controltower:UpdateAccountFactoryConfig",
	"cur:DescribeReportDefinitions",
	"ce:CreateAnomalyMonitor",
	"ce:CreateAnomalySubscription",
	"ce:CreateCostCategoryDefinition",
	"ce:CreateNotificationSubscription",
	"ce:CreateReport",
	"ce:DeleteAnomalyMonitor",
	"ce:DeleteAnomalySubscription",
	"ce:DeleteCostCategoryDefinition",
	"ce:DeleteNotificationSubscription",
	"ce:DeleteReport",
	"ce:DescribeCostCategoryDefinition",
	"ce:DescribeNotificationSubscription",
	"ce:DescribeReport",
	"ce:GetAnomalies",
	"ce:GetAnomalyMonitors",
	"ce:GetAnomalySubscriptions",
	"ce:GetCostAndUsage",
	"ce:GetCostAndUsageWithResources",
	"ce:GetCostCategories",
	"ce:GetCostForecast",
	"ce:GetDimensionValues",
	"ce:GetPreferences",
	"ce:GetReservationCoverage",
	"ce:GetReservationPurchaseRecommendation",
	"ce:GetReservationUtilization",
	"ce:GetRightsizingRecommendation",
	"ce:GetSavingsPlansCoverage",
	"ce:GetSavingsPlansPurchaseRecommendation",
	"ce:GetSavingsPlansUtilization",
	"ce:GetSavingsPlansUtilizationDetails",
	"ce:GetTags",
	"ce:GetUsageForecast",
	"ce:ListCostCategoryDefinitions",
	"ce:ProvideAnomalyFeedback",
	"ce:UpdateAnomalyMonitor",
	"ce:UpdateAnomalySubscription",
	"ce:UpdateCostCategoryDefinition",
	"ce:UpdateNotificationSubscription",
	"ce:UpdatePreferences",
	"ce:UpdateReport",
	"dataexchange:CreateAsset",
	"dataexchange:CreateDataSet",
	"dataexchange:CreateRevision",
	"dlm:CreateLifecyclePolicy",
	"dlm:GetLifecyclePolicies",
	"datapipeline:ActivatePipeline",
	"datapipeline:AddTags",
	"datapipeline:CreatePipeline",
	"datapipeline:DeactivatePipeline",
	"datapipeline:DeletePipeline",
	"datapipeline:DescribeObjects",
	"datapipeline:DescribePipelines",
	"datapipeline:EvaluateExpression",
	"datapipeline:GetAccountLimits",
	"datapipeline:GetPipelineDefinition",
	"datapipeline:ListPipelines",
	"datapipeline:PollForTask",
	"datapipeline:PutAccountLimits",
	"datapipeline:PutPipelineDefinition",
	"datapipeline:QueryObjects",
	"datapipeline:RemoveTags",
	"datapipeline:ReportTaskProgress",
	"datapipeline:ReportTaskRunnerHeartbeat",
	"datapipeline:SetStatus",
	"datapipeline:SetTaskStatus",
	"datapipeline:ValidatePipelineDefinition",
	"dms:CreateEndpoint",
	"dms:CreateEventSubscription",
	"dms:CreateReplicationInstance",
	"dms:CreateReplicationSubnetGroup",
	"dms:DescribeAccountAttributes",
	"dms:DescribeCertificates",
	"dms:DescribeConnections",
	"dms:DescribeEndpointTypes",
	"dms:DescribeEndpoints",
	"dms:DescribeEventCategories",
	"dms:DescribeEventSubscriptions",
	"dms:DescribeEvents",
	"dms:DescribeOrderableReplicationInstances",
	"dms:DescribeReplicationInstances",
	"dms:DescribeReplicationSubnetGroups",
	"dms:DescribeReplicationTasks",
	"dms:ImportCertificate",
	"dms:ModifyEventSubscription",
	"dms:ModifyReplicationSubnetGroup",
	"dbqms:CreateFavoriteQuery",
	"dbqms:CreateQueryHistory",
	"dbqms:CreateTab",
	"dbqms:DeleteFavoriteQueries",
	"dbqms:DeleteQueryHistory",
	"dbqms:DeleteTab",
	"dbqms:DescribeFavoriteQueries",
	"dbqms:DescribeQueryHistory",
	"dbqms:DescribeTabs",
	"dbqms:GetQueryString",
	"dbqms:UpdateFavoriteQuery",
	"dbqms:UpdateQueryHistory",
	"dbqms:UpdateTab",
	"deepcomposer:AssociateCoupon",
	"deeplens:AssociateServiceRoleToAccount",
	"deeplens:CreateDeviceCertificates",
	"deeplens:CreateModel",
	"deeplens:CreateProject",
	"deeplens:GetAssociatedResources",
	"deeplens:GetDeploymentStatus",
	"deeplens:ImportProjectFromTemplate",
	"deeplens:ListDeployments",
	"deeplens:ListDevices",
	"deeplens:ListModels",
	"deeplens:ListProjects",
	"deeplens:RegisterDevice",
	"deepracer:CreateAccountResources",
	"deepracer:DeleteAccountResources",
	"deepracer:GetAccountResources",
	"deepracer:GetAlias",
	"deepracer:ListLeaderboards",
	"deepracer:ListModels",
	"deepracer:ListTracks",
	"deepracer:SetAlias",
	"deepracer:TestRewardFunction",
	"detective:AcceptInvitation",
	"detective:CreateGraph",
	"detective:DisassociateMembership",
	"detective:GetPricingInformation [permission only]",
	"detective:ListGraphs",
	"detective:ListInvitations",
	"detective:RejectInvitation",
	"devicefarm:CreateInstanceProfile",
	"devicefarm:CreateProject",
	"devicefarm:CreateTestGridProject",
	"devicefarm:CreateVPCEConfiguration",
	"devicefarm:GetAccountSettings",
	"devicefarm:GetOfferingStatus",
	"devicefarm:ListDeviceInstances",
	"devicefarm:ListDevices",
	"devicefarm:ListInstanceProfiles",
	"devicefarm:ListOfferingPromotions",
	"devicefarm:ListOfferingTransactions",
	"devicefarm:ListOfferings",
	"devicefarm:ListProjects",
	"devicefarm:ListTestGridProjects",
	"devicefarm:ListVPCEConfigurations",
	"devicefarm:PurchaseOffering",
	"devicefarm:RenewOffering",
	"devicefarm:Device Pool as filter",
	"devicefarm:Device Selection Configuration as filter",
	"devops-guru:DescribeAccountHealth",
	"devops-guru:DescribeAccountOverview",
	"devops-guru:DescribeAnomaly",
	"devops-guru:DescribeFeedback",
	"devops-guru:DescribeInsight",
	"devops-guru:DescribeResourceCollectionHealth",
	"devops-guru:DescribeServiceIntegration",
	"devops-guru:GetCostEstimation",
	"devops-guru:GetResourceCollection",
	"devops-guru:ListAnomaliesForInsight",
	"devops-guru:ListEvents",
	"devops-guru:ListInsights",
	"devops-guru:ListNotificationChannels",
	"devops-guru:ListRecommendations",
	"devops-guru:PutFeedback",
	"devops-guru:SearchInsights",
	"devops-guru:StartCostEstimation",
	"devops-guru:UpdateResourceCollection",
	"devops-guru:UpdateServiceIntegration",
	"directconnect:CreateDirectConnectGateway",
	"directconnect:DeleteDirectConnectGatewayAssociationProposal",
	"directconnect:DescribeLocations",
	"directconnect:DescribeVirtualGateways",
	"directconnect:UpdateDirectConnectGatewayAssociation",
	"ds:CheckAlias [permission only]",
	"ds:ConnectDirectory",
	"ds:CreateDirectory",
	"ds:CreateIdentityPoolDirectory [permission only]",
	"ds:CreateMicrosoftAD",
	"ds:DescribeDirectories",
	"ds:DescribeSnapshots",
	"ds:DescribeTrusts",
	"ds:GetDirectoryLimits",
	"ds:ListLogSubscriptions",
	"dynamodb:DescribeLimits",
	"dynamodb:DescribeReservedCapacity",
	"dynamodb:DescribeReservedCapacityOfferings",
	"dynamodb:ListBackups",
	"dynamodb:ListContributorInsights",
	"dynamodb:ListExports",
	"dynamodb:ListGlobalTables",
	"dynamodb:ListStreams",
	"dynamodb:ListTables",
	"dynamodb:PurchaseReservedCapacityOfferings",
	"dax:CreateParameterGroup",
	"dax:CreateSubnetGroup",
	"dax:DeleteParameterGroup",
	"dax:DeleteSubnetGroup",
	"dax:DescribeDefaultParameters",
	"dax:DescribeEvents",
	"dax:DescribeParameterGroups",
	"dax:DescribeParameters",
	"dax:DescribeSubnetGroups",
	"dax:UpdateParameterGroup",
	"dax:UpdateSubnetGroup",
	"ec2:AdvertiseByoipCidr",
	"ec2:CancelBundleTask",
	"ec2:CancelConversionTask",
	"ec2:CancelReservedInstancesListing",
	"ec2:CreateDefaultSubnet",
	"ec2:CreateDefaultVpc",
	"ec2:CreateSpotDatafeedSubscription",
	"ec2:DeleteNetworkInterfacePermission",
	"ec2:DeletePlacementGroup",
	"ec2:DeleteSpotDatafeedSubscription",
	"ec2:DeprovisionByoipCidr",
	"ec2:DeregisterInstanceEventNotificationAttributes",
	"ec2:DescribeAccountAttributes",
	"ec2:DescribeAddresses",
	"ec2:DescribeAddressesAttribute",
	"ec2:DescribeAggregateIdFormat",
	"ec2:DescribeAvailabilityZones",
	"ec2:DescribeBundleTasks",
	"ec2:DescribeByoipCidrs",
	"ec2:DescribeCapacityReservations",
	"ec2:DescribeCarrierGateways",
	"ec2:DescribeClassicLinkInstances",
	"ec2:DescribeClientVpnAuthorizationRules",
	"ec2:DescribeClientVpnConnections",
	"ec2:DescribeClientVpnEndpoints",
	"ec2:DescribeClientVpnRoutes",
	"ec2:DescribeClientVpnTargetNetworks",
	"ec2:DescribeCoipPools",
	"ec2:DescribeConversionTasks",
	"ec2:DescribeCustomerGateways",
	"ec2:DescribeDhcpOptions",
	"ec2:DescribeEgressOnlyInternetGateways",
	"ec2:DescribeElasticGpus",
	"ec2:DescribeExportImageTasks",
	"ec2:DescribeExportTasks",
	"ec2:DescribeFastSnapshotRestores",
	"ec2:DescribeFleetHistory",
	"ec2:DescribeFleetInstances",
	"ec2:DescribeFleets",
	"ec2:DescribeFlowLogs",
	"ec2:DescribeFpgaImageAttribute",
	"ec2:DescribeFpgaImages",
	"ec2:DescribeHostReservationOfferings",
	"ec2:DescribeHostReservations",
	"ec2:DescribeHosts",
	"ec2:DescribeIamInstanceProfileAssociations",
	"ec2:DescribeIdFormat",
	"ec2:DescribeIdentityIdFormat",
	"ec2:DescribeImageAttribute",
	"ec2:DescribeImages",
	"ec2:DescribeImportImageTasks",
	"ec2:DescribeImportSnapshotTasks",
	"ec2:DescribeInstanceAttribute",
	"ec2:DescribeInstanceCreditSpecifications",
	"ec2:DescribeInstanceEventNotificationAttributes",
	"ec2:DescribeInstanceStatus",
	"ec2:DescribeInstanceTypeOfferings",
	"ec2:DescribeInstanceTypes",
	"ec2:DescribeInstances",
	"ec2:DescribeInternetGateways",
	"ec2:DescribeIpv6Pools",
	"ec2:DescribeKeyPairs",
	"ec2:DescribeLaunchTemplateVersions",
	"ec2:DescribeLaunchTemplates",
	"ec2:DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations",
	"ec2:DescribeLocalGatewayRouteTableVpcAssociations",
	"ec2:DescribeLocalGatewayRouteTables",
	"ec2:DescribeLocalGatewayVirtualInterfaceGroups",
	"ec2:DescribeLocalGatewayVirtualInterfaces",
	"ec2:DescribeLocalGateways",
	"ec2:DescribeManagedPrefixLists",
	"ec2:DescribeMovingAddresses",
	"ec2:DescribeNatGateways",
	"ec2:DescribeNetworkAcls",
	"ec2:DescribeNetworkInsightsAnalyses",
	"ec2:DescribeNetworkInsightsPaths",
	"ec2:DescribeNetworkInterfaceAttribute",
	"ec2:DescribeNetworkInterfacePermissions",
	"ec2:DescribeNetworkInterfaces",
	"ec2:DescribePlacementGroups",
	"ec2:DescribePrefixLists",
	"ec2:DescribePrincipalIdFormat",
	"ec2:DescribePublicIpv4Pools",
	"ec2:DescribeRegions",
	"ec2:DescribeReplaceRootVolumeTasks",
	"ec2:DescribeReservedInstances",
	"ec2:DescribeReservedInstancesListings",
	"ec2:DescribeReservedInstancesModifications",
	"ec2:DescribeReservedInstancesOfferings",
	"ec2:DescribeRouteTables",
	"ec2:DescribeScheduledInstanceAvailability",
	"ec2:DescribeScheduledInstances",
	"ec2:DescribeSecurityGroupReferences",
	"ec2:DescribeSecurityGroupRules",
	"ec2:DescribeSecurityGroups",
	"ec2:DescribeSnapshotAttribute",
	"ec2:DescribeSnapshots",
	"ec2:DescribeSpotDatafeedSubscription",
	"ec2:DescribeSpotFleetInstances",
	"ec2:DescribeSpotFleetRequestHistory",
	"ec2:DescribeSpotFleetRequests",
	"ec2:DescribeSpotInstanceRequests",
	"ec2:DescribeSpotPriceHistory",
	"ec2:DescribeStaleSecurityGroups",
	"ec2:DescribeStoreImageTasks",
	"ec2:DescribeSubnets",
	"ec2:DescribeTags",
	"ec2:DescribeTrafficMirrorFilters",
	"ec2:DescribeTrafficMirrorSessions",
	"ec2:DescribeTrafficMirrorTargets",
	"ec2:DescribeTransitGatewayAttachments",
	"ec2:DescribeTransitGatewayConnectPeers",
	"ec2:DescribeTransitGatewayConnects",
	"ec2:DescribeTransitGatewayMulticastDomains",
	"ec2:DescribeTransitGatewayPeeringAttachments",
	"ec2:DescribeTransitGatewayRouteTables",
	"ec2:DescribeTransitGatewayVpcAttachments",
	"ec2:DescribeTransitGateways",
	"ec2:DescribeVolumeAttribute",
	"ec2:DescribeVolumeStatus",
	"ec2:DescribeVolumes",
	"ec2:DescribeVolumesModifications",
	"ec2:DescribeVpcAttribute",
	"ec2:DescribeVpcClassicLink",
	"ec2:DescribeVpcClassicLinkDnsSupport",
	"ec2:DescribeVpcEndpointConnectionNotifications",
	"ec2:DescribeVpcEndpointConnections",
	"ec2:DescribeVpcEndpointServiceConfigurations",
	"ec2:DescribeVpcEndpointServicePermissions",
	"ec2:DescribeVpcEndpointServices",
	"ec2:DescribeVpcEndpoints",
	"ec2:DescribeVpcPeeringConnections",
	"ec2:DescribeVpcs",
	"ec2:DescribeVpnConnections",
	"ec2:DescribeVpnGateways",
	"ec2:DisableEbsEncryptionByDefault",
	"ec2:DisableSerialConsoleAccess",
	"ec2:DisassociateVpcCidrBlock",
	"ec2:EnableEbsEncryptionByDefault",
	"ec2:EnableSerialConsoleAccess",
	"ec2:GetCoipPoolUsage",
	"ec2:GetDefaultCreditSpecification",
	"ec2:GetEbsDefaultKmsKeyId",
	"ec2:GetEbsEncryptionByDefault",
	"ec2:GetSerialConsoleAccessStatus",
	"ec2:ImportVolume",
	"ec2:ModifyAddressAttribute",
	"ec2:ModifyAvailabilityZoneGroup",
	"ec2:ModifyDefaultCreditSpecification",
	"ec2:ModifyEbsDefaultKmsKeyId",
	"ec2:ModifyIdFormat",
	"ec2:ModifyIdentityIdFormat",
	"ec2:MoveAddressToVpc",
	"ec2:ProvisionByoipCidr",
	"ec2:PurchaseScheduledInstances",
	"ec2:RegisterImage",
	"ec2:RegisterInstanceEventNotificationAttributes",
	"ec2:ResetAddressAttribute",
	"ec2:ResetEbsDefaultKmsKeyId",
	"ec2:RestoreAddressToClassic",
	"ec2:EC2-Classic-EBS",
	"ec2:EC2-Classic-InstanceStore",
	"ec2:EC2-VPC-EBS",
	"ec2:EC2-VPC-EBS-Subnet",
	"ec2:EC2-VPC-InstanceStore",
	"ec2:EC2-VPC-InstanceStore-Subnet",
	"ec2:WithdrawByoipCidr",
	"autoscaling:DescribeAccountLimits",
	"autoscaling:DescribeAdjustmentTypes",
	"autoscaling:DescribeAutoScalingGroups",
	"autoscaling:DescribeAutoScalingInstances",
	"autoscaling:DescribeAutoScalingNotificationTypes",
	"autoscaling:DescribeInstanceRefreshes",
	"autoscaling:DescribeLaunchConfigurations",
	"autoscaling:DescribeLifecycleHookTypes",
	"autoscaling:DescribeLifecycleHooks",
	"autoscaling:DescribeLoadBalancerTargetGroups",
	"autoscaling:DescribeLoadBalancers",
	"autoscaling:DescribeMetricCollectionTypes",
	"autoscaling:DescribeNotificationConfigurations",
	"autoscaling:DescribePolicies",
	"autoscaling:DescribeScalingActivities",
	"autoscaling:DescribeScalingProcessTypes",
	"autoscaling:DescribeScheduledActions",
	"autoscaling:DescribeTags",
	"autoscaling:DescribeTerminationPolicyTypes",
	"autoscaling:DescribeWarmPool",
	"imagebuilder:ListComponents",
	"imagebuilder:ListContainerRecipes",
	"imagebuilder:ListDistributionConfigurations",
	"imagebuilder:ListImagePipelines",
	"imagebuilder:ListImageRecipes",
	"imagebuilder:ListImages",
	"imagebuilder:ListInfrastructureConfigurations",
	"elasticbeanstalk:CheckDNSAvailability",
	"elasticbeanstalk:CreateStorageLocation",
	"elasticbeanstalk:DescribeAccountAttributes",
	"elasticbeanstalk:ListPlatformBranches",
	"ecr:DeleteRegistryPolicy",
	"ecr:DescribeRegistry",
	"ecr:GetAuthorizationToken",
	"ecr:GetRegistryPolicy",
	"ecr:PutRegistryPolicy",
	"ecr:PutReplicationConfiguration",
	"ecr-public:GetAuthorizationToken",
	"ecs:CreateCapacityProvider",
	"ecs:CreateCluster",
	"ecs:CreateTaskSet",
	"ecs:DeleteAccountSetting",
	"ecs:DeregisterTaskDefinition",
	"ecs:DescribeTaskDefinition",
	"ecs:DiscoverPollEndpoint",
	"ecs:ListAccountSettings",
	"ecs:ListClusters",
	"ecs:ListServices",
	"ecs:ListTaskDefinitionFamilies",
	"ecs:ListTaskDefinitions",
	"ecs:PutAccountSetting",
	"ecs:PutAccountSettingDefault",
	"ecs:RegisterTaskDefinition",
	"elasticfilesystem:CreateFileSystem",
	"elasticfilesystem:DescribeAccountPreferences",
	"elasticfilesystem:DescribeMountTargetSecurityGroups",
	"elasticfilesystem:ModifyMountTargetSecurityGroups",
	"elasticfilesystem:PutAccountPreferences",
	"elastic-inference:DescribeAcceleratorOfferings",
	"elastic-inference:DescribeAcceleratorTypes",
	"elastic-inference:DescribeAccelerators",
	"elastic-inference:ListTagsForResource",
	"elastic-inference:TagResource",
	"elastic-inference:UntagResource",
	"eks:CreateCluster",
	"eks:DescribeAddonVersions",
	"eks:ListClusters",
	"elasticloadbalancing:DescribeInstanceHealth",
	"elasticloadbalancing:DescribeLoadBalancerAttributes",
	"elasticloadbalancing:DescribeLoadBalancerPolicies",
	"elasticloadbalancing:DescribeLoadBalancerPolicyTypes",
	"elasticloadbalancing:DescribeLoadBalancers",
	"elasticloadbalancing:DescribeAccountLimits",
	"elasticloadbalancing:DescribeListenerCertificates",
	"elasticloadbalancing:DescribeListeners",
	"elasticloadbalancing:DescribeLoadBalancerAttributes",
	"elasticloadbalancing:DescribeLoadBalancers",
	"elasticloadbalancing:DescribeRules",
	"elasticloadbalancing:DescribeSSLPolicies",
	"elasticloadbalancing:DescribeTargetGroupAttributes",
	"elasticloadbalancing:DescribeTargetGroups",
	"elasticloadbalancing:DescribeTargetHealth",
	"elasticloadbalancing:SetWebAcl [permission only]",
	"elasticmapreduce:CreateRepository",
	"elasticmapreduce:CreateSecurityConfiguration",
	"elasticmapreduce:CreateStudio",
	"elasticmapreduce:DeleteRepository",
	"elasticmapreduce:DeleteSecurityConfiguration",
	"elasticmapreduce:DescribeRepository",
	"elasticmapreduce:DescribeSecurityConfiguration",
	"elasticmapreduce:GetBlockPublicAccessConfiguration",
	"elasticmapreduce:LinkRepository",
	"elasticmapreduce:ListClusters",
	"elasticmapreduce:ListEditors",
	"elasticmapreduce:ListNotebookExecutions",
	"elasticmapreduce:ListRepositories",
	"elasticmapreduce:ListSecurityConfigurations",
	"elasticmapreduce:ListStudioSessionMappings",
	"elasticmapreduce:ListStudios",
	"elasticmapreduce:PutBlockPublicAccessConfiguration",
	"elasticmapreduce:RunJobFlow",
	"elasticmapreduce:UnlinkRepository",
	"elasticmapreduce:UpdateRepository",
	"elasticmapreduce:ViewEventsFromAllClustersInConsole [permission only]",
	"elastictranscoder:ListJobsByStatus",
	"elastictranscoder:ListPipelines",
	"elastictranscoder:ListPresets",
	"elastictranscoder:TestRole",
	"elasticache:DescribeCacheEngineVersions",
	"elasticache:DescribeEngineDefaultParameters",
	"elasticache:DescribeEvents",
	"elasticache:DescribeReservedCacheNodesOfferings",
	"elasticache:DescribeServiceUpdates",
	"es:AcceptInboundCrossClusterSearchConnection",
	"es:CreateElasticsearchServiceRole",
	"es:CreatePackage",
	"es:DeleteElasticsearchServiceRole",
	"es:DeleteInboundCrossClusterSearchConnection",
	"es:DeleteOutboundCrossClusterSearchConnection",
	"es:DeletePackage",
	"es:DescribeElasticsearchInstanceTypeLimits",
	"es:DescribeInboundCrossClusterSearchConnections",
	"es:DescribeOutboundCrossClusterSearchConnections",
	"es:DescribePackages",
	"es:DescribeReservedElasticsearchInstanceOfferings",
	"es:DescribeReservedElasticsearchInstances",
	"es:GetPackageVersionHistory",
	"es:ListDomainNames",
	"es:ListDomainsForPackage",
	"es:ListElasticsearchInstanceTypeDetails",
	"es:ListElasticsearchInstanceTypes",
	"es:ListElasticsearchVersions",
	"es:PurchaseReservedElasticsearchInstanceOffering",
	"es:RejectInboundCrossClusterSearchConnection",
	"es:UpdatePackage",
	"elemental-appliances-software:ListQuotes",
	"elemental-activations:CompleteAccountRegistration",
	"elemental-activations:CompleteFileUpload",
	"elemental-activations:DownloadSoftware",
	"elemental-activations:GenerateLicenses",
	"elemental-activations:StartAccountRegistration",
	"elemental-activations:StartFileUpload",
	"mediaconnect:AddFlowMediaStreams",
	"mediaconnect:AddFlowOutputs",
	"mediaconnect:AddFlowSources",
	"mediaconnect:AddFlowVpcInterfaces",
	"mediaconnect:CreateFlow",
	"mediaconnect:DeleteFlow",
	"mediaconnect:DescribeFlow",
	"mediaconnect:DescribeOffering",
	"mediaconnect:DescribeReservation",
	"mediaconnect:GrantFlowEntitlements",
	"mediaconnect:ListEntitlements",
	"mediaconnect:ListFlows",
	"mediaconnect:ListOfferings",
	"mediaconnect:ListReservations",
	"mediaconnect:ListTagsForResource",
	"mediaconnect:PurchaseOffering",
	"mediaconnect:RemoveFlowMediaStream",
	"mediaconnect:RemoveFlowOutput",
	"mediaconnect:RemoveFlowSource",
	"mediaconnect:RemoveFlowVpcInterface",
	"mediaconnect:RevokeFlowEntitlement",
	"mediaconnect:StartFlow",
	"mediaconnect:StopFlow",
	"mediaconnect:TagResource",
	"mediaconnect:UntagResource",
	"mediaconnect:UpdateFlow",
	"mediaconnect:UpdateFlowEntitlement",
	"mediaconnect:UpdateFlowMediaStream",
	"mediaconnect:UpdateFlowOutput",
	"mediaconnect:UpdateFlowSource",
	"mediaconvert:AssociateCertificate",
	"mediaconvert:CreatePreset",
	"mediaconvert:CreateQueue",
	"mediaconvert:DescribeEndpoints",
	"mediaconvert:DisassociateCertificate",
	"mediaconvert:ListJobTemplates",
	"mediaconvert:ListPresets",
	"mediaconvert:ListQueues",
	"medialive:ListChannels",
	"medialive:ListInputDeviceTransfers",
	"medialive:ListInputDevices",
	"medialive:ListInputSecurityGroups",
	"medialive:ListInputs",
	"medialive:ListMultiplexPrograms",
	"medialive:ListMultiplexes",
	"medialive:ListOfferings",
	"medialive:ListReservations",
	"mediapackage:CreateChannel",
	"mediapackage:CreateHarvestJob",
	"mediapackage:CreateOriginEndpoint",
	"mediapackage:ListChannels",
	"mediapackage:ListHarvestJobs",
	"mediapackage:ListOriginEndpoints",
	"mediapackage-vod:CreateAsset",
	"mediapackage-vod:CreatePackagingConfiguration",
	"mediapackage-vod:CreatePackagingGroup",
	"mediapackage-vod:ListAssets",
	"mediapackage-vod:ListPackagingConfigurations",
	"mediapackage-vod:ListPackagingGroups",
	"mediastore:CreateContainer",
	"mediastore:DeleteContainer",
	"mediastore:DeleteContainerPolicy",
	"mediastore:DeleteCorsPolicy",
	"mediastore:DeleteLifecyclePolicy",
	"mediastore:DeleteMetricPolicy",
	"mediastore:DeleteObject",
	"mediastore:DescribeContainer",
	"mediastore:DescribeObject",
	"mediastore:GetContainerPolicy",
	"mediastore:GetCorsPolicy",
	"mediastore:GetLifecyclePolicy",
	"mediastore:GetMetricPolicy",
	"mediastore:GetObject",
	"mediastore:ListContainers",
	"mediastore:ListItems",
	"mediastore:ListTagsForResource",
	"mediastore:PutContainerPolicy",
	"mediastore:PutCorsPolicy",
	"mediastore:PutLifecyclePolicy",
	"mediastore:PutMetricPolicy",
	"mediastore:PutObject",
	"mediastore:StartAccessLogging",
	"mediastore:StopAccessLogging",
	"mediastore:TagResource",
	"mediastore:UntagResource",
	"mediatailor:ListPlaybackConfigurations",
	"mediatailor:ListTagsForResource",
	"mediatailor:TagResource",
	"mediatailor:UntagResource",
	"elemental-support-cases:CheckCasePermission",
	"elemental-support-cases:CreateCase",
	"elemental-support-cases:GetCase",
	"elemental-support-cases:GetCases",
	"elemental-support-cases:UpdateCase",
	"elemental-support-content:Query",
	"emr-containers:CreateVirtualCluster",
	"emr-containers:ListVirtualClusters",
	"events:ListApiDestinations",
	"events:ListArchives",
	"events:ListConnections",
	"events:ListEventBuses",
	"events:ListEventSources",
	"events:ListPartnerEventSources",
	"events:ListReplays",
	"events:ListRuleNamesByTarget",
	"events:ListRules",
	"events:PutPartnerEvents",
	"events:PutPermission",
	"events:RemovePermission",
	"events:TestEventPattern",
	"schemas:GetDiscoveredSchema",
	"fis:ListActions",
	"fis:ListExperimentTemplates",
	"fis:ListExperiments",
	"fms:AssociateAdminAccount",
	"fms:DeleteNotificationChannel",
	"fms:DisassociateAdminAccount",
	"fms:GetAdminAccount",
	"fms:GetNotificationChannel",
	"fms:ListAppsLists",
	"fms:ListMemberAccounts",
	"fms:ListPolicies",
	"fms:ListProtocolsLists",
	"fms:PutNotificationChannel",
	"forecast:ListDatasetGroups",
	"forecast:ListDatasetImportJobs",
	"forecast:ListDatasets",
	"forecast:ListForecastExportJobs",
	"forecast:ListForecasts",
	"forecast:ListPredictorBacktestExportJobs",
	"forecast:ListPredictors",
	"frauddetector:BatchCreateVariable",
	"frauddetector:CreateVariable",
	"frauddetector:DeleteEvent",
	"frauddetector:GetEventPrediction",
	"frauddetector:GetKMSEncryptionKey",
	"frauddetector:PutKMSEncryptionKey",
	"frauddetector:UpdateModelVersionStatus",
	"freertos:DescribeHardwarePlatform",
	"freertos:GetSoftwareURL",
	"freertos:GetSoftwareURLForConfiguration",
	"freertos:ListFreeRTOSVersions",
	"freertos:ListHardwarePlatforms",
	"freertos:ListHardwareVendors",
	"freertos:ListSoftwareConfigurations",
	"fsx:DescribeBackups",
	"fsx:DescribeDataRepositoryTasks",
	"fsx:DescribeFileSystems",
	"gamelift:AcceptMatch",
	"gamelift:CreateAlias",
	"gamelift:CreateBuild",
	"gamelift:CreateFleet",
	"gamelift:CreateGameServerGroup",
	"gamelift:CreateGameSession",
	"gamelift:CreateGameSessionQueue",
	"gamelift:CreateMatchmakingConfiguration",
	"gamelift:CreateMatchmakingRuleSet",
	"gamelift:CreatePlayerSession",
	"gamelift:CreatePlayerSessions",
	"gamelift:CreateScript",
	"gamelift:CreateVpcPeeringAuthorization",
	"gamelift:CreateVpcPeeringConnection",
	"gamelift:DeleteVpcPeeringAuthorization",
	"gamelift:DeleteVpcPeeringConnection",
	"gamelift:DescribeEC2InstanceLimits",
	"gamelift:DescribeFleetAttributes",
	"gamelift:DescribeFleetCapacity",
	"gamelift:DescribeFleetUtilization",
	"gamelift:DescribeGameSessionDetails",
	"gamelift:DescribeGameSessionPlacement",
	"gamelift:DescribeGameSessionQueues",
	"gamelift:DescribeGameSessions",
	"gamelift:DescribeMatchmaking",
	"gamelift:DescribeMatchmakingConfigurations",
	"gamelift:DescribeMatchmakingRuleSets",
	"gamelift:DescribePlayerSessions",
	"gamelift:DescribeVpcPeeringAuthorizations",
	"gamelift:DescribeVpcPeeringConnections",
	"gamelift:GetGameSessionLogUrl",
	"gamelift:ListAliases",
	"gamelift:ListBuilds",
	"gamelift:ListFleets",
	"gamelift:ListGameServerGroups",
	"gamelift:ListScripts",
	"gamelift:SearchGameSessions",
	"gamelift:StartMatchBackfill",
	"gamelift:StartMatchmaking",
	"gamelift:StopGameSessionPlacement",
	"gamelift:StopMatchmaking",
	"gamelift:UpdateGameSession",
	"gamelift:ValidateMatchmakingRuleSet",
	"glacier:GetDataRetrievalPolicy",
	"glacier:ListProvisionedCapacity",
	"glacier:ListVaults",
	"glacier:PurchaseProvisionedCapacity",
	"glacier:SetDataRetrievalPolicy",
	"globalaccelerator:AdvertiseByoipCidr",
	"globalaccelerator:CreateAccelerator",
	"globalaccelerator:CreateCustomRoutingAccelerator",
	"globalaccelerator:DeprovisionByoipCidr",
	"globalaccelerator:ListAccelerators",
	"globalaccelerator:ListByoipCidrs",
	"globalaccelerator:ListCustomRoutingAccelerators",
	"globalaccelerator:ListCustomRoutingPortMappingsByDestination",
	"globalaccelerator:ProvisionByoipCidr",
	"globalaccelerator:WithdrawByoipCidr",
	"glue:BatchGetCrawlers",
	"glue:BatchGetDevEndpoints",
	"glue:BatchGetJobs",
	"glue:BatchGetTriggers",
	"glue:BatchGetWorkflows",
	"glue:BatchStopJobRun",
	"glue:CheckSchemaVersionValidity",
	"glue:CreateClassifier",
	"glue:CreateCrawler",
	"glue:CreateDevEndpoint",
	"glue:CreateJob",
	"glue:CreateMLTransform",
	"glue:CreateScript",
	"glue:CreateSecurityConfiguration",
	"glue:CreateTrigger",
	"glue:CreateWorkflow",
	"glue:DeleteClassifier",
	"glue:DeleteCrawler",
	"glue:DeleteDevEndpoint",
	"glue:DeleteJob",
	"glue:DeleteSecurityConfiguration",
	"glue:DeleteTrigger",
	"glue:DeleteWorkflow",
	"glue:GetClassifier",
	"glue:GetClassifiers",
	"glue:GetCrawler",
	"glue:GetCrawlerMetrics",
	"glue:GetCrawlers",
	"glue:GetDataCatalogEncryptionSettings",
	"glue:GetDataflowGraph",
	"glue:GetDevEndpoint",
	"glue:GetDevEndpoints",
	"glue:GetJob",
	"glue:GetJobBookmark",
	"glue:GetJobRun",
	"glue:GetJobRuns",
	"glue:GetJobs",
	"glue:GetMLTransforms",
	"glue:GetMapping",
	"glue:GetPlan",
	"glue:GetSecurityConfiguration",
	"glue:GetSecurityConfigurations",
	"glue:GetTrigger",
	"glue:GetTriggers",
	"glue:GetWorkflow",
	"glue:GetWorkflowRun",
	"glue:GetWorkflowRunProperties",
	"glue:GetWorkflowRuns",
	"glue:ListCrawlers",
	"glue:ListDevEndpoints",
	"glue:ListJobs",
	"glue:ListMLTransforms",
	"glue:ListRegistries",
	"glue:ListTriggers",
	"glue:ListWorkflows",
	"glue:PutDataCatalogEncryptionSettings",
	"glue:PutWorkflowRunProperties",
	"glue:ResetJobBookmark",
	"glue:ResumeWorkflowRun",
	"glue:StartCrawler",
	"glue:StartCrawlerSchedule",
	"glue:StartJobRun",
	"glue:StartTrigger",
	"glue:StartWorkflowRun",
	"glue:StopCrawler",
	"glue:StopCrawlerSchedule",
	"glue:StopTrigger",
	"glue:StopWorkflowRun",
	"glue:UpdateClassifier",
	"glue:UpdateCrawler",
	"glue:UpdateCrawlerSchedule",
	"glue:UpdateDevEndpoint",
	"glue:UpdateJob",
	"glue:UpdateTrigger",
	"glue:UpdateWorkflow",
	"databrew:CreateDataset",
	"databrew:CreateProfileJob",
	"databrew:CreateProject",
	"databrew:CreateRecipe",
	"databrew:CreateRecipeJob",
	"databrew:CreateSchedule",
	"databrew:ListDatasets",
	"databrew:ListJobs",
	"databrew:ListProjects",
	"databrew:ListRecipes",
	"groundstation:CreateConfig",
	"groundstation:CreateDataflowEndpointGroup",
	"groundstation:CreateMissionProfile",
	"groundstation:GetMinuteUsage",
	"groundstation:ListConfigs",
	"groundstation:ListContacts",
	"groundstation:ListDataflowEndpointGroups",
	"groundstation:ListGroundStations",
	"groundstation:ListMissionProfiles",
	"groundstation:ListSatellites",
	"groundstation:ReserveContact",
	"groundtruthlabeling:AssociatePatchToManifestJob",
	"groundtruthlabeling:DescribeConsoleJob",
	"groundtruthlabeling:ListDatasetObjects",
	"groundtruthlabeling:RunFilterOrSampleDatasetJob",
	"groundtruthlabeling:RunGenerateManifestByCrawlingJob",
	"guardduty:AcceptInvitation",
	"guardduty:ArchiveFindings",
	"guardduty:CreateDetector",
	"guardduty:CreateIPSet",
	"guardduty:CreateMembers",
	"guardduty:CreatePublishingDestination",
	"guardduty:CreateSampleFindings",
	"guardduty:CreateThreatIntelSet",
	"guardduty:DeclineInvitations",
	"guardduty:DeleteInvitations",
	"guardduty:DeleteMembers",
	"guardduty:DescribeOrganizationConfiguration",
	"guardduty:DisableOrganizationAdminAccount",
	"guardduty:DisassociateFromMasterAccount",
	"guardduty:DisassociateMembers",
	"guardduty:EnableOrganizationAdminAccount",
	"guardduty:GetFindings",
	"guardduty:GetFindingsStatistics",
	"guardduty:GetInvitationsCount",
	"guardduty:GetMasterAccount",
	"guardduty:GetMemberDetectors",
	"guardduty:GetMembers",
	"guardduty:GetUsageStatistics",
	"guardduty:InviteMembers",
	"guardduty:ListDetectors",
	"guardduty:ListFilters",
	"guardduty:ListFindings",
	"guardduty:ListIPSets",
	"guardduty:ListInvitations",
	"guardduty:ListMembers",
	"guardduty:ListOrganizationAdminAccounts",
	"guardduty:ListPublishingDestinations",
	"guardduty:ListThreatIntelSets",
	"guardduty:StartMonitoringMembers",
	"guardduty:StopMonitoringMembers",
	"guardduty:UnarchiveFindings",
	"guardduty:UpdateFindingsFeedback",
	"guardduty:UpdateMemberDetectors",
	"guardduty:UpdateOrganizationConfiguration",
	"health:DescribeAffectedAccountsForOrganization",
	"health:DescribeAffectedEntitiesForOrganization",
	"health:DescribeEntityAggregates",
	"health:DescribeEventAggregates",
	"health:DescribeEventDetailsForOrganization",
	"health:DescribeEventTypes",
	"health:DescribeEvents",
	"health:DescribeEventsForOrganization",
	"health:DescribeHealthServiceStatusForOrganization",
	"health:DisableHealthServiceAccessForOrganization",
	"health:EnableHealthServiceAccessForOrganization",
	"healthlake:CreateFHIRDatastore",
	"healthlake:ListFHIRDatastores",
	"honeycode:ApproveTeamAssociation",
	"honeycode:CreateTeam",
	"honeycode:CreateTenant",
	"honeycode:DeregisterGroups",
	"honeycode:DescribeTeam",
	"honeycode:ListDomains",
	"honeycode:ListGroups",
	"honeycode:ListTeamAssociations",
	"honeycode:ListTenants",
	"honeycode:RegisterDomainForVerification",
	"honeycode:RegisterGroups",
	"honeycode:RejectTeamAssociation",
	"honeycode:RestartDomainVerification",
	"honeycode:UpdateTeam",
	"access-analyzer:CancelPolicyGeneration",
	"access-analyzer:GetGeneratedPolicy",
	"access-analyzer:ListAnalyzers",
	"access-analyzer:ListPolicyGenerations",
	"access-analyzer:StartPolicyGeneration",
	"access-analyzer:ValidatePolicy",
	"iam:CreateAccountAlias",
	"iam:DeleteAccountAlias",
	"iam:DeleteAccountPasswordPolicy",
	"iam:GenerateCredentialReport",
	"iam:GetAccountAuthorizationDetails",
	"iam:GetAccountPasswordPolicy",
	"iam:GetAccountSummary",
	"iam:GetContextKeysForCustomPolicy",
	"iam:GetCredentialReport",
	"iam:GetOrganizationsAccessReport",
	"iam:GetServiceLastAccessedDetails",
	"iam:GetServiceLastAccessedDetailsWithEntities",
	"iam:ListAccountAliases",
	"iam:ListGroups",
	"iam:ListOpenIDConnectProviders",
	"iam:ListPolicies",
	"iam:ListRoles",
	"iam:ListSAMLProviders",
	"iam:ListServerCertificates",
	"iam:ListUsers",
	"iam:ListVirtualMFADevices",
	"iam:SetSecurityTokenServicePreferences",
	"iam:SimulateCustomPolicy",
	"iam:UpdateAccountPasswordPolicy",
	"identitystore:DescribeGroup",
	"identitystore:DescribeUser",
	"identitystore:ListGroups",
	"identitystore:ListUsers",
	"importexport:CancelJob",
	"importexport:CreateJob",
	"importexport:GetShippingLabel",
	"importexport:GetStatus",
	"importexport:ListJobs",
	"importexport:UpdateJob",
	"inspector:AddAttributesToFindings",
	"inspector:CreateAssessmentTarget",
	"inspector:CreateAssessmentTemplate",
	"inspector:CreateExclusionsPreview",
	"inspector:CreateResourceGroup",
	"inspector:DeleteAssessmentRun",
	"inspector:DeleteAssessmentTarget",
	"inspector:DeleteAssessmentTemplate",
	"inspector:DescribeAssessmentRuns",
	"inspector:DescribeAssessmentTargets",
	"inspector:DescribeAssessmentTemplates",
	"inspector:DescribeCrossAccountAccessRole",
	"inspector:DescribeExclusions",
	"inspector:DescribeFindings",
	"inspector:DescribeResourceGroups",
	"inspector:DescribeRulesPackages",
	"inspector:GetAssessmentReport",
	"inspector:GetExclusionsPreview",
	"inspector:GetTelemetryMetadata",
	"inspector:ListAssessmentRunAgents",
	"inspector:ListAssessmentRuns",
	"inspector:ListAssessmentTargets",
	"inspector:ListAssessmentTemplates",
	"inspector:ListEventSubscriptions",
	"inspector:ListExclusions",
	"inspector:ListFindings",
	"inspector:ListRulesPackages",
	"inspector:ListTagsForResource",
	"inspector:PreviewAgents",
	"inspector:RegisterCrossAccountAccessRole",
	"inspector:RemoveAttributesFromFindings",
	"inspector:SetTagsForResource",
	"inspector:StartAssessmentRun",
	"inspector:StopAssessmentRun",
	"inspector:SubscribeToEvent",
	"inspector:UnsubscribeFromEvent",
	"inspector:UpdateAssessmentTarget",
	"iot:AttachThingPrincipal",
	"iot:CancelAuditMitigationActionsTask",
	"iot:CancelAuditTask",
	"iot:CancelDetectMitigationActionsTask",
	"iot:ClearDefaultAuthorizer",
	"iot:CreateAuditSuppression",
	"iot:CreateCertificateFromCsr",
	"iot:CreateKeysAndCertificate",
	"iot:DeleteAccountAuditConfiguration",
	"iot:DeleteAuditSuppression",
	"iot:DeleteRegistrationCode",
	"iot:DeleteV2LoggingLevel",
	"iot:DescribeAccountAuditConfiguration",
	"iot:DescribeAuditFinding",
	"iot:DescribeAuditMitigationActionsTask",
	"iot:DescribeAuditSuppression",
	"iot:DescribeAuditTask",
	"iot:DescribeDefaultAuthorizer",
	"iot:DescribeDetectMitigationActionsTask",
	"iot:DescribeEndpoint",
	"iot:DescribeEventConfigurations",
	"iot:DescribeThingRegistrationTask",
	"iot:DetachThingPrincipal",
	"iot:GetIndexingConfiguration",
	"iot:GetLoggingOptions",
	"iot:GetRegistrationCode",
	"iot:GetV2LoggingOptions",
	"iot:ListAttachedPolicies",
	"iot:ListAuditFindings",
	"iot:ListAuditMitigationActionsExecutions",
	"iot:ListAuditMitigationActionsTasks",
	"iot:ListAuditSuppressions",
	"iot:ListAuditTasks",
	"iot:ListAuthorizers",
	"iot:ListBillingGroups",
	"iot:ListCACertificates",
	"iot:ListCertificates",
	"iot:ListCertificatesByCA",
	"iot:ListCustomMetrics",
	"iot:ListDetectMitigationActionsTasks",
	"iot:ListDimensions",
	"iot:ListDomainConfigurations",
	"iot:ListFleetMetrics",
	"iot:ListIndices",
	"iot:ListJobTemplates",
	"iot:ListJobs",
	"iot:ListMitigationActions",
	"iot:ListOTAUpdates",
	"iot:ListOutgoingCertificates",
	"iot:ListPolicies",
	"iot:ListPolicyPrincipals",
	"iot:ListPrincipalPolicies",
	"iot:ListPrincipalThings",
	"iot:ListProvisioningTemplates",
	"iot:ListRoleAliases",
	"iot:ListScheduledAudits",
	"iot:ListStreams",
	"iot:ListThingGroups",
	"iot:ListThingPrincipals",
	"iot:ListThingRegistrationTaskReports",
	"iot:ListThingRegistrationTasks",
	"iot:ListThingTypes",
	"iot:ListThings",
	"iot:ListTopicRuleDestinations",
	"iot:ListTopicRules",
	"iot:ListTunnels",
	"iot:ListV2LoggingLevels",
	"iot:OpenTunnel",
	"iot:RegisterCACertificate",
	"iot:RegisterCertificate",
	"iot:RegisterCertificateWithoutCA",
	"iot:RegisterThing",
	"iot:SetLoggingOptions",
	"iot:SetV2LoggingLevel",
	"iot:SetV2LoggingOptions",
	"iot:StartAuditMitigationActionsTask",
	"iot:StartOnDemandAuditTask",
	"iot:StartThingRegistrationTask",
	"iot:StopThingRegistrationTask",
	"iot:UpdateAccountAuditConfiguration",
	"iot:UpdateAuditSuppression",
	"iot:UpdateEventConfigurations",
	"iot:UpdateIndexingConfiguration",
	"iot:ValidateSecurityProfileBehaviors",
	"iot1click:ClaimDevicesByClaimCode",
	"iot1click:ListDevices",
	"iot1click:ListProjects",
	"iotanalytics:DescribeLoggingOptions",
	"iotanalytics:ListChannels",
	"iotanalytics:ListDatasets",
	"iotanalytics:ListDatastores",
	"iotanalytics:ListPipelines",
	"iotanalytics:PutLoggingOptions",
	"iotanalytics:RunPipelineActivity",
	"iotdeviceadvisor:CreateSuiteDefinition",
	"iotdeviceadvisor:ListSuiteDefinitions",
	"iotdeviceadvisor:StartSuiteRun",
	"iotwireless:GetLogLevelsByResourceTypes",
	"iotwireless:GetServiceEndpoint",
	"iotwireless:ListDestinations",
	"iotwireless:ListDeviceProfiles",
	"iotwireless:ListPartnerAccounts",
	"iotwireless:ListServiceProfiles",
	"iotwireless:ListWirelessDevices",
	"iotwireless:ListWirelessGatewayTaskDefinitions",
	"iotwireless:ListWirelessGateways",
	"iotwireless:ResetAllResourceLogLevels",
	"iotwireless:UpdateLogLevelsByResourceTypes",
	"iot-device-tester:CheckVersion",
	"iot-device-tester:DownloadTestSuite",
	"iot-device-tester:LatestIdt",
	"iot-device-tester:SendMetrics",
	"iot-device-tester:SupportedVersion",
	"iotevents:DescribeLoggingOptions",
	"iotevents:ListAlarmModels",
	"iotevents:ListDetectorModels",
	"iotevents:ListInputRoutings",
	"iotevents:ListInputs",
	"iotevents:PutLoggingOptions",
	"iotfleethub:CreateApplication",
	"iotfleethub:ListApplications",
	"greengrass:AssociateServiceRoleToAccount",
	"greengrass:CreateConnectorDefinition",
	"greengrass:CreateCoreDefinition",
	"greengrass:CreateDeviceDefinition",
	"greengrass:CreateFunctionDefinition",
	"greengrass:CreateGroup",
	"greengrass:CreateLoggerDefinition",
	"greengrass:CreateResourceDefinition",
	"greengrass:CreateSoftwareUpdateJob",
	"greengrass:CreateSubscriptionDefinition",
	"greengrass:DisassociateServiceRoleFromAccount",
	"greengrass:GetServiceRoleForAccount",
	"greengrass:ListBulkDeployments",
	"greengrass:ListConnectorDefinitions",
	"greengrass:ListCoreDefinitions",
	"greengrass:ListDeviceDefinitions",
	"greengrass:ListFunctionDefinitions",
	"greengrass:ListGroups",
	"greengrass:ListLoggerDefinitions",
	"greengrass:ListResourceDefinitions",
	"greengrass:ListSubscriptionDefinitions",
	"greengrass:StartBulkDeployment",
	"greengrass:CreateDeployment",
	"greengrass:ListComponents",
	"greengrass:ListCoreDevices",
	"greengrass:ListDeployments",
	"iotsitewise:CreateAssetModel",
	"iotsitewise:CreateGateway",
	"iotsitewise:CreatePortal",
	"iotsitewise:DescribeDefaultEncryptionConfiguration",
	"iotsitewise:DescribeLoggingOptions",
	"iotsitewise:ListAssetModels",
	"iotsitewise:ListGateways",
	"iotsitewise:ListPortals",
	"iotsitewise:PutDefaultEncryptionConfiguration",
	"iotsitewise:PutLoggingOptions",
	"iotthingsgraph:AssociateEntityToThing",
	"iotthingsgraph:CreateFlowTemplate",
	"iotthingsgraph:CreateSystemInstance",
	"iotthingsgraph:CreateSystemTemplate",
	"iotthingsgraph:DeleteNamespace",
	"iotthingsgraph:DescribeNamespace",
	"iotthingsgraph:DissociateEntityFromThing",
	"iotthingsgraph:GetEntities",
	"iotthingsgraph:GetNamespaceDeletionStatus",
	"iotthingsgraph:GetUploadStatus",
	"iotthingsgraph:ListFlowExecutionMessages",
	"iotthingsgraph:SearchEntities",
	"iotthingsgraph:SearchFlowTemplates",
	"iotthingsgraph:SearchSystemInstances",
	"iotthingsgraph:SearchSystemTemplates",
	"iotthingsgraph:SearchThings",
	"iotthingsgraph:UploadEntityDefinitions",
	"iq:CreateProject",
	"iq-permission:ApproveAccessGrant",
	"kendra:CreateIndex",
	"kendra:ListIndices",
	"kms:ConnectCustomKeyStore",
	"kms:CreateCustomKeyStore",
	"kms:CreateKey",
	"kms:DeleteCustomKeyStore",
	"kms:DescribeCustomKeyStores",
	"kms:DisconnectCustomKeyStore",
	"kms:GenerateRandom",
	"kms:ListAliases",
	"kms:ListKeys",
	"kms:UpdateCustomKeyStore",
	"kinesis:DescribeLimits",
	"kinesis:DisableEnhancedMonitoring",
	"kinesis:EnableEnhancedMonitoring",
	"kinesis:ListShards",
	"kinesis:ListStreamConsumers",
	"kinesis:ListStreams",
	"kinesis:UpdateShardCount",
	"kinesisanalytics:CreateApplication",
	"kinesisanalytics:DiscoverInputSchema",
	"kinesisanalytics:ListApplications",
	"kinesisanalytics:CreateApplication",
	"kinesisanalytics:DiscoverInputSchema",
	"kinesisanalytics:ListApplications",
	"firehose:ListDeliveryStreams",
	"kinesisvideo:ListSignalingChannels",
	"kinesisvideo:ListStreams",
	"lakeformation:BatchGrantPermissions",
	"lakeformation:BatchRevokePermissions",
	"lakeformation:DeregisterResource",
	"lakeformation:DescribeResource",
	"lakeformation:GetDataAccess",
	"lakeformation:GetDataLakeSettings",
	"lakeformation:GetEffectivePermissionsForPath",
	"lakeformation:GrantPermissions",
	"lakeformation:ListPermissions",
	"lakeformation:ListResources",
	"lakeformation:PutDataLakeSettings",
	"lakeformation:RegisterResource",
	"lakeformation:RevokePermissions",
	"lakeformation:UpdateResource",
	"lambda:CreateEventSourceMapping",
	"lambda:GetAccountSettings",
	"lambda:ListCodeSigningConfigs",
	"lambda:ListEventSourceMappings",
	"lambda:ListFunctions",
	"lambda:ListLayerVersions",
	"lambda:ListLayers",
	"launchwizard:DeleteApp",
	"launchwizard:DescribeProvisionedApp",
	"launchwizard:DescribeProvisioningEvents",
	"launchwizard:GetInfrastructureSuggestion",
	"launchwizard:GetIpAddress",
	"launchwizard:GetResourceCostEstimate",
	"launchwizard:ListProvisionedApps",
	"launchwizard:StartProvisioning",
	"lex:GetBotAliases",
	"lex:GetBots",
	"lex:GetBuiltinIntent",
	"lex:GetBuiltinIntents",
	"lex:GetBuiltinSlotTypes",
	"lex:GetImport",
	"lex:GetIntents",
	"lex:GetSlotTypes",
	"lex:StartImport",
	"lex:CreateUploadUrl",
	"lex:ListBots",
	"lex:ListBuiltInIntents",
	"lex:ListBuiltInSlotTypes",
	"lex:ListExports",
	"lex:ListImports",
	"license-manager:CheckInLicense",
	"license-manager:CheckoutLicense",
	"license-manager:CreateLicense",
	"license-manager:CreateLicenseConfiguration",
	"license-manager:CreateLicenseManagerReportGenerator",
	"license-manager:DeleteToken",
	"license-manager:ExtendLicenseConsumption",
	"license-manager:GetAccessToken",
	"license-manager:GetServiceSettings",
	"license-manager:ListDistributedGrants",
	"license-manager:ListLicenseConfigurations",
	"license-manager:ListLicenseSpecificationsForResource",
	"license-manager:ListLicenses",
	"license-manager:ListReceivedGrants",
	"license-manager:ListReceivedLicenses",
	"license-manager:ListResourceInventory",
	"license-manager:ListTokens",
	"license-manager:UpdateServiceSettings",
	"lightsail:CreateContactMethod",
	"lightsail:CreateContainerServiceRegistryLogin",
	"lightsail:ExportSnapshot",
	"lightsail:GetActiveNames",
	"lightsail:GetAlarms",
	"lightsail:GetAutoSnapshots",
	"lightsail:GetBlueprints",
	"lightsail:GetBundles",
	"lightsail:GetCertificates",
	"lightsail:GetContactMethods",
	"lightsail:GetContainerAPIMetadata",
	"lightsail:GetContainerImages",
	"lightsail:GetContainerLog",
	"lightsail:GetContainerServiceDeployments",
	"lightsail:GetContainerServiceMetricData",
	"lightsail:GetContainerServicePowers",
	"lightsail:GetContainerServices",
	"lightsail:GetDisks",
	"lightsail:GetDistributionBundles",
	"lightsail:GetDistributionLatestCacheReset",
	"lightsail:GetDistributionMetricData",
	"lightsail:GetDistributions",
	"lightsail:GetOperation",
	"lightsail:GetOperations",
	"lightsail:GetRegions",
	"lightsail:GetRelationalDatabaseBlueprints",
	"lightsail:GetRelationalDatabaseBundles",
	"lightsail:GetRelationalDatabaseEvents",
	"lightsail:GetRelationalDatabaseLogEvents",
	"lightsail:GetRelationalDatabaseLogStreams",
	"lightsail:GetRelationalDatabaseMasterUserPassword",
	"lightsail:GetRelationalDatabaseMetricData",
	"lightsail:GetRelationalDatabaseParameters",
	"lightsail:IsVpcPeered",
	"lightsail:PeerVpc",
	"lightsail:UnpeerVpc",
	"lightsail:UpdateRelationalDatabaseParameters",
	"geo:ListGeofenceCollections",
	"geo:ListMaps",
	"geo:ListPlaceIndexes",
	"geo:ListRouteCalculators",
	"geo:ListTrackers",
	"lookoutequipment:DescribeDataIngestionJob",
	"lookoutequipment:ListDatasets",
	"lookoutequipment:ListInferenceSchedulers",
	"lookoutequipment:ListModels",
	"lookoutmetrics:GetSampleData",
	"lookoutmetrics:ListAnomalyDetectors",
	"lookoutvision:CreateDataset",
	"lookoutvision:DeleteDataset",
	"lookoutvision:DescribeDataset",
	"lookoutvision:DescribeTrialDetection",
	"lookoutvision:ListDatasetEntries",
	"lookoutvision:ListModels",
	"lookoutvision:ListProjects",
	"lookoutvision:ListTrialDetections",
	"lookoutvision:StartTrialDetection",
	"lookoutvision:UpdateDatasetEntries",
	"machinelearning:DescribeBatchPredictions",
	"machinelearning:DescribeDataSources",
	"machinelearning:DescribeEvaluations",
	"machinelearning:DescribeMLModels",
	"macie2:AcceptInvitation",
	"macie2:CreateInvitations",
	"macie2:CreateSampleFindings",
	"macie2:DeclineInvitations",
	"macie2:DeleteInvitations",
	"macie2:DescribeBuckets",
	"macie2:DescribeOrganizationConfiguration",
	"macie2:DisableMacie",
	"macie2:DisableOrganizationAdminAccount",
	"macie2:DisassociateFromAdministratorAccount",
	"macie2:DisassociateFromMasterAccount",
	"macie2:EnableMacie",
	"macie2:EnableOrganizationAdminAccount",
	"macie2:GetAdministratorAccount",
	"macie2:GetBucketStatistics",
	"macie2:GetClassificationExportConfiguration",
	"macie2:GetFindingStatistics",
	"macie2:GetFindings",
	"macie2:GetFindingsPublicationConfiguration",
	"macie2:GetInvitationsCount",
	"macie2:GetMacieSession",
	"macie2:GetMasterAccount",
	"macie2:GetUsageStatistics",
	"macie2:GetUsageTotals",
	"macie2:ListClassificationJobs",
	"macie2:ListCustomDataIdentifiers",
	"macie2:ListFindings",
	"macie2:ListFindingsFilters",
	"macie2:ListInvitations",
	"macie2:ListMembers",
	"macie2:ListOrganizationAdminAccounts",
	"macie2:ListTagsForResource",
	"macie2:PutClassificationExportConfiguration",
	"macie2:PutFindingsPublicationConfiguration",
	"macie2:SearchResources",
	"macie2:TagResource",
	"macie2:TestCustomDataIdentifier",
	"macie2:UntagResource",
	"macie2:UpdateMacieSession",
	"macie2:UpdateMemberSession",
	"macie2:UpdateOrganizationConfiguration",
	"macie:AssociateMemberAccount",
	"macie:AssociateS3Resources",
	"macie:DisassociateMemberAccount",
	"macie:DisassociateS3Resources",
	"macie:ListMemberAccounts",
	"macie:ListS3Resources",
	"macie:UpdateS3Resources",
	"managedblockchain:CreateNetwork",
	"managedblockchain:ListInvitations",
	"managedblockchain:ListNetworks",
	"grafana:CreateWorkspace",
	"grafana:ListWorkspaces",
	"aps:CreateWorkspace",
	"aps:ListWorkspaces",
	"kafka:BatchAssociateScramSecret",
	"kafka:BatchDisassociateScramSecret",
	"kafka:CreateCluster",
	"kafka:CreateConfiguration",
	"kafka:DeleteCluster",
	"kafka:DeleteConfiguration",
	"kafka:DescribeCluster",
	"kafka:DescribeClusterOperation",
	"kafka:DescribeConfiguration",
	"kafka:DescribeConfigurationRevision",
	"kafka:GetBootstrapBrokers",
	"kafka:GetCompatibleKafkaVersions",
	"kafka:ListClusterOperations",
	"kafka:ListClusters",
	"kafka:ListConfigurationRevisions",
	"kafka:ListConfigurations",
	"kafka:ListKafkaVersions",
	"kafka:ListNodes",
	"kafka:ListScramSecrets",
	"kafka:RebootBroker",
	"kafka:UpdateBrokerCount",
	"kafka:UpdateBrokerStorage",
	"kafka:UpdateBrokerType",
	"kafka:UpdateClusterConfiguration",
	"kafka:UpdateClusterKafkaVersion",
	"kafka:UpdateConfiguration",
	"kafka:UpdateMonitoring",
	"airflow:ListEnvironments",
	"aws-marketplace:AcceptAgreementApprovalRequest",
	"aws-marketplace:CancelAgreementRequest",
	"aws-marketplace:DescribeAgreement",
	"aws-marketplace:GetAgreementApprovalRequest",
	"aws-marketplace:GetAgreementRequest",
	"aws-marketplace:GetAgreementTerms",
	"aws-marketplace:ListAgreementApprovalRequests",
	"aws-marketplace:ListAgreementRequests",
	"aws-marketplace:RejectAgreementApprovalRequest",
	"aws-marketplace:SearchAgreements",
	"aws-marketplace:Subscribe",
	"aws-marketplace:Unsubscribe",
	"aws-marketplace:UpdateAgreementApprovalRequest",
	"aws-marketplace:ViewSubscriptions",
	"aws-marketplace:CancelChangeSet",
	"aws-marketplace:CompleteTask",
	"aws-marketplace:DescribeChangeSet",
	"aws-marketplace:DescribeEntity",
	"aws-marketplace:DescribeTask",
	"aws-marketplace:ListChangeSets",
	"aws-marketplace:ListEntities",
	"aws-marketplace:ListTasks",
	"aws-marketplace:StartChangeSet",
	"aws-marketplace:UpdateTask",
	"marketplacecommerceanalytics:GenerateDataSet",
	"marketplacecommerceanalytics:StartSupportDataExport",
	"aws-marketplace:GetEntitlements",
	"aws-marketplace:DescribeBuilds",
	"aws-marketplace:ListBuilds",
	"aws-marketplace:StartBuild",
	"aws-marketplace-management:uploadFiles",
	"aws-marketplace-management:viewMarketing",
	"aws-marketplace-management:viewReports",
	"aws-marketplace-management:viewSettings",
	"aws-marketplace-management:viewSupport",
	"aws-marketplace:BatchMeterUsage",
	"aws-marketplace:MeterUsage",
	"aws-marketplace:RegisterUsage",
	"aws-marketplace:ResolveCustomer",
	"aws-marketplace:DescribeProcurementSystemConfiguration",
	"aws-marketplace:PutProcurementSystemConfiguration",
	"mechanicalturk:AcceptQualificationRequest",
	"mechanicalturk:ApproveAssignment",
	"mechanicalturk:AssociateQualificationWithWorker",
	"mechanicalturk:CreateAdditionalAssignmentsForHIT",
	"mechanicalturk:CreateHIT",
	"mechanicalturk:CreateHITType",
	"mechanicalturk:CreateHITWithHITType",
	"mechanicalturk:CreateQualificationType",
	"mechanicalturk:CreateWorkerBlock",
	"mechanicalturk:DeleteHIT",
	"mechanicalturk:DeleteQualificationType",
	"mechanicalturk:DeleteWorkerBlock",
	"mechanicalturk:DisassociateQualificationFromWorker",
	"mechanicalturk:GetAccountBalance",
	"mechanicalturk:GetAssignment",
	"mechanicalturk:GetFileUploadURL",
	"mechanicalturk:GetHIT",
	"mechanicalturk:GetQualificationScore",
	"mechanicalturk:GetQualificationType",
	"mechanicalturk:ListAssignmentsForHIT",
	"mechanicalturk:ListBonusPayments",
	"mechanicalturk:ListHITs",
	"mechanicalturk:ListHITsForQualificationType",
	"mechanicalturk:ListQualificationRequests",
	"mechanicalturk:ListQualificationTypes",
	"mechanicalturk:ListReviewPolicyResultsForHIT",
	"mechanicalturk:ListReviewableHITs",
	"mechanicalturk:ListWorkerBlocks",
	"mechanicalturk:ListWorkersWithQualificationType",
	"mechanicalturk:NotifyWorkers",
	"mechanicalturk:RejectAssignment",
	"mechanicalturk:RejectQualificationRequest",
	"mechanicalturk:SendBonus",
	"mechanicalturk:SendTestEventNotification",
	"mechanicalturk:UpdateExpirationForHIT",
	"mechanicalturk:UpdateHITReviewStatus",
	"mechanicalturk:UpdateHITTypeOfHIT",
	"mechanicalturk:UpdateNotificationSettings",
	"mechanicalturk:UpdateQualificationType",
	"ec2messages:AcknowledgeMessage",
	"ec2messages:DeleteMessage",
	"ec2messages:FailMessage",
	"ec2messages:GetEndpoint",
	"ec2messages:GetMessages",
	"ec2messages:SendReply",
	"mgh:CreateHomeRegionControl",
	"mgh:DescribeApplicationState",
	"mgh:DescribeHomeRegionControls",
	"mgh:GetHomeRegion",
	"mgh:ListApplicationStates",
	"mgh:ListMigrationTasks",
	"mgh:ListProgressUpdateStreams",
	"mgh:NotifyApplicationState",
	"mobileanalytics:GetFinancialReports",
	"mobileanalytics:GetReports",
	"mobileanalytics:PutEvents",
	"mobilehub:CreateProject",
	"mobilehub:CreateServiceRole",
	"mobilehub:DeleteProjectSnapshot",
	"mobilehub:DeployToStage",
	"mobilehub:DescribeBundle",
	"mobilehub:ExportBundle",
	"mobilehub:GetProjectSnapshot",
	"mobilehub:ImportProject",
	"mobilehub:InstallBundle",
	"mobilehub:ListAvailableConnectors",
	"mobilehub:ListAvailableFeatures",
	"mobilehub:ListAvailableRegions",
	"mobilehub:ListBundles",
	"mobilehub:ListProjectSnapshots",
	"mobilehub:ListProjects",
	"mobilehub:ValidateProject",
	"mobilehub:VerifyServiceRole",
	"monitron:CreateProject",
	"monitron:ListProjects",
	"mq:CreateBroker",
	"mq:CreateConfiguration",
	"mq:DescribeBrokerEngineTypes",
	"mq:DescribeBrokerInstanceOptions",
	"mq:ListBrokers",
	"mq:ListConfigurations",
	"network-firewall:ListRuleGroups",
	"networkmanager:CreateGlobalNetwork",
	"nimble:GetFeatureMap",
	"nimble:ListStudios",
	"opsworks:CreateStack",
	"opsworks:CreateUserProfile",
	"opsworks:DeleteUserProfile",
	"opsworks:DescribeMyUserProfile",
	"opsworks:DescribeOperatingSystems",
	"opsworks:DescribeUserProfiles",
	"opsworks:UpdateMyUserProfile",
	"opsworks:UpdateUserProfile",
	"opsworks-cm:AssociateNode",
	"opsworks-cm:CreateBackup",
	"opsworks-cm:CreateServer",
	"opsworks-cm:DeleteBackup",
	"opsworks-cm:DeleteServer",
	"opsworks-cm:DescribeAccountAttributes",
	"opsworks-cm:DescribeBackups",
	"opsworks-cm:DescribeEvents",
	"opsworks-cm:DescribeNodeAssociationStatus",
	"opsworks-cm:DescribeServers",
	"opsworks-cm:DisassociateNode",
	"opsworks-cm:ExportServerEngineAttribute",
	"opsworks-cm:ListTagsForResource",
	"opsworks-cm:RestoreServer",
	"opsworks-cm:StartMaintenance",
	"opsworks-cm:TagResource",
	"opsworks-cm:UntagResource",
	"opsworks-cm:UpdateServer",
	"opsworks-cm:UpdateServerEngineAttributes",
	"organizations:CreateAccount",
	"organizations:CreateGovCloudAccount",
	"organizations:CreateOrganization",
	"organizations:CreatePolicy",
	"organizations:DeleteOrganization",
	"organizations:DescribeCreateAccountStatus",
	"organizations:DescribeOrganization",
	"organizations:DisableAWSServiceAccess",
	"organizations:EnableAWSServiceAccess",
	"organizations:EnableAllFeatures",
	"organizations:LeaveOrganization",
	"organizations:ListAWSServiceAccessForOrganization",
	"organizations:ListAccounts",
	"organizations:ListCreateAccountStatus",
	"organizations:ListDelegatedAdministrators",
	"organizations:ListHandshakesForAccount",
	"organizations:ListHandshakesForOrganization",
	"organizations:ListPolicies",
	"organizations:ListRoots",
	"outposts:CreateOutpost",
	"outposts:DeleteOutpost",
	"outposts:DeleteSite",
	"outposts:GetOutpost",
	"outposts:GetOutpostInstanceTypes",
	"outposts:ListOutposts",
	"outposts:ListSites",
	"outposts:ListTagsForResource",
	"outposts:TagResource",
	"outposts:UntagResource",
	"panorama:CreateApp",
	"panorama:CreateAppDeployment",
	"panorama:CreateDeploymentConfiguration",
	"panorama:CreateDevice",
	"panorama:CreateDeviceUpdate",
	"panorama:CreateModel",
	"panorama:DescribeAppDeployment",
	"panorama:DescribeDeviceUpdate",
	"panorama:DescribeSoftware",
	"panorama:GetDeploymentConfiguration",
	"panorama:GetWebSocketURL",
	"panorama:ListAppDeploymentOperations",
	"panorama:ListApps",
	"panorama:ListDeploymentConfigurations",
	"panorama:ListDeviceUpdates",
	"panorama:ListDevices",
	"panorama:ListModels",
	"panorama:UpdateDevice",
	"personalize:ListBatchInferenceJobs",
	"personalize:ListCampaigns",
	"personalize:ListDatasetGroups",
	"personalize:ListDatasetImportJobs",
	"personalize:ListDatasets",
	"personalize:ListEventTrackers",
	"personalize:ListFilters",
	"personalize:ListRecipes",
	"personalize:ListSchemas",
	"personalize:ListSolutionVersions",
	"personalize:ListSolutions",
	"mobiletargeting:CreateApp",
	"mobiletargeting:CreateEmailTemplate",
	"mobiletargeting:CreatePushTemplate",
	"mobiletargeting:CreateRecommenderConfiguration",
	"mobiletargeting:CreateSmsTemplate",
	"mobiletargeting:CreateVoiceTemplate",
	"mobiletargeting:GetRecommenderConfigurations",
	"ses:CreateConfigurationSet",
	"ses:CreateDedicatedIpPool",
	"ses:CreateEmailIdentity",
	"ses:GetAccount",
	"ses:GetBlacklistReports",
	"ses:GetDedicatedIp",
	"ses:GetDeliverabilityDashboardOptions",
	"ses:GetDomainDeliverabilityCampaign",
	"ses:ListConfigurationSets",
	"ses:ListDedicatedIpPools",
	"ses:ListDeliverabilityTestReports",
	"ses:ListDomainDeliverabilityCampaigns",
	"ses:ListEmailIdentities",
	"ses:PutAccountDedicatedIpWarmupAttributes",
	"ses:PutAccountSendingAttributes",
	"ses:PutDedicatedIpWarmupAttributes",
	"ses:PutDeliverabilityDashboardOption",
	"sms-voice:CreateConfigurationSet",
	"sms-voice:CreateConfigurationSetEventDestination",
	"sms-voice:DeleteConfigurationSet",
	"sms-voice:DeleteConfigurationSetEventDestination",
	"sms-voice:GetConfigurationSetEventDestinations",
	"sms-voice:ListConfigurationSets",
	"sms-voice:SendVoiceMessage",
	"sms-voice:UpdateConfigurationSetEventDestination",
	"polly:DescribeVoices",
	"polly:GetSpeechSynthesisTask",
	"polly:ListLexicons",
	"polly:ListSpeechSynthesisTasks",
	"pricing:DescribeServices",
	"pricing:GetAttributeValues",
	"pricing:GetProducts",
	"aws-marketplace:AssociateProductsWithPrivateMarketplace",
	"aws-marketplace:CreatePrivateMarketplace",
	"aws-marketplace:CreatePrivateMarketplaceRequests",
	"aws-marketplace:DescribePrivateMarketplaceRequests",
	"aws-marketplace:DescribePrivateMarketplaceStatus",
	"aws-marketplace:DisassociateProductsFromPrivateMarketplace",
	"aws-marketplace:ListPrivateMarketplaceRequests",
	"proton:CreateEnvironmentAccountConnection",
	"proton:DeleteAccountRoles",
	"proton:GetAccountRoles",
	"proton:GetAccountSettings",
	"proton:ListEnvironmentTemplates",
	"proton:ListEnvironments",
	"proton:ListServiceInstances",
	"proton:ListServiceTemplates",
	"proton:ListServices",
	"proton:UpdateAccountRoles",
	"proton:UpdateAccountSettings",
	"purchase-orders:ModifyPurchaseOrders",
	"purchase-orders:ViewPurchaseOrders",
	"qldb:ListJournalS3Exports",
	"qldb:ListLedgers",
	"quicksight:CreateAccountCustomization",
	"quicksight:CreateCustomPermissions",
	"quicksight:CreateDataSource",
	"quicksight:CreateVPCConnection",
	"quicksight:DeleteCustomPermissions",
	"quicksight:DeleteVPCConnection",
	"quicksight:DescribeAccountSettings",
	"quicksight:DescribeCustomPermissions",
	"quicksight:GetAnonymousUserEmbedUrl",
	"quicksight:GetGroupMapping",
	"quicksight:GetSessionEmbedUrl",
	"quicksight:ListCustomPermissions",
	"quicksight:ListDataSets",
	"quicksight:ListDataSources",
	"quicksight:ListIngestions",
	"quicksight:ListNamespaces",
	"quicksight:SearchDirectoryGroups",
	"quicksight:SetGroupMapping",
	"quicksight:Subscribe",
	"quicksight:Unsubscribe",
	"quicksight:UpdateAccountSettings",
	"quicksight:UpdateCustomPermissions",
	"rds:CancelExportTask",
	"rds:CreateCustomAvailabilityZone",
	"rds:CreateDBProxy",
	"rds:CrossRegionCommunication",
	"rds:DeleteCustomAvailabilityZone",
	"rds:DeleteDBInstanceAutomatedBackup",
	"rds:DeleteInstallationMedia",
	"rds:DescribeAccountAttributes",
	"rds:DescribeCertificates",
	"rds:DescribeCustomAvailabilityZones",
	"rds:DescribeDBEngineVersions",
	"rds:DescribeEngineDefaultClusterParameters",
	"rds:DescribeEngineDefaultParameters",
	"rds:DescribeEventCategories",
	"rds:DescribeEvents",
	"rds:DescribeExportTasks",
	"rds:DescribeInstallationMedia",
	"rds:DescribeOrderableDBInstanceOptions",
	"rds:DescribeReservedDBInstancesOfferings",
	"rds:DescribeSourceRegions",
	"rds:ImportInstallationMedia",
	"rds:ModifyCertificates",
	"rds:StartExportTask",
	"redshift:AcceptReservedNodeExchange",
	"redshift:CancelQuery",
	"redshift:CancelQuerySession",
	"redshift:CreateSavedQuery",
	"redshift:CreateScheduledAction",
	"redshift:DeleteSavedQueries",
	"redshift:DeleteScheduledAction",
	"redshift:DescribeAccountAttributes",
	"redshift:DescribeClusterDbRevisions",
	"redshift:DescribeClusterParameterGroups",
	"redshift:DescribeClusterSecurityGroups",
	"redshift:DescribeClusterSnapshots",
	"redshift:DescribeClusterSubnetGroups",
	"redshift:DescribeClusterTracks",
	"redshift:DescribeClusterVersions",
	"redshift:DescribeClusters",
	"redshift:DescribeDefaultClusterParameters",
	"redshift:DescribeEventCategories",
	"redshift:DescribeEventSubscriptions",
	"redshift:DescribeEvents",
	"redshift:DescribeHsmClientCertificates",
	"redshift:DescribeHsmConfigurations",
	"redshift:DescribeNodeConfigurationOptions",
	"redshift:DescribeOrderableClusterOptions",
	"redshift:DescribeQuery",
	"redshift:DescribeReservedNodeOfferings",
	"redshift:DescribeReservedNodes",
	"redshift:DescribeSavedQueries",
	"redshift:DescribeScheduledActions",
	"redshift:DescribeSnapshotCopyGrants",
	"redshift:DescribeStorage",
	"redshift:DescribeTable",
	"redshift:DescribeTableRestoreStatus",
	"redshift:ExecuteQuery",
	"redshift:FetchResults",
	"redshift:GetReservedNodeExchangeOfferings",
	"redshift:ListDatabases",
	"redshift:ListSavedQueries",
	"redshift:ListSchemas",
	"redshift:ListTables",
	"redshift:ModifyClusterMaintenance",
	"redshift:ModifySavedQuery",
	"redshift:ModifyScheduledAction",
	"redshift:PurchaseReservedNodeOffering",
	"redshift:ViewQueriesFromConsole",
	"redshift:ViewQueriesInConsole",
	"redshift-data:CancelStatement",
	"redshift-data:DescribeStatement",
	"redshift-data:DescribeTable",
	"redshift-data:ExecuteStatement",
	"redshift-data:GetStatementResult",
	"redshift-data:ListDatabases",
	"redshift-data:ListSchemas",
	"redshift-data:ListStatements",
	"redshift-data:ListTables",
	"rekognition:CompareFaces",
	"rekognition:DescribeProjects",
	"rekognition:DetectFaces",
	"rekognition:DetectLabels",
	"rekognition:DetectModerationLabels",
	"rekognition:DetectProtectiveEquipment",
	"rekognition:DetectText",
	"rekognition:GetCelebrityInfo",
	"rekognition:GetCelebrityRecognition",
	"rekognition:GetContentModeration",
	"rekognition:GetFaceDetection",
	"rekognition:GetFaceSearch",
	"rekognition:GetLabelDetection",
	"rekognition:GetPersonTracking",
	"rekognition:GetSegmentDetection",
	"rekognition:GetTextDetection",
	"rekognition:RecognizeCelebrities",
	"rekognition:StartCelebrityRecognition",
	"rekognition:StartContentModeration",
	"rekognition:StartFaceDetection",
	"rekognition:StartLabelDetection",
	"rekognition:StartPersonTracking",
	"rekognition:StartSegmentDetection",
	"rekognition:StartTextDetection",
	"ram:CreateResourceShare",
	"ram:EnableSharingWithAwsOrganization",
	"ram:GetResourcePolicies",
	"ram:GetResourceShareAssociations",
	"ram:GetResourceShareInvitations",
	"ram:GetResourceShares",
	"ram:ListPermissions",
	"ram:ListPrincipals",
	"ram:ListResourceTypes",
	"ram:ListResources",
	"tag:DescribeReportCreation",
	"tag:GetComplianceSummary",
	"tag:GetResources",
	"tag:GetTagKeys",
	"tag:GetTagValues",
	"tag:StartReportCreation",
	"tag:TagResources",
	"tag:UntagResources",
	"resource-groups:CreateGroup",
	"resource-groups:ListGroups",
	"resource-groups:SearchResources",
	"robomaker:BatchDeleteWorlds",
	"robomaker:BatchDescribeSimulationJob",
	"robomaker:CreateDeploymentJob",
	"robomaker:CreateFleet",
	"robomaker:CreateRobot",
	"robomaker:CreateRobotApplication",
	"robomaker:CreateSimulationApplication",
	"robomaker:CreateSimulationJob",
	"robomaker:CreateWorldTemplate",
	"robomaker:ListDeploymentJobs",
	"robomaker:ListFleets",
	"robomaker:ListRobotApplications",
	"robomaker:ListRobots",
	"robomaker:ListSimulationApplications",
	"robomaker:ListSimulationJobBatches",
	"robomaker:ListSimulationJobs",
	"robomaker:ListSupportedAvailabilityZones [permission only]",
	"robomaker:ListWorldExportJobs",
	"robomaker:ListWorldGenerationJobs",
	"robomaker:ListWorldTemplates",
	"robomaker:ListWorlds",
	"robomaker:StartSimulationJobBatch",
	"robomaker:UpdateRobotDeployment [permission only]",
	"route53:CreateHealthCheck",
	"route53:CreateReusableDelegationSet",
	"route53:CreateTrafficPolicy",
	"route53:GetAccountLimit",
	"route53:GetCheckerIpRanges",
	"route53:GetGeoLocation",
	"route53:GetHealthCheckCount",
	"route53:GetHostedZoneCount",
	"route53:GetTrafficPolicyInstanceCount",
	"route53:ListGeoLocations",
	"route53:ListHealthChecks",
	"route53:ListHostedZones",
	"route53:ListHostedZonesByName",
	"route53:ListReusableDelegationSets",
	"route53:ListTrafficPolicies",
	"route53:ListTrafficPolicyInstances",
	"route53:TestDNSAnswer",
	"route53domains:AcceptDomainTransferFromAnotherAwsAccount",
	"route53domains:CancelDomainTransferToAnotherAwsAccount",
	"route53domains:CheckDomainAvailability",
	"route53domains:CheckDomainTransferability",
	"route53domains:DeleteTagsForDomain",
	"route53domains:DisableDomainAutoRenew",
	"route53domains:DisableDomainTransferLock",
	"route53domains:EnableDomainAutoRenew",
	"route53domains:EnableDomainTransferLock",
	"route53domains:GetContactReachabilityStatus",
	"route53domains:GetDomainDetail",
	"route53domains:GetDomainSuggestions",
	"route53domains:GetOperationDetail",
	"route53domains:ListDomains",
	"route53domains:ListOperations",
	"route53domains:ListTagsForDomain",
	"route53domains:RegisterDomain",
	"route53domains:RejectDomainTransferFromAnotherAwsAccount",
	"route53domains:RenewDomain",
	"route53domains:ResendContactReachabilityEmail",
	"route53domains:RetrieveDomainAuthCode",
	"route53domains:TransferDomain",
	"route53domains:TransferDomainToAnotherAwsAccount",
	"route53domains:UpdateDomainContact",
	"route53domains:UpdateDomainContactPrivacy",
	"route53domains:UpdateDomainNameservers",
	"route53domains:UpdateTagsForDomain",
	"route53domains:ViewBilling",
	"route53resolver:ListFirewallDomainLists",
	"route53resolver:ListFirewallRuleGroupAssociations",
	"route53resolver:ListFirewallRuleGroups",
	"route53resolver:ListResolverEndpoints",
	"route53resolver:ListResolverRuleAssociations",
	"route53resolver:ListResolverRules",
	"s3:CreateJob",
	"s3:GetAccessPoint",
	"s3:GetAccountPublicAccessBlock",
	"s3:ListAccessPoints",
	"s3:ListAccessPointsForObjectLambda",
	"s3:ListAllMyBuckets",
	"s3:ListJobs",
	"s3:ListStorageLensConfigurations",
	"s3:PutAccountPublicAccessBlock",
	"s3:PutStorageLensConfiguration",
	"s3-outposts:GetAccessPoint",
	"s3-outposts:ListAccessPoints",
	"s3-outposts:ListEndpoints",
	"s3-outposts:ListRegionalBuckets",
	"sagemaker:DisableSagemakerServicecatalogPortfolio",
	"sagemaker:EnableSagemakerServicecatalogPortfolio",
	"sagemaker:GetSagemakerServicecatalogPortfolioStatus",
	"sagemaker:GetSearchSuggestions",
	"sagemaker:ListActions",
	"sagemaker:ListAlgorithms",
	"sagemaker:ListAppImageConfigs",
	"sagemaker:ListApps",
	"sagemaker:ListArtifacts",
	"sagemaker:ListAssociations",
	"sagemaker:ListAutoMLJobs",
	"sagemaker:ListCandidatesForAutoMLJob",
	"sagemaker:ListCodeRepositories",
	"sagemaker:ListCompilationJobs",
	"sagemaker:ListContexts",
	"sagemaker:ListDataQualityJobDefinitions",
	"sagemaker:ListDeviceFleets",
	"sagemaker:ListDevices",
	"sagemaker:ListDomains",
	"sagemaker:ListEdgePackagingJobs",
	"sagemaker:ListEndpointConfigs",
	"sagemaker:ListEndpoints",
	"sagemaker:ListExperiments",
	"sagemaker:ListFeatureGroups",
	"sagemaker:ListFlowDefinitions",
	"sagemaker:ListHumanLoops",
	"sagemaker:ListHumanTaskUis",
	"sagemaker:ListHyperParameterTuningJobs",
	"sagemaker:ListImages",
	"sagemaker:ListLabelingJobs",
	"sagemaker:ListModelBiasJobDefinitions",
	"sagemaker:ListModelExplainabilityJobDefinitions",
	"sagemaker:ListModelPackageGroups",
	"sagemaker:ListModelPackages",
	"sagemaker:ListModelQualityJobDefinitions",
	"sagemaker:ListModels",
	"sagemaker:ListMonitoringExecutions",
	"sagemaker:ListMonitoringSchedules",
	"sagemaker:ListNotebookInstanceLifecycleConfigs",
	"sagemaker:ListNotebookInstances",
	"sagemaker:ListPipelines",
	"sagemaker:ListProcessingJobs",
	"sagemaker:ListProjects",
	"sagemaker:ListSubscribedWorkteams",
	"sagemaker:ListTrainingJobs",
	"sagemaker:ListTransformJobs",
	"sagemaker:ListTrialComponents",
	"sagemaker:ListTrials",
	"sagemaker:ListUserProfiles",
	"sagemaker:ListWorkforces",
	"sagemaker:ListWorkteams",
	"sagemaker:RenderUiTemplate",
	"sagemaker:Search",
	"savingsplans:CreateSavingsPlan",
	"savingsplans:DescribeSavingsPlansOfferingRates",
	"savingsplans:DescribeSavingsPlansOfferings",
	"secretsmanager:GetRandomPassword",
	"secretsmanager:ListSecrets",
	"sts:DecodeAuthorizationMessage",
	"sts:GetAccessKeyInfo",
	"sts:GetCallerIdentity",
	"sts:GetServiceBearerToken",
	"sts:GetSessionToken",
	"sms:CreateApp",
	"sms:CreateReplicationJob",
	"sms:DeleteApp",
	"sms:DeleteAppLaunchConfiguration",
	"sms:DeleteAppReplicationConfiguration",
	"sms:DeleteAppValidationConfiguration",
	"sms:DeleteReplicationJob",
	"sms:DeleteServerCatalog",
	"sms:DisassociateConnector",
	"sms:GenerateChangeSet",
	"sms:GenerateTemplate",
	"sms:GetApp",
	"sms:GetAppLaunchConfiguration",
	"sms:GetAppReplicationConfiguration",
	"sms:GetAppValidationConfiguration",
	"sms:GetAppValidationOutput",
	"sms:GetConnectors",
	"sms:GetMessages [permission only]",
	"sms:GetReplicationJobs",
	"sms:GetReplicationRuns",
	"sms:GetServers",
	"sms:ImportAppCatalog",
	"sms:ImportServerCatalog",
	"sms:LaunchApp",
	"sms:ListApps",
	"sms:NotifyAppValidationOutput",
	"sms:PutAppLaunchConfiguration",
	"sms:PutAppReplicationConfiguration",
	"sms:PutAppValidationConfiguration",
	"sms:SendMessage [permission only]",
	"sms:StartAppReplication",
	"sms:StartOnDemandAppReplication",
	"sms:StartOnDemandReplicationRun",
	"sms:StopAppReplication",
	"sms:TerminateApp",
	"sms:UpdateApp",
	"sms:UpdateReplicationJob",
	"serverlessrepo:CreateApplication",
	"serverlessrepo:ListApplications",
	"serverlessrepo:SearchApplications",
	"servicecatalog:AssociateBudgetWithResource",
	"servicecatalog:AssociateProductWithPortfolio",
	"servicecatalog:BatchAssociateServiceActionWithProvisioningArtifact",
	"servicecatalog:BatchDisassociateServiceActionFromProvisioningArtifact",
	"servicecatalog:CopyProduct",
	"servicecatalog:CreateProvisionedProductPlan",
	"servicecatalog:CreateServiceAction",
	"servicecatalog:CreateTagOption",
	"servicecatalog:DeleteConstraint",
	"servicecatalog:DeleteProvisionedProductPlan",
	"servicecatalog:DeleteServiceAction",
	"servicecatalog:DeleteTagOption",
	"servicecatalog:DescribeConstraint",
	"servicecatalog:DescribeCopyProductStatus",
	"servicecatalog:DescribePortfolioShareStatus",
	"servicecatalog:DescribeProductView",
	"servicecatalog:DescribeProvisionedProduct",
	"servicecatalog:DescribeProvisionedProductPlan",
	"servicecatalog:DescribeRecord",
	"servicecatalog:DescribeServiceAction",
	"servicecatalog:DescribeServiceActionExecutionParameters",
	"servicecatalog:DescribeTagOption",
	"servicecatalog:DisableAWSOrganizationsAccess",
	"servicecatalog:DisassociateBudgetFromResource",
	"servicecatalog:DisassociateProductFromPortfolio",
	"servicecatalog:EnableAWSOrganizationsAccess",
	"servicecatalog:ExecuteProvisionedProductPlan",
	"servicecatalog:ExecuteProvisionedProductServiceAction",
	"servicecatalog:GetAWSOrganizationsAccessStatus",
	"servicecatalog:GetProvisionedProductOutputs",
	"servicecatalog:ListAcceptedPortfolioShares",
	"servicecatalog:ListApplications",
	"servicecatalog:ListAttributeGroups",
	"servicecatalog:ListBudgetsForResource",
	"servicecatalog:ListConstraintsForPortfolio",
	"servicecatalog:ListOrganizationPortfolioAccess",
	"servicecatalog:ListPortfolios",
	"servicecatalog:ListProvisionedProductPlans",
	"servicecatalog:ListProvisioningArtifactsForServiceAction",
	"servicecatalog:ListRecordHistory",
	"servicecatalog:ListResourcesForTagOption",
	"servicecatalog:ListServiceActions",
	"servicecatalog:ListStackInstancesForProvisionedProduct",
	"servicecatalog:ListTagOptions",
	"servicecatalog:ScanProvisionedProducts",
	"servicecatalog:SearchProducts",
	"servicecatalog:SearchProductsAsAdmin",
	"servicecatalog:SearchProvisionedProducts",
	"servicecatalog:SyncResource",
	"servicecatalog:TerminateProvisionedProduct",
	"servicecatalog:UpdateConstraint",
	"servicecatalog:UpdateProvisionedProduct",
	"servicecatalog:UpdateProvisionedProductProperties",
	"servicecatalog:UpdateServiceAction",
	"servicecatalog:UpdateTagOption",
	"servicequotas:AssociateServiceQuotaTemplate",
	"servicequotas:DeleteServiceQuotaIncreaseRequestFromTemplate",
	"servicequotas:DisassociateServiceQuotaTemplate",
	"servicequotas:GetAWSDefaultServiceQuota",
	"servicequotas:GetAssociationForServiceQuotaTemplate",
	"servicequotas:GetRequestedServiceQuotaChange",
	"servicequotas:GetServiceQuota",
	"servicequotas:GetServiceQuotaIncreaseRequestFromTemplate",
	"servicequotas:ListAWSDefaultServiceQuotas",
	"servicequotas:ListRequestedServiceQuotaChangeHistory",
	"servicequotas:ListRequestedServiceQuotaChangeHistoryByQuota",
	"servicequotas:ListServiceQuotaIncreaseRequestsInTemplate",
	"servicequotas:ListServiceQuotas",
	"servicequotas:ListServices",
	"servicequotas:ListTagsForResource",
	"servicequotas:TagResource",
	"servicequotas:UntagResource",
	"ses:CloneReceiptRuleSet",
	"ses:CreateConfigurationSet",
	"ses:CreateConfigurationSetEventDestination",
	"ses:CreateConfigurationSetTrackingOptions",
	"ses:CreateReceiptFilter",
	"ses:CreateReceiptRule",
	"ses:CreateReceiptRuleSet",
	"ses:CreateTemplate",
	"ses:DeleteConfigurationSet",
	"ses:DeleteConfigurationSetEventDestination",
	"ses:DeleteConfigurationSetTrackingOptions",
	"ses:DeleteCustomVerificationEmailTemplate",
	"ses:DeleteReceiptFilter",
	"ses:DeleteReceiptRule",
	"ses:DeleteReceiptRuleSet",
	"ses:DeleteTemplate",
	"ses:DescribeActiveReceiptRuleSet",
	"ses:DescribeConfigurationSet",
	"ses:DescribeReceiptRule",
	"ses:DescribeReceiptRuleSet",
	"ses:GetAccountSendingEnabled",
	"ses:GetCustomVerificationEmailTemplate",
	"ses:GetSendQuota",
	"ses:GetSendStatistics",
	"ses:GetTemplate",
	"ses:ListConfigurationSets",
	"ses:ListCustomVerificationEmailTemplates",
	"ses:ListIdentities",
	"ses:ListIdentityPolicies",
	"ses:ListReceiptFilters",
	"ses:ListReceiptRuleSets",
	"ses:ListTemplates",
	"ses:ListVerifiedEmailAddresses",
	"ses:PutConfigurationSetDeliveryOptions",
	"ses:ReorderReceiptRuleSet",
	"ses:SendTemplatedEmail",
	"ses:SetActiveReceiptRuleSet",
	"ses:SetReceiptRulePosition",
	"ses:TestRenderTemplate",
	"ses:UpdateAccountSendingEnabled",
	"ses:UpdateConfigurationSetEventDestination",
	"ses:UpdateConfigurationSetReputationMetricsEnabled",
	"ses:UpdateConfigurationSetSendingEnabled",
	"ses:UpdateConfigurationSetTrackingOptions",
	"ses:UpdateCustomVerificationEmailTemplate",
	"ses:UpdateReceiptRule",
	"ses:UpdateTemplate",
	"ses:VerifyDomainIdentity",
	"ses:VerifyEmailAddress",
	"ses:VerifyEmailIdentity",
	"ssmmessages:CreateControlChannel",
	"ssmmessages:CreateDataChannel",
	"ssmmessages:OpenControlChannel",
	"ssmmessages:OpenDataChannel",
	"shield:AssociateDRTLogBucket",
	"shield:AssociateDRTRole",
	"shield:AssociateProactiveEngagementDetails",
	"shield:CreateProtection",
	"shield:CreateProtectionGroup",
	"shield:CreateSubscription",
	"shield:DeleteSubscription",
	"shield:DescribeAttackStatistics",
	"shield:DescribeDRTAccess",
	"shield:DescribeEmergencyContactSettings",
	"shield:DescribeSubscription",
	"shield:DisableProactiveEngagement",
	"shield:DisassociateDRTLogBucket",
	"shield:DisassociateDRTRole",
	"shield:EnableProactiveEngagement",
	"shield:GetSubscriptionState",
	"shield:ListAttacks",
	"shield:ListProtectionGroups",
	"shield:ListProtections",
	"shield:UpdateEmergencyContactSettings",
	"shield:UpdateSubscription",
	"signer:GetSigningPlatform",
	"signer:ListSigningJobs",
	"signer:ListSigningPlatforms",
	"signer:ListSigningProfiles",
	"signer:PutSigningProfile",
	"ses:CreateConfigurationSet",
	"ses:CreateContactList",
	"ses:CreateDedicatedIpPool",
	"ses:CreateEmailIdentity",
	"ses:CreateEmailTemplate",
	"ses:CreateImportJob",
	"ses:DeleteSuppressedDestination",
	"ses:GetAccount",
	"ses:GetBlacklistReports",
	"ses:GetDedicatedIp",
	"ses:GetDeliverabilityDashboardOptions",
	"ses:GetDomainDeliverabilityCampaign",
	"ses:GetSuppressedDestination",
	"ses:ListConfigurationSets",
	"ses:ListContactLists",
	"ses:ListCustomVerificationEmailTemplates",
	"ses:ListDedicatedIpPools",
	"ses:ListDeliverabilityTestReports",
	"ses:ListDomainDeliverabilityCampaigns",
	"ses:ListEmailIdentities",
	"ses:ListEmailTemplates",
	"ses:ListImportJobs",
	"ses:ListSuppressedDestinations",
	"ses:PutAccountDedicatedIpWarmupAttributes",
	"ses:PutAccountDetails",
	"ses:PutAccountSendingAttributes",
	"ses:PutAccountSuppressionAttributes",
	"ses:PutDedicatedIpWarmupAttributes",
	"ses:PutDeliverabilityDashboardOption",
	"ses:PutSuppressedDestination",
	"ses:SendBulkEmail",
	"swf:ListDomains",
	"swf:RegisterDomain",
	"sdb:ListDomains",
	"snowball:CancelCluster",
	"snowball:CancelJob",
	"snowball:CreateAddress",
	"snowball:CreateCluster",
	"snowball:CreateJob",
	"snowball:CreateReturnShippingLabel",
	"snowball:DescribeAddress",
	"snowball:DescribeAddresses",
	"snowball:DescribeCluster",
	"snowball:DescribeJob",
	"snowball:DescribeReturnShippingLabel",
	"snowball:GetJobManifest",
	"snowball:GetJobUnlockCode",
	"snowball:GetSnowballUsage",
	"snowball:GetSoftwareUpdates",
	"snowball:ListClusterJobs",
	"snowball:ListClusters",
	"snowball:ListCompatibleImages",
	"snowball:ListJobs",
	"snowball:UpdateCluster",
	"snowball:UpdateJob",
	"snowball:UpdateJobShipmentState",
	"sns:CheckIfPhoneNumberIsOptedOut",
	"sns:CreatePlatformApplication",
	"sns:CreatePlatformEndpoint",
	"sns:CreateSMSSandboxPhoneNumber",
	"sns:DeleteEndpoint",
	"sns:DeletePlatformApplication",
	"sns:DeleteSMSSandboxPhoneNumber",
	"sns:GetEndpointAttributes",
	"sns:GetPlatformApplicationAttributes",
	"sns:GetSMSAttributes",
	"sns:GetSMSSandboxAccountStatus",
	"sns:GetSubscriptionAttributes",
	"sns:ListEndpointsByPlatformApplication",
	"sns:ListOriginationNumbers",
	"sns:ListPhoneNumbersOptedOut",
	"sns:ListPlatformApplications",
	"sns:ListSMSSandboxPhoneNumbers",
	"sns:ListSubscriptions",
	"sns:ListTopics",
	"sns:OptInPhoneNumber",
	"sns:SetEndpointAttributes",
	"sns:SetPlatformApplicationAttributes",
	"sns:SetSMSAttributes",
	"sns:SetSubscriptionAttributes",
	"sns:Unsubscribe",
	"sns:VerifySMSSandboxPhoneNumber",
	"sqs:ListQueues",
	"sso:AssociateDirectory",
	"sso:AssociateProfile",
	"sso:CreateApplicationInstance",
	"sso:CreateApplicationInstanceCertificate",
	"sso:CreateManagedApplicationInstance",
	"sso:CreateProfile",
	"sso:CreateTrust",
	"sso:DeleteApplicationInstance",
	"sso:DeleteApplicationInstanceCertificate",
	"sso:DeleteManagedApplicationInstance",
	"sso:DeletePermissionsPolicy",
	"sso:DeleteProfile",
	"sso:DescribePermissionsPolicies",
	"sso:DescribeRegisteredRegions",
	"sso:DisassociateDirectory",
	"sso:DisassociateProfile",
	"sso:GetApplicationInstance",
	"sso:GetApplicationTemplate",
	"sso:GetManagedApplicationInstance",
	"sso:GetMfaDeviceManagementForDirectory",
	"sso:GetPermissionSet",
	"sso:GetPermissionsPolicy",
	"sso:GetProfile",
	"sso:GetSSOStatus",
	"sso:GetSharedSsoConfiguration",
	"sso:GetSsoConfiguration",
	"sso:GetTrust",
	"sso:ImportApplicationInstanceServiceProviderMetadata",
	"sso:ListApplicationInstanceCertificates",
	"sso:ListApplicationInstances",
	"sso:ListApplicationTemplates",
	"sso:ListApplications",
	"sso:ListDirectoryAssociations",
	"sso:ListInstances",
	"sso:ListProfileAssociations",
	"sso:ListProfiles",
	"sso:PutMfaDeviceManagementForDirectory",
	"sso:PutPermissionsPolicy",
	"sso:SearchGroups",
	"sso:SearchUsers",
	"sso:StartSSO",
	"sso:UpdateApplicationInstanceActiveCertificate",
	"sso:UpdateApplicationInstanceDisplayData",
	"sso:UpdateApplicationInstanceResponseConfiguration",
	"sso:UpdateApplicationInstanceResponseSchemaConfiguration",
	"sso:UpdateApplicationInstanceSecurityConfiguration",
	"sso:UpdateApplicationInstanceServiceProviderConfiguration",
	"sso:UpdateApplicationInstanceStatus",
	"sso:UpdateDirectoryAssociation",
	"sso:UpdateManagedApplicationInstanceStatus",
	"sso:UpdateProfile",
	"sso:UpdateSSOConfiguration",
	"sso:UpdateTrust",
	"sso-directory:AddMemberToGroup",
	"sso-directory:CompleteVirtualMfaDeviceRegistration",
	"sso-directory:CompleteWebAuthnDeviceRegistration",
	"sso-directory:CreateAlias",
	"sso-directory:CreateBearerToken",
	"sso-directory:CreateExternalIdPConfigurationForDirectory",
	"sso-directory:CreateGroup",
	"sso-directory:CreateProvisioningTenant",
	"sso-directory:CreateUser",
	"sso-directory:DeleteBearerToken",
	"sso-directory:DeleteExternalIdPCertificate",
	"sso-directory:DeleteExternalIdPConfigurationForDirectory",
	"sso-directory:DeleteGroup",
	"sso-directory:DeleteMfaDeviceForUser",
	"sso-directory:DeleteProvisioningTenant",
	"sso-directory:DeleteUser",
	"sso-directory:DescribeDirectory",
	"sso-directory:DescribeGroup",
	"sso-directory:DescribeGroups",
	"sso-directory:DescribeProvisioningTenant",
	"sso-directory:DescribeUser",
	"sso-directory:DescribeUserByUniqueAttribute",
	"sso-directory:DescribeUsers",
	"sso-directory:DisableExternalIdPConfigurationForDirectory",
	"sso-directory:DisableUser",
	"sso-directory:EnableExternalIdPConfigurationForDirectory",
	"sso-directory:EnableUser",
	"sso-directory:GetAWSSPConfigurationForDirectory",
	"sso-directory:GetUserPoolInfo",
	"sso-directory:ImportExternalIdPCertificate",
	"sso-directory:IsMemberInGroup",
	"sso-directory:ListBearerTokens",
	"sso-directory:ListExternalIdPCertificates",
	"sso-directory:ListExternalIdPConfigurationsForDirectory",
	"sso-directory:ListGroupsForMember",
	"sso-directory:ListGroupsForUser",
	"sso-directory:ListMembersInGroup",
	"sso-directory:ListMfaDevicesForUser",
	"sso-directory:ListProvisioningTenants",
	"sso-directory:RemoveMemberFromGroup",
	"sso-directory:SearchGroups",
	"sso-directory:SearchUsers",
	"sso-directory:StartVirtualMfaDeviceRegistration",
	"sso-directory:StartWebAuthnDeviceRegistration",
	"sso-directory:UpdateExternalIdPConfigurationForDirectory",
	"sso-directory:UpdateGroup",
	"sso-directory:UpdateGroupDisplayName",
	"sso-directory:UpdateMfaDeviceForUser",
	"sso-directory:UpdatePassword",
	"sso-directory:UpdateUser",
	"sso-directory:UpdateUserName",
	"sso-directory:VerifyEmail",
	"states:ListActivities",
	"states:ListStateMachines",
	"states:SendTaskFailure",
	"states:SendTaskHeartbeat",
	"states:SendTaskSuccess",
	"storagegateway:ActivateGateway",
	"storagegateway:CreateTapePool",
	"storagegateway:DeleteTapeArchive",
	"storagegateway:DescribeTapeArchives",
	"storagegateway:ListGateways",
	"sumerian:Login",
	"support:AddAttachmentsToSet",
	"support:AddCommunicationToCase",
	"support:CreateCase",
	"support:DescribeAttachment",
	"support:DescribeCaseAttributes",
	"support:DescribeCases",
	"support:DescribeCommunications",
	"support:DescribeIssueTypes",
	"support:DescribeServices",
	"support:DescribeSeverityLevels",
	"support:DescribeSupportLevel",
	"support:DescribeTrustedAdvisorCheckRefreshStatuses",
	"support:DescribeTrustedAdvisorCheckResult",
	"support:DescribeTrustedAdvisorCheckSummaries",
	"support:DescribeTrustedAdvisorChecks",
	"support:InitiateCallForCase",
	"support:InitiateChatForCase",
	"support:PutCaseAttributes",
	"support:RateCaseCommunication",
	"support:RefreshTrustedAdvisorCheck",
	"support:ResolveCase",
	"support:SearchForCases",
	"ssm:CancelCommand",
	"ssm:CancelMaintenanceWindowExecution",
	"ssm:CreateActivation",
	"ssm:CreateMaintenanceWindow",
	"ssm:CreateOpsItem",
	"ssm:CreateOpsMetadata",
	"ssm:CreatePatchBaseline",
	"ssm:DeleteActivation",
	"ssm:DeleteInventory",
	"ssm:DescribeActivations",
	"ssm:DescribeAssociationExecutionTargets",
	"ssm:DescribeAssociationExecutions",
	"ssm:DescribeAutomationExecutions",
	"ssm:DescribeAutomationStepExecutions",
	"ssm:DescribeAvailablePatches",
	"ssm:DescribeInstanceInformation",
	"ssm:DescribeInstancePatchStates",
	"ssm:DescribeInstancePatchStatesForPatchGroup",
	"ssm:DescribeInstancePatches",
	"ssm:DescribeInstanceProperties",
	"ssm:DescribeInventoryDeletions",
	"ssm:DescribeMaintenanceWindowExecutionTaskInvocations",
	"ssm:DescribeMaintenanceWindowExecutionTasks",
	"ssm:DescribeMaintenanceWindowSchedule",
	"ssm:DescribeMaintenanceWindows",
	"ssm:DescribeMaintenanceWindowsForTarget",
	"ssm:DescribeOpsItems",
	"ssm:DescribeParameters",
	"ssm:DescribePatchBaselines",
	"ssm:DescribePatchGroupState",
	"ssm:DescribePatchGroups",
	"ssm:DescribePatchProperties",
	"ssm:DescribeSessions",
	"ssm:GetAutomationExecution",
	"ssm:GetCommandInvocation",
	"ssm:GetConnectionStatus",
	"ssm:GetDeployablePatchSnapshotForInstance",
	"ssm:GetInventory",
	"ssm:GetInventorySchema",
	"ssm:GetMaintenanceWindowExecution",
	"ssm:GetMaintenanceWindowExecutionTask",
	"ssm:GetMaintenanceWindowExecutionTaskInvocation",
	"ssm:GetManifest",
	"ssm:ListAssociationVersions",
	"ssm:ListAssociations",
	"ssm:ListCommandInvocations",
	"ssm:ListCommands",
	"ssm:ListComplianceItems",
	"ssm:ListComplianceSummaries",
	"ssm:ListDocuments",
	"ssm:ListInventoryEntries",
	"ssm:ListOpsItemEvents",
	"ssm:ListOpsItemRelatedItems",
	"ssm:ListOpsMetadata",
	"ssm:ListResourceComplianceSummaries",
	"ssm:ListResourceDataSync",
	"ssm:PutConfigurePackageResult",
	"ssm:PutInventory",
	"ssm:SendAutomationSignal",
	"ssm:StopAutomationExecution",
	"ssm:UpdateInstanceInformation",
	"ssm-incidents:CreateReplicationSet",
	"ssm-incidents:CreateResponsePlan",
	"ssm-incidents:ListIncidentRecords",
	"ssm-incidents:ListReplicationSets",
	"ssm-incidents:ListResponsePlans",
	"ssm-contacts:ListContacts",
	"ssm-contacts:ListEngagements",
	"resource-explorer:ListResourceTypes",
	"resource-explorer:ListResources",
	"resource-explorer:ListTags",
	"textract:AnalyzeDocument",
	"textract:DetectDocumentText",
	"textract:GetDocumentAnalysis",
	"textract:GetDocumentTextDetection",
	"textract:StartDocumentAnalysis",
	"textract:StartDocumentTextDetection",
	"timestream:CancelQuery",
	"timestream:DescribeEndpoints",
	"timestream:ListDatabases",
	"timestream:SelectValues",
	"tiros:CreateQuery",
	"tiros:GetQueryAnswer",
	"tiros:GetQueryExplanation",
	"transcribe:CreateLanguageModel",
	"transcribe:CreateMedicalVocabulary",
	"transcribe:CreateVocabulary",
	"transcribe:CreateVocabularyFilter",
	"transcribe:DeleteLanguageModel",
	"transcribe:DeleteMedicalTranscriptionJob",
	"transcribe:DeleteMedicalVocabulary",
	"transcribe:DeleteTranscriptionJob",
	"transcribe:DeleteVocabulary",
	"transcribe:DeleteVocabularyFilter",
	"transcribe:DescribeLanguageModel",
	"transcribe:GetMedicalTranscriptionJob",
	"transcribe:GetMedicalVocabulary",
	"transcribe:GetTranscriptionJob",
	"transcribe:GetVocabulary",
	"transcribe:GetVocabularyFilter",
	"transcribe:ListLanguageModels",
	"transcribe:ListMedicalTranscriptionJobs",
	"transcribe:ListMedicalVocabularies",
	"transcribe:ListTranscriptionJobs",
	"transcribe:ListVocabularies",
	"transcribe:ListVocabularyFilters",
	"transcribe:StartMedicalStreamTranscription",
	"transcribe:StartMedicalStreamTranscriptionWebSocket",
	"transcribe:StartMedicalTranscriptionJob",
	"transcribe:StartStreamTranscription",
	"transcribe:StartStreamTranscriptionWebSocket",
	"transcribe:StartTranscriptionJob",
	"transcribe:UpdateMedicalVocabulary",
	"transcribe:UpdateVocabulary",
	"transcribe:UpdateVocabularyFilter",
	"transfer:CreateServer",
	"transfer:DescribeSecurityPolicy",
	"transfer:ListSecurityPolicies",
	"transfer:ListServers",
	"translate:CreateParallelData",
	"translate:DeleteParallelData",
	"translate:DeleteTerminology",
	"translate:DescribeTextTranslationJob",
	"translate:GetParallelData",
	"translate:GetTerminology",
	"translate:ImportTerminology",
	"translate:ListParallelData",
	"translate:ListTerminologies",
	"translate:ListTextTranslationJobs",
	"translate:StartTextTranslationJob",
	"translate:StopTextTranslationJob",
	"translate:TranslateText",
	"translate:UpdateParallelData",
	"trustedadvisor:DescribeAccount",
	"trustedadvisor:DescribeAccountAccess",
	"trustedadvisor:DescribeChecks",
	"trustedadvisor:DescribeNotificationPreferences",
	"trustedadvisor:DescribeOrganization",
	"trustedadvisor:DescribeOrganizationAccounts",
	"trustedadvisor:DescribeReports",
	"trustedadvisor:DescribeServiceMetadata",
	"trustedadvisor:GenerateReport",
	"trustedadvisor:ListAccountsForParent",
	"trustedadvisor:ListOrganizationalUnitsForParent",
	"trustedadvisor:ListRoots",
	"trustedadvisor:SetAccountAccess",
	"trustedadvisor:SetOrganizationAccess",
	"trustedadvisor:UpdateNotificationPreferences",
	"waf:GetChangeToken",
	"waf:GetChangeTokenStatus",
	"waf:ListActivatedRulesInRuleGroup",
	"waf:ListByteMatchSets",
	"waf:ListGeoMatchSets",
	"waf:ListIPSets",
	"waf:ListLoggingConfigurations",
	"waf:ListRateBasedRules",
	"waf:ListRegexMatchSets",
	"waf:ListRegexPatternSets",
	"waf:ListRuleGroups",
	"waf:ListRules",
	"waf:ListSizeConstraintSets",
	"waf:ListSqlInjectionMatchSets",
	"waf:ListSubscribedRuleGroups",
	"waf:ListWebACLs",
	"waf:ListXssMatchSets",
	"waf-regional:GetChangeToken",
	"waf-regional:GetChangeTokenStatus",
	"waf-regional:ListActivatedRulesInRuleGroup",
	"waf-regional:ListByteMatchSets",
	"waf-regional:ListGeoMatchSets",
	"waf-regional:ListIPSets",
	"waf-regional:ListLoggingConfigurations",
	"waf-regional:ListRateBasedRules",
	"waf-regional:ListRegexMatchSets",
	"waf-regional:ListRegexPatternSets",
	"waf-regional:ListRuleGroups",
	"waf-regional:ListRules",
	"waf-regional:ListSizeConstraintSets",
	"waf-regional:ListSqlInjectionMatchSets",
	"waf-regional:ListSubscribedRuleGroups",
	"waf-regional:ListWebACLs",
	"waf-regional:ListXssMatchSets",
	"wafv2:CheckCapacity",
	"wafv2:DescribeManagedRuleGroup",
	"wafv2:ListAvailableManagedRuleGroups",
	"wafv2:ListIPSets",
	"wafv2:ListLoggingConfigurations",
	"wafv2:ListRegexPatternSets",
	"wafv2:ListRuleGroups",
	"wafv2:ListWebACLs",
	"wellarchitected:CreateWorkload",
	"wellarchitected:GetLensVersionDifference",
	"wellarchitected:ListLenses",
	"wellarchitected:ListNotifications",
	"wellarchitected:ListShareInvitations",
	"wellarchitected:ListWorkloads",
	"workdocs:AbortDocumentVersionUpload",
	"workdocs:ActivateUser",
	"workdocs:AddResourcePermissions",
	"workdocs:AddUserToGroup",
	"workdocs:CheckAlias",
	"workdocs:CreateComment",
	"workdocs:CreateCustomMetadata",
	"workdocs:CreateFolder",
	"workdocs:CreateInstance",
	"workdocs:CreateLabels",
	"workdocs:CreateNotificationSubscription",
	"workdocs:CreateUser",
	"workdocs:DeactivateUser",
	"workdocs:DeleteComment",
	"workdocs:DeleteCustomMetadata",
	"workdocs:DeleteDocument",
	"workdocs:DeleteFolder",
	"workdocs:DeleteFolderContents",
	"workdocs:DeleteInstance",
	"workdocs:DeleteLabels",
	"workdocs:DeleteNotificationSubscription",
	"workdocs:DeleteUser",
	"workdocs:DeregisterDirectory",
	"workdocs:DescribeActivities",
	"workdocs:DescribeAvailableDirectories",
	"workdocs:DescribeComments",
	"workdocs:DescribeDocumentVersions",
	"workdocs:DescribeFolderContents",
	"workdocs:DescribeGroups",
	"workdocs:DescribeInstances",
	"workdocs:DescribeNotificationSubscriptions",
	"workdocs:DescribeResourcePermissions",
	"workdocs:DescribeRootFolders",
	"workdocs:DescribeUsers",
	"workdocs:DownloadDocumentVersion",
	"workdocs:GetCurrentUser",
	"workdocs:GetDocument",
	"workdocs:GetDocumentPath",
	"workdocs:GetDocumentVersion",
	"workdocs:GetFolder",
	"workdocs:GetFolderPath",
	"workdocs:GetGroup",
	"workdocs:GetResources",
	"workdocs:InitiateDocumentVersionUpload",
	"workdocs:RegisterDirectory",
	"workdocs:RemoveAllResourcePermissions",
	"workdocs:RemoveResourcePermission",
	"workdocs:UpdateDocument",
	"workdocs:UpdateDocumentVersion",
	"workdocs:UpdateFolder",
	"workdocs:UpdateInstanceAlias",
	"workdocs:UpdateUser",
	"worklink:CreateFleet",
	"worklink:ListFleets",
	"workmail:CreateOrganization",
	"workmail:DescribeDirectories",
	"workmail:DescribeKmsKeys",
	"workmail:DescribeOrganizations",
	"workmail:ListOrganizations",
	"workspaces:CreateConnectionAlias",
	"workspaces:CreateIpGroup",
	"workspaces:CreateTags",
	"workspaces:DeleteTags",
	"workspaces:DescribeAccount",
	"workspaces:DescribeAccountModifications",
	"workspaces:DescribeConnectionAliases",
	"workspaces:DescribeTags",
	"workspaces:DescribeWorkspaceBundles",
	"workspaces:DescribeWorkspaceDirectories",
	"workspaces:DescribeWorkspaceImages",
	"workspaces:DescribeWorkspaces",
	"workspaces:DescribeWorkspacesConnectionStatus",
	"workspaces:ImportWorkspaceImage",
	"workspaces:ListAvailableManagementCidrRanges",
	"workspaces:ModifyAccount",
	"wam:AuthenticatePackager",
	"xray:BatchGetTraces",
	"xray:GetEncryptionConfig",
	"xray:GetGroups",
	"xray:GetInsight",
	"xray:GetInsightEvents",
	"xray:GetInsightImpactGraph",
	"xray:GetInsightSummaries",
	"xray:GetSamplingRules",
	"xray:GetSamplingStatisticSummaries",
	"xray:GetSamplingTargets",
	"xray:GetServiceGraph",
	"xray:GetTimeSeriesServiceStatistics",
	"xray:GetTraceGraph",
	"xray:GetTraceSummaries",
	"xray:PutEncryptionConfig",
	"xray:PutTelemetryRecords",
	"xray:PutTraceSegments",
}
