unit Horse.Core.Files;

{$IF DEFINED(FPC)}
  {$MODE DELPHI}{$H+}
{$ENDIF}

interface

uses
{$IF DEFINED(FPC)}
  SysUtils, Classes, Generics.Collections;
{$ELSE}
  System.SysUtils, System.Classes, System.Generics.Collections;
{$ENDIF}

type THorseCoreFile = class

  private
    FFileType: TDictionary<string, String>;
    FFileName: string;
    FFileStream: TStream;
    FFreeContentStream: Boolean;

  protected
    procedure InitializeFileType;
    function GetExtInfo: string;

  public
    function ContentType: string;
    function ContentStream: TStream;
    function Size: Int64;

    property FreeContentStream: Boolean read FFreeContentStream write FFreeContentStream;

    constructor Create(const AFileName: String);
    destructor Destroy; override;
end;

implementation

{$IFNDEF FPC}
  {$IF (CompilerVersion >= 32.0)}
  uses
    System.Net.Mime;
  {$ENDIF}
{$ENDIF}

{ THorseCoreFile }

function THorseCoreFile.ContentType: string;
{$IFNDEF FPC}
  {$IF CompilerVersion >= 32.0}
  var
    LKind: TMimeTypes.TKind;
  {$ENDIF}
{$ENDIF}
begin
  {$IF DEFINED(FPC)}
  InitializeFileType;
  FFileType.TryGetValue(GetExtInfo, Result);
  {$ELSE}
    {$IF CompilerVersion >= 32.0}
    TMimeTypes.Default.GetFileInfo(FFileName, Result, LKind);
    {$ELSE}
    InitializeFileType;
    FFileType.TryGetValue(GetExtInfo, Result);
    {$ENDIF}
  {$ENDIF}
end;

constructor THorseCoreFile.Create(const AFileName: String);
begin
  if AFileName = EmptyStr then
    raise Exception.Create('Invalid FileName');

  if not FileExists(AFileName) then
    raise Exception.Create('File not exist');

  FFileName := AFileName;
  FFreeContentStream := True;
end;

destructor THorseCoreFile.Destroy;
begin
  FFileType.Free;
  if FFreeContentStream then
    FFileStream.Free;
  inherited;
end;

function THorseCoreFile.GetExtInfo: string;
begin
  Result := ExtractFileExt(FFileName.Trim.ToLower);
  if (Result <> '') and (Result[Low(Result)] = '.') then
    Result := Result.Substring(1);
end;

procedure THorseCoreFile.InitializeFileType;
begin
  if not Assigned(FFileType) then
    FFileType := TDictionary<String, String>.Create;
{$REGION 'MIME CONST'}
  FFileType.AddOrSetValue('ez', 'application/andrew-inset');
  FFileType.AddOrSetValue('aw', 'application/applixware');
  FFileType.AddOrSetValue('atom', 'application/atom+xml');
  FFileType.AddOrSetValue('atomcat', 'application/atomcat+xml');
  FFileType.AddOrSetValue('atomsvc', 'application/atomsvc+xml');
  FFileType.AddOrSetValue('bson', 'application/bson');
  FFileType.AddOrSetValue('ccxml', 'application/ccxml+xml');
  FFileType.AddOrSetValue('cdmia', 'application/cdmi-capability');
  FFileType.AddOrSetValue('cdmic', 'application/cdmi-container');
  FFileType.AddOrSetValue('cdmid', 'application/cdmi-domain');
  FFileType.AddOrSetValue('cdmio', 'application/cdmi-object');
  FFileType.AddOrSetValue('cdmiq', 'application/cdmi-queue');
  FFileType.AddOrSetValue('cu', 'application/cu-seeme');
  FFileType.AddOrSetValue('davmount', 'application/davmount+xml');
  FFileType.AddOrSetValue('dbk', 'application/docbook+xml');
  FFileType.AddOrSetValue('dssc', 'application/dssc+der');
  FFileType.AddOrSetValue('xdssc', 'application/dssc+xml');
  FFileType.AddOrSetValue('ecma', 'application/ecmascript');
  FFileType.AddOrSetValue('emma', 'application/emma+xml');
  FFileType.AddOrSetValue('epub', 'application/epub+zip');
  FFileType.AddOrSetValue('exi', 'application/exi');
  FFileType.AddOrSetValue('pfr', 'application/font-tdpfr');
  FFileType.AddOrSetValue('gml', 'application/gml+xml');
  FFileType.AddOrSetValue('gpx', 'application/gpx+xml');
  FFileType.AddOrSetValue('gxf', 'application/gxf');
  FFileType.AddOrSetValue('stk', 'application/hyperstudio');
  FFileType.AddOrSetValue('ink', 'application/inkml+xml');
  FFileType.AddOrSetValue('inkml', 'application/inkml+xml');
  FFileType.AddOrSetValue('ipfix', 'application/ipfix');
  FFileType.AddOrSetValue('jar', 'application/java-archive');
  FFileType.AddOrSetValue('ser', 'application/java-serialized-object');
  FFileType.AddOrSetValue('class', 'application/java-vm');
  FFileType.AddOrSetValue('js', 'application/javascript');
  FFileType.AddOrSetValue('json', 'application/json');
  FFileType.AddOrSetValue('map', 'application/json');
  FFileType.AddOrSetValue('jsonml', 'application/jsonml+json');
  FFileType.AddOrSetValue('lostxml', 'application/lost+xml');
  FFileType.AddOrSetValue('hqx', 'application/mac-binhex40');
  FFileType.AddOrSetValue('cpt', 'application/mac-compactpro');
  FFileType.AddOrSetValue('mads', 'application/mads+xml');
  FFileType.AddOrSetValue('mrc', 'application/marc');
  FFileType.AddOrSetValue('mrcx', 'application/marcxml+xml');
  FFileType.AddOrSetValue('ma', 'application/mathematica');
  FFileType.AddOrSetValue('nb', 'application/mathematica');
  FFileType.AddOrSetValue('mb', 'application/mathematica');
  FFileType.AddOrSetValue('mathml', 'application/mathml+xml');
  FFileType.AddOrSetValue('mbox', 'application/mbox');
  FFileType.AddOrSetValue('mscml', 'application/mediaservercontrol+xml');
  FFileType.AddOrSetValue('metalink', 'application/metalink+xml');
  FFileType.AddOrSetValue('meta4', 'application/metalink4+xml');
  FFileType.AddOrSetValue('mets', 'application/mets+xml');
  FFileType.AddOrSetValue('mods', 'application/mods+xml');
  FFileType.AddOrSetValue('m21', 'application/mp21');
  FFileType.AddOrSetValue('mp21', 'application/mp21');
  FFileType.AddOrSetValue('mp4s', 'application/mp4');
  FFileType.AddOrSetValue('doc', 'application/msword');
  FFileType.AddOrSetValue('dot', 'application/msword');
  FFileType.AddOrSetValue('mxf', 'application/mxf');
  FFileType.AddOrSetValue('bin', 'application/octet-stream');
  FFileType.AddOrSetValue('bpk', 'application/octet-stream');
  FFileType.AddOrSetValue('class', 'application/octet-stream');
  FFileType.AddOrSetValue('deploy', 'application/octet-stream');
  FFileType.AddOrSetValue('dist', 'application/octet-stream');
  FFileType.AddOrSetValue('distz', 'application/octet-stream');
  FFileType.AddOrSetValue('dmg', 'application/octet-stream');
  FFileType.AddOrSetValue('dms', 'application/octet-stream');
  FFileType.AddOrSetValue('dump', 'application/octet-stream');
  FFileType.AddOrSetValue('elc', 'application/octet-stream');
  FFileType.AddOrSetValue('iso', 'application/octet-stream');
  FFileType.AddOrSetValue('lha', 'application/octet-stream');
  FFileType.AddOrSetValue('lrf', 'application/octet-stream');
  FFileType.AddOrSetValue('lzh', 'application/octet-stream');
  FFileType.AddOrSetValue('mar', 'application/octet-stream');
  FFileType.AddOrSetValue('pkg', 'application/octet-stream');
  FFileType.AddOrSetValue('so', 'application/octet-stream');
  FFileType.AddOrSetValue('oda', 'application/oda');
  FFileType.AddOrSetValue('opf', 'application/oebps-package+xml');
  FFileType.AddOrSetValue('ogx', 'application/ogg');
  FFileType.AddOrSetValue('omdoc', 'application/omdoc+xml');
  FFileType.AddOrSetValue('onetoc', 'application/onenote');
  FFileType.AddOrSetValue('onetoc2', 'application/onenote');
  FFileType.AddOrSetValue('onetmp', 'application/onenote');
  FFileType.AddOrSetValue('onepkg', 'application/onenote');
  FFileType.AddOrSetValue('oxps', 'application/oxps');
  FFileType.AddOrSetValue('xer', 'application/patch-ops-error+xml');
  FFileType.AddOrSetValue('pdf', 'application/pdf');
  FFileType.AddOrSetValue('pgp', 'application/pgp-encrypted');
  FFileType.AddOrSetValue('asc', 'application/pgp-signature');
  FFileType.AddOrSetValue('sig', 'application/pgp-signature');
  FFileType.AddOrSetValue('prf', 'application/pics-rules');
  FFileType.AddOrSetValue('p10', 'application/pkcs10');
  FFileType.AddOrSetValue('p7m', 'application/pkcs7-mime');
  FFileType.AddOrSetValue('p7c', 'application/pkcs7-mime');
  FFileType.AddOrSetValue('p7s', 'application/pkcs7-signature');
  FFileType.AddOrSetValue('p8', 'application/pkcs8');
  FFileType.AddOrSetValue('ac', 'application/pkix-attr-cert');
  FFileType.AddOrSetValue('cer', 'application/pkix-cert');
  FFileType.AddOrSetValue('crl', 'application/pkix-crl');
  FFileType.AddOrSetValue('pkipath', 'application/pkix-pkipath');
  FFileType.AddOrSetValue('pki', 'application/pkixcmp');
  FFileType.AddOrSetValue('pls', 'application/pls+xml');
  FFileType.AddOrSetValue('ai', 'application/postscript');
  FFileType.AddOrSetValue('eps', 'application/postscript');
  FFileType.AddOrSetValue('ps', 'application/postscript');
  FFileType.AddOrSetValue('cww', 'application/prs.cww');
  FFileType.AddOrSetValue('pskcxml', 'application/pskc+xml');
  FFileType.AddOrSetValue('rdf', 'application/rdf+xml');
  FFileType.AddOrSetValue('rif', 'application/reginfo+xml');
  FFileType.AddOrSetValue('rnc', 'application/relax-ng-compact-syntax');
  FFileType.AddOrSetValue('rl', 'application/resource-lists+xml');
  FFileType.AddOrSetValue('rld', 'application/resource-lists-diff+xml');
  FFileType.AddOrSetValue('rs', 'application/rls-services+xml');
  FFileType.AddOrSetValue('gbr', 'application/rpki-ghostbusters');
  FFileType.AddOrSetValue('mft', 'application/rpki-manifest');
  FFileType.AddOrSetValue('roa', 'application/rpki-roa');
  FFileType.AddOrSetValue('rsd', 'application/rsd+xml');
  FFileType.AddOrSetValue('rss', 'application/rss+xml');
  FFileType.AddOrSetValue('rtf', 'application/rtf');
  FFileType.AddOrSetValue('sbml', 'application/sbml+xml');
  FFileType.AddOrSetValue('scq', 'application/scvp-cv-request');
  FFileType.AddOrSetValue('scs', 'application/scvp-cv-response');
  FFileType.AddOrSetValue('spq', 'application/scvp-vp-request');
  FFileType.AddOrSetValue('spp', 'application/scvp-vp-response');
  FFileType.AddOrSetValue('sdp', 'application/sdp');
  FFileType.AddOrSetValue('setpay', 'application/set-payment-initiation');
  FFileType.AddOrSetValue('setreg', 'application/set-registration-initiation');
  FFileType.AddOrSetValue('shf', 'application/shf+xml');
  FFileType.AddOrSetValue('smi', 'application/smil+xml');
  FFileType.AddOrSetValue('smil', 'application/smil+xml');
  FFileType.AddOrSetValue('soap', 'application/soap+xml');
  FFileType.AddOrSetValue('rq', 'application/sparql-query');
  FFileType.AddOrSetValue('srx', 'application/sparql-results+xml');
  FFileType.AddOrSetValue('gram', 'application/srgs');
  FFileType.AddOrSetValue('grxml', 'application/srgs+xml');
  FFileType.AddOrSetValue('sru', 'application/sru+xml');
  FFileType.AddOrSetValue('ssdl', 'application/ssdl+xml');
  FFileType.AddOrSetValue('ssml', 'application/ssml+xml');
  FFileType.AddOrSetValue('tei', 'application/tei+xml');
  FFileType.AddOrSetValue('teicorpus', 'application/tei+xml');
  FFileType.AddOrSetValue('tfi', 'application/thraud+xml');
  FFileType.AddOrSetValue('tsd', 'application/timestamped-data');
  FFileType.AddOrSetValue('plb', 'application/vnd.3gpp.pic-bw-large');
  FFileType.AddOrSetValue('psb', 'application/vnd.3gpp.pic-bw-small');
  FFileType.AddOrSetValue('pvb', 'application/vnd.3gpp.pic-bw-var');
  FFileType.AddOrSetValue('tcap', 'application/vnd.3gpp2.tcap');
  FFileType.AddOrSetValue('pwn', 'application/vnd.3m.post-it-notes');
  FFileType.AddOrSetValue('aso', 'application/vnd.accpac.simply.aso');
  FFileType.AddOrSetValue('imp', 'application/vnd.accpac.simply.imp');
  FFileType.AddOrSetValue('acu', 'application/vnd.acucobol');
  FFileType.AddOrSetValue('atc', 'application/vnd.acucorp');
  FFileType.AddOrSetValue('acutc', 'application/vnd.acucorp');
  FFileType.AddOrSetValue('air', 'application/vnd.adobe.air-application-installer-package+zip');
  FFileType.AddOrSetValue('fcdt', 'application/vnd.adobe.formscentral.fcdt');
  FFileType.AddOrSetValue('fxp', 'application/vnd.adobe.fxp');
  FFileType.AddOrSetValue('fxpl', 'application/vnd.adobe.fxp');
  FFileType.AddOrSetValue('xdp', 'application/vnd.adobe.xdp+xml');
  FFileType.AddOrSetValue('xfdf', 'application/vnd.adobe.xfdf');
  FFileType.AddOrSetValue('ahead', 'application/vnd.ahead.space');
  FFileType.AddOrSetValue('azf', 'application/vnd.airzip.filesecure.azf');
  FFileType.AddOrSetValue('azs', 'application/vnd.airzip.filesecure.azs');
  FFileType.AddOrSetValue('azw', 'application/vnd.amazon.ebook');
  FFileType.AddOrSetValue('acc', 'application/vnd.americandynamics.acc');
  FFileType.AddOrSetValue('ami', 'application/vnd.amiga.ami');
  FFileType.AddOrSetValue('apk', 'application/vnd.android.package-archive');
  FFileType.AddOrSetValue('cii', 'application/vnd.anser-web-certificate-issue-initiation');
  FFileType.AddOrSetValue('fti', 'application/vnd.anser-web-funds-transfer-initiation');
  FFileType.AddOrSetValue('atx', 'application/vnd.antix.game-component');
  FFileType.AddOrSetValue('mpkg', 'application/vnd.apple.installer+xml');
  FFileType.AddOrSetValue('m3u8', 'application/vnd.apple.mpegurl');
  FFileType.AddOrSetValue('swi', 'application/vnd.aristanetworks.swi');
  FFileType.AddOrSetValue('iota', 'application/vnd.astraea-software.iota');
  FFileType.AddOrSetValue('aep', 'application/vnd.audiograph');
  FFileType.AddOrSetValue('mpm', 'application/vnd.blueice.multipass');
  FFileType.AddOrSetValue('bmi', 'application/vnd.bmi');
  FFileType.AddOrSetValue('rep', 'application/vnd.businessobjects');
  FFileType.AddOrSetValue('cdxml', 'application/vnd.chemdraw+xml');
  FFileType.AddOrSetValue('mmd', 'application/vnd.chipnuts.karaoke-mmd');
  FFileType.AddOrSetValue('cdy', 'application/vnd.cinderella');
  FFileType.AddOrSetValue('cla', 'application/vnd.claymore');
  FFileType.AddOrSetValue('rp9', 'application/vnd.cloanto.rp9');
  FFileType.AddOrSetValue('c4g', 'application/vnd.clonk.c4group');
  FFileType.AddOrSetValue('c4d', 'application/vnd.clonk.c4group');
  FFileType.AddOrSetValue('c4f', 'application/vnd.clonk.c4group');
  FFileType.AddOrSetValue('c4p', 'application/vnd.clonk.c4group');
  FFileType.AddOrSetValue('c4u', 'application/vnd.clonk.c4group');
  FFileType.AddOrSetValue('c11amc', 'application/vnd.cluetrust.cartomobile-config');
  FFileType.AddOrSetValue('c11amz', 'application/vnd.cluetrust.cartomobile-config-pkg');
  FFileType.AddOrSetValue('csp', 'application/vnd.commonspace');
  FFileType.AddOrSetValue('cdbcmsg', 'application/vnd.contact.cmsg');
  FFileType.AddOrSetValue('cmc', 'application/vnd.cosmocaller');
  FFileType.AddOrSetValue('clkx', 'application/vnd.crick.clicker');
  FFileType.AddOrSetValue('clkk', 'application/vnd.crick.clicker.keyboard');
  FFileType.AddOrSetValue('clkp', 'application/vnd.crick.clicker.palette');
  FFileType.AddOrSetValue('clkt', 'application/vnd.crick.clicker.template');
  FFileType.AddOrSetValue('clkw', 'application/vnd.crick.clicker.wordbank');
  FFileType.AddOrSetValue('wbs', 'application/vnd.criticaltools.wbs+xml');
  FFileType.AddOrSetValue('pml', 'application/vnd.ctc-posml');
  FFileType.AddOrSetValue('ppd', 'application/vnd.cups-ppd');
  FFileType.AddOrSetValue('car', 'application/vnd.curl.car');
  FFileType.AddOrSetValue('pcurl', 'application/vnd.curl.pcurl');
  FFileType.AddOrSetValue('dart', 'application/vnd.dart');
  FFileType.AddOrSetValue('rdz', 'application/vnd.data-vision.rdz');
  FFileType.AddOrSetValue('uvf', 'application/vnd.dece.data');
  FFileType.AddOrSetValue('uvvf', 'application/vnd.dece.data');
  FFileType.AddOrSetValue('uvd', 'application/vnd.dece.data');
  FFileType.AddOrSetValue('uvvd', 'application/vnd.dece.data');
  FFileType.AddOrSetValue('uvt', 'application/vnd.dece.ttml+xml');
  FFileType.AddOrSetValue('uvvt', 'application/vnd.dece.ttml+xml');
  FFileType.AddOrSetValue('uvx', 'application/vnd.dece.unspecified');
  FFileType.AddOrSetValue('uvvx', 'application/vnd.dece.unspecified');
  FFileType.AddOrSetValue('uvz', 'application/vnd.dece.zip');
  FFileType.AddOrSetValue('uvvz', 'application/vnd.dece.zip');
  FFileType.AddOrSetValue('fe_launch', 'application/vnd.denovo.fcselayout-link');
  FFileType.AddOrSetValue('dna', 'application/vnd.dna');
  FFileType.AddOrSetValue('mlp', 'application/vnd.dolby.mlp');
  FFileType.AddOrSetValue('dpg', 'application/vnd.dpgraph');
  FFileType.AddOrSetValue('dfac', 'application/vnd.dreamfactory');
  FFileType.AddOrSetValue('kpxx', 'application/vnd.ds-keypoint');
  FFileType.AddOrSetValue('ait', 'application/vnd.dvb.ait');
  FFileType.AddOrSetValue('svc', 'application/vnd.dvb.service');
  FFileType.AddOrSetValue('geo', 'application/vnd.dynageo');
  FFileType.AddOrSetValue('mag', 'application/vnd.ecowin.chart');
  FFileType.AddOrSetValue('nml', 'application/vnd.enliven');
  FFileType.AddOrSetValue('esf', 'application/vnd.epson.esf');
  FFileType.AddOrSetValue('msf', 'application/vnd.epson.msf');
  FFileType.AddOrSetValue('qam', 'application/vnd.epson.quickanime');
  FFileType.AddOrSetValue('slt', 'application/vnd.epson.salt');
  FFileType.AddOrSetValue('ssf', 'application/vnd.epson.ssf');
  FFileType.AddOrSetValue('es3', 'application/vnd.eszigno3+xml');
  FFileType.AddOrSetValue('et3', 'application/vnd.eszigno3+xml');
  FFileType.AddOrSetValue('ez2', 'application/vnd.ezpix-album');
  FFileType.AddOrSetValue('ez3', 'application/vnd.ezpix-package');
  FFileType.AddOrSetValue('fdf', 'application/vnd.fdf');
  FFileType.AddOrSetValue('mseed', 'application/vnd.fdsn.mseed');
  FFileType.AddOrSetValue('seed', 'application/vnd.fdsn.seed');
  FFileType.AddOrSetValue('dataless', 'application/vnd.fdsn.seed');
  FFileType.AddOrSetValue('json', 'application/vnd.embarcadero.firedac+json');
  FFileType.AddOrSetValue('xml', 'application/vnd.embarcadero.firedac+xml');
  FFileType.AddOrSetValue('bin', 'application/vnd.embarcadero.firedac+bin');
  FFileType.AddOrSetValue('gph', 'application/vnd.flographit');
  FFileType.AddOrSetValue('ftc', 'application/vnd.fluxtime.clip');
  FFileType.AddOrSetValue('fm', 'application/vnd.framemaker');
  FFileType.AddOrSetValue('frame', 'application/vnd.framemaker');
  FFileType.AddOrSetValue('maker', 'application/vnd.framemaker');
  FFileType.AddOrSetValue('book', 'application/vnd.framemaker');
  FFileType.AddOrSetValue('fnc', 'application/vnd.frogans.fnc');
  FFileType.AddOrSetValue('ltf', 'application/vnd.frogans.ltf');
  FFileType.AddOrSetValue('fsc', 'application/vnd.fsc.weblaunch');
  FFileType.AddOrSetValue('oas', 'application/vnd.fujitsu.oasys');
  FFileType.AddOrSetValue('oa2', 'application/vnd.fujitsu.oasys2');
  FFileType.AddOrSetValue('oa3', 'application/vnd.fujitsu.oasys3');
  FFileType.AddOrSetValue('fg5', 'application/vnd.fujitsu.oasysgp');
  FFileType.AddOrSetValue('bh2', 'application/vnd.fujitsu.oasysprs');
  FFileType.AddOrSetValue('ddd', 'application/vnd.fujixerox.ddd');
  FFileType.AddOrSetValue('xdw', 'application/vnd.fujixerox.docuworks');
  FFileType.AddOrSetValue('xbd', 'application/vnd.fujixerox.docuworks.binder');
  FFileType.AddOrSetValue('fzs', 'application/vnd.fuzzysheet');
  FFileType.AddOrSetValue('txd', 'application/vnd.genomatix.tuxedo');
  FFileType.AddOrSetValue('ggb', 'application/vnd.geogebra.file');
  FFileType.AddOrSetValue('ggt', 'application/vnd.geogebra.tool');
  FFileType.AddOrSetValue('gex', 'application/vnd.geometry-explorer');
  FFileType.AddOrSetValue('gre', 'application/vnd.geometry-explorer');
  FFileType.AddOrSetValue('gxt', 'application/vnd.geonext');
  FFileType.AddOrSetValue('g2w', 'application/vnd.geoplan');
  FFileType.AddOrSetValue('g3w', 'application/vnd.geospace');
  FFileType.AddOrSetValue('gmx', 'application/vnd.gmx');
  FFileType.AddOrSetValue('kml', 'application/vnd.google-earth.kml+xml');
  FFileType.AddOrSetValue('kmz', 'application/vnd.google-earth.kmz');
  FFileType.AddOrSetValue('gqf', 'application/vnd.grafeq');
  FFileType.AddOrSetValue('gqs', 'application/vnd.grafeq');
  FFileType.AddOrSetValue('gac', 'application/vnd.groove-account');
  FFileType.AddOrSetValue('ghf', 'application/vnd.groove-help');
  FFileType.AddOrSetValue('gim', 'application/vnd.groove-identity-message');
  FFileType.AddOrSetValue('grv', 'application/vnd.groove-injector');
  FFileType.AddOrSetValue('gtm', 'application/vnd.groove-tool-message');
  FFileType.AddOrSetValue('tpl', 'application/vnd.groove-tool-template');
  FFileType.AddOrSetValue('vcg', 'application/vnd.groove-vcard');
  FFileType.AddOrSetValue('hal', 'application/vnd.hal+xml');
  FFileType.AddOrSetValue('zmm', 'application/vnd.handheld-entertainment+xml');
  FFileType.AddOrSetValue('hbci', 'application/vnd.hbci');
  FFileType.AddOrSetValue('les', 'application/vnd.hhe.lesson-player');
  FFileType.AddOrSetValue('hpgl', 'application/vnd.hp-hpgl');
  FFileType.AddOrSetValue('hpid', 'application/vnd.hp-hpid');
  FFileType.AddOrSetValue('hps', 'application/vnd.hp-hps');
  FFileType.AddOrSetValue('jlt', 'application/vnd.hp-jlyt');
  FFileType.AddOrSetValue('pcl', 'application/vnd.hp-pcl');
  FFileType.AddOrSetValue('pclxl', 'application/vnd.hp-pclxl');
  FFileType.AddOrSetValue('sfd-hdstx', 'application/vnd.hydrostatix.sof-data');
  FFileType.AddOrSetValue('mpy', 'application/vnd.ibm.minipay');
  FFileType.AddOrSetValue('afp', 'application/vnd.ibm.modcap');
  FFileType.AddOrSetValue('listafp', 'application/vnd.ibm.modcap');
  FFileType.AddOrSetValue('list3820', 'application/vnd.ibm.modcap');
  FFileType.AddOrSetValue('irm', 'application/vnd.ibm.rights-management');
  FFileType.AddOrSetValue('sc', 'application/vnd.ibm.secure-container');
  FFileType.AddOrSetValue('icc', 'application/vnd.iccprofile');
  FFileType.AddOrSetValue('icm', 'application/vnd.iccprofile');
  FFileType.AddOrSetValue('igl', 'application/vnd.igloader');
  FFileType.AddOrSetValue('ivp', 'application/vnd.immervision-ivp');
  FFileType.AddOrSetValue('ivu', 'application/vnd.immervision-ivu');
  FFileType.AddOrSetValue('igm', 'application/vnd.insors.igm');
  FFileType.AddOrSetValue('xpw', 'application/vnd.intercon.formnet');
  FFileType.AddOrSetValue('xpx', 'application/vnd.intercon.formnet');
  FFileType.AddOrSetValue('i2g', 'application/vnd.intergeo');
  FFileType.AddOrSetValue('qbo', 'application/vnd.intu.qbo');
  FFileType.AddOrSetValue('qfx', 'application/vnd.intu.qfx');
  FFileType.AddOrSetValue('rcprofile', 'application/vnd.ipunplugged.rcprofile');
  FFileType.AddOrSetValue('irp', 'application/vnd.irepository.package+xml');
  FFileType.AddOrSetValue('xpr', 'application/vnd.is-xpr');
  FFileType.AddOrSetValue('fcs', 'application/vnd.isac.fcs');
  FFileType.AddOrSetValue('jam', 'application/vnd.jam');
  FFileType.AddOrSetValue('rms', 'application/vnd.jcp.javame.midlet-rms');
  FFileType.AddOrSetValue('jisp', 'application/vnd.jisp');
  FFileType.AddOrSetValue('joda', 'application/vnd.joost.joda-archive');
  FFileType.AddOrSetValue('ktz', 'application/vnd.kahootz');
  FFileType.AddOrSetValue('ktr', 'application/vnd.kahootz');
  FFileType.AddOrSetValue('karbon', 'application/vnd.kde.karbon');
  FFileType.AddOrSetValue('chrt', 'application/vnd.kde.kchart');
  FFileType.AddOrSetValue('kfo', 'application/vnd.kde.kformula');
  FFileType.AddOrSetValue('flw', 'application/vnd.kde.kivio');
  FFileType.AddOrSetValue('kon', 'application/vnd.kde.kontour');
  FFileType.AddOrSetValue('kpr', 'application/vnd.kde.kpresenter');
  FFileType.AddOrSetValue('kpt', 'application/vnd.kde.kpresenter');
  FFileType.AddOrSetValue('ksp', 'application/vnd.kde.kspread');
  FFileType.AddOrSetValue('kwd', 'application/vnd.kde.kword');
  FFileType.AddOrSetValue('kwt', 'application/vnd.kde.kword');
  FFileType.AddOrSetValue('htke', 'application/vnd.kenameaapp');
  FFileType.AddOrSetValue('kia', 'application/vnd.kidspiration');
  FFileType.AddOrSetValue('kne', 'application/vnd.kinar');
  FFileType.AddOrSetValue('knp', 'application/vnd.kinar');
  FFileType.AddOrSetValue('skp', 'application/vnd.koan');
  FFileType.AddOrSetValue('skd', 'application/vnd.koan');
  FFileType.AddOrSetValue('skt', 'application/vnd.koan');
  FFileType.AddOrSetValue('skm', 'application/vnd.koan');
  FFileType.AddOrSetValue('sse', 'application/vnd.kodak-descriptor');
  FFileType.AddOrSetValue('lasxml', 'application/vnd.las.las+xml');
  FFileType.AddOrSetValue('lbd', 'application/vnd.llamagraphics.life-balance.desktop');
  FFileType.AddOrSetValue('lbe', 'application/vnd.llamagraphics.life-balance.exchange+xml');
  FFileType.AddOrSetValue('123', 'application/vnd.lotus-1-2-3');
  FFileType.AddOrSetValue('apr', 'application/vnd.lotus-approach');
  FFileType.AddOrSetValue('pre', 'application/vnd.lotus-freelance');
  FFileType.AddOrSetValue('nsf', 'application/vnd.lotus-notes');
  FFileType.AddOrSetValue('org', 'application/vnd.lotus-organizer');
  FFileType.AddOrSetValue('scm', 'application/vnd.lotus-screencam');
  FFileType.AddOrSetValue('lwp', 'application/vnd.lotus-wordpro');
  FFileType.AddOrSetValue('portpkg', 'application/vnd.macports.portpkg');
  FFileType.AddOrSetValue('mcd', 'application/vnd.mcd');
  FFileType.AddOrSetValue('mc1', 'application/vnd.medcalcdata');
  FFileType.AddOrSetValue('cdkey', 'application/vnd.mediastation.cdkey');
  FFileType.AddOrSetValue('mwf', 'application/vnd.mfer');
  FFileType.AddOrSetValue('mfm', 'application/vnd.mfmp');
  FFileType.AddOrSetValue('flo', 'application/vnd.micrografx.flo');
  FFileType.AddOrSetValue('igx', 'application/vnd.micrografx.igx');
  FFileType.AddOrSetValue('mif', 'application/vnd.mif');
  FFileType.AddOrSetValue('daf', 'application/vnd.mobius.daf');
  FFileType.AddOrSetValue('dis', 'application/vnd.mobius.dis');
  FFileType.AddOrSetValue('mbk', 'application/vnd.mobius.mbk');
  FFileType.AddOrSetValue('mqy', 'application/vnd.mobius.mqy');
  FFileType.AddOrSetValue('msl', 'application/vnd.mobius.msl');
  FFileType.AddOrSetValue('plc', 'application/vnd.mobius.plc');
  FFileType.AddOrSetValue('txf', 'application/vnd.mobius.txf');
  FFileType.AddOrSetValue('mpn', 'application/vnd.mophun.application');
  FFileType.AddOrSetValue('mpc', 'application/vnd.mophun.certificate');
  FFileType.AddOrSetValue('xul', 'application/vnd.mozilla.xul+xml');
  FFileType.AddOrSetValue('cil', 'application/vnd.ms-artgalry');
  FFileType.AddOrSetValue('cab', 'application/vnd.ms-cab-compressed');
  FFileType.AddOrSetValue('xls', 'application/vnd.ms-excel');
  FFileType.AddOrSetValue('xlm', 'application/vnd.ms-excel');
  FFileType.AddOrSetValue('xla', 'application/vnd.ms-excel');
  FFileType.AddOrSetValue('xlc', 'application/vnd.ms-excel');
  FFileType.AddOrSetValue('xlt', 'application/vnd.ms-excel');
  FFileType.AddOrSetValue('xlw', 'application/vnd.ms-excel');
  FFileType.AddOrSetValue('xlam', 'application/vnd.ms-excel.addin.macroenabled.12');
  FFileType.AddOrSetValue('xlsb', 'application/vnd.ms-excel.sheet.binary.macroenabled.12');
  FFileType.AddOrSetValue('xlsm', 'application/vnd.ms-excel.sheet.macroenabled.12');
  FFileType.AddOrSetValue('xltm', 'application/vnd.ms-excel.template.macroenabled.12');
  FFileType.AddOrSetValue('eot', 'application/vnd.ms-fontobject');
  FFileType.AddOrSetValue('chm', 'application/vnd.ms-htmlhelp');
  FFileType.AddOrSetValue('ims', 'application/vnd.ms-ims');
  FFileType.AddOrSetValue('lrm', 'application/vnd.ms-lrm');
  FFileType.AddOrSetValue('thmx', 'application/vnd.ms-officetheme');
  FFileType.AddOrSetValue('cat', 'application/vnd.ms-pki.seccat');
  FFileType.AddOrSetValue('stl', 'application/vnd.ms-pki.stl');
  FFileType.AddOrSetValue('ppt', 'application/vnd.ms-powerpoint');
  FFileType.AddOrSetValue('pps', 'application/vnd.ms-powerpoint');
  FFileType.AddOrSetValue('pot', 'application/vnd.ms-powerpoint');
  FFileType.AddOrSetValue('ppam', 'application/vnd.ms-powerpoint.addin.macroenabled.12');
  FFileType.AddOrSetValue('pptm', 'application/vnd.ms-powerpoint.presentation.macroenabled.12');
  FFileType.AddOrSetValue('sldm', 'application/vnd.ms-powerpoint.slide.macroenabled.12');
  FFileType.AddOrSetValue('ppsm', 'application/vnd.ms-powerpoint.slideshow.macroenabled.12');
  FFileType.AddOrSetValue('potm', 'application/vnd.ms-powerpoint.template.macroenabled.12');
  FFileType.AddOrSetValue('mpp', 'application/vnd.ms-project');
  FFileType.AddOrSetValue('mpt', 'application/vnd.ms-project');
  FFileType.AddOrSetValue('docm', 'application/vnd.ms-word.document.macroenabled.12');
  FFileType.AddOrSetValue('dotm', 'application/vnd.ms-word.template.macroenabled.12');
  FFileType.AddOrSetValue('wps', 'application/vnd.ms-works');
  FFileType.AddOrSetValue('wks', 'application/vnd.ms-works');
  FFileType.AddOrSetValue('wcm', 'application/vnd.ms-works');
  FFileType.AddOrSetValue('wdb', 'application/vnd.ms-works');
  FFileType.AddOrSetValue('wpl', 'application/vnd.ms-wpl');
  FFileType.AddOrSetValue('xps', 'application/vnd.ms-xpsdocument');
  FFileType.AddOrSetValue('mseq', 'application/vnd.mseq');
  FFileType.AddOrSetValue('mus', 'application/vnd.musician');
  FFileType.AddOrSetValue('msty', 'application/vnd.muvee.style');
  FFileType.AddOrSetValue('taglet', 'application/vnd.mynfc');
  FFileType.AddOrSetValue('nlu', 'application/vnd.neurolanguage.nlu');
  FFileType.AddOrSetValue('ntf', 'application/vnd.nitf');
  FFileType.AddOrSetValue('nitf', 'application/vnd.nitf');
  FFileType.AddOrSetValue('nnd', 'application/vnd.noblenet-directory');
  FFileType.AddOrSetValue('nns', 'application/vnd.noblenet-sealer');
  FFileType.AddOrSetValue('nnw', 'application/vnd.noblenet-web');
  FFileType.AddOrSetValue('ngdat', 'application/vnd.nokia.n-gage.data');
  FFileType.AddOrSetValue('n-gage', 'application/vnd.nokia.n-gage.symbian.install');
  FFileType.AddOrSetValue('rpst', 'application/vnd.nokia.radio-preset');
  FFileType.AddOrSetValue('rpss', 'application/vnd.nokia.radio-presets');
  FFileType.AddOrSetValue('edm', 'application/vnd.novadigm.edm');
  FFileType.AddOrSetValue('edx', 'application/vnd.novadigm.edx');
  FFileType.AddOrSetValue('FExt', 'application/vnd.novadigm.FExt');
  FFileType.AddOrSetValue('odc', 'application/vnd.oasis.opendocument.chart');
  FFileType.AddOrSetValue('otc', 'application/vnd.oasis.opendocument.chart-template');
  FFileType.AddOrSetValue('odb', 'application/vnd.oasis.opendocument.database');
  FFileType.AddOrSetValue('odf', 'application/vnd.oasis.opendocument.formula');
  FFileType.AddOrSetValue('odft', 'application/vnd.oasis.opendocument.formula-template');
  FFileType.AddOrSetValue('odg', 'application/vnd.oasis.opendocument.graphics');
  FFileType.AddOrSetValue('otg', 'application/vnd.oasis.opendocument.graphics-template');
  FFileType.AddOrSetValue('odi', 'application/vnd.oasis.opendocument.image');
  FFileType.AddOrSetValue('oti', 'application/vnd.oasis.opendocument.image-template');
  FFileType.AddOrSetValue('odp', 'application/vnd.oasis.opendocument.presentation');
  FFileType.AddOrSetValue('otp', 'application/vnd.oasis.opendocument.presentation-template');
  FFileType.AddOrSetValue('ods', 'application/vnd.oasis.opendocument.spreadsheet');
  FFileType.AddOrSetValue('ots', 'application/vnd.oasis.opendocument.spreadsheet-template');
  FFileType.AddOrSetValue('odt', 'application/vnd.oasis.opendocument.text');
  FFileType.AddOrSetValue('odm', 'application/vnd.oasis.opendocument.text-master');
  FFileType.AddOrSetValue('ott', 'application/vnd.oasis.opendocument.text-template');
  FFileType.AddOrSetValue('oth', 'application/vnd.oasis.opendocument.text-web');
  FFileType.AddOrSetValue('xo', 'application/vnd.olpc-sugar');
  FFileType.AddOrSetValue('dd2', 'application/vnd.oma.dd2+xml');
  FFileType.AddOrSetValue('oxt', 'application/vnd.openofficeorg.extension');
  FFileType.AddOrSetValue('pptx', 'application/vnd.openxmlformats-officedocument.presentationml.presentation');
  FFileType.AddOrSetValue('sldx', 'application/vnd.openxmlformats-officedocument.presentationml.slide');
  FFileType.AddOrSetValue('ppsx', 'application/vnd.openxmlformats-officedocument.presentationml.slideshow');
  FFileType.AddOrSetValue('potx', 'application/vnd.openxmlformats-officedocument.presentationml.template');
  FFileType.AddOrSetValue('xlsx', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
  FFileType.AddOrSetValue('xltx', 'application/vnd.openxmlformats-officedocument.spreadsheetml.template');
  FFileType.AddOrSetValue('docx', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document');
  FFileType.AddOrSetValue('dotx', 'application/vnd.openxmlformats-officedocument.wordprocessingml.template');
  FFileType.AddOrSetValue('mgp', 'application/vnd.osgeo.mapguide.package');
  FFileType.AddOrSetValue('dp', 'application/vnd.osgi.dp');
  FFileType.AddOrSetValue('esa', 'application/vnd.osgi.subsystem');
  FFileType.AddOrSetValue('pdb', 'application/vnd.palm');
  FFileType.AddOrSetValue('pqa', 'application/vnd.palm');
  FFileType.AddOrSetValue('oprc', 'application/vnd.palm');
  FFileType.AddOrSetValue('paw', 'application/vnd.pawaafile');
  FFileType.AddOrSetValue('str', 'application/vnd.pg.format');
  FFileType.AddOrSetValue('ei6', 'application/vnd.pg.osasli');
  FFileType.AddOrSetValue('efif', 'application/vnd.picsel');
  FFileType.AddOrSetValue('wg', 'application/vnd.pmi.widget');
  FFileType.AddOrSetValue('plf', 'application/vnd.pocketlearn');
  FFileType.AddOrSetValue('pbd', 'application/vnd.powerbuilder6');
  FFileType.AddOrSetValue('box', 'application/vnd.previewsystems.box');
  FFileType.AddOrSetValue('mgz', 'application/vnd.proteus.magazine');
  FFileType.AddOrSetValue('qps', 'application/vnd.publishare-delta-tree');
  FFileType.AddOrSetValue('ptid', 'application/vnd.pvi.ptid1');
  FFileType.AddOrSetValue('qxd', 'application/vnd.quark.quarkxpress');
  FFileType.AddOrSetValue('qxt', 'application/vnd.quark.quarkxpress');
  FFileType.AddOrSetValue('qwd', 'application/vnd.quark.quarkxpress');
  FFileType.AddOrSetValue('qwt', 'application/vnd.quark.quarkxpress');
  FFileType.AddOrSetValue('qxl', 'application/vnd.quark.quarkxpress');
  FFileType.AddOrSetValue('qxb', 'application/vnd.quark.quarkxpress');
  FFileType.AddOrSetValue('bed', 'application/vnd.realvnc.bed');
  FFileType.AddOrSetValue('mxl', 'application/vnd.recordare.musicxml');
  FFileType.AddOrSetValue('musicxml', 'application/vnd.recordare.musicxml+xml');
  FFileType.AddOrSetValue('cryptonote', 'application/vnd.rig.cryptonote');
  FFileType.AddOrSetValue('cod', 'application/vnd.rim.cod');
  FFileType.AddOrSetValue('rm', 'application/vnd.rn-realmedia');
  FFileType.AddOrSetValue('rmvb', 'application/vnd.rn-realmedia-vbr');
  FFileType.AddOrSetValue('link66', 'application/vnd.route66.link66+xml');
  FFileType.AddOrSetValue('st', 'application/vnd.sailingtracker.track');
  FFileType.AddOrSetValue('see', 'application/vnd.seemail');
  FFileType.AddOrSetValue('sema', 'application/vnd.sema');
  FFileType.AddOrSetValue('semd', 'application/vnd.semd');
  FFileType.AddOrSetValue('semf', 'application/vnd.semf');
  FFileType.AddOrSetValue('ifm', 'application/vnd.shana.informed.formdata');
  FFileType.AddOrSetValue('itp', 'application/vnd.shana.informed.formtemplate');
  FFileType.AddOrSetValue('iif', 'application/vnd.shana.informed.interchange');
  FFileType.AddOrSetValue('ipk', 'application/vnd.shana.informed.package');
  FFileType.AddOrSetValue('twd', 'application/vnd.simtech-mindmapper');
  FFileType.AddOrSetValue('twds', 'application/vnd.simtech-mindmapper');
  FFileType.AddOrSetValue('mmf', 'application/vnd.smaf');
  FFileType.AddOrSetValue('teacher', 'application/vnd.smart.teacher');
  FFileType.AddOrSetValue('sdkm', 'application/vnd.solent.sdkm+xml');
  FFileType.AddOrSetValue('sdkd', 'application/vnd.solent.sdkm+xml');
  FFileType.AddOrSetValue('dxp', 'application/vnd.spotfire.dxp');
  FFileType.AddOrSetValue('sfs', 'application/vnd.spotfire.sfs');
  FFileType.AddOrSetValue('sdc', 'application/vnd.stardivision.calc');
  FFileType.AddOrSetValue('sda', 'application/vnd.stardivision.draw');
  FFileType.AddOrSetValue('sdd', 'application/vnd.stardivision.impress');
  FFileType.AddOrSetValue('smf', 'application/vnd.stardivision.math');
  FFileType.AddOrSetValue('sdw', 'application/vnd.stardivision.writer');
  FFileType.AddOrSetValue('vor', 'application/vnd.stardivision.writer');
  FFileType.AddOrSetValue('sgl', 'application/vnd.stardivision.writer-global');
  FFileType.AddOrSetValue('smzip', 'application/vnd.stepmania.package');
  FFileType.AddOrSetValue('sm', 'application/vnd.stepmania.stepchart');
  FFileType.AddOrSetValue('sxc', 'application/vnd.sun.xml.calc');
  FFileType.AddOrSetValue('stc', 'application/vnd.sun.xml.calc.template');
  FFileType.AddOrSetValue('sxd', 'application/vnd.sun.xml.draw');
  FFileType.AddOrSetValue('std', 'application/vnd.sun.xml.draw.template');
  FFileType.AddOrSetValue('sxi', 'application/vnd.sun.xml.impress');
  FFileType.AddOrSetValue('sti', 'application/vnd.sun.xml.impress.template');
  FFileType.AddOrSetValue('sxm', 'application/vnd.sun.xml.math');
  FFileType.AddOrSetValue('sxw', 'application/vnd.sun.xml.writer');
  FFileType.AddOrSetValue('sxg', 'application/vnd.sun.xml.writer.global');
  FFileType.AddOrSetValue('stw', 'application/vnd.sun.xml.writer.template');
  FFileType.AddOrSetValue('sus', 'application/vnd.sus-calendar');
  FFileType.AddOrSetValue('susp', 'application/vnd.sus-calendar');
  FFileType.AddOrSetValue('svd', 'application/vnd.svd');
  FFileType.AddOrSetValue('sis', 'application/vnd.symbian.install');
  FFileType.AddOrSetValue('sisx', 'application/vnd.symbian.install');
  FFileType.AddOrSetValue('xsm', 'application/vnd.syncml+xml');
  FFileType.AddOrSetValue('bdm', 'application/vnd.syncml.dm+wbxml');
  FFileType.AddOrSetValue('xdm', 'application/vnd.syncml.dm+xml');
  FFileType.AddOrSetValue('tao', 'application/vnd.tao.intent-module-archive');
  FFileType.AddOrSetValue('pcap', 'application/vnd.tcpdump.pcap');
  FFileType.AddOrSetValue('cap', 'application/vnd.tcpdump.pcap');
  FFileType.AddOrSetValue('dmp', 'application/vnd.tcpdump.pcap');
  FFileType.AddOrSetValue('tmo', 'application/vnd.tmobile-livetv');
  FFileType.AddOrSetValue('tpt', 'application/vnd.trid.tpt');
  FFileType.AddOrSetValue('mxs', 'application/vnd.triscape.mxs');
  FFileType.AddOrSetValue('tra', 'application/vnd.trueapp');
  FFileType.AddOrSetValue('ufd', 'application/vnd.ufdl');
  FFileType.AddOrSetValue('ufdl', 'application/vnd.ufdl');
  FFileType.AddOrSetValue('utz', 'application/vnd.uiq.theme');
  FFileType.AddOrSetValue('umj', 'application/vnd.umajin');
  FFileType.AddOrSetValue('unityweb', 'application/vnd.unity');
  FFileType.AddOrSetValue('uoml', 'application/vnd.uoml+xml');
  FFileType.AddOrSetValue('vcx', 'application/vnd.vcx');
  FFileType.AddOrSetValue('vsd', 'application/vnd.visio');
  FFileType.AddOrSetValue('vst', 'application/vnd.visio');
  FFileType.AddOrSetValue('vss', 'application/vnd.visio');
  FFileType.AddOrSetValue('vsw', 'application/vnd.visio');
  FFileType.AddOrSetValue('vis', 'application/vnd.visionary');
  FFileType.AddOrSetValue('vsf', 'application/vnd.vsf');
  FFileType.AddOrSetValue('wbxml', 'application/vnd.wap.wbxml');
  FFileType.AddOrSetValue('wmlc', 'application/vnd.wap.wmlc');
  FFileType.AddOrSetValue('wmlsc', 'application/vnd.wap.wmlscriptc');
  FFileType.AddOrSetValue('wtb', 'application/vnd.webturbo');
  FFileType.AddOrSetValue('nbp', 'application/vnd.wolfram.player');
  FFileType.AddOrSetValue('wpd', 'application/vnd.wordperfect');
  FFileType.AddOrSetValue('wqd', 'application/vnd.wqd');
  FFileType.AddOrSetValue('stf', 'application/vnd.wt.stf');
  FFileType.AddOrSetValue('xar', 'application/vnd.xara');
  FFileType.AddOrSetValue('xfdl', 'application/vnd.xfdl');
  FFileType.AddOrSetValue('hvd', 'application/vnd.yamaha.hv-dic');
  FFileType.AddOrSetValue('hvs', 'application/vnd.yamaha.hv-script');
  FFileType.AddOrSetValue('hvp', 'application/vnd.yamaha.hv-voice');
  FFileType.AddOrSetValue('osf', 'application/vnd.yamaha.openscoreformat');
  FFileType.AddOrSetValue('osfpvg', 'application/vnd.yamaha.openscoreformat.osfpvg+xml');
  FFileType.AddOrSetValue('saf', 'application/vnd.yamaha.smaf-audio');
  FFileType.AddOrSetValue('spf', 'application/vnd.yamaha.smaf-phrase');
  FFileType.AddOrSetValue('cmp', 'application/vnd.yellowriver-custom-menu');
  FFileType.AddOrSetValue('zir', 'application/vnd.zul');
  FFileType.AddOrSetValue('zirz', 'application/vnd.zul');
  FFileType.AddOrSetValue('zaz', 'application/vnd.zzazz.deck+xml');
  FFileType.AddOrSetValue('vxml', 'application/voicexml+xml');
  FFileType.AddOrSetValue('wgt', 'application/widget');
  FFileType.AddOrSetValue('hlp', 'application/winhlp');
  FFileType.AddOrSetValue('wsdl', 'application/wsdl+xml');
  FFileType.AddOrSetValue('wspolicy', 'application/wspolicy+xml');
  FFileType.AddOrSetValue('7z', 'application/x-7z-compressed');
  FFileType.AddOrSetValue('abw', 'application/x-abiword');
  FFileType.AddOrSetValue('ace', 'application/x-ace-compressed');
  FFileType.AddOrSetValue('dmg', 'application/x-apple-diskimage');
  FFileType.AddOrSetValue('aab', 'application/x-authorware-bin');
  FFileType.AddOrSetValue('x32', 'application/x-authorware-bin');
  FFileType.AddOrSetValue('u32', 'application/x-authorware-bin');
  FFileType.AddOrSetValue('vox', 'application/x-authorware-bin');
  FFileType.AddOrSetValue('aam', 'application/x-authorware-map');
  FFileType.AddOrSetValue('aas', 'application/x-authorware-seg');
  FFileType.AddOrSetValue('bcpio', 'application/x-bcpio');
  FFileType.AddOrSetValue('torrent', 'application/x-bittorrent');
  FFileType.AddOrSetValue('blb', 'application/x-blorb');
  FFileType.AddOrSetValue('blorb', 'application/x-blorb');
  FFileType.AddOrSetValue('bz', 'application/x-bzip');
  FFileType.AddOrSetValue('bz2', 'application/x-bzip2');
  FFileType.AddOrSetValue('boz', 'application/x-bzip2');
  FFileType.AddOrSetValue('cbr', 'application/x-cbr');
  FFileType.AddOrSetValue('cba', 'application/x-cbr');
  FFileType.AddOrSetValue('cbt', 'application/x-cbr');
  FFileType.AddOrSetValue('cbz', 'application/x-cbr');
  FFileType.AddOrSetValue('cb7', 'application/x-cbr');
  FFileType.AddOrSetValue('vcd', 'application/x-cdlink');
  FFileType.AddOrSetValue('cfs', 'application/x-cfs-compressed');
  FFileType.AddOrSetValue('chat', 'application/x-chat');
  FFileType.AddOrSetValue('pgn', 'application/x-chess-pgn');
  FFileType.AddOrSetValue('nsc', 'application/x-conference');
  FFileType.AddOrSetValue('cpio', 'application/x-cpio');
  FFileType.AddOrSetValue('csh', 'application/x-csh');
  FFileType.AddOrSetValue('deb', 'application/x-debian-package');
  FFileType.AddOrSetValue('udeb', 'application/x-debian-package');
  FFileType.AddOrSetValue('dgc', 'application/x-dgc-compressed');
  FFileType.AddOrSetValue('dir', 'application/x-director');
  FFileType.AddOrSetValue('dcr', 'application/x-director');
  FFileType.AddOrSetValue('dxr', 'application/x-director');
  FFileType.AddOrSetValue('cst', 'application/x-director');
  FFileType.AddOrSetValue('cct', 'application/x-director');
  FFileType.AddOrSetValue('cxt', 'application/x-director');
  FFileType.AddOrSetValue('w3d', 'application/x-director');
  FFileType.AddOrSetValue('fgd', 'application/x-director');
  FFileType.AddOrSetValue('swa', 'application/x-director');
  FFileType.AddOrSetValue('wad', 'application/x-doom');
  FFileType.AddOrSetValue('ncx', 'application/x-dtbncx+xml');
  FFileType.AddOrSetValue('dtb', 'application/x-dtbook+xml');
  FFileType.AddOrSetValue('res', 'application/x-dtbresource+xml');
  FFileType.AddOrSetValue('dvi', 'application/x-dvi');
  FFileType.AddOrSetValue('evy', 'application/x-envoy');
  FFileType.AddOrSetValue('eva', 'application/x-eva');
  FFileType.AddOrSetValue('bdf', 'application/x-font-bdf');
  FFileType.AddOrSetValue('gsf', 'application/x-font-ghostscript');
  FFileType.AddOrSetValue('psf', 'application/x-font-linux-psf');
  FFileType.AddOrSetValue('otf', 'application/x-font-otf');
  FFileType.AddOrSetValue('pcf', 'application/x-font-pcf');
  FFileType.AddOrSetValue('snf', 'application/x-font-snf');
  FFileType.AddOrSetValue('ttf', 'application/x-font-ttf');
  FFileType.AddOrSetValue('ttc', 'application/x-font-ttf');
  FFileType.AddOrSetValue('pfa', 'application/x-font-type1');
  FFileType.AddOrSetValue('pfb', 'application/x-font-type1');
  FFileType.AddOrSetValue('pfm', 'application/x-font-type1');
  FFileType.AddOrSetValue('afm', 'application/x-font-type1');
  FFileType.AddOrSetValue('woff', 'application/x-font-woff');
  FFileType.AddOrSetValue('arc', 'application/x-freearc');
  FFileType.AddOrSetValue('spl', 'application/x-futuresplash');
  FFileType.AddOrSetValue('gca', 'application/x-gca-compressed');
  FFileType.AddOrSetValue('ulx', 'application/x-glulx');
  FFileType.AddOrSetValue('gnumeric', 'application/x-gnumeric');
  FFileType.AddOrSetValue('gramps', 'application/x-gramps-xml');
  FFileType.AddOrSetValue('gtar', 'application/x-gtar');
  FFileType.AddOrSetValue('hdf', 'application/x-hdf');
  FFileType.AddOrSetValue('install', 'application/x-install-instructions');
  FFileType.AddOrSetValue('iso', 'application/x-iso9660-image');
  FFileType.AddOrSetValue('jnlp', 'application/x-java-jnlp-file');
  FFileType.AddOrSetValue('latex', 'application/x-latex');
  FFileType.AddOrSetValue('lzh', 'application/x-lzh-compressed');
  FFileType.AddOrSetValue('lha', 'application/x-lzh-compressed');
  FFileType.AddOrSetValue('mie', 'application/x-mie');
  FFileType.AddOrSetValue('prc', 'application/x-mobipocket-ebook');
  FFileType.AddOrSetValue('mobi', 'application/x-mobipocket-ebook');
  FFileType.AddOrSetValue('application', 'application/x-ms-application');
  FFileType.AddOrSetValue('lnk', 'application/x-ms-shortcut');
  FFileType.AddOrSetValue('wmd', 'application/x-ms-wmd');
  FFileType.AddOrSetValue('wmz', 'application/x-ms-wmz');
  FFileType.AddOrSetValue('xbap', 'application/x-ms-xbap');
  FFileType.AddOrSetValue('mdb', 'application/x-msaccess');
  FFileType.AddOrSetValue('obd', 'application/x-msbinder');
  FFileType.AddOrSetValue('crd', 'application/x-mscardfile');
  FFileType.AddOrSetValue('clp', 'application/x-msclip');
  FFileType.AddOrSetValue('exe', 'application/x-msdownload');
  FFileType.AddOrSetValue('dll', 'application/x-msdownload');
  FFileType.AddOrSetValue('com', 'application/x-msdownload');
  FFileType.AddOrSetValue('bat', 'application/x-msdownload');
  FFileType.AddOrSetValue('msi', 'application/x-msdownload');
  FFileType.AddOrSetValue('mvb', 'application/x-msmediaview');
  FFileType.AddOrSetValue('m13', 'application/x-msmediaview');
  FFileType.AddOrSetValue('m14', 'application/x-msmediaview');
  FFileType.AddOrSetValue('wmf', 'application/x-msmetafile');
  FFileType.AddOrSetValue('wmz', 'application/x-msmetafile');
  FFileType.AddOrSetValue('emf', 'application/x-msmetafile');
  FFileType.AddOrSetValue('emz', 'application/x-msmetafile');
  FFileType.AddOrSetValue('mny', 'application/x-msmoney');
  FFileType.AddOrSetValue('pub', 'application/x-mspublisher');
  FFileType.AddOrSetValue('scd', 'application/x-msschedule');
  FFileType.AddOrSetValue('trm', 'application/x-msterminal');
  FFileType.AddOrSetValue('wri', 'application/x-mswrite');
  FFileType.AddOrSetValue('nc', 'application/x-netcdf');
  FFileType.AddOrSetValue('cdf', 'application/x-netcdf');
  FFileType.AddOrSetValue('nzb', 'application/x-nzb');
  FFileType.AddOrSetValue('p12', 'application/x-pkcs12');
  FFileType.AddOrSetValue('pfx', 'application/x-pkcs12');
  FFileType.AddOrSetValue('p7b', 'application/x-pkcs7-certificates');
  FFileType.AddOrSetValue('spc', 'application/x-pkcs7-certificates');
  FFileType.AddOrSetValue('p7r', 'application/x-pkcs7-certreqresp');
  FFileType.AddOrSetValue('rar', 'application/x-rar-compressed');
  FFileType.AddOrSetValue('ris', 'application/x-research-info-systems');
  FFileType.AddOrSetValue('sh', 'application/x-sh');
  FFileType.AddOrSetValue('shar', 'application/x-shar');
  FFileType.AddOrSetValue('swf', 'application/x-shockwave-flash');
  FFileType.AddOrSetValue('xap', 'application/x-silverlight-app');
  FFileType.AddOrSetValue('sql', 'application/x-sql');
  FFileType.AddOrSetValue('sit', 'application/x-stuffit');
  FFileType.AddOrSetValue('sitx', 'application/x-stuffitx');
  FFileType.AddOrSetValue('srt', 'application/x-subrip');
  FFileType.AddOrSetValue('sv4cpio', 'application/x-sv4cpio');
  FFileType.AddOrSetValue('sv4crc', 'application/x-sv4crc');
  FFileType.AddOrSetValue('t3', 'application/x-t3vm-image');
  FFileType.AddOrSetValue('gam', 'application/x-tads');
  FFileType.AddOrSetValue('tar', 'application/x-tar');
  FFileType.AddOrSetValue('tcl', 'application/x-tcl');
  FFileType.AddOrSetValue('tex', 'application/x-tex');
  FFileType.AddOrSetValue('tfm', 'application/x-tex-tfm');
  FFileType.AddOrSetValue('texinfo', 'application/x-texinfo');
  FFileType.AddOrSetValue('texi', 'application/x-texinfo');
  FFileType.AddOrSetValue('obj', 'application/x-tgif');
  FFileType.AddOrSetValue('ustar', 'application/x-ustar');
  FFileType.AddOrSetValue('src', 'application/x-wais-source');
  FFileType.AddOrSetValue('der', 'application/x-x509-ca-cert');
  FFileType.AddOrSetValue('crt', 'application/x-x509-ca-cert');
  FFileType.AddOrSetValue('fig', 'application/x-xfig');
  FFileType.AddOrSetValue('xlf', 'application/x-xliff+xml');
  FFileType.AddOrSetValue('xpi', 'application/x-xpinstall');
  FFileType.AddOrSetValue('xz', 'application/x-xz');
  FFileType.AddOrSetValue('yaml', 'application/x-yaml');
  FFileType.AddOrSetValue('z1', 'application/x-zmachine');
  FFileType.AddOrSetValue('z2', 'application/x-zmachine');
  FFileType.AddOrSetValue('z3', 'application/x-zmachine');
  FFileType.AddOrSetValue('z4', 'application/x-zmachine');
  FFileType.AddOrSetValue('z5', 'application/x-zmachine');
  FFileType.AddOrSetValue('z6', 'application/x-zmachine');
  FFileType.AddOrSetValue('z7', 'application/x-zmachine');
  FFileType.AddOrSetValue('z8', 'application/x-zmachine');
  FFileType.AddOrSetValue('xaml', 'application/xaml+xml');
  FFileType.AddOrSetValue('xdf', 'application/xcap-diff+xml');
  FFileType.AddOrSetValue('xenc', 'application/xenc+xml');
  FFileType.AddOrSetValue('xhtml', 'application/xhtml+xml');
  FFileType.AddOrSetValue('xht', 'application/xhtml+xml');
  FFileType.AddOrSetValue('xml', 'application/xml');
  FFileType.AddOrSetValue('xsl', 'application/xml');
  FFileType.AddOrSetValue('dtd', 'application/xml-dtd');
  FFileType.AddOrSetValue('xop', 'application/xop+xml');
  FFileType.AddOrSetValue('xpl', 'application/xproc+xml');
  FFileType.AddOrSetValue('xslt', 'application/xslt+xml');
  FFileType.AddOrSetValue('xspf', 'application/xspf+xml');
  FFileType.AddOrSetValue('mxml', 'application/xv+xml');
  FFileType.AddOrSetValue('xhvml', 'application/xv+xml');
  FFileType.AddOrSetValue('xvml', 'application/xv+xml');
  FFileType.AddOrSetValue('xvm', 'application/xv+xml');
  FFileType.AddOrSetValue('yang', 'application/yang');
  FFileType.AddOrSetValue('yin', 'application/yin+xml');
  FFileType.AddOrSetValue('zip', 'application/zip');
  FFileType.AddOrSetValue('adp', 'audio/adpcm');
  FFileType.AddOrSetValue('au', 'audio/basic');
  FFileType.AddOrSetValue('snd', 'audio/basic');
  FFileType.AddOrSetValue('mid', 'audio/midi');
  FFileType.AddOrSetValue('midi', 'audio/midi');
  FFileType.AddOrSetValue('kar', 'audio/midi');
  FFileType.AddOrSetValue('rmi', 'audio/midi');
  FFileType.AddOrSetValue('mp4a', 'audio/mp4');
  FFileType.AddOrSetValue('mpga', 'audio/mpeg');
  FFileType.AddOrSetValue('mp2', 'audio/mpeg');
  FFileType.AddOrSetValue('mp2a', 'audio/mpeg');
  FFileType.AddOrSetValue('mp3', 'audio/mpeg');
  FFileType.AddOrSetValue('m2a', 'audio/mpeg');
  FFileType.AddOrSetValue('m3a', 'audio/mpeg');
  FFileType.AddOrSetValue('oga', 'audio/ogg');
  FFileType.AddOrSetValue('ogg', 'audio/ogg');
  FFileType.AddOrSetValue('spx', 'audio/ogg');
  FFileType.AddOrSetValue('s3m', 'audio/s3m');
  FFileType.AddOrSetValue('sil', 'audio/silk');
  FFileType.AddOrSetValue('uva', 'audio/vnd.dece.audio');
  FFileType.AddOrSetValue('uvva', 'audio/vnd.dece.audio');
  FFileType.AddOrSetValue('eol', 'audio/vnd.digital-winds');
  FFileType.AddOrSetValue('dra', 'audio/vnd.dra');
  FFileType.AddOrSetValue('dts', 'audio/vnd.dts');
  FFileType.AddOrSetValue('dtshd', 'audio/vnd.dts.hd');
  FFileType.AddOrSetValue('lvp', 'audio/vnd.lucent.voice');
  FFileType.AddOrSetValue('pya', 'audio/vnd.ms-playready.media.pya');
  FFileType.AddOrSetValue('ecelp4800', 'audio/vnd.nuera.ecelp4800');
  FFileType.AddOrSetValue('ecelp7470', 'audio/vnd.nuera.ecelp7470');
  FFileType.AddOrSetValue('ecelp9600', 'audio/vnd.nuera.ecelp9600');
  FFileType.AddOrSetValue('rip', 'audio/vnd.rip');
  FFileType.AddOrSetValue('weba', 'audio/webm');
  FFileType.AddOrSetValue('aac', 'audio/x-aac');
  FFileType.AddOrSetValue('aif', 'audio/x-aiff');
  FFileType.AddOrSetValue('aiff', 'audio/x-aiff');
  FFileType.AddOrSetValue('aifc', 'audio/x-aiff');
  FFileType.AddOrSetValue('caf', 'audio/x-caf');
  FFileType.AddOrSetValue('flac', 'audio/x-flac');
  FFileType.AddOrSetValue('mka', 'audio/x-matroska');
  FFileType.AddOrSetValue('m3u', 'audio/x-mpegurl');
  FFileType.AddOrSetValue('wax', 'audio/x-ms-wax');
  FFileType.AddOrSetValue('wma', 'audio/x-ms-wma');
  FFileType.AddOrSetValue('ram', 'audio/x-pn-realaudio');
  FFileType.AddOrSetValue('ra', 'audio/x-pn-realaudio');
  FFileType.AddOrSetValue('rmp', 'audio/x-pn-realaudio-plugin');
  FFileType.AddOrSetValue('wav', 'audio/x-wav');
  FFileType.AddOrSetValue('xm', 'audio/xm');
  FFileType.AddOrSetValue('cdx', 'chemical/x-cdx');
  FFileType.AddOrSetValue('cif', 'chemical/x-cif');
  FFileType.AddOrSetValue('cmdf', 'chemical/x-cmdf');
  FFileType.AddOrSetValue('cml', 'chemical/x-cml');
  FFileType.AddOrSetValue('csml', 'chemical/x-csml');
  FFileType.AddOrSetValue('xyz', 'chemical/x-xyz');
  FFileType.AddOrSetValue('bmp', 'image/bmp');
  FFileType.AddOrSetValue('cgm', 'image/cgm');
  FFileType.AddOrSetValue('g3', 'image/g3fax');
  FFileType.AddOrSetValue('gif', 'image/gif');
  FFileType.AddOrSetValue('ief', 'image/ief');
  FFileType.AddOrSetValue('jpeg', 'image/jpeg');
  FFileType.AddOrSetValue('jpg', 'image/jpeg');
  FFileType.AddOrSetValue('jpe', 'image/jpeg');
  FFileType.AddOrSetValue('ktx', 'image/ktx');
  FFileType.AddOrSetValue('png', 'image/png');
  FFileType.AddOrSetValue('btif', 'image/prs.btif');
  FFileType.AddOrSetValue('sgi', 'image/sgi');
  FFileType.AddOrSetValue('svg', 'image/svg+xml');
  FFileType.AddOrSetValue('svgz', 'image/svg+xml');
  FFileType.AddOrSetValue('tiff', 'image/tiff');
  FFileType.AddOrSetValue('tif', 'image/tiff');
  FFileType.AddOrSetValue('psd', 'image/vnd.adobe.photoshop');
  FFileType.AddOrSetValue('uvi', 'image/vnd.dece.graphic');
  FFileType.AddOrSetValue('uvvi', 'image/vnd.dece.graphic');
  FFileType.AddOrSetValue('uvg', 'image/vnd.dece.graphic');
  FFileType.AddOrSetValue('uvvg', 'image/vnd.dece.graphic');
  FFileType.AddOrSetValue('sub', 'image/vnd.dvb.subtitle');
  FFileType.AddOrSetValue('djvu', 'image/vnd.djvu');
  FFileType.AddOrSetValue('djv', 'image/vnd.djvu');
  FFileType.AddOrSetValue('dwg', 'image/vnd.dwg');
  FFileType.AddOrSetValue('dxf', 'image/vnd.dxf');
  FFileType.AddOrSetValue('fbs', 'image/vnd.fastbidsheet');
  FFileType.AddOrSetValue('fpx', 'image/vnd.fpx');
  FFileType.AddOrSetValue('fst', 'image/vnd.fst');
  FFileType.AddOrSetValue('mmr', 'image/vnd.fujixerox.edmics-mmr');
  FFileType.AddOrSetValue('rlc', 'image/vnd.fujixerox.edmics-rlc');
  FFileType.AddOrSetValue('mdi', 'image/vnd.ms-modi');
  FFileType.AddOrSetValue('wdp', 'image/vnd.ms-photo');
  FFileType.AddOrSetValue('npx', 'image/vnd.net-fpx');
  FFileType.AddOrSetValue('wbmp', 'image/vnd.wap.wbmp');
  FFileType.AddOrSetValue('xif', 'image/vnd.xiff');
  FFileType.AddOrSetValue('webp', 'image/webp');
  FFileType.AddOrSetValue('3ds', 'image/x-3ds');
  FFileType.AddOrSetValue('ras', 'image/x-cmu-raster');
  FFileType.AddOrSetValue('cmx', 'image/x-cmx');
  FFileType.AddOrSetValue('fh', 'image/x-freehand');
  FFileType.AddOrSetValue('fhc', 'image/x-freehand');
  FFileType.AddOrSetValue('fh4', 'image/x-freehand');
  FFileType.AddOrSetValue('fh5', 'image/x-freehand');
  FFileType.AddOrSetValue('fh7', 'image/x-freehand');
  FFileType.AddOrSetValue('ico', 'image/x-icon');
  FFileType.AddOrSetValue('sid', 'image/x-mrsid-image');
  FFileType.AddOrSetValue('pcx', 'image/x-pcx');
  FFileType.AddOrSetValue('pic', 'image/x-pict');
  FFileType.AddOrSetValue('pct', 'image/x-pict');
  FFileType.AddOrSetValue('pnm', 'image/x-portable-anymap');
  FFileType.AddOrSetValue('pbm', 'image/x-portable-bitmap');
  FFileType.AddOrSetValue('pgm', 'image/x-portable-graymap');
  FFileType.AddOrSetValue('ppm', 'image/x-portable-pixmap');
  FFileType.AddOrSetValue('rgb', 'image/x-rgb');
  FFileType.AddOrSetValue('tga', 'image/x-tga');
  FFileType.AddOrSetValue('xbm', 'image/x-xbitmap');
  FFileType.AddOrSetValue('xpm', 'image/x-xpixmap');
  FFileType.AddOrSetValue('xwd', 'image/x-xwindowdump');
  FFileType.AddOrSetValue('eml', 'message/rfc822');
  FFileType.AddOrSetValue('mime', 'message/rfc822');
  FFileType.AddOrSetValue('igs', 'model/iges');
  FFileType.AddOrSetValue('iges', 'model/iges');
  FFileType.AddOrSetValue('msh', 'model/mesh');
  FFileType.AddOrSetValue('mesh', 'model/mesh');
  FFileType.AddOrSetValue('silo', 'model/mesh');
  FFileType.AddOrSetValue('dae', 'model/vnd.collada+xml');
  FFileType.AddOrSetValue('dwf', 'model/vnd.dwf');
  FFileType.AddOrSetValue('gdl', 'model/vnd.gdl');
  FFileType.AddOrSetValue('gtw', 'model/vnd.gtw');
  FFileType.AddOrSetValue('mts', 'model/vnd.mts');
  FFileType.AddOrSetValue('vtu', 'model/vnd.vtu');
  FFileType.AddOrSetValue('wrl', 'model/vrml');
  FFileType.AddOrSetValue('vrml', 'model/vrml');
  FFileType.AddOrSetValue('x3db', 'model/x3d+binary');
  FFileType.AddOrSetValue('x3dbz', 'model/x3d+binary');
  FFileType.AddOrSetValue('x3dv', 'model/x3d+vrml');
  FFileType.AddOrSetValue('x3dvz', 'model/x3d+vrml');
  FFileType.AddOrSetValue('x3d', 'model/x3d+xml');
  FFileType.AddOrSetValue('x3dz', 'model/x3d+xml');
  FFileType.AddOrSetValue('appcache', 'text/cache-manifest');
  FFileType.AddOrSetValue('manifest', 'text/cache-manifest');
  FFileType.AddOrSetValue('ics', 'text/calendar');
  FFileType.AddOrSetValue('ifb', 'text/calendar');
  FFileType.AddOrSetValue('cmd', 'text/cmd');
  FFileType.AddOrSetValue('css', 'text/css');
  FFileType.AddOrSetValue('csv', 'text/csv');
  FFileType.AddOrSetValue('html', 'text/html');
  FFileType.AddOrSetValue('htm', 'text/html');
  FFileType.AddOrSetValue('n3', 'text/n3');
  FFileType.AddOrSetValue('txt', 'text/plain');
  FFileType.AddOrSetValue('text', 'text/plain');
  FFileType.AddOrSetValue('conf', 'text/plain');
  FFileType.AddOrSetValue('def', 'text/plain');
  FFileType.AddOrSetValue('list', 'text/plain');
  FFileType.AddOrSetValue('log', 'text/plain');
  FFileType.AddOrSetValue('in', 'text/plain');
  FFileType.AddOrSetValue('dsc', 'text/prs.lines.tag');
  FFileType.AddOrSetValue('rtx', 'text/richtext');
  FFileType.AddOrSetValue('sgml', 'text/sgml');
  FFileType.AddOrSetValue('sgm', 'text/sgml');
  FFileType.AddOrSetValue('tsv', 'text/tab-separated-values');
  FFileType.AddOrSetValue('t', 'text/troff');
  FFileType.AddOrSetValue('tr', 'text/troff');
  FFileType.AddOrSetValue('roff', 'text/troff');
  FFileType.AddOrSetValue('man', 'text/troff');
  FFileType.AddOrSetValue('me', 'text/troff');
  FFileType.AddOrSetValue('ms', 'text/troff');
  FFileType.AddOrSetValue('ttl', 'text/turtle');
  FFileType.AddOrSetValue('uri', 'text/uri-list');
  FFileType.AddOrSetValue('uris', 'text/uri-list');
  FFileType.AddOrSetValue('urls', 'text/uri-list');
  FFileType.AddOrSetValue('vcard', 'text/vcard');
  FFileType.AddOrSetValue('curl', 'text/vnd.curl');
  FFileType.AddOrSetValue('dcurl', 'text/vnd.curl.dcurl');
  FFileType.AddOrSetValue('scurl', 'text/vnd.curl.scurl');
  FFileType.AddOrSetValue('mcurl', 'text/vnd.curl.mcurl');
  FFileType.AddOrSetValue('sub', 'text/vnd.dvb.subtitle');
  FFileType.AddOrSetValue('fly', 'text/vnd.fly');
  FFileType.AddOrSetValue('flx', 'text/vnd.fmi.flexstor');
  FFileType.AddOrSetValue('gv', 'text/vnd.graphviz');
  FFileType.AddOrSetValue('3dml', 'text/vnd.in3d.3dml');
  FFileType.AddOrSetValue('spot', 'text/vnd.in3d.spot');
  FFileType.AddOrSetValue('jad', 'text/vnd.sun.j2me.app-descriptor');
  FFileType.AddOrSetValue('wml', 'text/vnd.wap.wml');
  FFileType.AddOrSetValue('wmls', 'text/vnd.wap.wmlscript');
  FFileType.AddOrSetValue('s', 'text/x-asm');
  FFileType.AddOrSetValue('asm', 'text/x-asm');
  FFileType.AddOrSetValue('c', 'text/x-c');
  FFileType.AddOrSetValue('cc', 'text/x-c');
  FFileType.AddOrSetValue('cxx', 'text/x-c');
  FFileType.AddOrSetValue('cpp', 'text/x-c');
  FFileType.AddOrSetValue('h', 'text/x-c');
  FFileType.AddOrSetValue('hh', 'text/x-c');
  FFileType.AddOrSetValue('dic', 'text/x-c');
  FFileType.AddOrSetValue('f', 'text/x-fortran');
  FFileType.AddOrSetValue('for', 'text/x-fortran');
  FFileType.AddOrSetValue('f77', 'text/x-fortran');
  FFileType.AddOrSetValue('f90', 'text/x-fortran');
  FFileType.AddOrSetValue('java', 'text/x-java-source');
  FFileType.AddOrSetValue('opml', 'text/x-opml');
  FFileType.AddOrSetValue('p', 'text/x-pascal');
  FFileType.AddOrSetValue('pas', 'text/x-pascal');
  FFileType.AddOrSetValue('nfo', 'text/x-nfo');
  FFileType.AddOrSetValue('etx', 'text/x-setext');
  FFileType.AddOrSetValue('sfv', 'text/x-sfv');
  FFileType.AddOrSetValue('uu', 'text/x-uuencode');
  FFileType.AddOrSetValue('vcs', 'text/x-vcalendar');
  FFileType.AddOrSetValue('vcf', 'text/x-vcard');
  FFileType.AddOrSetValue('vcf', 'text/x-yaml');
  FFileType.AddOrSetValue('xml', 'text/xml');
  FFileType.AddOrSetValue('xsl', 'text/xml');
  FFileType.AddOrSetValue('dtd', 'text/xml-dtd');
  FFileType.AddOrSetValue('yaml', 'text/yaml');
  FFileType.AddOrSetValue('3gp', 'video/3gpp');
  FFileType.AddOrSetValue('3g2', 'video/3gpp2');
  FFileType.AddOrSetValue('h261', 'video/h261');
  FFileType.AddOrSetValue('h263', 'video/h263');
  FFileType.AddOrSetValue('h264', 'video/h264');
  FFileType.AddOrSetValue('jpgv', 'video/jpeg');
  FFileType.AddOrSetValue('jpm', 'video/jpm');
  FFileType.AddOrSetValue('jpgm', 'video/jpm');
  FFileType.AddOrSetValue('mj2', 'video/mj2');
  FFileType.AddOrSetValue('mjp2', 'video/mj2');
  FFileType.AddOrSetValue('mp4', 'video/mp4');
  FFileType.AddOrSetValue('mp4v', 'video/mp4');
  FFileType.AddOrSetValue('mpg4', 'video/mp4');
  FFileType.AddOrSetValue('mpeg', 'video/mpeg');
  FFileType.AddOrSetValue('mpg', 'video/mpeg');
  FFileType.AddOrSetValue('mpe', 'video/mpeg');
  FFileType.AddOrSetValue('m1v', 'video/mpeg');
  FFileType.AddOrSetValue('m2v', 'video/mpeg');
  FFileType.AddOrSetValue('ogv', 'video/ogg');
  FFileType.AddOrSetValue('qt', 'video/quicktime');
  FFileType.AddOrSetValue('mov', 'video/quicktime');
  FFileType.AddOrSetValue('uvh', 'video/vnd.dece.hd');
  FFileType.AddOrSetValue('uvvh', 'video/vnd.dece.hd');
  FFileType.AddOrSetValue('uvm', 'video/vnd.dece.mobile');
  FFileType.AddOrSetValue('uvvm', 'video/vnd.dece.mobile');
  FFileType.AddOrSetValue('uvp', 'video/vnd.dece.pd');
  FFileType.AddOrSetValue('uvvp', 'video/vnd.dece.pd');
  FFileType.AddOrSetValue('uvs', 'video/vnd.dece.sd');
  FFileType.AddOrSetValue('uvvs', 'video/vnd.dece.sd');
  FFileType.AddOrSetValue('uvv', 'video/vnd.dece.video');
  FFileType.AddOrSetValue('uvvv', 'video/vnd.dece.video');
  FFileType.AddOrSetValue('dvb', 'video/vnd.dvb.file');
  FFileType.AddOrSetValue('fvt', 'video/vnd.fvt');
  FFileType.AddOrSetValue('mxu', 'video/vnd.mpegurl');
  FFileType.AddOrSetValue('m4u', 'video/vnd.mpegurl');
  FFileType.AddOrSetValue('pyv', 'video/vnd.ms-playready.media.pyv');
  FFileType.AddOrSetValue('uvu', 'video/vnd.uvvu.mp4');
  FFileType.AddOrSetValue('uvvu', 'video/vnd.uvvu.mp4');
  FFileType.AddOrSetValue('viv', 'video/vnd.vivo');
  FFileType.AddOrSetValue('webm', 'video/webm');
  FFileType.AddOrSetValue('f4v', 'video/x-f4v');
  FFileType.AddOrSetValue('fli', 'video/x-fli');
  FFileType.AddOrSetValue('flv', 'video/x-flv');
  FFileType.AddOrSetValue('m4v', 'video/x-m4v');
  FFileType.AddOrSetValue('mkv', 'video/x-matroska');
  FFileType.AddOrSetValue('mk3d', 'video/x-matroska');
  FFileType.AddOrSetValue('mks', 'video/x-matroska');
  FFileType.AddOrSetValue('mng', 'video/x-mng');
  FFileType.AddOrSetValue('asf', 'video/x-ms-asf');
  FFileType.AddOrSetValue('asx', 'video/x-ms-asf');
  FFileType.AddOrSetValue('vob', 'video/x-ms-vob');
  FFileType.AddOrSetValue('wm', 'video/x-ms-wm');
  FFileType.AddOrSetValue('wmv', 'video/x-ms-wmv');
  FFileType.AddOrSetValue('wmx', 'video/x-ms-wmx');
  FFileType.AddOrSetValue('wvx', 'video/x-ms-wvx');
  FFileType.AddOrSetValue('avi', 'video/x-msvideo');
  FFileType.AddOrSetValue('movie', 'video/x-sgi-movie');
  FFileType.AddOrSetValue('smv', 'video/x-smv');
  FFileType.AddOrSetValue('ice', 'x-conference/x-cooltalk');
{$ENDREGION}
end;

function THorseCoreFile.Size: Int64;
begin
  result := ContentStream.Size;
end;

function THorseCoreFile.ContentStream: TStream;
begin
  if not Assigned(FFileStream) then
    FFileStream := TFileStream.Create(FFileName, fmOpenRead or fmShareDenyWrite);
  result := FFileStream;
end;

end.
