# Usage: cmake [[-D <var>=<value>] ...] -P run_tests.cmake -- [test1 ...]
# Run the given test cases and report a summary result.  Meant as an
# alternative to src/tests/run-tests for use with CMake builds.
# Variables with special meanings to this script:
# QUIET
#     If set to ON, only display a summary result and override the setting
#     for VERBOSE.  The default value is OFF.
# VERBOSE
#     If set to ON, causes test cases to be run with the -v flag to produce
#     verbose output, and all output from the test cases will be echoed by
#     this script.  If set to OFF, test cases will not be run with the -v flag,
#     and output from the test cases will not be echoed.  The default value
#     is ON if VERBOSE is set and non-empty in the environment.
# FORCE_PATH
#     If set to ON, casues test cases to be run with the -f flag (so that they
#     will use the same paths as the game's main.c would use even if alternate
#     paths have been configured for the test cases).  The default value is
#     ON if FORCE_PATH is set and non-empty in the environment.
# USECOLOR
#     If set to ON, use ANSI terminal color sequences to highlight some parts
#     of the output generated by this script.  The default value is OFF
#     when IF(WIN32) evaulates to true; otherwise the default value is ON.
# TOOL
#     If set, the test case will be run as ${TOOL} test_case_path
#     [arg_for_test_case ...].  To pass arguments to the tool itself, have
#     TOOL be a CMake-style list:  i.e. tool_path;tool_arg_1[;tool_arg_2...].

CMAKE_POLICY(VERSION 3.5)

# Set up defaults for the variables influenced by the environment or by
# what the user has set.
SET(_QUIET OFF)
SET(_VERBOSE OFF)
IF(DEFINED ENV{VERBOSE})
    STRING(LENGTH "$ENV{VERBOSE}" _RESLENGTH)
    IF(_RESLENGTH GREATER 0)
        SET(_VERBOSE ON)
    ENDIF()
ENDIF()
SET(_FORCE_PATH OFF)
IF(DEFINED ENV{FORCE_PATH})
    STRING(LENGTH "$ENV{FORCE_PATH}" _RESLENGTH)
    IF(_RESLENGTH GREATER 0)
        SET(_FORCE_PATH ON)
    ENDIF()
ENDIF()
IF(WIN32)
    SET(_USECOLOR OFF)
ELSE()
    SET(_USECOLOR ON)
ENDIF()

# Modify those variables by what the user has set.
IF(DEFINED QUIET)
    IF(QUIET)
        SET(_QUIET ON)
        SET(_VERBOSE OFF)
    ELSEIF(DEFINED VERBOSE)
        IF(VERBOSE)
            SET(_QUIET OFF)
            SET(_VERBOSE ON)
        ELSE()
            SET(_VERBOSE OFF)
        ENDIF()
    ENDIF()
ELSEIF(DEFINED VERBOSE)
    IF(VERBOSE)
        SET(_QUIET OFF)
        SET(_VERBOSE ON)
    ELSE()
        SET(_VERBOSE OFF)
    ENDIF()
ENDIF()
IF(DEFINED FORCE_PATH)
    IF(FORCE_PATH)
        SET(_FORCE_PATH ON)
    ELSE()
        SET(_FORCE_PATH OFF)
    ENDIF()
ENDIF()
IF(DEFINED USECOLOR)
    IF(USECOLOR)
        SET(_USECOLOR ON)
    ELSE()
        SET(_USECOLOR OFF)
    ENDIF()
ENDIF()

# Skip over the arguments to CMake.
SET(_X 0)
WHILE(_X LESS CMAKE_ARGC)
    IF(CMAKE_ARGV${_X} STREQUAL "--")
        BREAK()
    ENDIF()
    MATH(EXPR _X "${_X} + 1")
ENDWHILE()
IF(_X LESS CMAKE_ARGC)
    MATH(EXPR _STARTX "${_X} + 1")
ELSE()
    SET(_STARTX CMAKE_ARGC)
ENDIF()

# Remember the length of the longest name for a test suite excluding the
# initial path components that are common to all the test suites.
SET(_MAXLENGTH 0)
UNSET(_COMMON)
SET(_X ${_STARTX})
WHILE(_X LESS CMAKE_ARGC)
    STRING(LENGTH "${CMAKE_ARGV${_X}}" _THISLENGTH)
    IF(_MAXLENGTH LESS _THISLENGTH)
        SET(_MAXLENGTH ${_THISLENGTH})
    ENDIF()
    IF(DEFINED _COMMON)
        IF((NOT (_COMMON STREQUAL "")) AND (NOT ("${CMAKE_ARGV${_X}}" MATCHES "^${_COMMON}/")))
            SET(_LOOKING ON)
            WHILE(_LOOKING)
                STRING(FIND "${_COMMON}" "/" _SLASH REVERSE)
                IF(_SLASH EQUAL -1)
                    SET(_COMMON "")
                    SET(_LOOKING OFF)
                ELSE()
                    STRING(SUBSTRING "${_COMMON}" 0 ${_SLASH} _COMMON)
                    IF("${CMAKE_ARGV${_X}}" MATCHES "^${_COMMON}/")
                        SET(_LOOKING OFF)
                    ENDIF()
                ENDIF()
            ENDWHILE()
        ENDIF()
    ELSE()
        SET(_COMMON "${CMAKE_ARGV${_X}}")
    ENDIF()
    MATH(EXPR _X "${_X} + 1")
ENDWHILE()
IF(DEFINED _COMMON)
    STRING(LENGTH "${_COMMON}" _THISLENGTH)
    IF(_THISLENGTH LESS _MAXLENGTH)
        MATH(EXPR _MAXLENGTH "${_MAXLENGTH} - ${_THISLENGTH} - 1")
    ENDIF()
ENDIF()

# Store some strings for ANSI color.
STRING(ASCII 27 _ESC)
SET(_RED "${_ESC}[01;31m")
SET(_GREEN "${_ESC}[01;32m")
SET(_YELLOW "${_ESC}[01;33m")
SET(_RESET "${_ESC}[0m")

# Remember how long a newline is.
STRING(LENGTH "\n" _NLLENGTH)

# Run each test suite accumulating the number of test cases run and run with
# passing results.
IF(NOT _QUIET)
    MATH(EXPR _COUNT "${CMAKE_ARGC} - ${_STARTX}")
    MESSAGE("Running ${_COUNT} suites:")
ENDIF()
MATH(EXPR _DISLENGTH "${_MAXLENGTH} + 8")
SET(_PASS 0)
SET(_TOTAL 0)
SET(_EXITCODE 0)
IF(_VERBOSE)
    IF(_FORCE_PATH)
        SET(_TESTOPTS "-vf")
    ELSE()
        SET(_TESTOPTS "-v")
    ENDIF()
ELSEIF(_FORCE_PATH)
    SET(_TESTOPTS "-f")
ELSE()
    UNSET(_TESTOPTS)
ENDIF()
SET(_X ${_STARTX})
WHILE(_X LESS CMAKE_ARGC)
    # Use FILE(TO_NATIVE_PATH ...) rather than CMAKE_PATH(CONVERT ...) since
    # the latter requires at least CMake 3.20.
    FILE(TO_NATIVE_PATH "${CMAKE_ARGV${_X}}" _CONVERTED_PATH)
    IF(DEFINED TOOL)
        IF(DEFINED _TESTOPTS)
            EXECUTE_PROCESS(COMMAND ${TOOL} "${_CONVERTED_PATH}" ${_TESTOPTS}
                WORKING_DIRECTORY . RESULT_VARIABLE _RUN_RESULT
                OUTPUT_VARIABLE _RUN_OUTPUT ERROR_VARIABLE _RUN_OUTPUT)
        ELSE()
            EXECUTE_PROCESS(COMMAND ${TOOL} "${_CONVERTED_PATH}"
                WORKING_DIRECTORY . RESULT_VARIABLE _RUN_RESULT
                OUTPUT_VARIABLE _RUN_OUTPUT ERROR_VARIABLE _RUN_OUTPUT)
        ENDIF()
    ELSEIF(DEFINED _TESTOPTS)
        EXECUTE_PROCESS(COMMAND "${_CONVERTED_PATH}" ${_TESTOPTS}
            WORKING_DIRECTORY . RESULT_VARIABLE _RUN_RESULT
            OUTPUT_VARIABLE _RUN_OUTPUT ERROR_VARIABLE _RUN_OUTPUT)
    ELSE()
        EXECUTE_PROCESS(COMMAND "${_CONVERTED_PATH}" WORKING_DIRECTORY .
            RESULT_VARIABLE _RUN_RESULT OUTPUT_VARIABLE _RUN_OUTPUT
            ERROR_VARIABLE _RUN_OUTPUT)
    ENDIF()
    IF(NOT (_RUN_RESULT MATCHES "^[+-]?[0-9]+$"))
        IF(NOT (_RUN_RESULT STREQUAL ""))
            # Print out string result codes; record them as an exit code of
            # one in _EXITCODE.
            IF(_USECOLOR)
                MESSAGE("${_RED}${_CONVERTED_PATH}: Suite died: ${_RUN_RESULT}${_RESET}")
            ELSE()
                MESSAGE("${_CONVERTED_PATH}: Suite died: ${_RUN_RESULT}")
            ENDIF()
            SET(_EXITCODE 1)
            SET(_RUN_RESULT 1)
            # We do not know how many tests the suite had, but at least one
            # failed.
            MATH(EXPR _TOTAL "${_TOTAL} + 1")
        ELSE()
            SET(_RUN_RESULT 0)
        ENDIF()
    ELSEIF(_RUN_RESULT)
        # Non-zero integer exit codes get a generic message and are directly
        # recorded in _EXITCODE.
        IF(_USECOLOR)
            MESSAGE("${_RED}${_CONVERTED_PATH}: Suite died${_RESET}")
        ELSE()
            MESSAGE("${_CONVERTED_PATH}: Suite died")
        ENDIF()
        SET(_EXITCODE ${_RUN_RESULT})
        # We do not know how many tests the suite had, but at least one failed.
        MATH(EXPR _TOTAL "${_TOTAL} + 1")
    ENDIF()
    IF(NOT _RUN_RESULT)
        STRING(REGEX MATCHALL "[^:\n]+ finished: [0-9]+/[0-9]+ passed\r?\n" _SUMMARY_LINES ${_RUN_OUTPUT})
        LIST(LENGTH _SUMMARY_LINES _COUNT)
        IF(_COUNT)
            LIST(GET _SUMMARY_LINES -1 _LAST_LINE)
            STRING(REGEX REPLACE "[^:\n]+ finished: ([0-9]+)/[0-9]+ passed\r?\n" "\\1" _THISPASS ${_LAST_LINE})
            STRING(REGEX REPLACE "[^:\n]+ finished: [0-9]+/([0-9]+) passed\r?\n" "\\1" _THISTOTAL ${_LAST_LINE})
            # Tally the results
            MATH(EXPR _PASS "${_PASS} + ${_THISPASS}")
            MATH(EXPR _TOTAL "${_TOTAL} + ${_THISTOTAL}")
            IF(NOT _QUIET)
                IF(_VERBOSE)
                    # Echo the output from test case.
                    SET(_MATCHCOUNT 0)
                    STRING(LENGTH "${_RUN_OUTPUT}" _RESLENGTH)
                    WHILE(_RESLENGTH GREATER 0)
                        STRING(FIND "${_RUN_OUTPUT}" "\n" _ENDLINE)
                        IF(_ENDLINE EQUAL -1)
                            SET(_LINE "{_RUN_OUTPUT}")
                            SET(_RESLENGTH 0)
                        ELSE()
                            STRING(SUBSTRING "${_RUN_OUTPUT}" 0 ${_ENDLINE} _LINE)
                            MATH(EXPR _ENDLINE "${_ENDLINE} + ${_NLLENGTH}")
                            STRING(SUBSTRING "${_RUN_OUTPUT}" ${_ENDLINE} -1
                                _RUN_OUTPUT)
                            STRING(LENGTH "${_RUN_OUTPUT}" _RESLENGTH)
                        ENDIF()
                        IF(_LINE MATCHES "[^:\n]+ finished: [0-9]+/[0-9]+ passed\r?")
                            # But strip out the line corresponding to
                            # _LAST_LINE since it will printed out separately
                            # below.
                            MATH(EXPR _MATCHCOUNT "${_MATCHCOUNT} + 1")
                            IF (NOT (_MATCHCOUNT EQUAL _COUNT))
                                MESSAGE("${_LINE}")
                            ENDIF()
                        ELSE()
                            MESSAGE("${_LINE}")
                        ENDIF()
                    ENDWHILE()
                ENDIF()
                STRING(REGEX REPLACE "([^:\n]+) finished: [0-9]+/[0-9]+ passed\r?\n" "\\1" _NAME ${_LAST_LINE})
                IF(_THISTOTAL)
                    MATH(EXPR _PERCENT "(100 * ${_THISPASS}) / ${_THISTOTAL}")
                ELSE()
                    SET(_PERCENT 100)
                ENDIF()
                SET(_RESULT "${_THISPASS}/${_THISTOTAL}")
                STRING(LENGTH "${_NAME}" _NAMLENGTH)
                STRING(LENGTH "${_RESULT}" _RESLENGTH)
                MATH(EXPR _PADLENGTH "${_DISLENGTH} - ${_NAMLENGTH} - ${_RESLENGTH}")
                IF(_PADLENGTH GREATER 1)
                    # STRING(REPEAT ...) require CMake 3.15 so use this
                    # instead.
                    SET(_Y 2)
                    SET(_PAD "  ")
                    WHILE(_Y LESS _PADLENGTH)
                        STRING(APPEND _PAD " ")
                        MATH(EXPR _Y "${_Y} + 1")
                    ENDWHILE()
                ELSE()
                    SET(_PAD " ")
                ENDIF()
                IF(_USECOLOR)
                    IF(_THISPASS EQUAL _THISTOTAL)
                        SET(_COLOR "${_GREEN}")
                    ELSEIF(_PERCENT GREATER 89)
                        SET(_COLOR "${_YELLOW}")
                    ELSE()
                        SET(_COLOR "${_RED}")
                    ENDIF()
                    IF(_VERBOSE)
                        MESSAGE("    ${_NAME} finished: ${_COLOR}${_RESULT}${_RESET} passed")
                    ELSE()
                        MESSAGE("    ${_NAME}${_PAD}${_COLOR}${_RESULT}${_RESET} passed")
                    ENDIF()
                ELSE()
                    IF(_VERBOSE)
                        MESSAGE("    ${_NAME} finished: ${_RESULT} passed")
                    ELSE()
                        MESSAGE("    ${_NAME}${_PAD}${_RESULT} passed")
                    ENDIF()
                ENDIF()
            ENDIF()
        ELSE()
            IF(_USECOLOR)
                MESSAGE("${_RED}${_CONVERTED_PATH}: Malformed output${_RESET}")
            ELSE()
                MESSAGE("${_CONVERTED_PATH}: Malformed output")
            ENDIF()
            SET(_EXITCODE 1)
        ENDIF()
    ENDIF()
    MATH(EXPR _X "${_X} + 1")
ENDWHILE()

# Print a summary of all the test results.
IF(_TOTAL)
    MATH(EXPR _PERCENT "(100 * ${_PASS}) / ${_TOTAL}")
    MATH(EXPR _DEC1 "((1000 * ${_PASS}) / ${_TOTAL}) % 10")
ELSE()
    SET(_PERCENT 100)
    SET(_DEC1 0)
ENDIF()
IF(_USECOLOR)
    IF(_PASS EQUAL _TOTAL)
        SET(_COLOR "${_GREEN}")
    ELSEIF(_PERCENT GREATER 89)
        SET(_COLOR "${_YELLOW}")
    ELSE()
        SET(_COLOR "${_RED}")
    ENDIF()
    MESSAGE("Total: ${_COLOR}${_PASS}/${_TOTAL}${_RESET} passed (${_COLOR}${_PERCENT}.${_DEC1}%${_RESET})")
ELSE()
    MESSAGE("Total: ${_PASS}/${_TOTAL} passed (${_PERCENT}.${_DEC1}%)")
ENDIF()

IF(_EXITCODE)
    # Cause CMake to exit with an exit code of 1.  Apparently, CMake does not
    # provide a way to exit with other non-zero exit codes.
    MESSAGE(FATAL_ERROR "Non-zero result: ${_EXITCODE}")
ENDIF()
