/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package de.zray.renderbackends.opengl;

import de.zray.renderbackends.opengl.debug.GLDebugRenderer;
import de.zray.se.world.World;
import de.zray.se.EngineSettings;
import de.zray.se.graphics.Camera;
import de.zray.se.logger.SELogger;
import de.zray.se.renderbackend.RenderBackend;
import org.lwjgl.glfw.*;
import org.lwjgl.system.*;

import java.nio.*;
import org.joml.Vector3d;

import static org.lwjgl.glfw.Callbacks.*;
import static org.lwjgl.glfw.GLFW.*;
import org.lwjgl.opengl.GL;
import static org.lwjgl.opengl.GL11.*;
import static org.lwjgl.system.MemoryStack.*;
import static org.lwjgl.system.MemoryUtil.*;
import org.lwjgl.opengl.GLCapabilities;

/**
 *
 * @author vortex
 */
public class GLRenderer implements RenderBackend{
    private long window = -1;
    private float aspectRatio = 1;
    private final String windowTitle = EngineSettings.get().title+" "+EngineSettings.get().version;
    private int windowW = EngineSettings.get().window.resX;
    private int windowH = EngineSettings.get().window.resY;
    private boolean closeRequested = false;
    private World currentWorld;

    private GLDebugRenderer dRenderer = new GLDebugRenderer();
    private GLRenderLight lightRender = new GLRenderLight();
    private GLRendererMesh meshRender = new GLRendererMesh();
    
    
    @Override
    public boolean init() {
        GLFWErrorCallback.createPrint(System.err).set();
        if ( !glfwInit() ){
            SELogger.get().dispatchMsg(this, SELogger.SELogType.ERROR, new String[]{"Unable to initialize GLFW"}, true);
            return false;
        }
        glfwWindowHint(GLFW_VISIBLE, GLFW_FALSE);
        glfwWindowHint(GLFW_RESIZABLE, GLFW_TRUE);
        window = glfwCreateWindow(windowW, windowH, windowTitle, NULL, NULL);
        if ( window == NULL ){
            SELogger.get().dispatchMsg(this, SELogger.SELogType.ERROR, new String[]{"Failed to create the GLFW window"}, true);
            return false;
        }
        glfwMakeContextCurrent(window);
        GL.createCapabilities();

        glfwSetWindowSizeCallback(window, (window, w, h) -> {
            calcWindowProps(w, h);
        });
        
        try ( MemoryStack stack = stackPush() ) {
            IntBuffer pWidth = stack.mallocInt(1);
            IntBuffer pHeight = stack.mallocInt(1);
            glfwGetWindowSize(window, pWidth, pHeight);
            GLFWVidMode vidmode = glfwGetVideoMode(glfwGetPrimaryMonitor());
            glfwSetWindowPos(window, (vidmode.width() - pWidth.get(0)) / 2, (vidmode.height() - pHeight.get(0)) / 2);
        }

        glfwSwapInterval(0);
        glfwShowWindow(window);
        glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
        calcWindowProps(windowW, windowH);
        return true;
    }

    private void calcWindowProps(int w, int h){
        aspectRatio = (float)w/(float)h;
        windowW = w;
        windowH = h;
        glViewport(0, 0, windowW, windowH);
    }
    
    @Override
    public boolean isInited() {
        return (window != -1);
    }

       
    @Override
    public void renderWorld(EngineSettings.DebugMode dMode) {
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        glEnable(GL_DEPTH_TEST);
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        applyCameraPositioning(currentWorld.getCurrentCamera());
        lightRender.renderLightSources(currentWorld);
        meshRender.renderActors(currentWorld);
        if(dMode == EngineSettings.DebugMode.DEBUG_AND_OBJECTS){
            renderDebug();
        }
        applyCamera(currentWorld.getCurrentCamera());
        glfwSwapBuffers(window);
    }

    @Override
    public void shutdown() {
        glfwFreeCallbacks(window);
        glfwDestroyWindow(window);
        glfwTerminate();
        glfwSetErrorCallback(null).free();
    }

    @Override
    public boolean closeRequested() {
        return (closeRequested || glfwWindowShouldClose(window));
    }

    @Override
    public boolean isReady() {
        return true;
    }

    @Override
    public void backendSwitchRequested(){
        //Do nothing
    }
    
    @Override
    public void setCurrentWorld(World world){
        this.currentWorld = world;
    }
    
    private void applyCameraPositioning(Camera cam){
        switch(cam.getViewMode()){
            case EGO :
                glRotated(cam.getRotation().x, 1, 0, 0);
                glRotated(cam.getRotation().y, 0, 1, 0);
                glRotated(cam.getRotation().z, 0, 0, 1);
                glTranslatef(-cam.getPosition().x, -cam.getPosition().y, -cam.getPosition().z);
                break;
            case THIRDPERSON :
                break;
        }
    }
    
    private void applyCamera(Camera cam){
        if(cam == null){
            applayEmptyCamera();
            return;
        }
        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        switch (cam.getCameraMode()) {
            case ORTHOGRAPHIC:
                glOrtho(0, windowW, windowH, 0, cam.getNear(), cam.getFar());
                break;
            case PERSPECTIVE:
                GLUProject.gluPerspective(cam.getFOV(), aspectRatio, cam.getNear(), cam.getFar());
                break;
        }
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
    }
    
    private void applayEmptyCamera(){
        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        glOrtho(0, windowW, windowH, 0, 1, 100);
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
    }

    @Override
    public int getWidth() {
        return windowW;
    }

    @Override
    public int getHeight() {
        return windowH;
    }
    
    @Override
    public void requestClose() {
        closeRequested = true;
        glfwSetWindowShouldClose(window, true);
    }

    @Override
    public void renderDebug() {
        dRenderer.render(currentWorld);
    }

    @Override
    public boolean pick(int pointerX, int pointerY) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public boolean rayPick(Vector3d ray) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public boolean featureTest() {
        GLFWErrorCallback.createPrint(System.err).set();
        if (!glfwInit()){
            SELogger.get().dispatchMsg(this, SELogger.SELogType.ERROR, new String[]{"Unable to initialize GLFW"}, true);
            return false;
        }
        glfwWindowHint(GLFW_VISIBLE, GLFW_FALSE);
        window = glfwCreateWindow(windowW, windowH, windowTitle, NULL, NULL);
        if ( window == NULL ){
            SELogger.get().dispatchMsg(this, SELogger.SELogType.ERROR, new String[]{"Failed to create the GLFW window"}, true);
            return false;
        }

        glfwMakeContextCurrent(window);
        GLCapabilities caps = GL.createCapabilities();
        
        boolean glSupported = caps.OpenGL15;
        if(glSupported){
            SELogger.get().dispatchMsg(GLRenderer.class, SELogger.SELogType.INFO, new String[]{"OpenGL 1.5 supported!"}, false);
        } else {
            SELogger.get().dispatchMsg(GLRenderer.class, SELogger.SELogType.ERROR, new String[]{"OpenGL 1.5 not supported!"}, false);
        }
        
        glfwFreeCallbacks(window);
        glfwDestroyWindow(window);
        glfwTerminate();
        glfwSetErrorCallback(null).free();
        window = -1;
        
        return glSupported;
    }

    @Override
    public String getClassAsString() {
        return GLRenderer.class.toString();
    }

    @Override
    public long getWindow() {
        return window;
    }
}
