<?php

declare(strict_types=1);

namespace WsdlToPhp\PhpGenerator\Tests\Element;

use InvalidArgumentException;
use TypeError;
use WsdlToPhp\PhpGenerator\Element\PhpClass;
use WsdlToPhp\PhpGenerator\Element\PhpConstant;
use WsdlToPhp\PhpGenerator\Element\PhpVariable;
use WsdlToPhp\PhpGenerator\Tests\TestCase;

/**
 * @internal
 * @coversDefaultClass
 */
class PhpConstantTest extends TestCase
{
    public function testGetPhpDeclarationNullValue()
    {
        $constant = new PhpConstant('foo');

        $this->assertSame('define(\'foo\', null);', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationTrueValue()
    {
        $constant = new PhpConstant('foo', true);

        $this->assertSame('define(\'foo\', true);', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationFalseValue()
    {
        $constant = new PhpConstant('foo', false);

        $this->assertSame('define(\'foo\', false);', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationStringOneValue()
    {
        $constant = new PhpConstant('foo', '1');

        $this->assertSame('define(\'foo\', \'1\');', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationNumberOneValue()
    {
        $constant = new PhpConstant('foo', 1);

        $this->assertSame('define(\'foo\', 1);', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationParenthesisValue()
    {
        $constant = new PhpConstant('foo', 'NCSA Common (Apache default)');

        $this->assertSame('define(\'foo\', \'NCSA Common (Apache default)\');', $constant->getPhpDeclaration());
    }

    public function testExceptionForNonScalerValue()
    {
        $this->expectException(InvalidArgumentException::class);

        new PhpConstant('Foo', []);
    }

    public function testGetPhpDeclarationNullValueForClass()
    {
        $constant = new PhpConstant('foo', null, new PhpClass('bar'));

        $this->assertSame('const FOO = null;', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationParenthesisValueForClass()
    {
        $constant = new PhpConstant('foo', 'NCSA Common (Apache default)', new PhpClass('bar'));

        $this->assertSame('const FOO = \'NCSA Common (Apache default)\';', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationTrueValueForClass()
    {
        $constant = new PhpConstant('foo', true, new PhpClass('Bar'));

        $this->assertSame('const FOO = true;', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationFalseValueForClass()
    {
        $constant = new PhpConstant('foo', false, new PhpClass('Bar'));

        $this->assertSame('const FOO = false;', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationStringOneValueForClass()
    {
        $constant = new PhpConstant('foo', '1', new PhpClass('Bar'));

        $this->assertSame('const FOO = \'1\';', $constant->getPhpDeclaration());
    }

    public function testGetPhpDeclarationNumberOneValueForClass()
    {
        $constant = new PhpConstant('foo', 1, new PhpClass('Bar'));

        $this->assertSame('const FOO = 1;', $constant->getPhpDeclaration());
    }

    public function testAddChild()
    {
        $this->expectException(InvalidArgumentException::class);

        $constant = new PhpVariable('Foo', 'bar');

        $constant->addChild(new PhpVariable('Bar', 'foo'));
    }

    public function testToStringNullValue()
    {
        $constant = new PhpConstant('foo');

        $this->assertSame('define(\'foo\', null);', $constant->toString());
    }

    public function testToStringNullValueMatchesStringCasting()
    {
        $constant = new PhpConstant('foo');

        $this->assertSame((string) $constant, $constant->toString());
    }

    public function testToStringNullValueForClass()
    {
        $constant = new PhpConstant('foo', null, new PhpClass('bar'));

        $this->assertSame('const FOO = null;', $constant->toString());
    }

    public function testGetChildrenTypes()
    {
        $constant = new PhpConstant('foo', null, new PhpClass('bar'));

        $this->assertSame([], $constant->getChildrenTypes());
    }

    public function testExceptionMessageOnName()
    {
        $this->expectException(TypeError::class);

        new PhpConstant(0);
    }

    public function testExceptionMessageOnValue()
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Value of type "object" is not a valid scalar value for PhpConstant object');

        new PhpConstant('Foo', new \stdClass());
    }
}
