using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using System.Web;
using System.Drawing;
using System.Reflection;
using System.Collections.Concurrent;
using System.Text;


namespace VedAstro.Library
{

    /// <summary>
    /// Central place to for all things related to generating an Events Chart in SVG
    /// </summary>
    public static class EventsChartFactory
    {
        //1 GENERATE DATA FOR EVENT ROWS
        public const int SingleRowHeight = 15;
        public const int SummaryRowHeight = 22;

        /// <summary>
        /// Height used in virtual space for generating life events,
        /// will throw error if exceeds, hence 10k pixel space
        /// </summary>
        private const int MaxLifeEventCanvasHeight = 10000;

        /// <summary>
        /// Width of the blue background behind life event descriptions
        /// </summary>
        private const int DescriptionBackgroundWidth = 100; //for 24 characters

        /// <summary>
        /// max Y axis used in chart set by life events, and other as generated
        /// used to make final SVG height
        /// </summary>
        public static int MaxYAxis = 0;

        //px width & height of each slice of time
        //used when generating dasa rows
        //note: changes needed only here
        private const int WidthPerSlice = 1;

        /// <summary>
        /// Filled when first events come out fresh from oven unsorted
        /// declare static, stops GC
        /// </summary>
        public static List<Event> UnsortedEventList { get; set; }



        //▒█▀▀█ ▒█░▒█ ▒█▀▀█ ▒█░░░ ▀█▀ ▒█▀▀█ 
        //▒█▄▄█ ▒█░▒█ ▒█▀▀▄ ▒█░░░ ▒█░ ▒█░░░ 
        //▒█░░░ ░▀▄▄▀ ▒█▄▄█ ▒█▄▄█ ▄█▄ ▒█▄▄█


        /// <summary>
        /// Main method that starts making chart
        /// </summary>
        public static string GenerateEventsChartSvg(EventsChart chartSpecs)
        {

            var inputPerson = chartSpecs.Person;
            var timeRange = chartSpecs.TimeRange;
            var daysPerPixel = chartSpecs.DaysPerPixel;
            var inputedEventTags = chartSpecs.EventTagList;
            var summaryOptions = chartSpecs.Options;

            //ACT I : declare the components
            string svgHead = null;
            string contentHead = null;
            string timeHeader = null;
            string eventRows = null;
            string summaryRow = null;
            string nowLine = null;
            string contentTail = null;
            string cursorLine = null;
            string lifeEvents = null;
            string border = null;
            string jsCode = null;
            string svgTail = null;
            string svgDefinitions = null;


            //ACT II : fill the components in order
            GenerateComponents(inputPerson, timeRange.start, timeRange.end, daysPerPixel, inputedEventTags, summaryOptions);


            //ACT III : compile in right placement
            var final =
                $@" <!--MADE BY MACHINES FOR HUMAN EYES-->
                    {svgHead}
                        {svgDefinitions}

                        <!--inside border-->
                        {contentHead}
                            {timeHeader}
                            {eventRows}
                            {lifeEvents}
                            {summaryRow}
                            {nowLine}
                        {contentTail}

                        <!--outside border-->
                        {border} <!--border painted on top events-->
                        {cursorLine} <!--events legend painted on top border-->
                        {jsCode} <!--place last-->
                    {svgTail}
                ";

            //ACT IV : take a bow & exit the stage.
            return final;


            void GenerateComponents(Person inputPerson, Time startTime, Time endTime, double daysPerPixel, List<EventTag> inputedEventTags, ChartOptions summaryOptions)
            {
                //STEP 1: USER INPUT > USABLE DATA
                var svgBackgroundColor = "transparent"; //blends nicely with site's back
                var randomId = Tools.GenerateId();

                // One precision value for generating all dasa components,
                // because misalignment occurs if use different precision
                // note: precision = time slice count, each slice = 1 pixel (zoom level)
                double eventsPrecisionHours = Tools.DaysToHours(daysPerPixel);

                //generate time slice only once for all rows
                var timeSlices = Time.GetTimeListFromRange(startTime, endTime, eventsPrecisionHours);
                var svgWidth = timeSlices.Count; //slice per px
                var svgTotalWidth = svgWidth + 10; //add little for wiggle room

                int verticalYAxis; //last position of element above is set here
                verticalYAxis = 0; //start at 0 y axis

                //rows are dynamically generated as needed, hence the extra logic below
                //list of rows to generate
                EventsChartFactory.UnsortedEventList = EventManager.CalculateEvents(eventsPrecisionHours, startTime, endTime, inputPerson, inputedEventTags);

                //STEP 2: DATA > SVG COMPONENTS
                timeHeader = GenerateTimeHeaderRow(timeSlices, daysPerPixel, WidthPerSlice, ref verticalYAxis);

                //note : avg speed 30s
                eventRows = GenerateEventRows(
                    eventsPrecisionHours,
                    inputPerson,
                    timeSlices,
                    summaryOptions, ref verticalYAxis);

                var padding = 1;//space between rows
                summaryRow = GenerateSummaryRow(SummaryRowData, SummaryRowHeight, SingleRowHeight, padding, ref verticalYAxis);

                nowLine = MakeNowLine(startTime, verticalYAxis, timeSlices);

                lifeEvents = GetLifeEventLinesSvg(inputPerson, verticalYAxis, startTime, timeSlices);

                border = GetBorderSvg(timeSlices, verticalYAxis);

                cursorLine = GetTimeCursorLine(verticalYAxis);

                //get max height
                MaxYAxis = verticalYAxis > MaxYAxis ? verticalYAxis : MaxYAxis;

                //note: if width & height not hard set, parent div clips it
                var svgTotalHeight = MaxYAxis + 80; //little padding

                //set custom/global styles for main SVG element
                var svgStyle = $@"
                width:{svgTotalWidth}px;
                font-family:Calibri;
                height:{svgTotalHeight}px;
                background:{svgBackgroundColor};";

                //place birth time in chart to be used by EventsChart.js for google event detection
                var birthTimeStdText = inputPerson.BirthTimeString;

                svgHead = $"<svg birthtime=\"{birthTimeStdText}\" birthlocation=\"{inputPerson.GetBirthLocation().Name()}\" class=\"EventChartHolder\" id=\"{randomId}\" style=\"{svgStyle}\" xmlns=\"http://www.w3.org/2000/svg\" xmlns:xlink=\"http://www.w3.org/1999/xlink\">";//much needed for use tags to work

                //inject JS code to be able to run direct in browser without website
                jsCode = GetJsCodeSvg(randomId);

                //place all content except border & cursor time legend inside group for padding
                const int contentPadding = 2;//todo move to central place
                contentHead = $"<g class=\"EventChartContent\" transform=\"matrix(1, 0, 0, 1, {contentPadding}, {contentPadding})\">";

                svgTail = "</svg>";
                contentTail = "</g>";

                //filters used for extra styling
                svgDefinitions = @"
                        <defs>
                            <filter id=""glow"">
                                <feGaussianBlur in=""SourceAlpha"" stdDeviation=""1"" result=""blur"" />
                                <feFlood flood-color=""white"" flood-opacity=""1"" result=""color"" />
                                <feComposite in=""color"" in2=""blur"" operator=""in"" result=""blur"" />
                                <feMerge>
                                    <feMergeNode in=""blur"" />
                                    <feMergeNode in=""SourceGraphic"" />
                                </feMerge>
                            </filter>
                        </defs>";

            }

        }


        /// <summary>
        /// Generates events chart with specs and generated SVG chart
        /// </summary>
        public static EventsChart GenerateEventsChart(Person foundPerson, TimeRange timeRange, double daysPerPixel, List<EventTag> eventTags, ChartOptions summaryOptions)
        {
            //a new chart is born
            var newChartId = Tools.GenerateId();
            var newChart = new EventsChart(newChartId, "", foundPerson, timeRange, daysPerPixel, eventTags, summaryOptions);

            //from person get SVG chart as string
            var eventsChartSvgString = GenerateEventsChartSvg(newChart);

            //combine chart and data
            newChart.ContentSvg = eventsChartSvgString;

            return newChart;
        }


        /// <summary>
        /// wraps a list of svg elements inside 1 main svg element
        /// if width not set defaults to 1000px, and height to 1000px
        /// </summary>
        public static string WrapSvgElements(string svgClass, string combinedSvgString, int svgWidth, int svgTotalHeight, string randomId, string svgBackgroundColor = "#f0f9ff")
        {
            //use random id for each chart, done so that js can uniquely
            //manipulate 1 chart in page of multiple charts

            var fontsSVG = @"<font horiz-adv-x=""2048"">
<!-- Calibri is a trademark of the Microsoft group of companies. -->
<!-- Copyright: Copyright 2024 Adobe System Incorporated. All rights reserved. -->
<font-face font-family=""Calibri-Bold"" units-per-em=""2048"" underline-position=""-232"" underline-thickness=""134""/>
<missing-glyph horiz-adv-x=""1038"" d=""M987,1294l0,-1294l-945,0l0,1294M879,101l0,1092l-732,0l0,-1092M780,845C780,807 775,773 764,744C753,715 737,690 718,669C698,648 674,632 647,620C619,608 589,600 556,596l-5,-121C550,466 543,460 529,457C514,453 495,451 470,451C455,451 442,451 432,452C421,453 413,454 407,456C400,457 396,460 393,463C390,466 389,470 389,475l-8,157C379,659 383,677 394,687C404,697 419,702 439,702l13,0C477,702 498,706 515,713C532,720 545,729 556,741C567,753 575,767 580,784C585,800 587,817 587,836C587,877 575,910 550,935C525,960 486,972 435,972C412,972 391,970 373,966C355,961 339,957 326,952C313,947 302,942 293,938C284,933 278,931 275,931C272,931 270,932 267,934C264,935 262,938 260,943C258,948 257,955 256,964C255,973 254,985 254,1000C254,1017 255,1031 257,1040C258,1049 262,1056 267,1061C272,1066 281,1072 294,1078C307,1083 322,1088 340,1093C358,1098 378,1102 400,1105C422,1108 445,1110 468,1110C525,1110 573,1103 613,1089C652,1075 684,1056 709,1032C734,1008 752,980 763,948C774,915 780,881 780,845M574,284C574,267 572,252 569,240C566,227 560,217 552,210C544,202 534,196 521,193C508,190 492,188 473,188C454,188 438,190 425,193C412,196 401,202 393,210C385,217 379,227 376,240C373,252 371,267 371,284C371,303 373,318 376,331C379,343 385,353 393,361C401,369 412,375 425,378C438,381 454,383 473,383C492,383 508,381 521,378C534,375 544,369 552,361C560,353 566,343 569,331C572,318 574,303 574,284z""/>
<glyph unicode=""A"" horiz-adv-x=""1241"" d=""M1201,98C1209,74 1214,55 1216,41C1218,27 1215,17 1208,10C1201,3 1188,-2 1169,-3C1150,-5 1125,-6 1092,-6C1058,-6 1032,-5 1013,-4C994,-3 979,-1 969,2C959,5 952,9 948,14C944,19 941,26 938,35l-87,260l-486,0l-82,-253C280,33 277,25 273,19C268,12 261,7 252,4C242,0 228,-3 211,-4C193,-5 170,-6 141,-6C110,-6 86,-5 69,-3C52,-1 40,4 33,12C26,19 24,30 26,44C28,58 33,77 41,100l399,1148C444,1259 449,1269 454,1276C459,1283 468,1288 480,1292C491,1295 507,1298 528,1299C549,1300 576,1300 610,1300C649,1300 681,1300 704,1299C727,1298 746,1295 759,1292C772,1288 781,1282 787,1275C792,1268 797,1258 801,1245M607,1047l-1,0l-183,-550l367,0z""/>
<glyph unicode=""B"" horiz-adv-x=""1148"" d=""M1078,388C1078,342 1072,300 1059,263C1046,226 1029,193 1007,164C984,135 958,111 927,90C896,69 861,52 824,39C787,26 747,16 705,10C662,3 615,0 563,0l-344,0C197,0 179,7 164,20C149,33 141,54 141,83l0,1128C141,1240 149,1262 164,1275C179,1288 197,1294 219,1294l325,0C623,1294 691,1287 746,1274C801,1261 847,1241 885,1214C922,1187 951,1152 971,1111C991,1070 1001,1021 1001,965C1001,934 997,904 989,877C981,849 969,824 954,801C939,778 920,758 898,741C875,724 850,710 821,699C858,692 892,681 923,664C954,647 981,626 1004,600C1027,573 1045,542 1058,507C1071,472 1078,432 1078,388M738,933C738,959 734,982 726,1003C718,1024 706,1041 690,1055C674,1068 654,1079 630,1086C605,1093 573,1097 532,1097l-133,0l0,-336l147,0C584,761 614,766 637,775C660,784 679,796 694,812C709,827 720,846 727,867C734,888 738,910 738,933M805,379C805,409 800,436 790,460C780,483 765,503 746,519C727,535 702,547 672,556C642,565 603,569 555,569l-156,0l0,-368l190,0C626,201 657,205 682,213C707,220 728,232 747,247C765,262 779,280 790,303C800,325 805,350 805,379z""/>
<glyph unicode=""C"" horiz-adv-x=""1084"" d=""M1036,196C1036,179 1036,165 1035,154C1034,142 1032,132 1030,124C1028,115 1025,108 1022,102C1019,96 1013,89 1006,82C999,74 985,64 964,52C943,40 917,29 887,18C856,7 822,-3 783,-10C744,-17 701,-21 656,-21C567,-21 487,-7 416,20C345,47 284,88 234,143C184,197 146,265 119,346C92,427 79,522 79,630C79,740 94,838 123,923C152,1008 193,1080 246,1138C299,1196 362,1240 436,1270C509,1300 590,1315 679,1315C715,1315 750,1312 783,1306C816,1300 847,1292 876,1283C904,1273 929,1262 952,1249C975,1236 991,1226 1000,1217C1009,1208 1015,1200 1018,1194C1021,1188 1024,1180 1026,1171C1028,1162 1030,1151 1031,1138C1032,1125 1032,1110 1032,1091C1032,1071 1031,1054 1030,1040C1029,1026 1026,1015 1023,1006C1020,997 1016,991 1011,987C1006,983 1001,981 995,981C985,981 972,987 957,999C942,1010 922,1023 898,1038C873,1052 844,1065 811,1077C777,1088 737,1094 690,1094C639,1094 593,1084 553,1063C512,1042 478,1012 450,973C421,934 400,886 385,831C370,776 363,713 363,644C363,568 371,502 387,447C402,391 425,345 454,309C483,273 518,246 559,229C600,211 646,202 697,202C744,202 784,208 818,219C852,230 881,242 906,255C930,268 950,280 966,291C981,302 993,307 1002,307C1009,307 1014,306 1018,303C1022,300 1025,295 1028,287C1031,279 1033,268 1034,254C1035,239 1036,220 1036,196z""/>
<glyph unicode=""D"" horiz-adv-x=""1291"" d=""M1216,666C1216,547 1201,446 1170,362C1139,277 1094,208 1036,155C978,102 908,63 825,38C742,13 643,0 528,0l-309,0C197,0 179,7 164,20C149,33 141,54 141,83l0,1128C141,1240 149,1262 164,1275C179,1288 197,1294 219,1294l332,0C666,1294 764,1281 845,1254C925,1227 993,1187 1048,1134C1103,1081 1144,1015 1173,937C1202,859 1216,769 1216,666M944,657C944,718 937,775 923,828C908,881 885,926 854,965C822,1004 781,1034 732,1056C682,1077 617,1088 536,1088l-133,0l0,-880l137,0C612,208 673,217 722,236C771,255 813,283 846,321C879,358 903,405 920,461C936,517 944,582 944,657z""/>
<glyph unicode=""E"" horiz-adv-x=""999"" d=""M908,103C908,84 907,68 906,55C904,42 901,31 898,23C895,15 891,9 886,6C881,2 875,0 869,0l-650,0C197,0 179,7 164,20C149,33 141,54 141,83l0,1128C141,1240 149,1262 164,1275C179,1288 197,1294 219,1294l646,0C871,1294 876,1292 881,1289C886,1286 890,1280 893,1272C896,1263 899,1253 901,1240C902,1227 903,1210 903,1190C903,1171 902,1156 901,1143C899,1130 896,1119 893,1111C890,1103 886,1097 881,1094C876,1090 871,1088 865,1088l-462,0l0,-316l391,0C800,772 806,770 811,767C816,763 820,757 823,750C826,742 829,732 831,719C832,706 833,690 833,671C833,652 832,636 831,623C829,610 826,600 823,593C820,585 816,579 811,576C806,573 800,571 794,571l-391,0l0,-365l466,0C875,206 881,204 886,201C891,197 895,191 898,183C901,175 904,165 906,152C907,139 908,122 908,103z""/>
<glyph unicode=""F"" horiz-adv-x=""940"" d=""M867,1186C867,1166 866,1149 865,1136C863,1122 860,1111 857,1103C853,1095 849,1089 844,1085C839,1081 834,1079 828,1079l-423,0l0,-348l397,0C808,731 813,729 818,726C823,723 827,717 831,709C834,701 837,690 839,677C840,664 841,647 841,627C841,607 840,590 839,577C837,564 834,553 831,544C827,535 823,529 818,526C813,522 808,520 802,520l-397,0l0,-482C405,31 403,24 399,19C395,14 388,9 378,6C367,2 354,-1 337,-3C320,-5 299,-6 273,-6C248,-6 227,-5 210,-3C193,-1 179,2 169,6C159,9 152,14 148,19C143,24 141,31 141,38l0,1173C141,1240 149,1262 164,1275C179,1288 197,1294 219,1294l609,0C834,1294 839,1292 844,1289C849,1285 853,1279 857,1271C860,1262 863,1251 865,1237C866,1223 867,1206 867,1186z""/>
<glyph unicode=""I"" horiz-adv-x=""546"" d=""M405,36C405,29 403,23 399,18C394,13 387,8 377,5C367,2 354,-1 337,-3C320,-5 299,-6 273,-6C248,-6 227,-5 210,-3C193,-1 179,2 169,5C159,8 152,13 148,18C143,23 141,29 141,36l0,1222C141,1265 143,1271 148,1276C152,1281 159,1286 170,1289C180,1292 193,1295 210,1297C227,1299 248,1300 273,1300C299,1300 320,1299 337,1297C354,1295 367,1292 377,1289C387,1286 394,1281 399,1276C403,1271 405,1265 405,1258z""/>
<glyph unicode=""L"" horiz-adv-x=""866"" d=""M852,110C852,90 851,73 850,60C848,46 845,35 842,26C839,17 835,10 830,6C825,2 819,0 812,0l-593,0C197,0 179,7 164,20C149,33 141,54 141,83l0,1175C141,1265 143,1271 148,1276C152,1281 159,1286 169,1289C179,1292 193,1295 210,1297C227,1299 248,1300 273,1300C299,1300 320,1299 337,1297C354,1295 367,1292 377,1289C387,1286 394,1281 399,1276C403,1271 405,1265 405,1258l0,-1041l407,0C819,217 825,215 830,212C835,208 839,202 842,194C845,185 848,174 850,161C851,147 852,130 852,110z""/>
<glyph unicode=""M"" horiz-adv-x=""1790"" d=""M1649,36C1649,29 1647,23 1644,18C1640,13 1633,8 1624,5C1614,2 1601,-1 1585,-3C1569,-5 1549,-6 1524,-6C1500,-6 1480,-5 1464,-3C1448,-1 1435,2 1426,5C1417,8 1410,13 1406,18C1402,23 1400,29 1400,36l0,1053l-2,0l-375,-1052C1020,28 1016,21 1010,16C1004,10 996,5 986,2C975,-1 962,-3 946,-4C930,-5 911,-6 888,-6C865,-6 846,-5 830,-3C814,-2 801,1 791,5C780,8 772,13 766,18C760,23 756,30 754,37l-362,1052l-2,0l0,-1053C390,29 388,23 385,18C381,13 374,8 364,5C354,2 341,-1 326,-3C310,-5 290,-6 265,-6C241,-6 221,-5 205,-3C189,-1 176,2 167,5C157,8 150,13 147,18C143,23 141,29 141,36l0,1153C141,1223 150,1249 168,1267C186,1285 210,1294 240,1294l172,0C443,1294 469,1292 491,1287C513,1282 532,1273 548,1262C564,1250 577,1235 588,1216C599,1197 608,1173 616,1145l280,-771l4,0l290,769C1199,1171 1208,1195 1219,1214C1229,1233 1241,1249 1255,1261C1268,1273 1284,1282 1303,1287C1322,1292 1343,1294 1368,1294l177,0C1563,1294 1579,1292 1592,1287C1605,1282 1615,1276 1624,1267C1632,1258 1638,1247 1643,1234C1647,1221 1649,1206 1649,1189z""/>
<glyph unicode=""N"" horiz-adv-x=""1349"" d=""M1208,92C1208,77 1205,63 1200,51C1195,39 1188,29 1179,21C1170,13 1159,7 1147,4C1134,0 1122,-2 1109,-2l-112,0C974,-2 954,0 937,5C920,10 904,18 890,31C875,43 861,60 848,81C835,102 820,129 803,162l-322,605C462,803 443,842 424,884C405,925 387,966 372,1005l-2,0C373,957 375,909 376,862C377,814 378,765 378,714l0,-677C378,30 376,24 373,19C369,14 363,9 354,6C345,2 332,-1 317,-3C302,-5 282,-6 258,-6C235,-6 215,-5 200,-3C185,-1 173,2 164,6C155,9 149,14 146,19C143,24 141,30 141,37l0,1163C141,1231 150,1255 169,1271C187,1286 209,1294 236,1294l141,0C402,1294 424,1292 441,1288C458,1283 474,1276 488,1266C501,1256 514,1242 526,1225C538,1207 550,1185 563,1159l252,-473C830,657 844,629 859,602C873,574 887,546 900,519C913,491 926,464 939,437C952,410 964,384 976,357l1,0C975,404 974,452 973,503C972,554 971,602 971,648l0,607C971,1262 973,1268 977,1273C981,1278 988,1283 997,1287C1006,1291 1019,1294 1034,1296C1049,1297 1069,1298 1093,1298C1116,1298 1135,1297 1150,1296C1165,1294 1177,1291 1186,1287C1194,1283 1200,1278 1203,1273C1206,1268 1208,1262 1208,1255z""/>
<glyph unicode=""O"" horiz-adv-x=""1385"" d=""M1305,662C1305,555 1292,459 1265,374C1238,289 1199,218 1146,159C1093,100 1028,55 951,24C873,-7 783,-23 681,-23C580,-23 493,-10 418,17C343,43 280,83 230,137C180,191 143,260 118,343C93,426 80,525 80,639C80,744 93,838 120,922C147,1005 186,1076 239,1135C292,1194 357,1239 435,1270C512,1301 602,1317 705,1317C803,1317 890,1304 965,1278C1040,1252 1102,1212 1153,1158C1203,1104 1241,1036 1267,954C1292,871 1305,774 1305,662M1029,648C1029,716 1024,778 1013,834C1002,889 984,937 959,977C933,1016 899,1047 856,1069C813,1090 760,1101 696,1101C631,1101 577,1089 534,1065C491,1040 456,1008 429,967C402,926 384,879 373,825C362,770 356,713 356,652C356,581 361,518 372,462C383,405 401,357 426,317C451,277 485,247 528,226C571,205 624,194 689,194C754,194 808,206 851,230C894,254 929,287 956,328C983,369 1002,418 1013,473C1024,528 1029,586 1029,648z""/>
<glyph unicode=""P"" horiz-adv-x=""1090"" d=""M1027,901C1027,828 1016,764 993,708C970,652 937,605 894,567C851,528 798,499 735,479C672,459 597,449 512,449l-108,0l0,-413C404,29 402,23 398,18C393,13 386,8 376,5C366,2 353,-1 336,-3C319,-5 298,-6 272,-6C247,-6 226,-5 209,-3C192,-1 178,2 168,5C158,8 151,13 147,18C143,23 141,29 141,36l0,1164C141,1231 149,1255 166,1271C182,1286 203,1294 230,1294l305,0C566,1294 595,1293 623,1291C650,1288 683,1283 722,1276C761,1268 800,1254 840,1233C879,1212 913,1186 941,1155C969,1123 990,1086 1005,1044C1020,1001 1027,954 1027,901M752,882C752,927 744,965 728,994C712,1023 692,1045 669,1059C646,1073 621,1082 596,1086C570,1089 543,1091 516,1091l-112,0l0,-439l118,0C564,652 599,658 628,669C656,680 679,696 697,717C715,737 729,761 738,790C747,818 752,849 752,882z""/>
<glyph unicode=""R"" horiz-adv-x=""1153"" d=""M1101,34C1101,27 1100,21 1097,16C1094,11 1088,6 1078,3C1068,0 1053,-3 1034,-4C1015,-5 988,-6 955,-6C927,-6 905,-5 888,-4C871,-3 858,0 849,4C839,7 832,12 828,18C824,23 821,30 818,38l-116,289C688,360 674,389 661,414C648,439 633,461 617,478C600,495 581,507 560,516C539,525 514,529 486,529l-82,0l0,-493C404,29 402,23 398,18C393,13 386,8 376,5C366,2 353,-1 336,-3C319,-5 298,-6 272,-6C247,-6 226,-5 209,-3C192,-1 178,2 168,5C158,8 151,13 147,18C143,23 141,29 141,36l0,1175C141,1240 149,1262 164,1275C179,1288 197,1294 219,1294l335,0C588,1294 616,1293 638,1292C660,1291 680,1289 698,1287C750,1280 797,1267 839,1249C880,1231 916,1208 945,1179C974,1150 996,1115 1011,1076C1026,1036 1034,990 1034,939C1034,896 1029,856 1018,821C1007,785 990,753 969,725C948,697 921,673 890,652C859,631 823,615 783,602C802,593 821,581 838,568C855,555 871,539 886,520C901,501 915,479 928,455C941,430 954,403 967,372l109,-255C1086,92 1093,73 1096,62C1099,50 1101,41 1101,34M762,912C762,955 752,991 733,1020C714,1049 682,1070 638,1082C625,1085 610,1088 593,1090C576,1092 552,1093 522,1093l-118,0l0,-367l134,0C575,726 608,731 636,740C664,749 687,761 706,778C725,794 739,813 748,836C757,859 762,884 762,912z""/>
<glyph unicode=""T"" horiz-adv-x=""1014"" d=""M1002,1186C1002,1167 1001,1150 1000,1137C998,1123 995,1112 992,1104C989,1095 985,1089 980,1085C975,1081 969,1079 963,1079l-324,0l0,-1043C639,29 637,23 633,18C628,13 621,8 611,5C601,2 588,-1 571,-3C554,-5 532,-6 507,-6C482,-6 461,-5 444,-3C427,-1 413,2 403,5C393,8 386,13 382,18C377,23 375,29 375,36l0,1043l-324,0C44,1079 39,1081 34,1085C29,1089 25,1095 22,1104C19,1112 16,1123 15,1137C13,1150 12,1167 12,1186C12,1206 13,1223 15,1237C16,1251 19,1262 22,1271C25,1279 29,1285 34,1289C39,1292 44,1294 51,1294l912,0C969,1294 975,1292 980,1289C985,1285 989,1279 992,1271C995,1262 998,1251 1000,1237C1001,1223 1002,1206 1002,1186z""/>
<glyph unicode=""U"" horiz-adv-x=""1337"" d=""M1197,482C1197,403 1185,332 1162,269C1139,206 1104,153 1059,110C1014,67 958,34 891,11C824,-12 748,-23 661,-23C580,-23 507,-13 442,8C377,28 323,59 278,100C233,141 199,193 176,255C152,316 140,388 140,471l0,787C140,1265 142,1271 146,1276C150,1281 157,1286 168,1289C178,1292 191,1295 208,1297C225,1299 246,1300 272,1300C297,1300 318,1299 335,1297C352,1295 365,1292 375,1289C385,1286 392,1281 397,1276C401,1271 403,1265 403,1258l0,-764C403,443 409,398 422,361C435,323 453,292 477,267C500,242 529,224 562,212C595,199 631,193 672,193C713,193 750,199 783,212C816,225 843,243 866,268C889,292 906,322 919,358C931,393 937,434 937,479l0,779C937,1265 939,1271 943,1276C947,1281 954,1286 964,1289C974,1292 988,1295 1005,1297C1022,1299 1043,1300 1068,1300C1093,1300 1114,1299 1131,1297C1147,1295 1160,1292 1170,1289C1180,1286 1187,1281 1191,1276C1195,1271 1197,1265 1197,1258z""/>
<glyph unicode=""Y"" horiz-adv-x=""1064"" d=""M664,491l0,-455C664,29 662,23 658,18C653,13 646,8 636,5C626,2 613,-1 596,-3C579,-5 557,-6 532,-6C506,-6 485,-5 468,-3C451,-1 438,2 428,5C417,8 410,13 406,18C402,23 400,29 400,36l0,455l-354,705C33,1221 25,1240 22,1254C18,1267 20,1278 28,1285C35,1292 49,1296 70,1298C91,1299 119,1300 156,1300C187,1300 211,1299 230,1298C248,1297 262,1294 273,1291C284,1287 292,1282 297,1276C302,1270 306,1262 311,1253l144,-318C469,904 483,872 497,838C510,804 524,768 538,731l2,0C553,767 566,802 579,836C592,870 605,902 618,933l142,316C763,1260 768,1268 773,1275C778,1281 785,1286 795,1290C805,1294 819,1297 836,1298C853,1299 876,1300 905,1300C943,1300 973,1299 994,1298C1015,1296 1029,1292 1037,1285C1044,1278 1046,1267 1042,1253C1038,1239 1030,1220 1018,1196z""/>
<glyph unicode=""0"" horiz-adv-x=""1038"" d=""M982,652C982,549 974,455 957,372C940,289 913,218 876,159C838,100 789,55 729,24C668,-7 594,-23 507,-23C418,-23 345,-8 287,23C229,53 183,97 149,154C115,211 91,280 78,363C64,445 57,538 57,642C57,745 66,838 83,922C100,1005 127,1076 165,1135C202,1194 251,1239 312,1270C372,1301 445,1317 532,1317C621,1317 694,1302 753,1272C811,1241 857,1198 891,1141C925,1084 949,1014 962,932C975,849 982,756 982,652M717,639C717,700 715,755 712,802C709,849 704,889 697,924C690,959 681,988 671,1012C660,1036 648,1055 633,1070C618,1084 602,1094 583,1100C564,1106 544,1109 521,1109C482,1109 449,1100 424,1081C398,1062 378,1033 363,995C348,957 337,910 331,853C325,796 322,730 322,655C322,563 326,487 333,427C340,366 352,318 368,283C383,247 403,222 428,208C453,193 482,186 517,186C543,186 566,190 586,199C606,207 624,219 639,236C654,253 666,274 676,299C686,324 694,354 701,387C707,420 711,458 714,500C716,542 717,588 717,639z""/>
<glyph unicode=""2"" horiz-adv-x=""1038"" d=""M956,107C956,88 955,71 954,58C952,44 949,33 946,24C943,15 938,9 933,6C928,2 922,0 915,0l-744,0C156,0 144,1 133,4C122,7 114,12 107,20C100,27 96,38 93,53C90,68 88,87 88,110C88,132 89,151 91,167C93,182 97,196 102,209C107,221 114,233 123,245C131,256 142,269 155,283l224,240C424,570 460,612 487,651C514,689 536,724 551,756C566,787 577,816 583,843C588,870 591,895 591,919C591,941 588,962 581,982C574,1001 563,1018 550,1033C536,1048 519,1059 498,1068C477,1076 453,1080 425,1080C386,1080 351,1075 321,1065C290,1055 264,1044 241,1032C218,1019 198,1008 183,998C168,987 156,982 147,982C141,982 136,984 132,988C127,992 124,999 121,1008C118,1017 116,1030 115,1046C113,1061 112,1080 112,1103C112,1118 113,1131 114,1142C115,1152 116,1161 118,1169C120,1176 123,1183 126,1189C129,1194 135,1201 144,1209C152,1217 167,1227 190,1240C212,1252 239,1264 272,1276C305,1287 341,1297 380,1305C419,1313 460,1317 503,1317C570,1317 629,1309 680,1292C730,1275 772,1251 805,1221C838,1190 863,1154 880,1113C896,1071 904,1026 904,978C904,936 900,895 892,854C884,813 867,770 842,723C817,676 781,624 735,567C689,510 628,443 552,368l-151,-155l510,0C918,213 924,211 930,207C935,203 940,197 944,189C948,180 951,169 953,156C955,142 956,126 956,107z""/>
<glyph unicode=""4"" horiz-adv-x=""1038"" d=""M996,372C996,338 993,312 986,295C979,277 969,268 957,268l-115,0l0,-234C842,27 840,22 836,17C832,12 825,7 815,4C805,1 792,-2 776,-3C760,-5 739,-6 714,-6C690,-6 670,-5 654,-3C637,-2 624,1 615,4C605,7 598,12 595,17C591,22 589,27 589,34l0,234l-495,0C85,268 76,269 69,272C62,274 56,279 51,288C46,296 42,308 40,324C37,339 36,360 36,386C36,407 37,426 38,442C39,457 40,471 43,484C45,497 48,509 53,520C57,531 62,543 69,556l402,709C474,1271 480,1276 488,1281C496,1285 507,1289 522,1292C536,1295 554,1297 576,1298C598,1299 625,1300 656,1300C690,1300 719,1299 743,1298C767,1296 786,1293 801,1290C815,1286 825,1281 832,1276C839,1271 842,1264 842,1257l0,-779l115,0C968,478 977,470 985,454C992,437 996,410 996,372M589,1073l-2,0l-341,-595l343,0z""/>
</font>

	<font horiz-adv-x=""2048"">
<!-- Calibri is a trademark of the Microsoft group of companies. -->
<!-- Copyright: Copyright 2024 Adobe System Incorporated. All rights reserved. -->
<font-face font-family=""Calibri-Light"" units-per-em=""2048"" underline-position=""-261"" underline-thickness=""104""/>
<missing-glyph horiz-adv-x=""1038"" d=""M973,1294l0,-1294l-927,0l0,1294M897,70l0,1154l-778,0l0,-1154M760,860C760,820 754,784 742,753C730,722 713,695 692,674C670,653 644,636 615,625C586,613 554,607 519,606l-5,-170C514,431 510,427 503,425C496,422 485,421 471,421C464,421 459,421 454,422C449,423 445,424 442,425C439,426 436,428 435,430C433,431 432,433 432,436l-4,200C428,651 432,663 441,670C450,677 461,680 475,680l20,0C528,680 555,685 577,694C598,703 616,715 629,730C642,745 652,764 658,785C663,806 666,829 666,853C666,906 651,948 622,979C593,1010 545,1025 480,1025C454,1025 431,1022 410,1017C389,1011 370,1005 354,998C338,991 325,984 314,979C303,973 296,970 291,970C289,970 287,971 285,972C282,973 280,975 279,978C277,981 276,985 275,990C274,995 273,1001 273,1009C273,1017 274,1024 276,1031C277,1037 281,1043 286,1048C291,1053 300,1058 313,1065C326,1071 342,1077 361,1083C379,1089 399,1094 422,1099C444,1103 466,1105 488,1105C534,1105 574,1099 608,1086C642,1073 670,1056 693,1034C715,1012 732,986 743,956C754,926 760,894 760,860M536,267C536,255 535,245 533,237C531,229 528,223 523,218C518,213 512,209 505,207C497,205 487,204 476,204C464,204 454,205 446,207C438,209 432,213 427,218C422,223 418,229 416,237C414,245 413,255 413,267C413,279 414,289 416,297C418,305 422,312 427,317C432,322 438,325 446,327C454,329 464,330 476,330C487,330 497,329 505,327C512,325 518,322 523,317C528,312 531,305 533,297C535,289 536,279 536,267z""/>
<glyph unicode="" "" horiz-adv-x=""463""/>
<glyph unicode=""A"" horiz-adv-x=""1154"" d=""M1108,41C1112,31 1114,23 1114,17C1113,10 1111,5 1107,2C1102,-1 1096,-3 1087,-4C1078,-5 1066,-6 1052,-6C1038,-6 1027,-5 1019,-4C1010,-3 1004,-2 999,0C994,2 991,5 989,8C986,11 984,15 982,20l-124,345l-568,0l-119,-343C170,17 168,13 165,10C162,7 159,4 154,2C149,-1 143,-3 135,-4C126,-5 116,-6 103,-6C89,-6 77,-5 68,-4C59,-3 52,0 48,3C43,6 41,11 41,17C40,23 42,31 46,41l453,1231C501,1277 504,1282 508,1286C512,1289 517,1292 524,1294C530,1296 538,1298 547,1299C556,1300 566,1300 577,1300C589,1300 600,1300 609,1299C618,1298 625,1296 632,1294C638,1292 643,1289 647,1286C650,1282 653,1277 655,1272M571,1163l-1,0l-244,-696l495,0z""/>
<glyph unicode=""E"" horiz-adv-x=""1001"" d=""M883,54C883,45 882,38 881,31C880,24 878,19 875,14C872,9 868,6 864,4C860,1 856,0 851,0l-606,0C233,0 221,5 208,14C195,23 189,39 189,63l0,1168C189,1255 195,1272 208,1281C221,1290 233,1294 245,1294l597,0C847,1294 851,1293 855,1291C859,1288 862,1285 865,1280C868,1275 870,1270 871,1263C872,1256 873,1247 873,1238C873,1229 872,1222 871,1215C870,1208 868,1203 865,1199C862,1194 859,1191 855,1189C851,1186 847,1185 842,1185l-531,0l0,-455l458,0C774,730 779,729 783,727C787,724 790,721 793,717C796,712 798,707 799,700C800,693 801,684 801,675C801,666 800,659 799,653C798,646 796,641 793,637C790,633 787,630 783,628C779,626 774,625 769,625l-458,0l0,-516l540,0C856,109 860,108 864,106C868,103 872,100 875,96C878,91 880,86 881,79C882,72 883,63 883,54z""/>
<glyph unicode=""G"" horiz-adv-x=""1285"" d=""M1125,1131C1125,1122 1124,1114 1123,1107C1122,1100 1120,1094 1118,1089C1115,1084 1112,1080 1109,1077C1105,1074 1101,1073 1097,1073C1089,1073 1076,1080 1058,1093C1039,1106 1015,1121 984,1137C953,1152 916,1167 873,1180C829,1193 777,1200 717,1200C640,1200 572,1186 512,1158C452,1129 401,1090 360,1041C319,992 287,933 266,866C244,798 233,725 233,647C233,560 245,482 270,414C295,345 329,288 373,241C416,194 468,158 529,133C589,108 655,96 726,96C772,96 819,102 866,113C913,124 958,142 999,165l0,419l-330,0C658,584 650,589 645,598C640,607 637,621 637,638C637,647 638,656 639,663C640,670 642,675 645,680C648,684 651,687 655,690C659,692 664,693 669,693l396,0C1071,693 1077,692 1084,690C1091,688 1097,685 1103,680C1108,675 1113,669 1116,661C1119,652 1121,642 1121,630l0,-481C1121,133 1119,119 1114,108C1109,97 1097,86 1079,76C1061,65 1037,54 1007,42C977,29 946,19 914,11C882,2 850,-4 818,-8C785,-13 753,-15 722,-15C624,-15 537,1 460,33C383,64 318,109 265,167C212,224 171,293 143,373C115,452 101,540 101,636C101,737 116,829 147,912C177,995 220,1065 275,1124C330,1183 395,1228 472,1261C549,1293 633,1309 726,1309C773,1309 817,1305 859,1296C900,1287 938,1277 971,1265C1004,1252 1033,1239 1056,1225C1079,1210 1095,1199 1104,1191C1112,1182 1118,1174 1121,1166C1124,1158 1125,1146 1125,1131z""/>
<glyph unicode=""T"" horiz-adv-x=""990"" d=""M973,1238C973,1229 972,1222 971,1215C970,1208 968,1203 965,1199C962,1194 959,1191 955,1189C951,1186 947,1185 942,1185l-386,0l0,-1165C556,16 555,12 553,9C551,6 548,3 543,1C538,-1 532,-3 524,-4C516,-5 506,-6 495,-6C484,-6 474,-5 466,-4C458,-3 452,-1 447,1C442,3 439,6 437,9C435,12 434,16 434,20l0,1165l-386,0C43,1185 38,1186 35,1189C31,1191 28,1194 25,1199C22,1203 20,1208 19,1215C18,1222 17,1229 17,1238C17,1247 18,1256 19,1263C20,1270 22,1275 25,1280C28,1285 31,1288 35,1291C38,1293 43,1294 48,1294l894,0C947,1294 951,1293 955,1291C959,1288 962,1285 965,1280C968,1275 970,1270 971,1263C972,1256 973,1247 973,1238z""/>
<glyph unicode=""a"" horiz-adv-x=""964"" d=""M801,20C801,13 799,8 795,5C790,1 785,-2 778,-3C771,-5 761,-6 749,-6C738,-6 728,-5 720,-3C712,-2 706,1 702,5C697,8 695,13 695,20l0,101C654,77 610,43 561,19C512,-5 461,-17 408,-17C361,-17 318,-11 281,2C243,14 211,32 185,55C159,78 139,105 125,138C111,171 104,207 104,248C104,298 114,342 135,379C156,416 185,446 223,471C261,496 307,514 360,527C413,539 473,545 538,545l144,0l0,78C682,662 678,696 670,726C662,755 649,780 631,800C613,820 590,835 561,845C532,855 497,860 456,860C416,860 380,855 348,846C316,836 288,825 264,813C240,801 220,790 205,781C189,771 178,766 171,766C167,766 163,767 160,768C157,769 154,772 152,776C149,779 148,784 147,790C146,795 145,802 145,810C145,824 146,835 148,844C150,852 155,860 164,869C172,877 187,887 208,898C229,909 254,920 282,930C309,939 339,947 371,953C403,959 435,962 467,962C529,962 581,955 624,940C667,925 701,903 728,875C754,847 773,813 784,772C795,731 801,684 801,632M682,447l-165,0C469,447 427,443 391,434C355,425 325,413 301,396C277,379 259,359 248,336C236,313 230,286 230,256C230,204 247,163 280,132C313,101 359,85 419,85C466,85 511,97 552,122C593,146 636,183 682,234z""/>
<glyph unicode=""e"" horiz-adv-x=""1012"" d=""M905,519C905,495 899,479 886,470C873,461 861,456 849,456l-617,0C232,400 238,349 249,304C260,259 278,220 303,188C328,155 361,130 402,113C443,96 493,87 552,87C594,87 632,91 665,98C698,105 727,114 752,123C776,132 796,140 812,147C827,154 838,158 845,158C849,158 853,157 856,155C859,153 862,150 864,146C866,141 868,136 869,129C870,122 870,114 870,104C870,99 870,95 869,91C868,86 868,82 867,79C866,75 864,71 862,68C860,65 857,61 854,58C851,55 840,49 823,41C806,32 783,24 755,16C727,7 695,0 658,-7C621,-14 581,-17 538,-17C468,-17 406,-6 352,15C298,36 253,67 216,108C179,149 152,199 133,260C114,321 104,391 104,471C104,548 114,616 134,677C153,738 181,789 218,832C255,874 299,906 351,929C402,951 460,962 523,962C591,962 649,951 697,928C745,905 785,875 816,837C847,798 869,754 884,703C898,652 905,599 905,543M782,554C784,650 762,725 716,779C670,833 603,860 515,860C470,860 430,852 396,835C361,818 332,795 309,767C285,739 267,707 254,670C241,633 234,594 232,554z""/>
<glyph unicode=""l"" horiz-adv-x=""452"" d=""M286,20C286,16 285,12 283,9C281,6 278,3 274,1C269,-1 263,-3 256,-4C248,-5 238,-6 226,-6C215,-6 205,-5 198,-4C190,-3 184,-1 179,1C174,3 171,6 170,9C168,12 167,16 167,20l0,1348C167,1372 168,1376 170,1379C171,1382 174,1385 179,1388C184,1390 190,1392 198,1393C205,1394 215,1394 226,1394C238,1394 248,1394 256,1393C263,1392 269,1390 274,1388C278,1385 281,1382 283,1379C285,1376 286,1372 286,1368z""/>
<glyph unicode=""m"" horiz-adv-x=""1620"" d=""M1457,20C1457,16 1456,12 1454,9C1452,6 1449,3 1445,1C1440,-1 1434,-3 1427,-4C1419,-5 1409,-6 1398,-6C1386,-6 1376,-5 1369,-4C1361,-3 1355,-1 1350,1C1345,3 1342,6 1341,9C1339,12 1338,16 1338,20l0,557C1338,620 1334,659 1327,694C1319,729 1307,759 1291,784C1274,809 1254,828 1229,842C1204,855 1174,862 1140,862C1098,862 1056,846 1013,813C970,780 922,732 871,669l0,-649C871,16 870,12 868,9C866,6 863,3 859,1C854,-1 848,-3 841,-4C833,-5 823,-6 812,-6C801,-6 791,-5 784,-4C776,-3 770,-1 765,1C760,3 757,6 756,9C754,12 753,16 753,20l0,557C753,620 749,659 741,694C733,729 721,759 705,784C688,809 668,828 643,842C618,855 588,862 554,862C512,862 470,846 427,813C384,780 337,732 286,669l0,-649C286,16 285,12 283,9C281,6 278,3 274,1C269,-1 263,-3 256,-4C248,-5 238,-6 226,-6C215,-6 205,-5 198,-4C190,-3 184,-1 179,1C174,3 171,6 170,9C168,12 167,16 167,20l0,905C167,929 168,933 170,936C171,939 174,942 179,945C184,947 190,949 197,950C204,951 213,951 224,951C235,951 245,951 252,950C259,949 265,947 269,945C273,942 276,939 278,936C279,933 280,929 280,925l0,-128C332,855 381,897 428,923C475,949 521,962 567,962C605,962 639,958 669,949C698,940 724,927 747,911C770,894 789,875 805,852C821,829 834,804 844,777C875,812 905,841 933,865C960,888 987,907 1012,922C1037,936 1061,946 1084,953C1107,959 1130,962 1153,962C1210,962 1258,952 1297,932C1336,912 1367,885 1391,851C1414,817 1431,777 1442,732C1452,687 1457,639 1457,588z""/>
<glyph unicode=""p"" horiz-adv-x=""1064"" d=""M957,486C957,409 949,339 932,277C915,215 890,162 857,119C824,76 783,42 734,19C685,-5 630,-17 567,-17C540,-17 515,-14 492,-8C469,-3 446,6 424,18C401,30 379,45 357,63C334,81 311,103 286,128l0,-467C286,-343 285,-347 283,-350C281,-353 278,-356 274,-358C269,-361 263,-362 256,-363C248,-364 238,-365 226,-365C215,-365 205,-364 198,-363C190,-362 184,-361 179,-358C174,-356 171,-353 170,-350C168,-347 167,-343 167,-339l0,1264C167,930 168,934 170,937C171,940 174,943 179,945C183,947 189,949 196,950C203,951 211,951 221,951C231,951 239,951 246,950C253,949 258,947 263,945C268,943 271,940 273,937C275,934 276,930 276,925l0,-131C304,823 331,848 356,869C381,890 407,907 432,921C457,935 483,945 510,952C537,959 565,962 595,962C660,962 716,950 762,925C808,900 846,865 875,822C904,779 925,728 938,671C951,613 957,551 957,486M832,467C832,518 828,567 820,614C812,661 799,702 780,739C761,776 735,805 704,827C672,849 632,860 584,860C560,860 537,857 514,850C491,843 467,832 443,817C419,802 394,782 368,758C342,733 315,703 286,667l0,-394C336,212 384,166 430,134C475,101 523,85 573,85C619,85 659,96 692,119C725,141 751,170 772,207C793,243 808,284 818,330C827,375 832,421 832,467z""/>
<glyph unicode=""t"" horiz-adv-x=""673"" d=""M603,74C603,60 602,49 600,40C597,31 594,25 589,20C584,15 576,10 567,6C557,1 546,-2 533,-5C520,-8 507,-11 493,-12C479,-14 465,-15 452,-15C407,-15 369,-9 338,3C307,15 281,33 261,58C241,82 227,113 218,150C209,187 204,231 204,282l0,563l-135,0C60,845 52,849 47,858C42,866 39,878 39,895C39,904 40,911 41,918C42,924 44,929 47,934C50,938 53,941 57,943C60,945 64,946 69,946l135,0l0,222C204,1172 205,1176 207,1179C208,1182 211,1185 216,1188C221,1190 227,1192 235,1193C242,1194 252,1194 263,1194C275,1194 285,1194 293,1193C300,1192 306,1190 311,1188C315,1185 318,1182 320,1179C322,1176 323,1172 323,1168l0,-222l250,0C578,946 582,945 586,943C589,941 592,938 595,934C598,929 600,924 601,918C602,911 603,904 603,895C603,878 600,866 595,858C589,849 582,845 573,845l-250,0l0,-547C323,229 333,177 353,141C372,105 408,87 460,87C476,87 491,89 504,92C517,95 528,99 538,103C548,107 557,111 564,114C571,117 577,119 582,119C585,119 588,118 591,117C594,116 596,113 598,110C599,106 601,101 602,96C603,90 603,83 603,74z""/>
<glyph unicode="":"" horiz-adv-x=""538"" d=""M373,806C373,788 372,773 370,761C367,748 363,738 356,731C349,724 341,719 330,716C319,713 304,711 287,711C270,711 256,713 245,716C234,719 226,724 219,731C212,738 208,748 206,761C203,773 202,788 202,806C202,825 203,841 206,854C208,866 212,876 219,883C226,890 234,896 245,899C256,902 270,903 287,903C304,903 319,902 330,899C341,896 349,890 356,883C363,876 367,866 370,854C372,841 373,825 373,806M373,89C373,70 372,55 370,43C367,30 363,20 356,13C349,6 341,1 330,-2C319,-5 304,-7 287,-7C270,-7 256,-5 245,-2C234,1 226,6 219,13C212,20 208,30 206,43C203,55 202,70 202,89C202,108 203,123 206,136C208,149 212,159 219,166C226,173 234,178 245,181C256,184 270,185 287,185C304,185 319,184 330,181C341,178 349,173 356,166C363,159 367,149 370,136C372,123 373,108 373,89z""/>
<glyph unicode=""/"" horiz-adv-x=""742"" d=""M122,-232C119,-240 116,-246 113,-251C109,-256 105,-259 100,-262C95,-265 89,-266 82,-267C75,-268 66,-269 56,-269C46,-269 37,-268 30,-266C22,-264 16,-261 11,-257C6,-254 4,-249 3,-243C2,-238 3,-231 6,-224l610,1648C619,1432 622,1438 625,1443C628,1448 633,1451 638,1454C643,1457 649,1459 656,1460C663,1461 672,1461 682,1461C692,1461 701,1460 708,1458C715,1456 721,1453 726,1450C730,1446 733,1441 734,1436C735,1430 735,1423 732,1416z""/>
<glyph unicode=""0"" horiz-adv-x=""1038"" d=""M959,649C959,552 951,462 936,381C921,300 895,230 860,171C824,112 777,66 720,33C662,0 591,-17 506,-17C428,-17 362,-3 308,26C253,54 209,96 176,152C142,208 118,277 103,360C88,442 80,537 80,645C80,742 88,831 103,913C118,994 144,1065 179,1124C214,1183 261,1229 319,1262C376,1295 447,1311 532,1311C610,1311 676,1297 731,1269C785,1240 829,1198 863,1142C897,1086 922,1017 937,935C952,852 959,757 959,649M831,638C831,704 829,764 824,818C819,871 811,919 800,961C789,1003 776,1040 760,1071C743,1102 724,1127 701,1148C678,1168 651,1183 622,1193C593,1202 560,1207 523,1207C457,1207 404,1192 363,1161C322,1130 290,1088 267,1037C244,986 228,927 220,861C212,795 208,727 208,656C208,557 214,472 225,401C236,329 254,270 279,224C304,177 335,143 374,121C413,98 460,87 515,87C559,87 597,94 630,108C663,122 691,142 714,167C737,192 757,222 772,257C787,292 799,329 808,370C817,411 823,454 826,499C829,544 831,591 831,638z""/>
<glyph unicode=""1"" horiz-adv-x=""1038"" d=""M918,50C918,41 917,33 915,27C913,20 910,15 907,11C904,7 900,4 896,3C892,1 888,0 884,0l-656,0C224,0 220,1 217,3C213,4 209,7 206,11C203,15 200,20 198,27C196,33 195,41 195,50C195,58 196,65 198,72C199,79 202,84 205,89C208,93 211,96 215,99C219,101 223,102 228,102l287,0l0,1062l-274,-167C230,990 222,987 215,986C208,985 203,986 199,989C195,992 192,998 191,1006C190,1013 189,1022 189,1033C189,1042 189,1049 190,1056C191,1062 192,1067 194,1072C196,1077 199,1081 202,1084C205,1087 208,1091 213,1094l310,198C525,1293 528,1295 531,1296C534,1297 538,1299 543,1300C548,1301 553,1301 558,1302C563,1302 570,1302 577,1302C588,1302 598,1301 606,1300C614,1299 620,1297 625,1295C630,1293 633,1290 635,1287C637,1284 638,1280 638,1276l0,-1174l246,0C889,102 893,101 898,99C902,96 906,93 909,89C912,84 914,79 916,72C917,65 918,58 918,50z""/>
<glyph unicode=""2"" horiz-adv-x=""1038"" d=""M907,56C907,48 906,41 905,34C903,27 901,21 898,16C895,11 891,7 886,4C881,1 876,0 870,0l-699,0C165,0 159,1 154,3C149,5 144,8 140,13C136,17 133,23 131,30C129,37 128,46 128,56C128,65 128,72 129,79C130,86 131,92 133,97C135,102 138,108 141,113C144,118 149,124 154,130l265,280C485,479 538,541 577,596C616,651 645,700 666,744C687,788 700,828 707,864C713,900 716,934 716,966C716,998 711,1029 701,1058C690,1087 675,1112 655,1134C635,1155 610,1173 581,1186C551,1199 517,1205 478,1205C436,1205 398,1199 365,1188C332,1176 303,1163 278,1149C253,1134 232,1121 215,1110C198,1098 186,1092 178,1092C174,1092 170,1093 167,1095C164,1097 161,1100 159,1105C156,1109 155,1114 154,1121C153,1128 152,1135 152,1144C152,1152 152,1159 153,1164C154,1169 155,1174 156,1179C157,1183 159,1187 162,1191C164,1195 169,1200 177,1207C184,1214 197,1223 216,1234C234,1245 257,1257 285,1269C313,1280 344,1290 378,1299C412,1307 447,1311 484,1311C545,1311 598,1302 643,1284C688,1266 725,1242 755,1211C785,1180 807,1145 822,1105C837,1064 844,1022 844,977C844,936 841,895 834,854C827,813 812,768 789,720C766,671 732,618 689,559C645,500 587,433 514,356l-236,-247l592,0C875,109 880,108 885,106C890,103 894,100 897,95C900,90 903,85 905,78C906,71 907,64 907,56z""/>
<glyph unicode=""3"" horiz-adv-x=""1038"" d=""M894,365C894,309 885,258 867,211C848,164 821,124 785,90C749,56 704,30 650,11C596,-8 534,-17 463,-17C420,-17 380,-13 343,-6C306,1 273,10 244,20C215,30 190,40 171,51C152,62 140,69 135,73C130,77 127,81 125,85C122,89 120,94 119,99C117,104 116,110 115,118C114,125 113,134 113,145C113,165 115,179 120,186C124,193 130,197 137,197C143,197 155,191 174,180C192,169 216,156 245,143C274,130 307,117 346,106C384,95 426,89 473,89C521,89 563,95 599,108C635,121 665,139 690,162C715,185 733,212 746,244C758,275 764,310 764,347C764,387 756,423 740,456C724,488 701,516 671,539C641,562 604,580 561,593C518,606 469,612 415,612l-128,0C283,612 279,613 275,615C270,617 266,620 263,625C259,629 256,634 254,641C251,648 250,656 250,666C250,675 251,682 253,689C255,695 258,700 261,705C264,709 268,712 273,714C277,716 282,717 287,717l113,0C447,717 489,723 527,736C564,749 596,767 623,791C650,814 670,843 685,876C699,909 706,945 706,986C706,1016 701,1044 692,1071C682,1098 668,1121 649,1141C630,1161 606,1177 577,1188C548,1199 514,1205 475,1205C434,1205 397,1199 364,1187C331,1175 301,1162 276,1148C250,1133 229,1120 213,1108C196,1096 185,1090 178,1090C174,1090 170,1091 167,1092C164,1093 161,1096 159,1100C156,1104 155,1110 154,1117C153,1124 152,1133 152,1144C152,1151 152,1158 153,1164C154,1169 155,1174 157,1179C159,1184 161,1188 164,1193C167,1197 171,1202 177,1208C183,1214 195,1223 214,1235C233,1247 256,1259 283,1270C310,1281 340,1291 375,1299C410,1307 447,1311 486,1311C545,1311 597,1303 641,1287C684,1271 720,1249 749,1221C778,1193 799,1160 813,1122C827,1083 834,1042 834,997C834,956 829,917 818,882C807,846 790,814 769,787C748,759 722,736 691,717C660,698 624,686 584,679l0,-2C629,673 670,662 708,645C745,628 778,605 806,578C833,551 855,519 871,483C886,446 894,407 894,365z""/>
<glyph unicode=""4"" horiz-adv-x=""1038"" d=""M970,365C970,350 967,337 961,328C954,318 945,313 933,313l-164,0l0,-293C769,16 768,12 766,9C764,6 761,3 756,1C751,-1 745,-3 738,-4C730,-5 720,-6 708,-6C697,-6 687,-5 680,-4C672,-3 666,-1 661,1C656,3 653,6 652,9C650,12 649,16 649,20l0,293l-556,0C87,313 82,314 77,316C72,317 68,320 65,324C61,328 58,333 57,340C55,347 54,355 54,365C54,374 54,381 55,388C56,395 57,401 59,407C60,413 63,419 66,425C69,430 72,436 76,443l501,829C580,1277 585,1281 590,1285C595,1288 601,1291 609,1294C616,1296 625,1298 636,1299C647,1300 659,1300 673,1300C688,1300 702,1299 714,1298C726,1297 736,1295 744,1293C752,1290 758,1287 763,1283C767,1279 769,1275 769,1270l0,-853l164,0C944,417 953,413 960,404C967,395 970,382 970,365M649,1190l-2,0l-465,-773l467,0z""/>
<glyph unicode=""8"" horiz-adv-x=""1038"" d=""M942,324C942,269 932,221 913,179C893,136 865,101 828,72C791,43 745,21 692,6C639,-9 578,-17 510,-17C447,-17 389,-10 338,4C287,17 243,37 207,64C170,91 142,124 123,163C103,202 93,246 93,297C93,340 100,379 115,414C130,449 151,481 178,510C205,539 238,567 277,592C316,617 359,641 408,664C365,685 327,708 294,732C261,756 233,782 211,809C188,836 171,866 160,897C149,928 143,961 143,996C143,1038 151,1078 166,1116C181,1154 205,1188 236,1217C267,1246 307,1269 356,1286C405,1303 462,1311 528,1311C590,1311 644,1303 690,1288C735,1272 773,1251 803,1224C833,1197 855,1166 870,1131C885,1096 892,1059 892,1020C892,984 886,950 874,918C862,885 845,855 822,826C799,797 772,771 739,746C706,721 668,697 625,675C676,650 722,626 762,601C801,576 834,549 861,521C888,492 908,462 922,430C935,398 942,363 942,324M766,1007C766,1038 761,1066 750,1091C739,1116 722,1137 701,1155C680,1172 654,1186 623,1195C592,1204 556,1209 517,1209C436,1209 375,1191 333,1155C290,1118 269,1068 269,1005C269,976 274,950 285,925C295,900 311,877 332,855C353,832 379,811 412,790C444,769 482,747 527,725C604,762 663,805 704,852C745,899 766,950 766,1007M814,306C814,340 808,371 795,398C782,425 763,451 738,475C712,498 680,521 641,544C602,566 557,589 506,614C458,591 416,568 380,546C344,523 314,500 291,476C268,452 250,427 239,400C227,373 221,344 221,311C221,240 246,184 297,145C347,105 422,85 521,85C617,85 690,105 740,145C789,184 814,238 814,306z""/>
</font>

	<font horiz-adv-x=""2048"">
<!-- Calibri is a trademark of the Microsoft group of companies. -->
<!-- Copyright: Copyright 2024 Adobe System Incorporated. All rights reserved. -->
<font-face font-family=""Calibri"" units-per-em=""2048"" underline-position=""-232"" underline-thickness=""134""/>
<missing-glyph horiz-adv-x=""1038"" d=""M978,1294l0,-1294l-934,0l0,1294M891,81l0,1132l-762,0l0,-1132M767,855C767,816 761,781 750,750C738,719 722,694 701,673C680,652 655,635 626,623C597,611 566,604 532,602l-5,-152C527,443 522,439 512,436C502,433 488,432 471,432C461,432 453,432 446,433C439,434 434,435 430,436C425,437 422,439 420,441C418,443 417,446 417,450l-5,184C411,654 415,668 424,676C433,684 446,688 462,688l18,0C510,688 535,692 555,701C575,709 591,720 604,734C616,748 625,765 630,785C635,804 638,825 638,847C638,896 624,934 597,963C569,992 525,1006 464,1006C439,1006 417,1004 397,999C377,994 359,988 344,982C329,975 316,969 307,964C297,959 290,956 285,956C283,956 281,957 279,958C276,959 274,962 272,966C270,969 269,974 268,981C267,987 266,995 266,1006C266,1017 267,1027 269,1034C270,1041 274,1048 279,1053C284,1058 293,1063 306,1069C319,1075 335,1081 354,1087C372,1092 392,1097 414,1101C436,1105 458,1107 481,1107C531,1107 574,1100 610,1087C646,1074 676,1056 699,1033C722,1010 739,984 750,953C761,922 767,890 767,855M549,273C549,259 548,247 546,238C543,229 539,221 534,215C528,209 520,205 511,202C501,199 489,198 475,198C460,198 448,199 439,202C429,205 421,209 415,215C409,221 405,229 402,238C399,247 398,259 398,273C398,287 399,299 402,309C405,318 409,326 415,332C421,338 429,342 439,345C448,348 460,349 475,349C489,349 501,348 511,345C520,342 528,338 534,332C539,326 543,318 546,309C548,299 549,287 549,273z""/>
<glyph unicode="" "" horiz-adv-x=""463""/>
<glyph unicode=""D"" horiz-adv-x=""1260"" d=""M1167,666C1167,554 1153,456 1124,373C1095,289 1054,220 999,165C944,110 876,68 796,41C716,14 621,0 510,0l-274,0C221,0 206,5 193,16C179,26 172,44 172,70l0,1154C172,1250 179,1268 193,1279C206,1289 221,1294 236,1294l293,0C641,1294 736,1280 813,1251C890,1222 955,1180 1008,1126C1060,1071 1100,1005 1127,928C1154,851 1167,763 1167,666M986,659C986,729 977,794 960,855C943,915 916,967 879,1011C842,1055 795,1089 738,1114C681,1139 608,1151 519,1151l-175,0l0,-1007l177,0C604,144 673,154 730,175C787,196 834,227 872,270C910,313 939,366 958,431C977,495 986,571 986,659z""/>
<glyph unicode=""I"" horiz-adv-x=""516"" d=""M344,26C344,21 343,16 340,12C337,8 333,5 326,3C319,0 311,-2 300,-3C289,-5 275,-6 258,-6C242,-6 228,-5 217,-3C206,-2 197,0 190,3C183,5 179,8 176,12C173,16 172,21 172,26l0,1242C172,1273 174,1278 177,1282C180,1286 185,1289 192,1292C199,1294 208,1296 219,1298C230,1299 243,1300 258,1300C275,1300 289,1299 300,1298C311,1296 319,1294 326,1292C333,1289 337,1286 340,1282C343,1278 344,1273 344,1268z""/>
<glyph unicode=""J"" horiz-adv-x=""653"" d=""M484,326C484,275 479,228 470,185C461,142 445,106 422,76C399,46 370,23 333,7C296,-10 250,-18 195,-18C176,-18 158,-16 140,-13C121,-10 104,-6 89,-1C74,4 60,9 49,15C38,20 30,25 27,29C23,33 20,37 18,41C15,45 13,50 12,56C10,61 9,68 8,77C7,85 7,95 7,107C7,122 8,134 9,144C10,153 11,161 14,167C17,172 20,176 24,179C27,181 32,182 37,182C42,182 49,180 58,175C66,170 76,164 88,158C99,152 113,147 128,142C143,137 161,134 182,134C205,134 224,137 240,143C256,148 269,159 280,174C291,189 299,211 304,238C309,265 312,300 312,343l0,926C312,1274 313,1278 316,1282C319,1286 323,1289 330,1292C337,1295 346,1297 357,1298C368,1299 381,1300 398,1300C414,1300 428,1299 439,1298C450,1297 459,1295 466,1292C472,1289 477,1286 480,1282C483,1278 484,1274 484,1269z""/>
<glyph unicode=""K"" horiz-adv-x=""1064"" d=""M1016,28C1016,23 1015,18 1013,14C1010,9 1006,6 999,3C992,0 983,-2 972,-3C961,-5 946,-6 928,-6C905,-6 886,-5 871,-3C856,-1 845,3 838,8C831,13 825,19 820,26l-476,647l0,-647C344,21 343,17 340,13C337,8 333,5 326,3C319,0 311,-2 300,-3C289,-5 275,-6 258,-6C242,-6 228,-5 217,-3C206,-2 197,0 190,3C183,5 179,8 176,13C173,17 172,21 172,26l0,1242C172,1273 173,1278 176,1282C179,1286 183,1289 190,1292C197,1294 206,1296 217,1298C228,1299 242,1300 258,1300C275,1300 289,1299 300,1298C311,1296 319,1294 326,1292C333,1289 337,1286 340,1282C343,1278 344,1273 344,1268l0,-575l458,575C806,1274 811,1279 816,1283C821,1287 828,1290 836,1293C843,1295 853,1297 864,1298C875,1299 889,1300 906,1300C923,1300 937,1299 948,1298C959,1296 967,1294 974,1291C980,1288 984,1285 987,1281C989,1277 990,1273 990,1268C990,1259 988,1251 984,1242C979,1233 971,1221 959,1206l-429,-513l462,-614C1003,62 1010,50 1013,44C1015,37 1016,32 1016,28z""/>
<glyph unicode=""M"" horiz-adv-x=""1751"" d=""M1579,26C1579,21 1578,16 1575,12C1572,8 1568,5 1561,3C1554,0 1546,-2 1536,-3C1525,-5 1512,-6 1495,-6C1480,-6 1467,-5 1456,-3C1445,-2 1436,0 1430,3C1423,5 1419,8 1416,12C1413,16 1412,21 1412,26l0,1127l-2,0l-461,-1132C947,16 944,12 941,9C937,6 932,3 925,1C918,-2 909,-3 900,-4C890,-5 878,-6 865,-6C851,-6 839,-5 829,-4C819,-3 811,-1 804,2C797,4 792,7 789,10C785,13 782,17 781,21l-441,1132l-1,0l0,-1127C339,21 338,16 335,12C332,8 328,5 321,3C314,0 306,-2 295,-3C284,-5 271,-6 254,-6C238,-6 225,-5 214,-3C203,-2 195,0 189,3C182,5 178,8 176,12C173,16 172,21 172,26l0,1190C172,1244 179,1264 194,1276C209,1288 225,1294 243,1294l104,0C368,1294 387,1292 403,1288C419,1284 433,1278 445,1269C457,1260 467,1249 475,1236C483,1223 490,1207 496,1189l375,-937l5,0l390,934C1273,1206 1281,1223 1290,1237C1299,1251 1308,1262 1319,1271C1329,1279 1340,1285 1353,1289C1366,1292 1380,1294 1397,1294l109,0C1516,1294 1526,1293 1535,1290C1544,1287 1551,1282 1558,1276C1564,1269 1569,1261 1573,1252C1577,1242 1579,1230 1579,1216z""/>
<glyph unicode=""N"" horiz-adv-x=""1322"" d=""M1150,71C1150,58 1148,46 1144,37C1139,28 1134,20 1127,14C1120,8 1112,4 1103,1C1094,-2 1086,-3 1077,-3l-55,0C1005,-3 990,-1 977,3C964,6 951,13 940,23C928,32 916,45 905,62C893,79 880,100 866,126l-408,736C437,900 415,940 394,982C372,1023 352,1064 333,1103l-2,0C332,1055 333,1006 334,956C335,906 335,856 335,807l0,-781C335,21 334,17 331,13C328,8 324,5 318,3C311,0 303,-2 293,-3C282,-5 269,-6 253,-6C237,-6 224,-5 214,-3C203,-2 195,0 189,3C183,5 179,8 176,13C173,17 172,21 172,26l0,1194C172,1247 179,1266 194,1277C209,1288 225,1294 242,1294l82,0C343,1294 360,1292 373,1289C386,1286 397,1280 408,1273C418,1265 428,1254 438,1241C447,1227 458,1210 469,1189l314,-568C802,586 821,553 839,520C857,487 874,454 891,422C908,390 924,359 941,328C957,297 973,265 989,234l1,0C989,287 988,342 988,399C987,456 987,510 987,563l0,702C987,1270 988,1274 991,1278C994,1282 998,1286 1005,1289C1011,1292 1019,1294 1030,1295C1040,1296 1053,1297 1070,1297C1085,1297 1097,1296 1108,1295C1119,1294 1127,1292 1133,1289C1139,1286 1143,1282 1146,1278C1149,1274 1150,1270 1150,1265z""/>
<glyph unicode=""O"" horiz-adv-x=""1356"" d=""M1256,662C1256,557 1244,463 1219,379C1194,295 1158,224 1109,165C1060,106 998,60 924,29C850,-3 764,-19 665,-19C568,-19 484,-4 413,25C342,54 283,96 237,151C190,206 156,275 133,357C110,439 99,533 99,640C99,742 111,835 136,918C161,1001 198,1072 247,1131C296,1189 358,1234 432,1266C506,1297 592,1313 691,1313C786,1313 869,1299 940,1270C1011,1241 1070,1200 1117,1145C1164,1090 1199,1022 1222,941C1245,860 1256,767 1256,662M1075,650C1075,723 1069,792 1056,855C1043,918 1021,972 991,1019C960,1065 920,1101 869,1127C818,1153 756,1166 681,1166C606,1166 544,1152 493,1124C442,1096 401,1059 370,1012C338,965 315,911 302,849C288,786 281,721 281,652C281,576 287,506 300,442C313,378 334,323 364,277C394,230 434,194 485,169C535,143 598,130 674,130C749,130 813,144 864,172C915,200 957,238 988,286C1019,333 1042,388 1055,451C1068,514 1075,580 1075,650z""/>
<glyph unicode=""P"" horiz-adv-x=""1058"" d=""M967,915C967,850 956,792 935,740C914,688 883,644 844,607C804,570 755,542 698,522C640,502 571,492 491,492l-147,0l0,-466C344,21 343,16 340,12C337,8 332,5 326,3C319,0 311,-2 300,-3C289,-5 275,-6 258,-6C241,-6 228,-5 217,-3C206,-2 197,0 190,3C183,5 179,8 176,12C173,16 172,21 172,26l0,1194C172,1247 179,1266 193,1277C207,1288 223,1294 240,1294l277,0C545,1294 572,1293 598,1291C623,1288 654,1283 689,1276C724,1268 759,1254 796,1233C832,1212 863,1186 888,1155C913,1124 933,1089 947,1049C960,1008 967,964 967,915M786,901C786,954 776,998 757,1033C737,1068 713,1095 684,1112C655,1129 625,1140 594,1145C563,1150 532,1152 503,1152l-159,0l0,-519l155,0C551,633 594,640 629,653C663,666 692,685 715,709C738,732 756,761 768,794C780,827 786,862 786,901z""/>
<glyph unicode=""R"" horiz-adv-x=""1112"" d=""M1035,26C1035,21 1034,16 1032,12C1030,8 1026,5 1019,2C1012,-1 1003,-3 991,-4C979,-5 963,-6 943,-6C926,-6 912,-5 901,-4C890,-3 881,0 874,3C867,6 862,10 859,15C855,20 852,27 849,35l-119,305C716,375 702,407 687,436C672,465 654,490 633,511C612,532 588,548 560,560C532,571 498,577 459,577l-115,0l0,-551C344,21 343,16 340,12C337,8 332,5 326,3C319,0 311,-2 300,-3C289,-5 275,-6 258,-6C241,-6 228,-5 217,-3C206,-2 197,0 190,3C183,5 179,8 176,12C173,16 172,21 172,26l0,1198C172,1250 179,1268 193,1279C206,1289 221,1294 236,1294l275,0C544,1294 571,1293 593,1292C614,1290 634,1288 651,1286C701,1277 745,1264 784,1245C822,1226 854,1203 880,1174C906,1145 926,1113 939,1076C952,1039 958,998 958,953C958,910 952,871 941,837C929,802 912,772 890,746C868,719 842,696 811,677C780,658 746,641 708,628C729,619 749,607 766,593C783,578 800,561 815,541C830,521 844,498 857,472C870,446 884,417 897,384l116,-285C1022,75 1028,58 1031,49C1034,39 1035,31 1035,26M776,935C776,986 765,1029 742,1064C719,1099 681,1124 628,1139C611,1144 593,1147 572,1149C551,1151 523,1152 489,1152l-145,0l0,-436l168,0C557,716 597,722 630,733C663,744 690,759 712,779C734,798 750,821 761,848C771,875 776,904 776,935z""/>
<glyph unicode=""S"" horiz-adv-x=""941"" d=""M867,363C867,302 856,248 834,201C811,154 780,114 741,81C701,48 654,23 601,6C547,-11 489,-19 427,-19C384,-19 344,-15 307,-8C270,-1 237,8 208,19C179,30 154,41 135,52C115,63 101,73 94,81C86,89 80,99 77,112C73,124 71,140 71,161C71,176 72,188 73,198C74,207 76,215 79,221C82,227 85,231 89,234C93,236 98,237 103,237C112,237 126,231 143,220C160,209 181,196 208,183C235,170 267,157 305,146C342,134 386,128 435,128C472,128 507,133 538,143C569,153 595,167 618,186C640,204 657,226 669,253C681,280 687,310 687,344C687,381 679,412 662,438C645,464 623,487 596,507C569,526 538,544 503,561C468,577 432,594 395,611C358,628 323,646 288,667C253,688 222,712 195,740C168,768 146,801 129,839C112,876 103,921 103,974C103,1028 113,1076 133,1119C152,1161 180,1196 215,1225C250,1254 291,1276 340,1291C388,1306 440,1313 496,1313C525,1313 554,1311 583,1306C612,1301 639,1294 665,1286C690,1277 713,1268 733,1258C753,1247 766,1239 773,1233C779,1226 783,1221 785,1218C787,1214 789,1209 790,1204C791,1198 792,1191 793,1183C794,1175 794,1165 794,1152C794,1140 794,1129 793,1120C792,1111 790,1103 788,1097C786,1090 783,1086 780,1083C776,1080 772,1078 767,1078C760,1078 748,1083 733,1092C717,1101 698,1112 675,1124C652,1135 626,1146 595,1156C564,1165 529,1170 490,1170C454,1170 423,1165 396,1156C369,1146 347,1133 330,1117C313,1101 300,1082 291,1060C282,1038 278,1015 278,990C278,954 286,923 303,897C320,871 342,848 370,828C397,808 429,790 464,773C499,756 535,740 572,723C609,706 645,687 680,667C715,646 747,622 775,595C802,567 825,534 842,497C859,459 867,414 867,363z""/>
<glyph unicode=""T"" horiz-adv-x=""998"" d=""M983,1220C983,1207 982,1196 981,1187C980,1178 978,1170 975,1165C972,1159 968,1155 964,1152C959,1149 954,1148 949,1148l-364,0l0,-1122C585,21 584,16 581,12C578,8 574,5 567,3C560,0 552,-2 541,-3C530,-5 516,-6 499,-6C483,-6 469,-5 458,-3C447,-2 438,0 431,3C424,5 420,8 417,12C414,16 413,21 413,26l0,1122l-364,0C44,1148 39,1149 35,1152C30,1155 27,1159 24,1165C21,1170 19,1178 18,1187C16,1196 15,1207 15,1220C15,1233 16,1244 18,1254C19,1263 21,1271 24,1277C27,1283 30,1287 35,1290C39,1293 44,1294 49,1294l900,0C954,1294 959,1293 964,1290C968,1287 972,1283 975,1277C978,1271 980,1263 981,1254C982,1244 983,1233 983,1220z""/>
<glyph unicode=""V"" horiz-adv-x=""1162"" d=""M695,30C692,23 689,17 684,12C679,7 673,4 665,1C656,-2 646,-3 633,-4C620,-5 603,-6 584,-6C569,-6 557,-6 546,-5C535,-5 525,-4 517,-3C509,-2 502,0 497,2C491,3 486,5 482,8C478,11 475,14 473,18C470,21 468,26 466,31l-423,1202C38,1248 35,1260 35,1269C34,1278 37,1284 43,1289C49,1294 59,1297 72,1298C85,1299 103,1300 126,1300C145,1300 160,1300 171,1299C182,1298 190,1296 196,1293C202,1290 207,1287 210,1283C213,1278 215,1273 218,1266l371,-1089l1,0l361,1087C953,1271 955,1277 958,1282C961,1287 965,1290 972,1293C979,1296 988,1298 1000,1299C1012,1300 1028,1300 1048,1300C1069,1300 1085,1299 1097,1298C1109,1296 1117,1293 1122,1288C1126,1283 1128,1277 1127,1268C1126,1259 1122,1247 1117,1232z""/>
<glyph unicode=""W"" horiz-adv-x=""1822"" d=""M1411,38C1408,29 1404,21 1399,16C1394,10 1386,6 1377,3C1368,0 1356,-3 1343,-4C1329,-5 1312,-6 1292,-6C1271,-6 1254,-5 1240,-4C1225,-3 1213,0 1204,3C1195,6 1188,10 1183,16C1178,21 1175,29 1172,38l-271,977l-2,0l-250,-977C647,29 644,21 639,16C634,10 628,6 619,3C610,0 599,-3 586,-4C572,-5 554,-6 533,-6C511,-6 493,-5 478,-4C463,-3 451,0 442,3C433,6 426,10 421,16C416,21 412,29 410,38l-342,1195C64,1248 62,1260 62,1269C61,1278 64,1284 70,1289C75,1294 84,1297 97,1298C110,1299 127,1300 148,1300C169,1300 185,1300 197,1299C209,1298 218,1296 225,1293C231,1290 235,1287 238,1282C240,1277 242,1271 243,1264l291,-1084l1,0l276,1082C813,1269 816,1276 819,1281C822,1286 826,1289 833,1292C840,1295 849,1297 862,1298C874,1299 890,1300 910,1300C929,1300 944,1299 955,1298C966,1297 974,1295 981,1292C988,1289 993,1286 996,1281C999,1276 1001,1269 1003,1262l297,-1082l2,0l285,1083C1589,1270 1591,1276 1593,1281C1595,1286 1599,1289 1606,1292C1612,1295 1621,1297 1633,1298C1644,1299 1660,1300 1679,1300C1699,1300 1715,1299 1727,1298C1738,1296 1747,1293 1752,1288C1757,1283 1760,1277 1760,1268C1759,1259 1757,1247 1754,1232z""/>
<glyph unicode=""a"" horiz-adv-x=""981"" d=""M831,24C831,16 828,10 823,6C818,2 810,-1 801,-3C792,-5 778,-6 760,-6C743,-6 729,-5 719,-3C708,-1 701,2 696,6C691,10 689,16 689,24l0,90C650,72 606,39 558,16C509,-7 458,-19 404,-19C357,-19 314,-13 276,0C237,12 205,30 178,53C151,76 130,105 115,139C100,173 92,212 92,255C92,306 102,350 123,387C144,424 173,455 212,480C251,505 298,523 354,536C410,548 473,554 543,554l124,0l0,70C667,659 663,689 656,716C649,743 637,765 621,783C604,800 583,814 557,823C531,832 499,836 461,836C420,836 384,831 352,822C319,812 291,801 267,790C242,778 222,767 206,758C189,748 177,743 169,743C164,743 159,744 155,747C151,750 148,754 145,759C142,764 139,771 138,780C137,788 136,797 136,807C136,824 137,837 140,847C142,856 148,865 157,874C166,883 181,893 203,905C225,916 250,927 279,937C308,946 339,954 373,961C407,967 441,970 476,970C541,970 596,963 641,948C686,933 723,912 751,884C779,855 799,820 812,778C825,736 831,687 831,631M667,435l-141,0C481,435 441,431 408,424C375,416 347,405 325,390C303,375 287,357 277,336C266,315 261,290 261,263C261,216 276,179 306,152C335,124 377,110 430,110C473,110 514,121 551,143C588,165 626,199 667,244z""/>
<glyph unicode=""b"" horiz-adv-x=""1076"" d=""M981,487C981,409 973,339 956,277C939,214 914,161 881,117C848,73 807,39 759,16C711,-7 656,-19 594,-19C565,-19 539,-16 515,-10C490,-5 466,4 443,17C420,30 396,46 373,65C350,84 325,108 299,135l0,-110C299,20 298,15 295,11C292,7 288,4 282,2C276,-1 269,-3 260,-4C251,-5 239,-6 225,-6C212,-6 200,-5 191,-4C182,-3 174,-1 168,2C162,4 158,7 156,11C154,15 153,20 153,25l0,1336C153,1366 154,1371 157,1375C159,1379 163,1382 170,1385C176,1388 184,1390 195,1391C206,1392 219,1393 235,1393C252,1393 265,1392 276,1391C287,1390 295,1388 301,1385C307,1382 311,1379 314,1375C317,1371 318,1366 318,1361l0,-539C345,849 371,872 396,891C421,910 445,925 469,937C493,948 517,957 541,962C565,967 590,970 617,970C682,970 738,957 785,931C831,905 869,870 898,827C927,783 948,732 961,673C974,614 981,552 981,487M807,468C807,514 804,559 797,602C790,645 777,684 760,717C743,750 720,777 691,798C662,818 627,828 584,828C563,828 542,825 521,819C500,813 479,803 458,789C437,775 415,757 392,735C369,713 344,685 318,652l0,-359C363,238 407,196 448,167C489,138 532,123 577,123C618,123 654,133 683,153C712,173 736,200 755,233C773,266 786,303 795,344C803,385 807,426 807,468z""/>
<glyph unicode=""c"" horiz-adv-x=""866"" d=""M800,162C800,151 800,141 799,133C798,124 797,117 796,112C794,106 792,101 790,97C787,92 782,86 774,78C765,69 751,59 731,47C711,34 689,23 664,14C639,4 611,-4 582,-10C553,-16 522,-19 491,-19C426,-19 369,-8 319,13C269,34 227,66 194,107C160,148 134,198 117,258C99,317 90,386 90,464C90,553 101,629 123,693C144,756 174,808 212,849C249,890 294,920 345,940C396,959 451,969 510,969C539,969 567,966 594,961C621,956 645,949 668,940C691,931 711,921 729,910C746,899 759,889 767,881C775,873 781,867 784,862C787,857 789,852 791,846C793,839 794,832 795,824C796,816 796,806 796,794C796,768 793,750 787,740C781,729 774,724 765,724C755,724 744,730 731,741C718,752 701,764 681,777C661,790 637,803 609,814C580,825 547,830 508,830C429,830 368,800 326,739C283,678 262,589 262,473C262,415 268,364 279,321C290,277 306,240 327,211C348,182 375,160 406,146C437,131 472,124 512,124C550,124 583,130 612,142C641,154 666,167 687,182C708,196 725,209 740,221C754,232 765,238 773,238C778,238 782,237 785,234C788,231 791,227 794,221C796,214 798,206 799,197C800,187 800,175 800,162z""/>
<glyph unicode=""d"" horiz-adv-x=""1076"" d=""M924,25C924,20 923,15 921,11C918,7 914,4 909,2C903,-1 895,-3 886,-4C877,-5 865,-6 852,-6C838,-6 826,-5 817,-4C808,-3 800,-1 794,2C788,4 784,7 781,11C778,15 777,20 777,25l0,119C730,93 681,53 630,24C579,-5 523,-19 462,-19C396,-19 340,-6 293,20C246,45 209,80 180,124C151,167 129,219 116,278C103,337 96,399 96,464C96,541 104,611 121,674C138,736 162,789 195,833C228,877 268,911 317,935C365,958 421,970 484,970C537,970 585,959 629,936C672,913 715,879 758,834l0,523C758,1362 759,1366 762,1371C764,1375 768,1378 775,1381C782,1383 790,1385 801,1387C811,1388 824,1389 840,1389C857,1389 870,1388 881,1387C892,1385 900,1383 906,1381C912,1378 917,1375 920,1371C923,1366 924,1362 924,1357M758,658C713,713 670,756 629,785C587,814 543,828 498,828C456,828 420,818 391,798C362,778 338,752 320,719C301,686 288,649 280,608C271,567 267,525 267,482C267,437 271,392 278,349C285,306 297,267 314,234C331,200 354,173 383,153C412,132 448,122 491,122C513,122 534,125 555,131C575,137 596,147 617,161C638,175 661,193 684,216C707,238 732,265 758,298z""/>
<glyph unicode=""e"" horiz-adv-x=""1019"" d=""M921,514C921,488 915,470 902,459C889,448 874,442 857,442l-590,0C267,392 272,347 282,307C292,267 309,233 332,204C355,175 386,153 423,138C460,123 506,115 560,115C603,115 641,119 674,126C707,133 736,140 761,149C785,158 805,166 821,173C836,180 848,183 856,183C861,183 865,182 869,180C872,177 875,174 877,169C879,164 881,158 882,150C883,141 883,131 883,119C883,110 883,103 882,97C881,90 881,85 880,80C879,75 877,70 875,66C872,62 869,58 866,55C862,51 851,45 833,37C815,28 792,20 763,12C734,4 701,-3 664,-9C626,-16 586,-19 543,-19C469,-19 404,-9 349,12C293,33 246,63 208,104C170,145 141,196 122,257C103,318 93,390 93,471C93,548 103,618 123,680C143,741 172,794 210,837C247,880 293,913 346,936C399,959 459,970 525,970C596,970 656,959 706,936C755,913 796,883 828,845C860,806 884,761 899,710C914,658 921,603 921,544M755,563C757,650 738,718 698,767C657,816 597,841 518,841C477,841 442,833 411,818C380,803 355,782 334,757C313,732 297,702 286,669C275,635 268,600 267,563z""/>
<glyph unicode=""f"" horiz-adv-x=""625"" d=""M646,1295C646,1282 645,1272 644,1265C643,1257 641,1251 640,1246C638,1241 636,1238 633,1237C630,1235 627,1234 623,1234C618,1234 613,1236 606,1239C599,1242 590,1245 580,1249C570,1252 558,1256 545,1259C531,1262 515,1263 497,1263C472,1263 451,1259 434,1251C417,1243 403,1231 392,1214C381,1197 374,1175 369,1148C364,1121 362,1088 362,1049l0,-98l201,0C568,951 573,950 577,948C580,945 584,941 587,936C590,930 592,923 594,914C595,905 596,895 596,882C596,858 593,841 587,830C581,819 573,814 563,814l-201,0l0,-789C362,20 361,15 358,12C355,8 351,5 345,2C338,-1 330,-3 319,-4C308,-5 295,-6 279,-6C263,-6 250,-5 239,-4C228,-3 220,-1 214,2C207,5 203,8 200,12C197,15 196,20 196,25l0,789l-127,0C58,814 50,819 45,830C40,841 37,858 37,882C37,895 38,905 39,914C40,923 42,930 45,936C48,941 51,945 55,948C59,950 64,951 69,951l127,0l0,93C196,1107 202,1161 214,1206C225,1251 243,1287 267,1316C291,1345 321,1366 358,1380C394,1393 437,1400 486,1400C509,1400 532,1398 554,1394C576,1389 593,1385 605,1380C617,1375 625,1370 629,1366C633,1362 636,1357 639,1351C642,1344 644,1337 645,1328C646,1319 646,1308 646,1295z""/>
<glyph unicode=""g"" horiz-adv-x=""964"" d=""M905,884C905,861 902,844 896,834C889,823 881,818 871,818l-129,0C765,794 782,768 791,739C800,710 805,679 805,648C805,596 797,550 780,510C763,470 740,436 709,409C678,381 641,360 598,345C555,330 508,323 456,323C419,323 385,328 352,338C319,347 293,359 275,374C263,362 253,348 246,333C238,318 234,300 234,280C234,257 245,237 267,222C288,207 317,198 353,197l235,-10C633,186 674,180 711,169C748,158 781,142 808,122C835,101 857,76 872,47C887,17 895,-18 895,-57C895,-98 886,-138 869,-175C852,-212 825,-245 790,-273C754,-302 708,-324 653,-340C598,-357 532,-365 457,-365C384,-365 323,-359 272,-346C221,-334 179,-317 146,-296C113,-275 90,-249 75,-219C60,-190 53,-158 53,-123C53,-101 56,-80 61,-59C66,-38 75,-19 86,0C97,19 110,37 127,54C143,71 162,87 183,104C150,121 126,142 111,167C95,192 87,220 87,249C87,290 95,326 112,358C129,390 149,419 174,444C153,469 137,497 125,528C113,559 107,596 107,640C107,691 116,737 133,778C150,819 174,853 205,881C236,909 273,931 316,946C359,961 405,968 456,968C483,968 509,967 533,964C556,961 578,956 599,951l272,0C882,951 891,946 897,935C902,924 905,907 905,884M647,647C647,708 630,756 597,791C563,825 515,842 453,842C421,842 393,837 370,826C346,815 326,801 311,782C295,763 283,742 276,718C268,693 264,668 264,641C264,582 281,535 315,501C348,467 395,450 456,450C489,450 517,455 541,466C565,476 585,490 601,509C616,527 628,548 636,572C643,596 647,621 647,647M730,-66C730,-27 714,3 683,24C651,45 608,56 554,57l-233,8C300,48 282,33 269,18C255,3 244,-12 236,-25C228,-39 223,-53 220,-66C217,-79 215,-93 215,-107C215,-150 237,-183 281,-205C325,-228 386,-239 465,-239C515,-239 557,-234 591,-224C624,-215 651,-202 672,-186C693,-170 708,-152 717,-131C726,-110 730,-89 730,-66z""/>
<glyph unicode=""h"" horiz-adv-x=""1076"" d=""M928,25C928,20 927,15 924,12C921,8 917,5 911,2C905,-1 897,-3 886,-4C875,-5 862,-6 846,-6C829,-6 816,-5 805,-4C794,-3 786,-1 780,2C774,5 770,8 767,12C764,15 763,20 763,25l0,527C763,603 759,645 751,676C743,707 731,734 716,757C701,780 681,797 657,809C632,821 604,827 572,827C531,827 489,812 448,783C407,754 363,711 318,654l0,-629C318,20 317,15 314,12C311,8 307,5 301,2C295,-1 287,-3 276,-4C265,-5 252,-6 235,-6C219,-6 206,-5 195,-4C184,-3 176,-1 170,2C163,5 159,8 157,12C154,15 153,20 153,25l0,1336C153,1366 154,1371 157,1375C159,1379 163,1382 170,1385C176,1388 184,1390 195,1391C206,1392 219,1393 235,1393C252,1393 265,1392 276,1391C287,1390 295,1388 301,1385C307,1382 311,1379 314,1375C317,1371 318,1366 318,1361l0,-539C365,872 413,909 461,934C509,958 557,970 606,970C666,970 717,960 758,940C799,919 832,892 857,858C882,824 901,784 912,739C923,693 928,638 928,573z""/>
<glyph unicode=""i"" horiz-adv-x=""470"" d=""M318,25C318,20 317,15 314,12C311,8 307,5 301,2C295,-1 287,-3 276,-4C265,-5 252,-6 235,-6C219,-6 206,-5 195,-4C184,-3 176,-1 170,2C163,5 159,8 157,12C154,15 153,20 153,25l0,900C153,930 154,934 157,938C159,942 163,945 170,948C176,951 184,953 195,954C206,955 219,956 235,956C252,956 265,955 276,954C287,953 295,951 301,948C307,945 311,942 314,938C317,934 318,930 318,925M337,1229C337,1190 330,1164 315,1150C300,1136 273,1129 234,1129C195,1129 169,1136 155,1150C140,1163 133,1189 133,1227C133,1266 140,1292 155,1306C170,1320 197,1327 236,1327C275,1327 301,1320 316,1307C330,1293 337,1267 337,1229z""/>
<glyph unicode=""k"" horiz-adv-x=""931"" d=""M892,26C892,21 891,16 888,12C885,8 881,5 875,2C868,-1 859,-3 848,-4C837,-5 822,-6 805,-6C787,-6 772,-5 761,-4C749,-3 739,-2 731,1C723,3 717,6 712,11C707,15 702,20 698,26l-380,498l0,-499C318,20 317,15 314,12C311,8 307,5 301,2C295,-1 287,-3 276,-4C265,-5 252,-6 235,-6C219,-6 206,-5 195,-4C184,-3 176,-1 170,2C163,5 159,8 157,12C154,15 153,20 153,25l0,1336C153,1366 154,1371 157,1375C159,1379 163,1382 170,1385C176,1388 184,1390 195,1391C206,1392 219,1393 235,1393C252,1393 265,1392 276,1391C287,1390 295,1388 301,1385C307,1382 311,1379 314,1375C317,1371 318,1366 318,1361l0,-814l340,374C663,928 669,933 675,938C681,942 688,946 697,949C705,952 715,954 726,955C737,956 751,956 767,956C784,956 798,956 809,955C820,954 830,952 837,950C844,947 849,944 852,941C855,937 856,932 856,926C856,918 854,910 850,902C845,894 838,885 828,874l-326,-326l366,-475C877,62 883,53 887,46C890,39 892,32 892,26z""/>
<glyph unicode=""l"" horiz-adv-x=""470"" d=""M318,25C318,20 317,15 314,12C311,8 307,5 301,2C295,-1 287,-3 276,-4C265,-5 252,-6 235,-6C219,-6 206,-5 195,-4C184,-3 176,-1 170,2C163,5 159,8 157,12C154,15 153,20 153,25l0,1336C153,1366 154,1371 157,1375C159,1379 163,1382 170,1385C176,1388 184,1390 195,1391C206,1392 219,1393 235,1393C252,1393 265,1392 276,1391C287,1390 295,1388 301,1385C307,1382 311,1379 314,1375C317,1371 318,1366 318,1361z""/>
<glyph unicode=""m"" horiz-adv-x=""1636"" d=""M1488,25C1488,20 1487,15 1484,12C1481,8 1477,5 1471,2C1465,-1 1457,-3 1446,-4C1435,-5 1422,-6 1406,-6C1389,-6 1376,-5 1365,-4C1354,-3 1346,-1 1340,2C1333,5 1329,8 1326,12C1323,15 1322,20 1322,25l0,547C1322,610 1319,645 1312,676C1305,707 1295,734 1280,757C1265,780 1247,797 1224,809C1201,821 1175,827 1144,827C1106,827 1068,812 1030,783C991,754 949,711 903,654l0,-629C903,20 902,15 899,12C896,8 892,5 886,2C879,-1 871,-3 860,-4C849,-5 836,-6 820,-6C805,-6 792,-5 781,-4C770,-3 761,-1 755,2C748,5 744,8 742,12C739,15 738,20 738,25l0,547C738,610 734,645 727,676C720,707 709,734 694,757C679,780 661,797 639,809C616,821 590,827 559,827C521,827 483,812 444,783C405,754 363,711 318,654l0,-629C318,20 317,15 314,12C311,8 307,5 301,2C295,-1 287,-3 276,-4C265,-5 252,-6 235,-6C219,-6 206,-5 195,-4C184,-3 176,-1 170,2C163,5 159,8 157,12C154,15 153,20 153,25l0,900C153,930 154,935 156,939C158,942 162,946 168,949C174,952 182,954 191,955C200,956 213,956 228,956C243,956 255,956 265,955C274,954 282,952 287,949C292,946 296,942 299,939C301,935 302,930 302,925l0,-119C353,863 402,904 450,931C497,957 545,970 594,970C631,970 665,966 695,957C724,948 750,936 773,921C796,905 815,886 831,865C847,843 860,819 871,792C901,825 930,852 957,875C984,898 1010,916 1035,930C1060,944 1084,954 1108,961C1131,967 1155,970 1179,970C1237,970 1286,960 1325,940C1364,919 1396,892 1421,858C1445,824 1462,784 1473,739C1483,693 1488,645 1488,594z""/>
<glyph unicode=""n"" horiz-adv-x=""1076"" d=""M928,25C928,20 927,15 924,12C921,8 917,5 911,2C905,-1 897,-3 886,-4C875,-5 862,-6 846,-6C829,-6 816,-5 805,-4C794,-3 786,-1 780,2C774,5 770,8 767,12C764,15 763,20 763,25l0,527C763,603 759,645 751,676C743,707 731,734 716,757C701,780 681,797 657,809C632,821 604,827 572,827C531,827 489,812 448,783C407,754 363,711 318,654l0,-629C318,20 317,15 314,12C311,8 307,5 301,2C295,-1 287,-3 276,-4C265,-5 252,-6 235,-6C219,-6 206,-5 195,-4C184,-3 176,-1 170,2C163,5 159,8 157,12C154,15 153,20 153,25l0,900C153,930 154,935 156,939C158,942 162,946 168,949C174,952 182,954 191,955C200,956 213,956 228,956C243,956 255,956 265,955C274,954 282,952 287,949C292,946 296,942 299,939C301,935 302,930 302,925l0,-119C353,863 403,904 454,931C504,957 555,970 606,970C666,970 717,960 758,940C799,919 832,892 857,858C882,824 901,784 912,739C923,693 928,638 928,574z""/>
<glyph unicode=""o"" horiz-adv-x=""1080"" d=""M990,485C990,412 980,344 961,283C942,221 913,168 875,123C836,78 788,44 731,19C673,-6 606,-19 530,-19C456,-19 392,-8 337,14C282,36 236,68 199,110C162,152 135,203 117,263C99,323 90,391 90,467C90,540 100,608 119,670C138,731 166,784 205,829C243,874 291,908 348,933C405,958 472,970 549,970C623,970 688,959 743,937C798,915 843,883 880,841C917,799 944,748 963,688C981,628 990,560 990,485M816,474C816,523 812,569 803,612C794,655 779,693 758,726C737,759 709,785 674,804C639,823 595,832 542,832C493,832 452,823 417,806C382,789 353,764 330,733C307,701 291,663 280,620C269,577 263,529 263,478C263,429 268,382 277,339C286,296 301,258 322,226C343,193 371,168 406,149C441,130 485,120 538,120C586,120 628,129 663,146C698,163 727,188 750,219C773,250 789,288 800,331C811,374 816,422 816,474z""/>
<glyph unicode=""p"" horiz-adv-x=""1076"" d=""M981,489C981,410 973,340 956,277C939,214 914,161 881,118C848,74 807,40 758,17C709,-7 654,-19 591,-19C564,-19 540,-16 517,-11C494,-6 472,3 451,14C429,25 407,40 386,57C365,74 342,95 318,118l0,-450C318,-337 317,-342 314,-346C311,-350 307,-353 301,-356C295,-359 287,-361 276,-362C265,-363 252,-364 235,-364C219,-364 206,-363 195,-362C184,-361 176,-359 170,-356C163,-353 159,-350 157,-346C154,-342 153,-337 153,-332l0,1257C153,931 154,936 156,940C158,943 162,946 168,949C174,952 182,954 191,955C200,956 212,956 225,956C239,956 251,956 260,955C269,954 276,952 282,949C288,946 292,943 295,940C297,936 298,931 298,925l0,-121C325,832 352,856 377,877C402,898 428,915 454,929C479,942 506,953 533,960C560,967 588,970 618,970C683,970 739,957 785,932C831,907 869,872 898,828C927,784 948,733 961,675C974,616 981,554 981,489M807,470C807,516 804,561 797,604C790,647 778,685 761,718C744,751 721,778 692,798C663,818 628,828 585,828C564,828 543,825 522,819C501,812 480,802 459,789C438,775 415,757 392,735C369,712 344,685 318,652l0,-358C363,239 406,196 447,167C488,138 530,123 575,123C616,123 652,133 682,153C711,173 735,200 754,233C772,266 785,304 794,345C803,386 807,428 807,470z""/>
<glyph unicode=""r"" horiz-adv-x=""714"" d=""M679,866C679,851 679,839 678,829C677,819 676,811 674,806C672,800 670,796 667,793C664,790 659,788 654,788C649,788 642,790 635,793C627,796 618,799 609,802C599,805 588,807 576,810C564,813 551,814 537,814C520,814 504,811 488,804C472,797 455,786 438,771C420,756 401,735 382,710C363,685 341,654 318,617l0,-592C318,20 317,15 314,12C311,8 307,5 301,2C295,-1 287,-3 276,-4C265,-5 252,-6 235,-6C219,-6 206,-5 195,-4C184,-3 176,-1 170,2C163,5 159,8 157,12C154,15 153,20 153,25l0,900C153,930 154,935 156,939C158,942 162,946 168,949C174,952 182,954 191,955C200,956 213,956 228,956C243,956 255,956 265,955C274,954 282,952 287,949C292,946 296,942 299,939C301,935 302,930 302,925l0,-131C327,830 350,859 372,882C393,905 414,923 433,936C452,949 472,958 491,963C510,968 529,970 548,970C557,970 567,970 578,969C589,968 600,966 612,963C624,960 635,957 645,954C654,951 661,947 665,944C669,941 672,938 673,935C674,932 676,928 677,923C678,918 678,912 679,903C679,894 679,881 679,866z""/>
<glyph unicode=""s"" horiz-adv-x=""801"" d=""M718,270C718,224 710,183 693,147C676,111 651,81 620,56C589,31 551,13 508,0C465,-13 417,-19 365,-19C333,-19 303,-16 274,-11C245,-6 219,0 196,8C173,15 153,23 137,32C121,40 109,47 102,54C95,61 89,70 86,82C83,94 81,110 81,131C81,144 82,154 83,163C84,172 86,179 88,184C90,189 93,193 97,196C100,198 104,199 109,199C116,199 127,195 142,186C156,177 174,167 195,156C216,145 240,136 269,127C298,118 331,113 368,113C396,113 421,116 444,122C467,128 486,137 503,149C520,160 533,175 542,193C551,211 555,232 555,257C555,282 549,304 536,321C523,338 505,354 484,367C463,380 439,392 412,403C385,413 358,424 330,435C301,446 274,459 247,474C220,488 195,505 174,526C153,547 136,571 123,600C110,629 103,663 103,703C103,738 110,772 124,805C137,837 158,865 185,890C212,914 247,933 288,948C329,963 376,970 431,970C455,970 479,968 503,964C527,960 549,955 568,949C587,943 604,937 618,930C631,923 642,916 649,911C656,906 660,901 663,897C665,893 667,889 668,884C669,879 670,872 671,865C672,858 672,849 672,838C672,827 672,817 671,809C670,800 668,793 666,788C663,783 660,779 657,777C654,774 650,773 646,773C640,773 631,777 620,784C609,791 594,799 576,808C558,816 537,824 513,831C488,838 460,842 429,842C401,842 376,839 355,833C334,826 316,817 303,806C289,794 279,780 272,764C265,748 261,731 261,712C261,686 268,664 281,647C294,629 312,613 333,600C354,587 379,575 406,564C433,553 461,542 490,531C518,520 546,507 574,493C601,479 626,462 647,442C668,422 686,398 699,370C712,342 718,309 718,270z""/>
<glyph unicode=""t"" horiz-adv-x=""686"" d=""M621,91C621,72 620,56 617,45C614,34 610,25 605,20C600,15 592,10 581,5C570,0 558,-3 545,-6C531,-9 516,-12 501,-14C486,-16 470,-17 455,-17C408,-17 368,-11 335,2C302,14 274,33 253,58C232,83 216,114 207,153C197,191 192,236 192,288l0,526l-126,0C56,814 48,819 42,830C36,841 33,858 33,882C33,895 34,905 36,914C37,923 39,930 42,936C45,941 48,945 53,948C57,950 62,951 67,951l125,0l0,214C192,1170 193,1174 196,1178C198,1182 202,1186 209,1189C215,1192 223,1194 234,1195C245,1196 258,1197 274,1197C291,1197 304,1196 315,1195C326,1194 334,1192 340,1189C346,1186 350,1182 353,1178C356,1174 357,1170 357,1165l0,-214l231,0C593,951 598,950 602,948C606,945 610,941 613,936C616,930 618,923 619,914C620,905 621,895 621,882C621,858 618,841 612,830C606,819 598,814 588,814l-231,0l0,-502C357,250 366,203 385,172C403,140 436,124 483,124C498,124 512,126 524,129C536,132 547,135 556,138C565,141 573,145 580,148C587,151 593,152 598,152C601,152 605,151 608,150C611,148 613,145 615,140C616,135 618,129 619,121C620,113 621,103 621,91z""/>
<glyph unicode=""u"" horiz-adv-x=""1076"" d=""M924,25C924,20 923,15 921,12C918,8 914,5 908,2C902,-1 894,-3 885,-4C875,-5 863,-6 849,-6C834,-6 821,-5 812,-4C802,-3 794,-1 789,2C783,5 779,8 777,12C775,15 774,20 774,25l0,119C723,87 672,46 622,20C572,-6 521,-19 470,-19C410,-19 360,-9 319,11C278,31 244,58 219,93C194,127 176,167 165,213C154,258 148,314 148,379l0,546C148,930 149,935 152,939C154,942 158,946 165,949C172,952 180,954 191,955C202,956 215,956 231,956C247,956 260,956 271,955C282,954 290,952 297,949C303,946 307,942 310,939C313,935 314,930 314,925l0,-524C314,348 318,306 326,275C333,243 345,216 361,194C376,171 396,154 420,142C444,129 472,123 504,123C545,123 587,138 628,167C669,196 712,239 758,296l0,629C758,930 759,935 762,939C764,942 768,946 775,949C782,952 790,954 801,955C811,956 824,956 841,956C857,956 870,956 881,955C892,954 900,952 906,949C912,946 917,942 920,939C923,935 924,930 924,925z""/>
<glyph unicode=""v"" horiz-adv-x=""925"" d=""M891,927C891,924 891,922 891,919C890,916 890,912 889,909C888,906 888,902 887,898C886,894 884,890 883,885l-307,-855C573,23 570,17 566,12C561,7 555,4 546,1C537,-2 526,-3 513,-4C500,-5 483,-6 463,-6C443,-6 426,-5 413,-4C400,-3 389,-1 381,2C372,5 366,8 361,13C356,18 353,23 350,30l-306,855C41,893 39,900 38,906C36,912 35,917 35,920C34,923 34,925 34,927C34,932 35,937 38,941C41,945 45,948 52,951C58,953 66,954 77,955C87,956 100,956 115,956C134,956 150,956 162,955C173,954 182,952 189,949C195,946 200,943 203,939C206,935 208,930 211,924l254,-742l4,-12l3,12l251,742C724,930 727,935 730,939C733,943 738,946 745,949C751,952 760,954 771,955C782,956 796,956 814,956C829,956 842,956 852,955C862,954 870,953 876,950C881,947 885,944 888,941C890,937 891,932 891,927z""/>
<glyph unicode=""w"" horiz-adv-x=""1464"" d=""M1412,927C1412,922 1411,917 1410,910C1409,903 1406,895 1403,885l-265,-855C1136,23 1133,17 1129,12C1124,7 1118,4 1111,1C1103,-2 1092,-3 1079,-4C1066,-5 1049,-6 1029,-6C1008,-6 991,-5 977,-4C963,-3 952,-1 944,2C935,5 929,8 925,13C921,18 918,23 916,30l-189,653l-2,9l-2,-9l-175,-653C546,23 543,17 539,12C534,7 528,4 519,1C510,-2 499,-3 486,-4C472,-5 455,-6 435,-6C414,-6 398,-5 385,-4C372,-3 361,-1 353,2C344,5 338,8 334,13C330,18 327,23 325,30l-263,855C59,895 56,903 55,910C54,917 53,922 53,927C53,933 54,938 57,942C60,945 64,948 71,951C77,953 85,954 96,955C107,956 120,956 135,956C154,956 169,956 180,955C191,954 200,952 206,950C212,947 216,944 219,940C222,935 224,930 226,924l217,-742l2,-9l2,9l199,742C647,930 650,935 653,940C656,944 660,947 666,950C672,952 680,954 691,955C701,956 714,956 731,956C747,956 760,956 771,955C781,954 789,952 795,950C801,947 805,944 808,940C811,936 813,931 814,926l215,-744l2,-9l1,9l214,742C1247,930 1250,935 1253,940C1256,944 1260,947 1267,950C1273,952 1282,954 1293,955C1304,956 1318,956 1335,956C1351,956 1364,956 1374,955C1383,954 1391,953 1397,950C1402,947 1406,944 1409,941C1411,938 1412,933 1412,927z""/>
<glyph unicode=""y"" horiz-adv-x=""927"" d=""M563,0l-120,-331C439,-342 429,-350 413,-355C396,-361 371,-364 338,-364C321,-364 307,-363 296,-361C285,-360 277,-357 272,-353C266,-349 263,-344 262,-337C261,-330 263,-322 267,-313l124,313C385,3 379,7 374,13C369,19 365,25 363,32l-321,860C37,906 34,917 34,925C34,933 37,939 42,944C47,949 56,952 68,954C80,955 96,956 116,956C136,956 152,956 163,955C174,954 183,952 190,949C197,946 202,943 205,938C208,933 211,926 214,917l257,-722l3,0l248,726C726,934 731,942 737,946C742,949 751,952 762,954C773,955 790,956 811,956C830,956 845,955 857,954C869,952 878,949 884,944C889,939 892,933 892,925C892,917 890,907 886,895z""/>
<glyph unicode="","" horiz-adv-x=""511"" d=""M371,143C371,122 370,103 368,86C366,69 363,52 358,37C353,22 347,7 340,-8C332,-23 322,-39 311,-56l-137,-206C171,-267 167,-271 163,-274C158,-277 153,-280 147,-283C140,-286 133,-287 125,-288C116,-289 106,-290 93,-290C82,-290 72,-289 65,-288C57,-287 51,-286 48,-283C45,-281 43,-278 43,-274C42,-270 43,-265 46,-260l133,275l0,128C179,161 181,175 185,186C188,197 194,205 202,210C210,215 220,219 232,221C244,223 259,224 276,224C293,224 307,223 319,221C331,219 341,215 349,210C356,205 362,197 366,186C369,175 371,161 371,143z""/>
<glyph unicode=""."" horiz-adv-x=""517"" d=""M366,109C366,62 358,30 343,15C327,-1 298,-9 257,-9C217,-9 189,-1 174,14C159,29 151,60 151,106C151,153 159,185 175,201C190,216 219,224 260,224C300,224 328,216 343,201C358,186 366,155 366,109z""/>
<glyph unicode=""1"" horiz-adv-x=""1038"" d=""M927,68C927,55 926,45 924,36C922,27 919,20 916,15C913,10 909,6 905,4C900,1 896,0 891,0l-676,0C210,0 206,1 202,4C198,6 194,10 191,15C187,20 184,27 182,36C180,45 179,55 179,68C179,80 180,90 182,99C184,108 187,115 190,121C193,126 196,131 201,134C205,137 210,138 215,138l270,0l0,984l-250,-149C222,966 212,962 205,961C197,960 191,961 186,966C181,970 178,977 177,987C175,997 174,1010 174,1025C174,1036 175,1046 176,1054C177,1062 178,1069 180,1074C182,1079 185,1084 189,1088C192,1092 197,1096 203,1100l298,191C504,1293 507,1295 511,1296C515,1297 520,1299 526,1300C532,1301 539,1302 547,1303C555,1303 565,1303 577,1303C593,1303 606,1302 617,1301C628,1300 636,1298 642,1296C648,1293 652,1290 654,1287C656,1283 657,1279 657,1275l0,-1137l234,0C896,138 901,137 906,134C911,131 915,126 918,121C921,115 923,108 925,99C926,90 927,80 927,68z""/>
<glyph unicode=""2"" horiz-adv-x=""1038"" d=""M924,74C924,62 923,51 922,42C920,33 918,25 915,19C912,12 908,8 903,5C898,2 892,0 886,0l-715,0C162,0 154,1 147,4C140,6 134,10 129,15C124,20 120,28 118,38C115,48 114,60 114,75C114,88 115,100 116,110C117,120 119,129 122,137C125,144 130,152 135,160C140,167 146,175 154,184l251,266C463,511 510,566 545,615C580,664 607,708 626,748C645,788 657,824 663,857C669,890 672,920 672,949C672,978 667,1005 658,1031C649,1056 635,1079 618,1098C600,1117 578,1133 551,1144C524,1155 494,1161 459,1161C418,1161 382,1155 350,1144C317,1133 289,1120 265,1107C240,1094 220,1081 204,1070C187,1059 175,1053 167,1053C162,1053 158,1054 155,1057C151,1060 148,1064 146,1070C143,1076 141,1084 140,1094C139,1104 138,1116 138,1130C138,1140 138,1149 139,1156C140,1163 141,1170 143,1175C144,1180 146,1185 149,1190C152,1195 157,1201 165,1208C173,1215 187,1224 207,1236C226,1248 251,1260 281,1272C310,1283 343,1293 379,1301C414,1309 452,1313 491,1313C554,1313 609,1304 656,1287C703,1269 742,1245 773,1214C804,1183 827,1148 842,1107C857,1066 865,1023 865,977C865,936 861,895 854,854C847,813 831,768 808,721C784,673 750,620 705,562C660,503 601,436 527,360l-205,-214l563,0C890,146 896,145 901,142C906,139 910,134 914,128C917,122 920,115 922,106C923,97 924,86 924,74z""/>
<glyph unicode=""3"" horiz-adv-x=""1038"" d=""M914,370C914,311 904,257 883,210C862,162 833,121 794,87C755,53 708,27 651,9C594,-10 530,-19 459,-19C416,-19 375,-15 338,-8C300,-1 266,7 237,17C208,26 183,36 164,47C145,57 133,64 128,69C123,74 119,78 116,83C113,88 111,93 109,100C107,107 106,115 105,125C104,134 103,146 103,160C103,184 105,201 110,210C115,219 121,224 130,224C136,224 148,219 166,208C183,197 206,186 234,174C261,161 294,150 331,139C368,128 409,123 454,123C498,123 537,129 570,140C603,151 631,167 654,188C677,209 694,233 705,262C716,290 722,321 722,354C722,391 715,424 701,453C686,482 665,508 638,529C610,550 576,567 536,578C495,589 449,595 398,595l-123,0C270,595 265,596 260,599C255,601 250,605 247,610C243,615 240,622 238,631C235,640 234,651 234,665C234,678 235,688 237,697C239,705 242,712 246,717C249,722 253,725 258,728C263,730 268,731 274,731l113,0C431,731 471,737 506,748C541,759 570,776 595,797C619,818 638,844 651,874C664,903 670,936 670,973C670,1000 666,1025 657,1050C648,1074 635,1095 618,1113C601,1131 579,1145 552,1156C525,1166 493,1171 458,1171C419,1171 384,1165 352,1154C319,1142 290,1129 265,1116C240,1103 219,1090 202,1079C185,1067 174,1061 167,1061C162,1061 158,1062 155,1064C151,1065 148,1069 146,1074C143,1079 142,1086 141,1095C140,1104 139,1115 139,1130C139,1140 139,1149 140,1157C141,1164 142,1171 144,1177C146,1182 148,1187 151,1192C154,1197 158,1202 165,1209C171,1215 184,1224 203,1236C222,1248 246,1260 275,1272C303,1283 336,1293 373,1301C410,1309 449,1313 492,1313C552,1313 605,1305 650,1290C695,1275 733,1253 764,1225C794,1197 817,1164 832,1125C847,1086 854,1042 854,995C854,954 849,917 838,882C827,847 812,816 791,789C770,762 745,739 715,720C685,701 650,687 611,680l0,-2C656,673 697,662 734,645C771,627 802,605 829,578C856,551 877,520 892,485C907,449 914,411 914,370z""/>
<glyph unicode=""4"" horiz-adv-x=""1038"" d=""M979,368C979,346 976,329 969,316C962,303 953,297 941,297l-146,0l0,-272C795,20 794,15 791,12C788,8 784,5 777,2C770,-1 762,-3 751,-4C740,-5 727,-6 710,-6C694,-6 681,-5 670,-4C659,-3 651,-1 645,2C638,5 634,8 632,12C629,15 628,20 628,25l0,272l-534,0C86,297 79,298 74,300C69,302 64,306 60,311C55,316 52,324 51,334C49,344 48,357 48,373C48,386 48,397 49,407C50,417 51,426 53,435C55,443 58,451 61,459C64,466 68,474 73,483l466,786C542,1274 547,1279 554,1283C560,1287 568,1290 578,1293C588,1296 600,1298 615,1299C630,1300 647,1300 667,1300C689,1300 708,1299 725,1298C741,1296 754,1294 764,1291C774,1288 782,1285 787,1281C792,1276 795,1271 795,1265l0,-826l146,0C952,439 962,433 969,422C976,410 979,392 979,368M628,1149l-2,0l-421,-710l423,0z""/>
<glyph unicode=""5"" horiz-adv-x=""1038"" d=""M918,416C918,347 906,285 883,231C860,177 827,132 784,95C741,58 691,29 632,10C573,-9 507,-19 435,-19C395,-19 357,-16 321,-10C285,-5 253,2 225,11C197,19 174,27 156,35C138,43 127,49 122,54C117,59 113,63 111,67C109,71 107,76 106,82C104,87 103,94 102,103C101,112 101,122 101,134C101,145 102,156 103,165C104,174 105,181 108,187C111,192 114,196 118,199C121,202 125,203 130,203C137,203 147,199 161,191C175,182 194,173 218,164C241,154 271,145 306,136C341,127 382,123 430,123C475,123 516,128 553,139C590,150 623,166 650,189C677,212 698,240 713,275C728,309 736,351 736,400C736,441 730,478 717,510C704,541 684,568 657,589C630,610 595,625 554,636C513,646 463,651 406,651C365,651 330,649 301,645C272,641 244,639 219,639C200,639 187,643 179,652C171,661 167,677 167,702l0,527C167,1251 172,1267 183,1278C193,1289 208,1294 227,1294l573,0C805,1294 811,1293 816,1290C821,1287 825,1282 828,1276C831,1270 834,1262 836,1253C837,1244 838,1233 838,1220C838,1196 835,1177 828,1164C821,1151 812,1144 800,1144l-483,0l0,-363C340,784 364,786 389,787C413,788 441,788 473,788C547,788 612,779 667,762C722,744 769,719 806,687C843,654 871,615 890,569C909,523 918,472 918,416z""/>
<glyph unicode=""6"" horiz-adv-x=""1038"" d=""M957,419C957,362 948,308 931,255C914,202 887,155 851,115C815,74 769,42 714,18C659,-7 593,-19 518,-19C464,-19 416,-12 375,1C334,14 298,33 267,58C236,83 211,113 191,149C170,185 154,226 142,272C129,318 120,368 115,423C110,478 107,536 107,599C107,654 110,710 116,767C122,824 133,878 148,931C163,984 184,1033 211,1080C237,1126 270,1166 310,1201C349,1235 397,1262 452,1282C507,1302 572,1312 645,1312C670,1312 695,1311 720,1308C745,1305 769,1301 790,1296C811,1291 829,1285 844,1279C859,1273 868,1268 873,1265C878,1261 881,1257 884,1253C886,1248 888,1244 889,1239C890,1234 891,1228 892,1222C893,1215 893,1207 893,1198C893,1186 893,1176 893,1167C892,1158 891,1151 889,1146C886,1141 883,1137 880,1134C876,1131 871,1130 865,1130C858,1130 848,1132 835,1137C822,1142 807,1147 789,1153C771,1158 749,1163 724,1168C698,1173 668,1175 634,1175C572,1175 519,1162 474,1137C429,1112 393,1078 365,1035C336,992 315,943 302,887C288,830 280,771 278,710C295,720 315,730 337,740C358,750 382,759 408,767C433,775 460,782 489,787C518,792 548,794 581,794C651,794 710,785 758,766C806,747 845,720 875,687C904,653 925,613 938,568C951,522 957,472 957,419M783,403C783,442 779,477 772,509C764,540 751,567 733,589C715,611 691,628 661,640C631,652 594,658 550,658C525,658 501,656 476,652C451,647 428,641 405,634C382,626 360,617 339,607C318,596 298,585 281,573C281,487 287,415 298,357C309,299 325,253 346,218C367,183 393,158 424,143C455,128 492,120 534,120C577,120 614,128 645,144C676,160 702,182 723,209C744,236 759,266 769,300C778,333 783,368 783,403z""/>
</font>
";

            //create the final svg that will be displayed
            var svgTotalWidth = svgWidth + 10; //add little for wiggle room
            var svgBody = $"<svg class=\"{svgClass}\" id=\"{randomId}\"" +
                          //$" width=\"100%\"" +
                          //$" height=\"100%\"" +
                          $" style=\"" +
                          //note: if width & height not hard set, parent div clips it
                          $"width:{svgTotalWidth}px;" +
                          $"height:{svgTotalHeight}px;" +
                          $"background:{svgBackgroundColor};" +
                          $"\" " +//end of style tag
                          $"xmlns=\"http://www.w3.org/2000/svg\" " +
                          $"xmlns:xlink=\"http://www.w3.org/1999/xlink\">" + //much needed for use tags to work
                                                                             //$"<title>{chartTitle}</title>" + //title visible in browser when open direct
                          $"{combinedSvgString}</svg>";

            return svgBody;
        }




        //▒█▀▀█ ▒█▀▀█ ▀█▀ ▒█░░▒█ ░█▀▀█ ▀▀█▀▀ ▒█▀▀▀ 
        //▒█▄▄█ ▒█▄▄▀ ▒█░ ░▒█▒█░ ▒█▄▄█ ░▒█░░ ▒█▀▀▀ 
        //▒█░░░ ▒█░▒█ ▄█▄ ░░▀▄▀░ ▒█░▒█ ░▒█░░ ▒█▄▄▄



        private static string GetJsCodeSvg(string randomId)
        {
            string jsCode = "";

            //add last to load last
            //load order is important
            jsCode += "<script href=\"https://code.jquery.com/jquery-3.6.3.min.js\" />";
            jsCode += "<script href=\"https://cdn.jsdelivr.net/npm/@svgdotjs/svg.js@3.0/dist/svg.min.js\" />";//used in events chart inside js
            jsCode += "<script href=\"https://www.vedastro.org/js/VedAstro.js\" />";

            //random id is created here to link svg element with JS instance
            jsCode += $@"
	                    <script type=""text/ecmascript"">
		                    <![CDATA[

                               //START THE BEAST
		                       var index = new EventsChart(""{randomId}"");
			                ]]>
	                    </script>
                            ";

            return jsCode;
        }

        private static string GetBorderSvg(List<Time> timeSlices, int svgTotalHeight)
        {
            //save a copy of the number of time slices used to calculate the svg total width later
            var dasaSvgWidth = timeSlices.Count;

            //add border around svg element
            //note:compensate for padding, makes border fit nicely around content
            var borderWidth = dasaSvgWidth + 2; //contentPadding = 2 todo centralize
            var roundedBorder = 3;
            svgTotalHeight += 10; //adjust
            var compiledRow = $"<rect class=\"EventChartBorder\" rx=\"{roundedBorder}\" width=\"{borderWidth}\" height=\"{svgTotalHeight}\" style=\"stroke-width: 2; fill: none; paint-order: stroke; stroke:#333;\"></rect>";

            return compiledRow;
        }

        private static string MakeNowLine(Time startTime, int verticalYAxis, List<Time> timeSlices)
        {
            //the height for all lines, cursor, now & life events line
            //place below the last row
            var nowLineY = verticalYAxis + 2; //space between rows


            //get now line position
            var clientNowTime = startTime.StdTimeNowAtOffset; //now time at client
            var nowLinePosition = GetLinePosition(timeSlices, clientNowTime);
            //skip now line if beyond chart on both sides
            var beyondLimit = nowLinePosition == 0 || nowLinePosition == (timeSlices.Count - 1);
            var nowLineHeight = nowLineY + 6; //space between icon & last row
            var nowLineSvgGroup = GetNowLine(nowLineHeight, nowLinePosition, beyondLimit); //hides if beyond limit


            return nowLineSvgGroup;
        }

        /// <summary>
        /// Get color based on nature
        /// </summary>
        private static string GetColor(EventNature? eventNature) => GetColor(eventNature.ToString());

        /// <summary>
        /// Get color based on event same summary color
        /// return in CSS #FFFFFF
        /// </summary>
        public static string GetColor(double value, double min = -2, double max = 2)
        {
            if (value < min || value > max)
            {
                throw new ArgumentOutOfRangeException(nameof(value), "Value must be within the range of min and max.");
            }

            // Normalize the value to a 0-1 range
            double normalizedValue = (value - min) / (max - min);

            // Calculate the RGB components based on the normalized value
            int r = (int)(255 * Math.Max(0, 1 - normalizedValue));
            int g = (int)(255 * Math.Max(0, normalizedValue));
            int b = 0;

            // Convert the RGB components to a CSS color string
            string color = $"#{r:X2}{g:X2}{b:X2}";

            return color;
        }

        /// <summary>
        /// Get color based on nature
        /// </summary>
        private static string GetColor(string nature)
        {
            switch (nature)
            {
                case "Good": return "#00FF00"; //green
                case "Neutral": return "#D3D3D3"; //grey
                case "Bad": return "#FF0000"; //red
                default: throw new Exception($"Invalid Nature: {nature}");
            }
        }

        /// <summary>
        /// return in CSS #FFFFFF
        /// convert value coming in, to a percentage based on min & max
        /// this will make setting color based on value easier & accurate
        /// note, this will cause color to be relative to only what is visible in chart atm!
        /// </summary>
        private static string GetSummaryColor(double totalNature, double minValue, double maxValue)
        {

            string colorHex;
            //GOOD
            if (totalNature >= 0)
            {
                //note: higher number is lighter color lower is darker
                var color255 = (int)totalNature.Remap(0, maxValue, 0, 255);
                if (color255 <= 10) { return "#FFFFFF"; } //if very close to 0, return greenish white color (lighter)
                if (color255 >= 220) { return "#00DD00"; } //if very close to 255, return darker green (DD) to highlight

                //this is done to invert the color,
                //so that higher score leads to darker green
                color255 = 255 - color255;

                colorHex = $"{color255:X}";//convert from 255 to FF
                var summaryColor = $"#{colorHex}FF{colorHex}";
                return summaryColor; //green
            }
            //BAD
            else
            {
                //note: colors here are rendered opposite compared to good
                //because lower number here is worse, as such needs to be dark red.
                var color255 = (int)totalNature.Remap(minValue, 0, 0, 255);
                if (color255 <= 10) { return "#DD0000"; } //if very close to 255, return darker red (DD) to highlight
                if (color255 >= 220) { return "#FFFFFF"; } //if very close to 0, return white color
                colorHex = $"{color255:X}";//convert from 255 to FF
                var summaryColor = $"#FF{colorHex}{colorHex}";
                return summaryColor; //red
            }

        }

        /// <summary>
        /// Generates individual life event line svg
        /// Incrementing line height also increments icon position below
        /// </summary>
        private static string GenerateLifeEventLine(LifeEvent lifeEvent, int lineHeight, DateTimeOffset lifeEvtTime, int positionX, out int totalHeight)
        {
            //shorten the event name if too long & add ellipsis at end
            //else text goes outside box
            var formattedEventName = ShortenName(lifeEvent.Name, 14);

            int iconYAxis = lineHeight + 7; //start icon at end of line + 7 padding
            int verticalLineHeight = iconYAxis + 2; //extra px added cause line moved up -2 
            var iconXAxis = $"-7.5"; //use negative to move center under the line
            var nameTextHeight = 15;

            //only show name label for Major and Normal events
            var isMajor = lifeEvent.Weight == "Major";
            var isNormal = lifeEvent.Weight == "Normal";
            var isShowName = isNormal || isMajor; //show name for normal and major only
            var evtNameStyle = isShowName ? "" : "display: none;";

            //display description only for major events and when text not Empty
            var notEmpty = !string.IsNullOrEmpty(lifeEvent.Description);
            var descriptionDisplayStyle = isMajor && notEmpty ? "" : "display:none;";

            //convert event description text to image 
            string descriptionTextSvg = StringToSvgTextBox(lifeEvent.Description, out var boxHeightPx);

            //set width of event name box based on if desc box is shown below (only shows for Major type)
            var withoutDescBox = (GetTextWidthPx(formattedEventName) + 14);
            var withDescBox = 82;  //82 = 100 - icon = text space left
            var nameBackgroundWidth = isMajor ? withDescBox : withoutDescBox;

            //background height set according to text
            var descBackgroundHeight = boxHeightPx + 5; //5 for padding

            //set total height so caller can mark occupied space
            totalHeight = (isMajor ? descBackgroundHeight : 0) + nameTextHeight;

            var iconSvg = $@"
                                <!-- POINTER LINE -->
                                <rect class=""vertical-line"" y=""-2"" fill=""#1E1EEA"" width=""2"" height=""{verticalLineHeight}""></rect>

                                <!-- EVENT ICON LABEL -->
                                <g transform=""translate({iconXAxis},{iconYAxis})"">
                                    <g class=""name-label"" style="" cursor: pointer; "" >
                                        <!-- EVENT NAME-->
                                        <g transform=""translate(18,0)"" style=""{evtNameStyle}"">
						                    <rect class=""background"" x=""0"" y=""0"" style=""fill: blue; opacity: 0.8;"" width=""{nameBackgroundWidth}"" height=""{nameTextHeight}"" rx=""2"" ry=""2"" />
						                    <text class=""event-name"" x=""3"" y=""11"" style=""fill:#FFFFFF; font-size:12px;"">{HttpUtility.HtmlEncode(formattedEventName)}</text>
					                    </g>

                                        <!-- EVENT ICON-->
				                        <g class=""nature-icon"" transform=""translate(8,8)"">
                                            <rect class=""background"" fill=""{GetColor(lifeEvent.Nature)}"" x=""-7.5"" y=""-8"" width=""15"" height=""15"" rx=""2"" ry=""2""/>
					                        <path d=""M2-0.2H0.7C0.5-0.2,0.3,0,0.3,0.2v1.3c0,0.2,0.2,0.4,0.4,0.4H2c0.2,0,0.4-0.2,0.4-0.4V0.2C2.5,0,2.3-0.2,2-0.2z M2-4.5v0.4 h-3.6v-0.4c0-0.2-0.2-0.4-0.4-0.4c-0.2,0-0.4,0.2-0.4,0.4v0.4h-0.4c-0.5,0-0.9,0.4-0.9,0.9v6.1c0,0.5,0.4,0.9,0.9,0.9h6.3 c0.5,0,0.9-0.4,0.9-0.9v-6.1c0-0.5-0.4-0.9-0.9-0.9H3.1v-0.4c0-0.2-0.2-0.4-0.4-0.4S2-4.8,2-4.5z M2.9,2.9h-5.4 c-0.2,0-0.4-0.2-0.4-0.4v-4.4h6.3v4.4C3.4,2.7,3.2,2.9,2.9,2.9z""/>
				                        </g>
                                    </g>

                                    <!-- DESCRIPTION -->
		                            <g class=""description-label"" style=""{descriptionDisplayStyle}"" transform=""translate(0,{17})"">
                                        <rect class=""background"" style=""fill: blue; opacity: 0.8;"" width=""{DescriptionBackgroundWidth}"" height=""{descBackgroundHeight}"" rx=""2"" ry=""2""/>
                                        <text transform=""translate(2,11)"">
                                            {descriptionTextSvg}
			                            </text>
		                            </g>
                                </g>
                                ";

            //put together icon + line + event data
            var lifeEventLine = $@"<g eventname=""{lifeEvent.DescriptionHtmlSafe}"" eventweight=""{lifeEvent.Weight}"" class=""LifeEventLines"" stdtime=""{lifeEvtTime:dd/MM/yyyy}"" 
                              transform=""translate({positionX}, 0)"">{iconSvg}</g>";

            return lifeEventLine;


        }

        /// <summary>
        /// Given a text will shorten it with ellipsis at the back
        /// </summary>
        private static string ShortenName(string rawInput, int nameCharLimit)
        {
            //if no changes needed return as is (default)
            var returnVal = rawInput;
            //const int nameCharLimit = 14; //any number of char above this limit will be replaced with  ellipsis  "..."

            //if input is above limit
            //replace extra chars with ...
            var isAbove = rawInput.Length > nameCharLimit;
            if (isAbove)
            {
                //cut the extra chars out
                returnVal = returnVal.Substring(0, nameCharLimit);

                //add ellipsis 
                returnVal += "...";
            }

            return returnVal;
        }

        private static double GetTextWidthPx(string textInput)
        {

            SizeF size;
            using (var graphics = Graphics.FromHwnd(IntPtr.Zero))
            {
                size = graphics.MeasureString(textInput,
                    new Font(
                        familyName: "Calibri",
                        emSize: 10,
                        style: FontStyle.Regular,
                        unit: GraphicsUnit.Pixel));
            }

            var widthPx = Math.Round(size.Width);

            return widthPx;
        }

        /// <summary>
        /// Input sting to get nicely wrapped text in SVG
        /// Note: this is only an invisible structured text, other styling have to be added separately
        /// </summary>
        /// <returns></returns>
        private static string StringToSvgTextBox(string inputStr, out int boxHeightPx)
        {
            //shorten text length & add ellipsis at end
            inputStr = ShortenName(inputStr, 44);

            const int characterLimit = 22;
            //chop string to pieces, to wrap text nicely (new line)
            var splitStringRaw = Tools.SplitByCharCount(inputStr, characterLimit);

            //convert raw string to be HTML safe (handles special chars like &,!,%)
            var splitStringRaw2 = splitStringRaw.Select(rawStr => HttpUtility.HtmlEncode(rawStr)).ToList();

            //remove white space characters at the front
            var splitString = splitStringRaw2.Select(x => x.Trim()).ToList();

            //NOTE : ellipses is 3 chars but takes space of 1, hence below logic
            //if last string is ellipses by itself, then push it up (else a new row just for ellipses, not nice)
            if (splitString.Any() && splitString.Last() == "...")
            {
                //remove last
                splitString.RemoveAt(splitString.Count - 1);

                //add ellipses to back of last text (3 chars but takes space of 1)
                splitString[^1] += "...";
            }

            //make a new holder for each line
            var compiledSvgLines = "";
            const int lineHeight = 12;
            var nextYAxis = 0; //start with 0, at top
            foreach (var strLine in splitString)
            {
                var newLineSvg = $@"<tspan x=""0"" y=""{nextYAxis}"" fill=""#FFFFFF"" font-size=""10"">{strLine}</tspan>";

                //add together with other lines
                compiledSvgLines += newLineSvg;

                //set next line to come underneath this (increment y axis)
                nextYAxis += lineHeight;
            }

            //save to be used to draw border around
            boxHeightPx = nextYAxis;

            return compiledSvgLines;
        }

        /// <summary>
        /// Returns a person's life events in SVG group to be placed inside events chart
        /// gets person's life events as lines for the events chart
        /// </summary>
        private static string GetLifeEventLinesSvg(Person person, int verticalYAxis, Time startTime, List<Time> timeSlices)
        {
            //each time slice is 1 px in width
            //NOTE: add extra for life event boxes on the edges
            //that expand more in virtual space than rendered
            var maxSlices = timeSlices.Count + 1000;

            // place to keep track of occupied space represents pixels (x-axis, y-axis)
            bool[,] virtualSpace = new bool[maxSlices, MaxLifeEventCanvasHeight];

            //use offset of input time, this makes sure life event lines
            //are placed on event chart correctly, since event chart is based on input offset
            var lineHeight = verticalYAxis + 9; //space between icon & last row
            var inputOffset = startTime.GetStdDateTimeOffset().Offset; //timezone the chart will be in

            //sort by earliest to latest event
            var lifeEventList = person.LifeEventList;
            lifeEventList.Sort((x, y) => x.CompareTo(y));

            var listRowData = new List<string>();
            foreach (var lifeEvent in lifeEventList)
            {
                //get timezone at place event happened
                var lifeEvtTime = lifeEvent.StartTime.GetStdDateTimeOffset();//time at the place of event with correct standard timezone
                var startTimeInputOffset = lifeEvtTime.ToOffset(inputOffset); //change to output offset, to match chart
                var positionX = GetLinePosition(timeSlices, startTimeInputOffset);

                //if line is not in report time range, don't generate it
                if (positionX == 0) { continue; }

                //no print minor events
                if (lifeEvent.Weight == "Minor") { continue; }

                //gets row number (pixel), that is free to occupy
                var relativeRowNumber = GetRowNumber(positionX); //start at 0 index

                //add line height to avoid chart above to row number (y position)
                var finalYPosition = relativeRowNumber + lineHeight;

                //put together icon + line + event data
                var generateLifeEventLine = GenerateLifeEventLine(lifeEvent, finalYPosition, lifeEvtTime, positionX, out var lifeEventHeight);

                //mark as occupied for future ref
                MarkRowNumber(positionX, relativeRowNumber, DescriptionBackgroundWidth, lifeEventHeight, 15, 15);

                //save position Y if it is the highest so far, save it to be used to draw final SVG height
                MaxYAxis = finalYPosition > MaxYAxis ? finalYPosition : MaxYAxis;

                //add empty row before if 1st
                while (relativeRowNumber > (listRowData.Count - 1)) { listRowData.Add(""); }

                //save the SVG under its row number with others to be placed z-index wise later
                listRowData[relativeRowNumber] += generateLifeEventLine;

            }

            //place each row in a group and add to final list
            //NOTE:
            //we stack the row from last to first, so that the top most row, is painted last,
            //thus appearing above the lines of the events below
            var finalSvg = "";
            int rowNum = (listRowData.Count - 1); //0 index
            for (; rowNum >= 0; rowNum--)
            {
                var rowEventIcons = listRowData[rowNum];
                var wrap = $@"<g id=""row{rowNum}"">{rowEventIcons}</g>";
                finalSvg += wrap;

            }

            //wrap in a group so that can be hidden/shown as needed
            //add transform matrix to adjust for border shift
            const int contentPadding = 2;
            var wrapperGroup = $"<g id=\"LifeEventLinesHolder\" transform=\"matrix(1, 0, 0, 1, {contentPadding}, {contentPadding})\">{finalSvg}</g>";

            return wrapperGroup;

            //-------------------------

            void MarkRowNumber(int startCol, int startRow, int widthPx, int heightPx, int verticalPadding, int horizontalPadding)
            {
                //mark all occupied space
                var lastColumn = startCol + widthPx + horizontalPadding;
                var lastRow = startRow + heightPx + verticalPadding;

                //mark all columns
                for (int colIndex = startCol; colIndex <= lastColumn; colIndex++)
                {
                    //mark all rows
                    for (int rowIndex = startRow; rowIndex <= lastRow; rowIndex++)
                    {
                        //mark true for occupied
                        virtualSpace[colIndex, rowIndex] = true;
                    }
                }

            }

            //given a column number will find row in it that is free (row = pixel)
            //start at 0 index
            int GetRowNumber(int columnIndex)
            {

                //scan till find row that is empty at fixed column (x-axis)
                for (int rowIndex = 0; rowIndex < MaxLifeEventCanvasHeight; rowIndex++)
                {
                    var isOccupied = virtualSpace[columnIndex, rowIndex];

                    //not occupied with padding included return as available number
                    if (!isOccupied) { return rowIndex; }
                }

                throw new Exception("End of Virtual Rows!");
            }

        }

        /// <summary>
        /// add in the cursor line (moves with cursor via JS)
        /// note: template cursor line is duplicated to dynamically generate legend box
        /// </summary>
        private static string GetTimeCursorLine(int verticalYAxis)
        {
            //add in the cursor line (moves with cursor via JS)
            //note: - template cursor line is duplicated to dynamically generate legend box
            //      - placed last so that show on top of others
            var lineHeight = verticalYAxis + 8;//minor adjust


            return $@"
                        <g id=""CursorLine"" x=""0"" y=""0"">
                            <!--place where asset icons are stored-->
                            <g id=""CursorLineLegendIcons"" style=""display:none;"">
                                <circle id=""CursorLineBadIcon"" cx=""6.82"" cy=""7.573"" r=""4.907"" fill=""red""></circle>
                                <circle id=""CursorLineGoodIcon"" cx=""6.82"" cy=""7.573"" r=""4.907"" fill=""green""></circle>
                                <g id=""CursorLineClockIcon"" fill=""#fff"" transform=""matrix(0.5, 0, 0, 0.5, 2, 3)"" width=""12"" height=""12"">
				                    <path d=""M10 0a10 10 0 1 0 10 10A10 10 0 0 0 10 0zm2.5 14.5L9 11V4h2v6l3 3z""/>
			                    </g>
                                <path id=""CursorLineSumIcon"" transform=""matrix(0.045, 0, 0, 0.045, -14, -4)"" fill=""#fff"" d=""M437 122c-15 2-26 5-38 10-38 16-67 51-75 91-4 17-4 36 0 54 10 48 47 86 95 98 11 2 15 3 30 3s19-1 30-3c48-12 86-50 96-98 3-18 3-37 0-54-10-47-48-86-95-98-10-2-16-3-29-3h-14zm66 59c2 2 3 3 4 6s1 17 0 20c-2 7-11 9-15 2-1-2-1-3-1-7v-5h-37-37s8 11 18 23l21 25c1 2 1 5 1 7-1 1-10 13-21 26l-19 24c0 1 13 1 37 1h37v-5c0-6 1-9 5-11 5-2 11 1 11 8 1 1 1 6 1 10-1 7-1 8-2 10s-3 4-7 4h-52-50l-2-1c-4-3-5-7-3-11 0 0 11-14 24-29l22-28-22-28c-13-16-24-29-24-30-2-3-1-7 2-9 2-3 2-3 55-3h51l3 1z"" stroke=""none"" fill-rule=""nonzero""/>
                            </g>

                            <g id=""IBeam"">
			                    <rect width=""20"" height=""2"" style=""fill:black;"" x=""-9"" y=""0""></rect>
			                    <rect width=""2"" height=""{lineHeight}"" style=""fill:black;"" x=""0"" y=""0""></rect>
			                    <rect width=""20"" height=""2"" style=""fill:black;"" x=""-9"" y=""{lineHeight - 1}""></rect>
		                    </g>
                            <!--TIME ROW-->
		                    <g id=""TimeRowLegendTemplate"" transform=""matrix(1, 0, 0, 1, 10, 26)"" style=""display:none;"">
                                <!--BACKGROUND-->
                                <rect opacity=""0.80"" fill=""blue"" x=""-1"" y=""0"" width=""160"" height=""12"" rx=""2"" ry=""2""></rect>
			                    <!--TIME-->
                                <text fill=""#FFFFFF"" font-size=""11"" font-weight=""100"" x=""14"" y=""9"">Template</text>
                                <!--CLOCK ICON-->
                                <g fill=""#fff"" transform=""matrix(0.5, 0, 0, 0.5, 2, 1)"" width=""12"" height=""12"">
				                    <path d=""M10 0a10 10 0 1 0 10 10A10 10 0 0 0 10 0zm2.5 14.5L9 11V4h2v6l3 3z""></path>
			                    </g>
                            </g>

                            <!--cloned to CursorLineLegendClone -->
		                    <g id=""CursorLineLegendTemplate"" transform=""matrix(1, 0, 0, 1, 10, 26)"" style=""display:none;"">
                                <!--BACKGROUND-->
                                <rect opacity=""0.80"" fill=""blue"" x=""-1"" y=""0"" width=""160"" height=""15"" rx=""2"" ry=""2""></rect>
			                    <!--EVENT NAME-->
                                <text class=""event-name"" fill=""#FFFFFF"" font-size=""11"" font-weight=""400"" x=""14"" y=""11"">Template</text>
                                <!--ICON SET DYNAMIC BY JS-->
                                <circle class=""icon-holder"" cx=""6.82"" cy=""7.573"" r=""4.907"" fill=""grey""></circle>
                                <text class=""nature-score"" fill=""#FFFFFF"" font-size=""7"" font-weight=""100"" x=""5.3"" y=""9.9""></text>
                            </g>
                            <!--place where dynamic event names are placed by JS-->
                            <g id=""CursorLineLegendHolder"" transform=""matrix(1, 0, 0, 1, 0, 4)""></g>
                            <g id=""CursorLineLegendDescriptionHolder"" transform=""matrix(1, 0, 0, 1, 0, 0)"" style=""display:none;"">
			                    <rect id=""CursorLineLegendDescriptionBackground"" fill=""#0096FF"" x=""170"" y=""11.244"" width=""150"" height=""0"" rx=""2"" ry=""2""></rect>
                                
                                <!--PREDICTION TEXT INSERTED BY JS AT RUNTIME-->                                
                                <g id=""CursorLineLegendDescription"" transform=""matrix(1, 0, 0, 1, 0, 0)""></g>                                
		                    </g>
                       </g>
                        ";
        }

        /// <summary>
        /// get now line SVG group
        /// if set hide will set display as hidden
        /// </summary>
        private static string GetNowLine(int positionX, int positionY, bool hideOnLoad = false)
        {

            //add hide style if caller specifies to hide on load
            var extraClass = hideOnLoad ? "style=\"display: none;\"" : "";

            positionX += 10; //adjust

            return $@"
                       <g id=""NowVerticalLine"" x=""0"" y=""0"" {extraClass} transform=""matrix(1, 0, 0, 1, {positionY}, 0)"">
		                    <rect width=""2"" height=""{positionX}"" style="" fill:blue; stroke-width:0.5px; stroke:#000;""></rect>
		                    <g transform=""matrix(2, 0, 0, 2, -12, {positionX})"">
			                    <rect style=""fill: blue; opacity: 0.8; filter:url(#glow);"" x=""0"" y=""0"" width=""13.5"" height=""9.95"" rx=""2.5"" ry=""2.5""></rect>
			                    <text style=""fill:#FFF; font-size: 5px; filter:url(#glow);"" x=""1.135"" y=""6.367"">NOW</text>
		                    </g>
	                    </g>
                        ";
        }

        /// <summary>
        /// gets line position given a date
        /// finds most closest time slice, else return 0 means none found
        /// note:
        /// - make sure offset in time list and input time matches
        /// - tries to get nearest day first, then tries month to nearest year
        /// </summary>
        private static int GetLinePosition(List<Time> timeSliceList, DateTimeOffset inputTime)
        {
            //input timezone must be converted output timezone (timeSliceList)
            var outputTimezone = timeSliceList[0].GetStdDateTimeOffset().Offset;
            //translate event timezone to match chart timezone
            inputTime = inputTime.ToOffset(outputTimezone);

#if DEBUG
            if (outputTimezone.Ticks != inputTime.Offset.Ticks) { Console.WriteLine($"Life Event Offset Translated:{outputTimezone}->{inputTime.Offset}"); }
#endif

            var nowHour = inputTime.Hour;
            var nowDay = inputTime.Day;
            var nowYear = inputTime.Year;
            var nowMonth = inputTime.Month;

            //if nearest hour is possible then end here
            var nearestHour = GetNearestHour();
            if (nearestHour != 0) { return nearestHour; }

            //else try get nearest day
            var nearestDay = GetNearestDay();
            if (nearestDay != 0) { return nearestDay; }

            //else try get nearest month
            var nearestMonth = GetNearestMonth();
            if (nearestMonth != 0) { return nearestMonth; }

            //else try get nearest year
            var nearestYear = GetNearestYear();
            if (nearestYear != 0) { return nearestYear; }


            //if control reaches here then now time not found in time slices
            //this is possible when viewing old charts as such set now line to 0
            return 0;

            int GetNearestMonth()
            {
                //go through the list and find where the slice is closest to now
                var slicePosition = 0;
                foreach (var time in timeSliceList)
                {

                    //if same year and same month then send this slice position
                    //as the correct one
                    var sameYear = time.StdYear() == nowYear;
                    var sameMonth = time.StdMonth() == nowMonth;
                    if (sameMonth && sameYear)
                    {
                        return slicePosition;
                    }

                    //move to next slice position
                    slicePosition++;
                }

                return 0;
            }

            int GetNearestDay()
            {
                //go through the list and find where the slice is closest to now
                var slicePosition = 0;
                foreach (var time in timeSliceList)
                {

                    //if same year and same month then send this slice position
                    //as the correct one
                    var sameDay = time.StdDate() == nowDay;
                    var sameYear = time.StdYear() == nowYear;
                    var sameMonth = time.StdMonth() == nowMonth;
                    if (sameMonth && sameYear && sameDay)
                    {
                        return slicePosition;
                    }

                    //move to next slice position
                    slicePosition++;
                }

                return 0;
            }

            int GetNearestHour()
            {
                //go through the list and find where the slice is closest to now
                var slicePosition = 0;
                foreach (var time in timeSliceList)
                {

                    //if same year and same month then send this slice position
                    //as the correct one
                    var sameHour = time.GetStdDateTimeOffset().Hour == nowHour;
                    var sameDay = time.StdDate() == nowDay;
                    var sameYear = time.StdYear() == nowYear;
                    var sameMonth = time.StdMonth() == nowMonth;
                    if (sameDay && sameHour && sameMonth && sameYear)
                    {
                        return slicePosition;
                    }

                    //move to next slice position
                    slicePosition++;
                }

                return 0;
            }

            int GetNearestYear()
            {

                //go through the list and find where the slice is closest to now
                var slicePosition = 0;
                foreach (var time in timeSliceList)
                {

                    //if same year and same month then send this slice position
                    //as the correct one
                    var sameYear = time.StdYear() == nowYear;
                    if (sameYear)
                    {
                        return slicePosition;
                    }

                    //move to next slice position
                    slicePosition++;
                }

                return 0;
            }
        }

        private static string GenerateTimeHeaderRow(List<Time> timeSlices, double daysPerPixel, int _widthPerSlice, ref int headerY)
        {
            var dasaSvgWidth = 0; //will be filled when calling row generator
            var compiledRow = "";

            var beginYear = timeSlices[0].StdYear();
            var endYear = timeSlices.Last().StdYear();
            var difYears = endYear - beginYear;

            //header rows are dynamically generated as needed, hence the extra logic below
            var headerGenerator = new List<Func<List<Time>, int, int, int, string>>();
            var showYearRow = daysPerPixel <= 15;
            if (difYears >= 10 && !showYearRow) { headerGenerator.Add(GenerateDecadeRowSvg); }
            if (difYears is >= 5 and < 10) { headerGenerator.Add(Generate5YearRowSvg); }
            if (showYearRow) { headerGenerator.Add(GenerateYearRowSvg); }
            if (daysPerPixel <= 1.3) { headerGenerator.Add(GenerateMonthRowSvg); }
            if (daysPerPixel <= 0.07) { headerGenerator.Add(GenerateDateRowSvg); }
            if (daysPerPixel <= 0.001) { headerGenerator.Add(GenerateHourRowSvg); }

            var padding = 2;//space between rows
            int headerHeight = 11;
            foreach (var generator in headerGenerator)
            {
                compiledRow += generator(timeSlices, headerY, 0, headerHeight);

                //update for next generator
                headerY = headerY + headerHeight + padding;
            }

            return compiledRow;


            string GenerateYearRowSvg(List<Time> timeSlices, int yAxis, int xAxis, int rowHeight)
            {

                //generate the row for each time slice
                var rowHtml = "";
                var previousYear = 0; //start 0 for first draw
                var yearBoxWidthCount = 0;
                int rectWidth = 0;
                int childAxisX = 0;

                foreach (var slice in timeSlices)
                {

                    //only generate new year box when year changes or at
                    //end of time slices to draw the last year box
                    var lastTimeSlice = timeSlices.IndexOf(slice) == timeSlices.Count - 1;
                    var yearChanged = previousYear != slice.StdYear();
                    if (yearChanged || lastTimeSlice)
                    {
                        //and it is in the beginning
                        if (previousYear == 0)
                        {
                            yearBoxWidthCount = 0; //reset width
                        }
                        else
                        {
                            //generate previous year data first before resetting
                            childAxisX += rectWidth; //use previous rect width to position this
                            rectWidth = yearBoxWidthCount * _widthPerSlice; //calculate new rect width
                            var textX = rectWidth / 2; //center of box divide 2
                            var rect = $"<g transform=\"matrix(1, 0, 0, 1, {childAxisX}, 0)\">" + //y is 0 because already set in parent group
                                                $"<rect " +
                                                    $"fill=\"#0d6efd\" x=\"0\" y=\"0\" width=\"{rectWidth}\" height=\"{rowHeight}\" " + $" style=\"paint-order: stroke; stroke: rgb(255, 255, 255); stroke-opacity: 1; stroke-linejoin: round;\"/>" +
                                                    $"<text x=\"{textX}\" y=\"{9}\" width=\"{rectWidth}\" fill=\"white\"" +
                                                        $" style=\"fill: rgb(255, 255, 255);" +
                                                        $" font-size: 10px;" +
                                                        $" font-weight: 700;" +
                                                        $" text-anchor: middle;" +
                                                        $" white-space: pre;\"" +
                                                        //$" transform=\"matrix(0.966483, 0, 0, 0.879956, 2, -6.779947)\"" +
                                                        $">" +
                                                        $"{previousYear}" + //previous year generate at begin of new year
                                                    $"</text>" +
                                             $"</g>";


                            //add to final return
                            rowHtml += rect;

                            //reset width
                            yearBoxWidthCount = 0;

                        }
                    }
                    //year same as before
                    else
                    {
                        //update width only, position is same
                        //as when created the year box
                        //yearBoxWidthCount *= _widthPerSlice;

                    }

                    //update previous year for next slice
                    previousYear = slice.StdYear();

                    yearBoxWidthCount++;


                }

                //wrap all the rects inside a svg so they can me moved together
                //svg tag here acts as group, svg nesting
                rowHtml = $"<g transform=\"matrix(1, 0, 0, 1, {xAxis}, {yAxis})\">{rowHtml}</g>";

                return rowHtml;
            }

            string GenerateDecadeRowSvg(List<Time> timeSlices, int yAxis, int xAxis, int rowHeight)
            {

                //generate the row for each time slice
                var rowHtml = "";
                var previousYear = 0; //start 0 for first draw
                var yearBoxWidthCount = 0;
                int rectWidth = 0;
                int childAxisX = 0;
                const int decade = 10;

                var beginYear = timeSlices[0].StdYear();
                var endYear = beginYear + decade; //decade


                foreach (var slice in timeSlices)
                {

                    //only generate new year box when year changes or at
                    //end of time slices to draw the last year box
                    var lastTimeSlice = timeSlices.IndexOf(slice) == timeSlices.Count - 1;
                    var yearChanged = previousYear != slice.StdYear();
                    if (yearChanged || lastTimeSlice)
                    {
                        //is this slice end year & last month (month for accuracy, otherwise border at jan not december)
                        //todo begging of box is not beginning of year, possible solution month
                        //var isLastMonth = slice.GetStdMonth() is 10 or 11 or 12; //use oct & nov in-case december is not generated at low precision 
                        var isEndYear = endYear == slice.StdYear();
                        if (isEndYear)
                        {
                            //generate previous year data first before resetting
                            childAxisX += rectWidth; //use previous rect width to position this
                            rectWidth = yearBoxWidthCount * _widthPerSlice; //calculate new rect width
                            var textX = rectWidth / 2; //center of box divide 2
                            var rect = $"<g transform=\"matrix(1, 0, 0, 1, {childAxisX}, 0)\">" + //y is 0 because already set in parent group
                                       $"<rect " +
                                       $"fill=\"#0d6efd\" x=\"0\" y=\"0\" width=\"{rectWidth}\" height=\"{rowHeight}\" " + $" style=\"paint-order: stroke; stroke: rgb(255, 255, 255); stroke-opacity: 1; stroke-linejoin: round;\"/>" +
                                       $"<text x=\"{textX}\" y=\"{9}\" width=\"{rectWidth}\" fill=\"white\"" +
                                       $" style=\"fill: rgb(255, 255, 255);" +
                                       $" font-size: 10px;" +
                                       $" font-weight: 700;" +
                                       $" text-anchor: middle;" +
                                       $" white-space: pre;\"" +
                                       //$" transform=\"matrix(0.966483, 0, 0, 0.879956, 2, -6.779947)\"" +
                                       $">" +
                                       $"{beginYear} - {endYear}" + //previous year generate at begin of new year
                                       $"</text>" +
                                       $"</g>";


                            //add to final return
                            rowHtml += rect;

                            //reset width
                            yearBoxWidthCount = 0;

                            //set new begin & end
                            beginYear = endYear + 1;
                            endYear = beginYear + decade;

                        }

                    }

                    //update previous year for next slice
                    previousYear = slice.StdYear();

                    yearBoxWidthCount++;

                }

                //wrap all the rects inside a svg so they can me moved together
                //svg tag here acts as group, svg nesting
                rowHtml = $"<g transform=\"matrix(1, 0, 0, 1, {xAxis}, {yAxis})\">{rowHtml}</g>";

                return rowHtml;
            }

            string Generate5YearRowSvg(List<Time> timeSlices, int yAxis, int xAxis, int rowHeight)
            {

                //generate the row for each time slice
                var rowHtml = "";
                var previousYear = 0; //start 0 for first draw
                var yearBoxWidthCount = 0;
                int rectWidth = 0;
                int childAxisX = 0;
                //int rowHeight = 11;

                const int yearRange = 5;

                var beginYear = timeSlices[0].StdYear();
                var endYear = beginYear + yearRange;


                foreach (var slice in timeSlices)
                {

                    //only generate new year box when year changes or at
                    //end of time slices to draw the last year box
                    var lastTimeSlice = timeSlices.IndexOf(slice) == timeSlices.Count - 1;
                    var yearChanged = previousYear != slice.StdYear();
                    if (yearChanged || lastTimeSlice)
                    {
                        //is this slice end year
                        var isEndYear = endYear == slice.StdYear();
                        if (isEndYear)
                        {
                            //generate previous year data first before resetting
                            childAxisX += rectWidth; //use previous rect width to position this
                            rectWidth = yearBoxWidthCount * _widthPerSlice; //calculate new rect width
                            var textX = rectWidth / 2; //center of box divide 2
                            var rect = $"<g transform=\"matrix(1, 0, 0, 1, {childAxisX}, 0)\">" + //y is 0 because already set in parent group
                                       $"<rect " +
                                       $"fill=\"#0d6efd\" x=\"0\" y=\"0\" width=\"{rectWidth}\" height=\"{rowHeight}\" " + $" style=\"paint-order: stroke; stroke: rgb(255, 255, 255); stroke-opacity: 1; stroke-linejoin: round;\"/>" +
                                       $"<text x=\"{textX}\" y=\"{9}\" width=\"{rectWidth}\" fill=\"white\"" +
                                       $" style=\"fill: rgb(255, 255, 255);" +
                                       $" font-size: 10px;" +
                                       $" font-weight: 700;" +
                                       $" text-anchor: middle;" +
                                       $" white-space: pre;\"" +
                                       //$" transform=\"matrix(0.966483, 0, 0, 0.879956, 2, -6.779947)\"" +
                                       $">" +
                                       $"{beginYear} - {endYear}" + //previous year generate at begin of new year
                                       $"</text>" +
                                       $"</g>";


                            //add to final return
                            rowHtml += rect;

                            //reset width
                            yearBoxWidthCount = 0;

                            //set new begin & end
                            beginYear = endYear + 1;
                            endYear = beginYear + yearRange;

                        }

                    }

                    //update previous year for next slice
                    previousYear = slice.StdYear();

                    yearBoxWidthCount++;

                }

                //wrap all the rects inside a svg so they can me moved together
                //svg tag here acts as group, svg nesting
                rowHtml = $"<g transform=\"matrix(1, 0, 0, 1, {xAxis}, {yAxis})\">{rowHtml}</g>";

                return rowHtml;
            }

            string GenerateMonthRowSvg(List<Time> timeSlices, int yAxis, int xAxis, int rowHeight)
            {

                //generate the row for each time slice
                var rowHtml = "";
                var previousMonth = 0; //start 0 for first draw
                var yearBoxWidthCount = 0;
                int rectWidth = 0;
                int childAxisX = 0;
                //int rowHeight = 11;

                foreach (var slice in timeSlices)
                {

                    //only generate new year box when year changes or at
                    //end of time slices to draw the last year box
                    var lastTimeSlice = timeSlices.IndexOf(slice) == timeSlices.Count - 1;
                    var monthChanged = previousMonth != slice.StdMonth();
                    if (monthChanged || lastTimeSlice)
                    {
                        //and it is in the beginning
                        if (previousMonth == 0)
                        {
                            yearBoxWidthCount = 0; //reset width
                        }
                        else
                        {
                            //generate previous month data first before resetting
                            childAxisX += rectWidth; //use previous rect width to position this
                            rectWidth = yearBoxWidthCount * _widthPerSlice; //calculate new rect width
                            var textX = rectWidth / 2; //center of box divide 2
                            var rect = $"<g transform=\"matrix(1, 0, 0, 1, {childAxisX}, 0)\">" + //y is 0 because already set in parent group
                                       $"<rect " +
                                       $"fill=\"#0d6efd\" x=\"0\" y=\"0\" width=\"{rectWidth}\" height=\"{rowHeight}\" " + $" style=\"paint-order: stroke; stroke: rgb(255, 255, 255); stroke-opacity: 1; stroke-linejoin: round;\"/>" +
                                       $"<text x=\"{textX}\" y=\"{9}\" width=\"{rectWidth}\" fill=\"white\"" +
                                       $" style=\"fill: rgb(255, 255, 255);" +
                                       $" font-size: 10px;" +
                                       $" font-weight: 700;" +
                                       $" text-anchor: middle;" +
                                       $" white-space: pre;\"" +
                                       //$" transform=\"matrix(0.966483, 0, 0, 0.879956, 2, -6.779947)\"" +
                                       $">" +
                                       $"{GetMonthName(previousMonth)}" + //previous year generate at begin of new year
                                       $"</text>" +
                                       $"</g>";


                            //add to final return
                            rowHtml += rect;

                            //reset width
                            yearBoxWidthCount = 0;

                        }
                    }
                    //year same as before
                    else
                    {
                        //update width only, position is same
                        //as when created the year box
                        //yearBoxWidthCount *= _widthPerSlice;

                    }

                    //update previous month for next slice
                    previousMonth = slice.StdMonth();

                    yearBoxWidthCount++;


                }

                //wrap all the rects inside a svg so they can me moved together
                //svg tag here acts as group, svg nesting
                rowHtml = $"<g transform=\"matrix(1, 0, 0, 1, {xAxis}, {yAxis})\">{rowHtml}</g>";

                return rowHtml;

                string GetMonthName(int monthNum)
                {
                    switch (monthNum)
                    {
                        case 1: return "JAN";
                        case 2: return "FEB";
                        case 3: return "MAR";
                        case 4: return "APR";
                        case 5: return "MAY";
                        case 6: return "JUN";
                        case 7: return "JUL";
                        case 8: return "AUG";
                        case 9: return "SEP";
                        case 10: return "OCT";
                        case 11: return "NOV";
                        case 12: return "DEC";
                        default: throw new Exception($"Invalid Month: {monthNum}");
                    }
                }
            }

            string GenerateDateRowSvg(List<Time> timeSlices, int yAxis, int xAxis, int rowHeight)
            {

                //generate the row for each time slice
                var rowHtml = "";
                var previousDate = 0; //start 0 for first draw
                var dateBoxWidthCount = 0;
                int rectWidth = 0;
                int childAxisX = 0;
                //int rowHeight = 11;

                foreach (var slice in timeSlices)
                {

                    //only generate new date box when date changes or at
                    //end of time slices to draw the last date box
                    var lastTimeSlice = timeSlices.IndexOf(slice) == timeSlices.Count - 1;
                    var dateChanged = previousDate != slice.StdDate();
                    if (dateChanged || lastTimeSlice)
                    {
                        //and it is in the beginning
                        if (previousDate == 0)
                        {
                            dateBoxWidthCount = 0; //reset width
                        }
                        else
                        {
                            //generate previous date data first before resetting
                            childAxisX += rectWidth; //use previous rect width to position this
                            rectWidth = dateBoxWidthCount * _widthPerSlice; //calculate new rect width
                            var textX = rectWidth / 2; //center of box divide 2
                            var rect = $"<g transform=\"matrix(1, 0, 0, 1, {childAxisX}, 0)\">" + //y is 0 because already set in parent group
                                       $"<rect " +
                                       $"fill=\"#0d6efd\" x=\"0\" y=\"0\" width=\"{rectWidth}\" height=\"{rowHeight}\" " + $" style=\"paint-order: stroke; stroke: rgb(255, 255, 255); stroke-opacity: 1; stroke-linejoin: round;\"/>" +
                                       $"<text x=\"{textX}\" y=\"{9}\" width=\"{rectWidth}\" fill=\"white\"" +
                                       $" style=\"fill: rgb(255, 255, 255);" +
                                       $" font-size: 10px;" +
                                       $" font-weight: 700;" +
                                       $" text-anchor: middle;" +
                                       $" white-space: pre;\"" +
                                       //$" transform=\"matrix(0.966483, 0, 0, 0.879956, 2, -6.779947)\"" +
                                       $">" +
                                       $"{previousDate}" + //previous date generate at begin of new date
                                       $"</text>" +
                                       $"</g>";


                            //add to final return
                            rowHtml += rect;

                            //reset width
                            dateBoxWidthCount = 0;

                        }
                    }

                    //update previous date for next slice
                    previousDate = slice.StdDate();

                    dateBoxWidthCount++;

                }

                //wrap all the rects inside a svg so they can me moved together
                //svg tag here acts as group, svg nesting
                rowHtml = $"<g transform=\"matrix(1, 0, 0, 1, {xAxis}, {yAxis})\">{rowHtml}</g>";

                return rowHtml;

            }

            string GenerateHourRowSvg(List<Time> timeSlices, int yAxis, int xAxis, int rowHeight)
            {

                //generate the row for each time slice
                var rowHtml = "";
                var previousHour = -1; //so that hour 0 is counted
                var hourBoxWidthCount = 0;
                int rectWidth = 0;
                int childAxisX = 0;
                //int rowHeight = 11;

                foreach (var slice in timeSlices)
                {

                    //only generate new date box when hour changes or at
                    //end of time slices to draw the last hour box
                    var isLastTimeSlice = timeSlices.IndexOf(slice) == timeSlices.Count - 1;
                    var hourChanged = previousHour != slice.StdHour();
                    if (hourChanged || isLastTimeSlice)
                    {
                        //and it is in the beginning
                        if (previousHour == -1)
                        {
                            hourBoxWidthCount = 0; //reset width
                        }
                        else
                        {
                            //generate previous hour data first before resetting
                            childAxisX += rectWidth; //use previous rect width to position this
                            rectWidth = hourBoxWidthCount * _widthPerSlice; //calculate new rect width
                            var textX = rectWidth / 2; //center of box divide 2
                            var rect = $"<g transform=\"matrix(1, 0, 0, 1, {childAxisX}, 0)\">" + //y is 0 because already set in parent group
                                       $"<rect " +
                                       $"fill=\"#0d6efd\" x=\"0\" y=\"0\" width=\"{rectWidth}\" height=\"{rowHeight}\" " + $" style=\"paint-order: stroke; stroke: rgb(255, 255, 255); stroke-opacity: 1; stroke-linejoin: round;\"/>" +
                                       $"<text x=\"{textX}\" y=\"{9}\" width=\"{rectWidth}\" fill=\"white\"" +
                                       $" style=\"fill: rgb(255, 255, 255);" +
                                       $" font-size: 10px;" +
                                       $" font-weight: 700;" +
                                       $" text-anchor: middle;" +
                                       $" white-space: pre;\"" +
                                       //$" transform=\"matrix(0.966483, 0, 0, 0.879956, 2, -6.779947)\"" +
                                       $">" +
                                       $"{previousHour}" + //previous hour generate at begin of new hour
                                       $"</text>" +
                                       $"</g>";


                            //add to final return
                            rowHtml += rect;

                            //reset width
                            hourBoxWidthCount = 0;

                        }
                    }

                    //update previous hour for next slice
                    previousHour = slice.StdHour();

                    hourBoxWidthCount++;
                }

                //wrap all the rects inside a svg so they can me moved together
                //svg tag here acts as group, svg nesting
                rowHtml = $"<g transform=\"matrix(1, 0, 0, 1, {xAxis}, {yAxis})\">{rowHtml}</g>";

                return rowHtml;

            }

        }

        private static Dictionary<int, SumData> SummaryRowData { get; set; }

        /// <summary>
        /// Generate rows based of inputed events
        /// </summary>
        private static string GenerateEventRows(double hoursPerPixel, Person inputPerson, List<Time> timeSlices, ChartOptions options, ref int yAxis)
        {
            const int widthPerSlice = 1;


            //sort event by duration, so that events are ordered nicely in chart
            //todo events are breaking up between rows
            //todo order by planet strength
            var eventList = EventsChartFactory.UnsortedEventList.OrderByDescending(x => x.DurationMin).ToList();

            //1. FILTER
            //remove events 85% shorter duration than time shown by 1 pixel
            //exp: if event is only 10min but 1 pixel = 60min,
            //then the small event is going to overpower the larger events
            var removedCount = eventList.RemoveAll(evt => evt.DurationHour < (hoursPerPixel * 0.85));
            Console.WriteLine($"!! REMOVED EVENT TOO SHORT COUNT : {removedCount} !!");

            //todo make data visible to end user
            Console.WriteLine($"Number Of Events In Chart :{eventList.Count}");

            //2. STACK & GENERATED ROWS FROM ABOVE DATA
            var padding = 1;//space between rows
            var compiledRow = "";
            if (eventList.Any())
            {

                //x axis, total nature score, planet name
                SummaryRowData = new Dictionary<int, SumData>(); //note: summary data is filled when generating rows

                //generate svg for each row & add to final row
                compiledRow += GenerateMultipleRowSvg(eventList, timeSlices, yAxis, 0, options, out int finalHeight);

                //set y axis (horizontal) for next row
                yAxis = yAxis + finalHeight + padding;

            }

            return compiledRow;



            //█░░ █▀█ █▀▀ ▄▀█ █░░   █▀▀ █░█ █▄░█ █▀▀ ▀█▀ █ █▀█ █▄░█ █▀
            //█▄▄ █▄█ █▄▄ █▀█ █▄▄   █▀░ █▄█ █░▀█ █▄▄ ░█░ █ █▄█ █░▀█ ▄█


            //height not known until generated
            //returns the final dynamic height of this event row
            string GenerateMultipleRowSvg(List<Event> eventList, List<Time> timeSlices, int yAxis, int xAxis, ChartOptions summaryOptions, out int finalHeight)
            {
                //generate the row for each time slice
                var rowHtml = "";
                var horizontalPosition = 0; //distance from left
                var verticalPosition = 0; //distance from top

                //height of each row
                var spaceBetweenRow = 1;

                //used to determine final height
                var highestTimeSlice = 0;
                var multipleEventCount = 0;

                //start as empty event
                var prevEventList = new Dictionary<int, EventName>();

                //convert each time slice to a stack of event rects
                foreach (var slice in timeSlices)
                {
                    //get events that occurred at this time slice
                    var foundEventList = eventList.FindAll(tempEvent => tempEvent.IsOccurredAtTime(slice));

                    //generate rect for each event & stack from top to bottom (vertical)
                    foreach (var foundEvent in foundEventList)
                    {
                        //every time a rect is added, we keep track of it in a list to generate the summary row at last
                        //based on event nature minus or plus 1
                        double natureScore = 0;

                        //calculate accurate nature score
                        natureScore = CalculateNatureScore(foundEvent, inputPerson, summaryOptions.SelectedAlgorithm);
                        foundEvent.NatureScore = natureScore;
                    }

                    //get max & min to set color boundaries, helps to render dynamic color range based on changing algo count
                    double maxNatureScore = 0;
                    double minNatureScore = 0;
                    if (foundEventList.Any())
                    {
                        maxNatureScore = foundEventList.Aggregate((l, r) => l.NatureScore > r.NatureScore ? l : r).NatureScore;
                        minNatureScore = foundEventList.Aggregate((l, r) => l.NatureScore < r.NatureScore ? l : r).NatureScore;
                    }

                    //generate each event rect at a time
                    foreach (var foundEvent in foundEventList)
                    {
                        //if current event is different from event has changed, so draw a black line
                        int finalYAxis = yAxis + verticalPosition;
                        var prevExist = prevEventList.TryGetValue(finalYAxis, out var prevEventName);
                        var isNewEvent = prevExist && (prevEventName != foundEvent.Name);
                        var color = isNewEvent ? "black" : GetSummaryColor(foundEvent.NatureScore, minNatureScore, maxNatureScore);

                        //save current event too previous to draw border later
                        //border ONLY for top 3 rows (long duration events), lower row borders block color
                        prevEventList[finalYAxis] = foundEvent.Name;

                        //if (finalYAxis <= 29)
                        //{
                        //    prevEventList[finalYAxis] = foundEvent.Name;
                        //}

                        //generate and add to row
                        //the hard coded attribute names used here are used in App.js
                        var rect = $"<rect " +
                                   $"eventname=\"{foundEvent?.FormattedName}\" " +
                                   $"eventdescription=\"{foundEvent?.Description}\" " +
                                   $"naturescore=\"{foundEvent?.NatureScore}\" " +
                                   $"age=\"{inputPerson.GetAge(slice)}\" " +
                                   $"stdtime=\"{slice.GetStdDateTimeOffset().ToString(Time.DateTimeFormat)}\" " +
                                   $"x=\"{horizontalPosition}\" " +
                                   $"y=\"{finalYAxis}\" " + //y axis placed here instead of parent group, so that auto legend can use the y axis
                                   $"width=\"{widthPerSlice}\" " +
                                   $"height=\"{SingleRowHeight}\" " +
                                   $"fill=\"{color}\" />";

                        //add rect to return list
                        rowHtml += rect;


                        //STAGE 2 : SAVE SUMMARY DATA

                        //compile nature score for making summary row later (defaults to 0)
                        var previousNatureScoreSum = (SummaryRowData.ContainsKey(horizontalPosition) ? SummaryRowData[horizontalPosition].NatureScore : 0);

                        var x = new SumData
                        {
                            BirthTime = inputPerson.BirthTime,
                            Planet = Tools.GetPlanetFromName(foundEvent.FormattedName),
                            NatureScore = foundEvent.NatureScore + previousNatureScoreSum //combine current with other events on same time slice
                        };
                        SummaryRowData[horizontalPosition] = x;


                        //increment vertical position for next
                        //element to be placed beneath this one
                        verticalPosition += SingleRowHeight + spaceBetweenRow;

                        multipleEventCount++; //include this in count

                    }


                    //set position for next element in time slice
                    horizontalPosition += widthPerSlice;

                    //reset vertical position for next time slice
                    verticalPosition = 0;

                    //safe only the highest row (last row in to be added) used for calculating final height
                    var multipleRowH = (multipleEventCount * (SingleRowHeight + spaceBetweenRow)) - spaceBetweenRow; //minus 1 to compensate for last row
                    var thisSliceHeight = multipleEventCount > 1 ? multipleRowH : SingleRowHeight; //different height calc for multiple & single row
                    highestTimeSlice = thisSliceHeight > highestTimeSlice ? thisSliceHeight : highestTimeSlice;
                    multipleEventCount = 0; //reset

                }

                //wrap all the rects inside a svg so they can be moved together
                //note: use group instead of svg because editing capabilities
                rowHtml = $"<g class=\"EventListHolder\" transform=\"matrix(1, 0, 0, 1, {xAxis}, 0)\">{rowHtml}</g>";

                //send height of tallest time slice aka the
                //final height of this gochara row to caller
                finalHeight = highestTimeSlice;

                return rowHtml;
            }

        }

        private static string GenerateSummaryRow(Dictionary<int, SumData> summaryRowData, int summaryRowHeight, int singleRowHeight, int padding, ref int yAxis)
        {

            //min & max used to calculate color later
            var rawMaxValue = summaryRowData?.Values?.Max(x => x.NatureScore) ?? 0;
            var rawMinValue = summaryRowData?.Values?.Min(x => x.NatureScore) ?? 0;

            //lower ceiling height by percentage, to make lower colors pop
            //NOTE: percentage set base on experimentation
            var maxValue = (rawMaxValue - (rawMaxValue * 0.4)); //40% drop from top
            var minValue = (rawMinValue - (rawMinValue * 0.2)); //20% up from bottom

#if DEBUG
            Console.WriteLine($"GenerateSummaryRow : MAX:{maxValue}, MIN:{minValue}");
#endif

            var rowHtml = "";
            //STEP 1 : generate color summary
            var colorRow = "";
            foreach (var summarySlice in summaryRowData)
            {
                int xAxis = summarySlice.Key;
                //total nature score is sum of negative & positive 1s of all events
                //that occurred at this point in time, possible negative number
                //exp: -4 bad + 5 good = 1 total nature score
                double totalNatureScore = summarySlice.Value.NatureScore;



                var summaryColor = GetSummaryColor(totalNatureScore, minValue, maxValue);
                var rect = $"<rect " +
                           $"x=\"{xAxis}\" " +
                           $"y=\"{yAxis}\" " + //y-axis placed here instead of parent group, so that auto legend can use the y-axis
                           $"width=\"{WidthPerSlice}\" " +
                           $"height=\"{summaryRowHeight}\" " +
                           $"fill=\"{summaryColor}\" />";

                //add rect to row
                colorRow += rect;
            }

            rowHtml += $"<g id=\"ColorRow\">{colorRow}</g>";


            //STEP 2 : generate graph summary
            //var barChartRow = "";
            //foreach (var summarySlice in summaryRowData)
            //{
            //    int xAxis = summarySlice.Key;
            //    double totalNatureScore = summarySlice.Value.NatureScore; //possible negative
            //    var barHeight = (int)totalNatureScore.Remap(minValue, maxValue, 0, 30);
            //    var rect = $"<rect " +
            //               $"x=\"{xAxis}\" " +
            //               $"y=\"{yAxis}\" " + //y axis placed here instead of parent group, so that auto legend can use the y axis
            //               $"width=\"{widthPerSlice}\" " +
            //               $"height=\"{barHeight}\" " +
            //               $"fill=\"black\" />";

            //    //add rect to row
            //    barChartRow += rect;
            //}

            ////note: chart is flipped 180, to start bar from bottom to top
            ////default hidden
            //rowHtml += $"<g id=\"BarChartRow\" transform=\"matrix(1, 0, 0, 1, 0, 20)\">{barChartRow}</g>";


            //STEP 3 : generate color summary SMART
            //var colorRowSmart = "";
            //foreach (var summarySlice in summaryRowData)
            //{
            //    int xAxis = summarySlice.Key;
            //    //total nature score is sum of negative & positive 1s of all events
            //    //that occurred at this point in time, possible negative number
            //    //exp: -4 bad + 5 good = 1 total nature score
            //    double totalNatureScore = summarySlice.Value.NatureScore;
            //    var planetPowerFactor = GetPlanetPowerFactor(summarySlice.Value.Planet, summarySlice.Value.BirthTime);
            //    var smartNatureScore = totalNatureScore * planetPowerFactor;
            //    var rect = $"<rect " +
            //               $"x=\"{xAxis}\" " +
            //               $"y=\"{yAxis}\" " + //y axis placed here instead of parent group, so that auto legend can use the y axis
            //               $"width=\"{widthPerSlice}\" " +
            //               $"height=\"{singleRowHeight}\" " +
            //               $"fill=\"{GetSummaryColor(smartNatureScore, -100, 100)}\" />";

            //    //add rect to row
            //    colorRowSmart += rect;
            //}
            ////note: chart is flipped 180, to start bar from bottom to top
            ////default hidden
            //rowHtml += $"<g id=\"BarChartRowSmart\" transform=\"matrix(1, 0, 0, 1, 0, 43)\">{colorRowSmart}</g>";


            //STEP 4 : final wrapper
            //add in "Smart Summary" label above row
            float aboveRow = yAxis - singleRowHeight - padding;
            rowHtml += $@"
                    <g id=""SummaryLabel"" transform=""matrix(1, 0, 0, 1, 0, {aboveRow})"">
				        <rect style=""fill: blue; opacity: 0.80;"" x=""-1"" y=""0"" width=""100"" height=""15"" rx=""2"" ry=""2""/>
				        <text style=""fill:#FFFFFF; font-size:11px; font-weight:400;"" x=""16"" y=""11"">Smart Summary</text>
				        <path transform=""matrix(0.045, 0, 0, 0.045, -14, -4)"" fill=""#fff"" d=""M437 122c-15 2-26 5-38 10-38 16-67 51-75 91-4 17-4 36 0 54 10 48 47 86 95 98 11 2 15 3 30 3s19-1 30-3c48-12 86-50 96-98 3-18 3-37 0-54-10-47-48-86-95-98-10-2-16-3-29-3h-14zm66 59c2 2 3 3 4 6s1 17 0 20c-2 7-11 9-15 2-1-2-1-3-1-7v-5h-37-37s8 11 18 23l21 25c1 2 1 5 1 7-1 1-10 13-21 26l-19 24c0 1 13 1 37 1h37v-5c0-6 1-9 5-11 5-2 11 1 11 8 1 1 1 6 1 10-1 7-1 8-2 10s-3 4-7 4h-52-50l-2-1c-4-3-5-7-3-11 0 0 11-14 24-29l22-28-22-28c-13-16-24-29-24-30-2-3-1-7 2-9 2-3 2-3 55-3h51l3 1z"" stroke=""none"" fill-rule=""nonzero""/>
			        </g>";

            //return compiled rects as 1 row in a group for easy debugging & edits
            rowHtml = $"<g id=\"SummaryRow\">{rowHtml}</g>";

            //add in summary row height
            yAxis += 15;

            return rowHtml;
        }

        /// <summary>
        /// Intelligently calculates summary score
        /// </summary>
        private static double CalculateNatureScore(Event foundEvent, Person person, List<AlgorithmFuncs> selectedAlgorithm)
        {
            //add together all score of selected algorithms
            var final = 0.0;
            foreach (var algorithm in selectedAlgorithm)
            {
                final += algorithm.Invoke(foundEvent, person);
            }


            //rounding
            final = Math.Round(final, 2);

            return final;
        }


        public static object GenerateEventsChartForChat(Time birthTime, TimeRange timeRangeParsed)
        {
            throw new NotImplementedException();
        }
    }
}
