/*
 *  Copyright 2022 UnitTestBot contributors (utbot.org)
 * <p>
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 * <p>
 *  http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.jacodb.impl.features

import org.jacodb.api.*
import org.jacodb.api.ext.jvmPrimitiveNames
import org.jacodb.impl.fs.PersistenceClassSource
import org.jacodb.impl.fs.className
import org.jacodb.impl.storage.executeQueries
import org.jacodb.impl.storage.jooq.tables.references.BUILDERS
import org.jacodb.impl.storage.jooq.tables.references.CLASSES
import org.jacodb.impl.storage.jooq.tables.references.SYMBOLS
import org.jacodb.impl.storage.runBatch
import org.jacodb.impl.storage.withoutAutoCommit
import org.jooq.DSLContext
import org.objectweb.asm.Opcodes
import org.objectweb.asm.Type
import org.objectweb.asm.tree.ClassNode
import org.objectweb.asm.tree.MethodNode

private val MethodNode.isGetter: Boolean
    get() {
        return name.startsWith("get")
    }

private val Int.isPublic get() = this and Opcodes.ACC_PUBLIC != 0
private val Int.isStatic get() = this and Opcodes.ACC_STATIC != 0

private data class BuilderMethod(
    val callerClass: String,
    val methodOffset: Int,
    val priority: Int
)

class BuildersIndexer(val persistence: JcDatabasePersistence, private val location: RegisteredLocation) :
    ByteCodeIndexer {

    // class -> (caller_class, offset, priority)
    private val potentialBuilders = hashMapOf<String, HashSet<BuilderMethod>>()

    override fun index(classNode: ClassNode) {
        val callerClass = classNode.name
        classNode.methods.forEachIndexed { index, methodNode ->
            val isStatic = methodNode.access.isStatic
            if (methodNode.access.isPublic && !methodNode.isGetter) {
                val returnType = Type.getMethodType(methodNode.desc).returnType.internalName
                if (
                    !jvmPrimitiveNames.contains(returnType) && // not interesting in primitives
                    !returnType.startsWith("[") && // not interesting in arrays
                    !returnType.startsWith("java/") // not interesting in java package classes
                ) {
                    val noParams = Type.getArgumentTypes(methodNode.desc).isNullOrEmpty()
                    val isBuildName = methodNode.name.equals("build")
                    val priority = when {
                        isStatic && noParams && returnType == callerClass -> 15
                        isStatic && noParams -> 10
                        isBuildName && noParams -> 7
                        isStatic -> 5
                        isBuildName -> 3
                        else -> 0
                    }
                    potentialBuilders.getOrPut(returnType) { hashSetOf() }
                        .add(BuilderMethod(callerClass, index, priority))
                }
            }
        }
    }


    override fun flush(jooq: DSLContext) {
        jooq.withoutAutoCommit { conn ->
            conn.runBatch(BUILDERS) {
                potentialBuilders.forEach { (calleeClass, builders) ->
                    val calleeId = calleeClass.className
                    builders.forEach {
                        val (callerClass, offset, priority) = it
                        val callerId = callerClass.className
                        setString(1, calleeId)
                        setString(2, callerId)
                        setInt(3, priority)
                        setInt(4, offset)
                        setLong(5, location.id)
                        addBatch()
                    }
                }
            }
        }
    }

}

data class BuildersResponse(
    val methodOffset: Int,
    val priority: Int,
    val source: ClassSource
)

object Builders : JcFeature<Set<String>, BuildersResponse> {

    fun create(jooq: DSLContext, drop: Boolean) {
        if (drop) {
            jooq.executeQueries(dropScheme)
        }
        jooq.executeQueries(createScheme)
    }

    private val createScheme = """
        CREATE TABLE IF NOT EXISTS "Builders"(
            "class_name"                    VARCHAR(256) NOT NULL,
            "builder_class_name"            VARCHAR(256) NOT NULL,
            "priority"                      INTEGER NOT NULL,
            "offset"      					INTEGER NOT NULL,
            "location_id"                   BIGINT NOT NULL,
        CONSTRAINT "fk_location_id" FOREIGN KEY ("location_id") REFERENCES "BytecodeLocations" ("id") ON DELETE CASCADE
        );
    """.trimIndent()

    private val createIndex = """
            CREATE INDEX IF NOT EXISTS "BuildersSearch" ON "Builders"(location_id, class_name, priority);
            CREATE INDEX IF NOT EXISTS "BuildersSorting" ON "Builders"(priority);
            CREATE INDEX IF NOT EXISTS "BuildersJoin" ON "Builders"(builder_class_name);
    """.trimIndent()

    private val dropScheme = """
            DROP TABLE IF EXISTS "Builders";
            DROP INDEX IF EXISTS "BuildersSearch";
            DROP INDEX IF EXISTS "BuildersSorting";
            DROP INDEX IF EXISTS "BuildersJoin";
    """.trimIndent()

    override fun onSignal(signal: JcSignal) {
        when (signal) {
            is JcSignal.BeforeIndexing -> {
                signal.jcdb.persistence.write {
                    if (signal.clearOnStart) {
                        it.executeQueries(dropScheme)
                    }
                    it.executeQueries(createScheme)
                }
            }

            is JcSignal.LocationRemoved -> {
                signal.jcdb.persistence.write {
                    it.deleteFrom(BUILDERS).where(BUILDERS.LOCATION_ID.eq(signal.location.id)).execute()
                }
            }

            is JcSignal.AfterIndexing -> {
                signal.jcdb.persistence.write {
                    it.executeQueries(createIndex)
                }
            }

            is JcSignal.Drop -> {
                signal.jcdb.persistence.write {
                    it.deleteFrom(BUILDERS).execute()
                }
            }

            else -> Unit
        }
    }

    override suspend fun query(classpath: JcClasspath, req: Set<String>): Sequence<BuildersResponse> {
        return syncQuery(classpath, req)
    }

    fun syncQuery(classpath: JcClasspath, req: Set<String>): Sequence<BuildersResponse> {
        val locationIds = classpath.registeredLocations.map { it.id }
        val persistence = classpath.db.persistence
        return sequence {
            val result = persistence.read { jooq ->
                jooq.select(BUILDERS.OFFSET, SYMBOLS.NAME, CLASSES.ID, CLASSES.LOCATION_ID, BUILDERS.PRIORITY)
                    .from(BUILDERS)
                    .join(SYMBOLS).on(SYMBOLS.NAME.eq(BUILDERS.BUILDER_CLASS_NAME))
                    .join(CLASSES).on(CLASSES.NAME.eq(SYMBOLS.ID).and(BUILDERS.LOCATION_ID.eq(CLASSES.LOCATION_ID)))
                    .where(
                        BUILDERS.CLASS_NAME.`in`(req).and(BUILDERS.LOCATION_ID.`in`(locationIds))


                    )
                    .limit(100)
                    .fetch()
                    .mapNotNull { (offset, className, classId, locationId, priority) ->
                        BuildersResponse(
                            source = PersistenceClassSource(
                                classpath.db,
                                locationId = locationId!!,
                                classId = classId!!,
                                className = className!!,
                            ),
                            methodOffset = offset!!,
                            priority = priority ?: 0
                        )
                    }.sortedByDescending { it.priority }
            }
            yieldAll(result)
        }

    }

    override fun newIndexer(jcdb: JcDatabase, location: RegisteredLocation) =
        BuildersIndexer(jcdb.persistence, location)


}