<?php

/**
 * The quasi-random generator class.
 */

namespace CryptoManana\Randomness;

use CryptoManana\Core\Abstractions\Randomness\AbstractGenerator as RandomnessSource;
use CryptoManana\Core\Interfaces\Randomness\SeedableGeneratorInterface as SeedAction;
use CryptoManana\Core\StringBuilder as StringBuilder;

/**
 * Class QuasiRandom - The quasi-random generator object.
 *
 * @package CryptoManana\Randomness
 */
class QuasiRandom extends RandomnessSource implements SeedAction
{
    /**
     * The maximum supported integer.
     */
    const QUASI_INT_MAX = 32767;

    /**
     * The minimum supported integer.
     */
    const QUASI_INT_MIN = -32768;

    /**
     * The quasi algorithm default leap step.
     */
    const QUASI_LEAP_STEP = 1;

    /**
     * The quasi algorithm default skip step.
     */
    const QUASI_SKIP_STEP = 1;

    /**
     * The generated integer sequence property storage for all instances.
     *
     * @var array Quasi-random integer sequence.
     */
    protected static $randomNumbers = [];

    /**
     * The generated byte sequence property storage for all instances.
     *
     * @var array Quasi-random byte sequence as integers.
     */
    protected static $randomBytes = [];

    /**
     * Pointer to the next number from the quasi sequence.
     *
     * @uses QuasiRandom::$randomNumbers For fetching.
     *
     * @var int The next position for access.
     */
    protected static $lastNumberIndex = 0;

    /**
     * Pointer to the next byte from the quasi sequence.
     *
     * @uses QuasiRandom::$randomBytes For fetching.
     *
     * @var int The next position for access.
     */
    protected static $lastByteIndex = 0;

    /**
     * The initialization seed value property storage for all instances.
     *
     * @var bool|int The generator's seed value.
     */
    protected static $seed = false;

    /**
     * Custom pseudo-random number generator used for the shuffling of internal sets.
     *
     * @param int $minimum The lowest value to be returned.
     * @param int $maximum The highest value to be returned.
     *
     * @return int Randomly generated integer number.
     */
    protected static function internalNumberGenerator($minimum = 0, $maximum = 65535)
    {
        // Seed is multiplied by a prime number and divided by modulus of a prime number
        self::$seed = (self::$seed * 127) % 2976221 + 1;

        // Return number in the supported range
        return self::$seed % ($maximum - $minimum + 1) + $minimum;
    }

    /**
     * Validates the given seed value and converts it to an integer.
     *
     * @param int|mixed $seed The initialization value.
     *
     * @return int The valid initialization value.
     * @throws \Exception Validation errors.
     */
    protected static function validateSeedValue($seed)
    {
        $seed = filter_var(
            $seed,
            FILTER_VALIDATE_INT,
            [
                "options" => [
                    "min_range" => self::QUASI_INT_MIN,
                    "max_range" => self::QUASI_INT_MAX,
                ],
            ]
        );

        if ($seed === false) {
            throw new \DomainException(
                "The provided seed value is of invalid type or is out of the supported range."
            );
        }

        return $seed;
    }

    /**
     * Reset internal pointers to the first element of each internal set.
     */
    protected static function resetQuasiIndexes()
    {
        self::$lastNumberIndex = 0;
        self::$lastByteIndex = 0;
    }

    /**
     * Generate an internal quasi-random numerical sequence.
     *
     * @param string $setName The internal static set name as a string.
     * @param int $seed The used seed value for the internal generator.
     * @param int $from The starting value in the range.
     * @param int $to The ending value in the range.
     * @param int $leapStep The leap step used.
     * @param int $skipStep The skip step used.
     */
    protected static function createQuasiSequence($setName, $seed, $from, $to, $leapStep, $skipStep)
    {
        // Set the seed value used for shuffling operations
        self::$seed = $seed;

        // Generate range with leap step
        self::${$setName} = range($from, $to, $leapStep);

        // Shuffle/scramble the numeric set
        $count = count(self::${$setName});
        $lastIndex = $count - 1;

        for ($currentIndex = 0; $currentIndex < $count; $currentIndex++) {
            $randomIndex = self::internalNumberGenerator(0, $lastIndex);

            $tmp = self::${$setName}[$currentIndex];
            self::${$setName}[$currentIndex] = self::${$setName}[$randomIndex];
            self::${$setName}[$randomIndex] = $tmp;
        }

        // Skip a few points
        array_splice(self::${$setName}, 0, $skipStep);
    }

    /**
     * Generate all needed quasi-random internal sequences.
     *
     * @param int $seed The used seed value for the internal generator.
     */
    protected static function generateSequences($seed)
    {
        // Calculate the skip step for usage
        $skipStep = (abs($seed) % 2 === 0) ? self::QUASI_SKIP_STEP : 0;

        // Generate range with leap step, from -32768 to 32767 => 16 bits
        self::createQuasiSequence(
            'randomNumbers',
            /** @see QuasiRandom::$randomNumbers */
            $seed,
            self::QUASI_INT_MIN,
            self::QUASI_INT_MAX,
            self::QUASI_LEAP_STEP,
            $skipStep
        );

        // Generate range with leap step, from 0 to 255 => 8 bits
        self::createQuasiSequence(
            'randomBytes',
            /** @see QuasiRandom::$randomBytes */
            $seed,
            0,
            255,
            self::QUASI_LEAP_STEP,
            $skipStep
        );

        // Reset to always start from the first element of both internal sets
        self::resetQuasiIndexes();
    }

    /**
     * Lookup method for searching in the internal integer set.
     *
     * @param int $minimum The lowest value to be returned.
     * @param int $maximum The highest value to be returned.
     *
     * @return int|null An proper integer number or `null` if not found.
     */
    protected static function lookupSequence($minimum, $maximum)
    {
        // Get the internal set size
        $internalCount = count(self::$randomNumbers);

        // If the whole sequence has been iterated at last call
        if (self::$lastNumberIndex === $internalCount) {
            self::$lastNumberIndex = 0; // Start over
        }

        // Lookup the sequence
        for ($i = self::$lastNumberIndex; $i < $internalCount; $i++) {
            // Update the lookup index and fetch the next number
            $number = self::$randomNumbers[$i];
            self::$lastNumberIndex = $i + 1;

            // If the number is in range, then return it
            if ($number >= $minimum && $number <= $maximum) {
                return $number;
            }
        }

        // Mark as not found on this iteration
        return null;
    }

    /**
     * Internal static method for single point consumption of the randomness source that outputs integers.
     *
     * @param int $minimum The lowest value to be returned.
     * @param int $maximum The highest value to be returned.
     *
     * @return int Randomly generated integer number.
     */
    protected static function getInteger($minimum, $maximum)
    {
        // Speed optimization for internal byte generation faster access
        if ($minimum === 0 && $maximum === 255) {
            $integer = unpack('C', self::getEightBits(1));

            return reset($integer); // Always an integer
        }

        do {
            // Searches for a proper number in the supported range
            $tmpNumber = self::lookupSequence($minimum, $maximum);
        } while ($tmpNumber === null);

        return $tmpNumber;
    }

    /**
     * Internal static method for single point consumption of the randomness source that outputs bytes.
     *
     * @param int $count The output string length based on the requested number of bytes.
     *
     * @return string Randomly generated string containing the requested number of bytes.
     */
    protected static function getEightBits($count)
    {
        $tmpBytes = '';

        $internalCount = count(self::$randomBytes);

        // Lookup the sequence
        for ($i = 1; $i <= $count; $i++) {
            // If the whole sequence has been iterated at last call
            if (self::$lastByteIndex === $internalCount) {
                self::$lastByteIndex = 0; // Start over
            }

            // Update the lookup index and fetch the next byte
            $byte = self::$randomBytes[self::$lastByteIndex];
            self::$lastByteIndex = self::$lastByteIndex + 1;

            $tmpBytes .= StringBuilder::getChr($byte);
        }

        return $tmpBytes;
    }

    /**
     * The maximum supported integer.
     *
     * @return int The upper integer generation border.
     */
    public function getMaxNumber()
    {
        return self::QUASI_INT_MAX;
    }

    /**
     * The minimum supported integer.
     *
     * @return int The lower integer generation border.
     */
    public function getMinNumber()
    {
        return self::QUASI_INT_MIN;
    }

    /**
     * The quasi-random generator constructor.
     *
     * Note: This type of generator is auto-seeded on the first object creation.
     */
    public function __construct()
    {
        parent::__construct();

        if (self::$seed === false) {
            self::setSeed();
        }
    }

    /**
     * Get debug information for the class instance.
     *
     * @return array Debug information.
     */
    public function __debugInfo()
    {
        return [
            'systemPrecision' => self::$systemPrecision,
            'quasiNumbersCount' => count(self::$randomNumbers),
            'quasiBytesCount' => count(self::$randomBytes),
            'seed' => self::$seed,
        ];
    }

    /**
     * Seed the generator initialization or invoke auto-seeding.
     *
     * Note: Invokes auto-seeding if the `null` value is passed.
     *
     * @param null|int $seed The initialization value.
     *
     * @throws \Exception Validation errors.
     */
    public static function setSeed($seed = null)
    {
        // If the seed is the same, just reset internal pointers
        if (!is_bool(self::$seed) && self::$seed === $seed) {
            self::resetQuasiIndexes();

            return;
        }

        // Seed the internal generator used for shuffling
        if (!is_null($seed)) {
            // Validate the input seed value
            $seed = self::validateSeedValue($seed);

            // Save original value before transformations
            $originalSeedValue = $seed;

            // Allow negative values and use them as initial for the internal shuffling
            $seed = ($seed < 0) ? QuasiRandom::QUASI_INT_MAX + 1 + (int)abs($seed) : (int)$seed;
        } else {
            // Get the seed value in the supported range
            $seed = (time() + 42) % (QuasiRandom::QUASI_INT_MAX + 1);

            // Save the auto-seed value
            $originalSeedValue = $seed;
        }

        // Generate internal sequences
        self::generateSequences($seed);

        // Set the used seed value for history
        self::$seed = $originalSeedValue;
    }

    /**
     * Generate a random integer number in a certain range.
     *
     * Note: Passing `null` will use the default parameter value.
     *
     * @param null|int $from The lowest value to be returned (default => 0).
     * @param null|int $to The highest value to be returned (default => $this->getMaxNumber()).
     *
     * @return int Randomly generated integer number.
     * @throws \Exception Validation errors.
     */
    public function getInt($from = 0, $to = null)
    {
        $from = ($from === null) ? 0 : $from;
        $to = ($to === null) ? $this->getMaxNumber() : $to;

        $this->validateIntegerRange($from, $to);

        // Speed optimization for boolean type generation
        if ($from === 0 && $to === 1) {
            return (int)$this->getBool();
        }

        return self::getInteger($from, $to);
    }

    /**
     * Generate a random byte string.
     *
     * Note: PHP represents bytes as characters to make byte strings.
     *
     * @param int $length The output string length (default => 1).
     *
     * @return string Randomly generated string containing the requested number of bytes.
     * @throws \Exception Validation errors.
     */
    public function getBytes($length = 1)
    {
        $this->validatePositiveInteger($length);

        return self::getEightBits($length);
    }

    /**
     * Generate a random boolean.
     *
     * @return bool Randomly generated boolean value.
     * @throws \Exception Validation errors.
     */
    public function getBool()
    {
        /**
         * {@internal Complete override because of the `$number % 2` is not with ~0.5 probability in the numeric set. }}
         */
        return self::getInteger(0, 255) > 127; // ~0.5 probability
    }
}
