<?php

declare(strict_types=1);

namespace TomasVotruba\PHPStanBodyscan\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\ConsoleOutput;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use TomasVotruba\PHPStanBodyscan\Logger;
use TomasVotruba\PHPStanBodyscan\OutputFormatter\JsonOutputFormatter;
use TomasVotruba\PHPStanBodyscan\OutputFormatter\TableOutputFormatter;
use TomasVotruba\PHPStanBodyscan\PHPStanConfigFactory;
use TomasVotruba\PHPStanBodyscan\Process\AnalyseProcessFactory;
use TomasVotruba\PHPStanBodyscan\Process\PHPStanResultResolver;
use TomasVotruba\PHPStanBodyscan\Utils\FileLoader;
use TomasVotruba\PHPStanBodyscan\ValueObject\BodyscanResult;
use TomasVotruba\PHPStanBodyscan\ValueObject\LevelResult;
use Webmozart\Assert\Assert;

final class RunCommand extends Command
{
    /**
     * @var array<int, string>
     */
    private const DOT_STATES = ['   ', '.  ', '.. ', '...', '....', '.....'];

    public function __construct(
        private readonly SymfonyStyle $symfonyStyle,
        private readonly AnalyseProcessFactory $analyseProcessFactory,
        private readonly PHPStanConfigFactory $phpStanConfigFactory,
        private readonly JsonOutputFormatter $jsonOutputFormatter,
        private readonly TableOutputFormatter $tableOutputFormatter,
        private readonly PHPStanResultResolver $phpStanResultResolver
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this->setName('run');
        $this->setAliases(['scan', 'analyse']);
        $this->setDescription('Check classes that are not used in any config and in the code');

        $this->addOption('min-level', null, InputOption::VALUE_REQUIRED, 'Min PHPStan level to run', 0);
        $this->addOption('max-level', null, InputOption::VALUE_REQUIRED, 'Max PHPStan level to run', 8);
        $this->addOption('timeout', null, InputOption::VALUE_OPTIONAL, 'Set PHPStan process timeout in seconds');
        $this->addOption('env-file', null, InputOption::VALUE_REQUIRED, 'Path to project .env file');
        $this->addOption('json', null, InputOption::VALUE_NONE, 'Show result in JSON');

        $this->addOption(
            'bare',
            null,
            InputOption::VALUE_NONE,
            'Without any extensions, without ignores, without baselines, just pure PHPStan'
        );

        $this->addOption('no-ignore', null, InputOption::VALUE_NONE, 'Run PHPStan without any ignores/baselines');
    }

    /**
     * @param ConsoleOutput $output
     * @return Command::*
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        /** @var string $projectDirectory */
        $projectDirectory = getcwd();

        $minPhpStanLevel = (int) $input->getOption('min-level');
        $maxPhpStanLevel = (int) $input->getOption('max-level');
        Assert::lessThanEq($minPhpStanLevel, $maxPhpStanLevel);
        $phpStanTimeout = $input->getOption('timeout') ? (int) $input->getOption('timeout') : null;
        $isBare = (bool) $input->getOption('bare');
        $isJson = (bool) $input->getOption('json');
        $isNoIgnore = (bool) $input->getOption('no-ignore');

        // silence output till the end to avoid invalid json format
        if ($isJson) {
            $this->symfonyStyle->setVerbosity(OutputInterface::VERBOSITY_QUIET);
        }

        $envVariables = $this->loadEnvVariables($input);

        // 1. prepare empty phpstan config
        // no baselines, ignores etc. etc :)
        $phpstanConfig = $this->phpStanConfigFactory->create($projectDirectory, [], $isBare, $isNoIgnore);
        file_put_contents($projectDirectory . '/phpstan-bodyscan.neon', $phpstanConfig->getFileContents());

        $levelResults = [];

        if ($isBare) {
            // temporarily disable project PHPStan extensions
            $phpstanExtensionFile = $projectDirectory . '/vendor/phpstan/extension-installer/src/GeneratedConfig.php';
            if (file_exists($phpstanExtensionFile)) {
                $this->symfonyStyle->writeln('Disabling PHPStan extensions...');
                $this->symfonyStyle->newLine();
                rename($phpstanExtensionFile, $phpstanExtensionFile . '.bak');
            }
        }

        // 2. measure phpstan levels
        for ($phpStanLevel = $minPhpStanLevel; $phpStanLevel <= $maxPhpStanLevel; ++$phpStanLevel) {
            $infoMessage = '<info>' . sprintf('Running PHPStan level %d%s', $phpStanLevel, $isBare ?
                    ' without extensions' : '') . '</info>';

            $section = $output->section();

            for ($i = 0; $i < 20; ++$i) {
                $stateIndex = $i % count(self::DOT_STATES);
                $section->overwrite($infoMessage . ': ' . self::DOT_STATES[$stateIndex]);
                usleep(700_000);
            }

            $levelResult = $this->measureErrorCountInLevel($phpStanLevel, $projectDirectory, $envVariables, $phpStanTimeout);
            $levelResults[] = $levelResult;

            $section->overwrite(sprintf($infoMessage . ': found %d errors', $levelResult->getErrorCount()));
        }

        if ($isBare) {
            // restore PHPStan extension file
            $this->symfonyStyle->writeln('Restoring PHPStan extensions...');
            $this->symfonyStyle->newLine();

            $phpstanExtensionFile = $projectDirectory . '/vendor/phpstan/extension-installer/src/GeneratedConfig.php';
            rename($phpstanExtensionFile . '.bak', $phpstanExtensionFile);
        }

        $bodyscanResult = new BodyscanResult($levelResults);

        // 3. tidy up temporary config
        unlink($projectDirectory . '/phpstan-bodyscan.neon');

        if ($isJson) {
            $this->jsonOutputFormatter->outputResult($bodyscanResult);
        } else {
            $this->tableOutputFormatter->outputResult($bodyscanResult);
        }

        return self::SUCCESS;
    }

    /**
     * @param array<string, mixed> $envVariables
     */
    private function measureErrorCountInLevel(
        int $phpStanLevel,
        string $projectDirectory,
        array $envVariables,
        ?int $phpStanTimeout
    ): LevelResult {
        $process = $this->analyseProcessFactory->create($projectDirectory, $phpStanLevel, $envVariables, $phpStanTimeout);
        $process->run();

        $result = $this->phpStanResultResolver->resolve($process);
        $fileErrorCount = (int) $result['totals']['file_errors'];

        Logger::log(sprintf(
            'Project directory "%s" - PHPStan level %d: %d errors',
            $projectDirectory,
            $phpStanLevel,
            $fileErrorCount
        ));

        return new LevelResult($phpStanLevel, $fileErrorCount);
    }

    /**
     * @return array<string, string>
     */
    private function loadEnvVariables(InputInterface $input): array
    {
        $envFile = $input->getOption('env-file');
        if (! is_string($envFile)) {
            return [];
        }

        $this->symfonyStyle->note(sprintf('Adding envs from "%s" file:', $envFile));
        $this->symfonyStyle->newLine();

        return FileLoader::resolveEnvVariablesFromFile($envFile);
    }
}
