use std::ops::BitAnd;

use base::prelude::*;

use tracing::{event, Level};

use super::super::event::OutputEvent;
use super::super::*;
use super::alarm::{Alarm, AlarmDetails, Alarmer, BadMemOp};
use super::alarmunit::AlarmUnit;
use super::context::Context;
use super::control::{
    ControlRegisters, ControlUnit, DeviceManager, OpcodeResult, ProgramCounterChange, TrapCircuit,
};
use super::exchanger::exchanged_value_for_load;
use super::io::{TransferFailed, Unit};
use super::memory::{MemoryMapped, MemoryOpFailure, MemoryUnit, MetaBitChange};

#[derive(Debug, PartialEq, Eq)]
enum TransferOutcome {
    /// Indicates a successful transfer operation.
    Success {
        /// Indicates that the memory location's meta bit was set.
        /// This allows the trap circuit to be triggered if necessary.
        metabit_was_set: bool,

        /// Describes the output (if any) generated by this
        /// instruction.
        output: Option<OutputEvent>,
    },

    /// When the outcome of the TSD is dismiss and wait, we don't
    /// trigger the trap circuit (not because we think the TX-2 behaved
    /// this way, but because it keeps the code simpler and we don't
    /// know if the oposite behaviour is needed).
    DismissAndWait,
}

type OpConversion = fn(Address) -> BadMemOp;

fn bad_write(addr: Address) -> BadMemOp {
    BadMemOp::Write(addr.into())
}

impl ControlUnit {
    /// Implements the IOS opcode
    pub(crate) fn op_ios(
        &mut self,
        ctx: &Context,
        mem: &mut MemoryUnit,
        devices: &mut DeviceManager,
    ) -> Result<OpcodeResult, Alarm> {
        let j = self.regs.n.index_address();
        let cf = self.regs.n.configuration();

        if cf & 1 != 0 {
            // Setting the report bit in the configuration value
            // causes the device's status word from before any mode
            // change to be copied into the E register (as stated in
            // section 4-3.6 of the User Handbook).
            //
            // Note that this means that the current sequence and the
            // sequence for which we are generating a report word are
            // different (in general).
            let flag_raised: bool = self.regs.flags.current_flag_state(&j);
            mem.set_e_register(devices.report(
                ctx,
                self.regs.k,
                j,
                flag_raised,
                &mut self.alarm_unit,
            )?);
        }
        let mut dismiss_reason: Option<&str> = if cf & 0o20 != 0 {
            Some("dismiss bit set in config")
        } else {
            None
        };

        let operand = self.regs.n.operand_address_and_defer_bit();
        let result = match u32::from(operand) {
            0o20_000 => devices.disconnect(ctx, &j, &mut self.alarm_unit),
            0o30_000..=0o37_777 => {
                let mode: Unsigned12Bit = Unsigned12Bit::try_from(operand & 0o07_777).unwrap();
                ControlUnit::connect_unit(
                    ctx,
                    devices,
                    &mut self.regs,
                    &mut self.trap,
                    j,
                    mode,
                    &mut self.alarm_unit,
                )
            }
            0o40_000 => {
                self.regs.flags.lower(&j);
                if self.regs.k == Some(j) {
                    // J is the current sequence, so the flag is
                    // lowered but don't perform a drop-out (see User
                    // Handbook page 4-7).
                    self.regs.current_sequence_is_runnable = true;
                }
                Ok(())
            }
            0o50_000 => {
                self.regs.flags.raise(&j);
                if Some(j) == self.regs.k {
                    dismiss_reason = None;
                }
                Ok(())
            }
            0o60_000..=0o60777 => {
                // Select unit XXX
                Err(self.alarm_unit.always_fire(Alarm {
                    sequence: self.regs.k,
                    details: AlarmDetails::ROUNDTUITAL(format!(
                        "IOS operand {:o}: Select Unit command is not yet implemented.",
                        operand
                    )),
                }))
            }
            _ => {
                let command: u8 = (u32::from(operand) >> 12) as u8;
                self.alarm_unit.fire_if_not_masked(Alarm {
                    sequence: self.regs.k,
                    details: AlarmDetails::IOSAL {
                        unit: j,
                        operand: Some(operand),
                        message: format!(
                            "IOS operand {:o} has unrecognised leading command digit {:o}",
                            operand, command,
                        ),
                    },
                })?;
                // IOSAL is masked.  Just do nothing.
                Ok(())
            }
        };
        if let Some(reason) = dismiss_reason {
            self.dismiss_unless_held(reason);
        }
        result.map(|()| OpcodeResult {
            program_counter_change: None,
            // poll_order_change doesn't always need to be set, but
            // false positives cost us only compute efficiency.
            poll_order_change: Some(j),
            output: None,
        })
    }

    fn connect_unit(
        ctx: &Context,
        devices: &mut DeviceManager,
        regs: &mut ControlRegisters,
        trap: &mut TrapCircuit,
        unit: Unsigned6Bit,
        mode: Unsigned12Bit,
        alarm_unit: &mut AlarmUnit,
    ) -> Result<(), Alarm> {
        let maybe_flag_change: Option<FlagChange> = match u8::from(unit) {
            0o42 => {
                trap.connect(ctx, mode);
                None
            }
            _ => devices.connect(ctx, regs.k, &unit, mode, alarm_unit)?,
        };
        if let Some(FlagChange::Raise) = maybe_flag_change {
            regs.flags.raise(&unit);
        }
        Ok(())
    }

    pub(crate) fn op_tsd(
        &mut self,
        ctx: &Context,
        devices: &mut DeviceManager,
        execution_address: Address,
        mem: &mut MemoryUnit,
    ) -> Result<OpcodeResult, Alarm> {
        fn make_tsd_qsal(seq: Option<SequenceNumber>, inst: Instruction, op: BadMemOp) -> Alarm {
            Alarm {
                sequence: seq,
                details: AlarmDetails::QSAL(inst, op, "TSD address is not mapped".to_string()),
            }
        }

        let result: Result<TransferOutcome, Alarm> = if let Some(unit) = self.regs.k {
            let target: Address = self.operand_address_with_optional_defer_and_index(ctx, mem)?;
            let not_mapped = |op_conv: OpConversion| -> Alarm {
                let op: BadMemOp = op_conv(target);
                make_tsd_qsal(self.regs.k, self.regs.n, op)
            };

            let meta_op: MetaBitChange = if self.trap.set_metabits_of_operands() {
                MetaBitChange::Set
            } else {
                MetaBitChange::None
            };
            // There are no sequence numbers below 0o40, besides 0.
            if matches!(u8::from(unit), 0 | 0o75 | 0o76) {
                // Non-INOUT sequences just cycle the target location;
                // see section 4-1 (page 4-3) of the Users Handbook;
                // also pages 4-2 and 4-9).
                match mem.cycle_word(ctx, &target) {
                    Ok(extra_bits) => Ok(TransferOutcome::Success {
                        metabit_was_set: extra_bits.meta,
                        output: None,
                    }),
                    Err(MemoryOpFailure::ReadOnly(_address, extra_bits)) => {
                        // The read-only case is not an error, it's
                        // normal.  The TSD instruction simply has no
                        // effect when the target address is
                        // read-only.
                        // TODO: should there be an effect on the E register?
                        Ok(TransferOutcome::Success {
                            metabit_was_set: extra_bits.meta,
                            output: None, // not an INOUT unit anyway
                        })
                    }
                    Err(MemoryOpFailure::NotMapped(_)) => {
                        self.alarm_unit.fire_if_not_masked(not_mapped(bad_write))?;
                        // QSAL is masked, carry on.
                        Ok(TransferOutcome::Success {
                            metabit_was_set: false, // act as if metabit unset
                            output: None,
                        })
                    }
                }
            } else {
                devices.mark_device_changed(unit);
                match devices.get_mut(&unit) {
                    None => {
                        event!(Level::WARN, "TSD on unknown unit {:o}", unit);
                        Ok(TransferOutcome::DismissAndWait)
                    }
                    Some(device) => {
                        let is_input_unit = device.is_input_unit;
                        if !device.connected {
                            // If the unit is not connected, perform
                            // dismiss and wait.  This requirement is
                            // described in section 4-3.7 of the Users
                            // Handbook.
                            Ok(TransferOutcome::DismissAndWait)
                        } else if device.in_maintenance {
                            event!(
                                Level::WARN,
                                "TSD on unit {:o}, but it is in maintenance",
                                unit
                            );
                            Ok(TransferOutcome::DismissAndWait)
                        } else {
                            // We're actually going to do the (input or output) transfer.
                            // First load into the M register the existing contents of
                            // memory.
                            let transfer_mode = device.transfer_mode(&mut self.alarm_unit)?;
                            let (m_register, extra_bits) = self
                                .fetch_operand_from_address_without_exchange(
                                    ctx,
                                    mem,
                                    &target,
                                    &UpdateE::No,
                                )?;
                            if is_input_unit {
                                // In read operations, data is transferred
                                // from the I/O device's buffer over the
                                // IOBM bus, into the E register.  See
                                // figure 15-18 in Volume 2 of the TX-2
                                // Techical Manual.
                                match device.read(ctx, &mut self.alarm_unit) {
                                    Ok(masked_word) => {
                                        const UPDATE_E_YES: UpdateE = UpdateE::Yes;
                                        let newval: Unsigned36Bit =
                                            masked_word.apply(Unsigned36Bit::ZERO);
                                        match transfer_mode {
                                            TransferMode::Assembly => {
                                                let bits: Unsigned6Bit =
                                                    newval.bitand(Unsigned6Bit::MAX);
                                                self.memory_store_without_exchange(
                                                    ctx,
                                                    mem,
                                                    &target,
                                                    &cycle_and_splay(m_register, bits),
                                                    &UPDATE_E_YES,
                                                    &meta_op,
                                                )?;
                                            }
                                            TransferMode::Exchange => {
                                                self.memory_store_with_exchange(
                                                    ctx,
                                                    mem,
                                                    &target,
                                                    &newval,
                                                    &m_register,
                                                    &UPDATE_E_YES,
                                                    &meta_op,
                                                )?;
                                            }
                                        }
                                        Ok(TransferOutcome::Success {
                                            metabit_was_set: extra_bits.meta,
                                            output: None, // because this is a read unit.
                                        })
                                    }
                                    Err(TransferFailed::BufferNotFree) => {
                                        Ok(TransferOutcome::DismissAndWait)
                                    }
                                    Err(TransferFailed::Alarm(alarm)) => {
                                        return Err(alarm);
                                    }
                                }
                            } else {
                                // In write operations, data is
                                // transferred from the E register to
                                // the I/O device over the E bus.  See
                                // figure 15-17 in Volume 2 of the
                                // TX-2 Techical Manual.
                                mem.set_e_register(exchanged_value_for_load(
                                    &self.get_config(),
                                    &m_register,
                                    &mem.get_e_register(),
                                ));
                                match transfer_mode {
                                    TransferMode::Exchange => {
                                        match device.write(ctx, mem.get_e_register()) {
                                            Err(TransferFailed::BufferNotFree) => {
                                                Ok(TransferOutcome::DismissAndWait)
                                            }
                                            Err(TransferFailed::Alarm(alarm)) => Err(alarm),
                                            Ok(maybe_output) => Ok(TransferOutcome::Success {
                                                metabit_was_set: extra_bits.meta,
                                                output: maybe_output,
                                            }),
                                        }
                                    }
                                    TransferMode::Assembly => Err(self
                                        .alarm_unit
                                        .always_fire(Alarm {
                                        sequence: self.regs.k,
                                        details: AlarmDetails::ROUNDTUITAL(
                                            "TSD output in assembly mode is not yet implemented."
                                                .to_string(),
                                        ),
                                    })),
                                }
                            }
                        }
                    }
                }
            }
        } else {
            Err(self.alarm_unit.always_fire(Alarm {
                sequence: self.regs.k,
                details: AlarmDetails::BUGAL {
                instr: Some(self.regs.n),
                message: "Executed TSD instruction while the K register is None (i.e. there is no current sequence)".to_string(),
                }}))
        };
        match result {
            Ok(TransferOutcome::Success {
                metabit_was_set,
                output,
            }) => {
                if metabit_was_set && self.trap.trap_on_operand() {
                    self.raise_trap();
                }
                Ok(OpcodeResult {
                    program_counter_change: None,
                    poll_order_change: self.regs.k,
                    output,
                })
            }
            Ok(TransferOutcome::DismissAndWait) => {
                // In the dismiss and wait case, the
                // sequence is dismissed even if the hold
                // bit is set (Users Handbook, section
                // 4-3.2).  The hold bit only governs what
                // happens following the completion of an
                // instruction.
                self.dismiss("TSD while data was not ready caused dismiss-and-wait");
                Ok(OpcodeResult {
                    program_counter_change: Some(ProgramCounterChange::DismissAndWait(
                        execution_address,
                    )),
                    poll_order_change: self.regs.k,
                    output: None,
                })
            }
            Err(e) => Err(e),
        }
    }
}
