export declare namespace ej {
  
  export namespace base {
    
    /**
     * Ajax class provides ability to make asynchronous HTTP request to the server
     * ```typescript
     *   var ajax = new Ajax("index.html", "GET", true);
     *   ajax.send().then(
     *               function (value) {
     *                   console.log(value);
     *               },
     *               function (reason) {
     *                   console.log(reason);
     *               });
     * ```
     */
    export class Ajax {
        /**
         * Specifies the URL to which request to be sent.
         * @default null
         */
        url: string;
        /**
         * Specifies which HTTP request method to be used. For ex., GET, POST
         * @default GET
         */
        type: string;
        /**
         * Specifies the data to be sent.
         * @default null
         */
        data: string | Object;
        /**
         * A boolean value indicating whether the request should be sent asynchronous or not.
         * @default true
         */
        mode: boolean;
        /**
         * Specifies the callback for creating the XMLHttpRequest object.
         * @default null
         */
        httpRequest: XMLHttpRequest;
        private options;
        private contentType;
        private dataType;
        /**
         * Constructor for Ajax class
         * @param  {string|Object} options?
         * @param  {string} type?
         * @param  {boolean} async?
         * @returns defaultType
         */
        constructor(options?: string | Object, type?: string, async?: boolean);
        /**
         * Send the request to server.
         * @param {any} data - To send the user data
         * @return {Promise}
         */
        send(data?: string | Object): Promise<Ajax>;
        /**
         * Specifies the callback function to be triggered before sending request to sever.
         * This can be used to modify the XMLHttpRequest object before it is sent.
         * @event
         */
        beforeSend: Function;
        /**
         * Specifies callback function to be triggered after XmlHttpRequest is succeeded.
         * The callback will contain server response as the parameter.
         * @event
         */
        onSuccess: Function;
        /**
         * Triggers when XmlHttpRequest is failed.
         * @event
         */
        onFailure: Function;
        private successHandler(data);
        private failureHandler(reason);
        private stateChange(resolve, reject);
        /**
         * To get the response header from XMLHttpRequest
         * @param  {string} key Key to search in the response header
         * @returns {string}
         */
        getResponseHeader(key: string): string;
    }
    export interface HeaderOptions {
        readyState?: number;
        getResponseHeader?: Function;
        setRequestHeader?: Function;
        overrideMimeType?: Function;
    }
    
    /**
     * Interface for a class Animation
     */
    export interface AnimationModel {
    
        /**
         * Specify the type of animation
         * @default : 'FadeIn';
         */
        name?: ej.base.Effect;
    
        /**
         * Specify the duration to animate
         * @default : 400;
         */
        duration?: number;
    
        /**
         * Specify the animation timing function
         * @default : 'ease';
         */
        timingFunction?: string;
    
        /**
         * Specify the delay to start animation
         * @default : 0;
         */
        delay?: number;
    
        /**
         * Triggers when animation is in-progress
         * @event
         */
        progress?: ej.base.EmitType<AnimationOptions>;
    
        /**
         * Triggers when the animation is started
         * @event
         */
        begin?: ej.base.EmitType<AnimationOptions>;
    
        /**
         * Triggers when animation is completed
         * @event
         */
        end?: ej.base.EmitType<AnimationOptions>;
    
        /**
         * Triggers when animation is failed due to any scripts
         * @event
         */
        fail?: ej.base.EmitType<AnimationOptions>;
    
    }
    
    /**
     * Animation effect names
     */
    export type Effect = 'FadeIn' | 'FadeOut' | 'FadeZoomIn' | 'FadeZoomOut' | 'FlipLeftDownIn' | 'FlipLeftDownOut' | 'FlipLeftUpIn' | 'FlipLeftUpOut' | 'FlipRightDownIn' | 'FlipRightDownOut' | 'FlipRightUpIn' | 'FlipRightUpOut' | 'FlipXDownIn' | 'FlipXDownOut' | 'FlipXUpIn' | 'FlipXUpOut' | 'FlipYLeftIn' | 'FlipYLeftOut' | 'FlipYRightIn' | 'FlipYRightOut' | 'SlideBottomIn' | 'SlideBottomOut' | 'SlideDown' | 'SlideLeft' | 'SlideLeftIn' | 'SlideLeftOut' | 'SlideRight' | 'SlideRightIn' | 'SlideRightOut' | 'SlideTopIn' | 'SlideTopOut' | 'SlideUp' | 'ZoomIn' | 'ZoomOut';
    /**
     * The Animation framework provide options to animate the html DOM elements
     * ```typescript
     *   let animeObject = new Animation({
     *      name: 'SlideLeftIn',
     *      duration: 1000
     *   });
     *   animeObject.animate('#anime1');
     *   animeObject.animate('#anime2', { duration: 500 });
     * ```
     */
    export class Animation extends Base<HTMLElement> implements ej.base.INotifyPropertyChanged {
        /**
         * Specify the type of animation
         * @default : 'FadeIn';
         */
        name: ej.base.Effect;
        /**
         * Specify the duration to animate
         * @default : 400;
         */
        duration: number;
        /**
         * Specify the animation timing function
         * @default : 'ease';
         */
        timingFunction: string;
        /**
         * Specify the delay to start animation
         * @default : 0;
         */
        delay: number;
        /**
         * Triggers when animation is in-progress
         * @event
         */
        progress: ej.base.EmitType<AnimationOptions>;
        /**
         * Triggers when the animation is started
         * @event
         */
        begin: ej.base.EmitType<AnimationOptions>;
        /**
         * Triggers when animation is completed
         * @event
         */
        end: ej.base.EmitType<AnimationOptions>;
        /**
         * Triggers when animation is failed due to any scripts
         * @event
         */
        fail: ej.base.EmitType<AnimationOptions>;
        /**
         * @private
         */
        easing: {
            [key: string]: string;
        };
        constructor(options: AnimationModel);
        /**
         * Applies animation to the current element.
         * @param {string | HTMLElement} element - Element which needs to be animated.
         * @param {AnimationModel} options - Overriding default animation settings.
         * @return {void}
         */
        animate(element: string | HTMLElement, options?: AnimationModel): void;
        /**
         * Stop the animation effect on animated element.
         * @param {HTMLElement} element - Element which needs to be stop the animation.
         * @param {AnimationOptions} model - Handling the animation model at stop function.
         * @return {void}
         */
        static stop(element: HTMLElement, model?: AnimationOptions): void;
        /**
         * Set delay to animation element
         * @param {AnimationModel} model
         * @returns {void}
         */
        private static delayAnimation(model);
        /**
         * Triggers animation
         * @param {AnimationModel} model
         * @returns {void}
         */
        private static applyAnimation(model);
        /**
         * Returns Animation Model
         * @param {AnimationModel} options
         * @returns {AnimationModel}
         */
        private getModel(options);
        /**
         * @private
         */
        onPropertyChanged(newProp: AnimationModel, oldProp: AnimationModel): void;
        /**
         * Returns module name as animation
         * @private
         */
        getModuleName(): string;
        /**
         * @private
         */
        destroy(): void;
    }
    /**
     * Animation event argument for progress event handler
     * @private
     */
    export interface AnimationOptions extends AnimationModel {
        /**
         * Get current time-stamp in progress EventHandler
         */
        timeStamp?: number;
        /**
         * Get current animation element in progress EventHandler
         */
        element?: HTMLElement;
    }
    /**
     * Ripple provides material theme's wave effect when an element is clicked
     * ```html
     * <div id='ripple'></div>
     * <script>
     *   rippleEffect(document.getElementById('ripple'));
     * </script>
     * ```
     * @private
     * @param HTMLElement element - Target element
     * @param RippleOptions rippleOptions - Ripple options .
     */
    export function rippleEffect(element: HTMLElement, rippleOptions?: RippleOptions, done?: Function): () => void;
    /**
     * Ripple method arguments to handle ripple effect
     * @private
     */
    export interface RippleOptions {
        /**
         * Get selector child elements for ripple effect
         */
        selector?: string;
        /**
         * Get ignore elements to prevent ripple effect
         */
        ignore?: string;
        /**
         * Override the enableRipple method
         */
        rippleFlag?: boolean;
        /**
         * Set ripple effect from center position
         */
        isCenterRipple?: boolean;
        /**
         * Set ripple duration
         */
        duration?: number;
    }
    export let isRippleEnabled: boolean;
    /**
     * Animation Module provides support to enable ripple effect functionality to Essential JS 2 components.
     * @param {boolean} isRipple Specifies the boolean value to enable or disable ripple effect.
     * @returns {boolean}
     */
    export function enableRipple(isRipple: boolean): boolean;
    
    export interface DomElements extends HTMLElement {
        ej2_instances: Object[];
    }
    export interface AngularEventEmitter {
        subscribe?: (generatorOrNext?: any, error?: any, complete?: any) => any;
    }
    export type EmitType<T> = AngularEventEmitter & ((arg?: T, ...rest: any[]) => void);
    /**
     * Base library module is common module for Framework modules like touch,keyboard and etc.,
     * @private
     */
    export abstract class Base<ElementType extends HTMLElement> {
        element: ElementType;
        isDestroyed: boolean;
        protected isProtectedOnChange: boolean;
        protected properties: {
            [key: string]: Object;
        };
        protected changedProperties: {
            [key: string]: Object;
        };
        protected oldProperties: {
            [key: string]: Object;
        };
        protected refreshing: boolean;
        protected finalUpdate: Function;
        protected modelObserver: Observer;
        protected childChangedProperties: {
            [key: string]: Object;
        };
        protected abstract getModuleName(): string;
        protected abstract onPropertyChanged(newProperties: Object, oldProperties?: Object): void;
        /** Property base section */
        /**
         * Function used to set bunch of property at a time.
         * @private
         * @param  {Object} prop - JSON object which holds components properties.
         * @param  {boolean} muteOnChange? - Specifies to true when we set properties.
         */
        setProperties(prop: Object, muteOnChange?: boolean): void;
        /**
         * Calls for child element data bind
         * @param {Object} obj
         * @param {Object} parent
         * @returns {void}
         */
        private static callChildDataBind(obj, parent);
        protected clearChanges(): void;
        /**
         * Bind property changes immediately to components
         */
        dataBind(): void;
        protected saveChanges(key: string, newValue: string, oldValue: string): void;
        /** Event Base Section */
        /**
         * Adds the handler to the given event listener.
         * @param {string} eventName - A String that specifies the name of the event
         * @param {Function} listener - Specifies the call to run when the event occurs.
         * @return {void}
         */
        addEventListener(eventName: string, handler: Function): void;
        /**
         * Removes the handler from the given event listener.
         * @param {string} eventName - A String that specifies the name of the event to remove
         * @param {Function} listener - Specifies the function to remove
         * @return {void}
         */
        removeEventListener(eventName: string, handler: Function): void;
        /**
         * Triggers the handlers in the specified event.
         * @private
         * @param {string} eventName - Specifies the event to trigger for the specified component properties.
         * Can be a custom event, or any of the standard events.
         * @param {Event} eventProp - Additional parameters to pass on to the event properties
         * @return {void}
         */
        trigger(eventName: string, eventProp?: Object): void;
        /**
         * Base constructor accept options and element
         */
        constructor(options: Object, element: ElementType | string);
        /**
         * To maintain instance in base class
         */
        protected addInstance(): void;
        /**
         * To remove the instance from the element
         */
        protected destroy(): void;
    }
    
    /**
     * Get configuration details for Browser
     * @private
     */
    export class Browser {
        private static uA;
        private static extractBrowserDetail();
        /**
         * To get events from the browser
         * @param {string} event - type of event triggered.
         * @returns {Boolean}
         */
        private static getEvent(event);
        /**
         * To get the Touch start event from browser
         * @returns {string}
         */
        private static getTouchStartEvent();
        /**
         * To get the Touch end event from browser
         * @returns {string}
         */
        private static getTouchEndEvent();
        /**
         * To get the Touch move event from browser
         * @returns {string}
         */
        private static getTouchMoveEvent();
        /**
         * To get the value based on provided key and regX
         * @param {string} key
         * @param {RegExp} regX
         * @returns {Object}
         */
        private static getValue(key, regX);
        /**
         * Property specifies the userAgent of the browser. Default userAgent value is based on the browser.
         * Also we can set our own userAgent.
         */
        static userAgent: string;
        /**
         * Property is to get the browser information like Name, Version and Language
         * @returns BrowserInfo
         */
        static readonly info: BrowserInfo;
        /**
         * Property is to get whether the userAgent is based IE.
         */
        static readonly isIE: Boolean;
        /**
         * Property is to get whether the browser has touch support.
         */
        static readonly isTouch: Boolean;
        /**
         * Property is to get whether the browser has Pointer support.
         */
        static readonly isPointer: Boolean;
        /**
         * Property is to get whether the browser has MSPointer support.
         */
        static readonly isMSPointer: Boolean;
        /**
         * Property is to get whether the userAgent is device based.
         */
        static readonly isDevice: Boolean;
        /**
         * Property is to get whether the userAgent is IOS.
         */
        static readonly isIos: Boolean;
        /**
         * Property is to get whether the userAgent is Ios7.
         */
        static readonly isIos7: Boolean;
        /**
         * Property is to get whether the userAgent is Android.
         */
        static readonly isAndroid: Boolean;
        /**
         * Property is to identify whether application ran in web view.
         */
        static readonly isWebView: Boolean;
        /**
         * Property is to get whether the userAgent is Windows.
         */
        static readonly isWindows: Boolean;
        /**
         * Property is to get the touch start event. It returns event name based on browser.
         */
        static readonly touchStartEvent: string;
        /**
         * Property is to get the touch move event. It returns event name based on browser.
         */
        static readonly touchMoveEvent: string;
        /**
         * Property is to get the touch end event. It returns event name based on browser.
         */
        static readonly touchEndEvent: string;
    }
    export interface BrowserDetails {
        isAndroid?: Boolean;
        isDevice?: Boolean;
        isIE?: Boolean;
        isIos?: Boolean;
        isIos7?: Boolean;
        isMSPointer?: Boolean;
        isPointer?: Boolean;
        isTouch?: Boolean;
        isWebView?: Boolean;
        isWindows?: Boolean;
        info?: BrowserInfo;
        touchStartEvent?: string;
        touchMoveEvent?: string;
        touchEndEvent?: string;
    }
    export interface BrowserInfo {
        name?: string;
        version?: string;
        culture?: {
            name?: string;
            language?: string;
        };
    }
    
    /**
     * @private
     */
    export class CanvasRenderer {
        private canvasObj;
        /**
         * Specifies root id of the canvas element
         * @default null
         */
        private rootId;
        /**
         * Specifies the height of the canvas element.
         * @default null
         */
        height: number;
        /**
         * Specifies the width of the canvas element.
         * @default null
         */
        width: number;
        /**
         * Specifies the context of the canvas.
         * @default null
         */
        ctx: CanvasRenderingContext2D;
        /**
         * Holds the context of the rendered canvas as string.
         * @default null
         */
        dataUrl: string;
        constructor(rootID: string);
        private getOptionValue<T>(options, key);
        /**
         * To create a Html5 canvas element
         * @param {BaseAttibutes} options - Options to create canvas
         * @return {HTMLCanvasElement}
         */
        createCanvas(options: BaseAttibutes): HTMLCanvasElement;
        /**
         * To set the width and height for the Html5 canvas element
         * @param {number} width - width of the canvas
         * @param {number} height - height of the canvas
         * @return {void}
         */
        setCanvasSize(width: number, height: number): void;
        private setAttributes(options);
        /**
         * To draw a line
         * @param {LineAttributes} options - required options to draw a line on the canvas
         * @return {void}
         */
        drawLine(options: LineAttributes): void;
        /**
         * To draw a rectangle
         * @param {RectAttributes} options - required options to draw a rectangle on the canvas
         * @return {void}
         */
        drawRectangle(options: RectAttributes): void;
        private drawCornerRadius(options);
        /**
         * To draw a path on the canvas
         * @param {PathAttributes} options - options needed to draw path
         * @param {Int32Array} canvasTranslate - Array of numbers to translate the canvas
         * @return {void}
         */
        drawPath(options: PathAttributes, canvasTranslate: Int32Array): void;
        /**
         * To draw a text
         * @param {TextAttributes} options - options required to draw text
         * @param {string} label - Specifies the text which has to be drawn on the canvas
         * @return {void}
         */
        drawText(options: TextAttributes, label: string): void;
        /**
         * To draw circle on the canvas
         * @param {CircleAttributes} options - required options to draw the circle
         * @return {void}
         */
        drawCircle(options: CircleAttributes): void;
        /**
         * To draw polyline
         * @param {PolylineAttributes} options - options needed to draw polyline
         * @return {void}
         */
        drawPolyline(options: PolylineAttributes): void;
        /**
         * To draw an ellipse on the canvas
         * @param {EllipseAttributes} options - options needed to draw ellipse
         * @return {void}
         */
        drawEllipse(options: EllipseAttributes): void;
        /**
         * To draw an image
         * @param {ImageAttributes} options - options required to draw an image on the canvas
         * @return {void}
         */
        drawImage(options: ImageAttributes): void;
        /**
         * To create a linear gradient
         * @param {string[]} colors - Specifies the colors required to create linear gradient
         * @return {string}
         */
        createLinearGradient(colors: GradientColor[]): string;
        /**
         * To create a radial gradient
         * @param {string[]} colors - Specifies the colors required to create linear gradient
         * @return {string}
         */
        createRadialGradient(colors: GradientColor[]): string;
        private setGradientValues(colors, myGradient);
        /**
         * To set the attributes to the element
         * @param {SVGCanvasAttributes} options - Attributes to set for the element
         * @param {HTMLElement} element - The element to which the attributes need to be set
         * @return {HTMLElement}
         */
        setElementAttributes(options: SVGCanvasAttributes, element: HTMLElement): HTMLElement;
        /**
         * To update the values of the canvas element attributes
         * @param {SVGCanvasAttributes} options - Specifies the colors required to create gradient
         * @return {void}
         */
        updateCanvasAttributes(options: SVGCanvasAttributes): void;
    }
    
    /**
     * To detect the changes for inner properties.
     * @private
     */
    export class ChildProperty<T> {
        private parentObj;
        private controlParent;
        private propName;
        private isParentArray;
        protected properties: {
            [key: string]: Object;
        };
        protected changedProperties: {
            [key: string]: Object;
        };
        protected childChangedProperties: {
            [key: string]: Object;
        };
        protected oldProperties: {
            [key: string]: Object;
        };
        protected finalUpdate: Function;
        private callChildDataBind;
        constructor(parent: T, propName: string, defaultValue: Object, isArray?: boolean);
        /**
         * Updates the property changes
         * @param {boolean} val
         * @param {string} propName
         * @returns {void}
         */
        private updateChange(val, propName);
        /**
         * Updates time out duration
         */
        private updateTimeOut();
        /**
         * Clears changed properties
         */
        private clearChanges();
        /**
         * Set property changes
         * @param {Object} prop
         * @param {boolean} muteOnChange
         * {void}
         */
        protected setProperties(prop: Object, muteOnChange: boolean): void;
        /**
         * Binds data
         */
        protected dataBind(): void;
        /**
         * Saves changes to newer values
         * @param {string} key
         * @param {Object} newValue
         * @param {Object} oldValue
         * @returns {void}
         */
        protected saveChanges(key: string, newValue: Object, oldValue: Object): void;
    }
    
    /**
     * Interface for a class Component
     */
    export interface ComponentModel {
    
        /**
         * Enable or disable persisting component's state between page reloads.
         * @default false
         */
        enablePersistence?: boolean;
    
        /**
         * Enable or disable rendering component in right to left direction.
         * @default false
         */
        enableRtl?: boolean;
    
        /**
         * Overrides the global culture and localization value for this component. Default global culture is 'en-US'.
         * @default undefined
         */
        locale?: string;
    
    }
    
    /**
     * Base class for all Essential JavaScript components
     */
    export abstract class Component<ElementType extends HTMLElement> extends Base<ElementType> {
        element: ElementType;
        private detectFunction;
        /**
         * Enable or disable persisting component's state between page reloads.
         * @default false
         */
        enablePersistence: boolean;
        /**
         * Enable or disable rendering component in right to left direction.
         * @default false
         */
        enableRtl: boolean;
        /**
         * Overrides the global culture and localization value for this component. Default global culture is 'en-US'.
         * @default undefined
         */
        locale: string;
        protected needsID: boolean;
        protected moduleLoader: ModuleLoader;
        protected localObserver: Observer;
        protected abstract render(): void;
        protected abstract preRender(): void;
        protected abstract getPersistData(): string;
        protected injectedModules: Function[];
        protected requiredModules(): ej.base.ModuleDeclaration[];
        /**
         * Destroys the sub modules while destroying the widget
         */
        protected destroy(): void;
        /**
         * Applies all the pending property changes and render the component again.
         */
        refresh(): void;
        /**
         * Appends the component within the given HTML element
         * @param {string | HTMLElement} selector - Target element where component needs to be appended.
         */
        appendTo(selector?: string | HTMLElement): void;
        /**
         * When invoked, applies the pending property changes immediately to the component.
         */
        dataBind(): void;
        /**
         * Attach one or more  event handler to the current component context.
         * It is used for internal handling event internally within the component only.
         * @param {BoundOptions[]| string} event - It is  optional type either to  Set the collection of event list or the eventName.
         * @param {Function} handler - optional parameter Specifies the handler to run when the event occurs
         * @param {Object} context - optional parameter Specifies the context to be bind in the handler.
         * @return {void}
         * @private
         */
        on(event: BoundOptions[] | string, handler?: Function, context?: Object): void;
        /**
         * To remove one or more event handler that has been attached with the on() method.
         * @param {BoundOptions[]| string} event - It is  optional type either to  Set the collection of event list or the eventName.
         * @param {Function} handler - optional parameter Specifies the function to run when the event occurs
         * @return {void}
         * @private
         */
        off(event: BoundOptions[] | string, handler?: Function): void;
        /**
         * To notify the handlers in the specified event.
         * @param {string} property - Specifies the event to be notify.
         * @param {Object} argument - Additional parameters to pass while calling the handler.
         * @return {void}
         * @private
         */
        notify(property: string, argument: Object): void;
        /**
         * Get injected modules
         * @private
         */
        getInjectedModules(): Function[];
        /**
         * Dynamically injects the required modules to the component.
         */
        static Inject(...moduleList: Function[]): void;
        /**
         * Initialize the constructor for component base
         */
        constructor(options?: Object, selector?: string | ElementType);
        private injectModules();
        private mergePersistData();
        private setPersistData();
        protected clearTemplate(templateName?: string[]): void;
        protected addOnPersist(options: string[]): string;
        protected getActualProperties<T>(obj: T): T;
        protected ignoreOnPersist(options: string[]): string;
        protected iterateJsonProperties(obj: {
            [key: string]: Object;
        }, ignoreList: string[]): Object;
    }
    
    /**
     * Function to create Html element.
     * @param tagName - Name of the tag, id and class names.
     * @param properties - Object to set properties in the element.
     * @param properties.id - To set the id to the created element.
     * @param properties.className - To add classes to the element.
     * @param properties.innerHTML - To set the innerHTML to element.
     * @param properties.styles - To set the some custom styles to element.
     * @param properties.attrs - To set the attributes to element.
     * @private
     */
    export function createElement(tagName: string, properties?: {
        id?: string;
        className?: string;
        innerHTML?: string;
        styles?: string;
        attrs?: {
            [key: string]: string;
        };
    }): HTMLElement;
    /**
     * The function used to add the classes to array of elements
     * @param  {Element[]|NodeList} elements - An array of elements that need to add a list of classes
     * @param  {string|string[]} classes - String or array of string that need to add an individual element as a class
     * @private
     */
    export function addClass(elements: Element[] | NodeList, classes: string | string[]): Element[] | NodeList;
    /**
     * The function used to add the classes to array of elements
     * @param  {Element[]|NodeList} elements - An array of elements that need to remove a list of classes
     * @param  {string|string[]} classes - String or array of string that need to add an individual element as a class
     * @private
     */
    export function removeClass(elements: Element[] | NodeList, classes: string | string[]): Element[] | NodeList;
    /**
     * The function used to check element is visible or not.
     * @param  {Element|Node} element - An element the need to check visibility
     * @private
     */
    export function isVisible(element: Element | Node): Boolean;
    /**
     * The function used to insert an array of elements into a first of the element.
     * @param  {Element[]|NodeList} fromElements - An array of elements that need to prepend.
     * @param  {Element} toElement - An element that is going to prepend.
     * @private
     */
    export function prepend(fromElements: Element[] | NodeList, toElement: Element): Element[] | NodeList;
    /**
     * The function used to insert an array of elements into last of the element.
     * @param  {Element[]|NodeList} fromElements - An array of elements that need to append.
     * @param  {Element} toElement - An element that is going to prepend.
     * @private
     */
    export function append(fromElements: Element[] | NodeList, toElement: Element): Element[] | NodeList;
    /**
     * The function used to remove the element from the
     * @param  {Element|Node|HTMLElement} element - An element that is going to detach from the Dom
     * @private
     */
    export function detach(element: Element | Node | HTMLElement): Element;
    /**
     * The function used to remove the element from Dom also clear the bounded events
     * @param  {Element|Node|HTMLElement} element - An element remove from the Dom
     * @private
     */
    export function remove(element: Element | Node | HTMLElement): void;
    /**
     * The function helps to set multiple attributes to an element
     * @param  {Element|Node} element - An element that need to set attributes.
     * @param  {{[key:string]:string}} attributes - JSON Object that is going to as attributes.
     * @private
     */
    export function attributes(element: Element | Node, attributes: {
        [key: string]: string;
    }): Element;
    /**
     * The function selects the element from giving context.
     * @param  {string} selector - Selector string need fetch element from the
     * @param  {Document|Element=document} context - It is an optional type, That specifies a Dom context.
     * @private
     */
    export function select(selector: string, context?: Document | Element): Element;
    /**
     * The function selects an array of element from the given context.
     * @param  {string} selector - Selector string need fetch element from the
     * @param  {Document|Element=document} context - It is an optional type, That specifies a Dom context.
     * @private
     */
    export function selectAll(selector: string, context?: Document | Element): HTMLElement[];
    /**
     * Returns single closest parent element based on class selector.
     * @param  {Element} element - An element that need to find the closest element.
     * @param  {string} selector - A classSelector of closest element.
     * @private
     */
    export function closest(element: Element | Node, selector: string): Element;
    /**
     * Returns all sibling elements of the given element.
     * @param  {Element|Node} element - An element that need to get siblings.
     * @private
     */
    export function siblings(element: Element | Node): Element[];
    /**
     * set the value if not exist. Otherwise set the existing value
     * @param  {HTMLElement} element - An element to which we need to set value.
     * @param  {string} property - Property need to get or set.
     * @param  {string} value - value need to set.
     * @private
     */
    export function getAttributeOrDefault(element: HTMLElement, property: string, value: string): string;
    /**
     * Set the style attributes to Html element.
     * @param {HTMLElement} element - Element which we want to set attributes
     * @param {any} attrs - Set the given attributes to element
     * @return {void}
     * @private
     */
    export function setStyleAttribute(element: HTMLElement, attrs: {
        [key: string]: Object;
    }): void;
    /**
     * Method for add and remove classes to a dom element.
     * @param {Element} element - Element for add and remove classes
     * @param {string[]} addClasses - List of classes need to be add to the element
     * @param {string[]} removeClasses - List of classes need to be remove from the element
     * @return {void}
     * @private
     */
    export function classList(element: Element, addClasses: string[], removeClasses: string[]): void;
    /**
     * Method to check whether the element matches the given selector.
     * @param {Element} element - Element to compare with the selector.
     * @param {string} selector - String selector which element will satisfy.
     * @return {void}
     * @private
     */
    export function matches(element: Element, selector: string): boolean;
    
    /**
     * Interface for a class Position
     */
    export interface PositionModel {
    
        /**
         * Specifies the left position of cursor in draggable.
         */
        left?: number;
    
        /**
         * Specifies the left position of cursor in draggable.
         */
        top?: number;
    
    }
    
    /**
     * Interface for a class Draggable
     */
    export interface DraggableModel {
    
        /**
         * Defines the distance between the cursor and the draggable element.
         */
        cursorAt?: PositionModel;
    
        /**
         * If `clone` set to true, drag operations are performed in duplicate element of the draggable element. 
         * @default true
         */
        clone?: boolean;
    
        /**
         * Defines the parent  element in which draggable element movement will be restricted.
         */
        dragArea?: HTMLElement | string;
    
        /**
         * Specifies the callback function for drag event.
         * @event
         */
        drag?: Function;
    
        /**
         * Specifies the callback function for dragStart event.
         * @event
         */
        dragStart?: Function;
    
        /**
         * Specifies the callback function for dragStop event.
         * @event
         */
        dragStop?: Function;
    
        /**
         * Defines the minimum distance draggable element to be moved to trigger the drag operation.
         * @default 1
         */
        distance?: number;
    
        /**
         * Defines the child element selector which will act as drag handle.
         */
        handle?: string;
    
        /**
         * Defines the child element selector which will prevent dragging of element.
         */
        abort?: string;
    
        /**
         * Defines the callback function for customizing the cloned  element.
         */
        helper?: Function;
    
        /**
         * Defines the scope value to group sets of draggable and droppable items. 
         * A draggable with the same scope value will be accepted by the droppable.
         */
        scope?: string;
    
        /**
         * Specifies the dragTarget by which the clone element is positioned if not given current context element will be considered.
         * @private
         */
        dragTarget?: string;
    
        /**
         * Defines the axis to limit the draggable element drag path.The possible axis path values are   
         * * `x` - Allows drag movement in horizontal direction only. 
         * * `y` - Allows drag movement in vertical direction only.
         */
        axis?: DragDirection;
    
        /**
         * Defines the function to change the position value.
         * @private
         */
        queryPositionInfo?: Function;
    
        /**
         * Defines whether the drag clone element will be split form the cursor pointer.
         * @private
         */
        enableTailMode?: boolean;
    
        /**
         * Defines whether to skip the previous drag movement comparison.
         * @private
         */
        skipDistanceCheck?: boolean;
    
        /**
         * @private
         */
        preventDefault?: boolean;
    
    }
    
    /**
     * Specifies the Direction in which drag movement happen.
     */
    export type DragDirection = 'x' | 'y';
    /**
     * Specifies the position coordinates
     */
    export class Position extends ej.base.ChildProperty<Position> {
        /**
         * Specifies the left position of cursor in draggable.
         */
        left: number;
        /**
         * Specifies the left position of cursor in draggable.
         */
        top: number;
    }
    /**
     * Coordinates for element position
     * @private
     */
    export interface Coordinates {
        /**
         * Defines the x Coordinate of page.
         */
        pageX: number;
        /**
         * Defines the y Coordinate of page.
         */
        pageY: number;
        /**
         * Defines the x Coordinate of client.
         */
        clientX: number;
        /**
         * Defines the y Coordinate of client.
         */
        clientY: number;
    }
    /**
     * Interface to specify the drag data in the droppable.
     */
    export interface DropInfo {
        /**
         * Specifies the current draggable element
         */
        draggable?: HTMLElement;
        /**
         * Specifies the current helper element.
         */
        helper?: HTMLElement;
        /**
         * Specifies the drag target element
         */
        draggedElement?: HTMLElement;
    }
    export interface DropObject {
        target: HTMLElement;
        instance: DropOption;
    }
    /**
     * Used to access values
     * @private
     */
    export interface DragPosition {
        left: string;
        top: string;
    }
    /**
     * Used for accessing the interface.
     * @private
     */
    export interface Instance extends HTMLElement {
        /**
         * Specifies current instance collection in element
         */
        ej2_instances: {
            [key: string]: Object;
        }[];
    }
    /**
     * Droppable function to be invoked from draggable
     * @private
     */
    export interface DropOption {
        /**
         * Used to triggers over function while draggable element is over the droppable element.
         */
        intOver: Function;
        /**
         * Used to triggers out function while draggable element is out of the droppable element.
         */
        intOut: Function;
        /**
         * Used to triggers  out function while draggable element is dropped on the droppable element.
         */
        intDrop: Function;
        /**
         * Specifies the information about the drag element.
         */
        dragData: DropInfo;
        /**
         * Specifies the status of the drag of drag stop calling.
         */
        dragStopCalled: boolean;
    }
    /**
     * Drag Event arguments
     */
    export interface DragEventArgs {
        /**
         * Specifies the actual event.
         */
        event?: MouseEvent & TouchEvent;
        /**
         * Specifies the current drag element.
         */
        element?: HTMLElement;
        /**
         * Specifies the current target element.
         */
        target?: HTMLElement;
    }
    /**
     * Draggable Module provides support to enable draggable functionality in Dom Elements.
     * ```html
     * <div id='drag'>Draggable</div>
     * <script>
     * var ele = document.getElementById('drag');
     * var drag:Draggable = new Draggable(ele,{
     *     clone:false,
     *     drag: function(e) {
     *      //drag handler code.
     *      },
     *     handle:'.class'
     * });
     * </script>
     * ```
     */
    export class Draggable extends Base<HTMLElement> implements ej.base.INotifyPropertyChanged {
        /**
         * Defines the distance between the cursor and the draggable element.
         */
        cursorAt: PositionModel;
        /**
         * If `clone` set to true, drag operations are performed in duplicate element of the draggable element.
         * @default true
         */
        clone: boolean;
        /**
         * Defines the parent  element in which draggable element movement will be restricted.
         */
        dragArea: HTMLElement | string;
        /**
         * Specifies the callback function for drag event.
         * @event
         */
        drag: Function;
        /**
         * Specifies the callback function for dragStart event.
         * @event
         */
        dragStart: Function;
        /**
         * Specifies the callback function for dragStop event.
         * @event
         */
        dragStop: Function;
        /**
         * Defines the minimum distance draggable element to be moved to trigger the drag operation.
         * @default 1
         */
        distance: number;
        /**
         * Defines the child element selector which will act as drag handle.
         */
        handle: string;
        /**
         * Defines the child element selector which will prevent dragging of element.
         */
        abort: string;
        /**
         * Defines the callback function for customizing the cloned  element.
         */
        helper: Function;
        /**
         * Defines the scope value to group sets of draggable and droppable items.
         * A draggable with the same scope value will be accepted by the droppable.
         */
        scope: string;
        /**
         * Specifies the dragTarget by which the clone element is positioned if not given current context element will be considered.
         * @private
         */
        dragTarget: string;
        /**
         * Defines the axis to limit the draggable element drag path.The possible axis path values are
         * * `x` - Allows drag movement in horizontal direction only.
         * * `y` - Allows drag movement in vertical direction only.
         */
        axis: DragDirection;
        /**
         * Defines the function to change the position value.
         * @private
         */
        queryPositionInfo: Function;
        /**
         * Defines whether the drag clone element will be split form the cursor pointer.
         * @private
         */
        enableTailMode: boolean;
        /**
         * Defines whether to skip the previous drag movement comparison.
         * @private
         */
        skipDistanceCheck: boolean;
        /**
         * @private
         */
        preventDefault: boolean;
        private target;
        private initialPosition;
        private relativeXPosition;
        private relativeYPosition;
        private margin;
        private offset;
        private position;
        private dragLimit;
        private borderWidth;
        private padding;
        private pageX;
        private diffX;
        private diffY;
        private pageY;
        private helperElement;
        private hoverObject;
        private parentClientRect;
        droppables: {
            [key: string]: DropInfo;
        };
        constructor(element: HTMLElement, options?: DraggableModel);
        protected bind(): void;
        private static getDefaultPosition();
        private toggleEvents(isUnWire?);
        private initialize(evt);
        private intDragStart(evt);
        private getProcessedPositionValue(value);
        private calculateParentPosition(ele);
        private intDrag(evt);
        private getDocumentWidthHeight(str);
        private intDragStop(evt);
        private intDestroy(evt);
        onPropertyChanged(newProp: DraggableModel, oldProp: DraggableModel): void;
        getModuleName(): string;
        private setDragArea();
        private getProperTargetElement(evt);
        private getMousePosition(evt);
        private getCoordinates(evt);
        private getHelperElement(evt);
        private setGlobalDroppables(reset, drag?, helper?);
        private checkTargetElement(evt);
        private getDropInstance(ele);
        destroy(): void;
    }
    
    /**
     * Interface for a class Droppable
     */
    export interface DroppableModel {
    
        /**
         * Defines the selector for draggable element to be accepted by the droppable.
         */
        accept?: string;
    
        /**
         * Defines the scope value to group sets of draggable and droppable items. 
         * A draggable with the same scope value will only be accepted by the droppable.
         */
        scope?: string;
    
        /**
         * Specifies the callback function, which will be triggered while drag element is dropped in droppable.
         * @event
         */
        drop?: (args: DropEventArgs) => void;
    
        /**
         * Specifies the callback function, which will be triggered while drag element is moved over droppable element.
         * @event
         */
        over?: Function;
    
        /**
         * Specifies the callback function, which will be triggered while drag element is moved out of droppable element.
         * @event
         */
        out?: Function;
    
    }
    
    /**
     * Droppable arguments in drop callback.
     * @private
     */
    export interface DropData {
        /**
         * Specifies that current element can be dropped.
         */
        canDrop: boolean;
        /**
         * Specifies target to drop.
         */
        target: HTMLElement;
    }
    export interface DropEvents extends MouseEvent, TouchEvent {
        dropTarget?: HTMLElement;
    }
    /**
     * Interface for drop event args
     */
    export interface DropEventArgs {
        /**
         * Specifies the original mouse or touch  event arguments.
         */
        event?: MouseEvent & TouchEvent;
        /**
         * Specifies the target element.
         */
        target?: HTMLElement;
        /**
         *  Specifies the dropped element.
         */
        droppedElement?: HTMLElement;
        /**
         * Specifies the dragData
         */
        dragData?: DropInfo;
    }
    /**
     * Droppable Module provides support to enable droppable functionality in Dom Elements.
     * ```html
     * <div id='drop'>Droppable</div>
     * <script>
     * let ele:HTMLElement = document.getElementById('drop');
     * var drag:Droppable = new Droppable(ele,{
     *     accept:'.drop',
     *     drop: function(e) {
     *      //drop handler code.
     *     }
     * });
     * </script>
     * ```
     */
    export class Droppable extends Base<HTMLElement> implements ej.base.INotifyPropertyChanged {
        /**
         * Defines the selector for draggable element to be accepted by the droppable.
         */
        accept: string;
        /**
         * Defines the scope value to group sets of draggable and droppable items.
         * A draggable with the same scope value will only be accepted by the droppable.
         */
        scope: string;
        /**
         * Specifies the callback function, which will be triggered while drag element is dropped in droppable.
         * @event
         */
        drop: (args: DropEventArgs) => void;
        /**
         * Specifies the callback function, which will be triggered while drag element is moved over droppable element.
         * @event
         */
        over: Function;
        /**
         * Specifies the callback function, which will be triggered while drag element is moved out of droppable element.
         * @event
         */
        out: Function;
        private mouseOver;
        dragData: {
            [key: string]: DropInfo;
        };
        constructor(element: HTMLElement, options?: DroppableModel);
        protected bind(): void;
        private wireEvents();
        onPropertyChanged(newProp: DroppableModel, oldProp: DroppableModel): void;
        getModuleName(): string;
        private dragStopCalled;
        intOver(event: MouseEvent & TouchEvent, element?: Element): void;
        intOut(event: MouseEvent & TouchEvent, element?: Element): void;
        private intDrop(evt, element?);
        private isDropArea(evt, helper, element?);
        destroy(): void;
    }
    
    /**
     * EventHandler class provides option to add, remove, clear and trigger events to a HTML DOM element
     * @private
     * ```html
     * <div id="Eventdiv">  </div>
     * <script>
     *   let node: HTMLElement = document.querySelector("#Eventdiv");
     *   EventHandler.addEventListener(node, "click", function(){
     *       // click handler function code
     *   });
     *   EventHandler.addEventListener(node, "onmouseover", function(){
     *       // mouseover handler function code
     *   });
     *   EventHandler.removeEventListener(node, "click", function(){
     *       // click handler function code
     *   });
     *   eventObj.clearEvents();
     * </script>
     * ```
     */
    export class EventHandler {
        private static addOrGetEventData(element);
        /**
         * Add an event to the specified DOM element.
         * @param {any} element - Target HTML DOM element
         * @param {string} eventName - A string that specifies the name of the event
         * @param {Function} listener - Specifies the function to run when the event occurs
         * @param {Object} bindTo - A object that binds 'this' variable in the event handler
         * @param {number} debounce - Specifies at what interval given event listener should be triggered.
         * @return {Function}
         */
        static add(element: Element | HTMLElement | Document, eventName: string, listener: Function, bindTo?: Object, intDebounce?: number): Function;
        /**
         * Remove an event listener that has been attached before.
         * @param {any} element - Specifies the target html element to remove the event
         * @param {string} eventName - A string that specifies the name of the event to remove
         * @param {Function} listener - Specifies the function to remove
         * @return {void}
         */
        static remove(element: Element | HTMLElement | Document, eventName: string, listener: Function): void;
        /**
         * Clear all the event listeners that has been previously attached to the element.
         * @param {any} element - Specifies the target html element to clear the events
         * @return {void}
         */
        static clearEvents(element: Element): void;
        /**
         * Trigger particular event of the element.
         * @param {any} element - Specifies the target html element to trigger the events
         * @param {string} eventName - Specifies the event to trigger for the specified element.
         * Can be a custom event, or any of the standard events.
         * @param {any} eventProp - Additional parameters to pass on to the event properties
         * @return {void}
         */
        static trigger(element: HTMLElement, eventName: string, eventProp?: Object): void;
    }
    /**
     * Common Event argument for all base Essential JavaScript 2 Events.
     * @private
     */
    export interface BaseEventArgs {
        /**
         * Specifies name of the event.
         */
        name?: string;
    }
    
    /**
     * Specifies the observer used for external change detection.
     */
    export let onIntlChange: Observer;
    /**
     * Specifies the default rtl status for EJ2 components.
     */
    export let rightToLeft: boolean;
    /**
     * Interface for dateFormatOptions
     * @private
     */
    export interface DateFormatOptions {
        /**
         * Specifies the skeleton for date formatting.
         */
        skeleton?: string;
        /**
         * Specifies the type of date formatting either date, dateTime or time.
         */
        type?: string;
        /**
         * Specifies custom date formatting to be used.
         */
        format?: string;
    }
    /**
     * Interface for numberFormatOptions
     * @private
     */
    export interface NumberFormatOptions {
        /**
         * Specifies minimum fraction digits in formatted value.
         */
        minimumFractionDigits?: number;
        /**
         * Specifies maximum fraction digits in formatted value.
         */
        maximumFractionDigits?: number;
        /**
         * Specifies minimum significant digits in formatted value.
         */
        minimumSignificantDigits?: number;
        /**
         * Specifies maximum significant digits in formatted value.
         */
        maximumSignificantDigits?: number;
        /**
         * Specifies whether to use grouping or not in formatted value,
         */
        useGrouping?: boolean;
        /**
         * Specifies the skeleton for perform formatting.
         */
        skeleton?: string;
        /**
         * Specifies the currency code to be used for formatting.
         */
        currency?: string;
        /**
         * Specifies minimum integer digits in formatted value.
         */
        minimumIntegerDigits?: number;
        /**
         * Specifies custom number format for formatting.
         */
        format?: string;
    }
    /**
     * Specifies the CLDR data loaded for internationalization functionalities.
     * @private
     */
    export let cldrData: Object;
    /**
     * Specifies the default culture value to be considered.
     * @private
     */
    export let defaultCulture: string;
    /**
     * Specifies default currency code to be considered
     * @private
     */
    export let defaultCurrencyCode: string;
    /**
     * Internationalization class provides support to parse and format the number and date object to the desired format.
     * ```typescript
     * // To set the culture globally
     * setCulture('en-GB');
     *
     * // To set currency code globally
     * setCurrencyCode('EUR');
     *
     * //Load cldr data
     * loadCldr(gregorainData);
     * loadCldr(timeZoneData);
     * loadCldr(numbersData);
     * loadCldr(numberSystemData);
     *
     * // To use formatter in component side
     * let Intl:Internationalization = new Internationalization();
     *
     * // Date formatting
     * let dateFormatter: Function = Intl.getDateFormat({skeleton:'long',type:'dateTime'});
     * dateFormatter(new Date('11/2/2016'));
     * dateFormatter(new Date('25/2/2030'));
     * Intl.formatDate(new Date(),{skeleton:'E'});
     *
     * //Number formatting
     * let numberFormatter: Function = Intl.getNumberFormat({skeleton:'C5'})
     * numberFormatter(24563334);
     * Intl.formatNumber(123123,{skeleton:'p2'});
     *
     * // Date parser
     * let dateParser: Function = Intl.getDateParser({skeleton:'short',type:'time'});
     * dateParser('10:30 PM');
     * Intl.parseDate('10',{skeleton:'H'});
     * ```
     */
    export class Internationalization {
        culture: string;
        constructor(cultureName?: string);
        /**
         * Returns the format function for given options.
         * @param {DateFormatOptions} options - Specifies the format options in which the format function will return.
         * @returns {Function}
         */
        getDateFormat(options?: DateFormatOptions): Function;
        /**
         * Returns the format function for given options.
         * @param {NumberFormatOptions} options - Specifies the format options in which the format function will return.
         * @returns {Function}
         */
        getNumberFormat(options?: ej.base.NumberFormatOptions): Function;
        /**
         * Returns the parser function for given options.
         * @param {DateFormatOptions} options - Specifies the format options in which the parser function will return.
         * @returns {Function}
         * @private
         */
        getDateParser(options?: DateFormatOptions): Function;
        /**
         * Returns the parser function for given options.
         * @param {NumberFormatOptions} options - Specifies the format options in which the parser function will return.
         * @returns {Function}
         */
        getNumberParser(options?: ej.base.NumberFormatOptions): Function;
        /**
         * Returns the formatted string based on format options.
         * @param {Number} value - Specifies the number to format.
         * @param {NumberFormatOptions} option - Specifies the format options in which the number will be formatted.
         * @returns {string}
         */
        formatNumber(value: Number, option?: ej.base.NumberFormatOptions): string;
        /**
         * Returns the formatted date string based on format options.
         * @param {Number} value - Specifies the number to format.
         * @param {DateFormatOptions} option - Specifies the format options in which the number will be formatted.
         * @returns {string}
         */
        formatDate(value: Date, option?: DateFormatOptions): string;
        /**
         * Returns the date object for given date string and options.
         * @param {string} value - Specifies the string to parse.
         * @param {DateFormatOptions} option - Specifies the parse options in which the date string will be parsed.
         * @returns {Date}
         */
        parseDate(value: string, option?: DateFormatOptions): Date;
        /**
         * Returns the number object from the given string value and options.
         * @param {string} value - Specifies the string to parse.
         * @param {NumberFormatOptions} option - Specifies the parse options in which the  string number  will be parsed.
         * @returns {number}
         */
        parseNumber(value: string, option?: ej.base.NumberFormatOptions): number;
        /**
         * Returns Native Date Time Pattern
         * @param {DateFormatOptions} option - Specifies the parse options for resultant date time pattern.
         * @param {boolean} isExcelFormat - Specifies format value to be converted to excel pattern.
         * @returns {string}
         * @private
         */
        getDatePattern(option: DateFormatOptions, isExcelFormat?: boolean): string;
        /**
         * Returns Native Number Pattern
         * @param {NumberFormatOptions} option - Specifies the parse options for resultant number pattern.
         * @returns {string}
         * @private
         */
        getNumberPattern(option: ej.base.NumberFormatOptions): string;
        private getCulture();
    }
    /**
     * Set the default culture to all EJ2 components
     * @param {string} cultureName - Specifies the culture name to be set as default culture.
     */
    export function setCulture(cultureName: string): void;
    /**
     * Set the default currency code to all EJ2 components
     * @param {string} currencyCode Specifies the culture name to be set as default culture.
     * @returns {void}
     */
    export function setCurrencyCode(currencyCode: string): void;
    /**
     * Load the CLDR data into context
     * @param {Object[]} obj Specifies the CLDR data's to be used for formatting and parser.
     * @returns {void}
     */
    export function loadCldr(...data: Object[]): void;
    /**
     * To enable or disable RTL functionality for all components globally.
     * @param {boolean} status - Optional argument Specifies the status value to enable or disable rtl option.
     * @returns {void}
     */
    export function enableRtl(status?: boolean): void;
    /**
     * To get the numeric CLDR object for given culture
     * @param {string} locale - Specifies the locale for which numericObject to be returned.
     * @ignore
     * @private
     */
    export function getNumericObject(locale: string, type?: string): Object;
    /**
     * To get the default date CLDR object.
     * @ignore
     * @private
     */
    export function getDefaultDateObject(): Object;
    
    /**
     * Interface for a class KeyboardEvents
     */
    export interface KeyboardEventsModel {
    
        /**
         * Specifies key combination and it respective action name.
         * @default null
         */
        keyConfigs?: { [key: string]: string };
    
        /**
         * Specifies on which event keyboardEvents class should listen for key press. For ex., `keyup`, `keydown` or `keypress`
         * @default keyup
         */
        eventName?: string;
    
        /**
         * Specifies the listener when keyboard actions is performed. 
         * @event
         */
        keyAction?: ej.base.EmitType<ej.base.KeyboardEventArgs>;
    
    }
    
    export interface KeyboardEventArgs extends KeyboardEvent {
        action: string;
    }
    /**
     * KeyboardEvents class enables you to bind key action desired key combinations for ex., Ctrl+A, Delete, Alt+Space etc.
     * ```html
     * <div id='testEle'>  </div>;
     * <script>
     *   let node: HTMLElement = document.querySelector('#testEle');
     *   let kbInstance = new KeyboardEvents({
     *       element: node,
     *       keyConfigs:{ selectAll : 'ctrl+a' },
     *       keyAction: function (e:KeyboardEvent, action:string) {
     *           // handler function code
     *       }
     *   });
     * </script>
     * ```
     */
    export class KeyboardEvents extends Base<HTMLElement> implements ej.base.INotifyPropertyChanged {
        /**
         * Specifies key combination and it respective action name.
         * @default null
         */
        keyConfigs: {
            [key: string]: string;
        };
        /**
         * Specifies on which event keyboardEvents class should listen for key press. For ex., `keyup`, `keydown` or `keypress`
         * @default keyup
         */
        eventName: string;
        /**
         * Specifies the listener when keyboard actions is performed.
         * @event
         */
        keyAction: ej.base.EmitType<ej.base.KeyboardEventArgs>;
        /**
         * Initializes the KeyboardEvents
         * @param {HTMLElement} element
         * @param {KeyboardEventsModel} options
         */
        constructor(element: HTMLElement, options?: KeyboardEventsModel);
        /**
         * Unwire bound events and destroy the instance.
         * @return {void}
         */
        destroy(): void;
        /**
         * Function can be used to specify certain action if a property is changed
         * @param newProp
         * @param oldProp
         * @returns {void}
         * @private
         */
        onPropertyChanged(newProp: KeyboardEventsModel, oldProp?: KeyboardEventsModel): void;
        protected bind(): void;
        /**
         * To get the module name, returns 'keyboard'.
         * @private
         */
        getModuleName(): string;
        /**
         * Wiring event handlers to events
         */
        private wireEvents();
        /**
         * Unwiring event handlers to events
         */
        private unwireEvents();
        /**
         * To handle a key press event returns null
         */
        private keyPressHandler;
        private static configCache;
        /**
         * To get the key configuration data
         * @param {string} config - configuration data
         * returns {KeyData}
         */
        private static getKeyConfigData(config);
        private static getKeyCode(keyVal);
    }
    
    /**
     * L10n modules provides localized text for different culture.
     * ```typescript
     *
     * //load global locale object common for all components.
     * L10n.load({
     *    'fr-BE': {
     *       'button': {
     *            'check': 'vérifié'
     *        }
     *    }
     * });
     * //set globale default locale culture.
     * setCulture('fr-BE');
     * let instance: ej.base.L10n = new L10n('button', {
     *    check: 'checked'
     * });
     * //Get locale text for current property.
     * instance.getConstant('check');
     * //Change locale culture in a component.
     * instance.setLocale('en-US');
     * ```
     */
    export class L10n {
        private static locale;
        private controlName;
        private localeStrings;
        private currentLocale;
        /**
         * Constructor
         */
        constructor(controlName: string, localeStrings: Object, locale?: string);
        /**
         * Sets the locale text
         * @param {string} locale
         * @returns {void}
         */
        setLocale(locale: string): void;
        /**
         * Sets the global locale for all components.
         * @param {Object} localeObject - specifies the localeObject to be set as global locale.
         */
        static load(localeObject: Object): void;
        /**
         * Returns current locale text for the property based on the culture name and control name.
         * @param {string} propertyName - specifies the property for which localize text to be returned.
         * @return string
         */
        getConstant(prop: string): string;
        /**
         * Returns the control constant object for current object and the locale specified.
         * @param {Object} curObject
         * @param {string} locale
         * @returns {Object}
         */
        private intGetControlConstant(curObject, locale);
    }
    
    export interface ModuleDeclaration {
        args: Object[];
        member: string;
        isProperty?: boolean;
    }
    export interface IParent {
        [key: string]: any;
    }
    export class ModuleLoader {
        private parent;
        private loadedModules;
        constructor(parent: IParent);
        /**
         * Inject required modules in component library
         * @return {void}
         * @param {ej.base.ModuleDeclaration[]} requiredModules - Array of modules to be required
         * @param {Function[]} moduleList - Array of modules to be injected from sample side
         */
        inject(requiredModules: ej.base.ModuleDeclaration[], moduleList: Function[]): void;
        /**
         * To remove the created object while destroying the control
         * @return {void}
         */
        clean(): void;
        /**
         * Removes all unused modules
         * @param {ej.base.ModuleDeclaration[]} moduleList
         * @returns {void}
         */
        private clearUnusedModule(moduleList);
        /**
         * To get the name of the member.
         * @param {string} name
         * @returns {string}
         */
        private getMemberName(name);
        /**
         * Returns boolean based on whether the module specified is loaded or not
         * @param {string} modName
         * @returns {boolean}
         */
        private isModuleLoaded(modName);
    }
    
    /**
     * Method used to create property. General syntax below.
     * @param  {T} defaultValue? - Specifies the default value of property.
     * ```
     * @Property('TypeScript')
     * propertyName: Type;
     * ```
     * @private
     */
    export function Property<T>(defaultValue?: T | Object): PropertyDecorator;
    /**
     * Method used to create complex property. General syntax below.
     * @param  {T} defaultValue - Specifies the default value of property.
     * @param  {Function} type - Specifies the class type of complex object.
     * ```
     * @Complex<Type>({},Type)
     * propertyName: Type;
     * ```
     * @private
     */
    export function Complex<T>(defaultValue: T, type: Function): PropertyDecorator;
    /**
     * Method used to create complex Factory property. General syntax below.
     * @param  {Function} defaultType - Specifies the default value of property.
     * @param  {Function} type - Specifies the class factory type of complex object.
     * ```
     * @ComplexFactory(defaultType, factoryFunction)
     * propertyName: Type1 | Type2;
     * ```
     * @private
     */
    export function ComplexFactory(type: Function): PropertyDecorator;
    /**
     * Method used to create complex array property. General syntax below.
     * @param  {T[]} defaultValue - Specifies the default value of property.
     * @param  {Function} type - Specifies the class type of complex object.
     * ```
     * @Collection([], Type);
     * propertyName: Type;
     * ```
     * @private
     */
    export function Collection<T>(defaultValue: T[], type: Function): PropertyDecorator;
    /**
     * Method used to create complex factory array property. General syntax below.
     * @param  {T[]} defaultType - Specifies the default type of property.
     * @param  {Function} type - Specifies the class type of complex object.
     * ```
     * @Collection([], Type);
     * propertyName: Type;
     * ```
     * @private
     */
    export function CollectionFactory(type: Function): PropertyDecorator;
    /**
     * Method used to create event property. General syntax below.
     * @param  {Function} defaultValue? - Specifies the default value of property.
     * @param  {boolean} isComplex? - Specifies the whether it is complex object.
     * ```
     * @Event(()=>{return true;})
     * ```
     * @private
     */
    export function Event(): PropertyDecorator;
    /**
     * NotifyPropertyChanges is triggers the call back when the property has been changed.
     *
     * ```
     *  @NotifyPropertyChanges
     * class DemoClass implements ej.base.INotifyPropertyChanged {
     *
     *     @Property()
     *     property1: string;
     *
     *     dataBind: () => void;
     *
     *     constructor() { }
     *
     *     onPropertyChanged(newProp: any, oldProp: any) {
     *         // Called when property changed
     *     }
     * }
     * ```
     * @private
     */
    export function NotifyPropertyChanges(classConstructor: Function): void;
    /**
     * Interface to notify the changed properties
     */
    export interface INotifyPropertyChanged {
        onPropertyChanged(newProperties: Object, oldProperties?: Object): void;
    }
    /**
     * Method used to create builder for the components
     * @param {any} component -specifies the target component for which builder to be created.
     * @private
     */
    export function CreateBuilder<T>(component: T): Object;
    
    /**
     * Observer is used to perform event handling based the object.
     * ```
     * //Creating observer instance.
     * let observer:Observer = Observer(this);
     * let handler: Function = (a:number, b: number): number => {return a + b; }
     * //add handler to event.
     * observe.on('eventname', handler);
     * //remove handler from event.
     * observe.off('eventname', handler);
     * //notify the handlers in event.
     * observe.notify('eventname');
     * ```
     *
     */
    export interface BoundOptions {
        handler?: Function;
        context?: Object;
        event?: string;
    }
    export class Observer {
        private context;
        private boundedEvents;
        constructor(context?: Object);
        /**
         * To attach handler for given property in current context.
         * @param {string} property - specifies the name of the event.
         * @param {Function} handler - Specifies the handler function to be called while event notified.
         * @param {Object} context - Specifies the context binded to the handler.
         * @return {void}
         */
        on(property: string, handler: Function, context?: Object): void;
        /**
         * To remove handlers from a event attached using on() function.
         * @param {string} eventName - specifies the name of the event.
         * @param {Function} handler - Optional argument specifies the handler function to be called while event notified.
         * @return {void}
         */
        off(property: string, handler?: Function): void;
        /**
         * To notify the handlers in the specified event.
         * @param {string} property - Specifies the event to be notify.
         *  @param {Object} args - Additional parameters to pass while calling the handler.
         * @return {void}
         */
        notify(property: string, argument?: Object): void;
        /**
         * To destroy handlers in the event
         */
        destroy(): void;
        /**
         * Returns if the property exists.
         */
        private notExist(prop);
        /**
         * Returns if the handler is present.
         */
        private isHandlerPresent(boundedEvents, handler);
    }
    
    /**
     * This has the basic properties required for SvgRenderer and CanvasRenderer
     * @private
     */
    export interface BaseAttibutes {
        /**
         * Specifies the ID of an element 
         */
        id?: string;
        /**
         * Specifies the fill color value 
         */
        fill?: string;
        /**
         * Specifies the border color value
         */
        stroke?: string;
        /**
         * Specifies the width of the border
         */
        'stroke-width'?: number;
        /**
         * Specifies the opacity value of an element
         */
        opacity?: number;
        /**
         * Height of the element
         */
        height?: number;
        /**
         * Width of the element
         */
        width?: number;
        /**
         * X value of the element 
         */
        x?: number;
        /**
         * Y value of the element 
         */
        y?: number;
        /**
         * Specifies the dash array value of an element
         */
        'stroke-dasharray'?: string;
        /**
         * Property to specify CSS styles for the elements
         */
        style?: string;
        /**
         * Color of the element 
         */
        color?: string;
        /**
         * Specifies the name of the class
         */
        className?: string;
        /**
         * Specifies the transformation value 
         */
        transform?: string;
        /**
         * Specifies the fill opacity of a shape/element
         */
        'fill-opacity'?: number;
        /**
         * Type of pointer for an element 
         */
        pointer?: string;
        /**
         * Specifies the plot value 
         */
        plot?: string;
        /**
         * Visibility of an element 
         */
        visibility?: string;
        /**
         * Specifies the clip path of an element
         */
        'clip-path'?: string;
    }
    /**
     * This has the properties for a SVG element
     * @private
     */
    export interface SVGAttributes extends BaseAttibutes {
        /**
         * View box property of an element
         */
        viewBox?: string;
        /**
         * Specifies the xmlns link property of a SVG element 
         */
        xmlns?: string;
    }
    /**
     * Properties required to render a circle
     * @private
     */
    export interface CircleAttributes extends BaseAttibutes {
        /**
         * Center x value of a circle
         */
        cx?: number;
        /**
         * Center y value of a circle
         */
        cy?: number;
        /**
         * Radius value of a circle
         */
        r?: number;
    }
    /**
     * Properties required to render a line
     * @private
     */
    export interface LineAttributes extends BaseAttibutes {
        /**
         * Specifies the value of x1
         */
        x1?: number;
        /**
         * Specifies the value of x2
         */
        x2?: number;
        /**
         * Specifies the value of y1
         */
        y1?: number;
        /**
         * Specifies the value of y2
         */
        y2?: number;
    }
    /**
     * Properties required to render a rectangle
     * @private
     */
    export interface RectAttributes extends BaseAttibutes {
        /**
         * Corner radius value of a rectangle
         */
        rx?: number;
    }
    /**
     * Properties required to render path
     * @private
     */
    export interface PathAttributes extends BaseAttibutes {
        /**
         * Specifies the d value of a path
         */
        d?: string;
        /**
         * Inner radius value of a path
         */
        innerR?: number;
        /**
         * Value of cx in path
         */
        cx?: number;
        /**
         * Value of cy in path
         */
        cy?: number;
        /**
         * Radius value of a path
         */
        r?: number;
        /**
         * Specifies the start value
         */
        start?: number;
        /**
         * Specifies the end value
         */
        end?: number;
        /**
         * Specifies the radius value
         */
        radius?: number;
        /**
         * Specifies the direction of path
         */
        counterClockWise?: boolean;
    }
    /**
     * Properties required to render a polyline
     * @private
     */
    export interface PolylineAttributes extends BaseAttibutes {
        /**
         * Points required to draw a polyline
         */
        points?: string;
    }
    /**
     * Properties required to render ellipse
     * @private
     */
    export interface EllipseAttributes extends CircleAttributes {
        /**
         * Specifies the rx value
         */
        rx?: number;
        /**
         * Specifies the ry value
         */
        ry?: number;
    }
    /**
     * Properties required to render a pattern
     * @private
     */
    export interface PatternAttributes extends BaseAttibutes {
        /**
         * Units to render a pattern
         */
        patternUnits?: string;
    }
    /**
     * Properties required to render an image
     * @private
     */
    export interface ImageAttributes extends BaseAttibutes {
        /**
         * Specifies the link to render it as image
         */
        href?: string;
        /**
         * Ratio value to render an image
         */
        preserveAspectRatio?: string;
    }
    /**
     * Properties required to render text
     * @private
     */
    export interface TextAttributes extends BaseAttibutes {
        /**
         * Size of the text
         */
        'font-size'?: string;
        /**
         * Font family of the text
         */
        'font-family'?: string;
        /**
         * Font style of the text
         */
        'font-style'?: string;
        /**
         * Weight of the text
         */
        'font-weight'?: string;
        /**
         * Specifies the text anchor value
         */
        'text-anchor'?: string;
        /**
         * Specifies the baseline value
         */
        'baseline'?: string;
        /**
         * Angle of rotation
         */
        'labelRotation'?: number;
    }
    /**
     * Properties required to render radial gradient
     * @private
     */
    export interface RadialGradient {
        /**
         * Specifies the id of the radial gradient
         */
        id?: string;
        /**
         * Specifies the cx value
         */
        cx?: string;
        /**
         * Specifies the cy value
         */
        cy?: string;
        /**
         * Specifies the radius value
         */
        r?: string;
        /**
         * Specifies the fx value
         */
        fx?: string;
        /**
         * Specifies the fy value
         */
        fy?: string;
    }
    /**
     * Properties required to render linear gradient
     * @private
     */
    export interface LinearGradient {
        /**
         * Id of the linear gradient
         */
        id?: string;
        /**
         * Specifies the x1 value
         */
        x1?: string;
        /**
         * Specifies the x2 value
         */
        x2?: string;
        /**
         * Specifies the y1 value
         */
        y1?: string;
        /**
         * Specifies the y2 value
         */
        y2?: string;
    }
    /**
     * Properties required to render a circle
     */
    export interface SVGCanvasAttributes {
        /**
         * To specify a new property
         */
        [key: string]: string;
    }
    /**
     * Properties required to render a gradient
     * @private
     */
    export interface GradientColor {
        /**
         * Specifies the color value of the gradient
         */
        color?: string;
        /**
         * Specifies the colorstop value of the gradient
         */
        colorStop?: string;
    }
    
    export class SvgRenderer {
        private svgLink;
        private svgObj;
        private rootId;
        /**
         * Specifies the height of the canvas element.
         * @default null
         */
        height: number;
        /**
         * Specifies the width of the canvas element.
         * @default null
         */
        width: number;
        constructor(rootID: string);
        private getOptionValue<T>(options, key);
        /**
         * To create a Html5 SVG element
         * @param {SVGAttributes} options - Options to create SVG
         * @return {Element}
         */
        createSvg(options: SVGAttributes): Element;
        private setSVGSize(width, height);
        /**
         * To draw a path
         * @param {PathAttributes} options - Options to draw a path in SVG
         * @return {Element}
         */
        drawPath(options: PathAttributes): Element;
        /**
         * To draw a line
         * @param {LineAttributes} options - Options to draw a line in SVG
         * @return {Element}
         */
        drawLine(options: LineAttributes): Element;
        /**
         * To draw a rectangle
         * @param {BaseAttibutes} options - Required options to draw a rectangle in SVG
         * @return {Element}
         */
        drawRectangle(options: RectAttributes): Element;
        /**
         * To draw a circle
         * @param {CircleAttributes} options - Required options to draw a circle in SVG
         * @return {Element}
         */
        drawCircle(options: CircleAttributes): Element;
        /**
         * To draw a polyline
         * @param {PolylineAttributes} options - Options required to draw a polyline
         * @return {Element}
         */
        drawPolyline(options: PolylineAttributes): Element;
        /**
         * To draw an ellipse
         * @param {EllipseAttributes} options - Options required to draw an ellipse
         * @return {Element}
         */
        drawEllipse(options: EllipseAttributes): Element;
        /**
         * To draw a polygon
         * @param {PolylineAttributes} options - Options needed to draw a polygon in SVG
         * @return {Element}
         */
        drawPolygon(options: PolylineAttributes): Element;
        /**
         * To draw an image
         * @param {ImageAttributes} options - Required options to draw an image in SVG
         * @return {Element}
         */
        drawImage(options: ImageAttributes): Element;
        /**
         * To draw a text
         * @param {TextAttributes} options - Options needed to draw a text in SVG
         * @return {Element}
         */
        createText(options: TextAttributes, label: string): Element;
        /**
         * To create a tSpan
         * @param {TextAttributes} options - Options to create tSpan
         * @param {string} label - The text content which is to be rendered in the tSpan
         * @return {Element}
         */
        createTSpan(options: TextAttributes, label: string): Element;
        /**
         * To create a title
         * @param {string} text - The text content which is to be rendered in the title
         * @return {Element}
         */
        createTitle(text: string): Element;
        /**
         * To create defs element in SVG
         * @return {Element}
         */
        createDefs(): Element;
        /**
         * To create clip path in SVG
         * @param {BaseAttibutes} options - Options needed to create clip path
         * @return {Element}
         */
        createClipPath(options: BaseAttibutes): Element;
        /**
         * To create foreign object in SVG
         * @param {BaseAttibutes} options - Options needed to create foreign object
         * @return {Element}
         */
        createForeignObject(options: BaseAttibutes): Element;
        /**
         * To create group element in SVG
         * @param {BaseAttibutes} options - Options needed to create group
         * @return {Element}
         */
        createGroup(options: BaseAttibutes): Element;
        /**
         * To create pattern in SVG
         * @param {PatternAttributes} options - Required options to create pattern in SVG
         * @param {string} type - Specifies the name of the pattern
         * @return {Element}
         */
        createPattern(options: PatternAttributes, element: string): Element;
        /**
         * To create radial gradient in SVG
         * @param {string[]} colors - Specifies the colors required to create radial gradient
         * @param {string[]} colorStop - Specifies the colorstop required to create radial gradient
         * @param {string} name - Specifies the name of the gradient
         * @param {RadialGradient} options - value for radial gradient
         * @return {string}
         */
        createRadialGradient(colors: GradientColor[], name: string, options: RadialGradient): string;
        /**
         * To create linear gradient in SVG
         * @param {string[]} colors - Array of string specifies the values for color
         * @param {string[]} colors - Array of string specifies the values for colorStop
         * @param {string} name - Specifies the name of the gradient
         * @param {LinearGradient} options - Specifies the options for gradient
         * @return {string}
         */
        createLinearGradient(colors: GradientColor[], name: string, options: LinearGradient): string;
        /**
         * To render the gradient element in SVG
         * @param {string} gradientType - Specifies the type of the gradient
         * @param {RadialGradient | LinearGradient} options - Options required to render a gradient
         * @param {string[]} colors - Array of string specifies the values for color
         * @param {string[]} colorStop - Array of string specifies the values for colorStop
         * @return {Element}
         */
        drawGradient(gradientType: string, options: RadialGradient | LinearGradient, colors: GradientColor[]): Element;
        /**
         * To render a clip path
         * @param {BaseAttibutes} options - Options required to render a clip path
         * @return {Element}
         */
        drawClipPath(options: BaseAttibutes): Element;
        /**
         * To create circular clip path in SVG
         * @param {CircleAttributes} options - Options required to create circular clip path
         * @return {Element}
         */
        drawCircularClipPath(options: CircleAttributes): Element;
        /**
         * To set the attributes to the element
         * @param {SVGCanvasAttributes} options - Attributes to set for the element
         * @param {Element} element - The element to which the attributes need to be set
         * @return {Element}
         */
        setElementAttributes(options: SVGCanvasAttributes, element: Element): Element;
    }
    
    /**
     * Interface for Template Engine.
     */
    export interface ITemplateEngine {
        compile: (templateString: string, helper?: Object) => (data: Object | JSON) => string;
    }
    /**
     * Compile the template string into template function.
     * @param  {string} templateString - The template string which is going to convert.
     * @param  {Object} helper? - Helper functions as an object.
     * @private
     */
    export function compile(templateString: string, helper?: Object): (data: Object | JSON, component?: any, propName?: any) => HTMLCollection;
    /**
     * Set your custom template engine for template rendering.
     * @param  {ITemplateEngine} classObj - Class object for custom template.
     * @private
     */
    export function setTemplateEngine(classObj: ITemplateEngine): void;
    /**
     * Get current template engine for template rendering.
     * @param  {ITemplateEngine} classObj - Class object for custom template.
     * @private
     */
    export function getTemplateEngine(): (template: string, helper?: Object) => (data: Object | JSON) => string;
    
    /**
     * The function to set regular expression for template expression string.
     * @param  {RegExp} value - Value expression.
     * @private
     */
    export function expression(value?: RegExp): RegExp;
    /**
     * Compile the template string into template function.
     * @param  {string} template - The template string which is going to convert.
     * @param  {Object} helper? - Helper functions as an object.
     * @private
     */
    export function compile(template: string, helper?: Object): () => string;
    
    /**
     * Interface for a class SwipeSettings
     * @private
     */
    export interface SwipeSettingsModel {
    
        /**
         * Property specifies minimum distance of swipe moved.
         */
        swipeThresholdDistance?: number;
    
    }
    
    /**
     * Interface for a class Touch
     */
    export interface TouchModel {
    
        /**
         * Specifies the callback function for tap event.
         * @event
         */
        tap?: ej.base.EmitType<TapEventArgs>;
    
        /**
         * Specifies the callback function for tapHold event.
         * @event
         */
        tapHold?: ej.base.EmitType<TapEventArgs>;
    
        /**
         * Specifies the callback function for swipe event.
         * @event
         */
        swipe?: ej.base.EmitType<SwipeEventArgs>;
    
        /**
         * Specifies the callback function for scroll event.
         * @event
         */
        scroll?: ej.base.EmitType<ScrollEventArgs>;
    
        /**
         * Specifies the time delay for tap.
         * @default 350
         */
        tapThreshold?: number;
    
        /**
         * Specifies the time delay for tap hold.
         * @default 750
         */
        tapHoldThreshold?: number;
    
        /**
         * Customize the swipe event configuration.
         * @default { swipeThresholdDistance: 50 }
         */
        swipeSettings?: SwipeSettingsModel;
    
    }
    
    /**
     * SwipeSettings is a framework module that provides support to handle swipe event like swipe up, swipe right, etc..,
     * @private
     */
    export class SwipeSettings extends ej.base.ChildProperty<SwipeSettings> {
        /**
         * Property specifies minimum distance of swipe moved.
         */
        swipeThresholdDistance: number;
    }
    /**
     * Touch class provides support to handle the touch event like tap, double tap, tap hold, etc..,
     * ```typescript
     *    let node: HTMLElement;
     * let touchObj: Touch = new Touch({
     *    element: node,
     *    tap: function (e) {
     *        // tap handler function code
     *    }
     *    tapHold: function (e) {
     *        // tap hold handler function code
     *    }
     *    scroll: function (e) {
     *        // scroll handler function code
     *    }
     *    swipe: function (e) {
     *        // swipe handler function code
     *    }
     * });
     * ```
     */
    export class Touch extends Base<HTMLElement> implements ej.base.INotifyPropertyChanged {
        private isTouchMoved;
        private startPoint;
        private movedPoint;
        private endPoint;
        private startEventData;
        private lastTapTime;
        private lastMovedPoint;
        private scrollDirection;
        private hScrollLocked;
        private vScrollLocked;
        private defaultArgs;
        private distanceX;
        private distanceY;
        private movedDirection;
        private tStampStart;
        private touchAction;
        private timeOutTap;
        private modeClear;
        private timeOutTapHold;
        /**
         * Specifies the callback function for tap event.
         * @event
         */
        tap: ej.base.EmitType<TapEventArgs>;
        /**
         * Specifies the callback function for tapHold event.
         * @event
         */
        tapHold: ej.base.EmitType<TapEventArgs>;
        /**
         * Specifies the callback function for swipe event.
         * @event
         */
        swipe: ej.base.EmitType<SwipeEventArgs>;
        /**
         * Specifies the callback function for scroll event.
         * @event
         */
        scroll: ej.base.EmitType<ScrollEventArgs>;
        /**
         * Specifies the time delay for tap.
         * @default 350
         */
        tapThreshold: number;
        /**
         * Specifies the time delay for tap hold.
         * @default 750
         */
        tapHoldThreshold: number;
        /**
         * Customize the swipe event configuration.
         * @default { swipeThresholdDistance: 50 }
         */
        swipeSettings: SwipeSettingsModel;
        private tapCount;
        constructor(element: HTMLElement, options?: TouchModel);
        /**
         * @private
         * @param newProp
         * @param oldProp
         */
        onPropertyChanged(newProp: TouchModel, oldProp: TouchModel): void;
        protected bind(): void;
        /**
         * To destroy the touch instance.
         * @return {void}
         */
        destroy(): void;
        private wireEvents();
        private unwireEvents();
        /**
         * Returns module name as touch
         * @returns {string}
         * @private
         */
        getModuleName(): string;
        /**
         * Returns if the HTML element is Scrollable.
         * @param {HTMLElement} element - HTML Element to check if Scrollable.
         * @returns {boolean}
         */
        private isScrollable(element);
        private startEvent;
        private moveEvent;
        private tapHoldEvent(evt);
        private endEvent;
        private calcPoints(evt);
        private calcScrollPoints(evt);
        private getVelocity(pnt);
        private checkSwipe(ele, flag);
    }
    /**
     * The argument type of `Tap` Event
     * @private
     */
    export interface TapEventArgs extends ej.base.BaseEventArgs {
        /**
         * Original native event Object.
         */
        originalEvent: TouchEventArgs | MouseEventArgs;
        /**
         * Tap Count.
         */
        tapCount?: number;
    }
    /**
     * The argument type of `Scroll` Event
     * @private
     */
    export interface ScrollEventArgs extends ej.base.BaseEventArgs {
        /**
         * Event argument for start event.
         */
        startEvents: TouchEventArgs | MouseEventArgs;
        /**
         * Original native event object for scroll.
         */
        originalEvent: TouchEventArgs | MouseEventArgs;
        /**
         * X position when scroll started.
         */
        startX: number;
        /**
         * Y position when scroll started.
         */
        startY: number;
        /**
         * The direction scroll.
         */
        scrollDirection: string;
        /**
         * The total traveled distance from X position
         */
        distanceX: number;
        /**
         * The total traveled distance from Y position
         */
        distanceY: number;
        /**
         * The velocity of scroll.
         */
        velocity: number;
    }
    /**
     * The argument type of `Swipe` Event
     * @private
     */
    export interface SwipeEventArgs extends ej.base.BaseEventArgs {
        /**
         * Event argument for start event.
         */
        startEvents: TouchEventArgs | MouseEventArgs;
        /**
         * Original native event object  for swipe.
         */
        originalEvent: TouchEventArgs | MouseEventArgs;
        /**
         * X position when swipe started.
         */
        startX: number;
        /**
         * Y position when swipe started.
         */
        startY: number;
        /**
         * The direction swipe.
         */
        swipeDirection: string;
        /**
         * The total traveled distance from X position
         */
        distanceX: number;
        /**
         * The total traveled distance from Y position
         */
        distanceY: number;
        /**
         * The velocity of swipe.
         */
        velocity: number;
    }
    export interface TouchEventArgs extends MouseEvent {
        /**
         * A TouchList with touched points.
         */
        changedTouches: MouseEventArgs[] | TouchEventArgs[];
        /**
         * Cancel the default action.
         */
        preventDefault(): void;
        /**
         * The horizontal coordinate point of client area.
         */
        clientX: number;
        /**
         * The vertical coordinate point of client area.
         */
        clientY: number;
    }
    export interface MouseEventArgs extends MouseEvent {
        /**
         * A TouchList with touched points.
         */
        changedTouches: MouseEventArgs[] | TouchEventArgs[];
        /**
         * Cancel the default action.
         */
        preventDefault(): void;
        /**
         * The horizontal coordinate point of client area.
         */
        clientX: number;
        /**
         * The vertical coordinate point of client area.
         */
        clientY: number;
    }
    
    /**
     * Common utility methods
     */
    export interface IKeyValue extends CSSStyleDeclaration {
        [key: string]: any;
    }
    /**
     * Create Instance from constructor function with desired parameters.
     * @param {Function} classFunction - Class function to which need to create instance
     * @param {any[]} params - Parameters need to passed while creating instance
     * @return {any}
     * @private
     */
    export function createInstance(classFunction: Function, params: any[]): any;
    /**
     * To run a callback function immediately after the browser has completed other operations.
     * @param {Function} handler - callback function to be triggered.
     * @return {Function}
     * @private
     */
    export function setImmediate(handler: Function): Function;
    /**
     * To get nameSpace value from the desired object.
     * @param {string} nameSpace - String value to the get the inner object
     * @param {any} obj - Object to get the inner object value.
     * @return {any}
     * @private
     */
    export function getValue(nameSpace: string, obj: any): any;
    /**
     * To set value for the nameSpace in desired object.
     * @param {string} nameSpace - String value to the get the inner object
     * @param {any} value - Value that you need to set.
     * @param {any} obj - Object to get the inner object value.
     * @return {void}
     * @private
     */
    export function setValue(nameSpace: string, value: any, obj: any): any;
    /**
     * Delete an item from Object
     * @param {any} obj - Object in which we need to delete an item.
     * @param {string} params - String value to the get the inner object
     * @return {void}
     * @private
     */
    export function deleteObject(obj: any, key: string): void;
    /**
     * Check weather the given argument is only object.
     * @param {any} obj - Object which is need to check.
     * @return {boolean}
     * @private
     */
    export function isObject(obj: any): boolean;
    /**
     * To get enum value by giving the string.
     * @param {any} enumObject - Enum object.
     * @param {string} enumValue - Enum value to be searched
     * @return {any}
     * @private
     */
    export function getEnumValue(enumObject: any, enumValue: string | number): any;
    /**
     * Merge the source object into destination object.
     * @param {any} source - source object which is going to merge with destination object
     * @param {any} destination - object need to be merged
     * @return {void}
     * @private
     */
    export function merge(source: Object, destination: Object): void;
    /**
     * Extend the two object with newer one.
     * @param {any} copied - Resultant object after merged
     * @param {Object} first - First object need to merge
     * @param {Object} second - Second object need to merge
     * @return {Object}
     * @private
     */
    export function extend(copied: Object, first: Object, second?: Object, deep?: boolean): Object;
    /**
     * To check whether the object is null or undefined.
     * @param {Object} value - To check the object is null or undefined
     * @return {boolean}
     * @private
     */
    export function isNullOrUndefined(value: Object): boolean;
    /**
     * To check whether the object is undefined.
     * @param {Object} value - To check the object is undefined
     * @return {boolean}
     * @private
     */
    export function isUndefined(value: Object): boolean;
    /**
     * To return the generated unique name
     * @param {string} definedName - To concatenate the unique id to provided name
     * @return {string}
     * @private
     */
    export function getUniqueID(definedName?: string): string;
    /**
     * It limits the rate at which a function can fire. The function will fire only once every provided second instead of as quickly.
     * @param {Function} eventFunction - Specifies the function to run when the event occurs
     * @param {number} delay - A number that specifies the milliseconds for function delay call option
     * @return {Function}
     * @private
     */
    export function debounce(eventFunction: Function, delay: number): Function;
    /**
     * To convert the object to string for query url
     * @param  {Object} data
     * @returns string
     * @private
     */
    export function queryParams(data: any): string;
    /**
     * To check whether the given array contains object.
     * @param {T[]} value- Specifies the T type array to be checked.
     * @private
     */
    export function isObjectArray<T>(value: T[]): boolean;
    /**
     * To check whether the  child element is descendant to parent element or parent and child are same element.
     * @param{Element} - Specifies the child element to compare with parent.
     * @param{Element} - Specifies the parent element.
     * @return boolean
     * @private
     */
    export function compareElementParent(child: Element, parent: Element): boolean;
    /**
     * To throw custom error message.
     * @param{string} - Specifies the error message to be thrown.
     * @private
     */
    export function throwError(message: string): void;
    /**
     * This function is used to print given element
     * @param{Element} element - Specifies the print content element.
     * @param{Window} printWindow - Specifies the print window.
     * @private
     */
    export function print(element: Element, printWindow?: Window): Window;
    /**
     * Function to normalize the units applied to the element.
     * @param  {number|string} value
     * @return {string} result
     * @private
     */
    export function formatUnit(value: number | string): string;
    /**
     * Function to fetch the Instances of a HTML element for the given component.
     * @param {string | HTMLElement} element
     * @param {any} component
     * @return {Object} inst
     * @private
     */
    export function getInstance(element: string | HTMLElement, component: any): Object;
    /**
     * Function to add instances for the given element.
     * @param {string | HTMLElement} element
     * @param {Object} instance
     * @return {void}
     * @private
     */
    export function addInstance(element: string | HTMLElement, instance: Object): void;
    
    export const basicPatterns: string[];
    /**
     * Interface for Date Format Options Module.
     * @private
     */
    export interface FormatOptions {
        month?: Object;
        weekday?: Object;
        pattern?: string;
        designator?: Object;
        timeZone?: any;
        era?: Object;
        hour12?: boolean;
        numMapper?: NumberMapper;
        dateSeperator?: string;
    }
    export const datePartMatcher: {
        [key: string]: Object;
    };
    /**
     * Date Format is a framework provides support for date formatting.
     * @private
     */
    export class DateFormat {
        /**
         * Returns the formatter function for given skeleton.
         * @param {string} -  Specifies the culture name to be which formatting.
         * @param {DateFormatOptions} - Specific the format in which date  will format.
         * @param {cldr} - Specifies the global cldr data collection.
         * @return Function.
         */
        static dateFormat(culture: string, option: DateFormatOptions, cldr: Object): Function;
        /**
         * Returns formatted date string based on options passed.
         * @param {Date} value
         * @param {FormatOptions} options
         */
        private static intDateFormatter(value, options);
        /**
         * Returns two digit numbers for given value and length
         */
        private static checkTwodigitNumber(val, len);
        /**
         * Returns the value of the Time Zone.
         * @param {number} tVal
         * @param {string} pattern
         * @private
         */
        static getTimeZoneValue(tVal: number, pattern: string): string;
    }
    
    /**
     * Date Parser.
     * @private
     */
    export class DateParser {
        /**
         * Returns the parser function for given skeleton.
         * @param {string} -  Specifies the culture name to be which formatting.
         * @param {DateFormatOptions} - Specific the format in which string date  will be parsed.
         * @param {cldr} - Specifies the global cldr data collection.
         *  @return Function.
         */
        static dateParser(culture: string, option: DateFormatOptions, cldr: Object): Function;
        /**
         * Returns date object for provided date options
         * @param {DateParts} options
         * @param {Date} value
         * @returns {Date}
         */
        private static getDateObject(options, value?);
        /**
         * Returns date parsing options for provided value along with parse and numeric options
         * @param {string} value
         * @param {ParseOptions} parseOptions
         * @param {NumericOptions} num
         * @returns {DateParts}
         */
        private static internalDateParse(value, parseOptions, num);
        /**
         * Returns parsed number for provided Numeric string and Numeric Options
         * @param {string} value
         * @param {NumericOptions} option
         * @returns {number}
         */
        private static internalNumberParser(value, option);
        /**
         * Returns parsed time zone RegExp for provided hour format and time zone
         * @param {string} hourFormat
         * @param {base.TimeZoneOptions} tZone
         * @param {string} nRegex
         * @returns {string}
         */
        private static parseTimeZoneRegx(hourFormat, tZone, nRegex);
        /**
         * Returns zone based value.
         * @param {boolean} flag
         * @param {string} val1
         * @param {string} val2
         * @param {NumericOptions} num
         * @returns {number}
         */
        private static getZoneValue(flag, val1, val2, num);
    }
    
    /**
     * Date base common constants and function for date parser and formatter.
     */
    export namespace IntlBase {
        const negativeDataRegex: RegExp;
        const customRegex: RegExp;
        const latnParseRegex: RegExp;
        const defaultCurrency: string;
        interface NumericSkeleton {
            type?: string;
            isAccount?: boolean;
            fractionDigits?: number;
        }
        interface GenericFormatOptions {
            nData?: NegativeData;
            pData?: NegativeData;
            zeroData?: NegativeData;
        }
        interface GroupSize {
            primary?: number;
            secondary?: number;
        }
        interface NegativeData extends FormatParts {
            nlead?: string;
            nend?: string;
            groupPattern?: string;
            minimumFraction?: number;
            maximumFraction?: number;
        }
        const formatRegex: RegExp;
        const dateParseRegex: RegExp;
        const basicPatterns: string[];
        interface Dependables {
            parserObject?: Object;
            dateObject?: Object;
            numericObject?: Object;
        }
        interface TimeZoneOptions {
            hourFormat?: string;
            gmtFormat?: string;
            gmtZeroFormat?: string;
        }
        const defaultObject: Object;
        const monthIndex: Object;
        /**
         *
         */
        const month: string;
        const days: string;
        /**
         * Default numerber Object
         */
        const patternMatcher: {
            [key: string]: Object;
        };
        /**
         * Returns the resultant pattern based on the skeleton, dateObject and the type provided
         * @private
         * @param {string} skeleton
         * @param {Object} dateObject
         * @param {string} type
         * @returns {string}
         */
        function getResultantPattern(skeleton: string, dateObject: Object, type: string): string;
        /**
         * Returns the dependable object for provided cldr data and culture
         * @private
         * @param {Object} cldr
         * @param {string} culture
         * @param {boolean} isNumber
         * @returns {Dependables}
         */
        function getDependables(cldr: Object, culture: string, isNumber?: boolean): Dependables;
        /**
         * Returns the symbol pattern for provided parameters
         * @private
         * @param {string} type
         * @param {string} numSystem
         * @param {Object} obj
         * @param {boolean} isAccount
         * @returns {string}
         */
        function getSymbolPattern(type: string, numSystem: string, obj: Object, isAccount: boolean): string;
        /**
         * Returns proper numeric skeleton
         * @private
         * @param {string} skeleton
         * @returns {NumericSkeleton}
         */
        function getProperNumericSkeleton(skeleton: string): NumericSkeleton;
        /**
         * Returns format data for number formatting like minimum fraction, maximum fraction, etc..,
         * @private
         * @param {string} pattern
         * @param {boolean} needFraction
         * @param {string} cSymbol
         * @param {boolean} fractionOnly
         * @returns {NegativeData}
         */
        function getFormatData(pattern: string, needFraction: boolean, cSymbol: string, fractionOnly?: boolean): NegativeData;
        /**
         * Returns currency symbol based on currency code
         * @private
         * @param {Object} numericObject
         * @param {string} currencyCode
         * @returns {string}
         */
        function getCurrencySymbol(numericObject: Object, currencyCode: string): string;
        /**
         * Returns formatting options for custom number format
         * @private
         * @param {string} format
         * @param {CommonOptions} dOptions
         * @param {Dependables} obj
         * @returns {GenericFormatOptions}
         */
        function customFormat(format: string, dOptions: CommonOptions, obj: Dependables): GenericFormatOptions;
        /**
         * Returns formatting options for currency or percent type
         * @private
         * @param {string[]} parts
         * @param {string} actual
         * @param {string} symbol
         * @returns {NegativeData}
         */
        function isCurrencyPercent(parts: string[], actual: string, symbol: string): NegativeData;
        /**
         * Returns culture based date separator
         * @private
         * @param {Object} dateObj
         * @returns {string}
         */
        function getDateSeparator(dateObj: Object): string;
        /**
         * Returns Native Date Time pattern
         * @private
         * @param {string} culture
         * @param {DateFormatOptions} options
         * @param {Object} cldr
         * @returns {string}
         */
        function getActualDateTimeFormat(culture: string, options: DateFormatOptions, cldr?: Object, isExcelFormat?: boolean): string;
        /**
         * Returns Native Number pattern
         * @private
         * @param {string} culture
         * @param {NumberFormatOptions} options
         * @param {Object} cldr
         * @returns {string}
         */
        function getActualNumberFormat(culture: string, options: ej.base.NumberFormatOptions, cldr?: Object): string;
    }
    
    /**
     * Interface for default formatting options
     * @private
     */
    export interface FormatParts extends NumberFormatOptions {
        groupOne?: boolean;
        isPercent?: boolean;
        isCurrency?: boolean;
        isNegative?: boolean;
        groupData?: GroupDetails;
        groupSeparator?: string;
    }
    /**
     * Interface for common formatting options
     */
    export interface CommonOptions {
        numberMapper?: NumberMapper;
        currencySymbol?: string;
        percentSymbol?: string;
        minusSymbol?: string;
    }
    /**
     * Interface for grouping process
     */
    export interface GroupDetails {
        primary?: number;
        secondary?: number;
    }
    /**
     * Module for number formatting.
     * @private
     */
    export class NumberFormat {
        /**
         * Returns the formatter function for given skeleton.
         * @param {string} culture -  Specifies the culture name to be which formatting.
         * @param {NumberFormatOptions} option - Specific the format in which number  will format.
         * @param {Object} object- Specifies the global cldr data collection.
         * @return Function.
         */
        static numberFormatter(culture: string, option: ej.base.NumberFormatOptions, cldr: Object): Function;
        /**
         * Returns grouping details for the pattern provided
         * @param {string} pattern
         * @returns {GroupDetails}
         */
        static getGroupingDetails(pattern: string): GroupDetails;
        /**
         * Returns if the provided integer range is valid.
         * @param {number} val1
         * @param {number} val2
         * @param {boolean} checkbothExist
         * @param {boolean} isFraction
         * @returns {boolean}
         */
        private static checkValueRange(val1, val2, checkbothExist, isFraction?);
        /**
         * Check if the provided fraction range is valid
         * @param {number} val
         * @param {string} text
         * @param {boolean} isFraction
         * @returns {void}
         */
        private static checkRange(val, text, isFraction?);
        /**
         * Returns formatted numeric string for provided formatting options
         * @param {number} value
         * @param {base.GenericFormatOptions} fOptions
         * @param {CommonOptions} dOptions
         * @returns {string}
         */
        private static intNumberFormatter(value, fOptions, dOptions);
        /**
         * Returns significant digits processed numeric string
         * @param {number} value
         * @param {number} min
         * @param {number} max
         * @returns {string}
         */
        private static processSignificantDigits(value, min, max);
        /**
         * Returns grouped numeric string
         * @param {string} val
         * @param {number} level1
         * @param {string} sep
         * @param {string} decimalSymbol
         * @param {number} level2
         * @returns {string}
         */
        private static groupNumbers(val, level1, sep, decimalSymbol, level2?);
        /**
         * Returns fraction processed numeric string
         * @param {number} value
         * @param {number} min
         * @param {number} max
         * @returns {string}
         */
        private static processFraction(value, min, max);
        /**
         * Returns integer processed numeric string
         * @param {string} value
         * @param {number} min
         * @returns {string}
         */
        private static processMinimumIntegers(value, min);
    }
    
    /**
     * interface for Numeric Formatting Parts
     */
    export interface NumericParts {
        symbolRegex?: RegExp;
        nData?: any;
        pData?: any;
        infinity?: string;
        type?: string;
        fractionDigits?: number;
        isAccount?: boolean;
        custom?: boolean;
    }
    /**
     * Module for Number Parser.
     * @private
     */
    export class NumberParser {
        /**
         * Returns the parser function for given skeleton.
         * @param {string} -  Specifies the culture name to be which formatting.
         * @param {NumberFormatOptions} - Specific the format in which number  will parsed.
         * @param {cldr} - Specifies the global cldr data collection.
         * @return Function.
         */
        static numberParser(culture: string, option: ej.base.NumberFormatOptions, cldr: Object): Function;
        /**
         * Returns parsed number for the provided formatting options
         * @param {string} value
         * @param {NumericParts} options
         * @param {NumericOptions} numOptions
         * @returns {number}
         */
        private static getParsedNumber(value, options, numOptions);
    }
    
    /**
     * Interface for numeric Options
     */
    export interface NumericOptions {
        numericPair?: Object;
        numericRegex?: string;
        numberParseRegex?: RegExp;
        symbolNumberSystem?: Object;
        symbolMatch?: Object;
        numberSystem?: string;
    }
    /**
     * Interface for numeric object
     */
    export interface NumericObject {
        obj?: Object;
        nSystem?: string;
    }
    /**
     * Interface for number mapper
     */
    export interface NumberMapper {
        mapper?: Object;
        timeSeparator?: string;
        numberSymbols?: Object;
        numberSystem?: string;
    }
    /**
     * Interface for parser base
     * @private
     */
    export class ParserBase {
        static nPair: string;
        static nRegex: string;
        static numberingSystems: Object;
        /**
         * Returns the cldr object for the culture specifies
         * @param {Object} obj - Specifies the object from which culture object to be acquired.
         * @param {string} cName - Specifies the culture name.
         * @returns {Object}
         */
        static getMainObject(obj: Object, cName: string): Object;
        /**
         * Returns the numbering system object from given cldr data.
         * @param {Object} obj - Specifies the object from which number system is acquired.
         * @returns {Object}
         */
        static getNumberingSystem(obj: Object): Object;
        /**
         * Returns the reverse of given object keys or keys specified.
         * @param {Object} prop - Specifies the object to be reversed.
         * @param {number[]} keys - Optional parameter specifies the custom keyList for reversal.
         * @returns {Object}
         */
        static reverseObject(prop: Object, keys?: number[]): Object;
        /**
         * Returns the symbol regex by skipping the escape sequence.
         * @param {string[]} props - Specifies the array values to be skipped.
         * @returns {RegExp}
         */
        static getSymbolRegex(props: string[]): RegExp;
        private static getSymbolMatch(prop);
        /**
         * Returns regex string for provided value
         * @param {string} val
         * @returns {string}
         */
        private static constructRegex(val);
        /**
         * Returns the replaced value of matching regex and obj mapper.
         * @param {string} value - Specifies the  values to be replaced.
         * @param {RegExp} regex - Specifies the  regex to search.
         * @param {Object} obj - Specifies the  object matcher to be replace value parts.
         * @returns {string}
         */
        static convertValueParts(value: string, regex: RegExp, obj: Object): string;
        /**
         * Returns default numbering system object for formatting from cldr data
         * @param {Object} obj
         * @returns {NumericObject}
         */
        static getDefaultNumberingSystem(obj: Object): NumericObject;
        /**
         * Returns the replaced value of matching regex and obj mapper.
         */
        static getCurrentNumericOptions(curObj: Object, numberSystem: Object, needSymbols?: boolean): Object;
        /**
         * Returns number mapper object for the provided cldr data
         * @param {Object} curObj
         * @param {Object} numberSystem
         * @param {boolean} isNumber
         * @returns {NumberMapper}
         */
        static getNumberMapper(curObj: Object, numberSystem: Object, isNumber?: boolean): NumberMapper;
    }
    
  }
  export namespace buttons {
    
    /**
     * Interface for a class Button
     */
    export interface ButtonModel extends ej.base.ComponentModel{
    
        /**
         * Positions the icon before/after the text content in the Button.
         * The possible values are:
         * * Left: The icon will be positioned to the left of the text content.
         * * Right: The icon will be positioned to the right of the text content.
         * @default "left"
         */
        iconPosition?: IconPosition;
    
        /**
         * Defines class/multiple classes separated by a space for the Button that is used to include an icon.
         * Buttons can also include font icon and sprite image.
         * @default ""
         */
        iconCss?: string;
    
        /**
         * Specifies a value that indicates whether the Button is `disabled` or not.
         * @default false.
         */
        disabled?: boolean;
    
        /**
         * Allows the appearance of the Button to be enhanced and visually appealing when set to `true`.
         * @default false
         */
        isPrimary?: boolean;
    
        /**
         * Defines class/multiple classes separated by a space in the Button element. The Button types, styles, and
         * size can be defined by using [`this`](http://ej2.syncfusion.com/documentation/button/howto.html).
         * @default ""
         */
        cssClass?: string;
    
        /**
         * Defines the `content` of the Button element that can either be a text or HTML elements.
         * @default ""
         */
        content?: string;
    
        /**
         * Makes the Button toggle, when set to `true`. When you click it, the state changes from normal to active.
         * @default false
         */
        isToggle?: boolean;
    
    }
    
    export type IconPosition = 'Left' | 'Right';
    /**
     * The Button is a graphical user interface element that triggers an event on its click action. It can contain a text, an image, or both.
     * ```html
     * <button id="button">Button</button>
     * ```
     * ```typescript
     * <script>
     * var btnObj = new Button();
     * btnObj.appendTo("#button");
     * </script>
     * ```
     */
    export class Button extends ej.base.Component<HTMLButtonElement> implements ej.base.INotifyPropertyChanged {
        /**
         * Positions the icon before/after the text content in the Button.
         * The possible values are:
         * * Left: The icon will be positioned to the left of the text content.
         * * Right: The icon will be positioned to the right of the text content.
         * @default "left"
         */
        iconPosition: IconPosition;
        /**
         * Defines class/multiple classes separated by a space for the Button that is used to include an icon.
         * Buttons can also include font icon and sprite image.
         * @default ""
         */
        iconCss: string;
        /**
         * Specifies a value that indicates whether the Button is `disabled` or not.
         * @default false.
         */
        disabled: boolean;
        /**
         * Allows the appearance of the Button to be enhanced and visually appealing when set to `true`.
         * @default false
         */
        isPrimary: boolean;
        /**
         * Defines class/multiple classes separated by a space in the Button element. The Button types, styles, and
         * size can be defined by using [`this`](http://ej2.syncfusion.com/documentation/button/howto.html).
         * @default ""
         */
        cssClass: string;
        /**
         * Defines the `content` of the Button element that can either be a text or HTML elements.
         * @default ""
         */
        content: string;
        /**
         * Makes the Button toggle, when set to `true`. When you click it, the state changes from normal to active.
         * @default false
         */
        isToggle: boolean;
        /**
         * Constructor for creating the widget
         * @param  {ButtonModel} options?
         * @param  {string|HTMLButtonElement} element?
         */
        constructor(options?: ej.buttons.ButtonModel, element?: string | HTMLButtonElement);
        protected preRender(): void;
        /**
         * Initialize the control rendering
         * @returns void
         * @private
         */
        render(): void;
        private initialize();
        private controlStatus(disabled);
        private setIconCss();
        protected wireEvents(): void;
        protected unWireEvents(): void;
        private btnClickHandler();
        /**
         * Destroys the widget.
         * @returns void
         */
        destroy(): void;
        /**
         * Get component name.
         * @returns string
         * @private
         */
        getModuleName(): string;
        /**
         * Get the properties to be maintained in the persisted state.
         * @returns string
         */
        getPersistData(): string;
        /**
         * Called internally if any of the property value changed.
         * @param  {Button} newProp
         * @param  {Button} oldProp
         * @returns void
         * @private
         */
        onPropertyChanged(newProp: ej.buttons.Button, oldProp: ej.buttons.Button): void;
    }
    
    /**
     * Interface for a class CheckBox
     */
    export interface CheckBoxModel extends ej.base.ComponentModel{
    
        /**
         * Triggers when the CheckBox state has been changed by user interaction.
         * @event
         */
        change?: ej.base.EmitType<ChangeEventArgs>;
    
        /**
         * Specifies a value that indicates whether the CheckBox is `checked` or not.
         * When set to `true`, the CheckBox will be in `checked` state.
         * @default false
         */
        checked?: boolean;
    
        /**
         * Defines class/multiple classes separated by a space in the CheckBox element.
         * You can add custom styles to the CheckBox by using this property.
         * @default ''
         */
        cssClass?: string;
    
        /**
         * Specifies a value that indicates whether the CheckBox is `disabled` or not.
         * When set to `true`, the CheckBox will be in `disabled` state.
         * @default false
         */
        disabled?: boolean;
    
        /**
         * Specifies a value that indicates whether the CheckBox is in `indeterminate` state or not.
         * When set to `true`, the CheckBox will be in `indeterminate` state.
         * @default false
         */
        indeterminate?: boolean;
    
        /**
         * Defines the caption for the CheckBox, that describes the purpose of the CheckBox.
         * @default ''
         */
        label?: string;
    
        /**
         * Positions label `before`/`after` the CheckBox.
         * The possible values are:
         * * Before - The label is positioned to left of the CheckBox.
         * * After - The label is positioned to right of the CheckBox.
         * @default 'After'
         */
        labelPosition?: LabelPosition;
    
        /**
         * Defines `name` attribute for the CheckBox.
         * It is used to reference form data (CheckBox value) after a form is submitted.
         * @default ''
         */
        name?: string;
    
        /**
         * Defines `value` attribute for the CheckBox.
         * It is a form data passed to the server when submitting the form.
         * @default ''
         */
        value?: string;
    
    }
    
    export type LabelPosition = 'After' | 'Before';
    /**
     * The CheckBox is a graphical user interface element that allows you to select one or more options from the choices.
     * It contains checked, unchecked, and indeterminate states.
     * ```html
     * <input type="checkbox" id="checkbox"/>
     * <script>
     * var checkboxObj = new CheckBox({ label: "Default" });
     * checkboxObj.appendTo("#checkbox");
     * </script>
     * ```
     */
    export class CheckBox extends ej.base.Component<HTMLInputElement> implements ej.base.INotifyPropertyChanged {
        private tagName;
        private isKeyPressed;
        private keyboardModule;
        /**
         * Triggers when the CheckBox state has been changed by user interaction.
         * @event
         */
        change: ej.base.EmitType<ChangeEventArgs>;
        /**
         * Specifies a value that indicates whether the CheckBox is `checked` or not.
         * When set to `true`, the CheckBox will be in `checked` state.
         * @default false
         */
        checked: boolean;
        /**
         * Defines class/multiple classes separated by a space in the CheckBox element.
         * You can add custom styles to the CheckBox by using this property.
         * @default ''
         */
        cssClass: string;
        /**
         * Specifies a value that indicates whether the CheckBox is `disabled` or not.
         * When set to `true`, the CheckBox will be in `disabled` state.
         * @default false
         */
        disabled: boolean;
        /**
         * Specifies a value that indicates whether the CheckBox is in `indeterminate` state or not.
         * When set to `true`, the CheckBox will be in `indeterminate` state.
         * @default false
         */
        indeterminate: boolean;
        /**
         * Defines the caption for the CheckBox, that describes the purpose of the CheckBox.
         * @default ''
         */
        label: string;
        /**
         * Positions label `before`/`after` the CheckBox.
         * The possible values are:
         * * Before - The label is positioned to left of the CheckBox.
         * * After - The label is positioned to right of the CheckBox.
         * @default 'After'
         */
        labelPosition: LabelPosition;
        /**
         * Defines `name` attribute for the CheckBox.
         * It is used to reference form data (CheckBox value) after a form is submitted.
         * @default ''
         */
        name: string;
        /**
         * Defines `value` attribute for the CheckBox.
         * It is a form data passed to the server when submitting the form.
         * @default ''
         */
        value: string;
        /**
         * Constructor for creating the widget
         * @private
         */
        constructor(options?: CheckBoxModel, element?: string | HTMLInputElement);
        private changeState(state?);
        private clickHandler(event);
        /**
         * Destroys the widget.
         * @returns void
         */
        destroy(): void;
        private focusHandler();
        private focusOutHandler();
        /**
         * Gets the module name.
         * @private
         */
        protected getModuleName(): string;
        /**
         * Gets the properties to be maintained in the persistence state.
         * @private
         */
        getPersistData(): string;
        private getWrapper();
        private initialize();
        private initWrapper();
        private keyDownHandler();
        private labelMouseHandler(e);
        private mouseDownHandler();
        /**
         * Called internally if any of the property value changes.
         * @private
         */
        onPropertyChanged(newProp: CheckBoxModel, oldProp: CheckBoxModel): void;
        /**
         * Initialize Angular, React and Unique ID support.
         * @private
         */
        protected preRender(): void;
        /**
         * Initialize the control rendering
         * @private
         */
        protected render(): void;
        private setDisabled();
        private setText(text);
        protected unWireEvents(): void;
        protected wireEvents(): void;
    }
    export interface ChangeEventArgs extends ej.base.BaseEventArgs {
        /** Returns the event parameters of the CheckBox. */
        event?: Event;
        /** Returns the checked value of the CheckBox. */
        checked?: boolean;
    }
    
    /**
     * Initialize wrapper element for angular.
     * @private
     */
    export function wrapperInitialize(tag: string, type: string, element: HTMLInputElement, WRAPPER: string): HTMLInputElement;
    export function getTextNode(element: HTMLElement): Node;
    /**
     * Creates CheckBox component UI with theming and ripple support.
     * @private
     */
    export function createCheckBox(enableRipple?: boolean, options?: CheckBoxUtilModel): Element;
    export function rippleMouseHandler(e: MouseEvent, rippleSpan: Element): void;
    export interface CheckBoxUtilModel {
        checked?: boolean;
        label?: string;
        enableRtl?: boolean;
        cssClass?: string;
    }
    
    /**
     * Interface for a class RadioButton
     */
    export interface RadioButtonModel extends ej.base.ComponentModel{
    
        /**
         * Event trigger when the RadioButton state has been changed by user interaction.
         * @event
         */
        change?: ej.base.EmitType<ChangeArgs>;
    
        /**
         * Specifies a value that indicates whether the RadioButton is `checked` or not.
         * When set to `true`, the RadioButton will be in `checked` state.
         * @default false
         */
        checked?: boolean;
    
        /**
         * Defines class/multiple classes separated by a space in the RadioButton element.
         * You can add custom styles to the RadioButton by using this property.
         * @default ''
         */
        cssClass?: string;
    
        /**
         * Specifies a value that indicates whether the RadioButton is `disabled` or not.
         * When set to `true`, the RadioButton will be in `disabled` state.
         * @default false
         */
        disabled?: boolean;
    
        /**
         * Defines the caption for the RadioButton, that describes the purpose of the RadioButton.
         * @default ''
         */
        label?: string;
    
        /**
         * Positions label `before`/`after` the RadioButton.
         * The possible values are:
         * * Before: The label is positioned to left of the RadioButton.
         * * After: The label is positioned to right of the RadioButton.
         * @default 'After'
         */
        labelPosition?: RadioLabelPosition;
    
        /**
         * Defines `name` attribute for the RadioButton.
         * It is used to reference form data (RadioButton value) after a form is submitted.
         * @default ''
         */
        name?: string;
    
        /**
         * Defines `value` attribute for the RadioButton.
         * It is a form data passed to the server when submitting the form.
         * @default ''
         */
        value?: string;
    
    }
    
    export type RadioLabelPosition = 'After' | 'Before';
    /**
     * The RadioButton is a graphical user interface element that allows you to select one option from the choices.
     * It contains checked and unchecked states.
     * ```html
     * <input type="radio" id="radio"/>
     * <script>
     * var radioObj = new RadioButton({ label: "Default" });
     * radioObj.appendTo("#radio");
     * </script>
     * ```
     */
    export class RadioButton extends ej.base.Component<HTMLInputElement> implements ej.base.INotifyPropertyChanged {
        private tagName;
        private isKeyPressed;
        /**
         * Event trigger when the RadioButton state has been changed by user interaction.
         * @event
         */
        change: ej.base.EmitType<ChangeArgs>;
        /**
         * Specifies a value that indicates whether the RadioButton is `checked` or not.
         * When set to `true`, the RadioButton will be in `checked` state.
         * @default false
         */
        checked: boolean;
        /**
         * Defines class/multiple classes separated by a space in the RadioButton element.
         * You can add custom styles to the RadioButton by using this property.
         * @default ''
         */
        cssClass: string;
        /**
         * Specifies a value that indicates whether the RadioButton is `disabled` or not.
         * When set to `true`, the RadioButton will be in `disabled` state.
         * @default false
         */
        disabled: boolean;
        /**
         * Defines the caption for the RadioButton, that describes the purpose of the RadioButton.
         * @default ''
         */
        label: string;
        /**
         * Positions label `before`/`after` the RadioButton.
         * The possible values are:
         * * Before: The label is positioned to left of the RadioButton.
         * * After: The label is positioned to right of the RadioButton.
         * @default 'After'
         */
        labelPosition: RadioLabelPosition;
        /**
         * Defines `name` attribute for the RadioButton.
         * It is used to reference form data (RadioButton value) after a form is submitted.
         * @default ''
         */
        name: string;
        /**
         * Defines `value` attribute for the RadioButton.
         * It is a form data passed to the server when submitting the form.
         * @default ''
         */
        value: string;
        /**
         * Constructor for creating the widget
         * @private
         */
        constructor(options?: RadioButtonModel, element?: string | HTMLInputElement);
        private changeHandler(event);
        private updateChange(state);
        /**
         * Destroys the widget.
         * @returns void
         */
        destroy(): void;
        private focusHandler();
        private focusOutHandler();
        protected getModuleName(): string;
        /**
         * Gets the properties to be maintained in the persistence state.
         * @private
         */
        getPersistData(): string;
        private getLabel();
        private initialize();
        private initWrapper();
        private keyDownHandler();
        private labelRippleHandler(e);
        private mouseDownHandler();
        /**
         * Called internally if any of the property value changes.
         * @private
         */
        onPropertyChanged(newProp: RadioButtonModel, oldProp: RadioButtonModel): void;
        /**
         * Initialize checked Property, Angular and React and Unique ID support.
         * @private
         */
        protected preRender(): void;
        /**
         * Initialize the control rendering
         * @private
         */
        protected render(): void;
        private setDisabled();
        private setText(text);
        protected unWireEvents(): void;
        protected wireEvents(): void;
    }
    export interface ChangeArgs extends ej.base.BaseEventArgs {
        /** Returns the value of the RadioButton. */
        value?: string;
        /** Returns the event parameters of the RadioButton. */
        event?: Event;
    }
    
  }
  export namespace calendars {
    
    /**
     * Interface for a class Calendar
     */
    export interface CalendarModel extends ej.base.ComponentModel{
    
        /**
         * Gets or sets the selected date of the Calendar.
         * @default null
         */
        value?: Date;
    
        /**
         * Gets or sets the minimum date that can be selected in the Calendar.
         * @default new Date(1900, 00, 01)
         */
        min?: Date;
    
        /**
         * Gets or sets the maximum date that can be selected in the Calendar.
         * @default new Date(2099, 11, 31)
         */
        max?: Date;
    
        /**
         * Gets or sets the Calendar's first day of the week. By default, the first day of the week will be based on the current culture.
         * @default 0
         */
        firstDayOfWeek?: number;
    
        /**
         * Specifies the initial view of the Calendar when it is opened.
         * With the help of this property, initial view can be changed to year or decade view.
         * @default month
         */
        start?: ej.calendars.CalendarView;
    
        /**
         * Sets the maximum level of view (month, year, decade) in the Calendar.
         * Depth view should be smaller than the start view to restrict its view navigation.
         * @default month
         */
        depth?: ej.calendars.CalendarView;
    
        /**
         * Determines whether the week number of the Calendar is to be displayed or not.
         * The week number is displayed in every week row.
         * @default false
         */
        weekNumber?: boolean;
    
        /**
         * Specifies whether the today button is displayed or not.
         * @default true
         */
        showTodayButton?: boolean;
    
        /**
         * Triggers when Calendar is created.
         * @event 
         */
        created?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when Calendar is destroyed.
         * @event 
         */
        destroyed?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the Calendar value is changed.
         * @event  
         */
        change?: ej.base.EmitType<ChangedEventArgs>;
    
        /**
         * Triggers when the Calendar is navigated to another level or within the same level of view.
         * @event
         */
        navigated?: ej.base.EmitType<NavigatedEventArgs>;
    
        /**
         * Triggers when each day cell of the Calendar is rendered.
         * @event
         */
        renderDayCell?: ej.base.EmitType<RenderDayCellEventArgs>;
    
    }
    
    /**
     * Specifies the view of the calendar.
     */
    export type CalendarView = 'Month' | 'Year' | 'Decade';
    /**
     * Represents the Calendar component that allows the user to select a date.
     * ```html
     * <div id="calendar"/>
     * ```
     * ```typescript
     * <script>
     *   var calendarObj = new Calendar({ value: new Date() });
     *   calendarObj.appendTo("#calendar");
     * </script>
     * ```
     */
    export class Calendar extends ej.base.Component<HTMLElement> implements ej.base.INotifyPropertyChanged {
        protected headerElement: HTMLElement;
        protected contentElement: HTMLElement;
        protected table: HTMLElement;
        protected tableHeadElement: HTMLElement;
        protected tableBodyElement: Element;
        protected nextIcon: HTMLElement;
        protected previousIcon: HTMLElement;
        protected headerTitleElement: HTMLElement;
        protected todayElement: HTMLElement;
        protected footer: HTMLElement;
        protected keyboardModule: ej.base.KeyboardEvents;
        protected globalize: ej.base.Internationalization;
        protected currentDate: Date;
        protected changedArgs: ChangedEventArgs;
        protected navigatedArgs: NavigatedEventArgs;
        private renderDaycellArg;
        private effect;
        protected previousDate: Date;
        protected changeHandler: Function;
        protected navigateHandler: Function;
        protected navigatePreviousHandler: Function;
        protected navigateNextHandler: Function;
        private l10;
        protected todayDisabled: boolean;
        protected todayDate: Date;
        protected calendarElement: HTMLElement;
        protected keyConfigs: {
            [key: string]: string;
        };
        /**
         * Gets or sets the selected date of the Calendar.
         * @default null
         */
        value: Date;
        /**
         * Gets or sets the minimum date that can be selected in the Calendar.
         * @default new Date(1900, 00, 01)
         */
        min: Date;
        /**
         * Gets or sets the maximum date that can be selected in the Calendar.
         * @default new Date(2099, 11, 31)
         */
        max: Date;
        /**
         * Gets or sets the Calendar's first day of the week. By default, the first day of the week will be based on the current culture.
         * @default 0
         */
        firstDayOfWeek: number;
        /**
         * Specifies the initial view of the Calendar when it is opened.
         * With the help of this property, initial view can be changed to year or decade view.
         * @default month
         */
        start: ej.calendars.CalendarView;
        /**
         * Sets the maximum level of view (month, year, decade) in the Calendar.
         * Depth view should be smaller than the start view to restrict its view navigation.
         * @default month
         */
        depth: ej.calendars.CalendarView;
        /**
         * Determines whether the week number of the Calendar is to be displayed or not.
         * The week number is displayed in every week row.
         * @default false
         */
        weekNumber: boolean;
        /**
         * Specifies whether the today button is displayed or not.
         * @default true
         */
        showTodayButton: boolean;
        /**
         * Triggers when Calendar is created.
         * @event
         */
        created: ej.base.EmitType<Object>;
        /**
         * Triggers when Calendar is destroyed.
         * @event
         */
        destroyed: ej.base.EmitType<Object>;
        /**
         * Triggers when the Calendar value is changed.
         * @event
         */
        change: ej.base.EmitType<ChangedEventArgs>;
        /**
         * Triggers when the Calendar is navigated to another level or within the same level of view.
         * @event
         */
        navigated: ej.base.EmitType<NavigatedEventArgs>;
        /**
         * Triggers when each day cell of the Calendar is rendered.
         * @event
         */
        renderDayCell: ej.base.EmitType<RenderDayCellEventArgs>;
        /**
         * Initialized new instance of Calendar Class.
         * Constructor for creating the widget
         * @param  {CalendarModel} options?
         * @param  {string|HTMLElement} element?
         */
        constructor(options?: CalendarModel, element?: string | HTMLElement);
        /**
         * To Initialize the control rendering.
         * @returns void
         * @private
         */
        protected render(): void;
        protected processDate(): void;
        private validateDate();
        private minMaxUpdate();
        protected header(): void;
        protected content(): void;
        private getCultureValues();
        private contentHdr();
        private contentBody();
        private updateFooter();
        private contentFooter();
        protected wireEvents(): void;
        protected todayButtonClick(): void;
        protected keyActionHandle(e: ej.base.KeyboardEventArgs): void;
        protected KeyboardNavigate(number: number, currentView: number, e: KeyboardEvent, max: Date, min: Date): void;
        /**
         * Initialize the event handler
         * @private
         */
        protected preRender(): void;
        private minMaxDate(localDate);
        protected renderMonths(e?: Event): void;
        private renderDays(currentDate, e?);
        private renderYears(e?);
        private renderDecades(e?);
        private dayCell(localDate);
        protected firstDay(date: Date): Date;
        protected lastDay(date: Date): Date;
        protected checkDateValue(value: Date): Date;
        private findlastDay(date);
        private removeTheadEle();
        private renderTemplate(elements, numCells, classNm, e?);
        private clickHandler(e);
        private contentClick(e, view, ele);
        private switchView(view, e?);
        /**
         * To get component name
         * @private
         */
        protected getModuleName(): string;
        /**
         * Gets the properties to be maintained upon browser refresh.
         * @returns string
         */
        getPersistData(): string;
        /**
         * Called internally if any of the property value changed.
         * returns void
         * @private
         */
        onPropertyChanged(newProp: CalendarModel, oldProp: CalendarModel): void;
        private setvalue();
        private titleUpdate(date, view);
        protected setActiveDescendant(): string;
        private iconHandler();
        /**
         * Destroys the widget.
         * @returns void
         */
        destroy(): void;
        private title(e?);
        private getViewNumber(stringVal);
        private navTitle(e?);
        private previous();
        protected navigatePrevious(e: MouseEvent | KeyboardEvent): void;
        private next();
        protected navigateNext(eve: MouseEvent | KeyboardEvent): void;
        /**
         * This method is used to navigate to the month/year/decade view of the Calendar.
         * @param  {string} view - Specifies the view of the Calendar.
         * @param  {Date} date - Specifies the focused date in a view.
         * @returns void
         */
        navigateTo(view: ej.calendars.CalendarView, date: Date): void;
        /**
         * Gets the current view of the Calendar.
         * @returns string
         */
        currentView(): string;
        private getDateVal(date);
        private getCultureObjects(ld, c);
        private getWeek(d);
        private setTime(date, time);
        protected addMonths(date: Date, i: number): void;
        protected addYears(date: Date, i: number): void;
        protected getIdValue(e: MouseEvent, element: Element): Date;
        private selectDate(e, date, element);
        private setAriaActivedescendant();
        protected previousIconHandler(disabled: boolean): void;
        protected renderDayCellEvent(args: RenderDayCellEventArgs): void;
        protected navigatedEvent(eve: MouseEvent | KeyboardEvent): void;
        private triggerNavigate(event);
        protected changeEvent(e: Event): void;
        private triggerChange(e);
        protected nextIconHandler(disabled: boolean): void;
        private compare(startDate, endDate, modifier);
        protected isMinMaxRange(date: Date): boolean;
        private compareYear(start, end);
        private compareDecade(start, end);
        private shiftArray(array, i);
        protected addDay(date: Date, i: number, e: KeyboardEvent, max: Date, min: Date): void;
        private findNextTD(date, column, max, min);
        private getMaxDays(d);
        private setDateDecade(date, year);
        private setDateYear(date, value);
        private compareMonth(start, end);
    }
    export interface RenderDayCellEventArgs extends ej.base.BaseEventArgs {
        /** Specifies whether to disable the current date or not. */
        isDisabled?: boolean;
        /** Specifies the day cell element. */
        element?: HTMLElement;
        /** Defines the current date of the Calendar. */
        date?: Date;
        /** Defines whether the current date is out of range (less than min or greater than max) or not. */
        isOutOfRange?: boolean;
    }
    export interface ChangedEventArgs extends ej.base.BaseEventArgs {
        /** Defines the selected date of the Calendar. */
        value?: Date;
        /** Defines the event of the Calendar. */
        event?: KeyboardEvent | MouseEvent;
    }
    export interface NavigatedEventArgs extends ej.base.BaseEventArgs {
        /** Defines the current view of the Calendar. */
        view?: string;
        /** Defines the focused date in a view. */
        date?: Date;
        /** Defines the event of the Calendar. */
        event?: KeyboardEvent | MouseEvent;
    }
    
    /**
     * Interface for a class DatePicker
     */
    export interface DatePickerModel extends CalendarModel{
    
        /**
         * Specifies the width of the DatePicker component.
         * @default null
         */
        width?: number | string;
    
        /**
         * Specifies the root CSS class of the DatePicker that allows to
         * customize the appearance by overriding the styles.
         * @default null
         */
        cssClass?: string;
    
        /**
         * Specifies the component to act as strict. So that, it allows to enter only a valid date value within a specified range or else it 
         * will resets to previous value. By default, strictMode is in false.
         * @default false
         */
        strictMode?: boolean;
    
        /**
         * Specifies the format of the value that to be displayed in component. By default, the format is
         * based on the culture. 
         * @default null
         */
        format?: string;
    
        /**
         * Specifies the component to be disabled or not.
         * @default true
         */
        enabled?: boolean;
    
        /**
         * Specifies whether to show or hide the clear Icon
         * @default false
         */
        showClearButton?: boolean;
    
        /**
         * specifies the z-index value of the datePicker popup element.
         * @default 1000
         */
        zIndex?: number;
    
        /**
         * Specifies the component in readonly state. 
         * @default false
         */
        readonly?: boolean;
    
        /**
         * Specifies the placeholder text that displayed in textbox.
         * @default null
         */
        placeholder?: string;
    
        /**
         * Specifies the placeholder text to be floated.
         * @default Auto
         */
        floatLabelType?: any;
    
        /**
         * Triggers when the popup gets open.
         * @event 
         */
        open?: ej.base.EmitType<PreventableEventArgs | PopupObjectArgs>;
    
        /**
         * Triggers when the popup gets close.
         * @event 
         */
        close?: ej.base.EmitType<PreventableEventArgs | PopupObjectArgs>;
    
        /**
         * Triggers when the control loses the focus.
         * @event 
         */
        blur?: ej.base.EmitType<Object>;
    
        /**
         *  Triggers when the control gets focus.
         * @event 
         */
        focus?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the component is created.
         * @event 
         */
        created?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the component is destroyed.
         * @event 
         */
        destroyed?: ej.base.EmitType<Object>;
    
    }
    
    /**
     * Represents the DatePicker component that allows user to select
     * or enter a date value.
     * ```html
     * <input id="datepicker"/>
     * ````
     * ````typescript
     * <script>
     *   var datePickerObject = new DatePicker({ value: new Date() });
     *   datePickerObject.appendTo("#datepicker");
     * </script>
     * ```
     */
    export class DatePicker extends Calendar implements ej.inputs.IInput {
        private popupObj;
        protected inputWrapper: ej.inputs.InputObject;
        private modal;
        protected inputElement: HTMLInputElement;
        protected popupWrapper: HTMLElement;
        protected changedArgs: ChangedEventArgs;
        protected previousDate: Date;
        private keyboardModules;
        private calendarKeyboardModules;
        private previousEleValue;
        private ngTag;
        protected dateTimeFormat: string;
        private inputEleCopy;
        protected l10n: ej.base.L10n;
        private preventArgs;
        private isDateIconClicked;
        private index;
        protected keyConfigs: {
            [key: string]: string;
        };
        protected calendarKeyConfigs: {
            [key: string]: string;
        };
        /**
         * Specifies the width of the DatePicker component.
         * @default null
         */
        width: number | string;
        /**
         * Specifies the root CSS class of the DatePicker that allows to
         * customize the appearance by overriding the styles.
         * @default null
         */
        cssClass: string;
        /**
         * Specifies the component to act as strict. So that, it allows to enter only a valid date value within a specified range or else it
         * will resets to previous value. By default, strictMode is in false.
         * @default false
         */
        strictMode: boolean;
        /**
         * Specifies the format of the value that to be displayed in component. By default, the format is
         * based on the culture.
         * @default null
         */
        format: string;
        /**
         * Specifies the component to be disabled or not.
         * @default true
         */
        enabled: boolean;
        /**
         * Specifies whether to show or hide the clear Icon
         * @default false
         */
        showClearButton: boolean;
        /**
         * specifies the z-index value of the datePicker popup element.
         * @default 1000
         */
        zIndex: number;
        /**
         * Specifies the component in readonly state.
         * @default false
         */
        readonly: boolean;
        /**
         * Specifies the placeholder text that displayed in textbox.
         * @default null
         */
        placeholder: string;
        /**
         * Specifies the placeholder text to be floated.
         * @default Auto
         */
        floatLabelType: ej.inputs.FloatLabelType;
        /**
         * Triggers when the popup gets open.
         * @event
         */
        open: ej.base.EmitType<PreventableEventArgs | PopupObjectArgs>;
        /**
         * Triggers when the popup gets close.
         * @event
         */
        close: ej.base.EmitType<PreventableEventArgs | PopupObjectArgs>;
        /**
         * Triggers when the control loses the focus.
         * @event
         */
        blur: ej.base.EmitType<Object>;
        /**
         *  Triggers when the control gets focus.
         * @event
         */
        focus: ej.base.EmitType<Object>;
        /**
         * Triggers when the component is created.
         * @event
         */
        created: ej.base.EmitType<Object>;
        /**
         * Triggers when the component is destroyed.
         * @event
         */
        destroyed: ej.base.EmitType<Object>;
        /**
         * Constructor for creating the widget.
         */
        constructor(options?: ej.calendars.DatePickerModel, element?: string | HTMLInputElement);
        /**
         * To Initialize the control rendering.
         * @return void
         * @private
         */
        render(): void;
        private initialize();
        private createInput();
        protected updateInput(): void;
        protected bindEvents(): void;
        private bindClearEvent();
        protected resetHandler(e?: MouseEvent): void;
        private clear(event);
        private dateIconHandler(e?);
        private CalendarKeyActionHandle(e);
        private inputFocusHandler();
        private inputBlurHandler();
        private documentHandler(e);
        protected inputKeyActionHandle(e: ej.base.KeyboardEventArgs): void;
        protected strictModeUpdate(): void;
        private createCalendar();
        private modelHeader();
        private changeTrigger();
        protected navigatedEvent(): void;
        protected changeEvent(e?: MouseEvent): void;
        protected selectCalendar(e?: Event): void;
        protected isCalendar(): boolean;
        protected setWidth(width: number | string): void;
        /**
         * Shows the Calendar.
         * @returns void
         */
        show(): void;
        /**
         * Hide the Calendar.
         * @returns void
         */
        hide(): void;
        /**
         * Sets the focus to widget for interaction.
         * @returns void
         */
        focusIn(triggerEvent?: boolean): void;
        /**
         * Remove the focus from widget, if the widget is in focus state.
         * @returns void
         */
        focusOut(): void;
        /**
         * Gets the current view of the DatePicker.
         * @returns string
         */
        currentView(): string;
        /**
         * This method used to navigate to the month/year/decade view of the DatePicker.
         * @param  {string} view - Specifies the view of the calendar.
         * @param  {Date} date - Specifies the focused date in a view.
         * @returns void
         */
        navigateTo(view: ej.calendars.CalendarView, date: Date): void;
        /**
         * To destroy the widget.
         * @returns void
         */
        destroy(): void;
        /**
         * Initialize the event handler
         * @private
         */
        protected preRender(): void;
        private checkHtmlAttributes();
        /**
         * To get component name.
         * @private
         */
        protected getModuleName(): string;
        private disabledDates();
        private setAriaAttributes();
        protected errorClass(): void;
        /**
         * Called internally if any of the property value changed.
         * returns void
         * @private
         */
        onPropertyChanged(newProp: ej.calendars.DatePickerModel, oldProp: ej.calendars.DatePickerModel): void;
    }
    export interface PopupObjectArgs {
        /** Defines the DatePicker popup element. */
        popup: ej.popups.Popup;
    }
    export interface PreventableEventArgs {
        /** Prevents the default action */
        preventDefault?: Function;
    }
    
    /**
     * Interface for a class Presets
     */
    export interface PresetsModel {
    
        /**
         * Defines the label string of the preset range. 
         */
        label?: string;
    
        /**
         * Defines the start date of the preset range 
         */
        start?: Date;
    
        /**
         * Defines the end date of the preset range 
         */
        end?: Date;
    
    }
    
    /**
     * Interface for a class DateRangePicker
     */
    export interface DateRangePickerModel extends CalendarModel{
    
        /**
         * Gets or sets the start date of the date range selection.
         * @default null
         */
        startDate?: Date;
    
        /**
         * Gets or sets the end date of the date range selection.
         * @default null
         */
        endDate?: Date;
    
        /**
         * Set the predefined ranges which let the user pick required range easily in a component.
         * @default null
         */
        presets?: PresetsModel[];
    
        /**
         * Specifies the width of the DateRangePicker component.
         * @default ''
         */
        width?: number | string;
    
        /**
         * specifies the z-index value of the dateRangePicker popup element.
         * @default 1000
         */
        zIndex?: number;
    
        /**
         * Specifies whether to show or hide the clear Icon
         * @default false
         */
        showClearButton?: boolean;
    
        /**
         *  Sets the root CSS class to the DateRangePicker which allows you to customize the appearance.
         * @default ''    
         */
        cssClass?: string;
    
        /**
         * Sets or gets the string that used between the start and end date string. 
         * @default '-'
         */
        separator?: string;
    
        /**
         *  Specifies the minimum span that can be allowed in date range selection.
         * @default null    
         */
        minDays?: number;
    
        /**
         *  Specifies the maximum span that can be allowed in a date range selection.
         * @default null
         */
        maxDays?: number;
    
        /**
         * Specifies the component to act as strict which allows entering only a valid date range in a DateRangePicker.
         * @default false
         */
        strictMode?: boolean;
    
        /**
         * Sets or gets the required date format to the start and end date string.
         * @default null
         */
        format?: string;
    
        /**
         * Specifies the component to be disabled which prevents the DateRangePicker from user interactions. 
         * @default true
         */
        enabled?: boolean;
    
        /**
         * Denies the editing the ranges in the DateRangePicker component. 
         * @default false
         */
        readonly?: boolean;
    
        /**
         * Specifies the placeholder text that need to be displayed in the DateRangePicker component.
         * @default ''
         */
        placeholder?: string;
    
        /**
         * Specifies the initial view of the Calendar when it is opened.
         * With the help of this property, initial view can be changed to year or decade view.
         * @default month
         * @hidden
         */
        start?: ej.calendars.CalendarView;
    
        /**
         * Sets the maximum level of view (month, year, decade) in the Calendar.
         * Depth view should be smaller than the start view to restrict its view navigation.
         * @default month
         * @hidden
         */
        depth?: ej.calendars.CalendarView;
    
        /**
         * Gets or sets the selected date of the Calendar
         * @default null
         * @hidden
         */
        value?: Date;
    
        /**
         * Triggers when the DateRangePicker is created
         * @event 
         */
        created?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the DateRangePicker is destroyed.
         * @event 
         */
        destroyed?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the DateRangePicker is opened.
         * @event 
         */
        open?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the DateRangePicker is closed.
         * @event 
         */
        close?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the date range value is changed in DateRangePicker.
         * @event 
         */
        change?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the Calendar is navigated to another level or within the same level of view.
         * @event
         * @hidden
         */
        navigated?: ej.base.EmitType<NavigatedEventArgs>;
    
        /**
         * Triggers on selecting the start and end date.
         * @event 
         */
        select?: ej.base.EmitType<Object>;
    
    }
    
    export class Presets extends ej.base.ChildProperty<Presets> {
        /**
         * Defines the label string of the preset range.
         */
        label: string;
        /**
         * Defines the start date of the preset range
         */
        start: Date;
        /**
         * Defines the end date of the preset range
         */
        end: Date;
    }
    export interface RangeEventArgs extends ej.base.BaseEventArgs {
        /** Defines the value string in the input element */
        value: string;
        /** Defines the start date  */
        startDate: Date;
        /** Defines the end date  */
        endDate: Date;
        /** Defines the day span between the range */
        daySpan: number;
    }
    export interface RangePopupEventArgs extends ej.base.BaseEventArgs {
        /** Defines the range string in the input element */
        date: string;
        /** Defines the DateRangePicker model */
        model: DateRangePickerModel;
        /** Defines the popup element */
        popup: ej.popups.Popup;
    }
    /**
     * Represents the DateRangePicker component that allows user to select the date range from the calendar
     * or entering the range through the input element.
     * ```html
     * <input id="daterangepicker"/>
     * ```
     * ```typescript
     * <script>
     *   var dateRangePickerObj = new DateRangePicker({ startDate: new Date("05/07/2017"), endDate: new Date("10/07/2017") });
     *   dateRangePickerObj.appendTo("#daterangepicker");
     * </script>
     * ```
     */
    export class DateRangePicker extends Calendar {
        private popupObj;
        private inputWrapper;
        private popupWrapper;
        private rightCalendar;
        private leftCalendar;
        private deviceCalendar;
        private leftCalCurrentDate;
        private rightCalCurrentDate;
        private leftCalPrevIcon;
        private leftCalNextIcon;
        private rightCalPrevIcon;
        private rightCalNextIcon;
        private inputKeyboardModule;
        protected leftKeyboardModule: ej.base.KeyboardEvents;
        protected rightKeyboardModule: ej.base.KeyboardEvents;
        private previousStartValue;
        private previousEndValue;
        private applyButton;
        private cancelButton;
        private startButton;
        private endButton;
        private cloneElement;
        private l10n;
        private isCustomRange;
        private isCustomWindow;
        private presetsItem;
        private liCollections;
        private activeIndex;
        private presetElement;
        private previousEleValue;
        private targetElement;
        private isTab;
        private disabledDayCnt;
        private angularTag;
        private inputElement;
        private modal;
        private isKeyPopup;
        private dateDisabled;
        private navNextFunction;
        private navPrevFunction;
        private deviceNavNextFunction;
        private deviceNavPrevFunction;
        private isRangeIconClicked;
        private isMaxDaysClicked;
        private popupKeyboardModule;
        private presetKeyboardModule;
        private btnKeyboardModule;
        private virtualRenderCellArgs;
        private disabledDays;
        private isMobile;
        private presetKeyConfig;
        private keyInputConfigs;
        private defaultConstant;
        /**
         * Gets or sets the start date of the date range selection.
         * @default null
         */
        startDate: Date;
        /**
         * Gets or sets the end date of the date range selection.
         * @default null
         */
        endDate: Date;
        /**
         * Set the predefined ranges which let the user pick required range easily in a component.
         * @default null
         */
        presets: PresetsModel[];
        /**
         * Specifies the width of the DateRangePicker component.
         * @default ''
         */
        width: number | string;
        /**
         * specifies the z-index value of the dateRangePicker popup element.
         * @default 1000
         */
        zIndex: number;
        /**
         * Specifies whether to show or hide the clear Icon
         * @default false
         */
        showClearButton: boolean;
        /**
         *  Sets the root CSS class to the DateRangePicker which allows you to customize the appearance.
         * @default ''
         */
        cssClass: string;
        /**
         * Sets or gets the string that used between the start and end date string.
         * @default '-'
         */
        separator: string;
        /**
         *  Specifies the minimum span that can be allowed in date range selection.
         * @default null
         */
        minDays: number;
        /**
         *  Specifies the maximum span that can be allowed in a date range selection.
         * @default null
         */
        maxDays: number;
        /**
         * Specifies the component to act as strict which allows entering only a valid date range in a DateRangePicker.
         * @default false
         */
        strictMode: boolean;
        /**
         * Sets or gets the required date format to the start and end date string.
         * @default null
         */
        format: string;
        /**
         * Specifies the component to be disabled which prevents the DateRangePicker from user interactions.
         * @default true
         */
        enabled: boolean;
        /**
         * Denies the editing the ranges in the DateRangePicker component.
         * @default false
         */
        readonly: boolean;
        /**
         * Specifies the placeholder text that need to be displayed in the DateRangePicker component.
         * @default ''
         */
        placeholder: string;
        /**
         * Specifies the initial view of the Calendar when it is opened.
         * With the help of this property, initial view can be changed to year or decade view.
         * @default month
         * @hidden
         */
        start: ej.calendars.CalendarView;
        /**
         * Sets the maximum level of view (month, year, decade) in the Calendar.
         * Depth view should be smaller than the start view to restrict its view navigation.
         * @default month
         * @hidden
         */
        depth: ej.calendars.CalendarView;
        /**
         * Gets or sets the selected date of the Calendar
         * @default null
         * @hidden
         */
        value: Date;
        /**
         * Triggers when the DateRangePicker is created
         * @event
         */
        created: ej.base.EmitType<Object>;
        /**
         * Triggers when the DateRangePicker is destroyed.
         * @event
         */
        destroyed: ej.base.EmitType<Object>;
        /**
         * Triggers when the DateRangePicker is opened.
         * @event
         */
        open: ej.base.EmitType<Object>;
        /**
         * Triggers when the DateRangePicker is closed.
         * @event
         */
        close: ej.base.EmitType<Object>;
        /**
         * Triggers when the date range value is changed in DateRangePicker.
         * @event
         */
        change: ej.base.EmitType<Object>;
        /**
         * Triggers when the Calendar is navigated to another level or within the same level of view.
         * @event
         * @hidden
         */
        navigated: ej.base.EmitType<NavigatedEventArgs>;
        /**
         * Triggers on selecting the start and end date.
         * @event
         */
        select: ej.base.EmitType<Object>;
        /**
         * Constructor for creating the widget
         */
        constructor(options?: DateRangePickerModel, element?: string | HTMLInputElement);
        /**
         * To Initialize the control rendering.
         * @return void
         * @private
         */
        protected render(): void;
        /**
         * Initialize the event handler
         * @returns void
         * @private
         */
        protected preRender(): void;
        private initProperty();
        private initialize();
        private processPresets();
        protected bindEvents(): void;
        private bindClearEvent();
        protected resetHandler(e: MouseEvent): void;
        private clear();
        private rangeIconHandler(e);
        private checkHtmlAttributes();
        private createPopup();
        private renderControl();
        private clearCalendarEvents();
        private updateNavIcons();
        private calendarIconEvent();
        private bindCalendarEvents();
        private calendarIconRipple();
        private deviceCalendarEvent();
        private deviceNavNext(e);
        private deviceNavPrevious(e);
        private updateDeviceCalendar(calendar);
        private deviceHeaderClick(event);
        private inputBlurHandler();
        private clearRange();
        private errorClass();
        private keyCalendarUpdate(isLeftCalendar, ele);
        private navInCalendar(e, isLeftCalendar, leftLimit, rightLimit, ele);
        private keyInputHandler(e);
        private keyNavigation(calendar, e);
        private inputHandler(e);
        private bindCalendarCellEvents(calendar?);
        private removeFocusedDate();
        private hoverSelection(event, element);
        private updateRange(elementCollection);
        private checkMinMaxDays();
        private rangeArgs();
        private otherMonthSelect(ele, isStartDate, sameDate?);
        private selectRange(event, element);
        private selectableDates();
        private updateMinMaxDays(calendar);
        private removeClassDisabled();
        private updateHeader();
        private removeSelection();
        private addSelectedAttributes(ele, date, isStartDate, sameDate?);
        private removeSelectedAttributes();
        private updateCalendarElement(calendar);
        private navPreviousMonth(event);
        private deviceNavigation(ele?);
        private updateControl(calendar);
        private navNextMonth(event);
        private compareMonths(start, end);
        private isPopupOpen();
        private createHeader();
        private disableInput();
        private validateMinMax();
        private validateDates();
        private validateRangeStrict();
        private validateRange();
        private validateMinMaxDays();
        private renderCalendar();
        private isSameMonth(start, end);
        private selectNextMonth();
        private selectStartMonth();
        private createCalendar();
        private createControl();
        private cancelFunction(eve?);
        private deviceHeaderUpdate();
        private applyFunction(eve?);
        private onMouseClick(event, item?);
        private onMouseOver(event);
        private onMouseLeave(event);
        private setListSelection(li, event);
        private removeListSelection();
        private applyPresetRange(values);
        private renderCustomPopup();
        private listRippleEffect();
        private createPresets();
        private wireListEvents();
        private unWireListEvents();
        private renderPopup();
        private calendarFocus();
        private presetHeight();
        private presetKeyActionHandler(e);
        private listMoveDown(e);
        private listMoveUp(e);
        private getHoverLI();
        private getActiveLI();
        private popupKeyBoardHandler();
        private setScrollPosition();
        private popupKeyActionHandle(e);
        private iconBlurHandler();
        private removeIconTabindex();
        private addIconTabindex();
        private documentHandler(e);
        private createInput();
        private refreshControl();
        private updateInput();
        private isDateDisabled(date);
        private disabledDateRender();
        private virtualRenderCellEvent(args);
        private disabledDates();
        private changeTrigger();
        /**
         * This method is used to navigate to the month/year/decade view of the Calendar.
         * @param  {string} view - Specifies the view of the Calendar.
         * @param  {Date} date - Specifies the focused date in a view.
         * @returns void
         * @hidden
         */
        navigateTo(view: ej.calendars.CalendarView, value: Date): void;
        /**
         * To destroy the widget.
         * @returns void
         */
        destroy(): void;
        /**
         * To get component name
         * @returns string
         * @private
         */
        protected getModuleName(): string;
        /**
         * Return the properties that are maintained upon browser refresh.
         * @returns string
         */
        getPersistData(): string;
        /**
         * Return the selected range and day span in the DateRangePicker.
         * @returns Object
         */
        getSelectedRange(): Object;
        /**
         * To open the Popup container in the DateRangePicker component.
         * @returns void
         */
        show(element?: HTMLElement): void;
        /**
         * To close the Popup container in the DateRangePicker component.
         * @returns void
         */
        hide(): void;
        private setLocale();
        /**
         * Called internally if any of the property value changed.
         * returns void
         * @private
         */
        onPropertyChanged(newProp: DateRangePickerModel, oldProp: DateRangePickerModel): void;
    }
    
    /**
     * Interface for a class DateTimePicker
     */
    export interface DateTimePickerModel extends DatePickerModel{
    
        /**
         * Specifies the format of the time value that to be displayed in time popup list.
         * @default null
         */
        timeFormat?: string;
    
        /**
         * Specifies the time interval between the two adjacent time values in the time popup list . 
         * @default 30
         */
        step?: number;
    
        /**
         * specifies the z-index value of the popup element.
         * @default 1000
         */
        zIndex?: number;
    
        /**
         * Specifies whether to show or hide the clear icon in textbox.
         * @default false
         */
        showClearButton?: boolean;
    
        /**
         * Specifies the placeholder text that to be is displayed in textbox.
         * @default null
         */
        placeholder?: string;
    
        /**
         * Specifies the component whether to act as strict or not. So that, it allows to enter only a valid datetime value
         * or else it will resets to previous value. By default, strictMode is in false.
         * @default false
         */
        strictMode?: boolean;
    
        /**
         * Triggers when popup is opened.
         * @event 
         */
        open?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when popup is closed.
         * @event 
         */
        close?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when input loses the focus.
         * @event 
         */
        blur?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when input gets focus.
         * @event 
         */
        focus?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when DateTimePicker is created.
         * @event 
         */
        created?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when DateTimePicker is destroyed.
         * @event 
         */
        destroyed?: ej.base.EmitType<Object>;
    
    }
    
    /**
     * Represents the DateTimePicker component that allows user to select
     * or enter a date time value.
     * ```html
     * <input id="dateTimePicker"/>
     * ````
     * ````typescript
     * <script>
     *   var dateTimePickerObject = new DateTimePicker({ value: new Date() });
     *   dateTimePickerObject.appendTo("#dateTimePicker");
     * </script>
     * ```
     */
    export class DateTimePicker extends DatePicker {
        private timeIcon;
        private cloneElement;
        private dateTimeWrapper;
        private rippleFn;
        private listWrapper;
        private liCollections;
        private timeCollections;
        private listTag;
        private selectedElement;
        private containerStyle;
        private popupObject;
        protected timeModal: HTMLElement;
        private isNavigate;
        private timeValue;
        protected l10n: ej.base.L10n;
        private keyboardHandler;
        protected inputEvent: ej.base.KeyboardEvents;
        private activeIndex;
        private valueWithMinutes;
        private previousDateTime;
        private initValue;
        private isValidState;
        protected timekeyConfigure: {
            [key: string]: string;
        };
        /**
         * Specifies the format of the time value that to be displayed in time popup list.
         * @default null
         */
        timeFormat: string;
        /**
         * Specifies the time interval between the two adjacent time values in the time popup list .
         * @default 30
         */
        step: number;
        /**
         * specifies the z-index value of the popup element.
         * @default 1000
         */
        zIndex: number;
        /**
         * Specifies whether to show or hide the clear icon in textbox.
         * @default false
         */
        showClearButton: boolean;
        /**
         * Specifies the placeholder text that to be is displayed in textbox.
         * @default null
         */
        placeholder: string;
        /**
         * Specifies the component whether to act as strict or not. So that, it allows to enter only a valid datetime value
         * or else it will resets to previous value. By default, strictMode is in false.
         * @default false
         */
        strictMode: boolean;
        /**
         * Triggers when popup is opened.
         * @event
         */
        open: ej.base.EmitType<Object>;
        /**
         * Triggers when popup is closed.
         * @event
         */
        close: ej.base.EmitType<Object>;
        /**
         * Triggers when input loses the focus.
         * @event
         */
        blur: ej.base.EmitType<Object>;
        /**
         * Triggers when input gets focus.
         * @event
         */
        focus: ej.base.EmitType<Object>;
        /**
         * Triggers when DateTimePicker is created.
         * @event
         */
        created: ej.base.EmitType<Object>;
        /**
         * Triggers when DateTimePicker is destroyed.
         * @event
         */
        destroyed: ej.base.EmitType<Object>;
        /**
         * Constructor for creating the widget
         */
        constructor(options?: DateTimePickerModel, element?: string | HTMLInputElement);
        /**
         * Sets the focus to widget for interaction.
         * @returns void
         */
        focusIn(): void;
        /**
         * Remove the focus from widget, if the widget is in focus state.
         * @returns void
         */
        focusOut(): void;
        /**
         * To destroy the widget.
         * @returns void
         */
        destroy(): void;
        /**
         * To Initialize the control rendering.
         * @return void
         * @private
         */
        render(): void;
        private setValue();
        private validateMinMaxRange(value);
        private checkValidState(value);
        private checkErrorState();
        private validateValue(value);
        private disablePopupButton(isDisable);
        private getFormattedValue(value);
        private isDateObject(value);
        private createInputElement();
        private renderTimeIcon();
        private bindInputEvents();
        private unBindInputEvents();
        private cldrTimeFormat();
        private cldrDateTimeFormat();
        private getCldrFormat(type);
        private isNullOrEmpty(value);
        protected getCultureTimeObject(ld: Object, c: string): Object;
        private timeHandler(e?);
        private dateHandler(e?);
        show(type?: string): void;
        toggle(): void;
        private listCreation();
        private popupCreation(type);
        private openPopup();
        private documentClickHandler(event);
        private isTimePopupOpen();
        private isDatePopupOpen();
        private renderPopup();
        private setPopupWidth();
        protected wireTimeListEvents(): void;
        protected unWireTimeListEvents(): void;
        private onMouseOver(event);
        private onMouseLeave();
        private setTimeHover(li, className);
        protected getPopupHeight(): number;
        protected changeEvent(): void;
        private updateValue();
        private setTimeScrollPosition();
        private setInputValue(type);
        private getFullDateTime();
        private combineDateTime(value);
        private onMouseClick(event);
        private setSelection(li, event);
        private setTimeActiveClass();
        private setTimeActiveDescendant();
        protected addTimeSelection(): void;
        protected removeTimeSelection(): void;
        protected removeTimeHover(className: string): void;
        protected getTimeHoverItem(className: string): Element[];
        protected isValidLI(li: Element | HTMLElement): boolean;
        private calculateStartEnd(value, range, method);
        private startTime(date);
        private endTime(date);
        hide(): void;
        private closePopup();
        protected preRender(): void;
        protected getProperty(date: DateTimePickerModel, val: string): void;
        protected checkAttributes(): void;
        private getTimeActiveElement();
        protected createDateObj(val: Date | string): Date;
        private getDateObject(text);
        protected findNextTimeElement(event: ej.base.KeyboardEventArgs): void;
        protected setTimeValue(date: Date, value: Date): Date;
        protected timeElementValue(value: Date): Date;
        protected timeKeyHandler(event: ej.base.KeyboardEventArgs): void;
        protected TimeKeyActionHandle(event: ej.base.KeyboardEventArgs): void;
        protected inputKeyAction(event: ej.base.KeyboardEventArgs): void;
        onPropertyChanged(newProp: DateTimePickerModel, oldProp: DateTimePickerModel): void;
        /**
         * To get component name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    /**
     * Interface for a class TimePicker
     */
    export interface TimePickerModel extends ej.base.ComponentModel{
    
        /**
         * Gets or sets the width of the TimePicker component. The width of the popup is based on the width of the component.
         * @default '100%'
         */
        width?: string | number;
    
        /**
         * Specifies the root CSS class of the TimePicker that allows to
         * customize the appearance by overriding the styles.
         * @default null
         */
        cssClass?: string;
    
        /**
         * Specifies the component to act as strict so that, it allows to enter only a valid time value within a specified range or else 
         * resets to previous value. By default, strictMode is in false.
         * @default false
         */
        strictMode?: boolean;
    
        /**
         * Specifies the format of value that is to be displayed in component. By default, the format is
         * based on the culture. 
         * @default null
         */
        format?: string;
    
        /**
         * Specifies whether the component to be disabled or not.
         * @default true
         */
        enabled?: boolean;
    
        /**
         * Specifies the component in readonly state. 
         * @default false
         */
        readonly?: boolean;
    
        /**
         * Specifies the placeholder text to be floated.
         */
        floatLabelType?: ej.inputs.FloatLabelType;
    
        /**
         * Specifies the placeholder text that is displayed in textbox.
         * @default null
         */
        placeholder?: string;
    
        /**
         * specifies the z-index value of the timePicker popup element.
         * @default 1000
         */
        zIndex?: number;
    
        /**
         * Specifies whether to show or hide the clear Icon
         * @default false
         */
        showClearButton?: boolean;
    
        /**
         * Specifies the time interval between the two adjacent time values in the popup list . 
         * @default 30
         */
        step?: number;
    
        /**
         * Specifies the scroll bar position if there is no value is selected in the popup list or
         *  the given value is not present in the popup list. 
         * @default null
         */
        scrollTo?: Date;
    
        /**
         * Gets or sets the value of the component. The value is parsed based on the format. 
         * @default null
         */
        value?: Date;
    
        /**
         * Gets or sets the minimum time value that can be allowed to select in TimePicker.
         * @default 00:00
         */
        min?: Date;
    
        /**
         * Gets or sets the maximum time value that can be allowed to select in TimePicker.
         * @default 00:00
         */
        max?: Date;
    
        /**
         * Specifies the component to be rendered in right-to-left direction.
         * @default false
         */
        enableRtl?: boolean;
    
        /**
         * Triggers when the value is changed.
         * @event  
         */
        change?: ej.base.EmitType<ChangeEventArgs>;
    
        /**
         * Triggers when the component is created.
         * @event
         */
        created?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the component is destroyed.
         * @event
         */
        destroyed?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the popup is opened.
         * @event
         */
        open?: ej.base.EmitType<PopupEventArgs>;
    
        /**
         * Triggers while rendering the each popup list item.
         * @event
         */
        itemRender?: ej.base.EmitType<ItemEventArgs>;
    
        /**
         * Triggers when the popup is closed.
         * @event
         */
        close?: ej.base.EmitType<PopupEventArgs>;
    
        /**
         * Triggers when the control loses the focus.
         * @event
         */
        blur?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the control gets focused.
         * @event
         */
        focus?: ej.base.EmitType<Object>;
    
    }
    
    export interface ChangeEventArgs {
        /** Defines the boolean that returns true when the value is changed by user interaction, otherwise returns false. */
        isInteracted: boolean;
        /** Defines the selected time value of the TimePicker. */
        value: Date;
        /** Defines the selected time value as string. */
        text: string;
        /** Defines the type of the event. */
        e: ej.base.KeyboardEventArgs | FocusEvent | MouseEvent;
    }
    /**
     * Interface for before list item render .
     * @private
     */
    export interface ItemEventArgs extends ej.base.BaseEventArgs {
        /** Defines the created LI element. */
        element: HTMLElement;
        /** Defines the displayed text value in a popup list. */
        text: string;
        /** Defines the Date object of displayed text in a popup list. */
        value: Date;
        /** Specifies whether to disable the current time value or not. */
        isDisabled: Boolean;
    }
    export interface CursorPositionDetails {
        /** Defines the text selection starting position. */
        start: number;
        /** Defines the text selection end position. */
        end: number;
    }
    export interface MeridianText {
        /** Defines the culture specific meridian text for AM. */
        am: string;
        /** Defines the culture specific meridian text for PM. */
        pm: string;
    }
    export interface PopupEventArgs {
        /** Defines the TimePicker popup element. */
        popup: ej.popups.Popup;
    }
    export namespace TimePickerBase {
        function createListItems(min: Date, max: Date, globalize: ej.base.Internationalization, timeFormat: string, step: number): {
            collection: number[];
            list: HTMLElement;
        };
    }
    /**
     * TimePicker is an intuitive interface component which provides an options to select a time value
     * from popup list or to set a desired time value.
     * ```
     * <input id='timepicker' type='text'/>
     * <script>
     *   var timePickerObj = new TimePicker({ value: new Date() });
     *   timePickerObj.appendTo('#timepicker');
     * </script>
     * ```
     */
    export class TimePicker extends ej.base.Component<HTMLElement> implements ej.inputs.IInput {
        private inputWrapper;
        private popupWrapper;
        private cloneElement;
        private listWrapper;
        private listTag;
        private selectedElement;
        private liCollections;
        protected inputElement: HTMLInputElement;
        private popupObj;
        protected inputEvent: ej.base.KeyboardEvents;
        protected globalize: ej.base.Internationalization;
        private defaultCulture;
        private containerStyle;
        private rippleFn;
        private l10n;
        private cursorDetails;
        private activeIndex;
        private timeCollections;
        private isNavigate;
        private disableItemCollection;
        protected isPreventBlur: boolean;
        private isTextSelected;
        private prevValue;
        private inputStyle;
        private angularTag;
        private valueWithMinutes;
        private prevDate;
        private initValue;
        private initMin;
        private initMax;
        private isIconClicked;
        protected keyConfigure: {
            [key: string]: string;
        };
        /**
         * Gets or sets the width of the TimePicker component. The width of the popup is based on the width of the component.
         * @default '100%'
         */
        width: string | number;
        /**
         * Specifies the root CSS class of the TimePicker that allows to
         * customize the appearance by overriding the styles.
         * @default null
         */
        cssClass: string;
        /**
         * Specifies the component to act as strict so that, it allows to enter only a valid time value within a specified range or else
         * resets to previous value. By default, strictMode is in false.
         * @default false
         */
        strictMode: boolean;
        /**
         * Specifies the format of value that is to be displayed in component. By default, the format is
         * based on the culture.
         * @default null
         */
        format: string;
        /**
         * Specifies whether the component to be disabled or not.
         * @default true
         */
        enabled: boolean;
        /**
         * Specifies the component in readonly state.
         * @default false
         */
        readonly: boolean;
        /**
         * Specifies the placeholder text to be floated.
         */
        floatLabelType: ej.inputs.FloatLabelType;
        /**
         * Specifies the placeholder text that is displayed in textbox.
         * @default null
         */
        placeholder: string;
        /**
         * specifies the z-index value of the timePicker popup element.
         * @default 1000
         */
        zIndex: number;
        /**
         * Specifies whether to show or hide the clear Icon
         * @default false
         */
        showClearButton: boolean;
        /**
         * Specifies the time interval between the two adjacent time values in the popup list .
         * @default 30
         */
        step: number;
        /**
         * Specifies the scroll bar position if there is no value is selected in the popup list or
         *  the given value is not present in the popup list.
         * @default null
         */
        scrollTo: Date;
        /**
         * Gets or sets the value of the component. The value is parsed based on the format.
         * @default null
         */
        value: Date;
        /**
         * Gets or sets the minimum time value that can be allowed to select in TimePicker.
         * @default 00:00
         */
        min: Date;
        /**
         * Gets or sets the maximum time value that can be allowed to select in TimePicker.
         * @default 00:00
         */
        max: Date;
        /**
         * Specifies the component to be rendered in right-to-left direction.
         * @default false
         */
        enableRtl: boolean;
        /**
         * Triggers when the value is changed.
         * @event
         */
        change: ej.base.EmitType<ChangeEventArgs>;
        /**
         * Triggers when the component is created.
         * @event
         */
        created: ej.base.EmitType<Object>;
        /**
         * Triggers when the component is destroyed.
         * @event
         */
        destroyed: ej.base.EmitType<Object>;
        /**
         * Triggers when the popup is opened.
         * @event
         */
        open: ej.base.EmitType<PopupEventArgs>;
        /**
         * Triggers while rendering the each popup list item.
         * @event
         */
        itemRender: ej.base.EmitType<ItemEventArgs>;
        /**
         * Triggers when the popup is closed.
         * @event
         */
        close: ej.base.EmitType<PopupEventArgs>;
        /**
         * Triggers when the control loses the focus.
         * @event
         */
        blur: ej.base.EmitType<Object>;
        /**
         * Triggers when the control gets focused.
         * @event
         */
        focus: ej.base.EmitType<Object>;
        /**
         * Constructor for creating the widget
         */
        constructor(options?: TimePickerModel, element?: string | HTMLInputElement);
        /**
         * Initialize the event handler
         * @private
         */
        protected preRender(): void;
        protected render(): void;
        private validateDisable();
        private initialize();
        private checkDateValue(value);
        private createInputElement();
        destroy(): void;
        private popupCreation();
        protected getPopupHeight(): number;
        private generateList();
        private popupCalculation();
        private isEmptyList();
        private renderPopup();
        private getFormattedValue(value);
        private getDateObject(text);
        private checkErrorState(val);
        private validateInterval();
        private disableTimeIcon();
        private disableElement(element);
        private enableElement(element);
        private selectInputText();
        private getMeridianText();
        private getCursorSelection();
        private getActiveElement();
        private isNullOrEmpty(value);
        private setWidth();
        private setScrollPosition();
        private findScrollTop(element);
        private setScrollTo();
        private getText();
        private getValue(value);
        private cldrDateFormat();
        private cldrTimeFormat();
        private dateToNumeric();
        private getExactDateTime(value);
        private setValue(value);
        private updatePlaceHolder();
        private popupHandler(e);
        private mouseDownHandler();
        private mouseUpHandler(event);
        private focusSelection();
        private inputHandler(event);
        private onMouseClick(event);
        private closePopup(delay?);
        private checkValueChange(event, isNavigation);
        private onMouseOver(event);
        private setHover(li, className);
        private setSelection(li, event);
        private onMouseLeave();
        private scrollHandler();
        private setMinMax(minVal, maxVal);
        protected validateMinMax(dateVal: Date | string, minVal: Date, maxVal: Date): Date | string;
        private valueIsDisable(value);
        protected validateState(val: string | Date): boolean;
        protected strictOperation(minimum: Date, maximum: Date, dateVal: Date | string, val: Date): Date | string;
        protected bindEvents(): void;
        protected unBindEvents(): void;
        private bindClearEvent();
        protected clearHandler(e: MouseEvent): void;
        private clear(event);
        protected setZIndex(): void;
        protected checkAttributes(): void;
        protected setCurrentDate(value: Date): Date;
        protected getSeparator(): string;
        protected getTextFormat(): number;
        protected updateValue(value: string | Date, event: ej.base.KeyboardEventArgs | FocusEvent): void;
        protected previousState(date: Date): string;
        protected resetState(): void;
        protected objToString(val: Date): string;
        protected checkValue(value: string | Date): string;
        protected validateValue(date: Date, value: string | Date): string;
        protected findNextElement(event: ej.base.KeyboardEventArgs): void;
        protected elementValue(value: Date): void;
        private validLiElement(index, backward?);
        protected keyHandler(event: ej.base.KeyboardEventArgs): void;
        protected setPopupPosition(): number;
        protected getCultureTimeObject(ld: Object, c: string): Object;
        protected getCultureDateObject(ld: Object, c: string): Object;
        protected wireListEvents(): void;
        protected unWireListEvents(): void;
        protected valueProcess(event: ej.base.KeyboardEventArgs | FocusEvent | MouseEvent, value: Date): void;
        protected changeEvent(event: ej.base.KeyboardEventArgs | FocusEvent | MouseEvent): void;
        protected updateInput(isUpdate: boolean, date: Date): void;
        protected setActiveDescendant(): void;
        protected removeSelection(): void;
        protected removeHover(className: string): void;
        protected getHoverItem(className: string): Element[];
        private setActiveClass();
        protected addSelection(): void;
        protected isValidLI(li: Element | HTMLElement): boolean;
        protected createDateObj(val: Date | string): Date;
        protected TimeParse(today: string, val: Date | string): Date;
        protected createListItems(): void;
        private documentClickHandler(event);
        protected setEnableRtl(): void;
        protected setEnable(): void;
        protected getProperty(date: TimePickerModel, val: string): void;
        /**
         * Focuses out the TimePicker textbox element.
         * @returns void
         */
        focusOut(): void;
        private isPopupOpen();
        /**
         * Focused the TimePicker textbox element.
         * @returns void
         */
        focusIn(e?: MouseEvent): void;
        /**
         * Hides the TimePicker popup.
         * @returns void
         */
        hide(): void;
        /**
         * Opens the popup to show the list items.
         * @returns void
         */
        show(): void;
        /**
         * Gets the properties to be maintained upon browser refresh.
         * @returns string
         */
        getPersistData(): string;
        /**
         * To get component name
         * @private
         */
        protected getModuleName(): string;
        /**
         * Called internally if any of the property value changed.
         * returns void
         * @private
         */
        onPropertyChanged(newProp: TimePickerModel, oldProp: TimePickerModel): void;
    }
    
  }
  export namespace charts {
    
    /**
     * Interface for a class AccumulationChart
     */
    export interface AccumulationChartModel extends ej.base.ComponentModel{
    
        /**
         * The width of the chart as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, chart will render to the full width of its parent element.
         * @default null
         */
    
        width?: string;
    
        /**
         * The height of the chart as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, chart will render to the full height of its parent element.
         * @default null
         */
    
        height?: string;
    
        /**
         * Title for accumulation chart
         * @default null
         */
        title?: string;
    
        /**
         * Options for customizing the `title` of accumulation chart.
         */
    
        titleStyle?: FontModel;
    
        /**
         * Options for customizing the legend of accumulation chart.
         */
        legendSettings?: LegendSettingsModel;
    
        /**
         * Options for customizing the tooltip of accumulation chart.
         */
    
        tooltip?:  TooltipSettingsModel;
    
        /**
         * Specifies whether point has to get selected or not. Takes value either 'None 'or 'Point'
         * @default None
         */
        selectionMode?: AccumulationSelectionMode;
    
        /**
         * If set true, enables the multi selection in accumulation chart. It requires `selectionMode` to be `Point`.
         * @default false
         */
        isMultiSelect?: boolean;
    
        /**
         * Specifies the point indexes to be selected while loading a accumulation chart.
         * It requires `selectionMode` to be `Point`.
         * ```html
         * <div id='Pie'></div>
         * ```
         * ```typescript
         * let pie: AccumulationChart = new AccumulationChart({
         * ...
         *   selectionMode: 'Point',
         *   selectedDataIndexes: [ { series: 0, point: 1},
         *                          { series: 2, point: 3} ],
         * ...
         * });
         * pie.appendTo('#Pie');
         * ```
         * @default []
         */
        selectedDataIndexes?: IndexesModel[];
    
        /**
         *  Options to customize the left, right, top and bottom margins of accumulation chart.
         */
    
        margin?: MarginModel;
    
        /**
         * If set true, labels for the point will be placed smartly without overlapping.
         * @default true
         */
        enableSmartLabels?: boolean;
    
        /**
         * Options for customizing the color and width of the chart border.
         */
    
        border?: BorderModel;
    
        /**
         * The background color of the chart, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
        background?: string;
    
        /**
         * The configuration for series in accumulation chart.
         */
    
        series?: AccumulationSeriesModel[];
    
        /**
         * The configuration for annotation in chart.
         */
    
        annotations?: AccumulationAnnotationSettingsModel[];
    
        /**
         * Specifies the theme for accumulation chart.
         * @default 'Material'
         */
        theme?: AccumulationTheme;
    
        /**
         * Triggers after accumulation chart loaded.
         * @event
         */
        loaded?: ej.base.EmitType<IAccLoadedEventArgs>;
    
        /**
         * Triggers before accumulation chart load.
         * @event
         */
        load?: ej.base.EmitType<IAccLoadedEventArgs>;
    
        /**
         * Triggers before the series gets rendered.
         * @event
         */
        seriesRender?: ej.base.EmitType<IAccSeriesRenderEventArgs>;
    
        /**
         * Triggers before the legend gets rendered.
         * @event
         */
        legendRender?: ej.base.EmitType<ILegendRenderEventArgs>;
    
        /**
         * Triggers before the data label for series gets rendered.
         * @event
         */
        textRender?: ej.base.EmitType<IAccTextRenderEventArgs>;
    
        /**
         * Triggers before the tooltip for series gets rendered.
         * @event
         */
        tooltipRender?: ej.base.EmitType<IAccTooltipRenderEventArgs>;
    
        /**
         * Triggers before each points for series gets rendered.
         * @event
         */
    
        pointRender?: ej.base.EmitType<IAccPointRenderEventArgs>;
    
        /**
         * Triggers before the annotation gets rendered.
         * @event
         */
    
        annotationRender?: ej.base.EmitType<IAnnotationRenderEventArgs>;
    
        /**
         * Triggers before the prints gets started.
         * @event
         */
    
        beforePrint?: ej.base.EmitType<IPrintEventArgs>;
    
        /**
         * Triggers on hovering the accumulation chart.
         * @event
         */
    
        chartMouseMove?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on clicking the accumulation chart.
         * @event
         */
    
        chartMouseClick?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on point click.
         * @event
         */
    
        pointClick?: ej.base.EmitType<IPointEventArgs>;
    
        /**
         * Triggers on point move.
         * @event
         */
    
        pointMove?: ej.base.EmitType<IPointEventArgs>;
    
        /**
         * Triggers after animation gets completed for series.
         * @event
         */
        animationComplete?: ej.base.EmitType<IAccAnimationCompleteEventArgs>;
    
        /**
         * Triggers on mouse down.
         * @event
         */
    
        chartMouseDown?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers while cursor leaves the accumulation chart.
         * @event
         */
    
        chartMouseLeave?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on mouse up.
         * @event
         */
    
        chartMouseUp?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers after window resize.
         * @event
         */
    
        resized?: ej.base.EmitType<IAccResizeEventArgs>;
    
        /**
         * Defines the currencyCode format of the accumulation chart
         * @private
         */
    
        currencyCode?: string;
    
    }
    /**
     * AccumulationChart file
     */
    
    /**
     * Represents the AccumulationChart control.
     * ```html
     * <div id="accumulation"/>
     * <script>
     *   var accObj = new AccumulationChart({ });
     *   accObj.appendTo("#accumulation");
     * </script>
     * ```
     */
    export class AccumulationChart extends ej.base.Component<HTMLElement> implements ej.base.INotifyPropertyChanged {
        /**
         * `accBaseModue` is used to define the common functionalities of accumulation series
         * @private
         */
        accBaseModule: AccumulationBase;
        /**
         * `pieSeriesModule` is used to render pie series.
         * @private
         */
        pieSeriesModule: PieSeries;
        /**
         * `funnelSeriesModule` is used to render funnel series.
         * @private
         */
        funnelSeriesModule: FunnelSeries;
        /**
         * `pyramidSeriesModule` is used to render funnel series.
         * @private
         */
        pyramidSeriesModule: PyramidSeries;
        /**
         * `accumulationLegendModule` is used to manipulate and add legend in accumulation chart.
         */
        accumulationLegendModule: AccumulationLegend;
        /**
         * `accumulationDataLabelModule` is used to manipulate and add dataLabel in accumulation chart.
         */
        accumulationDataLabelModule: AccumulationDataLabel;
        /**
         * `accumulationTooltipModule` is used to manipulate and add tooltip in accumulation chart.
         */
        accumulationTooltipModule: AccumulationTooltip;
        /**
         * `accumulationSelectionModule` is used to manipulate and add selection in accumulation chart.
         */
        accumulationSelectionModule: AccumulationSelection;
        /**
         * `annotationModule` is used to manipulate and add annotation in chart.
         */
        annotationModule: AccumulationAnnotation;
        /**
         * The width of the chart as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, chart will render to the full width of its parent element.
         * @default null
         */
        width: string;
        /**
         * The height of the chart as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, chart will render to the full height of its parent element.
         * @default null
         */
        height: string;
        /**
         * Title for accumulation chart
         * @default null
         */
        title: string;
        /**
         * Options for customizing the `title` of accumulation chart.
         */
        titleStyle: FontModel;
        /**
         * Options for customizing the legend of accumulation chart.
         */
        legendSettings: LegendSettingsModel;
        /**
         * Options for customizing the tooltip of accumulation chart.
         */
        tooltip: TooltipSettingsModel;
        /**
         * Specifies whether point has to get selected or not. Takes value either 'None 'or 'Point'
         * @default None
         */
        selectionMode: AccumulationSelectionMode;
        /**
         * If set true, enables the multi selection in accumulation chart. It requires `selectionMode` to be `Point`.
         * @default false
         */
        isMultiSelect: boolean;
        /**
         * Specifies the point indexes to be selected while loading a accumulation chart.
         * It requires `selectionMode` to be `Point`.
         * ```html
         * <div id='Pie'></div>
         * ```
         * ```typescript
         * let pie: AccumulationChart = new AccumulationChart({
         * ...
         *   selectionMode: 'Point',
         *   selectedDataIndexes: [ { series: 0, point: 1},
         *                          { series: 2, point: 3} ],
         * ...
         * });
         * pie.appendTo('#Pie');
         * ```
         * @default []
         */
        selectedDataIndexes: IndexesModel[];
        /**
         *  Options to customize the left, right, top and bottom margins of accumulation chart.
         */
        margin: MarginModel;
        /**
         * If set true, labels for the point will be placed smartly without overlapping.
         * @default true
         */
        enableSmartLabels: boolean;
        /**
         * Options for customizing the color and width of the chart border.
         */
        border: BorderModel;
        /**
         * The background color of the chart, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
        background: string;
        /**
         * The configuration for series in accumulation chart.
         */
        series: AccumulationSeriesModel[];
        /**
         * The configuration for annotation in chart.
         */
        annotations: AccumulationAnnotationSettingsModel[];
        /**
         * Specifies the theme for accumulation chart.
         * @default 'Material'
         */
        theme: AccumulationTheme;
        /**
         * Triggers after accumulation chart loaded.
         * @event
         */
        loaded: ej.base.EmitType<IAccLoadedEventArgs>;
        /**
         * Triggers before accumulation chart load.
         * @event
         */
        load: ej.base.EmitType<IAccLoadedEventArgs>;
        /**
         * Triggers before the series gets rendered.
         * @event
         */
        seriesRender: ej.base.EmitType<IAccSeriesRenderEventArgs>;
        /**
         * Triggers before the legend gets rendered.
         * @event
         */
        legendRender: ej.base.EmitType<ILegendRenderEventArgs>;
        /**
         * Triggers before the data label for series gets rendered.
         * @event
         */
        textRender: ej.base.EmitType<IAccTextRenderEventArgs>;
        /**
         * Triggers before the tooltip for series gets rendered.
         * @event
         */
        tooltipRender: ej.base.EmitType<IAccTooltipRenderEventArgs>;
        /**
         * Triggers before each points for series gets rendered.
         * @event
         */
        pointRender: ej.base.EmitType<IAccPointRenderEventArgs>;
        /**
         * Triggers before the annotation gets rendered.
         * @event
         */
        annotationRender: ej.base.EmitType<IAnnotationRenderEventArgs>;
        /**
         * Triggers before the prints gets started.
         * @event
         */
        beforePrint: ej.base.EmitType<IPrintEventArgs>;
        /**
         * Triggers on hovering the accumulation chart.
         * @event
         */
        chartMouseMove: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on clicking the accumulation chart.
         * @event
         */
        chartMouseClick: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on point click.
         * @event
         */
        pointClick: ej.base.EmitType<IPointEventArgs>;
        /**
         * Triggers on point move.
         * @event
         */
        pointMove: ej.base.EmitType<IPointEventArgs>;
        /**
         * Triggers after animation gets completed for series.
         * @event
         */
        animationComplete: ej.base.EmitType<IAccAnimationCompleteEventArgs>;
        /**
         * Triggers on mouse down.
         * @event
         */
        chartMouseDown: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers while cursor leaves the accumulation chart.
         * @event
         */
        chartMouseLeave: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on mouse up.
         * @event
         */
        chartMouseUp: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers after window resize.
         * @event
         */
        resized: ej.base.EmitType<IAccResizeEventArgs>;
        /**
         * Defines the currencyCode format of the accumulation chart
         * @private
         */
        private currencyCode;
        /** @private */
        svgObject: Element;
        /** @private */
        initialClipRect: Rect;
        /** @private */
        availableSize: Size;
        /** @private */
        renderer: ej.base.SvgRenderer;
        /** @private */
        intl: ej.base.Internationalization;
        /** @private */
        visibleSeries: AccumulationSeries[];
        /** @private */
        seriesCounts: number;
        /** @private explode radius internal property */
        explodeDistance: number;
        /** @private Mouse position x */
        mouseX: number;
        /** @private Mouse position y */
        mouseY: number;
        private resizeTo;
        /** @private */
        center: ChartLocation;
        /** @private */
        readonly type: AccumulationType;
        /** @private */
        isTouch: boolean;
        /** @private */
        animateSeries: boolean;
        private titleCollection;
        /** @private */
        themeStyle: IThemeStyle;
        /**
         * Constructor for creating the AccumulationChart widget
         * @private
         */
        constructor(options?: AccumulationChartModel, element?: string | HTMLElement);
        /**
         *  To create svg object, renderer and binding events for the container.
         */
        protected preRender(): void;
        /**
         * Themeing for chart goes here
         */
        private setTheme();
        /**
         * To render the accumulation chart elements
         */
        protected render(): void;
        /**
         * Method to unbind events for accumulation chart
         */
        private unWireEvents();
        /**
         * Method to bind events for the accumulation chart
         */
        private wireEvents();
        /**
         * Method to set mouse x, y from events
         */
        private setMouseXY(e);
        /**
         * Handles the mouse end.
         * @return {boolean}
         * @private
         */
        accumulationMouseEnd(e: PointerEvent): boolean;
        /**
         * Handles the mouse start.
         * @return {boolean}
         * @private
         */
        accumulationMouseStart(e: PointerEvent): boolean;
        /**
         * Handles the accumulation chart resize.
         * @return {boolean}
         * @private
         */
        accumulationResize(e: Event): boolean;
        /**
         * Handles the export method for chart control.
         */
        export(type: ExportType, fileName: string, orientation?: ej.pdfexport.PdfPageOrientation): void;
        /**
         * Handles the print method for accumulation chart control.
         */
        print(id?: string[] | string | Element): void;
        /**
         * Applying styles for accumulation chart element
         */
        private setStyle(element);
        /**
         * Method to set the annotation content dynamically for accumulation.
         */
        setAnnotationValue(annotationIndex: number, content: string): void;
        /**
         * Handles the mouse move on accumulation chart.
         * @return {boolean}
         * @private
         */
        accumulationMouseMove(e: PointerEvent): boolean;
        titleTooltip(event: Event, x: number, y: number, isTouch?: boolean): void;
        /**
         * Handles the mouse click on accumulation chart.
         * @return {boolean}
         * @private
         */
        accumulationOnMouseClick(e: PointerEvent): boolean;
        private triggerPointEvent(event, element);
        /**
         * Handles the mouse right click on accumulation chart.
         * @return {boolean}
         * @private
         */
        accumulationRightClick(event: MouseEvent | PointerEvent): boolean;
        /**
         * Handles the mouse leave on accumulation chart.
         * @return {boolean}
         * @private
         */
        accumulationMouseLeave(e: PointerEvent): boolean;
        /**
         * Method to set culture for chart
         */
        private setCulture();
        /**
         * Method to create SVG element for accumulation chart.
         */
        private createPieSvg();
        /**
         * To Remove the SVG from accumulation chart.
         * @return {boolean}
         * @private
         */
        removeSvg(): void;
        /**
         * Method to create the secondary element for tooltip, datalabel and annotaitons.
         */
        private createSecondaryElement();
        /**
         * Method to find visible series based on series types
         */
        private calculateVisibleSeries();
        /**
         * To find points from dataSource
         */
        private processData();
        /**
         * To refresh the accumulation chart
         * @private
         */
        refreshChart(): void;
        /**
         * Method to find groupped points
         */
        private doGrouppingProcess();
        /**
         * Method to calculate bounds for accumulation chart
         */
        private calculateBounds();
        private calculateLegendBounds();
        /**
         * To render elements for accumulation chart
         * @private
         */
        renderElements(): void;
        /**
         * To set the left and top position for data label template for center aligned chart
         */
        private setSecondaryElementPosition();
        /**
         * To render the annotaitions for accumulation series.
         */
        private renderAnnotation();
        /**
         * Method to process the explode in accumulation chart
         */
        private processExplode();
        /**
         * Method to render series for accumulation chart
         */
        private renderSeries();
        /**
         * Method to render border for accumulation chart
         */
        private renderBorder();
        /**
         * Method to render legend for accumulation chart
         */
        private renderLegend();
        /**
         * To process the selection in accumulation chart
         */
        private processSelection();
        /**
         * To render title for accumulation chart
         */
        private renderTitle();
        /**
         * To get the series parent element
         * @private
         */
        getSeriesElement(): Element;
        /**
         * To refresh the all visible series points
         * @private
         */
        refreshSeries(): void;
        /**
         * To refresh points label region and visible
         * @private
         */
        refreshPoints(points: AccPoints[]): void;
        /**
         * To get Module name
         *  @private
         */
        getModuleName(): string;
        /**
         * To destroy the accumulationcharts
         * @private
         */
        destroy(): void;
        /**
         * To provide the array of modules needed for control rendering
         * @return {ej.base.ModuleDeclaration[]}
         * @private
         */
        requiredModules(): ej.base.ModuleDeclaration[];
        /**
         * To find datalabel visibility in series
         */
        private findDatalabelVisibility();
        /**
         * Get the properties to be maintained in the persisted state.
         * @private
         */
        getPersistData(): string;
        /**
         * Called internally if any of the property value changed.
         * @private
         */
        onPropertyChanged(newProp: AccumulationChartModel, oldProp: AccumulationChartModel): void;
    }
    
    /**
     * AccumulationChart annotation properties
     */
    
    /**
     * `AccumulationAnnotation` module handles the annotation for accumulation chart.
     */
    export class AccumulationAnnotation extends AnnotationBase {
        private pie;
        private annotations;
        private parentElement;
        /**
         * Constructor for accumulation chart annotation.
         * @private.
         */
        constructor(control: AccumulationChart, annotations: AccumulationAnnotationSettings[]);
        /**
         * Method to render the annotation for accumulation chart
         * @param element
         */
        renderAnnotations(element: Element): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the annotation.
         * @return {void}
         * @private
         */
        destroy(control: AccumulationChart): void;
    }
    
    /**
     * Interface for a class AccumulationAnnotationSettings
     */
    export interface AccumulationAnnotationSettingsModel {
    
        /**
         * Content of the annotation, which accepts the id of the custom element.
         * @default null
         */
        content?: string;
    
        /**
         * if set coordinateUnit as `Pixel` X specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        x?: string | Date | number;
    
        /**
         * if set coordinateUnit as `Pixel` Y specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        y?: string | number;
    
        /**
         * Specifies the coordinate units of the annotation. They are
         * * Pixel - Annotation renders based on x and y pixel value.
         * * Point - Annotation renders based on x and y axis value.
         * @default 'Pixel'
         */
    
        coordinateUnits?: Units;
    
        /**
         * Specifies the regions of the annotation. They are
         * * Chart - Annotation renders based on chart coordinates.
         * * Series - Annotation renders based on series coordinates.
         * @default 'Chart'
         */
    
        region?: Regions;
    
        /**
         * Specifies the position of the annotation. They are
         * * Top - Align the annotation element as top side.
         * * Bottom - Align the annotation element as bottom side.
         * * Middle - Align the annotation element as mid point.
         * @default 'Middle'
         */
    
        verticalAlignment?: Position;
    
        /**
         * Specifies the alignment of the annotation. They are
         * * Near - Align the annotation element as top side.
         * * Far - Align the annotation element as bottom side.
         * * Center - Align the annotation element as mid point.
         * @default 'Center'
         */
    
        horizontalAlignment?: Alignment;
    
        /**
         * Information about annotation for assistive technology.
         * @default null
         */
        description?: string;
    
    }
    
    /**
     * Interface for a class AccumulationDataLabelSettings
     */
    export interface AccumulationDataLabelSettingsModel {
    
        /**
         * If set true, data label for series gets render.
         * @default false
         */
    
        visible?: boolean;
    
        /**
         * The DataSource field which contains the data label value.
         * @default null
         */
    
        name?: string;
    
        /**
         * The background color of the data label, which accepts value in hex, rgba as a valid CSS color string.
         * @default 'transparent'
         */
    
        fill?: string;
    
        /**
         * Specifies the position of data label. They are.
         * * Outside - Places label outside the point.
         * * Inside - Places label inside the point.
         * @default 'Inside'
         */
    
        position?: AccumulationLabelPosition;
    
        /**
         * The roundedCornerX for the data label. It requires `border` values not to be null.
         * @default 5
         */
        rx?: number;
    
        /**
         * The roundedCornerY for the data label. It requires `border` values not to be null.
         * @default 5
         */
        ry?: number;
    
        /**
         * Option for customizing the border lines.
         */
    
        border?: BorderModel;
    
        /**
         * Option for customizing the data label text.
         */
    
        font?: FontModel;
    
        /**
         * Options for customize the connector line in series.
         * This property is applicable for Pie, Funnel and Pyramid series.
         * The default connector length for Pie series is '4%'. For other series, it is null.
         */
        connectorStyle?: ConnectorModel;
    
        /**
         * Custom template to format the data label content. Use ${point.x} and ${point.y} as a placeholder
         * text to display the corresponding data point.
         * @default null
         */
    
        template?: string;
    
    }
    
    /**
     * Interface for a class AccPoints
     */
    export interface AccPointsModel {
    
    }
    
    /**
     * Interface for a class AccumulationSeries
     */
    export interface AccumulationSeriesModel {
    
        /**
         * Specifies the dataSource for the series. It can be an array of JSON objects or an instance of DataManager.
         * ```html
         * <div id='Pie'></div> 
         * ```
         * ```typescript
         * let dataManager: ej.data.DataManager = new DataManager({
         *         url: 'http://mvc.syncfusion.com/Services/Northwnd.svc/Tasks/'
         * });
         * let query: ej.data.Query = new Query().take(50).where('Estimate', 'greaterThan', 0, false);
         * let pie: AccumulationChart = new AccumulationChart({
         * ...
         *     series: [{
         *        dataSource: dataManager,
         *        xName: 'Id',
         *        yName: 'Estimate',
         *        query: query
         *    }],
         * ...
         * });
         * pie.appendTo('#Pie');
         * ```
         * @default ''
         */
    
        dataSource?: Object | ej.data.DataManager;
    
        /**
         * Specifies Query to select data from dataSource. This property is applicable only when the dataSource is `ej.DataManager`.
         * @default null
         */
        query?: ej.data.Query;
    
        /**
         * The DataSource field which contains the x value.
         * @default ''
         */
    
        xName?: string;
    
        /**
         * Specifies the series name
         * @default ''
         */
    
        name?: string;
    
        /**
         * The DataSource field which contains the y value.
         * @default ''
         */
    
        yName?: string;
    
        /**
         * Specifies the series visibility.
         * @default true
         */
    
        visible?: boolean;
    
        /**
         * Options for customizing the border of the series.
         */
    
        border?: BorderModel;
    
        /**
         * Options for customizing the animation for series.
         */
    
        animation?: AnimationModel;
    
        /**
         * The shape of the legend. Each series has its own legend shape. They are
         * * Circle - Renders a circle.
         * * Rectangle - Renders a rectangle.
         * * Triangle - Renders a triangle.
         * * Diamond - Renders a diamond.
         * * Cross - Renders a cross.
         * * HorizontalLine - Renders a horizontalLine.
         * * VerticalLine - Renders a verticalLine.
         * * Pentagon - Renders a pentagon.
         * * InvertedTriangle - Renders a invertedTriangle.
         * * SeriesType -Render a legend shape based on series type. 
         * @default 'SeriesType'
         */
    
        legendShape?: LegendShape;
    
        /**
         * The DataSource field that contains the color value of point
         * It is applicable for series
         * @default ''
         */
    
        pointColorMapping?: string;
    
        /**
         * Custom style for the selected series or points.
         * @default null
         */
        selectionStyle?: string;
    
        /**
         * AccumulationSeries y values less than groupTo are combined into single slice named others
         * @default null
         */
        groupTo?: string;
    
        /**
         * The data label for the series.
         */
        dataLabel?: AccumulationDataLabelSettingsModel;
    
        /**
         * Palette for series points.
         * @default []
         */
        palettes?: string[];
    
        /**
         * Start angle for a series.
         * @default 0
         */
        startAngle?: number;
    
        /**
         * End angle for a series.
         * @default 360
         */
        endAngle?: number;
    
        /**
         * Radius of the pie series and its values in percentage.
         * @default '80%'
         */
        radius?: string;
    
        /**
         * When the innerRadius value is greater than 0 percentage, a donut will appear in pie series. It takes values only in percentage.
         * @default '0'
         */
        innerRadius?: string;
    
        /**
         * Specify the type of the series in accumulation chart. 
         * @default 'Pie'
         */
        type?: AccumulationType;
    
        /**
         * To enable or disable tooltip for a series.
         * @default true
         */
        enableTooltip?: boolean;
    
        /**
         * If set true, series points will be exploded on mouse click or touch.
         * @default false
         */
        explode?: boolean;
    
        /**
         * Distance of the point from the center, which takes values in both pixels and percentage. 
         * @default '30%'
         */
        explodeOffset?: string;
    
        /**
         * If set true, all the points in the series will get exploded on load.
         * @default false
         */
        explodeAll?: boolean;
    
        /**
         * Index of the point, to be exploded on load. 
         * @default null
         */
        explodeIndex?: number;
    
        /**
         * options to customize the empty points in series
         */
        emptyPointSettings?: EmptyPointSettingsModel;
    
        /**
         * Defines the distance between the segments of a funnel/pyramid series. The range will be from 0 to 1
         * @default 0
         */
        gapRatio?: number;
    
        /**
         * Defines the width of the funnel/pyramid with respect to the chart area
         * @default '80%'
         */
        width?: string;
    
        /**
         * Defines the height of the funnel/pyramid with respect to the chart area
         * @default '80%'
         */
        height?: string;
    
        /**
         * Defines the width of the funnel neck with respect to the chart area
         * @default '20%'
         */
        neckWidth?: string;
    
        /**
         * Defines the height of the funnel neck with respect to the chart area
         * @default '20%'
         */
        neckHeight?: string;
    
        /**
         * Defines how the values have to be reflected, whether through height/surface of the segments
         * @default 'Linear'
         */
        pyramidMode?: PyramidModes;
    
        /**
         * The opacity of the series.
         * @default 1.
         */
        opacity?: number;
    
    }
    /**
     * AccumulationChart base file
     */
    
    /**
     * Annotation for accumulation series
     */
    export class AccumulationAnnotationSettings extends ej.base.ChildProperty<AccumulationAnnotationSettings> {
        /**
         * Content of the annotation, which accepts the id of the custom element.
         * @default null
         */
        content: string;
        /**
         * if set coordinateUnit as `Pixel` X specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        x: string | Date | number;
        /**
         * if set coordinateUnit as `Pixel` Y specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        y: string | number;
        /**
         * Specifies the coordinate units of the annotation. They are
         * * Pixel - Annotation renders based on x and y pixel value.
         * * Point - Annotation renders based on x and y axis value.
         * @default 'Pixel'
         */
        coordinateUnits: Units;
        /**
         * Specifies the regions of the annotation. They are
         * * Chart - Annotation renders based on chart coordinates.
         * * Series - Annotation renders based on series coordinates.
         * @default 'Chart'
         */
        region: Regions;
        /**
         * Specifies the position of the annotation. They are
         * * Top - Align the annotation element as top side.
         * * Bottom - Align the annotation element as bottom side.
         * * Middle - Align the annotation element as mid point.
         * @default 'Middle'
         */
        verticalAlignment: Position;
        /**
         * Specifies the alignment of the annotation. They are
         * * Near - Align the annotation element as top side.
         * * Far - Align the annotation element as bottom side.
         * * Center - Align the annotation element as mid point.
         * @default 'Center'
         */
        horizontalAlignment: Alignment;
        /**
         * Information about annotation for assistive technology.
         * @default null
         */
        description: string;
    }
    /**
     * Configures the dataLabel in accumulation chart.
     */
    export class AccumulationDataLabelSettings extends ej.base.ChildProperty<AccumulationDataLabelSettings> {
        /**
         * If set true, data label for series gets render.
         * @default false
         */
        visible: boolean;
        /**
         * The DataSource field which contains the data label value.
         * @default null
         */
        name: string;
        /**
         * The background color of the data label, which accepts value in hex, rgba as a valid CSS color string.
         * @default 'transparent'
         */
        fill: string;
        /**
         * Specifies the position of data label. They are.
         * * Outside - Places label outside the point.
         * * Inside - Places label inside the point.
         * @default 'Inside'
         */
        position: AccumulationLabelPosition;
        /**
         * The roundedCornerX for the data label. It requires `border` values not to be null.
         * @default 5
         */
        rx: number;
        /**
         * The roundedCornerY for the data label. It requires `border` values not to be null.
         * @default 5
         */
        ry: number;
        /**
         * Option for customizing the border lines.
         */
        border: BorderModel;
        /**
         * Option for customizing the data label text.
         */
        font: FontModel;
        /**
         * Options for customize the connector line in series.
         * This property is applicable for Pie, Funnel and Pyramid series.
         * The default connector length for Pie series is '4%'. For other series, it is null.
         */
        connectorStyle: ConnectorModel;
        /**
         * Custom template to format the data label content. Use ${point.x} and ${point.y} as a placeholder
         * text to display the corresponding data point.
         * @default null
         */
        template: string;
    }
    /**
     * Points model for the series.
     */
    export class AccPoints {
        x: Object;
        y: number;
        visible: boolean;
        text: string;
        originalText: string;
        /** @private */
        label: string;
        color: string;
        symbolLocation: ChartLocation;
        index: number;
        /** @private */
        midAngle: number;
        /** @private */
        endAngle: number;
        /** @private */
        labelAngle: number;
        /** @private */
        region: Rect;
        /** @private */
        labelRegion: Rect;
        /** @private */
        labelVisible: boolean;
        /** @private */
        labelPosition: AccumulationLabelPosition;
        /** @private */
        yRatio: number;
        /** @private */
        heightRatio: number;
        /** @private */
        labelOffset: ChartLocation;
        regions: Rect[];
    }
    /**
     *  Configures the series in accumulation chart.
     */
    export class AccumulationSeries extends ej.base.ChildProperty<AccumulationSeries> {
        /**
         * Specifies the dataSource for the series. It can be an array of JSON objects or an instance of DataManager.
         * ```html
         * <div id='Pie'></div>
         * ```
         * ```typescript
         * let dataManager: ej.data.DataManager = new DataManager({
         *         url: 'http://mvc.syncfusion.com/Services/Northwnd.svc/Tasks/'
         * });
         * let query: ej.data.Query = new Query().take(50).where('Estimate', 'greaterThan', 0, false);
         * let pie: AccumulationChart = new AccumulationChart({
         * ...
         *     series: [{
         *        dataSource: dataManager,
         *        xName: 'Id',
         *        yName: 'Estimate',
         *        query: query
         *    }],
         * ...
         * });
         * pie.appendTo('#Pie');
         * ```
         * @default ''
         */
        dataSource: Object | ej.data.DataManager;
        /**
         * Specifies Query to select data from dataSource. This property is applicable only when the dataSource is `ej.DataManager`.
         * @default null
         */
        query: ej.data.Query;
        /**
         * The DataSource field which contains the x value.
         * @default ''
         */
        xName: string;
        /**
         * Specifies the series name
         * @default ''
         */
        name: string;
        /**
         * The DataSource field which contains the y value.
         * @default ''
         */
        yName: string;
        /**
         * Specifies the series visibility.
         * @default true
         */
        visible: boolean;
        /**
         * Options for customizing the border of the series.
         */
        border: BorderModel;
        /**
         * Options for customizing the animation for series.
         */
        animation: AnimationModel;
        /**
         * The shape of the legend. Each series has its own legend shape. They are
         * * Circle - Renders a circle.
         * * Rectangle - Renders a rectangle.
         * * Triangle - Renders a triangle.
         * * Diamond - Renders a diamond.
         * * Cross - Renders a cross.
         * * HorizontalLine - Renders a horizontalLine.
         * * VerticalLine - Renders a verticalLine.
         * * Pentagon - Renders a pentagon.
         * * InvertedTriangle - Renders a invertedTriangle.
         * * SeriesType -Render a legend shape based on series type.
         * @default 'SeriesType'
         */
        legendShape: LegendShape;
        /**
         * The DataSource field that contains the color value of point
         * It is applicable for series
         * @default ''
         */
        pointColorMapping: string;
        /**
         * Custom style for the selected series or points.
         * @default null
         */
        selectionStyle: string;
        /**
         * AccumulationSeries y values less than groupTo are combined into single slice named others
         * @default null
         */
        groupTo: string;
        /**
         * The data label for the series.
         */
        dataLabel: AccumulationDataLabelSettingsModel;
        /**
         * Palette for series points.
         * @default []
         */
        palettes: string[];
        /**
         * Start angle for a series.
         * @default 0
         */
        startAngle: number;
        /**
         * End angle for a series.
         * @default 360
         */
        endAngle: number;
        /**
         * Radius of the pie series and its values in percentage.
         * @default '80%'
         */
        radius: string;
        /**
         * When the innerRadius value is greater than 0 percentage, a donut will appear in pie series. It takes values only in percentage.
         * @default '0'
         */
        innerRadius: string;
        /**
         * Specify the type of the series in accumulation chart.
         * @default 'Pie'
         */
        type: AccumulationType;
        /**
         * To enable or disable tooltip for a series.
         * @default true
         */
        enableTooltip: boolean;
        /**
         * If set true, series points will be exploded on mouse click or touch.
         * @default false
         */
        explode: boolean;
        /**
         * Distance of the point from the center, which takes values in both pixels and percentage.
         * @default '30%'
         */
        explodeOffset: string;
        /**
         * If set true, all the points in the series will get exploded on load.
         * @default false
         */
        explodeAll: boolean;
        /**
         * Index of the point, to be exploded on load.
         * @default null
         */
        explodeIndex: number;
        /**
         * options to customize the empty points in series
         */
        emptyPointSettings: EmptyPointSettingsModel;
        /**
         * Defines the distance between the segments of a funnel/pyramid series. The range will be from 0 to 1
         * @default 0
         */
        gapRatio: number;
        /**
         * Defines the width of the funnel/pyramid with respect to the chart area
         * @default '80%'
         */
        width: string;
        /**
         * Defines the height of the funnel/pyramid with respect to the chart area
         * @default '80%'
         */
        height: string;
        /**
         * Defines the width of the funnel neck with respect to the chart area
         * @default '20%'
         */
        neckWidth: string;
        /**
         * Defines the height of the funnel neck with respect to the chart area
         * @default '20%'
         */
        neckHeight: string;
        /**
         * Defines how the values have to be reflected, whether through height/surface of the segments
         * @default 'Linear'
         */
        pyramidMode: PyramidModes;
        /**
         * The opacity of the series.
         * @default 1.
         */
        opacity: number;
        /** @private */
        points: AccPoints[];
        /** @private */
        dataModule: Data;
        /** @private */
        sumOfPoints: number;
        /** @private */
        index: number;
        private sumOfClub;
        /** @private */
        resultData: Object;
        /** @private */
        lastGroupTo: string;
        /** @private */
        isRectSeries: boolean;
        /** @private */
        clipRect: Rect;
        /**
         * To find the max bounds of the data label to place smart legend
         *  @private
         */
        labelBound: Rect;
        /**
         * To find the max bounds of the accumulation segment to place smart legend
         *  @private
         */
        accumulationBound: Rect;
        /**
         * Defines the funnel size
         * @private
         */
        triangleSize: Size;
        /**
         * Defines the size of the funnel neck
         * @private
         */
        neckSize: Size;
        /** @private To refresh the Datamanager for series */
        refreshDataManager(accumulation: AccumulationChart): void;
        /**
         * To get points on dataManager is success
         * @private
         */
        dataManagerSuccess(e: {
            result: Object;
            count: number;
        }, accumulation: AccumulationChart): void;
        /** @private To find points from result data */
        getPoints(result: Object, accumulation: AccumulationChart): void;
        /**
         * Method to set point index and color
         */
        private pushPoints(point, colors);
        /**
         * Method to find club point
         */
        private isClub(point, clubValue);
        /**
         * Method to find sum of points in the series
         */
        private findSumOfPoints(result);
        /**
         * Method to set points x, y and text from data source
         */
        private setPoints(data, i, colors);
        /**
         * Method render the series elements for accumulation chart
         * @private
         */
        renderSeries(accumulation: AccumulationChart): void;
        /**
         * Method render the points elements for accumulation chart series.
         */
        private renderPoints(accumulation, seriesGroup);
        /**
         * Method render the datalabel elements for accumulation chart.
         */
        private renderDataLabel(accumulation, datalabelGroup);
        /**
         * To find maximum bounds for smart legend placing
         * @private
         */
        findMaxBounds(totalbound: Rect, bound: Rect): void;
        /**
         * To set empty point value for null points
         * @private
         */
        setAccEmptyPoint(point: AccPoints, i: number, data: Object, colors: string[]): void;
        /**
         * To find point is empty
         */
        private isEmpty(point);
    }
    /**
     * method to get series from index
     * @private
     */
    export function getSeriesFromIndex(index: number, visibleSeries: AccumulationSeries[]): AccumulationSeries;
    /**
     * method to get point from index
     * @private
     */
    export function pointByIndex(index: number, points: AccPoints[]): AccPoints;
    
    /**
     * Accumulation charts Enum file
     */
    /**
     * Defines the Accumulation Chart series type.
     */
    export type AccumulationType = 'Pie' | 'Funnel' | 'Pyramid';
    /**
     * Defines the AccumulationLabelPosition. They are
     * * Inside - Define the data label position for the accumulation series Inside.
     * * Outside - Define the data label position for the accumulation series Outside.
     * *
     */
    export type AccumulationLabelPosition = 'Inside' | 'Outside';
    /**
     * Defines the ConnectorType. They are
     * * Line - Accumulation series Connector line type as Straight line.
     * * Curve - Accumulation series Connector line type as Curved line.
     * *
     */
    export type ConnectorType = 'Line' | 'Curve';
    /**
     * Defines the SelectionMode, They are.
     * * none - Disable the selection.
     * * point - To select a point.
     */
    export type AccumulationSelectionMode = 'None' | 'Point';
    /**
     * Defines Theme of the accumulation chart. They are
     * * Material - Render a accumulation chart with Material theme.
     * * Fabric - Render a accumulation chart with fabric theme.
     */
    export type AccumulationTheme = 'Material' | 'Fabric' | 'Bootstrap' | 'Highcontrast';
    /**
     * Defines the empty point mode of the chart.
     * * Zero - Used to display empty points as zero.
     * * Drop - Used to ignore the empty point while rendering.
     * * Average - Used to display empty points as previous and next point average.
     */
    export type AccEmptyPointMode = 'Zero' | 'Drop' | 'Average';
    /**
     * Defines the mode of the pyramid
     * * Linear - Height of the pyramid segments reflects the values
     * * Surface - Surface/Area of the  pyramid segments reflects the values
     */
    export type PyramidModes = 'Linear' | 'Surface';
    
    /**
     * Interface for Accumulation chart
     */
    
    /**
     * Accumulation Chart SeriesRender event arguments.
     */
    export interface IAccSeriesRenderEventArgs {
        /** Defines the current series */
        series: AccumulationSeries;
        /** Defines the current data object */
        data: Object;
        /** Defines the current series name */
        name: string;
    }
    /**
     * Accumulation Chart TextRender event arguments.
     */
    export interface IAccTextRenderEventArgs extends IChartEventArgs {
        /** Defines the current series */
        series: AccumulationSeriesModel;
        /** Defines the current point */
        point: AccPoints;
        /** Defines the current text */
        text: string;
        /** Defines the current fill color */
        color: string;
        /** Defines the current label border */
        border: BorderModel;
        /** Defines the current text template */
        template: string;
    }
    /**
     * Accumulation Chart TooltipRender event arguments.
     */
    export interface IAccTooltipRenderEventArgs extends IChartEventArgs {
        /** Defines the current tooltip content */
        content?: string | HTMLElement;
        /** Defines the current tooltip text style */
        textStyle?: FontModel;
        /** Defines the current tooltip series */
        series: AccumulationSeries;
        /** Defines the current tooltip point */
        point: AccPoints;
    }
    /**
     * Accumulation Chart AnimationComplete event arguments.
     */
    export interface IAccAnimationCompleteEventArgs extends IChartEventArgs {
        /** Defines the current animation series */
        series: AccumulationSeries;
        /** Defines the accumulation chart instance */
        accumulation: AccumulationChart;
    }
    /**
     * Accumulation Chart PointRender event arguments.
     */
    export interface IAccPointRenderEventArgs extends IChartEventArgs {
        /** Defines the current series of the point */
        series: AccumulationSeries;
        /** Defines the current point */
        point: AccPoints;
        /** Defines the current point fill color */
        fill: string;
        /** Defines the current point border color */
        border: BorderModel;
        /** Defines the current point height */
        height?: number;
        /** Defines the current point width */
        width?: number;
    }
    /**
     * Accumulation Chart Load or Loaded event arguments.
     */
    export interface IAccLoadedEventArgs extends IChartEventArgs {
        /** Defines the accumulation chart instance */
        accumulation: AccumulationChart;
    }
    /**
     * Accumulation Chart Resize event arguments.
     */
    export interface IAccResizeEventArgs {
        /** Defines the name of the Event */
        name: string;
        /** Defines the previous size of the accumulation chart */
        previousSize: Size;
        /** Defines the current size of the accumulation chart */
        currentSize: Size;
        /** Defines the accumulation chart instance */
        accumulation: AccumulationChart;
    }
    
    /**
     * Accumulation Base used to do some base calculation for accumulation chart.
     */
    export class AccumulationBase {
        /** @private */
        constructor(accumulation: AccumulationChart);
        private pieCenter;
        /**
         * Gets the center of the pie
         * @private
         */
        /**
         * Sets the center of the pie
         * @private
         */
        center: ChartLocation;
        private pieRadius;
        /**
         * Gets the radius of the pie
         * @private
         */
        /**
         * Sets the radius of the pie
         * @private
         */
        radius: number;
        private pieLabelRadius;
        /**
         * Gets the label radius of the pie
         * @private
         */
        /**
         * Sets the label radius of the pie
         * @private
         */
        labelRadius: number;
        /** @private */
        protected accumulation: AccumulationChart;
        /**
         * Checks whether the series is circular or not
         * @private
         */
        protected isCircular(): boolean;
        /**
         * To process the explode on accumulation chart loading
         * @private
         */
        processExplode(event: Event): void;
        /**
         * To invoke the explode on accumulation chart loading
         * @private
         */
        invokeExplode(): void;
        /**
         * To deExplode all points in the series
         */
        private deExplodeAll(index);
        /**
         * To explode point by index
         * @private
         */
        explodePoints(index: number, chart: AccumulationChart, explode?: boolean): void;
        /**
         * To check point is exploded by id
         */
        private isExplode(id);
        /**
         * To deExplode the point by index
         */
        private deExplodeSlice(index, sliceId, center);
        /**
         * To translate the point elements by index and position
         */
        private setTranslate(index, sliceId, position);
        /**
         * To translate the point element by id and position
         */
        private setElementTransform(id, position);
        /**
         * To translate the point elements by index position
         */
        private explodeSlice(index, translate, sliceId, center);
    }
    
    /**
     * AccumulationDataLabel module used to render `dataLabel`.
     */
    export class AccumulationDataLabel extends AccumulationBase {
        /** @private */
        titleRect: Rect;
        /** @private */
        areaRect: Rect;
        /** @private */
        clearTooltip: number;
        private id;
        marginValue: number;
        constructor(accumulation: AccumulationChart);
        /**
         * Method to get datalabel text location.
         * @private
         */
        getDataLabelPosition(point: AccPoints, dataLabel: AccumulationDataLabelSettingsModel, textSize: Size, points: AccPoints[], parent: Element, id: string): void;
        /**
         * Method to get datalabel bound.
         */
        private getLabelRegion(point, position, textSize, labelRadius, margin, endAngle?);
        /**
         * Method to get datalabel smart position.
         */
        private getSmartLabel(point, dataLabel, textSize, points, parent, id);
        /**
         * To find trimmed datalabel tooltip needed.
         * @return {void}
         * @private
         */
        move(e: Event, x: number, y: number, isTouch?: boolean): void;
        /**
         * To find previous valid label point
         */
        private findPreviousPoint(points, index, position);
        /**
         * To find current point datalabel is overlapping with other points
         */
        private isOverlapping(currentPoint, points);
        /**
         * To get text trimmed while exceeds the accumulation chart area.
         */
        private textTrimming(point, rect, font, position);
        /**
         * To set point label visible and region to disable.
         */
        private setPointVisibileFalse(point);
        /**
         * To set datalabel angle position for outside labels
         */
        private setOuterSmartLabel(previousPoint, point, border, labelRadius, textsize, margin);
        /**
         * Sets smart label positions for funnel and pyramid series
         */
        private setSmartLabelForSegments(point, prevPoint, distance, textSize, margin);
        /**
         * To find connector line overlapping.
         */
        private isConnectorLineOverlapping(point, previous);
        /**
         * To find two rectangle intersect
         */
        private isLineRectangleIntersect(line1, line2, rect);
        /**
         * To find two line intersect
         */
        private isLinesIntersect(point1, point2, point11, point12);
        /**
         * To get two rectangle overlapping angles.
         */
        private getOverlappedAngle(first, second, angle, padding);
        /**
         * To get connector line path
         */
        private getConnectorPath(label, point, dataLabel, end?);
        /**
         * Finds the curved path for funnel/pyramid data label connectors
         */
        private getPolyLinePath(start, end);
        /**
         * Finds the bezier point for funnel/pyramid data label connectors
         */
        private getBezierPoint(t, controlPoints, index, count);
        /**
         * To get label edges based on the center and label rect position.
         */
        private getEdgeOfLabel(labelshape, angle, middle, border?);
        /**
         * Finds the distance between the label position and the edge/center of the funnel/pyramid
         */
        private getLabelDistance(point, dataLabel);
        /**
         * Finds the label position / beginning of the connector(ouside funnel labels)
         */
        private getLabelLocation(point, position?);
        /**
         * Finds the beginning of connector line
         */
        private getConnectorStartPoint(point, connector);
        /**
         * To find area rect based on margin, available size.
         * @private
         */
        findAreaRect(): void;
        /**
         * To render the data labels from series points.
         */
        renderDataLabel(point: AccPoints, dataLabel: AccumulationDataLabelSettingsModel, parent: Element, points: AccPoints[], series: number, templateElement?: HTMLElement): void;
        /**
         * To find the template element size
         * @param element
         * @param point
         * @param argsData
         */
        private getTemplateSize(element, point, argsData);
        /**
         * To set the template element style
         * @param childElement
         * @param point
         * @param parent
         * @param labelColor
         * @param fill
         */
        private setTemplateStyle(childElement, point, parent, labelColor, fill);
        /**
         * To find saturated color for datalabel
         */
        private getSaturatedColor(point, color);
        /**
         * Animates the data label template.
         * @return {void}.
         * @private
         */
        doTemplateAnimation(accumulation: AccumulationChart, element: Element): void;
        /**
         * To find background color for the datalabel
         */
        private getLabelBackground(point);
        /**
         * To correct the padding between datalabel regions.
         */
        private correctLabelRegion(labelRegion, textSize, padding?);
        /**
         * To get the dataLabel module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the data label.
         * @return {void}
         * @private
         */
        destroy(accumulation: AccumulationChart): void;
    }
    
    /**
     * Defines the behavior of a funnel series
     */
    
    /**
     * FunnelSeries module used to render `Funnel` Series.
     */
    export class FunnelSeries extends TriangularBase {
        /**
         * Defines the path of a funnel segment
         */
        private getSegmentData(point, series, chart);
        /**
         * Renders a funnel segment
         * @private
         */
        renderPoint(point: AccPoints, series: AccumulationSeries, chart: AccumulationChart, options: PathOption): void;
        /**
         * To get the module name of the funnel series.
         */
        protected getModuleName(): string;
        /**
         * To destroy the funnel series.
         * @return {void}
         * @private
         */
        destroy(accumulation: AccumulationChart): void;
    }
    
    /**
     * AccumulationLegend module used to render `Legend` for Accumulation chart.
     */
    export class AccumulationLegend extends BaseLegend {
        titleRect: Rect;
        private totalRowCount;
        private maxColumnWidth;
        /**
         * Constructor for Accumulation Legend.
         * @param chart
         */
        constructor(chart: AccumulationChart);
        /**
         * Get the legend options.
         * @return {void}
         * @private
         */
        getLegendOptions(chart: AccumulationChart, series: AccumulationSeries[]): void;
        /**
         * To find legend bounds for accumulation chart.
         * @private
         */
        getLegendBounds(availableSize: Size, legendBounds: Rect, legend: LegendSettingsModel): void;
        /**
         * To find maximum column size for legend
         */
        private getMaxColumn(columns, width, padding, rowWidth);
        /**
         * To find available width from legend x position.
         */
        private getAvailWidth(tx, width, legendX);
        /**
         * To find legend rendering locations from legend options.
         * @private
         */
        getRenderPoint(legendOption: LegendOptions, start: ChartLocation, textPadding: number, prevLegend: LegendOptions, rect: Rect, count: number, firstLegend: number): void;
        /**
         * finding the smart legend place according to positions.
         * @return {void}
         * @private
         */
        getSmartLegendLocation(labelBound: Rect, legendBound: Rect, margin: MarginModel): void;
        /**
         * To get title rect.
         */
        private getTitleRect(accumulation);
        /**
         * To get legend by index
         */
        private legendByIndex(index, legendCollections);
        /**
         * To show or hide the legend on clicking the legend.
         * @return {void}
         */
        click(event: Event): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the Legend.
         * @return {void}
         * @private
         */
        destroy(chart: AccumulationChart): void;
    }
    
    /**
     * PieBase class used to do pie base calculations.
     */
    export class PieBase extends AccumulationBase {
        protected startAngle: number;
        protected totalAngle: number;
        protected innerRadius: number;
        center: ChartLocation;
        radius: number;
        labelRadius: number;
        /**
         * To initialize the property values.
         * @private
         */
        initProperties(chart: AccumulationChart, series: AccumulationSeries): void;
        /**
         * To find the center of the accumulation.
         * @private
         */
        findCenter(accumulation: AccumulationChart, series: AccumulationSeries): void;
        /**
         * To find angles from series.
         */
        private initAngles(series);
        /**
         * To calculate data-label bound
         * @private
         */
        defaultLabelBound(series: AccumulationSeries, visible: boolean, position: AccumulationLabelPosition): void;
        /**
         * To calculate series bound
         * @private
         */
        getSeriesBound(series: AccumulationSeries): Rect;
        /**
         * To get rect location size from angle
         */
        private getRectFromAngle(angle);
        /**
         * To get path arc direction
         */
        protected getPathArc(center: ChartLocation, start: number, end: number, radius: number, innerRadius: number): string;
        /**
         * To get pie direction
         */
        protected getPiePath(center: ChartLocation, start: ChartLocation, end: ChartLocation, radius: number, clockWise: number): string;
        /**
         * To get doughnut direction
         */
        protected getDoughnutPath(center: ChartLocation, start: ChartLocation, end: ChartLocation, radius: number, innerStart: ChartLocation, innerEnd: ChartLocation, innerRadius: number, clockWise: number): string;
        /**
         * Method to start animation for pie series.
         */
        protected doAnimation(slice: Element, series: AccumulationSeries): void;
    }
    
    /**
     * AccumulationChart series file
     */
    
    /**
     * PieSeries module used to render `Pie` Series.
     */
    export class PieSeries extends PieBase {
        /**
         * To get path option, degree, symbolLocation from the point.
         * @private
         */
        renderPoint(point: AccPoints, series: AccumulationSeries, chart: AccumulationChart, option: PathOption): PathOption;
        /**
         * To get path option from the point.
         */
        private getPathOption(point, degree);
        /**
         * To animate the pie series.
         * @private
         */
        animateSeries(accumulation: AccumulationChart, option: AnimationModel, series: AccumulationSeries, slice: Element): void;
        /**
         * To get the module name of the Pie series.
         */
        protected getModuleName(): string;
        /**
         * To destroy the pie series.
         * @return {void}
         * @private
         */
        destroy(accumulation: AccumulationChart): void;
    }
    
    /**
     * Defines the behavior of a pyramid series
     */
    
    /**
     * PyramidSeries module used to render `Pyramid` Series.
     */
    export class PyramidSeries extends TriangularBase {
        /**
         * Defines the path of a pyramid segment
         */
        private getSegmentData(point, series, chart);
        /**
         * Initializes the size of the pyramid segments
         * @private
         */
        protected initializeSizeRatio(points: AccPoints[], series: AccumulationSeries): void;
        /**
         * Defines the size of the pyramid segments, the surface of that will reflect the values
         */
        private calculateSurfaceSegments(series);
        /**
         * Finds the height of pyramid segment
         */
        private getSurfaceHeight(y, surface);
        /**
         * Solves quadratic equation
         */
        private solveQuadraticEquation(a, b, c);
        /**
         * Renders a pyramid segment
         */
        private renderPoint(point, series, chart, options);
        /**
         * To get the module name of the Pyramid series.
         */
        protected getModuleName(): string;
        /**
         * To destroy the pyramid series
         * @return {void}
         * @private
         */
        destroy(accumulation: AccumulationChart): void;
    }
    
    /**
     * Defines the common behavior of funnel and pyramid series
     */
    
    /**
     * TriangularBase is used to calculate base functions for funnel/pyramid series.
     */
    export class TriangularBase extends AccumulationBase {
        /**
         * Initializes the properties of funnel/pyramid series
         * @private
         */
        initProperties(chart: AccumulationChart, series: AccumulationSeries): void;
        /**
         * Initializes the size of the pyramid/funnel segments
         * @private
         */
        protected initializeSizeRatio(points: AccPoints[], series: AccumulationSeries, reverse?: boolean): void;
        /**
         * Marks the label location from the set of points that forms a pyramid/funnel segment
         * @private
         */
        protected setLabelLocation(series: AccumulationSeries, point: AccPoints, points: ChartLocation[]): void;
        /**
         * Finds the path to connect the list of points
         * @private
         */
        protected findPath(locations: ChartLocation[]): string;
        /**
         * To calculate data-label bounds
         * @private
         */
        defaultLabelBound(series: AccumulationSeries, visible: boolean, position: AccumulationLabelPosition, chart: AccumulationChart): void;
    }
    
    /**
     * `AccumulationSelection` module handles the selection for accumulation chart.
     * @private
     */
    export class AccumulationSelection extends BaseSelection {
        private renderer;
        /** @private */
        rectPoints: Rect;
        selectedDataIndexes: Indexes[];
        private series;
        constructor(accumulation: AccumulationChart);
        /**
         * To initialize the private variables
         */
        private initPrivateVariables(accumulation);
        /**
         * Invoke selection for rendered chart.
         * @param  {AccumulationChart} chart - Define the chart to invoke the selection.
         * @return {void}
         */
        invokeSelection(accumulation: AccumulationChart): void;
        /**
         * To get series selection style by series.
         */
        private generateStyle(series);
        /**
         * To get elements by index, series
         */
        private findElements(accumulation, series, index);
        /**
         * To get series point element by index
         */
        private getElementByIndex(index);
        /**
         * To calculate selected elements on mouse click or touch
         * @private
         */
        calculateSelectedElements(accumulation: AccumulationChart, event: Event): void;
        /**
         * To perform the selection process based on index and element.
         */
        private performSelection(index, accumulation, element?);
        /**
         * To select the element by index. Adding or removing selection style class name.
         */
        private selection(accumulation, index, selectedElements);
        /**
         * To redraw the selection process on accumulation chart refresh.
         * @private
         */
        redrawSelection(accumulation: AccumulationChart, oldMode: AccumulationSelectionMode): void;
        /**
         * To remove the selected elements style classes by indexes.
         */
        private removeSelectedElements(accumulation, indexes);
        /**
         * To perform the selection for legend elements.
         * @private
         */
        legendSelection(accumulation: AccumulationChart, series: number, pointIndex: number): void;
        /**
         * To select the element by selected data indexes.
         */
        private selectDataIndex(indexes, accumulation);
        /**
         * To remove the selection styles for multi selection process.
         */
        private removeMultiSelectEelments(accumulation, index, currentIndex, seriesCollection);
        /**
         * To apply the opacity effect for accumulation chart series elements.
         */
        private blurEffect(pieId, visibleSeries);
        /**
         * To check selection elements by style class name.
         */
        private checkSelectionElements(element, className, visibility);
        /**
         * To apply selection style for elements.
         */
        private applyStyles(elements, index);
        /**
         * To get selection style class name by id
         */
        private getSelectionClass(id);
        /**
         * To remove selection style for elements.
         */
        private removeStyles(elements, index);
        /**
         * To apply or remove selected elements index.
         */
        private addOrRemoveIndex(indexes, index, add?);
        /**
         * To check two index, point and series are equal
         */
        private checkEquals(first, second);
        /**
         * To check selected points are visibility
         */
        private checkPointVisibility(selectedDataIndexes);
        /**
         * Get module name.
         */
        getModuleName(): string;
        /**
         * To destroy the selection.
         * @return {void}
         * @private
         */
        destroy(accumulation: AccumulationChart): void;
    }
    
    /**
     * `AccumulationTooltip` module is used to render tooltip for accumulation chart.
     */
    export class AccumulationTooltip extends BaseTooltip {
        accumulation: AccumulationChart;
        constructor(accumulation: AccumulationChart);
        /**
         * @hidden
         */
        private addEventListener();
        private mouseLeaveHandler(e);
        private mouseUpHandler(e);
        private mouseMoveHandler(e);
        /**
         * Renders the tooltip.
         * @param  {PointerEvent} event - Mouse move event.
         * @return {void}
         */
        tooltip(event: PointerEvent | TouchEvent): void;
        private renderSeriesTooltip(e, chart, isFirst, tooltipDiv);
        private drawMarker(isBottom, size);
        private getPieData(e, chart, x, y);
        /**
         * To get series from index
         */
        private getSeriesFromIndex(index, visibleSeries);
        private getTemplateText(data);
        private getTooltipText(data, tooltip);
        private findHeader(data);
        private parseTemplate(point, series, format);
        /**
         * Removes the tooltip on mouse leave.
         * @return {void}
         * @private
         */
        removeTooltip(duration: number): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the Tooltip.
         * @return {void}
         * @private
         */
        destroy(chart: AccumulationChart): void;
    }
    
    /**
     * Interface for a class CrosshairSettings
     */
    export interface CrosshairSettingsModel {
    
        /**
         * If set to true, crosshair line becomes visible.
         * @default false
         */
        enable?: boolean;
    
        /**
         * Options to customize the crosshair line.
         */
        line?: BorderModel;
    
        /**
         * Specifies the line type. Horizontal mode enables the horizontal line and Vertical mode enables the vertical line. They are,
         * * None: Hides both vertical and horizontal crosshair lines.
         * * Both: Shows both vertical and horizontal crosshair lines.
         * * Vertical: Shows the vertical line.
         * * Horizontal: Shows the horizontal line.
         * @default Both
         */
        lineType?: LineType;
    
    }
    
    /**
     * Interface for a class ZoomSettings
     */
    export interface ZoomSettingsModel {
    
        /**
         * If set to true, chart can be zoomed by a rectangular selecting region on the plot area.
         * @default false
         */
    
        enableSelectionZooming?: boolean;
    
        /**
         * If  to true, chart can be pinched to zoom in / zoom out.
         * @default false
         */
    
        enablePinchZooming?: boolean;
    
        /**
         * If set to true, chart can be zoomed by using mouse wheel.
         * @default false
         */
    
        enableMouseWheelZooming?: boolean;
    
        /**
         * If set to true, zooming will be performed on mouse up. It requires `enableSelectionZooming` to be true.
         * ```html 
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *    zoomSettings: {
         *      enableSelectionZooming: true,
         *      enableDeferredZooming: false
         *    }
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default true
         */
    
        enableDeferredZooming?: boolean;
    
        /**
         * Specifies whether to allow zooming vertically or horizontally or in both ways.They are,
         * * x,y: Chart can be zoomed both vertically and horizontally.
         * * x: Chart can be zoomed horizontally.
         * * y: Chart can be zoomed  vertically.
         *  It requires `enableSelectionZooming` to be true.
         * ```html 
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *    zoomSettings: {
         *      enableSelectionZooming: true,
         *      mode: 'XY'
         *    }
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default 'XY'
         */
        mode?: ZoomMode;
    
        /**
         * Specifies the toolkit options for the zooming as follows:
         * * Zoom
         * * ZoomIn
         * * ZoomOut
         * * Pan
         * * Reset
         * @default '["Zoom", "ZoomIn", "ZoomOut", "Pan", "Reset"]' 
         */
    
        toolbarItems?: ToolbarItems[];
    
        /**
         * Specifies whether chart needs to be panned by default.
         * @default false.
         */
    
        enablePan?: boolean;
    
    }
    
    /**
     * Interface for a class Chart
     */
    export interface ChartModel extends ej.base.ComponentModel{
    
        /**
         * The width of the chart as a string accepts input as both like '100px' or '100%'.
         * If specified as '100%, chart renders to the full width of its parent element.
         * @default null
         */
    
        width?: string;
    
        /**
         * The height of the chart as a string accepts input both as '100px' or '100%'.
         * If specified as '100%, chart renders to the full height of its parent element.
         * @default null
         */
    
        height?: string;
    
        /**
         * Title of the chart
         * @default ''
         */
    
        title?: string;
    
        /**
         * Options for customizing the title of the Chart.
         */
    
        titleStyle?: FontModel;
    
        /**
         *  Options to customize left, right, top and bottom margins of the chart.
         */
    
        margin?: MarginModel;
    
        /**
         * Options for customizing the color and width of the chart border.
         */
    
        border?: BorderModel;
    
        /**
         * The background color of the chart that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        background?: string;
    
        /**
         * Options for configuring the border and background of the chart area.
         */
    
        chartArea?: ChartAreaModel;
    
        /**
         * Options to configure the horizontal axis.
         */
    
        primaryXAxis?: AxisModel;
    
        /**
         * Options to configure the vertical axis.
         */
    
        primaryYAxis?: AxisModel;
    
        /**
         * Options to split Chart into multiple plotting areas horizontally.
         * Each object in the collection represents a plotting area in the Chart.
         */
    
        rows?: RowModel[];
    
        /**
         * Options to split chart into multiple plotting areas vertically.
         * Each object in the collection represents a plotting area in the chart.
         */
    
        columns?: ColumnModel[];
    
        /**
         * Secondary axis collection for the chart.
         */
    
        axes?: AxisModel[];
    
        /**
         * The configuration for series in the chart.
         */
    
        series?: SeriesModel[];
    
        /**
         * The configuration for annotation in chart.
         */
    
        annotations?: ChartAnnotationSettingsModel[];
    
        /**
         * Palette for the chart series.
         * @default []
         */
        palettes?: string[];
    
        /**
         * Specifies the theme for the chart.
         * @default 'Material'
         */
        theme?: ChartTheme;
    
        /**
         * Options for customizing the tooltip of the chart.
         */
    
        tooltip?: TooltipSettingsModel;
    
        /**
         * Options for customizing the crosshair of the chart.
         */
        crosshair?: CrosshairSettingsModel;
    
        /**
         * Options for customizing the legend of the chart.
         */
        legendSettings?: LegendSettingsModel;
    
        /**
         * Options to enable the zooming feature in the chart.
         */
        zoomSettings?: ZoomSettingsModel;
    
        /**
         * Specifies whether series or data point has to be selected. They are,
         * * none: Disables the selection.
         * * series: selects a series.
         * * point: selects a point.
         * * cluster: selects a cluster of point
         * * dragXY: selects points by dragging with respect to both horizontal and vertical axes
         * * dragX: selects points by dragging with respect to horizontal axis.
         * * dragY: selects points by dragging with respect to vertical axis.
         * @default None
         */
        selectionMode?: SelectionMode;
    
        /**
         * If set true, enables the multi selection in chart. It requires `selectionMode` to be `Point` | `Series` | or `Cluster`.
         * @default false
         */
        isMultiSelect?: boolean;
    
        /**
         * Specifies the point indexes to be selected while loading a chart.
         * It requires `selectionMode` to be `Point` | `Series` | or `Cluster`.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *   selectionMode: 'Point',
         *   selectedDataIndexes: [ { series: 0, point: 1},
         *                          { series: 2, point: 3} ],
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default []
         */
        selectedDataIndexes?: IndexesModel[];
    
        /**
         * Specifies whether a grouping separator should be used for a number.
         * @default false
         */
        useGroupingSeparator?: boolean;
    
        /**
         * It specifies whether the chart should be render in transposed manner or not.
         * @default false
         */
        isTransposed?: boolean;
    
        /**
         * Defines the collection of technical indicators, that are used in financial markets
         */
        indicators?: TechnicalIndicatorModel[];
    
        /**
         * Description for chart.
         * @default null
         */
        description?: string;
    
        /**
         * TabIndex value for the chart.
         * @default 1
         */
        tabIndex?: number;
    
        /**
         * To enable the side by side placing the points for column type series.
         * @default true
         */
        enableSideBySidePlacement?: boolean;
    
        /**
         * Triggers before the annotation gets rendered.
         * @event
         */
    
        annotationRender?: ej.base.EmitType<IAnnotationRenderEventArgs>;
    
        /**
         * Triggers before the prints gets started.
         * @event
         */
    
        beforePrint?: ej.base.EmitType<IPrintEventArgs>;
    
        /**
         * Triggers after chart load.
         * @event
         */
        loaded?: ej.base.EmitType<ILoadedEventArgs>;
    
        /**
         * Triggers before chart load.
         * @event
         */
        load?: ej.base.EmitType<ILoadedEventArgs>;
    
        /**
         * Triggers after animation is completed for the series.
         * @event
         */
        animationComplete?: ej.base.EmitType<IAnimationCompleteEventArgs>;
    
        /**
         * Triggers before the legend is rendered.
         * @event
         */
        legendRender?: ej.base.EmitType<ILegendRenderEventArgs>;
    
        /**
         * Triggers before the data label for series is rendered.
         * @event
         */
    
        textRender?: ej.base.EmitType<ITextRenderEventArgs>;
    
        /**
         * Triggers before each points for the series is rendered.
         * @event
         */
    
        pointRender?: ej.base.EmitType<IPointRenderEventArgs>;
    
        /**
         * Triggers before the series is rendered.
         * @event
         */
    
        seriesRender?: ej.base.EmitType<ISeriesRenderEventArgs>;
    
        /**
         * Triggers before each axis label is rendered.
         * @event
         */
        axisLabelRender?: ej.base.EmitType<IAxisLabelRenderEventArgs>;
    
        /**
         * Triggers before each axis range is rendered.
         * @event
         */
        axisRangeCalculated?: ej.base.EmitType<IAxisRangeCalculatedEventArgs>;
    
        /**
         * Triggers before each axis multi label is rendered.
         * @event
         */
        axisMultiLabelRender?: ej.base.EmitType<IAxisMultiLabelRenderEventArgs>;
    
        /**
         * Triggers before the tooltip for series is rendered.
         * @event
         */
    
        tooltipRender?: ej.base.EmitType<ITooltipRenderEventArgs>;
    
        /**
         * Triggers on hovering the chart.
         * @event
         */
    
        chartMouseMove?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on clicking the chart.
         * @event
         */
    
        chartMouseClick?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on point click.
         * @event
         */
    
        pointClick?: ej.base.EmitType<IPointEventArgs>;
    
        /**
         * Triggers on point move.
         * @event
         */
    
        pointMove?: ej.base.EmitType<IPointEventArgs>;
    
        /**
         * Triggers when cursor leaves the chart.
         * @event
         */
    
        chartMouseLeave?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on mouse down.
         * @event
         */
    
        chartMouseDown?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on mouse up.
         * @event
         */
    
        chartMouseUp?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers after the drag selection is completed.
         * @event
         */
    
        dragComplete?: ej.base.EmitType<IDragCompleteEventArgs>;
    
        /**
         * Triggers after the zoom selection is completed.
         * @event
         */
    
        zoomComplete?: ej.base.EmitType<IZoomCompleteEventArgs>;
    
        /**
         * Defines the currencyCode format of the chart
         * @private
         */
        currencyCode?: string;
    
    }
    
    /**
     * Configures the crosshair in the chart.
     */
    export class CrosshairSettings extends ej.base.ChildProperty<CrosshairSettings> {
        /**
         * If set to true, crosshair line becomes visible.
         * @default false
         */
        enable: boolean;
        /**
         * Options to customize the crosshair line.
         */
        line: BorderModel;
        /**
         * Specifies the line type. Horizontal mode enables the horizontal line and Vertical mode enables the vertical line. They are,
         * * None: Hides both vertical and horizontal crosshair lines.
         * * Both: Shows both vertical and horizontal crosshair lines.
         * * Vertical: Shows the vertical line.
         * * Horizontal: Shows the horizontal line.
         * @default Both
         */
        lineType: LineType;
    }
    /**
     * Configures the zooming behavior for the chart.
     */
    export class ZoomSettings extends ej.base.ChildProperty<ZoomSettings> {
        /**
         * If set to true, chart can be zoomed by a rectangular selecting region on the plot area.
         * @default false
         */
        enableSelectionZooming: boolean;
        /**
         * If  to true, chart can be pinched to zoom in / zoom out.
         * @default false
         */
        enablePinchZooming: boolean;
        /**
         * If set to true, chart can be zoomed by using mouse wheel.
         * @default false
         */
        enableMouseWheelZooming: boolean;
        /**
         * If set to true, zooming will be performed on mouse up. It requires `enableSelectionZooming` to be true.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *    zoomSettings: {
         *      enableSelectionZooming: true,
         *      enableDeferredZooming: false
         *    }
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default true
         */
        enableDeferredZooming: boolean;
        /**
         * Specifies whether to allow zooming vertically or horizontally or in both ways.They are,
         * * x,y: Chart can be zoomed both vertically and horizontally.
         * * x: Chart can be zoomed horizontally.
         * * y: Chart can be zoomed  vertically.
         *  It requires `enableSelectionZooming` to be true.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *    zoomSettings: {
         *      enableSelectionZooming: true,
         *      mode: 'XY'
         *    }
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default 'XY'
         */
        mode: ZoomMode;
        /**
         * Specifies the toolkit options for the zooming as follows:
         * * Zoom
         * * ZoomIn
         * * ZoomOut
         * * Pan
         * * Reset
         * @default '["Zoom", "ZoomIn", "ZoomOut", "Pan", "Reset"]'
         */
        toolbarItems: ToolbarItems[];
        /**
         * Specifies whether chart needs to be panned by default.
         * @default false.
         */
        enablePan: boolean;
    }
    /**
     * Represents the Chart control.
     * ```html
     * <div id="chart"/>
     * <script>
     *   var chartObj = new Chart({ isResponsive : true });
     *   chartObj.appendTo("#chart");
     * </script>
     * ```
     */
    export class Chart extends ej.base.Component<HTMLElement> implements ej.base.INotifyPropertyChanged {
        /**
         * `lineSeriesModule` is used to add line series to the chart.
         */
        lineSeriesModule: LineSeries;
        /**
         * `multiColoredLineSeriesModule` is used to add multi colored line series to the chart.
         */
        multiColoredLineSeriesModule: MultiColoredLineSeries;
        /**
         * `multiColoredAreaSeriesModule` is used to add multi colored area series to the chart.
         */
        multiColoredAreaSeriesModule: MultiColoredAreaSeries;
        /**
         * `columnSeriesModule` is used to add column series to the chart.
         */
        columnSeriesModule: ColumnSeries;
        /**
         * `areaSeriesModule` is used to add area series in the chart.
         */
        areaSeriesModule: AreaSeries;
        /**
         * `barSeriesModule` is used to add bar series to the chart.
         */
        barSeriesModule: BarSeries;
        /**
         * `stackingColumnSeriesModule` is used to add stacking column series in the chart.
         */
        stackingColumnSeriesModule: StackingColumnSeries;
        /**
         * `stackingAreaSeriesModule` is used to add stacking area series to the chart.
         */
        stackingAreaSeriesModule: StackingAreaSeries;
        /**
         * 'CandleSeriesModule' is used to add candle series in the chart.
         */
        candleSeriesModule: CandleSeries;
        /**
         * `stackingBarSeriesModule` is used to add stacking bar series to the chart.
         */
        stackingBarSeriesModule: StackingBarSeries;
        /**
         * `stepLineSeriesModule` is used to add step line series to the chart.
         */
        stepLineSeriesModule: StepLineSeries;
        /**
         * `stepAreaSeriesModule` is used to add step area series to the chart.
         */
        stepAreaSeriesModule: StepAreaSeries;
        /**
         * `polarSeriesModule` is used to add polar series in the chart.
         */
        polarSeriesModule: PolarSeries;
        /**
         *  `radarSeriesModule` is used to add radar series in the chart.
         */
        radarSeriesModule: RadarSeries;
        /**
         * `splineSeriesModule` is used to add spline series to the chart.
         */
        splineSeriesModule: SplineSeries;
        /**
         * `splineAreaSeriesModule` is used to add spline area series to the chart.
         */
        splineAreaSeriesModule: SplineAreaSeries;
        /**
         * `scatterSeriesModule` is used to add scatter series to the chart.
         */
        scatterSeriesModule: ScatterSeries;
        /**
         * `boxAndWhiskerSeriesModule` is used to add line series to the chart.
         */
        boxAndWhiskerSeriesModule: BoxAndWhiskerSeries;
        /**
         * `rangeColumnSeriesModule` is used to add rangeColumn series to the chart.
         */
        rangeColumnSeriesModule: RangeColumnSeries;
        /**
         * hiloSeriesModule is used to add hilo series in chart
         */
        hiloSeriesModule: HiloSeries;
        /**
         * hiloOpenCloseSeriesModule is used to add hilo series in chart
         */
        hiloOpenCloseSeriesModule: HiloOpenCloseSeries;
        /**
         * `waterfallSeries` is used to add waterfall series in chart.
         */
        waterfallSeriesModule: WaterfallSeries;
        /**
         * `bubbleSeries` is used to add bubble series in chart.
         */
        bubbleSeriesModule: BubbleSeries;
        /**
         * `rangeAreaSeriesModule` is used to add rangeArea series in chart.
         */
        rangeAreaSeriesModule: RangeAreaSeries;
        /**
         * `tooltipModule` is used to manipulate and add tooltip to the series.
         */
        tooltipModule: ej.popups.Tooltip;
        /**
         * `crosshairModule` is used to manipulate and add crosshair to the chart.
         */
        crosshairModule: Crosshair;
        /**
         * `errorBarModule` is used to manipulate and add errorBar for series.
         */
        errorBarModule: ErrorBar;
        /**
         * `dataLabelModule` is used to manipulate and add data label to the series.
         */
        dataLabelModule: DataLabel;
        /**
         * `datetimeModule` is used to manipulate and add dateTime axis to the chart.
         */
        dateTimeModule: DateTime;
        /**
         * `categoryModule` is used to manipulate and add category axis to the chart.
         */
        categoryModule: Category;
        /**
         * `dateTimeCategoryModule` is used to manipulate date time and category axis
         */
        dateTimeCategoryModule: DateTimeCategory;
        /**
         * `logarithmicModule` is used to manipulate and add log axis to the chart.
         */
        logarithmicModule: Logarithmic;
        /**
         * `legendModule` is used to manipulate and add legend to the chart.
         */
        legendModule: Legend;
        /**
         * `zoomModule` is used to manipulate and add zooming to the chart.
         */
        zoomModule: Zoom;
        /**
         * `selectionModule` is used to manipulate and add selection to the chart.
         */
        selectionModule: Selection;
        /**
         * `annotationModule` is used to manipulate and add annotation in chart.
         */
        annotationModule: ChartAnnotation;
        /**
         * `stripLineModule` is used to manipulate and add stripLine in chart.
         */
        stripLineModule: StripLine;
        /**
         * `multiLevelLabelModule` is used to manipulate and add multiLevelLabel in chart.
         */
        multiLevelLabelModule: MultiLevelLabel;
        /**
         * 'TrendlineModule' is used to predict the market trend using trendlines
         */
        trendLineModule: Trendlines;
        /**
         * `sMAIndicatorModule` is used to predict the market trend using SMA approach
         */
        sMAIndicatorModule: SmaIndicator;
        /**
         * `eMAIndicatorModule` is used to predict the market trend using EMA approach
         */
        eMAIndicatorModule: EmaIndicator;
        /**
         * `tMAIndicatorModule` is used to predict the market trend using TMA approach
         */
        tMAIndicatorModule: TmaIndicator;
        /**
         * `accumulationDistributionIndicatorModule` is used to predict the market trend using Accumulation Distribution approach
         */
        accumulationDistributionIndicatorModule: AccumulationDistributionIndicator;
        /**
         * `atrIndicatorModule` is used to predict the market trend using ATR approach
         */
        atrIndicatorModule: AtrIndicator;
        /**
         * `rSIIndicatorModule` is used to predict the market trend using RSI approach
         */
        rsiIndicatorModule: RsiIndicator;
        /**
         * `macdIndicatorModule` is used to predict the market trend using Macd approach
         */
        macdIndicatorModule: MacdIndicator;
        /**
         * `stochasticIndicatorModule` is used to predict the market trend using Stochastic approach
         */
        stochasticIndicatorModule: StochasticIndicator;
        /**
         * `momentumIndicatorModule` is used to predict the market trend using Momentum approach
         */
        momentumIndicatorModule: MomentumIndicator;
        /**
         * `bollingerBandsModule` is used to predict the market trend using Bollinger approach
         */
        bollingerBandsModule: BollingerBands;
        /**
         * The width of the chart as a string accepts input as both like '100px' or '100%'.
         * If specified as '100%, chart renders to the full width of its parent element.
         * @default null
         */
        width: string;
        /**
         * The height of the chart as a string accepts input both as '100px' or '100%'.
         * If specified as '100%, chart renders to the full height of its parent element.
         * @default null
         */
        height: string;
        /**
         * Title of the chart
         * @default ''
         */
        title: string;
        /**
         * Options for customizing the title of the Chart.
         */
        titleStyle: FontModel;
        /**
         *  Options to customize left, right, top and bottom margins of the chart.
         */
        margin: MarginModel;
        /**
         * Options for customizing the color and width of the chart border.
         */
        border: BorderModel;
        /**
         * The background color of the chart that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        background: string;
        /**
         * Options for configuring the border and background of the chart area.
         */
        chartArea: ChartAreaModel;
        /**
         * Options to configure the horizontal axis.
         */
        primaryXAxis: AxisModel;
        /**
         * Options to configure the vertical axis.
         */
        primaryYAxis: AxisModel;
        /**
         * Options to split Chart into multiple plotting areas horizontally.
         * Each object in the collection represents a plotting area in the Chart.
         */
        rows: RowModel[];
        /**
         * Options to split chart into multiple plotting areas vertically.
         * Each object in the collection represents a plotting area in the chart.
         */
        columns: ColumnModel[];
        /**
         * Secondary axis collection for the chart.
         */
        axes: AxisModel[];
        /**
         * The configuration for series in the chart.
         */
        series: SeriesModel[];
        /**
         * The configuration for annotation in chart.
         */
        annotations: ChartAnnotationSettingsModel[];
        /**
         * Palette for the chart series.
         * @default []
         */
        palettes: string[];
        /**
         * Specifies the theme for the chart.
         * @default 'Material'
         */
        theme: ChartTheme;
        /**
         * Options for customizing the tooltip of the chart.
         */
        tooltip: TooltipSettingsModel;
        /**
         * Options for customizing the crosshair of the chart.
         */
        crosshair: CrosshairSettingsModel;
        /**
         * Options for customizing the legend of the chart.
         */
        legendSettings: LegendSettingsModel;
        /**
         * Options to enable the zooming feature in the chart.
         */
        zoomSettings: ZoomSettingsModel;
        /**
         * Specifies whether series or data point has to be selected. They are,
         * * none: Disables the selection.
         * * series: selects a series.
         * * point: selects a point.
         * * cluster: selects a cluster of point
         * * dragXY: selects points by dragging with respect to both horizontal and vertical axes
         * * dragX: selects points by dragging with respect to horizontal axis.
         * * dragY: selects points by dragging with respect to vertical axis.
         * @default None
         */
        selectionMode: SelectionMode;
        /**
         * If set true, enables the multi selection in chart. It requires `selectionMode` to be `Point` | `Series` | or `Cluster`.
         * @default false
         */
        isMultiSelect: boolean;
        /**
         * Specifies the point indexes to be selected while loading a chart.
         * It requires `selectionMode` to be `Point` | `Series` | or `Cluster`.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *   selectionMode: 'Point',
         *   selectedDataIndexes: [ { series: 0, point: 1},
         *                          { series: 2, point: 3} ],
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default []
         */
        selectedDataIndexes: IndexesModel[];
        /**
         * Specifies whether a grouping separator should be used for a number.
         * @default false
         */
        useGroupingSeparator: boolean;
        /**
         * It specifies whether the chart should be render in transposed manner or not.
         * @default false
         */
        isTransposed: boolean;
        /**
         * Defines the collection of technical indicators, that are used in financial markets
         */
        indicators: TechnicalIndicatorModel[];
        /**
         * Description for chart.
         * @default null
         */
        description: string;
        /**
         * TabIndex value for the chart.
         * @default 1
         */
        tabIndex: number;
        /**
         * To enable the side by side placing the points for column type series.
         * @default true
         */
        enableSideBySidePlacement: boolean;
        /**
         * Triggers before the annotation gets rendered.
         * @event
         */
        annotationRender: ej.base.EmitType<IAnnotationRenderEventArgs>;
        /**
         * Triggers before the prints gets started.
         * @event
         */
        beforePrint: ej.base.EmitType<IPrintEventArgs>;
        /**
         * Triggers after chart load.
         * @event
         */
        loaded: ej.base.EmitType<ILoadedEventArgs>;
        /**
         * Triggers before chart load.
         * @event
         */
        load: ej.base.EmitType<ILoadedEventArgs>;
        /**
         * Triggers after animation is completed for the series.
         * @event
         */
        animationComplete: ej.base.EmitType<IAnimationCompleteEventArgs>;
        /**
         * Triggers before the legend is rendered.
         * @event
         */
        legendRender: ej.base.EmitType<ILegendRenderEventArgs>;
        /**
         * Triggers before the data label for series is rendered.
         * @event
         */
        textRender: ej.base.EmitType<ITextRenderEventArgs>;
        /**
         * Triggers before each points for the series is rendered.
         * @event
         */
        pointRender: ej.base.EmitType<IPointRenderEventArgs>;
        /**
         * Triggers before the series is rendered.
         * @event
         */
        seriesRender: ej.base.EmitType<ISeriesRenderEventArgs>;
        /**
         * Triggers before each axis label is rendered.
         * @event
         */
        axisLabelRender: ej.base.EmitType<IAxisLabelRenderEventArgs>;
        /**
         * Triggers before each axis range is rendered.
         * @event
         */
        axisRangeCalculated: ej.base.EmitType<IAxisRangeCalculatedEventArgs>;
        /**
         * Triggers before each axis multi label is rendered.
         * @event
         */
        axisMultiLabelRender: ej.base.EmitType<IAxisMultiLabelRenderEventArgs>;
        /**
         * Triggers before the tooltip for series is rendered.
         * @event
         */
        tooltipRender: ej.base.EmitType<ITooltipRenderEventArgs>;
        /**
         * Triggers on hovering the chart.
         * @event
         */
        chartMouseMove: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on clicking the chart.
         * @event
         */
        chartMouseClick: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on point click.
         * @event
         */
        pointClick: ej.base.EmitType<IPointEventArgs>;
        /**
         * Triggers on point move.
         * @event
         */
        pointMove: ej.base.EmitType<IPointEventArgs>;
        /**
         * Triggers when cursor leaves the chart.
         * @event
         */
        chartMouseLeave: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on mouse down.
         * @event
         */
        chartMouseDown: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on mouse up.
         * @event
         */
        chartMouseUp: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers after the drag selection is completed.
         * @event
         */
        dragComplete: ej.base.EmitType<IDragCompleteEventArgs>;
        /**
         * Triggers after the zoom selection is completed.
         * @event
         */
        zoomComplete: ej.base.EmitType<IZoomCompleteEventArgs>;
        /**
         * Defines the currencyCode format of the chart
         * @private
         */
        private currencyCode;
        private htmlObject;
        private getElement;
        private elementSize;
        private isLegend;
        /**
         * localization object
         * @private
         */
        localeObject: ej.base.L10n;
        /**
         * It contains default values of localization values
         */
        private defaultLocalConstants;
        /**
         * Gets the current visible axis of the Chart.
         * @hidden
         */
        axisCollections: Axis[];
        /**
         * Gets the current visible series of the Chart.
         * @hidden
         */
        visibleSeries: Series[];
        /**
         * Render panel for chart.
         * @hidden
         */
        chartAxisLayoutPanel: CartesianAxisLayoutPanel | PolarRadarPanel;
        /**
         * Gets all the horizontal axis of the Chart.
         * @hidden
         */
        horizontalAxes: Axis[];
        /**
         * Gets all the vertical axis of the Chart.
         * @hidden
         */
        verticalAxes: Axis[];
        /**
         * Gets the inverted chart.
         * @hidden
         */
        requireInvertedAxis: boolean;
        /** @private */
        svgObject: Element;
        /** @private */
        isTouch: boolean;
        /** @private */
        renderer: ej.base.SvgRenderer;
        /** @private */
        initialClipRect: Rect;
        /** @private */
        seriesElements: Element;
        /** @private */
        indicatorElements: Element;
        /** @private */
        trendLineElements: Element;
        /** @private */
        visibleSeriesCount: number;
        /** @private */
        intl: ej.base.Internationalization;
        /** @private */
        dataLabelCollections: Rect[];
        /** @private */
        dataLabelElements: Element;
        /** @private */
        mouseX: number;
        /** @private */
        mouseY: number;
        /** @private */
        animateSeries: boolean;
        /** @private */
        availableSize: Size;
        /** @private */
        delayRedraw: boolean;
        /** @private */
        isDoubleTap: boolean;
        /** @private */
        mouseDownX: number;
        /** @private */
        mouseDownY: number;
        /** @private */
        previousMouseMoveX: number;
        /** @private */
        previousMouseMoveY: number;
        /** @private */
        private threshold;
        /** @private */
        isChartDrag: boolean;
        private resizeTo;
        /** @private */
        disableTrackTooltip: boolean;
        /** @private */
        startMove: boolean;
        /** @private */
        yAxisElements: Element;
        /** @private */
        radius: number;
        /** @private */
        chartAreaType: string;
        /**
         * `markerModule` is used to manipulate and add marker to the series.
         * @private
         */
        markerRender: Marker;
        private titleCollection;
        /** @private */
        themeStyle: IThemeStyle;
        /**
         * Constructor for creating the widget
         * @hidden
         */
        constructor(options?: ChartModel, element?: string | HTMLElement);
        /**
         * Initialize the event handler.
         */
        protected preRender(): void;
        private initPrivateVariable();
        /**
         * To Initialize the control rendering.
         */
        protected render(): void;
        /**
         * Gets the localized label by locale keyword.
         * @param  {string} key
         * @return {string}
         */
        getLocalizedLabel(key: string): string;
        /**
         * Refresh the chart bounds.
         * @private
         */
        refreshBound(): void;
        private renderElements();
        /**
         * To render the legend
         */
        private renderAxes();
        /**
         * To render the legend
         */
        private renderLegend();
        /**
         * To set the left and top position for data label template for center aligned chart
         */
        private setSecondaryElementPosition();
        private initializeModuleElements();
        private hasTrendlines();
        private renderSeriesElements(axisElement);
        private renderSeries();
        private initializeIndicator();
        private initializeTrendLine();
        private appendElementsAfterSeries(axisElement);
        private applyZoomkit();
        private renderAnnotation();
        private performSelection();
        private processData();
        private initializeDataModule(series);
        private calculateBounds();
        /**
         * Handles the print method for chart control.
         */
        print(id?: string[] | string | Element): void;
        /**
         * Handles the export method for chart control.
         * @param type
         * @param fileName
         */
        export(type: ExportType, fileName: string, orientation?: ej.pdfexport.PdfPageOrientation): void;
        /**
         * Defines the trendline initialization
         */
        private initTrendLines();
        private calculateAreaType();
        private calculateVisibleAxis();
        private initAxis(series, axis, isSeries);
        private initTechnicalIndicators();
        /** @private */
        refreshTechnicalIndicator(series: SeriesBase): void;
        private calculateVisibleSeries();
        private renderTitle();
        private renderBorder();
        private renderAreaBorder();
        /**
         * To add series for the chart
         * @param {SeriesModel[]} seriesCollection - Defines the series collection to be added in chart.
         * @return {void}.
         */
        addSeries(seriesCollection: SeriesModel[]): void;
        /**
         * To Remove series for the chart
         * @param index - Defines the series index to be remove in chart series
         * @return {void}
         */
        removeSeries(index: number): void;
        /**
         * To destroy the widget
         * @method destroy
         * @return {void}.
         * @member of Chart
         */
        destroy(): void;
        /**
         * Get component name
         */
        getModuleName(): string;
        /**
         * Get the properties to be maintained in the persisted state.
         * @private
         */
        getPersistData(): string;
        /**
         * Method to create SVG element.
         */
        private createChartSvg();
        /**
         * Method to bind events for chart
         */
        private unWireEvents();
        private wireEvents();
        private chartRightClick(event);
        private setStyle(element);
        /**
         * Finds the orientation.
         * @return {boolean}
         * @private
         */
        isOrientation(): boolean;
        /**
         * Handles the long press on chart.
         * @return {boolean}
         * @private
         */
        longPress(e?: ej.base.TapEventArgs): boolean;
        /**
         * To find mouse x, y for aligned chart element svg position
         */
        private setMouseXY(pageX, pageY);
        /**
         * Handles the chart resize.
         * @return {boolean}
         * @private
         */
        chartResize(e: Event): boolean;
        /**
         * Handles the mouse move.
         * @return {boolean}
         * @private
         */
        mouseMove(e: PointerEvent): boolean;
        /**
         * Handles the mouse leave.
         * @return {boolean}
         * @private
         */
        mouseLeave(e: PointerEvent): boolean;
        /**
         * Handles the mouse leave on chart.
         * @return {boolean}
         * @private
         */
        chartOnMouseLeave(e: PointerEvent | TouchEvent): boolean;
        /**
         * Handles the mouse click on chart.
         * @return {boolean}
         * @private
         */
        chartOnMouseClick(e: PointerEvent | TouchEvent): boolean;
        private triggerPointEvent(event);
        /**
         * Handles the mouse move on chart.
         * @return {boolean}
         * @private
         */
        chartOnMouseMove(e: PointerEvent | TouchEvent): boolean;
        private titleTooltip(event, x, y, isTouch?);
        private axisTooltip(event, x, y, isTouch?);
        private findAxisLabel(text);
        /**
         * Handles the mouse down on chart.
         * @return {boolean}
         * @private
         */
        chartOnMouseDown(e: PointerEvent): boolean;
        /**
         * Handles the mouse up.
         * @return {boolean}
         * @private
         */
        mouseEnd(e: PointerEvent): boolean;
        /**
         * Handles the mouse up.
         * @return {boolean}
         * @private
         */
        chartOnMouseUp(e: PointerEvent | TouchEvent): boolean;
        /**
         * Method to set culture for chart
         */
        private setCulture();
        /**
         * Method to set the annotation content dynamically for chart.
         */
        setAnnotationValue(annotationIndex: number, content: string): void;
        /**
         * Method to set locale constants
         */
        private setLocaleConstants();
        /**
         * Themeing for chart goes here
         */
        private setTheme();
        /**
         * To provide the array of modules needed for control rendering
         * @return {ej.base.ModuleDeclaration[]}
         * @private
         */
        requiredModules(): ej.base.ModuleDeclaration[];
        private findAxisModule(modules);
        private findIndicatorModules(modules);
        private findTrendLineModules(modules);
        private findStriplineVisibility(striplines);
        /**
         * To Remove the SVG.
         * @return {boolean}
         * @private
         */
        removeSvg(): void;
        private refreshDefinition(definitions);
        /**
         * Refresh the axis default value.
         * @return {boolean}
         * @private
         */
        refreshAxis(): void;
        private axisChange(axis);
        /**
         * Called internally if any of the property value changed.
         * @private
         */
        onPropertyChanged(newProp: ChartModel, oldProp: ChartModel): void;
    }
    
    /**
     * `ChartAnnotation` module handles the annotation for chart.
     */
    export class ChartAnnotation extends AnnotationBase {
        private chart;
        private annotations;
        private parentElement;
        /**
         * Constructor for chart annotation.
         * @private.
         */
        constructor(control: Chart, annotations: ChartAnnotationSettings[]);
        /**
         * Method to render the annotation for chart
         * @param element
         * @private
         */
        renderAnnotations(element: Element): void;
        /**
         * To destroy the annotation.
         * @return {void}
         * @private
         */
        destroy(control: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * Common axis classes
     * @private
     */
    export class NiceInterval extends Double {
        /**
         * Method to calculate numeric datetime interval
         */
        calculateDateTimeNiceInterval(axis: Axis, size: Size, start: number, end: number): number;
        /**
         * To get the skeleton for the DateTime axis.
         * @return {string}
         *  @private
         */
        getSkeleton(axis: Axis): string;
    }
    
    /**
     * Interface for a class Row
     */
    export interface RowModel {
    
        /**
         * The height of the row as a string accept input both as '100px' and '100%'.
         * If specified as '100%, row renders to the full height of its chart.
         * @default '100%'
         */
    
        height?: string;
    
        /**
         * Options to customize the border of the rows.
         */
    
        border?: BorderModel;
    
    }
    
    /**
     * Interface for a class Column
     */
    export interface ColumnModel {
    
        /**
         * The width of the column as a string accepts input both as like '100px' or '100%'.
         * If specified as '100%, column renders to the full width of its chart.
         * @default '100%'
         */
    
        width?: string;
    
        /**
         * Options to customize the border of the columns.
         */
    
        border?: BorderModel;
    
    }
    
    /**
     * Interface for a class MajorGridLines
     */
    export interface MajorGridLinesModel {
    
        /**
         * The width of the line in pixels.
         * @default 1
         */
    
        width?: number;
    
        /**
         * The dash array of the grid lines.
         * @default ''
         */
    
        dashArray?: string;
    
        /**
         * The color of the major grid line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
    
        color?: string;
    
    }
    
    /**
     * Interface for a class MinorGridLines
     */
    export interface MinorGridLinesModel {
    
        /**
         * The width of the line in pixels.
         * @default 0.7
         */
    
        width?: number;
    
        /**
         * The dash array of grid lines.
         * @default ''
         */
    
        dashArray?: string;
    
        /**
         * The color of the minor grid line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
    
        color?: string;
    
    }
    
    /**
     * Interface for a class AxisLine
     */
    export interface AxisLineModel {
    
        /**
         * The width of the line in pixels.
         * @default 1
         */
    
        width?: number;
    
        /**
         * The dash array of the axis line.
         * @default ''
         */
    
        dashArray?: string;
    
        /**
         * The color of the axis line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
    
        color?: string;
    
    }
    
    /**
     * Interface for a class MajorTickLines
     */
    export interface MajorTickLinesModel {
    
        /**
         * The width of the tick lines in pixels.
         * @default 1
         */
    
        width?: number;
    
        /**
         * The height of the ticks in pixels.
         * @default 5
         */
    
        height?: number;
    
        /**
         * The color of the major tick line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
    
        color?: string;
    
    }
    
    /**
     * Interface for a class MinorTickLines
     */
    export interface MinorTickLinesModel {
    
        /**
         * The width of the tick line in pixels.
         * @default 0.7
         */
    
        width?: number;
    
        /**
         * The height of the ticks in pixels.
         * @default 5
         */
    
        height?: number;
    
        /**
         * The color of the minor tick line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
    
        color?: string;
    
    }
    
    /**
     * Interface for a class CrosshairTooltip
     */
    export interface CrosshairTooltipModel {
    
        /**
         * If set to true, crosshair ToolTip will be visible.
         *  @default false
         */
    
        enable?: Boolean;
    
        /**
         * The fill color of the ToolTip accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
    
        fill?: string;
    
        /**
         * Options to customize the crosshair ToolTip text.
         */
    
        textStyle?: FontModel;
    
    }
    
    /**
     * Interface for a class Axis
     */
    export interface AxisModel {
    
        /**
         * Options to customize the axis label.
         */
    
        labelStyle?: FontModel;
    
        /**
         * Options to customize the crosshair ToolTip.
         */
    
        crosshairTooltip?: CrosshairTooltipModel;
    
        /**
         * Specifies the title of an axis.
         * @default ''
         */
    
        title?: string;
    
        /**
         * Options for customizing the axis title.
         */
    
        titleStyle?: FontModel;
    
        /**
         * Used to format the axis label that accepts any global string format like 'C', 'n1', 'P' etc.
         * It also accepts placeholder like '{value}°C' in which value represent the axis label, e.g, 20°C.
         * @default ''
         */
    
        labelFormat?: string;
    
        /**
         * Specifies the skeleton format in which the dateTime format will process.
         * @default ''
         */
    
        skeleton?: string;
    
        /**
         * It specifies the type of format to be used in dateTime format process.
         * @default 'DateTime'
         */
    
        skeletonType?: SkeletonType;
    
        /**
         * Left and right padding for the plot area in pixels.
         * @default 0
         */
    
        plotOffset?: number;
    
        /**
         * Specifies indexed category  axis.
         * @default false
         */
    
        isIndexed?: boolean;
    
        /**
         * The base value for logarithmic axis. It requires `valueType` to be `Logarithmic`.
         * @default 10
         */
        logBase?: number;
    
        /**
         * Specifies the index of the column where the axis is associated,
         * when the chart area is divided into multiple plot areas by using `columns`.
         * ```html 
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     columns: [{ width: '50%' },
         *               { width: '50%' }],
         *     axes: [{
         *                name: 'xAxis 1',
         *                columnIndex: 1,
         *     }],
         * ... 
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default 0
         */
    
        columnIndex?: number;
    
        /**
         * Specifies the index of the row where the axis is associated, when the chart area is divided into multiple plot areas by using `rows`.
         * ```html 
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     rows: [{ height: '50%' },
         *            { height: '50%' }],
         *     axes: [{
         *                name: 'yAxis 1',
         *                rowIndex: 1,
         *      }],
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default 0
         */
    
        rowIndex?: number;
    
        /**
         * Specifies the number of `columns` or `rows` an axis has to span horizontally or vertically.
         * @default 1
         */
    
        span?: number;
    
        /**
         * With this property, you can request axis to calculate intervals approximately equal to your specified interval.
         * @default null
         */
    
        desiredIntervals?: number;
    
        /**
         * The maximum number of label count per 100 pixels with respect to the axis length.
         * @default 3
         */
    
        maximumLabels?: number;
    
        /**
         * The axis is scaled by this factor. When zoomFactor is 0.5, the chart is scaled by 200% along this axis. Value ranges from 0 to 1.
         * @default 1
         */
    
        zoomFactor?: number;
    
        /**
         * Position of the zoomed axis. Value ranges from 0 to 1.
         * @default 0
         */
    
        zoomPosition?: number;
    
        /**
         * If set to true, the axis will render at the opposite side of its default position.
         * @default false
         */
    
        opposedPosition?: boolean;
    
        /**
         * If set to true, axis interval will be calculated automatically with respect to the zoomed range.
         * @default true
         */
    
        enableAutoIntervalOnZooming?: boolean;
    
        /**
         * Specifies the padding for the axis range in terms of interval.They are,
         * * none: Padding cannot be applied to the axis.
         * * normal: Padding is applied to the axis based on the range calculation.
         * * additional: Interval of the axis is added as padding to the minimum and maximum values of the range.
         * * round: Axis range is rounded to the nearest possible value divided by the interval.
         * @default 'Auto'
         */
    
        rangePadding?: ChartRangePadding;
    
        /**
         * Specifies the type of data the axis is handling.
         * * Double:  Renders a numeric axis.
         * * DateTime: Renders a dateTime axis.
         * * Category: Renders a category axis.
         * * Logarithmic: Renders a log axis.
         * @default 'Double'
         */
    
        valueType?: ValueType;
    
        /**
         * Specifies the position of labels at the edge of the axis.They are,
         * * None: No action will be performed.
         * * Hide: Edge label will be hidden.
         * * Shift: Shifts the edge labels.
         * @default 'None'
         */
    
        edgeLabelPlacement?: EdgeLabelPlacement;
    
        /**
         * Specifies the types like `Years`, `Months`, `Days`, `Hours`, `Minutes`, `Seconds` in date time axis.They are,
         * * Auto: Defines the interval of the axis based on data.
         * * Years: Defines the interval of the axis in years.
         * * Months: Defines the interval of the axis in months.
         * * Days: Defines the interval of the axis in days.
         * * Hours: Defines the interval of the axis in hours.
         * * Minutes: Defines the interval of the axis in minutes.
         * @default 'Auto'
         */
    
        intervalType?: IntervalType;
    
        /**
         * Specifies the placement of a label for category axis. They are,
         * * betweenTicks: Renders the label between the ticks.
         * * onTicks: Renders the label on the ticks.
         * @default 'BetweenTicks'
         */
    
        labelPlacement?: LabelPlacement;
    
        /**
         * Specifies the placement of a ticks to the axis line. They are,
         * * inside: Renders the ticks inside to the axis line.
         * * outside: Renders the ticks outside to the axis line.
         * @default 'Outside'
         */
    
        tickPosition?: AxisPosition;
    
        /**
         * Specifies the placement of a labels to the axis line. They are,
         * * inside: Renders the labels inside to the axis line.
         * * outside: Renders the labels outside to the axis line.
         * @default 'Outside'
         */
    
        labelPosition?: AxisPosition;
    
        /**
         * Unique identifier of an axis.
         * To associate an axis with the series, set this name to the xAxisName/yAxisName properties of the series.
         * @default ''
         */
    
        name?: string;
    
        /**
         * If set to true, axis label will be visible.
         * @default true
         */
    
        visible?: boolean;
    
        /**
         * Specifies the number of minor ticks per interval.
         * @default 0
         */
    
        minorTicksPerInterval?: number;
    
        /**
         * The angle to which the axis label gets rotated.
         * @default 0
         */
    
        labelRotation?: number;
    
        /**
         * Specifies the value at which the axis line has to be intersect with the vertical axis or vice versa.
         * @default null
         */
    
        crossesAt?: Object;
    
        /**
         * Specifies whether axis elements like axis labels, axis title, etc has to be crossed with axis line
         * @default true
         */
    
        placeNextToAxisLine?: boolean;
    
        /**
         * Specifies axis name with which the axis line has to be crossed
         * @default null
         */
    
        crossesInAxis?: string;
    
        /**
         * Specifies the minimum range of an axis.
         * @default null
         */
    
        minimum?: Object;
    
        /**
         * Specifies the maximum range of an axis.
         * @default null
         */
    
        maximum?: Object;
    
        /**
         * Specifies the interval for an axis.
         * @default null
         */
    
        interval?: number;
    
        /**
         * Options for customizing major tick lines.
         */
    
        majorTickLines?: MajorTickLinesModel;
    
        /**
         * Options for customizing minor tick lines.
         */
    
        minorTickLines?: MinorTickLinesModel;
    
        /**
         * Options for customizing major grid lines.
         */
    
        majorGridLines?: MajorGridLinesModel;
    
        /**
         * Options for customizing minor grid lines.
         */
    
        minorGridLines?: MinorGridLinesModel;
    
        /**
         * Options for customizing axis lines.
         */
    
        lineStyle?: AxisLineModel;
    
        /**
         * Specifies the actions like `Hide`, `Rotate45`, and `Rotate90` when the axis labels intersect with each other.They are,
         * * None: Shows all the labels.
         * * Hide: Hides the label when it intersects.
         * * Rotate45: Rotates the label to 45 degree when it intersects.
         * * Rotate90: Rotates the label to 90 degree when it intersects.
         * @default Hide
         */
    
        labelIntersectAction?: LabelIntersectAction;
    
        /**
         * It specifies whether the axis to be rendered in inversed manner or not.
         * @default false
         */
        isInversed?: boolean;
    
        /**
         * The polar radar radius position.
         * @default 100
         */
    
        coefficient?: number;
    
        /**
         * The start angle for the series.
         * @default 0
         */
    
        startAngle?: number;
    
        /**
         * Description for axis and its element.
         * @default null
         */
        description?: string;
    
        /**
         * TabIndex value for the axis.
         * @default 2
         */
        tabIndex?: number;
    
        /**
         * Specifies the stripLine collection for the axis
         */
        stripLines?: StripLineSettingsModel[];
    
        /**
         * Specifies the multi level labels collection for the axis
         */
        multiLevelLabels?: MultiLevelLabelsModel[];
    
        /**
         * Border of the multi level labels.
         */
        border?: LabelBorderModel;
    
    }
    
    /**
     * Interface for a class VisibleLabels
     * @private
     */
    export interface VisibleLabelsModel {
    
    }
    
    /**
     * Configures the `rows` of the chart.
     */
    export class Row extends ej.base.ChildProperty<Row> {
        /**
         * The height of the row as a string accept input both as '100px' and '100%'.
         * If specified as '100%, row renders to the full height of its chart.
         * @default '100%'
         */
        height: string;
        /**
         * Options to customize the border of the rows.
         */
        border: BorderModel;
        /** @private */
        axes: Axis[];
        /** @private */
        computedHeight: number;
        /** @private */
        computedTop: number;
        /** @private */
        nearSizes: number[];
        /** @private */
        farSizes: number[];
        /**
         * Measure the row size
         * @return {void}
         * @private
         */
        computeSize(axis: Axis, clipRect: Rect): void;
    }
    /**
     * Configures the `columns` of the chart.
     */
    export class Column extends ej.base.ChildProperty<Column> {
        /**
         * The width of the column as a string accepts input both as like '100px' or '100%'.
         * If specified as '100%, column renders to the full width of its chart.
         * @default '100%'
         */
        width: string;
        /**
         * Options to customize the border of the columns.
         */
        border: BorderModel;
        /** @private */
        axes: Axis[];
        /** @private */
        computedWidth: number;
        /** @private */
        computedLeft: number;
        /** @private */
        nearSizes: number[];
        /** @private */
        farSizes: number[];
        /** @private */
        private padding;
        /**
         * Measure the column size
         * @return {void}
         * @private
         */
        computeSize(axis: Axis, clipRect: Rect): void;
    }
    /**
     * Configures the major grid lines in the `axis`.
     */
    export class MajorGridLines extends ej.base.ChildProperty<MajorGridLines> {
        /**
         * The width of the line in pixels.
         * @default 1
         */
        width: number;
        /**
         * The dash array of the grid lines.
         * @default ''
         */
        dashArray: string;
        /**
         * The color of the major grid line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        color: string;
    }
    /**
     * Configures the minor grid lines in the `axis`.
     */
    export class MinorGridLines extends ej.base.ChildProperty<MinorGridLines> {
        /**
         * The width of the line in pixels.
         * @default 0.7
         */
        width: number;
        /**
         * The dash array of grid lines.
         * @default ''
         */
        dashArray: string;
        /**
         * The color of the minor grid line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        color: string;
    }
    /**
     * Configures the axis line of a chart.
     */
    export class AxisLine extends ej.base.ChildProperty<AxisLine> {
        /**
         * The width of the line in pixels.
         * @default 1
         */
        width: number;
        /**
         * The dash array of the axis line.
         * @default ''
         */
        dashArray: string;
        /**
         * The color of the axis line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        color: string;
    }
    /**
     * Configures the major tick lines.
     */
    export class MajorTickLines extends ej.base.ChildProperty<MajorTickLines> {
        /**
         * The width of the tick lines in pixels.
         * @default 1
         */
        width: number;
        /**
         * The height of the ticks in pixels.
         * @default 5
         */
        height: number;
        /**
         * The color of the major tick line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        color: string;
    }
    /**
     * Configures the minor tick lines.
     */
    export class MinorTickLines extends ej.base.ChildProperty<MinorTickLines> {
        /**
         * The width of the tick line in pixels.
         * @default 0.7
         */
        width: number;
        /**
         * The height of the ticks in pixels.
         * @default 5
         */
        height: number;
        /**
         * The color of the minor tick line that accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        color: string;
    }
    /**
     * Configures the crosshair ToolTip.
     */
    export class CrosshairTooltip extends ej.base.ChildProperty<CrosshairTooltip> {
        /**
         * If set to true, crosshair ToolTip will be visible.
         *  @default false
         */
        enable: Boolean;
        /**
         * The fill color of the ToolTip accepts value in hex and rgba as a valid CSS color string.
         * @default null
         */
        fill: string;
        /**
         * Options to customize the crosshair ToolTip text.
         */
        textStyle: FontModel;
    }
    /**
     * Configures the axes in the chart.
     */
    export class Axis extends ej.base.ChildProperty<Axis> {
        /**
         * Options to customize the axis label.
         */
        labelStyle: FontModel;
        /**
         * Options to customize the crosshair ToolTip.
         */
        crosshairTooltip: CrosshairTooltipModel;
        /**
         * Specifies the title of an axis.
         * @default ''
         */
        title: string;
        /**
         * Options for customizing the axis title.
         */
        titleStyle: FontModel;
        /**
         * Used to format the axis label that accepts any global string format like 'C', 'n1', 'P' etc.
         * It also accepts placeholder like '{value}°C' in which value represent the axis label, e.g, 20°C.
         * @default ''
         */
        labelFormat: string;
        /**
         * Specifies the skeleton format in which the dateTime format will process.
         * @default ''
         */
        skeleton: string;
        /**
         * It specifies the type of format to be used in dateTime format process.
         * @default 'DateTime'
         */
        skeletonType: SkeletonType;
        /**
         * Left and right padding for the plot area in pixels.
         * @default 0
         */
        plotOffset: number;
        /**
         * Specifies indexed category  axis.
         * @default false
         */
        isIndexed: boolean;
        /**
         * The base value for logarithmic axis. It requires `valueType` to be `Logarithmic`.
         * @default 10
         */
        logBase: number;
        /**
         * Specifies the index of the column where the axis is associated,
         * when the chart area is divided into multiple plot areas by using `columns`.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     columns: [{ width: '50%' },
         *               { width: '50%' }],
         *     axes: [{
         *                name: 'xAxis 1',
         *                columnIndex: 1,
         *     }],
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default 0
         */
        columnIndex: number;
        /**
         * Specifies the index of the row where the axis is associated, when the chart area is divided into multiple plot areas by using `rows`.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     rows: [{ height: '50%' },
         *            { height: '50%' }],
         *     axes: [{
         *                name: 'yAxis 1',
         *                rowIndex: 1,
         *      }],
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default 0
         */
        rowIndex: number;
        /**
         * Specifies the number of `columns` or `rows` an axis has to span horizontally or vertically.
         * @default 1
         */
        span: number;
        /**
         * With this property, you can request axis to calculate intervals approximately equal to your specified interval.
         * @default null
         */
        desiredIntervals: number;
        /**
         * The maximum number of label count per 100 pixels with respect to the axis length.
         * @default 3
         */
        maximumLabels: number;
        /**
         * The axis is scaled by this factor. When zoomFactor is 0.5, the chart is scaled by 200% along this axis. Value ranges from 0 to 1.
         * @default 1
         */
        zoomFactor: number;
        /**
         * Position of the zoomed axis. Value ranges from 0 to 1.
         * @default 0
         */
        zoomPosition: number;
        /**
         * If set to true, the axis will render at the opposite side of its default position.
         * @default false
         */
        opposedPosition: boolean;
        /**
         * If set to true, axis interval will be calculated automatically with respect to the zoomed range.
         * @default true
         */
        enableAutoIntervalOnZooming: boolean;
        /**
         * Specifies the padding for the axis range in terms of interval.They are,
         * * none: Padding cannot be applied to the axis.
         * * normal: Padding is applied to the axis based on the range calculation.
         * * additional: Interval of the axis is added as padding to the minimum and maximum values of the range.
         * * round: Axis range is rounded to the nearest possible value divided by the interval.
         * @default 'Auto'
         */
        rangePadding: ChartRangePadding;
        /**
         * Specifies the type of data the axis is handling.
         * * Double:  Renders a numeric axis.
         * * DateTime: Renders a dateTime axis.
         * * Category: Renders a category axis.
         * * Logarithmic: Renders a log axis.
         * @default 'Double'
         */
        valueType: ValueType;
        /**
         * Specifies the position of labels at the edge of the axis.They are,
         * * None: No action will be performed.
         * * Hide: Edge label will be hidden.
         * * Shift: Shifts the edge labels.
         * @default 'None'
         */
        edgeLabelPlacement: EdgeLabelPlacement;
        /**
         * Specifies the types like `Years`, `Months`, `Days`, `Hours`, `Minutes`, `Seconds` in date time axis.They are,
         * * Auto: Defines the interval of the axis based on data.
         * * Years: Defines the interval of the axis in years.
         * * Months: Defines the interval of the axis in months.
         * * Days: Defines the interval of the axis in days.
         * * Hours: Defines the interval of the axis in hours.
         * * Minutes: Defines the interval of the axis in minutes.
         * @default 'Auto'
         */
        intervalType: IntervalType;
        /**
         * Specifies the placement of a label for category axis. They are,
         * * betweenTicks: Renders the label between the ticks.
         * * onTicks: Renders the label on the ticks.
         * @default 'BetweenTicks'
         */
        labelPlacement: LabelPlacement;
        /**
         * Specifies the placement of a ticks to the axis line. They are,
         * * inside: Renders the ticks inside to the axis line.
         * * outside: Renders the ticks outside to the axis line.
         * @default 'Outside'
         */
        tickPosition: AxisPosition;
        /**
         * Specifies the placement of a labels to the axis line. They are,
         * * inside: Renders the labels inside to the axis line.
         * * outside: Renders the labels outside to the axis line.
         * @default 'Outside'
         */
        labelPosition: AxisPosition;
        /**
         * Unique identifier of an axis.
         * To associate an axis with the series, set this name to the xAxisName/yAxisName properties of the series.
         * @default ''
         */
        name: string;
        /**
         * If set to true, axis label will be visible.
         * @default true
         */
        visible: boolean;
        /**
         * Specifies the number of minor ticks per interval.
         * @default 0
         */
        minorTicksPerInterval: number;
        /**
         * The angle to which the axis label gets rotated.
         * @default 0
         */
        labelRotation: number;
        /**
         * Specifies the value at which the axis line has to be intersect with the vertical axis or vice versa.
         * @default null
         */
        crossesAt: Object;
        /**
         * Specifies whether axis elements like axis labels, axis title, etc has to be crossed with axis line
         * @default true
         */
        placeNextToAxisLine: boolean;
        /**
         * Specifies axis name with which the axis line has to be crossed
         * @default null
         */
        crossesInAxis: string;
        /**
         * Specifies the minimum range of an axis.
         * @default null
         */
        minimum: Object;
        /**
         * Specifies the maximum range of an axis.
         * @default null
         */
        maximum: Object;
        /**
         * Specifies the interval for an axis.
         * @default null
         */
        interval: number;
        /**
         * Options for customizing major tick lines.
         */
        majorTickLines: MajorTickLinesModel;
        /**
         * Options for customizing minor tick lines.
         */
        minorTickLines: MinorTickLinesModel;
        /**
         * Options for customizing major grid lines.
         */
        majorGridLines: MajorGridLinesModel;
        /**
         * Options for customizing minor grid lines.
         */
        minorGridLines: MinorGridLinesModel;
        /**
         * Options for customizing axis lines.
         */
        lineStyle: AxisLineModel;
        /**
         * Specifies the actions like `Hide`, `Rotate45`, and `Rotate90` when the axis labels intersect with each other.They are,
         * * None: Shows all the labels.
         * * Hide: Hides the label when it intersects.
         * * Rotate45: Rotates the label to 45 degree when it intersects.
         * * Rotate90: Rotates the label to 90 degree when it intersects.
         * @default Hide
         */
        labelIntersectAction: LabelIntersectAction;
        /**
         * It specifies whether the axis to be rendered in inversed manner or not.
         * @default false
         */
        isInversed: boolean;
        /**
         * The polar radar radius position.
         * @default 100
         */
        coefficient: number;
        /**
         * The start angle for the series.
         * @default 0
         */
        startAngle: number;
        /**
         * Description for axis and its element.
         * @default null
         */
        description: string;
        /**
         * TabIndex value for the axis.
         * @default 2
         */
        tabIndex: number;
        /**
         * Specifies the stripLine collection for the axis
         */
        stripLines: StripLineSettingsModel[];
        /**
         * Specifies the multi level labels collection for the axis
         */
        multiLevelLabels: MultiLevelLabelsModel[];
        /**
         * Border of the multi level labels.
         */
        border: LabelBorderModel;
        /** @private */
        visibleRange: VisibleRangeModel;
        /** @private */
        visibleLabels: VisibleLabels[];
        /** @private */
        actualRange: VisibleRangeModel;
        /** @private */
        series: Series[];
        /** @private */
        doubleRange: DoubleRange;
        /** @private */
        maxLabelSize: Size;
        /** @private */
        rotatedLabel: string;
        /** @private */
        rect: Rect;
        /** @private */
        axisBottomLine: BorderModel;
        /** @private */
        orientation: Orientation;
        /** @private */
        intervalDivs: number[];
        /** @private */
        actualIntervalType: IntervalType;
        /** @private */
        labels: string[];
        /** @private */
        format: Function;
        /** @private */
        baseModule: Double | DateTime | Category | DateTimeCategory;
        /** @private */
        startLabel: string;
        /** @private */
        endLabel: string;
        /** @private */
        angle: number;
        /** @private */
        dateTimeInterval: number;
        /** @private */
        isStack100: boolean;
        /** @private */
        crossInAxis: this;
        /** @private */
        crossAt: number;
        /** @private */
        updatedRect: Rect;
        /** @private */
        multiLevelLabelHeight: number;
        /**
         * The function used to find tick size.
         * @return {number}
         * @private
         */
        findTickSize(crossAxis: Axis): number;
        /**
         * The function used to find axis position.
         * @return {number}
         * @private
         */
        isInside(range: VisibleRangeModel): boolean;
        /**
         * The function used to find label Size.
         * @return {number}
         * @private
         */
        findLabelSize(crossAxis: Axis, innerPadding: number): number;
        /**
         * The function used to find axis position.
         * @return {number}
         * @private
         */
        updateCrossValue(chart: Chart): void;
        private findDifference(crossAxis);
        /**
         * The function used to find whether the range is set.
         * @return {boolean}
         * @private
         */
        setRange(): boolean;
        /**
         * Calculate visible range for axis.
         * @return {void}
         * @private
         */
        calculateVisibleRange(size: Size): void;
        /**
         * Calculate desired interval for the axis.
         * @return {void}
         * @private
         */
        getActualDesiredIntervalsCount(availableSize: Size): number;
        /**
         * Triggers the event.
         * @return {void}
         * @private
         */
        triggerLabelRender(chart: Chart, tempInterval: number, text: string, labelStyle: FontModel): void;
        /**
         * Triggers the event.
         * @return {void}
         * @private
         */
        triggerRangeRender(chart: Chart, minimum: number, maximum: number, interval: number): void;
        /**
         * Calculate padding for the axis.
         * @return {string}
         * @private
         */
        getRangePadding(chart: Chart): string;
        /**
         * Calculate maximum label width for the axis.
         * @return {void}
         * @private
         */
        getMaxLabelWidth(chart: Chart): void;
        /**
         * Finds the multiple rows for axis.
         * @return {void}
         */
        private findMultiRows(length, currentX, currentLabel);
        /**
         * Finds the default module for axis.
         * @return {void}
         * @private
         */
        getModule(chart: Chart): void;
    }
    /** @private */
    export interface VisibleRangeModel {
        min?: number;
        max?: number;
        interval?: number;
        delta?: number;
    }
    /** @private */
    export class VisibleLabels {
        text: string | string[];
        value: number;
        labelStyle: FontModel;
        size: Size;
        index: number;
        originalText: string;
        constructor(text: string | string[], value: number, labelStyle: FontModel, size?: Size, index?: number);
    }
    
    export class CartesianAxisLayoutPanel {
        private chart;
        private initialClipRect;
        private htmlObject;
        private element;
        private padding;
        /** @private */
        leftSize: number;
        /** @private */
        rightSize: number;
        /** @private */
        topSize: number;
        /** @private */
        bottomSize: number;
        /** @private */
        seriesClipRect: Rect;
        /** @private */
        constructor(chartModule?: Chart);
        /**
         * Measure the axis size.
         * @return {void}
         * @private
         */
        measureAxis(rect: Rect): void;
        private measureRowAxis(chart, rect);
        private measureColumnAxis(chart, rect);
        /**
         * Measure the column and row in chart.
         * @return {void}
         * @private
         */
        measureDefinition(definition: Row | Column, chart: Chart, size: Size, clipRect: Rect): void;
        /**
         * Measure the axis.
         * @return {void}
         * @private
         */
        private calculateAxisSize(rect);
        /**
         * Measure the axis.
         * @return {void}
         * @private
         */
        measure(): void;
        private crossAt(chart);
        private updateCrossAt(axis, crossAt);
        private pushAxis(definition, axis);
        private arrangeAxis(definition);
        private getActualColumn(axis);
        private getActualRow(axis);
        /**
         * Measure the row size.
         * @return {void}
         */
        private calculateRowSize(rect);
        /**
         * Measure the row size.
         * @return {void}
         */
        private calculateColumnSize(rect);
        /**
         * To render the axis element.
         * @return {void}
         * @private
         */
        renderAxes(): Element;
        private findAxisPosition(axis);
        private drawBottomLine(definition, index, isRow);
        private drawAxisLine(axis, index, plotX, plotY, parent, rect);
        private drawYAxisGridLine(axis, index, parent, rect);
        private isBorder(axis, index, value);
        /**
         * To render the axis label.
         * @return {void}
         * @private
         */
        drawYAxisLabels(axis: Axis, index: number, parent: Element, rect: Rect): void;
        /**
         * To render the axis label border.
         * @return {void}
         * @private
         */
        private drawYAxisBorder(axis, index, parent, rect);
        private drawYAxisTitle(axis, index, parent, rect);
        private drawXAxisGridLine(axis, index, parent, rect);
        private drawAxisMinorLine(axis, tempInterval, minorGird, minorTick, rect, labelIndex);
        private findLogNumeric(axis, logPosition, logInterval, value, labelIndex);
        /**
         * To render the axis label.
         * @return {void}
         * @private
         */
        drawXAxisLabels(axis: Axis, index: number, parent: Element, rect: Rect): void;
        /**
         * To render the axis label border.
         * @return {void}
         * @private
         */
        private drawXAxisBorder(axis, index, parent, axisRect);
        /**
         * To create border element for axis.
         * @return {void}
         * @private
         */
        private createAxisBorderElement(axis, index, labelBorder, parent);
        private findAxisLabel(axis, label, width);
        private drawXAxisTitle(axis, index, parent, rect);
        private renderTickLine(axis, index, majorTick, minorTick, parent);
        private renderGridLine(axis, index, majorGrid, minorGird);
        private findParentNode(chart, label, axis, index);
        private createZoomingLabel(chart, labelElement, axis, index, rect);
    }
    
    /**
     * `Category` module is used to render category axis.
     */
    export class Category extends NiceInterval {
        /**
         * Constructor for the category module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * The function to calculate the range and labels for the axis.
         * @return {void}
         */
        calculateRangeAndInterval(size: Size, axis: Axis): void;
        /**
         * Actual Range for the axis.
         * @private
         */
        protected getActualRange(axis: Axis, size: Size): void;
        /**
         * Padding for the axis.
         * @private
         */
        protected applyRangePadding(axis: Axis, size: Size): void;
        /**
         * Calculate label for the axis.
         * @private
         */
        protected calculateVisibleLabels(axis: Axis): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the category axis.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `DateTime` module is used to render datetime axis.
     */
    export class DateTime extends NiceInterval {
        private start;
        private end;
        /**
         * Constructor for the dateTime module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * The function to calculate the range and labels for the axis.
         * @return {void}
         */
        calculateRangeAndInterval(size: Size, axis: Axis): void;
        /**
         * Actual Range for the axis.
         * @private
         */
        protected getActualRange(axis: Axis, size: Size): void;
        /**
         * Apply padding for the range.
         * @private
         */
        protected applyRangePadding(axis: Axis, size: Size): void;
        private getYear(minimum, maximum, rangePadding, interval);
        private getMonth(minimum, maximum, rangePadding, interval);
        private getDay(minimum, maximum, rangePadding, interval);
        private getHour(minimum, maximum, rangePadding, interval);
        /**
         * Calculate visible range for axis.
         * @private
         */
        protected calculateVisibleRange(size: Size, axis: Axis): void;
        /**
         * Calculate visible labels for the axis.
         * @private
         */
        protected calculateVisibleLabels(axis: Axis): void;
        /** @private */
        increaseDateTimeInterval(axis: Axis, value: number, interval: number): Date;
        private alignRangeStart(axis, sDate, intervalSize, intervalType);
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the category axis.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Category module is used to render category axis.
     */
    export class DateTimeCategory extends Category {
        private axisSize;
        /**
         * Constructor for the category module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * The function to calculate the range and labels for the axis.
         * @return {void}
         * @private
         */
        calculateRangeAndInterval(size: Size, axis: Axis): void;
        /**
         * Calculate label for the axis.
         * @private
         */
        protected calculateVisibleLabels(axis: Axis): void;
        /**
         * get same interval
         */
        private sameInterval(currentDate, previousDate, type, index);
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the category axis.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Numeric module is used to render numeric axis.
     */
    export class Double {
        /** @private */
        chart: Chart;
        /** @private */
        min: Object;
        /** @private */
        max: Object;
        private paddingInterval;
        /**
         * Constructor for the dateTime module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * Numeric Nice Interval for the axis.
         * @private
         */
        protected calculateNumericNiceInterval(axis: Axis, delta: number, size: Size): number;
        /**
         * Actual Range for the axis.
         * @private
         */
        protected getActualRange(axis: Axis, size: Size): void;
        /**
         * Range for the axis.
         * @private
         */
        protected initializeDoubleRange(axis: Axis): void;
        /**
         * The function to calculate the range and labels for the axis.
         * @return {void}
         * @private
         */
        calculateRangeAndInterval(size: Size, axis: Axis): void;
        /**
         * Calculate Range for the axis.
         * @private
         */
        protected calculateRange(axis: Axis, size: Size): void;
        private findMinMax(min, max);
        /**
         * Apply padding for the range.
         * @private
         */
        protected applyRangePadding(axis: Axis, size: Size): void;
        private updateActualRange(axis, minimum, maximum, interval);
        private findAdditional(axis, start, end, interval);
        private findNormal(axis, start, end, interval, size);
        /**
         * Calculate visible range for axis.
         * @private
         */
        protected calculateVisibleRange(size: Size, axis: Axis): void;
        /**
         * Calculate label for the axis.
         * @private
         */
        protected calculateVisibleLabels(axis: Axis, chart: Chart): void;
        /**
         * Format of the axis label.
         * @private
         */
        protected getFormat(axis: Axis): string;
        /**
         * Formatted the axis label.
         * @private
         */
        protected formatValue(axis: Axis, isCustom: boolean, format: string, tempInterval: number): string;
    }
    
    /**
     * `Logarithmic` module is used to render log axis.
     */
    export class Logarithmic extends Double {
        /**
         * Constructor for the logerithmic module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * The method to calculate the range and labels for the axis.
         * @return {void}
         */
        calculateRangeAndInterval(size: Size, axis: Axis): void;
        /**
         * Calculates actual range for the axis.
         * @private
         */
        protected getActualRange(axis: Axis, size: Size): void;
        /**
         * Calculates visible range for the axis.
         * @private
         */
        protected calculateVisibleRange(size: Size, axis: Axis): void;
        /**
         * Calculates log iInteval for the axis.
         * @private
         */
        protected calculateLogNiceInterval(delta: number, size: Size, axis: Axis): number;
        /**
         * Calculates labels for the axis.
         * @private
         */
        protected calculateVisibleLabels(axis: Axis, chart: Chart): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the category axis.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * MultiLevel Labels src
     */
    
    /**
     * `MultiLevelLabel` module is used to render the multi level label in chart.
     */
    export class MultiLevelLabel {
        /** @private */
        chart: Chart;
        /** @private */
        xAxisPrevHeight: number[];
        /** @private */
        xAxisMultiLabelHeight: number[];
        /** @private */
        yAxisPrevHeight: number[];
        /** @private */
        yAxisMultiLabelHeight: number[];
        /** @private */
        multiElements: Element;
        /**
         * Constructor for the logerithmic module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * Finds multilevel label height
         * @return {void}
         */
        getMultilevelLabelsHeight(axis: Axis): void;
        /**
         * render x axis multi level labels
         * @private
         * @return {void}
         */
        renderXAxisMultiLevelLabels(axis: Axis, index: number, parent: Element, axisRect: Rect): void;
        /**
         * render x axis multi level labels border
         * @private
         * @return {void}
         */
        private renderXAxisLabelBorder(labelIndex, gap, axis, startX, startY, labelSize, textOptions, axisRect, alignment, path, isOutside, opposedPosition);
        /**
         * render y axis multi level labels
         * @private
         * @return {void}
         */
        renderYAxisMultiLevelLabels(axis: Axis, index: number, parent: Element, rect: Rect): void;
        /**
         * render y axis multi level labels border
         * @private
         * @return {void}
         */
        private renderYAxisLabelBorder(labelIndex, gap, axis, endY, startX, startY, labelSize, textOptions, rect, alignment, path, isOutside, opposedPosition);
        /**
         * create cliprect
         * @return {void}
         * @private
         */
        createClipRect(x: number, y: number, height: number, width: number, clipId: string, axisId: string): void;
        /**
         * create borer element
         * @return {void}
         * @private
         */
        createBorderElement(borderIndex: number, axisIndex: number, axis: Axis, path: string): void;
        /**
         * Triggers the event.
         * @return {void}
         * @private
         */
        triggerMultiLabelRender(axis: Axis, text: string, textStyle: FontModel, textAlignment: Alignment): IAxisMultiLabelRenderEventArgs;
        /**
         * To get the module name for `MultiLevelLabel`.
         * @private
         */
        getModuleName(): string;
        /**
         * To destroy the `MultiLevelLabel` module.
         * @private
         */
        destroy(): void;
    }
    
    export class PolarRadarPanel extends LineBase {
        private initialClipRect;
        private htmlObject;
        private element;
        private centerX;
        private centerY;
        private startAngle;
        /** @private */
        seriesClipRect: Rect;
        /**
         * Measure the polar radar axis size.
         * @return {void}
         * @private
         */
        measureAxis(rect: Rect): void;
        private measureRowAxis(chart, rect);
        private measureColumnAxis(chart, rect);
        /**
         * Measure the column and row in chart.
         * @return {void}
         * @private
         */
        measureDefinition(definition: Row | Column, chart: Chart, size: Size, clipRect: Rect): void;
        /**
         * Measure the axis.
         * @return {void}
         * @private
         */
        private calculateAxisSize();
        /**
         * Measure the axis.
         * @return {void}
         * @private
         */
        measure(): void;
        /**
         * Measure the row size.
         * @return {void}
         */
        private calculateRowSize(rect);
        /**
         * Measure the row size.
         * @return {void}
         */
        private calculateColumnSize(rect);
        /**
         * To render the axis element.
         * @return {void}
         * @private
         */
        renderAxes(): Element;
        private drawYAxisLine(axis, index, plotX, plotY);
        drawYAxisLabels(axis: Axis, index: number): void;
        private drawYAxisGridLine(axis, index);
        private drawXAxisGridLine(axis, index);
        private drawAxisMinorLine(axis, tempInterval, minorGird, minorTick);
        /**
         * To render the axis label.
         * @return {void}
         * @private
         */
        drawXAxisLabels(axis: Axis, index: number): void;
        private renderTickLine(axis, index, majorTickLine, minorTickLine?);
        private renderGridLine(axis, index, majorGrid, minorGird);
    }
    
    /**
     * `StripLine` module is used to render the stripLine in chart.
     */
    export class StripLine {
        private measureStripLine(axis, stripline, seriesClipRect);
        /**
         * To get from to value from start, end, size, start from axis
         */
        private getFromTovalue(start, end, size, startFromAxis, axis);
        private getToValue(to, from, size, axis, end);
        private findValue(value, axis);
        /**
         * To render striplines based start and end.
         * @private
         * @param chart
         * @param position
         * @param axes
         */
        renderStripLine(chart: Chart, position: ZIndex, axes: Axis[]): void;
        private renderRectangle(stripline, rect, id, parent, chart);
        private renderText(stripline, rect, id, parent, chart, axis);
        private invertAlignment(anchor);
        private factor(anchor);
        private getTextStart(xy, size, textAlignment);
        /**
         * To get the module name for `StripLine`.
         * @private
         */
        getModuleName(): string;
        /**
         * To destroy the `StripLine` module.
         * @private
         */
        destroy(): void;
    }
    
    /**
     * `Legend` module is used to render legend for the chart.
     */
    export class Legend extends BaseLegend {
        constructor(chart: Chart);
        /**
         * Binding events for legend module.
         */
        private addEventListener();
        /**
         * UnBinding events for legend module.
         */
        private removeEventListener();
        /**
         * To handle mosue move for legend module
         */
        private mouseMove(e);
        /**
         * To handle mosue end for legend module
         */
        private mouseEnd(e);
        /**
         * Get the legend options.
         * @return {void}
         * @private
         */
        getLegendOptions(visibleSeriesCollection: Series[], chart: Chart): void;
        /** @private */
        getLegendBounds(availableSize: Size, legendBounds: Rect, legend: LegendSettingsModel): void;
        /** @private */
        getRenderPoint(legendOption: LegendOptions, start: ChartLocation, textPadding: number, prevLegend: LegendOptions, rect: Rect, count: number, firstLegend: number): void;
        /** @private */
        LegendClick(seriesIndex: number): void;
        private refreshSeries(seriesCollection);
        /**
         * To show the tooltip for the trimmed text in legend.
         * @return {void}
         */
        click(event: Event): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * To destroy the Legend.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Interface for a class ChartAnnotationSettings
     */
    export interface ChartAnnotationSettingsModel {
    
        /**
         * if set coordinateUnit as `Pixel` X specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        x?: string | Date | number;
    
        /**
         * if set coordinateUnit as `Pixel` Y specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        y?: string | number;
    
        /**
         * Content of the annotation, which accepts the id of the custom element.
         * @default null
         */
        content?: string;
    
        /**
         * Specifies the alignment of the annotation. They are
         * * Near - Align the annotation element as left side.
         * * Far - Align the annotation element as right side.
         * * Center - Align the annotation element as mid point.
         * @default 'Center'
         */
    
        horizontalAlignment?: Alignment;
    
        /**
         * Specifies the coordinate units of the annotation. They are
         * * Pixel - Annotation renders based on x and y pixel value.
         * * Point - Annotation renders based on x and y axis value.
         * @default 'Pixel'
         */
    
        coordinateUnits?: Units;
    
        /**
         * Specifies the regions of the annotation. They are
         * * Chart - Annotation renders based on chart coordinates.
         * * Series - Annotation renders based on series coordinates.
         * @default 'Chart'
         */
    
        region?: Regions;
    
        /**
         * Specifies the position of the annotation. They are
         * * Top - Align the annotation element as top side.
         * * Bottom - Align the annotation element as bottom side.
         * * Middle - Align the annotation element as mid point.
         * @default 'Middle'
         */
    
        verticalAlignment?: Position;
    
        /**
         * The name of horizontal axis associated with the annotation.
         * It requires `axes` of chart.
         * @default null
         */
    
        xAxisName?: string;
    
        /**
         * The name of vertical axis associated with the annotation.
         * It requires `axes` of chart.
         * @default null
         */
    
        yAxisName?: string;
    
        /**
         * Information about annotation for assistive technology.
         * @default null
         */
        description?: string;
    
    }
    
    /**
     * Interface for a class LabelBorder
     */
    export interface LabelBorderModel {
    
        /**
         * The color of the border that accepts value in hex and rgba as a valid CSS color string.
         * @default ''
         */
        color?: string;
    
        /**
         * The width of the border in pixels.
         * @default 1
         */
        width?: number;
    
        /**
         * Border type for labels
         * * Rectangle 
         * * Without Top Border
         * * Without Top and BottomBorder
         * * Without Border
         * * Brace
         * * CurlyBrace
         * @default 'Rectangle'
         */
        type?: BorderType;
    
    }
    
    /**
     * Interface for a class MultiLevelCategories
     */
    export interface MultiLevelCategoriesModel {
    
        /**
         * Start value of the multi level labels
         * @default null
         */
        start?: number | Date | string;
    
        /**
         * End value of the multi level labels
         * @default null
         */
        end?: number | Date | string;
    
        /**
         * multi level labels text.
         * @default ''
         */
        text?: string;
    
        /**
         * Maximum width of the text for multi level labels.
         * @default null
         */
        maximumTextWidth?: number;
    
    }
    
    /**
     * Interface for a class StripLineSettings
     */
    export interface StripLineSettingsModel {
    
        /**
         * If set true, strip line for axis renders.
         * @default true
         */
        visible?: boolean;
    
        /**
         *  If set true, strip line get render from axis origin.
         *  @default false
         */
        startFromAxis?: boolean;
    
        /**
         * Start value of the strip line.
         * @default null
         */
        start?: number | Date | string;
    
        /**
         * End value of the strip line.
         * @default null
         */
        end?: number | Date | string;
    
        /**
         * Size of the strip line, when it starts from the origin.
         * @default null
         */
        size?: number;
    
        /**
         * Color of the strip line.
         * @default '#808080'
         */
        color?: string;
    
        /**
         * Border of the strip line.
         */
        border?: BorderModel;
    
        /**
         * Strip line text.
         * @default ''
         */
        text?: string;
    
        /**
         * The angle to which the strip line text gets rotated.
         * @default null
         */
        rotation?: number;
    
        /**
         * Defines the position of the strip line text horizontally. They are, 
         * * Start: Places the strip line text at the start.
         * * Middle: Places the strip line text in the middle.
         * * End: Places the strip line text at the end.
         * @default 'Middle'
         */
        horizontalAlignment?: Anchor;
    
        /**
         * Defines the position of the strip line text vertically. They are, 
         * * Start: Places the strip line text at the start.
         * * Middle: Places the strip line text in the middle.
         * * End: Places the strip line text at the end.
         * @default 'Middle'
         */
        verticalAlignment?: Anchor;
    
        /**
         * Options to customize the strip line text.
         */
        textStyle?: FontModel;
    
        /**
         * Specifies the order of the strip line. They are,
         * * Behind: Places the strip line behind the series elements.
         * * Over: Places the strip line over the series elements.
         * @default 'Behind'
         */
        zIndex?: ZIndex;
    
        /**
         * Strip line Opacity
         * @default 1
         */
        opacity?: number;
    
    }
    
    /**
     * Interface for a class MultiLevelLabels
     */
    export interface MultiLevelLabelsModel {
    
        /**
         * Defines the position of the multi level labels. They are, 
         * * Near: Places the multi level labels at Near.
         * * Center: Places the multi level labels at Center.
         * * Far: Places the multi level labels at Far.
         * @default 'Center'
         */
        alignment?: Alignment;
    
        /**
         * Defines the textOverFlow for multi level labels. They are, 
         * * Trim: Trim textOverflow for multi level labels.
         * * Wrap: Wrap textOverflow for multi level labels.
         * * none: None textOverflow for multi level labels.
         * @default 'Wrap'
         */
        overflow?: TextOverflow;
    
        /**
         * Options to customize the multi level labels.
         */
        textStyle?: FontModel;
    
        /**
         * Border of the multi level labels.
         */
        border?: LabelBorderModel;
    
        /**
         * multi level categories for multi level labels.
         */
        categories?: MultiLevelCategoriesModel[];
    
    }
    
    /**
     * Configures the Annotation for chart.
     */
    export class ChartAnnotationSettings extends ej.base.ChildProperty<ChartAnnotationSettings> {
        /**
         * if set coordinateUnit as `Pixel` X specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        x: string | Date | number;
        /**
         * if set coordinateUnit as `Pixel` Y specifies the axis value
         * else is specifies pixel or percentage of coordinate
         * @default 0
         */
        y: string | number;
        /**
         * Content of the annotation, which accepts the id of the custom element.
         * @default null
         */
        content: string;
        /**
         * Specifies the alignment of the annotation. They are
         * * Near - Align the annotation element as left side.
         * * Far - Align the annotation element as right side.
         * * Center - Align the annotation element as mid point.
         * @default 'Center'
         */
        horizontalAlignment: Alignment;
        /**
         * Specifies the coordinate units of the annotation. They are
         * * Pixel - Annotation renders based on x and y pixel value.
         * * Point - Annotation renders based on x and y axis value.
         * @default 'Pixel'
         */
        coordinateUnits: Units;
        /**
         * Specifies the regions of the annotation. They are
         * * Chart - Annotation renders based on chart coordinates.
         * * Series - Annotation renders based on series coordinates.
         * @default 'Chart'
         */
        region: Regions;
        /**
         * Specifies the position of the annotation. They are
         * * Top - Align the annotation element as top side.
         * * Bottom - Align the annotation element as bottom side.
         * * Middle - Align the annotation element as mid point.
         * @default 'Middle'
         */
        verticalAlignment: Position;
        /**
         * The name of horizontal axis associated with the annotation.
         * It requires `axes` of chart.
         * @default null
         */
        xAxisName: string;
        /**
         * The name of vertical axis associated with the annotation.
         * It requires `axes` of chart.
         * @default null
         */
        yAxisName: string;
        /**
         * Information about annotation for assistive technology.
         * @default null
         */
        description: string;
    }
    /**
     * label border properties.
     */
    export class LabelBorder extends ej.base.ChildProperty<LabelBorder> {
        /**
         * The color of the border that accepts value in hex and rgba as a valid CSS color string.
         * @default ''
         */
        color: string;
        /**
         * The width of the border in pixels.
         * @default 1
         */
        width: number;
        /**
         * Border type for labels
         * * Rectangle
         * * Without Top Border
         * * Without Top and BottomBorder
         * * Without Border
         * * Brace
         * * CurlyBrace
         * @default 'Rectangle'
         */
        type: BorderType;
    }
    /**
     * categories for multi level labels
     */
    export class MultiLevelCategories extends ej.base.ChildProperty<MultiLevelCategories> {
        /**
         * Start value of the multi level labels
         * @default null
         */
        start: number | Date | string;
        /**
         * End value of the multi level labels
         * @default null
         */
        end: number | Date | string;
        /**
         * multi level labels text.
         * @default ''
         */
        text: string;
        /**
         * Maximum width of the text for multi level labels.
         * @default null
         */
        maximumTextWidth: number;
    }
    /**
     * Strip line properties
     */
    export class StripLineSettings extends ej.base.ChildProperty<StripLineSettings> {
        /**
         * If set true, strip line for axis renders.
         * @default true
         */
        visible: boolean;
        /**
         *  If set true, strip line get render from axis origin.
         *  @default false
         */
        startFromAxis: boolean;
        /**
         * Start value of the strip line.
         * @default null
         */
        start: number | Date | string;
        /**
         * End value of the strip line.
         * @default null
         */
        end: number | Date | string;
        /**
         * Size of the strip line, when it starts from the origin.
         * @default null
         */
        size: number;
        /**
         * Color of the strip line.
         * @default '#808080'
         */
        color: string;
        /**
         * Border of the strip line.
         */
        border: BorderModel;
        /**
         * Strip line text.
         * @default ''
         */
        text: string;
        /**
         * The angle to which the strip line text gets rotated.
         * @default null
         */
        rotation: number;
        /**
         * Defines the position of the strip line text horizontally. They are,
         * * Start: Places the strip line text at the start.
         * * Middle: Places the strip line text in the middle.
         * * End: Places the strip line text at the end.
         * @default 'Middle'
         */
        horizontalAlignment: Anchor;
        /**
         * Defines the position of the strip line text vertically. They are,
         * * Start: Places the strip line text at the start.
         * * Middle: Places the strip line text in the middle.
         * * End: Places the strip line text at the end.
         * @default 'Middle'
         */
        verticalAlignment: Anchor;
        /**
         * Options to customize the strip line text.
         */
        textStyle: FontModel;
        /**
         * Specifies the order of the strip line. They are,
         * * Behind: Places the strip line behind the series elements.
         * * Over: Places the strip line over the series elements.
         * @default 'Behind'
         */
        zIndex: ZIndex;
        /**
         * Strip line Opacity
         * @default 1
         */
        opacity: number;
    }
    /**
     * MultiLevelLabels properties
     */
    export class MultiLevelLabels extends ej.base.ChildProperty<MultiLevelLabels[]> {
        /**
         * Defines the position of the multi level labels. They are,
         * * Near: Places the multi level labels at Near.
         * * Center: Places the multi level labels at Center.
         * * Far: Places the multi level labels at Far.
         * @default 'Center'
         */
        alignment: Alignment;
        /**
         * Defines the textOverFlow for multi level labels. They are,
         * * Trim: Trim textOverflow for multi level labels.
         * * Wrap: Wrap textOverflow for multi level labels.
         * * none: None textOverflow for multi level labels.
         * @default 'Wrap'
         */
        overflow: TextOverflow;
        /**
         * Options to customize the multi level labels.
         */
        textStyle: FontModel;
        /**
         * Border of the multi level labels.
         */
        border: LabelBorderModel;
        /**
         * multi level categories for multi level labels.
         */
        categories: MultiLevelCategoriesModel[];
    }
    
    /**
     * `AreaSeries` module is used to render the area series.
     */
    export class AreaSeries extends MultiColoredSeries {
        /**
         * Render Area series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * To destroy the area series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
    }
    
    /**
     * `BarSeries` module is used to render the bar series.
     */
    export class BarSeries extends ColumnBase {
        /**
         * Render Bar series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the bar series.
         * @return {void}
         * @private
         */
        protected destroy(chart: Chart): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
    }
    
    /**
     * `BoxAndWhiskerSeries` module is used to render the box and whisker series.
     */
    export class BoxAndWhiskerSeries extends ColumnBase {
        /**
         * Render BoxAndWhisker series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * update the tip region fo box plot
         * @param series
         * @param point
         * @param sideBySideInfo
         */
        private updateTipRegion(series, point, sideBySideInfo);
        /**
         * Update tip size to tip regions
         * @param series
         * @param point
         * @param region
         * @param isInverted
         */
        private updateTipSize(series, point, region, isInverted);
        /**
         * Calculation for path direction performed here
         * @param point
         * @param series
         * @param median
         * @param average
         */
        getPathString(point: Points, series: Series, median: ChartLocation, average: ChartLocation): string;
        /**
         * Rendering for box and whisker append here.
         * @param series
         * @param point
         * @param rect
         * @param argsData
         * @param direction
         */
        renderBoxAndWhisker(series: Series, point: Points, rect: Rect, argsData: IPointRenderEventArgs, direction: string): void;
        /**
         * To find the box plot values
         * @param yValues
         * @param point
         * @param mode
         */
        findBoxPlotValues(yValues: number[], point: Points, mode: BoxPlotMode): void;
        /**
         * to find the exclusive quartile values
         * @param yValues
         * @param count
         * @param percentile
         */
        private getExclusiveQuartileValue(yValues, count, percentile);
        /**
         * to find the inclusive quartile values
         * @param yValues
         * @param count
         * @param percentile
         */
        private getInclusiveQuartileValue(yValues, count, percentile);
        /**
         * To find the quartile values
         * @param yValues
         * @param count
         * @param lowerQuartile
         * @param upperQuartile
         */
        private getQuartileValues(yValues, count, quartile);
        /**
         * To find the min, max and outlier values
         * @param yValues
         * @param lowerQuartile
         * @param upperQuartile
         * @param minimum
         * @param maximum
         * @param outliers
         */
        private getMinMaxOutlier(yValues, count, quartile);
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the candle series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `BubbleSeries` module is used to render the bubble series.
     */
    export class BubbleSeries {
        /**
         * Render the Bubble series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * To destroy the Bubble.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
    }
    
    /**
     * `CandleSeries` module is used to render the candle series.
     */
    export class CandleSeries extends ColumnBase {
        /**
         * Render Candle series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * Trigger point rendering event
         */
        protected triggerPointRenderEvent(series: Series, point: Points): IPointRenderEventArgs;
        /**
         * Find the color of the candle
         * @param series
         * @private
         */
        private getCandleColor(point, series);
        /**
         * Finds the path of the candle shape
         * @param Series
         * @private
         */
        getPathString(topRect: Rect, midRect: Rect, series: Series): string;
        /**
         * Draws the candle shape
         * @param series
         * @private
         */
        drawCandle(series: Series, point: Points, rect: Rect, argsData: IPointRenderEventArgs, direction: string): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the candle series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Interface for a class DataLabelSettings
     */
    export interface DataLabelSettingsModel {
    
        /**
         * If set true, data label for series renders.
         * @default false
         */
    
        visible?: boolean;
    
        /**
         * The DataSource field that contains the data label value.
         * @default null
         */
    
        name?: string;
    
        /**
         * The background color of the data label accepts value in hex and rgba as a valid CSS color string.
         * @default 'transparent'
         */
    
        fill?: string;
    
        /**
         * The opacity for the background.
         * @default 1
         */
    
        opacity?: number;
    
        /**
         * Specifies the position of the data label. They are,
         * * Outer: Positions the label outside the point.
         * * top: Positions the label on top of the point.
         * * Bottom: Positions the label at the bottom of the point.
         * * Middle: Positions the label to the middle of the point.
         * * Auto: Positions the label based on series.
         * @default 'Auto'
         */
    
        position?: LabelPosition;
    
        /**
         * The roundedCornerX for the data label. It requires `border` values not to be null.
         * @default 5
         */
        rx?: number;
    
        /**
         * The roundedCornerY for the data label. It requires `border` values not to be null.
         * @default 5
         */
        ry?: number;
    
        /**
         * Specifies the alignment for data Label. They are,
         * * Near: Aligns the label to the left of the point.
         * * Center: Aligns the label to the center of the point.
         * * Far: Aligns the label to the right of the point.
         * @default 'Center'
         */
        alignment?: Alignment;
    
        /**
         * Option for customizing the border lines.
         */
    
        border?: BorderModel;
    
        /**
         * Margin configuration for the data label.
         */
    
        margin?: MarginModel;
    
        /**
         * Option for customizing the data label text.
         */
    
        font?: FontModel;
    
        /**
         * Custom template to show the data label. Use ${point.x} and ${point.y} as a placeholder
         * text to display the corresponding data point.
         * @default null
         */
    
        template?: string;
    
    }
    
    /**
     * Interface for a class MarkerSettings
     */
    export interface MarkerSettingsModel {
    
        /**
         * If set to true the marker for series is rendered. This is applicable only for line and area type series.
         * @default false
         */
    
        visible?: boolean;
    
        /**
         * The different shape of a marker:
         * * Circle
         * * Rectangle
         * * Triangle
         * * Diamond
         * * HorizontalLine
         * * VerticalLine
         * * Pentagon
         * * InvertedTriangle
         * * Image
         * @default 'Circle'
         */
    
        shape?: ChartShape;
    
        /**
         * The URL for the Image that is to be displayed as a marker.  It requires marker `shape` value to be an `Image`.
         * @default ''
         */
    
        imageUrl?: string;
    
        /**
         * The height of the marker in pixels.
         * @default 5
         */
    
        height?: number;
    
        /**
         * The width of the marker in pixels.
         * @default 5
         */
    
        width?: number;
    
        /**
         * Options for customizing the border of a marker.
         */
    
        border?: BorderModel;
    
        /**
         *  The fill color of the marker that accepts value in hex and rgba as a valid CSS color string. By default, it will take series' color.
         * @default null
         */
    
        fill?: string;
    
        /**
         * The opacity of the marker.
         * @default 1
         */
    
        opacity?: number;
    
        /**
         * The data label for the series.
         */
    
        dataLabel?: DataLabelSettingsModel;
    
    }
    
    /**
     * Interface for a class Points
     * @private
     */
    export interface PointsModel {
    
    }
    
    /**
     * Interface for a class Trendline
     */
    export interface TrendlineModel {
    
        /**
         * Defines the name of trendline
         * @default ''
         */
        name?: string;
    
        /**
         * Defines the type of the trendline
         * @default 'Linear'
         */
        type?: TrendlineTypes;
    
        /**
         * Defines the period, the price changes over which will be considered to predict moving average trend line
         * @default 2
         */
        period?: number;
    
        /**
         * Defines the polynomial order of the polynomial trendline
         * @default 2
         */
        polynomialOrder?: number;
    
        /**
         * Defines the period, by which the trend has to backward forecast
         * @default 0
         */
        backwardForecast?: number;
    
        /**
         * Defines the period, by which the trend has to forward forecast
         * @default 0
         */
        forwardForecast?: number;
    
        /**
         * Options to customize the animation for trendlines
         */
        animation?: AnimationModel;
    
        /**
         * Options to customize the marker for trendlines
         */
        marker?: MarkerSettingsModel;
    
        /**
         * Enables/disables tooltip for trendlines
         * @default true
         */
        enableTooltip?: boolean;
    
        /**
         * Defines the intercept of the trendline
         * @default null
         */
        intercept?: number;
    
        /**
         * Defines the fill color of trendline
         * @default ''
         */
        fill?: string;
    
        /**
         * Defines the width of the trendline
         * @default 1
         */
        width?: number;
    
        /**
         * Sets the legend shape of the trendline
         * @default 'SeriesType'
         */
        legendShape?: LegendShape;
    
    }
    
    /**
     * Interface for a class ErrorBarCapSettings
     */
    export interface ErrorBarCapSettingsModel {
    
        /**
         * The width of the error bar in pixels.
         * @default 1
         */
    
        width?: number;
    
        /**
         * The length of the error bar in pixels.
         * @default 10
         */
    
        length?: number;
    
        /**
         *  The stroke color of the cap, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
    
        color?: string;
    
        /**
         * The opacity of the cap.
         * @default 1
         */
    
        opacity?: number;
    
    }
    
    /**
     * Interface for a class ChartSegment
     */
    export interface ChartSegmentModel {
    
        /**
         * Defines the starting point of region.
         * @default null
         */
    
        value?: Object;
    
        /**
         * Defines the color of a region.
         * @default null
         */
    
        color?: string;
    
        /**
         * Defines the pattern of dashes and gaps to stroke.
         * @default '0'
         */
    
        dashArray?: string;
    
    }
    
    /**
     * Interface for a class ErrorBarSettings
     * @private
     */
    export interface ErrorBarSettingsModel {
    
        /**
         * If set true, error bar for data gets rendered.
         * @default false
         */
    
        visible?: boolean;
    
        /**
         * The type of the error bar . They are
         * * Fixed -  Renders a fixed type error bar.
         * * Percentage - Renders a percentage type error bar.
         * * StandardDeviation - Renders a standard deviation type error bar.
         * * StandardError -Renders a standard error type error bar.
         * * Custom -Renders a custom type error bar.
         * @default 'Fixed'
         */
    
        type?: ErrorBarType;
    
        /**
         * The direction of the error bar . They are
         * * both -  Renders both direction of error bar.
         * * minus - Renders minus direction of error bar.
         * * plus - Renders plus direction error bar.
         * @default 'Both'
         */
    
        direction?: ErrorBarDirection;
    
        /**
         * The mode of the error bar . They are
         * * Vertical -  Renders a vertical error bar.
         * * Horizontal - Renders a horizontal error bar.
         * * Both - Renders both side error bar.
         * @default 'Vertical'
         */
    
        mode?: ErrorBarMode;
    
        /**
         *  The color for stroke of the error bar, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
    
        color?: string;
    
        /**
         * The vertical error of the error bar.
         * @default 1
         */
    
        verticalError?: number;
    
        /**
         * The stroke width of the error bar..
         * @default 1
         */
    
        width?: number;
    
        /**
         * The horizontal error of the error bar.
         * @default 1
         */
    
        horizontalError?: number;
    
        /**
         * The vertical positive error of the error bar.
         * @default 3
         */
    
        verticalPositiveError?: number;
    
        /**
         * The vertical negative error of the error bar.
         * @default 3
         */
    
        verticalNegativeError?: number;
    
        /**
         * The horizontal positive error of the error bar.
         * @default 1
         */
    
        horizontalPositiveError?: number;
    
        /**
         * The horizontal negative error of the error bar.
         * @default 1
         */
    
        horizontalNegativeError?: number;
    
        /**
         * Options for customizing the cap of the error bar.
         */
        errorBarCap?: ErrorBarCapSettingsModel;
    
    }
    
    /**
     * Interface for a class SeriesBase
     */
    export interface SeriesBaseModel {
    
        /**
         * The DataSource field that contains the x value.
         * It is applicable for series and technical indicators
         * @default ''
         */
    
        xName?: string;
    
        /**
         * The DataSource field that contains the high value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
    
        high?: string;
    
        /**
         * The DataSource field that contains the low value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
    
        low?: string;
    
        /**
         * The DataSource field that contains the open value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
    
        open?: string;
    
        /**
         * The DataSource field that contains the close value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
    
        close?: string;
    
        /**
         * Defines the data source field that contains the volume value in candle charts
         * It is applicable for financial series and technical indicators
         * @default ''
         */
    
        volume?: string;
    
        /**
         * The DataSource field that contains the color value of point
         * It is applicable for series
         * @default ''
         */
    
        pointColorMapping?: string;
    
        /**
         * The name of the horizontal axis associated with the series. It requires `axes` of the chart.
         * It is applicable for series and technical indicators
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     columns: [{ width: '50%' },
         *               { width: '50%' }],
         *     axes: [{
         *                name: 'xAxis 1',
         *                columnIndex: 1,
         *            }],
         *     series: [{
         *                dataSource: data,
         *                xName: 'x', yName: 'y',
         *                xAxisName: 'xAxis 1',
         *     }],
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default null
         */
    
        xAxisName?: string;
    
        /**
         * The name of the vertical axis associated with the series. It requires `axes` of the chart.
         * It is applicable for series and technical indicators
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     rows: [{ height: '50%' },
         *            { height: '50%' }],
         *     axes: [{
         *                name: 'yAxis 1',
         *                rowIndex: 1,
         *            }],
         *     series: [{
         *                dataSource: data,
         *                xName: 'x', yName: 'y',
         *                yAxisName: 'yAxis 1'
         *     }],
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default null
         */
    
        yAxisName?: string;
    
        /**
         * Options to customizing animation for the series.
         */
    
        animation?: AnimationModel;
    
        /**
         * The fill color for the series that accepts value in hex and rgba as a valid CSS color string.
         * It also represents the color of the signal lines in technical indicators.
         * For technical indicators, the default value is 'blue' and for series, it has null.
         * @default null
         */
    
        fill?: string;
    
        /**
         * The stroke width for the series that is applicable only for `Line` type series.
         * It also represents the stroke width of the signal lines in technical indicators.
         * @default 1
         */
    
        width?: number;
    
        /**
         * Defines the pattern of dashes and gaps to stroke the lines in `Line` type series.
         * @default '0'
         */
    
        dashArray?: string;
    
        /**
         * Specifies the DataSource for the series. It can be an array of JSON objects or an instance of DataManager.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let dataManager: ej.data.DataManager = new DataManager({
         *         url: 'http://mvc.syncfusion.com/Services/Northwnd.svc/Tasks/'
         * });
         * let query: ej.data.Query = new Query().take(50).where('Estimate', 'greaterThan', 0, false);
         * let chart: Chart = new Chart({
         * ...
         *     series: [{
         *        dataSource: dataManager,
         *        xName: 'Id',
         *        yName: 'Estimate',
         *        query: query
         *    }],
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default ''
         */
    
        dataSource?: Object | ej.data.DataManager;
    
        /**
         * Specifies query to select data from DataSource. This property is applicable only when the DataSource is `ej.DataManager`.
         * @default null
         */
        query?: ej.data.Query;
    
        /**
         * Defines the collection of regions that helps to differentiate a line series.
         */
        segments?: ChartSegmentModel[];
    
        /**
         * Defines the axis, based on which the line series will be split.
         */
        segmentAxis?: Segment;
    
    }
    
    /**
     * Interface for a class Series
     */
    export interface SeriesModel extends SeriesBaseModel{
    
        /**
         * The name of the series visible in legend.
         * @default ''
         */
    
        name?: string;
    
        /**
         * The DataSource field that contains the y value.
         * @default ''
         */
    
        yName?: string;
    
        /**
         * Type of series to be drawn in radar or polar series. They are
         *  'Line'
         *  'Column'
         *  'Area'
         *  'Scatter'
         *  'Spline'
         *  'StackingColumn'
         *  'StackingArea'
         *  'RangeColumn'
         *  'SplineArea'
         * @default 'Line'
         */
        drawType?: ChartDrawType;
    
        /**
         * Specifies whether to join start and end point of a line/area series used in polar/radar chart to form a closed path.
         * @default true
         */
        isClosed?: boolean;
    
        /**
         * This property is used in financial charts to visualize the price movements in stock.
         * It defines the color of the candle/point, when the opening price is less than the closing price.
         * @default '#2ecd71'
         */
    
        bearFillColor?: string;
    
        /**
         * This property is used in financial charts to visualize the price movements in stock.
         * It defines the color of the candle/point, when the opening price is higher than the closing price.
         * @default '#e74c3d'
         */
    
        bullFillColor?: string;
    
        /**
         * This property is applicable for candle series.
         * It enables/disables to visually compare the current values with the previous values in stock.
         * @default false
         */
        enableSolidCandles?: boolean;
    
        /**
         * The DataSource field that contains the size value of y
         * @default ''
         */
    
        size?: string;
    
        /**
         * This property allows grouping series in `stacked column / bar` charts.
         * Any string value can be provided to the stackingGroup property.
         * If any two or above series have the same value, those series will be grouped together.
         * @default ''
         */
    
        stackingGroup?: string;
    
        /**
         * Specifies the visibility of series.
         * @default true
         */
    
        visible?: boolean;
    
        /**
         * Options to customizing the border of the series. This is applicable only for `Column` and `Bar` type series.
         */
    
        border?: BorderModel;
    
        /**
         * The opacity of the series.
         * @default 1
         */
        opacity?: number;
    
        /**
         * The type of the series are
         * * Line
         * * Column
         * * Area
         * * Bar
         * * StackingColumn
         * * StackingArea
         * * StackingBar
         * * StepLine
         * * StepArea
         * * Scatter
         * * Spline
         * * StackingColumn100
         * * StackingBar100
         * * StackingArea100
         * * RangeColumn
         * * Hilo
         * * HiloOpenClose
         * * Waterfall
         * * RangeArea 
         * * Bubble
         * * Candle 
         * * Polar 
         * * Radar 
         * * BoxAndWhisker
         * @default 'Line'
         */
    
        type?: ChartSeriesType;
    
        /**
         * Options for displaying and customizing error bar for individual point in a series.
         */
        errorBar?: ErrorBarSettingsModel;
    
        /**
         * Options for displaying and customizing markers for individual points in a series.
         */
        marker?: MarkerSettingsModel;
    
        /**
         * Defines the collection of trendlines that are used to predict the trend
         */
        trendlines?: TrendlineModel[];
    
        /**
         * If set true, the Tooltip for series will be visible.
         * @default true
         */
        enableTooltip?: boolean;
    
        /**
         * The shape of the legend. Each series has its own legend shape. They are,
         * * Circle
         * * Rectangle
         * * Triangle
         * * Diamond
         * * Cross
         * * HorizontalLine
         * * VerticalLine
         * * Pentagon
         * * InvertedTriangle
         * * SeriesType
         * @default 'SeriesType'
         */
    
        legendShape?: LegendShape;
    
        /**
         * Custom style for the selected series or points.
         * @default null
         */
        selectionStyle?: string;
    
        /**
         * Minimum radius
         * @default 1
         */
        minRadius?: number;
    
        /**
         * Maximum radius
         * @default 3
         */
        maxRadius?: number;
    
        /**
         * Defines type of spline to be rendered.
         * @default 'Natural'
         */
        splineType?: SplineType;
    
        /**
         * It defines tension of cardinal spline types
         * @default 0.5
         */
        cardinalSplineTension?: number;
    
        /**
         * options to customize the empty points in series
         */
        emptyPointSettings?: EmptyPointSettingsModel;
    
        /**
         * If set true, the mean value for box and whisker will be visible.
         * @default true
         */
        showMean?: boolean;
    
        /**
         * The mode of the box and whisker char series. They are,
         * Exclusive
         * Inclusive
         * Normal
         * @default 'Normal'
         */
        boxPlotMode?: BoxPlotMode;
    
        /**
         * To render the column series points with particular column width.
         * @default 0.7
         */
        columnWidth?: number;
    
        /**
         * To render the column series points with particular column spacing. It takes value from 0 - 1.
         * @default 0
         */
        columnSpacing?: number;
    
        /**
         * Defines the visual representation of the negative changes in waterfall charts.
         * @default '#C64E4A'
         */
        negativeFillColor?: string;
    
        /**
         * Defines the visual representation of the summaries in waterfall charts.
         * @default '#4E81BC'
         */
        summaryFillColor?: string;
    
        /**
         * Defines the collection of indexes of the intermediate summary columns in waterfall charts.
         * @default []
         */
        intermediateSumIndexes?: number[];
    
        /**
         * Defines the collection of indexes of the overall summary columns in waterfall charts.
         * @default []
         */
        sumIndexes?: number[];
    
        /**
         * Defines the appearance of line connecting adjacent points in waterfall charts.
         */
    
        connector?: ConnectorModel;
    
        /**
         * To render the column series points with particular rounded corner.
         */
        cornerRadius?: CornerRadiusModel;
    
    }
    
    /**
     * Configures the data label in the series.
     */
    export class DataLabelSettings extends ej.base.ChildProperty<DataLabelSettings> {
        /**
         * If set true, data label for series renders.
         * @default false
         */
        visible: boolean;
        /**
         * The DataSource field that contains the data label value.
         * @default null
         */
        name: string;
        /**
         * The background color of the data label accepts value in hex and rgba as a valid CSS color string.
         * @default 'transparent'
         */
        fill: string;
        /**
         * The opacity for the background.
         * @default 1
         */
        opacity: number;
        /**
         * Specifies the position of the data label. They are,
         * * Outer: Positions the label outside the point.
         * * top: Positions the label on top of the point.
         * * Bottom: Positions the label at the bottom of the point.
         * * Middle: Positions the label to the middle of the point.
         * * Auto: Positions the label based on series.
         * @default 'Auto'
         */
        position: LabelPosition;
        /**
         * The roundedCornerX for the data label. It requires `border` values not to be null.
         * @default 5
         */
        rx: number;
        /**
         * The roundedCornerY for the data label. It requires `border` values not to be null.
         * @default 5
         */
        ry: number;
        /**
         * Specifies the alignment for data Label. They are,
         * * Near: Aligns the label to the left of the point.
         * * Center: Aligns the label to the center of the point.
         * * Far: Aligns the label to the right of the point.
         * @default 'Center'
         */
        alignment: Alignment;
        /**
         * Option for customizing the border lines.
         */
        border: BorderModel;
        /**
         * Margin configuration for the data label.
         */
        margin: MarginModel;
        /**
         * Option for customizing the data label text.
         */
        font: FontModel;
        /**
         * Custom template to show the data label. Use ${point.x} and ${point.y} as a placeholder
         * text to display the corresponding data point.
         * @default null
         */
        template: string;
    }
    /**
     *  Configures the marker in the series.
     */
    export class MarkerSettings extends ej.base.ChildProperty<MarkerSettings> {
        /**
         * If set to true the marker for series is rendered. This is applicable only for line and area type series.
         * @default false
         */
        visible: boolean;
        /**
         * The different shape of a marker:
         * * Circle
         * * Rectangle
         * * Triangle
         * * Diamond
         * * HorizontalLine
         * * VerticalLine
         * * Pentagon
         * * InvertedTriangle
         * * Image
         * @default 'Circle'
         */
        shape: ChartShape;
        /**
         * The URL for the Image that is to be displayed as a marker.  It requires marker `shape` value to be an `Image`.
         * @default ''
         */
        imageUrl: string;
        /**
         * The height of the marker in pixels.
         * @default 5
         */
        height: number;
        /**
         * The width of the marker in pixels.
         * @default 5
         */
        width: number;
        /**
         * Options for customizing the border of a marker.
         */
        border: BorderModel;
        /**
         *  The fill color of the marker that accepts value in hex and rgba as a valid CSS color string. By default, it will take series' color.
         * @default null
         */
        fill: string;
        /**
         * The opacity of the marker.
         * @default 1
         */
        opacity: number;
        /**
         * The data label for the series.
         */
        dataLabel: DataLabelSettingsModel;
    }
    /**
     * Points model for the series.
     * @private
     */
    export class Points {
        x: Object;
        y: Object;
        visible: boolean;
        text: string;
        color: string;
        open: Object;
        close: Object;
        symbolLocations: ChartLocation[];
        xValue: number;
        yValue: number;
        index: number;
        regions: Rect[];
        percent: string;
        high: Object;
        low: Object;
        volume: Object;
        size: Object;
        isEmpty: boolean;
        regionData: PolarArc;
        minimum: number;
        maximum: number;
        upperQuartile: number;
        lowerQuartile: number;
        median: number;
        outliers: number[];
        average: number;
        error: number;
        interior: string;
        marker: MarkerSettingsModel;
    }
    /**
     * Defines the behavior of the Trendlines
     */
    export class Trendline extends ej.base.ChildProperty<Trendline> {
        /**
         * Defines the name of trendline
         * @default ''
         */
        name: string;
        /**
         * Defines the type of the trendline
         * @default 'Linear'
         */
        type: TrendlineTypes;
        /**
         * Defines the period, the price changes over which will be considered to predict moving average trend line
         * @default 2
         */
        period: number;
        /**
         * Defines the polynomial order of the polynomial trendline
         * @default 2
         */
        polynomialOrder: number;
        /**
         * Defines the period, by which the trend has to backward forecast
         * @default 0
         */
        backwardForecast: number;
        /**
         * Defines the period, by which the trend has to forward forecast
         * @default 0
         */
        forwardForecast: number;
        /**
         * Options to customize the animation for trendlines
         */
        animation: AnimationModel;
        /**
         * Options to customize the marker for trendlines
         */
        marker: MarkerSettingsModel;
        /**
         * Enables/disables tooltip for trendlines
         * @default true
         */
        enableTooltip: boolean;
        /**
         * Defines the intercept of the trendline
         * @default null
         */
        intercept: number;
        /**
         * Defines the fill color of trendline
         * @default ''
         */
        fill: string;
        /**
         * Defines the width of the trendline
         * @default 1
         */
        width: number;
        /**
         * Sets the legend shape of the trendline
         * @default 'SeriesType'
         */
        legendShape: LegendShape;
        /** @private */
        targetSeries: Series;
        /** @private */
        trendLineElement: Element;
        /** @private */
        points: Points[];
        /** @private */
        clipRectElement: Element;
        /** @private */
        clipRect: Rect;
        /** @private */
        polynomialSlopes: number[];
        /** @private */
        sourceIndex: number;
        /** @private */
        index: number;
        /** @private */
        setDataSource(series: Series, chart: Chart): void;
    }
    /**
     * Configures Error bar in series.
     */
    export class ErrorBarCapSettings extends ej.base.ChildProperty<ErrorBarCapSettings> {
        /**
         * The width of the error bar in pixels.
         * @default 1
         */
        width: number;
        /**
         * The length of the error bar in pixels.
         * @default 10
         */
        length: number;
        /**
         *  The stroke color of the cap, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
        color: string;
        /**
         * The opacity of the cap.
         * @default 1
         */
        opacity: number;
    }
    export class ChartSegment extends ej.base.ChildProperty<ChartSegment> {
        /**
         * Defines the starting point of region.
         * @default null
         */
        value: Object;
        /**
         * Defines the color of a region.
         * @default null
         */
        color: string;
        /**
         * Defines the pattern of dashes and gaps to stroke.
         * @default '0'
         */
        dashArray: string;
        /** @private */
        startValue: number;
        /** @private */
        endValue: number;
    }
    export class ErrorBarSettings extends ej.base.ChildProperty<ErrorBarSettings> {
        /**
         * If set true, error bar for data gets rendered.
         * @default false
         */
        visible: boolean;
        /**
         * The type of the error bar . They are
         * * Fixed -  Renders a fixed type error bar.
         * * Percentage - Renders a percentage type error bar.
         * * StandardDeviation - Renders a standard deviation type error bar.
         * * StandardError -Renders a standard error type error bar.
         * * Custom -Renders a custom type error bar.
         * @default 'Fixed'
         */
        type: ErrorBarType;
        /**
         * The direction of the error bar . They are
         * * both -  Renders both direction of error bar.
         * * minus - Renders minus direction of error bar.
         * * plus - Renders plus direction error bar.
         * @default 'Both'
         */
        direction: ErrorBarDirection;
        /**
         * The mode of the error bar . They are
         * * Vertical -  Renders a vertical error bar.
         * * Horizontal - Renders a horizontal error bar.
         * * Both - Renders both side error bar.
         * @default 'Vertical'
         */
        mode: ErrorBarMode;
        /**
         *  The color for stroke of the error bar, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
        color: string;
        /**
         * The vertical error of the error bar.
         * @default 1
         */
        verticalError: number;
        /**
         * The stroke width of the error bar..
         * @default 1
         */
        width: number;
        /**
         * The horizontal error of the error bar.
         * @default 1
         */
        horizontalError: number;
        /**
         * The vertical positive error of the error bar.
         * @default 3
         */
        verticalPositiveError: number;
        /**
         * The vertical negative error of the error bar.
         * @default 3
         */
        verticalNegativeError: number;
        /**
         * The horizontal positive error of the error bar.
         * @default 1
         */
        horizontalPositiveError: number;
        /**
         * The horizontal negative error of the error bar.
         * @default 1
         */
        horizontalNegativeError: number;
        /**
         * Options for customizing the cap of the error bar.
         */
        errorBarCap: ErrorBarCapSettingsModel;
    }
    /**
     * Defines the common behavior of Series and Technical Indicators
     */
    export class SeriesBase extends ej.base.ChildProperty<SeriesBase> {
        /**
         * The DataSource field that contains the x value.
         * It is applicable for series and technical indicators
         * @default ''
         */
        xName: string;
        /**
         * The DataSource field that contains the high value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
        high: string;
        /**
         * The DataSource field that contains the low value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
        low: string;
        /**
         * The DataSource field that contains the open value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
        open: string;
        /**
         * The DataSource field that contains the close value of y
         * It is applicable for series and technical indicators
         * @default ''
         */
        close: string;
        /**
         * Defines the data source field that contains the volume value in candle charts
         * It is applicable for financial series and technical indicators
         * @default ''
         */
        volume: string;
        /**
         * The DataSource field that contains the color value of point
         * It is applicable for series
         * @default ''
         */
        pointColorMapping: string;
        /**
         * The name of the horizontal axis associated with the series. It requires `axes` of the chart.
         * It is applicable for series and technical indicators
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     columns: [{ width: '50%' },
         *               { width: '50%' }],
         *     axes: [{
         *                name: 'xAxis 1',
         *                columnIndex: 1,
         *            }],
         *     series: [{
         *                dataSource: data,
         *                xName: 'x', yName: 'y',
         *                xAxisName: 'xAxis 1',
         *     }],
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default null
         */
        xAxisName: string;
        /**
         * The name of the vertical axis associated with the series. It requires `axes` of the chart.
         * It is applicable for series and technical indicators
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *     rows: [{ height: '50%' },
         *            { height: '50%' }],
         *     axes: [{
         *                name: 'yAxis 1',
         *                rowIndex: 1,
         *            }],
         *     series: [{
         *                dataSource: data,
         *                xName: 'x', yName: 'y',
         *                yAxisName: 'yAxis 1'
         *     }],
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default null
         */
        yAxisName: string;
        /**
         * Options to customizing animation for the series.
         */
        animation: AnimationModel;
        /**
         * The fill color for the series that accepts value in hex and rgba as a valid CSS color string.
         * It also represents the color of the signal lines in technical indicators.
         * For technical indicators, the default value is 'blue' and for series, it has null.
         * @default null
         */
        fill: string;
        /**
         * The stroke width for the series that is applicable only for `Line` type series.
         * It also represents the stroke width of the signal lines in technical indicators.
         * @default 1
         */
        width: number;
        /**
         * Defines the pattern of dashes and gaps to stroke the lines in `Line` type series.
         * @default '0'
         */
        dashArray: string;
        /**
         * Specifies the DataSource for the series. It can be an array of JSON objects or an instance of DataManager.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let dataManager: ej.data.DataManager = new DataManager({
         *         url: 'http://mvc.syncfusion.com/Services/Northwnd.svc/Tasks/'
         * });
         * let query: ej.data.Query = new Query().take(50).where('Estimate', 'greaterThan', 0, false);
         * let chart: Chart = new Chart({
         * ...
         *     series: [{
         *        dataSource: dataManager,
         *        xName: 'Id',
         *        yName: 'Estimate',
         *        query: query
         *    }],
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         * @default ''
         */
        dataSource: Object | ej.data.DataManager;
        /**
         * Specifies query to select data from DataSource. This property is applicable only when the DataSource is `ej.DataManager`.
         * @default null
         */
        query: ej.data.Query;
        /**
         * Defines the collection of regions that helps to differentiate a line series.
         */
        segments: ChartSegmentModel[];
        /**
         * Defines the axis, based on which the line series will be split.
         */
        segmentAxis: Segment;
        /**
         * Process data for the series.
         * @hidden
         */
        processJsonData(): void;
        private pushData(point, i);
        /** @private */
        protected dataPoint(i: number, textMappingName: string): Points;
        /**
         * To set empty point value based on empty point mode
         * @private
         */
        setEmptyPoint(point: Points, i: number): void;
        private findVisibility(point);
        /**
         * To get Y min max for the provided point seriesType XY
         */
        private setXYMinMax(yValue);
        /**
         * To get Y min max for the provided point seriesType XY
         */
        private setHiloMinMax(high, low);
        /**
         * Finds the type of the series
         * @private
         */
        private getSeriesType();
        /** @private */
        protected pushCategoryData(point: Points, index: number, pointX: string): void;
        /**
         * To find average of given property
         */
        private getAverage(member, i, data?);
        /**
         * To find the control points for spline.
         * @return {void}
         * @private
         */
        refreshDataManager(chart: Chart): void;
        private dataManagerSuccess(e, chart, isRemoteData?);
        private refreshChart(isRemoteData);
        /** @private */
        xMin: number;
        /** @private */
        xMax: number;
        /** @private */
        yMin: number;
        /** @private */
        yMax: number;
        /** @private */
        xAxis: Axis;
        /** @private */
        yAxis: Axis;
        /** @private */
        chart: Chart;
        /** @private */
        currentViewData: Object;
        /** @private */
        clipRect: Rect;
        /** @private */
        xData: number[];
        /** @private */
        yData: number[];
        /** @private */
        index: number;
        /** @private */
        dataModule: Data;
        /** @private */
        points: Points[];
        /** @private */
        seriesType: SeriesValueType;
        /** @private */
        sizeMax: number;
        /** @private */
        private recordsCount;
    }
    /**
     *  Configures the series in charts.
     */
    export class Series extends SeriesBase {
        /**
         * The name of the series visible in legend.
         * @default ''
         */
        name: string;
        /**
         * The DataSource field that contains the y value.
         * @default ''
         */
        yName: string;
        /**
         * Type of series to be drawn in radar or polar series. They are
         *  'Line'
         *  'Column'
         *  'Area'
         *  'Scatter'
         *  'Spline'
         *  'StackingColumn'
         *  'StackingArea'
         *  'RangeColumn'
         *  'SplineArea'
         * @default 'Line'
         */
        drawType: ChartDrawType;
        /**
         * Specifies whether to join start and end point of a line/area series used in polar/radar chart to form a closed path.
         * @default true
         */
        isClosed: boolean;
        /**
         * This property is used in financial charts to visualize the price movements in stock.
         * It defines the color of the candle/point, when the opening price is less than the closing price.
         * @default '#2ecd71'
         */
        bearFillColor: string;
        /**
         * This property is used in financial charts to visualize the price movements in stock.
         * It defines the color of the candle/point, when the opening price is higher than the closing price.
         * @default '#e74c3d'
         */
        bullFillColor: string;
        /**
         * This property is applicable for candle series.
         * It enables/disables to visually compare the current values with the previous values in stock.
         * @default false
         */
        enableSolidCandles: boolean;
        /**
         * The DataSource field that contains the size value of y
         * @default ''
         */
        size: string;
        /**
         * This property allows grouping series in `stacked column / bar` charts.
         * Any string value can be provided to the stackingGroup property.
         * If any two or above series have the same value, those series will be grouped together.
         * @default ''
         */
        stackingGroup: string;
        /**
         * Specifies the visibility of series.
         * @default true
         */
        visible: boolean;
        /**
         * Options to customizing the border of the series. This is applicable only for `Column` and `Bar` type series.
         */
        border: BorderModel;
        /**
         * The opacity of the series.
         * @default 1
         */
        opacity: number;
        /**
         * The type of the series are
         * * Line
         * * Column
         * * Area
         * * Bar
         * * StackingColumn
         * * StackingArea
         * * StackingBar
         * * StepLine
         * * StepArea
         * * Scatter
         * * Spline
         * * StackingColumn100
         * * StackingBar100
         * * StackingArea100
         * * RangeColumn
         * * Hilo
         * * HiloOpenClose
         * * Waterfall
         * * RangeArea
         * * Bubble
         * * Candle
         * * Polar
         * * Radar
         * * BoxAndWhisker
         * @default 'Line'
         */
        type: ChartSeriesType;
        /**
         * Options for displaying and customizing error bar for individual point in a series.
         */
        errorBar: ErrorBarSettingsModel;
        /**
         * Options for displaying and customizing markers for individual points in a series.
         */
        marker: MarkerSettingsModel;
        /**
         * Defines the collection of trendlines that are used to predict the trend
         */
        trendlines: TrendlineModel[];
        /**
         * If set true, the Tooltip for series will be visible.
         * @default true
         */
        enableTooltip: boolean;
        /**
         * The shape of the legend. Each series has its own legend shape. They are,
         * * Circle
         * * Rectangle
         * * Triangle
         * * Diamond
         * * Cross
         * * HorizontalLine
         * * VerticalLine
         * * Pentagon
         * * InvertedTriangle
         * * SeriesType
         * @default 'SeriesType'
         */
        legendShape: LegendShape;
        /**
         * Custom style for the selected series or points.
         * @default null
         */
        selectionStyle: string;
        /**
         * Minimum radius
         * @default 1
         */
        minRadius: number;
        /**
         * Maximum radius
         * @default 3
         */
        maxRadius: number;
        /**
         * Defines type of spline to be rendered.
         * @default 'Natural'
         */
        splineType: SplineType;
        /**
         * It defines tension of cardinal spline types
         * @default 0.5
         */
        cardinalSplineTension: number;
        /**
         * options to customize the empty points in series
         */
        emptyPointSettings: EmptyPointSettingsModel;
        /**
         * If set true, the mean value for box and whisker will be visible.
         * @default true
         */
        showMean: boolean;
        /**
         * The mode of the box and whisker char series. They are,
         * Exclusive
         * Inclusive
         * Normal
         * @default 'Normal'
         */
        boxPlotMode: BoxPlotMode;
        /**
         * To render the column series points with particular column width.
         * @default 0.7
         */
        columnWidth: number;
        /**
         * To render the column series points with particular column spacing. It takes value from 0 - 1.
         * @default 0
         */
        columnSpacing: number;
        /**
         * Defines the visual representation of the negative changes in waterfall charts.
         * @default '#C64E4A'
         */
        negativeFillColor: string;
        /**
         * Defines the visual representation of the summaries in waterfall charts.
         * @default '#4E81BC'
         */
        summaryFillColor: string;
        /**
         * Defines the collection of indexes of the intermediate summary columns in waterfall charts.
         * @default []
         */
        intermediateSumIndexes: number[];
        /**
         * Defines the collection of indexes of the overall summary columns in waterfall charts.
         * @default []
         */
        sumIndexes: number[];
        /**
         * Defines the appearance of line connecting adjacent points in waterfall charts.
         */
        connector: ConnectorModel;
        /**
         * To render the column series points with particular rounded corner.
         */
        cornerRadius: CornerRadiusModel;
        visibleSeriesCount: number;
        /** @private */
        position: number;
        /** @private */
        rectCount: number;
        /** @private */
        seriesElement: Element;
        /** @private */
        errorBarElement: Element;
        /** @private */
        symbolElement: Element;
        /** @private */
        shapeElement: Element;
        /** @private */
        textElement: Element;
        /** @private */
        pathElement: Element;
        /** @private */
        sourceIndex: number;
        /** @private */
        category: SeriesCategories;
        /** @private */
        isRectSeries: boolean;
        /** @private */
        clipRectElement: Element;
        /** @private */
        stackedValues: StackValues;
        /** @private */
        interior: string;
        /** @private */
        drawPoints: ControlPoints[];
        /** @private */
        delayedAnimation: boolean;
        constructor(parent: any, propName: string, defaultValue: Object, isArray?: boolean);
        /**
         * Refresh the axis label.
         * @return {boolean}
         * @private
         */
        refreshAxisLabel(): void;
        /**
         * To get the series collection.
         * @return {void}
         * @private
         */
        findSeriesCollection(column: Column, row: Row, isStack: boolean): Series[];
        /**
         * To get the column type series.
         * @return {void}
         * @private
         */
        private rectSeriesInChart(series, isStack);
        /**
         * To calculate the stacked values.
         * @return {void}
         * @private
         */
        calculateStackedValue(isStacking100: boolean, chart: Chart): void;
        private calculateStackingValues(seriesCollection, isStacking100);
        private findFrequencies(seriesCollection);
        /**
         * To find the log values.
         * @return {void}
         * @private
         */
        logWithIn(value: number, axis: Axis): number;
        /** @private */
        renderSeries(chart: Chart, index: number): void;
        /**
         * To create seris element.
         * @return {void}
         * @private
         */
        createSeriesElements(chart: Chart): void;
        /**
         * To append the series.
         * @return {void}
         * @private
         */
        appendSeriesElement(element: Element, chart: Chart): void;
        /**
         * To perform animation for chart series.
         * @return {void}
         * @private
         */
        performAnimation(chart: Chart, type: string, errorBar: ErrorBarSettingsModel, marker: MarkerSettingsModel, dataLabel: DataLabelSettingsModel): void;
        /**
         * To set border color for empty point
         * @private
         */
        setPointColor(point: Points, color: string): string;
        /**
         * To set border color for empty point
         * @private
         */
        setBorderColor(point: Points, border: BorderModel): BorderModel;
    }
    
    /**
     * Column Series Base
     */
    export class ColumnBase {
        /**
         * To get the position of the column series.
         * @return {DoubleRange}
         * @private
         */
        protected getSideBySideInfo(series: Series): DoubleRange;
        /**
         * To get the rect values.
         * @return {Rect}
         * @private
         */
        protected getRectangle(x1: number, y1: number, x2: number, y2: number, series: Series): Rect;
        /**
         * To get the position of each series.
         * @return {void}
         * @private
         */
        private getSideBySidePositions(series);
        private findRectPosition(seriesCollection);
        /**
         * Updates the symbollocation for points
         * @return void
         * @private
         */
        protected updateSymbolLocation(point: Points, rect: Rect, series: Series): void;
        /**
         * Update the region for the point.
         * @return {void}
         * @private
         */
        protected updateXRegion(point: Points, rect: Rect, series: Series): void;
        /**
         * Update the region for the point in bar series.
         * @return {void}
         * @private
         */
        protected updateYRegion(point: Points, rect: Rect, series: Series): void;
        /**
         * To trigger the point rendering event.
         * @return {void}
         * @private
         */
        protected triggerEvent(series: Series, point: Points, fill: string, border: BorderModel): IPointRenderEventArgs;
        /**
         * To draw the rectangle for points.
         * @return {void}
         * @private
         */
        protected drawRectangle(series: Series, point: Points, rect: Rect, argsData: IPointRenderEventArgs): void;
        /**
         * To animate the series.
         * @return {void}
         * @private
         */
        animate(series: Series): void;
        /**
         * To animate the series.
         * @return {void}
         * @private
         */
        private animateRect(element, series, point);
        /**
         * To get rounded rect path direction
         */
        private calculateRoundedRectPath(rect, topLeft, topRight, bottomLeft, bottomRight);
    }
    export interface RectPosition {
        position: number;
        rectCount: number;
    }
    
    /**
     * `ColumnSeries` Module used to render the column series.
     */
    export class ColumnSeries extends ColumnBase {
        /**
         * Render Column series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the column series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `DataLabel` module is used to render data label for the data point.
     */
    export class DataLabel {
        private chart;
        private margin;
        private isShape;
        private locationX;
        private locationY;
        private fontBackground;
        private borderWidth;
        private markerHeight;
        private commonId;
        private yAxisInversed;
        private inverted;
        private errorHeight;
        private chartBackground;
        /**
         * Constructor for the data label module.
         * @private
         */
        constructor(chart: Chart);
        private initPrivateVariables(series, marker);
        private calculateErrorHeight(series, position);
        private isRectSeries(series);
        /**
         * Render the data label for series.
         * @return {void}
         */
        render(series: Series, chart: Chart, dataLabel: DataLabelSettingsModel): void;
        /**
         * Render the data label template.
         * @return {void}
         * @private
         */
        private createDataLabelTemplate(parentElement, series, dataLabel, point, data, labelIndex);
        private calculateTextPosition(point, series, textSize, dataLabel, labelIndex);
        private calculatePolarRectPosition(location, position, series, point, size, labelIndex);
        /**
         * Get the label location
         */
        private getLabelLocation(point, series, textSize, labelIndex);
        private calculateRectPosition(labelLocation, rect, isMinus, position, series, textSize, labelIndex, point);
        private calculatePathPosition(labelLocation, position, series, point, size, labelIndex);
        private isDataLabelShape(style);
        private calculateRectActualPosition(labelLocation, rect, isMinus, series, size, labelIndex, point);
        private calculateAlignment(value, labelLocation, alignment, isMinus);
        private calculateTopAndOuterPosition(location, rect, position, series, index, extraSpace, isMinus);
        /**
         * Updates the label location
         */
        private updateLabelLocation(position, location, extraSpace, margin, rect, top, inside?);
        private calculatePathActualPosition(y, markerSize, series, point, size, labelIndex);
        /**
         * Animates the data label.
         * @param  {Series} series - Data label of the series gets animated.
         * @return {void}
         */
        doDataLabelAnimation(series: Series, element?: Element): void;
        private getPosition(index);
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the dataLabel for series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `ErrorBar` module is used to render the error bar for series.
     */
    export class ErrorBar {
        private chart;
        errorHeight: number;
        error: number;
        positiveHeight: number;
        negativeHeight: number;
        /**
         * Constructor for the error bar module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * Render the error bar for series.
         * @return {void}
         */
        render(series: Series): void;
        private renderErrorBar(series);
        private findLocation(point, series, isInverted, x1, y1);
        private calculateFixedValue(point, series, isInverted, errorX, errorY, xAxis, yAxis);
        private calculatePercentageValue(point, series, isInverted, errorX, errorY, xAxis, yAxis);
        private calculateStandardDeviationValue(point, series, isInverted, errorX, errorY, xAxis, yAxis);
        private calculateStandardErrorValue(point, series, isInverted, errorX, errorY, xAxis, yAxis);
        private calculateCustomValue(point, series, isInverted, errorX, errorY, xAxis, yAxis);
        private getHorizontalDirection(start, mid, end, direction, errorMode, capLength);
        private getVerticalDirection(start, mid, end, direction, errorMode, capLength);
        private getBothDirection(start, mid, end, direction, errorMode, capLength);
        private getErrorDirection(start, mid, end, series, isInverted);
        meanCalculation(series: Series, mode: ErrorBarMode): Mean;
        private createElement(series);
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doErrorBarAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the errorBar for series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `HiloOpenCloseSeries` module is used to render the hiloOpenClose series.
     */
    export class HiloOpenCloseSeries extends ColumnBase {
        /**
         * Render HiloOpenCloseSeries series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * Updates the tick region
         */
        private updateTickRegion(horizontal, region, borderWidth);
        /**
         * Trigger point rendering event
         */
        private triggerPointRenderEvent(series, point);
        /**
         * To draw the rectangle for points.
         * @return {void}
         * @private
         */
        protected drawHiloOpenClosePath(series: Series, point: Points, open: ChartLocation, close: ChartLocation, rect: Rect, argsData: IPointRenderEventArgs): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the column series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `HiloSeries` module is used to render the hilo series.
     */
    export class HiloSeries extends ColumnBase {
        /**
         * Render Hiloseries.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * To trigger the point rendering event.
         * @return {void}
         * @private
         */
        private triggerPointRenderEvent(series, point);
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the Hilo series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Base for line type series.
     */
    export class LineBase {
        chart: Chart;
        private padding;
        /** @private */
        constructor(chartModule?: Chart);
        /**
         * To improve the chart performance.
         * @return {void}
         * @private
         */
        improveChartPerformance(series: Series): Points[];
        /**
         * To generate the line path direction
         * @param firstPoint
         * @param secondPoint
         * @param series
         * @param isInverted
         * @param getPointLocation
         * @param startPoint
         */
        getLineDirection(firstPoint: Points, secondPoint: Points, series: Series, isInverted: Boolean, getPointLocation: Function, startPoint: string): string;
        /**
         * To append the line path.
         * @return {void}
         * @private
         */
        appendLinePath(options: PathOption, series: Series, clipRect: string): void;
        /**
         * To render the marker for the series.
         * @return {void}
         * @private
         */
        renderMarker(series: Series): void;
        /**
         * To do the progressive animation.
         * @return {void}
         * @private
         */
        doProgressiveAnimation(series: Series, option: AnimationModel): void;
        /**
         * To store the symbol location and region
         * @param point
         * @param series
         * @param isInverted
         * @param getLocation
         */
        storePointLocation(point: Points, series: Series, isInverted: boolean, getLocation: Function): void;
        /**
         * To do the linear animation.
         * @return {void}
         * @private
         */
        doLinearAnimation(series: Series, animation: AnimationModel): void;
    }
    
    /**
     * `LineSeries` module used to render the line series.
     */
    export class LineSeries extends LineBase {
        /**
         * Render Line Series.
         * @return {void}.
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the line series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Marker Module used to render the marker for line type series.
     */
    export class MarkerExplode extends ChartData {
        private markerExplode;
        private isRemove;
        /** @private */
        elementId: string;
        /**
         * Constructor for the marker module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        /**
         * @hidden
         */
        private mouseUpHandler();
        /**
         * @hidden
         */
        private mouseMoveHandler();
        private markerMove(remove);
        private drawTrackBall(series, point, location, index);
        /**
         * @hidden
         */
        removeHighlightedMarker(): void;
    }
    
    /**
     * Marker module used to render the marker for line type series.
     */
    export class Marker extends MarkerExplode {
        /**
         * Constructor for the marker module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * Render the marker for series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        private renderMarker(series, point, location, index);
        createElement(series: Series): void;
        private getRangeLowPoint(region, series);
        /**
         * Animates the marker.
         * @return {void}.
         * @private
         */
        doMarkerAnimation(series: Series): void;
    }
    
    /**
     * `MultiColoredAreaSeries` module used to render the area series with multi color.
     */
    export class MultiColoredAreaSeries extends MultiColoredSeries {
        /**
         * Render Area series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * To Store the path directions of the area
         */
        private generatePathOption(options, series, point, direction, id);
        /**
         * To destroy the area series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
    }
    
    /**
     * Base class for multi colored series
     */
    export class MultiColoredSeries extends LineBase {
        /**
         * To Generate the area path direction
         * @param xValue
         * @param yValue
         * @param series
         * @param isInverted
         * @param getPointLocation
         * @param startPoint
         * @param startPath
         */
        getAreaPathDirection(xValue: number, yValue: number, series: Series, isInverted: boolean, getPointLocation: Function, startPoint: ChartLocation, startPath: string): string;
        /**
         * To Generate the empty point direction
         * @param firstPoint
         * @param secondPoint
         * @param series
         * @param isInverted
         * @param getPointLocation
         */
        getAreaEmptyDirection(firstPoint: ChartLocation, secondPoint: ChartLocation, series: Series, isInverted: boolean, getPointLocation: Function): string;
        /**
         * To set point color
         * @param points
         */
        setPointColor(currentPoint: Points, previous: Points, series: Series, isXSegment: boolean, segments: ChartSegmentModel[]): boolean;
        sortSegments(series: Series, chartSegments: ChartSegmentModel[]): ChartSegmentModel[];
        /**
         * Segment calculation performed here
         * @param series
         * @param options
         * @param chartSegments
         */
        applySegmentAxis(series: Series, options: PathOption[], segments: ChartSegmentModel[]): void;
        private includeSegment(segments, axis, series, length);
        /**
         * To create clip rect for segment axis
         * @param startValue
         * @param endValue
         * @param series
         * @param index
         * @param isX
         * @param chart
         */
        createClipRect(startValue: number, endValue: number, series: Series, index: number, isX: boolean): string;
        /**
         * To get exact value from segment value
         * @param segmentValue
         * @param axis
         * @param chart
         */
        getAxisValue(segmentValue: Object, axis: Axis, chart: Chart): number;
    }
    
    /**
     * `MultiColoredLineSeries` used to render the line series with multi color.
     */
    export class MultiColoredLineSeries extends MultiColoredSeries {
        /**
         * Render Line Series.
         * @return {void}.
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the line series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `PolarSeries` module is used to render the polar series.
     */
    export class PolarSeries extends PolarRadarPanel {
        /**
         * Render Polar Series.
         * @return {void}.
         * @private
         */
        render(series: Series): void;
        /**
         * Render Column DrawType.
         * @return {void}.
         * @private
         */
        columnDrawTypeRender(series: Series): void;
        /**
         * To trigger the point rendering event.
         * @return {void}
         * @private
         */
        triggerEvent(chart: Chart, series: Series, point: Points): IPointRenderEventArgs;
        /** get position for column drawtypes
         * @return {void}.
         * @private
         */
        getSeriesPosition(series: Series): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To do the Polar Radar draw type column animation.
         * @return {void}
         * @private
         */
        doPolarRadarAnimation(animateElement: Element, delay: number, duration: number, series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the polar series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `RadarSeries` module is used to render the radar series.
     */
    export class RadarSeries extends PolarSeries {
        /**
         * Render radar Series.
         * @return {void}.
         * @private
         */
        render(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the radar series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `RangeAreaSeries` module is used to render the range area series.
     */
    export class RangeAreaSeries extends LineBase {
        /**
         * Render RangeArea Series.
         * @return {void}.
         * @private
         */
        render(series: Series): void;
        /**
         * path for rendering the low points
         * @return {void}.
         * @private
         */
        protected closeRangeAreaPath(visiblePoints: Points[], point: Points, series: Series, direction: string, i: number): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the line series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `RangeColumnSeries` module is used to render the range column series.
     */
    export class RangeColumnSeries extends ColumnBase {
        /**
         * Render Range Column series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the range column series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `ScatterSeries` module is used to render the scatter series.
     */
    export class ScatterSeries {
        /**
         * Render the scatter series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the scatter.
         * @return {void}
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `SplineAreaSeries` module used to render the spline area series.
     */
    export class SplineAreaSeries extends SplineBase {
        /**
         * Render the splineArea series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the spline.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * render Line series
     */
    export class SplineBase extends LineBase {
        private splinePoints;
        /** @private */
        constructor(chartModule?: Chart);
        /**
         * To find the control points for spline.
         * @return {void}
         * @private
         */
        findSplinePoint(series: Series): void;
        protected getPreviousIndex(points: Points[], i: number, series: Series): number;
        getNextIndex(points: Points[], i: number, series: Series): number;
        filterEmptyPoints(series: Series): Points[];
        /**
         * To find the natural spline.
         * @return {void}
         * @private
         */
        findSplineCoefficients(points: Points[], series: Series): number[];
        /**
         * To find the control points for spline.
         * @return {void}
         * @private
         */
        getControlPoints(point1: Points, point2: Points, ySpline1: number, ySpline2: number, series: Series): ControlPoints;
        /**
         * calculate datetime interval in hours
         *
         */
        protected dateTimeInterval(series: Series): number;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
    }
    
    /**
     * `SplineSeries` module is used to render the spline series.
     */
    export class SplineSeries extends SplineBase {
        /**
         * Render the spline series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the spline.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `StackingAreaSeries` module used to render the Stacking Area series.
     */
    export class StackingAreaSeries extends LineBase {
        /**
         * Render the Stacking area series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the stacking area.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To find previous visible series
         */
        private getPreviousSeries(series);
    }
    
    /**
     * `StackingBarSeries` module is used to render the stacking bar series.
     */
    export class StackingBarSeries extends ColumnBase {
        /**
         * Render the Stacking bar series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * To destroy the stacking bar.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
    }
    
    /**
     * `StackingColumnSeries` module used to render the stacking column series.
     */
    export class StackingColumnSeries extends ColumnBase {
        /**
         * Render the Stacking column series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the stacking column.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `StepAreaSeries` Module used to render the step area series.
     */
    export class StepAreaSeries extends LineBase {
        /**
         * Render StepArea series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the step Area series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `StepLineSeries` module is used to render the step line series.
     */
    export class StepLineSeries extends LineBase {
        /**
         * Render the Step line series.
         * @return {void}
         * @private
         */
        render(series: Series, xAxis: Axis, yAxis: Axis, isInverted: boolean): void;
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * To destroy the step line series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `WaterfallSeries` module is used to render the waterfall series.
     */
    export class WaterfallSeries extends ColumnBase {
        /**
         * Render waterfall series.
         * @return {void}
         * @private
         */
        render(series: Series): void;
        /**
         * To check intermediateSumIndex in waterfall series.
         * @return boolean
         * @private
         */
        private isIntermediateSum(series, index);
        /**
         * To check sumIndex in waterfall series.
         * @return boolean
         * @private
         */
        private isSumIndex(series, index);
        /**
         * To trigger the point rendering event for waterfall series.
         * @return IPointRenderEventArgs
         * @private
         */
        private triggerPointRenderEvent(series, point);
        /**
         * Add sumIndex and intermediateSumIndex data.
         * @return {object[]}
         * @private
         */
        processWaterfallData(json: Object[], series: Series): Object[];
        /**
         * Animates the series.
         * @param  {Series} series - Defines the series to animate.
         * @return {void}
         */
        doAnimation(series: Series): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the waterfall series.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `AccumulationDistributionIndicator` module is used to render accumulation distribution indicator.
     */
    export class AccumulationDistributionIndicator extends TechnicalAnalysis {
        /**
         * Defines the predictions using accumulation distribution approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         *  Calculates the Accumulation Distribution values
         * @private
         */
        private calculateADPoints(indicator, validData);
        /**
         * To destroy the Accumulation Distribution Technical Indicator.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `AtrIndicator` module is used to render ATR indicator.
     */
    export class AtrIndicator extends TechnicalAnalysis {
        /**
         * Defines the predictions using Average True Range approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         *  To calculate Average True Range indicator points
         * @private
         */
        private calculateATRPoints(indicator, validData);
        /**
         * To destroy the Average true range indicator.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `BollingerBands` module is used to render bollinger band indicator.
     */
    export class BollingerBands extends TechnicalAnalysis {
        /**
         * Initializes the series collection to represent bollinger band
         */
        initSeriesCollection(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Defines the predictions using Bollinger Band Approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * To destroy the Bollinger Band.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `EmaIndicator` module is used to render EMA indicator.
     */
    export class EmaIndicator extends TechnicalAnalysis {
        /**
         * Defines the predictions based on EMA approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * To destroy the EMA Indicator
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * Technical Analysis module helps to predict the market trend
     */
    export class TechnicalAnalysis extends LineBase {
        /**
         * Defines the collection of series, that are used to represent the given technical indicator
         * @private
         */
        initSeriesCollection(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Initializes the properties of the given series
         * @private
         */
        protected setSeriesProperties(series: Series, indicator: TechnicalIndicator, name: string, fill: string, width: number, chart: Chart): void;
        /**
         * Creates the elements of a technical indicator
         * @private
         */
        createIndicatorElements(chart: Chart, indicator: TechnicalIndicator, index: number): void;
        protected getDataPoint(x: Object, y: Object, sourcePoint: Points, series: Series, index: number, indicator?: TechnicalIndicator): Points;
        protected getRangePoint(x: Object, high: Object, low: Object, sourcePoint: Points, series: Series, index: number, indicator?: TechnicalIndicator): Points;
        protected setSeriesRange(points: Points[], indicator: TechnicalIndicator, series?: Series): void;
    }
    
    /**
     * `MacdIndicator` module is used to render MACD indicator.
     */
    export class MacdIndicator extends TechnicalAnalysis {
        /**
         * Defines the collection of series to represent the MACD indicator
         * @private
         */
        initSeriesCollection(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Defines the predictions using MACD approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Calculates the EMA values for the given period
         */
        private calculateEMAValues(period, validData, field);
        /**
         * Defines the MACD Points
         */
        private getMACDPoints(indicator, macdPoints, validData, series);
        /**
         * Calculates the signal points
         */
        private getSignalPoints(indicator, signalEma, validData, series);
        /**
         * Calculates the MACD values
         */
        private getMACDVales(indicator, shortEma, longEma);
        /**
         * Calculates the Histogram Points
         */
        private getHistogramPoints(indicator, macdPoints, signalEma, validData, series);
        /**
         * To destroy the MACD Indicator.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `MomentumIndicator` module is used to render Momentum indicator.
     */
    export class MomentumIndicator extends TechnicalAnalysis {
        /**
         * Defines the collection of series to represent a momentum indicator
         * @private
         */
        initSeriesCollection(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Defines the predictions using momentum approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * To destroy the momentum indicator
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `RsiIndicator` module is used to render RSI indicator.
     */
    export class RsiIndicator extends TechnicalAnalysis {
        /**
         * Initializes the series collection to represent the RSI Indicator
         * @private
         */
        initSeriesCollection(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Defines the predictions using RSI approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * To destroy the RSI Indicator.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `SmaIndicator` module is used to render SMA indicator.
     */
    export class SmaIndicator extends TechnicalAnalysis {
        /**
         * Defines the predictions based on SMA approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * To destroy the SMA indicator
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `StochasticIndicator` module is used to render stochastic indicator.
     */
    export class StochasticIndicator extends TechnicalAnalysis {
        /**
         * Defines the collection of series that represents the stochastic indicator
         * @private
         */
        initSeriesCollection(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Defines the predictions based on stochastic approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * Calculates the SMA Values
         * @private
         */
        private smaCalculation(period, kPeriod, data, sourceSeries);
        /**
         * Calculates the period line values.
         * @private
         */
        private calculatePeriod(period, kPeriod, data, series);
        /**
         * To destroy the Stocastic Indicator.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * Interface for a class TechnicalIndicator
     */
    export interface TechnicalIndicatorModel extends SeriesBaseModel{
    
        /**
         * Defines the type of the technical indicator
         * @default 'Sma'
         */
        type?: TechnicalIndicators;
    
        /**
         * Defines the period, the price changes over which will be considered to predict the trend
         * @default 14
         */
        period?: number;
    
        /**
         * Defines the look back period, the price changes over which will define the %K value in stochastic indicators
         * @default 14
         */
        kPeriod?: number;
    
        /**
         * Defines the period, the price changes over which will define the %D value in stochastic indicators
         * @default 3
         */
        dPeriod?: number;
    
        /**
         * Defines the over-bought(threshold) values. It is applicable for RSI and stochastic indicators
         * @default 80
         */
        overBought?: number;
    
        /**
         * Defines the over-sold(threshold) values. It is applicable for RSI and stochastic indicators
         * @default 20
         */
        overSold?: number;
    
        /**
         * Sets the standard deviation values that helps to define the upper and lower bollinger bands
         * @default 2
         */
        standardDeviation?: number;
    
        /**
         * Defines the field to compare the current value with previous values
         * @default 'Close'
         */
        field?: FinancialDataFields;
    
        /**
         * Sets the slow period to define the Macd line
         * @default 12
         */
        slowPeriod?: number;
    
        /**
         * Sets the fast period to define the Macd line
         * @default 26
         */
        fastPeriod?: number;
    
        /**
         * Enables/Disables the over-bought and over-sold regions
         * @default true
         */
        showZones?: boolean;
    
        /**
         * Defines the appearance of the the MacdLine of Macd indicator
         * @default { color: '#ff9933', width: 2 }
         */
        macdLine?: ConnectorModel;
    
        /**
         * Defines the type of the Macd indicator.
         * @default 'Both'
         */
        macdType?: MacdType;
    
        /**
         * Defines the color of the positive bars in Macd indicators
         */
        macdPositiveColor?: string;
    
        /**
         * Defines the color of the negative bars in Macd indicators
         */
        macdNegativeColor?: string;
    
        /**
         * Options for customizing the BollingerBand in the indicator.
         * @default 'rgba(211,211,211,0.25)'
         */
    
        bandColor?: string;
    
        /**
         * Defines the appearance of the upper line in technical indicators
         */
        upperLine?: ConnectorModel;
    
        /**
         * Defines the appearance of lower line in technical indicators
         */
    
        lowerLine?: ConnectorModel;
    
        /**
         * Defines the appearance of period line in technical indicators
         */
    
        periodLine?: ConnectorModel;
    
        /**
         * Defines the name of the series, the data of which has to be depicted as indicator
         * @default ''
         */
        seriesName?: string;
    
    }
    
    /**
     * Defines how to represent the market trend using technical indicators
     */
    export class TechnicalIndicator extends SeriesBase {
        /**
         * Defines the type of the technical indicator
         * @default 'Sma'
         */
        type: TechnicalIndicators;
        /**
         * Defines the period, the price changes over which will be considered to predict the trend
         * @default 14
         */
        period: number;
        /**
         * Defines the look back period, the price changes over which will define the %K value in stochastic indicators
         * @default 14
         */
        kPeriod: number;
        /**
         * Defines the period, the price changes over which will define the %D value in stochastic indicators
         * @default 3
         */
        dPeriod: number;
        /**
         * Defines the over-bought(threshold) values. It is applicable for RSI and stochastic indicators
         * @default 80
         */
        overBought: number;
        /**
         * Defines the over-sold(threshold) values. It is applicable for RSI and stochastic indicators
         * @default 20
         */
        overSold: number;
        /**
         * Sets the standard deviation values that helps to define the upper and lower bollinger bands
         * @default 2
         */
        standardDeviation: number;
        /**
         * Defines the field to compare the current value with previous values
         * @default 'Close'
         */
        field: FinancialDataFields;
        /**
         * Sets the slow period to define the Macd line
         * @default 12
         */
        slowPeriod: number;
        /**
         * Sets the fast period to define the Macd line
         * @default 26
         */
        fastPeriod: number;
        /**
         * Enables/Disables the over-bought and over-sold regions
         * @default true
         */
        showZones: boolean;
        /**
         * Defines the appearance of the the MacdLine of Macd indicator
         * @default { color: '#ff9933', width: 2 }
         */
        macdLine: ConnectorModel;
        /**
         * Defines the type of the Macd indicator.
         * @default 'Both'
         */
        macdType: MacdType;
        /**
         * Defines the color of the positive bars in Macd indicators
         */
        macdPositiveColor: string;
        /**
         * Defines the color of the negative bars in Macd indicators
         */
        macdNegativeColor: string;
        /**
         * Options for customizing the BollingerBand in the indicator.
         * @default 'rgba(211,211,211,0.25)'
         */
        bandColor: string;
        /**
         * Defines the appearance of the upper line in technical indicators
         */
        upperLine: ConnectorModel;
        /**
         * Defines the appearance of lower line in technical indicators
         */
        lowerLine: ConnectorModel;
        /**
         * Defines the appearance of period line in technical indicators
         */
        periodLine: ConnectorModel;
        /**
         * Defines the name of the series, the data of which has to be depicted as indicator
         * @default ''
         */
        seriesName: string;
        /** @private */
        targetSeries: Series[];
        /** @private */
        sourceSeries: Series;
        /** @private */
        indicatorElement: Element;
        /** @private */
        clipRectElement: Element;
        /** @private */
        clipRect: Rect;
        /** @private */
        setDataSource(series: Series, chart: Chart): void;
    }
    
    /**
     * `TmaIndicator` module is used to render TMA indicator.
     */
    export class TmaIndicator extends TechnicalAnalysis {
        /**
         * Defines the predictions based on TMA approach
         * @private
         */
        initDataSource(indicator: TechnicalIndicator, chart: Chart): void;
        /**
         * To destroy the TMA indicator.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
    }
    
    /**
     * `Trendline` module is used to render 6 types of trendlines in chart.
     */
    export class Trendlines {
        /**
         * Defines the collection of series, that are used to represent a trendline
         * @private
         */
        initSeriesCollection(trendline: Trendline, chart: Chart): void;
        /**
         * Initializes the properties of the trendline series
         */
        private setSeriesProperties(series, trendline, name, fill, width, chart);
        /**
         * Creates the elements of a trendline
         */
        private createTrendLineElements(chart, trendline, index, element, clipRectElement);
        /**
         * Defines the data point of trendline
         */
        private getDataPoint(x, y, sourcePoint, series, index);
        /**
         * Finds the slope and intercept of trendline
         */
        private findSlopeIntercept(xValues, yValues, trendline, points);
        /**
         * Defines the points to draw the trendlines
         */
        initDataSource(trendline: Trendline, chart: Chart): void;
        /**
         * Calculation of exponential points
         */
        private setExponentialRange(points, trendline, series);
        /**
         * Calculation of logarithmic points
         */
        private setLogarithmicRange(points, trendline, series);
        /**
         * Calculation of polynomial points
         */
        private setPolynomialRange(points, trendline, series);
        /**
         * Calculation of power points
         */
        private setPowerRange(points, trendline, series);
        /**
         * Calculation of linear points
         */
        private setLinearRange(points, trendline, series);
        /**
         * Calculation of moving average points
         */
        private setMovingAverageRange(points, trendline, series);
        /**
         * Calculation of logarithmic points
         */
        private getLogarithmicPoints(trendline, points, xValues, yValues, series, slopeInterceptLog);
        /**
         * Defines the points based on data point
         */
        private getPowerPoints(trendline, points, xValues, yValues, series, slopeInterceptPower);
        /**
         * Get the polynomial points based on polynomial slopes
         */
        private getPolynomialPoints(trendline, points, xValues, yValues, series);
        /**
         * Defines the moving average points
         */
        private getMovingAveragePoints(trendline, points, xValues, yValues, series);
        /**
         * Defines the linear points
         */
        private getLinearPoints(trendline, points, xValues, yValues, series, slopeInterceptLinear);
        /**
         * Defines the exponential points
         */
        private getExponentialPoints(trendline, points, xValues, yValues, series, slopeInterceptExp);
        /**
         * Defines the points based on data point
         */
        private getPoints(trendline, points, xValues, yValues, series);
        /**
         * Defines the polynomial value of y
         */
        private getPolynomialYValue(slopes, x);
        /**
         * Defines the gauss jordan elimination
         */
        private gaussJordanElimination(matrix, polynomialSlopes);
        /**
         * Defines the trendline elements
         */
        getTrendLineElements(series: Series, chart: Chart): void;
        /**
         * To destroy the trendline
         */
        destroy(chart: Chart): void;
        /**
         * Get module name
         */
        protected getModuleName(): string;
    }
    /** @private */
    export interface SlopeIntercept {
        slope?: number;
        intercept?: number;
    }
    
    /**
     * `Crosshair` module is used to render the crosshair for chart.
     */
    export class Crosshair {
        private elementID;
        private elementSize;
        private crosshairInterval;
        private arrowLocation;
        private isTop;
        private isBottom;
        private isLeft;
        private isRight;
        private valueX;
        private valueY;
        private rx;
        private ry;
        private chart;
        /**
         * Constructor for crosshair module.
         * @private
         */
        constructor(chart: Chart);
        /**
         * @hidden
         */
        private addEventListener();
        private mouseUpHandler();
        private mouseLeaveHandler();
        private mouseMoveHandler(event);
        /**
         * Handles the long press on chart.
         * @return {boolean}
         * @private
         */
        private longPress();
        /**
         * Renders the crosshair.
         * @return {void}
         */
        crosshair(): void;
        private renderCrosshairLine(options, crossGroup);
        private renderAxisTooltip(chart, chartRect, axisGroup);
        private getAxisText(axis);
        private tooltipLocation(text, axis, bounds, axisRect);
        private stopAnimation();
        /**
         * Removes the crosshair on mouse leave.
         * @return {void}
         * @private
         */
        removeCrosshair(duration: number): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the crosshair.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `Selection` module handles the selection for chart.
     * @private
     */
    export class Selection extends BaseSelection {
        private renderer;
        private isSeriesMode;
        private resizing;
        /** @private */
        rectPoints: Rect;
        private closeIconId;
        private closeIcon;
        private draggedRectGroup;
        private draggedRect;
        /** @private */
        selectedDataIndexes: Indexes[];
        private series;
        private dragging;
        private dragRect;
        private rectGrabbing;
        private resizeMode;
        private chart;
        /**
         * Constructor for selection module.
         * @private.
         */
        constructor(chart: Chart);
        /**
         * Binding events for selection module.
         */
        private addEventListener();
        /**
         * Chart mouse down
         */
        private mousedown(e);
        /**
         * UnBinding events for selection module.
         */
        private removeEventListener();
        /**
         * To find private variable values
         */
        private initPrivateVariables(chart);
        /**
         * Method to select the point and series.
         * @return {void}
         */
        invokeSelection(chart: Chart): void;
        private generateStyle(series);
        private selectDataIndex(chart, indexes);
        private getElementByIndex(chart, index, suffix?);
        private getClusterElements(chart, index);
        private findElements(chart, series, index, suffix?);
        /**
         * To find the selected element.
         * @return {void}
         * @private
         */
        calculateSelectedElements(event: Event): void;
        private performSelection(index, chart, element?);
        private selection(chart, index, selectedElements);
        private clusterSelection(chart, series, index);
        private removeMultiSelectEelments(chart, index, currentIndex, seriesCollection);
        private blurEffect(chartId, visibleSeries);
        private checkSelectionElements(element, className, visibility);
        private applyStyles(elements);
        private getSelectionClass(id);
        private removeStyles(elements);
        private addOrRemoveIndex(indexes, index, add?);
        private toEquals(first, second, checkSeriesOnly);
        /**
         * To redraw the selected points.
         * @return {void}
         * @private
         */
        redrawSelection(chart: Chart, oldMode: SelectionMode): void;
        /** @private */
        legendSelection(chart: Chart, series: number): void;
        private getSeriesElements(series);
        private indexFinder(id);
        /**
         * Drag selection that returns the selected data.
         * @return {void}
         * @private
         */
        calculateDragSelectedElements(chart: Chart, dragRect: Rect): void;
        private removeOffset(rect, clip);
        /**
         * Method to draw dragging rect.
         * @return {void}
         * @private
         */
        drawDraggingRect(chart: Chart, dragRect: Rect): void;
        private createCloseButton(x, y);
        /**
         * Method to remove dragged element.
         * @return {void}
         * @private
         */
        removeDraggedElements(chart: Chart, event: Event): void;
        /**
         * Method to resize the drag rect.
         * @return {void}
         * @private
         */
        resizingSelectionRect(chart: Chart, location: ChartLocation, tapped?: boolean): void;
        private findResizeMode(chartSvgObject, rect, location);
        private changeCursorStyle(isResize, rectelement, cursorStyle);
        private removeSelectedElements(chart, index, seriesCollection);
        private setAttributes(ele, object);
        /**
         * Method to move the dragged rect.
         * @return {void}
         * @private
         */
        draggedRectMoved(chart: Chart, grabbedPoint: Rect, doDrawing?: boolean): void;
        /**
         * To complete the selection.
         * @return {void}
         * @private
         */
        completeSelection(e: Event): void;
        private getDragRect(chart, seriesClipRect);
        /** @private */
        dragStart(chart: Chart, seriesClipRect: Rect, mouseDownX: number, mouseDownY: number, event: Event): void;
        /** @private */
        mouseMove(event: PointerEvent | TouchEvent): void;
        /**
         * Get module name.
         * @private
         */
        getModuleName(): string;
        /**
         * To destroy the selection.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * `Tooltip` module is used to render the tooltip for chart series.
     */
    export class Tooltip extends BaseTooltip {
        /**
         * Constructor for tooltip module.
         * @private.
         */
        constructor(chart: Chart);
        /**
         * @hidden
         */
        private addEventListener();
        private mouseUpHandler();
        private mouseLeaveHandler();
        private mouseMoveHandler();
        /**
         * Handles the long press on chart.
         * @return {boolean}
         * @private
         */
        private longPress();
        /**
         * Renders the tooltip.
         * @return {void}
         */
        tooltip(): void;
        private findHeader(data);
        private renderSeriesTooltip(chart, isFirst, tooltipDiv);
        private findData(data, previous);
        private getSymbolLocation(data);
        private getTooltipText(pointData);
        private getTemplateText(data);
        private findMouseValue(data, chart);
        private renderGroupedTooltip(chart, isFirst, tooltipDiv);
        private getBoxLocation(data);
        private drawMarker(isBottom, isRight, size);
        private findColor(data, series);
        private parseTemplate(point, series, format, xAxis, yAxis);
        private formatPointValue(point, axis, dataValue, isXPoint, isYPoint);
        private getFormat(chart, series);
        private getIndicatorTooltipFormat(series, chart, format);
        removeHighlightedMarker(data: PointData[]): void;
        /**
         * Removes the tooltip on mouse leave.
         * @return {void}
         * @private
         */
        removeTooltip(duration: number): void;
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the tooltip.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Zooming Toolkit created here
     * @private
     */
    export class Toolkit {
        private chart;
        private selectionColor;
        private fillColor;
        private elementOpacity;
        private elementId;
        private zoomInElements;
        private zoomOutElements;
        private zoomElements;
        private panElements;
        /** @private */
        constructor(chart: Chart);
        /**
         * To create the pan button.
         * @return {void}
         * @private
         */
        createPanButton(childElement: Element, parentElement: Element, chart: Chart): void;
        /**
         * To create the zoom button.
         * @return {void}
         * @private
         */
        createZoomButton(childElement: Element, parentElement: Element, chart: Chart): void;
        /**
         * To create the ZoomIn button.
         * @return {void}
         * @private
         */
        createZoomInButton(childElement: Element, parentElement: Element, chart: Chart): void;
        /**
         * To create the ZoomOut button.
         * @return {void}
         * @private
         */
        createZoomOutButton(childElement: Element, parentElement: Element, chart: Chart): void;
        /**
         * To create the Reset button.
         * @return {void}
         * @private
         */
        createResetButton(childElement: Element, parentElement: Element, chart: Chart, isDevice: Boolean): void;
        /**
         * To bind events.
         * @return {void}
         * @private
         */
        wireEvents(element: Element, process: Function): void;
        /**
         * To show tooltip.
         * @return {void}
         * @private
         */
        private showTooltip(event);
        /** @private */
        removeTooltip(): void;
        /** @private */
        reset(): boolean;
        private zoomIn(e);
        private zoomOut(e);
        private zoom(e);
        /** @private */
        pan(): boolean;
        private zoomInOutCalculation(scale, chart, axes, mode);
        private applySelection(elements, color);
    }
    
    /**
     * `Zooming` module handles the zooming for chart.
     */
    export class Zoom {
        private chart;
        private zooming;
        private elementId;
        /** @private */
        zoomingRect: Rect;
        /** @private */
        toolkit: Toolkit;
        /** @private */
        toolkitElements: Element;
        /** @private */
        isPanning: boolean;
        /** @private */
        isZoomed: boolean;
        /** @private */
        isPointer: Boolean;
        /** @private */
        pinchTarget: Element;
        /** @private */
        isDevice: Boolean;
        /** @private */
        browserName: string;
        /** @private */
        touchStartList: ITouches[] | TouchList;
        /** @private */
        touchMoveList: ITouches[] | TouchList;
        /** @private */
        offset: Rect;
        /** @private */
        zoomAxes: IZoomAxisRange[];
        /** @private */
        isIOS: Boolean;
        /** @private */
        performedUI: boolean;
        private zoomkitOpacity;
        private wheelEvent;
        private cancelEvent;
        /**
         * Constructor for Zooming module.
         * @private.
         */
        constructor(chart: Chart);
        /**
         * Function that handles the Rectangular zooming.
         * @return {void}
         */
        renderZooming(e: PointerEvent | TouchEvent, chart: Chart, isTouch: boolean): void;
        private drawZoomingRectangle(chart);
        private doPan(chart, axes);
        /**
         * Redraw the chart on zooming.
         * @return {void}
         * @private
         */
        performZoomRedraw(chart: Chart): void;
        private refreshAxis(layout, chart, axes);
        private doZoom(chart, axes, bounds);
        /**
         * Function that handles the Mouse wheel zooming.
         * @return {void}
         * @private
         */
        performMouseWheelZooming(e: WheelEvent, mouseX: number, mouseY: number, chart: Chart, axes: AxisModel[]): void;
        /**
         * Function that handles the Pinch zooming.
         * @return {void}
         * @private
         */
        performPinchZooming(e: TouchEvent, chart: Chart): boolean;
        private calculatePinchZoomFactor(chart, pinchRect);
        private setTransform(transX, transY, scaleX, scaleY, chart, isPinch);
        private calculateZoomAxesRange(chart, axes);
        private showZoomingToolkit(chart);
        /**
         * To the show the zooming toolkit.
         * @return {void}
         * @private
         */
        applyZoomToolkit(chart: Chart, axes: AxisModel[]): void;
        /**
         * Return boolean property to show zooming toolkit.
         * @return {void}
         * @private
         */
        isAxisZoomed(axes: AxisModel[]): boolean;
        private zoomToolkitMove(e);
        private zoomToolkitLeave(e);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        /**
         * Handles the mouse wheel on chart.
         * @return {boolean}
         * @private
         */
        chartMouseWheel(e: WheelEvent): boolean;
        /**
         * @hidden
         */
        private mouseMoveHandler(e);
        /**
         * @hidden
         */
        private mouseDownHandler(e);
        /**
         * @hidden
         */
        private mouseUpHandler(e);
        /**
         * @hidden
         */
        private mouseCancelHandler(e);
        /**
         * Handles the touch pointer.
         * @return {boolean}
         * @private
         */
        addTouchPointer(touchList: ITouches[], e: PointerEvent, touches: TouchList): ITouches[];
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the zooming.
         * @return {void}
         * @private
         */
        destroy(chart: Chart): void;
    }
    
    /**
     * Numeric Range.
     * @private
     */
    export class DoubleRange {
        private mStart;
        private mEnd;
        /** @private */
        readonly start: number;
        /** @private */
        readonly end: number;
        /** @private */
        readonly delta: number;
        /** @private */
        readonly median: number;
        constructor(start: number, end: number);
    }
    
    /**
     * Defines Orientation of axis. They are
     * * horizontal
     * * vertical
     * @private
     */
    export type Orientation = 'Horizontal' | 'Vertical';
    /**
     * Defines area type of chart. They are
     * * none
     * * cartesianAxes
     * * polarAxes
     * @private
     */
    export type ChartAreaType = 'CartesianAxes' | 'PolarAxes';
    /**
     * Defines series type of chart. They are
     * * xy
     * * highLow
     * @private
     */
    export type SeriesValueType = 'XY' | 'HighLow' | 'HighLowOpenClose' | 'BoxPlot';
    /**
     * Defines the range padding of axis. They are
     * * none - Padding cannot be applied to the axis.
     * * normal - Padding is applied to the axis based on the range calculation.
     * * additional - Interval of the axis is added as padding to the minimum and maximum values of the range.
     * * round - Axis range is rounded to the nearest possible value divided by the interval.
     */
    export type ChartRangePadding = 'Auto' | 'None' | 'Normal' | 'Additional' | 'Round';
    /**
     * Defines the segment axis. They are,
     * * X - Segment calculation rendered based on horizontal axis
     * * Y - Segment calculation rendered based on vertical axis
     */
    export type Segment = 'X' | 'Y';
    /**
     * Defines the type series in chart. They are
     * * line - Renders the line series.
     * * column - Renders the column series.
     * * area - Renders the area series.
     * * pie - Renders the pie series.
     * * polar - Renders the polar series.
     * * radar - Renders the radar series.
     * * bar - Renders the stacking column series
     * * stackingColumn - Renders the stacking column series.
     * * stackingArea - Renders the stacking area series
     * * stackingBar - Renders the stacking bar series.
     * * StackingColumn100 - Renders the stacking column series.
     * * StackingArea100 - Renders the stacking area 100 percent series
     * * StackingBar100 - Renders the stacking bar 100 percent series.
     * * stepLine -  Renders the step line series.
     * * stepArea -  Renders the step area series.
     * * scatter - Renders the scatter series.
     * * spline - Renders the spline series
     * * rangeColumn - Renders the rangeColumn series.
     * * hilo - Renders the hilo series
     * * hiloOpenClose - Renders the HiloOpenClose Series
     * * Waterfall - Renders the Waterfall Series
     * * rangeArea - Renders the rangeArea series.
     */
    export type ChartSeriesType = 'Line' | 'Column' | 'Area' | 'Bar' | 'StackingColumn' | 'StackingArea' | 'StackingBar' | 'StepLine' | 'StepArea' | 'SplineArea' | 'Scatter' | 'Spline' | 'StackingColumn100' | 'StackingBar100' | 'StackingArea100' | 'RangeColumn' | 'Hilo' | 'HiloOpenClose' | 'Waterfall' | 'RangeArea' | 'Bubble' | 'Candle' | 'Polar' | 'Radar' | 'BoxAndWhisker' | 'MultiColoredLine' | 'MultiColoredArea';
    /**
     * * Type of series to be drawn in radar or polar series. They are
     * * line - Renders the line series.
     * * column - Renders the column series.
     * * area - Renders the area series.
     * * scatter - Renders the scatter series.
     * * spline - Renders the spline series.
     * * stackingColumn - Renders the stacking column series.
     * * stackingArea - Renders the stacking area series.
     * * rangeColumn - Renders the range column series.
     * * splineArea - Renders the spline area series.
     */
    export type ChartDrawType = 'Line' | 'Column' | 'StackingColumn' | 'Area' | 'Scatter' | 'RangeColumn' | 'Spline' | 'SplineArea' | 'StackingArea';
    /**
     * Defines the Edge Label Placement for an axis. They are
     * * none - No action will be perform.
     * * hide - Edge label will be hidden.
     * * shift - Shift the edge labels.
     */
    export type EdgeLabelPlacement = 'None' | 'Hide' | 'Shift';
    /**
     * Defines the Label Placement for category axis. They are
     * * betweenTicks - Render the label between the ticks.
     * * onTicks - Render the label on the ticks.
     */
    export type LabelPlacement = 'BetweenTicks' | 'OnTicks';
    /**
     * Defines the shape of marker. They are
     * * circle - Renders a circle.
     * * rectangle - Renders a rectangle.
     * * triangle - Renders a triangle.
     * * diamond - Renders a diamond.
     * * cross - Renders a cross.
     * * horizontalLine - Renders a horizontalLine.
     * * verticalLine - Renders a verticalLine.
     * * pentagon- Renders a pentagon.
     * * invertedTriangle - Renders a invertedTriangle.
     * * image - Renders a image.
     */
    export type ChartShape = 'Circle' | 'Rectangle' | 'Triangle' | 'Diamond' | 'Cross' | 'HorizontalLine' | 'VerticalLine' | 'Pentagon' | 'InvertedTriangle' | 'Image';
    /**
     * Defines the type of axis. They are
     * * double -  Renders a numeric axis.
     * * dateTime - Renders a dateTime axis.
     * * category - Renders a category axis.
     * * logarithmic - Renders a log axis.
     * * DateTimeCategory - Renders a datetime DateTimeCategory axis
     */
    export type ValueType = 'Double' | 'DateTime' | 'Category' | 'Logarithmic' | 'DateTimeCategory';
    /**
     * Defines the type of error bar. They are
     * * fixed -  Renders a fixed type error bar.
     * * percentage - Renders a percentage type error bar.
     * * standardDeviation - Renders a standard deviation type error bar.
     * * standardError -Renders a standard error type error bar.
     * * custom -Renders a custom type error bar.
     */
    export type ErrorBarType = 'Fixed' | 'Percentage' | 'StandardDeviation' | 'StandardError' | 'Custom';
    /**
     * Defines the direction of error bar. They are
     * * both -  Renders both direction of error bar.
     * * minus - Renders minus direction of error bar.
     * * plus - Renders plus direction error bar.
     */
    export type ErrorBarDirection = 'Both' | 'Minus' | 'Plus';
    /**
     * Defines the modes of error bar. They are
     * * vertical -  Renders a vertical error bar.
     * * horizontal - Renders a horizontal error bar.
     * * both - Renders both side error bar.
     */
    export type ErrorBarMode = 'Vertical' | 'Horizontal' | 'Both';
    /**
     * Defines the interval type of datetime axis. They are
     * * auto - Define the interval of the axis based on data.
     * * years - Define the interval of the axis in years.
     * * months - Define the interval of the axis in months.
     * * days - Define the interval of the axis in days.
     * * hours - Define the interval of the axis in hours.
     * * minutes - Define the interval of the axis in minutes.
     */
    export type IntervalType = 'Auto' | 'Years' | 'Months' | 'Days' | 'Hours' | 'Minutes' | 'Seconds';
    /**
     * Defines the mode of line in crosshair. They are
     * * none - Hides both vertical and horizontal crosshair line.
     * * both - Shows both vertical and horizontal crosshair line.
     * * vertical - Shows the vertical line.
     * * horizontal - Shows the horizontal line.
     */
    export type LineType = 'None' | 'Both' | 'Vertical' | 'Horizontal';
    export type MacdType = 'Line' | 'Histogram' | 'Both';
    /**
     * Defines the position of the legend. They are
     * * auto - Places the legend based on area type.
     * * top - Displays the legend on the top of chart.
     * * left - Displays the legend on the left of chart.
     * * bottom - Displays the legend on the bottom of chart.
     * * right - Displays the legend on the right of chart.
     * * custom - Displays the legend  based on given x and y value.
     */
    export type LegendPosition = 'Auto' | 'Top' | 'Left' | 'Bottom' | 'Right' | 'Custom';
    /**
     * Defines the shape of legend. They are
     * * circle - Renders a circle.
     * * rectangle - Renders a rectangle.
     * * triangle - Renders a triangle.
     * * diamond - Renders a diamond.
     * * cross - Renders a cross.
     * * horizontalLine - Renders a horizontalLine.
     * * verticalLine - Renders a verticalLine.
     * * pentagon - Renders a pentagon.
     * * invertedTriangle - Renders a invertedTriangle.
     * * SeriesType -Render a legend shape based on series type.
     */
    export type LegendShape = 'Circle' | 'Rectangle' | 'Triangle' | 'Diamond' | 'Cross' | 'HorizontalLine' | 'VerticalLine' | 'Pentagon' | 'InvertedTriangle' | 'SeriesType';
    /**
     * Defines the zooming mode, They are.
     * * x,y - Chart will be zoomed with respect to both vertical and horizontal axis.
     * * x - Chart will be zoomed with respect to horizontal axis.
     * * y - Chart will be zoomed with respect to vertical axis.
     */
    export type ZoomMode = 'XY' | 'X' | 'Y';
    /**
     * Defines the ZoomingToolkit, They are.
     * * zoom - Renders the zoom button.
     * * zoomIn - Renders the zoomIn button.
     * * zoomOut - Renders the zoomOut button.
     * * pan - Renders the pan button.
     * * reset - Renders the reset button.
     */
    export type ToolbarItems = 'Zoom' | 'ZoomIn' | 'ZoomOut' | 'Pan' | 'Reset';
    /**
     * Defines the SelectionMode, They are.
     * * none - Disable the selection.
     * * series - To select a series.
     * * point - To select a point.
     * * cluster - To select a cluster of point
     * * dragXY - To select points, by dragging with respect to both horizontal and vertical axis
     * * dragX - To select points, by dragging with respect to horizontal axis.
     * * dragY - To select points, by dragging with respect to vertical axis.
     */
    export type SelectionMode = 'None' | 'Series' | 'Point' | 'Cluster' | 'DragXY' | 'DragY' | 'DragX';
    /**
     * Defines the LabelPosition, They are.
     * * outer - Position the label outside the point.
     * * top - Position the label on top of the point.
     * * bottom - Position the label on bottom of the point.
     * * middle - Position the label to middle of the point.
     * * auto - Position the label based on series.
     */
    export type LabelPosition = 'Outer' | 'Top' | 'Bottom' | 'Middle' | 'Auto';
    /**
     * Defines the Alignment. They are
     * * none - Shows all the labels.
     * * hide - Hide the label when it intersect.
     * * rotate45 - Rotate the label to 45 degree when it intersect.
     * * rotate90 - Rotate the label to 90 degree when it intersect.
     * *
     */
    export type LabelIntersectAction = 'None' | 'Hide' | 'Trim' | 'Wrap' | 'MultipleRows' | 'Rotate45' | 'Rotate90';
    /**
     * Defines the Position. They are
     * * inside - Place the ticks or labels inside to the axis line.
     * * outside - Place the ticks or labels outside to the axis line.
     * *
     */
    export type AxisPosition = 'Inside' | 'Outside';
    /**
     * Defines Theme of the chart. They are
     * * Material - Render a chart with Material theme.
     * * Fabric - Render a chart with Fabric theme
     */
    export type ChartTheme = 'Material' | 'Fabric' | 'Bootstrap' | 'Highcontrast';
    /**
     *  Specifies the order of the strip line. `Over` | `Behind`.
     * * Over - Places the strip line over the series elements.
     * * Behind - laces the strip line behind the series elements.
     */
    export type ZIndex = 'Over' | 'Behind';
    /**
     * Defines the strip line text position.
     * * Start - Places the strip line text at the start.
     * * Middle - Places the strip line text in the middle.
     * * End - Places the strip line text at the end.
     */
    export type Anchor = 'Start' | 'Middle' | 'End';
    /**
     * Defines the empty point mode of the chart.
     * * Gap - Used to display empty points as space.
     * * Zero - Used to display empty points as zero.
     * * Drop - Used to ignore the empty point while rendering.
     * * Average - Used to display empty points as previous and next point average.
     */
    export type EmptyPointMode = 'Gap' | 'Zero' | 'Drop' | 'Average';
    /**
     * Defines the type of technical indicators. They are
     * * Sma - Predicts the trend using Simple Moving Average approach
     * * Ema - Predicts the trend using Exponential Moving Average approach
     * * Tma - Predicts the trend using Triangle Moving Average approach
     * * Atr - Predicts the trend using Average True Range approach
     * * AccumulationDistribution - Predicts the trend using Accumulation Distribution approach
     * * Momentum - Predicts the trend using Momentum approach
     * * Rsi - Predicts the trend using RSI approach
     * * Macd - Predicts the trend using Moving Average Convergence Divergence approach
     * * Stochastic - Predicts the trend using Stochastic approach
     * * BollingerBands - Predicts the trend using Bollinger approach
     */
    export type TechnicalIndicators = 'Sma' | 'Ema' | 'Tma' | 'Momentum' | 'Atr' | 'AccumulationDistribution' | 'BollingerBands' | 'Macd' | 'Stochastic' | 'Rsi';
    /**
     * Defines the type of trendlines. They are
     * * Linear - Defines the linear trendline
     * * Exponential - Defines the exponential trendline
     * * Polynomial - Defines the polynomial trendline
     * * Power - Defines the power trendline
     * * Logarithmic - Defines the logarithmic trendline
     * * MovingAverage - Defines the moving average trendline
     */
    export type TrendlineTypes = 'Linear' | 'Exponential' | 'Polynomial' | 'Power' | 'Logarithmic' | 'MovingAverage';
    /** @private */
    export type SeriesCategories = 'TrendLine' | 'Indicator' | 'Series';
    /**
     * Defines the financial data fields
     * * High - Represents the highest price in the stocks over time
     * * Low - Represents the lowest price in the stocks over time
     * * Open - Represents the opening price in the stocks over time
     * * Close - Represents the closing price in the stocks over time
     */
    export type FinancialDataFields = 'High' | 'Low' | 'Open' | 'Close';
    /**
     * It defines type of spline.
     * Natural - Used to render Natural spline.
     * Cardinal - Used to render cardinal spline.
     * Clamped - Used to render Clamped spline
     * Monotonic - Used to render monotonic spline
     */
    export type SplineType = 'Natural' | 'Monotonic' | 'Cardinal' | 'Clamped';
    /**
     * Defines the BoxPlotMode for box and whisker chart series, They are.
     * * exclusive - Series render based on exclusive mode.
     * * inclusive - Series render based on inclusive mode.
     * * normal - Series render based on normal mode.
     */
    export type BoxPlotMode = 'Exclusive' | 'Inclusive' | 'Normal';
    /**
     * Defines the skeleton type for the axis.
     * * Date - it formats date only.
     * * DateTime - it formats date and time.
     * * Time - it formats time only.
     */
    export type SkeletonType = 'Date' | 'DateTime' | 'Time';
    /**
     * Defines border type for multi level labels.
     *  * Rectangle
     *  * Brace
     *  * WithoutBorder
     *  * Without top Border
     *  * Without top and bottom border
     *  * Curly brace
     */
    export type BorderType = 'Rectangle' | 'Brace' | 'WithoutBorder' | 'WithoutTopBorder' | 'WithoutTopandBottomBorder' | 'CurlyBrace';
    
    /**
     * To get the data on mouse move.
     * @private
     */
    export class ChartData {
        /** @private */
        chart: Chart;
        lierIndex: number;
        /** @private */
        currentPoints: PointData[] | AccPointData[];
        /** @private */
        previousPoints: PointData[] | AccPointData[];
        /**
         * Constructor for the data.
         * @private
         */
        constructor(chart: Chart);
        /**
         * Method to get the Data.
         * @private
         */
        getData(): PointData;
        isSelected(chart: Chart): boolean;
        private getRectPoint(series, rect, x, y);
        /**
         * Checks whether the region contains a point
         */
        private checkRegionContainsPoint(regionRect, rect, x, y);
        private getClosest(series, value);
        getClosestX(chart: Chart, series: Series): PointData;
    }
    
    /**
     * Annotation Module handles the Annotation for chart and accumulation series.
     */
    export class AnnotationBase {
        private control;
        private annotation;
        private isChart;
        /**
         * Constructor for chart and accumulation annotation
         * @param control
         */
        constructor(control: Chart | AccumulationChart);
        /**
         * Method to render the annotation for chart and accumulation series.
         * @private
         * @param annotation
         * @param index
         */
        render(annotation: AccumulationAnnotationSettings | ChartAnnotationSettings, index: number): HTMLElement;
        /**
         * Method to calculate the location for annotation - coordinate unit as pixel.
         * @private
         * @param location
         */
        setAnnotationPixelValue(location: ChartLocation): boolean;
        /**
         * Method to calculate the location for annotation - coordinate unit as point.
         * @private
         * @param location
         */
        setAnnotationPointValue(location: ChartLocation): boolean;
        /**
         * To process the annotation for accumulation chart
         * @param annotation
         * @param index
         * @param parentElement
         */
        processAnnotation(annotation: ChartAnnotationSettings | AccumulationAnnotationSettings, index: number, parentElement: HTMLElement): void;
        /**
         * Method to calculate the location for annotation - coordinate unit as point in accumulation chart.
         * @private
         * @param location
         */
        setAccumulationPointValue(location: ChartLocation): boolean;
        /**
         * Method to set the element style for accumulation / chart annotation.
         * @private
         * @param location
         * @param element
         * @param parentElement
         */
        setElementStyle(location: ChartLocation, element: HTMLElement, parentElement: HTMLElement): void;
        /**
         * Method to calculate the alignment value for annotation.
         * @private
         * @param alignment
         * @param size
         * @param value
         */
        setAlignmentValue(alignment: Alignment | Position, size: number, value: number): number;
    }
    
    /**
     * Interface for a class Location
     */
    export interface LocationModel {
    
        /**
         * X coordinate of the legend in pixels.
         * @default 0
         */
        x?: number;
    
        /**
         * Y coordinate of the legend in pixels.
         * @default 0
         */
        y?: number;
    
    }
    
    /**
     * Interface for a class LegendSettings
     */
    export interface LegendSettingsModel {
    
        /**
         * If set to true, legend will be visible.
         * @default true
         */
        visible?: boolean;
    
        /**
         * The height of the legend in pixels.
         * @default null
         */
        height?: string;
    
        /**
         * The width of the legend in pixels.
         * @default null
         */
        width?: string;
    
        /**
         * Specifies the location of the legend, relative to the chart.
         * If x is 20, legend moves by 20 pixels to the right of the chart. It requires the `position` to be `Custom`.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *   legendSettings: {
         *     visible: true,
         *     position: 'Custom',
         *     location: { x: 100, y: 150 },
         *   },
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         */
        location?: LocationModel;
    
        /**
         * Position of the legend in the chart are,
         * * Auto: Places the legend based on area type.
         * * Top: Displays the legend at the top of the chart.
         * * Left: Displays the legend at the left of the chart.
         * * Bottom: Displays the legend at the bottom of the chart.
         * * Right: Displays the legend at the right of the chart.
         * * Custom: Displays the legend  based on the given x and y values.
         * @default 'Auto'
         */
        position?: LegendPosition;
    
        /**
         * Option to customize the padding between legend items.
         * @default 8
         */
        padding?: number;
    
        /**
         * Legend in chart can be aligned as follows:
         * * Near: Aligns the legend to the left of the chart.
         * * Center: Aligns the legend to the center of the chart.
         * * Far: Aligns the legend to the right of the chart.
         * @default 'Center'
         */
        alignment?: Alignment;
    
        /**
         * Options to customize the legend text.
         */
        textStyle?: FontModel;
    
        /**
         * Shape height of the legend in pixels.
         * @default 10
         */
        shapeHeight?: number;
    
        /**
         * Shape width of the legend in pixels.
         * @default 10
         */
        shapeWidth?: number;
    
        /**
         * Options to customize the border of the legend.
         */
        border?: BorderModel;
    
        /**
         * Padding between the legend shape and text.
         * @default 5
         */
        shapePadding?: number;
    
        /**
         * The background color of the legend that accepts value in hex and rgba as a valid CSS color string.
         * @default 'transparent'
         */
        background?: string;
    
        /**
         * Opacity of the legend.
         * @default 1
         */
        opacity?: number;
    
        /**
         * If set to true, series' visibility collapses based on the legend visibility.
         * @default true
         */
        toggleVisibility?: boolean;
    
        /**
         * Description for legends.
         * @default null
         */
        description?: string;
    
        /**
         * TabIndex value for the legend.
         * @default 3
         */
        tabIndex?: number;
    
    }
    
    /**
     * Interface for a class BaseLegend
     * @private
     */
    export interface BaseLegendModel {
    
    }
    
    /**
     * Interface for a class LegendOptions
     * @private
     */
    export interface LegendOptionsModel {
    
    }
    
    /**
     * Configures the location for the legend.
     */
    export class Location extends ej.base.ChildProperty<Location> {
        /**
         * X coordinate of the legend in pixels.
         * @default 0
         */
        x: number;
        /**
         * Y coordinate of the legend in pixels.
         * @default 0
         */
        y: number;
    }
    /**
     * Configures the legends in charts.
     */
    export class LegendSettings extends ej.base.ChildProperty<LegendSettings> {
        /**
         * If set to true, legend will be visible.
         * @default true
         */
        visible: boolean;
        /**
         * The height of the legend in pixels.
         * @default null
         */
        height: string;
        /**
         * The width of the legend in pixels.
         * @default null
         */
        width: string;
        /**
         * Specifies the location of the legend, relative to the chart.
         * If x is 20, legend moves by 20 pixels to the right of the chart. It requires the `position` to be `Custom`.
         * ```html
         * <div id='Chart'></div>
         * ```
         * ```typescript
         * let chart: Chart = new Chart({
         * ...
         *   legendSettings: {
         *     visible: true,
         *     position: 'Custom',
         *     location: { x: 100, y: 150 },
         *   },
         * ...
         * });
         * chart.appendTo('#Chart');
         * ```
         */
        location: LocationModel;
        /**
         * Position of the legend in the chart are,
         * * Auto: Places the legend based on area type.
         * * Top: Displays the legend at the top of the chart.
         * * Left: Displays the legend at the left of the chart.
         * * Bottom: Displays the legend at the bottom of the chart.
         * * Right: Displays the legend at the right of the chart.
         * * Custom: Displays the legend  based on the given x and y values.
         * @default 'Auto'
         */
        position: LegendPosition;
        /**
         * Option to customize the padding between legend items.
         * @default 8
         */
        padding: number;
        /**
         * Legend in chart can be aligned as follows:
         * * Near: Aligns the legend to the left of the chart.
         * * Center: Aligns the legend to the center of the chart.
         * * Far: Aligns the legend to the right of the chart.
         * @default 'Center'
         */
        alignment: Alignment;
        /**
         * Options to customize the legend text.
         */
        textStyle: FontModel;
        /**
         * Shape height of the legend in pixels.
         * @default 10
         */
        shapeHeight: number;
        /**
         * Shape width of the legend in pixels.
         * @default 10
         */
        shapeWidth: number;
        /**
         * Options to customize the border of the legend.
         */
        border: BorderModel;
        /**
         * Padding between the legend shape and text.
         * @default 5
         */
        shapePadding: number;
        /**
         * The background color of the legend that accepts value in hex and rgba as a valid CSS color string.
         * @default 'transparent'
         */
        background: string;
        /**
         * Opacity of the legend.
         * @default 1
         */
        opacity: number;
        /**
         * If set to true, series' visibility collapses based on the legend visibility.
         * @default true
         */
        toggleVisibility: boolean;
        /**
         * Description for legends.
         * @default null
         */
        description: string;
        /**
         * TabIndex value for the legend.
         * @default 3
         */
        tabIndex: number;
    }
    /**
     * Legend base class for Chart and Accumulation chart.
     * @private
     */
    export class BaseLegend {
        protected chart: Chart | AccumulationChart;
        protected legend: LegendSettingsModel;
        protected maxItemHeight: number;
        protected isPaging: boolean;
        private clipPathHeight;
        protected totalPages: number;
        protected isVertical: boolean;
        private rowCount;
        private columnCount;
        private pageButtonSize;
        protected pageXCollections: number[];
        protected maxColumns: number;
        private isTrimmed;
        maxWidth: number;
        protected legendID: string;
        private clipRect;
        private legendTranslateGroup;
        private currentPage;
        private isChartControl;
        protected library: Legend | AccumulationLegend;
        /**  @private */
        position: LegendPosition;
        /**
         * Gets the legend bounds in chart.
         * @private
         */
        legendBounds: Rect;
        /** @private */
        legendCollections: LegendOptions[];
        /** @private */
        clearTooltip: number;
        /**
         * Constructor for the dateTime module.
         * @private
         */
        constructor(chart?: Chart | AccumulationChart);
        /**
         * Calculate the bounds for the legends.
         * @return {void}
         * @private
         */
        calculateLegendBounds(rect: Rect, availableSize: Size): void;
        /**
         * To find legend position based on available size for chart and accumulation chart
         */
        private getPosition(position, availableSize);
        /**
         * To set bounds for chart and accumulation chart
         */
        protected setBounds(computedWidth: number, computedHeight: number, legend: LegendSettingsModel, legendBounds: Rect): void;
        /**
         * To find legend location based on position, alignment for chart and accumulation chart
         */
        private getLocation(position, alignment, legendBounds, rect, availableSize);
        /**
         * To find legend alignment for chart and accumulation chart
         */
        private alignLegend(start, size, legendSize, alignment);
        /**
         * Renders the legend.
         * @return {void}
         * @private
         */
        renderLegend(chart: Chart | AccumulationChart, legend: LegendSettingsModel, legendBounds: Rect): void;
        /**
         * To find first valid legend text index for chart and accumulation chart
         */
        private findFirstLegendPosition(legendCollection);
        /**
         * To create legend rendering elements for chart and accumulation chart
         */
        private createLegendElements(chart, legendBounds, legendGroup, legend, id);
        /**
         * To render legend symbols for chart and accumulation chart
         */
        private renderSymbol(legendOption, group, i);
        /**
         * To render legend text for chart and accumulation chart
         */
        private renderText(chart, legendOption, group, textOptions, i);
        /**
         * To render legend paging elements for chart and accumulation chart
         */
        private renderPagingElements(chart, bounds, textOption, legendGroup);
        /**
         * To translate legend pages for chart and accumulation chart
         */
        protected translatePage(pagingText: Element, page: number, pageNumber: number): void;
        /**
         * To change legend pages for chart and accumulation chart
         */
        protected changePage(event: Event, pageUp: boolean): void;
        /**
         * To find legend elements id based on chart or accumulation chart
         * @private
         */
        generateId(option: LegendOptions, prefix: string, count: number): string;
        /**
         * To show or hide trimmed text tooltip for legend.
         * @return {void}
         * @private
         */
        move(event: Event): void;
    }
    /**
     * Class for legend options
     * @private
     */
    export class LegendOptions {
        render: boolean;
        text: string;
        fill: string;
        shape: LegendShape;
        visible: boolean;
        type: ChartSeriesType | AccumulationType;
        textSize: Size;
        location: ChartLocation;
        pointIndex?: number;
        seriesIndex?: number;
        markerShape?: ChartShape;
        markerVisibility?: boolean;
        constructor(text: string, fill: string, shape: LegendShape, visible: boolean, type: ChartSeriesType | AccumulationType, markerShape?: ChartShape, markerVisibility?: boolean, pointIndex?: number, seriesIndex?: number);
    }
    
    /**
     * Interface for a class Connector
     */
    export interface ConnectorModel {
    
        /**
         * specifies the type of the connector line. They are
         * * Smooth
         * * Line
         * @default 'Line'
         */
        type?: ConnectorType;
    
        /**
         * Color of the connector line.
         * @default null
         */
        color?: string;
    
        /**
         * Width of the connector line in pixels.
         * @default 1
         */
        width?: number;
    
        /**
         * Length of the connector line in pixels.
         * @default null
         */
        length?: string;
    
        /**
         * dashArray of the connector line.
         * @default ''
         */
        dashArray?: string;
    
    }
    
    /**
     * Interface for a class Font
     */
    export interface FontModel {
    
        /**
         * Font size for the text.
         * @default '16px'
         */
        size?: string;
    
        /**
         * Color for the text.
         * @default ''
         */
        color?: string;
    
        /**
         * FontFamily for the text.
         */
        fontFamily?: string;
    
        /**
         * FontWeight for the text.
         * @default 'Normal'
         */
        fontWeight?: string;
    
        /**
         * FontStyle for the text.
         * @default 'Normal'
         */
        fontStyle?: string;
    
        /**
         * Opacity for the text.
         * @default 1
         */
        opacity?: number;
    
        /**
         * text alignment
         * @default 'Center'
         */
        textAlignment?: Alignment;
    
        /**
         * Specifies the chart title text overflow
         * @default 'Trim'
         */
        textOverflow?: TextOverflow;
    
    }
    
    /**
     * Interface for a class Border
     */
    export interface BorderModel {
    
        /**
         * The color of the border that accepts value in hex and rgba as a valid CSS color string.
         * @default ''
         */
        color?: string;
    
        /**
         * The width of the border in pixels.
         * @default 1
         */
        width?: number;
    
    }
    
    /**
     * Interface for a class ChartArea
     */
    export interface ChartAreaModel {
    
        /**
         * Options to customize the border of the chart area.
         */
        border?: BorderModel;
    
        /**
         * The background of the chart area that accepts value in hex and rgba as a valid CSS color string..
         * @default transparent
         */
        background?: string;
    
        /**
         * The opacity for background.
         * @default 1
         */
        opacity?: number;
    
    }
    
    /**
     * Interface for a class Margin
     */
    export interface MarginModel {
    
        /**
         * Left margin in pixels.
         * @default 10
         */
        left?: number;
    
        /**
         * Right margin in pixels.
         * @default 10
         */
        right?: number;
    
        /**
         * Top margin in pixels.
         * @default 10
         */
        top?: number;
    
        /**
         * Bottom margin in pixels.
         * @default 10
         */
        bottom?: number;
    
    }
    
    /**
     * Interface for a class Animation
     */
    export interface AnimationModel {
    
        /**
         * If set to true, series gets animated on initial loading.
         * @default true
         */
    
        enable?: boolean;
    
        /**
         * The duration of animation in milliseconds.
         * @default 1000
         */
    
        duration?: number;
    
        /**
         * The option to delay animation of the series.
         * @default 0
         */
    
        delay?: number;
    
    }
    
    /**
     * Interface for a class Indexes
     * @private
     */
    export interface IndexesModel {
    
        /**
         * Specifies the series index
         * @default 0
         */
        series?: number;
    
        /**
         * Specifies the point index
         * @default 0
         */
        point?: number;
    
    }
    
    /**
     * Interface for a class CornerRadius
     */
    export interface CornerRadiusModel {
    
        /**
         * Specifies the top left corner radius value
         * @default 0
         */
        topLeft?: number;
    
        /**
         * Specifies the top right corner radius value
         * @default 0
         */
        topRight?: number;
    
        /**
         * Specifies the bottom left corner radius value
         * @default 0
         */
        bottomLeft?: number;
    
        /**
         * Specifies the bottom right corner radius value
         * @default 0
         */
        bottomRight?: number;
    
    }
    
    /**
     * Interface for a class Index
     * @private
     */
    export interface IndexModel {
    
    }
    
    /**
     * Interface for a class EmptyPointSettings
     */
    export interface EmptyPointSettingsModel {
    
        /**
         * To customize the fill color of empty points.
         * @default null
         */
        fill?: string;
    
        /**
         * Options to customize the border of empty points.
         * @default "{color: 'transparent', width: 0}"
         */
        border?: BorderModel;
    
        /**
         * To customize the mode of empty points.
         * @default Gap
         */
        mode?: EmptyPointMode | AccEmptyPointMode;
    
    }
    
    /**
     * Interface for a class TooltipSettings
     */
    export interface TooltipSettingsModel {
    
        /**
         * Enables / Disables the visibility of the tooltip.
         * @default false.
         */
    
        enable?: boolean;
    
        /**
         * If set to true, a single ToolTip will be displayed for every index.
         * @default false.
         */
    
        shared?: boolean;
    
        /**
         * The fill color of the tooltip that accepts value in hex and rgba as a valid CSS color string. 
         */
    
        fill?: string;
    
        /**
         * Header for tooltip. 
         */
    
        header?: string;
    
        /**
         * The fill color of the tooltip that accepts value in hex and rgba as a valid CSS color string. 
         */
    
        opacity?: number;
    
        /**
         * Options to customize the ToolTip text.
         */
    
        textStyle?: FontModel;
    
        /**
         * Format the ToolTip content.
         * @default null.
         */
    
        format?: string;
    
        /**
         * Custom template to format the ToolTip content. Use ${x} and ${y} as the placeholder text to display the corresponding data point.
         * @default null.
         */
    
        template?: string;
    
        /**
         * If set to true, ToolTip will animate while moving from one point to another.
         * @default true.
         */
        enableAnimation?: boolean;
    
        /**
         * Options to customize tooltip borders.
         */
        border?: BorderModel;
    
    }
    
    /**
     * Defines the appearance of the connectors
     */
    export class Connector extends ej.base.ChildProperty<Connector> {
        /**
         * specifies the type of the connector line. They are
         * * Smooth
         * * Line
         * @default 'Line'
         */
        type: ConnectorType;
        /**
         * Color of the connector line.
         * @default null
         */
        color: string;
        /**
         * Width of the connector line in pixels.
         * @default 1
         */
        width: number;
        /**
         * Length of the connector line in pixels.
         * @default null
         */
        length: string;
        /**
         * dashArray of the connector line.
         * @default ''
         */
        dashArray: string;
    }
    /**
     * Configures the fonts in charts.
     */
    export class Font extends ej.base.ChildProperty<Font> {
        /**
         * Font size for the text.
         * @default '16px'
         */
        size: string;
        /**
         * Color for the text.
         * @default ''
         */
        color: string;
        /**
         * FontFamily for the text.
         */
        fontFamily: string;
        /**
         * FontWeight for the text.
         * @default 'Normal'
         */
        fontWeight: string;
        /**
         * FontStyle for the text.
         * @default 'Normal'
         */
        fontStyle: string;
        /**
         * Opacity for the text.
         * @default 1
         */
        opacity: number;
        /**
         * text alignment
         * @default 'Center'
         */
        textAlignment: Alignment;
        /**
         * Specifies the chart title text overflow
         * @default 'Trim'
         */
        textOverflow: TextOverflow;
    }
    /**
     * Configures the borders in the chart.
     */
    export class Border extends ej.base.ChildProperty<Border> {
        /**
         * The color of the border that accepts value in hex and rgba as a valid CSS color string.
         * @default ''
         */
        color: string;
        /**
         * The width of the border in pixels.
         * @default 1
         */
        width: number;
    }
    /**
     * Configures the chart area.
     */
    export class ChartArea extends ej.base.ChildProperty<ChartArea> {
        /**
         * Options to customize the border of the chart area.
         */
        border: BorderModel;
        /**
         * The background of the chart area that accepts value in hex and rgba as a valid CSS color string..
         * @default transparent
         */
        background: string;
        /**
         * The opacity for background.
         * @default 1
         */
        opacity: number;
    }
    /**
     * Configures the chart margins.
     */
    export class Margin extends ej.base.ChildProperty<Margin> {
        /**
         * Left margin in pixels.
         * @default 10
         */
        left: number;
        /**
         * Right margin in pixels.
         * @default 10
         */
        right: number;
        /**
         * Top margin in pixels.
         * @default 10
         */
        top: number;
        /**
         * Bottom margin in pixels.
         * @default 10
         */
        bottom: number;
    }
    /**
     * Configures the animation behavior for chart series.
     */
    export class Animation extends ej.base.ChildProperty<Animation> {
        /**
         * If set to true, series gets animated on initial loading.
         * @default true
         */
        enable: boolean;
        /**
         * The duration of animation in milliseconds.
         * @default 1000
         */
        duration: number;
        /**
         * The option to delay animation of the series.
         * @default 0
         */
        delay: number;
    }
    /** @private */
    export class Indexes extends ej.base.ChildProperty<Indexes> {
        /**
         * Specifies the series index
         * @default 0
         */
        series: number;
        /**
         * Specifies the point index
         * @default 0
         */
        point: number;
    }
    /**
     * Column series rounded corner options
     */
    export class CornerRadius extends ej.base.ChildProperty<CornerRadius> {
        /**
         * Specifies the top left corner radius value
         * @default 0
         */
        topLeft: number;
        /**
         * Specifies the top right corner radius value
         * @default 0
         */
        topRight: number;
        /**
         * Specifies the bottom left corner radius value
         * @default 0
         */
        bottomLeft: number;
        /**
         * Specifies the bottom right corner radius value
         * @default 0
         */
        bottomRight: number;
    }
    /**
     * @private
     */
    export class Index {
        series: number;
        point: number;
        constructor(seriesIndex: number, pointIndex?: number);
    }
    /**
     * Configures the Empty Points of series
     */
    export class EmptyPointSettings extends ej.base.ChildProperty<EmptyPointSettings> {
        /**
         * To customize the fill color of empty points.
         * @default null
         */
        fill: string;
        /**
         * Options to customize the border of empty points.
         * @default "{color: 'transparent', width: 0}"
         */
        border: BorderModel;
        /**
         * To customize the mode of empty points.
         * @default Gap
         */
        mode: EmptyPointMode | AccEmptyPointMode;
    }
    /**
     * Configures the ToolTips in the chart.
     */
    export class TooltipSettings extends ej.base.ChildProperty<TooltipSettings> {
        /**
         * Enables / Disables the visibility of the tooltip.
         * @default false.
         */
        enable: boolean;
        /**
         * If set to true, a single ToolTip will be displayed for every index.
         * @default false.
         */
        shared: boolean;
        /**
         * The fill color of the tooltip that accepts value in hex and rgba as a valid CSS color string.
         */
        fill: string;
        /**
         * Header for tooltip.
         */
        header: string;
        /**
         * The fill color of the tooltip that accepts value in hex and rgba as a valid CSS color string.
         */
        opacity: number;
        /**
         * Options to customize the ToolTip text.
         */
        textStyle: FontModel;
        /**
         * Format the ToolTip content.
         * @default null.
         */
        format: string;
        /**
         * Custom template to format the ToolTip content. Use ${x} and ${y} as the placeholder text to display the corresponding data point.
         * @default null.
         */
        template: string;
        /**
         * If set to true, ToolTip will animate while moving from one point to another.
         * @default true.
         */
        enableAnimation: boolean;
        /**
         * Options to customize tooltip borders.
         */
        border: BorderModel;
    }
    
    /**
     * Specifies the chart constant value
     */
    /** @private */
    export const loaded: string;
    /** @private */
    export const load: string;
    /** @private */
    export const animationComplete: string;
    /** @private */
    export const legendRender: string;
    /** @private */
    export const textRender: string;
    /** @private */
    export const pointRender: string;
    /** @private */
    export const seriesRender: string;
    /** @private */
    export const axisLabelRender: string;
    /** @private */
    export const axisRangeCalculated: string;
    /** @private */
    export const axisMultiLabelRender: string;
    /** @private */
    export const tooltipRender: string;
    /** @private */
    export const chartMouseMove: string;
    /** @private */
    export const chartMouseClick: string;
    /** @private */
    export const pointClick: string;
    /** @private */
    export const pointMove: string;
    /** @private */
    export const chartMouseLeave: string;
    /** @private */
    export const chartMouseDown: string;
    /** @private */
    export const chartMouseUp: string;
    /** @private */
    export const zoomComplete: string;
    /** @private */
    export const dragComplete: string;
    /** @private */
    export const resized: string;
    /** @private */
    export const beforePrint: string;
    /** @private */
    export const annotationRender: string;
    
    /**
     * data module is used to generate query and dataSource
     */
    export class Data {
        private dataManager;
        private query;
        /**
         * Constructor for data module
         * @private
         */
        constructor(dataSource?: Object | ej.data.DataManager, query?: ej.data.Query);
        /**
         * The function used to initialize dataManager and query
         * @return {void}
         * @private
         */
        initDataManager(dataSource: Object | ej.data.DataManager, query: ej.data.Query): void;
        /**
         * The function used to generate updated Query from chart model
         * @return {void}
         * @private
         */
        generateQuery(): ej.data.Query;
        /**
         * The function used to get dataSource by executing given Query
         * @param  {Query} query - A Query that specifies to generate dataSource
         * @return {void}
         * @private
         */
        getData(query: ej.data.Query): Promise<Object>;
    }
    
    export interface IChartEventArgs {
        /** Defines the name of the event */
        name: string;
        /** Defines the event cancel status */
        cancel: boolean;
    }
    export interface ILegendRenderEventArgs extends IChartEventArgs {
        /** Defines the current legend text */
        text: string;
        /** Defines the current legend fill color */
        fill: string;
        /** Defines the current legend shape */
        shape: LegendShape;
        /** Defines the current legend marker shape */
        markerShape?: ChartShape;
    }
    export interface IAccLegendRenderEventArgs extends IChartEventArgs {
        /** Defines the current legend shape */
        shape: LegendShape;
        /** Defines the current legend fill color */
        fill: string;
        /** Defines the current legend text */
        text: string;
    }
    export interface ITextRenderEventArgs extends IChartEventArgs {
        /** Defines the current series of the label */
        series: SeriesModel;
        /** Defines the current point of the label */
        point: Points;
        /** Defines the current text */
        text: string;
        /** Defines the current label fill color */
        color: string;
        /** Defines the current label border */
        border: BorderModel;
        /** Defines the current label template */
        template: string;
    }
    export interface IAnnotationRenderEventArgs extends IChartEventArgs {
        /** Defines the current annotation content */
        content: HTMLElement;
        /** Defines the current annotation location */
        location: ChartLocation;
    }
    export interface IZoomCompleteEventArgs extends IChartEventArgs {
        /** Defines the zoomed axis */
        axis: AxisModel;
        /** Defines the previous zoom factor */
        previousZoomFactor: number;
        /** Defines the previous zoom position */
        previousZoomPosition: number;
        /** Defines the current zoom factor */
        currentZoomFactor: number;
        /** Defines the current zoom position */
        currentZoomPosition: number;
    }
    export interface IPointRenderEventArgs extends IChartEventArgs {
        /** Defines the current series of the point */
        series: Series;
        /** Defines the current point */
        point: Points;
        /** Defines the current point fill color */
        fill: string;
        /** Defines the current point border */
        border: BorderModel;
        /** Defines the current point height */
        height?: number;
        /** Defines the current point width */
        width?: number;
        /** Defines the current point marker shape */
        shape?: ChartShape;
    }
    export interface ISeriesRenderEventArgs {
        /** Defines the current series */
        series: Series;
        /** Defines the current series data object */
        data: Object;
        /** Defines name of the event */
        name: string;
        /** Defines the current series fill */
        fill: string;
    }
    export interface IAxisLabelRenderEventArgs extends IChartEventArgs {
        /** Defines the current axis */
        axis: Axis;
        /** Defines axis current label text */
        text: string;
        /** Defines axis current label value */
        value: number;
        /** Defines axis current label font style */
        labelStyle: FontModel;
    }
    export interface IAxisRangeCalculatedEventArgs extends IChartEventArgs {
        /** Defines the current axis */
        axis: Axis;
        /** Defines axis current range */
        minimum: number;
        /** Defines axis current range */
        maximum: number;
        /** Defines axis current interval */
        interval: number;
    }
    export interface IAxisMultiLabelRenderEventArgs extends IChartEventArgs {
        /** Defines the current axis */
        axis: Axis;
        /** Defines axis current label text */
        text: string;
        /** Defines font style for multi labels */
        textStyle: FontModel;
        /** Defines text alignment for multi labels */
        alignment: Alignment;
    }
    export interface ITooltipRenderEventArgs extends IChartEventArgs {
        /** Defines tooltip text collections */
        text?: string;
        /** Defines tooltip text style */
        textStyle?: FontModel;
        /** Defines current tooltip series */
        series: Series | AccumulationSeries;
        /** Defines current tooltip point */
        point: Points | AccPoints;
    }
    export interface IMouseEventArgs extends IChartEventArgs {
        /** Defines current mouse event target id */
        target: string;
        /** Defines current mouse x location */
        x: number;
        /** Defines current mouse y location */
        y: number;
    }
    export interface IPointEventArgs extends IChartEventArgs {
        /** Defines the current series */
        series: SeriesModel;
        /** Defines the current point */
        point: Points;
        /** Defines the point index */
        pointIndex: number;
        /** Defines the series index */
        seriesIndex: number;
        /** Defines the current chart instance */
        chart: Chart;
        /** Defines current mouse x location */
        x: number;
        /** Defines current mouse y location */
        y: number;
    }
    /** @private */
    export interface IFontMapping {
        size?: string;
        color?: string;
        fontWeight?: string;
        fontStyle?: string;
        fontFamily?: string;
    }
    /** @private */
    export interface ITouches {
        pageX?: number;
        pageY?: number;
        pointerId?: number;
    }
    /** @private */
    export interface IShapes {
        renderOption?: Object;
        functionName?: string;
    }
    /** @private */
    export interface IZoomAxisRange {
        actualMin?: number;
        actualDelta?: number;
        min?: number;
        delta?: number;
    }
    export interface IDragCompleteEventArgs extends IChartEventArgs {
        /** Defines current selected Data X, Y values */
        selectedDataValues: {
            x: string;
            y: number;
        }[][];
    }
    export interface ILoadedEventArgs extends IChartEventArgs {
        /** Defines the current chart instance */
        chart: Chart;
    }
    export interface IAnimationCompleteEventArgs extends IChartEventArgs {
        /** Defines the current animation series */
        series: Series;
    }
    export interface IPrintEventArgs extends IChartEventArgs {
        htmlContent: Element;
    }
    /** @private */
    export interface IBoxPlotQuartile {
        minimum: number;
        maximum: number;
        outliers: number[];
        upperQuartile: number;
        lowerQuartile: number;
        average: number;
        median: number;
    }
    /** @private */
    /**
     * Specifies the Theme style for chart and accumulation.
     */
    export interface IThemeStyle {
        axisLabel: string;
        axisTitle: string;
        axisLine: string;
        majorGridLine: string;
        minorGridLine: string;
        majorTickLine: string;
        minorTickLine: string;
        chartTitle: string;
        legendLabel: string;
        background: string;
        areaBorder: string;
        errorBar: string;
        crosshairLine: string;
        crosshairFill: string;
        crosshairLabel: string;
        tooltipFill: string;
        tooltipBoldLabel: string;
        tooltipLightLabel: string;
        tooltipHeaderLine: string;
        markerShadow: string;
        selectionRectFill: string;
        selectionRectStroke: string;
        selectionCircleStroke: string;
    }
    
    /**
     * Specifies Chart Themes
     */
    export namespace Theme {
        /** @private */
        let axisLabelFont: IFontMapping;
        /** @private */
        let axisTitleFont: IFontMapping;
        /** @private */
        let chartTitleFont: IFontMapping;
        /** @private */
        let crosshairLabelFont: IFontMapping;
        /** @private */
        let tooltipLabelFont: IFontMapping;
        /** @private */
        let legendLabelFont: IFontMapping;
        /** @private */
        let stripLineLabelFont: IFontMapping;
    }
    /** @private */
    export function getSeriesColor(theme: ChartTheme | AccumulationTheme): string[];
    /** @private */
    export function getThemeColor(theme: ChartTheme | AccumulationTheme): IThemeStyle;
    
    /**
     * Selection Module handles the selection for chart.
     * @private
     */
    export class BaseSelection {
        protected styleId: string;
        protected unselected: string;
        protected control: Chart | AccumulationChart;
        constructor(control: Chart | AccumulationChart);
        /**
         * To create selection styles for series
         */
        protected seriesStyles(): void;
        /**
         * To concat indexes
         */
        protected concatIndexes(userIndexes: IndexesModel[], localIndexes: Indexes[]): Indexes[];
        /**
         * Selected points series visibility checking on legend click
         */
        protected checkVisibility(selectedIndexes: Indexes[]): boolean;
        /**
         * To add svg element style class
         * @private
         */
        addSvgClass(element: Element, className: string): void;
        /**
         * To remove svg element style class
         * @private
         */
        removeSvgClass(element: Element, className: string): void;
        /**
         * To get children from parent element
         */
        protected getChildren(parent: Element): Element[];
    }
    
    /**
     * Tooltip Module used to render the tooltip for series.
     */
    export class BaseTooltip extends ChartData {
        element: HTMLElement;
        elementSize: Size;
        textStyle: FontModel;
        isRemove: boolean;
        toolTipInterval: number;
        padding: number;
        arrowPadding: number;
        textElements: Element[];
        templateFn: Function;
        rx: number;
        ry: number;
        isComplete: boolean;
        inverted: boolean;
        formattedText: string[];
        header: string;
        markerPoint: number[];
        /** @private */
        valueX: number;
        /** @private */
        valueY: number;
        tipRadius: number;
        control: AccumulationChart | Chart;
        /**
         * Constructor for tooltip module.
         * @private.
         */
        constructor(chart: Chart | AccumulationChart);
        getElement(id: string): HTMLElement;
        /**
         * Renders the tooltip.
         * @return {void}
         * @private
         */
        getTooltipElement(isTooltip: HTMLElement): HTMLDivElement;
        private createElement(chart);
        pushData(data: PointData | AccPointData, isFirst: boolean, tooltipDiv: HTMLDivElement, isChart: boolean): boolean;
        renderTooltip(data: PointData | AccPointData, areaRect: Rect, location: ChartLocation, textCollection: string, isFirst: boolean, cartesain: boolean): Side;
        renderTemplate(data: PointData | AccPointData, rect: Rect, location: ChartLocation, point: Points | AccPoints, isFirst: boolean): void;
        renderTooltipElement(chart: Chart | AccumulationChart, pointData: PointData | AccPointData, areaRect: Rect, location: ChartLocation, cartesain: boolean, isFirst: boolean): Side;
        private changeText(point, isBottom, isRight, rect);
        renderText(isRender: boolean, chart: Chart | AccumulationChart): void;
        private createTemplate(point, data, areaRect, location, parent, isFirst);
        private sharedTooltipLocation(bounds, x, y);
        private seriesTooltipLocation(pointData, bounds, symbolLocation, arrowLocation, tipLocation);
        progressAnimation(element: HTMLElement, tooltipGroup: HTMLElement, series: Series | AccumulationSeries, opacity: number, rectOpacity: number, timeStamp: number, isRect: boolean, shared: boolean): void;
        endAnimation(element: HTMLElement, tooltipGroup: HTMLElement, series: Series | AccumulationSeries, shared: boolean): void;
        removeHighlight(chart: Chart | AccumulationChart): void;
        highlightPoint(series: Series | AccumulationSeries, pointIndex: number, highlight: boolean): void;
        highlightPoints(item: PointData | AccPointData): void;
        private animateTooltipDiv(tooltipDiv, rect);
        private updateDiv(tooltipDiv, x, y);
        triggerEvent(point: PointData | AccPointData, isFirst: boolean, textCollection: string, firstText?: boolean): boolean;
        removeText(): void;
        private updateTemplateFn(chart);
        stopAnimation(): void;
    }
    export class Side {
        isRight: boolean;
        isBottom: boolean;
        constructor(bottom: boolean, right: boolean);
    }
    
    /**
     * Defines Coordinate units of an annotation. They are
     * * Pixel
     * * Point
     */
    export type Units = 'Pixel' | 'Point';
    /**
     * Defines the Alignment. They are
     * * near - Align the element to the left.
     * * center - Align the element to the center.
     * * far - Align the element to the right.
     * *
     */
    export type Alignment = 'Near' | 'Center' | 'Far';
    /**
     * Defines regions of an annotation. They are
     * * Chart
     * * Series
     */
    export type Regions = 'Chart' | 'Series';
    /**
     * Defines the Position. They are
     * * top - Align the element to the top.
     * * middle - Align the element to the center.
     * * bottom - Align the element to the bottom.
     * *
     */
    export type Position = 'Top' | 'Middle' | 'Bottom';
    /**
     * Defines the export file format.
     * * PNG - export the image file format as png.
     * * JPEG - export the image file format as jpeg.
     */
    export type ExportType = 'PNG' | 'JPEG' | 'SVG' | 'PDF';
    /**
     * Defines the Text overflow.
     * * None - Shown the chart title with overlap if exceed.
     * * Wrap - Shown the chart title with wrap if exceed.
     * * Trim - Shown the chart title with trim if exceed.
     */
    export type TextOverflow = 'None' | 'Wrap' | 'Trim';
    
    /**
     * Annotation Module handles the Annotation for chart and accumulation series.
     */
    export class ExportUtils {
        private control;
        private printWindow;
        /**
         * Constructor for chart and accumulation annotation
         * @param control
         */
        constructor(control: Chart | AccumulationChart);
        /**
         * To print the accumulation and chart elements
         * @param elements
         */
        print(elements?: string[] | string | Element): void;
        /**
         * To get the html string of the chart and accumulation
         * @param elements
         * @private
         */
        getHTMLContent(elements?: string[] | string | Element): Element;
        /**
         * To export the file as image/svg format
         * @param type
         * @param fileName
         */
        export(type: ExportType, fileName: string, orientation?: ej.pdfexport.PdfPageOrientation): void;
        /**
         * To trigger the download element
         * @param fileName
         * @param type
         * @param url
         */
        triggerDownload(fileName: string, type: ExportType, url: string, isDownload: boolean): void;
    }
    
    /**
     * Methods for calculating the text size.
     */
    /**
     * Function to measure the height and width of the text.
     * @param  {string} text
     * @param  {FontModel} font
     * @param  {string} id
     * @returns no
     * @private
     */
    export function measureText(text: string, font: FontModel): Size;
    /**
     * Function to sort the dataSource, by default it sort the data in ascending order.
     * @param  {Object} data
     * @param  {string} fields
     * @param  {boolean} isDescending
     * @returns Object
     */
    export function sort(data: Object[], fields: string[], isDescending?: boolean): Object[];
    /** @private */
    export function rotateTextSize(font: FontModel, text: string, angle: number, chart: Chart): Size;
    /** @private */
    export function removeElement(id: string): void;
    /** @private */
    export function logBase(value: number, base: number): number;
    /** @private */
    export function showTooltip(text: string, x: number, y: number, areaWidth: number, id: string, element: Element, isTouch?: boolean): void;
    /** @private */
    export function inside(value: number, range: VisibleRangeModel): boolean;
    /** @private */
    export function withIn(value: number, range: VisibleRangeModel): boolean;
    /** @private */
    export function withInRange(previousPoint: Points, currentPoint: Points, nextPoint: Points, series: Series): boolean;
    /** @private */
    export function sum(values: number[]): number;
    /** @private */
    export function subArraySum(values: Object[], first: number, last: number, index: number[], series: Series): number;
    /** @private */
    export function subtractThickness(rect: Rect, thickness: Thickness): Rect;
    /** @private */
    export function subtractRect(rect: Rect, thickness: Rect): Rect;
    /** @private */
    export function degreeToLocation(degree: number, radius: number, center: ChartLocation): ChartLocation;
    /** @private */
    export function getAngle(center: ChartLocation, point: ChartLocation): number;
    /** @private */
    export function subArray(values: number[], index: number): number[];
    /** @private */
    export function valueToCoefficient(value: number, axis: Axis): number;
    /** @private */
    export function TransformToVisible(x: number, y: number, xAxis: Axis, yAxis: Axis, isInverted?: boolean, series?: Series): ChartLocation;
    /**
     * method to find series, point index by element id
     * @private
     */
    export function indexFinder(id: string): Index;
    /** @private */
    export function CoefficientToVector(coefficient: number, startAngle: number): ChartLocation;
    /** @private */
    export function valueToPolarCoefficient(value: number, axis: Axis): number;
    /** @private */
    export class Mean {
        verticalStandardMean: number;
        horizontalStandardMean: number;
        verticalSquareRoot: number;
        horizontalSquareRoot: number;
        verticalMean: number;
        horizontalMean: number;
        constructor(verticalStandardMean: number, verticalSquareRoot: number, horizontalStandardMean: number, horizontalSquareRoot: number, verticalMean: number, horizontalMean: number);
    }
    /** @private */
    export class PolarArc {
        startAngle: number;
        endAngle: number;
        innerRadius: number;
        radius: number;
        currentXPosition: number;
        constructor(startAngle?: number, endAngle?: number, innerRadius?: number, radius?: number, currentXPosition?: number);
    }
    /** @private */
    export function createTooltip(id: string, text: string, top: number, left: number, fontSize: string): void;
    /** @private */
    export function createZoomingLabels(chart: Chart, axis: Axis, parent: Element, index: number, isVertical: boolean, rect: Rect): Element;
    /** @private */
    export function withInBounds(x: number, y: number, bounds: Rect, width?: number, height?: number): boolean;
    /** @private */
    export function getValueXByPoint(value: number, size: number, axis: Axis): number;
    /** @private */
    export function getValueYByPoint(value: number, size: number, axis: Axis): number;
    /** @private */
    export function findClipRect(series: Series): void;
    /** @private */
    export function firstToLowerCase(str: string): string;
    /** @private */
    export function getMinPointsDelta(axis: Axis, seriesCollection: Series[]): number;
    /** @private */
    export function getAnimationFunction(effect: string): Function;
    /**
     * Animation Effect Calculation Started Here
     * @param currentTime
     * @param startValue
     * @param endValue
     * @param duration
     * @private
     */
    export function linear(currentTime: number, startValue: number, endValue: number, duration: number): number;
    /**
     * Animation Effect Calculation End
     * @private
     */
    export function markerAnimate(element: Element, delay: number, duration: number, series: Series | AccumulationSeries, pointIndex: number, point: ChartLocation, isLabel: boolean): void;
    /**
     * Animation for template
     * @private
     */
    export function templateAnimate(element: Element, delay: number, duration: number, name: ej.base.Effect, isRemove?: boolean): void;
    /** @private */
    export function drawSymbol(location: ChartLocation, shape: string, size: Size, url: string, options: PathOption, label: string): Element;
    /** @private */
    export function calculateShapes(location: ChartLocation, size: Size, shape: string, options: PathOption, url: string): IShapes;
    /** @private */
    export function getRectLocation(startLocation: ChartLocation, endLocation: ChartLocation, outerRect: Rect): Rect;
    /** @private */
    export function minMax(value: number, min: number, max: number): number;
    /** @private */
    export function getElement(id: string): Element;
    /** @private */
    export function getTemplateFunction(template: string): Function;
    /** @private */
    export function createTemplate(childElement: HTMLElement, pointIndex: number, content: string, chart: Chart | AccumulationChart, point?: Points | AccPoints, series?: Series | AccumulationSeries): HTMLElement;
    /** @private */
    export function getFontStyle(font: FontModel): string;
    /** @private */
    export function measureElementRect(element: HTMLElement): ClientRect;
    /** @private */
    export function findlElement(elements: NodeList, id: string): Element;
    /** @private */
    export function getPoint(x: number, y: number, xAxis: Axis, yAxis: Axis, isInverted?: boolean, series?: Series): ChartLocation;
    /** @private */
    export function appendElement(child: Element, parent: Element): void;
    /** @private */
    export function getDraggedRectLocation(x1: number, y1: number, x2: number, y2: number, outerRect: Rect): Rect;
    /** @private */
    export function checkBounds(start: number, size: number, min: number, max: number): number;
    /** @private */
    export function getLabelText(currentPoint: Points, series: Series, chart: Chart): string[];
    /** @private */
    export function stopTimer(timer: number): void;
    /** @private */
    export function isCollide(rect: Rect, collections: Rect[], clipRect: Rect): boolean;
    /** @private */
    export function isOverlap(currentRect: Rect, rect: Rect): boolean;
    /** @private */
    export function containsRect(currentRect: Rect, rect: Rect): boolean;
    /** @private */
    export function calculateRect(location: ChartLocation, textSize: Size, margin: MarginModel): Rect;
    /** @private */
    export function convertToHexCode(value: ColorValue): string;
    /** @private */
    export function componentToHex(value: number): string;
    /** @private */
    export function convertHexToColor(hex: string): ColorValue;
    /** @private */
    export function colorNameToHex(color: string): string;
    /** @private */
    export function getSaturationColor(color: string, factor: number): string;
    /** @private */
    export function getMedian(values: number[]): number;
    /** @private */
    export function calculateLegendShapes(location: ChartLocation, size: Size, shape: string, options: PathOption): IShapes;
    /** @private */
    export function textTrim(maxWidth: number, text: string, font: FontModel): string;
    /** @private */
    export function stringToNumber(value: string, containerSize: number): number;
    /** @private */
    export function findDirection(rX: number, rY: number, rect: Rect, arrowLocation: ChartLocation, arrowPadding: number, top: boolean, bottom: boolean, left: boolean, tipX: number, tipY: number, tipRadius?: number): string;
    /** @private */
    export function textElement(options: TextOption, font: FontModel, color: string, parent: HTMLElement | Element, isMinus?: boolean): Element;
    /**
     * Method to calculate the width and height of the chart
     */
    export function calculateSize(chart: Chart | AccumulationChart): void;
    export function createSvg(chart: Chart | AccumulationChart): void;
    /**
     * To calculate chart title and height
     * @param title
     * @param style
     * @param width
     */
    export function getTitle(title: string, style: FontModel, width: number): string[];
    /**
     * Method to calculate x position of title
     */
    export function titlePositionX(chartSize: Size, leftPadding: number, rightPadding: number, titleStyle: FontModel): number;
    /**
     * Method to find new text and element size based on textOverflow
     */
    export function textWrap(currentLabel: string, maximumWidth: number, font: FontModel): string[];
    /** @private */
    export class CustomizeOption {
        id: string;
        constructor(id?: string);
    }
    /** @private */
    export class StackValues {
        startValues?: number[];
        endValues?: number[];
        constructor(startValue?: number[], endValue?: number[]);
    }
    /** @private */
    export class TextOption extends CustomizeOption {
        anchor: string;
        text: string | string[];
        transform: string;
        x: number;
        y: number;
        baseLine: string;
        constructor(id?: string, x?: number, y?: number, anchor?: string, text?: string | string[], transform?: string, baseLine?: string);
    }
    /** @private */
    export class PathOption extends CustomizeOption {
        opacity: number;
        fill: string;
        stroke: string;
        ['stroke-width']: number;
        ['stroke-dasharray']: string;
        d: string;
        constructor(id: string, fill: string, width: number, color: string, opacity?: number, dashArray?: string, d?: string);
    }
    /** @private */
    export class RectOption extends PathOption {
        x: number;
        y: number;
        height: number;
        width: number;
        rx: number;
        ry: number;
        transform: string;
        constructor(id: string, fill: string, border: BorderModel, opacity: number, rect: Rect, rx?: number, ry?: number, transform?: string, dashArray?: string);
    }
    /** @private */
    export class CircleOption extends PathOption {
        cy: number;
        cx: number;
        r: number;
        constructor(id: string, fill: string, border: BorderModel, opacity: number, cx: number, cy: number, r: number);
    }
    /** @private */
    export class PolygonOption {
        id: string;
        points: string;
        fill: string;
        constructor(id: string, points: string, fill: string);
    }
    /** @private */
    export class Size {
        height: number;
        width: number;
        constructor(width: number, height: number);
    }
    /** @private */
    export class Rect {
        x: number;
        y: number;
        height: number;
        width: number;
        constructor(x: number, y: number, width: number, height: number);
    }
    /** @private */
    export class ChartLocation {
        x: number;
        y: number;
        constructor(x: number, y: number);
    }
    /** @private */
    export class Thickness {
        left: number;
        right: number;
        top: number;
        bottom: number;
        constructor(left: number, right: number, top: number, bottom: number);
    }
    /** @private */
    export class ColorValue {
        r: number;
        g: number;
        b: number;
        constructor(r?: number, g?: number, b?: number);
    }
    /** @private */
    export class PointData {
        point: Points;
        series: Series;
        lierIndex: number;
        constructor(point: Points, series: Series, index?: number);
    }
    /** @private */
    export class AccPointData {
        point: AccPoints;
        series: AccumulationSeries;
        constructor(point: AccPoints, series: AccumulationSeries, index?: number);
    }
    /** @private */
    export class ControlPoints {
        controlPoint1: ChartLocation;
        controlPoint2: ChartLocation;
        constructor(controlPoint1: ChartLocation, controlPoint2: ChartLocation);
    }
    
  }
  export namespace circulargauge {
    
    /**
     * Interface for a class CircularGauge
     */
    export interface CircularGaugeModel extends ej.base.ComponentModel{
    
        /**
         * The width of the circular gauge as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, gauge will render to the full width of its parent element.
         * @default null
         */
    
        width?: string;
    
        /**
         * The height of the circular gauge as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, gauge will render to the full height of its parent element.
         * @default null
         */
    
        height?: string;
    
        /**
         * Options for customizing the color and width of the gauge border.
         */
    
        border?: BorderModel;
    
        /**
         * The background color of the gauge, which accepts value in hex, rgba as a valid CSS color string.
         * @default 'transparent'
         */
        background?: string;
    
        /**
         * Title for gauge
         * @default ''
         */
    
        title?: string;
    
        /**
         * Options for customizing the title of Gauge.
         */
    
        titleStyle?: FontModel;
    
        /**
         *  Options to customize the left, right, top and bottom margins of the gauge.
         */
    
        margin?: MarginModel;
    
        /**
         * Options for customizing the axes of gauge
         */
    
        axes?: AxisModel[];
    
        /**
         * Options for customizing the tooltip of gauge.
         */
    
        tooltip?: TooltipSettingsModel;
    
        /**
         * If set true, pointers can able to drag on interaction.
         * @default true
         */
        enablePointerDrag?: boolean;
    
        /**
         * X coordinate of the circular gauge center point, which takes values either in pixels or in percentage.
         * @default null
         */
    
        centerX?: string;
    
        /**
         * Y coordinate of the circular gauge center point, which takes values either in pixels or in percentage.
         * @default null
         */
    
        centerY?: string;
    
        /**
         * Specifies the theme for circular gauge.
         * * Material - Gauge render with material theme.
         * * Fabric - Gauge render with fabric theme.
         * @default Material
         */
        theme?: GaugeTheme;
    
        /**
         * Specifies whether a grouping separator should be used for a number.
         * @default false
         */
        useGroupingSeparator?: boolean;
    
        /**
         * Information about gauge for assistive technology.
         * @default null
         */
        description?: string;
    
        /**
         * TabIndex value for the gauge.
         * @default 1
         */
        tabIndex?: number;
    
        /**
         * Triggers after gauge loaded.
         * @event
         */
        loaded?: ej.base.EmitType<ILoadedEventArgs>;
    
        /**
         * Triggers before gauge load.
         * @event
         */
        load?: ej.base.EmitType<ILoadedEventArgs>;
    
        /**
         * Triggers after animation gets completed for pointers.
         * @event
         */
        animationComplete?: ej.base.EmitType<IAnimationCompleteEventArgs>;
    
        /**
         * Triggers before each axis label gets rendered.
         * @event
         */
        axisLabelRender?: ej.base.EmitType<IAxisLabelRenderEventArgs>;
    
        /**
         * Triggers before each annotation gets rendered.
         * @event
         */
        annotationRender?: ej.base.EmitType<IAnnotationRenderEventArgs>;
    
        /**
         * Triggers before the tooltip for pointer gets rendered.
         * @event
         */
    
        tooltipRender?: ej.base.EmitType<ITooltipRenderEventArgs>;
    
        /**
         * Triggers before the pointer is dragged.
         * @event
         */
    
        dragStart?: ej.base.EmitType<IPointerDragEventArgs>;
    
        /**
         * Triggers while dragging the pointers.
         * @event
         */
    
        dragMove?: ej.base.EmitType<IPointerDragEventArgs>;
    
        /**
         * Triggers after the pointer is dragged.
         * @event
         */
    
        dragEnd?: ej.base.EmitType<IPointerDragEventArgs>;
    
        /**
         * Triggers on hovering the circular gauge.
         * @event
         */
    
        gaugeMouseMove?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers while cursor leaves the circular gauge.
         * @event
         */
    
        gaugeMouseLeave?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on mouse down.
         * @event
         */
    
        gaugeMouseDown?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers on mouse up.
         * @event
         */
    
        gaugeMouseUp?: ej.base.EmitType<IMouseEventArgs>;
    
        /**
         * Triggers after window resize.
         * @event
         */
    
        resized?: ej.base.EmitType<IResizeEventArgs>;
    
    }
    /**
     * Circular Gauge
     */
    
    /**
     * Represents the Circular gauge control.
     * ```html
     * <div id="gauge"/>
     * <script>
     *   var gaugeObj = new CircularGauge();
     *   gaugeObj.appendTo("#gauge");
     * </script>
     * ```
     */
    export class CircularGauge extends ej.base.Component<HTMLElement> implements ej.base.INotifyPropertyChanged {
        /**
         * annotationModule is used to add annotation in gauge.
         */
        annotationsModule: Annotations;
        /**
         * `tooltipModule` is used to show the tooltip to the circular gauge..
         */
        tooltipModule: GaugeTooltip;
        /**
         * The width of the circular gauge as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, gauge will render to the full width of its parent element.
         * @default null
         */
        width: string;
        /**
         * The height of the circular gauge as a string in order to provide input as both like '100px' or '100%'.
         * If specified as '100%, gauge will render to the full height of its parent element.
         * @default null
         */
        height: string;
        /**
         * Options for customizing the color and width of the gauge border.
         */
        border: BorderModel;
        /**
         * The background color of the gauge, which accepts value in hex, rgba as a valid CSS color string.
         * @default 'transparent'
         */
        background: string;
        /**
         * Title for gauge
         * @default ''
         */
        title: string;
        /**
         * Options for customizing the title of Gauge.
         */
        titleStyle: FontModel;
        /**
         *  Options to customize the left, right, top and bottom margins of the gauge.
         */
        margin: MarginModel;
        /**
         * Options for customizing the axes of gauge
         */
        axes: AxisModel[];
        /**
         * Options for customizing the tooltip of gauge.
         */
        tooltip: TooltipSettingsModel;
        /**
         * If set true, pointers can able to drag on interaction.
         * @default true
         */
        enablePointerDrag: boolean;
        /**
         * X coordinate of the circular gauge center point, which takes values either in pixels or in percentage.
         * @default null
         */
        centerX: string;
        /**
         * Y coordinate of the circular gauge center point, which takes values either in pixels or in percentage.
         * @default null
         */
        centerY: string;
        /**
         * Specifies the theme for circular gauge.
         * * Material - Gauge render with material theme.
         * * Fabric - Gauge render with fabric theme.
         * @default Material
         */
        theme: GaugeTheme;
        /**
         * Specifies whether a grouping separator should be used for a number.
         * @default false
         */
        useGroupingSeparator: boolean;
        /**
         * Information about gauge for assistive technology.
         * @default null
         */
        description: string;
        /**
         * TabIndex value for the gauge.
         * @default 1
         */
        tabIndex: number;
        /**
         * Triggers after gauge loaded.
         * @event
         */
        loaded: ej.base.EmitType<ILoadedEventArgs>;
        /**
         * Triggers before gauge load.
         * @event
         */
        load: ej.base.EmitType<ILoadedEventArgs>;
        /**
         * Triggers after animation gets completed for pointers.
         * @event
         */
        animationComplete: ej.base.EmitType<IAnimationCompleteEventArgs>;
        /**
         * Triggers before each axis label gets rendered.
         * @event
         */
        axisLabelRender: ej.base.EmitType<IAxisLabelRenderEventArgs>;
        /**
         * Triggers before each annotation gets rendered.
         * @event
         */
        annotationRender: ej.base.EmitType<IAnnotationRenderEventArgs>;
        /**
         * Triggers before the tooltip for pointer gets rendered.
         * @event
         */
        tooltipRender: ej.base.EmitType<ITooltipRenderEventArgs>;
        /**
         * Triggers before the pointer is dragged.
         * @event
         */
        dragStart: ej.base.EmitType<IPointerDragEventArgs>;
        /**
         * Triggers while dragging the pointers.
         * @event
         */
        dragMove: ej.base.EmitType<IPointerDragEventArgs>;
        /**
         * Triggers after the pointer is dragged.
         * @event
         */
        dragEnd: ej.base.EmitType<IPointerDragEventArgs>;
        /**
         * Triggers on hovering the circular gauge.
         * @event
         */
        gaugeMouseMove: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers while cursor leaves the circular gauge.
         * @event
         */
        gaugeMouseLeave: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on mouse down.
         * @event
         */
        gaugeMouseDown: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers on mouse up.
         * @event
         */
        gaugeMouseUp: ej.base.EmitType<IMouseEventArgs>;
        /**
         * Triggers after window resize.
         * @event
         */
        resized: ej.base.EmitType<IResizeEventArgs>;
        /** @private */
        renderer: ej.base.SvgRenderer;
        /** @private */
        svgObject: Element;
        /** @private */
        availableSize: Size;
        /** @private */
        intl: ej.base.Internationalization;
        /** @private */
        private resizeTo;
        /** @private */
        midPoint: GaugeLocation;
        /** @private */
        activePointer: Pointer;
        /** @private */
        activeAxis: Axis;
        /** @private */
        animatePointer: boolean;
        /** @private */
        /**
         * Render axis panel for gauge.
         * @hidden
         */
        gaugeAxisLayoutPanel: AxisLayoutPanel;
        /**
         * Constructor for creating the widget
         * @hidden
         */
        constructor(options?: CircularGaugeModel, element?: string | HTMLElement);
        /**
         *  To create svg object, renderer and binding events for the container.
         */
        protected preRender(): void;
        private themeEffect();
        private setThemeColors(labelcolor, others);
        /**
         * To render the circular gauge elements
         */
        protected render(): void;
        /**
         * Method to unbind events for circular gauge
         */
        private unWireEvents();
        /**
         * Method to bind events for circular gauge
         */
        private wireEvents();
        /**
         * Handles the mouse move.
         * @return {boolean}
         * @private
         */
        mouseMove(e: PointerEvent): boolean;
        /**
         * Handles the mouse leave.
         * @return {boolean}
         * @private
         */
        mouseLeave(e: PointerEvent): boolean;
        /**
         * Handles the mouse right click.
         * @return {boolean}
         * @private
         */
        gaugeRightClick(event: MouseEvent | PointerEvent): boolean;
        /**
         * Handles the pointer draf while mouse move on gauge.
         * @private
         */
        pointerDrag(location: GaugeLocation): void;
        /**
         * Handles the mouse down on gauge.
         * @return {boolean}
         * @private
         */
        gaugeOnMouseDown(e: PointerEvent): boolean;
        /**
         * Handles the mouse end.
         * @return {boolean}
         * @private
         */
        mouseEnd(e: PointerEvent): boolean;
        /**
         * Handles the mouse event arguments.
         * @return {IMouseEventArgs}
         * @private
         */
        private getMouseArgs(e, type, name);
        /**
         * Handles the gauge resize.
         * @return {boolean}
         * @private
         */
        gaugeResize(e: Event): boolean;
        /**
         * Applying styles for circular gauge elements
         */
        private setGaugeStyle(element);
        /**
         * Method to set culture for gauge
         */
        private setCulture();
        /**
         * Methods to create svg element for circular gauge.
         */
        private createSvg();
        /**
         * To Remove the SVG from circular gauge.
         * @return {boolean}
         * @private
         */
        removeSvg(): void;
        /**
         * To initialize the circular gauge private variable.
         * @private
         */
        private initPrivateVariable();
        /**
         * To calculate the size of the circular gauge element.
         */
        private calculateSvgSize();
        /**
         * Method to calculate the availble size for circular gauge.
         */
        private calculateBounds();
        /**
         * To render elements for circular gauge
         */
        private renderElements();
        /**
         * Method to render the title for circular gauge.
         */
        private renderTitle();
        /**
         * Method to render the border for circular gauge.
         */
        private renderBorder();
        /**
         * Method to set the pointer value dynamically for circular gauge.
         */
        setPointerValue(axisIndex: number, pointerIndex: number, value: number): void;
        /**
         * Method to set the annotation content dynamically for circular gauge.
         */
        setAnnotationValue(axisIndex: number, annotationIndex: number, content: string): void;
        /**
         * Method to set the range values dynamically for circular gauge.
         */
        setRangeValue(axisIndex: number, rangeIndex: number, start: number, end: number): void;
        /**
         * To destroy the widget
         * @method destroy
         * @return {void}
         * @member of Circular-Gauge
         */
        destroy(): void;
        /**
         * To provide the array of modules needed for control rendering
         * @return {ej.base.ModuleDeclaration[]}
         * @private
         */
        requiredModules(): ej.base.ModuleDeclaration[];
        /**
         * Get the properties to be maintained in the persisted state.
         * @private
         */
        getPersistData(): string;
        /**
         * Called internally if any of the property value changed.
         * @private
         */
        onPropertyChanged(newProp: CircularGaugeModel, oldProp: CircularGaugeModel): void;
        /**
         * Get component name for circular gauge
         * @private
         */
        getModuleName(): string;
    }
    
    /**
     * Annotation Module handles the annotation of the axis.
     */
    export class Annotations {
        private gauge;
        private elementId;
        /**
         * Constructor for Annotation module.
         * @private.
         */
        constructor(gauge: CircularGauge);
        /**
         * Method to render the annotation for circular gauge.
         */
        renderAnnotation(axis: Axis, index: number): void;
        /**
         * Method to create annotation template for circular gauge.
         */
        createTemplate(element: HTMLElement, annotationIndex: number, axisIndex: number): void;
        /**
         * Method to update the annotation location for circular gauge.
         */
        private updateLocation(element, axis, annotation);
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the annotation.
         * @return {void}
         * @private
         */
        destroy(gauge: CircularGauge): void;
    }
    
    /**
     * Interface for a class Line
     */
    export interface LineModel {
    
        /**
         * The width of the line in pixels.
         * @default 1
         */
    
        width?: number;
    
        /**
         * The dash array of the axis line.
         * @default ''
         */
    
        dashArray?: string;
    
        /**
         * The color of the axis line, which accepts value in hex, rgba as a valid CSS color string.
         */
    
        color?: string;
    
    }
    
    /**
     * Interface for a class Label
     */
    export interface LabelModel {
    
        /**
         * The font of the axis labels
         */
    
        font?: FontModel;
    
        /**
         * To format the axis label, which accepts any global string format like 'C', 'n1', 'P' etc.
         * Also accepts placeholder like '{value}°C' in which value represent the axis label e.g. 20°C.
         * @default ''
         */
    
        format?: string;
    
        /**
         * Specifies the position of the labels. They are,
         * * inside -  Places the labels inside the axis.
         * * outside - Places the labels outside of the axis.
         * @default 'Inside'
         */
    
        position?: Position;
    
        /**
         * Specifies the label of an axis, which must get hide when an axis makes a complete circle. They are
         * * first -  Hides the 1st label on intersect.
         * * last - Hides the last label on intersect.
         * * none - Places both the labels.
         * @default 'None'
         */
    
        hiddenLabel?: HiddenLabel;
    
        /**
         * if set true, the labels will get rotated along the axis.
         * @default false
         */
        autoAngle?: boolean;
    
        /**
         * If set true, labels takes the range color.
         */
    
        useRangeColor?: boolean;
    
        /**
         * Distance of the labels from axis in pixel.
         * @default 0
         */
    
        offset?: number;
    
    }
    
    /**
     * Interface for a class Range
     */
    export interface RangeModel {
    
        /**
         * Specifies the minimum value of the range.
         * @default null
         */
    
        start?: number;
    
        /**
         * Specifies the maximum value of the range.
         * @default null
         */
    
        end?: number;
    
        /**
         * The radius of the range in pixels or in percentage.
         * @default null
         */
    
        radius?: string;
    
        /**
         * Specifies the start width of the ranges
         */
    
        startWidth?: number | string;
    
        /**
         * Specifies the end width of the ranges
         */
    
        endWidth?: number | string;
    
        /**
         * Specifies the color of the ranges
         */
    
        color?: string;
    
    }
    
    /**
     * Interface for a class Tick
     */
    export interface TickModel {
    
        /**
         * The width of the ticks in pixels.
         * @default null
         */
    
        width?: number;
    
        /**
         * The height of the line in pixels.
         * @default null
         */
    
        height?: number;
    
        /**
         * Specifies the interval of the tick line.
         * @default null
         */
    
        interval?: number;
    
        /**
         * Distance of the ticks from axis in pixel.
         * @default 0
         */
    
        offset?: number;
    
        /**
         * The color of the tick line, which accepts value in hex, rgba as a valid CSS color string.
         */
    
        color?: string;
    
        /**
         * Specifies the position of the ticks. They are
         * * inside -  Places the ticks inside the axis.
         * * outside - Places the ticks outside of the axis.
         * @default 'Inside'
         */
    
        position?: Position;
    
        /**
         * If set true, major ticks takes the range color.
         */
    
        useRangeColor?: boolean;
    
    }
    
    /**
     * Interface for a class Cap
     */
    export interface CapModel {
    
        /**
         * The color of the cap.
         * @default 'transparent'
         */
    
        color?: string;
    
        /**
         * Options for customizing the border of the cap.
         */
    
        border?: BorderModel;
    
        /**
         * Radius of the cap in pixels.
         * @default 8
         */
    
        radius?: number;
    
    }
    
    /**
     * Interface for a class NeedleTail
     */
    export interface NeedleTailModel {
    
        /**
         * The color of the back needle.
         */
    
        color?: string;
    
        /**
         * Options for customizing the border of the back needle.
         */
    
        border?: BorderModel;
    
        /**
         * The radius of the back needle in pixels or in percentage.
         * @default 0%
         */
    
        length?: string;
    
    }
    
    /**
     * Interface for a class Animation
     */
    export interface AnimationModel {
    
        /**
         * If set true, pointers get animate on initial loading.
         * @default true
         */
    
        enable?: boolean;
    
        /**
         * Duration of animation in milliseconds.
         * @default 1000
         */
    
        duration?: number;
    
    }
    
    /**
     * Interface for a class Annotation
     */
    export interface AnnotationModel {
    
        /**
         * Content of the annotation, which accepts the id of the custom element.
         * @default null
         */
        content?: string;
    
        /**
         * Angle for annotation with respect to axis.
         * @default 90
         */
        angle?: number;
    
        /**
         * Radius for annotation with respect to axis.
         * @default '50%'
         */
        radius?: string;
    
        /**
         * Order of an annotation in an axis.
         * @default '-1'
         */
        zIndex?: string;
    
        /**
         * Rotates the annotation along the axis.
         * @default false
         */
        autoAngle?: boolean;
    
        /**
         * Options for customizing the annotation text.
         */
    
        textStyle?: FontModel;
    
        /**
         * Information about annotation for assistive technology.
         * @default null
         */
        description?: string;
    
    }
    
    /**
     * Interface for a class Pointer
     */
    export interface PointerModel {
    
        /**
         * Specifies the value of the pointer.
         * @default null
         */
    
        value?: number;
    
        /**
         * Specifies the type of pointer for an axis.
         * * needle -  Renders a needle.
         * * marker - Renders a marker.
         * * rangeBar - Renders a rangeBar.
         * @default 'Needle'
         */
    
        type?: PointerType;
    
        /**
         * The URL for the Image that is to be displayed as pointer.
         * It requires marker shape value to be Image.
         * @default null
         */
        imageUrl?: string;
    
        /**
         * Length of the pointer in pixels or in percentage.
         * @default null
         */
        radius?: string;
    
        /**
         * Width of the pointer in pixels.
         * @default 20
         */
        pointerWidth?: number;
    
        /**
         * Options for customizing the cap
         */
    
        cap?: CapModel;
    
        /**
         * Options for customizing the back needle.
         */
    
        needleTail?: NeedleTailModel;
    
        /**
         * The color of the pointer.
         */
    
        color?: string;
    
        /**
         * Options for customizing the border of the needle.
         */
    
        border?: BorderModel;
    
        /**
         * Configures the animation of pointers.
         */
    
        animation?: AnimationModel;
    
        /**
         * Specifies the shape of the marker. They are
         * * circle - Renders a circle.
         * * rectangle - Renders a rectangle.
         * * triangle - Renders a triangle.
         * * diamond - Renders a diamond.
         * * invertedTriangle - Renders a invertedTriangle.
         * * image - Renders a image.
         * @default 'Circle'
         */
    
        markerShape?: GaugeShape;
    
        /**
         * The height of the marker in pixels.
         * @default 5
         */
    
        markerHeight?: number;
    
        /**
         * Information about pointer for assistive technology.
         * @default null
         */
        description?: string;
    
        /**
         * The width of the marker in pixels.
         * @default 5
         */
    
        markerWidth?: number;
    
    }
    
    /**
     * Interface for a class Axis
     */
    export interface AxisModel {
    
        /**
         * Specifies the minimum value of an axis.
         * @default null
         */
    
        minimum?: number;
    
        /**
         * Specifies the maximum value of an axis.
         * @default null
         */
    
        maximum?: number;
    
        /**
         * Radius of an axis in pixels or in percentage.
         * @default null
         */
    
        radius?: string;
    
        /**
         * Options for customizing the axis lines.
         */
    
        lineStyle?: LineModel;
    
        /**
         * Options for customizing the ranges of an axis
         */
    
        ranges?: RangeModel[];
    
        /**
         * Options for customizing the pointers of an axis
         */
    
        pointers?: PointerModel[];
    
        /**
         * ‘Annotation’ module is used to handle annotation action for an axis.
         */
    
        annotations?: AnnotationModel[];
    
        /**
         * Options for customizing the major tick lines.
         */
    
        majorTicks?: TickModel;
    
        /**
         * Options for customizing the minor tick lines.
         */
    
        minorTicks?: TickModel;
    
        /**
         * The start angle of an axis
         * @default 200
         */
    
        startAngle?: number;
    
        /**
         * The end angle of an axis
         * @default 160
         */
    
        endAngle?: number;
    
        /**
         * Specifies the direction of an axis. They are
         * * clockWise -  Renders the axis in clock wise direction.
         * * antiClockWise - Renders the axis in anti-clock wise direction.
         * @default 'clockWise'
         */
    
        direction?: GaugeDirection;
    
        /**
         * The background color of the axis, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
        background?: string;
    
        /**
         * Options to customize the axis label.
         */
    
        labelStyle?: LabelModel;
    
    }
    
    export class AxisLayoutPanel {
        private gauge;
        private farSizes;
        private axisRenderer;
        pointerRenderer: PointerRenderer;
        constructor(gauge: CircularGauge);
        /**
         * Measure the calculate the axis size and radius.
         * @return {void}
         * @private
         */
        measureAxis(rect: Rect): void;
        /**
         * Measure to calculate the axis radius of the circular gauge.
         * @return {void}
         * @private
         */
        private calculateAxesRadius();
        /**
         * Measure to calculate the axis size.
         * @return {void}
         * @private
         */
        private measureAxisSize(gauge, rect);
        /**
         * Calculate the axis values of the circular gauge.
         * @return {void}
         * @private
         */
        private calculateAxisValues(rect?);
        /**
         * Calculate the visible range of an axis.
         * @return {void}
         * @private
         */
        private calculateVisibleRange(axis, rect);
        /**
         * Calculate the numeric intervals of an axis range.
         * @return {void}
         * @private
         */
        private calculateNumericInterval(axis, rect);
        /**
         * Calculate the nice interval of an axis range.
         * @return {void}
         * @private
         */
        private calculateNiceInterval(maxValue, minValue, radius, degree);
        /**
         * Calculate the visible labels of an axis.
         * @return {void}
         * @private
         */
        private calculateVisibleLabels(axis);
        /**
         * Measure the axes available size.
         * @return {void}
         * @private
         */
        private computeSize(axes, rect);
        /**
         * To render the axis element of the circular gauge.
         * @return {void}
         * @private
         */
        renderAxes(): void;
        /**
         * Calculate maximum label width for the axis.
         * @return {void}
         */
        private getMaxLabelWidth(gauge, axis);
    }
    
    /**
     * Specifies the Axis rendering for circular gauge
     */
    export class AxisRenderer {
        private majorValues;
        private gauge;
        /**
         * Constructor for axis renderer.
         * @private.
         */
        constructor(gauge: CircularGauge);
        /**
         * Method to render the axis element of the circular gauge.
         * @return {void}
         * @private
         */
        drawAxisOuterLine(axis: Axis, index: number, element: Element, gauge: CircularGauge): void;
        /**
         * Method to render the axis line of the circular gauge.
         * @return {void}
         * @private
         */
        drawAxisLine(axis: Axis, index: number, element: Element, gauge: CircularGauge): void;
        /**
         * Method to render the axis labels of the circular gauge.
         * @return {void}
         * @private
         */
        drawAxisLabels(axis: Axis, index: number, element: Element, gauge: CircularGauge): void;
        /**
         * Method to find the anchor of the axis label.
         * @private
         */
        private findAnchor(location, style, angle, label);
        /**
         * Method to render the axis minor tick lines of the circular gauge.
         * @return {void}
         * @private
         */
        drawMinorTickLines(axis: Axis, index: number, element: Element, gauge: CircularGauge): void;
        /**
         * Method to render the axis major tick lines of the circular gauge.
         * @return {void}
         * @private
         */
        drawMajorTickLines(axis: Axis, index: number, element: Element, gauge: CircularGauge): void;
        /**
         * Method to calcualte the tick elements for the circular gauge.
         * @return {void}
         * @private
         */
        calculateTicks(value: number, options: Tick, axis: Axis): string;
        /**
         * Method to render the axis range of the circular gauge.
         * @return {void}
         * @private
         */
        drawAxisRange(axis: Axis, index: number, element: Element, gauge: CircularGauge): void;
        /**
         * Method to calculate the radius of the axis range.
         * @return {void}
         */
        private calculateRangeRadius(axis, range);
        /**
         * Method to get the range color of the circular gauge.
         * @return {void}
         * @private
         */
        setRangeColor(axis: Axis): void;
    }
    
    /**
     * Configures the axis line.
     */
    export class Line extends ej.base.ChildProperty<Line> {
        /**
         * The width of the line in pixels.
         * @default 1
         */
        width: number;
        /**
         * The dash array of the axis line.
         * @default ''
         */
        dashArray: string;
        /**
         * The color of the axis line, which accepts value in hex, rgba as a valid CSS color string.
         */
        color: string;
    }
    /**
     * Configures the axis label.
     */
    export class Label extends ej.base.ChildProperty<Label> {
        /**
         * The font of the axis labels
         */
        font: FontModel;
        /**
         * To format the axis label, which accepts any global string format like 'C', 'n1', 'P' etc.
         * Also accepts placeholder like '{value}°C' in which value represent the axis label e.g. 20°C.
         * @default ''
         */
        format: string;
        /**
         * Specifies the position of the labels. They are,
         * * inside -  Places the labels inside the axis.
         * * outside - Places the labels outside of the axis.
         * @default 'Inside'
         */
        position: Position;
        /**
         * Specifies the label of an axis, which must get hide when an axis makes a complete circle. They are
         * * first -  Hides the 1st label on intersect.
         * * last - Hides the last label on intersect.
         * * none - Places both the labels.
         * @default 'None'
         */
        hiddenLabel: HiddenLabel;
        /**
         * if set true, the labels will get rotated along the axis.
         * @default false
         */
        autoAngle: boolean;
        /**
         * If set true, labels takes the range color.
         */
        useRangeColor: boolean;
        /**
         * Distance of the labels from axis in pixel.
         * @default 0
         */
        offset: number;
    }
    /**
     * Configures the ranges of an axis.
     */
    export class Range extends ej.base.ChildProperty<Range> {
        /**
         * Specifies the minimum value of the range.
         * @default null
         */
        start: number;
        /**
         * Specifies the maximum value of the range.
         * @default null
         */
        end: number;
        /**
         * The radius of the range in pixels or in percentage.
         * @default null
         */
        radius: string;
        /**
         * Specifies the start width of the ranges
         */
        startWidth: number | string;
        /**
         * Specifies the end width of the ranges
         */
        endWidth: number | string;
        /**
         * Specifies the color of the ranges
         */
        color: string;
        /** @private */
        currentRadius: number;
        /** @private */
        rangeColor: string;
    }
    /**
     * Configures the major and minor tick lines of an axis.
     */
    export class Tick extends ej.base.ChildProperty<Tick> {
        /**
         * The width of the ticks in pixels.
         * @default null
         */
        width: number;
        /**
         * The height of the line in pixels.
         * @default null
         */
        height: number;
        /**
         * Specifies the interval of the tick line.
         * @default null
         */
        interval: number;
        /**
         * Distance of the ticks from axis in pixel.
         * @default 0
         */
        offset: number;
        /**
         * The color of the tick line, which accepts value in hex, rgba as a valid CSS color string.
         */
        color: string;
        /**
         * Specifies the position of the ticks. They are
         * * inside -  Places the ticks inside the axis.
         * * outside - Places the ticks outside of the axis.
         * @default 'Inside'
         */
        position: Position;
        /**
         * If set true, major ticks takes the range color.
         */
        useRangeColor: boolean;
    }
    /**
     * Configures the needle cap in pointer.
     */
    export class Cap extends ej.base.ChildProperty<Cap> {
        /**
         * The color of the cap.
         * @default 'transparent'
         */
        color: string;
        /**
         * Options for customizing the border of the cap.
         */
        border: BorderModel;
        /**
         * Radius of the cap in pixels.
         * @default 8
         */
        radius: number;
    }
    /**
     * Configures the back needle in pointers.
     */
    export class NeedleTail extends ej.base.ChildProperty<NeedleTail> {
        /**
         * The color of the back needle.
         */
        color: string;
        /**
         * Options for customizing the border of the back needle.
         */
        border: BorderModel;
        /**
         * The radius of the back needle in pixels or in percentage.
         * @default 0%
         */
        length: string;
    }
    /**
     * Configures the animation of pointers.
     */
    export class Animation extends ej.base.ChildProperty<Animation> {
        /**
         * If set true, pointers get animate on initial loading.
         * @default true
         */
        enable: boolean;
        /**
         * Duration of animation in milliseconds.
         * @default 1000
         */
        duration: number;
    }
    /**
     * ‘Annotation’ module is used to handle annotation action for an axis.
     */
    export class Annotation extends ej.base.ChildProperty<Annotation> {
        /**
         * Content of the annotation, which accepts the id of the custom element.
         * @default null
         */
        content: string;
        /**
         * Angle for annotation with respect to axis.
         * @default 90
         */
        angle: number;
        /**
         * Radius for annotation with respect to axis.
         * @default '50%'
         */
        radius: string;
        /**
         * Order of an annotation in an axis.
         * @default '-1'
         */
        zIndex: string;
        /**
         * Rotates the annotation along the axis.
         * @default false
         */
        autoAngle: boolean;
        /**
         * Options for customizing the annotation text.
         */
        textStyle: FontModel;
        /**
         * Information about annotation for assistive technology.
         * @default null
         */
        description: string;
    }
    /**
     * Configures the pointers of an axis.
     */
    export class Pointer extends ej.base.ChildProperty<Pointer> {
        /**
         * Specifies the value of the pointer.
         * @default null
         */
        value: number;
        /**
         * Specifies the type of pointer for an axis.
         * * needle -  Renders a needle.
         * * marker - Renders a marker.
         * * rangeBar - Renders a rangeBar.
         * @default 'Needle'
         */
        type: PointerType;
        /**
         * The URL for the Image that is to be displayed as pointer.
         * It requires marker shape value to be Image.
         * @default null
         */
        imageUrl: string;
        /**
         * Length of the pointer in pixels or in percentage.
         * @default null
         */
        radius: string;
        /**
         * Width of the pointer in pixels.
         * @default 20
         */
        pointerWidth: number;
        /**
         * Options for customizing the cap
         */
        cap: CapModel;
        /**
         * Options for customizing the back needle.
         */
        needleTail: NeedleTailModel;
        /**
         * The color of the pointer.
         */
        color: string;
        /**
         * Options for customizing the border of the needle.
         */
        border: BorderModel;
        /**
         * Configures the animation of pointers.
         */
        animation: AnimationModel;
        /**
         * Specifies the shape of the marker. They are
         * * circle - Renders a circle.
         * * rectangle - Renders a rectangle.
         * * triangle - Renders a triangle.
         * * diamond - Renders a diamond.
         * * invertedTriangle - Renders a invertedTriangle.
         * * image - Renders a image.
         * @default 'Circle'
         */
        markerShape: GaugeShape;
        /**
         * The height of the marker in pixels.
         * @default 5
         */
        markerHeight: number;
        /**
         * Information about pointer for assistive technology.
         * @default null
         */
        description: string;
        /**
         * The width of the marker in pixels.
         * @default 5
         */
        markerWidth: number;
        /** @private */
        currentValue: number;
        /** @private */
        pathElement: Element[];
        /** @private */
        currentRadius: number;
    }
    /**
     * Configures an axis in a gauge.
     */
    export class Axis extends ej.base.ChildProperty<Axis> {
        /**
         * Specifies the minimum value of an axis.
         * @default null
         */
        minimum: number;
        /**
         * Specifies the maximum value of an axis.
         * @default null
         */
        maximum: number;
        /**
         * Radius of an axis in pixels or in percentage.
         * @default null
         */
        radius: string;
        /**
         * Options for customizing the axis lines.
         */
        lineStyle: LineModel;
        /**
         * Options for customizing the ranges of an axis
         */
        ranges: RangeModel[];
        /**
         * Options for customizing the pointers of an axis
         */
        pointers: PointerModel[];
        /**
         * ‘Annotation’ module is used to handle annotation action for an axis.
         */
        annotations: AnnotationModel[];
        /**
         * Options for customizing the major tick lines.
         */
        majorTicks: TickModel;
        /**
         * Options for customizing the minor tick lines.
         */
        minorTicks: TickModel;
        /**
         * The start angle of an axis
         * @default 200
         */
        startAngle: number;
        /**
         * The end angle of an axis
         * @default 160
         */
        endAngle: number;
        /**
         * Specifies the direction of an axis. They are
         * * clockWise -  Renders the axis in clock wise direction.
         * * antiClockWise - Renders the axis in anti-clock wise direction.
         * @default 'clockWise'
         */
        direction: GaugeDirection;
        /**
         * The background color of the axis, which accepts value in hex, rgba as a valid CSS color string.
         * @default null
         */
        background: string;
        /**
         * Options to customize the axis label.
         */
        labelStyle: LabelModel;
        /** @private */
        currentRadius: number;
        /** @private */
        visibleRange: VisibleRangeModel;
        /** @private */
        visibleLabels: VisibleLabels[];
        /** @private */
        maxLabelSize: Size;
        /** @private */
        rect: Rect;
        /** @private */
        nearSize: number;
        /** @private */
        farSize: number;
    }
    /** @private */
    export interface VisibleRangeModel {
        min?: number;
        max?: number;
        interval?: number;
    }
    
    /**
     * Specifies the Axis rendering for circular gauge
     */
    export class PointerRenderer {
        private gauge;
        /**
         * Constructor for pointer renderer.
         * @private.
         */
        constructor(gauge: CircularGauge);
        /**
         * Method to render the axis pointers of the circular gauge.
         * @return {void}
         * @private
         */
        drawPointers(axis: Axis, axisIndex: number, element: Element, gauge: CircularGauge): void;
        /**
         * Measure the pointer length of the circular gauge.
         * @return {void}
         */
        private calculatePointerRadius(axis, pointer);
        /**
         * Method to render the needle pointer of the ciruclar gauge.
         * @return {void}
         */
        private drawNeedlePointer(axis, axisIndex, index, parentElement, gauge);
        /**
         * Method to set the pointer value of the circular gauge.
         * @return {void}
         * @private
         */
        setPointerValue(axis: Axis, pointer: Pointer, value: number): void;
        /**
         * Method to render the marker pointer of the ciruclar gauge.
         * @return {void}
         */
        private drawMarkerPointer(axis, axisIndex, index, parentElement, gauge);
        /**
         * Method to render the range bar pointer of the ciruclar gauge.
         * @return {void}
         */
        private drawRangeBarPointer(axis, axisIndex, index, parentElement, gauge);
        /**
         * Method to perform the animation of the pointer in circular gauge.
         * @return {void}
         */
        private doPointerAnimation(pointer, axis);
        /**
         * Perform the needle and marker pointer animation for circular gauge.
         * @return {void}
         * @private
         */
        performNeedleAnimation(element: HTMLElement, start: number, end: number, axis: Axis, pointer: Pointer, radius?: number, innerRadius?: number): void;
        /**
         * Perform the range bar pointer animation for circular gauge.
         * @return {void}
         * @private
         */
        performRangeBarAnimation(element: HTMLElement, start: number, end: number, axis: Axis, pointer: Pointer, radius: number, innerRadius?: number): void;
    }
    
    /**
     * Interface for a class Border
     */
    export interface BorderModel {
    
        /**
         * The color of the border, which accepts value in hex, rgba as a valid CSS color string.
         */
        color?: string;
    
        /**
         * The width of the border in pixels.
         */
        width?: number;
    
    }
    
    /**
     * Interface for a class Font
     */
    export interface FontModel {
    
        /**
         * Font size for text.
         */
        size?: string;
    
        /**
         * Color for text.
         */
        color?: string;
    
        /**
         * FontFamily for text.
         */
        fontFamily?: string;
    
        /**
         * FontWeight for text.
         */
        fontWeight?: string;
    
        /**
         * FontStyle for text.
         */
        fontStyle?: string;
    
        /**
         * Opacity for text.
         */
        opacity?: number;
    
    }
    
    /**
     * Interface for a class Margin
     */
    export interface MarginModel {
    
        /**
         * Left margin in pixels.
         * @default 10
         */
        left?: number;
    
        /**
         * Right margin in pixels.
         * @default 10
         */
        right?: number;
    
        /**
         * Top margin in pixels.
         * @default 10
         */
        top?: number;
    
        /**
         * Bottom margin in pixels.
         * @default 10
         */
        bottom?: number;
    
    }
    
    /**
     * Interface for a class TooltipSettings
     */
    export interface TooltipSettingsModel {
    
        /**
         * Enable / Disable the visibility of tooltip.
         * @default false
         */
    
        enable?: boolean;
    
        /**
         * The fill color of the tooltip, which accepts value in hex, rgba as a valid CSS color string. 
         */
    
        fill?: string;
    
        /**
         * Options to customize the tooltip text.
         */
    
        textStyle?: FontModel;
    
        /**
         * Format of the tooltip content.
         * @default null
         */
    
        format?: string;
    
        /**
         * Custom template to format the tooltip content. Use ${x} and ${y} as a placeholder text to display the corresponding data point.
         * @default null
         */
    
        template?: string;
    
        /**
         * If set true, tooltip will animate, while moving from one point to another.
         * @default true
         */
        enableAnimation?: boolean;
    
        /**
         * Options to customize the border for tooltip.
         */
        border?: BorderModel;
    
    }
    
    /**
     * Configures the borders in circular gauge.
     */
    export class Border extends ej.base.ChildProperty<Border> {
        /**
         * The color of the border, which accepts value in hex, rgba as a valid CSS color string.
         */
        color: string;
        /**
         * The width of the border in pixels.
         */
        width: number;
    }
    /**
     * Configures the fonts in circular gauge.
     */
    export class Font extends ej.base.ChildProperty<Font> {
        /**
         * Font size for text.
         */
        size: string;
        /**
         * Color for text.
         */
        color: string;
        /**
         * FontFamily for text.
         */
        fontFamily: string;
        /**
         * FontWeight for text.
         */
        fontWeight: string;
        /**
         * FontStyle for text.
         */
        fontStyle: string;
        /**
         * Opacity for text.
         */
        opacity: number;
    }
    /**
     * Configures the margin of circular gauge.
     */
    export class Margin extends ej.base.ChildProperty<Margin> {
        /**
         * Left margin in pixels.
         * @default 10
         */
        left: number;
        /**
         * Right margin in pixels.
         * @default 10
         */
        right: number;
        /**
         * Top margin in pixels.
         * @default 10
         */
        top: number;
        /**
         * Bottom margin in pixels.
         * @default 10
         */
        bottom: number;
    }
    /**
     * Configures the tooltip in circular gauge.
     */
    export class TooltipSettings extends ej.base.ChildProperty<TooltipSettings> {
        /**
         * Enable / Disable the visibility of tooltip.
         * @default false
         */
        enable: boolean;
        /**
         * The fill color of the tooltip, which accepts value in hex, rgba as a valid CSS color string.
         */
        fill: string;
        /**
         * Options to customize the tooltip text.
         */
        textStyle: FontModel;
        /**
         * Format of the tooltip content.
         * @default null
         */
        format: string;
        /**
         * Custom template to format the tooltip content. Use ${x} and ${y} as a placeholder text to display the corresponding data point.
         * @default null
         */
        template: string;
        /**
         * If set true, tooltip will animate, while moving from one point to another.
         * @default true
         */
        enableAnimation: boolean;
        /**
         * Options to customize the border for tooltip.
         */
        border: BorderModel;
    }
    
    /**
     * Specifies the gauge constant value
     */
    /** @private */
    export const loaded: string;
    /** @private */
    export const load: string;
    /** @private */
    export const animationComplete: string;
    /** @private */
    export const axisLabelRender: string;
    /** @private */
    export const tooltipRender: string;
    /** @private */
    export const annotationRender: string;
    /** @private */
    export const gaugeMouseMove: string;
    /** @private */
    export const gaugeMouseLeave: string;
    /** @private */
    export const gaugeMouseDown: string;
    /** @private */
    export const gaugeMouseUp: string;
    /** @private */
    export const dragStart: string;
    /** @private */
    export const dragMove: string;
    /** @private */
    export const dragEnd: string;
    /** @private */
    export const resized: string;
    
    /**
     * interface doc
     */
    
    /**
     * Specifies Circular-Gauge Events
     */
    export interface ICircularGaugeEventArgs {
        /**
         * name of the event
         */
        name: string;
        /**
         * to cancel the event
         */
        cancel: boolean;
    }
    /**
     * Specifies Loaded event arguments for circular gauge.
     */
    export interface ILoadedEventArgs extends ICircularGaugeEventArgs {
        /**
         * gauge event argument
         */
        gauge: CircularGauge;
    }
    /**
     * Specifies AnimationComplete event arguments for circular gauge.
     */
    export interface IAnimationCompleteEventArgs extends ICircularGaugeEventArgs {
        /**
         * axis event argument
         */
        axis: Axis;
        /**
         * pointer event argument
         */
        pointer: Pointer;
    }
    /**
     * Specifies AxisLabelRender event arguments for circular gauge.
     */
    export interface IAxisLabelRenderEventArgs extends ICircularGaugeEventArgs {
        /**
         * axis event argument
         */
        axis: Axis;
        /**
         * text event argument
         */
        text: string;
        /**
         * value event argument
         */
        value: number;
    }
    /**
     * Specifies TooltipRender event arguments for circular gauge.
     */
    export interface ITooltipRenderEventArgs extends ICircularGaugeEventArgs {
        /**
         * content event argument
         */
        content?: string | HTMLElement;
        /**
         * textStyle event argument
         */
        textStyle?: FontModel;
        /**
         * border event argument
         */
        border?: BorderModel;
        /**
         * axis event argument
         */
        axis: Axis;
        /**
         * pointer event argument
         */
        pointer: Pointer;
    }
    /**
     * Specifies AnnotationRender event arguments for circular gauge.
     */
    export interface IAnnotationRenderEventArgs extends ICircularGaugeEventArgs {
        /**
         * content event argument
         */
        content?: string;
        /**
         * textStyle event argument
         */
        textStyle?: FontModel;
        /**
         * axis event argument
         */
        axis: Axis;
        /**
         * annotation event argument
         */
        annotation: Annotation;
    }
    /**
     * Specifies DragStart, DragMove and DragEnd events arguments for circular gauge.
     */
    export interface IPointerDragEventArgs {
        /**
         * name event argument
         */
        name: string;
        /**
         * axis event argument
         */
        axis: Axis;
        /**
         * pointer event argument
         */
        pointer: Pointer;
        /**
         * currentValue event argument
         */
        currentValue: number;
        /**
         * previousValue event argument
         */
        previousValue?: number;
    }
    /**
     * Specifies Resize event arguments for circular gauge.
     */
    export interface IResizeEventArgs {
        /**
         * name event argument
         */
        name: string;
        /**
         * previousSize event argument
         */
        previousSize: Size;
        /**
         * currentSize event argument
         */
        currentSize: Size;
        /**
         * gauge event argument
         */
        gauge: CircularGauge;
    }
    /**
     * Specifies Mouse events arguments for circular gauge.
     */
    export interface IMouseEventArgs extends ICircularGaugeEventArgs {
        /**
         * target event argument
         */
        target: Element;
        /**
         * x event argument
         */
        x: number;
        /**
         * y event argument
         */
        y: number;
    }
    /**
     * Specifies visible point
     */
    export interface IVisiblePointer {
        /**
         * axisIndex event argument
         */
        axisIndex?: number;
        /**
         * pointerIndex event argument
         */
        pointerIndex?: number;
    }
    /**
     * Specifies font mapping
     */
    export interface IFontMapping {
        /**
         * size event argument
         */
        size?: string;
        /**
         * color event argument
         */
        color?: string;
        /**
         * fontWeight event argument
         */
        fontWeight?: string;
        /**
         * fontStyle event argument
         */
        fontStyle?: string;
        /**
         * fontFamily event argument
         */
        fontFamily?: string;
    }
    
    /**
     * Specifies gauge Themes
     */
    export namespace Theme {
        /** @private */
        let axisLabelFont: IFontMapping;
        /** @private */
        let axisLineColor: string;
        /** @private */
        let tickLineColor: string;
        /** @private */
        let pointerColor: string;
    }
    /** @private */
    export function getRangePalette(theme: string): string[];
    
    /**
     * Tooltip Module handles the tooltip of the circular gauge
     */
    export class GaugeTooltip {
        private gauge;
        private element;
        private currentAxis;
        private tooltip;
        private currentPointer;
        private borderStyle;
        private textStyle;
        ejTooltip: ej.popups.Tooltip;
        /**
         * Constructor for Tooltip module.
         * @private.
         */
        constructor(gauge: CircularGauge);
        /**
         * Method to render the tooltip for circular gauge.
         */
        renderTooltip(target: Element): void;
        /**
         * Method to append the target element for tooltip
         */
        private appendTargetElement(angle);
        /**
         * Method to find the position of the tooltip anchor for circular gauge.
         */
        private findPosition(angle);
        /**
         * Method to perform the tooltip for circular gauge.
         */
        private onBeforeRender(args);
        /**
         * Method to customize the tooltip elements for circular gauge.
         */
        private tooltipCustomization(args);
        /**
         * Get module name.
         */
        protected getModuleName(): string;
        /**
         * To destroy the tooltip.
         * @return {void}
         * @private
         */
        destroy(gauge: CircularGauge): void;
    }
    
    /**
     * Defines position of the axis ticks / labels. They are
     * * inside
     * * outside
     * @private
     */
    export type Position = 'Inside' | 'Outside';
    /**
     * Defines Pointer type of the axis. They are
     * * needle
     * * marker
     * * rangeBar
     * @private
     */
    export type PointerType = 'Needle' | 'Marker' | 'RangeBar';
    /**
     * Defines Direction of the gauge. They are
     * * ClockWise
     * * AntiClockWise
     * @private
     */
    export type GaugeDirection = 'ClockWise' | 'AntiClockWise';
    /**
     * Defines Theme of the gauge. They are
     * * Material
     * * Fabric
     * @private
     */
    export type GaugeTheme = 'Material' | 'Bootstrap' | 'Highcontrast' | 'Fabric';
    /**
     * Defines Hidden label of the axis. They are
     * * First
     * * Last
     * @private
     */
    export type HiddenLabel = 'First' | 'Last' | 'None';
    /**
     * Defines the shape of marker. They are
     * * circle - Renders a circle.
     * * rectangle - Renders a rectangle.
     * * triangle - Renders a triangle.
     * * diamond - Renders a diamond.
     * * cross - Renders a cross.
     * * horizontalLine - Renders a horizontalLine.
     * * verticalLine - Renders a verticalLine.
     * * pentagon- Renders a pentagon.
     * * invertedTriangle - Renders a invertedTriangle.
     * * image - Renders a image.
     */
    export type GaugeShape = 'Circle' | 'Rectangle' | 'Triangle' | 'Diamond' | 'InvertedTriangle' | 'Image';
    
    /**
     * Function to measure the height and width of the text.
     * @param  {string} text
     * @param  {FontModel} font
     * @param  {string} id
     * @returns Size
     * @private
     */
    export function measureText(text: string, font: FontModel): Size;
    /**
     * Function to find number from string
     * * @returns number
     * @private
     */
    export function toPixel(value: string, maxDimension: number): number;
    /**
     * Function to get the style from FontModel.
     * @returns string
     * @private
     */
    export function getFontStyle(font: FontModel): string;
    /**
     * Function to set style to the element.
     * @private
     */
    export function setStyles(element: HTMLElement, fill: string, border: BorderModel): void;
    /**
     * Function to measure the element rect.
     * @returns ClientRect
     * @private
     */
    export function measureElementRect(element: HTMLElement): ClientRect;
    /**
     * Function to convert the number from string.
     * @returns number
     * @private
     */
    export function stringToNumber(value: string, containerSize: number): number;
    /**
     * Function to create the text element.
     * @returns Element
     * @private
     */
    export function textElement(options: TextOption, font: FontModel, color: string, parent: HTMLElement | Element, styles?: string): Element;
    /**
     * Function to append the path to the element.
     * @returns Element
     * @private
     */
    export function appendPath(options: PathOption, element: Element, gauge: CircularGauge, functionName?: string): Element;
    /**
     * Function to calculate the sum of array values.
     * @returns number
     * @private
     */
    export function calculateSum(from: number, to: number, values: number[]): number;
    /**
     * Function to calculate the value for linear animation effect
     * @param currentTime
     * @param startValue
     * @param endValue
     * @param duration
     * @private
     */
    export function linear(currentTime: number, startValue: number, endValue: number, duration: number): number;
    /**
     * Function to get the angle from value for circular gauge.
     * @returns number
     * @private
     */
    export function getAngleFromValue(value: number, maximumValue: number, minimumValue: number, startAngle: number, endAngle: number, isClockWise: boolean): number;
    /**
     * Function to get the degree for circular gauge.
     * @returns number
     * @private
     */
    export function getDegree(startAngle: number, endAngle: number): number;
    /**
     * Function to get the value from angle for circular gauge.
     * @returns number
     * @private
     */
    export function getValueFromAngle(angle: number, maximumValue: number, minimumValue: number, startAngle: number, endAngle: number, isClockWise: boolean): number;
    /**
     * Function to check whether it's a complete circle for circular gauge.
     * @returns boolean
     * @private
     */
    export function isCompleteAngle(startAngle: number, endAngle: number): boolean;
    /**
     * Function to get angle from location for circular gauge.
     * @returns number
     * @private
     */
    export function getAngleFromLocation(center: GaugeLocation, point: GaugeLocation): number;
    /**
     * Function to get the location from angle for circular gauge.
     * @returns GaugeLocation
     * @private
     */
    export function getLocationFromAngle(degree: number, radius: number, center: GaugeLocation): GaugeLocation;
    /**
     * Function to get the path direction of the circular gauge.
     * @returns string
     * @private
     */
    export function getPathArc(center: GaugeLocation, start: number, end: number, radius: number, startWidth?: number, endWidth?: number): string;
    /**
     * Function to get the range path direction of the circular gauge.
     * @returns string
     * @private
     */
    export function getRangePath(start: GaugeLocation, end: GaugeLocation, innerStart: GaugeLocation, innerEnd: GaugeLocation, radius: number, startRadius: number, endRadius: number, clockWise: number): string;
    /**
     * Function to calculate the complete path arc of the circular gauge.
     * @returns string
     * @private
     */
    export function getCompleteArc(center: GaugeLocation, start: number, end: number, radius: number, innerRadius: number): string;
    /**
     * Function to get the circular path direction of the circular gauge.
     * @returns string
     * @private
     */
    export function getCirclePath(start: GaugeLocation, end: GaugeLocation, radius: number, clockWise: number): string;
    /**
     * Function to get the complete path direction of the circular gauge.
     * @returns string
     * @private
     */
    export function getCompletePath(center: GaugeLocation, start: GaugeLocation, end: GaugeLocation, radius: number, innerStart: GaugeLocation, innerEnd: GaugeLocation, innerRadius: number, clockWise: number): string;
    /**
     * Function to get element from id.
     * @returns Element
     * @private
     */
    export function getElement(id: string): Element;
    /**
     * Function to compile the template function for circular gauge.
     * @returns Function
     * @private
     */
    export function getTemplateFunction(template: string): Function;
    /**
     * Function to remove the element from id.
     * @private
     */
    export function removeElement(id: string): void;
    /**
     * Function to get current point for circular gauge using element id.
     * @returns IVisiblePointer
     * @private
     */
    export function getPointer(targetId: string, gauge: CircularGauge): IVisiblePointer;
    /**
     * Function to convert the label using formar for cirular gauge.
     * @returns string
     * @private
     */
    export function getLabelFormat(format: string): string;
    /**
     * Function to calculate the marker shape for circular gauge.
     * @returns PathOption
     * @private
     */
    export function calculateShapes(location: GaugeLocation, shape: string, size: Size, url: string, options: PathOption): PathOption;
    /**
     * Function to get range color from value for circular gauge.
     * @returns string
     * @private
     */
    export function getRangeColor(value: number, ranges: Range[], color: string): string;
    /** @private */
    export class CustomizeOption {
        id: string;
        constructor(id?: string);
    }
    /** @private */
    export class PathOption extends CustomizeOption {
        opacity: number;
        fill: string;
        stroke: string;
        ['stroke-width']: number;
        ['stroke-dasharray']: string;
        d: string;
        transform: string;
        style: string;
        constructor(id: string, fill: string, width: number, color: string, opacity?: number, dashArray?: string, d?: string, transform?: string, style?: string);
    }
    /** @private */
    export class RectOption extends CustomizeOption {
        x: number;
        y: number;
        height: number;
        width: number;
        opacity: number;
        fill: string;
        stroke: string;
        ['stroke-width']: number;
        constructor(id: string, fill: string, border: BorderModel, opacity: number, rect: Rect);
    }
    /**
     * Internal class size
     */
    export class Size {
        /**
         * Specifies the height.
         */
        height: number;
        /**
         * Specifies the width.
         */
        width: number;
        constructor(width: number, height: number);
    }
    /** @private */
    export class GaugeLocation {
        x: number;
        y: number;
        constructor(x: number, y: number);
    }
    /** @private */
    export class Rect {
        x: number;
        y: number;
        height: number;
        width: number;
        constructor(x: number, y: number, width: number, height: number);
    }
    /** @private */
    export class TextOption extends CustomizeOption {
        anchor: string;
        text: string;
        transform: string;
        x: number;
        y: number;
        baseLine: string;
        constructor(id?: string, x?: number, y?: number, anchor?: string, text?: string, transform?: string, baseLine?: string);
    }
    /** @private */
    export class VisibleLabels {
        text: string;
        value: number;
        size: Size;
        constructor(text: string, value: number, size?: Size);
    }
    
  }
  export namespace compression {
    
    /**
     * represent compression stream writer
     * ```typescript
     * let compressedWriter = new CompressedStreamWriter();
     * let text: string = 'Hello world!!!';
     * compressedWriter.write(text, 0, text.length);
     * compressedWriter.close();
     * ```
     */
    export class CompressedStreamWriter {
        private stream;
        private pendingBuffer;
        private pendingBufLength;
        private pendingBufCache;
        private pendingBufBitsInCache;
        private treeLiteral;
        private treeDistances;
        private treeCodeLengths;
        private bufferPosition;
        private arrLiterals;
        private arrDistances;
        private extraBits;
        private currentHash;
        private hashHead;
        private hashPrevious;
        private matchStart;
        private matchLength;
        private matchPrevAvail;
        private blockStart;
        private stringStart;
        private lookAhead;
        private dataWindow;
        private inputBuffer;
        private totalBytesIn;
        private inputOffset;
        private inputEnd;
        private windowSize;
        private windowMask;
        private hashSize;
        private hashMask;
        private hashShift;
        private maxDist;
        private checkSum;
        private noWrap;
        /**
         * get compressed data
         */
        readonly compressedData: Uint8Array[];
        readonly getCompressedString: string;
        /**
         * Initializes compressor and writes ZLib header if needed.
         * @param {boolean} noWrap - optional if true, ZLib header and checksum will not be written.
         */
        constructor(noWrap?: boolean);
        /**
         * Compresses data and writes it to the stream.
         * @param {Uint8Array} data - data to compress
         * @param {number} offset - offset in data
         * @param {number} length - length of the data
         * @returns {void}
         */
        write(data: Uint8Array | string, offset: number, length: number): void;
        /**
         * write ZLib header to the compressed data
         * @return {void}
         */
        writeZLibHeader(): void;
        /**
         *  Write Most Significant Bytes in to stream
         * @param {number} s - check sum value
         */
        pendingBufferWriteShortBytes(s: number): void;
        private compressData(finish);
        private compressSlow(flush, finish);
        private discardMatch();
        private matchPreviousAvailable();
        private matchPreviousBest(prevMatch, prevLen);
        private lookAheadCompleted(finish);
        private huffmanIsFull(finish);
        private fillWindow();
        private slideWindow();
        private insertString();
        private findLongestMatch(curMatch);
        private updateHash();
        private huffmanTallyLit(literal);
        private huffmanTallyDist(dist, len);
        private huffmanFlushBlock(stored, storedOffset, storedLength, lastBlock);
        private huffmanFlushStoredBlock(stored, storedOffset, storedLength, lastBlock);
        private huffmanLengthCode(len);
        private huffmanDistanceCode(distance);
        private huffmanSendAllTrees(blTreeCodes);
        private huffmanReset();
        private huffmanCompressBlock();
        /**
         * write bits in to internal buffer
         * @param {number} b - source of bits
         * @param {number} count - count of bits to write
         */
        pendingBufferWriteBits(b: number, count: number): void;
        private pendingBufferFlush(isClose?);
        private pendingBufferFlushBits();
        private pendingBufferWriteByteBlock(data, offset, length);
        private pendingBufferWriteShort(s);
        private pendingBufferAlignToByte();
        /**
         * close the stream and write all pending buffer in to stream
         * @returns {void}
         */
        close(): void;
        /**
         * release allocated un-managed resource
         * @returns {void}
         */
        destroy(): void;
    }
    /**
     * represent the Huffman Tree
     */
    export class CompressorHuffmanTree {
        private codeFrequency;
        private codes;
        private codeLength;
        private lengthCount;
        private codeMinCount;
        private codeCount;
        private maxLength;
        private writer;
        private static reverseBits;
        static huffCodeLengthOrders: number[];
        readonly treeLength: number;
        readonly codeLengths: Uint8Array;
        readonly codeFrequencies: Uint16Array;
        /**
         * Create new Huffman Tree
         * @param {CompressedStreamWriter} writer instance
         * @param {number} elementCount - element count
         * @param {number} minCodes - minimum count
         * @param {number} maxLength - maximum count
         */
        constructor(writer: CompressedStreamWriter, elementCount: number, minCodes: number, maxLength: number);
        setStaticCodes(codes: Int16Array, lengths: Uint8Array): void;
        /**
         * reset all code data in tree
         * @returns {void}
         */
        reset(): void;
        /**
         * write code to the compressor output stream
         * @param {number} code - code to be written
         * @returns {void}
         */
        writeCodeToStream(code: number): void;
        /**
         * calculate code from their frequencies
         * @returns {void}
         */
        buildCodes(): void;
        static bitReverse(value: number): number;
        /**
         * calculate length of compressed data
         * @returns {number}
         */
        getEncodedLength(): number;
        /**
         * calculate code frequencies
         * @param {CompressorHuffmanTree} blTree
         * @returns {void}
         */
        calculateBLFreq(blTree: CompressorHuffmanTree): void;
        /**
         * @param {CompressorHuffmanTree} blTree - write tree to output stream
         * @returns {void}
         */
        writeTree(blTree: CompressorHuffmanTree): void;
        /**
         * Build huffman tree
         * @returns {void}
         */
        buildTree(): void;
        private constructHuffmanTree(arrTree, treeLength, values, nodesCount, child);
        private buildLength(child);
        private recreateTree(child, overflow, numLeafs);
        private calculateOptimalCodeLength(child, overflow, numNodes);
    }
    /**
     * Checksum calculator, based on Adler32 algorithm.
     */
    export class ChecksumCalculator {
        private static checkSumBitOffset;
        private static checksumBase;
        private static checksumIterationCount;
        /**
         * Updates checksum by calculating checksum of the
         * given buffer and adding it to current value.
         * @param {number} checksum - current checksum.
         * @param {Uint8Array} buffer - data byte array.
         * @param {number} offset - offset in the buffer.
         * @param {number} length - length of data to be used from the stream.
         * @returns {number}
         */
        static checksumUpdate(checksum: number, buffer: Uint8Array, offset: number, length: number): number;
    }
    
    /**
     * class provide compression library
     * ```typescript
     * let archive = new ZipArchive();
     * archive.compressionLevel = 'Normal';
     * let archiveItem = new ZipArchiveItem(archive, 'directoryName\fileName.txt');
     * archive.addItem(archiveItem);
     * archive.save(fileName.zip);
     * ```
     */
    export class ZipArchive {
        private files;
        private level;
        /**
         * gets compression level
         */
        /**
         * sets compression level
         */
        compressionLevel: CompressionLevel;
        /**
         * gets items count
         */
        readonly length: number;
        /**
         * constructor for creating ZipArchive instance
         */
        constructor();
        /**
         * add new item to archive
         * @param {ZipArchiveItem} item - item to be added
         * @returns {void}
         */
        addItem(item: ZipArchiveItem): void;
        /**
         * add new directory to archive
         * @param directoryName directoryName to be created
         * @returns {void}
         */
        addDirectory(directoryName: string): void;
        /**
         * gets item at specified index
         * @param {number} index - item index
         * @returns {ZipArchiveItem}
         */
        getItem(index: number): ZipArchiveItem;
        /**
         * determines whether an element is in the collection
         * @param {string | ZipArchiveItem} item - item to search
         * @returns {boolean}
         */
        contains(item: string | ZipArchiveItem): boolean;
        /**
         * save archive with specified file name
         * @param {string} fileName save archive with specified file name
         * @returns {Promise<ZipArchive>}
         */
        save(fileName: string): Promise<ZipArchive>;
        /**
         * Save archive as blob
         * @return {Promise<Blob>}
         */
        saveAsBlob(): Promise<Blob>;
        private saveInternal(fileName, skipFileSave);
        /**
         * release allocated un-managed resource
         * @returns {void}
         */
        destroy(): void;
        private getCompressedData(item);
        private compressData(input, data, crc32Table);
        private constructZippedObject(zipParts, data, dirLength, isDirectory);
        private writeHeader(data, date);
        private writeZippedContent(fileName, zipData, localDirLen, skipFileSave);
        private writeCentralDirectory(data, localHeader, offset, externalFileAttribute);
        private writeFooter(zipData, centralLength, localLength);
        private getArrayBuffer(input);
        private getBytes(value, offset);
        private getModifiedTime(date);
        private getModifiedDate(date);
        private calculateCrc32Value(crc32Value, input, crc32Table);
    }
    /**
     * Class represent unique ZipArchive item
     * ```typescript
     * let archiveItem = new ZipArchiveItem(archive, 'directoryName\fileName.txt');
     * ```
     */
    export class ZipArchiveItem {
        data: Blob | ArrayBuffer;
        private fileName;
        /**
         * Get the name of archive item
         * @returns string
         */
        /**
         * Set the name of archive item
         * @param  {string} value
         */
        name: string;
        /**
         * constructor for creating {ZipArchiveItem} instance
         * @param {Blob|ArrayBuffer} data file data
         * @param {itemName} itemName absolute file path
         */
        constructor(data: Blob | ArrayBuffer, itemName: string);
        /**
         * release allocated un-managed resource
         * @returns {void}
         */
        destroy(): void;
    }
    export interface CompressedData {
        fileName: string;
        compressedData: Uint8Array[] | string;
        uncompressedDataSize: number;
        compressedSize: number;
        crc32Value: number;
        compressionType: string;
        isDirectory: boolean;
    }
    export interface ZippedObject {
        localHeader: string;
        centralDir: string;
        compressedData: CompressedData;
    }
    /**
     * Compression level.
     */
    export type CompressionLevel = 'NoCompression' | 'Normal';
    
  }
  export namespace data {
    
    /**
     * Adaptors are specific data source type aware interfaces that are used by DataManager to communicate with DataSource.
     * This is the base adaptor class that other adaptors can extend.
     * @hidden
     */
    export class Adaptor {
        /**
         * Specifies the datasource option.
         * @default null
         */
        dataSource: DataOptions;
        /**
         * It contains the datamanager operations list like group, searches, etc.,
         * @default null
         * @hidden
         */
        pvt: PvtOptions;
        /**
         * Constructor for Adaptor class
         * @param  {DataOptions} ds?
         * @hidden
         * @returns aggregates
         */
        constructor(ds?: DataOptions);
        protected options: RemoteOptions;
        /**
         * Returns the data from the query processing.
         * @param  {Object} data
         * @param  {DataOptions} ds?
         * @param  {Query} query?
         * @param  {XMLHttpRequest} xhr?
         * @returns Object
         */
        processResponse(data: Object, ds?: DataOptions, query?: ej.data.Query, xhr?: XMLHttpRequest): Object;
        /**
         * Specifies the type of adaptor.
         * @default Adaptor
         */
        type: Object;
    }
    /**
     * JsonAdaptor is used to process JSON data. It contains methods to process the given JSON data based on the queries.
     * @hidden
     */
    export class JsonAdaptor extends Adaptor {
        /**
         * Process the JSON data based on the provided queries.
         * @param  {DataManager} dataManager
         * @param  {Query} query
         * @returns Object
         */
        processQuery(dataManager: ej.data.DataManager, query: ej.data.Query): Object;
        /**
         * Performs batch update in the JSON array which add, remove and update records.
         * @param  {DataManager} dm
         * @param  {CrudOptions} changes
         * @param  {RemoteArgs} e
         */
        batchRequest(dm: ej.data.DataManager, changes: CrudOptions, e: RemoteArgs): CrudOptions;
        /**
         * Performs filter operation with the given data and where query.
         * @param  {Object[]} ds
         * @param  {{validate:Function}} e
         */
        onWhere(ds: Object[], e: {
            validate: Function;
        }): Object[];
        /**
         * Returns aggregate function based on the aggregate type.
         * @param  {Object[]} ds
         * @param  {{field:string} e
         * @param  {string}} type
         */
        onAggregates(ds: Object[], e: {
            field: string;
            type: string;
        }): Function;
        /**
         * Performs search operation based on the given query.
         * @param  {Object[]} ds
         * @param  {QueryOptions} e
         */
        onSearch(ds: Object[], e: QueryOptions): Object[];
        /**
         * Sort the data with given direction and field.
         * @param  {Object[]} ds
         * @param  {{comparer:(a:Object} e
         * @param  {Object} b
         */
        onSortBy(ds: Object[], e: {
            comparer: (a: Object, b: Object) => number;
            fieldName: string;
        }, query: ej.data.Query): Object[];
        /**
         * Group the data based on the given query.
         * @param  {Object[]} ds
         * @param  {QueryOptions} e
         * @param  {Query} query
         */
        onGroup(ds: Object[], e: QueryOptions, query: ej.data.Query): Object[];
        /**
         * Retrieves records based on the given page index and size.
         * @param  {Object[]} ds
         * @param  {{pageSize:number} e
         * @param  {number}} pageIndex
         * @param  {Query} query
         */
        onPage(ds: Object[], e: {
            pageSize: number;
            pageIndex: number;
        }, query: ej.data.Query): Object[];
        /**
         * Retrieves records based on the given start and end index from query.
         * @param  {Object[]} ds
         * @param  {{start:number} e
         * @param  {number}} end
         */
        onRange(ds: Object[], e: {
            start: number;
            end: number;
        }): Object[];
        /**
         * Picks the given count of records from the top of the datasource.
         * @param  {Object[]} ds
         * @param  {{nos:number}} e
         */
        onTake(ds: Object[], e: {
            nos: number;
        }): Object[];
        /**
         * Skips the given count of records from the data source.
         * @param  {Object[]} ds
         * @param  {{nos:number}} e
         */
        onSkip(ds: Object[], e: {
            nos: number;
        }): Object[];
        /**
         * Selects specified columns from the data source.
         * @param  {Object[]} ds
         * @param  {{fieldNames:string}} e
         */
        onSelect(ds: Object[], e: {
            fieldNames: string[] | Function;
        }): Object[];
        /**
         * Inserts new record in the table.
         * @param  {DataManager} dm
         * @param  {Object} data
         * @param  {number} position
         */
        insert(dm: ej.data.DataManager, data: Object, tableName?: string, query?: ej.data.Query, position?: number): Object;
        /**
         * Remove the data from the dataSource based on the key field value.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {Object} value
         * @param  {string} tableName?
         * @returns null
         */
        remove(dm: ej.data.DataManager, keyField: string, value: Object, tableName?: string): Object[];
        /**
         * Updates existing record and saves the changes to the table.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {Object} value
         * @param  {string} tableName?
         * @returns null
         */
        update(dm: ej.data.DataManager, keyField: string, value: Object, tableName?: string): void;
    }
    /**
     * URL Adaptor of DataManager can be used when you are required to use remote service to retrieve data.
     * It interacts with server-side for all DataManager Queries and CRUD operations.
     * @hidden
     */
    export class UrlAdaptor extends Adaptor {
        /**
         * Process the query to generate request body.
         * @param  {DataManager} dm
         * @param  {Query} query
         * @param  {Object[]} hierarchyFilters?
         * @returns p
         */
        processQuery(dm: ej.data.DataManager, query: ej.data.Query, hierarchyFilters?: Object[]): Object;
        private getRequestQuery(options, query, singles, request, request1);
        /**
         * Convert the object from processQuery to string which can be added query string.
         * @param  {Object} req
         * @param  {Query} query
         * @param  {DataManager} dm
         */
        convertToQueryString(request: Object, query: ej.data.Query, dm: ej.data.DataManager): string;
        /**
         * Return the data from the data manager processing.
         * @param  {DataResult} data
         * @param  {DataOptions} ds?
         * @param  {Query} query?
         * @param  {XMLHttpRequest} xhr?
         * @param  {Object} request?
         * @param  {CrudOptions} changes?
         */
        processResponse(data: DataResult, ds?: DataOptions, query?: ej.data.Query, xhr?: XMLHttpRequest, request?: Object, changes?: CrudOptions): DataResult;
        /**
         * Add the group query to the adaptor`s option.
         * @param  {Object[]} e
         * @returns void
         */
        onGroup(e: QueryOptions[]): QueryOptions[];
        /**
         * Add the aggregate query to the adaptor`s option.
         * @param  {Aggregates[]} e
         * @returns void
         */
        onAggregates(e: Aggregates[]): void;
        /**
         * Prepare the request body based on the newly added, removed and updated records.
         * The result is used by the batch request.
         * @param  {DataManager} dm
         * @param  {CrudOptions} changes
         * @param  {Object} e
         */
        batchRequest(dm: ej.data.DataManager, changes: CrudOptions, e: Object): Object;
        /**
         * Method will trigger before send the request to server side.
         * Used to set the custom header or modify the request options.
         * @param  {DataManager} dm
         * @param  {XMLHttpRequest} request
         * @returns void
         */
        beforeSend(dm: ej.data.DataManager, request: XMLHttpRequest): void;
        /**
         * Prepare and returns request body which is used to insert a new record in the table.
         * @param  {DataManager} dm
         * @param  {Object} data
         * @param  {string} tableName
         */
        insert(dm: ej.data.DataManager, data: Object, tableName: string): Object;
        /**
         * Prepare and return request body which is used to remove record from the table.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {number|string} value
         * @param  {string} tableName
         */
        remove(dm: ej.data.DataManager, keyField: string, value: number | string, tableName: string): Object;
        /**
         * Prepare and return request body which is used to update record.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {Object} value
         * @param  {string} tableName
         */
        update(dm: ej.data.DataManager, keyField: string, value: Object, tableName: string): Object;
        /**
         * To generate the predicate based on the filtered query.
         * @param  {Object[]|string[]|number[]} data
         * @param  {Query} query
         * @hidden
         */
        getFiltersFrom(data: Object[] | string[] | number[], query: ej.data.Query): Predicate;
        protected getAggregateResult(pvt: PvtOptions, data: DataResult, args: DataResult, groupDs?: Object[]): DataResult;
        protected getQueryRequest(query: ej.data.Query): Requests;
        private addParams(options);
    }
    /**
     * OData Adaptor that is extended from URL Adaptor, is used for consuming data through OData Service.
     * @hidden
     */
    export class ODataAdaptor extends UrlAdaptor {
        private getModuleName;
        protected options: RemoteOptions;
        constructor();
        /**
         * Generate request string based on the filter criteria from query.
         * @param  {Predicate} pred
         * @param  {boolean} requiresCast?
         */
        onPredicate(predicate: Predicate, query: ej.data.Query | boolean, requiresCast?: boolean): string;
        /**
         * Generate request string based on the multiple filter criteria from query.
         * @param  {Predicate} pred
         * @param  {boolean} requiresCast?
         */
        onComplexPredicate(predicate: Predicate, query: ej.data.Query, requiresCast?: boolean): string;
        /**
         * Generate query string based on the multiple filter criteria from query.
         * @param  {Predicate} filter
         * @param  {boolean} requiresCast?
         */
        onEachWhere(filter: Predicate, query: ej.data.Query, requiresCast?: boolean): string;
        /**
         * Generate query string based on the multiple filter criteria from query.
         * @param  {string[]} filters
         */
        onWhere(filters: string[]): string;
        /**
         * Generate query string based on the multiple search criteria from query.
         * @param  {{fields:string[]} e
         * @param  {string} operator
         * @param  {string} key
         * @param  {boolean}} ignoreCase
         */
        onEachSearch(e: {
            fields: string[];
            operator: string;
            key: string;
            ignoreCase: boolean;
        }): void;
        /**
         * Generate query string based on the search criteria from query.
         * @param  {Object} e
         */
        onSearch(e: Object): string;
        /**
         * Generate query string based on multiple sort criteria from query.
         * @param  {QueryOptions} e
         */
        onEachSort(e: QueryOptions): string;
        /**
         * Returns sort query string.
         * @param  {string[]} e
         */
        onSortBy(e: string[]): string;
        /**
         * Adds the group query to the adaptor option.
         * @param  {Object[]} e
         * @returns string
         */
        onGroup(e: QueryOptions[]): QueryOptions[];
        /**
         * Returns the select query string.
         * @param  {string[]} e
         */
        onSelect(e: string[]): string;
        /**
         * Add the aggregate query to the adaptor option.
         * @param  {Object[]} e
         * @returns string
         */
        onAggregates(e: Object[]): string;
        /**
         * Returns the query string which requests total count from the data source.
         * @param  {boolean} e
         * @returns string
         */
        onCount(e: boolean): string;
        /**
         * Method will trigger before send the request to server side.
         * Used to set the custom header or modify the request options.
         * @param  {DataManager} dm
         * @param  {XMLHttpRequest} request
         * @param  {Ajax} settings?
         */
        beforeSend(dm: ej.data.DataManager, request: XMLHttpRequest, settings?: ej.base.Ajax): void;
        /**
         * Returns the data from the query processing.
         * @param  {DataResult} data
         * @param  {DataOptions} ds?
         * @param  {Query} query?
         * @param  {XMLHttpRequest} xhr?
         * @param  {Ajax} request?
         * @param  {CrudOptions} changes?
         * @returns aggregateResult
         */
        processResponse(data: DataResult, ds?: DataOptions, query?: ej.data.Query, xhr?: XMLHttpRequest, request?: ej.base.Ajax, changes?: CrudOptions): Object;
        /**
         * Converts the request object to query string.
         * @param  {Object} req
         * @param  {Query} query
         * @param  {DataManager} dm
         * @returns tableName
         */
        convertToQueryString(request: Object, query: ej.data.Query, dm: ej.data.DataManager): string;
        /**
         * Prepare and returns request body which is used to insert a new record in the table.
         * @param  {DataManager} dm
         * @param  {Object} data
         * @param  {string} tableName?
         */
        insert(dm: ej.data.DataManager, data: Object, tableName?: string): Object;
        /**
         * Prepare and return request body which is used to remove record from the table.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {number} value
         * @param  {string} tableName?
         */
        remove(dm: ej.data.DataManager, keyField: string, value: number, tableName?: string): Object;
        /**
         * Updates existing record and saves the changes to the table.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {Object} value
         * @param  {string} tableName?
         * @returns this
         */
        update(dm: ej.data.DataManager, keyField: string, value: Object, tableName?: string): Object;
        /**
         * Prepare the request body based on the newly added, removed and updated records.
         * The result is used by the batch request.
         * @param  {DataManager} dm
         * @param  {CrudOptions} changes
         * @param  {RemoteArgs} e
         * @returns {Object}
         */
        batchRequest(dm: ej.data.DataManager, changes: CrudOptions, e: RemoteArgs): Object;
        /**
         * Generate the string content from the removed records.
         * The result will be send during batch update.
         * @param  {Object[]} arr
         * @param  {RemoteArgs} e
         * @returns this
         */
        generateDeleteRequest(arr: Object[], e: RemoteArgs): string;
        /**
         * Generate the string content from the inserted records.
         * The result will be send during batch update.
         * @param  {Object[]} arr
         * @param  {RemoteArgs} e
         */
        generateInsertRequest(arr: Object[], e: RemoteArgs): string;
        /**
         * Generate the string content from the updated records.
         * The result will be send during batch update.
         * @param  {Object[]} arr
         * @param  {RemoteArgs} e
         */
        generateUpdateRequest(arr: Object[], e: RemoteArgs): string;
        private static getField(prop);
        private generateBodyContent(arr, e, stat);
        protected processBatchResponse(data: DataResult, query?: ej.data.Query, xhr?: XMLHttpRequest, request?: ej.base.Ajax, changes?: CrudOptions): CrudOptions | DataResult;
    }
    /**
     * The OData v4 is an improved version of OData protocols.
     * The DataManager uses the ODataV4Adaptor to consume OData v4 services.
     * @hidden
     */
    export class ODataV4Adaptor extends ODataAdaptor {
        /**
         * @hidden
         */
        protected getModulename(): string;
        protected options: RemoteOptions;
        /**
         * Returns the query string which requests total count from the data source.
         * @param  {boolean} e
         * @returns string
         */
        onCount(e: boolean): string;
        /**
         * Generate request string based on the filter criteria from query.
         * @param  {Predicate} pred
         * @param  {boolean} requiresCast?
         */
        onPredicate(predicate: Predicate, query: ej.data.Query | boolean, requiresCast?: boolean): string;
        /**
         *  Generate query string based on the multiple search criteria from query.
         * @param  {{fields:string[]} e
         * @param  {string} operator
         * @param  {string} key
         * @param  {boolean}} ignoreCase
         */
        onEachSearch(e: {
            fields: string[];
            operator: string;
            key: string;
            ignoreCase: boolean;
        }): void;
        /**
         *  Generate query string based on the search criteria from query.
         * @param  {Object} e
         */
        onSearch(e: Object): string;
        /**
         * Method will trigger before send the request to server side.
         * Used to set the custom header or modify the request options.
         * @param  {DataManager} dm
         * @param  {XMLHttpRequest} request
         * @param  {Ajax} settings
         * @returns void
         */
        beforeSend(dm: ej.data.DataManager, request: XMLHttpRequest, settings: ej.base.Ajax): void;
        /**
         * Returns the data from the query processing.
         * @param  {DataResult} data
         * @param  {DataOptions} ds?
         * @param  {Query} query?
         * @param  {XMLHttpRequest} xhr?
         * @param  {Ajax} request?
         * @param  {CrudOptions} changes?
         * @returns aggregateResult
         */
        processResponse(data: DataResult, ds?: DataOptions, query?: ej.data.Query, xhr?: XMLHttpRequest, request?: ej.base.Ajax, changes?: CrudOptions): Object;
    }
    /**
     * The Web API is a programmatic interface to define the request and response messages system that is mostly exposed in JSON or XML.
     * The DataManager uses the WebApiAdaptor to consume Web API.
     * Since this adaptor is targeted to interact with Web API created using OData endpoint, it is extended from ODataAdaptor
     * @hidden
     */
    export class WebApiAdaptor extends ODataAdaptor {
        /**
         * Prepare and returns request body which is used to insert a new record in the table.
         * @param  {DataManager} dm
         * @param  {Object} data
         * @param  {string} tableName?
         */
        insert(dm: ej.data.DataManager, data: Object, tableName?: string): Object;
        /**
         * Prepare and return request body which is used to remove record from the table.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {number} value
         * @param  {string} tableName?
         */
        remove(dm: ej.data.DataManager, keyField: string, value: number, tableName?: string): Object;
        /**
         * Prepare and return request body which is used to update record.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {Object} value
         * @param  {string} tableName?
         */
        update(dm: ej.data.DataManager, keyField: string, value: Object, tableName?: string): Object;
        /**
         * Method will trigger before send the request to server side.
         * Used to set the custom header or modify the request options.
         * @param  {DataManager} dm
         * @param  {XMLHttpRequest} request
         * @param  {Ajax} settings
         * @returns void
         */
        beforeSend(dm: ej.data.DataManager, request: XMLHttpRequest, settings: ej.base.Ajax): void;
        /**
         * Returns the data from the query processing.
         * @param  {DataResult} data
         * @param  {DataOptions} ds?
         * @param  {Query} query?
         * @param  {XMLHttpRequest} xhr?
         * @param  {Ajax} request?
         * @param  {CrudOptions} changes?
         * @returns aggregateResult
         */
        processResponse(data: DataResult, ds?: DataOptions, query?: ej.data.Query, xhr?: XMLHttpRequest, request?: ej.base.Ajax, changes?: CrudOptions): Object;
    }
    /**
     * WebMethodAdaptor can be used by DataManager to interact with web method.
     * @hidden
     */
    export class WebMethodAdaptor extends UrlAdaptor {
        /**
         * Prepare the request body based on the query.
         * The query information can be accessed at the WebMethod using variable named `value`.
         * @param  {DataManager} dm
         * @param  {Query} query
         * @param  {Object[]} hierarchyFilters?
         * @returns application
         */
        processQuery(dm: ej.data.DataManager, query: ej.data.Query, hierarchyFilters?: Object[]): Object;
    }
    /**
     * RemoteSaveAdaptor, extended from JsonAdaptor and it is used for binding local data and performs all DataManager queries in client-side.
     * It interacts with server-side only for CRUD operations.
     * @hidden
     */
    export class RemoteSaveAdaptor extends JsonAdaptor {
        /**
         * @hidden
         */
        constructor();
        /**
         * Prepare the request body based on the newly added, removed and updated records.
         * Also perform the changes in the locally cached data to sync with the remote data.
         * The result is used by the batch request.
         * @param  {DataManager} dm
         * @param  {CrudOptions} changes
         * @param  {RemoteArgs} e
         */
        batchRequest(dm: ej.data.DataManager, changes: CrudOptions, e: RemoteArgs): Object;
    }
    /**
     * Cache Adaptor is used to cache the data of the visited pages. It prevents new requests for the previously visited pages.
     * You can configure cache page size and duration of caching by using cachingPageSize and timeTillExpiration properties of the DataManager
     * @hidden
     */
    export class CacheAdaptor extends UrlAdaptor {
        private cacheAdaptor;
        private pageSize;
        private guidId;
        private isCrudAction;
        private isInsertAction;
        /**
         * Constructor for CacheAdaptor class.
         * @param  {CacheAdaptor} adaptor?
         * @param  {number} timeStamp?
         * @param  {number} pageSize?
         * @hidden
         */
        constructor(adaptor?: CacheAdaptor, timeStamp?: number, pageSize?: number);
        /**
         * It will generate the key based on the URL when we send a request to server.
         * @param  {string} url
         * @param  {Query} query?
         * @hidden
         */
        generateKey(url: string, query: ej.data.Query): string;
        /**
         * Process the query to generate request body.
         * If the data is already cached, it will return the cached data.
         * @param  {DataManager} dm
         * @param  {Query} query?
         * @param  {Object[]} hierarchyFilters?
         */
        processQuery(dm: ej.data.DataManager, query?: ej.data.Query, hierarchyFilters?: Object[]): Object;
        /**
         * Returns the data from the query processing.
         * It will also cache the data for later usage.
         * @param  {DataResult} data
         * @param  {DataManager} ds?
         * @param  {Query} query?
         * @param  {XMLHttpRequest} xhr?
         * @param  {Ajax} request?
         * @param  {CrudOptions} changes?
         */
        processResponse(data: DataResult, ds?: ej.data.DataManager, query?: ej.data.Query, xhr?: XMLHttpRequest, request?: ej.base.Ajax, changes?: CrudOptions): DataResult;
        /**
         * Method will trigger before send the request to server side. Used to set the custom header or modify the request options.
         * @param  {DataManager} dm
         * @param  {XMLHttpRequest} request
         * @param  {Ajax} settings?
         */
        beforeSend(dm: ej.data.DataManager, request: XMLHttpRequest, settings?: ej.base.Ajax): void;
        /**
         * Updates existing record and saves the changes to the table.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {Object} value
         * @param  {string} tableName
         */
        update(dm: ej.data.DataManager, keyField: string, value: Object, tableName: string): Object;
        /**
         * Prepare and returns request body which is used to insert a new record in the table.
         * @param  {DataManager} dm
         * @param  {Object} data
         * @param  {string} tableName?
         */
        insert(dm: ej.data.DataManager, data: Object, tableName?: string): Object;
        /**
         * Prepare and return request body which is used to remove record from the table.
         * @param  {DataManager} dm
         * @param  {string} keyField
         * @param  {Object} value
         * @param  {string} tableName?
         */
        remove(dm: ej.data.DataManager, keyField: string, value: Object, tableName?: string): Object[];
        /**
         * Prepare the request body based on the newly added, removed and updated records.
         * The result is used by the batch request.
         * @param  {DataManager} dm
         * @param  {CrudOptions} changes
         * @param  {RemoteArgs} e
         */
        batchRequest(dm: ej.data.DataManager, changes: CrudOptions, e: RemoteArgs): CrudOptions;
    }
    /**
     * @hidden
     */
    export interface CrudOptions {
        changedRecords?: Object[];
        addedRecords?: Object[];
        deletedRecords?: Object[];
        action?: string;
        table?: string;
        key?: string;
    }
    /**
     * @hidden
     */
    export interface PvtOptions {
        groups?: QueryOptions[];
        aggregates?: Aggregates[];
        search?: Object | Predicate;
        changeSet?: number;
        searches?: Object[];
    }
    /**
     * @hidden
     */
    export interface DataResult {
        nodeType?: number;
        addedRecords?: Object[];
        d?: DataResult | Object[];
        Count?: number;
        count?: number;
        result?: Object;
        results?: Object[] | DataResult;
        aggregate?: DataResult;
        aggregates?: Aggregates;
        value?: Object;
        Items?: Object[] | DataResult;
        keys?: string[];
        groupDs?: Object[];
    }
    /**
     * @hidden
     */
    export interface Requests {
        sorts: QueryOptions[];
        groups: QueryOptions[];
        filters: QueryOptions[];
        searches: QueryOptions[];
        aggregates: QueryOptions[];
    }
    /**
     * @hidden
     */
    export interface RemoteArgs {
        guid?: string;
        url?: string;
        key?: string;
        cid?: number;
        cSet?: string;
    }
    /**
     * @hidden
     */
    export interface RemoteOptions {
        from?: string;
        requestType?: string;
        sortBy?: string;
        select?: string;
        skip?: string;
        group?: string;
        take?: string;
        search?: string;
        count?: string;
        where?: string;
        aggregates?: string;
        expand?: string;
        accept?: string;
        multipartAccept?: string;
        batch?: string;
        changeSet?: string;
        batchPre?: string;
        contentId?: string;
        batchContent?: string;
        changeSetContent?: string;
        batchChangeSetContentType?: string;
    }
    
    /**
     * DataManager is used to manage and manipulate relational data.
     */
    export class DataManager {
        /** @hidden */
        adaptor: AdaptorOptions;
        /** @hidden */
        defaultQuery: ej.data.Query;
        /** @hidden */
        dataSource: DataOptions;
        /** @hidden */
        dateParse: boolean;
        /** @hidden */
        ready: Promise<ej.base.Ajax>;
        private isDataAvailable;
        private requests;
        /**
         * Constructor for DataManager class
         * @param  {DataOptions|JSON[]} dataSource?
         * @param  {Query} query?
         * @param  {AdaptorOptions|string} adaptor?
         * @hidden
         */
        constructor(dataSource?: DataOptions | JSON[] | Object[], query?: ej.data.Query, adaptor?: AdaptorOptions | string);
        /**
         * Overrides DataManager's default query with given query.
         * @param  {Query} query - Defines the new default query.
         */
        setDefaultQuery(query: ej.data.Query): ej.data.DataManager;
        /**
         * Executes the given query with local data source.
         * @param  {Query} query - Defines the query to retrieve data.
         */
        executeLocal(query?: ej.data.Query): Object[];
        /**
         * Executes the given query with either local or remote data source.
         * It will be executed as asynchronously and returns Promise object which will be resolved or rejected after action completed.
         * @param  {Query|Function} query - Defines the query to retrieve data.
         * @param  {Function} done - Defines the callback function and triggers when the Promise is resolved.
         * @param  {Function} fail - Defines the callback function and triggers when the Promise is rejected.
         * @param  {Function} always - Defines the callback function and triggers when the Promise is resolved or rejected.
         */
        executeQuery(query: ej.data.Query | Function, done?: Function, fail?: Function, always?: Function): Promise<ej.base.Ajax>;
        private static getDeferedArgs(query, result, args?);
        private static nextTick(fn);
        private extendRequest(url, fnSuccess, fnFail);
        private makeRequest(url, deffered, args?, query?);
        private beforeSend(request, settings?);
        /**
         * Save bulk changes to the given table name.
         * User can add a new record, edit an existing record, and delete a record at the same time.
         * If the datasource from remote, then updated in a single post.
         * @param  {Object} changes - Defines the CrudOptions.
         * @param  {string} key - Defines the column field.
         * @param  {string|Query} tableName - Defines the table name.
         * @param  {Query} query - Sets default query for the DataManager.
         */
        saveChanges(changes: Object, key?: string, tableName?: string | Query, query?: ej.data.Query): Promise<Object> | Object;
        /**
         * Inserts new record in the given table.
         * @param  {Object} data - Defines the data to insert.
         * @param  {string|Query} tableName - Defines the table name.
         * @param  {Query} query - Sets default query for the DataManager.
         */
        insert(data: Object, tableName?: string | Query, query?: ej.data.Query, position?: number): Object | Promise<Object>;
        /**
         * Removes data from the table with the given key.
         * @param  {string} keyField - Defines the column field.
         * @param  {Object} value - Defines the value to find the data in the specified column.
         * @param  {string|Query} tableName - Defines the table name
         * @param  {Query} query - Sets default query for the DataManager.
         */
        remove(keyField: string, value: Object, tableName?: string | Query, query?: ej.data.Query): Object | Promise<Object>;
        /**
         * Updates existing record in the given table.
         * @param  {string} keyField - Defines the column field.
         * @param  {Object} value - Defines the value to find the data in the specified column.
         * @param  {string|Query} tableName - Defines the table name
         * @param  {Query} query - Sets default query for the DataManager.
         */
        update(keyField: string, value: Object, tableName?: string | Query, query?: ej.data.Query): Object | Promise<Object>;
        private doAjaxRequest(res);
    }
    /**
     * Deferred is used to handle asynchronous operation.
     */
    export class Deferred {
        /**
         * Resolve a Deferred object and call doneCallbacks with the given args.
         */
        resolve: Function;
        /**
         * Reject a Deferred object and call failCallbacks with the given args.
         */
        reject: Function;
        /**
         * Promise is an object that represents a value that may not be available yet, but will be resolved at some point in the future.
         */
        promise: Promise<Object>;
        /**
         * Defines the callback function triggers when the Deferred object is resolved.
         */
        then: Function;
        /**
         * Defines the callback function triggers when the Deferred object is rejected.
         */
        catch: Function;
    }
    /**
     * @hidden
     */
    export interface DataOptions {
        url?: string;
        adaptor?: AdaptorOptions;
        insertUrl?: string;
        removeUrl?: string;
        updateUrl?: string;
        crudUrl?: string;
        batchUrl?: string;
        json?: Object[];
        headers?: Object[];
        accept?: boolean;
        data?: JSON;
        timeTillExpiration?: number;
        cachingPageSize?: number;
        enableCaching?: boolean;
        requestType?: string;
        key?: string;
        crossDomain?: boolean;
        jsonp?: string;
        dataType?: string;
        offline?: boolean;
        requiresFormat?: boolean;
    }
    /**
     * @hidden
     */
    export interface ReturnOption {
        result?: ReturnOption;
        count?: number;
        url?: string;
        aggregates?: Aggregates;
    }
    /**
     * @hidden
     */
    export interface RequestOptions {
        xhr?: XMLHttpRequest;
        count?: number;
        result?: ReturnOption;
        request?: ej.base.Ajax;
        aggregates?: Aggregates;
        actual?: Object;
        virtualSelectRecords?: Object;
        error?: string;
    }
    /**
     * @hidden
     */
    export interface AdaptorOptions {
        processQuery?: Function;
        processResponse?: Function;
        beforeSend?: Function;
        batchRequest?: Function;
        insert?: Function;
        remove?: Function;
        update?: Function;
    }
    
    /**
     * Query class is used to build query which is used by the DataManager to communicate with datasource.
     */
    export class Query {
        /** @hidden */
        queries: QueryOptions[];
        /** @hidden */
        key: string;
        /** @hidden */
        fKey: string;
        /** @hidden */
        fromTable: string;
        /** @hidden */
        lookups: string[];
        /** @hidden */
        expands: Object[];
        /** @hidden */
        sortedColumns: Object[];
        /** @hidden */
        groupedColumns: Object[];
        /** @hidden */
        subQuerySelector: Function;
        /** @hidden */
        subQuery: ej.data.Query;
        /** @hidden */
        isChild: boolean;
        /** @hidden */
        params: ParamOption[];
        /** @hidden */
        isCountRequired: boolean;
        /** @hidden */
        dataManager: ej.data.DataManager;
        /**
         * Constructor for Query class.
         * @param  {string|string[]} from?
         * @hidden
         */
        constructor(from?: string | string[]);
        /**
         * Sets the primary key.
         * @param  {string} field - Defines the column field.
         */
        setKey(field: string): ej.data.Query;
        /**
         * Sets default DataManager to execute query.
         * @param  {DataManager} dataManager - Defines the DataManager.
         */
        using(dataManager: ej.data.DataManager): ej.data.Query;
        /**
         * Executes query with the given DataManager.
         * @param  {DataManager} dataManager - Defines the DataManager.
         * @param  {Function} done - Defines the success callback.
         * @param  {Function} fail - Defines the failure callback.
         * @param  {Function} always - Defines the callback which will be invoked on either success or failure.
         *
         * <pre>
         * let dataManager: ej.data.DataManager = new DataManager([{ ID: '10' }, { ID: '2' }, { ID: '1' }, { ID: '20' }]);
         * let query: ej.data.Query = new Query();
         * query.sortBy('ID', (x: string, y: string): number => { return parseInt(x, 10) - parseInt(y, 10) });
         * let promise: Promise< Object > = query.execute(dataManager);
         * promise.then((e: { result: Object }) => { });
         * </pre>
         */
        execute(dataManager?: ej.data.DataManager, done?: Function, fail?: Function, always?: Function): Promise<Object>;
        /**
         * Executes query with the local datasource.
         * @param  {DataManager} dataManager - Defines the DataManager.
         */
        executeLocal(dataManager?: ej.data.DataManager): Object[];
        /**
         * Creates deep copy of the Query object.
         */
        clone(): ej.data.Query;
        /**
         * Specifies the name of table to retrieve data in query execution.
         * @param  {string} tableName - Defines the table name.
         */
        from(tableName: string): ej.data.Query;
        /**
         * Adds additional parameter which will be sent along with the request which will be generated while DataManager execute.
         * @param  {string} key - Defines the key of additional parameter.
         * @param  {Function|string} value - Defines the value for the key.
         */
        addParams(key: string, value: Function | string): ej.data.Query;
        /**
         * Expands the related table.
         * @param  {string|Object[]} tables
         */
        expand(tables: string | Object[]): ej.data.Query;
        /**
         * Filter data with given filter criteria.
         * @param  {string|Predicate} fieldName - Defines the column field or Predicate.
         * @param  {string} operator - Defines the operator how to filter data.
         * @param  {string|number|boolean} value - Defines the values to match with data.
         * @param  {boolean} ignoreCase - If ignore case set to false, then filter data with exact match or else
         * filter data with case insensitive.
         */
        where(fieldName: string | Predicate | Predicate[], operator?: string, value?: string | Date | number | boolean, ignoreCase?: boolean, ignoreAccent?: boolean): ej.data.Query;
        /**
         * Search data with given search criteria.
         * @param  {string|number|boolean} searchKey - Defines the search key.
         * @param  {string|string[]} fieldNames - Defines the collection of column fields.
         * @param  {string} operator - Defines the operator how to search data.
         * @param  {boolean} ignoreCase - If ignore case set to false, then filter data with exact match or else
         * filter data with case insensitive.
         */
        search(searchKey: string | number | boolean, fieldNames?: string | string[], operator?: string, ignoreCase?: boolean, ignoreAccent?: boolean): ej.data.Query;
        /**
         * Sort the data with given sort criteria.
         * By default, sort direction is ascending.
         * @param  {string|string[]} fieldName - Defines the single or collection of column fields.
         * @param  {string|Function} comparer - Defines the sort direction or custom sort comparer function.
         */
        sortBy(fieldName: string | string[], comparer?: string | Function, isFromGroup?: boolean): ej.data.Query;
        /**
         * Sorts data in descending order.
         * @param  {string} fieldName - Defines the column field.
         */
        sortByDesc(fieldName: string): ej.data.Query;
        /**
         * Groups data with the given field name.
         * @param  {string} fieldName - Defines the column field.
         */
        group(fieldName: string, fn?: Function, format?: string | ej.base.NumberFormatOptions | ej.base.DateFormatOptions): ej.data.Query;
        /**
         * Gets data based on the given page index and size.
         * @param  {number} pageIndex - Defines the current page index.
         * @param  {number} pageSize - Defines the no of records per page.
         */
        page(pageIndex: number, pageSize: number): ej.data.Query;
        /**
         * Gets data based on the given start and end index.
         * @param  {number} start - Defines the start index of the datasource.
         * @param  {number} end - Defines the end index of the datasource.
         */
        range(start: number, end: number): ej.data.Query;
        /**
         * Gets data from the top of the data source based on given number of records count.
         * @param  {number} nos - Defines the no of records to retrieve from datasource.
         */
        take(nos: number): ej.data.Query;
        /**
         * Skips data with given number of records count from the top of the data source.
         * @param  {number} nos - Defines the no of records skip in the datasource.
         */
        skip(nos: number): ej.data.Query;
        /**
         * Selects specified columns from the data source.
         * @param  {string|string[]} fieldNames - Defines the collection of column fields.
         */
        select(fieldNames: string | string[]): ej.data.Query;
        /**
         * Gets the records in hierarchical relationship from two tables. It requires the foreign key to relate two tables.
         * @param  {Query} query - Defines the query to relate two tables.
         * @param  {Function} selectorFn - Defines the custom function to select records.
         */
        hierarchy(query: ej.data.Query, selectorFn: Function): ej.data.Query;
        /**
         * Sets the foreign key which is used to get data from the related table.
         * @param  {string} key - Defines the foreign key.
         */
        foreignKey(key: string): ej.data.Query;
        /**
         * It is used to get total number of records in the DataManager execution result.
         */
        requiresCount(): ej.data.Query;
        /**
         * Aggregate the data with given type and field name.
         * @param  {string} type - Defines the aggregate type.
         * @param  {string} field - Defines the column field to aggregate.
         */
        aggregate(type: string, field: string): ej.data.Query;
        /**
         * Pass array of filterColumn query for performing filter operation.
         * @param  {QueryOptions[]} queries
         * @param  {string} name
         * @hidden
         */
        static filterQueries(queries: QueryOptions[], name: string): QueryOptions[];
        /**
         * To get the list of queries which is already filtered in current data source.
         * @param  {Object[]} queries
         * @param  {string[]} singles
         * @hidden
         */
        static filterQueryLists(queries: Object[], singles: string[]): Object;
    }
    /**
     * Predicate class is used to generate complex filter criteria.
     * This will be used by DataManager to perform multiple filtering operation.
     */
    export class Predicate {
        /** @hidden */
        field: string;
        /** @hidden */
        operator: string;
        /** @hidden */
        value: string | number | Date | boolean | Predicate | Predicate[];
        /** @hidden */
        condition: string;
        /** @hidden */
        ignoreCase: boolean;
        /** @hidden */
        ignoreAccent: boolean;
        /** @hidden */
        isComplex: boolean;
        /** @hidden */
        predicates: Predicate[];
        /** @hidden */
        comparer: Function;
        [x: string]: string | number | Date | boolean | Predicate | Predicate[] | Function;
        /**
         * Constructor for Predicate class.
         * @param  {string|Predicate} field
         * @param  {string} operator
         * @param  {string|number|boolean|Predicate|Predicate[]} value
         * @param  {boolean=false} ignoreCase
         * @hidden
         */
        constructor(field: string | Predicate, operator: string, value: string | number | Date | boolean | Predicate | Predicate[], ignoreCase?: boolean, ignoreAccent?: boolean);
        /**
         * Adds n-number of new predicates on existing predicate with “and” condition.
         * @param  {Object[]} args - Defines the collection of predicates.
         */
        static and(...args: Object[]): Predicate;
        /**
         * Adds new predicate on existing predicate with “and” condition.
         * @param  {string} field - Defines the column field.
         * @param  {string} operator - Defines the operator how to filter data.
         * @param  {string} value - Defines the values to match with data.
         * @param  {boolean} ignoreCase? - If ignore case set to false, then filter data with exact match or else
         * filter data with case insensitive.
         */
        and(field: string | Predicate, operator?: string, value?: string | number, ignoreCase?: boolean, ignoreAccent?: boolean): Predicate;
        /**
         * Adds n-number of new predicates on existing predicate with “or” condition.
         * @param  {Object[]} args - Defines the collection of predicates.
         */
        static or(...args: Object[]): Predicate;
        /**
         * Adds new predicate on existing predicate with “or” condition.
         * @param  {string} field - Defines the column field.
         * @param  {string} operator - Defines the operator how to filter data.
         * @param  {string} value - Defines the values to match with data.
         * @param  {boolean} ignoreCase? - If ignore case set to false, then filter data with exact match or else
         * filter data with case insensitive.
         */
        or(field: string | Predicate, operator?: string, value?: string | number, ignoreCase?: boolean, ignoreAccent?: boolean): Predicate;
        /**
         * Converts plain JavaScript object to Predicate object.
         * @param  {Predicate[]|Predicate} json - Defines single or collection of Predicate.
         */
        static fromJson(json: Predicate[] | Predicate): Predicate[];
        /**
         * Validate the record based on the predicates.
         * @param  {Object} record - Defines the datasource record.
         */
        validate(record: Object): boolean;
        /**
         * Converts predicates to plain JavaScript.
         * This method is uses Json stringify when serializing Predicate object.
         */
        toJson(): Object;
        private static combinePredicates(predicates, operator);
        private static combine(pred, field, operator, value, condition, ignoreCase?, ignoreAccent?);
        private static fromJSONData(json);
    }
    /**
     * @hidden
     */
    export interface QueryOptions {
        fn?: string;
        e?: QueryOptions;
        fieldNames?: string | string[];
        operator?: string;
        searchKey?: string | number | boolean;
        ignoreCase?: boolean;
        ignoreAccent?: boolean;
        comparer?: string | Function;
        format?: string | ej.base.NumberFormatOptions | ej.base.DateFormatOptions;
        direction?: string;
        pageIndex?: number;
        pageSize?: number;
        start?: number;
        end?: number;
        nos?: number;
        field?: string;
        fieldName?: string;
        type?: Object;
        name?: string | string[];
        filter?: Object;
        key?: string;
        value?: string | number | Date | boolean | Predicate | Predicate[];
        isComplex?: boolean;
        predicates?: Predicate[];
        condition?: string;
    }
    /**
     * @hidden
     */
    export interface QueryList {
        onSelect?: QueryOptions;
        onPage?: QueryOptions;
        onSkip?: QueryOptions;
        onTake?: QueryOptions;
        onRange?: QueryOptions;
    }
    /**
     * @hidden
     */
    export interface ParamOption {
        key: string;
        value?: string;
        fn?: Function;
    }
    
    /**
     * Data manager common utility methods.
     * @hidden
     */
    export class DataUtil {
        /**
         * Specifies the value which will be used to adjust the date value to server timezone.
         * @default null
         */
        static serverTimezoneOffset: number;
        /**
         * Returns the value by invoking the provided parameter function.
         * If the paramater is not of type function then it will be returned as it is.
         * @param  {Function|string|string[]|number} value
         * @param  {Object} inst?
         * @hidden
         */
        static getValue<T>(value: T | Function, inst?: Object): T;
        /**
         * Returns true if the input string ends with given string.
         * @param  {string} input
         * @param  {string} substr
         */
        static endsWith(input: string, substr: string): boolean;
        /**
         * Returns true if the input string starts with given string.
         * @param  {string} str
         * @param  {string} startstr
         */
        static startsWith(input: string, start: string): boolean;
        /**
         * To return the sorting function based on the string.
         * @param  {string} order
         * @hidden
         */
        static fnSort(order: string): Function;
        /**
         * Comparer function which is used to sort the data in ascending order.
         * @param  {string|number} x
         * @param  {string|number} y
         * @returns number
         */
        static fnAscending(x: string | number, y: string | number): number;
        /**
         * Comparer function which is used to sort the data in descending order.
         * @param  {string|number} x
         * @param  {string|number} y
         * @returns number
         */
        static fnDescending(x: string | number, y: string | number): number;
        private static extractFields(obj, fields);
        /**
         * Select objects by given fields from jsonArray.
         * @param  {Object[]} jsonArray
         * @param  {string[]} fields
         */
        static select(jsonArray: Object[], fields: string[]): Object[];
        /**
         * Group the input data based on the field name.
         * It also performs aggregation of the grouped records based on the aggregates paramater.
         * @param  {Object[]} jsonArray
         * @param  {string} field?
         * @param  {Object[]} agg?
         * @param  {number} level?
         * @param  {Object[]} groupDs?
         */
        static group(jsonArray: Object[], field?: string, aggregates?: Object[], level?: number, groupDs?: Object[], format?: Function): Object[];
        /**
         * It is used to categorize the multiple items based on a specific field in jsonArray.
         * The hierarchical queries are commonly required when you use foreign key binding.
         * @param  {string} fKey
         * @param  {string} from
         * @param  {Object[]} source
         * @param  {Group} lookup?
         * @param  {string} pKey?
         * @hidden
         */
        static buildHierarchy(fKey: string, from: string, source: Group, lookup?: Group, pKey?: string): void;
        /**
         * Throw error with the given string as message.
         * @param  {string} er
         */
        static throwError: Function;
        static aggregates: Aggregates;
        /**
         * The method used to get the field names which started with specified characters.
         * @param  {Object} obj
         * @param  {string[]} fields?
         * @param  {string} prefix?
         * @hidden
         */
        static getFieldList(obj: Object, fields?: string[], prefix?: string): string[];
        /**
         * Gets the value of the property in the given object.
         * The complex object can be accessed by providing the field names concatenated with dot(.).
         * @param  {string} nameSpace - The name of the property to be accessed.
         * @param  {Object} from - Defines the source object.
         */
        static getObject(nameSpace: string, from: Object): Object;
        /**
         * To set value for the nameSpace in desired object.
         * @param {string} nameSpace - String value to the get the inner object.
         * @param {Object} value - Value that you need to set.
         * @param {Object} obj - Object to get the inner object value.
         * @return { [key: string]: Object; } | Object
         * @hidden
         */
        static setValue(nameSpace: string, value: Object, obj: Object): {
            [key: string]: Object;
        } | Object;
        /**
         * Sort the given data based on the field and comparer.
         * @param  {Object[]} ds - Defines the input data.
         * @param  {string} field - Defines the field to be sorted.
         * @param  {Function} comparer - Defines the comparer function used to sort the records.
         */
        static sort(ds: Object[], field: string, comparer: Function): Object[];
        static ignoreDiacritics(value: string | number | boolean): string | Object;
        private static merge(left, right, fieldName, comparer);
        private static getVal(array, index, field?);
        private static toLowerCase(val);
        /**
         * Specifies the Object with filter operators.
         */
        static operatorSymbols: {
            [key: string]: string;
        };
        /**
         * Specifies the Object with filter operators which will be used for OData filter query generation.
         * * It will be used for date/number type filter query.
         */
        static odBiOperator: {
            [key: string]: string;
        };
        /**
         * Specifies the Object with filter operators which will be used for OData filter query generation.
         * It will be used for string type filter query.
         */
        static odUniOperator: {
            [key: string]: string;
        };
        /**
         * Specifies the Object with filter operators which will be used for ODataV4 filter query generation.
         * It will be used for string type filter query.
         */
        static odv4UniOperator: {
            [key: string]: string;
        };
        static diacritics: {
            [key: string]: string;
        };
        static fnOperators: Operators;
        /**
         * To perform the filter operation with specified adaptor and returns the result.
         * @param  {Object} adaptor
         * @param  {string} fnName
         * @param  {Object} param1?
         * @param  {Object} param2?
         * @hidden
         */
        static callAdaptorFunction(adaptor: Object, fnName: string, param1?: Object, param2?: Object): Object;
        /**
         * To perform the parse operation on JSON data, like convert to string from JSON or convert to JSON from string.
         */
        static parse: ParseOption;
        /**
         * Checks wheather the given input is a plain object or not.
         * @param  {Object|Object[]} obj
         */
        static isPlainObject(obj: Object | Object[]): boolean;
        /**
         * Returns true when the browser cross origin request.
         */
        static isCors(): boolean;
        /**
         * Generate random GUID value which will be prefixed with the given value.
         * @param  {string} prefix
         */
        static getGuid(prefix: string): string;
        /**
         * Checks wheather the given value is null or not.
         * @param  {string|Object} val
         * @returns boolean
         */
        static isNull(val: string | Object): boolean;
        /**
         * To get the required items from collection of objects.
         * @param  {Object[]} array
         * @param  {string} field
         * @param  {Function} comparer
         * @returns Object
         * @hidden
         */
        static getItemFromComparer(array: Object[], field: string, comparer: Function): Object;
        /**
         * To get distinct values of Array or Array of Objects.
         * @param  {Object[]} json
         * @param  {string} field
         * @param  {boolean} requiresCompleteRecord
         * @returns Object[]
         * * distinct array of objects is return when requiresCompleteRecord set as true.
         * @hidden
         */
        static distinct(json: Object[], fieldName: string, requiresCompleteRecord?: boolean): Object[];
        /**
         * @hidden
         */
        static dateParse: DateParseOption;
    }
    /**
     * @hidden
     */
    export interface Aggregates {
        sum?: Function;
        average?: Function;
        min?: Function;
        max?: Function;
        truecount?: Function;
        falsecount?: Function;
        count?: Function;
        type?: string;
        field?: string;
    }
    /**
     * @hidden
     */
    export interface Operators {
        equal?: Function;
        notequal?: Function;
        lessthan?: Function;
        greaterthan?: Function;
        lessthanorequal?: Function;
        greaterthanorequal?: Function;
        contains?: Function;
        notnull?: Function;
        isnull?: Function;
        startswith?: Function;
        endswith?: Function;
        processSymbols?: Function;
        processOperator?: Function;
    }
    /**
     * @hidden
     */
    export interface Group {
        GroupGuid?: string;
        level?: number;
        childLevels?: number;
        records?: Object[];
        key?: string;
        count?: number;
        items?: Object[];
        aggregates?: Object;
        field?: string;
        result?: Object;
    }
    /**
     * @hidden
     */
    export interface ParseOption {
        parseJson?: Function;
        iterateAndReviveArray?: Function;
        iterateAndReviveJson?: Function;
        jsonReviver?: (key: string, value: Object) => Object;
        isJson?: Function;
        isGuid?: Function;
        replacer?: Function;
        jsonReplacer?: Function;
        arrayReplacer?: Function;
    }
    /**
     * @hidden
     */
    export interface DateParseOption {
        addSelfOffset?: (input: Date) => Date;
        toUTC?: (input: Date) => Date;
        toTimeZone?: (input: Date, offset?: number, utc?: boolean) => Date;
    }
    
  }
  export namespace dropdowns {
    
    /**
     * Interface for a class AutoComplete
     */
    export interface AutoCompleteModel extends ComboBoxModel{
    
        /**
         * The `fields` property maps the columns of the data table and binds the data to the component.
         * * text - Maps the text column from data table for each list item
         * * value - Maps the value column from data table for each list item
         * * iconCss - Maps the icon class column from data table for each list item
         * * groupBy - Group the list items with it's related items by mapping groupBy field
         * 
         * > For more details about the field mapping refer to [`Data binding`](./data-binding.html) documentation.
         * @default { value: null, iconCss: null, groupBy: null}
         */
        fields?: FieldSettingsModel;
    
        /**
         * When set to ‘false’, consider the [`case-sensitive`](./filtering.html#case-sensitive-filtering) 
         * on performing the search to find suggestions.
         * By default consider the casing.
         * @default true.
         */
        ignoreCase?: boolean;
    
        /**
         * Allows you to either show or hide the popup button on the component.
         * @default false.
         */
        showPopupButton?: boolean;
    
        /**
         * When set to ‘true’, highlight the searched characters on suggested list items.
         * > For more details about the highlight refer to [`Custom highlight search`](./how-to.html#custom-highlight-search) documentation. 
         * @default false.
         */
        highlight?: boolean;
    
        /**
         * Supports the [`specified number`](./filtering.html#filter-item-count) 
         * of list items on the suggestion popup.
         * @default 20.
         */
        suggestionCount?: number;
    
        /**
         * Allows additional HTML attributes such as title, name, etc., and
         * accepts n number of attributes in a key-value pair format.
         * 
         * {% codeBlock src="autocomplete/html-attributes-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="autocomplete/html-attributes-api/index.html" %}{% endcodeBlock %}
         * @default {}.
         */
        htmlAttributes?: { [key: string]: string; };
    
        /**
         * Accepts the external [`query`](./api-query.html)
         * that execute along with data processing.
         * 
         * {% codeBlock src="autocomplete/query-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="autocomplete/query-api/index.html" %}{% endcodeBlock %}
         * @default null.
         */
        query?: ej.data.Query;
    
        /**
         * Allows you to set [`the minimum search character length']
         * (./filtering.html#limit-the-minimum-filter-character),
         * the search action will perform after typed minimum characters.
         * @default 1.
         */
        minLength?: number;
    
        /**
         * Determines on which filter type, the component needs to be considered on search action. 
         * The available [`FilterType`](./filtering.html#change-the-filter-type) 
         * and its supported data types are 
         * 
         * <table> 
         * <tr> 
         * <td colSpan=1 rowSpan=1> 
         * FilterType<br/></td><td colSpan=1 rowSpan=1> 
         * Description<br/></td><td colSpan=1 rowSpan=1> 
         * Supported Types<br/></td></tr> 
         * <tr> 
         * <td colSpan=1 rowSpan=1> 
         * StartsWith<br/></td><td colSpan=1 rowSpan=1> 
         * Checks whether a value begins with the specified value.<br/></td><td colSpan=1 rowSpan=1> 
         * String<br/></td></tr> 
         * <tr> 
         * <td colSpan=1 rowSpan=1> 
         * EndsWith<br/></td><td colSpan=1 rowSpan=1> 
         * Checks whether a value ends with specified value.<br/><br/></td><td colSpan=1 rowSpan=1> 
         * <br/>String<br/></td></tr> 
         * <tr> 
         * <td colSpan=1 rowSpan=1> 
         * Contains<br/></td><td colSpan=1 rowSpan=1> 
         * Checks whether a value contains with specified value.<br/><br/></td><td colSpan=1 rowSpan=1> 
         * <br/>String<br/></td></tr> 
         * </table> 
         * 
         * {% codeBlock src="autocomplete/filter-type-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="autocomplete/filter-type-api/index.html" %}{% endcodeBlock %}
         * 
         * The default value set to `Contains`, all the suggestion items which contain typed characters to listed in the suggestion popup.
         * @default 'Contains'.
         */
        filterType?: FilterType;
    
        /**
         * Triggers on typing a character in the component.
         * @event
         */
        filtering?: ej.base.EmitType<FilteringArgs>;
    
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        index?: number;
    
        /**
         * Specifies whether to display the floating label above the input element.
         * Possible values are:
         * * Never: The label will never float in the input when the placeholder is available.
         * * Always: The floating label will always float above the input.
         * * Auto: The floating label will float above the input after focusing or entering a value in the input.
         * 
         * {% codeBlock src="autocomplete/float-label-type-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="autocomplete/float-label-type-api/index.html" %}{% endcodeBlock %}
         * 
         *  @default 'Never'.
         */
        floatLabelType?: ej.inputs.FloatLabelType;
    
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        valueTemplate?: string;
    
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        filterBarPlaceholder?: string;
    
        /**
         * Not applicable to this component. 
         * @default false.
         * @private
         */
        allowFiltering?: boolean;
    
        /**
         * Not applicable to this component. 
         * @default null.
         * @private
         */
        text?: string;
    
    }
    
    export type FilterType = 'Contains' | 'StartsWith' | 'EndsWith';
    /**
     * The AutoComplete component provides the matched suggestion list when type into the input,
     * from which the user can select one.
     * ```html
     * <input id="list" type="text"/>
     * ```
     * ```typescript
     *   let atcObj:AutoComplete = new AutoComplete();
     *   atcObj.appendTo("#list");
     * ```
     */
    export class AutoComplete extends ComboBox {
        private isFiltered;
        /**
         * The `fields` property maps the columns of the data table and binds the data to the component.
         * * text - Maps the text column from data table for each list item
         * * value - Maps the value column from data table for each list item
         * * iconCss - Maps the icon class column from data table for each list item
         * * groupBy - Group the list items with it's related items by mapping groupBy field
         *
         * > For more details about the field mapping refer to [`Data binding`](./data-binding.html) documentation.
         * @default { value: null, iconCss: null, groupBy: null}
         */
        fields: FieldSettingsModel;
        /**
         * When set to ‘false’, consider the [`case-sensitive`](./filtering.html#case-sensitive-filtering)
         * on performing the search to find suggestions.
         * By default consider the casing.
         * @default true.
         */
        ignoreCase: boolean;
        /**
         * Allows you to either show or hide the popup button on the component.
         * @default false.
         */
        showPopupButton: boolean;
        /**
         * When set to ‘true’, highlight the searched characters on suggested list items.
         * > For more details about the highlight refer to [`Custom highlight search`](./how-to.html#custom-highlight-search) documentation.
         * @default false.
         */
        highlight: boolean;
        /**
         * Supports the [`specified number`](./filtering.html#filter-item-count)
         * of list items on the suggestion popup.
         * @default 20.
         */
        suggestionCount: number;
        /**
         * Allows additional HTML attributes such as title, name, etc., and
         * accepts n number of attributes in a key-value pair format.
         *
         * {% codeBlock src="autocomplete/html-attributes-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="autocomplete/html-attributes-api/index.html" %}{% endcodeBlock %}
         * @default {}.
         */
        htmlAttributes: {
            [key: string]: string;
        };
        /**
         * Accepts the external [`query`](./api-query.html)
         * that execute along with data processing.
         *
         * {% codeBlock src="autocomplete/query-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="autocomplete/query-api/index.html" %}{% endcodeBlock %}
         * @default null.
         */
        query: ej.data.Query;
        /**
         * Allows you to set [`the minimum search character length']
         * (./filtering.html#limit-the-minimum-filter-character),
         * the search action will perform after typed minimum characters.
         * @default 1.
         */
        minLength: number;
        /**
         * Determines on which filter type, the component needs to be considered on search action.
         * The available [`FilterType`](./filtering.html#change-the-filter-type)
         * and its supported data types are
         *
         * <table>
         * <tr>
         * <td colSpan=1 rowSpan=1>
         * FilterType<br/></td><td colSpan=1 rowSpan=1>
         * Description<br/></td><td colSpan=1 rowSpan=1>
         * Supported Types<br/></td></tr>
         * <tr>
         * <td colSpan=1 rowSpan=1>
         * StartsWith<br/></td><td colSpan=1 rowSpan=1>
         * Checks whether a value begins with the specified value.<br/></td><td colSpan=1 rowSpan=1>
         * String<br/></td></tr>
         * <tr>
         * <td colSpan=1 rowSpan=1>
         * EndsWith<br/></td><td colSpan=1 rowSpan=1>
         * Checks whether a value ends with specified value.<br/><br/></td><td colSpan=1 rowSpan=1>
         * <br/>String<br/></td></tr>
         * <tr>
         * <td colSpan=1 rowSpan=1>
         * Contains<br/></td><td colSpan=1 rowSpan=1>
         * Checks whether a value contains with specified value.<br/><br/></td><td colSpan=1 rowSpan=1>
         * <br/>String<br/></td></tr>
         * </table>
         *
         * {% codeBlock src="autocomplete/filter-type-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="autocomplete/filter-type-api/index.html" %}{% endcodeBlock %}
         *
         * The default value set to `Contains`, all the suggestion items which contain typed characters to listed in the suggestion popup.
         * @default 'Contains'.
         */
        filterType: FilterType;
        /**
         * Triggers on typing a character in the component.
         * @event
         */
        filtering: ej.base.EmitType<FilteringArgs>;
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        index: number;
        /**
         * Specifies whether to display the floating label above the input element.
         * Possible values are:
         * * Never: The label will never float in the input when the placeholder is available.
         * * Always: The floating label will always float above the input.
         * * Auto: The floating label will float above the input after focusing or entering a value in the input.
         *
         * {% codeBlock src="autocomplete/float-label-type-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="autocomplete/float-label-type-api/index.html" %}{% endcodeBlock %}
         *
         *  @default 'Never'.
         */
        floatLabelType: ej.inputs.FloatLabelType;
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        valueTemplate: string;
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        filterBarPlaceholder: string;
        /**
         * Not applicable to this component.
         * @default false.
         * @private
         */
        allowFiltering: boolean;
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        text: string;
        /**
         * * Constructor for creating the widget
         */
        constructor(options?: AutoCompleteModel, element?: string | HTMLElement);
        /**
         * Initialize the event handler
         * @private
         */
        protected preRender(): void;
        protected getNgDirective(): string;
        protected getQuery(query: ej.data.Query): ej.data.Query;
        protected searchLists(e: ej.base.KeyboardEventArgs): void;
        private filterAction(dataSource, query?, fields?);
        protected clear(e?: MouseEvent, property?: string): void;
        protected onActionComplete(ulElement: HTMLElement, list: {
            [key: string]: Object;
        }[], e?: Object, isUpdated?: boolean): void;
        private postBackAction();
        protected setSelection(li: Element, e: MouseEvent | ej.base.KeyboardEventArgs | TouchEvent): void;
        protected showSpinner(): void;
        protected hideSpinner(): void;
        protected isFiltering(): boolean;
        protected renderPopup(): void;
        protected isEditTextBox(): boolean;
        protected isPopupButton(): boolean;
        protected isSelectFocusItem(element: Element): boolean;
        /**
         * Search the entered text and show it in the suggestion list if available.
         * @returns void.
         */
        showPopup(): void;
        /**
         * Hides the popup if it is in open state.
         * @returns void.
         */
        hidePopup(): void;
        /**
         * Dynamically change the value of properties.
         * @private
         */
        onPropertyChanged(newProp: AutoCompleteModel, oldProp: AutoCompleteModel): void;
        /**
         * Return the module name of this component.
         * @private
         */
        getModuleName(): string;
        /**
         * To initialize the control rendering
         * @private
         */
        render(): void;
    }
    export interface FilteringArgs extends FilteringEventArgs {
        /**
         * To prevent the internal filtering action.
         */
        preventDefaultAction: boolean;
    }
    
    /**
     * Interface for a class ComboBox
     */
    export interface ComboBoxModel extends ej.dropdowns.DropDownListModel{
    
        /**
         * Specifies whether suggest a first matched item in input when searching. No action happens when no matches found.
         * @default false.
         */
        autofill?: boolean;
    
        /**
         * Specifies whether the component allows user defined value which does not exist in data source.    
         * @default true.
         */
        allowCustom?: boolean;
    
        /**
         * Allows additional HTML attributes such as title, name, etc., and
         * accepts n number of attributes in a key-value pair format.
         * 
         * {% codeBlock src="combobox/html-attributes-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="combobox/html-attributes-api/index.html" %}{% endcodeBlock %}
         * @default {}.
         */
        htmlAttributes?: { [key: string]: string; };
    
        /**
         * When allowFiltering is set to true, show the filter bar (search box) of the component.
         * The filter action retrieves matched items through the `filtering` event based on
         * the characters typed in the search TextBox.
         * If no match is found, the value of the `noRecordsTemplate` property will be displayed.
         * 
         * {% codeBlock src="combobox/allow-filtering-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="combobox/allow-filtering-api/index.html" %}{% endcodeBlock %}
         * @default false.
         */
        allowFiltering?: boolean;
    
        /**
         * Accepts the external [`Query`](./api-query.html)
         * that execute along with [`data processing`](./data-binding.html).
         * 
         * {% codeBlock src="combobox/query-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="combobox/query-api/index.html" %}{% endcodeBlock %}
         * @default null.
         */
        query?: ej.data.Query;
    
        /**
         * Gets or sets the index of the selected item in the component.
         * 
         * {% codeBlock src="combobox/index-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="combobox/index-api/index.html" %}{% endcodeBlock %}
         * 
         * @default null.
         */
        index?: number;
    
        /**
         * Specifies whether to show or hide the clear button. 
         * When the clear button is clicked, `value`, `text`, and `index` properties are reset to null.
         * @default true.
         */
        showClearButton?: boolean;
    
        /**
         * Triggers on set a 
         * [`custom value`](./getting-started.html#custom-values) to this component.
         * @event
         */
        customValueSpecifier?: ej.base.EmitType<CustomValueSpecifierEventArgs>;
    
        /**
         * Triggers on typing a character in the component.
         * > For more details about the filtering refer to [`Filtering`](./filtering.html) documentation.
         * @event
         */
        filtering?: ej.base.EmitType<FilteringEventArgs>;
    
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        valueTemplate?: string;
    
        /**
         * Specifies whether to display the floating label above the input element.
         * Possible values are:
         * * Never: The label will never float in the input when the placeholder is available.
         * * Always: The floating label will always float above the input.
         * * Auto: The floating label will float above the input after focusing or entering a value in the input.
         * 
         * {% codeBlock src="combobox/float-label-type-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="combobox/float-label-type-api/index.html" %}{% endcodeBlock %}
         * 
         *  @default 'Never'.
         */
        floatLabelType?: ej.inputs.FloatLabelType;
    
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        filterBarPlaceholder?: string;
    
    }
    
    /**
     * The ComboBox component allows the user to type a value or choose an option from the list of predefined options.
     * ```html
     * <select id="list">
     *      <option value='1'>Badminton</option>
     *      <option value='2'>Basketball</option>
     *      <option value='3'>Cricket</option>
     *      <option value='4'>Football</option>
     *      <option value='5'>Tennis</option>
     * </select>
     * ```
     * ```typescript
     *   let games:ComboBox = new ComboBox();
     *   games.appendTo("#list");
     * ```
     */
    export class ComboBox extends DropDownList {
        /**
         * Specifies whether suggest a first matched item in input when searching. No action happens when no matches found.
         * @default false.
         */
        autofill: boolean;
        /**
         * Specifies whether the component allows user defined value which does not exist in data source.
         * @default true.
         */
        allowCustom: boolean;
        /**
         * Allows additional HTML attributes such as title, name, etc., and
         * accepts n number of attributes in a key-value pair format.
         *
         * {% codeBlock src="combobox/html-attributes-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="combobox/html-attributes-api/index.html" %}{% endcodeBlock %}
         * @default {}.
         */
        htmlAttributes: {
            [key: string]: string;
        };
        /**
         * When allowFiltering is set to true, show the filter bar (search box) of the component.
         * The filter action retrieves matched items through the `filtering` event based on
         * the characters typed in the search TextBox.
         * If no match is found, the value of the `noRecordsTemplate` property will be displayed.
         *
         * {% codeBlock src="combobox/allow-filtering-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="combobox/allow-filtering-api/index.html" %}{% endcodeBlock %}
         * @default false.
         */
        allowFiltering: boolean;
        /**
         * Accepts the external [`Query`](./api-query.html)
         * that execute along with [`data processing`](./data-binding.html).
         *
         * {% codeBlock src="combobox/query-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="combobox/query-api/index.html" %}{% endcodeBlock %}
         * @default null.
         */
        query: ej.data.Query;
        /**
         * Gets or sets the index of the selected item in the component.
         *
         * {% codeBlock src="combobox/index-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="combobox/index-api/index.html" %}{% endcodeBlock %}
         *
         * @default null.
         */
        index: number;
        /**
         * Specifies whether to show or hide the clear button.
         * When the clear button is clicked, `value`, `text`, and `index` properties are reset to null.
         * @default true.
         */
        showClearButton: boolean;
        /**
         * Triggers on set a
         * [`custom value`](./getting-started.html#custom-values) to this component.
         * @event
         */
        customValueSpecifier: ej.base.EmitType<CustomValueSpecifierEventArgs>;
        /**
         * Triggers on typing a character in the component.
         * > For more details about the filtering refer to [`Filtering`](./filtering.html) documentation.
         * @event
         */
        filtering: ej.base.EmitType<FilteringEventArgs>;
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        valueTemplate: string;
        /**
         * Specifies whether to display the floating label above the input element.
         * Possible values are:
         * * Never: The label will never float in the input when the placeholder is available.
         * * Always: The floating label will always float above the input.
         * * Auto: The floating label will float above the input after focusing or entering a value in the input.
         *
         * {% codeBlock src="combobox/float-label-type-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="combobox/float-label-type-api/index.html" %}{% endcodeBlock %}
         *
         *  @default 'Never'.
         */
        floatLabelType: ej.inputs.FloatLabelType;
        /**
         * Not applicable to this component.
         * @default null.
         * @private
         */
        filterBarPlaceholder: string;
        /**
         * *Constructor for creating the component
         */
        constructor(options?: ComboBoxModel, element?: string | HTMLElement);
        /**
         * Initialize the event handler
         * @private
         */
        protected preRender(): void;
        protected wireEvent(): void;
        private preventBlur(e);
        protected targetElement(): HTMLElement | HTMLInputElement;
        protected setOldText(text: string): void;
        protected setOldValue(value: string | number): void;
        private valueMuteChange(value);
        protected updateValues(): void;
        protected getAriaAttributes(): {
            [key: string]: string;
        };
        protected searchLists(e: ej.base.KeyboardEventArgs): void;
        protected getNgDirective(): string;
        protected setSearchBox(): ej.inputs.InputObject;
        protected onActionComplete(ulElement: HTMLElement, list: {
            [key: string]: Object;
        }[], e?: Object, isUpdated?: boolean): void;
        protected getFocusElement(): Element;
        protected setValue(e?: ej.base.KeyboardEventArgs): boolean;
        protected showSpinner(): void;
        protected hideSpinner(): void;
        protected setAutoFill(activeElement: Element, isHover?: boolean): void;
        private isAndroidAutoFill(value);
        protected isSelectFocusItem(element: Element): boolean;
        private inlineSearch(e?);
        protected incrementalSearch(e: ej.base.KeyboardEventArgs): void;
        private setAutoFillSelection(currentValue);
        protected getValueByText(text: string): string | number;
        protected unWireEvent(): void;
        protected setSelection(li: Element, e: MouseEvent | ej.base.KeyboardEventArgs | TouchEvent): void;
        protected selectCurrentItem(e: ej.base.KeyboardEventArgs): void;
        protected setHoverList(li: Element): void;
        private targetFocus(e);
        protected dropDownClick(e: MouseEvent): void;
        private customValue();
        /**
         * Dynamically change the value of properties.
         * @private
         */
        onPropertyChanged(newProp: ComboBoxModel, oldProp: ComboBoxModel): void;
        /**
         * To initialize the control rendering.
         * @private
         */
        render(): void;
        /**
         * Return the module name of this component.
         * @private
         */
        getModuleName(): string;
        /**
         * Hides the popup if it is in open state.
         * @returns void.
         */
        hidePopup(): void;
        /**
         * Sets the focus to the component for interaction.
         * @returns void.
         */
        focusIn(): void;
    }
    export interface CustomValueSpecifierEventArgs {
        /**
         * Gets the typed custom text to make a own text format and assign it to `item` argument.
         */
        text: string;
        /**
         * Sets the text custom format data for set a `value` and `text`.
         */
        item: {
            [key: string]: string | Object;
        };
    }
    
    export type HightLightType = 'Contains' | 'StartsWith' | 'EndsWith';
    /**
     * Function helps to find which highlightSearch is to call based on your data.
     * @param  {HTMLElement} content - Specifies an content element.
     * @param  {string} query - Specifies the string to be highlighted.
     * @param  {boolean} ignoreCase - Specifies the ignoreCase option.
     * @param  {HightLightType} type - Specifies the type of highlight.
     */
    export function highlightSearch(content: HTMLElement, query: string, ignoreCase: boolean, type?: HightLightType): void;
    
    export type SearchType = 'StartsWith' | 'Equal';
    /**
     * Search and focus the list item based on key code matches with list text content
     * @param  { number } keyCode - Specifies the key code which pressed on keyboard events.
     * @param  { HTMLElement[]] } items - Specifies an array of HTMLElement, from which matches find has done.
     * @param { number } selectedIndex - Specifies the selected item in list item, so that search will happen
     * after selected item otherwise it will do from initial.
     * @param  { boolean } ignoreCase - Specifies the case consideration when search has done.
     */
    export function incrementalSearch(keyCode: number, items: HTMLElement[], selectedIndex: number, ignoreCase: boolean): Element;
    export function Search(inputVal: string, items: HTMLElement[], searchType: SearchType, ignoreCase?: boolean): {
        [key: string]: Element | number;
    };
    
    /**
     * Interface for a class FieldSettings
     */
    export interface FieldSettingsModel {
    
        /**
         * Maps the text column from data table for each list item
         * @default null.
         */
        text?: string;
    
        /**
         * Maps the value column from data table for each list item
         * @default null.
         */
        value?: string;
    
        /**
         * Maps the icon class column from data table for each list item.
         * @default null.
         */
        iconCss?: string;
    
        /**
         * Group the list items with it's related items by mapping groupBy field.
         * @default null.
         */
        groupBy?: string;
    
    }
    
    /**
     * Interface for a class DropDownBase
     */
    export interface DropDownBaseModel extends ej.base.ComponentModel{
    
        /**
         * The `fields` property maps the columns of the data table and binds the data to the component.
         * * text - Maps the text column from data table for each list item.
         * * value - Maps the value column from data table for each list item.
         * * iconCss - Maps the icon class column from data table for each list item.
         * * groupBy - Group the list items with it's related items by mapping groupBy field.
         * ```html
         * <input type="text" tabindex="1" id="list"> </input>
         * ```
         * ```typescript  
         *   let customers: DropDownList = new DropDownList({
         *      dataSource:new DataManager({ url:'http://js.syncfusion.com/demos/ejServices/Wcf/Northwind.svc/' }),
         *      query: new Query().from('Customers').select(['ContactName', 'CustomerID']).take(5),
         *      fields: { text: 'ContactName', value: 'CustomerID' },
         *      placeholder: 'Select a customer'
         *   });
         *   customers.appendTo("#list");
         * ```
         * @default {text: null, value: null, iconCss: null, groupBy: null}
         */
        fields?: FieldSettingsModel;
    
        /**
         * When set to true, enables RTL mode of the component that 
         * displays the content in the right-to-left direction.
         * @default false.
         */
        enableRtl?: boolean;
    
        /**
         * Enable or disable persisting component's state between page reloads. 
         * If enabled, following list of states will be persisted.
         * 1. value
         * @default false.
         */
        enablePersistence?: boolean;
    
        /**
         * Accepts the template design and assigns it to each list item present in the popup.
         * We have built-in [`template engine`](./template-engine.html)
         * 
         * which provides options to compile template string into a executable function. 
         * For EX: We have expression evolution as like ES6 expression string literals. 
         * @default null.
         */
        itemTemplate?: string;
    
        /**
         * Accepts the template design and assigns it to the group headers present in the popup list.
         * @default null.
         */
        groupTemplate?: string;
    
        /**
         * Accepts the template design and assigns it to popup list of component
         * when no data is available on the component.
         * @default 'No Records Found'.
         */
        noRecordsTemplate?: string;
    
        /**
         * Accepts the template and assigns it to the popup list content of the component
         * when the data fetch request from the remote server fails.
         * @default 'The Request Failed'.
         */
        actionFailureTemplate?: string;
    
        /**
         * Specifies the `sortOrder` to sort the data source. The available type of sort orders are
         * * `None` - The data source is not sorting.
         * * `Ascending` - The data source is sorting with ascending order.
         * * `Descending` - The data source is sorting with descending order.
         * @default None.
         */
        sortOrder?: ej.lists.SortOrder;
    
        /**
         * Specifies a value that indicates whether the component is enabled or not.
         * @default true.
         */
        enabled?: boolean;
    
        /**
         * Accepts the list items either through local or remote service and binds it to the component.
         * It can be an array of JSON Objects or an instance of
         * [`DataManager`](./api-dataManager.html).
         * @default [].
         */
        dataSource?: { [key: string]: Object }[] | ej.data.DataManager | string[] | number[];
    
        /**
         * Accepts the external [`Query`](./api-query.html)
         * which will execute along with the data processing.
         * @default null.
         */
        query?: ej.data.Query;
    
        /**
         * specifies the z-index value of the component popup element.
         * @default 1000
         */
        zIndex?: number;
    
        /**
         * ignoreAccent set to true, then ignores the diacritic characters or accents when filtering.
         */
        ignoreAccent?: boolean;
    
        /**
         * Triggers before fetching data from the remote server.
         * @event
         */
        actionBegin?: ej.base.EmitType<Object>;
    
        /**
         * Triggers after data is fetched successfully from the remote server.
         * @event
         */
        actionComplete?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the data fetch request from the remote server fails.
         * @event
         */
        actionFailure?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when an item in the popup is selected by the user either with mouse/tap or with keyboard navigation.
         * @event
         */
        select?: ej.base.EmitType<SelectEventArgs>;
    
    }
    
    export class FieldSettings extends ej.base.ChildProperty<FieldSettings> {
        /**
         * Maps the text column from data table for each list item
         * @default null.
         */
        text: string;
        /**
         * Maps the value column from data table for each list item
         * @default null.
         */
        value: string;
        /**
         * Maps the icon class column from data table for each list item.
         * @default null.
         */
        iconCss: string;
        /**
         * Group the list items with it's related items by mapping groupBy field.
         * @default null.
         */
        groupBy: string;
    }
    export const dropDownBaseClasses: DropDownBaseClassList;
    export interface DropDownBaseClassList {
        root: string;
        rtl: string;
        content: string;
        selected: string;
        hover: string;
        noData: string;
        fixedHead: string;
        focus: string;
        li: string;
        disabled: string;
        group: string;
        grouping: string;
    }
    export interface SelectEventArgs {
        /**
         * If the event is triggered by interaction, it returns true. Otherwise, it returns false.
         */
        isInteracted: boolean;
        /**
         * Returns the selected list item
         */
        item: HTMLLIElement;
        /**
         * Returns the selected item as JSON Object from the data source.
         */
        itemData: FieldSettingsModel;
        /**
         * Specifies the original event arguments.
         */
        e: MouseEvent | KeyboardEvent | TouchEvent;
        /**
         * Illustrates whether the current action needs to be prevented or not.
         */
        cancel?: boolean;
    }
    /**
     * DropDownBase component will generate the list items based on given data and act as base class to drop-down related components
     */
    export class DropDownBase extends ej.base.Component<HTMLElement> implements ej.base.INotifyPropertyChanged {
        protected listData: {
            [key: string]: Object;
        }[];
        protected ulElement: HTMLElement;
        protected liCollections: HTMLElement[];
        private bindEvent;
        private scrollTimer;
        protected list: HTMLElement;
        protected fixedHeaderElement: HTMLElement;
        protected keyboardModule: ej.base.KeyboardEvents;
        protected enableRtlElements: HTMLElement[];
        protected rippleFun: Function;
        protected l10n: ej.base.L10n;
        protected item: HTMLLIElement;
        protected itemData: {
            [key: string]: Object;
        };
        protected isActive: boolean;
        protected isRequested: boolean;
        protected queryString: string;
        /**
         * The `fields` property maps the columns of the data table and binds the data to the component.
         * * text - Maps the text column from data table for each list item.
         * * value - Maps the value column from data table for each list item.
         * * iconCss - Maps the icon class column from data table for each list item.
         * * groupBy - Group the list items with it's related items by mapping groupBy field.
         * ```html
         * <input type="text" tabindex="1" id="list"> </input>
         * ```
         * ```typescript
         *   let customers: DropDownList = new DropDownList({
         *      dataSource:new DataManager({ url:'http://js.syncfusion.com/demos/ejServices/Wcf/Northwind.svc/' }),
         *      query: new Query().from('Customers').select(['ContactName', 'CustomerID']).take(5),
         *      fields: { text: 'ContactName', value: 'CustomerID' },
         *      placeholder: 'Select a customer'
         *   });
         *   customers.appendTo("#list");
         * ```
         * @default {text: null, value: null, iconCss: null, groupBy: null}
         */
        fields: FieldSettingsModel;
        /**
         * When set to true, enables RTL mode of the component that
         * displays the content in the right-to-left direction.
         * @default false.
         */
        enableRtl: boolean;
        /**
         * Enable or disable persisting component's state between page reloads.
         * If enabled, following list of states will be persisted.
         * 1. value
         * @default false.
         */
        enablePersistence: boolean;
        /**
         * Accepts the template design and assigns it to each list item present in the popup.
         * We have built-in [`template engine`](./template-engine.html)
         *
         * which provides options to compile template string into a executable function.
         * For EX: We have expression evolution as like ES6 expression string literals.
         * @default null.
         */
        itemTemplate: string;
        /**
         * Accepts the template design and assigns it to the group headers present in the popup list.
         * @default null.
         */
        groupTemplate: string;
        /**
         * Accepts the template design and assigns it to popup list of component
         * when no data is available on the component.
         * @default 'No Records Found'.
         */
        noRecordsTemplate: string;
        /**
         * Accepts the template and assigns it to the popup list content of the component
         * when the data fetch request from the remote server fails.
         * @default 'The Request Failed'.
         */
        actionFailureTemplate: string;
        /**
         * Specifies the `sortOrder` to sort the data source. The available type of sort orders are
         * * `None` - The data source is not sorting.
         * * `Ascending` - The data source is sorting with ascending order.
         * * `Descending` - The data source is sorting with descending order.
         * @default None.
         */
        sortOrder: ej.lists.SortOrder;
        /**
         * Specifies a value that indicates whether the component is enabled or not.
         * @default true.
         */
        enabled: boolean;
        /**
         * Accepts the list items either through local or remote service and binds it to the component.
         * It can be an array of JSON Objects or an instance of
         * [`DataManager`](./api-dataManager.html).
         * @default [].
         */
        dataSource: {
            [key: string]: Object;
        }[] | ej.data.DataManager | string[] | number[];
        /**
         * Accepts the external [`Query`](./api-query.html)
         * which will execute along with the data processing.
         * @default null.
         */
        query: ej.data.Query;
        /**
         * specifies the z-index value of the component popup element.
         * @default 1000
         */
        zIndex: number;
        /**
         * ignoreAccent set to true, then ignores the diacritic characters or accents when filtering.
         */
        ignoreAccent: boolean;
        /**
         * Triggers before fetching data from the remote server.
         * @event
         */
        actionBegin: ej.base.EmitType<Object>;
        /**
         * Triggers after data is fetched successfully from the remote server.
         * @event
         */
        actionComplete: ej.base.EmitType<Object>;
        /**
         * Triggers when the data fetch request from the remote server fails.
         * @event
         */
        actionFailure: ej.base.EmitType<Object>;
        /**
         * Triggers when an item in the popup is selected by the user either with mouse/tap or with keyboard navigation.
         * @event
         */
        select: ej.base.EmitType<SelectEventArgs>;
        /**
         * * Constructor for DropDownBase class
         */
        constructor(options?: DropDownBaseModel, element?: string | HTMLElement);
        protected getPropObject(prop: string, newProp: {
            [key: string]: string;
        }, oldProp: {
            [key: string]: string;
        }): {
            [key: string]: Object;
        };
        protected getValueByText(text: string, ignoreCase?: boolean): string | number;
        protected l10nUpdate(actionFailure?: boolean): void;
        protected getTextByValue(value: string | number): string;
        protected getFormattedValue(value: string): string | number;
        /**
         * Sets RTL to dropdownbase wrapper
         */
        protected setEnableRtl(): void;
        /**
         * Initialize the Component.
         */
        private initialize();
        /**
         * Get the properties to be maintained in persisted state.
         */
        protected getPersistData(): string;
        /**
         * Sets the enabled state to DropDownBase.
         */
        protected setEnabled(): void;
        private renderItemsBySelect();
        private getJSONfromOption(items, options, fields);
        /**
         * Execute before render the list items
         * @private
         */
        protected preRender(): void;
        /**
         * Creates the list items of DropDownBase component.
         */
        private setListData(dataSource, fields, query);
        protected showSpinner(): void;
        protected hideSpinner(): void;
        protected onActionFailure(e: Object): void;
        protected onActionComplete(ulElement: HTMLElement, list: {
            [key: string]: Object;
        }[], e?: Object): void;
        protected postRender(listElement: HTMLElement, list: {
            [key: string]: Object;
        }[], bindEvent: boolean): void;
        /**
         * Get the query to do the data operation before list item generation.
         */
        protected getQuery(query: ej.data.Query): ej.data.Query;
        /**
         * To render the template content for group header element.
         */
        private renderGroupTemplate(listEle);
        /**
         * To create the ul li list items
         */
        private createListItems(dataSource, fields);
        protected listOption(dataSource: {
            [key: string]: Object;
        }[], fields: FieldSettingsModel): {
            [key: string]: Object;
        };
        protected setFloatingHeader(e: Event): void;
        private scrollStop(e);
        /**
         * To render the list items
         */
        private renderItems(listData, fields);
        protected templateListItem(dataSource: {
            [key: string]: Object;
        }[], fields: FieldSettingsModel): HTMLElement;
        protected typeOfData(items: {
            [key: string]: Object;
        }[] | string[]): {
            [key: string]: Object;
        };
        protected setFixedHeader(): void;
        private getSortedDataSource(dataSource);
        /**
         * Return the index of item which matched with given value in data source
         */
        protected getIndexByValue(value: string | number): number;
        /**
         * To dispatch the event manually
         */
        protected dispatchEvent(element: HTMLElement, type: string): void;
        /**
         * To set the current fields
         */
        protected setFields(): void;
        /**
         * reset the items list.
         */
        protected resetList(dataSource?: {
            [key: string]: Object;
        }[] | ej.data.DataManager | string[] | number[], fields?: FieldSettingsModel, query?: ej.data.Query): void;
        protected updateSelection(): void;
        protected renderList(): void;
        protected updateDataSource(prop?: string): void;
        protected setUpdateInitial(props: string[], newProp: {
            [key: string]: string;
        }): void;
        /**
         * When property value changes happened, then onPropertyChanged method will execute the respective changes in this component.
         * @private
         */
        onPropertyChanged(newProp: DropDownBaseModel, oldProp: DropDownBaseModel): void;
        /**
         * Build and render the component
         * @private
         */
        render(isEmptyData?: boolean): void;
        /**
         * Return the module name of this component.
         * @private
         */
        getModuleName(): string;
        /**
         * Gets all the list items bound on this component.
         * @returns Element[].
         */
        getItems(): Element[];
        /**
         * Adds a new item to the popup list. By default, new item appends to the list as the last item,
         * but you can insert based on the index parameter.
         * @param  { Object[] } items - Specifies an array of JSON data or a JSON data.
         * @param { number } itemIndex - Specifies the index to place the newly added item in the popup list.
         * @return {void}.
         */
        addItem(items: {
            [key: string]: Object;
        }[] | {
            [key: string]: Object;
        }, itemIndex?: number): void;
        protected setZIndex(): void;
        protected updateActionCompleteData(li: HTMLElement, item: {
            [key: string]: Object;
        }): void;
        /**
         * Gets the data Object that matches the given value.
         * @param { string | number } value - Specifies the value of the list item.
         * @returns Object.
         */
        getDataByValue(value: string | number): {
            [key: string]: Object;
        };
        /**
         * Removes the component from the DOM and detaches all its related event handlers. It also removes the attributes and classes.
         * @method destroy
         * @return {void}.
         */
        destroy(): void;
    }
    
    /**
     * Interface for a class DropDownList
     */
    export interface DropDownListModel extends DropDownBaseModel{
    
        /**
         * Sets CSS classes to the root element of the component that allows customization of appearance.
         * @default null.
         */
        cssClass?: string;
    
        /**
         * Specifies the width of the component. By default, the component width sets based on the width of 
         * its parent container. You can also set the width in pixel values.
         * @default '100%'.
         */
        width?: string | number;
    
        /**
         * Specifies the height of the popup list.  
         * > For more details about the popup configuration refer to 
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         * @default '300px'.
         */
        popupHeight?: string | number;
    
        /**
         * Specifies the width of the popup list. By default, the popup width sets based on the width of 
         * the component.
         * > For more details about the popup configuration refer to 
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         * @default '100%'.
         */
        popupWidth?: string | number;
    
        /**
         * Specifies a short hint that describes the expected value of the DropDownList component.
         * @default null.
         */
        placeholder?: string;
    
        /**
         * Accepts the value to be displayed as a watermark text on the filter bar. 
         * @default null.
         */
        filterBarPlaceholder?: string;
    
        /**
         * Allows additional HTML attributes such as title, name, etc., and
         * accepts n number of attributes in a key-value pair format.
         * 
         * {% codeBlock src="dropdownlist/html-attributes-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="dropdownlist/html-attributes-api/index.html" %}{% endcodeBlock %}
         *
         * @default {}.
         */
        htmlAttributes?: { [key: string]: string; };
    
        /**
         * Accepts the external [`Query`](./api-query.html)
         * that execute along with data processing.
         * 
         * {% codeBlock src="dropdownlist/query-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="dropdownlist/query-api/index.html" %}{% endcodeBlock %}
         * 
         * @default null.
         */
        query?: ej.data.Query;
    
        /**
         * Accepts the template design and assigns it to the selected list item in the input element of the component.
         * For more details about the available template options refer to 
         * [`Template`](./templates.html) documentation.
         * 
         * We have built-in [`template engine`](./template-engine.html) 
         * which provides options to compile template string into a executable function. 
         * For EX: We have expression evolution as like ES6 expression string literals.
         * @default null.
         */
        valueTemplate?: string;
    
        /**
         * Accepts the template design and assigns it to the header container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         * @default null.
         */
        headerTemplate?: string;
    
        /**
         * Accepts the template design and assigns it to the footer container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         * @default null.
         */
        footerTemplate?: string;
    
        /**
         * When allowFiltering is set to true, show the filter bar (search box) of the component.
         * The filter action retrieves matched items through the `filtering` event based on
         * the characters typed in the search TextBox.
         * 
         * If no match is found, the value of the `noRecordsTemplate` property will be displayed.
         * > For more details about the filtering refer to [`Filtering`](./filtering.html) documentation.
         * 
         * {% codeBlock src="dropdownlist/allow-filtering-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="dropdownlist/allow-filtering-api/index.html" %}{% endcodeBlock %}
         * @default false.
         */
        allowFiltering?: boolean;
    
        /**
         * When set to true, the user interactions on the component are disabled.
         * @default false.
         */
        readonly?: boolean;
    
        /**
         * Gets or sets the display text of the selected item in the component.
         * @default null.
         */
        text?: string;
    
        /**
         * Gets or sets the value of the selected item in the component.
         * @default null.
         */
        value?: number | string;
    
        /**
         * Gets or sets the index of the selected item in the component.
         * 
         * {% codeBlock src="dropdownlist/index-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="dropdownlist/index-api/index.html" %}{% endcodeBlock %}
         * 
         * @default null.
         */
        index?: number;
    
        /**
         * Specifies whether to display the floating label above the input element.
         * Possible values are:
         * * Never: The label will never float in the input when the placeholder is available.
         * * Always: The floating label will always float above the input.
         * * Auto: The floating label will float above the input after focusing or entering a value in the input.
         * 
         * {% codeBlock src="dropdownlist/float-label-type-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="dropdownlist/float-label-type-api/index.html" %}{% endcodeBlock %}
         * 
         *  @default 'Never'.
         */
        floatLabelType?: ej.inputs.FloatLabelType;
    
        /**
         * Specifies whether to show or hide the clear button. 
         * When the clear button is clicked, `value`, `text`, and `index` properties are reset to null.
         * @default false.
         */
        showClearButton?: boolean;
    
        /**
         * Triggers on typing a character in the filter bar when the 
         * [`allowFiltering`](./api-dropDownList.html#allowfiltering) 
         * is enabled.
         * > For more details about the filtering refer to [`Filtering`](./filtering.html) documentation.
         * 
         * @event
         */
        filtering?: ej.base.EmitType<FilteringEventArgs>;
    
        /**
         * Triggers when an item in a popup is selected or when the model value is changed by user.
         * Use change event to 
         * [`Configure the Cascading DropDownList`](./how-to.html#configure-the-cascading-dropdownlist)
         * @event
         */
        change?: ej.base.EmitType<ChangeEventArgs>;
    
        /**
         * Triggers when the popup opens.
         * @event
         */
        open?: ej.base.EmitType<PopupEventArgs>;
    
        /**
         * Triggers when the popup is closed.
         * @event
         */
        close?: ej.base.EmitType<PopupEventArgs>;
    
        /**
         * Triggers when focus moves out from the component.
         * @event
         */
        blur?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the component is focused.
         * @event
         */
        focus?: ej.base.EmitType<Object>;
    
    }
    
    export interface ChangeEventArgs extends SelectEventArgs {
        /**
         * Returns the selected value
         */
        value: number | string;
    }
    export const dropDownListClasses: DropDownListClassList;
    /**
     * The DropDownList component contains a list of predefined values from which you can
     * choose a single value.
     * ```html
     * <input type="text" tabindex="1" id="list"> </input>
     * ```
     * ```typescript
     *   let dropDownListObj:DropDownList = new DropDownList();
     *   dropDownListObj.appendTo("#list");
     * ```
     */
    export class DropDownList extends DropDownBase implements ej.inputs.IInput {
        protected inputWrapper: ej.inputs.InputObject;
        protected inputElement: HTMLInputElement;
        private valueTempElement;
        private listObject;
        private header;
        private footer;
        protected selectedLI: HTMLElement;
        private listHeight;
        protected hiddenElement: HTMLSelectElement;
        protected isPopupOpen: boolean;
        private isDocumentClick;
        protected isInteracted: boolean;
        private isFilterFocus;
        protected beforePopupOpen: boolean;
        protected initial: boolean;
        private initRemoteRender;
        private searchBoxHeight;
        private popupObj;
        private backIconElement;
        private clearIconElement;
        private containerStyle;
        protected previousValue: string | number;
        protected activeIndex: number;
        protected filterInput: HTMLInputElement;
        private searchKeyModule;
        private tabIndex;
        private isNotSearchList;
        protected isTyped: boolean;
        protected isSelected: boolean;
        protected preventFocus: boolean;
        protected preventAutoFill: boolean;
        protected queryString: string;
        protected isValidKey: boolean;
        protected typedString: string;
        protected isEscapeKey: boolean;
        private isPreventBlur;
        protected isTabKey: boolean;
        private actionCompleteData;
        protected prevSelectPoints: {
            [key: string]: number;
        };
        protected isSelectCustom: boolean;
        protected isDropDownClick: boolean;
        protected preventAltUp: boolean;
        private searchKeyEvent;
        private filterInputObj;
        protected spinnerElement: HTMLElement;
        protected keyConfigure: {
            [key: string]: string;
        };
        protected isCustomFilter: boolean;
        /**
         * Sets CSS classes to the root element of the component that allows customization of appearance.
         * @default null.
         */
        cssClass: string;
        /**
         * Specifies the width of the component. By default, the component width sets based on the width of
         * its parent container. You can also set the width in pixel values.
         * @default '100%'.
         */
        width: string | number;
        /**
         * Specifies the height of the popup list.
         * > For more details about the popup configuration refer to
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         * @default '300px'.
         */
        popupHeight: string | number;
        /**
         * Specifies the width of the popup list. By default, the popup width sets based on the width of
         * the component.
         * > For more details about the popup configuration refer to
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         * @default '100%'.
         */
        popupWidth: string | number;
        /**
         * Specifies a short hint that describes the expected value of the DropDownList component.
         * @default null.
         */
        placeholder: string;
        /**
         * Accepts the value to be displayed as a watermark text on the filter bar.
         * @default null.
         */
        filterBarPlaceholder: string;
        /**
         * Allows additional HTML attributes such as title, name, etc., and
         * accepts n number of attributes in a key-value pair format.
         *
         * {% codeBlock src="dropdownlist/html-attributes-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="dropdownlist/html-attributes-api/index.html" %}{% endcodeBlock %}
         *
         * @default {}.
         */
        htmlAttributes: {
            [key: string]: string;
        };
        /**
         * Accepts the external [`Query`](./api-query.html)
         * that execute along with data processing.
         *
         * {% codeBlock src="dropdownlist/query-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="dropdownlist/query-api/index.html" %}{% endcodeBlock %}
         *
         * @default null.
         */
        query: ej.data.Query;
        /**
         * Accepts the template design and assigns it to the selected list item in the input element of the component.
         * For more details about the available template options refer to
         * [`Template`](./templates.html) documentation.
         *
         * We have built-in [`template engine`](./template-engine.html)
         * which provides options to compile template string into a executable function.
         * For EX: We have expression evolution as like ES6 expression string literals.
         * @default null.
         */
        valueTemplate: string;
        /**
         * Accepts the template design and assigns it to the header container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         * @default null.
         */
        headerTemplate: string;
        /**
         * Accepts the template design and assigns it to the footer container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         * @default null.
         */
        footerTemplate: string;
        /**
         * When allowFiltering is set to true, show the filter bar (search box) of the component.
         * The filter action retrieves matched items through the `filtering` event based on
         * the characters typed in the search TextBox.
         *
         * If no match is found, the value of the `noRecordsTemplate` property will be displayed.
         * > For more details about the filtering refer to [`Filtering`](./filtering.html) documentation.
         *
         * {% codeBlock src="dropdownlist/allow-filtering-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="dropdownlist/allow-filtering-api/index.html" %}{% endcodeBlock %}
         * @default false.
         */
        allowFiltering: boolean;
        /**
         * When set to true, the user interactions on the component are disabled.
         * @default false.
         */
        readonly: boolean;
        /**
         * Gets or sets the display text of the selected item in the component.
         * @default null.
         */
        text: string;
        /**
         * Gets or sets the value of the selected item in the component.
         * @default null.
         */
        value: number | string;
        /**
         * Gets or sets the index of the selected item in the component.
         *
         * {% codeBlock src="dropdownlist/index-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="dropdownlist/index-api/index.html" %}{% endcodeBlock %}
         *
         * @default null.
         */
        index: number;
        /**
         * Specifies whether to display the floating label above the input element.
         * Possible values are:
         * * Never: The label will never float in the input when the placeholder is available.
         * * Always: The floating label will always float above the input.
         * * Auto: The floating label will float above the input after focusing or entering a value in the input.
         *
         * {% codeBlock src="dropdownlist/float-label-type-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="dropdownlist/float-label-type-api/index.html" %}{% endcodeBlock %}
         *
         *  @default 'Never'.
         */
        floatLabelType: ej.inputs.FloatLabelType;
        /**
         * Specifies whether to show or hide the clear button.
         * When the clear button is clicked, `value`, `text`, and `index` properties are reset to null.
         * @default false.
         */
        showClearButton: boolean;
        /**
         * Triggers on typing a character in the filter bar when the
         * [`allowFiltering`](./api-dropDownList.html#allowfiltering)
         * is enabled.
         * > For more details about the filtering refer to [`Filtering`](./filtering.html) documentation.
         *
         * @event
         */
        filtering: ej.base.EmitType<FilteringEventArgs>;
        /**
         * Triggers when an item in a popup is selected or when the model value is changed by user.
         * Use change event to
         * [`Configure the Cascading DropDownList`](./how-to.html#configure-the-cascading-dropdownlist)
         * @event
         */
        change: ej.base.EmitType<ChangeEventArgs>;
        /**
         * Triggers when the popup opens.
         * @event
         */
        open: ej.base.EmitType<PopupEventArgs>;
        /**
         * Triggers when the popup is closed.
         * @event
         */
        close: ej.base.EmitType<PopupEventArgs>;
        /**
         * Triggers when focus moves out from the component.
         * @event
         */
        blur: ej.base.EmitType<Object>;
        /**
         * Triggers when the component is focused.
         * @event
         */
        focus: ej.base.EmitType<Object>;
        /**
         * * Constructor for creating the DropDownList component.
         */
        constructor(options?: ej.dropdowns.DropDownListModel, element?: string | HTMLElement);
        /**
         * Initialize the event handler.
         * @private
         */
        protected preRender(): void;
        private initializeData();
        protected setZIndex(): void;
        protected renderList(isEmptyData?: boolean): void;
        private floatLabelChange();
        protected resetHandler(e: MouseEvent): void;
        protected resetFocusElement(): void;
        protected clear(e?: MouseEvent | ej.base.KeyboardEventArgs, property?: string): void;
        private resetSelection();
        private setHTMLAttributes();
        protected getAriaAttributes(): {
            [key: string]: string;
        };
        protected setEnableRtl(): void;
        private setEnable();
        /**
         * Get the properties to be maintained in the persisted state.
         */
        protected getPersistData(): string;
        private preventTabIndex(element);
        protected targetElement(): HTMLElement | HTMLInputElement;
        protected getNgDirective(): string;
        protected getElementByText(text: string): Element;
        protected getElementByValue(value: string | number): Element;
        private initValue();
        protected updateValues(): void;
        protected onBlur(e: MouseEvent): void;
        protected focusOutAction(): void;
        protected onFocusOut(): void;
        protected onFocus(): void;
        protected wireEvent(): void;
        protected bindCommonEvent(): void;
        private bindClearEvent();
        protected unBindCommonEvent(): void;
        protected updateIconState(): void;
        /**
         * Event binding for list
         */
        private wireListEvents();
        private onSearch(e);
        protected onMouseClick(e: MouseEvent): void;
        private onMouseOver(e);
        private setHover(li);
        private onMouseLeave(e);
        private removeHover();
        protected isValidLI(li: Element | HTMLElement): boolean;
        protected incrementalSearch(e: ej.base.KeyboardEventArgs): void;
        protected hideSpinner(): void;
        protected showSpinner(): void;
        protected keyActionHandler(e: ej.base.KeyboardEventArgs): void;
        protected selectCurrentItem(e: ej.base.KeyboardEventArgs): void;
        protected isSelectFocusItem(element: Element): boolean;
        private getPageCount();
        private pageUpSelection(steps, event);
        private pageDownSelection(steps, event);
        protected unWireEvent(): void;
        /**
         * Event un binding for list items.
         */
        private unWireListEvents();
        protected onDocumentClick(e: MouseEvent): void;
        private activeStateChange();
        private focusDropDown(e?);
        protected dropDownClick(e: MouseEvent): void;
        protected updateSelectedItem(li: Element, e: MouseEvent | KeyboardEvent | TouchEvent, preventSelect?: boolean): void;
        protected activeItem(li: Element): void;
        protected setValue(e?: ej.base.KeyboardEventArgs): boolean;
        protected setSelection(li: Element, e: MouseEvent | ej.base.KeyboardEventArgs | TouchEvent): void;
        private setValueTemplate();
        protected removeSelection(): void;
        protected getItemData(): {
            [key: string]: string;
        };
        /**
         * To trigger the change event for list.
         */
        protected onChangeEvent(eve: MouseEvent | KeyboardEvent | TouchEvent): void;
        protected detachChangeEvent(eve: MouseEvent | KeyboardEvent | TouchEvent): void;
        protected setHiddenValue(): void;
        /**
         * Filter bar implementation
         */
        protected onFilterUp(e: ej.base.KeyboardEventArgs): void;
        protected onFilterDown(e: ej.base.KeyboardEventArgs): void;
        protected removeFillSelection(): void;
        protected getSelectionPoints(): {
            [key: string]: number;
        };
        protected searchLists(e: ej.base.KeyboardEventArgs): void;
        private filteringAction(dataSource, query?, fields?);
        protected setSearchBox(popupElement: HTMLElement): ej.inputs.InputObject;
        protected onInput(): void;
        protected onActionFailure(e: Object): void;
        protected onActionComplete(ulElement: HTMLElement, list: {
            [key: string]: Object;
        }[], e?: Object, isUpdated?: boolean): void;
        private addNewItem(listData, newElement);
        protected updateActionCompleteData(li: HTMLElement, item: {
            [key: string]: Object;
        }): void;
        private focusIndexItem();
        protected updateSelection(): void;
        protected removeFocus(): void;
        protected renderPopup(): void;
        private checkCollision(popupEle);
        private getOffsetValue(popupEle);
        private createPopup(element, offsetValue, left);
        private isEmptyList();
        protected getFocusElement(): void;
        private isFilterLayout();
        private scrollHandler();
        private setSearchBoxPosition();
        private setPopupPosition(border?);
        private setWidth();
        private scrollBottom(isInitial?);
        private scrollTop();
        protected isEditTextBox(): boolean;
        protected isFiltering(): boolean;
        protected isPopupButton(): boolean;
        protected setScrollPosition(e?: ej.base.KeyboardEventArgs): void;
        private clearText();
        private listScroll();
        private closePopup(delay?);
        private destroyPopup();
        private clickOnBackIcon();
        /**
         * To Initialize the control rendering
         * @private
         */
        render(): void;
        private setFooterTemplate(popupEle);
        protected setOldText(text: string): void;
        protected setOldValue(value: string | number): void;
        protected refreshPopup(): void;
        protected updateDataSource(prop?: string): void;
        /**
         * Dynamically change the value of properties.
         * @private
         */
        onPropertyChanged(newProp: ej.dropdowns.DropDownListModel, oldProp: ej.dropdowns.DropDownListModel): void;
        /**
         * Return the module name.
         * @private
         */
        getModuleName(): string;
        /**
         * Opens the popup that displays the list of items.
         * @returns void.
         */
        showPopup(): void;
        /**
         * Hides the popup if it is in an open state.
         * @returns void.
         */
        hidePopup(): void;
        /**
         * Sets the focus on the component for interaction.
         * @returns void.
         */
        focusIn(): void;
        /**
         * Moves the focus from the component if the component is already focused.
         * @returns void.
         */
        focusOut(): void;
        /**
         * Removes the component from the DOM and detaches all its related event handlers. Also it removes the attributes and classes.
         * @method destroy
         * @return {void}.
         */
        destroy(): void;
    }
    export interface DropDownListClassList {
        root: string;
        hover: string;
        selected: string;
        rtl: string;
        base: string;
        disable: string;
        input: string;
        inputFocus: string;
        li: string;
        icon: string;
        iconAnimation: string;
        value: string;
        focus: string;
        device: string;
        backIcon: string;
        filterBarClearIcon: string;
        filterInput: string;
        filterParent: string;
        mobileFilter: string;
        footer: string;
        header: string;
        clearIcon: string;
        clearIconHide: string;
        popupFullScreen: string;
        disableIcon: string;
    }
    export interface FilteringEventArgs {
        /**
         * Gets the `keyup` event arguments.
         */
        baseEventArgs: Object;
        /**
         * Illustrates whether the current action needs to be prevented or not.
         */
        cancel: boolean;
        /**
         * Search text value.
         */
        text: string;
        /**
         * To filter the data from given data source by using query
         * @param  {Object[] | ej.data.DataManager } dataSource - Set the data source to filter.
         * @param  {Query} query - Specify the query to filter the data.
         * @param  {FieldSettingsModel} fields - Specify the fields to map the column in the data table.
         * @return {void}.
         */
        updateData(dataSource: {
            [key: string]: Object;
        }[] | ej.data.DataManager | string[] | number[], query?: ej.data.Query, fields?: FieldSettingsModel): void;
    }
    export interface PopupEventArgs {
        /**
         * Specifies the popup Object.
         */
        popup: ej.popups.Popup;
        /**
         * Illustrates whether the current action needs to be prevented or not.
         */
        cancel?: boolean;
    }
    
    /**
     * The Multiselect enable CheckBoxSelection call this inject module.
     */
    export class CheckBoxSelection {
        private parent;
        private checkAllParent;
        private selectAllSpan;
        filterInput: HTMLInputElement;
        private filterParent;
        private filterInputObj;
        private backIconElement;
        private clearIconElement;
        private checkWrapper;
        list: HTMLElement;
        private activeLi;
        private activeEle;
        constructor(parent?: IMulitSelect);
        getModuleName(): string;
        addEventListener(): void;
        removeEventListener(): void;
        listOption(args: {
            [key: string]: Object;
        }): void;
        private checboxCreate(e);
        private setSelectAll();
        destroy(): void;
        listSelection(args: IUpdateListArgs): void;
        private validateCheckNode(checkWrap, isCheck, li?, e?, selectAll?);
        private clickHandler(e);
        private changeState(wrapper, state, e?, isPrevent?, selectAll?);
        protected setSearchBox(args: IUpdateListArgs): ej.inputs.InputObject | void;
        private clickOnBackIcon(e);
        private clearText(e);
        private setDeviceSearchBox();
        private setSearchBoxPosition();
        protected targetElement(): string;
        private onBlur(e);
        protected onDocumentClick(e: MouseEvent): void;
        private getFocus(e);
        private checkSelectAll(e);
        private setLocale(unSelect?);
        private getActiveList(args);
        private setReorder(args);
    }
    export interface ItemCreatedArgs {
        curData: {
            [key: string]: Object;
        };
        item: HTMLElement;
        text: string;
    }
    export interface IUpdateListArgs {
        module: string;
        enable: boolean;
        li: HTMLElement;
        e: MouseEvent | ej.base.KeyboardEventArgs;
        popupElement: HTMLElement;
        value: string;
        index: number;
    }
    
    /**
     * Specifies mulitselct interfaces.
     * @hidden
     */
    export interface IMulitSelect extends ej.base.Component<HTMLElement> {
        listCurrentOptions?: {
            [key: string]: Object;
        };
        inputElement?: HTMLInputElement;
        popupWrapper?: HTMLDivElement;
        selectAll?(state?: boolean): void;
        selectAllHeight?: number;
        searchBoxHeight?: number;
        onInput?(): void;
        filterInput?: HTMLInputElement;
        KeyUp?(e?: ej.base.KeyboardEventArgs): void;
        onKeyDown?(e?: ej.base.KeyboardEventArgs): void;
        mainList?: HTMLElement;
        list?: HTMLElement;
        targetElement?(): string;
        targetInputElement?: HTMLInputElement | string;
        selectAllText?: string;
        unSelectAllText?: string;
        popupObj?: ej.popups.Popup;
        onDocumentFocus?: boolean;
        hidePopup?(): void;
        refreshPopup?(): void;
        refreshListItems?(data?: string): void;
        filterBarPlaceholder?: string;
        overAllWrapper?: HTMLDivElement;
        templateList?: {
            [key: string]: Object;
        };
        itemTemplate?: string;
        headerTemplate?: string;
        mobFilter?: boolean;
        header?: HTMLElement;
        updateDelimView?(): void;
        updateValueState?(event?: ej.base.KeyboardEventArgs | MouseEvent, newVal?: [string | number], oldVal?: [string | number]): void;
        tempValues?: [number | string];
        value?: [number | string];
        refreshInputHight?(): void;
        refreshPlaceHolder?(): void;
        ulElement?: HTMLElement;
        hiddenElement?: HTMLSelectElement;
        dispatchEvent?(element?: HTMLElement, type?: string): void;
        inputFocus?: boolean;
        enableSelectionOrder?: boolean;
        focusAtFirstListItem(): void;
        isPopupOpen(): boolean;
        showSelectAll: boolean;
    }
    
    /**
     * Interface for a class MultiSelect
     */
    export interface MultiSelectModel extends DropDownBaseModel{
    
        /**
         * Sets the CSS classes to root element of this component which helps to customize the
         * complete styles.
         * @default null
         */
        cssClass?: string;
    
        /**
         * Gets or sets the width of the component. By default, it sizes based on its parent.
         * container dimension.
         * @default '100%'
         */
        width?: string | number;
    
        /**
         * Gets or sets the height of the popup list. By default it renders based on its list item.
         * > For more details about the popup configuration refer to 
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         * 
         * @default '300px'
         */
        popupHeight?: string | number;
    
        /**
         * Gets or sets the width of the popup list and percentage values has calculated based on input width.
         * > For more details about the popup configuration refer to 
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         * 
         * @default '100%'
         */
        popupWidth?: string | number;
    
        /**
         * Gets or sets the placeholder in the component to display the given information
         * in input when no item selected. 
         * @default null
         */
        placeholder?: string;
    
        /**
         * Accepts the value to be displayed as a watermark text on the filter bar. 
         * @default null.
         */
        filterBarPlaceholder?: string;
    
        /**
         * Gets or sets the additional attribute to `HtmlAttributes` property in MultiSelect,
         * which helps to add attribute like title, name etc, input should be key value pair.
         * 
         * {% codeBlock src="multiselect/html-attributes-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="multiselect/html-attributes-api/index.html" %}{% endcodeBlock %}
         * @default {}
         */
        htmlAttributes?: { [key: string]: string; };
    
        /**
         * Accepts the template design and assigns it to the selected list item in the input element of the component.
         * For more details about the available template options refer to 
         * [`Template`](./templates.html) documentation.
         * 
         * We have built-in [`template engine`](./template-engine.html) 
         * which provides options to compile template string into a executable function. 
         * For EX: We have expression evolution as like ES6 expression string literals.
         * @default null
         */
        valueTemplate?: string;
    
        /**
         * Accepts the template design and assigns it to the header container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         * 
         * @default null
         */
        headerTemplate?: string;
    
        /**
         * Accepts the template design and assigns it to the footer container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         * 
         * @default null
         */
        footerTemplate?: string;
    
        /**
         * Accepts the template design and assigns it to each list item present in the popup.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         * 
         * We have built-in [`template engine`](./template-engine.html) 
         * which provides options to compile template string into a executable function. 
         * For EX: We have expression evolution as like ES6 expression string literals.
         * @default null
         */
        itemTemplate?: string;
    
        /**
         * To enable the filtering option in this component. 
         * Filter action performs when type in search box and collect the matched item through `filtering` event.
         * If searching character does not match, `noRecordsTemplate` property value will be shown.
         * 
         * {% codeBlock src="multiselect/allow-filtering-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="multiselect/allow-filtering-api/index.html" %}{% endcodeBlock %}
         * 
         * @default false
         */
        allowFiltering?: boolean;
    
        /**
         * Allows user to add a 
         * [`custom value`](./custom-value.html), the value which is not present in the suggestion list. 
         * @default false
         */
        allowCustomValue?: boolean;
    
        /**
         * Enables close icon with the each selected item.
         * @default true
         */
        showClearButton?: boolean;
    
        /**
         * Sets limitation to the value selection.
         * based on the limitation, list selection will be prevented.
         * @default 1000
         */
        maximumSelectionLength?: number;
    
        /**
         * Gets or sets the `readonly` to input or not. Once enabled, just you can copy or highlight 
         * the text however tab key action will perform.
         * 
         * @default false
         */
        readonly?: boolean;
    
        /**
         * Selects the list item which maps the data `text` field in the component.
         * @default null
         */
        text?: string;
    
        /**
         * Selects the list item which maps the data `value` field in the component.
         * @default null
         */
        value?: [number | string];
    
        /**
         * Hides the selected item from the list item.
         * @default false
         */
        hideSelectedItem?: boolean;
    
        /**
         * Based on the property, when item get select popup visibility state will changed.
         * @default true
         */
        closePopupOnSelect?: boolean;
    
        /**
         * configures visibility mode for component interaction.
         * 
         *   - `Box` - selected items will be visualized in chip.
         * 
         *   - `Delimiter` - selected items will be visualized in text content.
         * 
         *   - `Default` - on `focus in` component will act in `box` mode.
         *    on `blur` component will act in `delimiter` mode.
         * 
         * {% codeBlock src="multiselect/visual-mode-api/index.ts" %}{% endcodeBlock %}
         * 
         * {% codeBlock src="multiselect/visual-mode-api/index.html" %}{% endcodeBlock %}
         * 
         * @default Default
         */
        mode?: visualMode;
    
        /**
         * Sets the delimiter character for 'default' and 'delimiter' visibility modes.
         * @default ,
         */
        delimiterChar?: string;
    
        /**
         * Sets [`case sensitive`](./filtering.html#case-sensitive-filtering) 
         * option for filter operation.
         * @default false
         */
        ignoreCase?: boolean;
    
        /**
         * Allows you to either show or hide the DropDown button on the component
         * 
         * @default false
         */
        showDropDownIcon?: boolean;
    
        /**
         * Allows you to either show or hide the selectAll option on the component.
         * 
         * @default false
         */
        showSelectAll?: boolean;
    
        /**
         * Specifies the selectAllText to be displayed on the component.
         * 
         * @default 'select All'
         */
        selectAllText?: string;
    
        /**
         * Specifies the UnSelectAllText to be displayed on the component.
         * 
         * @default 'select All'
         */
        unSelectAllText?: string;
    
        /**
         * Reorder the selected items in popup visibility state.
         * 
         * @default true
         */
        enableSelectionOrder?: boolean;
    
        /**
         * Whether to automatically open the popup when the control is clicked.
         * @default true
         */
        openOnClick?: boolean;
    
        /**
         * Fires each time when selection changes happened in list items after model and input value get affected.
         * @event
         */
        change?: ej.base.EmitType<MultiSelectChangeEventArgs>;
    
        /**
         * Fires before the selected item removed from the widget.
         * @event
         */
        removing?: ej.base.EmitType<RemoveEventArgs>;
    
        /**
         * Fires after the selected item removed from the widget.
         * @event
         */
        removed?: ej.base.EmitType<RemoveEventArgs>;
    
        /**
         * Fires when popup opens after animation completion.
         * @event
         */
        open?: ej.base.EmitType<PopupEventArgs>;
    
        /**
         * Fires when popup close after animation completion.
         * @event
         */
        close?: ej.base.EmitType<PopupEventArgs>;
    
        /**
         * Event triggers when the input get focus-out.
         * @event
         */
        blur?: ej.base.EmitType<Object>;
    
        /**
         * Event triggers when the input get focused.
         * @event
         */
        focus?: ej.base.EmitType<Object>;
    
        /**
         * Event triggers when the chip selection.
         * @event
         */
        chipSelection?: ej.base.EmitType<Object>;
    
        /**
         * Triggers event,when user types a text in search box.
         * > For more details about filtering, refer to [`Filtering`](./filtering.html) documentation.
         * 
         * @event
         */
        filtering?: ej.base.EmitType<FilteringEventArgs>;
    
        /**
         * Fires before set the selected item as chip in the component.
         * > For more details about chip customization refer [`Chip Customization`](./chip-customization.html)
         * 
         * @event
         */
        tagging?: ej.base.EmitType<TaggingEventArgs>;
    
        /**
         * Triggers when the [`customValue`](./custom-value.html) is selected.
         * @event
         */
        customValueSelection?: ej.base.EmitType<CustomValueEventArgs>;
    
    }
    
    export interface RemoveEventArgs extends SelectEventArgs {
    }
    /**
     * The Multiselect allows the user to pick a more than one value from list of predefined values.
     * ```html
     * <select id="list">
     *      <option value='1'>Badminton</option>
     *      <option value='2'>Basketball</option>
     *      <option value='3'>Cricket</option>
     *      <option value='4'>Football</option>
     *      <option value='5'>Tennis</option>
     * </select>
     * ```
     * ```typescript
     * <script>
     *   var multiselectObj = new Multiselect();
     *   multiselectObj.appendTo("#list");
     * </script>
     * ```
     */
    export class MultiSelect extends DropDownBase implements ej.inputs.IInput {
        private spinnerElement;
        private selectAllAction;
        private setInitialValue;
        private setDynValue;
        private listCurrentOptions;
        private targetInputElement;
        private selectAllHeight?;
        private searchBoxHeight?;
        private mobFilter?;
        private isFiltered;
        private isFirstClick;
        /**
         * Sets the CSS classes to root element of this component which helps to customize the
         * complete styles.
         * @default null
         */
        cssClass: string;
        /**
         * Gets or sets the width of the component. By default, it sizes based on its parent.
         * container dimension.
         * @default '100%'
         */
        width: string | number;
        /**
         * Gets or sets the height of the popup list. By default it renders based on its list item.
         * > For more details about the popup configuration refer to
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         *
         * @default '300px'
         */
        popupHeight: string | number;
        /**
         * Gets or sets the width of the popup list and percentage values has calculated based on input width.
         * > For more details about the popup configuration refer to
         * [`Popup Configuration`](./getting-started.html#configure-the-popup-list) documentation.
         *
         * @default '100%'
         */
        popupWidth: string | number;
        /**
         * Gets or sets the placeholder in the component to display the given information
         * in input when no item selected.
         * @default null
         */
        placeholder: string;
        /**
         * Accepts the value to be displayed as a watermark text on the filter bar.
         * @default null.
         */
        filterBarPlaceholder: string;
        /**
         * Gets or sets the additional attribute to `HtmlAttributes` property in MultiSelect,
         * which helps to add attribute like title, name etc, input should be key value pair.
         *
         * {% codeBlock src="multiselect/html-attributes-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="multiselect/html-attributes-api/index.html" %}{% endcodeBlock %}
         * @default {}
         */
        htmlAttributes: {
            [key: string]: string;
        };
        /**
         * Accepts the template design and assigns it to the selected list item in the input element of the component.
         * For more details about the available template options refer to
         * [`Template`](./templates.html) documentation.
         *
         * We have built-in [`template engine`](./template-engine.html)
         * which provides options to compile template string into a executable function.
         * For EX: We have expression evolution as like ES6 expression string literals.
         * @default null
         */
        valueTemplate: string;
        /**
         * Accepts the template design and assigns it to the header container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         *
         * @default null
         */
        headerTemplate: string;
        /**
         * Accepts the template design and assigns it to the footer container of the popup list.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         *
         * @default null
         */
        footerTemplate: string;
        /**
         * Accepts the template design and assigns it to each list item present in the popup.
         * > For more details about the available template options refer to [`Template`](./templates.html) documentation.
         *
         * We have built-in [`template engine`](./template-engine.html)
         * which provides options to compile template string into a executable function.
         * For EX: We have expression evolution as like ES6 expression string literals.
         * @default null
         */
        itemTemplate: string;
        /**
         * To enable the filtering option in this component.
         * Filter action performs when type in search box and collect the matched item through `filtering` event.
         * If searching character does not match, `noRecordsTemplate` property value will be shown.
         *
         * {% codeBlock src="multiselect/allow-filtering-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="multiselect/allow-filtering-api/index.html" %}{% endcodeBlock %}
         *
         * @default false
         */
        allowFiltering: boolean;
        /**
         * Allows user to add a
         * [`custom value`](./custom-value.html), the value which is not present in the suggestion list.
         * @default false
         */
        allowCustomValue: boolean;
        /**
         * Enables close icon with the each selected item.
         * @default true
         */
        showClearButton: boolean;
        /**
         * Sets limitation to the value selection.
         * based on the limitation, list selection will be prevented.
         * @default 1000
         */
        maximumSelectionLength: number;
        /**
         * Gets or sets the `readonly` to input or not. Once enabled, just you can copy or highlight
         * the text however tab key action will perform.
         *
         * @default false
         */
        readonly: boolean;
        /**
         * Selects the list item which maps the data `text` field in the component.
         * @default null
         */
        text: string;
        /**
         * Selects the list item which maps the data `value` field in the component.
         * @default null
         */
        value: [number | string];
        /**
         * Hides the selected item from the list item.
         * @default false
         */
        hideSelectedItem: boolean;
        /**
         * Based on the property, when item get select popup visibility state will changed.
         * @default true
         */
        closePopupOnSelect: boolean;
        /**
         * configures visibility mode for component interaction.
         *
         *   - `Box` - selected items will be visualized in chip.
         *
         *   - `Delimiter` - selected items will be visualized in text content.
         *
         *   - `Default` - on `focus in` component will act in `box` mode.
         *    on `blur` component will act in `delimiter` mode.
         *
         * {% codeBlock src="multiselect/visual-mode-api/index.ts" %}{% endcodeBlock %}
         *
         * {% codeBlock src="multiselect/visual-mode-api/index.html" %}{% endcodeBlock %}
         *
         * @default Default
         */
        mode: visualMode;
        /**
         * Sets the delimiter character for 'default' and 'delimiter' visibility modes.
         * @default ,
         */
        delimiterChar: string;
        /**
         * Sets [`case sensitive`](./filtering.html#case-sensitive-filtering)
         * option for filter operation.
         * @default false
         */
        ignoreCase: boolean;
        /**
         * Allows you to either show or hide the DropDown button on the component
         *
         * @default false
         */
        showDropDownIcon: boolean;
        /**
         * Allows you to either show or hide the selectAll option on the component.
         *
         * @default false
         */
        showSelectAll: boolean;
        /**
         * Specifies the selectAllText to be displayed on the component.
         *
         * @default 'select All'
         */
        selectAllText: string;
        /**
         * Specifies the UnSelectAllText to be displayed on the component.
         *
         * @default 'select All'
         */
        unSelectAllText: string;
        /**
         * Reorder the selected items in popup visibility state.
         *
         * @default true
         */
        enableSelectionOrder: boolean;
        /**
         * Whether to automatically open the popup when the control is clicked.
         * @default true
         */
        openOnClick: boolean;
        /**
         * Fires each time when selection changes happened in list items after model and input value get affected.
         * @event
         */
        change: ej.base.EmitType<MultiSelectChangeEventArgs>;
        /**
         * Fires before the selected item removed from the widget.
         * @event
         */
        removing: ej.base.EmitType<RemoveEventArgs>;
        /**
         * Fires after the selected item removed from the widget.
         * @event
         */
        removed: ej.base.EmitType<RemoveEventArgs>;
        /**
         * Fires when popup opens after animation completion.
         * @event
         */
        open: ej.base.EmitType<PopupEventArgs>;
        /**
         * Fires when popup close after animation completion.
         * @event
         */
        close: ej.base.EmitType<PopupEventArgs>;
        /**
         * Event triggers when the input get focus-out.
         * @event
         */
        blur: ej.base.EmitType<Object>;
        /**
         * Event triggers when the input get focused.
         * @event
         */
        focus: ej.base.EmitType<Object>;
        /**
         * Event triggers when the chip selection.
         * @event
         */
        chipSelection: ej.base.EmitType<Object>;
        /**
         * Triggers event,when user types a text in search box.
         * > For more details about filtering, refer to [`Filtering`](./filtering.html) documentation.
         *
         * @event
         */
        filtering: ej.base.EmitType<FilteringEventArgs>;
        /**
         * Fires before set the selected item as chip in the component.
         * > For more details about chip customization refer [`Chip Customization`](./chip-customization.html)
         *
         * @event
         */
        tagging: ej.base.EmitType<TaggingEventArgs>;
        /**
         * Triggers when the [`customValue`](./custom-value.html) is selected.
         * @event
         */
        customValueSelection: ej.base.EmitType<CustomValueEventArgs>;
        /**
         * Constructor for creating the DropDownList widget.
         */
        constructor(option?: MultiSelectModel, element?: string | HTMLElement);
        private isValidKey;
        private mainList;
        ulElement: HTMLElement;
        private mainData;
        private mainListCollection;
        private customValueFlag;
        private inputElement;
        private componentWrapper;
        private overAllWrapper;
        private searchWrapper;
        private viewWrapper;
        private chipCollectionWrapper;
        private overAllClear;
        private dropIcon;
        private hiddenElement;
        private delimiterWrapper;
        private popupObj;
        private inputFocus;
        private header;
        private footer;
        private initStatus;
        private popupWrapper;
        private keyCode;
        private beforePopupOpen;
        private remoteCustomValue;
        private filterAction;
        private enableRTL(state);
        requiredModules(): ej.base.ModuleDeclaration[];
        private updateHTMLAttribute();
        private updateReadonly(state);
        private updateClearButton(state);
        private updateCssClass();
        private onPopupShown();
        private focusAtFirstListItem();
        private focusAtLastListItem(data);
        protected getAriaAttributes(): {
            [key: string]: string;
        };
        private updateListARIA();
        private removelastSelection(e);
        protected onActionFailure(e: Object): void;
        protected targetElement(): string;
        protected onActionComplete(ulElement: HTMLElement, list: {
            [key: string]: Object;
        }[], e?: Object, isUpdated?: boolean): void;
        private refreshSelection();
        private checkSelectAll();
        private openClick(e);
        private KeyUp(e);
        protected getQuery(query: ej.data.Query): ej.data.Query;
        private dataUpdater(dataSource, query?, fields?);
        private tempQuery;
        private tempValues;
        private checkForCustomValue(query?, fields?);
        protected getNgDirective(): string;
        private wrapperClick(e);
        private enable(state);
        private scrollFocusStatus;
        private keyDownStatus;
        private onBlur(eve);
        private refreshInputHight();
        private validateValues(newValue, oldValue);
        private updateValueState(event, newVal, oldVal);
        private getPagingCount();
        private pageUpSelection(steps);
        private pageDownSelection(steps);
        private focusIn();
        private showDelimWrapper();
        private hideDelimWrapper();
        private expandTextbox();
        private isPopupOpen();
        private refreshPopup();
        private checkTextLength();
        private popupKeyActions(keyCode);
        private updateAriaAttribute();
        private onKeyDown(e);
        private selectByKey(e);
        private escapeAction();
        private scrollBottom(selectedLI, activeIndex);
        private scrollTop(selectedLI, activeIndex);
        private selectListByKey(e);
        private refreshListItems(data);
        private removeSelectedChip(e);
        private moveByTop(state);
        private moveByList(position);
        private moveBy(position);
        private chipClick(e);
        private removeChipSelection();
        private addChipSelection(element);
        private onChipRemove(e);
        private makeTextBoxEmpty();
        private refreshPlaceHolder();
        private removeValue(value, eve);
        private updateMainList(state, value);
        private removeChip(value);
        private updateChipStatus();
        private addValue(value, text, eve);
        private checkMaxSelection();
        private dispatchSelect(value, eve, element, isNotTrigger);
        private addChip(text, value, e?);
        private removeChipFocus();
        private onMobileChipInteraction(e);
        private getChip(data, value, e?);
        private calcPopupWidth();
        private mouseIn();
        private mouseOut();
        protected listOption(dataSource: {
            [key: string]: Object;
        }[], fields: FieldSettingsModel): {
            [key: string]: Object;
        };
        private renderPopup();
        private ClearAll(e);
        private windowResize();
        protected wireEvent(): void;
        private onInput();
        protected preRender(): void;
        private initializeData();
        private updateData(delimiterChar);
        private initialValueUpdate();
        protected isValidLI(li: Element | HTMLElement): boolean;
        protected updateListSelection(li: Element, e: MouseEvent | ej.base.KeyboardEventArgs): void;
        protected removeListSelection(): void;
        private removeHover();
        private removeFocus();
        private addListHover(li);
        private addListFocus(element);
        private addListSelection(element);
        private updateDelimeter(delimChar);
        private onMouseClick(e);
        private onMouseOver(e);
        private onMouseLeave(e);
        private onListMouseDown(e);
        private wireListEvents();
        private unwireListEvents();
        private hideOverAllClear();
        private showOverAllClear();
        protected showSpinner(): void;
        protected hideSpinner(): void;
        private updateDelimView();
        private unWireEvent();
        private selectAllItem(state);
        protected setZIndex(): void;
        protected updateDataSource(prop?: string): void;
        private onLoadSelect();
        /**
         * Get the properties to be maintained in the persisted state.
         */
        protected getPersistData(): string;
        /**
         * Dynamically change the value of properties.
         * @private
         */
        onPropertyChanged(newProp: MultiSelectModel, oldProp: MultiSelectModel): void;
        /**
         * Hides the popup, if the popup in a open state.
         * @returns void
         */
        hidePopup(): void;
        /**
         * Shows the popup, if the popup in a closed state.
         * @returns void
         */
        showPopup(): void;
        /**
         * Based on the state parameter, entire list item will be selected/deselected.
         * parameter
         * `true`   - Selects entire list items.
         * `false`  - Un Selects entire list items.
         * @returns void
         */
        selectAll(state: boolean): void;
        getModuleName(): string;
        /**
         * To Initialize the control rendering
         * @private
         */
        render(): void;
        private dropDownIcon();
        private initialUpdate();
        /**
         * Removes the component from the DOM and detaches all its related event handlers. Also it removes the attributes and classes.
         * @method destroy
         * @return {void}
         */
        destroy(): void;
    }
    export interface CustomValueEventArgs {
        /**
         * Gets the newly added data.
         */
        newData: Object;
        /**
         * Illustrates whether the current action needs to be prevented or not.
         */
        cancel: boolean;
    }
    export interface TaggingEventArgs {
        /**
         * If the event is triggered by interaction, it returns true. Otherwise, it returns false.
         */
        isInteracted: boolean;
        /**
         * Returns the selected item as JSON Object from the data source.
         */
        itemData: FieldSettingsModel;
        /**
         * Specifies the original event arguments.
         */
        e: MouseEvent | KeyboardEvent | TouchEvent;
        /**
         * To set the classes to chip element
         * @param  { string } classes - Specify the classes to chip element.
         * @return {void}.
         */
        setClass: Function;
    }
    export interface MultiSelectChangeEventArgs {
        /**
         * If the event is triggered by interaction, it returns true. Otherwise, it returns false.
         */
        isInteracted: boolean;
        /**
         * Returns the component initial Value.
         */
        oldValue: number[] | string[];
        /**
         * Returns the updated component Values.
         */
        value: number[] | string[];
        /**
         * Specifies the original event arguments.
         */
        e: MouseEvent | KeyboardEvent | TouchEvent;
    }
    export type visualMode = 'Default' | 'Delimiter' | 'Box' | 'CheckBox';
    
  }
  export namespace excelexport {
    
    /**
     * BlobHelper class
     */
    export class BlobHelper {
        private parts;
        private blob;
        append(part: any): void;
        getBlob(): Blob;
    }
    
    /**
     * CellStyle class
     */
    export class CellStyle {
        name: string;
        index: number;
        backColor: string;
        numFmtId: number;
        borders: Borders;
        fontName: string;
        fontSize: number;
        fontColor: string;
        italic: boolean;
        bold: boolean;
        hAlign: HAlignType;
        vAlign: VAlignType;
        underline: boolean;
        wrapText: boolean;
        numberFormat: string;
        type: string;
        isGlobalStyle: boolean;
        constructor();
    }
    export class Font {
        b: boolean;
        i: boolean;
        u: boolean;
        sz: number;
        name: string;
        color: string;
        constructor();
    }
    export class CellXfs {
        numFmtId: number;
        fontId: number;
        fillId: number;
        borderId: number;
        xfId: number;
        applyAlignment: number;
        alignment: Alignment;
    }
    export class Alignment {
        horizontal: string;
        vertical: string;
        wrapText: number;
    }
    export class CellStyleXfs {
        numFmtId: number;
        fontId: number;
        fillId: number;
        borderId: number;
        applyAlignment: number;
        alignment: Alignment;
    }
    export class CellStyles {
        name: string;
        xfId: number;
        constructor();
    }
    export class NumFmt {
        numFmtId: number;
        formatCode: string;
        constructor();
        constructor(id: number, code: string);
    }
    /**
     * Border class
     */
    export class Border {
        lineStyle: LineStyle;
        color: string;
        constructor();
        constructor(mLine: LineStyle, mColor: string);
    }
    /**
     * Border class
     */
    export class Borders {
        left: Border;
        right: Border;
        bottom: Border;
        top: Border;
        all: Border;
        constructor();
    }
    
    /**
     * Worksheet class
     */
    export class Cell {
        index: number;
        rowSpan: number;
        colSpan: number;
        value: string | Date | number | boolean;
        cellStyle: CellStyle;
        styleIndex: number;
        sharedStringIndex: number;
        saveType: string;
        type: string;
        refName: string;
    }
    /**
     * Rows class
     */
    export class Cells extends Array {
        add: (cell: Cell) => void;
    }
    
    /**
     * Column class
     */
    export class Column {
        index: number;
        width: number;
    }
    
    /**
     * CsvHelper class
     */
    export class CsvHelper {
        private isMicrosoftBrowser;
        private buffer;
        private csvStr;
        private formatter;
        private globalStyles;
        constructor(json: any);
        private parseWorksheet(json);
        private parseRows(rows);
        private parseRow(row);
        private parseCell(cell);
        private parseCellValue(value);
        /**
         * Saves the file with specified name and sends the file to client browser
         * @param  {string} fileName- file name to save.
         * @param  {Blob} buffer- the content to write in file
         */
        save(fileName: string): void;
        saveAsBlob(): Blob;
    }
    
    /**
     * LineStyle
     */
    export type LineStyle = 'thin' | 'thick' | 'medium' | 'none';
    export type HAlignType = 'center ' | 'justify' | 'left' | 'right' | 'general';
    export type VAlignType = 'bottom' | 'center' | 'top';
    export type HyperLinkType = 'none' | 'url' | 'file' | 'unc' | 'workbook';
    export type SaveType = 'xlsx' | 'csv';
    export type CellType = 'b' | 'e' | 'inlineStr' | 'n' | 's' | 'str';
    export type BlobSaveType = 'text/csv' | 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
    
    /**
     * Worksheet class
     */
    export class Row {
        height: number;
        index: number;
        cells: Cells;
        spans: string;
        grouping: Grouping;
    }
    /**
     * Rows class
     */
    export class Rows extends Array {
        add: (row: Row) => void;
    }
    
    /**
     * ValueFormatter class to globalize the value.
     * @hidden
     */
    export class ValueFormatter {
        private intl;
        constructor(cultureName?: string);
        getFormatFunction(format: ej.base.NumberFormatOptions | ej.base.DateFormatOptions): Function;
        toView(value: number | Date, format: Function): string | Object;
        displayText(value: any, format: ej.base.NumberFormatOptions | ej.base.DateFormatOptions): string;
    }
    
    /**
     * Workbook class
     */
    export class Workbook {
        private mArchive;
        private sharedString;
        private sharedStringCount;
        cellStyles: any;
        private worksheets;
        private builtInProperties;
        private mFonts;
        private mBorders;
        private mFills;
        private mNumFmt;
        private mStyles;
        private mCellXfs;
        private mCellStyleXfs;
        private mergeCells;
        private csvHelper;
        private mSaveType;
        private mHyperLinks;
        private unitsProportions;
        private hyperlinkStyle;
        private printTitles;
        private culture;
        private intl;
        private globalStyles;
        constructor(json: any, saveType: SaveType, culture?: string);
        private parserBuiltInProperties(jsonBuiltInProperties, builtInProperties);
        private parserWorksheets(json);
        private mergeOptions(fromJson, toJson);
        private applyProperties(sourceJson, destJson);
        private getCellName(row, column);
        private getColumnName(col);
        private parserPrintTitle(json, sheet);
        private parserFreezePanes(json, sheet);
        private parserColumns(json, sheet);
        private parserRows(json, sheet);
        private parserRow(json, rowIndex);
        private parseGrouping(json, row);
        private parseCells(json, row);
        private applyGlobalStyle(json, cellStyle);
        private compareStyleXfs(toCompareStyle);
        private parserCellStyle(json, cellStyle, cellType);
        private parserCellStyle(json, cellStyle, cellType, defStyleIndex);
        private switchNumberFormat(numberFormat, type);
        private getNumberFormat(numberFormat, type);
        private parserBorder(json, border);
        private processCellStyle(style);
        private isNewFont(toCompareStyle);
        private isNewBorder(toCompareStyle);
        private isAllBorder(toCompareBorder);
        private compareStyle(toCompareStyle);
        private contains(array, item);
        private getCellValueType(value);
        private parseCellType(cell);
        saveAsBlob(blobSaveType: BlobSaveType): Promise<{
            blobData: Blob;
        }>;
        save(fileName: string, proxyUrl?: string): void;
        private saveInternal();
        private saveWorkbook();
        private saveWorksheets();
        private saveWorksheet(sheet, index);
        private pixelsToColumnWidth(pixels);
        private trunc(x);
        private pixelsToRowHeight(pixels);
        private saveSheetRelations(sheet);
        private saveFreezePanes(sheet);
        private saveSharedString();
        private processString(value);
        private saveStyles();
        private saveNumberFormats();
        private saveFonts();
        private saveFills();
        private saveBorders();
        private saveCellStyles();
        private saveCellStyleXfs();
        private saveCellXfs();
        private saveApp(builtInProperties);
        private saveCore(builtInProperties);
        private saveTopLevelRelation();
        private saveWorkbookRelation();
        private saveContentType();
        private addToArchive(xmlString, itemName);
        private processMergeCells(cell, rowIndex, mergeCells);
        /**
         * Returns the tick count corresponding to the given year, month, and day.
         * @param year number value of year
         * @param month number value of month
         * @param day number value of day
         */
        private dateToTicks(year, month, day);
        /**
         * Return the tick count corresponding to the given hour, minute, second.
         * @param hour number value of hour
         * @param minute number value if minute
         * @param second number value of second
         */
        private timeToTicks(hour, minute, second);
        /**
         * Checks if given year is a leap year.
         * @param year Year value.
         */
        isLeapYear(year: number): boolean;
        /**
         * Converts `DateTime` to the equivalent OLE Automation date.
         */
        private toOADate(date);
    }
    export class BuiltInProperties {
        author: string;
        comments: string;
        category: string;
        company: string;
        manager: string;
        subject: string;
        title: string;
        createdDate: Date;
        modifiedDate: Date;
        tags: string;
        status: string;
    }
    
    /**
     * Worksheet class
     */
    export class Worksheet {
        isSummaryRowBelow: boolean;
        index: number;
        columns: Column[];
        rows: Rows;
        freezePanes: FreezePane;
        name: string;
        mergeCells: MergeCells;
        hyperLinks: HyperLink[];
    }
    export class HyperLink {
        ref: string;
        rId: number;
        toolTip: string;
        location: string;
        display: string;
        target: string;
        type: HyperLinkType;
    }
    export class Grouping {
        outlineLevel: number;
        isCollapsed: boolean;
        isHidden: boolean;
    }
    export class FreezePane {
        row: number;
        column: number;
        leftCell: string;
    }
    export class MergeCell {
        ref: string;
        x: number;
        width: number;
        y: number;
        height: number;
    }
    /**
     * MergeCells class
     */
    export class MergeCells extends Array {
        add: (mergeCell: MergeCell) => void;
        static isIntersecting(base: MergeCell, compare: MergeCell): boolean;
    }
    
    /**
     * Worksheets class
     */
    export class Worksheets extends Array<Worksheet> {
    }
    
  }
  export namespace fileutils {
    
    /**
     * Encoding class: Contains the details about encoding type, whether to write a Unicode byte order mark (BOM).
     * ```typescript
     * let encoding : Encoding = new Encoding();
     * encoding.type = 'Utf8';
     * encoding.getBytes('Encoding', 0, 5);
     * ```
     */
    export class Encoding {
        private emitBOM;
        private encodingType;
        /**
         * Gets a value indicating whether to write a Unicode byte order mark
         * @returns boolean- true to specify that a Unicode byte order mark is written; otherwise, false
         */
        readonly includeBom: boolean;
        /**
         * Gets the encoding type.
         * @returns EncodingType
         */
        /**
         * Sets the encoding type.
         * @param  {EncodingType} value
         */
        type: EncodingType;
        /**
         * Initializes a new instance of the Encoding class. A parameter specifies whether to write a Unicode byte order mark
         * @param  {boolean} includeBom?-true to specify that a Unicode byte order mark is written; otherwise, false.
         */
        constructor(includeBom?: boolean);
        /**
         * Initialize the includeBom to emit BOM or Not
         * @param  {boolean} includeBom
         */
        private initBOM(includeBom);
        /**
         * Calculates the number of bytes produced by encoding the characters in the specified string
         * @param  {string} chars - The string containing the set of characters to encode
         * @returns {number} - The number of bytes produced by encoding the specified characters
         */
        getByteCount(chars: string): number;
        /**
         * Return the Byte of character
         * @param  {number} codePoint
         * @returns {number}
         */
        private utf8Len(codePoint);
        /**
         * for 4 byte character return surrogate pair true, otherwise false
         * @param  {number} codeUnit
         * @returns {boolean}
         */
        private isHighSurrogate(codeUnit);
        /**
         * for 4byte character generate the surrogate pair
         * @param  {number} highCodeUnit
         * @param  {number} lowCodeUnit
         */
        private toCodepoint(highCodeUnit, lowCodeUnit);
        /**
         * private method to get the byte count for specific charindex and count
         * @param  {string} chars
         * @param  {number} charIndex
         * @param  {number} charCount
         */
        private getByteCountInternal(chars, charIndex, charCount);
        /**
         * Encodes a set of characters from the specified string into the ArrayBuffer.
         * @param  {string} s- The string containing the set of characters to encode
         * @param  {number} charIndex-The index of the first character to encode.
         * @param  {number} charCount- The number of characters to encode.
         * @returns {ArrayBuffer} - The ArrayBuffer that contains the resulting sequence of bytes.
         */
        getBytes(s: string, charIndex: number, charCount: number): ArrayBuffer;
        /**
         * Decodes a sequence of bytes from the specified ArrayBuffer into the string.
         * @param  {ArrayBuffer} bytes- The ArrayBuffer containing the sequence of bytes to decode.
         * @param  {number} index- The index of the first byte to decode.
         * @param  {number} count- The number of bytes to decode.
         * @returns {string} - The string that contains the resulting set of characters.
         */
        getString(bytes: ArrayBuffer, index: number, count: number): string;
        private getBytesOfAnsiEncoding(byteCount, s, charIndex, charCount);
        private getBytesOfUtf8Encoding(byteCount, s, charIndex, charCount);
        private getBytesOfUnicodeEncoding(byteCount, s, charIndex, charCount);
        private getStringOfUtf8Encoding(byteCal, index, count);
        private getStringofUnicodeEncoding(byteUni, index, count);
        /**
         * To clear the encoding instance
         * @return {void}
         */
        destroy(): void;
    }
    /**
     * EncodingType : Specifies the encoding type
     */
    export type EncodingType = 'Ansi' | 'Utf8' | 'Unicode';
    /**
     * To check the object is null or undefined and throw error if it is null or undefined
     * @param {Object} value - object to check is null or undefined
     * @return {boolean}
     * @throws {ArgumentException} - if the value is null or undefined
     * @private
     */
    export function validateNullOrUndefined(value: Object, message: string): void;
    
    /**
     * Save class provide method to save file
     * ```typescript
     * let blob : Blob = new Blob([''], { type: 'text/plain' });
     * Save.save('fileName.txt',blob);
     */
    export class Save {
        static isMicrosoftBrowser: boolean;
        /**
         * Initialize new instance of {save}
         */
        constructor();
        /**
         * Saves the file with specified name and sends the file to client browser
         * @param  {string} fileName- file name to save.
         * @param  {Blob} buffer- the content to write in file
         * @param  {boolean} isMicrosoftBrowser- specify whether microsoft browser or not
         * @returns {void}
         */
        static save(fileName: string, buffer: Blob): void;
        private static saveInternal(fileName, extension, buffer, downloadLink, hasDownloadAttribute);
        /**
         *
         * @param {string} extension - get mime type of the specified extension
         * @private
         */
        static getMimeType(extension: string): string;
    }
    
    /**
     * StreamWriter class contains the implementation for writing characters to a file in a particular encoding
     * ```typescript
     * let writer = new StreamWriter();
     * writer.write('Hello World');
     * writer.save('Sample.txt');
     * writer.dispose();
     * ```
     */
    export class StreamWriter {
        private bufferBlob;
        private bufferText;
        private enc;
        /**
         * Gets the content written to the StreamWriter as Blob.
         * @returns Blob
         */
        readonly buffer: Blob;
        /**
         * Gets the encoding.
         * @returns Encoding
         */
        readonly encoding: Encoding;
        /**
         * Initializes a new instance of the StreamWriter class by using the specified encoding.
         * @param  {Encoding} encoding?- The character encoding to use.
         */
        constructor(encoding?: Encoding);
        private init(encoding);
        /**
         * Private method to set Byte Order Mark(BOM) value based on EncodingType
         */
        private setBomByte();
        /**
         * Saves the file with specified name and sends the file to client browser
         * @param  {string} fileName - The file name to save
         * @returns {void}
         */
        save(fileName: string): void;
        /**
         * Writes the specified string.
         * @param  {string} value - The string to write. If value is null or undefined, nothing is written.
         * @returns {void}
         */
        write(value: string): void;
        private flush();
        /**
         * Writes the specified string followed by a line terminator
         * @param  {string} value - The string to write. If value is null or undefined, nothing is written
         * @returns {void}
         */
        writeLine(value: string): void;
        /**
         * Releases the resources used by the StreamWriter
         * @returns {void}
         */
        destroy(): void;
    }
    
  }
  export namespace grids {
    
    /**
     * Summary Action controller.
     */
    export class Aggregate implements IAction {
        private parent;
        private locator;
        private footerRenderer;
        constructor(parent: IGrid, locator?: ServiceLocator);
        getModuleName(): string;
        private initiateRender();
        private prepareSummaryInfo();
        private getFormatFromType(format, type);
        onPropertyChanged(e: NotifyArgs): void;
        addEventListener(): void;
        removeEventListener(): void;
        destroy(): void;
    }
    /**
     * @private
     */
    export function summaryIterator(aggregates: AggregateRowModel[], callback: Function): void;
    
    /**
     * `BatchEdit` module is used to handle batch editing actions.
     * @hidden
     */
    export class BatchEdit {
        private parent;
        private serviceLocator;
        private form;
        formObj: ej.inputs.FormValidator;
        private renderer;
        private focus;
        private dataBoundFunction;
        private cellDetails;
        private isColored;
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator, renderer?: EditRender);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private dataBound();
        /**
         * @hidden
         */
        destroy(): void;
        protected clickHandler(e: MouseEvent): void;
        protected dblClickHandler(e: MouseEvent): void;
        private onBeforeCellFocused(e);
        private onCellFocused(e);
        private isAddRow(index);
        private editCellFromIndex(rowIdx, cellIdx);
        closeEdit(): void;
        deleteRecord(fieldname?: string, data?: Object): void;
        addRecord(data?: Object): void;
        endEdit(data?: Object): void;
        private validateFormObj();
        batchSave(): void;
        getBatchChanges(): Object;
        private mergeBatchChanges(row, mRow, frzCols);
        /**
         * @hidden
         */
        removeRowObjectFromUID(uid: string): void;
        /**
         * @hidden
         */
        addRowObject(row: Row<Column>): void;
        private bulkDelete(fieldname?, data?);
        private refreshRowIdx();
        private getIndexFromData(data);
        private bulkAddRow(data?);
        private renderMovable(ele);
        private findNextEditableCell(columnIndex, isAdd);
        private checkNPCell(col);
        private getDefaultData();
        private setCellIdx(target);
        editCell(index: number, field: string, isAdd?: boolean): void;
        updateCell(rowIndex: number, field: string, value: string | number | boolean | Date): void;
        private setChanges(rowObj, field, value);
        updateRow(index: number, data: Object): void;
        private getCellIdx(uid);
        private refreshTD(td, column, rowObj, value);
        private getColIndex(cells, index);
        saveCell(isForceSave?: boolean): void;
        protected getDataByIndex(index: number): Object;
    }
    
    /**
     * @hidden
     * `CheckBoxFilter` module is used to handle filtering action.
     */
    export class CheckBoxFilter {
        protected sBox: HTMLElement;
        protected isExcel: boolean;
        protected id: string;
        protected colType: string;
        protected fullData: Object[];
        protected filteredData: Object[];
        protected isFiltered: boolean | number;
        protected dlg: Element;
        protected dialogObj: ej.popups.Dialog;
        protected cBox: HTMLElement;
        protected spinner: HTMLElement;
        protected searchBox: Element;
        protected sInput: HTMLInputElement;
        protected sIcon: Element;
        protected options: IFilterArgs;
        protected filterSettings: FilterSettings;
        protected existingPredicate: {
            [key: string]: PredicateModel[];
        };
        protected foreignKeyData: Object[];
        protected filterState: boolean;
        protected defaultConstants: Object;
        protected values: Object;
        private cBoxTrue;
        private cBoxFalse;
        private itemsCnt;
        private result;
        protected parent: IGrid;
        protected serviceLocator: ServiceLocator;
        protected localeObj: ej.base.L10n;
        protected valueFormatter: ValueFormatter;
        /**
         * Constructor for checkbox filtering module
         * @hidden
         */
        constructor(parent?: IGrid, filterSettings?: FilterSettings, serviceLocator?: ServiceLocator);
        protected initLocale(constants: Object): void;
        /**
         * To destroy the filter bar.
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private wireEvents();
        private unWireEvents();
        private searchBoxClick(e);
        private searchBoxKeyUp(e?);
        private updateSearchIcon();
        /**
         * Gets the localized label by locale keyword.
         * @param  {string} key
         * @return {string}
         */
        getLocalizedLabel(key: string): string;
        private updateDataSource();
        protected updateModel(options: IFilterArgs): void;
        protected getAndSetChkElem(options: IFilterArgs): HTMLElement;
        protected showDialog(options: IFilterArgs): void;
        private dialogCreated(e);
        openDialog(options: IFilterArgs): void;
        closeDialog(): void;
        protected clearFilter(): void;
        private btnClick(e);
        private fltrBtnHandler();
        private initiateFilter(fColl);
        private refreshCheckboxes();
        private getPredicateFromCols(columns);
        private getAllData();
        private dataSuccess(e);
        private processDataSource(query?, isInitial?);
        private updateResult();
        private clickHandler(e);
        private updateAllCBoxes(checked);
        private dialogOpen();
        private createCheckbox(value, checked);
        private updateIndeterminatenBtn();
        private createFilterItems(data, isInitial?);
        private getCheckedState(isColFiltered, value);
        static getDistinct(json: Object[], field: string, column?: Column, foreignKeyData?: Object[]): Object;
        static getPredicate(columns: PredicateModel[]): Predicate;
        private static generatePredicate(cols);
        private static getCaseValue(filter);
        private static updateDateFilter(filter);
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    /**
     * The `Clipboard` module is used to handle clipboard copy action.
     */
    export class Clipboard implements IAction {
        private clipBoardTextArea;
        private copyContent;
        private isSelect;
        private parent;
        /**
         * Constructor for the Grid clipboard module
         * @hidden
         */
        constructor(parent?: IGrid);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private initialEnd();
        private keyDownHandler(e);
        private setCopyData(withHeader?);
        private getCopyData(cells, isCell, splitKey, withHeader?);
        /**
         * Copy selected rows or cells data into clipboard.
         * @param {boolean} withHeader - Specifies whether the column header data need to be copied or not.
         */
        copy(withHeader?: boolean): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        /**
         * To destroy the clipboard
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private checkBoxSelection();
    }
    
    /**
     * The `ColumnChooser` module is used to show or hide columns dynamically.
     */
    export class ColumnChooser implements IAction {
        private dataManager;
        private column;
        private parent;
        private serviceLocator;
        private l10n;
        private dlgObj;
        private searchValue;
        private flag;
        private timer;
        getShowHideService: ShowHide;
        private showColumn;
        private hideColumn;
        private mainDiv;
        private innerDiv;
        private ulElement;
        private isDlgOpen;
        private dlghide;
        private initialOpenDlg;
        private stateChangeColumns;
        private dlgDiv;
        private isInitialOpen;
        private isCustomizeOpenCC;
        private cBoxTrue;
        private cBoxFalse;
        private searchBoxObj;
        /**
         * Constructor for the Grid ColumnChooser module
         * @hidden
         */
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator);
        private destroy();
        private rtlUpdate();
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private render();
        private clickHandler(e);
        private hideDialog();
        /**
         * To render columnChooser when showColumnChooser enabled.
         * @return {void}
         * @hidden
         */
        renderColumnChooser(x?: number, y?: number, target?: Element): void;
        /**
         * Column chooser can be displayed on screen by given position(X and Y axis).
         * @param  {number} X - Defines the X axis.
         * @param  {number} Y - Defines the Y axis.
         * @return {void}
         */
        openColumnChooser(X?: number, Y?: number): void;
        private enableAfterRenderEle(e);
        private customDialogOpen();
        private customDialogClose();
        private getColumns();
        private renderDlgContent();
        private renderChooserList();
        private confirmDlgBtnClick(args);
        private columnStateChange(stateColumns, state);
        private clearActions();
        private checkstatecolumn(isChecked, coluid);
        private columnChooserSearch(searchVal);
        private wireEvents();
        private unWireEvents();
        private checkBoxClickHandler(e);
        private refreshCheckboxList(gdCol, searchVal?);
        private refreshCheckboxState();
        private checkState(element, state);
        private createCheckBox(label, checked, uid);
        private renderCheckbox(column);
        private columnChooserManualSearch(e);
        private startTimer(e);
        private stopTimer();
        private addcancelIcon();
        private removeCancelIcon();
        private mOpenDlg();
        private getModuleName();
    }
    
    /**
     * 'column menu module used to handle column menu actions'
     */
    export class ColumnMenu implements IAction {
        private element;
        private gridID;
        private parent;
        private serviceLocator;
        private columnMenu;
        private l10n;
        private defaultItems;
        private localeText;
        private targetColumn;
        private disableItems;
        private hiddenItems;
        private headerCell;
        private isOpen;
        private eventArgs;
        private GROUP;
        private UNGROUP;
        private ASCENDING;
        private DESCENDING;
        private ROOT;
        private FILTER;
        private POP;
        private WRAP;
        private CHOOSER;
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator);
        wireEvents(): void;
        unwireEvents(): void;
        /**
         * To destroy the resize
         * @return {void}
         * @hidden
         */
        destroy(): void;
        columnMenuHandlerClick(e: Event): void;
        private openColumnMenu(e);
        columnMenuHandlerDown(e: Event): void;
        private getColumnMenuHandlers();
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private enableAfterRenderMenu(e);
        private render();
        private wireFilterEvents();
        private unwireFilterEvents();
        private beforeMenuItemRender(args);
        private columnMenuBeforeClose(args);
        private isChooserItem(item);
        private columnMenuBeforeOpen(args);
        private ensureDisabledStatus(item);
        private columnMenuItemClick(args);
        private columnMenuOnClose(args);
        private getDefaultItems();
        private getItems();
        private getDefaultItem(item);
        private getLocaleText(item);
        private generateID(item, append?);
        private getKeyFromId(id, append?);
        getColumnMenu(): HTMLElement;
        private getModuleName();
        private setLocaleKey();
        private getHeaderCell(e);
        private getColumn();
        private createChooserItems();
        private appendFilter(e);
        private getFilter(target, id, isClose?);
        private setPosition(li, ul);
        private filterPosition(e);
        private getDefault();
        private isFilterPopupOpen();
        private getFilterPop();
        private isFilterItemAdded();
    }
    
    /**
     * `CommandColumn` used to handle the command column actions.
     * @hidden
     */
    export class CommandColumn {
        private parent;
        private previousClickedTD;
        private locator;
        private clickedButton;
        constructor(parent: IGrid, locator?: ServiceLocator);
        private initiateRender();
        private commandClickHandler(e);
        /**
         * For internal use only - Get the module name.
         */
        private getModuleName();
        /**
         * To destroy CommandColumn.
         * @method destroy
         * @return {void}
         */
        private destroy();
        private removeEventListener();
        private addEventListener();
        private keyPressHandler(e);
    }
    
    export const menuClass: CMenuClassList;
    export interface CMenuClassList {
        header: string;
        content: string;
        edit: string;
        batchEdit: string;
        editIcon: string;
        pager: string;
        cancel: string;
        save: string;
        delete: string;
        copy: string;
        pdf: string;
        group: string;
        ungroup: string;
        csv: string;
        excel: string;
        fPage: string;
        lPage: string;
        nPage: string;
        pPage: string;
        ascending: string;
        descending: string;
        groupHeader: string;
        touchPop: string;
    }
    /**
     * The `ContextMenu` module is used to handle context menu actions.
     */
    export class ContextMenu implements IAction {
        private element;
        contextMenu: any;
        private defaultItems;
        private disableItems;
        private hiddenItems;
        private gridID;
        private parent;
        private serviceLocator;
        private l10n;
        private localeText;
        private targetColumn;
        private eventArgs;
        isOpen: boolean;
        row: HTMLTableRowElement;
        cell: HTMLTableCellElement;
        private keyPressHandlerFunction;
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private keyDownHandler(e);
        private render();
        private enableAfterRenderMenu(e);
        private getMenuItems();
        private getLastPage();
        private contextMenuOpen();
        private contextMenuItemClick(args);
        private contextMenuOnClose(args);
        private getLocaleText(item);
        private updateItemStatus();
        private contextMenuBeforeOpen(args);
        private ensureTarget(targetElement, selector);
        private ensureFrozenHeader(targetElement);
        private ensureDisabledStatus(item);
        /**
         * Gets the context menu element from the Grid.
         * @return {Element}
         */
        getContextMenu(): Element;
        /**
         * Destroys the context menu component in the Grid.
         * @method destroy
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private getModuleName();
        private generateID(item);
        private getKeyFromId(id);
        private buildDefaultItems(item);
        private getDefaultItems();
        private setLocaleKey();
        private getColumn(e);
        private selectRow(e, isSelectable);
    }
    
    /**
     * Grid data module is used to generate query and data source.
     * @hidden
     */
    export class Data implements IDataProcessor {
        dataManager: ej.data.DataManager;
        protected parent: IGrid;
        protected serviceLocator: ServiceLocator;
        protected dataState: PendingState;
        /**
         * Constructor for data module.
         * @hidden
         */
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator);
        /**
         * The function used to initialize dataManager and external query
         * @return {void}
         */
        private initDataManager();
        /**
         * The function is used to generate updated Query from Grid model.
         * @return {Query}
         * @hidden
         */
        generateQuery(skipPage?: boolean): ej.data.Query;
        protected aggregateQuery(query: ej.data.Query, isForeign?: boolean): ej.data.Query;
        protected pageQuery(query: ej.data.Query, skipPage?: boolean): ej.data.Query;
        protected groupQuery(query: ej.data.Query): ej.data.Query;
        protected sortQuery(query: ej.data.Query): ej.data.Query;
        protected searchQuery(query: ej.data.Query): ej.data.Query;
        protected filterQuery(query: ej.data.Query, column?: PredicateModel[], skipFoerign?: boolean): ej.data.Query;
        private fGeneratePredicate(col, predicateList);
        /**
         * The function is used to get dataManager promise by executing given Query.
         * @param  {Query} query - Defines the query which will execute along with data processing.
         * @return {Promise<Object>}
         * @hidden
         */
        getData(args?: {
            requestType?: string;
            foreignKeyData?: string[];
            data?: Object;
            index?: number;
        }, query?: ej.data.Query): Promise<Object>;
        private formatGroupColumn(value, field);
        private crudActions(args);
        /** @hidden */
        saveChanges(changes: Object, key: string): Promise<Object>;
        private getKey(keys);
        /** @hidden */
        isRemote(): boolean;
        private addRows(e);
        private removeRows(e);
        private getColumnByField(field);
        protected destroy(): void;
        getState(): PendingState;
        setState(state: PendingState): Object;
        getStateEventArgument(query: ej.data.Query): PendingState;
        private eventPromise(args, query?, key?);
    }
    
    /**
     * The `DetailRow` module is used to handle detail template and hierarchy Grid operations.
     */
    export class DetailRow {
        private aria;
        private parent;
        private focus;
        /**
         * Constructor for the Grid detail template module
         * @hidden
         */
        constructor(parent?: IGrid, locator?: ServiceLocator);
        private clickHandler(e);
        private toogleExpandcollapse(target);
        private isDetailRow(row);
        private destroy();
        private getTDfromIndex(index, className);
        /**
         * Expands a detail row with the given target.
         * @param  {Element} target - Defines the collapsed element to expand.
         * @return {void}
         */
        expand(target: number | Element): void;
        /**
         * Collapses a detail row with the given target.
         * @param  {Element} target - Defines the expanded element to collapse.
         * @return {void}
         */
        collapse(target: number | Element): void;
        /**
         * Expands all the detail rows of the Grid.
         * @return {void}
         */
        expandAll(): void;
        /**
         * Collapses all the detail rows of the Grid.
         * @return {void}
         */
        collapseAll(): void;
        private expandCollapse(isExpand);
        private keyPressHandler(e);
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    /**
     * `DialogEdit` module is used to handle dialog editing actions.
     * @hidden
     */
    export class DialogEdit extends NormalEdit {
        protected parent: IGrid;
        protected serviceLocator: ServiceLocator;
        protected renderer: EditRender;
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator, renderer?: EditRender);
        closeEdit(): void;
        addRecord(data?: Object, index?: number): void;
        endEdit(): void;
        deleteRecord(fieldname?: string, data?: Object): void;
        protected startEdit(tr?: Element): void;
    }
    
    /**
     * The `Edit` module is used to handle editing actions.
     */
    export class Edit implements IAction {
        private edit;
        protected renderer: EditRender;
        private editModule;
        /** @hidden */
        formObj: ej.inputs.FormValidator;
        mFormObj: ej.inputs.FormValidator;
        private editCellType;
        private editType;
        protected parent: IGrid;
        protected serviceLocator: ServiceLocator;
        protected l10n: ej.base.L10n;
        private dialogObj;
        private alertDObj;
        private actionBeginFunction;
        private actionCompleteFunction;
        private preventObj;
        /**
         * Constructor for the Grid editing module
         * @hidden
         */
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator);
        private updateColTypeObj();
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        /**
         * @hidden
         */
        onPropertyChanged(e: NotifyArgs): void;
        private updateEditObj();
        private initialEnd();
        private wireEvents();
        private unwireEvents();
        private tapEvent(e);
        private getUserAgent();
        /**
         * Edits any bound record in the Grid by TR element.
         * @param {HTMLTableRowElement} tr - Defines the table row to be edited.
         */
        startEdit(tr?: HTMLTableRowElement): void;
        /**
         * Cancels edited state.
         */
        closeEdit(): void;
        protected refreshToolbar(): void;
        /**
         * To adds a new row at the top with the given data. When data is not passed, it will add empty rows.
         * > `editSettings.allowEditing` should be true.
         * @param {Object} data - Defines the new add record data.
         * @param {number} index - Defines the row index to be added
         */
        addRecord(data?: Object, index?: number): void;
        /**
         * Deletes a record with the given options. If fieldname and data are not given, the Grid will delete the selected record.
         * > `editSettings.allowDeleting` should be true.
         * @param {string} fieldname - Defines the primary key field name of the column.
         * @param {Object} data - Defines the JSON data record to be deleted.
         */
        deleteRecord(fieldname?: string, data?: Object): void;
        /**
         * Deletes a visible row by TR element.
         * @param {HTMLTableRowElement} tr - Defines the table row element.
         */
        deleteRow(tr: HTMLTableRowElement): void;
        /**
         * If Grid is in editable state, you can save a record by invoking endEdit.
         */
        endEdit(): void;
        /**
         * To update the specified cell by given value without changing into edited state.
         * @param {number} rowIndex Defines the row index.
         * @param {string} field Defines the column field.
         * @param {string | number | boolean | Date} value - Defines the value to be changed.
         */
        updateCell(rowIndex: number, field: string, value: string | number | boolean | Date): void;
        /**
         * To update the specified row by given values without changing into edited state.
         * @param {number} index Defines the row index.
         * @param {Object} data Defines the data object to be updated.
         */
        updateRow(index: number, data: Object): void;
        /**
         * Resets added, edited, and deleted records in the batch mode.
         */
        batchCancel(): void;
        /**
         * Bulk saves added, edited, and deleted records in the batch mode.
         */
        batchSave(): void;
        /**
         * Changes a particular cell into edited state based on the row index and field name provided in the `batch` mode.
         * @param {number} index - Defines row index to edit a particular cell.
         * @param {string} field - Defines the field name of the column to perform batch edit.
         */
        editCell(index: number, field: string): void;
        /**
         * Checks the status of validation at the time of editing. If validation is passed, it returns true.
         * @return {boolean}
         */
        editFormValidate(): boolean;
        /**
         * Gets the added, edited,and deleted data before bulk save to the DataSource in batch mode.
         * @return {Object}
         */
        getBatchChanges(): Object;
        /**
         * Gets the current value of the edited component.
         */
        getCurrentEditCellData(): string;
        /**
         * Saves the cell that is currently edited. It does not save the value to the DataSource.
         */
        saveCell(): void;
        private endEditing();
        private showDialog(content, obj);
        getValueFromType(col: Column, value: string | Date | boolean): number | string | Date | boolean;
        private destroyToolTip();
        private createConfirmDlg();
        private createAlertDlg();
        private alertClick();
        private dlgWidget(btnOptions, name);
        private dlgCancel();
        private dlgOk(e);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private actionComplete(e);
        /**
         * @hidden
         */
        getCurrentEditedData(form: Element, editedData: Object): Object;
        /**
         * @hidden
         */
        onActionBegin(e: NotifyArgs): void;
        /**
         * @hidden
         */
        destroyWidgets(cols?: Column[]): void;
        /**
         * @hidden
         */
        destroyForm(): void;
        /**
         * To destroy the editing.
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private keyPressHandler(e);
        private preventBatch(args);
        private executeAction();
        /**
         * @hidden
         */
        applyFormValidation(cols?: Column[]): void;
        private createFormObj(form, rules);
        private valErrorPlacement(inputElement, error);
        private getElemTable(inputElement);
        private validationComplete(args);
        private createTooltip(element, error, name, display);
    }
    /** @hidden */
    export namespace Global {
        let timer: Object;
    }
    
    /**
     * @hidden
     * `ExcelExport` module is used to handle the Excel export action.
     */
    export class ExcelExport {
        private parent;
        private isExporting;
        private theme;
        private book;
        private workSheet;
        private rows;
        private columns;
        private styles;
        private data;
        private rowLength;
        private footer;
        private expType;
        private includeHiddenColumn;
        private isCsvExport;
        private isBlob;
        private blobPromise;
        private exportValueFormatter;
        private isElementIdChanged;
        private helper;
        private foreignKeyData;
        /**
         * Constructor for the Grid Excel Export module.
         * @hidden
         */
        constructor(parent?: IGrid);
        /**
         * For internal use only - Get the module name.
         */
        private getModuleName();
        private init(gObj);
        /**
         * Export Grid to Excel file.
         * @param  {exportProperties} exportProperties - Defines the export properties of the Grid.
         * @param  {isMultipleExport} isMultipleExport - Defines is multiple Grid's are exported.
         * @param  {workbook} workbook - Defined the Workbook if multiple Grid is exported.
         * @param  {isCsv} isCsv - true if export to CSV.
         * @return {Promise<any>}
         */
        Map(grid: IGrid, exportProperties: ExcelExportProperties, isMultipleExport: boolean, workbook: any, isCsv: boolean, isBlob: boolean): Promise<any>;
        private processRecords(gObj, exportProperties, isMultipleExport, workbook);
        private processInnerRecords(gObj, exportProperties, isMultipleExport, workbook, r);
        private processRecordContent(gObj, returnType, headerRow, isMultipleExport, currentViewRecords?);
        private processGroupedRows(gObj, dataSource, headerRow, level);
        private processRecordRows(gObj, record, headerRow, level);
        private processAggregates(gObj, rec, currentViewRecords?, indent?, byGroup?);
        private fillAggregates(gObj, cells, indent, customIndex?);
        private getAggreateValue(cellType, template, cell, row);
        private mergeOptions(JSON1, JSON2);
        private getColumnStyle(gObj, columnIndex);
        private processHeaderContent(gObj, headerRow, exportProperties, indent);
        private getHeaderThemeStyle(theme);
        private updateThemeStyle(themestyle, style);
        private getCaptionThemeStyle(theme);
        private getRecordThemeStyle(theme);
        private processExcelHeader(header);
        private updatedCellIndex(json);
        private processExcelFooter(footer);
        private getIndex(spannedCells, rowIndex, columnIndex);
        private parseStyles(gObj, col, style, index);
        /**
         * To destroy the excel export
         * @return {void}
         * @hidden
         */
        destroy(): void;
    }
    
    /**
     * @hidden
     * `ExcelFilter` module is used to handle filtering action.
     */
    export class ExcelFilter extends CheckBoxFilter {
        private datePicker;
        private actObj;
        private numericTxtObj;
        private dlgDiv;
        private l10n;
        private dlgObj;
        private menuEle;
        private customFilterOperators;
        private dropOptr;
        private optrData;
        private menuItem;
        private menu;
        private cmenu;
        protected menuObj: ContextMenu;
        private isCMenuOpen;
        private localeConstants;
        /**
         * Constructor for excel filtering module
         * @hidden
         */
        constructor(parent?: IGrid, filterSettings?: FilterSettings, serviceLocator?: ServiceLocator, customFltrOperators?: Object);
        private getCMenuDS(type, operator?);
        /**
         * To destroy the filter bar.
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private createMenu(type, isFiltered, isCheckIcon);
        private createMenuElem(val, className?, iconName?, isSubMenu?);
        private wireExEvents();
        private unwireExEvents();
        private clickExHandler(e);
        private destroyCMenu();
        private hoverHandler(e);
        private ensureTextFilter();
        private preventClose(args);
        private getContextBounds(context);
        private getCMenuYPosition(target, context);
        openDialog(options: IFilterArgs): void;
        closeDialog(): void;
        private selectHandler(e);
        private renderDialogue(e);
        private removeDialog();
        private clearBtnClick(field);
        private createdDialog(target, column);
        private renderCustomFilter(target, column);
        private filterBtnClick(col);
        /**
         * Filters grid row by column name with given options.
         * @param {string} fieldName - Defines the field name of the filter column.
         * @param {string} firstOperator - Defines the first operator by how to filter records.
         * @param {string | number | Date | boolean} firstValue - Defines the first value which is used to filter records.
         * @param  {string} predicate - Defines the relationship between one filter query with another by using AND or OR predicate.
         * @param {boolean} matchCase - If ignore case set to true, then filter records with exact match or else
         * filter records with case insensitive(uppercase and lowercase letters treated as same).
         * @param {boolean} ignoreAccent - If ignoreAccent set to true, then ignores the diacritic characters or accents when filtering.
         * @param {string} secondOperator - Defines the second operator by how to filter records.
         * @param {string | number | Date | boolean} secondValue - Defines the first value which is used to filter records.
         */
        private filterByColumn(fieldName, firstOperator, firstValue, predicate?, matchCase?, ignoreAccent?, secondOperator?, secondValue?);
        private renderOperatorUI(column, table, elementID, predicates, isFirst?);
        private dropDownOpen(args);
        private getSelectedValue(text);
        private dropSelectedVal(col, predicates, isFirst?);
        private getSelectedText(operator);
        private renderFilterUI(column, dlgConetntEle);
        private renderRadioButton(column, tr, predicates);
        private removeObjects(elements);
        private renderFlValueUI(column, optr, elementId, predicates, isFirst?);
        private renderMatchCase(column, tr, matchCase, elementId, predicates);
        private renderDate(options, column, inputValue, fValue, isRtl);
        private completeAction(e);
        private renderNumericTextBox(options, column, inputValue, fValue, isRtl);
        private renderAutoComplete(options, column, inputValue, fValue, isRtl);
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    /**
     * @hidden
     * `ExportHelper` for `PdfExport` & `ExcelExport`
     */
    export class ExportHelper {
        parent: IGrid;
        private colDepth;
        private hideColumnInclude;
        private foreignKeyData;
        constructor(parent: IGrid);
        static getQuery(parent: IGrid, data: Data): ej.data.Query;
        getFData(value: string, column: Column): Object;
        getColumnData(gridObj: Grid): Promise<Object>;
        getHeaders(column: any[], isHideColumnInclude?: boolean): {
            rows: any[];
            columns: Column[];
        };
        getConvertedWidth(input: string): number;
        private generateActualColumns(column, actualColumns);
        private removeCellFromRow(cells, cellIndex);
        private processHeaderCells(rows);
        private appendGridCells(cols, gridRows, index, isFirstObj, isFirstColumn, isLastColumn);
        private generateCell(gridColumn, cellType?, rowSpan?, className?, rowIndex?, columnIndex?);
        private processColumns(rows);
        private getCellCount(column, count);
        private measureColumnDepth(column);
        private checkDepth(col, index);
    }
    /**
     * @hidden
     * `ExportValueFormatter` for `PdfExport` & `ExcelExport`
     */
    export class ExportValueFormatter {
        private internationalization;
        private valueFormatter;
        constructor();
        private returnFormattedValue(args, customFormat);
        formatCellValue(args: any): string;
    }
    
    /**
     *
     * The `Filter` module is used to handle filtering action.
     */
    export class Filter implements IAction {
        private filterSettings;
        private element;
        private value;
        private predicate;
        private operator;
        private column;
        private fieldName;
        private matchCase;
        private ignoreAccent;
        private timer;
        private filterStatusMsg;
        private currentFilterObject;
        private isRemove;
        private contentRefresh;
        private values;
        private nextFlMenuOpen;
        private type;
        private filterModule;
        private filterOperators;
        private fltrDlgDetails;
        private customOperators;
        private parent;
        private serviceLocator;
        private l10n;
        private valueFormatter;
        private actualPredicate;
        /**
         * Constructor for Grid filtering module
         * @hidden
         */
        constructor(parent?: IGrid, filterSettings?: FilterSettings, serviceLocator?: ServiceLocator);
        /**
         * To render filter bar when filtering enabled.
         * @return {void}
         * @hidden
         */
        render(e?: NotifyArgs): void;
        /**
         * To destroy the filter bar.
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private generateRow(index?);
        private generateCells();
        private generateCell(column, cellType?);
        /**
         * To update filterSettings when applying filter.
         * @return {void}
         * @hidden
         */
        updateModel(): void;
        private getFilteredColsIndexByField(col);
        /**
         * To trigger action complete event.
         * @return {void}
         * @hidden
         */
        onActionComplete(e: NotifyArgs): void;
        private wireEvents();
        private unWireEvents();
        private enableAfterRender(e);
        private initialEnd();
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private filterMenuClose(args);
        /**
         * Filters the Grid row by fieldName, filterOperator, and filterValue.
         * @param  {string} fieldName - Defines the field name of the filter column.
         * @param  {string} filterOperator - Defines the operator to filter records.
         * @param  {string | number | Date | boolean} filterValue - Defines the value which is used to filter records.
         * @param  {string} predicate - Defines the relationship of one filter query with another by using AND or OR predicate.
         * @param  {boolean} matchCase - If match case is set to true, then the filter records
         * the exact match or <br> filters records that are case insensitive (uppercase and lowercase letters treated the same).
         * @param {boolean} ignoreAccent - If ignoreAccent set to true, then filter ignores the diacritic characters or accents while filtering.
         * @param  {string} actualFilterValue - Defines the actual filter value for the filter column.
         * @param  {string} actualOperator - Defines the actual filter operator for the filter column.
         * @return {void}
         */
        filterByColumn(fieldName: string, filterOperator: string, filterValue: string | number | Date | boolean, predicate?: string, matchCase?: boolean, ignoreAccent?: boolean, actualFilterValue?: Object, actualOperator?: Object): void;
        private onPropertyChanged(e);
        /**
         * Clears all the filtered rows of the Grid.
         * @return {void}
         */
        clearFiltering(): void;
        private checkAlreadyColFiltered(field);
        private columnMenuFilter(args);
        private filterDialogOpen(col, target, left?, top?);
        /**
         * Removes filtered column by field name.
         * @param  {string} field - Defines column field name to remove filter.
         * @param  {boolean} isClearFilterBar -  Specifies whether the filter bar value needs to be cleared.
         * @return {void}
         * @hidden
         */
        removeFilteredColsByField(field: string, isClearFilterBar?: boolean): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        private keyUpHandler(e);
        private updateCrossIcon(element);
        private updateFilterMsg();
        private setFormatForFlColumn(value, column);
        private checkForSkipInput(column, value);
        private processFilter(e);
        private startTimer(e);
        private stopTimer();
        private onTimerTick();
        private validateFilterValue(value);
        private getOperator(value);
        private columnPositionChanged(e);
        private getLocalizedCustomOperators();
        private filterIconClickHandler(e);
        private clickHandler(e);
        private filterHandler(args);
    }
    
    /**
     * `ForeignKey` module is used to handle foreign key column's actions.
     */
    export class ForeignKey extends Data {
        constructor(parent: IGrid, serviceLocator: ServiceLocator);
        private initEvent();
        private initForeignKeyColumns(columns);
        private getForeignKeyData(args);
        private generateQueryFormData(args);
        private genarateQuery(column, e?, fromData?, needQuery?);
        private genarateColumnQuery(column);
        private isFiltered(column);
        protected getModuleName(): string;
        protected destroy(): void;
        private destroyEvent();
    }
    
    /**
     * `Freeze` module is used to handle Frozen rows and columns.
     * @hidden
     */
    export class Freeze implements IAction {
        private locator;
        private parent;
        constructor(parent: IGrid, locator?: ServiceLocator);
        getModuleName(): string;
        addEventListener(): void;
        private wireEvents();
        private dblClickHandler(e);
        private instantiateRenderer();
        removeEventListener(): void;
        destroy(): void;
    }
    
    /**
     *
     * The `Group` module is used to handle group action.
     */
    export class Group implements IAction {
        private groupSettings;
        private element;
        private colName;
        private column;
        private isAppliedGroup;
        private isAppliedUnGroup;
        private visualElement;
        private helper;
        private dragStart;
        private drag;
        private dragStop;
        private drop;
        private parent;
        private serviceLocator;
        private contentRefresh;
        private sortedColumns;
        private l10n;
        private aria;
        private focus;
        /**
         * Constructor for Grid group module
         * @hidden
         */
        constructor(parent?: IGrid, groupSettings?: GroupSettingsModel, sortedColumns?: string[], serviceLocator?: ServiceLocator);
        private columnDrag(e);
        private columnDragStart(e);
        private columnDrop(e);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private initialEnd();
        private keyPressHandler(e);
        private clickHandler(e);
        private unGroupFromTarget(target);
        private toogleGroupFromHeader(target);
        private applySortFromTarget(target);
        /**
         * Expands or collapses grouped rows by target element.
         * @param  {Element} target - Defines the target element of the grouped row.
         * @return {void}
         */
        expandCollapseRows(target: Element): void;
        private expandCollapse(isExpand);
        /**
         * Expands all the grouped rows of the Grid.
         * @return {void}
         */
        expandAll(): void;
        /**
         * Collapses all the grouped rows of the Grid.
         * @return {void}
         */
        collapseAll(): void;
        /**
         * The function is used to render grouping
         * @return {Element}
         * @hidden
         */
        render(): void;
        private renderGroupDropArea();
        private updateGroupDropArea();
        private initDragAndDrop();
        private initializeGHeaderDrag();
        private initializeGHeaderDrop();
        /**
         * Groups a column by column name.
         * @param  {string} columnName - Defines the column name to group.
         * @return {void}
         */
        groupColumn(columnName: string): void;
        /**
         * Ungroups a column by column name.
         * @param  {string} columnName - Defines the column name to ungroup.
         * @return {void}
         */
        ungroupColumn(columnName: string): void;
        /**
         * The function used to update groupSettings
         * @return {void}
         * @hidden
         */
        updateModel(): void;
        /**
         * The function used to trigger onActionComplete
         * @return {void}
         * @hidden
         */
        onActionComplete(e: NotifyArgs): void;
        private groupAddSortingQuery(colName);
        private addColToGroupDrop(field);
        private refreshToggleBtn(isRemove?);
        private removeColFromGroupDrop(field);
        private onPropertyChanged(e);
        private updateGroupedColumn(isVisible);
        private updateButtonVisibility(isVisible, className);
        private enableAfterRender(e);
        /**
         * To destroy the reorder
         * @return {void}
         * @hidden
         */
        destroy(): void;
        /**
         * Clears all the grouped columns of the Grid.
         * @return {void}
         */
        clearGrouping(): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        private refreshSortIcons(e?);
        private getGHeaderCell(field);
    }
    
    /**
     * `InlineEdit` module is used to handle inline editing actions.
     * @hidden
     */
    export class InlineEdit extends NormalEdit {
        protected parent: IGrid;
        protected serviceLocator: ServiceLocator;
        protected renderer: EditRender;
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator, renderer?: EditRender);
        closeEdit(): void;
        addRecord(data?: Object, index?: number): void;
        endEdit(): void;
        deleteRecord(fieldname?: string, data?: Object): void;
        protected startEdit(tr?: Element): void;
    }
    
    /**
     * `NormalEdit` module is used to handle normal('inline, dialog, external') editing actions.
     * @hidden
     */
    export class NormalEdit {
        protected parent: IGrid;
        protected serviceLocator: ServiceLocator;
        protected renderer: EditRender;
        formObj: ej.inputs.FormValidator;
        protected previousData: Object;
        private editRowIndex;
        private rowIndex;
        private addedRowIndex;
        private uid;
        private args;
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator, renderer?: EditRender);
        protected clickHandler(e: MouseEvent): void;
        protected dblClickHandler(e: MouseEvent): void;
        /**
         * The function used to trigger editComplete
         * @return {void}
         * @hidden
         */
        editComplete(e: NotifyArgs): void;
        protected startEdit(tr: Element): void;
        protected endEdit(): void;
        private destroyElements();
        private editHandler(args);
        private edSucc(e, args);
        private edFail(e);
        private editSuccess(e, args);
        private editFailure(e);
        private refreshRow(data);
        protected closeEdit(): void;
        protected addRecord(data?: Object, index?: number): void;
        protected deleteRecord(fieldname?: string, data?: Object): void;
        private stopEditStatus();
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        /**
         * @hidden
         */
        destroy(): void;
    }
    
    /**
     * The `Page` module is used to render pager and handle paging action.
     */
    export class Page implements IAction {
        private element;
        private pageSettings;
        private isForceCancel;
        private parent;
        private pagerObj;
        private handlers;
        /**
         * Constructor for the Grid paging module
         * @hidden
         */
        constructor(parent?: IGrid, pageSettings?: PageSettingsModel);
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        /**
         * The function used to render pager from grid pageSettings
         * @return {void}
         * @hidden
         */
        render(): void;
        private onSelect(e);
        private addAriaAttr();
        private dataReady(e?);
        /**
         * Refreshes the page count, pager information, and external message.
         * @return {void}
         */
        refresh(): void;
        /**
         * Navigates to the target page according to the given number.
         * @param  {number} pageNo - Defines the page number to navigate.
         * @return {void}
         */
        goToPage(pageNo: number): void;
        /**
         * The function used to update pageSettings model
         * @return {void}
         * @hidden
         */
        updateModel(e?: NotifyArgs): void;
        /**
         * The function used to trigger onActionComplete
         * @return {void}
         * @hidden
         */
        onActionComplete(e: NotifyArgs): void;
        /**
         * @hidden
         */
        onPropertyChanged(e: NotifyArgs): void;
        private clickHandler(e);
        private keyPressHandler(e);
        /**
         * Defines the text of the external message.
         * @param  {string} message - Defines the message to update.
         * @return {void}
         */
        updateExternalMessage(message: string): void;
        private appendToElement(e?);
        private enableAfterRender(e?);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        /**
         * To destroy the pager
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private pagerDestroy();
    }
    
    /**
     * `PDF Export` module is used to handle the exportToPDF action.
     * @hidden
     */
    export class PdfExport {
        private parent;
        private isExporting;
        private data;
        private pdfDocument;
        private hideColumnInclude;
        private currentViewData;
        private customDataSource;
        private exportValueFormatter;
        private gridTheme;
        private isGrouping;
        private helper;
        private isBlob;
        private blobPromise;
        /**
         * Constructor for the Grid PDF Export module
         * @hidden
         */
        constructor(parent?: IGrid);
        /**
         * For internal use only - Get the module name.
         */
        private getModuleName();
        private init(parent);
        /**
         * Used to map the input data
         * @return {void}
         */
        Map(parent?: IGrid, pdfExportProperties?: PdfExportProperties, isMultipleExport?: boolean, pdfDoc?: Object, isBlob?: boolean): Promise<Object>;
        private processExport(gObj, returnType, pdfExportProperties, isMultipleExport);
        private getSummaryCaptionThemeStyle();
        private getHeaderThemeStyle();
        private processGroupedRecords(pdfGrid, dataSource, gridColumns, gObj, border, level, font, brush, backgroundBrush, returnType);
        private processGridHeaders(childLevels, pdfGrid, rows, gridColumns, border, headerFont, headerBrush);
        private processExportProperties(pdfExportProperties, dataSource, section);
        private drawPageTemplate(template, element);
        private processContentValidation(content);
        private drawText(pageTemplate, content);
        private drawPageNumber(documentHeader, content);
        private drawImage(documentHeader, content);
        private drawLine(documentHeader, content);
        private processAggregates(sRows, pdfGrid, border, font, brush, backgroundBrush, isCaption, captionRow?, groupIndex?);
        private getTemplateFunction(templateFn, index, leastCaptionSummaryIndex, column);
        private getSummaryWithoutTemplate(data);
        private setColumnProperties(gridColumns, pdfGrid);
        /**
         * set default style properties of each rows in exporting grid
         * @private
         */
        private setRecordThemeStyle(row, border);
        /**
         * generate the formatted cell values
         * @private
         */
        private processRecord(border, columns, gObj, dataSource, pdfGrid, groupIndex?);
        private processCellStyle(cell, args);
        /**
         * set text alignment of each columns in exporting grid
         * @private
         */
        private getHorizontalAlignment(textAlign, format?);
        /**
         * set vertical alignment of each columns in exporting grid
         * @private
         */
        private getVerticalAlignment(verticalAlign, format?, textAlign?);
        private getFontFamily(fontFamily);
        private getFont(content);
        private getPageNumberStyle(pageNumberType);
        private setContentFormat(content, format);
        private getPageSize(pageSize);
        private getDashStyle(dashStyle);
        private getPenFromContent(content);
        private getBrushFromContent(content);
        private hexToRgb(hex);
        /**
         * To destroy the pdf export
         * @return {void}
         * @hidden
         */
        destroy(): void;
    }
    
    /**
     *
     * The `Print` module is used to handle print action.
     */
    export class Print {
        private parent;
        private printWind;
        private scrollModule;
        private isAsyncPrint;
        private printing;
        private static printGridProp;
        /**
         * Constructor for the Grid print module
         * @hidden
         */
        constructor(parent?: IGrid, scrollModule?: Scroll);
        /**
         * By default, prints all the Grid pages and hides the pager.
         * > You can customize print options using the
         * [`printMode`](./api-grid.html#printmode-string).
         * @return {void}
         */
        print(): void;
        private onEmpty();
        private actionBegin();
        private renderPrintGrid();
        private contentReady();
        private printGrid();
        private removeColGroup(depth, element);
        private isPrintGrid();
        /**
         * To destroy the print
         * @return {void}
         * @hidden
         */
        destroy(): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    /**
     *
     * The `Reorder` module is used for reordering columns.
     */
    export class Reorder implements IAction {
        private element;
        private upArrow;
        private downArrow;
        private x;
        private timer;
        private parent;
        /**
         * Constructor for the Grid reorder module
         * @hidden
         */
        constructor(parent?: IGrid);
        private chkDropPosition(srcElem, destElem);
        private chkDropAllCols(srcElem, destElem);
        private findColParent(col, cols, parent);
        private getColumnsModel(cols);
        private headerDrop(e);
        private isActionPrevent(gObj);
        private moveColumns(destIndex, column);
        private targetParentContainerIndex(srcElem, destElem);
        private getHeaderCells();
        private getColParent(column, columns);
        /**
         * Changes the position of the Grid columns by field names.
         * @param  {string} fromFName - Defines the origin field name.
         * @param  {string} toFName - Defines the destination field name.
         * @return {void}
         */
        reorderColumns(fromFName: string, toFName: string): void;
        private enableAfterRender(e);
        private createReorderElement();
        /**
         * The function used to trigger onActionComplete
         * @return {void}
         * @hidden
         */
        onActionComplete(e: NotifyArgs): void;
        /**
         * To destroy the reorder
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private drag(e);
        private updateScrollPostion(e);
        private setScrollLeft(scrollElem, isLeft);
        private stopTimer();
        private updateArrowPosition(target, isLeft);
        private dragStart(e);
        private dragStop(e);
        private setDisplay(display);
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    export const resizeClassList: ResizeClasses;
    export interface ResizeClasses {
        root: string;
        suppress: string;
        icon: string;
        helper: string;
        header: string;
        cursor: string;
    }
    /**
     * `Resize` module is used to handle Resize to fit for columns.
     * @hidden
     * @private
     */
    export class Resize implements IAction {
        private content;
        private header;
        private pageX;
        private column;
        private element;
        private helper;
        private tapped;
        private isDblClk;
        private minMove;
        private parentElementWidth;
        private parent;
        private widthService;
        /**
         * Constructor for the Grid resize module
         * @hidden
         */
        constructor(parent?: IGrid);
        /**
         * Resize by field names.
         * @param  {string|string[]} fName - Defines the field name.
         * @return {void}
         */
        autoFitColumns(fName: string | string[]): void;
        private resizeColumn(fName, index, id?);
        /**
         * To destroy the resize
         * @return {void}
         * @hidden
         */
        destroy(): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        private findColumn(fName);
        /**
         * To create table for autofit
         * @hidden
         */
        protected createTable(table: Element, text: Element[], tag: string): number;
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        /**
         * @hidden
         */
        render(): void;
        private wireEvents();
        private unwireEvents();
        private getResizeHandlers();
        private setHandlerHeight();
        private callAutoFit(e);
        private resizeStart(e);
        private cancelResizeAction(removeEvents?);
        private getWidth(width, minWidth, maxWidth);
        private resizing(e);
        private resizeEnd(e);
        private getPointX(e);
        private refreshColumnWidth();
        private getTargetColumn(e);
        private updateCursor(action);
        private refresh();
        private appendHelper();
        private setHelperHeight();
        private getScrollBarWidth(height?);
        private removeHelper(e);
        private updateHelper();
        private calcPos(elem);
        private doubleTapEvent(e);
        private getUserAgent();
        private timeoutHandler();
    }
    
    /**
     *
     * Reorder module is used to handle row reordering.
     * @hidden
     */
    export class RowDD implements IAction {
        private selectedRows;
        private isDragStop;
        private helper;
        private dragStart;
        private drag;
        private dragStop;
        private parent;
        /**
         * Constructor for the Grid print module
         * @hidden
         */
        constructor(parent?: IGrid);
        private initializeDrag();
        private getElementFromPosition(element, event);
        /**
         * The function used to trigger onActionComplete
         * @return {void}
         * @hidden
         */
        onActionComplete(e: NotifyArgs): void;
        private getTargetIdx(targetRow);
        private columnDrop(e);
        private enableAfterRender(e);
        /**
         * To destroy the print
         * @return {void}
         * @hidden
         */
        destroy(): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    /**
     * The `Scroll` module is used to handle scrolling behaviour.
     */
    export class Scroll implements IAction {
        private parent;
        private lastScrollTop;
        private previousValues;
        private oneTimeReady;
        private content;
        private header;
        private widthService;
        private pageXY;
        /**
         * Constructor for the Grid scrolling.
         * @hidden
         */
        constructor(parent?: IGrid);
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        /**
         * @hidden
         */
        setWidth(): void;
        /**
         * @hidden
         */
        setHeight(): void;
        /**
         * @hidden
         */
        setPadding(): void;
        /**
         * @hidden
         */
        removePadding(rtl?: boolean): void;
        /**
         * Refresh makes the Grid adoptable with the height of parent container.
         *
         * > The [`height`](./api-grid.html#height) must be set to 100%.
         * @return
         */
        refresh(): void;
        private getThreshold();
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private onContentScroll(scrollTarget);
        private onFreezeContentScroll(scrollTarget);
        private onWheelScroll(scrollTarget);
        private onTouchScroll(scrollTarget);
        private setPageXY();
        private getPointXY(e);
        private wireEvents();
        /**
         * @hidden
         */
        getCssProperties(rtl?: boolean): ScrollCss;
        private ensureOverflow(content);
        private onPropertyChanged(e);
        /**
         * @hidden
         */
        destroy(): void;
        /**
         * Function to get the scrollbar width of the browser.
         * @return {number}
         * @hidden
         */
        static getScrollBarWidth(): number;
    }
    /**
     * @hidden
     */
    export interface ScrollCss {
        padding?: string;
        border?: string;
    }
    
    /**
     * The `Search` module is used to handle search action.
     */
    export class Search implements IAction {
        private parent;
        private refreshSearch;
        private actionCompleteFunc;
        /**
         * Constructor for Grid search module.
         * @hidden
         */
        constructor(parent?: IGrid);
        /**
         * Searches Grid records by given key.
         *
         * > You can customize the default search action by using [`searchSettings`](./api-grid.html#searchsettings-searchsettingsmodel).
         * @param  {string} searchString - Defines the key.
         * @return {void}
         */
        search(searchString: string): void;
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        /**
         * To destroy the print
         * @return {void}
         * @hidden
         */
        destroy(): void;
        /**
         * @hidden
         */
        onPropertyChanged(e: NotifyArgs): void;
        /**
         * The function used to trigger onActionComplete
         * @return {void}
         * @hidden
         */
        onSearchComplete(e: NotifyArgs): void;
        onActionComplete(e: NotifyArgs): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
    }
    
    /**
     * The `Selection` module is used to handle cell and row selection.
     */
    export class Selection implements IAction {
        /**
         * @hidden
         */
        selectedRowIndexes: number[];
        /**
         * @hidden
         */
        selectedRowCellIndexes: ISelectedCell[];
        /**
         * @hidden
         */
        selectedRecords: Element[];
        /**
         * @hidden
         */
        isRowSelected: boolean;
        /**
         * @hidden
         */
        isCellSelected: boolean;
        private selectionSettings;
        private prevRowIndex;
        private prevCIdxs;
        private prevECIdxs;
        private preventFocus;
        private selectedRowIndex;
        private isMultiShiftRequest;
        private isMultiCtrlRequest;
        private enableSelectMultiTouch;
        private element;
        private startIndex;
        private startCellIndex;
        private currentIndex;
        private isDragged;
        private isCellDrag;
        private x;
        private y;
        private target;
        private preSelectedCellIndex;
        private factory;
        private contentRenderer;
        private checkedTarget;
        private primaryKey;
        private chkField;
        private selectedRowState;
        private totalRecordsCount;
        private chkAllCollec;
        private isCheckedOnAdd;
        private persistSelectedData;
        private selectionRequest;
        private onDataBoundFunction;
        private actionBeginFunction;
        private actionCompleteFunction;
        private actionCompleteFunc;
        private parent;
        private focus;
        private checkAllRecords;
        private unSelectRowKey;
        /**
         * Constructor for the Grid selection module
         * @hidden
         */
        constructor(parent?: IGrid, selectionSettings?: SelectionSettings, locator?: ServiceLocator);
        private initializeSelection();
        /**
         * The function used to trigger onActionBegin
         * @return {void}
         * @hidden
         */
        onActionBegin(args: Object, type: string): void;
        private fDataUpdate(args);
        /**
         * The function used to trigger onActionComplete
         * @return {void}
         * @hidden
         */
        onActionComplete(args: Object, type: string): void;
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        /**
         * To destroy the selection
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private isEditing();
        private getSelectedMovableRow(index);
        /**
         * Selects a row by the given index.
         * @param  {number} index - Defines the row index.
         * @param  {boolean} isToggle - If set to true, then it toggles the selection.
         * @return {void}
         */
        selectRow(index: number, isToggle?: boolean): void;
        private addMovableArgs(targetObj, mRow);
        /**
         * Selects a range of rows from start and end row indexes.
         * @param  {number} startIndex - Specifies the start row index.
         * @param  {number} endIndex - Specifies the end row index.
         * @return {void}
         */
        selectRowsByRange(startIndex: number, endIndex?: number): void;
        /**
         * Selects a collection of rows by index.
         * @param  {number[]} rowIndexes - Specifies an array of row indexes.
         * @return {void}
         */
        selectRows(rowIndexes: number[]): void;
        /**
         * Select rows with existing row selection by passing row indexes.
         * @param  {number} startIndex - Specifies the row indexes.
         * @return {void}
         * @hidden
         */
        addRowsToSelection(rowIndexes: number[]): void;
        private getCollectionFromIndexes(startIndex, endIndex);
        private clearRow();
        private updateRowProps(startIndex);
        private updatePersistCollection(selectedRow, chkState);
        private updateCheckBoxes(row, chkState?);
        private updateRowSelection(selectedRow, startIndex);
        /**
         * Deselects the currently selected rows and cells.
         * @return {void}
         */
        clearSelection(): void;
        /**
         * Deselects the currently selected rows.
         * @return {void}
         */
        clearRowSelection(): void;
        private rowDeselect(type, rowIndex, data, row, foreignKeyData, mRow?);
        private getRowObj(row?);
        private getSelectedMovableCell(cellIndex);
        /**
         * Selects a cell by the given index.
         * @param  {IIndex} cellIndex - Defines the row and column indexes.
         * @param  {boolean} isToggle - If set to true, then it toggles the selection.
         * @return {void}
         */
        selectCell(cellIndex: IIndex, isToggle?: boolean): void;
        private getCellIndex(rIdx, cIdx);
        /**
         * Selects a range of cells from start and end indexes.
         * @param  {IIndex} startIndex - Specifies the row and column's start index.
         * @param  {IIndex} endIndex - Specifies the row and column's end index.
         * @return {void}
         */
        selectCellsByRange(startIndex: IIndex, endIndex?: IIndex): void;
        /**
         * Selects a collection of cells by row and column indexes.
         * @param  {ISelectedCell[]} rowCellIndexes - Specifies the row and column indexes.
         * @return {void}
         */
        selectCells(rowCellIndexes: ISelectedCell[]): void;
        /**
         * Select cells with existing cell selection by passing row and column index.
         * @param  {IIndex} startIndex - Defines the collection of row and column index.
         * @return {void}
         * @hidden
         */
        addCellsToSelection(cellIndexes: IIndex[]): void;
        private getColIndex(rowIndex, index);
        private getLastColIndex(rowIndex);
        private clearCell();
        private cellDeselect(type, cellIndexes, data, cells, foreignKeyData);
        private updateCellSelection(selectedCell, rowIndex?, cellIndex?);
        private addAttribute(cell);
        private updateCellProps(startIndex, endIndex);
        private addRowCellIndex(rowCellIndex);
        /**
         * Deselects the currently selected cells.
         * @return {void}
         */
        clearCellSelection(): void;
        private getSelectedCellsElement();
        private mouseMoveHandler(e);
        private selectLikeExcel(rowIndex, cellIndex);
        private mouseUpHandler(e);
        private mouseDownHandler(e);
        private clearSelAfterRefresh(e);
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        private columnPositionChanged();
        private refreshHeader();
        private rowsRemoved(e);
        dataReady(e: {
            requestType: string;
        }): void;
        private getCheckAllBox();
        private onPropertyChanged(e);
        private hidePopUp();
        private initialEnd();
        private checkBoxSelectionChanged();
        private initPerisistSelection();
        private ensureCheckboxFieldSelection();
        private dataSuccess(res);
        private refreshPersistSelection();
        private actionBegin(e);
        private actionComplete(e);
        private onDataBound();
        private checkSelectAllAction(checkState);
        private checkSelectAll(checkBox);
        private getCheckAllStatus(ele?);
        private checkSelect(checkBox);
        private moveIntoUncheckCollection(row);
        private triggerChkChangeEvent(checkBox, checkState);
        private setCheckAllState(isInteraction?);
        private clickHandler(e);
        private popUpClickHandler(e);
        private showPopup(e);
        private rowCellSelectionHandler(rowIndex, cellIndex);
        private onCellFocused(e);
        /**
         * Apply ctrl + A key selection
         * @return {void}
         * @hidden
         */
        ctrlPlusA(): void;
        private applySpaceSelection(target);
        private applyDownUpKey(rowIndex?, cellIndex?);
        private applyUpDown(rowIndex);
        private applyRightLeftKey(rowIndex?, cellIndex?);
        private applyHomeEndKey(rowIndex?, cellIndex?);
        /**
         * Apply shift+down key selection
         * @return {void}
         * @hidden
         */
        shiftDownKey(rowIndex?: number, cellIndex?: number): void;
        private applyShiftLeftRightKey(rowIndex?, cellIndex?);
        private applyCtrlHomeEndKey(rowIndex, cellIndex);
        private addRemoveClassesForRow(row, isAdd, clearAll, ...args);
        private isRowType();
        private isCellType();
        private isSingleSel();
        private getRenderer();
        /**
         * Gets the collection of selected records.
         * @return {Object[]}
         */
        getSelectedRecords(): Object[];
        private addEventListener_checkbox();
        removeEventListener_checkbox(): void;
        private actionCompleteHandler(e);
    }
    
    /**
     * The `ShowHide` module is used to control column visibility.
     */
    export class ShowHide {
        private parent;
        /**
         * Constructor for the show hide module.
         * @hidden
         */
        constructor(parent: IGrid);
        /**
         * Shows a column by column name.
         * @param  {string|string[]} columnName - Defines a single or collection of column names to show.
         * @param  {string} showBy - Defines the column key either as field name or header text.
         * @return {void}
         */
        show(columnName: string | string[], showBy?: string): void;
        /**
         * Hides a column by column name.
         * @param  {string|string[]} columnName - Defines a single or collection of column names to hide.
         * @param  {string} hideBy - Defines the column key either as field name or header text.
         * @return {void}
         */
        hide(columnName: string | string[], hideBy?: string): void;
        private getToggleFields(key);
        private getColumns(keys, getKeyBy?);
        /**
         * Shows or hides columns by given column collection.
         * @private
         * @param  {Column[]} columns - Specifies the columns.
         * @return {void}
         */
        setVisible(columns?: Column[]): void;
    }
    
    /**
     *
     * The `Sort` module is used to handle sorting action.
     */
    export class Sort implements IAction {
        private columnName;
        private direction;
        private isMultiSort;
        private lastSortedCol;
        private sortSettings;
        private enableSortMultiTouch;
        private contentRefresh;
        private isRemove;
        private sortedColumns;
        private isModelChanged;
        private aria;
        private focus;
        private parent;
        /**
         * Constructor for Grid sorting module
         * @hidden
         */
        constructor(parent?: IGrid, sortSettings?: SortSettings, sortedColumns?: string[], locator?: ServiceLocator);
        /**
         * The function used to update sortSettings
         * @return {void}
         * @hidden
         */
        updateModel(): void;
        /**
         * The function used to trigger onActionComplete
         * @return {void}
         * @hidden
         */
        onActionComplete(e: NotifyArgs): void;
        /**
         * Sorts a column with the given options.
         * @param {string} columnName - Defines the column name to sort.
         * @param {SortDirection} direction - Defines the direction of sorting field.
         * @param {boolean} isMultiSort - Specifies whether the previously sorted columns are to be maintained.
         * @return {void}
         */
        sortColumn(columnName: string, direction: SortDirection, isMultiSort?: boolean): void;
        private updateSortedCols(columnName, isMultiSort);
        /**
         * @hidden
         */
        onPropertyChanged(e: NotifyArgs): void;
        /**
         * Clears all the sorted columns of the Grid.
         * @return {void}
         */
        clearSorting(): void;
        private isActionPrevent();
        /**
         * Remove sorted column by field name.
         * @param {string} field - Defines the column field name to remove sort.
         * @return {void}
         * @hidden
         */
        removeSortColumn(field: string): void;
        private getSortedColsIndexByField(field, sortedColumns?);
        /**
         * For internal use only - Get the module name.
         * @private
         */
        protected getModuleName(): string;
        private initialEnd();
        /**
         * @hidden
         */
        addEventListener(): void;
        /**
         * @hidden
         */
        removeEventListener(): void;
        /**
         * To destroy the sorting
         * @return {void}
         * @hidden
         */
        destroy(): void;
        private clickHandler(e);
        private keyPressed(e);
        private initiateSort(target, e, column);
        private showPopUp(e);
        private popUpClickHandler(e);
        private addSortIcons();
        private removeSortIcons(position?);
        private getSortColumnFromField(field);
        private updateAriaAttr();
        private refreshSortIcons();
    }
    
    /**
     * The `Toolbar` module is used to handle ToolBar actions.
     * @hidden
     */
    export class Toolbar {
        private element;
        private predefinedItems;
        toolbar: any;
        private searchElement;
        private gridID;
        private parent;
        private serviceLocator;
        private l10n;
        private items;
        private searchBoxObj;
        constructor(parent?: IGrid, serviceLocator?: ServiceLocator);
        private render();
        /**
         * Gets the toolbar of the Grid.
         * @return {Element}
         * @hidden
         */
        getToolbar(): Element;
        /**
         * Destroys the ToolBar.
         * @method destroy
         * @return {void}
         */
        destroy(): void;
        private createToolbar();
        private refreshToolbarItems(args?);
        private getItems();
        private getItem(itemObject);
        private getItemObject(itemName);
        /**
         * Enables or disables ToolBar items.
         * @param {string[]} items - Defines the collection of itemID of ToolBar items.
         * @param {boolean} isEnable - Defines the items to be enabled or disabled.
         * @return {void}
         * @hidden
         */
        enableItems(items: string[], isEnable: boolean): void;
        private toolbarClickHandler(args);
        private modelChanged(e);
        protected onPropertyChanged(e: NotifyArgs): void;
        private keyUpHandler(e);
        private search();
        private updateSearchBox();
        private wireEvent();
        private unWireEvent();
        protected addEventListener(): void;
        protected removeEventListener(): void;
        /**
         * For internal use only - Get the module name.
         */
        private getModuleName();
    }
    
    /**
     * Virtual Scrolling class
     */
    export class VirtualScroll implements IAction {
        private parent;
        private blockSize;
        private locator;
        constructor(parent: IGrid, locator?: ServiceLocator);
        getModuleName(): string;
        private instantiateRenderer();
        ensurePageSize(): void;
        addEventListener(): void;
        removeEventListener(): void;
        private refreshVirtualElement(args);
        destroy(): void;
    }
    
    /** @hidden */
    export const created: string;
    /** @hidden */
    export const destroyed: string;
    /** @hidden */
    export const load: string;
    /** @hidden */
    export const rowDataBound: string;
    /** @hidden */
    export const queryCellInfo: string;
    /** @hidden */
    export const actionBegin: string;
    /** @hidden */
    export const actionComplete: string;
    /** @hidden */
    export const actionFailure: string;
    /** @hidden */
    export const dataBound: string;
    /** @hidden */
    export const rowSelecting: string;
    /** @hidden */
    export const rowSelected: string;
    /** @hidden */
    export const rowDeselecting: string;
    /** @hidden */
    export const rowDeselected: string;
    /** @hidden */
    export const cellSelecting: string;
    /** @hidden */
    export const cellSelected: string;
    /** @hidden */
    export const cellDeselecting: string;
    /** @hidden */
    export const cellDeselected: string;
    /** @hidden */
    export const columnDragStart: string;
    /** @hidden */
    export const columnDrag: string;
    /** @hidden */
    export const columnDrop: string;
    /** @hidden */
    export const rowDragStart: string;
    /** @hidden */
    export const rowDrag: string;
    /** @hidden */
    export const rowDrop: string;
    /** @hidden */
    export const beforePrint: string;
    /** @hidden */
    export const printComplete: string;
    /** @hidden */
    export const detailDataBound: string;
    /** @hidden */
    export const toolbarClick: string;
    /** @hidden */
    export const batchAdd: string;
    /** @hidden */
    export const batchCancel: string;
    /** @hidden */
    export const batchDelete: string;
    /** @hidden */
    export const beforeBatchAdd: string;
    /** @hidden */
    export const beforeBatchDelete: string;
    /** @hidden */
    export const beforeBatchSave: string;
    /** @hidden */
    export const beginEdit: string;
    /** @hidden */
    export const cellEdit: string;
    /** @hidden */
    export const cellSave: string;
    /** @hidden */
    export const endAdd: string;
    /** @hidden */
    export const endDelete: string;
    /** @hidden */
    export const endEdit: string;
    /** @hidden */
    export const recordDoubleClick: string;
    /** @hidden */
    export const recordClick: string;
    /** @hidden */
    export const beforeDataBound: string;
    /** @hidden */
    export const beforeOpenColumnChooser: string;
    /** @hidden */
    export const resizeStart: string;
    /** @hidden */
    export const onResize: string;
    /** @hidden */
    export const resizeStop: string;
    /** @hidden */
    export const checkBoxChange: string;
    /** @hidden */
    export const beforeCopy: string;
    /** @hidden */
    export const filterChoiceRequest: string;
    /** @hidden */
    export const filterAfterOpen: string;
    /** @hidden */
    export const filterBeforeOpen: string;
    /**
     * Specifies grid internal events
     */
    /** @hidden */
    export const initialLoad: string;
    /** @hidden */
    export const initialEnd: string;
    /** @hidden */
    export const dataReady: string;
    /** @hidden */
    export const contentReady: string;
    /** @hidden */
    export const uiUpdate: string;
    /** @hidden */
    export const onEmpty: string;
    /** @hidden */
    export const inBoundModelChanged: string;
    /** @hidden */
    export const modelChanged: string;
    /** @hidden */
    export const colGroupRefresh: string;
    /** @hidden */
    export const headerRefreshed: string;
    /** @hidden */
    export const pageBegin: string;
    /** @hidden */
    export const pageComplete: string;
    /** @hidden */
    export const sortBegin: string;
    /** @hidden */
    export const sortComplete: string;
    /** @hidden */
    export const filterBegin: string;
    /** @hidden */
    export const filterComplete: string;
    /** @hidden */
    export const searchBegin: string;
    /** @hidden */
    export const searchComplete: string;
    /** @hidden */
    export const reorderBegin: string;
    /** @hidden */
    export const reorderComplete: string;
    /** @hidden */
    export const rowDragAndDropBegin: string;
    /** @hidden */
    export const rowDragAndDropComplete: string;
    /** @hidden */
    export const groupBegin: string;
    /** @hidden */
    export const groupComplete: string;
    /** @hidden */
    export const ungroupBegin: string;
    /** @hidden */
    export const ungroupComplete: string;
    /** @hidden */
    export const rowSelectionBegin: string;
    /** @hidden */
    export const rowSelectionComplete: string;
    /** @hidden */
    export const columnSelectionBegin: string;
    /** @hidden */
    export const columnSelectionComplete: string;
    /** @hidden */
    export const cellSelectionBegin: string;
    /** @hidden */
    export const cellSelectionComplete: string;
    /** @hidden */
    export const beforeCellFocused: string;
    /** @hidden */
    export const cellFocused: string;
    /** @hidden */
    export const keyPressed: string;
    /** @hidden */
    export const click: string;
    /** @hidden */
    export const destroy: string;
    /** @hidden */
    export const columnVisibilityChanged: string;
    /** @hidden */
    export const scroll: string;
    /** @hidden */
    export const columnWidthChanged: string;
    /** @hidden */
    export const columnPositionChanged: string;
    /** @hidden */
    export const rowDragAndDrop: string;
    /** @hidden */
    export const rowsAdded: string;
    /** @hidden */
    export const rowsRemoved: string;
    /** @hidden */
    export const columnDragStop: string;
    /** @hidden */
    export const headerDrop: string;
    /** @hidden */
    export const dataSourceModified: string;
    /** @hidden */
    export const refreshComplete: string;
    /** @hidden */
    export const refreshVirtualBlock: string;
    /** @hidden */
    export const dblclick: string;
    /** @hidden */
    export const toolbarRefresh: string;
    /** @hidden */
    export const bulkSave: string;
    /** @hidden */
    export const autoCol: string;
    /** @hidden */
    export const tooltipDestroy: string;
    /** @hidden */
    export const updateData: string;
    /** @hidden */
    export const editBegin: string;
    /** @hidden */
    export const editComplete: string;
    /** @hidden */
    export const addBegin: string;
    /** @hidden */
    export const addComplete: string;
    /** @hidden */
    export const saveComplete: string;
    /** @hidden */
    export const deleteBegin: string;
    /** @hidden */
    export const deleteComplete: string;
    /** @hidden */
    export const preventBatch: string;
    /** @hidden */
    export const dialogDestroy: string;
    /** @hidden */
    export const crudAction: string;
    /** @hidden */
    export const addDeleteAction: string;
    /** @hidden */
    export const destroyForm: string;
    /** @hidden */
    export const doubleTap: string;
    /** @hidden */
    export const beforeExcelExport: string;
    /** @hidden */
    export const excelExportComplete: string;
    /** @hidden */
    export const excelQueryCellInfo: string;
    /** @hidden */
    export const beforePdfExport: string;
    /** @hidden */
    export const pdfExportComplete: string;
    /** @hidden */
    export const pdfQueryCellInfo: string;
    /** @hidden */
    export const accessPredicate: string;
    /** @hidden */
    export const contextMenuClick: string;
    /** @hidden */
    export const freezeRender: string;
    /** @hidden */
    export const freezeRefresh: string;
    /** @hidden */
    export const contextMenuOpen: string;
    /** @hidden */
    export const columnMenuClick: string;
    /** @hidden */
    export const columnMenuOpen: string;
    /** @hidden */
    export const filterOpen: string;
    /** @hidden */
    export const filterDialogCreated: string;
    /** @hidden */
    export const filterMenuClose: string;
    /** @hidden */
    export const initForeignKeyColumn: string;
    /** @hidden */
    export const getForeignKeyData: string;
    /** @hidden */
    export const generateQuery: string;
    /** @hidden */
    export const showEmptyGrid: string;
    /** @hidden */
    export const foreignKeyData: string;
    /** @hidden */
    export const dataStateChange: string;
    /** @hidden */
    export const dataSourceChanged: string;
    /** @hidden */
    export const rtlUpdated: string;
    
    /**
     * Defines Actions of the Grid. They are
     * * paging
     * * refresh
     * * sorting
     * * filtering
     * * selection
     * * rowdraganddrop
     * * reorder
     * * grouping
     * * ungrouping
     */
    export type Action = 'paging' | 'refresh' | 'sorting' | 'selection' | 'filtering' | 'searching' | 'rowdraganddrop' | 'reorder' | 'grouping' | 'ungrouping' | 'batchsave' | 'virtualscroll' | 'print';
    /**
     * Defines directions of Sorting. They are
     * * Ascending
     * * Descending
     */
    export type SortDirection = 'Ascending' | 'Descending';
    /**
     * Defines types of Selection. They are
     * * Single - Allows user to select a row or cell.
     * * Multiple - Allows user to select multiple rows or cells.
     */
    export type SelectionType = 'Single' | 'Multiple';
    /**
     * Defines alignments of text, they are
     * * Left
     * * Right
     * * Center
     * * Justify
     */
    export type TextAlign = 'Left' | 'Right' | 'Center' | 'Justify';
    /**
     * Defines types of Cell
     * @hidden
     */
    export enum CellType {
        /**  Defines CellType as Data */
        Data = 0,
        /**  Defines CellType as Header */
        Header = 1,
        /**  Defines CellType as Summary */
        Summary = 2,
        /**  Defines CellType as GroupSummary */
        GroupSummary = 3,
        /**  Defines CellType as CaptionSummary */
        CaptionSummary = 4,
        /**  Defines CellType as Filter */
        Filter = 5,
        /**  Defines CellType as Indent */
        Indent = 6,
        /**  Defines CellType as GroupCaption */
        GroupCaption = 7,
        /**  Defines CellType as GroupCaptionEmpty */
        GroupCaptionEmpty = 8,
        /**  Defines CellType as Expand */
        Expand = 9,
        /**  Defines CellType as HeaderIndent */
        HeaderIndent = 10,
        /**  Defines CellType as StackedHeader */
        StackedHeader = 11,
        /**  Defines CellType as DetailHeader */
        DetailHeader = 12,
        /**  Defines CellType as DetailExpand */
        DetailExpand = 13,
        /**  Defines CellType as CommandColumn */
        CommandColumn = 14,
    }
    /**
     * Defines modes of GridLine, They are
     * * Both - Displays both the horizontal and vertical grid lines.
     * * None - No grid lines are displayed.
     * * Horizontal - Displays the horizontal grid lines only.
     * * Vertical - Displays the vertical grid lines only.
     * * Default - Displays grid lines based on the theme.
     */
    export type GridLine = 'Both' | 'None' | 'Horizontal' | 'Vertical' | 'Default';
    /**
     * Defines types of Render
     * @hidden
     */
    export enum RenderType {
        /**  Defines RenderType as Header */
        Header = 0,
        /**  Defines RenderType as Content */
        Content = 1,
        /**  Defines RenderType as Summary */
        Summary = 2,
    }
    /**
     * Defines modes of Selection, They are
     * * Row
     * * Cell
     * * Both
     */
    export type SelectionMode = 'Cell' | 'Row' | 'Both';
    /**
     * Print mode options are
     * * AllPages - Print all pages records of the Grid.
     * * CurrentPage - Print current page records of the Grid.
     */
    export type PrintMode = 'AllPages' | 'CurrentPage';
    /**
     * Defines types of Filter
     * * Menu - Specifies the filter type as menu.
     * * Excel - Specifies the filter type as excel.
     * * FilterBar - Specifies the filter type as filter bar.
     * * CheckBox - Specifies the filter type as check box.
     */
    export type FilterType = 'FilterBar' | 'Excel' | 'Menu' | 'CheckBox';
    /**
     * Filter bar mode options are
     * * OnEnter - Initiate filter operation after Enter key is pressed.
     * * Immediate -  Initiate filter operation after certain time interval. By default time interval is 1500 ms.
     */
    export type FilterBarMode = 'OnEnter' | 'Immediate';
    /**
     * Defines the aggregate types.
     */
    export type AggregateType = 'Sum' | 'Average' | 'Max' | 'Min' | 'Count' | 'TrueCount' | 'FalseCount' | 'Custom';
    /**
     * Defines the wrap mode.
     * * Both -  Wraps both header and content.
     * * Header - Wraps header alone.
     * * Content - Wraps content alone.
     */
    export type WrapMode = 'Both' | 'Header' | 'Content';
    /**
     * Defines Multiple Export Type.
     */
    export type MultipleExportType = 'AppendToSheet' | 'NewSheet';
    /**
     * Defines Predefined toolbar items.
     * @hidden
     */
    export type ToolbarItems = 'Add' | 'Delete' | 'Update' | 'Cancel' | 'Edit' | 'Search' | 'ColumnChooser' | 'Print' | 'PdfExport' | 'ExcelExport' | 'CsvExport' | 'WordExport';
    /**
     * Defines the cell content's overflow mode. The available modes are
     * * `Clip` -  Truncates the cell content when it overflows its area.
     * * `Ellipsis` -  Displays ellipsis when the cell content overflows its area.
     * * `EllipsisWithTooltip` - Displays ellipsis when the cell content overflows its area
     * also it will display tooltip while hover on ellipsis applied cell.
     */
    export type ClipMode = 'Clip' | 'Ellipsis' | 'EllipsisWithTooltip';
    /**
     * Defines the Command Buttons type.
     * * Edit -  Edit the current record.
     * * Delete - Delete the current record.
     * * Save - Save the current edited record.
     * * Cancel - Cancel the edited state.
     */
    export type CommandButtonType = 'Edit' | 'Delete' | 'Save' | 'Cancel';
    /**
     * Defines the default items of context menu.
     */
    export type ContextMenuItem = 'AutoFitAll' | 'AutoFit' | 'Group' | 'Ungroup' | 'Edit' | 'Delete' | 'Save' | 'Cancel' | 'Copy' | 'PdfExport' | 'ExcelExport' | 'CsvExport' | 'SortAscending' | 'SortDescending' | 'FirstPage' | 'PrevPage' | 'LastPage' | 'NextPage';
    /**
     * Defines the default items of Column menu.
     */
    export type ColumnMenuItem = 'AutoFitAll' | 'AutoFit' | 'Group' | 'Ungroup' | 'SortAscending' | 'SortDescending' | 'ColumnChooser' | 'Filter';
    /**
     * Defines Predefined toolbar items.
     * @hidden
     */
    export enum ToolbarItem {
        Add = 0,
        Edit = 1,
        Update = 2,
        Delete = 3,
        Cancel = 4,
        Print = 5,
        Search = 6,
        ColumnChooser = 7,
        PdfExport = 8,
        ExcelExport = 9,
        CsvExport = 10,
        WordExport = 11,
    }
    export type PdfPageSize = 'Letter' | 'Note' | 'Legal' | 'A0' | 'A1' | 'A2' | 'A3' | 'A4' | 'A5' | 'A6' | 'A7' | 'A8' | 'A9' | 'B0' | 'B1' | 'B2' | 'B3' | 'B4' | 'B5' | 'Archa' | 'Archb' | 'Archc' | 'Archd' | 'Arche' | 'Flsa' | 'HalfLetter' | 'Letter11x17' | 'Ledger';
    export type PageOrientation = 'Landscape' | 'Portrait';
    export type ContentType = 'Image' | 'Line' | 'PageNumber' | 'Text';
    export type PdfPageNumberType = 'LowerLatin' | 'LowerRoman' | 'UpperLatin' | 'UpperRoman' | 'Numeric' | 'Arabic';
    export type PdfDashStyle = 'Solid' | 'Dash' | 'Dot' | 'DashDot' | 'DashDotDot';
    /**
     * Defines PDF horizontal alignment.
     */
    export type PdfHAlign = 'Left' | 'Right' | 'Center' | 'Justify';
    /**
     * Defines PDF vertical alignment.
     */
    export type PdfVAlign = 'Top' | 'Bottom' | 'Middle';
    /**
     * Defines Export Type.
     */
    export type ExportType = 'CurrentPage' | 'AllPages';
    /**
     * Defines Excel horizontal alignment.
     */
    export type ExcelHAlign = 'Left' | 'Right' | 'Center' | 'Fill';
    /**
     * Defines Excel vertical alignment.
     */
    export type ExcelVAlign = 'Top' | 'Bottom' | 'Center' | 'Justify';
    /**
     * Defines border line style.
     */
    export type BorderLineStyle = 'Thin' | 'Thick';
    export type CheckState = 'Check' | 'Uncheck' | 'Intermediate' | 'None';
    /**
     * Defines mode of cell selection.
     * * Ascending
     * * Descending
     */
    export type CellSelectionMode = 'Flow' | 'Box';
    /**
     * Defines modes of editing.
     * * Ascending
     * * Descending
     */
    export type EditMode = 'Normal' | 'Dialog' | 'Batch';
    
    /**
     * Interface for a class SortDescriptor
     */
    export interface SortDescriptorModel {
    
        /**
         * Defines the field name of sort column. 
         * @default ''
         */
        field?: string;
    
        /**
         * Defines the direction of sort column. 
         * @default ''
         */
        direction?: SortDirection;
    
    }
    
    /**
     * Interface for a class SortSettings
     */
    export interface SortSettingsModel {
    
        /**
         * Specifies the columns to sort at initial rendering of Grid.
         * Also user can get current sorted columns. 
         * @default []
         */
        columns?: SortDescriptorModel[];
    
        /**
         * If `allowUnsort` set to false the user can not get the grid in unsorted state by clicking the sorted column header.
         * @default true
         */
        allowUnsort?: boolean;
    
    }
    
    /**
     * Interface for a class Predicate
     */
    export interface PredicateModel {
    
        /**
         * Defines the field name of the filter column.  
         * @default ''
         */
        field?: string;
    
        /**
         * Defines the operator to filter records. The available operators and its supported data types are:
         * <table> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * Operator<br/></td><td colspan=1 rowspan=1> 
         * Description<br/></td><td colspan=1 rowspan=1> 
         * Supported Types<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * startswith<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value begins with the specified value.<br/></td><td colspan=1 rowspan=1> 
         * String<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * endswith<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value ends with the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * <br/>String<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * contains<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value contains the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * <br/>String<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * equal<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value is equal to the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * <br/>String | Number | Boolean | Date<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * notequal<br/></td><td colspan=1 rowspan=1> 
         * Checks for values that are not equal to the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * <br/>String | Number | Boolean | Date<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * greaterthan<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value is greater than the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * Number | Date<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * greaterthanorequal<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value is greater than or equal to the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * <br/>Number | Date<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * lessthan<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value is less than the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * <br/>Number | Date<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * lessthanorequal<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the value is less than or equal to the specified value.<br/><br/></td><td colspan=1 rowspan=1> 
         * <br/>Number | Date<br/></td></tr> 
         * </table> 
         * @default null
         */
        operator?: string;
    
        /**
         * Defines the value used to filter records. 
         * @default ''
         */
        value?: string | number | Date | boolean;
    
        /**
         * If match case set to true, then filter records with exact match or else  
         * filter records with case insensitive(uppercase and lowercase letters treated as same).  
         * @default null
         */
        matchCase?: boolean;
    
        /**
         * If ignoreAccent is set to true, then filter ignores the diacritic characters or accents while filtering.
         * @default false
         */
        ignoreAccent?: boolean;
    
        /**
         * Defines the relationship between one filter query and another by using AND or OR predicate.   
         * @default null
         */
        predicate?: string;
    
        /**
         * @hidden 
         * Defines the actual filter value for the filter column.  
         */
        actualFilterValue?: Object;
    
        /**
         * @hidden 
         * Defines the actual filter operator for the filter column.  
         */
        actualOperator?: Object;
    
        /**
         * @hidden 
         * Defines the type of the filter column.  
         */
        type?: string;
    
        /**
         * @hidden 
         * Defines the predicate of filter column.  
         */
        ejpredicate?: Object;
    
    }
    
    /**
     * Interface for a class FilterSettings
     */
    export interface FilterSettingsModel {
    
        /**
         * Specifies the columns to be filtered at initial rendering of the Grid. You can also get the columns that were currently filtered.
         * @default []
         */
        columns?: PredicateModel[];
    
        /**
         * Defines options for filtering type. The available options are 
         * * `Menu` - Specifies the filter type as menu. 
         * * `CheckBox` - Specifies the filter type as checkbox.      
         * * `FilterBar` - Specifies the filter type as filterbar.  
         * * `Excel` - Specifies the filter type as checkbox.      
         * @default FilterBar 
         */
        type?: FilterType;
    
        /**
         * Defines the filter bar modes. The available options are,
         * * `OnEnter`: Initiates filter operation after Enter key is pressed. 
         * * `Immediate`: Initiates filter operation after a certain time interval. By default, time interval is 1500 ms. 
         * @default OnEnter
         */
        mode?: FilterBarMode;
    
        /**
         * Shows or hides the filtered status message on the pager.  
         * @default true
         */
        showFilterBarStatus?: boolean;
    
        /**
         * Defines the time delay (in milliseconds) in filtering records when the `Immediate` mode of filter bar is set. 
         * @default 1500 
         */
        immediateModeDelay?: number;
    
        /**
         * The `operators` is used to override the default operators in filter menu. This should be defined by type wise
         * (string, number, date and boolean). Based on the column type, this customize operator list will render in filter menu.
         * 
         * > Check the [`Filter Menu Operator`](./how-to.html#customizing-filter-menu-operators-list) customization.
         * @default null
         */
        operators?: ICustomOptr;
    
        /**
         * If ignoreAccent set to true, then filter ignores the diacritic characters or accents while filtering.
         * 
         * > Check the [`Diacritics`](./filtering.html/#diacritics) filtering.
         * @default false
         */
        ignoreAccent?: boolean;
    
    }
    
    /**
     * Interface for a class SelectionSettings
     */
    export interface SelectionSettingsModel {
    
        /**
         * Grid supports row, cell, and both (row and cell) selection mode. 
         * @default Row
         */
        mode?: SelectionMode;
    
        /**
         * The cell selection modes are flow and box. It requires the selection 
         * [`mode`](./api-selectionSettings.html#mode-selectionmode) to be either cell or both.
         * * `Flow`: Selects the range of cells between start index and end index that also includes the other cells of the selected rows.
         * * `Box`: Selects the range of cells within the start and end column indexes that includes in between cells of rows within the range.
         * @default Flow
         */
        cellSelectionMode?: CellSelectionMode;
    
        /**
         * Defines options for selection type. They are 
         * * `Single`: Allows selection of only a row or a cell. 
         * * `Multiple`: Allows selection of multiple rows or cells. 
         * @default Single 
         */
        type?: SelectionType;
    
        /**
         * If 'checkboxOnly' set to true, then the Grid selection is allowed only through checkbox.
         * 
         * > To enable checkboxOnly selection, should specify the column type as`checkbox`.
         * @default false 
         */
        checkboxOnly?: boolean;
    
        /**
         * If 'persistSelection' set to true, then the Grid selection is persisted on all operations.
         * For persisting selection in the Grid, any one of the column should be enabled as a primary key.
         * @default false 
         */
        persistSelection?: boolean;
    
    }
    
    /**
     * Interface for a class SearchSettings
     */
    export interface SearchSettingsModel {
    
        /**
         * Specifies the collection of fields included in search operation. By default, bounded columns of the Grid are included.  
         * @default []
         */
        fields?: string[];
    
        /**
         * Specifies the key value to search Grid records at initial rendering. 
         * You can also get the current search key.
         * @default ''
         */
        key?: string;
    
        /**
         * Defines the operator to search records. The available operators are:
         * <table> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * Operator<br/></td><td colspan=1 rowspan=1> 
         * Description<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * startswith<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the string begins with the specified string.<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * endswith<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the string ends with the specified string.<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * contains<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the string contains the specified string. <br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * equal<br/></td><td colspan=1 rowspan=1> 
         * Checks whether the string is equal to the specified string.<br/></td></tr> 
         * <tr> 
         * <td colspan=1 rowspan=1> 
         * notequal<br/></td><td colspan=1 rowspan=1> 
         * Checks for strings not equal to the specified string. <br/></td></tr> 
         * </table> 
         * @default contains 
         */
        operator?: string;
    
        /**
         * If `ignoreCase` is set to false, searches records that match exactly, else  
         * searches records that are case insensitive(uppercase and lowercase letters treated the same).  
         * @default true 
         */
        ignoreCase?: boolean;
    
    }
    
    /**
     * Interface for a class RowDropSettings
     */
    export interface RowDropSettingsModel {
    
        /**
         * Defines the ID of droppable component on which row drop should occur.   
         * @default null
         */
        targetID?: string;
    
    }
    
    /**
     * Interface for a class TextWrapSettings
     */
    export interface TextWrapSettingsModel {
    
        /**
         * Defines the `wrapMode` of the Grid. The available modes are: 
         * * `Both`: Wraps both the header and content. 
         * * `Content`: Wraps the header alone.
         * * `Header`: Wraps the content alone. 
         * @default Both
         */
        wrapMode?: WrapMode;
    
    }
    
    /**
     * Interface for a class GroupSettings
     */
    export interface GroupSettingsModel {
    
        /**
         * If `showDropArea` is set to true, the group drop area element will be visible at the top of the Grid.     
         * @default true 
         */
        showDropArea?: boolean;
    
        /**
         * If `showToggleButton` set to true, then the toggle button will be showed in the column headers which can be used to group
         * or ungroup columns by clicking them.
         * @default false   
         */
        showToggleButton?: boolean;
    
        /**
         * If `showGroupedColumn` is set to false, it hides the grouped column after grouping.  
         * @default false  
         */
        showGroupedColumn?: boolean;
    
        /**
         * If `showUngroupButton` set to false, then ungroup button is hidden in dropped element.  
         * It can be used to ungroup the grouped column when click on ungroup button. 
         * @default true 
         */
        showUngroupButton?: boolean;
    
        /**
         * If `disablePageWiseAggregates` set to true, then the group aggregate value will
         * be calculated from the whole data instead of paged data and two requests will be made for each page
         * when Grid bound with remote service.
         * @default false
         */
        disablePageWiseAggregates?: boolean;
    
        /**
         * Specifies the column names to group at initial rendering of the Grid.  
         * You can also get the currently grouped columns.   
         * @default []  
         */
        columns?: string[];
    
        /**
         * The Caption Template allows user to display the string or HTML element in group caption.
         * > It accepts either the [template string](http://ej2.syncfusion.com/documentation/base/template-engine.html) or the HTML element ID.
         * @default ''
         */
        captionTemplate?: string;
    
    }
    
    /**
     * Interface for a class EditSettings
     */
    export interface EditSettingsModel {
    
        /**
         * If `allowAdding` is set to true, new records can be added to the Grid.  
         * @default false 
         */
        allowAdding?: boolean;
    
        /**
         * If `allowEditing` is set to true, values can be updated in the existing record.  
         * @default false 
         */
        allowEditing?: boolean;
    
        /**
         * If `allowDeleting` is set to true, existing record can be deleted from the Grid.    
         * @default false 
         */
        allowDeleting?: boolean;
    
        /**
         * Defines the mode to edit. The available editing modes are:
         * * Normal
         * * Dialog
         * * Batch
         * @default Normal 
         */
        mode?: EditMode;
    
        /**
         * If `allowEditOnDblClick` is set to false, Grid will not allow editing of a record on double click. 
         * @default true 
         */
        allowEditOnDblClick?: boolean;
    
        /**
         * if `showConfirmDialog` is set to false, confirm dialog does not show when batch changes are saved or discarded.
         * @default true 
         */
        showConfirmDialog?: boolean;
    
        /**
         * If `showDeleteConfirmDialog` is set to true, confirm dialog will show delete action. You can also cancel delete command.
         * @default false 
         */
        showDeleteConfirmDialog?: boolean;
    
    }
    
    /**
     * Interface for a class Grid
     */
    export interface GridModel extends ej.base.ComponentModel{
    
        /**
         * Defines the schema of dataSource. 
         * If the `columns` declaration is empty or undefined then the `columns` are automatically generated from data source.     
         * @default []   
         */
        columns?: Column[] | string[] | ColumnModel[];
    
        /**
         * If `enableAltRow` is set to true, the grid will render with `e-altrow` CSS class to the alternative tr elements.    
         * > Check the [`AltRow`](./row.html#styling-alternate-rows) to customize the styles of alternative rows.
         * @default true 
         */
        enableAltRow?: boolean;
    
        /**
         * If `enableHover` is set to true, the row hover is enabled in the Grid.
         * @default true     
         */
        enableHover?: boolean;
    
        /**
         * Enables or disables the key board interaction of Grid.          
         * @hidden 
         * @default true     
         */
        allowKeyboard?: boolean;
    
        /**
         * If `allowTextWrap` set to true,  
         * then text content will wrap to the next line when its text content exceeds the width of the Column Cells. 
         * @default false     
         */
        allowTextWrap?: boolean;
    
        /**
         * Configures the text wrap in the Grid.  
         * @default {wrapMode:"Both"}     
         */
        textWrapSettings?: TextWrapSettingsModel;
    
        /**
         * If `allowPaging` is set to true, the pager renders at the footer of the Grid. It is used to handle page navigation in the Grid.
         * 
         * > Check the [`Paging`](./paging.html) to configure the grid pager.
         * @default false     
         */
        allowPaging?: boolean;
    
        /**
         * Configures the pager in the Grid.  
         * @default {currentPage: 1, pageSize: 12, pageCount: 8, enableQueryString: false, pageSizes: false, template: null}     
         */
        pageSettings?: PageSettingsModel;
    
        /**
         * If `enableVirtualization` set to true, then the Grid will render only the rows visible within the view-port
         * and load subsequent rows on vertical scrolling. This helps to load large dataset in Grid.
         * @default false
         */
        enableVirtualization?: boolean;
    
        /**
         * If `enableColumnVirtualization` set to true, then the Grid will render only the columns visible within the view-port
         * and load subsequent columns on horizontal scrolling. This helps to load large dataset of columns in Grid.
         * @default false
         */
        enableColumnVirtualization?: boolean;
    
        /**
         * Configures the search behavior in the Grid. 
         * @default { ignoreCase: true, fields: [], operator: 'contains', key: '' }    
         */
        searchSettings?: SearchSettingsModel;
    
        /**
         * If `allowSorting` is set to true, it allows sorting of grid records when column header is clicked.  
         * 
         * > Check the [`Sorting`](./sorting.html) to customize its default behavior.
         * @default false    
         */
        allowSorting?: boolean;
    
        /**
         * If `allowMultiSorting` set to true, then it will allow the user to sort multiple column in the grid.
         * > `allowSorting` should be true.
         * @default false
         */
        allowMultiSorting?: boolean;
    
        /**
         * If `allowExcelExport` set to true, then it will allow the user to export grid to Excel file.
         * 
         * > Check the [`ExcelExport`](./excel-exporting.html) to configure exporting document.
         * @default false    
         */
        allowExcelExport?: boolean;
    
        /**
         * If `allowPdfExport` set to true, then it will allow the user to export grid to Pdf file.
         * 
         * > Check the [`Pdfexport`](./pdf-exporting.html) to configure the exporting document.
         * @default false    
         */
        allowPdfExport?: boolean;
    
        /**
         * Configures the sort settings.  
         * @default {columns:[]}    
         */
        sortSettings?: SortSettingsModel;
    
        /**
         * If `allowSelection` is set to true, it allows selection of (highlight row) Grid records by clicking it.  
         * @default true        
         */
        allowSelection?: boolean;
    
        /**
         * The `selectedRowIndex` allows you to select a row at initial rendering. 
         * You can also get the currently selected row index.
         * @default -1        
         */
        selectedRowIndex?: number;
    
        /**
         * Configures the selection settings.  
         * @default {mode: 'Row', cellSelectionMode: 'Flow', type: 'Single'}    
         */
        selectionSettings?: SelectionSettingsModel;
    
        /**
         * If `allowFiltering` set to true the filter bar will be displayed. 
         * If set to false the filter bar will not be displayed. 
         * Filter bar allows the user to filter grid records with required criteria.   
         * 
         * > Check the [`Filtering`](./filtering.html) to customize its default behavior.     
         * @default false    
         */
        allowFiltering?: boolean;
    
        /**
         * If `allowReordering` is set to true, Grid columns can be reordered. 
         * Reordering can be done by drag and drop of a particular column from one index to another index.  
         * > If Grid is rendered with stacked headers, reordering is allowed only at the same level as the column headers.
         * @default false    
         */
        allowReordering?: boolean;
    
        /**
         * If `allowResizing` is set to true, Grid columns can be resized.      
         * @default false    
         */
        allowResizing?: boolean;
    
        /**
         * If `allowRowDragAndDrop` is set to true, you can drag and drop grid rows at another grid.    
         * @default false    
         */
        allowRowDragAndDrop?: boolean;
    
        /**
         * Configures the row drop settings.  
         * @default {targetID: ''}   
         */
        rowDropSettings?: RowDropSettingsModel;
    
        /**
         * Configures the filter settings of the Grid.  
         * @default {columns: [], type: 'FilterBar', mode: 'Immediate', showFilterBarStatus: true, immediateModeDelay: 1500 , operators: {}}    
         */
        filterSettings?: FilterSettingsModel;
    
        /**
         * If `allowGrouping` set to true, then it will allow the user to dynamically group or ungroup columns.  
         * Grouping can be done by drag and drop columns from column header to group drop area. 
         * 
         * > Check the [`Grouping`](./grouping.html) to customize its default behavior.
         * @default false    
         */
        allowGrouping?: boolean;
    
        /**
         * If `showColumnMenu` set to true, then it will enable the column menu options in each columns.
         * 
         * > Check the [`Column menu`](./columns.html#column-menu) for its configuration.
         * @default false    
         */
        showColumnMenu?: boolean;
    
        /**
         * Configures the group settings. 
         * @default {showDropArea: true, showToggleButton: false, showGroupedColumn: false, showUngroupButton: true, columns: []}    
         */
        groupSettings?: GroupSettingsModel;
    
        /**
         * Configures the edit settings. 
         * @default { allowAdding: false, allowEditing: false, allowDeleting: false, mode:'Normal',
         * allowEditOnDblClick: true, showConfirmDialog: true, showDeleteConfirmDialog: false }    
         */
        editSettings?: EditSettingsModel;
    
        /**
         * Configures the Grid aggregate rows.
         * > Check the [`Aggregates`](./aggregates.html) for its configuration.
         * @default []
         */
        aggregates?: AggregateRowModel[];
    
        /**
         * If `showColumnChooser` is set to true, it allows you to dynamically show or hide columns.  
         * 
         * > Check the [`ColumnChooser`](./columns.html#column-chooser) for its configuration.
         * @default false    
         */
        showColumnChooser?: boolean;
    
        /**
         * Defines the scrollable height of the grid content.    
         * @default auto    
         */
        height?: string | number;
    
        /**
         * Defines the Grid width.    
         * @default auto    
         */
        width?: string | number;
    
        /**
         * Defines the mode of grid lines. The available modes are, 
         * * `Both`: Displays both horizontal and vertical grid lines. 
         * * `None`: No grid lines are displayed.
         * * `Horizontal`: Displays the horizontal grid lines only. 
         * * `Vertical`: Displays the vertical grid lines only.
         * * `Default`: Displays grid lines based on the theme.
         * @default Default
         */
        gridLines?: GridLine;
    
        /**
         * The row template that renders customized rows from the given template. 
         * By default, Grid renders a table row for every data source item.
         * > * It accepts either [template string](../base/template-engine.html) or HTML element ID.   
         * > * The row template must be a table row.  
         * 
         * > Check the [`Row Template`](./row.html) customization.
         */
        rowTemplate?: string;
    
        /**
         * The detail template allows you to show or hide additional information about a particular row.
         *  
         * > It accepts either the [template string](../base/template-engine.html) or the HTML element ID.
         * 
         * {% codeBlock src="grid/detail-template-api/index.ts" %}{% endcodeBlock %}
         */
        detailTemplate?: string;
    
        /**
         * Defines Grid options to render child Grid. 
         * It requires the [`queryString`](./api-grid.html#querystring-string) for parent 
         * and child relationship. 
         * 
         * > Check the [`Child Grid`](./hierarchy-grid.html) for its configuration.
         */
        childGrid?: GridModel;
    
        /**
         * Defines the relationship between parent and child datasource. It acts as the foreign key for parent datasource.       
         */
        queryString?: string;
    
        /**
         * Defines the print modes. The available print modes are   
         * * `AllPages`: Prints all pages of the Grid. 
         * * `CurrentPage`: Prints the current page of the Grid.
         * @default AllPages
         */
        printMode?: PrintMode;
    
        /**
         * It is used to render grid table rows. 
         * If the `dataSource` is an array of JavaScript objects, 
         * then Grid will create instance of [`DataManager`](../data/api-dataManager.html) 
         * from this `dataSource`. 
         * If the `dataSource` is an existing [`DataManager`](../data/api-dataManager.html),
         *  the Grid will not initialize a new one. 
         * 
         * > Check the available [`Adaptors`](../data/adaptors.html) to customize the data operation.
         * @default []    
         */
        dataSource?: Object | ej.data.DataManager | DataResult;
    
        /**
         * Defines the height of Grid rows.
         * @default null
         */
        rowHeight?: number;
    
        /**
         * Defines the external [`Query`](../data/api-query.html) 
         * that will be executed along with data processing.    
         * @default null    
         */
        query?: ej.data.Query;
    
        /**
         * Defines the currencyCode format of the Grid columns
         * @private
         */
        currencyCode?: string;
    
        /**
         * `toolbar` defines the ToolBar items of the Grid. 
         * It contains built-in and custom toolbar items. 
         * If a string value is assigned to the `toolbar` option, it is considered as the template for the whole Grid ToolBar. 
         * If an array value is assigned, it is considered as the list of built-in and custom toolbar items in the Grid's Toolbar. 
         * <br><br>     
         * The available built-in ToolBar items are:
         * * Add: Adds a new record.
         * * Edit: Edits the selected record.
         * * Update: Updates the edited record.
         * * Delete: Deletes the selected record.
         * * Cancel: Cancels the edit state.
         * * Search: Searches records by the given key.
         * * Print: Prints the Grid.
         * * ExcelExport - Export the Grid to Excel(excelExport() method manually to make export.)
         * * PdfExport - Export the Grid to PDF(pdfExport() method manually to make export.)
         * * CsvExport - Export the Grid to CSV(csvExport() method manually to make export.)<br><br>
         * The following code example implements the custom toolbar items.
         * 
         *  > Check the [`Toolbar`](./toolbar.html#custom-toolbar-items) to customize its default items.
         * 
         * {% codeBlock src="grid/toolbar-api/index.ts" %}{% endcodeBlock %}
         * @default null
         */
        toolbar?: ToolbarItems[] | string[] | ej.navigations.ItemModel[] | ToolbarItem[];
    
        /**
         * `contextMenuItems` defines both built-in and custom context menu items.
         * <br><br> 
         * The available built-in items are,  
         * * `AutoFitAll` - Auto fit the size of all columns. 
         * * `AutoFit` - Auto fit the current column.
         * * `Group` - Group by current column. 
         * * `Ungroup` - Ungroup by current column.
         * * `Edit` - Edit the current record.
         * * `Delete` - Delete the current record.
         * * `Save` - Save the edited record.
         * * `Cancel` - Cancel the edited state.
         * * `Copy` - Copy the selected records.
         * * `PdfExport` - Export the grid as Pdf format.
         * * `ExcelExport` - Export the grid as Excel format.
         * * `CsvExport` - Export the grid as CSV format.
         * * `SortAscending` - Sort the current column in ascending order.
         * * `SortDescending` - Sort the current column in descending order.
         * * `FirstPage` - Go to the first page.
         * * `PrevPage` - Go to the previous page.
         * * `LastPage` - Go to the last page.
         * * `NextPage` - Go to the next page.
         * 
         * @default null
         */
        contextMenuItems?: ContextMenuItem[] | ContextMenuItemModel[];
    
        /**
         * `columnMenuItems` defines both built-in and custom column menu items.
         * <br><br> 
         * The available built-in items are,
         * * `AutoFitAll` - Auto fit the size of all columns. 
         * * `AutoFit` - Auto fit the current column.
         * * `Group` - Group by current column. 
         * * `Ungroup` - Ungroup by current column.
         * * `SortAscending` - Sort the current column in ascending order.
         * * `SortDescending` - Sort the current column in descending order.
         * * `Filter` - Filter options will show based on filterSettings property like checkbox filter, excel filter, menu filter.
         * @default null
         */
        columnMenuItems?: ColumnMenuItem[] | ColumnMenuItemModel[];
    
        /**
         * @hidden
         * It used to render toolbar template
         * @default null
         */
        toolbarTemplate?: string;
    
        /**
         * @hidden
         * It used to render pager template
         * @default null
         */
        pagerTemplate?: string;
    
        /**
         * Gets or sets the number of frozen rows.
         * @default 0
         */
        frozenRows?: number;
    
        /**
         * Gets or sets the number of frozen columns.
         * @default 0
         */
        frozenColumns?: number;
    
        /**
         * Triggers when the component is created.
         * @event 
         */
        created?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when the component is destroyed. 
         * @event 
         */
        destroyed?: ej.base.EmitType<Object>;
    
        /**
         * This event allows customization of Grid properties before rendering.
         * @event 
         */
        load?: ej.base.EmitType<Object>;
    
        /**
         * Triggered every time a request is made to access row information, element, or data. 
         * This will be triggered before the row element is appended to the Grid element.
         * @event 
         */
        rowDataBound?: ej.base.EmitType<RowDataBoundEventArgs>;
    
        /**
         * Triggered every time a request is made to access cell information, element, or data.
         * This will be triggered before the cell element is appended to the Grid element.
         * @event 
         */
        queryCellInfo?: ej.base.EmitType<QueryCellInfoEventArgs>;
    
        /**
         * Triggers when Grid actions such as sorting, filtering, paging, grouping etc., starts. 
         * @event
         */
        actionBegin?: ej.base.EmitType<PageEventArgs | GroupEventArgs | FilterEventArgs | SearchEventArgs | SortEventArgs | AddEventArgs | SaveEventArgs | EditEventArgs | DeleteEventArgs>;
    
        /**
         * Triggers when Grid actions such as sorting, filtering, paging, grouping etc. are completed. 
         * @event 
         */
        actionComplete?: ej.base.EmitType<PageEventArgs | GroupEventArgs | FilterEventArgs | SearchEventArgs | SortEventArgs | AddEventArgs | SaveEventArgs | EditEventArgs | DeleteEventArgs>;
    
        /**
         * Triggers when any Grid action failed to achieve the desired results. 
         * @event 
         */
        actionFailure?: ej.base.EmitType<FailureEventArgs>;
    
        /**
         * Triggers when data source is populated in the Grid.
         * @event 
         */
        dataBound?: ej.base.EmitType<Object>;
    
        /**
         * Triggers when record is double clicked.
         * @event 
         */
        recordDoubleClick?: ej.base.EmitType<RecordDoubleClickEventArgs>;
    
        /**
         * Triggers before row selection occurs.
         * @event 
         */
        rowSelecting?: ej.base.EmitType<RowSelectingEventArgs>;
    
        /**
         * Triggers after a row is selected.
         * @event 
         */
        rowSelected?: ej.base.EmitType<RowSelectEventArgs>;
    
        /**
         * Triggers before deselecting the selected row.
         * @event 
         */
        rowDeselecting?: ej.base.EmitType<RowDeselectEventArgs>;
    
        /**
         * Triggers when a selected row is deselected.
         * @event 
         */
        rowDeselected?: ej.base.EmitType<RowDeselectEventArgs>;
    
        /**
         * Triggers before any cell selection occurs.
         * @event 
         */
        cellSelecting?: ej.base.EmitType<CellSelectingEventArgs>;
    
        /**
         * Triggers after a cell is selected.
         * @event 
         */
        cellSelected?: ej.base.EmitType<CellSelectEventArgs>;
    
        /**
         * Triggers before the selected cell is deselecting.
         * @event 
         */
        cellDeselecting?: ej.base.EmitType<CellDeselectEventArgs>;
    
        /**
         * Triggers when a particular selected cell is deselected.
         * @event 
         */
        cellDeselected?: ej.base.EmitType<CellDeselectEventArgs>;
    
        /**
         * Triggers when column header element drag (move) starts. 
         * @event  
         */
        columnDragStart?: ej.base.EmitType<ColumnDragEventArgs>;
    
        /**
         * Triggers when column header element is dragged (moved) continuously. 
         * @event  
         */
        columnDrag?: ej.base.EmitType<ColumnDragEventArgs>;
    
        /**
         * Triggers when a column header element is dropped on the target column. 
         * @event  
         */
        columnDrop?: ej.base.EmitType<ColumnDragEventArgs>;
    
        /**
         * Triggers after print action is completed.  
         * @event 
         */
        printComplete?: ej.base.EmitType<PrintEventArgs>;
    
        /**
         * Triggers before the print action starts.  
         * @event 
         */
        beforePrint?: ej.base.EmitType<PrintEventArgs>;
    
        /**
         * Triggers before exporting each cell to PDF document. You can also customize the PDF cells.
         * @event 
         */
        pdfQueryCellInfo?: ej.base.EmitType<PdfQueryCellInfoEventArgs>;
    
        /**
         * Triggers before exporting each cell to Excel file.
         * You can also customize the Excel cells.
         * @event
         */
        excelQueryCellInfo?: ej.base.EmitType<ExcelQueryCellInfoEventArgs>;
    
        /**
         * Triggers before Grid data is exported to Excel file.
         * @event
         */
        beforeExcelExport?: ej.base.EmitType<Object>;
    
        /**
         * Triggers after Grid data is exported to Excel file.
         * @event
         */
        excelExportComplete?: ej.base.EmitType<ExcelExportCompleteArgs>;
    
        /**
         * Triggers before Grid data is exported to PDF document.
         * @event
         */
        beforePdfExport?: ej.base.EmitType<Object>;
    
        /**
         * Triggers after Grid data is exported to PDF document.
         * @event
         */
        pdfExportComplete?: ej.base.EmitType<PdfExportCompleteArgs>;
    
        /**
         * Triggers after detail row expands.
         * > This event triggers at initial expand.  
         * @event 
         */
        detailDataBound?: ej.base.EmitType<DetailDataBoundEventArgs>;
    
        /**
         * Triggers when row element's drag(move) starts. 
         * @event  
         */
        rowDragStart?: ej.base.EmitType<RowDragEventArgs>;
    
        /**
         * Triggers when row elements are dragged (moved) continuously. 
         * @event  
         */
        rowDrag?: ej.base.EmitType<RowDragEventArgs>;
    
        /**
         * Triggers when row elements are dropped on the target row. 
         * @event  
         */
        rowDrop?: ej.base.EmitType<RowDragEventArgs>;
    
        /**
         * Triggers when toolbar item is clicked.
         * @event
         */
        toolbarClick?: ej.base.EmitType<ej.navigations.ClickEventArgs>;
    
        /**
         * Triggers before the columnChooser open.
         * @event
         */
        beforeOpenColumnChooser?: ej.base.EmitType<ColumnChooserEventArgs>;
    
        /**
         * Triggers when records are added in batch mode.   
         * @event
         */
        batchAdd?: ej.base.EmitType<BatchAddArgs>;
    
        /**
         * Triggers when records are deleted in batch mode.
         * @event
         */
        batchDelete?: ej.base.EmitType<BatchDeleteArgs>;
    
        /**
         * Triggers before records are added in batch mode.
         * @event
         */
        beforeBatchAdd?: ej.base.EmitType<BeforeBatchAddArgs>;
    
        /**
         * Triggers before records are deleted in batch mode.
         * @event
         */
        beforeBatchDelete?: ej.base.EmitType<BeforeBatchDeleteArgs>;
    
        /**
         * Triggers before records are saved in batch mode.
         * @event
         */
        beforeBatchSave?: ej.base.EmitType<BeforeBatchSaveArgs>;
    
        /**
         * Triggers before the record is to be edit.
         * @event
         */
        beginEdit?: ej.base.EmitType<BeginEditArgs>;
    
        /**
         * Triggers when the cell is being edited.
         * @event
         */
        cellEdit?: ej.base.EmitType<CellEditArgs>;
    
        /**
         * Triggers when cell is saved.
         * @event
         */
        cellSave?: ej.base.EmitType<CellSaveArgs>;
    
        /**
         * Triggers when column resize starts.
         * @event
         */
        resizeStart?: ej.base.EmitType<ResizeArgs>;
    
        /**
         * Triggers o