#!/usr/bin/python3
# An autogenerated selection of SurpriseDog's common functions relevant to this project.
# To see how this file was created visit: https://github.com/SurpriseDog/Star-Wrangler

import os
import re
import sys
import math
import time
import queue
import shutil
import random
import socket
import threading
import subprocess
from urllib.parse import urlparse


def quote(text):
    "Wrap a string in the minimum number of quotes to be quotable"
    for q in ('"', "'", "'''"):
        if q not in text:
            break
    else:
        return repr(text)
    if "\n" in text:
        q = "'''"
    return q + text + q


def set_volume(level=80):
    "Set computer master volume"
    srun("amixer -D pulse sset Master " + str(level) + "% on")


def srun(*cmds, **kargs):
    "Split all text before quick run"
    return quickrun(flatten([str(item).split() for item in cmds]), **kargs)


def get_volume():
    cur_level = []
    for line in srun('amixer -D pulse'):
        if 'Playback' in line and '%' in line:
            cur_level.append(int(re.split('[\\[\\]]', line)[1][:-1]))
    return int(sum(cur_level) / len(cur_level))


def play(filename, volume=80, player='', opts='', **kargs):
    '''Set the volume, play an audio file and then reset the volume to previous level
    Passes other args onto run'''
    current_vol = get_volume()
    set_volume(volume)

    ext = os.path.splitext(filename)[-1][1:].lower()
    if not player:
        if ext in ('ogg',):
            player = 'ogg123'
        elif ext in ('mp3',):
            player = 'mpg123'
        else:
            player = 'mpv'  # 'aplay'
    quickrun(player, opts, filename, **kargs)
    set_volume(current_vol)


def map_nested(func, array):
    "Apply a function to a nested array and return it"
    out = []
    for item in array:
        if type(item) not in (tuple, list):
            out.append(func(item))
        else:
            out.append(map_nested(func, item))
    return out


def bisect_small(lis, num):
    '''Given a sorted list, returns the index of the biggest number <= than num
    Unlike bisect will never return an index which doesn't exist'''
    end = len(lis) - 1
    for x in range(end + 1):
        if lis[x] >= num:
            return max(x - 1, 0)
    else:
        return end


def is_num(num):
    "Is the string a number?"
    if str(num).strip().replace('.', '', 1).replace('e', '', 1).isdigit():
        return True
    return False


class ConvertDataSize():
    '''
    Convert data size. Given a user input size like
    "80%+10G -1M" = 80% of the blocksize + 10 gigabytes - 1 Megabyte
    '''

    def __init__(self, blocksize=1e9, binary_prefix=1000, rounding=0):
        self.blocksize = blocksize                  # Device blocksize for multiplying by a percentage
        self.binary_prefix = binary_prefix          # 1000 or 1024 byte kilobytes
        self.rounding = rounding                    # Round to sector sizes

    def _process(self, arg):
        arg = arg.strip().upper().replace('B', '')
        if not arg:
            return 0

        start = arg[0]
        end = arg[-1]

        if start == '-':
            return self.blocksize - self._process(arg[1:])

        if '+' in arg:
            return sum(map(self._process, arg.split('+')))

        if '-' in arg:
            args = arg.split('-')
            val = self._process(args.pop(0))
            for a in args:
                val -= self._process(a)
                return val

        if end in 'KMGTPEZY':
            return self._process(arg[:-1]) * self.binary_prefix ** (' KMGTPEZY'.index(end))

        if end == '%':
            if arg.count('%') > 1:
                raise ValueError("Extra % in arg:", arg)        # pylint: disable=W0715
            arg = float(arg[:-1])
            if not 0 <= arg <= 100:
                print("Percentages must be between 0 and 100, not", str(arg) + '%')
                return None
            else:
                return int(arg / 100 * self.blocksize)

        if is_num(arg):
            return float(arg)
        else:
            print("Could not understand arg:", arg)
            return None

    def __call__(self, arg):
        "Pass string to convert"
        val = self._process(arg)
        if val is None:
            return None
        val = int(val)
        if self.rounding:
            return val // self.rounding * self.rounding
        else:
            return val


def unique_filename(filename):
    "Given a filename, add a counter if needed to ensure it is unique"
    if not os.path.exists(filename):
        return filename
    filename, ext = os.path.splitext(filename)
    if not filename.endswith('.'):
        filename = filename + '.'
    extra = 1
    while os.path.exists(filename + str(extra) + ext):
        extra += 1
    return filename + str(extra) + ext


def search_list(expr, the_list, get='list', func='match', ignorecase=True, searcher=None):
    '''Search for expression in each item in list (or dictionary!)
    get = 'list'    = Return all items found <Default>
          'first'   = Return the first value found, otherwise None
          'only'    = Return only one item, error if more
          'exactly' = Return one item, error if more or less than one

    searcher = Custom lamda function'''

    if not searcher:
        # func = dict(search='in').get('search', func)
        # Avoiding regex now in case substring has a regex escape character
        if ignorecase:
            expr = expr.lower()
        if func in ('in', 'search'):
            if ignorecase:
                def searcher(expr, item):         # pylint: disable=E0102
                    return expr in item.lower()
            else:
                def searcher(expr, item):         # pylint: disable=E0102
                    return expr in item
        elif func == 'match':
            if ignorecase:
                def searcher(expr, item):         # pylint: disable=E0102
                    return item.lower().startswith(expr)
            else:
                def searcher(expr, item):         # pylint: disable=E0102
                    return item.startswith(expr)
        else:
            # Could have nested these, but this is faster.
            raise ValueError("Unknown search type:", func)

    output = []
    for item in the_list:
        if searcher(expr, item):
            if isinstance(the_list, dict):
                output.append(the_list[item])
            else:
                output.append(item)
            if get == 'first':
                return output[0]
    else:
        if get == 'first':
            return None

    if get == 'one':
        if len(output) == 1:
            return output[0]
        elif not output:
            return None
        else:
            raise ValueError("Too many items found in list!")

    if get == 'only':
        if len(output) != 1:
            raise ValueError("Found", len(output), "items in list!")
        else:
            return output[0]

    return output


def crop(text, cut=64, ending='...'):
    "Crop text down a length with ending..."
    if len(text) <= cut:
        return text
    else:
        cut -= len(ending)
        cut = 0 if cut < 0 else cut
        return text[:cut] + ending


def check_internet(timeout=8, tries=1):
    "Check internet connection, return True if on"
    ips = ['8.8.8.8', '8.8.4.4', '1.1.1.1']
    for tri in range(tries):
        if tri:
            time.sleep(2)
        ip = random.choice(ips)
        try:
            socket.create_connection((ip, 53), timeout)
            return True
        except OSError:
            pass
    return False


def safe_filename(filename, src="/ ", dest="-_", no_http=True, length=200,
                  forbidden='''*?\\/:<>|'"''', replacement='.'):
    '''Convert urls and the like to safe filesystem names
    src, dest is the character translation table
    length is the max length allowed, set to 200 so rdiff-backup doesn't get upset
    forbidden characters are replaced with the replacement character'''
    if no_http:
        if filename.startswith("http") or filename.startswith("www."):
            netloc = urlparse(filename).netloc
            filename = filename[filename.find(netloc):]
            filename = re.sub("^www\\.", "", filename)
            filename = filename.strip('/')
    filename = filename.translate(filename.maketrans(src, dest)).strip()
    return ''.join(c if c not in forbidden else replacement for c in filename.strip())[:length]


def spawn(func, *args, daemon=True, delay=0, **kargs):
    '''Spawn a function to run seperately and return the que
    waits for delay seconds before running
    Get the results with que.get()
    daemon = running in background, will shutdown automatically when main thread exits
    Check if the thread is still running with thread.is_alive()
    print('func=', func, id(func))'''
    # replaces fork_cmd, mcall

    def worker():
        if delay:
            time.sleep(delay)
        ret = func(*args, **kargs)
        que.put(ret)

    que = queue.Queue()
    # print('args=', args)
    thread = threading.Thread(target=worker)
    thread.daemon = daemon
    thread.start()
    return que, thread


class DotDict(dict):
    '''
    Example:
    m = dotdict({'first_name': 'Eduardo'}, last_name='Pool', age=24, sports=['Soccer'])

    Modified from:
    https://stackoverflow.com/questions/2352181/how-to-use-a-dot-to-access-members-of-dictionary
    to set unlimited chained .variables like DotDict().tom.bob = 3
    '''

    def __init__(self, *args, **kwargs):
        super(DotDict, self).__init__(*args, **kwargs)
        for arg in args:
            if isinstance(arg, dict):
                for k, v in arg.items():
                    self[k] = v

        if kwargs:
            for k, v in kwargs.items():
                self[k] = v

    def __getattr__(self, attr):
        if attr in self:
            return self.get(attr)
        else:
            self[attr] = DotDict()
            return self[attr]

    def __setattr__(self, key, value):
        self.__setitem__(key, value)

    def __contains__(self, key):
        return bool(key in self.__dict__)

    def __setitem__(self, key, value):
        super(DotDict, self).__setitem__(key, value)
        self.__dict__.update({key: value})

    def __delattr__(self, item):
        self.__delitem__(item)

    def __delitem__(self, key):
        super(DotDict, self).__delitem__(key)
        del self.__dict__[key]


def error(*args, header='\nError:', err=RuntimeError, **kargs):
    eprint(*args, header=header, v=3, **kargs)
    raise err


def quickrun(*cmd, check=False, encoding='utf-8', errors='replace', mode='w', stdin=None,
             verbose=0, testing=False, ofile=None, trifecta=False, printme=False, hidewarning=False, **kargs):
    '''Run a command, list of commands as arguments or any combination therof and return
    the output is a list of decoded lines.
    check    = if the process exits with a non-zero exit code then quit
    testing  = Print command and don't do anything.
    ofile    = output file
    mode     = output file write mode
    trifecta = return (returncode, stdout, stderr)
    stdin    = standard input (auto converted to bytes)
    printme  = Print to stdout instead of returning it, returns code instead
    '''
    cmd = list(map(str, flatten(cmd)))
    if len(cmd) == 1:
        cmd = cmd[0]

    if testing:
        print("Not running command:", cmd)
        return []

    if verbose:
        print("Running command:", cmd)
        print("               =", ' '.join(cmd))

    if ofile:
        output = open(ofile, mode=mode)
    else:
        output = subprocess.PIPE

    if stdin:
        if type(stdin) != bytes:
            stdin = stdin.encode()

    if printme:
        if trifecta:
            error("quickrun cant use both printme and trifecta")
        # todo: make more realtime https://stackoverflow.com/questions/803265/getting-realtime-output-using-subprocess
        ret = subprocess.run(cmd, check=check, stdout=sys.stdout, stderr=sys.stderr, input=stdin, **kargs)
        code = ret.returncode

    else:
        # Run the command and get return value
        ret = subprocess.run(cmd, check=check, stdout=output, stderr=output, input=stdin, **kargs)
        code = ret.returncode
        stdout = ret.stdout.decode(encoding=encoding, errors=errors).splitlines() if ret.stdout else []
        stderr = ret.stderr.decode(encoding=encoding, errors=errors).splitlines() if ret.stderr else []

    if ofile:
        output.close()
        return []

    if trifecta:
        return code, stdout, stderr

    if code and not hidewarning:
        warn("Process returned code:", code)

    if printme:
        return ret.returncode

    if not hidewarning:
        for line in stderr:
            print(line)

    return stdout


def list_get(lis, index, default=''):
    '''Fetch a value from a list if it exists, otherwise return default
    Now accepts negative indexes'''

    length = len(lis)
    if -length <= index < length:
        return lis[index]
    else:
        return default


def percent(num, digits=0):
    if not digits:
        return str(int(num * 100)) + '%'
    else:
        return sig(num * 100, digits) + '%'


def flatten(tree):
    "Flatten a nested list, tuple or dict of any depth into a flat list"
    # For big data sets use this: https://stackoverflow.com/a/45323085/11343425
    out = []
    if isinstance(tree, dict):
        for key, val in tree.items():
            if type(val) in (list, tuple, dict):
                out += flatten(val)
            else:
                out.append({key: val})

    else:
        for item in tree:
            if type(item) in (list, tuple, dict):
                out += flatten(item)
            else:
                out.append(item)
    return out


def sorted_array(array, column=-1, reverse=False):
    "Return sorted 2d array line by line"
    pairs = [(line[column], index) for index, line in enumerate(array)]
    for _val, index in sorted(pairs, reverse=reverse):
        # print(index, val)
        yield array[index]


def avg(lis):
    "Average a list"
    return sum(lis) / len(lis)


def read_file(filename):
    "Read an entire file into text"
    with open(filename, 'r') as f:
        return f.read()


def dict_valtokey(dic, val):
    "Take a dictionary value and return the first key found:"
    for k, v in dic.items():
        if val == v:
            return k
    return None


def read_state(filename, multiline=False, forget=False, verbose=True, cleanup_age=86400):
    "todo make this a class"
    '''
    Maintains open file handles to read the state of a file without wasting resources
    forget =        open a file without maintaing open file handle
    multiline =     Return every stdout line instead of just the first.
    cleanup_age =   Minimum age to keep an old unaccessed file around before cleaning it up
    verbose =       1   Print a notification each time a new file opened
    verbose =       2   Print a notification each time a file is accesssed
    '''

    if verbose >= 2:
        print("Reading:", filename)

    # Open a file and don't add it to the log
    if forget:
        with open(filename, 'r') as f:
            if multiline:
                return list(map(str.strip, f.readlines()))
            else:
                return f.readline().strip()

    # Keep a dictionary of open files
    self = read_state
    now = time.time()
    if not hasattr(self, 'filenames'):
        self.filenames = dict()         # dictionary of filenames to file handles
        self.history = dict()           # When was the last time file was opened?
        self.last_cleanup = now         # Cleanup old files, occassionally
        # There is a limit to the number of open file handles.
        self.limit = 64                 # int(resource.getrlimit(resource.RLIMIT_NOFILE)[0] / 4)

    # Cleanup old unused file handles
    if cleanup_age and now - self.last_cleanup > cleanup_age / 2:
        self.last_cleanup = now
        for name in list(self.history.keys()):
            if name == filename:
                continue
            if now - self.history[name] > cleanup_age:
                print("Removing old file handle:", name)
                f = self.filenames[name]
                del self.filenames[name]
                del self.history[name]
                f.close()

    # Remove files if past the limit of file handles
    if len(self.filenames) > self.limit:
        earliest = sorted(list(self.history.values()))[0]
        name = dict_valtokey(self.history, earliest)
        print("\nToo many open handles! Removing:", name)
        f = self.filenames[name]
        f.close()
        del self.filenames[name]
        del self.history[name]

    # Open the file
    if filename not in self.filenames:
        if verbose:
            print("Opening", '#' + str(len(self.filenames) + 1) + ':', filename)
        try:
            f = open(filename, 'r')
        except BaseException:
            raise ValueError("Could not open: " + filename)
        self.filenames[filename] = f
    else:
        f = self.filenames[filename]
        f.seek(0)
    self.history[filename] = now

    # Return data
    if multiline:
        return list(map(str.strip, f.readlines()))
    else:
        return f.readline().strip()


def read_val(file):
    "Read a number from an open file handle"
    file.seek(0)
    return int(file.read())


class Eprinter:
    '''Drop in replace to print errors if verbose level higher than setup level
    To replace every print statement type: from common import eprint as print

    eprint(v=-1)    # Normally hidden messages
    eprint(v=0)     # Default level
    eprint(v=1)     # Priority messages
    eprint(v=2)     # Warnings
    eprint(v=3)     # Errors
    '''

    # Setup: eprint = Eprinter(<verbosity level>).eprint
    # Simple setup: from common import eprint
    # Usage: eprint(messages, v=1)

    # Don't forget they must end in 'm'
    BOLD = '\033[1m'
    WARNING = '\x1b[1;33;40m'
    FAIL = '\x1b[0;31;40m'
    END = '\x1b[0m'

    def __init__(self, verbose=0):
        self.level = verbose
        self.history = []

        # If string starts with '\n', look at history to make sure previous newlines don't exist
        self.autonewlines = True

    def newlines(self, num=1):
        "Print the required number of newlines after checking history to make sure they exist."
        lines = sum([1 for line in self.history[-num:] if not line.strip()])
        num -= lines
        if num:
            print('\n' * (num), end='')
        return num


    def eprint(self, *args, v=0, color=None, header=None, **kargs):
        '''Print to stderr
        Custom color example: color='1;33;40'
        More colors: https://stackoverflow.com/a/21786287/11343425
        '''
        verbose = v
        # Will print if verbose >= level
        if verbose < self.level:
            return 0

        if not color:
            if v == 2 and not color:
                color = f"{self.WARNING}"
            if v >= 3 and not color:
                color = f"{self.FAIL}" + f"{self.BOLD}"
        else:
            color = '\x1b[' + color + 'm'

        msg = ' '.join(map(str, args))
        if self.autonewlines:
            match = re.match('^\n*', msg)
            if match:
                num = self.newlines(match.span()[1])
                if num:
                    # print('created', num, 'newlines', repr(msg[:64]))
                    msg = msg.lstrip('\n')


        self.history += msg.splitlines()
        if len(self.history) > 64:
            self.history = self.history[64:]

        if header:
            msg = header + ' ' + msg
        if color:
            print(color + msg + f"{self.END}", file=sys.stderr, **kargs)
        else:
            print(msg, file=sys.stderr, **kargs)
        return len(msg)


def undent(text, tab=''):
    "Remove whitespace at the beginning of lines of text"
    return '\n'.join([tab + line.lstrip() for line in text.splitlines()])


def warn(*args, header="\n\nWarning:", sep=' ', delay=1 / 64, confirm=False):
    msg = undent(sep.join(list(map(str, args))))
    time.sleep(eprint(msg, header=header, v=2) * delay)
    if confirm:
        _nul = input()


def qwarn(*args, header="\n\nWarning:", sep=' '):
    "Quick warn for scripts without delay"
    msg = undent(sep.join(list(map(str, args))))
    eprint(msg, header=header, v=2)


def mkdir(target, exist_ok=True, **kargs):
    "Make a directory without fuss"
    os.makedirs(target, exist_ok=exist_ok, **kargs)


def sig(num, digits=3):
    "Return number formatted for significant digits"
    num = float(num)
    if num == 0:
        return '0'
    negative = '-' if num < 0 else ''
    num = abs(num)
    power = math.log(num, 10)
    if num < 1:
        num = int(10**(-int(power) + digits) * num)
        return negative + '0.' + '0' * -int(power) + str(int(num)).rstrip('0')
    elif power < digits - 1:
        return negative + ('{0:.' + str(digits) + 'g}').format(num)
    else:
        return negative + str(int(num))


def rfs(num, mult=1000, digits=3, order=' KMGTPEZYB', suffix='B', space=' '):
    '''A "readable" file size
    mult is the value of a kilobyte in the filesystem. (1000 or 1024)
    order is the name of each level
    suffix is a trailing character (B for Bytes)
    space is the space between '3.14 M' for 3.14 Megabytes
    '''
    if abs(num) < mult:
        return sig(num) + space + suffix

    # https://cmte.ieee.org/futuredirections/2020/12/01/what-about-brontobytes/
    bb = mult**9
    if bb <= num < 2 * bb:
        print("Fun Fact: The DNA of all the cells of 100 Brontosauruses " + \
              "combined contains around a BrontoByte of data storage")
    if num >= bb:
        # Comment this out when BrontoBytes become mainstream
        order = list(order)
        order[9] = 'BrontoBytes'
        suffix = ''

    # Faster than using math.log:
    for x in range(len(order) - 1, -1, -1):
        magnitude = mult**x
        if abs(num) >= magnitude:
            return sig(num / magnitude, digits) + space + (order[x] + suffix).rstrip()
    return str(num) + suffix        # Never called, but needed for pylint


def check_install(*programs, msg='', quitonerr=True):
    '''Check if program is installed (and reccomend procedure to install)
    programs is the list of programs to test
    prints msg if it can't find any and returns False'''

    errors = 0
    for program in programs:
        paths = shutil.which(program)
        if not paths:
            errors += 1
            print('\n', program, 'is not installed.')
    if errors:
        if msg:
            if type(msg) == str:
                print("To install type:", msg)
            else:
                print("To install type:")
                for m in msg:
                    print('\t' + m)
        else:
            print("Please install to continue...")
        if quitonerr:
            sys.exit(1)
        return False
    return True


def gohome():
    os.chdir(os.path.dirname(sys.argv[0]))


def itercount(start=0, step=1):
    "Save an import itertools"
    x = start
    while True:
        yield x
        x += step


eprint = Eprinter(verbose=1).eprint     # pylint: disable=C0103

'''
&&&&%%%%%&@@@@&&&%%%%##%%%#%%&@@&&&&%%%%%%/%&&%%%%%%%%%%%&&&%%%%%&&&@@@@&%%%%%%%
%%%%%%%%&@&(((((#%%&%%%%%%%%%&@@&&&&&&%%%&&&&&%%%%%%%%%%%&&&&%&%#((((/#@@%%%%%%%
&&%%%%%%&@(*,,,,,,,/%&%%%%%%%&@@&&&&&%%&&&&%%&&%%%%%%%%%%&&&%#*,,,,,,*/&@&%%%%%%
%%%%%%%&@&/*,,,*,*,,*/%&%%%%%&@@&&&&&&%%&&&&&&&%%%%%%&%%%&&%*,,,,,,,,**#@&&%%%%%
&&&&&%%&@#(**********,*(#&%%%&@&&&&%%%%%%%%%&&&%%%%%%&%&&#*****,*******#@&&%%%%%
&&&%%%&&#/***/*****/*,**,*%&%&@@&&&&&&&&&&&&&&&%%%%%%&&#*,,,*/******/***(%&%%%%%
&&&%%%&%/*****///////**,,,,*/%%&&@@@@@@@@@@@@@@@@&&%#*,,,*,*(///////*****#%&%%%%
@@&%%#&#/,,,*/(//((((//**,,*/#&@@@@@&&&&&&&&&&@@@@@%(/*,,**/(/(((/(//*,,*(&&%%%%
&&&%##&#*,,,*////((((/*///(&@&@@&&&#%((//(/###%&@&@@@@#//**//(#(///***,.,/&&%%%%
%%%%%#%#*,,,**////(///((#&&&%@&%%(/*,,......,,/(#%&&&@@@%((/(/#(///**,,,,(&%%%%%
&&%%%#%%/,..***//(#(#%%&@@@&@%(*.,,..       ...,.,/#@&@@@&&%#(((///**,..,#%%%%%%
%&%%%%%#*,****/(##&@@@&@@@@&%*,....           ....,,(&@@@@@@&@&%((//****,(%%%%%%
%&%%%%%#/,**/#&@@@&@@@@@@@&(*,......    .     ..,..,.(&@@@@@@@&@@@&%#**,*(%%%%%%
&&%%%%#&#(#&@@@&@@@@@@@@%((#@@%&&((,,,,,..,,(**(%@@&@%##(&@@@@@@@@&&@@%#(%%%%%%%
&&&%%%%%&&&&&&@@@@@@%###%@(,%&/@@&(%(/*,..,*/%##&&,%@(*&@#((%&@@@@@@&&@&%%%%&&%%
&&%%%%%%&&&@@@@@@@@#((*#@%,#%%&@#%(/**//,****/(#%%%&&%*(@@*/#(&@@@@@@@&&%%%%%%%%
&&&%%%%%&@@@@&%#/,,,,*,(/%&@@&((%(*,*,,*,**,,*,*#%(#@@&%((**,,,,*#(%&@@&&%%%%%%%
&&&%%%%%@@@@%*/*,...,*,,/*#(//#****,***********,**/#/##(/*,*,...,*/*/&@@&%%&%%%%
&&%%%%%%&@@@(//,....,,*/****/,,/**************/***/,,//**/**,....,*//&@@&%%&%%%%
&&&%%%%%&@@%(/*,. ...,****/*/(//*%&@@&%%%%%%&&&&//*/(*/**/**......,/*#&@&%&&&&%%
&&%%%%%%&@@%(**,,....,/**/((/,#&&&&&%#((((((%&&&@&%/*/(/**/*,. ..,,*/((#@&&&&&&%
%&%%%%%%&&#(/**,..,,,***/((,./%&%&&&@&(/#((#@@&&&%&%,,/((*,/*,,..,,,///(%&%&&&&&
&&%%%%%%&#,**,.,..,,*(//(/,,.,&&&@#&@@##%(#&@&%%@&&#.,,/(((//*,,..,,**,*&%&&%&&&
&&%##%%%#/**,,,,..,*/((((*...,,#&##%(#%%&%%###%(%&/,.. **((((/,...,,,,**(%%#%%%&
&&%####(**,,,.,,.,,/(/(//*,,..../%&(##%&&&%%(#%&#, .. .**//(/(*,,..,.,,**/((#%%%
&&&%#///*,........,/(((//**,.   ,,(#%%%%%%&#%##**.   ,,*//((((*,........,*//(%%%
%%%%(/**...       .,/(((///*., .,*(#(%%%%%%%%##/*,..,,*///((/*.      .....**/(%%
%%%%#(,..          .,/((/(//****,/(((###%#%(#///**,,**/((/((*,          .,.,(%%%
&&%%%#/*...          ,*/(/(/((%%&#&#(/%./.*%(#%#%#&&(((/(/*,.          ..,**(&%%
&&%%%%(*.....          ..*((/**(#&&&&&&&%%%&%&&&%(/,*/((*..           .,..*(&&%%
&&%%%%&#*.      .        */(#/*,,*/((%#%%%%%((**,.*/(#(/,       .       ,(%&%%&%
%%%%%%&%#//**,..           .**(((*,...,,**,,..*,/((/*,.          ...,,//(#%%%%%%
%%%&&&%(/*,**,..,,.,..       .,,**//**,*,,,*,////*,,.        .,.,...,,,**//#%&%%
%%%&&%#/*,*,.    ...      ..         ...  ,.. .       .       ...   ..,,*/(#%&%%
&&&&&%(((*.*... . .*,.   .           .*%%#(,.          .    .*,. ..,.,,**/(%#&%%

Generated by https://github.com/SurpriseDog/Star-Wrangler
a Python tool for picking only the required code from source files
written by SurpriseDog at: https://github.com/SurpriseDog
2022-06-30
'''
