/*
 * SonarQube PHP Plugin
 * Copyright (C) 2010-2025 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the Sonar Source-Available License Version 1, as published by SonarSource SA.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Sonar Source-Available License for more details.
 *
 * You should have received a copy of the Sonar Source-Available License
 * along with this program; if not, see https://sonarsource.com/license/ssal/
 */
package org.sonar.php.checks;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import org.sonar.check.Rule;
import org.sonar.php.cfg.LiveVariablesAnalysis;
import org.sonar.php.cfg.LiveVariablesAnalysis.LiveVariables;
import org.sonar.php.cfg.LiveVariablesAnalysis.VariableUsage;
import org.sonar.php.tree.symbols.Scope;
import org.sonar.php.utils.collections.ListUtils;
import org.sonar.plugins.php.api.cfg.CfgBlock;
import org.sonar.plugins.php.api.cfg.ControlFlowGraph;
import org.sonar.plugins.php.api.symbols.Symbol;
import org.sonar.plugins.php.api.tree.Tree;
import org.sonar.plugins.php.api.tree.Tree.Kind;
import org.sonar.plugins.php.api.tree.expression.AssignmentExpressionTree;
import org.sonar.plugins.php.api.tree.expression.ExpressionTree;
import org.sonar.plugins.php.api.tree.statement.ExpressionStatementTree;
import org.sonar.plugins.php.api.tree.statement.TryStatementTree;
import org.sonar.plugins.php.api.visitors.PHPSubscriptionCheck;
import org.sonar.plugins.php.api.visitors.PHPVisitorCheck;

@Rule(key = "S1854")
public class DeadStoreCheck extends PHPSubscriptionCheck {

  private static final Set<String> BASIC_LITERAL_VALUES = Set.of(
    "true",
    "false",
    "1",
    "0",
    "0.0",
    "-1",
    "null",
    "''",
    "array()",
    "[]",
    "\"\"");
  private static final String MESSAGE_TEMPLATE = "Remove this useless assignment to local variable '%s'.";

  @Override
  public List<Tree.Kind> nodesToVisit() {
    return Arrays.asList(
      Kind.FUNCTION_DECLARATION,
      Kind.FUNCTION_EXPRESSION,
      Kind.METHOD_DECLARATION);
  }

  @Override
  public void visitNode(Tree tree) {
    ControlFlowGraph cfg = ControlFlowGraph.build(tree, context());
    if (cfg == null) {
      return;
    }
    Scope scope = context().symbolTable().getScopeFor(tree);
    if (scope == null || scope.hasUnresolvedCompact()) {
      return;
    }
    if (containsTryCatchBlock(tree)) {
      return;
    }
    LiveVariablesAnalysis lva = LiveVariablesAnalysis.analyze(cfg, context().symbolTable());
    cfg.blocks().forEach(block -> verifyBlock(block, lva.getLiveVariables(block), lva.getReadSymbols()));
  }

  private static boolean containsTryCatchBlock(Tree tree) {
    TryVisitor tryVisitor = new TryVisitor();
    tree.accept(tryVisitor);
    return tryVisitor.hasTry;
  }

  /**
   * Bottom-up approach, keeping track of which variables will be read by successor elements.
   */
  private void verifyBlock(CfgBlock block, LiveVariables blockLiveVariables, Set<Symbol> readSymbols) {
    Set<Symbol> willBeRead = new HashSet<>(blockLiveVariables.getOut());
    for (Tree element : ListUtils.reverse(block.elements())) {
      Map<Symbol, VariableUsage> usagesInElement = blockLiveVariables.getVariableUsages(element);
      for (Map.Entry<Symbol, VariableUsage> symbolWithUsage : usagesInElement.entrySet()) {
        Symbol symbol = symbolWithUsage.getKey();
        if (outOfScope(readSymbols, symbol)) {
          // These cases are verified by other checks
          continue;
        }
        VariableUsage usage = symbolWithUsage.getValue();
        if (usage.isWrite() && !usage.isRead()) {
          if (!willBeRead.contains(symbol) && !shouldSkip(element, symbol)) {
            context().newIssue(this, element, String.format(MESSAGE_TEMPLATE, symbol.name()));
          }
          willBeRead.remove(symbol);
        } else if (usage.isRead()) {
          willBeRead.add(symbol);
        }
      }
    }
  }

  private static boolean outOfScope(Set<Symbol> readSymbols, Symbol symbol) {
    return !readSymbols.contains(symbol) || symbol.is(Symbol.Kind.PARAMETER);
  }

  private static boolean shouldSkip(Tree element, Symbol symbol) {
    return symbol.hasModifier("static") || symbol.hasModifier("global") || isInitializedToBasicValue(element) || isReferenceValue(symbol);
  }

  private static boolean isReferenceValue(Symbol symbol) {
    return symbol.declaration().getParent().is(Kind.REFERENCE_VARIABLE, Kind.ASSIGNMENT_BY_REFERENCE) ||
      symbol.usages().stream().map(Tree::getParent).map(Tree::getParent).anyMatch(t -> t.is(Kind.ASSIGNMENT_BY_REFERENCE));
  }

  private static boolean isInitializedToBasicValue(Tree element) {
    if (!element.is(Kind.EXPRESSION_STATEMENT)) {
      return false;
    }
    ExpressionTree inner = ((ExpressionStatementTree) element).expression();
    if (!inner.is(Kind.ASSIGNMENT)) {
      return false;
    }
    ExpressionTree rightmostValue = extractRightmostValue((AssignmentExpressionTree) inner);
    return BASIC_LITERAL_VALUES.contains(rightmostValue.toString().toLowerCase(Locale.ENGLISH));
  }

  /**
   * For "$a = $b = foo();", it will return the tree for "foo()"
   */
  private static ExpressionTree extractRightmostValue(AssignmentExpressionTree assignment) {
    AssignmentExpressionTree rightMostAssignment = assignment;
    ExpressionTree rightValue = rightMostAssignment.value();
    while (rightValue.is(Kind.ASSIGNMENT)) {
      rightMostAssignment = (AssignmentExpressionTree) rightValue;
      rightValue = rightMostAssignment.value();
    }
    return rightValue;
  }

  private static class TryVisitor extends PHPVisitorCheck {
    private boolean hasTry;

    @Override
    public void visitTryStatement(TryStatementTree tree) {
      hasTry = true;
    }
  }

}
