import cx from 'classnames';
import React, { useEffect, forwardRef, useImperativeHandle } from 'react';
import { Form, Input, Select, Tooltip, Button } from 'antd';
import { RiQuestionLine, RiEdit2Line, RiLock2Line } from 'react-icons/ri';

import { useTranslation } from 'react-i18next';

import { Setting } from '@leaa/api/src/entrys';
import { errorMsg } from '@leaa/dashboard/src/utils';
import { IOnValidateFormResult } from '@leaa/dashboard/src/interfaces';
import { SettingUpdateOne } from '@leaa/api/src/dtos/setting';

import { FormCard } from '@leaa/dashboard/src/components';
import { FORM_SIZE } from '@leaa/dashboard/src/constants';

import style from './style.module.less';

interface IProps {
  items?: Setting[];
  loading?: boolean;
  onClickLabelEditCallback: (setting: Setting) => void;
  className?: string;
}

const formItemLayout = {
  labelCol: {
    xs: { span: 24 },
    sm: { span: 8 },
  },
  wrapperCol: {
    xs: { span: 24 },
    sm: { span: 16 },
  },
};

export const buildTypeDom = (setting: Pick<Setting, 'type' | 'name' | 'options'>) => {
  let dom = <span>----</span>;

  const { type, name, options } = setting;

  if (type === 'input') {
    dom = <Input placeholder={name} />;
  }

  if (['radio'].includes(type)) {
    dom = (
      <Select>
        {options &&
          options.split(/[\n]/).map((option: string) => (
            <Select.Option key={option} value={option}>
              {option}
            </Select.Option>
          ))}
      </Select>
    );
    // dom = <Input.TextArea placeholder={name} rows={3} />;
  }

  if (['checkbox', 'textarea'].includes(type)) {
    dom = <Input.TextArea placeholder={name} rows={3} />;
  }

  return dom;
};

export const SettingListForm = forwardRef((props: IProps, ref: React.Ref<any>) => {
  const { t } = useTranslation();
  const [form] = Form.useForm();

  const buildLabelDom = (setting: Setting) => {
    return (
      <span>
        <Tooltip
          title={
            <>
              <RiQuestionLine /> {setting.description}
            </>
          }
          trigger="hover"
        >
          <div className={cx(style['label-box'])}>
            <Button
              type="link"
              onClick={() => props.onClickLabelEditCallback(setting)}
              className={cx(style['label-button'])}
            >
              <strong className={cx(style['label-text'])}>
                {setting.private ? <RiLock2Line className={style['private-icon']} /> : null}
                {setting.name}
              </strong>

              <RiEdit2Line />
            </Button>

            <code>{setting.slug}</code>
          </div>
        </Tooltip>
      </span>
    );
  };

  const onValidateForm = async (): IOnValidateFormResult<SettingUpdateOne[]> => {
    try {
      const result = await form.validateFields();

      return Object.keys(result).map((k) => ({ id: k, value: result[k] }));
    } catch (err) {
      return errorMsg(err.errorFields[0]?.errors[0]);
    }
  };

  const onRefreshForm = (items?: Setting[]) => {
    if (!items) return undefined;

    return items.forEach((item) => {
      form.setFieldsValue({
        [item.id]: item.value,
      });
    });
  };

  // eslint-disable-next-line react-hooks/exhaustive-deps
  useEffect(() => onRefreshForm(props.items), [form, props.items]);
  useImperativeHandle(ref, () => ({ form, onValidateForm }));

  return (
    <div className={cx(style['setting-list-form-wrapper'], props.className)}>
      <FormCard>
        <Form
          form={form}
          name="setting-list"
          layout="horizontal"
          hideRequiredMark
          {...formItemLayout}
          className={style['form-wrapper']}
          size={FORM_SIZE}
        >
          {props.items?.map((item) => (
            <Form.Item
              key={item.id}
              name={item.id}
              rules={[{ required: true }]}
              colon={false}
              label={buildLabelDom(item)}
            >
              {buildTypeDom(item)}
            </Form.Item>
          ))}
        </Form>
      </FormCard>
    </div>
  );
});
