"""This script generates a version 0 JSON file.

Run with roseau-load-flow==0.2.1, NOT WITH THE CURRENT VERSION::

    uv run --no-project --script roseau/load_flow/io/tests/data/generators/v0.py
"""

# Script metadata generated by `uv`
# /// script
# requires-python = ">=3.10,<3.12"
# dependencies = [
#     "roseau-load-flow==0.2.1",
# ]
# ///
from pathlib import Path

from shapely.geometry import Point

import roseau.load_flow as rlf

try:
    # Version 0 does not have NETWORK_JSON_VERSION
    from roseau.load_flow.io.dict import NETWORK_JSON_VERSION
except ImportError:
    NETWORK_JSON_VERSION = 0
else:
    raise ValueError(
        f"This script is meant to generate version 0 JSON files, current version: {NETWORK_JSON_VERSION}"
        f"\nExecute with `uv run --no-project --script roseau/load_flow/io/tests/data/generators/v0.py`"
    )

OUTPUT = Path(__file__).parent.parent / f"network_json_v{NETWORK_JSON_VERSION}.json"

# Buses
buses = {
    1: rlf.VoltageSource(
        id=1,
        n=4,
        ground=None,
        source_voltages=[
            11547.005383792515 + 0.0j,
            -5773.502691896258 + -10000.000000179687j,
            -5773.502691896258 + 10000.000000179687j,
        ],
        geometry=Point(0.0, 0.0),
    ),
    2: rlf.Bus(id=2, n=3, geometry=Point(0.0, 1.0)),
    3: rlf.Bus(id=3, n=4, geometry=Point(0.0, 1.0)),
    4: rlf.Bus(id=4, n=4, geometry=Point(0.0, 1.0)),
    5: rlf.Bus(id=5, n=4, geometry=Point(0.0, 1.0)),
    6: rlf.Bus(id=6, n=3, geometry=Point(0.0, 1.0)),
    7: rlf.Bus(id=7, n=4, geometry=Point(0.0, 1.0)),
    8: rlf.Bus(id=8, n=3, geometry=Point(0.0, 1.0)),
    9: rlf.Bus(id=9, n=4, geometry=Point(0.0, 1.0)),
    10: rlf.Bus(id=10, n=4, geometry=Point(0.0, 1.0)),
    11: rlf.Bus(id=11, n=4, geometry=Point(0.0, 1.0)),
    12: rlf.Bus(id=12, n=3, geometry=Point(0.0, 1.0)),
    13: rlf.Bus(id=13, n=4, geometry=Point(0.0, 1.0)),
}

# Grounds and potential references
ground = rlf.Ground()
for bus_id in (1, 3, 4, 5, 7, 9, 10, 11, 13):
    ground.connect(buses[bus_id])
potential_refs = [
    rlf.PotentialRef(element=ground),
    rlf.PotentialRef(element=buses[2]),
    rlf.PotentialRef(element=buses[6]),
    rlf.PotentialRef(element=buses[8]),
    rlf.PotentialRef(element=buses[12]),
]

# Loads
loads = [
    rlf.PowerLoad(
        id=1,
        bus=buses[2],
        n=4,
        s=[
            41916.482229647016 + 20958.241114823508j,
            41916.482230776804 + 20958.2411153884j,
            41916.4822307768 + 20958.241115388402j,
        ],
    ),
    rlf.PowerLoad(
        id=2,
        bus=buses[3],
        n=4,
        s=[
            40459.7989783205 + 20229.89948916025j,
            40459.79897941102 + 20229.89948970551j,
            40459.79897941102 + 20229.89948970551j,
        ],
    ),
    rlf.PowerLoad(
        id=3,
        bus=buses[4],
        n=4,
        s=[
            37922.04164877094 + 18961.020824385465j,
            37922.04164985974 + 18961.020824929874j,
            37922.04164980375 + 18961.02082490188j,
        ],
    ),
    rlf.PowerLoad(
        id=4,
        bus=buses[5],
        n=4,
        s=[
            40459.798978684 + 20229.899489342002j,
            40459.79897977451 + 20229.89948988726j,
            40459.798978684004 + 20229.899489342002j,
        ],
    ),
    rlf.PowerLoad(
        id=5,
        bus=buses[6],
        n=4,
        s=[
            41916.48223002361 + 20958.24111501181j,
            41916.4822311534 + 20958.241115576697j,
            41916.48223002363 + 20958.241115011813j,
        ],
    ),
    rlf.PowerLoad(
        id=6,
        bus=buses[7],
        n=4,
        s=[
            40932.79932474136 + 20466.399662370677j,
            40932.79932583017 + 20466.39966291509j,
            40932.79932479737 + 20466.39966239868j,
        ],
    ),
    rlf.PowerLoad(
        id=7,
        bus=buses[8],
        n=4,
        s=[
            41916.482229647016 + 20958.241114823508j,
            41916.482230776804 + 20958.241115388402j,
            41916.4822307768 + 20958.241115388402j,
        ],
    ),
    rlf.PowerLoad(
        id=8,
        bus=buses[9],
        n=4,
        s=[
            40459.79897832049 + 20229.899489160252j,
            40459.79897941102 + 20229.89948970551j,
            40459.79897941101 + 20229.899489705513j,
        ],
    ),
    rlf.PowerLoad(
        id=9,
        bus=buses[10],
        n=4,
        s=[
            37922.04164877094 + 18961.020824385465j,
            37922.04164985973 + 18961.020824929878j,
            37922.04164980376 + 18961.02082490188j,
        ],
    ),
    rlf.PowerLoad(
        id=10,
        bus=buses[11],
        n=4,
        s=[
            40459.798978684 + 20229.899489342002j,
            40459.79897977452 + 20229.899489887266j,
            40459.798978684004 + 20229.899489342002j,
        ],
    ),
    rlf.PowerLoad(
        id=11,
        bus=buses[12],
        n=4,
        s=[
            41916.48223002361 + 20958.24111501181j,
            41916.4822311534 + 20958.241115576693j,
            41916.48223002362 + 20958.241115011817j,
        ],
    ),
    rlf.PowerLoad(
        id=12,
        bus=buses[13],
        n=4,
        s=[
            40932.79932474137 + 20466.399662370684j,
            40932.79932583017 + 20466.399662915086j,
            40932.799324797365 + 20466.399662398682j,
        ],
    ),
]

# Transformers
tc = {
    "160kVA_Dd0": rlf.TransformerCharacteristics(
        type_name="160kVA_Dd0",
        windings="dd0",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Dd6": rlf.TransformerCharacteristics(
        type_name="160kVA_Dd6",
        windings="dd6",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Dyn11": rlf.TransformerCharacteristics(
        type_name="160kVA_Dyn11",
        windings="dyn11",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Dyn5": rlf.TransformerCharacteristics(
        type_name="160kVA_Dyn5",
        windings="dyn5",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Dzn0": rlf.TransformerCharacteristics(
        type_name="160kVA_Dzn0",
        windings="dzn0",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Dzn6": rlf.TransformerCharacteristics(
        type_name="160kVA_Dzn6",
        windings="dzn6",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Yd11": rlf.TransformerCharacteristics(
        type_name="160kVA_Yd11",
        windings="yd11",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Yd5": rlf.TransformerCharacteristics(
        type_name="160kVA_Yd5",
        windings="yd5",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Yyn0": rlf.TransformerCharacteristics(
        type_name="160kVA_Yyn0",
        windings="yyn0",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Yyn6": rlf.TransformerCharacteristics(
        type_name="160kVA_Yyn6",
        windings="yyn6",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Yzn11": rlf.TransformerCharacteristics(
        type_name="160kVA_Yzn11",
        windings="yzn11",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
    "160kVA_Yzn5": rlf.TransformerCharacteristics(
        type_name="160kVA_Yzn5",
        windings="yzn5",
        sn=160000.0,
        uhv=20000.0,
        ulv=400.0,
        i0=0.023,
        p0=460.0,
        psc=2350.0,
        vsc=0.04,
    ),
}
p = Point(0.0, 0.5)
transformers = [
    rlf.DeltaDeltaTransformer(
        id="tr1",
        bus1=buses[1],
        bus2=buses[2],
        phases1="abc",
        phases2="abc",
        transformer_characteristics=tc["160kVA_Dd0"],
        tap=1,
        geometry=p,
    ),
    rlf.WyeWyeTransformer(
        id="tr2",
        bus1=buses[1],
        bus2=buses[3],
        phases1="abcn",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Yyn0"],
        tap=1,
        geometry=p,
    ),
    rlf.DeltaZigzagTransformer(
        id="tr3",
        bus1=buses[1],
        bus2=buses[4],
        phases1="abc",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Dzn0"],
        tap=1,
        geometry=p,
    ),
    rlf.DeltaWyeTransformer(
        id="tr4",
        bus1=buses[1],
        bus2=buses[5],
        phases1="abc",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Dyn11"],
        tap=1,
        geometry=p,
    ),
    rlf.WyeDeltaTransformer(
        id="tr5",
        bus1=buses[1],
        bus2=buses[6],
        phases1="abcn",
        phases2="abc",
        transformer_characteristics=tc["160kVA_Yd11"],
        tap=1,
        geometry=p,
    ),
    rlf.WyeZigzagTransformer(
        id="tr6",
        bus1=buses[1],
        bus2=buses[7],
        phases1="abcn",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Yzn11"],
        tap=1,
        geometry=p,
    ),
    rlf.DeltaDeltaTransformer(
        id="tr7",
        bus1=buses[1],
        bus2=buses[8],
        phases1="abc",
        phases2="abc",
        transformer_characteristics=tc["160kVA_Dd6"],
        tap=1,
        geometry=p,
    ),
    rlf.WyeWyeTransformer(
        id="tr8",
        bus1=buses[1],
        bus2=buses[9],
        phases1="abcn",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Yyn6"],
        tap=1,
        geometry=p,
    ),
    rlf.DeltaZigzagTransformer(
        id="tr9",
        bus1=buses[1],
        bus2=buses[10],
        phases1="abc",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Dzn6"],
        tap=1,
        geometry=p,
    ),
    rlf.DeltaWyeTransformer(
        id="tr10",
        bus1=buses[1],
        bus2=buses[11],
        phases1="abc",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Dyn5"],
        tap=1,
        geometry=p,
    ),
    rlf.WyeDeltaTransformer(
        id="tr11",
        bus1=buses[1],
        bus2=buses[12],
        phases1="abcn",
        phases2="abc",
        transformer_characteristics=tc["160kVA_Yd5"],
        tap=1,
        geometry=p,
    ),
    rlf.WyeZigzagTransformer(
        id="tr12",
        bus1=buses[1],
        bus2=buses[13],
        phases1="abcn",
        phases2="abcn",
        transformer_characteristics=tc["160kVA_Yzn5"],
        tap=1,
        geometry=p,
    ),
]

en = rlf.ElectricalNetwork(
    buses=buses,
    branches=transformers,
    loads=loads,
    special_elements=[ground, *potential_refs],
)

# The first file we had stored the geometry as a WKT string
Point.__geo_interface__ = Point.wkt

en.to_json(OUTPUT)
