﻿using System.ComponentModel.DataAnnotations;
using System.Text;
using System.Net.Http.Headers;
using System.Text.Json;
using System.Text.Json.Serialization;

namespace Auxilium
{
    /// <summary>
    /// Represents a request to generate a chat completion.
    /// </summary>
    public class Chat
	{
		[JsonIgnore(Condition = JsonIgnoreCondition.WhenWritingNull)]
		public string name { get; set; }
        /// <summary>
        /// ID of the model to use. See the model endpoint compatibility table for details on which models work with the Chat API.
        /// </summary>
        [Required]
        public string model { get; set; } = "gpt-3.5-turbo";

        /// <summary>
        /// The messages to generate chat completions for, in the chat format.
        /// </summary>
        [Required]
        public List<ChatMessage> messages { get; set; } = new List<ChatMessage>();

        /// <summary>
        /// What sampling temperature to use, between 0 and 2. Higher values like 0.8 will make the output more random, while lower values like 0.2 will make it more focused and deterministic.
        /// </summary>
        public double temperature { get; set; } = 1;

        /// <summary>
        /// An alternative to sampling with temperature, called nucleus sampling, where the model considers the results of the tokens with top_p probability mass. So 0.1 means only the tokens comprising the top 10% probability mass are considered.
        /// </summary>
        public double top_p { get; set; } = 0.75;

        /// <summary>
        /// How many chat completion choices to generate for each input message.
        /// </summary>
        public int n { get; set; } = 1;

        /// <summary>
        /// If set, partial message deltas will be sent, like in ChatGPT. Tokens will be sent as data-only server-sent events as they become available, with the stream terminated by a data: [DONE] message. See the OpenAI Cookbook for example code.
        /// </summary>
        public bool stream { get; set; } = false;

        /// <summary>
        /// Up to 4 sequences where the API will stop generating further tokens.
        /// </summary>
        public string[] stop { get; set; } = null;

        /// <summary>
        /// The maximum number of tokens to generate in the chat completion.
        /// </summary>
        public int max_tokens { get; set; } = int.MaxValue;

        /// <summary>
        /// Number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the text so far, increasing the model's likelihood to talk about new topics.
        /// </summary>
        public double presence_penalty { get; set; } = 0;

        /// <summary>
        /// Number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in the text so far, decreasing the model's likelihood to repeat the same line verbatim.
        /// </summary>
        public double frequency_penalty { get; set; } = 0;

        /// <summary>
        /// Modify the likelihood of specified tokens appearing in the completion.
        /// </summary>
        public Dictionary<string, int> logit_bias = null;

        /// <summary>
        /// A unique identifier representing your end-user, which can help OpenAI to monitor and detect abuse. Learn more.
        /// </summary>
        public string user = null;
	}

    /// <summary>
    /// Represents a chat message.
    /// </summary>
    public class ChatMessage
    {
        /// <summary>
        /// The role of the message sender, either "user" or "assistant" or "system".
        /// </summary>
        public string role { get; set; }

        /// <summary>
        /// The content of the message.
        /// </summary>
        public string content { get; set; } = "";
    }

    /// <summary>
    /// Represents a chat completion choice.
    /// </summary>
    public class ChatCompletionChoice
    {
        /// <summary>
        /// The index of the choice.
        /// </summary>
        public int index { get; set; }

        /// <summary>
        /// The message generated by the model.
        /// </summary>
        public ChatMessage message { get; set; }

        /// <summary>
        /// The reason why the model stopped generating tokens.
        /// </summary>
        public string finish_reason { get; set; }
    }

    /// <summary>
    /// Represents usage statistics for a chat completion.
    /// </summary>
    public class ChatCompletionUsage
    {
        /// <summary>
        /// The number of tokens in the prompt.
        /// </summary>
        public int prompt_tokens { get; set; }

        /// <summary>
        /// The number of tokens in the generated completion.
        /// </summary>
        public int completion_tokens { get; set; }

        /// <summary>
        /// The total number of tokens in the prompt and generated completion.
        /// </summary>
        public int total_tokens { get; set; }
    }

    /// <summary>
    /// Represents a response from the ChatGPT API.
    /// </summary>
    public class ChatCompletionResponse
    {
        /// <summary>
        /// The ID of the chat completion.
        /// </summary>
        public string id { get; set; }

        /// <summary>
        /// The type of object returned by the API.
        /// </summary>
        public string @object { get; set; }

        /// <summary>
        /// The timestamp when the chat completion was created.
        /// </summary>
        public long created { get; set; }

        /// <summary>
        /// The list of chat completion choices generated by the model.
        /// </summary>
        public List<ChatCompletionChoice> choices { get; set; }

        /// <summary>
        /// Usage statistics for the chat completion.
        /// </summary>
        public ChatCompletionUsage usage { get; set; }
        public ChatCompletionResponseError error { get; set; }
        
    }
    public class ChatCompletionResponseError
    {
        public string message { get; set; }
        public string type { get; set; }
        public string param { get; set; }
        public string code { get; set; }
    }
    public static class ChatGPTAPI
    {
        private static readonly HttpClient client;

        static ChatGPTAPI()
        {
            client = new HttpClient();
        }

        public static async Task<ChatCompletionResponse> GenerateChatCompletion(Chat chat, string apiKey)
        {
            // Set the API endpoint URL
            string url = "https://api.openai.com/v1/chat/completions";

            // Set the API key in the Authorization header
            client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Bearer", apiKey);

            // Remove name before sending
            var chatSend = chat.Clone();
            chatSend.name = null;

			// Serialize the chat object to JSON
			string json = JsonSerializer.Serialize(chatSend, JsonSerializerOptions.Default);

            // Create a new HTTP request with the JSON payload
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Post, url);
            request.Content = new StringContent(json, Encoding.UTF8, "application/json");

            // Send the HTTP request and wait for the response
            HttpResponseMessage response = await client.SendAsync(request);
            string responseJson = await response.Content.ReadAsStringAsync();

            // Deserialize the JSON response to a ChatCompletionResponse object
            ChatCompletionResponse chatCompletion = JsonSerializer.Deserialize<ChatCompletionResponse>(responseJson);

            return chatCompletion;
        }
    }
	public static class ChatExtension
	{
		public static Chat Clone(this Chat source)
		{
			var serialized = JsonSerializer.Serialize(source);
			return JsonSerializer.Deserialize<Chat>(serialized);
		}
	}
}
