/*
 * Decompiled with CFR 0.152.
 */
package org.jcvi.jillion.sam;

import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Predicate;
import org.jcvi.jillion.core.Range;
import org.jcvi.jillion.core.io.FileUtil;
import org.jcvi.jillion.core.io.IOUtil;
import org.jcvi.jillion.core.qual.QualitySequence;
import org.jcvi.jillion.core.qual.QualitySequenceBuilder;
import org.jcvi.jillion.core.residue.nt.NucleotideSequence;
import org.jcvi.jillion.internal.core.io.OpenAwareInputStream;
import org.jcvi.jillion.internal.core.io.TextLineParser;
import org.jcvi.jillion.internal.sam.SamUtil;
import org.jcvi.jillion.sam.AbstractSamFileParser;
import org.jcvi.jillion.sam.BgzfInputStream;
import org.jcvi.jillion.sam.SamParser;
import org.jcvi.jillion.sam.SamRecord;
import org.jcvi.jillion.sam.SamRecordBuilder;
import org.jcvi.jillion.sam.SamVisitor;
import org.jcvi.jillion.sam.VirtualFileOffset;
import org.jcvi.jillion.sam.attribute.InvalidAttributeException;
import org.jcvi.jillion.sam.attribute.ReservedAttributeValidator;
import org.jcvi.jillion.sam.attribute.SamAttribute;
import org.jcvi.jillion.sam.attribute.SamAttributeKey;
import org.jcvi.jillion.sam.attribute.SamAttributeKeyFactory;
import org.jcvi.jillion.sam.attribute.SamAttributeType;
import org.jcvi.jillion.sam.attribute.SamAttributeValidator;
import org.jcvi.jillion.sam.cigar.Cigar;
import org.jcvi.jillion.sam.cigar.CigarOperation;
import org.jcvi.jillion.sam.header.SamHeader;
import org.jcvi.jillion.sam.header.SamHeaderBuilder;
import org.jcvi.jillion.sam.header.SamReferenceSequenceBuilder;

class BamFileParser
extends AbstractSamFileParser {
    private static final VirtualFileOffset BEGINNING_OF_FILE = new VirtualFileOffset(0L);
    protected final File bamFile;
    protected final SamAttributeValidator validator;
    protected final String[] refNames;
    protected final SamHeader header;

    public BamFileParser(File bamFile) throws IOException {
        this(bamFile, ReservedAttributeValidator.INSTANCE);
    }

    public BamFileParser(File bamFile, SamAttributeValidator validator) throws IOException {
        if (bamFile == null) {
            throw new NullPointerException("bam file can not be null");
        }
        if (!"bam".equals(FileUtil.getExtension(bamFile))) {
            throw new IllegalArgumentException("must be .bam file" + bamFile.getAbsolutePath());
        }
        if (!bamFile.exists()) {
            throw new FileNotFoundException(bamFile.getAbsolutePath());
        }
        if (!bamFile.canRead()) {
            throw new IllegalArgumentException("bam file not readable " + bamFile.getAbsolutePath());
        }
        if (validator == null) {
            throw new NullPointerException("validator can not be null");
        }
        this.bamFile = bamFile;
        this.validator = validator;
        try (BgzfInputStream in = new BgzfInputStream(bamFile);){
            this.verifyMagicNumber(in);
            SamHeaderBuilder headerBuilder = this.parseHeader(new TextLineParser(IOUtil.toInputStream(this.readPascalString(in))));
            this.refNames = this.parseReferenceNamesAndAddToHeader(in, headerBuilder);
            this.header = headerBuilder.build();
        }
    }

    @Override
    public SamHeader getHeader() throws IOException {
        return this.header;
    }

    @Override
    public boolean canParse() {
        return true;
    }

    private void verifyReferenceInHeader(String referenceName) {
        Objects.requireNonNull(referenceName);
        if (this.header.getReferenceSequence(referenceName) == null) {
            throw new IllegalArgumentException("no reference with name '" + referenceName + "' contained in Bam file");
        }
    }

    @Override
    public void parse(SamParser.SamParserOptions options, SamVisitor visitor) throws IOException {
        if (options.getReferenceName().isPresent()) {
            if (options.getReferenceRange().isPresent()) {
                this._parse(options.getReferenceName().get(), options.getReferenceRange().get(), options.shouldCreateMementos(), visitor);
            } else {
                this._parse(options.getReferenceName().get(), options.shouldCreateMementos(), visitor);
            }
            return;
        }
        Predicate<SamRecord> predicate = record -> true;
        this.accept(visitor, options.shouldCreateMementos(), predicate);
    }

    protected void _parse(String referenceName, Range alignmentRange, boolean shouldCreateMementos, SamVisitor visitor) throws IOException {
        this.verifyReferenceInHeader(referenceName);
        this.accept(visitor, shouldCreateMementos, SamUtil.alignsToReference(referenceName, alignmentRange));
    }

    protected void _parse(String referenceName, boolean shouldCreateMementos, SamVisitor visitor) throws IOException {
        this.verifyReferenceInHeader(referenceName);
        this.accept(visitor, shouldCreateMementos, SamUtil.alignsToReference(referenceName));
    }

    @Override
    public void parse(String referenceName, SamVisitor visitor) throws IOException {
        this._parse(referenceName, false, visitor);
    }

    @Override
    public void parse(String referenceName, Range alignmentRange, SamVisitor visitor) throws IOException {
        this._parse(referenceName, alignmentRange, false, visitor);
    }

    @Override
    public void parse(SamVisitor visitor) throws IOException {
        this.accept(visitor, false, record -> true);
    }

    @Override
    public void parse(SamVisitor visitor, SamVisitor.SamVisitorCallback.SamVisitorMemento memento) throws IOException {
        Objects.requireNonNull(visitor);
        Objects.requireNonNull(memento);
        if (!(memento instanceof BamFileMemento)) {
            throw new IllegalArgumentException("memento must be for bam file");
        }
        BamFileMemento bamMemento = (BamFileMemento)memento;
        if (this != bamMemento.parserInstance) {
            throw new IllegalArgumentException("memento must be for this exact bam parser instance");
        }
        if (bamMemento.encodedFileOffset == 0L) {
            this.parse(visitor);
            return;
        }
        VirtualFileOffset vfs = new VirtualFileOffset(bamMemento.encodedFileOffset);
        try (BgzfInputStream in = BgzfInputStream.create(this.bamFile, vfs);){
            AtomicBoolean keepParsing = new AtomicBoolean(true);
            this.parseBamRecords(visitor, record -> true, v -> true, in, keepParsing, new MementoLessBamCallback(keepParsing));
        }
    }

    private void accept(SamVisitor visitor, boolean enableMementos, Predicate<SamRecord> filter) throws IOException {
        if (visitor == null) {
            throw new NullPointerException("visitor can not be null");
        }
        try (BgzfInputStream in = new BgzfInputStream(this.bamFile);){
            this.parseBamFromBeginning(visitor, enableMementos, filter, vfs -> true, in);
        }
    }

    protected void parseBamFromBeginning(SamVisitor visitor, boolean enableMementos, Predicate<SamRecord> filter, Predicate<VirtualFileOffset> keepParsingPredicate, BgzfInputStream in) throws IOException {
        this.verifyMagicNumber(in);
        SamHeaderBuilder headerBuilder = this.parseHeader(new TextLineParser(IOUtil.toInputStream(this.readPascalString(in))));
        this.parseReferenceNamesAndAddToHeader(in, headerBuilder);
        AtomicBoolean keepParsing = new AtomicBoolean(true);
        AbstractBamCallback callback = enableMementos ? new BamCallback(keepParsing) : new MementoLessBamCallback(keepParsing);
        visitor.visitHeader(callback, this.header);
        this.parseBamRecords(visitor, filter, keepParsingPredicate, in, keepParsing, callback);
    }

    protected void parseBamRecords(SamVisitor visitor, Predicate<SamRecord> filter, Predicate<VirtualFileOffset> keepParsingPredicate, BgzfInputStream in, AtomicBoolean keepParsing, AbstractBamCallback callback) throws IOException {
        boolean canceledByPredicate = false;
        try {
            VirtualFileOffset start = in.getCurrentVirutalFileOffset();
            while (keepParsing.get() && in.hasMoreData()) {
                SamRecord record = this.parseNextSamRecord(in, this.refNames, this.header);
                VirtualFileOffset end = in.getCurrentVirutalFileOffset();
                if (keepParsingPredicate.test(start)) {
                    if (filter.test(record)) {
                        callback.updateCurrentPosition(start);
                        visitor.visitRecord(callback, record, start, end);
                    }
                } else {
                    keepParsing.set(false);
                    canceledByPredicate = true;
                }
                start = end;
            }
        }
        catch (EOFException eOFException) {
            // empty catch block
        }
        if (canceledByPredicate || keepParsing.get()) {
            visitor.visitEnd();
        } else {
            visitor.halted();
        }
    }

    private SamRecord parseNextSamRecord(InputStream in, String[] refNames, SamHeader header) throws IOException {
        int blockSize = this.getSignedInt(in);
        SamRecordBuilder builder = new SamRecordBuilder(header, this.validator);
        int refId = this.getSignedInt(in);
        if (refId >= 0) {
            builder.setReferenceName(refNames[refId]);
        }
        int startPos = this.getSignedInt(in) + 1;
        builder.setStartPosition(startPos);
        long binMqReadLength = this.getUnsignedInt(in);
        byte mapQuality = (byte)(binMqReadLength >> 8 & 0xFFL);
        builder.setMappingQuality(mapQuality);
        int readNameLength = (int)(binMqReadLength & 0xFFL);
        long flagsNumCigarOps = this.getUnsignedInt(in);
        int bitFlags = (int)(flagsNumCigarOps >> 16 & 0xFFFFL);
        builder.setFlags(bitFlags);
        int numCigarOps = (int)(flagsNumCigarOps & 0xFFFFL);
        int seqLength = this.getSignedInt(in);
        int nextRefId = this.getSignedInt(in);
        if (nextRefId >= 0) {
            builder.setNextReferenceName(refNames[nextRefId]);
        }
        builder.setNextPosition(this.getSignedInt(in) + 1);
        builder.setObservedTemplateLength(this.getSignedInt(in));
        String readId = this.readNullTerminatedString(in, readNameLength);
        builder.setQueryName(readId);
        if (numCigarOps > 0) {
            Cigar cigar = this.parseCigar(in, numCigarOps);
            builder.setCigar(cigar);
        }
        if (seqLength > 0) {
            NucleotideSequence seq = SamUtil.readBamEncodedSequence(in, seqLength);
            builder.setSequence(seq);
            builder.setQualities(this.readQualities(in, seqLength));
        }
        int bytesReadSoFar = 32 + 4 * numCigarOps + readNameLength + (seqLength + 1) / 2 + seqLength;
        this.parseAttributesIfAnyAndAddToBuilder(in, builder, blockSize, bytesReadSoFar);
        return builder.build();
    }

    private Cigar parseCigar(InputStream in, int numCigarOps) throws IOException {
        Cigar.Builder cigarBuilder = new Cigar.Builder(numCigarOps);
        for (int i = 0; i < numCigarOps; ++i) {
            long bits = this.getUnsignedInt(in);
            int opCode = (int)(bits & 0xFL);
            int length = (int)(bits >> 4);
            cigarBuilder.addElement(CigarOperation.parseBinary(opCode), length);
        }
        return cigarBuilder.build();
    }

    private void parseAttributesIfAnyAndAddToBuilder(InputStream in, SamRecordBuilder builder, int blockSize, int bytesReadSoFar) throws IOException {
        int attributeByteLength = blockSize - bytesReadSoFar;
        if (attributeByteLength > 0) {
            byte[] attributeBytes = new byte[attributeByteLength];
            IOUtil.blockingRead(in, attributeBytes);
            OpenAwareInputStream attributeStream = new OpenAwareInputStream(new ByteArrayInputStream(attributeBytes));
            while (attributeStream.isOpen()) {
                SamAttribute attribute = this.parseAttribute(attributeStream);
                try {
                    builder.addAttribute(attribute);
                }
                catch (InvalidAttributeException e) {
                    throw new IOException("invalid attribute " + attribute, e);
                }
            }
        }
    }

    private String[] parseReferenceNamesAndAddToHeader(InputStream in, SamHeaderBuilder headerBuilder) throws IOException {
        int referenceCount = this.getSignedInt(in);
        String[] refNames = new String[referenceCount];
        for (int i = 0; i < referenceCount; ++i) {
            String name;
            refNames[i] = name = this.readPascalString(in);
            int length = this.getSignedInt(in);
            if (headerBuilder.hasReferenceSequence(name)) continue;
            headerBuilder.addReferenceSequence(new SamReferenceSequenceBuilder(name, length).build());
        }
        return refNames;
    }

    private SamAttribute parseAttribute(OpenAwareInputStream in) throws IOException {
        SamAttributeKey key = SamAttributeKeyFactory.getKey((char)in.read(), (char)in.read());
        char type = (char)in.read();
        switch (type) {
            case 'i': {
                return new SamAttribute(key, SamAttributeType.SIGNED_INT, IOUtil.readSignedInt(in));
            }
            case 'I': {
                return new SamAttribute(key, SamAttributeType.UNSIGNED_INT, IOUtil.readUnsignedInt(in));
            }
            case 'Z': {
                return new SamAttribute(key, SamAttributeType.STRING, this.readNullTerminatedStringAttribute(in));
            }
            case 'B': {
                return this.handleArray(key, in);
            }
            case 'A': {
                return new SamAttribute(key, SamAttributeType.PRINTABLE_CHARACTER, Character.valueOf((char)in.read()));
            }
            case 'c': {
                return new SamAttribute(key, SamAttributeType.SIGNED_INT, in.read());
            }
            case 'C': {
                return new SamAttribute(key, SamAttributeType.SIGNED_INT, IOUtil.readUnsignedByte(in));
            }
            case 's': {
                return new SamAttribute(key, SamAttributeType.SIGNED_INT, IOUtil.readSignedShort(in));
            }
            case 'S': {
                return new SamAttribute(key, SamAttributeType.SIGNED_INT, IOUtil.readUnsignedShort(in));
            }
            case 'f': {
                return new SamAttribute(key, SamAttributeType.FLOAT, Float.valueOf(IOUtil.readFloat(in)));
            }
            case 'H': {
                return new SamAttribute(key, SamAttributeType.BYTE_ARRAY_IN_HEX, this.toByteArray(this.readNullTerminatedStringAttribute(in)));
            }
        }
        throw new IOException("unknown type : " + type);
    }

    private SamAttribute handleArray(SamAttributeKey key, OpenAwareInputStream in) throws IOException {
        char arrayType = (char)in.read();
        int length = IOUtil.readSignedInt(in);
        switch (arrayType) {
            case 'i': {
                return new SamAttribute(key, SamAttributeType.SIGNED_INT_ARRAY, IOUtil.readIntArray(in, length));
            }
            case 'I': {
                return new SamAttribute(key, SamAttributeType.UNSIGNED_INT_ARRAY, IOUtil.readIntArray(in, length));
            }
            case 'c': {
                return new SamAttribute(key, SamAttributeType.SIGNED_BYTE_ARRAY, IOUtil.readByteArray(in, length));
            }
            case 'C': {
                return new SamAttribute(key, SamAttributeType.UNSIGNED_BYTE_ARRAY, IOUtil.readByteArray(in, length));
            }
            case 's': {
                return new SamAttribute(key, SamAttributeType.SIGNED_SHORT_ARRAY, IOUtil.readShortArray(in, length));
            }
            case 'S': {
                return new SamAttribute(key, SamAttributeType.UNSIGNED_SHORT_ARRAY, IOUtil.readShortArray(in, length));
            }
            case 'f': {
                return new SamAttribute(key, SamAttributeType.FLOAT_ARRAY, IOUtil.readFloatArray(in, length));
            }
        }
        throw new IOException("unknown array type : " + arrayType);
    }

    private byte[] toByteArray(String hex) {
        byte[] array = new byte[hex.length() / 2];
        char[] chars = hex.toCharArray();
        for (int i = 0; i < chars.length; i += 2) {
            array[i] = Byte.parseByte(new String(chars, i, 2), 16);
        }
        return array;
    }

    private String readNullTerminatedStringAttribute(OpenAwareInputStream in) throws IOException {
        boolean done = false;
        StringBuilder builder = new StringBuilder();
        do {
            int value;
            if ((value = in.read()) == -1 || value == 0) {
                done = true;
                continue;
            }
            builder.append((char)value);
        } while (!done && in.isOpen());
        return builder.toString();
    }

    private QualitySequence readQualities(InputStream in, int seqLength) throws IOException {
        byte[] bytes = new byte[seqLength];
        IOUtil.blockingRead(in, bytes);
        if (bytes[0] == -1) {
            return null;
        }
        return new QualitySequenceBuilder(bytes).turnOffDataCompression(true).build();
    }

    private long getUnsignedInt(InputStream in) throws IOException {
        return IOUtil.readUnsignedInt(in, ByteOrder.LITTLE_ENDIAN);
    }

    private int getSignedInt(InputStream in) throws IOException {
        return (int)IOUtil.readUnsignedInt(in, ByteOrder.LITTLE_ENDIAN);
    }

    private void verifyMagicNumber(InputStream in) throws IOException {
        byte[] header = new byte[4];
        IOUtil.blockingRead(in, header);
        if (!SamUtil.matchesBamMagicNumber(header)) {
            throw new IOException("invalid bam magic number header : " + Arrays.toString(header));
        }
    }

    private String readPascalString(InputStream in) throws IOException {
        int length = this.getSignedInt(in);
        return this.readNullTerminatedString(in, length);
    }

    private String readNullTerminatedString(InputStream in, int lengthIncludingNull) throws IOException {
        if (lengthIncludingNull == 0) {
            return "";
        }
        byte[] data = new byte[lengthIncludingNull];
        IOUtil.blockingRead(in, data);
        return new String(data, 0, lengthIncludingNull - 1, IOUtil.UTF_8);
    }

    private static final class BamFileMemento
    implements SamVisitor.SamVisitorCallback.SamVisitorMemento {
        private final BamFileParser parserInstance;
        private final long encodedFileOffset;

        public BamFileMemento(BamFileParser parserInstance, long position) {
            this.parserInstance = parserInstance;
            this.encodedFileOffset = position;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + (this.parserInstance == null ? 0 : this.parserInstance.hashCode());
            result = 31 * result + (int)(this.encodedFileOffset ^ this.encodedFileOffset >>> 32);
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (!(obj instanceof BamFileMemento)) {
                return false;
            }
            BamFileMemento other = (BamFileMemento)obj;
            if (this.parserInstance != other.parserInstance) {
                return false;
            }
            return this.encodedFileOffset == other.encodedFileOffset;
        }
    }

    protected final class BamCallback
    extends AbstractBamCallback {
        private VirtualFileOffset offset;

        public BamCallback(AtomicBoolean keepParsing) {
            this(keepParsing, BEGINNING_OF_FILE);
        }

        public BamCallback(AtomicBoolean keepParsing, VirtualFileOffset currentPosition) {
            super(keepParsing);
            this.offset = currentPosition;
        }

        @Override
        public void updateCurrentPosition(VirtualFileOffset currentPosition) {
            this.offset = currentPosition;
        }

        @Override
        public boolean canCreateMemento() {
            return true;
        }

        @Override
        public SamVisitor.SamVisitorCallback.SamVisitorMemento createMemento() {
            return new BamFileMemento(BamFileParser.this, this.offset.getEncodedValue());
        }
    }

    protected abstract class AbstractBamCallback
    extends AbstractSamFileParser.AbstractCallback {
        public AbstractBamCallback(AtomicBoolean keepParsing) {
            super(keepParsing);
        }

        public abstract void updateCurrentPosition(VirtualFileOffset var1);
    }

    protected final class MementoLessBamCallback
    extends AbstractBamCallback {
        public MementoLessBamCallback(AtomicBoolean keepParsing) {
            super(keepParsing);
        }

        @Override
        public boolean canCreateMemento() {
            return false;
        }

        @Override
        public SamVisitor.SamVisitorCallback.SamVisitorMemento createMemento() {
            throw new UnsupportedOperationException();
        }

        @Override
        public void updateCurrentPosition(VirtualFileOffset currentPosition) {
        }
    }
}

