import type {
  ChatGPTAPI,
  ChatGPTUnofficialProxyAPI,
  ChatMessage as ChatResponseV4,
} from 'chatgpt';
import type {
  ChatGPTAPIBrowser,
  ChatResponse as ChatResponseV3,
} from 'chatgpt-v3';
import {
  APIBrowserOptions,
  APIOfficialOptions,
  APIOptions,
  APIUnofficialOptions,
} from './types';
import {logWithTime} from './utils';
import {DB} from './db';

class ChatGPT {
  debug: number;
  readonly apiType: string;
  protected _opts: APIOptions;
  protected _api:
    | ChatGPTAPI
    | ChatGPTAPIBrowser
    | ChatGPTUnofficialProxyAPI
    | undefined;
  protected _apiBrowser: ChatGPTAPIBrowser | undefined;
  protected _apiOfficial: ChatGPTAPI | undefined;
  protected _apiUnofficialProxy: ChatGPTUnofficialProxyAPI | undefined;
  protected _timeoutMs: number | undefined;
  protected _db: DB;

  constructor(apiOpts: APIOptions, db: DB, debug = 1) {
    this.debug = debug;
    this.apiType = apiOpts.type;
    this._opts = apiOpts;
    this._timeoutMs = undefined;
    this._db = db;
  }

  init = async () => {
    if (this._opts.type == 'browser') {
      const {ChatGPTAPIBrowser} = await import('chatgpt-v3');
      this._apiBrowser = new ChatGPTAPIBrowser(
        this._opts.browser as APIBrowserOptions
      );
      await this._apiBrowser.initSession();
      this._api = this._apiBrowser;
      this._timeoutMs = this._opts.browser?.timeoutMs;
    } else if (this._opts.type == 'official') {
      const {ChatGPTAPI} = await import('chatgpt');
      this._apiOfficial = new ChatGPTAPI({
        ...(this._opts.official as APIOfficialOptions),
        getMessageById: async (id) => {
          const message = await this._db.messageStore.get(id);
          return message as ChatResponseV4;
        },
        upsertMessage: async (message) => {
          await this._db.messageStore.set(message.id, message);
        },
      });
      this._api = this._apiOfficial;
      this._timeoutMs = this._opts.official?.timeoutMs;
    } else if (this._opts.type == 'unofficial') {
      const {ChatGPTUnofficialProxyAPI} = await import('chatgpt');
      this._apiUnofficialProxy = new ChatGPTUnofficialProxyAPI(
        this._opts.unofficial as APIUnofficialOptions
      );
      this._api = this._apiUnofficialProxy;
      this._timeoutMs = this._opts.unofficial?.timeoutMs;
    } else {
      throw new RangeError('Invalid API type');
    }
    logWithTime('🔮 ChatGPT API has started...');
  };

  sendMessage = async (
    text: string,
    chatId: number,
    messageId: string,
    replyId?: string,
    onProgress?: (res: ChatResponseV3 | ChatResponseV4) => void
  ) => {
    if (!this._api) return;

    const contextDB = await this._db.getContext(chatId);
    const parentIdFromReply = await this._db.getReplyId(replyId);

    const context = {
      conversationId: contextDB?.conversationId,
      parentMessageId: parentIdFromReply ?? contextDB?.parentMessageId,
    };

    let res: ChatResponseV3 | ChatResponseV4;
    if (this.apiType == 'official') {
      if (!this._apiOfficial) return;
      res = await this._apiOfficial.sendMessage(text, {
        ...context,
        onProgress,
        messageId,
        timeoutMs: this._timeoutMs,
      });
    } else {
      res = await this._api.sendMessage(text, {
        ...context,
        onProgress,
        messageId,
        timeoutMs: this._timeoutMs,
      });
    }

    const parentMessageId =
      this.apiType == 'browser'
        ? (res as ChatResponseV3).messageId
        : (res as ChatResponseV4).id;

    await this._db.updateContext(chatId, {
      conversationId: res.conversationId,
      parentMessageId,
    });

    return res;
  };

  resetThread = async (chatId: number) => {
    if (this._apiBrowser) {
      await this._apiBrowser.resetThread();
    }
    await this._db.clearContext(chatId);
  };

  refreshSession = async () => {
    if (this._apiBrowser) {
      await this._apiBrowser.refreshSession();
    }
  };
}

export {ChatGPT};
