import Foundation

/**
    A closure that, when evaluated, returns a dictionary of key-value
    pairs that can be accessed from within a group of shared examples.
*/
public typealias SharedExampleContext = () -> [String: Any]

/**
    A closure that is used to define a group of shared examples. This
    closure may contain any number of example and example groups.
*/
public typealias SharedExampleClosure = (@escaping SharedExampleContext) -> Void

#if canImport(Darwin)
// swiftlint:disable type_name
@objcMembers
internal class _WorldBase: NSObject {}
#else
internal class _WorldBase: NSObject {}
// swiftlint:enable type_name
#endif

/**
    A collection of state Quick builds up in order to work its magic.
    World is primarily responsible for maintaining a mapping of QuickSpec
    classes to root example groups for those classes.

    It also maintains a mapping of shared example names to shared
    example closures.

    You may configure how Quick behaves by calling the -[World configure:]
    method from within an overridden +[QuickConfiguration configure:] method.
*/
final internal class World: _WorldBase {
    /**
        The example group that is currently being run.
        The DSL requires that this group is correctly set in order to build a
        correct hierarchy of example groups and their examples.
    */
    internal var currentExampleGroup: ExampleGroup!

    /**
        The example metadata of the test that is currently being run.
        This is useful for using the Quick test metadata (like its name) at
        runtime.
    */
    internal var currentExampleMetadata: SyncExampleMetadata?

    internal var numberOfSyncExamplesRun = 0
    internal var numberOfExamplesRun: Int {
        numberOfSyncExamplesRun + AsyncWorld.sharedWorld.numberOfAsyncExamplesRun
    }

    /**
        A flag that indicates whether additional test suites are being run
        within this test suite. This is only true within the context of Quick
        functional tests.
    */
#if canImport(Darwin)
    // Convention of generating Objective-C selector has been changed on Swift 3
    @objc(isRunningAdditionalSuites)
    internal var isRunningAdditionalSuites = false
#else
    internal var isRunningAdditionalSuites = false
#endif

    private var specs: [String: ExampleGroup] = [:]
    private var sharedExamples: [String: SharedExampleClosure] = [:]
    private let configuration = QCKConfiguration()

    internal private(set) var isConfigurationFinalized = false

    internal var exampleHooks: ExampleHooks { return configuration.exampleHooks }
    internal var asyncExampleHooks: AsyncExampleHooks { return configuration.asyncExampleHooks }
    internal var suiteHooks: SuiteHooks { return configuration.suiteHooks }

    // MARK: Singleton Constructor

    private override init() {}

    static private(set) var sharedWorld = World()

    static func anotherWorld<T>(block: (World) -> T) -> T {
        let previous = sharedWorld
        defer { sharedWorld = previous }

        let newWorld = World()
        sharedWorld = newWorld
        return block(newWorld)
    }

    // MARK: Public Interface

    /**
        Exposes the World's QCKConfiguration object within the scope of the closure
        so that it may be configured. This method must not be called outside of
        an overridden +[QuickConfiguration configure:] method.

        - parameter closure:  A closure that takes a Configuration object that can
                         be mutated to change Quick's behavior.
    */
    internal func configure(_ closure: QuickConfigurer) {
        assert(
            !isConfigurationFinalized,
            // swiftlint:disable:next line_length
            "Quick cannot be configured outside of a +[QuickConfiguration configure:] method. You should not call -[World configure:] directly. Instead, subclass QuickConfiguration and override the +[QuickConfiguration configure:] method."
        )
        closure(configuration)
    }

    /**
        Finalizes the World's configuration.
        Any subsequent calls to World.configure() will raise.
    */
    internal func finalizeConfiguration() {
        isConfigurationFinalized = true
    }

    /**
     Returns `true` if the root example group for the given spec class has been already initialized.

     - parameter specClass: The QuickSpec class for which is checked for the existing root example group.
     - returns: Whether the root example group for the given spec class has been already initialized or not.
     */
    internal func isRootExampleGroupInitialized(forSpecClass specClass: QuickSpec.Type) -> Bool {
        let name = String(describing: specClass)
        return specs.keys.contains(name)
    }

    /**
        Returns an internally constructed root example group for the given
        QuickSpec class.

        A root example group with the description "root example group" is lazily
        initialized for each QuickSpec class. This root example group wraps the
        top level of a -[QuickSpec spec] method--it's thanks to this group that
        users can define beforeEach and it closures at the top level, like so:

            override class func spec() {
                // These belong to the root example group
                beforeEach {}
                it("is at the top level") {}
            }

        - parameter specClass: The QuickSpec class for which to retrieve the root example group.
        - returns: The root example group for the class.
    */
    internal func rootExampleGroup(forSpecClass specClass: QuickSpec.Type) -> ExampleGroup {
        let name = String(describing: specClass)

        if let group = specs[name] {
            return group
        } else {
            let group = ExampleGroup(
                description: "root example group",
                flags: [:],
                isInternalRootExampleGroup: true
            )
            specs[name] = group
            return group
        }
    }

    /**
        Returns all examples that should be run for a given spec class.
        There are two filtering passes that occur when determining which examples should be run.
        That is, these examples are the ones that are included by inclusion filters, and are
        not excluded by exclusion filters.

        - parameter specClass: The QuickSpec subclass for which examples are to be returned.
        - returns: A list of examples to be run as test invocations, along with whether to run the full test, or just mark it as skipped.
    */
    internal func examples(forSpecClass specClass: QuickSpec.Type) -> [ExampleWrapper] {
        // 1. Grab all included examples.
        let included = includedExamples()
        // 2. Grab the intersection of (a) examples for this spec, and (b) included examples.
        let spec = rootExampleGroup(forSpecClass: specClass).examples.map { example in
            return ExampleWrapper(
                example: example,
                runFullTest: included.first(where: { $0.example == example})?.runFullTest ?? false
            )
        }
        // 3. Remove all excluded examples.
        return spec.map { test -> ExampleWrapper in
            ExampleWrapper(example: test.example, runFullTest: test.runFullTest && !self.configuration.exclusionFilters.contains { $0(test.example) })
        }
    }

    // MARK: Internal

    internal func registerSharedExample(_ name: String, closure: @escaping SharedExampleClosure) {
        raiseIfSharedExampleAlreadyRegistered(name)
        sharedExamples[name] = closure
    }

    internal func sharedExample(_ name: String) -> SharedExampleClosure {
        raiseIfSharedExampleNotRegistered(name)
        return sharedExamples[name]!
    }

    internal var includedExampleCount: Int {
        return includedExamples().count
    }

    internal lazy var cachedIncludedExampleCount: Int = self.includedExampleCount

    internal var beforesCurrentlyExecuting: Bool {
        let suiteBeforesExecuting = suiteHooks.phase == .beforesExecuting
        let exampleBeforesExecuting = exampleHooks.phase == .beforesExecuting
        var groupBeforesExecuting = false
        if let runningExampleGroup = currentExampleMetadata?.group {
            groupBeforesExecuting = runningExampleGroup.phase == .beforesExecuting
        }

        return suiteBeforesExecuting || exampleBeforesExecuting || groupBeforesExecuting
    }

    internal var aftersCurrentlyExecuting: Bool {
        let suiteAftersExecuting = suiteHooks.phase == .aftersExecuting
        let exampleAftersExecuting = exampleHooks.phase == .aftersExecuting
        var groupAftersExecuting = false
        if let runningExampleGroup = currentExampleMetadata?.group {
            groupAftersExecuting = runningExampleGroup.phase == .aftersExecuting
        }

        return suiteAftersExecuting || exampleAftersExecuting || groupAftersExecuting
    }

    internal func performWithCurrentExampleGroup(_ group: ExampleGroup, closure: () -> Void) {
        let previousExampleGroup = currentExampleGroup
        currentExampleGroup = group

        closure()

        currentExampleGroup = previousExampleGroup
    }

    private func allExamples() -> [Example] {
        var all: [Example] = []
        for (_, group) in specs {
            group.walkDownExamples { all.append($0) }
        }
        return all
    }

    internal func hasFocusedExamples() -> Bool {
        return allExamples().contains { example in
            return self.configuration.inclusionFilters.contains { $0(example) }
        }
    }

    private func includedExamples() -> [ExampleWrapper] {
        let all = allExamples()
        let hasFocusedExamples = self.hasFocusedExamples() || AsyncWorld.sharedWorld.hasFocusedExamples()

        if !hasFocusedExamples && configuration.runAllWhenEverythingFiltered {
            return all.map { example in
                return ExampleWrapper(
                    example: example,
                    runFullTest: !self.configuration.exclusionFilters.contains { $0(example) }
                )
            }
        } else {
            return all.map { example in
                return ExampleWrapper(
                    example: example,
                    runFullTest: self.configuration.inclusionFilters.contains {
                        $0(example)
                    }
                )
            }
        }
    }

    private func raiseIfSharedExampleAlreadyRegistered(_ name: String) {
        if sharedExamples[name] != nil {
            raiseError("A shared example named '\(name)' has already been registered.")
        }
    }

    private func raiseIfSharedExampleNotRegistered(_ name: String) {
        if sharedExamples[name] == nil {
            raiseError("No shared example named '\(name)' has been registered. Registered shared examples: '\(Array(sharedExamples.keys))'")
        }
    }
}

#if canImport(Darwin)
// swiftlint:disable type_name
@objcMembers
internal class _ExampleWrapperBase: NSObject {}
#else
internal class _ExampleWrapperBase: NSObject {}
// swiftlint:enable type_name
#endif

final internal class ExampleWrapper: _ExampleWrapperBase {
    private(set) var example: Example
    private(set) var runFullTest: Bool

    init(example: Example, runFullTest: Bool) {
        self.example = example
        self.runFullTest = runFullTest
        super.init()
    }
}
