<?php
/**
 * K 方法类集合
 * 这里包含常用的方法函数，部分被helper转为全局函数，剩余部分可以通过K::方法名调用
 * [目录]
 * g 全局变量调用
 * config 配置调用
 */
class k
{
    /**
     * 实例化当前K类，使用此方法可以保证只实例化一次，不被多次实例
     */
    public static function getSelf()
    {
        $obj = self::g('k.object');
        if ($obj) {
            return $obj;
        }
        $obj = new self;
        return $obj;
    }

    /**
     * 全局变量调用
     * @param string $key 共用变量name
     * @param mixed $value 为null是获取该key值，或不为null则为设置该key
     * @param string $key_prefix key值前缀
     */
    public static function g(string $key = null, $value = null, string $key_prefix = '')
    {
        global $_G;
        $g = $_G;
        if ($key_prefix) {
            $key_prefix = trim($key_prefix, '.');
            if (!is_array($key)) {
                $key = $key ? $key_prefix . '.' . $key : $key_prefix;
            }
        }
        if (!$key) {
            return $g;
        }
        if ($key && null === $value) {
            // 设置数据
            if (is_array($key)) {
                foreach ($key as $k => $v) {
                    self::g($k, $v, $key_prefix);
                }
                return true;
            }
            // 获取数据
            if (strpos($key, '.')) {
                $keys = explode('.', $key);
                foreach ($keys as $k => $v) {
                    if (!isset($g[$v])) {
                        return null;
                    }
                    $g = $g[$v];
                }
                return $g;
            }
            return isset($g[$key]) ? $g[$key] : null;
        }
        //设置数据
        if (strpos($key, '.')) {
            $keys = explode('.', $key);
            $_g   = $g;
            $_k   = '';
            foreach ($keys as $k => $v) {
                $_k .= '[\'' . $v . '\']';
                if (!isset($_g[$v])) {
                    $_g[$v] = [];
                    eval('$g' . $_k . '=[];');
                }
                $_g = $_g[$v];
            }
            eval('$g' . $_k . '=$value;');
        } else {
            $g[$key] = $value;
        }
        $_G = $g;
        return true;
    }

    /**
     * 读写配置
     * @param string $key 配置名称
     * @param mixed $value 为null是获取该配置值，或不为null则为设置该配置
     */
    public static function config(string $key = null, $value = null)
    {
        return self::g($key, $value, 'config');
    }

    /**
     * 读写系统配置
     * @param string $key 配置名称
     * @param mixed $value 为null是获取该配置值，或不为null则为设置该配置
     */
    public static function sysConfig(string $key = null, $value = null)
    {
        return self::g($key, $value, 'sys.config');
    }

    /**
     * 追加系统配置
     * @param string $key 配置名称
     * @param mixed $data 追加的数据，字符串或数组
     * @return boolean true/false
     */
    public static function configConcat(string $key, $data = '')
    {
        $config = self::config($key);
        if (is_string($data)) {
            if (!$config) {
                $config = '';
            }
            if ($config && !is_string($config)) {
                return false;
            }
            $data = $config . $data;
        } elseif (is_array($data)) {
            if (!$config) {
                $config = [];
            }
            if ($config && !is_array($config)) {
                return false;
            }
            $data = array_merge($config, $data);
        } else {
            return false;
        }
        return self::config($key, $data);
    }

    /**
     * 快捷设置当前应用的临时配置,该配置仅在当前脚本运行的目录生效，用法跟config一样
     */
    public static function myConfig($key = null, $value = null)
    {
        return self::g($key, $value, '_myconfig._' . md5(self::appinfo('dir_path')));
    }

    /**
     * 快捷获取和设置当前应用信息
     * @param string $key 配置名称
     * @param mixed $value 若不为null则为设置该配置
     */
    public static function appInfo(string $key = null, $value = null)
    {
        return self::g($key, $value, 'appinfo');
    }

    /**
     * 加载app自定义class文件，支持完整命名或仅类名，当仅类名时是在当前应用common内搜索
     * @param string $name class类的名称或完整命名
     */
    public static function u($name)
    {
        if (strpos($name, '/') === false) {
            $name = "app/" . self::appinfo('app_name') . "/common/{$name}";
        }
        $classname = str_replace('/', '\\', $name);
        $classpath = PK_WWWROOT . $name . '.class.php';
        // 若不存在先引入
        if (!class_exists($classname, false)) {
            if (is_file($classpath)) {
                require $classpath;
            }
        }
        // 若不存在报bug
        if (!class_exists($classname, false)) {
            $err = '不存在的类引用';
            if (self::config('debug')) {
                $err .= "[className=\"{$classname}\",classPath=\"{$classpath}\"]";
            }
            self::error($err);
        }
        return (new $classname());
    }

    /**
     * 调用输入（get、post、request）
     * 举例input('a'); 获取_REQUEST内的a参数,默认被过滤处理
     * input('get.a'); 获取_GET内的a参数
     * input('get.a/s'); 获取a参数并转为字符串,目前仅支持字符串/s及数字/d
     * input('get.a/s,get.b/d') 获取get方法a参数及b参数，返回数组
     * input()，input('get.') input('post.') 获取所有被输入的参数
     * @param mixed $keys 参数集合或字符串，若为字符串多个用,分开
     * @param boolean $filter 是否对获取的数据安全过滤
     */
    public static function input($keys = null, $filter = true)
    {
        //input相关数据的统一整理处理
        $_input = function ($data, $keys, $filter = true) {
            $_data_clear_up = function ($keys, $data) {
                $keys = explode(',', $keys);
                $r    = [];
                foreach ($keys as $v) {
                    if (strpos($v, '/')) {
                        list($key, $type) = explode('/', $v);
                        $get              = isset($data[$key]) ? $data[$key] : null;
                        if ('d' == $type) {
                            $r[$key] = self::cnum($get, 0);
                        } elseif ('s' == $type) {
                            $r[$key] = (string) $get;
                        } else {
                            $r[$key] = $get;
                        }
                    } else {
                        $r[$v] = isset($data[$v]) ? $data[$v] : null;
                    }
                }
                return $r;
            };
            $filter_data = function ($data, $filter) {
                if (true === $filter) {
                    $filter = self::config('default_filter');
                }
                if ($filter) {
                    $filters = explode(',', $filter);
                    foreach ($filters as $func) {
                        if (!is_string($data)) {
                            foreach ($data as $k => $v) {
                                if (is_string($v) || is_numeric($v)) {
                                    $data[$k] = call_user_func_array($func, [$v]);
                                }
                            }
                        } else {
                            $data = call_user_func_array($func, [$data]);
                        }
                    }
                }
                return $data;
            };
            $r = null;
            if (!$keys) {
                $r = $data;
            } else {
                $r = $_data_clear_up($keys, $data);
            }
            if ($filter) {
                $r = $filter_data($r, $filter);
            }
            if (count($r) == 1 && $keys) {
                $r = reset($r);
            }
            return $r;
        };
        if (!$keys) {
            return $_input($_REQUEST, '', $filter);
        }
        if (is_string($keys)) {
            if ('get.' == $keys) {
                return $_input($_GET, '', $filter);
            } elseif ('post.' == $keys) {
                return $_input($_POST, '', $filter);
            }
            $keys = explode(',', $keys);
        }
        $data = [];
        foreach ($keys as $v) {
            $v  = trim($v);
            $do = 'request';
            if (strpos($v, 'get.') === 0) {
                $v  = substr($v, 4);
                $do = 'get';
            } elseif (strpos($v, 'post.') === 0) {
                $v  = substr($v, 5);
                $do = 'post';
            }
            $key = $v;
            if (strpos($v, '/')) {
                $key = current(explode('/', $v));
            }
            switch ($do) {
                case 'get':
                    $data[$key] = $_input($_GET, $v, $filter);
                    break;
                case 'post':
                    $data[$key] = $_input($_POST, $v, $filter);
                    break;
                default:
                    $data[$key] = $_input($_REQUEST, $v, $filter);
                    break;
            }
        }
        if (count($data) == 1) {
            return current($data);
        }
        return $data;
    }

    //加载php脚本文件
    public static function requires($filepath)
    {
        if (is_string($filepath)) {
            if (strpos($filepath, ',')) {
                $filepath = explode(',', $filepath);
            } else {
                $filepath = array($filepath);
            }
        }
        if (!is_array($filepath)) {
            self::error('k::require data illegal');
        }
        foreach ($filepath as $path) {
            self::requireOne($path);
        }
        return true;
    }

    //创建相对独立的加载环境
    public static function requireOne($path)
    {
        require $path;
    }

    // 链接数据库,$PATH为数据库的配置信息或配置路径
    /**
     * 初始化pdo对象
     * @param mixed $info 数据库信息或配置文件路径
     * @param boolean $return_pdo 是否返回pdo对象，否则返回pdo标识
     */
    public static function pdoInit($info, $return_pdo = false)
    {
        $path = $info;
        if (is_array($path)) {
            $name = md5(json_encode($path));
            $r    = $path;
        } else {
            $name = str_replace(['\\', '//'], '/', substr(dirname($path), strlen(PK_APP_PATH)));
            $name = md5(trim($name, '/'));
            $r    = include $path;
        }

        //初始化数据库信息
        foreach ($r as $k => $v) {
            k::g("db.ext.{$name}.{$k}", $v);
            $$k = $v;
        }

        if (!$type) {
            return false;
        }
        require PK_LIB_PATH . 'db' . PK_DS . $type . '.php';

        //设置全局pdo对象
        k::g("db.ext.{$name}.pdo", $pdo);
        if ($return_pdo) {
            return $pdo;
        }
        //返回该数据库的name标识
        return $name;
    }

    /**
     * 快速调用数据库
     * @param string $table 表名称
     * @param mixed $pdo_name 数据库信息/pdo标识
     * @param mixed $prefix 表前缀，null为默认前缀
     */
    public static function db(string $table = '', $pdo_name = '', $prefix = null)
    {
        if (is_array($pdo_name)) {
            $pdo_name = self::pdoInit($pdo_name);
        }
        return (new db($table, $pdo_name, $prefix));
    }

    /**
     * 返回pdo对象
     * @param string $name pdo标识
     */
    public static function pdo(string $name = '')
    {
        if (!$name) {
            return self::g('db.pdo');
        }
        if (strpos($name, '/') !== false || strlen($name) != 32) {
            $name = md5(trim($name, '/'));
        }
        return k::g("db.ext.{$name}.pdo");
    }

    /**
     * 返回最后一次pdo请求错误信息
     * @param string $name pdo标识
     */
    public static function pdoErrorInfo(string $name = '')
    {
        return self::pdo($name)->errorInfo();
    }

    /**
     * mysql数据库转义
     * @param string $str 待转义字符串
     * @param boolean $quto 是否添加''字符
     * @param string $bwf 字符串两边添加的字符
     * @param boolean $must 是否强制添加''，若为false则数字不添加
     */
    public static function quote(string $str, $quto = true, string $bwf = '', $must = true)
    {
        $str = self::g('db.pdo')->quote($str);
        $str = substr($str, 1, strlen($str) - 2);
        if (($quto && !is_numeric($str)) || $must) {
            $str = "'{$bwf}{$str}{$bwf}'";
        }
        return $str;
    }

    // 无论怎样该函数都会输出字符串空
    public static function return_string_null()
    {
        return '';
    }

    //获取变量值,模板中调用
    public static function template_vars($var_key)
    {
        $a = self::g('template_parse.' . $var_key);
        if (!is_array($a)) {
            return [];
        }
        return $a;
    }

    /**
     * 返回变量值，模板变量输出调用，默认html转义
     * @param string $var 变量名
     * @param boolean $htmlspecialchars 是否转义，默认转义
     */
    public static function t($var, $htmlspecialchars = true)
    {
        $r = '';
        if (is_string($var) || is_numeric($var)) {
            $r = (string) $var;
            if ($htmlspecialchars) {
                $r = htmlspecialchars($r, ENT_QUOTES);
            }
        } elseif (is_array($var) || is_object($var)) {
            $r = json_encode($var, 320);
            if ($htmlspecialchars) {
                $r = htmlspecialchars($r, ENT_QUOTES);
            }
        } else {
            $r = '';
        }
        return $r;
    }

    /**
     * 解析模板方法
     * @param string $path 模板路径或模板代码，若为代码则$type='code'
     * @param array $vars 模板初始化的变量
     * @param string $type 加载类型：file/code
     * @param boolean $parse 是否解析模板
     */
    public static function template(string $path, array $vars = [], string $type = 'file', $parse = true)
    {
        if ('code' == $type) {
            $html = $path;
        } else {
            $appDirPath = self::appinfo('dir_path');
            $htmPath    = $path;
            $phpPath    = '';
            if (strpos($path, '/') === false && strpos($path, '\\') === false) {
                $htmPath = $appDirPath . 'web' . PK_DS . str_replace('|', PK_DS, $path) . '.htm';
                // 是否有对应的php文件
                $phpPath = $appDirPath . 'php' . PK_DS . str_replace('|', PK_DS, $path) . '.php';
            }
            if (!file_exists($htmPath) && !file_exists($phpPath)) {
                self::error('没有找到WEB/PHP文件:' . $path);
            }
            $html = '';
            if (file_exists($htmPath)) {
                $html = file_get_contents($htmPath);
            }
            if (file_exists($phpPath)) {
                $html = "{include('{$phpPath}')}\n{$html}";
            }
        }
        if (!$parse) {
            return $html;
        }
        // 脚本缓存文件路径
        $md5        = md5(md5($path) . md5($html));
        $cache_path = PK_CACHE_PATH . 'php' . PK_DS . $md5 . '.php';
        $init       = function ($vars, $cache_path) {
            // 是否存在该缓存文件
            if (!file_exists($cache_path)) {
                return false;
            }
            ob_start();
            $__Z7xIAtVHr_vars = $vars;
            $__Z7xIAtVHr_path = $cache_path;
            (function () use ($__Z7xIAtVHr_vars, $__Z7xIAtVHr_path) {
                // 带入变量
                if ($__Z7xIAtVHr_vars) {
                    foreach ($__Z7xIAtVHr_vars as $__Z7xIAtVHr_k => $__Z7xIAtVHr_v) {
                        $$__Z7xIAtVHr_k = $__Z7xIAtVHr_v;
                    }
                }
                include ($__Z7xIAtVHr_path);
            })();
            $__Z7xIAtVHr_r = ob_get_contents();
            ob_end_clean();
            return $__Z7xIAtVHr_r;
        };
        // 若存在脚本文件则加载
        $r = $init($vars, $cache_path);
        if (false !== $r) {
            return $r;
        }

        // 开始生成PHP模板文件
        // 包含模板
        if (preg_match_all('/\{include\(([\s\S]+?)\)\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php include({$v}); ?>", $html);
            }
        }
        //原生php
        if (preg_match_all('/\{php\}([\s\S]+?)\{\/php\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php {$v}; ?>", $html);
            }
        }
        //替换变量
        if (preg_match_all('/\{(\$[a-z0-9_\-"\>\'\.\[\]]+?)\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php echo isset({$v})?{$v}:''; ?>", $html);
            }
        }
        //替换函数
        if (preg_match_all('/\{\:([a-z0-9_\\\:]+\(\))\:\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php echo {$v}; ?>", $html);
            }
        }
        if (preg_match_all('/\{\:([a-z0-9_\\\:]+\([\s\S]+?\))\:\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php echo {$v}; ?>", $html);
            }
        }

        //实现foreach
        if (preg_match_all('/\{foreach ([\s\S]+?)\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php foreach{$v}{ ?>", $html);
            }
            $html = str_replace('{/foreach}', '<?php } ?>', $html);
        }
        //实现for
        if (preg_match_all('/\{for ([\s\S]+?)\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php for{$v}{ ?>", $html);
            }
            $html = str_replace('{/for}', '<?php } ?>', $html);
        }
        //实现if
        if (preg_match_all('/\{if ([\s\S]+?)\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php if{$v}{ ?>", $html);
            }
            if (preg_match_all('/\{elseif ([\s\S]+?)\}/i', $html, $match2)) {
                foreach ($match2[1] as $k2 => $v2) {
                    $html = str_replace($match2[0][$k2], "<?php }elseif{$v2}{ ?>", $html);
                }
            }
            $html = str_replace('{else /}', '<?php }else{ ?>', $html);
            $html = str_replace('{/if}', '<?php } ?>', $html);
        }
        //实现switch
        if (preg_match_all('/\{switch ([\s\S]+?)\}/i', $html, $match)) {
            foreach ($match[1] as $k => $v) {
                $html = str_replace($match[0][$k], "<?php switch(strval{$v}){case null:break; ?>", $html);
            }
            if (preg_match_all('/\{case \'([\s\S]+?)\'\}/i', $html, $match2)) {
                foreach ($match2[1] as $k2 => $v2) {
                    $html = str_replace($match2[0][$k2], "<?php case '{$v2}': ?>", $html);
                }
            }
            $html = str_replace('{break /}', '<?php break; ?>', $html);
            $html = str_replace('{default /}', '<?php default: ?>', $html);
            $html = str_replace('{/switch}', '<?php } ?>', $html);
        }
        //存储模板文件
        if (file_put_contents($cache_path, $html) === false) {
            self::error("缓存文件写入失败[path={$cache_path}]");
        }
        return $init($vars, $cache_path);
    }

    /**
     * 强制返回整数或小数数字
     * @param mixed $str 待处理的字符
     * @param mixed $return 若$str不符合要求返回的值
     * @param boolean $int 是否强制整数
     * @param mixed $min 最小值
     * @param mixed $max 最大值
     * @param boolean $minormax 当$str小于最小值时取最小值，当大于最大值时取最大值
     */
    public static function cNum($str, $return = 0, $int = true, $min = false, $max = false, $minormax = false)
    {
        if (is_numeric($str)) {
            if ($int) {
                if (function_exists('bcadd')) {
                    $str = bcadd($str, 0, 0);
                } else {
                    $str = number_format($str, 0, '.', '');
                }
            }
            if (false !== $min && $str < $min) {
                $str = $minormax ? $min : $return;
            }
            if (false !== $max && $str > $max) {
                $str = $minormax ? $max : $return;
            }
        } else {
            $str = $return;
        }

        return $str;
    }

    /**
     * 返回符合条件的字符串
     * @param mixed $str 待处理字符串
     * @param mixed $return 若不符合条件返回的值
     * @param mixed $cstr 字符串处理白名单
     * @param mixed $minlen 字符串最小长度
     * @param mixed $maxlen 字符串最大长度
     */
    public static function cStr($str, $return = false, $cstr = true, $minlen = false, $maxlen = false)
    {
        if (!is_string($str)) {
            return $return;
        }
        if (true === $cstr) {
            $cstr = self::config('string_safestrs');
        }
        $len = strlen($str);
        if ($cstr) {
            for ($i = 0; $i < $len; $i++) {
                $chk = strpos($cstr, substr($str, $i, 1));
                if (false === $chk) {
                    return $return;
                }
            }
        }
        if ($minlen < $maxlen) {
            if ($len < $minlen) {
                return $return;
            }
            if ($len > $maxlen) {
                return $return;
            }
        } elseif ($minlen && $minlen == $maxlen) {
            if ($len != $maxlen) {
                return $return;
            }
        }
        return $str;
    }

    /**
     * 将数据以json形式输出
     * @param mixed $data 待输出的数据
     */
    public static function exitJson($data = null)
    {
        header('Content-Type:application/json;charset=utf-8');
        $str = json_encode($data, 320);
        echo $str;
        exit;
    }

    /**
     * 成功 输出json数据并退出
     * {"code":1,"time":1655182540,"msg":"","data":null}
     * @param mixed $msg msg消息
     * @param mixed $data data数据
     */
    public static function success($msg = null, $data = null)
    {
        return self::exitJson([
            'code' => 1,
            'msg'  => $msg,
            'data' => $data,
            'time' => time(),
        ]);
    }

    /**
     * 失败 输出json数据并退出
     * {"code":0,"time":1655182540,"msg":"","data":null}
     * @param mixed $msg msg消息
     * @param mixed $data data数据
     */
    public static function error($msg = null, $data = null)
    {
        return self::exitJson([
            'code' => 0,
            'msg'  => $msg,
            'data' => $data,
            'time' => time(),
        ]);
    }

    /**
     * 判断是否手机来访
     */
    public static function isMobile()
    {
        $all_http = isset($_SERVER['ALL_HTTP']) ? strtolower($_SERVER['ALL_HTTP']) : '';
        $accept   = isset($_SERVER['HTTP_ACCEPT']) ? strtolower($_SERVER['HTTP_ACCEPT']) : '';
        $agent    = isset($_SERVER['HTTP_USER_AGENT']) ? strtolower($_SERVER['HTTP_USER_AGENT']) : '';
        // 大于0则为手机来访
        $mobile_browser = 0;
        if (preg_match('/(up.browser|up.link|mmp|symbian|smartphone|midp|wap|phone|iphone|ipad|ipod|android|xoom)/i', $agent)) {
            $mobile_browser++;
        }

        if (strpos($accept, 'application/vnd.wap.xhtml+xml') !== false) {
            $mobile_browser++;
        }

        if (isset($_SERVER['HTTP_X_WAP_PROFILE'])) {
            $mobile_browser++;
        }

        if (isset($_SERVER['HTTP_PROFILE'])) {
            $mobile_browser++;
        }

        $mobile_ua     = substr($agent, 0, 4);
        $mobile_agents = [
            'w3c ', 'acs-', 'alav', 'alca', 'amoi', 'audi', 'avan', 'benq', 'bird', 'blac', 'blaz', 'brew', 'cell', 'cldc', 'cmd-', 'dang', 'doco', 'eric', 'hipt', 'inno', 'ipaq', 'java', 'jigs', 'kddi', 'keji', 'leno', 'lg-c', 'lg-d', 'lg-g', 'lge-', 'maui', 'maxo', 'midp', 'mits', 'mmef', 'mobi', 'mot-', 'moto', 'mwbp', 'nec-', 'newt', 'noki', 'oper', 'palm', 'pana', 'pant', 'phil', 'play', 'port', 'prox', 'qwap', 'sage', 'sams', 'sany', 'sch-', 'sec-', 'send', 'seri', 'sgh-', 'shar', 'sie-', 'siem', 'smal', 'smar', 'sony', 'sph-', 'symb', 't-mo', 'teli', 'tim-', 'tosh', 'tsm-', 'upg1', 'upsi', 'vk-v', 'voda', 'wap-', 'wapa', 'wapi', 'wapp', 'wapr', 'webc', 'winw', 'winw', 'xda', 'xda-'];
        if (in_array($mobile_ua, $mobile_agents)) {
            $mobile_browser++;
        }

        if (strpos($all_http, 'operamini') !== false) {
            $mobile_browser++;
        }

        if (strpos($agent, 'windows') !== false) {
            $mobile_browser = 0;
        }

        if (strpos($agent, 'windows phone') !== false) {
            $mobile_browser++;
        }

        if ($mobile_browser > 0) {
            return true;
        }
        return false;
    }

    /**
     * 前端静态资源加载
     * @param mixed $urls 资源相对框架的路径，多个地址用,分割或输入字符串，当输入一个文件时auto可设置，多个文件auto自动为ture
     * @param boolean $auto 是否自动生成html标签，若遇到无法识别则原字符串输出
     */
    public static function webUrl($urls, $auto = false)
    {
        $marks = [
            'js'   => '<script src="{$url}"></script>',
            'css'  => '<link rel="stylesheet" href="{$url}" />',
            'icon' => '<link rel="icon" href="{$url}" sizes="32x32" />',
        ];
        if (is_string($urls)) {
            $urls = explode(',', $urls);
        }
        if (count($urls) > 1) {
            $auto = true;
        }
        $html = '';
        foreach ($urls as $url) {
            if (strpos($url, 'https://') !== 0 && strpos($url, 'http://') !== 0 && strpos($url, '//') !== 0) {
                $url = self::config('www_dir') . $url;
                $url = str_replace('\\', '/', $url);
                $url = preg_replace('#[\/]+#i', '/', $url);
            }
            // 获取文件后缀
            $k = pathinfo($url, PATHINFO_EXTENSION);
            if (isset($marks[$k]) && $auto) {
                $html .= str_replace('{$url}', $url, $marks[$k] . "\r\n");
            } else {
                $html .= $url;
            }
        }
        return $html;
    }

    /**
     * 上传文件对象
     * @param array $config 上传配置
     * $config = [
     * // 是否允许上传文件
     * 'open' => 1,
     * // 一次最多上传的文件数,0为不限制
     * 'count' => 0,
     * // 文件允许的最大大小,b,0为不限制
     * 'size' => 0,
     * // 文件存储的路径,相对于www目录
     * 'savepath' => 'upload/',
     * // 存储的文件名,不存在则随机
     * 'filename' => '',
     * // 允许上传的文件后缀,为空为不限制
     * 'suffix' => '',
     * // 若存在相同文件返回已上传的同文件
     * 'no_repeat' => 1
     * ];
     */
    public static function upload($config = [])
    {
        return (new upload($config));
    }

    /**
     * 创建随机数
     * @param int $len 随机数的长度
     * @param string $str 生成随机数的因子
     */
    public static function randomString(int $len = 7, string $str = '')
    {
        if (!$str) {
            $str = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890';
        }
        $strlen = strlen($str);
        $nstr   = '';
        if ($len > 0) {
            for ($i = 0; $i < $len; $i++) {
                $nstr .= substr($str, mt_rand(0, $strlen - 1), 1);
            }
        }
        return $nstr;
    }

    /**
     * json或数组快捷处理对象
     * @param mixed $json 待处理的json字符串/数组/对象
     * @param string $key 要获取的key值，留空为返回该json字符串
     * @param mixed $value 要设置的key值
     * @param boolean $return_array 是否返回数组，默认返回字符串
     */
    public static function jsonData($json, string $key = '', $value = null, $return_array = false)
    {
        if (!is_string($json)) {
            $json = json_encode($json);
        }
        $data = json_decode($json, true);
        if ('' === $key) {
            return $data;
        }
        $keys = explode('.', $key);
        $_d   = $data;
        $str  = '';
        foreach ($keys as $k => $v) {
            $str .= "['{$v}']";
            if (!isset($_d[$v])) {
                if (null === $value) {
                    // 要获取的值为空
                    return null;
                }
                $_d[$v] = [];
                eval("\$data{$str}=[];");
            }
            $_d = $_d[$v];
        }
        // 获取key值
        if (null === $value) {
            return $_d;
        }
        // 设置key值
        eval("\$data{$str}=\$value;");
        if ($return_array) {
            return $data;
        }
        return json_encode($data, 320);
    }

    /**
     * curl
     * @param string $url 远程地址
     * @param mixed $data 发送的数据，字符串或数组，字符串需要urlencode
     * @param string $method get/post
     * @param int $timeout 超时时间，秒
     * @param array $headers header信息
     */
    public static function cUrl(string $url, $data = [], string $method = 'get', int $timeout = 10, $headers = [])
    {
        $method = strtoupper($method);
        if (is_array($data)) {
            $data = http_build_query($data);
        }
        $ch = curl_init();
        if (false === $ch) {
            $opts = [
                'http' => [
                    'method'  => $method,
                    'content' => $data,
                    'timeout' => $timeout,
                ],
            ];
            $context = stream_context_create($opts);
            return file_get_contents($url, false, $context);
        }
        if ('POST' == $method) {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        } else {
            $url .= (strpos($url, '?') ? '&' : '?') . $data;
        }
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if ($headers) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        curl_setopt($ch, CURLOPT_FAILONERROR, 0);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $timeout);
        curl_setopt($ch, CURLOPT_HEADER, 0);
        //跳过SSL验证
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        $file_contents = curl_exec($ch);
        curl_close($ch);
        return $file_contents;
    }

    /**
     * 格式化文件大小显示
     * @param int $size 初始大小bytes
     * @param boolean $suffix 是否添加后缀
     */
    public static function formatBytes($size, $suffix = true)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];
        for ($i = 0; $size >= 1024 && $i < 6; $i++) {
            $size /= 1024;
        }
        return round($size, 2) . ($suffix ? $units[$i] : '');
    }

    /**
     * 计算目录或文件大小
     * @param string $dir_path  目录/文件路径
     * @return int 单位bytes
     */
    public static function dirSize($dir_path)
    {
        if (is_file($dir_path)) {
            return filesize($dir_path);
        }
        if (!is_dir($dir_path)) {
            return 0;
        }
        $dirsize = 0;
        $dir     = opendir($dir_path);
        while ($fileName = readdir($dir)) {
            $file = $dir_path . PK_DS . $fileName;
            if ('.' != $fileName && '..' != $fileName) {
                if (is_dir($file)) {
                    $dirsize += self::dirSize($file);
                } else {
                    $dirsize += filesize($file);
                }
            }
        }
        closedir($dir);
        return $dirsize;
    }

    /**
     * 首字母头像
     * @param string $text 文本内容
     * @param int $whf 图片大小
     * @param mixed $rgb 背景rgb，数组或字符串
     */
    public static function letterAvatar(string $text, int $whf = 100, $rgb = '')
    {
        $text = strtoupper(mb_substr($text, 0, 1));
        $md5  = substr(md5($text), 2);
        if (!$rgb) {
            $rgb = [];
            for ($i = 0; $i < 3; $i++) {
                $a = substr($md5, $i * 10, 10);
                $b = 0;
                for ($j = 0; $j < 10; $j++) {
                    $b += ord(substr($a, $j, 1));
                }
                $b     = intval(($b - 480) * 0.3446);
                $rgb[] = $b;
            }
        } else {
            if (is_string($rgb)) {
                $rgb = explode(',', $rgb);
            }
        }
        $bg    = "rgb({$rgb[0]},{$rgb[1]},{$rgb[2]})";
        $f     = intval($whf / 2);
        $src   = base64_encode('<svg xmlns="http://www.w3.org/2000/svg" version="1.1" height="' . $whf . '" width="' . $whf . '"><rect fill="' . $bg . '" x="0" y="0" width="' . $whf . '" height="' . $whf . '"></rect><text x="' . $f . '" y="' . $f . '" font-size="' . $f . '" fill="#FFFFFF" style="dominant-baseline:middle;text-anchor:middle">' . $text . '</text></svg>');
        $value = 'data:image/svg+xml;base64,' . $src;
        return $value;
    }

    /**
     * 文件下载,服务器文件下载给客户
     * @param $filepath 下载的文件完整名称
     * @param $filename 下载后的文件名称
     * @param int $readsize 每次读取的字节数 默认4096
     */
    public static function sendFile($filepath, $filename = false, $readsize = 4096)
    {
        // 检测下载的文件是否可以被下载
        if (!is_file($filepath) || !is_readable($filepath)) {
            return false;
        }
        //设置下载文件的名称
        $suffix = pathinfo($filepath, PATHINFO_EXTENSION);
        if (!$filename) {
            $filename = date('YmdHis');
        }
        $filename = trim($filename, '.' . $suffix) . '.' . $suffix;
        // 设置头信息
        header('Content-Type:application/octet-stream');
        header('Accept-Ranges:bytes');
        header('Content-Length:' . filesize($filepath));
        header('Content-Disposition:attachment;filename=' . $filename);
        //获取文件内容
        $handle = fopen($filepath, 'rb');
        while (!feof($handle)) {
            echo fread($handle, $readsize);
        }
        fclose($handle);
        exit;
    }

    /**
     * 文件下载至服务器
     * @param $file_source 远程文件地址
     * @param $file_target 下载至本地的完整文件名
     * @param $file_cover 是否覆盖下载
     */
    public static function download($file_source, $file_target, $file_cover = true)
    {
        if (!is_dir(dirname($file_target)) && !mkdir(dirname($file_target, 0777, true))) {
            // 目标文件所在的目录不存在且无法创建
            self::error(self::language('目录创建失败'));
            return false;
        }
        if (!$file_cover && is_file($file_target)) {
            // 目标文件已存在且不允许覆盖
            self::error(self::language('已存在相同的文件'));
            return false;
        }
        $rh = fopen($file_source, 'rb');
        $wh = fopen($file_target, 'wb');
        if (false === $rh || false === $wh) {
            self::error(self::language('读取或打开文件失败'));
            return false;
        }
        while (!feof($rh)) {
            if (fwrite($wh, fread($rh, 1024)) === false) {
                self::error(self::language("文件无法被写入\"{$file_target}\""));
                return false;
            }
        }
        fclose($rh);
        fclose($wh);
        return true;
    }

    //时间转为人性化样式
    /**
     * @param $datetime 日期时间|时间戳|空为当前时间
     */
    public static function datetime2text($datetime = false)
    {
        if (!$datetime) {
            $datetime = time();
        }
        if (!self::cnum($datetime)) {
            $datetime = strtotime($datetime);
        }
        $time = time() - $datetime;
        if ($time < 60) {
            $r = self::language('刚刚');
        } elseif ($time < 3600) {
            //分
            $r = intval($time / 60) . ' ' . self::language('分钟前');
        } elseif ($time < 86400) {
            //时
            $r = intval($time / 3600) . ' ' . self::language('小时前');
        } elseif ($time < 86400 * 3) {
            //天
            $r = intval($time / 86400) . ' ' . self::language('天前');
        } else {
            $r = date('Y-m-d', $datetime);
        }
        return $r;
    }

    // 语言函数
    /**
     * @param $text 要匹配的语言文本
     * @param $default_language 要使用的语言包
     */
    public static function language($text, $default_language = null)
    {
        $r = self::g('language');
        if ($default_language) {
            $r = include PK_LANG_PATH . $default_language . '.php';
        }
        if (!$r) {
            return $text;
        }
        if (!is_string($text) && !is_numeric($text)) {
            $text = json_encode($text, 320);
        }
        $key = strtolower($text);
        foreach ($r as $k => $v) {
            if (self::in_arrays($key, $k, false, '|')) {
                return $v;
            }
        }
        return $text;
    }
    public static function l($text, $default_language = null)
    {
        return self::language($text, $default_language);
    }

    /**
     * 图片处理快捷函数，验证码请使用code对象
     * @param $file 要处理的图片文件路径
     */
    public static function image($file = null)
    {
        return (new image($file));
    }

    /*
     * 该函数的设置方法及参数同setcookie，仅传入$name将会取值
     * 该函数会自动带前缀
     * @expire 为多少秒后过期，负数或0为马上过期
     */
    public static function cookie($name, $value = null, $expire = null, $path = '/', $domain = null)
    {
        $name = self::config('cookie_prefix') . $name;
        if (null === $value) {
            return isset($_COOKIE[$name]) ? $_COOKIE[$name] : null;
        }
        return setcookie($name, $value, time() + $expire, $path, $domain);
    }

    /*
     * session，仅传入$name将会取值
     */
    public static function session($name, $value = null)
    {
        $name = self::config('cookie_prefix') . $name;
        if (null === $value) {
            return isset($_SESSION[$name]) ? $_SESSION[$name] : null;
        }
        return $_SESSION[$name] = $value;
    }
    /**
     * in_array函数
     * @param $needle 搜索的字符串或数组
     * @param $array 被搜索的字符串或数组
     * @param $allin 多个搜索对象，为true时所有都存在返回true，为false只要一个存在返回true
     * @param $delimiter 如果对象是字符串用什么把他分割成数组
     */
    public static function in_arrays($needle, $array, $allin = true, $delimiter = ',')
    {
        if (!is_array($needle)) {
            $needle = explode($delimiter, (string) $needle);
        }
        if (!is_array($array)) {
            $array = explode($delimiter, (string) $array);
        }
        foreach ($needle as $v) {
            if (in_array((string) $v, $array)) {
                if (!$allin) {
                    return true;
                }
            } else {
                if ($allin) {
                    return false;
                }
            }
        }
        if ($allin) {
            return true;
        }
        return false;
    }
    public static function inArray($needle, $array, $allin = true, $delimiter = ',')
    {
        return self::in_arrays($needle, $array, $allin, $delimiter);
    }
    /*
     * 删除文件夹及下面的所有文件,返回删除的文件总数
     */
    public static function unDir($dir)
    {
        $i  = 0;
        $dh = opendir($dir);
        while ($file = readdir($dh)) {
            if ('.' != $file && '..' != $file) {
                $fullpath = $dir . PK_DS . $file;
                if (!is_dir($fullpath)) {
                    if (unlink($fullpath)) {
                        $i++;
                    }
                } else {
                    $i += self::undir($fullpath);
                }
            }
        }
        closedir($dh);
        rmdir($dir);
        return $i;
    }

    /**
     * 判断用户浏览器的类型
     * @param mixed $agent HTTP_USER_AGENT
     */
    public static function getBrowserType($agent = null)
    {
        $agent = null !== $agent ? $agent : strtolower($_SERVER['HTTP_USER_AGENT']);
        $strs  = [
            'msie'         => 'ie',
            'edge'         => 'edge',
            'firefox'      => 'firefox',
            'chrome'       => 'chrome',
            'safari'       => 'safari',
            'opera'        => 'opera',
            'omniweb'      => 'omniweb',
            'netscape'     => 'netscape',
            'micromessage' => 'wechat',
        ];
        foreach ($strs as $k => $v) {
            if (strpos($agent, $k) !== false) {
                return $v;
            }
        }
        return 'unkown';
    }
    /**
     * html标签安全化处理
     * @param string $html 输入的html
     * @param string $marks html标签白名单,留空为保留所有，格式：<b><div>
     * @param string $attrs html标签属性白名单,留空为保留所有，格式：class,href
     */
    public static function html2safeHtml($html, $marks = '', $attrs = '')
    {
        //第一次过滤
        $html = strip_tags($html, $marks);
        //第二次过滤
        if (!$marks) {
            return $html;
        }
        if (preg_match_all('/<(?!\/)([\s\S]+?)>/', $html, $match)) {
            // 获取允许的属性数据
            $attrs = explode(',', $attrs);
            foreach ($match[1] as $value2) {
                //获取当前标签名
                $bqn = substr($value2, 0, strpos($value2, ' '));
                if (!$bqn) {
                    // 该标签无属性
                    continue;
                }
                $markhtml = "<{$bqn}";
                foreach ($attrs as $attr) {
                    if (preg_match('/' . $attr . '="(.{0}|[\s\S]+?)"/i', $value2, $match2)) {
                        if (strtolower($bqn) == 'a' && strtolower($attr) == 'href') {
                            // 禁止href js
                            if (strpos(str_replace([" ", "　", "\r", "\n", "\t"], '', strtolower($match2[1])), 'javascript:') !== false) {
                                $match2[0] = 'href="#"';
                            }
                        }
                        $markhtml .= " {$match2[0]}";
                    }
                }
                $markhtml .= '>';
                $html = str_replace("<{$value2}>", $markhtml, $html);
            }
        }
        return $html;
    }
    /**
     * @param $url http地址
     * @param $param get参数
     * @param $return 是否返回不退出
     */
    public static function goUrl($url, $param = false, $return = false)
    {
        $url = self::weburl($url);
        if (true === $param) {
            $param            = [];
            $param['referer'] = self::g('sys.config.location');
        }
        if (is_array($param)) {
            $param = http_build_query($param);
        }
        if (is_string($param) || is_numeric($param)) {
            if (strpos($url, '?')) {
                $url = rtrim($url, '&') . '&';
            } else {
                $url .= '?';
            }
            $url .= $param;
        }
        if ($return) {
            return $url;
        }
        header('Location:' . $url);
        echo '<script>location.href = "' . $url . '";</script>';
        exit;
    }
    /**
     * 伪静态地址生成
     * @param string $pathname 访问的实际路径，应用名称/入口文件
     * @param string $search 访问的get参数
     */
    public static function reWriteUrl($pathname = '', $search = '')
    {
        $rules    = self::g('route');
        $pathname = trim(self::weburl($pathname), '/');
        $search   = trim(trim($search, '?'), '&');
        $url      = self::weburl($pathname . '?' . $search);
        if (!is_array($rules)) {
            return $url;
        }
        // search整理
        $searchs  = [];
        $_searchs = explode('&', $search);
        foreach ($_searchs as $_v) {
            $_v = explode('=', $_v);
            if (count($_v) != 2) {
                continue;
            }
            $searchs[$_v[0]] = $_v[1];
        }
        foreach ($rules as $furl => $turl) {
            $turls = explode('?', $turl);
            if (count($turls) != 2) {
                continue;
            }
            $tpathname = trim($turls[0], '/');
            $tsearch   = trim(trim($turls[1], '?'), '&');
            if ($tpathname != $pathname) {
                continue;
            }
            $tsearchs  = [];
            $_tsearchs = explode('&', $tsearch);
            foreach ($_tsearchs as $_v) {
                $_v = explode('=', $_v);
                if (count($_v) != 2) {
                    continue;
                }
                $tsearchs[$_v[0]] = $_v[1];
            }
            // 必须包含真实地址的所有参数，否则放弃
            foreach ($tsearchs as $key => $value) {
                if (!isset($searchs[$key])) {
                    continue 2;
                }
            }
            // 判断非变量地址是否相等，不相等则放弃
            foreach ($tsearchs as $key => $value) {
                if (preg_replace('/\$([0-9]+)/i', '', $value) === '') {
                    continue;
                }
                if ($searchs[$key] != $value) {
                    continue 2;
                }
            }
            // 更改地址
            $rewriteurl = trim(rtrim(ltrim($furl, '^'), '$'), '/');
            for ($n = 1; true; $n++) {
                $cz = false;
                foreach ($tsearchs as $key => $value) {
                    if ('$' . $n == $value) {
                        $cz = true;
                        break;
                    }
                }
                if (!$cz) {
                    break;
                }
                $rewriteurl = preg_replace('/\(.*?\)/i', $searchs[$key], $rewriteurl, 1);
            }
            // 追加未被伪静态的search
            $rewriteurl_add = '';
            foreach ($searchs as $key => $value) {
                if (!isset($tsearchs[$key])) {
                    $rewriteurl_add .= $key . '=' . $value . '&';
                }
            }
            $rewriteurl_add = trim($rewriteurl_add, '&');
            if ($rewriteurl_add) {
                if (strpos($rewriteurl, '?') === false) {
                    $rewriteurl .= '?';
                } else {
                    $rewriteurl .= '&';
                }
                $rewriteurl .= $rewriteurl_add;
            }
            $rewriteurl = self::weburl($rewriteurl);
            return $rewriteurl;
        }
        return $url;
    }
    /**
     * 程序到该函数调用时消耗的时间,单位s
     * @param number $limit 保留几位小数，默认4
     */
    public static function runtime($limit = 4)
    {
        return round(microtime(true) - PK_START_MICROTIME, $limit);
    }
    /**
     * 获取html里面的img对象
     * @param $html 待处理的html
     * @param $savepath 图片存储的路径
     * @param $dbs 存入数据库的数据，支持变量替换
     * @param $norepeat 是否允许同一文件被重复转存，需$dbs设置后生效
     * @param $maxN 最多获取img的个数，0为全部获取
     * @param $noAlt 为不获取的alt属性内容
     * @param $noKeywords 为不获取的路径关键词
     * @return string 被处理后的$html
     */
    public static function imgsBase64ToFile($html, $savepath, $dbs = [], $norepeat = false, $maxN = 0, $noAlt = 'emotion,emoticon', $noKeywords = 'emotion,emoticon')
    {
        if (strpos($savepath, PK_WWWROOT) === 0) {
            $savepath = substr($savepath, strlen(PK_WWWROOT));
        }
        $savepath = str_replace('\\', '/', $savepath);
        $savepath = '/' . trim($savepath, '/') . '/';
        if (!is_dir(PK_WWWROOT . $savepath) && !mkdir(PK_WWWROOT . $savepath, 0777, true)) {
            return $html;
        }
        $imgs = self::getHtmlImgs($html, $maxN, $noAlt, $noKeywords, false);
        foreach ($imgs as $img) {
            if ('data:image/png;base64,' != substr($img['src'], 0, 22)) {
                continue;
            }
            $dir_path = $savepath;
            $filename = md5(self::randomString(9)) . '.png';
            $filepath = $dir_path . $filename;
            if (file_put_contents(PK_WWWROOT . $filepath, base64_decode(substr($img['src'], 22))) === false) {
                continue;
            }
            if ($dbs) {
                $vs = [
                    'dir_path' => $dir_path,
                    'filename' => $filename,
                    'filepath' => $filepath,
                    'filetype' => filetype(PK_WWWROOT . $filepath),
                    'filesize' => filesize(PK_WWWROOT . $filepath),
                    'md5_file' => md5_file(PK_WWWROOT . $filepath),
                ];
                if ($norepeat) {
                    foreach ($dbs as $db => $data) {
                        $db = self::db($db)->where('md5_file', $vs['md5_file'])->find();
                        if ($db) {
                            $url  = self::weburl($db['filepath']);
                            $html = str_replace("src=\"{$img['src']}\"", "src=\"{$url}\"", $html);
                            // 删除文件
                            unlink(PK_WWWROOT . $filepath);
                            continue 2;
                        }
                    }

                }
                foreach ($dbs as $db => $data) {
                    foreach ($vs as $k => $v) {
                        foreach ($data as $k2 => $v2) {
                            $data[$k2] = str_replace('{' . $k . '}', $v, $v2);
                        }
                    }
                    self::db($db)->insert($data);
                }
            }
            $url  = self::weburl($filepath);
            $html = str_replace("src=\"{$img['src']}\"", "src=\"{$url}\"", $html);
        }
        return $html;
    }
    /**
     * 获取html里面的img对象
     * @param $html 待处理的html
     * @param $maxN 最多获取img的个数，0为全部获取
     * @param $noAlt 为不获取的alt属性内容
     * @param $noKeywords 为不获取的路径关键词
     * @param $nobase64 是否获取base64的值，默认获取
     * @return array [["src"=>"","alt"=>""]]
     */
    public static function getHtmlImgs($html, $maxN = 0, $noAlt = 'emotion,emoticon', $noKeywords = 'emotion,emoticon', $nobase64 = false)
    {
        $array = [];
        if (!preg_match_all('#<img.*?src="(.*?)".*?\>#', $html, $marks)) {
            return $array;
        }
        $s          = 0;
        $noKeywords = explode(',', $noKeywords);
        foreach ($marks[0] as $value) {
            if ($maxN && $maxN <= $s) {
                break;
            }
            $value = substr($value, 5, strlen($value) - 6) . ' ';
            if (preg_match_all('/([a-z_][a-z0-9_\-]+)\="(.*?)"/i', $value, $attrs)) {
                $i = count($array);
                foreach ($attrs[1] as $k => $a) {
                    $a             = trim(strtolower($a));
                    $v             = trim($attrs[2][$k]);
                    $array[$i][$a] = $v;
                }
                if ($noAlt && isset($array[$i]['alt']) && self::in_arrays(strtolower($array[$i]['alt']), strtolower($noAlt))) {
                    unset($array[$i]);
                    break;
                }
                foreach ($noKeywords as $noKeyword) {
                    if ($noKeyword && isset($array[$i]['src']) && strpos(strtolower($array[$i]['src']), $noKeyword) !== false) {
                        unset($array[$i]);
                        break 2;
                    }
                }
                if ($nobase64 && isset($array[$i]['src']) && strpos($array[$i]['src'], 'data:image/png;base64,') === 0) {
                    unset($array[$i]);
                    break;
                }
                $s++;
            }
        }
        return $array;
    }

    /**
     * 获取html里面指定对象
     * @param $html 待处理的html
     * @param $mark 待获取的mark字符串
     * @param $maxN 最多获取$mark的个数，0为全部获取
     * @return array [[attr=>value,]]
     */
    public static function getHtmlMarks($html, $mark, $maxN = 0)
    {
        $array = [];
        if (!preg_match_all('#\<' . $mark . ' .*?\>#', $html, $marks)) {
            return $array;
        }
        $s = 0;
        foreach ($marks[0] as $value) {
            if ($maxN && $maxN <= $s) {
                break;
            }
            // 若为<p>这种则排除
            // if (!self::in_arrays(substr($value, strlen($mark) + 1, 1), [' ', '>'])) {
            //     continue;
            // }
            $value = substr($value, strlen($mark) + 2, strlen($value) - strlen($mark) - 3) . ' ';
            if (preg_match_all('/([a-z_][a-z0-9_\-]+)\="(.*?)"/i', $value, $attrs)) {
                $i = count($array);
                foreach ($attrs[1] as $k => $a) {
                    $a             = trim(strtolower($a));
                    $v             = trim($attrs[2][$k]);
                    $array[$i][$a] = $v;
                }
                $s++;
            }
        }
        return $array;
    }

    /**
     * 强制返回字符串，若不为字符串类型，无论如何都返回''
     * @param $str 待处理的对象
     */
    public static function returnString($str)
    {
        if (is_string($str)) {
            return $str;
        }
        return '';
    }
}
