<?php
class image
{
    protected $info = [
        'size'   => 0,
        'width'  => 0,
        'height' => 0,
        'bits'   => '',
        'mime'   => '',
    ];
    protected $file      = false;
    protected $image     = false;
    protected $error     = false;
    protected $watermark = false;
    protected $resize    = false;
    protected $fontpath  = PK_OTH_PATH . 'font' . PK_DS . 'default.ttf';

    /**
     * 图片路径
     * Image constructor.
     * @param $file
     */
    public function __construct($file = null)
    {
        if ($file) {
            $this->file($file);
        }
    }

    /**
     * 图片路径
     * @param $file 待处理的图片文件路径
     */
    public function file($file)
    {
        $this->file = $file;
        $info       = getimagesize($this->file);
        if (!$info) {
            $this->error = k::language('初始化文件失败');
            return $this;
        }

        $this->info['size']   = filesize($this->file);
        $this->info['width']  = $info[0];
        $this->info['height'] = $info[1];
        $this->info['bits']   = isset($info['bits']) ? $info['bits'] : '';
        $this->info['mime']   = isset($info['mime']) ? $info['mime'] : '';

        switch ($this->info['mime']) {
            case 'image/gif':
                $this->image = imagecreatefromgif($file);
                break;
            case 'image/png':
                $this->image = imagecreatefrompng($file);
                break;
            case 'image/jpeg':
            default:
                $this->image = imagecreatefromjpeg($file);
                break;
        }

        return $this;
    }
    /*
     * 获取当前图片文件的信息，返回数组
     */
    public function info($info = null)
    {
        if (!$info) {
            return $this->info;
        }
        return $this->info[$info];
    }

    /*
     * 获取错误信息
     */
    public function error()
    {
        return $this->error;
    }

    /*
     * 获取图片对象
     */
    public function image()
    {
        return $this->image;
    }

    /**
     * 重置尺寸，宽度或高度任何一项为0都将自适应填充
     * @param int $width 新宽度
     * @param int $height 新高度
     */
    public function resize($width = 0, $height = 0)
    {
        $this->resize = [
            'width'  => $width,
            'height' => $height,
        ];
        return $this;
    }

    /**
     * 水印设置
     * @param $text 水印图片的路径或水印文字
     * @param $position 水印的位置,0-9,0为随机，1上左，2上中，3上右，4中左，5中心，6中右，7下左，8下中，9下右
     * @param $type 水印图片还是文字，默认为文字,图片为img
     * @param $fontsize 水印文字大小，默认14
     * @param $fontcolor 水印文字颜色，默认白色，rgb格式或数组
     * @param $fontpath 水印文字文件路径，默认不支持中文
     */
    public function watermark($text, $position = 0, $type = 'txt', $fontsize = 16, $fontcolor = '255,255,255', $fontpath = '')
    {
        if (!$fontpath) {
            $fontpath = $this->fontpath;
        }
        $this->watermark = [
            // img or txt
            'type'      => $type,
            // path or text
            'text'      => $text,
            'position'  => $position,
            'fontsize'  => $fontsize,
            'fontpath'  => $fontpath,
            'fontcolor' => $fontcolor,
        ];
        return $this;
    }

    /*
     * 图片加水印处理
     * @param int $width
     * @param int $height
     */
    protected function _watermark()
    {
        if (!$this->watermark) {
            return $this;
        }
        //图片水印
        if ('img' == $this->watermark['type']) {
            // 获取图片水印的水印对象
            $markimage  = new image($this->watermark['text']);
            $markwidth  = $markimage->info('width');
            $markheight = $markimage->info('height');
        } else {
            // 文字水印
            $box        = imagettfbbox($this->watermark['fontsize'], 0, $this->watermark['fontpath'], $this->watermark['text']);
            $markwidth  = max($box[2], $box[4]) - min($box[0], $box[6]);
            $markheight = max($box[1], $box[3]) - min($box[5], $box[7]);
        }
        $markpos = $this->watermark['position'];
        if ($markpos < 1 || $markpos > 9) {
            $markpos = mt_rand(1, 9);
        }
        // 计算位置
        $x = 5;
        $y = 5;
        switch ($markpos) {
            case 2:
                $x = ($this->info['width'] - $markwidth) / 2;
                break;
            case 3:
                $x = $this->info['width'] - $markwidth - 5;
                $y = 15;
                break;
            case 4:
                $y = ($this->info['height'] - $markheight) / 2;
                break;
            case 5:
                $x = ($this->info['width'] - $markwidth) / 2;
                $y = ($this->info['height'] - $markheight) / 2;
                break;
            case 6:
                $x = $this->info['width'] - $markwidth - 5;
                $y = ($this->info['height'] - $markheight) / 2;
                break;
            case 7:
                $y = $this->info['height'] - $markheight - 5;
                break;
            case 8:
                $x = ($this->info['width'] - $markwidth) / 2;
                $y = $this->info['height'] - $markheight - 5;
                break;
            case 9:
                $x = $this->info['width'] - $markwidth - 5;
                $y = $this->info['height'] - $markheight - 5;
                break;
        }
        $image = imagecreatetruecolor($this->info['width'], $this->info['height']);
        imagecopy($image, $this->image, 0, 0, 0, 0, $this->info['width'], $this->info['height']);
        if ('img' == $this->watermark['type']) {
            imagecopy($image, $markimage->image(), $x, $y, 0, 0, $markwidth, $markheight);
        } else {
            $rgb = $this->watermark['fontcolor'];
            if (is_string($rgb)) {
                $rgb = explode(',', $rgb);
            }
            $color = imagecolorallocate($image, $rgb[0], $rgb[1], $rgb[2]);
            imagettftext($image, $this->watermark['fontsize'], 0, $x, $y, $color, $this->watermark['fontpath'], $this->watermark['text']);
        }
        $this->image = $image;
        return $this;
    }

    /*
     * 图片尺寸处理，默认为0，0为根据以设置的宽或高等比例扩大或缩小
     */
    protected function _resize()
    {
        $width  = $this->resize['width'];
        $height = $this->resize['height'];
        if ($width <= 0 && $height <= 0) {
            return $this;
        }
        if ($width <= 0) {
            $width = ($this->info['width'] / $this->info['height'] * $height);
        }
        if ($height <= 0) {
            $height = ($this->info['height'] / $this->info['width'] * $width);
        }
        $_image = imagecreatetruecolor($width, $height);
        imagecopyresampled($_image, $this->image, 0, 0, 0, 0, $width, $height, $this->info['width'], $this->info['height']);
        $this->image          = $_image;
        $this->info['width']  = $width;
        $this->info['height'] = $height;
        return $this;
    }

    /**
     * 初始化图片对象
     */
    public function init()
    {
        // 添加水印
        $this->_watermark();

        // 调整大小
        $this->_resize();
        return $this;
    }

    /**
     * 生成验证码
     */
    public function code($length = 4, $fontsize = 18, $width = 99, $height = 38, $randomString = '')
    {
        $code = k::randomString($length, $randomString);
        // 创建背景
        $image = imagecreatetruecolor($width, $height);
        $color = imagecolorallocate($image, mt_rand(157, 255), mt_rand(157, 255), mt_rand(157, 255));
        imagefilledrectangle($image, 0, $height, $width, 0, $color);
        //线条
        $lineN = mt_rand(3, 7);
        for ($i = 0; $i < $lineN; $i++) {
            $color = imagecolorallocate($image, mt_rand(0, 156), mt_rand(0, 156), mt_rand(0, 156));
            imageline($image, mt_rand(0, $width), mt_rand(0, $height), mt_rand(0, $width), mt_rand(0, $height), $color);
        }
        //雪花
        $snowN = mt_rand(10, 50);
        for ($i = 0; $i < $snowN; $i++) {
            $color = imagecolorallocate($image, mt_rand(200, 255), mt_rand(200, 255), mt_rand(200, 255));
            imagestring($image, mt_rand(1, 5), mt_rand(0, $width), mt_rand(0, $height), '*', $color);
        }
        // 生成验证码
        $_x = $width / $length;
        for ($i = 0; $i < $length; $i++) {
            $fontcolor = imagecolorallocate($image, mt_rand(0, 156), mt_rand(0, 156), mt_rand(0, 156));
            imagettftext($image, $fontsize, mt_rand(-30, 30), $_x * $i + mt_rand(1, 5), $height / 1.4, $fontcolor, $this->fontpath, $code[$i]);
        }
        // 输出验证码
        header('content-type:image/png');
        imagepng($image);
        imagedestroy($image);
        return $code;
    }

    /**
     * 保存新图片
     * @param $filename 保存的路径，留空则输出到浏览器
     * @param int $quality 图片质量 0 -100 png与gif无法设置
     */
    public function save($filename = null, $quality = 90)
    {
        if ($this->error) {
            return false;
        }

        if (!is_resource($this->image)) {
            $this->error = k::language('图片对象初始化失败');
            return false;
        }

        $this->init();

        if ($filename) {
            // 保存图片文件
            switch (strtolower(pathinfo($filename, PATHINFO_EXTENSION))) {
                case 'gif':
                    imagegif($this->image, $filename);
                    break;
                case 'png':
                    imagepng($this->image, $filename);
                    break;
                case 'jpg':
                case 'jpeg':
                default:
                    imagejpeg($this->image, $filename, $quality);
                    break;
            }
        } else {
            // 输出图片到浏览器
            switch ($this->info['mime']) {
                case 'image/gif':
                    header('content-type:image/gif');
                    imagegif($this->image);
                    break;
                case 'image/png':
                    header('content-type:image/png');
                    imagepng($this->image);
                    break;
                case 'image/jpeg':
                default:
                    header('content-type:image/jpeg');
                    imagejpeg($this->image, null, $quality);
                    break;
            }
        }
        imagedestroy($this->image);
        return true;
    }
}
