﻿//-----------------------------------------------------------------------
// <copyright file="ActivatorUtilitiesConstructorTypeAdapter.cs" company="P.O.S Informatique">
//     Copyright (c) P.O.S Informatique. All rights reserved.
// </copyright>
//-----------------------------------------------------------------------
namespace PosInformatique.AspNet.WebForms.DependencyInjection
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Globalization;
    using System.Linq;
    using System.Reflection;
    using Microsoft.Extensions.DependencyInjection;

    /// <summary>
    /// Adapter which "virtual" declare a <see cref="ActivatorUtilitiesConstructorAttribute"/> on the constructors
    /// generated by ASP .NET compiler (aspnet_compiler) which call constructors coded by the developers
    /// marked with the <see cref="ActivatorUtilitiesConstructorAttribute"/>.
    /// </summary>
    internal sealed class ActivatorUtilitiesConstructorTypeAdapter : TypeDelegator
    {
        /// <summary>
        /// Gets the constructors of the <see cref="Type"/> wrapped in the <see cref="TypeDelegator"/> instance.
        /// </summary>
        private readonly ConstructorInfo[] constructors;

        /// <summary>
        /// Initializes a new instance of the <see cref="ActivatorUtilitiesConstructorTypeAdapter"/> class
        /// which wraps the <paramref name="type"/>.
        /// </summary>
        /// <param name="type"><see cref="Type"/> to wrap.</param>
        public ActivatorUtilitiesConstructorTypeAdapter(Type type)
            : base(type)
        {
            // Retrieves all the constructors
            this.constructors = type.GetConstructors(BindingFlags.Public | BindingFlags.Instance | BindingFlags.Static | BindingFlags.NonPublic);

            for (int i = 0; i < this.constructors.Length; i++)
            {
                // Try to retrieve a constructor with the same argument in the base type.
                var parameters = this.constructors[i].GetParameters().Select(p => p.ParameterType).ToArray();
                var baseConstructor = this.typeImpl.BaseType.GetConstructor(BindingFlags.Public | BindingFlags.Instance, null, parameters, null);

                if (baseConstructor != null)
                {
                    // A public instance constructor with same argument has been found, determines if the ActivatorUtilitiesConstructorAttribute has been applied.
                    if (baseConstructor.IsDefined(typeof(ActivatorUtilitiesConstructorAttribute)))
                    {
                        // The ActivatorUtilitiesConstructorAttribute has been defined, in this case
                        // we create a new instance of the ActivatorUtilitiesConstructorInfo which wraps the original constructor.
                        this.constructors[i] = new ActivatorUtilitiesConstructorInfo(this.constructors[i]);
                    }
                }
            }
        }

        /// <inheritdoc />
        public override ConstructorInfo[] GetConstructors(BindingFlags bindingAttr)
        {
            var candidates = new List<ConstructorInfo>(this.constructors.Length);

            foreach (var constructor in this.constructors)
            {
                // Checks the visibility
                bool matchVisibility = false;
                if (bindingAttr.HasFlag(BindingFlags.Public) && constructor.IsPublic)
                {
                    matchVisibility = true;
                }
                else if (bindingAttr.HasFlag(BindingFlags.NonPublic) && !constructor.IsPublic)
                {
                    matchVisibility = true;
                }

                if (!matchVisibility)
                {
                    continue;
                }

                // Checks static/non static
                if (bindingAttr.HasFlag(BindingFlags.Static) && constructor.IsStatic)
                {
                    candidates.Add(constructor);
                }
                else if (bindingAttr.HasFlag(BindingFlags.Instance) && !constructor.IsStatic)
                {
                    candidates.Add(constructor);
                }
            }

            return candidates.ToArray();
        }

        /// <summary>
        /// Adapter implementation which "virtualize" a <see cref="ConstructorInfo"/> by adding
        /// the <see cref="ActivatorUtilitiesConstructorAttribute"/>.
        /// </summary>
        private sealed class ActivatorUtilitiesConstructorInfo : ConstructorInfo
        {
            /// <summary>
            /// <see cref="ConstructorInfo"/> to wrap and virtual add the <see cref="ActivatorUtilitiesConstructorAttribute"/>.
            /// </summary>
            private readonly ConstructorInfo constructor;

            /// <summary>
            /// Initializes a new instance of the <see cref="ActivatorUtilitiesConstructorInfo"/> class
            /// which wraps the specified <paramref name="constructor"/>.
            /// </summary>
            /// <param name="constructor"><see cref="ConstructorInfo"/> to wrap and virtual add the <see cref="ActivatorUtilitiesConstructorAttribute"/>.</param>
            public ActivatorUtilitiesConstructorInfo(ConstructorInfo constructor)
                : base()
            {
                this.constructor = constructor;
            }

            /// <inheritdoc />
            public override RuntimeMethodHandle MethodHandle
            {
                get => this.constructor.MethodHandle;
            }

            /// <inheritdoc />
            public override MethodAttributes Attributes
            {
                get => this.constructor.Attributes;
            }

            /// <inheritdoc />
            public override string Name
            {
                get => this.constructor.Name;
            }

            /// <inheritdoc />
            public override Type DeclaringType
            {
                get => this.constructor.DeclaringType;
            }

            /// <inheritdoc />
            public override Type ReflectedType
            {
                get => this.constructor.ReflectedType;
            }

            /// <inheritdoc />
            public override object[] GetCustomAttributes(bool inherit)
            {
                var existingAttributes = this.constructor.GetCustomAttributes(inherit);

                var newAttributes = new object[existingAttributes.Length + 1];
                existingAttributes.CopyTo(newAttributes, 0);

                newAttributes[existingAttributes.Length] = new ActivatorUtilitiesConstructorAttribute();

                return newAttributes;
            }

            /// <inheritdoc />
            public override object[] GetCustomAttributes(Type attributeType, bool inherit)
            {
                if (attributeType == typeof(ActivatorUtilitiesConstructorAttribute))
                {
                    return new[] { new ActivatorUtilitiesConstructorAttribute() };
                }

                return this.constructor.GetCustomAttributes(attributeType, inherit);
            }

            /// <inheritdoc />
            public override MethodImplAttributes GetMethodImplementationFlags()
            {
                return this.constructor.GetMethodImplementationFlags();
            }

            /// <inheritdoc />
            public override ParameterInfo[] GetParameters()
            {
                return this.constructor.GetParameters();
            }

            /// <inheritdoc />
            public override object Invoke(BindingFlags invokeAttr, Binder binder, object[] parameters, CultureInfo culture)
            {
                return this.constructor.Invoke(invokeAttr, binder, parameters, culture);
            }

            /// <inheritdoc />
            public override object Invoke(object obj, BindingFlags invokeAttr, Binder binder, object[] parameters, CultureInfo culture)
            {
                return this.constructor.Invoke(obj, invokeAttr, binder, parameters, culture);
            }

            /// <inheritdoc />
            public override bool IsDefined(Type attributeType, bool inherit)
            {
                if (attributeType == typeof(ActivatorUtilitiesConstructorAttribute))
                {
                    return true;
                }

                return this.constructor.IsDefined(attributeType, inherit);
            }
        }
    }
}
