/* Copyright 2023-2024 Norconex Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.norconex.crawler.core.spi;

import java.util.function.Predicate;

import org.apache.commons.collections4.MultiMapUtils;
import org.apache.commons.collections4.MultiValuedMap;

import com.norconex.commons.lang.ClassFinder;
import com.norconex.commons.lang.bean.BeanMapper;
import com.norconex.commons.lang.bean.spi.PolymorphicTypeProvider;
import com.norconex.commons.lang.event.EventListener;
import com.norconex.crawler.core.doc.operations.checksum.DocumentChecksummer;
import com.norconex.crawler.core.doc.operations.checksum.MetadataChecksummer;
import com.norconex.crawler.core.doc.operations.filter.DocumentFilter;
import com.norconex.crawler.core.doc.operations.filter.MetadataFilter;
import com.norconex.crawler.core.doc.operations.filter.ReferenceFilter;
import com.norconex.crawler.core.doc.operations.spoil.SpoiledReferenceStrategizer;
import com.norconex.crawler.core.event.listeners.StopCrawlerOnMaxEventListener;
import com.norconex.crawler.core.stop.CrawlerStopper;
import com.norconex.crawler.core.store.DataStoreEngine;

/**
 * <p>
 * For auto registering in {@link BeanMapper}.
 * </p>
 */
public class CrawlerCorePtProvider implements PolymorphicTypeProvider {

    @Override
    public MultiValuedMap<Class<?>, Class<?>> getPolymorphicTypes() {
        //NOTE:
        // CrawlSessionConfig and CrawlerConfig are not registered here.
        // We leave it to crawler implementation to register them as/if
        // required. If one day we make all crawlers configurable in same
        // crawl session, this is likely where we would add it.

        MultiValuedMap<Class<?>, Class<?>> map =
                MultiMapUtils.newListValuedHashMap();
        addPolyType(map, CrawlerStopper.class); //TODO really want it configurable? maybe default choses between file-based and store-based.
        addPolyType(map, DataStoreEngine.class);
        addPolyType(map, DocumentChecksummer.class);
        addPolyType(map, DocumentFilter.class);
        addPolyType(map, EventListener.class, "event.listeners");
        addPolyType(map, MetadataChecksummer.class);
        addPolyType(map, MetadataFilter.class);
        addPolyType(map, ReferenceFilter.class);
        addPolyType(map, SpoiledReferenceStrategizer.class);
        map.put(EventListener.class, StopCrawlerOnMaxEventListener.class);
        return map;
    }

    private void addPolyType(
            MultiValuedMap<Class<?>, Class<?>> polyTypes,
            Class<?> baseClass) {
        addPolyType(polyTypes, baseClass, null);
    }

    private void addPolyType(
            MultiValuedMap<Class<?>, Class<?>> polyTypes,
            Class<?> baseClass,
            String corePkg) {
        polyTypes.putAll(
                baseClass, ClassFinder.findSubTypes(
                        baseClass,
                        corePkg == null
                                ? nm -> nm
                                        .startsWith(baseClass.getPackageName())
                                : filter(corePkg)));
    }

    private Predicate<String> filter(String corePkg) {
        return nm -> nm.startsWith("com.norconex.crawler.core." + corePkg);
    }
}
