// Copyright (C) 2024 Jérôme "SirLynix" Leclercq (lynix680@gmail.com)
// This file is part of the "Nazara Engine - Network module"
// For conditions of distribution and use, see copyright notice in Export.hpp

#pragma once

#ifndef NAZARA_NETWORK_POSIX_SOCKETIMPL_HPP
#define NAZARA_NETWORK_POSIX_SOCKETIMPL_HPP

#include <Nazara/Network/Enums.hpp>
#include <Nazara/Network/IpAddress.hpp>
#include <Nazara/Network/SocketHandle.hpp>

#define NAZARA_NETWORK_POLL_SUPPORT 1

namespace Nz
{
	struct NetBuffer;

	struct PollSocket
	{
		SocketHandle fd;
		short events;
		short revents;
	};

	class SocketImpl
	{
		public:
			SocketImpl() = delete;
			~SocketImpl() = delete;

			static SocketHandle Accept(SocketHandle handle, IpAddress* address, SocketError* error);

			static SocketState Bind(SocketHandle handle, const IpAddress& address, SocketError* error);

			static SocketHandle Create(NetProtocol protocol, SocketType type, SocketError* error);

			static void ClearErrorCode(SocketHandle handle);
			static void Close(SocketHandle handle);

			static SocketState Connect(SocketHandle handle, const IpAddress& address, SocketError* error);

			static bool Initialize();

			static SocketError GetLastError(SocketHandle handle, SocketError* error = nullptr);
			static int GetLastErrorCode(SocketHandle handle, SocketError* error = nullptr);

			static SocketState Listen(SocketHandle handle, const IpAddress& address, unsigned int queueSize, SocketError* error);

			static std::size_t QueryAvailableBytes(SocketHandle handle, SocketError* error = nullptr);
			static bool QueryBroadcasting(SocketHandle handle, SocketError* error = nullptr);
			static bool QueryKeepAlive(SocketHandle handle, SocketError* error = nullptr);
			static std::size_t QueryMaxDatagramSize(SocketHandle handle, SocketError* error = nullptr);
			static bool QueryNoDelay(SocketHandle handle, SocketError* error = nullptr);
			static IpAddress QueryPeerAddress(SocketHandle handle, SocketError* error = nullptr);
			static IpAddress QuerySocketAddress(SocketHandle handle, SocketError* error = nullptr);
			static std::size_t QueryReceiveBufferSize(SocketHandle handle, SocketError* error = nullptr);
			static std::size_t QuerySendBufferSize(SocketHandle handle, SocketError* error = nullptr);

			static unsigned int Poll(PollSocket* fdarray, std::size_t nfds, int timeout, SocketError* error);
			static SocketState PollConnection(SocketHandle handle, const IpAddress& address, UInt64 msTimeout, SocketError* error);

			static bool Receive(SocketHandle handle, void* buffer, int length, int* read, SocketError* error);
			static bool ReceiveFrom(SocketHandle handle, void* buffer, int length, IpAddress* from, int* read, SocketError* error);
			static bool ReceiveMultiple(SocketHandle handle, NetBuffer* buffers, std::size_t bufferCount, IpAddress* from, int* read, SocketError* error);

			static bool Send(SocketHandle handle, const void* buffer, int length, int* sent, SocketError* error);
			static bool SendMultiple(SocketHandle handle, const NetBuffer* buffers, std::size_t bufferCount, const IpAddress& to, int* sent, SocketError* error);
			static bool SendTo(SocketHandle handle, const void* buffer, int length, const IpAddress& to, int* sent, SocketError* error);

			static bool SetBlocking(SocketHandle handle, bool blocking, SocketError* error = nullptr);
			static bool SetBroadcasting(SocketHandle handle, bool broadcasting, SocketError* error = nullptr);
			static bool SetIPv6Only(SocketHandle handle, bool ipv6only, SocketError* error = nullptr);
			static bool SetKeepAlive(SocketHandle handle, bool enabled, UInt64 msTime, UInt64 msInterval, SocketError* error = nullptr);
			static bool SetNoDelay(SocketHandle handle, bool nodelay, SocketError* error = nullptr);
			static bool SetReceiveBufferSize(SocketHandle handle, std::size_t size, SocketError* error = nullptr);
			static bool SetSendBufferSize(SocketHandle handle, std::size_t size, SocketError* error = nullptr);

			static SocketError TranslateErrorToSocketError(int error);
			static int TranslateNetProtocolToAF(NetProtocol protocol);
			static int TranslateSocketTypeToSock(SocketType type);

			static void Uninitialize();

			static SocketHandle InvalidHandle;

		private:
			using socketID = int;
			static socketID s_socket;
	};
}

#endif // NAZARA_NETWORK_POSIX_SOCKETIMPL_HPP
