// Copyright (C) 2024 Jérôme "SirLynix" Leclercq (lynix680@gmail.com)
// This file is part of the "Nazara Engine - Core module"
// For conditions of distribution and use, see copyright notice in Export.hpp

/**
 * The Whirlpool hashing function.
 *
 * <P>
 * <b>References</b>
 *
 * <P>
 * The Whirlpool algorithm was developed by
 * <a href="mailto:pbarreto@scopus.com.br">Paulo S. L. M. Barreto</a> and
 * <a href="mailto:vincent.rijmen@cryptomathic.com">Vincent Rijmen</a>.
 *
 * See
 *    P.S.L.M. Barreto, V. Rijmen,
 *    ``The Whirlpool hashing function,''
 *    NESSIE submission, 2000 (tweaked version, 2001),
 *    <https://www.cosic.esat.kuleuven.ac.be/nessie/workshop/submissions/whirlpool.zip>
 *
 * @author  Paulo S.L.M. Barreto
 * @author  Vincent Rijmen.
 *
 * @version 3.0 (2003.03.12)
 *
 * =============================================================================
 *
 * Differences from version 2.1:
 *
 * - Suboptimal diffusion matrix replaced by cir(1, 1, 4, 1, 8, 5, 2, 9).
 *
 * =============================================================================
 *
 * Differences from version 2.0:
 *
 * - Generation of ISO/IEC 10118-3 test vectors.
 * - Bug fix: nonzero carry was ignored when tallying the data length
 *    (this bug apparently only manifested itself when feeding data
 *    in pieces rather than in a single chunk at once).
 * - Support for MS Visual C++ 64-bit integer arithmetic.
 *
 * Differences from version 1.0:
 *
 * - Original S-box replaced by the tweaked, hardware-efficient version.
 *
 * =============================================================================
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <Nazara/Core/Hash/Whirlpool.hpp>
#include <cstring>

#define ONE64 0xFFffffffffffffffULL

#define T64(x) ((x) & ONE64)
#define ROTR64(v, n) (((v) >> (n)) | T64((v) << (64 - (n))))

#define R 10

namespace Nz
{
	namespace NAZARA_ANONYMOUS_NAMESPACE
	{
		const UInt64 C0[256] = {
			0X18186018C07830D8ULL, 0X23238C2305AF4626ULL, 0XC6C63FC67EF991B8ULL, 0XE8E887E8136FCDFBULL,
			0X878726874CA113CBULL, 0XB8B8DAB8A9626D11ULL, 0X0101040108050209ULL, 0X4F4F214F426E9E0DULL,
			0X3636D836ADEE6C9BULL, 0XA6A6A2A6590451FFULL, 0XD2D26FD2DEBDB90CULL, 0XF5F5F3F5FB06F70EULL,
			0X7979F979EF80F296ULL, 0X6F6FA16F5FCEDE30ULL, 0X91917E91FCEF3F6DULL, 0X52525552AA07A4F8ULL,
			0X60609D6027FDC047ULL, 0XBCBCCABC89766535ULL, 0X9B9B569BACCD2B37ULL, 0X8E8E028E048C018AULL,
			0XA3A3B6A371155BD2ULL, 0X0C0C300C603C186CULL, 0X7B7BF17BFF8AF684ULL, 0X3535D435B5E16A80ULL,
			0X1D1D741DE8693AF5ULL, 0XE0E0A7E05347DDB3ULL, 0XD7D77BD7F6ACB321ULL, 0XC2C22FC25EED999CULL,
			0X2E2EB82E6D965C43ULL, 0X4B4B314B627A9629ULL, 0XFEFEDFFEA321E15DULL, 0X575741578216AED5ULL,
			0X15155415A8412ABDULL, 0X7777C1779FB6EEE8ULL, 0X3737DC37A5EB6E92ULL, 0XE5E5B3E57B56D79EULL,
			0X9F9F469F8CD92313ULL, 0XF0F0E7F0D317FD23ULL, 0X4A4A354A6A7F9420ULL, 0XDADA4FDA9E95A944ULL,
			0X58587D58FA25B0A2ULL, 0XC9C903C906CA8FCFULL, 0X2929A429558D527CULL, 0X0A0A280A5022145AULL,
			0XB1B1FEB1E14F7F50ULL, 0XA0A0BAA0691A5DC9ULL, 0X6B6BB16B7FDAD614ULL, 0X85852E855CAB17D9ULL,
			0XBDBDCEBD8173673CULL, 0X5D5D695DD234BA8FULL, 0X1010401080502090ULL, 0XF4F4F7F4F303F507ULL,
			0XCBCB0BCB16C08BDDULL, 0X3E3EF83EEDC67CD3ULL, 0X0505140528110A2DULL, 0X676781671FE6CE78ULL,
			0XE4E4B7E47353D597ULL, 0X27279C2725BB4E02ULL, 0X4141194132588273ULL, 0X8B8B168B2C9D0BA7ULL,
			0XA7A7A6A7510153F6ULL, 0X7D7DE97DCF94FAB2ULL, 0X95956E95DCFB3749ULL, 0XD8D847D88E9FAD56ULL,
			0XFBFBCBFB8B30EB70ULL, 0XEEEE9FEE2371C1CDULL, 0X7C7CED7CC791F8BBULL, 0X6666856617E3CC71ULL,
			0XDDDD53DDA68EA77BULL, 0X17175C17B84B2EAFULL, 0X4747014702468E45ULL, 0X9E9E429E84DC211AULL,
			0XCACA0FCA1EC589D4ULL, 0X2D2DB42D75995A58ULL, 0XBFBFC6BF9179632EULL, 0X07071C07381B0E3FULL,
			0XADAD8EAD012347ACULL, 0X5A5A755AEA2FB4B0ULL, 0X838336836CB51BEFULL, 0X3333CC3385FF66B6ULL,
			0X636391633FF2C65CULL, 0X02020802100A0412ULL, 0XAAAA92AA39384993ULL, 0X7171D971AFA8E2DEULL,
			0XC8C807C80ECF8DC6ULL, 0X19196419C87D32D1ULL, 0X494939497270923BULL, 0XD9D943D9869AAF5FULL,
			0XF2F2EFF2C31DF931ULL, 0XE3E3ABE34B48DBA8ULL, 0X5B5B715BE22AB6B9ULL, 0X88881A8834920DBCULL,
			0X9A9A529AA4C8293EULL, 0X262698262DBE4C0BULL, 0X3232C8328DFA64BFULL, 0XB0B0FAB0E94A7D59ULL,
			0XE9E983E91B6ACFF2ULL, 0X0F0F3C0F78331E77ULL, 0XD5D573D5E6A6B733ULL, 0X80803A8074BA1DF4ULL,
			0XBEBEC2BE997C6127ULL, 0XCDCD13CD26DE87EBULL, 0X3434D034BDE46889ULL, 0X48483D487A759032ULL,
			0XFFFFDBFFAB24E354ULL, 0X7A7AF57AF78FF48DULL, 0X90907A90F4EA3D64ULL, 0X5F5F615FC23EBE9DULL,
			0X202080201DA0403DULL, 0X6868BD6867D5D00FULL, 0X1A1A681AD07234CAULL, 0XAEAE82AE192C41B7ULL,
			0XB4B4EAB4C95E757DULL, 0X54544D549A19A8CEULL, 0X93937693ECE53B7FULL, 0X222288220DAA442FULL,
			0X64648D6407E9C863ULL, 0XF1F1E3F1DB12FF2AULL, 0X7373D173BFA2E6CCULL, 0X12124812905A2482ULL,
			0X40401D403A5D807AULL, 0X0808200840281048ULL, 0XC3C32BC356E89B95ULL, 0XECEC97EC337BC5DFULL,
			0XDBDB4BDB9690AB4DULL, 0XA1A1BEA1611F5FC0ULL, 0X8D8D0E8D1C830791ULL, 0X3D3DF43DF5C97AC8ULL,
			0X97976697CCF1335BULL, 0X0000000000000000ULL, 0XCFCF1BCF36D483F9ULL, 0X2B2BAC2B4587566EULL,
			0X7676C57697B3ECE1ULL, 0X8282328264B019E6ULL, 0XD6D67FD6FEA9B128ULL, 0X1B1B6C1BD87736C3ULL,
			0XB5B5EEB5C15B7774ULL, 0XAFAF86AF112943BEULL, 0X6A6AB56A77DFD41DULL, 0X50505D50BA0DA0EAULL,
			0X45450945124C8A57ULL, 0XF3F3EBF3CB18FB38ULL, 0X3030C0309DF060ADULL, 0XEFEF9BEF2B74C3C4ULL,
			0X3F3FFC3FE5C37EDAULL, 0X55554955921CAAC7ULL, 0XA2A2B2A2791059DBULL, 0XEAEA8FEA0365C9E9ULL,
			0X656589650FECCA6AULL, 0XBABAD2BAB9686903ULL, 0X2F2FBC2F65935E4AULL, 0XC0C027C04EE79D8EULL,
			0XDEDE5FDEBE81A160ULL, 0X1C1C701CE06C38FCULL, 0XFDFDD3FDBB2EE746ULL, 0X4D4D294D52649A1FULL,
			0X92927292E4E03976ULL, 0X7575C9758FBCEAFAULL, 0X06061806301E0C36ULL, 0X8A8A128A249809AEULL,
			0XB2B2F2B2F940794BULL, 0XE6E6BFE66359D185ULL, 0X0E0E380E70361C7EULL, 0X1F1F7C1FF8633EE7ULL,
			0X6262956237F7C455ULL, 0XD4D477D4EEA3B53AULL, 0XA8A89AA829324D81ULL, 0X96966296C4F43152ULL,
			0XF9F9C3F99B3AEF62ULL, 0XC5C533C566F697A3ULL, 0X2525942535B14A10ULL, 0X59597959F220B2ABULL,
			0X84842A8454AE15D0ULL, 0X7272D572B7A7E4C5ULL, 0X3939E439D5DD72ECULL, 0X4C4C2D4C5A619816ULL,
			0X5E5E655ECA3BBC94ULL, 0X7878FD78E785F09FULL, 0X3838E038DDD870E5ULL, 0X8C8C0A8C14860598ULL,
			0XD1D163D1C6B2BF17ULL, 0XA5A5AEA5410B57E4ULL, 0XE2E2AFE2434DD9A1ULL, 0X616199612FF8C24EULL,
			0XB3B3F6B3F1457B42ULL, 0X2121842115A54234ULL, 0X9C9C4A9C94D62508ULL, 0X1E1E781EF0663CEEULL,
			0X4343114322528661ULL, 0XC7C73BC776FC93B1ULL, 0XFCFCD7FCB32BE54FULL, 0X0404100420140824ULL,
			0X51515951B208A2E3ULL, 0X99995E99BCC72F25ULL, 0X6D6DA96D4FC4DA22ULL, 0X0D0D340D68391A65ULL,
			0XFAFACFFA8335E979ULL, 0XDFDF5BDFB684A369ULL, 0X7E7EE57ED79BFCA9ULL, 0X242490243DB44819ULL,
			0X3B3BEC3BC5D776FEULL, 0XABAB96AB313D4B9AULL, 0XCECE1FCE3ED181F0ULL, 0X1111441188552299ULL,
			0X8F8F068F0C890383ULL, 0X4E4E254E4A6B9C04ULL, 0XB7B7E6B7D1517366ULL, 0XEBEB8BEB0B60CBE0ULL,
			0X3C3CF03CFDCC78C1ULL, 0X81813E817CBF1FFDULL, 0X94946A94D4FE3540ULL, 0XF7F7FBF7EB0CF31CULL,
			0XB9B9DEB9A1676F18ULL, 0X13134C13985F268BULL, 0X2C2CB02C7D9C5851ULL, 0XD3D36BD3D6B8BB05ULL,
			0XE7E7BBE76B5CD38CULL, 0X6E6EA56E57CBDC39ULL, 0XC4C437C46EF395AAULL, 0X03030C03180F061BULL,
			0X565645568A13ACDCULL, 0X44440D441A49885EULL, 0X7F7FE17FDF9EFEA0ULL, 0XA9A99EA921374F88ULL,
			0X2A2AA82A4D825467ULL, 0XBBBBD6BBB16D6B0AULL, 0XC1C123C146E29F87ULL, 0X53535153A202A6F1ULL,
			0XDCDC57DCAE8BA572ULL, 0X0B0B2C0B58271653ULL, 0X9D9D4E9D9CD32701ULL, 0X6C6CAD6C47C1D82BULL,
			0X3131C43195F562A4ULL, 0X7474CD7487B9E8F3ULL, 0XF6F6FFF6E309F115ULL, 0X464605460A438C4CULL,
			0XACAC8AAC092645A5ULL, 0X89891E893C970FB5ULL, 0X14145014A04428B4ULL, 0XE1E1A3E15B42DFBAULL,
			0X16165816B04E2CA6ULL, 0X3A3AE83ACDD274F7ULL, 0X6969B9696FD0D206ULL, 0X09092409482D1241ULL,
			0X7070DD70A7ADE0D7ULL, 0XB6B6E2B6D954716FULL, 0XD0D067D0CEB7BD1EULL, 0XEDED93ED3B7EC7D6ULL,
			0XCCCC17CC2EDB85E2ULL, 0X424215422A578468ULL, 0X98985A98B4C22D2CULL, 0XA4A4AAA4490E55EDULL,
			0X2828A0285D885075ULL, 0X5C5C6D5CDA31B886ULL, 0XF8F8C7F8933FED6BULL, 0X8686228644A411C2ULL,
		};

		const UInt64 C1[256] = {
			0XD818186018C07830ULL, 0X2623238C2305AF46ULL, 0XB8C6C63FC67EF991ULL, 0XFBE8E887E8136FCDULL,
			0XCB878726874CA113ULL, 0X11B8B8DAB8A9626DULL, 0X0901010401080502ULL, 0X0D4F4F214F426E9EULL,
			0X9B3636D836ADEE6CULL, 0XFFA6A6A2A6590451ULL, 0X0CD2D26FD2DEBDB9ULL, 0X0EF5F5F3F5FB06F7ULL,
			0X967979F979EF80F2ULL, 0X306F6FA16F5FCEDEULL, 0X6D91917E91FCEF3FULL, 0XF852525552AA07A4ULL,
			0X4760609D6027FDC0ULL, 0X35BCBCCABC897665ULL, 0X379B9B569BACCD2BULL, 0X8A8E8E028E048C01ULL,
			0XD2A3A3B6A371155BULL, 0X6C0C0C300C603C18ULL, 0X847B7BF17BFF8AF6ULL, 0X803535D435B5E16AULL,
			0XF51D1D741DE8693AULL, 0XB3E0E0A7E05347DDULL, 0X21D7D77BD7F6ACB3ULL, 0X9CC2C22FC25EED99ULL,
			0X432E2EB82E6D965CULL, 0X294B4B314B627A96ULL, 0X5DFEFEDFFEA321E1ULL, 0XD5575741578216AEULL,
			0XBD15155415A8412AULL, 0XE87777C1779FB6EEULL, 0X923737DC37A5EB6EULL, 0X9EE5E5B3E57B56D7ULL,
			0X139F9F469F8CD923ULL, 0X23F0F0E7F0D317FDULL, 0X204A4A354A6A7F94ULL, 0X44DADA4FDA9E95A9ULL,
			0XA258587D58FA25B0ULL, 0XCFC9C903C906CA8FULL, 0X7C2929A429558D52ULL, 0X5A0A0A280A502214ULL,
			0X50B1B1FEB1E14F7FULL, 0XC9A0A0BAA0691A5DULL, 0X146B6BB16B7FDAD6ULL, 0XD985852E855CAB17ULL,
			0X3CBDBDCEBD817367ULL, 0X8F5D5D695DD234BAULL, 0X9010104010805020ULL, 0X07F4F4F7F4F303F5ULL,
			0XDDCBCB0BCB16C08BULL, 0XD33E3EF83EEDC67CULL, 0X2D0505140528110AULL, 0X78676781671FE6CEULL,
			0X97E4E4B7E47353D5ULL, 0X0227279C2725BB4EULL, 0X7341411941325882ULL, 0XA78B8B168B2C9D0BULL,
			0XF6A7A7A6A7510153ULL, 0XB27D7DE97DCF94FAULL, 0X4995956E95DCFB37ULL, 0X56D8D847D88E9FADULL,
			0X70FBFBCBFB8B30EBULL, 0XCDEEEE9FEE2371C1ULL, 0XBB7C7CED7CC791F8ULL, 0X716666856617E3CCULL,
			0X7BDDDD53DDA68EA7ULL, 0XAF17175C17B84B2EULL, 0X454747014702468EULL, 0X1A9E9E429E84DC21ULL,
			0XD4CACA0FCA1EC589ULL, 0X582D2DB42D75995AULL, 0X2EBFBFC6BF917963ULL, 0X3F07071C07381B0EULL,
			0XACADAD8EAD012347ULL, 0XB05A5A755AEA2FB4ULL, 0XEF838336836CB51BULL, 0XB63333CC3385FF66ULL,
			0X5C636391633FF2C6ULL, 0X1202020802100A04ULL, 0X93AAAA92AA393849ULL, 0XDE7171D971AFA8E2ULL,
			0XC6C8C807C80ECF8DULL, 0XD119196419C87D32ULL, 0X3B49493949727092ULL, 0X5FD9D943D9869AAFULL,
			0X31F2F2EFF2C31DF9ULL, 0XA8E3E3ABE34B48DBULL, 0XB95B5B715BE22AB6ULL, 0XBC88881A8834920DULL,
			0X3E9A9A529AA4C829ULL, 0X0B262698262DBE4CULL, 0XBF3232C8328DFA64ULL, 0X59B0B0FAB0E94A7DULL,
			0XF2E9E983E91B6ACFULL, 0X770F0F3C0F78331EULL, 0X33D5D573D5E6A6B7ULL, 0XF480803A8074BA1DULL,
			0X27BEBEC2BE997C61ULL, 0XEBCDCD13CD26DE87ULL, 0X893434D034BDE468ULL, 0X3248483D487A7590ULL,
			0X54FFFFDBFFAB24E3ULL, 0X8D7A7AF57AF78FF4ULL, 0X6490907A90F4EA3DULL, 0X9D5F5F615FC23EBEULL,
			0X3D202080201DA040ULL, 0X0F6868BD6867D5D0ULL, 0XCA1A1A681AD07234ULL, 0XB7AEAE82AE192C41ULL,
			0X7DB4B4EAB4C95E75ULL, 0XCE54544D549A19A8ULL, 0X7F93937693ECE53BULL, 0X2F222288220DAA44ULL,
			0X6364648D6407E9C8ULL, 0X2AF1F1E3F1DB12FFULL, 0XCC7373D173BFA2E6ULL, 0X8212124812905A24ULL,
			0X7A40401D403A5D80ULL, 0X4808082008402810ULL, 0X95C3C32BC356E89BULL, 0XDFECEC97EC337BC5ULL,
			0X4DDBDB4BDB9690ABULL, 0XC0A1A1BEA1611F5FULL, 0X918D8D0E8D1C8307ULL, 0XC83D3DF43DF5C97AULL,
			0X5B97976697CCF133ULL, 0X0000000000000000ULL, 0XF9CFCF1BCF36D483ULL, 0X6E2B2BAC2B458756ULL,
			0XE17676C57697B3ECULL, 0XE68282328264B019ULL, 0X28D6D67FD6FEA9B1ULL, 0XC31B1B6C1BD87736ULL,
			0X74B5B5EEB5C15B77ULL, 0XBEAFAF86AF112943ULL, 0X1D6A6AB56A77DFD4ULL, 0XEA50505D50BA0DA0ULL,
			0X5745450945124C8AULL, 0X38F3F3EBF3CB18FBULL, 0XAD3030C0309DF060ULL, 0XC4EFEF9BEF2B74C3ULL,
			0XDA3F3FFC3FE5C37EULL, 0XC755554955921CAAULL, 0XDBA2A2B2A2791059ULL, 0XE9EAEA8FEA0365C9ULL,
			0X6A656589650FECCAULL, 0X03BABAD2BAB96869ULL, 0X4A2F2FBC2F65935EULL, 0X8EC0C027C04EE79DULL,
			0X60DEDE5FDEBE81A1ULL, 0XFC1C1C701CE06C38ULL, 0X46FDFDD3FDBB2EE7ULL, 0X1F4D4D294D52649AULL,
			0X7692927292E4E039ULL, 0XFA7575C9758FBCEAULL, 0X3606061806301E0CULL, 0XAE8A8A128A249809ULL,
			0X4BB2B2F2B2F94079ULL, 0X85E6E6BFE66359D1ULL, 0X7E0E0E380E70361CULL, 0XE71F1F7C1FF8633EULL,
			0X556262956237F7C4ULL, 0X3AD4D477D4EEA3B5ULL, 0X81A8A89AA829324DULL, 0X5296966296C4F431ULL,
			0X62F9F9C3F99B3AEFULL, 0XA3C5C533C566F697ULL, 0X102525942535B14AULL, 0XAB59597959F220B2ULL,
			0XD084842A8454AE15ULL, 0XC57272D572B7A7E4ULL, 0XEC3939E439D5DD72ULL, 0X164C4C2D4C5A6198ULL,
			0X945E5E655ECA3BBCULL, 0X9F7878FD78E785F0ULL, 0XE53838E038DDD870ULL, 0X988C8C0A8C148605ULL,
			0X17D1D163D1C6B2BFULL, 0XE4A5A5AEA5410B57ULL, 0XA1E2E2AFE2434DD9ULL, 0X4E616199612FF8C2ULL,
			0X42B3B3F6B3F1457BULL, 0X342121842115A542ULL, 0X089C9C4A9C94D625ULL, 0XEE1E1E781EF0663CULL,
			0X6143431143225286ULL, 0XB1C7C73BC776FC93ULL, 0X4FFCFCD7FCB32BE5ULL, 0X2404041004201408ULL,
			0XE351515951B208A2ULL, 0X2599995E99BCC72FULL, 0X226D6DA96D4FC4DAULL, 0X650D0D340D68391AULL,
			0X79FAFACFFA8335E9ULL, 0X69DFDF5BDFB684A3ULL, 0XA97E7EE57ED79BFCULL, 0X19242490243DB448ULL,
			0XFE3B3BEC3BC5D776ULL, 0X9AABAB96AB313D4BULL, 0XF0CECE1FCE3ED181ULL, 0X9911114411885522ULL,
			0X838F8F068F0C8903ULL, 0X044E4E254E4A6B9CULL, 0X66B7B7E6B7D15173ULL, 0XE0EBEB8BEB0B60CBULL,
			0XC13C3CF03CFDCC78ULL, 0XFD81813E817CBF1FULL, 0X4094946A94D4FE35ULL, 0X1CF7F7FBF7EB0CF3ULL,
			0X18B9B9DEB9A1676FULL, 0X8B13134C13985F26ULL, 0X512C2CB02C7D9C58ULL, 0X05D3D36BD3D6B8BBULL,
			0X8CE7E7BBE76B5CD3ULL, 0X396E6EA56E57CBDCULL, 0XAAC4C437C46EF395ULL, 0X1B03030C03180F06ULL,
			0XDC565645568A13ACULL, 0X5E44440D441A4988ULL, 0XA07F7FE17FDF9EFEULL, 0X88A9A99EA921374FULL,
			0X672A2AA82A4D8254ULL, 0X0ABBBBD6BBB16D6BULL, 0X87C1C123C146E29FULL, 0XF153535153A202A6ULL,
			0X72DCDC57DCAE8BA5ULL, 0X530B0B2C0B582716ULL, 0X019D9D4E9D9CD327ULL, 0X2B6C6CAD6C47C1D8ULL,
			0XA43131C43195F562ULL, 0XF37474CD7487B9E8ULL, 0X15F6F6FFF6E309F1ULL, 0X4C464605460A438CULL,
			0XA5ACAC8AAC092645ULL, 0XB589891E893C970FULL, 0XB414145014A04428ULL, 0XBAE1E1A3E15B42DFULL,
			0XA616165816B04E2CULL, 0XF73A3AE83ACDD274ULL, 0X066969B9696FD0D2ULL, 0X4109092409482D12ULL,
			0XD77070DD70A7ADE0ULL, 0X6FB6B6E2B6D95471ULL, 0X1ED0D067D0CEB7BDULL, 0XD6EDED93ED3B7EC7ULL,
			0XE2CCCC17CC2EDB85ULL, 0X68424215422A5784ULL, 0X2C98985A98B4C22DULL, 0XEDA4A4AAA4490E55ULL,
			0X752828A0285D8850ULL, 0X865C5C6D5CDA31B8ULL, 0X6BF8F8C7F8933FEDULL, 0XC28686228644A411ULL,
		};

		const UInt64 C2[256] = {
			0X30D818186018C078ULL, 0X462623238C2305AFULL, 0X91B8C6C63FC67EF9ULL, 0XCDFBE8E887E8136FULL,
			0X13CB878726874CA1ULL, 0X6D11B8B8DAB8A962ULL, 0X0209010104010805ULL, 0X9E0D4F4F214F426EULL,
			0X6C9B3636D836ADEEULL, 0X51FFA6A6A2A65904ULL, 0XB90CD2D26FD2DEBDULL, 0XF70EF5F5F3F5FB06ULL,
			0XF2967979F979EF80ULL, 0XDE306F6FA16F5FCEULL, 0X3F6D91917E91FCEFULL, 0XA4F852525552AA07ULL,
			0XC04760609D6027FDULL, 0X6535BCBCCABC8976ULL, 0X2B379B9B569BACCDULL, 0X018A8E8E028E048CULL,
			0X5BD2A3A3B6A37115ULL, 0X186C0C0C300C603CULL, 0XF6847B7BF17BFF8AULL, 0X6A803535D435B5E1ULL,
			0X3AF51D1D741DE869ULL, 0XDDB3E0E0A7E05347ULL, 0XB321D7D77BD7F6ACULL, 0X999CC2C22FC25EEDULL,
			0X5C432E2EB82E6D96ULL, 0X96294B4B314B627AULL, 0XE15DFEFEDFFEA321ULL, 0XAED5575741578216ULL,
			0X2ABD15155415A841ULL, 0XEEE87777C1779FB6ULL, 0X6E923737DC37A5EBULL, 0XD79EE5E5B3E57B56ULL,
			0X23139F9F469F8CD9ULL, 0XFD23F0F0E7F0D317ULL, 0X94204A4A354A6A7FULL, 0XA944DADA4FDA9E95ULL,
			0XB0A258587D58FA25ULL, 0X8FCFC9C903C906CAULL, 0X527C2929A429558DULL, 0X145A0A0A280A5022ULL,
			0X7F50B1B1FEB1E14FULL, 0X5DC9A0A0BAA0691AULL, 0XD6146B6BB16B7FDAULL, 0X17D985852E855CABULL,
			0X673CBDBDCEBD8173ULL, 0XBA8F5D5D695DD234ULL, 0X2090101040108050ULL, 0XF507F4F4F7F4F303ULL,
			0X8BDDCBCB0BCB16C0ULL, 0X7CD33E3EF83EEDC6ULL, 0X0A2D050514052811ULL, 0XCE78676781671FE6ULL,
			0XD597E4E4B7E47353ULL, 0X4E0227279C2725BBULL, 0X8273414119413258ULL, 0X0BA78B8B168B2C9DULL,
			0X53F6A7A7A6A75101ULL, 0XFAB27D7DE97DCF94ULL, 0X374995956E95DCFBULL, 0XAD56D8D847D88E9FULL,
			0XEB70FBFBCBFB8B30ULL, 0XC1CDEEEE9FEE2371ULL, 0XF8BB7C7CED7CC791ULL, 0XCC716666856617E3ULL,
			0XA77BDDDD53DDA68EULL, 0X2EAF17175C17B84BULL, 0X8E45474701470246ULL, 0X211A9E9E429E84DCULL,
			0X89D4CACA0FCA1EC5ULL, 0X5A582D2DB42D7599ULL, 0X632EBFBFC6BF9179ULL, 0X0E3F07071C07381BULL,
			0X47ACADAD8EAD0123ULL, 0XB4B05A5A755AEA2FULL, 0X1BEF838336836CB5ULL, 0X66B63333CC3385FFULL,
			0XC65C636391633FF2ULL, 0X041202020802100AULL, 0X4993AAAA92AA3938ULL, 0XE2DE7171D971AFA8ULL,
			0X8DC6C8C807C80ECFULL, 0X32D119196419C87DULL, 0X923B494939497270ULL, 0XAF5FD9D943D9869AULL,
			0XF931F2F2EFF2C31DULL, 0XDBA8E3E3ABE34B48ULL, 0XB6B95B5B715BE22AULL, 0X0DBC88881A883492ULL,
			0X293E9A9A529AA4C8ULL, 0X4C0B262698262DBEULL, 0X64BF3232C8328DFAULL, 0X7D59B0B0FAB0E94AULL,
			0XCFF2E9E983E91B6AULL, 0X1E770F0F3C0F7833ULL, 0XB733D5D573D5E6A6ULL, 0X1DF480803A8074BAULL,
			0X6127BEBEC2BE997CULL, 0X87EBCDCD13CD26DEULL, 0X68893434D034BDE4ULL, 0X903248483D487A75ULL,
			0XE354FFFFDBFFAB24ULL, 0XF48D7A7AF57AF78FULL, 0X3D6490907A90F4EAULL, 0XBE9D5F5F615FC23EULL,
			0X403D202080201DA0ULL, 0XD00F6868BD6867D5ULL, 0X34CA1A1A681AD072ULL, 0X41B7AEAE82AE192CULL,
			0X757DB4B4EAB4C95EULL, 0XA8CE54544D549A19ULL, 0X3B7F93937693ECE5ULL, 0X442F222288220DAAULL,
			0XC86364648D6407E9ULL, 0XFF2AF1F1E3F1DB12ULL, 0XE6CC7373D173BFA2ULL, 0X248212124812905AULL,
			0X807A40401D403A5DULL, 0X1048080820084028ULL, 0X9B95C3C32BC356E8ULL, 0XC5DFECEC97EC337BULL,
			0XAB4DDBDB4BDB9690ULL, 0X5FC0A1A1BEA1611FULL, 0X07918D8D0E8D1C83ULL, 0X7AC83D3DF43DF5C9ULL,
			0X335B97976697CCF1ULL, 0X0000000000000000ULL, 0X83F9CFCF1BCF36D4ULL, 0X566E2B2BAC2B4587ULL,
			0XECE17676C57697B3ULL, 0X19E68282328264B0ULL, 0XB128D6D67FD6FEA9ULL, 0X36C31B1B6C1BD877ULL,
			0X7774B5B5EEB5C15BULL, 0X43BEAFAF86AF1129ULL, 0XD41D6A6AB56A77DFULL, 0XA0EA50505D50BA0DULL,
			0X8A5745450945124CULL, 0XFB38F3F3EBF3CB18ULL, 0X60AD3030C0309DF0ULL, 0XC3C4EFEF9BEF2B74ULL,
			0X7EDA3F3FFC3FE5C3ULL, 0XAAC755554955921CULL, 0X59DBA2A2B2A27910ULL, 0XC9E9EAEA8FEA0365ULL,
			0XCA6A656589650FECULL, 0X6903BABAD2BAB968ULL, 0X5E4A2F2FBC2F6593ULL, 0X9D8EC0C027C04EE7ULL,
			0XA160DEDE5FDEBE81ULL, 0X38FC1C1C701CE06CULL, 0XE746FDFDD3FDBB2EULL, 0X9A1F4D4D294D5264ULL,
			0X397692927292E4E0ULL, 0XEAFA7575C9758FBCULL, 0X0C3606061806301EULL, 0X09AE8A8A128A2498ULL,
			0X794BB2B2F2B2F940ULL, 0XD185E6E6BFE66359ULL, 0X1C7E0E0E380E7036ULL, 0X3EE71F1F7C1FF863ULL,
			0XC4556262956237F7ULL, 0XB53AD4D477D4EEA3ULL, 0X4D81A8A89AA82932ULL, 0X315296966296C4F4ULL,
			0XEF62F9F9C3F99B3AULL, 0X97A3C5C533C566F6ULL, 0X4A102525942535B1ULL, 0XB2AB59597959F220ULL,
			0X15D084842A8454AEULL, 0XE4C57272D572B7A7ULL, 0X72EC3939E439D5DDULL, 0X98164C4C2D4C5A61ULL,
			0XBC945E5E655ECA3BULL, 0XF09F7878FD78E785ULL, 0X70E53838E038DDD8ULL, 0X05988C8C0A8C1486ULL,
			0XBF17D1D163D1C6B2ULL, 0X57E4A5A5AEA5410BULL, 0XD9A1E2E2AFE2434DULL, 0XC24E616199612FF8ULL,
			0X7B42B3B3F6B3F145ULL, 0X42342121842115A5ULL, 0X25089C9C4A9C94D6ULL, 0X3CEE1E1E781EF066ULL,
			0X8661434311432252ULL, 0X93B1C7C73BC776FCULL, 0XE54FFCFCD7FCB32BULL, 0X0824040410042014ULL,
			0XA2E351515951B208ULL, 0X2F2599995E99BCC7ULL, 0XDA226D6DA96D4FC4ULL, 0X1A650D0D340D6839ULL,
			0XE979FAFACFFA8335ULL, 0XA369DFDF5BDFB684ULL, 0XFCA97E7EE57ED79BULL, 0X4819242490243DB4ULL,
			0X76FE3B3BEC3BC5D7ULL, 0X4B9AABAB96AB313DULL, 0X81F0CECE1FCE3ED1ULL, 0X2299111144118855ULL,
			0X03838F8F068F0C89ULL, 0X9C044E4E254E4A6BULL, 0X7366B7B7E6B7D151ULL, 0XCBE0EBEB8BEB0B60ULL,
			0X78C13C3CF03CFDCCULL, 0X1FFD81813E817CBFULL, 0X354094946A94D4FEULL, 0XF31CF7F7FBF7EB0CULL,
			0X6F18B9B9DEB9A167ULL, 0X268B13134C13985FULL, 0X58512C2CB02C7D9CULL, 0XBB05D3D36BD3D6B8ULL,
			0XD38CE7E7BBE76B5CULL, 0XDC396E6EA56E57CBULL, 0X95AAC4C437C46EF3ULL, 0X061B03030C03180FULL,
			0XACDC565645568A13ULL, 0X885E44440D441A49ULL, 0XFEA07F7FE17FDF9EULL, 0X4F88A9A99EA92137ULL,
			0X54672A2AA82A4D82ULL, 0X6B0ABBBBD6BBB16DULL, 0X9F87C1C123C146E2ULL, 0XA6F153535153A202ULL,
			0XA572DCDC57DCAE8BULL, 0X16530B0B2C0B5827ULL, 0X27019D9D4E9D9CD3ULL, 0XD82B6C6CAD6C47C1ULL,
			0X62A43131C43195F5ULL, 0XE8F37474CD7487B9ULL, 0XF115F6F6FFF6E309ULL, 0X8C4C464605460A43ULL,
			0X45A5ACAC8AAC0926ULL, 0X0FB589891E893C97ULL, 0X28B414145014A044ULL, 0XDFBAE1E1A3E15B42ULL,
			0X2CA616165816B04EULL, 0X74F73A3AE83ACDD2ULL, 0XD2066969B9696FD0ULL, 0X124109092409482DULL,
			0XE0D77070DD70A7ADULL, 0X716FB6B6E2B6D954ULL, 0XBD1ED0D067D0CEB7ULL, 0XC7D6EDED93ED3B7EULL,
			0X85E2CCCC17CC2EDBULL, 0X8468424215422A57ULL, 0X2D2C98985A98B4C2ULL, 0X55EDA4A4AAA4490EULL,
			0X50752828A0285D88ULL, 0XB8865C5C6D5CDA31ULL, 0XED6BF8F8C7F8933FULL, 0X11C28686228644A4ULL,
		};

		const UInt64 C3[256] = {
			0X7830D818186018C0ULL, 0XAF462623238C2305ULL, 0XF991B8C6C63FC67EULL, 0X6FCDFBE8E887E813ULL,
			0XA113CB878726874CULL, 0X626D11B8B8DAB8A9ULL, 0X0502090101040108ULL, 0X6E9E0D4F4F214F42ULL,
			0XEE6C9B3636D836ADULL, 0X0451FFA6A6A2A659ULL, 0XBDB90CD2D26FD2DEULL, 0X06F70EF5F5F3F5FBULL,
			0X80F2967979F979EFULL, 0XCEDE306F6FA16F5FULL, 0XEF3F6D91917E91FCULL, 0X07A4F852525552AAULL,
			0XFDC04760609D6027ULL, 0X766535BCBCCABC89ULL, 0XCD2B379B9B569BACULL, 0X8C018A8E8E028E04ULL,
			0X155BD2A3A3B6A371ULL, 0X3C186C0C0C300C60ULL, 0X8AF6847B7BF17BFFULL, 0XE16A803535D435B5ULL,
			0X693AF51D1D741DE8ULL, 0X47DDB3E0E0A7E053ULL, 0XACB321D7D77BD7F6ULL, 0XED999CC2C22FC25EULL,
			0X965C432E2EB82E6DULL, 0X7A96294B4B314B62ULL, 0X21E15DFEFEDFFEA3ULL, 0X16AED55757415782ULL,
			0X412ABD15155415A8ULL, 0XB6EEE87777C1779FULL, 0XEB6E923737DC37A5ULL, 0X56D79EE5E5B3E57BULL,
			0XD923139F9F469F8CULL, 0X17FD23F0F0E7F0D3ULL, 0X7F94204A4A354A6AULL, 0X95A944DADA4FDA9EULL,
			0X25B0A258587D58FAULL, 0XCA8FCFC9C903C906ULL, 0X8D527C2929A42955ULL, 0X22145A0A0A280A50ULL,
			0X4F7F50B1B1FEB1E1ULL, 0X1A5DC9A0A0BAA069ULL, 0XDAD6146B6BB16B7FULL, 0XAB17D985852E855CULL,
			0X73673CBDBDCEBD81ULL, 0X34BA8F5D5D695DD2ULL, 0X5020901010401080ULL, 0X03F507F4F4F7F4F3ULL,
			0XC08BDDCBCB0BCB16ULL, 0XC67CD33E3EF83EEDULL, 0X110A2D0505140528ULL, 0XE6CE78676781671FULL,
			0X53D597E4E4B7E473ULL, 0XBB4E0227279C2725ULL, 0X5882734141194132ULL, 0X9D0BA78B8B168B2CULL,
			0X0153F6A7A7A6A751ULL, 0X94FAB27D7DE97DCFULL, 0XFB374995956E95DCULL, 0X9FAD56D8D847D88EULL,
			0X30EB70FBFBCBFB8BULL, 0X71C1CDEEEE9FEE23ULL, 0X91F8BB7C7CED7CC7ULL, 0XE3CC716666856617ULL,
			0X8EA77BDDDD53DDA6ULL, 0X4B2EAF17175C17B8ULL, 0X468E454747014702ULL, 0XDC211A9E9E429E84ULL,
			0XC589D4CACA0FCA1EULL, 0X995A582D2DB42D75ULL, 0X79632EBFBFC6BF91ULL, 0X1B0E3F07071C0738ULL,
			0X2347ACADAD8EAD01ULL, 0X2FB4B05A5A755AEAULL, 0XB51BEF838336836CULL, 0XFF66B63333CC3385ULL,
			0XF2C65C636391633FULL, 0X0A04120202080210ULL, 0X384993AAAA92AA39ULL, 0XA8E2DE7171D971AFULL,
			0XCF8DC6C8C807C80EULL, 0X7D32D119196419C8ULL, 0X70923B4949394972ULL, 0X9AAF5FD9D943D986ULL,
			0X1DF931F2F2EFF2C3ULL, 0X48DBA8E3E3ABE34BULL, 0X2AB6B95B5B715BE2ULL, 0X920DBC88881A8834ULL,
			0XC8293E9A9A529AA4ULL, 0XBE4C0B262698262DULL, 0XFA64BF3232C8328DULL, 0X4A7D59B0B0FAB0E9ULL,
			0X6ACFF2E9E983E91BULL, 0X331E770F0F3C0F78ULL, 0XA6B733D5D573D5E6ULL, 0XBA1DF480803A8074ULL,
			0X7C6127BEBEC2BE99ULL, 0XDE87EBCDCD13CD26ULL, 0XE468893434D034BDULL, 0X75903248483D487AULL,
			0X24E354FFFFDBFFABULL, 0X8FF48D7A7AF57AF7ULL, 0XEA3D6490907A90F4ULL, 0X3EBE9D5F5F615FC2ULL,
			0XA0403D202080201DULL, 0XD5D00F6868BD6867ULL, 0X7234CA1A1A681AD0ULL, 0X2C41B7AEAE82AE19ULL,
			0X5E757DB4B4EAB4C9ULL, 0X19A8CE54544D549AULL, 0XE53B7F93937693ECULL, 0XAA442F222288220DULL,
			0XE9C86364648D6407ULL, 0X12FF2AF1F1E3F1DBULL, 0XA2E6CC7373D173BFULL, 0X5A24821212481290ULL,
			0X5D807A40401D403AULL, 0X2810480808200840ULL, 0XE89B95C3C32BC356ULL, 0X7BC5DFECEC97EC33ULL,
			0X90AB4DDBDB4BDB96ULL, 0X1F5FC0A1A1BEA161ULL, 0X8307918D8D0E8D1CULL, 0XC97AC83D3DF43DF5ULL,
			0XF1335B97976697CCULL, 0X0000000000000000ULL, 0XD483F9CFCF1BCF36ULL, 0X87566E2B2BAC2B45ULL,
			0XB3ECE17676C57697ULL, 0XB019E68282328264ULL, 0XA9B128D6D67FD6FEULL, 0X7736C31B1B6C1BD8ULL,
			0X5B7774B5B5EEB5C1ULL, 0X2943BEAFAF86AF11ULL, 0XDFD41D6A6AB56A77ULL, 0X0DA0EA50505D50BAULL,
			0X4C8A574545094512ULL, 0X18FB38F3F3EBF3CBULL, 0XF060AD3030C0309DULL, 0X74C3C4EFEF9BEF2BULL,
			0XC37EDA3F3FFC3FE5ULL, 0X1CAAC75555495592ULL, 0X1059DBA2A2B2A279ULL, 0X65C9E9EAEA8FEA03ULL,
			0XECCA6A656589650FULL, 0X686903BABAD2BAB9ULL, 0X935E4A2F2FBC2F65ULL, 0XE79D8EC0C027C04EULL,
			0X81A160DEDE5FDEBEULL, 0X6C38FC1C1C701CE0ULL, 0X2EE746FDFDD3FDBBULL, 0X649A1F4D4D294D52ULL,
			0XE0397692927292E4ULL, 0XBCEAFA7575C9758FULL, 0X1E0C360606180630ULL, 0X9809AE8A8A128A24ULL,
			0X40794BB2B2F2B2F9ULL, 0X59D185E6E6BFE663ULL, 0X361C7E0E0E380E70ULL, 0X633EE71F1F7C1FF8ULL,
			0XF7C4556262956237ULL, 0XA3B53AD4D477D4EEULL, 0X324D81A8A89AA829ULL, 0XF4315296966296C4ULL,
			0X3AEF62F9F9C3F99BULL, 0XF697A3C5C533C566ULL, 0XB14A102525942535ULL, 0X20B2AB59597959F2ULL,
			0XAE15D084842A8454ULL, 0XA7E4C57272D572B7ULL, 0XDD72EC3939E439D5ULL, 0X6198164C4C2D4C5AULL,
			0X3BBC945E5E655ECAULL, 0X85F09F7878FD78E7ULL, 0XD870E53838E038DDULL, 0X8605988C8C0A8C14ULL,
			0XB2BF17D1D163D1C6ULL, 0X0B57E4A5A5AEA541ULL, 0X4DD9A1E2E2AFE243ULL, 0XF8C24E616199612FULL,
			0X457B42B3B3F6B3F1ULL, 0XA542342121842115ULL, 0XD625089C9C4A9C94ULL, 0X663CEE1E1E781EF0ULL,
			0X5286614343114322ULL, 0XFC93B1C7C73BC776ULL, 0X2BE54FFCFCD7FCB3ULL, 0X1408240404100420ULL,
			0X08A2E351515951B2ULL, 0XC72F2599995E99BCULL, 0XC4DA226D6DA96D4FULL, 0X391A650D0D340D68ULL,
			0X35E979FAFACFFA83ULL, 0X84A369DFDF5BDFB6ULL, 0X9BFCA97E7EE57ED7ULL, 0XB44819242490243DULL,
			0XD776FE3B3BEC3BC5ULL, 0X3D4B9AABAB96AB31ULL, 0XD181F0CECE1FCE3EULL, 0X5522991111441188ULL,
			0X8903838F8F068F0CULL, 0X6B9C044E4E254E4AULL, 0X517366B7B7E6B7D1ULL, 0X60CBE0EBEB8BEB0BULL,
			0XCC78C13C3CF03CFDULL, 0XBF1FFD81813E817CULL, 0XFE354094946A94D4ULL, 0X0CF31CF7F7FBF7EBULL,
			0X676F18B9B9DEB9A1ULL, 0X5F268B13134C1398ULL, 0X9C58512C2CB02C7DULL, 0XB8BB05D3D36BD3D6ULL,
			0X5CD38CE7E7BBE76BULL, 0XCBDC396E6EA56E57ULL, 0XF395AAC4C437C46EULL, 0X0F061B03030C0318ULL,
			0X13ACDC565645568AULL, 0X49885E44440D441AULL, 0X9EFEA07F7FE17FDFULL, 0X374F88A9A99EA921ULL,
			0X8254672A2AA82A4DULL, 0X6D6B0ABBBBD6BBB1ULL, 0XE29F87C1C123C146ULL, 0X02A6F153535153A2ULL,
			0X8BA572DCDC57DCAEULL, 0X2716530B0B2C0B58ULL, 0XD327019D9D4E9D9CULL, 0XC1D82B6C6CAD6C47ULL,
			0XF562A43131C43195ULL, 0XB9E8F37474CD7487ULL, 0X09F115F6F6FFF6E3ULL, 0X438C4C464605460AULL,
			0X2645A5ACAC8AAC09ULL, 0X970FB589891E893CULL, 0X4428B414145014A0ULL, 0X42DFBAE1E1A3E15BULL,
			0X4E2CA616165816B0ULL, 0XD274F73A3AE83ACDULL, 0XD0D2066969B9696FULL, 0X2D12410909240948ULL,
			0XADE0D77070DD70A7ULL, 0X54716FB6B6E2B6D9ULL, 0XB7BD1ED0D067D0CEULL, 0X7EC7D6EDED93ED3BULL,
			0XDB85E2CCCC17CC2EULL, 0X578468424215422AULL, 0XC22D2C98985A98B4ULL, 0X0E55EDA4A4AAA449ULL,
			0X8850752828A0285DULL, 0X31B8865C5C6D5CDAULL, 0X3FED6BF8F8C7F893ULL, 0XA411C28686228644ULL,
		};

		const UInt64 C4[256] = {
			0XC07830D818186018ULL, 0X05AF462623238C23ULL, 0X7EF991B8C6C63FC6ULL, 0X136FCDFBE8E887E8ULL,
			0X4CA113CB87872687ULL, 0XA9626D11B8B8DAB8ULL, 0X0805020901010401ULL, 0X426E9E0D4F4F214FULL,
			0XADEE6C9B3636D836ULL, 0X590451FFA6A6A2A6ULL, 0XDEBDB90CD2D26FD2ULL, 0XFB06F70EF5F5F3F5ULL,
			0XEF80F2967979F979ULL, 0X5FCEDE306F6FA16FULL, 0XFCEF3F6D91917E91ULL, 0XAA07A4F852525552ULL,
			0X27FDC04760609D60ULL, 0X89766535BCBCCABCULL, 0XACCD2B379B9B569BULL, 0X048C018A8E8E028EULL,
			0X71155BD2A3A3B6A3ULL, 0X603C186C0C0C300CULL, 0XFF8AF6847B7BF17BULL, 0XB5E16A803535D435ULL,
			0XE8693AF51D1D741DULL, 0X5347DDB3E0E0A7E0ULL, 0XF6ACB321D7D77BD7ULL, 0X5EED999CC2C22FC2ULL,
			0X6D965C432E2EB82EULL, 0X627A96294B4B314BULL, 0XA321E15DFEFEDFFEULL, 0X8216AED557574157ULL,
			0XA8412ABD15155415ULL, 0X9FB6EEE87777C177ULL, 0XA5EB6E923737DC37ULL, 0X7B56D79EE5E5B3E5ULL,
			0X8CD923139F9F469FULL, 0XD317FD23F0F0E7F0ULL, 0X6A7F94204A4A354AULL, 0X9E95A944DADA4FDAULL,
			0XFA25B0A258587D58ULL, 0X06CA8FCFC9C903C9ULL, 0X558D527C2929A429ULL, 0X5022145A0A0A280AULL,
			0XE14F7F50B1B1FEB1ULL, 0X691A5DC9A0A0BAA0ULL, 0X7FDAD6146B6BB16BULL, 0X5CAB17D985852E85ULL,
			0X8173673CBDBDCEBDULL, 0XD234BA8F5D5D695DULL, 0X8050209010104010ULL, 0XF303F507F4F4F7F4ULL,
			0X16C08BDDCBCB0BCBULL, 0XEDC67CD33E3EF83EULL, 0X28110A2D05051405ULL, 0X1FE6CE7867678167ULL,
			0X7353D597E4E4B7E4ULL, 0X25BB4E0227279C27ULL, 0X3258827341411941ULL, 0X2C9D0BA78B8B168BULL,
			0X510153F6A7A7A6A7ULL, 0XCF94FAB27D7DE97DULL, 0XDCFB374995956E95ULL, 0X8E9FAD56D8D847D8ULL,
			0X8B30EB70FBFBCBFBULL, 0X2371C1CDEEEE9FEEULL, 0XC791F8BB7C7CED7CULL, 0X17E3CC7166668566ULL,
			0XA68EA77BDDDD53DDULL, 0XB84B2EAF17175C17ULL, 0X02468E4547470147ULL, 0X84DC211A9E9E429EULL,
			0X1EC589D4CACA0FCAULL, 0X75995A582D2DB42DULL, 0X9179632EBFBFC6BFULL, 0X381B0E3F07071C07ULL,
			0X012347ACADAD8EADULL, 0XEA2FB4B05A5A755AULL, 0X6CB51BEF83833683ULL, 0X85FF66B63333CC33ULL,
			0X3FF2C65C63639163ULL, 0X100A041202020802ULL, 0X39384993AAAA92AAULL, 0XAFA8E2DE7171D971ULL,
			0X0ECF8DC6C8C807C8ULL, 0XC87D32D119196419ULL, 0X7270923B49493949ULL, 0X869AAF5FD9D943D9ULL,
			0XC31DF931F2F2EFF2ULL, 0X4B48DBA8E3E3ABE3ULL, 0XE22AB6B95B5B715BULL, 0X34920DBC88881A88ULL,
			0XA4C8293E9A9A529AULL, 0X2DBE4C0B26269826ULL, 0X8DFA64BF3232C832ULL, 0XE94A7D59B0B0FAB0ULL,
			0X1B6ACFF2E9E983E9ULL, 0X78331E770F0F3C0FULL, 0XE6A6B733D5D573D5ULL, 0X74BA1DF480803A80ULL,
			0X997C6127BEBEC2BEULL, 0X26DE87EBCDCD13CDULL, 0XBDE468893434D034ULL, 0X7A75903248483D48ULL,
			0XAB24E354FFFFDBFFULL, 0XF78FF48D7A7AF57AULL, 0XF4EA3D6490907A90ULL, 0XC23EBE9D5F5F615FULL,
			0X1DA0403D20208020ULL, 0X67D5D00F6868BD68ULL, 0XD07234CA1A1A681AULL, 0X192C41B7AEAE82AEULL,
			0XC95E757DB4B4EAB4ULL, 0X9A19A8CE54544D54ULL, 0XECE53B7F93937693ULL, 0X0DAA442F22228822ULL,
			0X07E9C86364648D64ULL, 0XDB12FF2AF1F1E3F1ULL, 0XBFA2E6CC7373D173ULL, 0X905A248212124812ULL,
			0X3A5D807A40401D40ULL, 0X4028104808082008ULL, 0X56E89B95C3C32BC3ULL, 0X337BC5DFECEC97ECULL,
			0X9690AB4DDBDB4BDBULL, 0X611F5FC0A1A1BEA1ULL, 0X1C8307918D8D0E8DULL, 0XF5C97AC83D3DF43DULL,
			0XCCF1335B97976697ULL, 0X0000000000000000ULL, 0X36D483F9CFCF1BCFULL, 0X4587566E2B2BAC2BULL,
			0X97B3ECE17676C576ULL, 0X64B019E682823282ULL, 0XFEA9B128D6D67FD6ULL, 0XD87736C31B1B6C1BULL,
			0XC15B7774B5B5EEB5ULL, 0X112943BEAFAF86AFULL, 0X77DFD41D6A6AB56AULL, 0XBA0DA0EA50505D50ULL,
			0X124C8A5745450945ULL, 0XCB18FB38F3F3EBF3ULL, 0X9DF060AD3030C030ULL, 0X2B74C3C4EFEF9BEFULL,
			0XE5C37EDA3F3FFC3FULL, 0X921CAAC755554955ULL, 0X791059DBA2A2B2A2ULL, 0X0365C9E9EAEA8FEAULL,
			0X0FECCA6A65658965ULL, 0XB9686903BABAD2BAULL, 0X65935E4A2F2FBC2FULL, 0X4EE79D8EC0C027C0ULL,
			0XBE81A160DEDE5FDEULL, 0XE06C38FC1C1C701CULL, 0XBB2EE746FDFDD3FDULL, 0X52649A1F4D4D294DULL,
			0XE4E0397692927292ULL, 0X8FBCEAFA7575C975ULL, 0X301E0C3606061806ULL, 0X249809AE8A8A128AULL,
			0XF940794BB2B2F2B2ULL, 0X6359D185E6E6BFE6ULL, 0X70361C7E0E0E380EULL, 0XF8633EE71F1F7C1FULL,
			0X37F7C45562629562ULL, 0XEEA3B53AD4D477D4ULL, 0X29324D81A8A89AA8ULL, 0XC4F4315296966296ULL,
			0X9B3AEF62F9F9C3F9ULL, 0X66F697A3C5C533C5ULL, 0X35B14A1025259425ULL, 0XF220B2AB59597959ULL,
			0X54AE15D084842A84ULL, 0XB7A7E4C57272D572ULL, 0XD5DD72EC3939E439ULL, 0X5A6198164C4C2D4CULL,
			0XCA3BBC945E5E655EULL, 0XE785F09F7878FD78ULL, 0XDDD870E53838E038ULL, 0X148605988C8C0A8CULL,
			0XC6B2BF17D1D163D1ULL, 0X410B57E4A5A5AEA5ULL, 0X434DD9A1E2E2AFE2ULL, 0X2FF8C24E61619961ULL,
			0XF1457B42B3B3F6B3ULL, 0X15A5423421218421ULL, 0X94D625089C9C4A9CULL, 0XF0663CEE1E1E781EULL,
			0X2252866143431143ULL, 0X76FC93B1C7C73BC7ULL, 0XB32BE54FFCFCD7FCULL, 0X2014082404041004ULL,
			0XB208A2E351515951ULL, 0XBCC72F2599995E99ULL, 0X4FC4DA226D6DA96DULL, 0X68391A650D0D340DULL,
			0X8335E979FAFACFFAULL, 0XB684A369DFDF5BDFULL, 0XD79BFCA97E7EE57EULL, 0X3DB4481924249024ULL,
			0XC5D776FE3B3BEC3BULL, 0X313D4B9AABAB96ABULL, 0X3ED181F0CECE1FCEULL, 0X8855229911114411ULL,
			0X0C8903838F8F068FULL, 0X4A6B9C044E4E254EULL, 0XD1517366B7B7E6B7ULL, 0X0B60CBE0EBEB8BEBULL,
			0XFDCC78C13C3CF03CULL, 0X7CBF1FFD81813E81ULL, 0XD4FE354094946A94ULL, 0XEB0CF31CF7F7FBF7ULL,
			0XA1676F18B9B9DEB9ULL, 0X985F268B13134C13ULL, 0X7D9C58512C2CB02CULL, 0XD6B8BB05D3D36BD3ULL,
			0X6B5CD38CE7E7BBE7ULL, 0X57CBDC396E6EA56EULL, 0X6EF395AAC4C437C4ULL, 0X180F061B03030C03ULL,
			0X8A13ACDC56564556ULL, 0X1A49885E44440D44ULL, 0XDF9EFEA07F7FE17FULL, 0X21374F88A9A99EA9ULL,
			0X4D8254672A2AA82AULL, 0XB16D6B0ABBBBD6BBULL, 0X46E29F87C1C123C1ULL, 0XA202A6F153535153ULL,
			0XAE8BA572DCDC57DCULL, 0X582716530B0B2C0BULL, 0X9CD327019D9D4E9DULL, 0X47C1D82B6C6CAD6CULL,
			0X95F562A43131C431ULL, 0X87B9E8F37474CD74ULL, 0XE309F115F6F6FFF6ULL, 0X0A438C4C46460546ULL,
			0X092645A5ACAC8AACULL, 0X3C970FB589891E89ULL, 0XA04428B414145014ULL, 0X5B42DFBAE1E1A3E1ULL,
			0XB04E2CA616165816ULL, 0XCDD274F73A3AE83AULL, 0X6FD0D2066969B969ULL, 0X482D124109092409ULL,
			0XA7ADE0D77070DD70ULL, 0XD954716FB6B6E2B6ULL, 0XCEB7BD1ED0D067D0ULL, 0X3B7EC7D6EDED93EDULL,
			0X2EDB85E2CCCC17CCULL, 0X2A57846842421542ULL, 0XB4C22D2C98985A98ULL, 0X490E55EDA4A4AAA4ULL,
			0X5D8850752828A028ULL, 0XDA31B8865C5C6D5CULL, 0X933FED6BF8F8C7F8ULL, 0X44A411C286862286ULL,
		};

		const UInt64 C5[256] = {
			0X18C07830D8181860ULL, 0X2305AF462623238CULL, 0XC67EF991B8C6C63FULL, 0XE8136FCDFBE8E887ULL,
			0X874CA113CB878726ULL, 0XB8A9626D11B8B8DAULL, 0X0108050209010104ULL, 0X4F426E9E0D4F4F21ULL,
			0X36ADEE6C9B3636D8ULL, 0XA6590451FFA6A6A2ULL, 0XD2DEBDB90CD2D26FULL, 0XF5FB06F70EF5F5F3ULL,
			0X79EF80F2967979F9ULL, 0X6F5FCEDE306F6FA1ULL, 0X91FCEF3F6D91917EULL, 0X52AA07A4F8525255ULL,
			0X6027FDC04760609DULL, 0XBC89766535BCBCCAULL, 0X9BACCD2B379B9B56ULL, 0X8E048C018A8E8E02ULL,
			0XA371155BD2A3A3B6ULL, 0X0C603C186C0C0C30ULL, 0X7BFF8AF6847B7BF1ULL, 0X35B5E16A803535D4ULL,
			0X1DE8693AF51D1D74ULL, 0XE05347DDB3E0E0A7ULL, 0XD7F6ACB321D7D77BULL, 0XC25EED999CC2C22FULL,
			0X2E6D965C432E2EB8ULL, 0X4B627A96294B4B31ULL, 0XFEA321E15DFEFEDFULL, 0X578216AED5575741ULL,
			0X15A8412ABD151554ULL, 0X779FB6EEE87777C1ULL, 0X37A5EB6E923737DCULL, 0XE57B56D79EE5E5B3ULL,
			0X9F8CD923139F9F46ULL, 0XF0D317FD23F0F0E7ULL, 0X4A6A7F94204A4A35ULL, 0XDA9E95A944DADA4FULL,
			0X58FA25B0A258587DULL, 0XC906CA8FCFC9C903ULL, 0X29558D527C2929A4ULL, 0X0A5022145A0A0A28ULL,
			0XB1E14F7F50B1B1FEULL, 0XA0691A5DC9A0A0BAULL, 0X6B7FDAD6146B6BB1ULL, 0X855CAB17D985852EULL,
			0XBD8173673CBDBDCEULL, 0X5DD234BA8F5D5D69ULL, 0X1080502090101040ULL, 0XF4F303F507F4F4F7ULL,
			0XCB16C08BDDCBCB0BULL, 0X3EEDC67CD33E3EF8ULL, 0X0528110A2D050514ULL, 0X671FE6CE78676781ULL,
			0XE47353D597E4E4B7ULL, 0X2725BB4E0227279CULL, 0X4132588273414119ULL, 0X8B2C9D0BA78B8B16ULL,
			0XA7510153F6A7A7A6ULL, 0X7DCF94FAB27D7DE9ULL, 0X95DCFB374995956EULL, 0XD88E9FAD56D8D847ULL,
			0XFB8B30EB70FBFBCBULL, 0XEE2371C1CDEEEE9FULL, 0X7CC791F8BB7C7CEDULL, 0X6617E3CC71666685ULL,
			0XDDA68EA77BDDDD53ULL, 0X17B84B2EAF17175CULL, 0X4702468E45474701ULL, 0X9E84DC211A9E9E42ULL,
			0XCA1EC589D4CACA0FULL, 0X2D75995A582D2DB4ULL, 0XBF9179632EBFBFC6ULL, 0X07381B0E3F07071CULL,
			0XAD012347ACADAD8EULL, 0X5AEA2FB4B05A5A75ULL, 0X836CB51BEF838336ULL, 0X3385FF66B63333CCULL,
			0X633FF2C65C636391ULL, 0X02100A0412020208ULL, 0XAA39384993AAAA92ULL, 0X71AFA8E2DE7171D9ULL,
			0XC80ECF8DC6C8C807ULL, 0X19C87D32D1191964ULL, 0X497270923B494939ULL, 0XD9869AAF5FD9D943ULL,
			0XF2C31DF931F2F2EFULL, 0XE34B48DBA8E3E3ABULL, 0X5BE22AB6B95B5B71ULL, 0X8834920DBC88881AULL,
			0X9AA4C8293E9A9A52ULL, 0X262DBE4C0B262698ULL, 0X328DFA64BF3232C8ULL, 0XB0E94A7D59B0B0FAULL,
			0XE91B6ACFF2E9E983ULL, 0X0F78331E770F0F3CULL, 0XD5E6A6B733D5D573ULL, 0X8074BA1DF480803AULL,
			0XBE997C6127BEBEC2ULL, 0XCD26DE87EBCDCD13ULL, 0X34BDE468893434D0ULL, 0X487A75903248483DULL,
			0XFFAB24E354FFFFDBULL, 0X7AF78FF48D7A7AF5ULL, 0X90F4EA3D6490907AULL, 0X5FC23EBE9D5F5F61ULL,
			0X201DA0403D202080ULL, 0X6867D5D00F6868BDULL, 0X1AD07234CA1A1A68ULL, 0XAE192C41B7AEAE82ULL,
			0XB4C95E757DB4B4EAULL, 0X549A19A8CE54544DULL, 0X93ECE53B7F939376ULL, 0X220DAA442F222288ULL,
			0X6407E9C86364648DULL, 0XF1DB12FF2AF1F1E3ULL, 0X73BFA2E6CC7373D1ULL, 0X12905A2482121248ULL,
			0X403A5D807A40401DULL, 0X0840281048080820ULL, 0XC356E89B95C3C32BULL, 0XEC337BC5DFECEC97ULL,
			0XDB9690AB4DDBDB4BULL, 0XA1611F5FC0A1A1BEULL, 0X8D1C8307918D8D0EULL, 0X3DF5C97AC83D3DF4ULL,
			0X97CCF1335B979766ULL, 0X0000000000000000ULL, 0XCF36D483F9CFCF1BULL, 0X2B4587566E2B2BACULL,
			0X7697B3ECE17676C5ULL, 0X8264B019E6828232ULL, 0XD6FEA9B128D6D67FULL, 0X1BD87736C31B1B6CULL,
			0XB5C15B7774B5B5EEULL, 0XAF112943BEAFAF86ULL, 0X6A77DFD41D6A6AB5ULL, 0X50BA0DA0EA50505DULL,
			0X45124C8A57454509ULL, 0XF3CB18FB38F3F3EBULL, 0X309DF060AD3030C0ULL, 0XEF2B74C3C4EFEF9BULL,
			0X3FE5C37EDA3F3FFCULL, 0X55921CAAC7555549ULL, 0XA2791059DBA2A2B2ULL, 0XEA0365C9E9EAEA8FULL,
			0X650FECCA6A656589ULL, 0XBAB9686903BABAD2ULL, 0X2F65935E4A2F2FBCULL, 0XC04EE79D8EC0C027ULL,
			0XDEBE81A160DEDE5FULL, 0X1CE06C38FC1C1C70ULL, 0XFDBB2EE746FDFDD3ULL, 0X4D52649A1F4D4D29ULL,
			0X92E4E03976929272ULL, 0X758FBCEAFA7575C9ULL, 0X06301E0C36060618ULL, 0X8A249809AE8A8A12ULL,
			0XB2F940794BB2B2F2ULL, 0XE66359D185E6E6BFULL, 0X0E70361C7E0E0E38ULL, 0X1FF8633EE71F1F7CULL,
			0X6237F7C455626295ULL, 0XD4EEA3B53AD4D477ULL, 0XA829324D81A8A89AULL, 0X96C4F43152969662ULL,
			0XF99B3AEF62F9F9C3ULL, 0XC566F697A3C5C533ULL, 0X2535B14A10252594ULL, 0X59F220B2AB595979ULL,
			0X8454AE15D084842AULL, 0X72B7A7E4C57272D5ULL, 0X39D5DD72EC3939E4ULL, 0X4C5A6198164C4C2DULL,
			0X5ECA3BBC945E5E65ULL, 0X78E785F09F7878FDULL, 0X38DDD870E53838E0ULL, 0X8C148605988C8C0AULL,
			0XD1C6B2BF17D1D163ULL, 0XA5410B57E4A5A5AEULL, 0XE2434DD9A1E2E2AFULL, 0X612FF8C24E616199ULL,
			0XB3F1457B42B3B3F6ULL, 0X2115A54234212184ULL, 0X9C94D625089C9C4AULL, 0X1EF0663CEE1E1E78ULL,
			0X4322528661434311ULL, 0XC776FC93B1C7C73BULL, 0XFCB32BE54FFCFCD7ULL, 0X0420140824040410ULL,
			0X51B208A2E3515159ULL, 0X99BCC72F2599995EULL, 0X6D4FC4DA226D6DA9ULL, 0X0D68391A650D0D34ULL,
			0XFA8335E979FAFACFULL, 0XDFB684A369DFDF5BULL, 0X7ED79BFCA97E7EE5ULL, 0X243DB44819242490ULL,
			0X3BC5D776FE3B3BECULL, 0XAB313D4B9AABAB96ULL, 0XCE3ED181F0CECE1FULL, 0X1188552299111144ULL,
			0X8F0C8903838F8F06ULL, 0X4E4A6B9C044E4E25ULL, 0XB7D1517366B7B7E6ULL, 0XEB0B60CBE0EBEB8BULL,
			0X3CFDCC78C13C3CF0ULL, 0X817CBF1FFD81813EULL, 0X94D4FE354094946AULL, 0XF7EB0CF31CF7F7FBULL,
			0XB9A1676F18B9B9DEULL, 0X13985F268B13134CULL, 0X2C7D9C58512C2CB0ULL, 0XD3D6B8BB05D3D36BULL,
			0XE76B5CD38CE7E7BBULL, 0X6E57CBDC396E6EA5ULL, 0XC46EF395AAC4C437ULL, 0X03180F061B03030CULL,
			0X568A13ACDC565645ULL, 0X441A49885E44440DULL, 0X7FDF9EFEA07F7FE1ULL, 0XA921374F88A9A99EULL,
			0X2A4D8254672A2AA8ULL, 0XBBB16D6B0ABBBBD6ULL, 0XC146E29F87C1C123ULL, 0X53A202A6F1535351ULL,
			0XDCAE8BA572DCDC57ULL, 0X0B582716530B0B2CULL, 0X9D9CD327019D9D4EULL, 0X6C47C1D82B6C6CADULL,
			0X3195F562A43131C4ULL, 0X7487B9E8F37474CDULL, 0XF6E309F115F6F6FFULL, 0X460A438C4C464605ULL,
			0XAC092645A5ACAC8AULL, 0X893C970FB589891EULL, 0X14A04428B4141450ULL, 0XE15B42DFBAE1E1A3ULL,
			0X16B04E2CA6161658ULL, 0X3ACDD274F73A3AE8ULL, 0X696FD0D2066969B9ULL, 0X09482D1241090924ULL,
			0X70A7ADE0D77070DDULL, 0XB6D954716FB6B6E2ULL, 0XD0CEB7BD1ED0D067ULL, 0XED3B7EC7D6EDED93ULL,
			0XCC2EDB85E2CCCC17ULL, 0X422A578468424215ULL, 0X98B4C22D2C98985AULL, 0XA4490E55EDA4A4AAULL,
			0X285D8850752828A0ULL, 0X5CDA31B8865C5C6DULL, 0XF8933FED6BF8F8C7ULL, 0X8644A411C2868622ULL,
		};

		const UInt64 C6[256] = {
			0X6018C07830D81818ULL, 0X8C2305AF46262323ULL, 0X3FC67EF991B8C6C6ULL, 0X87E8136FCDFBE8E8ULL,
			0X26874CA113CB8787ULL, 0XDAB8A9626D11B8B8ULL, 0X0401080502090101ULL, 0X214F426E9E0D4F4FULL,
			0XD836ADEE6C9B3636ULL, 0XA2A6590451FFA6A6ULL, 0X6FD2DEBDB90CD2D2ULL, 0XF3F5FB06F70EF5F5ULL,
			0XF979EF80F2967979ULL, 0XA16F5FCEDE306F6FULL, 0X7E91FCEF3F6D9191ULL, 0X5552AA07A4F85252ULL,
			0X9D6027FDC0476060ULL, 0XCABC89766535BCBCULL, 0X569BACCD2B379B9BULL, 0X028E048C018A8E8EULL,
			0XB6A371155BD2A3A3ULL, 0X300C603C186C0C0CULL, 0XF17BFF8AF6847B7BULL, 0XD435B5E16A803535ULL,
			0X741DE8693AF51D1DULL, 0XA7E05347DDB3E0E0ULL, 0X7BD7F6ACB321D7D7ULL, 0X2FC25EED999CC2C2ULL,
			0XB82E6D965C432E2EULL, 0X314B627A96294B4BULL, 0XDFFEA321E15DFEFEULL, 0X41578216AED55757ULL,
			0X5415A8412ABD1515ULL, 0XC1779FB6EEE87777ULL, 0XDC37A5EB6E923737ULL, 0XB3E57B56D79EE5E5ULL,
			0X469F8CD923139F9FULL, 0XE7F0D317FD23F0F0ULL, 0X354A6A7F94204A4AULL, 0X4FDA9E95A944DADAULL,
			0X7D58FA25B0A25858ULL, 0X03C906CA8FCFC9C9ULL, 0XA429558D527C2929ULL, 0X280A5022145A0A0AULL,
			0XFEB1E14F7F50B1B1ULL, 0XBAA0691A5DC9A0A0ULL, 0XB16B7FDAD6146B6BULL, 0X2E855CAB17D98585ULL,
			0XCEBD8173673CBDBDULL, 0X695DD234BA8F5D5DULL, 0X4010805020901010ULL, 0XF7F4F303F507F4F4ULL,
			0X0BCB16C08BDDCBCBULL, 0XF83EEDC67CD33E3EULL, 0X140528110A2D0505ULL, 0X81671FE6CE786767ULL,
			0XB7E47353D597E4E4ULL, 0X9C2725BB4E022727ULL, 0X1941325882734141ULL, 0X168B2C9D0BA78B8BULL,
			0XA6A7510153F6A7A7ULL, 0XE97DCF94FAB27D7DULL, 0X6E95DCFB37499595ULL, 0X47D88E9FAD56D8D8ULL,
			0XCBFB8B30EB70FBFBULL, 0X9FEE2371C1CDEEEEULL, 0XED7CC791F8BB7C7CULL, 0X856617E3CC716666ULL,
			0X53DDA68EA77BDDDDULL, 0X5C17B84B2EAF1717ULL, 0X014702468E454747ULL, 0X429E84DC211A9E9EULL,
			0X0FCA1EC589D4CACAULL, 0XB42D75995A582D2DULL, 0XC6BF9179632EBFBFULL, 0X1C07381B0E3F0707ULL,
			0X8EAD012347ACADADULL, 0X755AEA2FB4B05A5AULL, 0X36836CB51BEF8383ULL, 0XCC3385FF66B63333ULL,
			0X91633FF2C65C6363ULL, 0X0802100A04120202ULL, 0X92AA39384993AAAAULL, 0XD971AFA8E2DE7171ULL,
			0X07C80ECF8DC6C8C8ULL, 0X6419C87D32D11919ULL, 0X39497270923B4949ULL, 0X43D9869AAF5FD9D9ULL,
			0XEFF2C31DF931F2F2ULL, 0XABE34B48DBA8E3E3ULL, 0X715BE22AB6B95B5BULL, 0X1A8834920DBC8888ULL,
			0X529AA4C8293E9A9AULL, 0X98262DBE4C0B2626ULL, 0XC8328DFA64BF3232ULL, 0XFAB0E94A7D59B0B0ULL,
			0X83E91B6ACFF2E9E9ULL, 0X3C0F78331E770F0FULL, 0X73D5E6A6B733D5D5ULL, 0X3A8074BA1DF48080ULL,
			0XC2BE997C6127BEBEULL, 0X13CD26DE87EBCDCDULL, 0XD034BDE468893434ULL, 0X3D487A7590324848ULL,
			0XDBFFAB24E354FFFFULL, 0XF57AF78FF48D7A7AULL, 0X7A90F4EA3D649090ULL, 0X615FC23EBE9D5F5FULL,
			0X80201DA0403D2020ULL, 0XBD6867D5D00F6868ULL, 0X681AD07234CA1A1AULL, 0X82AE192C41B7AEAEULL,
			0XEAB4C95E757DB4B4ULL, 0X4D549A19A8CE5454ULL, 0X7693ECE53B7F9393ULL, 0X88220DAA442F2222ULL,
			0X8D6407E9C8636464ULL, 0XE3F1DB12FF2AF1F1ULL, 0XD173BFA2E6CC7373ULL, 0X4812905A24821212ULL,
			0X1D403A5D807A4040ULL, 0X2008402810480808ULL, 0X2BC356E89B95C3C3ULL, 0X97EC337BC5DFECECULL,
			0X4BDB9690AB4DDBDBULL, 0XBEA1611F5FC0A1A1ULL, 0X0E8D1C8307918D8DULL, 0XF43DF5C97AC83D3DULL,
			0X6697CCF1335B9797ULL, 0X0000000000000000ULL, 0X1BCF36D483F9CFCFULL, 0XAC2B4587566E2B2BULL,
			0XC57697B3ECE17676ULL, 0X328264B019E68282ULL, 0X7FD6FEA9B128D6D6ULL, 0X6C1BD87736C31B1BULL,
			0XEEB5C15B7774B5B5ULL, 0X86AF112943BEAFAFULL, 0XB56A77DFD41D6A6AULL, 0X5D50BA0DA0EA5050ULL,
			0X0945124C8A574545ULL, 0XEBF3CB18FB38F3F3ULL, 0XC0309DF060AD3030ULL, 0X9BEF2B74C3C4EFEFULL,
			0XFC3FE5C37EDA3F3FULL, 0X4955921CAAC75555ULL, 0XB2A2791059DBA2A2ULL, 0X8FEA0365C9E9EAEAULL,
			0X89650FECCA6A6565ULL, 0XD2BAB9686903BABAULL, 0XBC2F65935E4A2F2FULL, 0X27C04EE79D8EC0C0ULL,
			0X5FDEBE81A160DEDEULL, 0X701CE06C38FC1C1CULL, 0XD3FDBB2EE746FDFDULL, 0X294D52649A1F4D4DULL,
			0X7292E4E039769292ULL, 0XC9758FBCEAFA7575ULL, 0X1806301E0C360606ULL, 0X128A249809AE8A8AULL,
			0XF2B2F940794BB2B2ULL, 0XBFE66359D185E6E6ULL, 0X380E70361C7E0E0EULL, 0X7C1FF8633EE71F1FULL,
			0X956237F7C4556262ULL, 0X77D4EEA3B53AD4D4ULL, 0X9AA829324D81A8A8ULL, 0X6296C4F431529696ULL,
			0XC3F99B3AEF62F9F9ULL, 0X33C566F697A3C5C5ULL, 0X942535B14A102525ULL, 0X7959F220B2AB5959ULL,
			0X2A8454AE15D08484ULL, 0XD572B7A7E4C57272ULL, 0XE439D5DD72EC3939ULL, 0X2D4C5A6198164C4CULL,
			0X655ECA3BBC945E5EULL, 0XFD78E785F09F7878ULL, 0XE038DDD870E53838ULL, 0X0A8C148605988C8CULL,
			0X63D1C6B2BF17D1D1ULL, 0XAEA5410B57E4A5A5ULL, 0XAFE2434DD9A1E2E2ULL, 0X99612FF8C24E6161ULL,
			0XF6B3F1457B42B3B3ULL, 0X842115A542342121ULL, 0X4A9C94D625089C9CULL, 0X781EF0663CEE1E1EULL,
			0X1143225286614343ULL, 0X3BC776FC93B1C7C7ULL, 0XD7FCB32BE54FFCFCULL, 0X1004201408240404ULL,
			0X5951B208A2E35151ULL, 0X5E99BCC72F259999ULL, 0XA96D4FC4DA226D6DULL, 0X340D68391A650D0DULL,
			0XCFFA8335E979FAFAULL, 0X5BDFB684A369DFDFULL, 0XE57ED79BFCA97E7EULL, 0X90243DB448192424ULL,
			0XEC3BC5D776FE3B3BULL, 0X96AB313D4B9AABABULL, 0X1FCE3ED181F0CECEULL, 0X4411885522991111ULL,
			0X068F0C8903838F8FULL, 0X254E4A6B9C044E4EULL, 0XE6B7D1517366B7B7ULL, 0X8BEB0B60CBE0EBEBULL,
			0XF03CFDCC78C13C3CULL, 0X3E817CBF1FFD8181ULL, 0X6A94D4FE35409494ULL, 0XFBF7EB0CF31CF7F7ULL,
			0XDEB9A1676F18B9B9ULL, 0X4C13985F268B1313ULL, 0XB02C7D9C58512C2CULL, 0X6BD3D6B8BB05D3D3ULL,
			0XBBE76B5CD38CE7E7ULL, 0XA56E57CBDC396E6EULL, 0X37C46EF395AAC4C4ULL, 0X0C03180F061B0303ULL,
			0X45568A13ACDC5656ULL, 0X0D441A49885E4444ULL, 0XE17FDF9EFEA07F7FULL, 0X9EA921374F88A9A9ULL,
			0XA82A4D8254672A2AULL, 0XD6BBB16D6B0ABBBBULL, 0X23C146E29F87C1C1ULL, 0X5153A202A6F15353ULL,
			0X57DCAE8BA572DCDCULL, 0X2C0B582716530B0BULL, 0X4E9D9CD327019D9DULL, 0XAD6C47C1D82B6C6CULL,
			0XC43195F562A43131ULL, 0XCD7487B9E8F37474ULL, 0XFFF6E309F115F6F6ULL, 0X05460A438C4C4646ULL,
			0X8AAC092645A5ACACULL, 0X1E893C970FB58989ULL, 0X5014A04428B41414ULL, 0XA3E15B42DFBAE1E1ULL,
			0X5816B04E2CA61616ULL, 0XE83ACDD274F73A3AULL, 0XB9696FD0D2066969ULL, 0X2409482D12410909ULL,
			0XDD70A7ADE0D77070ULL, 0XE2B6D954716FB6B6ULL, 0X67D0CEB7BD1ED0D0ULL, 0X93ED3B7EC7D6EDEDULL,
			0X17CC2EDB85E2CCCCULL, 0X15422A5784684242ULL, 0X5A98B4C22D2C9898ULL, 0XAAA4490E55EDA4A4ULL,
			0XA0285D8850752828ULL, 0X6D5CDA31B8865C5CULL, 0XC7F8933FED6BF8F8ULL, 0X228644A411C28686ULL,
		};

		const UInt64 C7[256] = {
			0X186018C07830D818ULL, 0X238C2305AF462623ULL, 0XC63FC67EF991B8C6ULL, 0XE887E8136FCDFBE8ULL,
			0X8726874CA113CB87ULL, 0XB8DAB8A9626D11B8ULL, 0X0104010805020901ULL, 0X4F214F426E9E0D4FULL,
			0X36D836ADEE6C9B36ULL, 0XA6A2A6590451FFA6ULL, 0XD26FD2DEBDB90CD2ULL, 0XF5F3F5FB06F70EF5ULL,
			0X79F979EF80F29679ULL, 0X6FA16F5FCEDE306FULL, 0X917E91FCEF3F6D91ULL, 0X525552AA07A4F852ULL,
			0X609D6027FDC04760ULL, 0XBCCABC89766535BCULL, 0X9B569BACCD2B379BULL, 0X8E028E048C018A8EULL,
			0XA3B6A371155BD2A3ULL, 0X0C300C603C186C0CULL, 0X7BF17BFF8AF6847BULL, 0X35D435B5E16A8035ULL,
			0X1D741DE8693AF51DULL, 0XE0A7E05347DDB3E0ULL, 0XD77BD7F6ACB321D7ULL, 0XC22FC25EED999CC2ULL,
			0X2EB82E6D965C432EULL, 0X4B314B627A96294BULL, 0XFEDFFEA321E15DFEULL, 0X5741578216AED557ULL,
			0X155415A8412ABD15ULL, 0X77C1779FB6EEE877ULL, 0X37DC37A5EB6E9237ULL, 0XE5B3E57B56D79EE5ULL,
			0X9F469F8CD923139FULL, 0XF0E7F0D317FD23F0ULL, 0X4A354A6A7F94204AULL, 0XDA4FDA9E95A944DAULL,
			0X587D58FA25B0A258ULL, 0XC903C906CA8FCFC9ULL, 0X29A429558D527C29ULL, 0X0A280A5022145A0AULL,
			0XB1FEB1E14F7F50B1ULL, 0XA0BAA0691A5DC9A0ULL, 0X6BB16B7FDAD6146BULL, 0X852E855CAB17D985ULL,
			0XBDCEBD8173673CBDULL, 0X5D695DD234BA8F5DULL, 0X1040108050209010ULL, 0XF4F7F4F303F507F4ULL,
			0XCB0BCB16C08BDDCBULL, 0X3EF83EEDC67CD33EULL, 0X05140528110A2D05ULL, 0X6781671FE6CE7867ULL,
			0XE4B7E47353D597E4ULL, 0X279C2725BB4E0227ULL, 0X4119413258827341ULL, 0X8B168B2C9D0BA78BULL,
			0XA7A6A7510153F6A7ULL, 0X7DE97DCF94FAB27DULL, 0X956E95DCFB374995ULL, 0XD847D88E9FAD56D8ULL,
			0XFBCBFB8B30EB70FBULL, 0XEE9FEE2371C1CDEEULL, 0X7CED7CC791F8BB7CULL, 0X66856617E3CC7166ULL,
			0XDD53DDA68EA77BDDULL, 0X175C17B84B2EAF17ULL, 0X47014702468E4547ULL, 0X9E429E84DC211A9EULL,
			0XCA0FCA1EC589D4CAULL, 0X2DB42D75995A582DULL, 0XBFC6BF9179632EBFULL, 0X071C07381B0E3F07ULL,
			0XAD8EAD012347ACADULL, 0X5A755AEA2FB4B05AULL, 0X8336836CB51BEF83ULL, 0X33CC3385FF66B633ULL,
			0X6391633FF2C65C63ULL, 0X020802100A041202ULL, 0XAA92AA39384993AAULL, 0X71D971AFA8E2DE71ULL,
			0XC807C80ECF8DC6C8ULL, 0X196419C87D32D119ULL, 0X4939497270923B49ULL, 0XD943D9869AAF5FD9ULL,
			0XF2EFF2C31DF931F2ULL, 0XE3ABE34B48DBA8E3ULL, 0X5B715BE22AB6B95BULL, 0X881A8834920DBC88ULL,
			0X9A529AA4C8293E9AULL, 0X2698262DBE4C0B26ULL, 0X32C8328DFA64BF32ULL, 0XB0FAB0E94A7D59B0ULL,
			0XE983E91B6ACFF2E9ULL, 0X0F3C0F78331E770FULL, 0XD573D5E6A6B733D5ULL, 0X803A8074BA1DF480ULL,
			0XBEC2BE997C6127BEULL, 0XCD13CD26DE87EBCDULL, 0X34D034BDE4688934ULL, 0X483D487A75903248ULL,
			0XFFDBFFAB24E354FFULL, 0X7AF57AF78FF48D7AULL, 0X907A90F4EA3D6490ULL, 0X5F615FC23EBE9D5FULL,
			0X2080201DA0403D20ULL, 0X68BD6867D5D00F68ULL, 0X1A681AD07234CA1AULL, 0XAE82AE192C41B7AEULL,
			0XB4EAB4C95E757DB4ULL, 0X544D549A19A8CE54ULL, 0X937693ECE53B7F93ULL, 0X2288220DAA442F22ULL,
			0X648D6407E9C86364ULL, 0XF1E3F1DB12FF2AF1ULL, 0X73D173BFA2E6CC73ULL, 0X124812905A248212ULL,
			0X401D403A5D807A40ULL, 0X0820084028104808ULL, 0XC32BC356E89B95C3ULL, 0XEC97EC337BC5DFECULL,
			0XDB4BDB9690AB4DDBULL, 0XA1BEA1611F5FC0A1ULL, 0X8D0E8D1C8307918DULL, 0X3DF43DF5C97AC83DULL,
			0X976697CCF1335B97ULL, 0X0000000000000000ULL, 0XCF1BCF36D483F9CFULL, 0X2BAC2B4587566E2BULL,
			0X76C57697B3ECE176ULL, 0X82328264B019E682ULL, 0XD67FD6FEA9B128D6ULL, 0X1B6C1BD87736C31BULL,
			0XB5EEB5C15B7774B5ULL, 0XAF86AF112943BEAFULL, 0X6AB56A77DFD41D6AULL, 0X505D50BA0DA0EA50ULL,
			0X450945124C8A5745ULL, 0XF3EBF3CB18FB38F3ULL, 0X30C0309DF060AD30ULL, 0XEF9BEF2B74C3C4EFULL,
			0X3FFC3FE5C37EDA3FULL, 0X554955921CAAC755ULL, 0XA2B2A2791059DBA2ULL, 0XEA8FEA0365C9E9EAULL,
			0X6589650FECCA6A65ULL, 0XBAD2BAB9686903BAULL, 0X2FBC2F65935E4A2FULL, 0XC027C04EE79D8EC0ULL,
			0XDE5FDEBE81A160DEULL, 0X1C701CE06C38FC1CULL, 0XFDD3FDBB2EE746FDULL, 0X4D294D52649A1F4DULL,
			0X927292E4E0397692ULL, 0X75C9758FBCEAFA75ULL, 0X061806301E0C3606ULL, 0X8A128A249809AE8AULL,
			0XB2F2B2F940794BB2ULL, 0XE6BFE66359D185E6ULL, 0X0E380E70361C7E0EULL, 0X1F7C1FF8633EE71FULL,
			0X62956237F7C45562ULL, 0XD477D4EEA3B53AD4ULL, 0XA89AA829324D81A8ULL, 0X966296C4F4315296ULL,
			0XF9C3F99B3AEF62F9ULL, 0XC533C566F697A3C5ULL, 0X25942535B14A1025ULL, 0X597959F220B2AB59ULL,
			0X842A8454AE15D084ULL, 0X72D572B7A7E4C572ULL, 0X39E439D5DD72EC39ULL, 0X4C2D4C5A6198164CULL,
			0X5E655ECA3BBC945EULL, 0X78FD78E785F09F78ULL, 0X38E038DDD870E538ULL, 0X8C0A8C148605988CULL,
			0XD163D1C6B2BF17D1ULL, 0XA5AEA5410B57E4A5ULL, 0XE2AFE2434DD9A1E2ULL, 0X6199612FF8C24E61ULL,
			0XB3F6B3F1457B42B3ULL, 0X21842115A5423421ULL, 0X9C4A9C94D625089CULL, 0X1E781EF0663CEE1EULL,
			0X4311432252866143ULL, 0XC73BC776FC93B1C7ULL, 0XFCD7FCB32BE54FFCULL, 0X0410042014082404ULL,
			0X515951B208A2E351ULL, 0X995E99BCC72F2599ULL, 0X6DA96D4FC4DA226DULL, 0X0D340D68391A650DULL,
			0XFACFFA8335E979FAULL, 0XDF5BDFB684A369DFULL, 0X7EE57ED79BFCA97EULL, 0X2490243DB4481924ULL,
			0X3BEC3BC5D776FE3BULL, 0XAB96AB313D4B9AABULL, 0XCE1FCE3ED181F0CEULL, 0X1144118855229911ULL,
			0X8F068F0C8903838FULL, 0X4E254E4A6B9C044EULL, 0XB7E6B7D1517366B7ULL, 0XEB8BEB0B60CBE0EBULL,
			0X3CF03CFDCC78C13CULL, 0X813E817CBF1FFD81ULL, 0X946A94D4FE354094ULL, 0XF7FBF7EB0CF31CF7ULL,
			0XB9DEB9A1676F18B9ULL, 0X134C13985F268B13ULL, 0X2CB02C7D9C58512CULL, 0XD36BD3D6B8BB05D3ULL,
			0XE7BBE76B5CD38CE7ULL, 0X6EA56E57CBDC396EULL, 0XC437C46EF395AAC4ULL, 0X030C03180F061B03ULL,
			0X5645568A13ACDC56ULL, 0X440D441A49885E44ULL, 0X7FE17FDF9EFEA07FULL, 0XA99EA921374F88A9ULL,
			0X2AA82A4D8254672AULL, 0XBBD6BBB16D6B0ABBULL, 0XC123C146E29F87C1ULL, 0X535153A202A6F153ULL,
			0XDC57DCAE8BA572DCULL, 0X0B2C0B582716530BULL, 0X9D4E9D9CD327019DULL, 0X6CAD6C47C1D82B6CULL,
			0X31C43195F562A431ULL, 0X74CD7487B9E8F374ULL, 0XF6FFF6E309F115F6ULL, 0X4605460A438C4C46ULL,
			0XAC8AAC092645A5ACULL, 0X891E893C970FB589ULL, 0X145014A04428B414ULL, 0XE1A3E15B42DFBAE1ULL,
			0X165816B04E2CA616ULL, 0X3AE83ACDD274F73AULL, 0X69B9696FD0D20669ULL, 0X092409482D124109ULL,
			0X70DD70A7ADE0D770ULL, 0XB6E2B6D954716FB6ULL, 0XD067D0CEB7BD1ED0ULL, 0XED93ED3B7EC7D6EDULL,
			0XCC17CC2EDB85E2CCULL, 0X4215422A57846842ULL, 0X985A98B4C22D2C98ULL, 0XA4AAA4490E55EDA4ULL,
			0X28A0285D88507528ULL, 0X5C6D5CDA31B8865CULL, 0XF8C7F8933FED6BF8ULL, 0X86228644A411C286ULL,
		};

		const UInt64 rc[R + 1] = {
			0X0000000000000000ULL,
			0X1823C6E887B8014FULL,
			0X36A6D2F5796F9152ULL,
			0X60BC9B8EA30C7B35ULL,
			0X1DE0D7C22E4BFE57ULL,
			0X157737E59FF04ADAULL,
			0X58C9290AB1A06B85ULL,
			0XBD5D10F4CB3E0567ULL,
			0XE427418BA77D95D8ULL,
			0XFBEE7C66DD17479EULL,
			0XCA2DBF07AD5A8333ULL,
		};
	}

	void WhirlpoolHasher::Append(const UInt8* data, std::size_t len)
	{
		len *= 8; // Whirlpool works with bits

		int sourcePos = 0; /* index of leftmost source UInt8 containing data (1 to 8 bits). */
		int sourceGap = (8 - (static_cast<std::intmax_t>(len) & 7)) & 7; /* space on source[sourcePos]. */
		int bufferRem = m_bufferBits & 7; /* occupied bits on buffer[bufferPos]. */

		UInt32 b;
		UInt8* buffer = m_buffer;
		UInt8* bitLength = m_bitLength;
		std::size_t bufferBits = m_bufferBits;
		std::size_t bufferPos = m_bufferPos;

		// tally the length of the added data
		UInt64 value = len;
		UInt32 carry = 0;
		for (int i = 31; i >= 0 && (carry != 0 || value != 0ULL); i--)
		{
			carry += bitLength[i] + (static_cast<UInt32>(value) & 0xFF);
			bitLength[i] = static_cast<UInt8>(carry);

			carry >>= 8;
			value >>= 8;
		}

		// process data in chunks of 8 bits (a more efficient approach would be to take whole-word chunks)
		while (len > 8)
		{
			// N.B. at least source[sourcePos] and source[sourcePos+1] contain data.
			// take a byte from the source
			b = ((data[sourcePos] << sourceGap) & 0xFF) |
				((data[sourcePos + 1] & 0xFF) >> (8 - sourceGap));

			// process this byte
			buffer[bufferPos++] |= static_cast<UInt8>(b >> bufferRem);
			bufferBits += 8 - bufferRem; // bufferBits = 8*bufferPos;

			if (bufferBits == 512)
			{
				ProcessBuffer(); // process data block
				bufferBits = bufferPos = 0; // reset buffer
			}

			buffer[bufferPos] = static_cast<UInt8>(b << (8 - bufferRem));
			bufferBits += bufferRem;

			// proceed to remaining data
			len -= 8;
			sourcePos++;
		}

		// now 0 <= sourceBits <= 8;
		// furthermore, all data (if any is left) is in source[sourcePos].
		if (len > 0)
		{
			b = (data[sourcePos] << sourceGap) & 0xFF; // bits are left-justified on b.
			buffer[bufferPos] |= b >> bufferRem; // process the remaining bits
		}
		else
			b = 0;

		if (bufferRem + len < 8)
			// all remaining data fits on buffer[bufferPos], and there still remains some space.
			bufferBits += len;
		else
		{
			// buffer[bufferPos] is full
			bufferPos++;
			bufferBits += 8 - bufferRem; // bufferBits = 8*bufferPos;
			len -= 8 - bufferRem;

			// now 0 <= sourceBits < 8; furthermore, all data (if any is left) is in source[sourcePos].
			if (bufferBits == 512)
			{
				ProcessBuffer(); // process data block
				bufferBits = bufferPos = 0; // reset buffer
			}

			buffer[bufferPos] = UInt8(b << (8 - bufferRem));
			bufferBits += len;
		}

		m_bufferBits = bufferBits;
		m_bufferPos = bufferPos;
	}

	void WhirlpoolHasher::Begin()
	{
		std::memset(m_bitLength, 0, 32);
		m_bufferBits = m_bufferPos = 0;
		m_buffer[0] = 0; // it's only necessary to cleanup buffer[bufferPos]
		for (int i = 0; i < 8; i++)
			m_hash[i] = 0L; // initial value
	}

	ByteArray WhirlpoolHasher::End()
	{
		UInt8 result[64];

		UInt8 *buffer = m_buffer;
		UInt8 *bitLength = m_bitLength;
		std::size_t bufferBits = m_bufferBits;
		std::size_t bufferPos = m_bufferPos;
		UInt8 *digest = result;

		// append a '1'-bit
		buffer[bufferPos] |= 0x80U >> (bufferBits & 7);
		bufferPos++; // all remaining bits on the current UInt8 are set to zero.

		// pad with zero bits to complete (N*WBLOCKBITS - LENGTHBITS) bits:
		if (bufferPos > 32)
		{
			if (bufferPos < 64)
				std::memset(&buffer[bufferPos], 0, 64 - bufferPos);

			ProcessBuffer(); // process data block
			bufferPos = 0; // reset buffer
		}

		if (bufferPos < 32)
			std::memset(&buffer[bufferPos], 0, 32 - bufferPos);

		bufferPos = 32;

		std::memcpy(&buffer[32], bitLength, 32); // append bit length of hashed data

		ProcessBuffer(); // process data block

		// return the completed message digest
		for (int i = 0; i < 8; i++)
		{
			digest[0] = static_cast<UInt8>(m_hash[i] >> 56);
			digest[1] = static_cast<UInt8>(m_hash[i] >> 48);
			digest[2] = static_cast<UInt8>(m_hash[i] >> 40);
			digest[3] = static_cast<UInt8>(m_hash[i] >> 32);
			digest[4] = static_cast<UInt8>(m_hash[i] >> 24);
			digest[5] = static_cast<UInt8>(m_hash[i] >> 16);
			digest[6] = static_cast<UInt8>(m_hash[i] >>  8);
			digest[7] = static_cast<UInt8>(m_hash[i]      );
			digest += 8;
		}

		m_bufferBits = bufferBits;
		m_bufferPos = bufferPos;

		return ByteArray(&result[0], 64);
	}

	std::size_t WhirlpoolHasher::GetDigestLength() const
	{
		return 64;
	}

	const char* WhirlpoolHasher::GetHashName() const
	{
		return "Whirlpool";
	}


	void WhirlpoolHasher::ProcessBuffer()
	{
		NAZARA_USE_ANONYMOUS_NAMESPACE

		int i, r;
		UInt64 K[8];     // the round key
		UInt64 block[8]; // mu(buffer)
		UInt64 state[8]; // the cipher state
		UInt64 L[8];
		UInt8 *buffer = m_buffer;

		// map the buffer to a block
		for (i = 0; i < 8; i++, buffer += 8)
		{
			block[i] =
				((static_cast<UInt64>(buffer[0])        ) << 56) ^
				((static_cast<UInt64>(buffer[1]) & 0xFFL) << 48) ^
				((static_cast<UInt64>(buffer[2]) & 0xFFL) << 40) ^
				((static_cast<UInt64>(buffer[3]) & 0xFFL) << 32) ^
				((static_cast<UInt64>(buffer[4]) & 0xFFL) << 24) ^
				((static_cast<UInt64>(buffer[5]) & 0xFFL) << 16) ^
				((static_cast<UInt64>(buffer[6]) & 0xFFL) <<  8) ^
				((static_cast<UInt64>(buffer[7]) & 0xFFL)     );
		}

		// compute and apply K^0 to the cipher state
		state[0] = block[0] ^ (K[0] = m_hash[0]);
		state[1] = block[1] ^ (K[1] = m_hash[1]);
		state[2] = block[2] ^ (K[2] = m_hash[2]);
		state[3] = block[3] ^ (K[3] = m_hash[3]);
		state[4] = block[4] ^ (K[4] = m_hash[4]);
		state[5] = block[5] ^ (K[5] = m_hash[5]);
		state[6] = block[6] ^ (K[6] = m_hash[6]);
		state[7] = block[7] ^ (K[7] = m_hash[7]);

		// iterate over all rounds
		for (r = 1; r <= R; r++)
		{
			// compute K^r from K^{r-1}
			L[0] =
				C0[static_cast<int>(K[0] >> 56)    ] ^
				C1[static_cast<int>(K[7] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[6] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[5] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[4] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[3] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[2] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[1]      ) & 0xFF] ^
				rc[r];

			L[1] =
				C0[static_cast<int>(K[1] >> 56)    ] ^
				C1[static_cast<int>(K[0] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[7] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[6] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[5] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[4] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[3] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[2]      ) & 0xFF];

			L[2] =
				C0[static_cast<int>(K[2] >> 56)    ] ^
				C1[static_cast<int>(K[1] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[0] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[7] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[6] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[5] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[4] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[3]      ) & 0xFF];

			L[3] =
				C0[static_cast<int>(K[3] >> 56)    ] ^
				C1[static_cast<int>(K[2] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[1] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[0] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[7] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[6] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[5] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[4]      ) & 0xFF];

			L[4] =
				C0[static_cast<int>(K[4] >> 56)    ] ^
				C1[static_cast<int>(K[3] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[2] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[1] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[0] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[7] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[6] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[5]      ) & 0xFF];

			L[5] =
				C0[static_cast<int>(K[5] >> 56)    ] ^
				C1[static_cast<int>(K[4] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[3] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[2] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[1] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[0] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[7] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[6]      ) & 0xFF];
			L[6] =
				C0[static_cast<int>(K[6] >> 56)    ] ^
				C1[static_cast<int>(K[5] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[4] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[3] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[2] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[1] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[0] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[7]      ) & 0xFF];

			L[7] =
				C0[static_cast<int>(K[7] >> 56)    ] ^
				C1[static_cast<int>(K[6] >> 48) & 0xFF] ^
				C2[static_cast<int>(K[5] >> 40) & 0xFF] ^
				C3[static_cast<int>(K[4] >> 32) & 0xFF] ^
				C4[static_cast<int>(K[3] >> 24) & 0xFF] ^
				C5[static_cast<int>(K[2] >> 16) & 0xFF] ^
				C6[static_cast<int>(K[1] >>  8) & 0xFF] ^
				C7[static_cast<int>(K[0]      ) & 0xFF];

			K[0] = L[0];
			K[1] = L[1];
			K[2] = L[2];
			K[3] = L[3];
			K[4] = L[4];
			K[5] = L[5];
			K[6] = L[6];
			K[7] = L[7];

			// apply the r-th round transformation
			L[0] =
				C0[static_cast<int>(state[0] >> 56)    ] ^
				C1[static_cast<int>(state[7] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[6] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[5] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[4] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[3] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[2] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[1]      ) & 0xFF] ^
				K[0];

			L[1] =
				C0[static_cast<int>(state[1] >> 56)    ] ^
				C1[static_cast<int>(state[0] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[7] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[6] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[5] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[4] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[3] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[2]      ) & 0xFF] ^
				K[1];
			L[2] =
				C0[static_cast<int>(state[2] >> 56)    ] ^
				C1[static_cast<int>(state[1] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[0] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[7] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[6] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[5] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[4] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[3]      ) & 0xFF] ^

				K[2];
			L[3] =
				C0[static_cast<int>(state[3] >> 56)    ] ^
				C1[static_cast<int>(state[2] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[1] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[0] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[7] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[6] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[5] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[4]      ) & 0xFF] ^
				K[3];
			L[4] =
				C0[static_cast<int>(state[4] >> 56)    ] ^
				C1[static_cast<int>(state[3] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[2] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[1] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[0] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[7] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[6] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[5]      ) & 0xFF] ^
				K[4];

			L[5] =
				C0[static_cast<int>(state[5] >> 56)    ] ^
				C1[static_cast<int>(state[4] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[3] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[2] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[1] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[0] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[7] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[6]      ) & 0xFF] ^
				K[5];

			L[6] =
				C0[static_cast<int>(state[6] >> 56)    ] ^
				C1[static_cast<int>(state[5] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[4] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[3] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[2] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[1] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[0] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[7]      ) & 0xFF] ^
				K[6];

			L[7] =
				C0[static_cast<int>(state[7] >> 56)    ] ^
				C1[static_cast<int>(state[6] >> 48) & 0xFF] ^
				C2[static_cast<int>(state[5] >> 40) & 0xFF] ^
				C3[static_cast<int>(state[4] >> 32) & 0xFF] ^
				C4[static_cast<int>(state[3] >> 24) & 0xFF] ^
				C5[static_cast<int>(state[2] >> 16) & 0xFF] ^
				C6[static_cast<int>(state[1] >>  8) & 0xFF] ^
				C7[static_cast<int>(state[0]      ) & 0xFF] ^
				K[7];

			state[0] = L[0];
			state[1] = L[1];
			state[2] = L[2];
			state[3] = L[3];
			state[4] = L[4];
			state[5] = L[5];
			state[6] = L[6];
			state[7] = L[7];
		}

		// apply the Miyaguchi-Preneel compression function
		m_hash[0] ^= state[0] ^ block[0];
		m_hash[1] ^= state[1] ^ block[1];
		m_hash[2] ^= state[2] ^ block[2];
		m_hash[3] ^= state[3] ^ block[3];
		m_hash[4] ^= state[4] ^ block[4];
		m_hash[5] ^= state[5] ^ block[5];
		m_hash[6] ^= state[6] ^ block[6];
		m_hash[7] ^= state[7] ^ block[7];
	}
}

#undef ONE64
#undef ROTR64
#undef R
#undef T64
