/* Value Change Dump Program for libvmcu */

//vcd format description:
//https://zipcpu.com/blog/2017/07/31/vcd.html

// C Headers
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <time.h>

// libvmcu
#include "libvmcu_analyzer.h"
#include "libvmcu_system.h"

/* libvmcu Structures */
vmcu_model_t  *m328p  = NULL;
vmcu_report_t *report = NULL;
vmcu_system_t *sys    = NULL;

/* driver structures */
uint16_t* addresses;
int count_addresses;

static void cleanup(void);
static void vcdump(vmcu_system_t* sys, uint64_t cycles);
static void vcdump_inner(vmcu_system_t* sys, uint64_t cycles, FILE* fout);
static void print_binary(FILE* fout, uint8_t value);
static void update_values(uint8_t* values);

int main(const int argc, const char **argv) {

    if(argc < 4) {

        printf("Usage: ./vcdump <binary> <cycles> <values to be monitored>+\n");
        return EXIT_FAILURE;
    }

    const char* filename = argv[1];
    uint64_t cycles = (uint64_t) atoi(argv[2]);
    
    printf("cycles=%ld\n", cycles);

    count_addresses = argc - 3;
    addresses = malloc(sizeof(uint16_t)*count_addresses);

    for(int i=3; i < argc; i++){ 
    
        const char* str = argv[i];
    	sscanf(str, "%hx", &addresses[i-3]);
    }

    m328p = vmcu_model_ctor(VMCU_DEVICE_M328P);

    if((report = vmcu_analyze_file(filename, m328p)) == NULL)
        return EXIT_FAILURE;

    sys = vmcu_system_ctor(report);
    vcdump(sys, cycles);

    cleanup();
    return EXIT_SUCCESS;
}

static void vcdump(vmcu_system_t* sys, uint64_t cycles) {
	
	FILE* fout = fopen("out.vcd", "w");
	fprintf(fout, "$version Generated by VCDUMP $end\n");
	
	char text[100];
	time_t now = time(NULL);
	struct tm *t = localtime(&now);
	strftime(text, sizeof(text)-1, "%a %b %d %H:%M:%S %Y", t);
	
	fprintf(fout, "$date %s $end\n", text);
	fprintf(fout, "$timescale 1us $end\n");
	fprintf(fout, "$scope module TOP $end\n");
	
	for(int i=0;i < count_addresses; i++){
	
		uint16_t addr = addresses[i];
		
		char varname[10];
		sprintf(varname, "%x", addr);

		fprintf(fout, "$var real 8 0x%s 0x%s[7:0] $end\n", varname, varname);
	}
	
	fprintf(fout, "$upscope $end\n");
	
	fprintf(fout, "$enddefinitions $end\n");
	fprintf(fout, "$dumpvars\n");

	vcdump_inner(sys, cycles, fout);
	
	fclose(fout);
}

static void vcdump_inner(vmcu_system_t* sys, uint64_t cycles, FILE* fout){

	printf("starting vcd trace...\n");
	uint8_t* values = malloc(sizeof(uint8_t)*count_addresses);

	while(sys->cycles < cycles){
		
		update_values(values);
		vmcu_system_step(sys);
		
		//go through all values that changed
		for(int i=0; i < count_addresses; i++){
		
			uint16_t addr = addresses[i];
			uint8_t value = values[i];
			uint8_t new_value = vmcu_system_read_data(sys, addr);
			
			if(new_value == value && (sys->steps != 1)){ 
				continue; 
			}

			fprintf(fout, "#%ld ", sys->cycles);
			fprintf(fout, "r%.16g", value*1.0);
			fprintf(fout, " 0x%x\n", addr);
		}
	}
	
	printf("done\n");

	free(values);
}

static void print_binary(FILE* fout, uint8_t value) {
	
	fprintf(fout, "b");
	
	for(int i=7;i >= 0; i--){
		uint8_t bit = (value & (0x01 << i)) >> i;
		char str[2];
		sprintf(str,"%d", bit);
		
		fputs(str, fout);
	}
}

static void update_values(uint8_t* values) {

	for(int i=0; i < count_addresses; i++) {

		values[i] = vmcu_system_read_data(sys, addresses[i]);
	}
}

static void cleanup(void) {

    vmcu_report_dtor(report);
    vmcu_model_dtor(m328p);
    vmcu_system_dtor(sys);

    free(addresses);
}
