import {
  Box,
  Button,
  Flex,
  Icon,
  Image,
  Link,
  Skeleton,
  SkeletonCircle,
  Stack,
  Text,
  useColorModeValue,
} from "@chakra-ui/react";
import { collection, getDocs, limit, orderBy, query } from "firebase/firestore";
import { useEffect, useState, FC } from "react";
import { FaReddit } from "react-icons/fa";
import { Community } from "../../atoms/communitiesAtom";
import { firestore } from "../../firebase/clientApp";
import { useCommunityData } from "../../hooks/useCommunityData";

const Recommendation: FC = () => {
  const [communities, setCommunities] = useState<Community[]>([]);
  const [loading, setLoading] = useState(false);
  const { communityStateValue, onJoinOrLeaveCommunity } = useCommunityData();

  const getCommunityRecommendation = async () => {
    setLoading(true);
    try {
      const communityQuery = query(
        collection(firestore, "communities"),
        orderBy("numberOfMembers", "desc"),
        limit(5)
      );

      const communityDocs = await getDocs(communityQuery);
      const communities = communityDocs.docs.map((doc) => ({
        id: doc.id,
        ...doc.data(),
      }));

      setCommunities(communities as Community[]);
    } catch (error: any) {
      console.log("handle getComRecommendation", error.message);
    }
    setLoading(false);
  };

  useEffect(() => {
    getCommunityRecommendation();
  }, []);

  const bg = useColorModeValue("white", "gray.700");
  const dividerBg = useColorModeValue("gray.300", "gray.600");

  return (
    <Flex
      direction="column"
      bg={bg}
      borderRadius={4}
      border="1px solid"
      borderColor={dividerBg}
    >
      <Flex
        align="flex-end"
        color="white"
        p="6px 10px"
        height="70px"
        borderRadius="4px 4px 0px 0px"
        fontWeight="600"
        bgImage="url(/images/recCommsArt.png)"
        backgroundSize="cover"
        bgGradient="linear-gradient(to bottom , rgba(0,0,0,0), rgba(0,0,0,0.75)), url('images/recCommsArt.png')"
      >
        Top Communites
      </Flex>

      {/* Communities Links to join */}
      <Flex direction="column">
        {/* Loading skeloeto  */}
        {loading ? (
          <Stack mt={2} p={3}>
            <Flex justify="space-between" align="center">
              <SkeletonCircle size="10" />
              <Skeleton height="10px" width="70%" />
            </Flex>
            <Flex justify="space-between" align="center">
              <SkeletonCircle size="10" />
              <Skeleton height="10px" width="70%" />
            </Flex>
            <Flex justify="space-between" align="center">
              <SkeletonCircle size="10" />
              <Skeleton height="10px" width="70%" />
            </Flex>
          </Stack>
        ) : (
          <>
            {/* Communities data */}
            {communities.map((community, index) => {
              const isJoined = !!communityStateValue.mySnippets.find(
                (snippet) => snippet.communityId === community.id
              );
              return (
                <Box key={community.id}>
                  <Flex
                    position="relative"
                    align="center"
                    fontSize="10pt"
                    borderBottom="1px solid"
                    borderColor={dividerBg}
                    p="10px 12px"
                  >
                    <Flex width="100%" align="center">
                      {/* Index number */}
                      <Flex width="15%">
                        <Text>{index + 1}</Text>
                      </Flex>

                      {/* Img and name  */}
                      <Flex align="center" width="80%" gap={2}>
                        {/* community profile image  */}
                        {community.profileImageURL ? (
                          <Image
                            borderRadius="full"
                            boxSize="30px"
                            src={community.profileImageURL}
                            alt=""
                          />
                        ) : (
                          <Icon as={FaReddit} fontSize={30} color="brand.100" />
                        )}
                        {/* community name  */}
                        <Link
                          href={`r/${community.id}`}
                          style={{
                            whiteSpace: "nowrap",
                            overflow: "hidden",
                            textOverflow: "ellipsis",
                          }}
                        >{`r/${community.id}`}</Link>
                      </Flex>
                    </Flex>

                    {/* join button  */}
                    <Box position="absolute" right="10px">
                      <Button
                        height="22px"
                        fontSize="8pt"
                        onClick={(event) => {
                          event.stopPropagation();
                          onJoinOrLeaveCommunity(community, isJoined);
                        }}
                        variant={isJoined ? "outline" : "solid"}
                      >
                        {isJoined ? "Joined" : "Join"}
                      </Button>
                    </Box>
                  </Flex>
                </Box>
              );
            })}
            <Box p="10px 20px">
              <Button height="30px" width="100%">
                View All
              </Button>
            </Box>
          </>
        )}
      </Flex>
    </Flex>
  );
};

export default Recommendation;
