import {
  MockRequestResponse,
  SENDER_ADDRESS_MOCK,
  SENDER_ADDRESS_NO_0X_MOCK,
} from './types';

export const BUY_ERC20_TRANSACTION = {
  data: `0x5ae401dc0000000000000000000000000000000000000000000000000000000065fb43ab0000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000124b858183f00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000080000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}00000000000000000000000000000000000000000000000000071afd498d00000000000000000000000000000000000000000000000000005d4bc0025786653d0000000000000000000000000000000000000000000000000000000000000042c02aaa39b223fe8d0a0e5c4f27ead9083c756cc20001f4a0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000646b175474e89094c44da98b954eedeac495271d0f00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000`,
  from: SENDER_ADDRESS_MOCK,
  maxFeePerGas: '0x0',
  maxPriorityFeePerGas: '0x0',
  to: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
  value: '0x71afd498d0000',
};

export const BUY_ERC20_REQUEST_1_MOCK: MockRequestResponse = {
  request: {
    id: '0',
    jsonrpc: '2.0',
    method: 'infura_simulateTransactions',
    params: [
      {
        transactions: [BUY_ERC20_TRANSACTION],
        withCallTrace: true,
        withLogs: true,
      },
    ],
  },
  response: {
    jsonrpc: '2.0',
    result: {
      transactions: [
        {
          return:
            '0x00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000005dc32b358db8fcc9',
          status: '0x1',
          gasUsed: '0x4628e',
          gasLimit: '0x53cc5',
          fees: [
            {
              maxFeePerGas: '0x0',
              maxPriorityFeePerGas: '0x0',
              balanceNeeded: '0x71afd498d0000',
              currentBalance: '0x77f9fd8d99e7e0',
              error: '',
            },
          ],
          stateDiff: {
            post: {
              '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000b000b000302f9f6000000000000432ff2e932894fe0874ac6a384b0',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000000000441ce10709c0a0992095e8e5cd95ea470de',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000013a0b298f9b81d1e0000000000000000000000012aa0e421',
                },
              },
              '0x22832de1178e7f4a8d868411422c720220ddf3d2': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x00000000000000000000000000007708590f241159371f6dd86140347968995f',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fb3f0700000000000000037a742348f5c5c152',
                },
              },
              [SENDER_ADDRESS_MOCK]: {
                balance: '0x70df00440ce7e0',
                nonce: '0x4',
              },
              '0x6b175474e89094c44da98b954eedeac495271d0f': {
                storage: {
                  '0x930de7bb643d1647fc9c859fcc7a7c7bdd0a4a0c3a4bbd02f09ca0cc743567a9':
                    '0x00000000000000000000000000000000000000000000026a5b2c87ebb8bb8d1c',
                  '0xd5dc93f3552cd59356296ab764be72c9bd93d66c2b8576cd18a4677f92be097d':
                    '0x0000000000000000000000000000000000000000000000005dc32b358db8fcc9',
                },
              },
              '0x86e9bd5e42a9afde8d9c2594e84e49cc7718f381': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x000000000000000000000000000ef30a346933b4bd99d3bccde01f6d16e1452c',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fb3f070000000000000000015eea78185df6b5',
                },
              },
              '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48': {
                storage: {
                  '0x32b8c3e0163a3eceb0dd8f4a69ff6a8f786c1f0b2824e29def2a833031a4d374':
                    '0x00000000000000000000000000000000000000000000000000000077a52fa7b8',
                  '0x3dc5bf0351aa60e0f5e0c2094c7bce3615793f84804cfb4ae4169d81e5d802ee':
                    '0x000000000000000000000000000000000000000000000000000000203d220011',
                },
              },
              '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2': {
                balance: '0x27c965cfc53bf0f2c70b6',
                storage: {
                  '0x83f8975b25917b7df8ba465bc82d6b4472b7883d2ee49412f33c76db8bf49be0':
                    '0x00000000000000000000000000000000000000000000000bd3875ff74244a952',
                },
              },
              '0xd9e497bd8f491fe163b42a62c296fb54caea74b7': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000100010000fbc89c0000000000000000000010c700afe76cb29923e8',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x0000000000000000000000000000000000000000f1e2d8a8f477b67ef05de74c',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000000000000013ec29600000000000000011e28c9f4ab7ce8e4',
                },
              },
            },
            pre: {
              '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000b000b000302f9f6000000000000432fedbb64e891ccc2a450f04080',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000000000441ce10008cb545e85f10ea6254adf2328d',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000013a0b249d023d51e0000000000000000000000012aa0e421',
                },
              },
              '0x22832de1178e7f4a8d868411422c720220ddf3d2': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x000000000000000000000000000077084f83018a34e0b30b6c5cbbecc5beadbe',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fa28c700000000000000037a742348f5c5c152',
                },
              },
              [SENDER_ADDRESS_MOCK]: {
                balance: '0x77f9fd8d99e7e0',
                nonce: '0x3',
              },
              '0x6b175474e89094c44da98b954eedeac495271d0f': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x930de7bb643d1647fc9c859fcc7a7c7bdd0a4a0c3a4bbd02f09ca0cc743567a9':
                    '0x00000000000000000000000000000000000000000000026ab8efb321467489e5',
                },
              },
              '0x86e9bd5e42a9afde8d9c2594e84e49cc7718f381': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x000000000000000000000000000ef2efd2b96a4061ac9c50b8d33ded38fd6e81',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fb3d870000000000000000015eea78185df6b5',
                },
              },
              '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x32b8c3e0163a3eceb0dd8f4a69ff6a8f786c1f0b2824e29def2a833031a4d374':
                    '0x00000000000000000000000000000000000000000000000000000077a596c28e',
                  '0x3dc5bf0351aa60e0f5e0c2094c7bce3615793f84804cfb4ae4169d81e5d802ee':
                    '0x000000000000000000000000000000000000000000000000000000203cbae53b',
                },
              },
              '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2': {
                balance: '0x27c965cf538c1c59f70b6',
                nonce: '0x1',
                storage: {
                  '0x83f8975b25917b7df8ba465bc82d6b4472b7883d2ee49412f33c76db8bf49be0':
                    '0x00000000000000000000000000000000000000000000000bd38044f9f8b7a952',
                },
              },
              '0xd9e497bd8f491fe163b42a62c296fb54caea74b7': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000100010000fbc89c0000000000000000000010c7009243db83250922',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x0000000000000000000000000000000000000000f1e2d826b87f89c78bd2b02f',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000000000000013ec1b700000000000000011e28c9f4ab7ce8e4',
                },
              },
            },
          },
          callTrace: {
            from: SENDER_ADDRESS_MOCK,
            to: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
            type: 'CALL',
            gas: '0x1dcd6500',
            gasUsed: '0x4628e',
            value: '0x71afd498d0000',
            input: `0x5ae401dc0000000000000000000000000000000000000000000000000000000065fb43ab0000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000124b858183f00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000080000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}00000000000000000000000000000000000000000000000000071afd498d00000000000000000000000000000000000000000000000000005d4bc0025786653d0000000000000000000000000000000000000000000000000000000000000042c02aaa39b223fe8d0a0e5c4f27ead9083c756cc20001f4a0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000646b175474e89094c44da98b954eedeac495271d0f00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000`,
            output:
              '0x00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000005dc32b358db8fcc9',
            error: '',
            calls: [
              {
                from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                to: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                type: 'DELEGATECALL',
                gas: '0x1d55cb99',
                gasUsed: '0x4b605',
                value: '0x71afd498d0000',
                input: `0xb858183f00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000080000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}00000000000000000000000000000000000000000000000000071afd498d00000000000000000000000000000000000000000000000000005d4bc0025786653d0000000000000000000000000000000000000000000000000000000000000042c02aaa39b223fe8d0a0e5c4f27ead9083c756cc20001f4a0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000646b175474e89094c44da98b954eedeac495271d0f000000000000000000000000000000000000000000000000000000000000`,
                output:
                  '0x0000000000000000000000000000000000000000000000005dc32b358db8fcc9',
                error: '',
                calls: [
                  {
                    from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                    to: '0xdaecee3c08e953bd5f89a5cc90ac560413d709e3',
                    type: 'DELEGATECALL',
                    gas: '0x1ce04a20',
                    gasUsed: '0x6ec',
                    value: '0x71afd498d0000',
                    input:
                      '0x4e6c8ed800000000000000000000000041ff9aa7e16b8b1a8a8dc4f0efacd93d02d071c9000000000000000000000000c02aaa39b223fe8d0a0e5c4f27ead9083c756cc2000000000000000000000000a0b86991c6218b36c1d19d4a2e9eb0ce3606eb4800000000000000000000000000000000000000000000000000000000000001f4',
                    output:
                      '0x0000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d36',
                    error: '',
                    calls: null,
                  },
                  {
                    from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                    to: '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36',
                    type: 'CALL',
                    gas: '0x1ce031ca',
                    gasUsed: '0x284c3',
                    value: '0x0',
                    input: `0x128acb0800000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000071afd498d0000000000000000000000000000fffd8963efd1fc6a506488495d951d5263988d2500000000000000000000000000000000000000000000000000000000000000a000000000000000000000000000000000000000000000000000000000000000c000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000040000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}000000000000000000000000000000000000000000000000000000000000002bc02aaa39b223fe8d0a0e5c4f27ead9083c756cc20001f4a0b86991c6218b36c1d19d4a2e9eb0ce3606eb48000000000000000000000000000000000000000000`,
                    output:
                      '0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffff98e52a00000000000000000000000000000000000000000000000000071afd498d0000',
                    error: '',
                    calls: [
                      {
                        from: '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        to: '0x86e9bd5e42a9afde8d9c2594e84e49cc7718f381',
                        type: 'CALL',
                        gas: '0x1c6c7478',
                        gasUsed: '0x6469',
                        value: '0x0',
                        input:
                          '0x214a6fe20000000000000000000000000000000000000000000000000000000065fb3f07',
                        output: '0x',
                        error: '',
                        calls: [
                          {
                            from: '0x86e9bd5e42a9afde8d9c2594e84e49cc7718f381',
                            to: '0x556b9306565093c855aea9ae92a594704c2cd59e',
                            type: 'STATICCALL',
                            gas: '0x1bfaade4',
                            gasUsed: '0x2cb0',
                            value: null,
                            input:
                              '0xa15ea89f0000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d36',
                            output:
                              '0x000000000000000000000000000000000000000015ed3b13ee249a07ec6465fb0000000000000000000000000000000000000000000000000000000065fc2307',
                            error: '',
                            calls: null,
                          },
                        ],
                      },
                      {
                        from: '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        to: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                        type: 'CALL',
                        gas: '0x1c6ba4b3',
                        gasUsed: '0x9ecc',
                        value: '0x0',
                        input:
                          '0xa9059cbb00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd40000000000000000000000000000000000000000000000000000000000671ad6',
                        output:
                          '0x0000000000000000000000000000000000000000000000000000000000000001',
                        error: '',
                        calls: [
                          {
                            from: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                            to: '0x43506849d7c04f9138d1a2050bbf3a0c054402dd',
                            type: 'DELEGATECALL',
                            gas: '0x1bf9da45',
                            gasUsed: '0x8253',
                            value: '0x0',
                            input:
                              '0xa9059cbb00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd40000000000000000000000000000000000000000000000000000000000671ad6',
                            output:
                              '0x0000000000000000000000000000000000000000000000000000000000000001',
                            error: '',
                            calls: null,
                            logs: [
                              {
                                address:
                                  '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                                topics: [
                                  '0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef',
                                  '0x0000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d36',
                                  '0x00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4',
                                ],
                                data: '0x0000000000000000000000000000000000000000000000000000000000671ad6',
                              },
                            ],
                          },
                        ],
                      },
                      {
                        from: '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        to: '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2',
                        type: 'STATICCALL',
                        gas: '0x1c6afb5f',
                        gasUsed: '0x9e6',
                        value: null,
                        input:
                          '0x70a082310000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        output:
                          '0x00000000000000000000000000000000000000000000000bd38044f9f8b7a952',
                        error: '',
                        calls: null,
                      },
                      {
                        from: '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        to: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                        type: 'CALL',
                        gas: '0x1c6aee8b',
                        gasUsed: '0xa5b3',
                        value: '0x0',
                        input: `0x23a69e75ffffffffffffffffffffffffffffffffffffffffffffffffffffffffff98e52a00000000000000000000000000000000000000000000000000071afd498d0000000000000000000000000000000000000000000000000000000000000000006000000000000000000000000000000000000000000000000000000000000000c000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000040000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}000000000000000000000000000000000000000000000000000000000000002bc02aaa39b223fe8d0a0e5c4f27ead9083c756cc20001f4a0b86991c6218b36c1d19d4a2e9eb0ce3606eb48000000000000000000000000000000000000000000`,
                        output: '0x',
                        error: '',
                        calls: [
                          {
                            from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                            to: '0xdaecee3c08e953bd5f89a5cc90ac560413d709e3',
                            type: 'DELEGATECALL',
                            gas: '0x1bf939da',
                            gasUsed: '0x7c8',
                            value: '0x0',
                            input:
                              '0x8bdb192500000000000000000000000041ff9aa7e16b8b1a8a8dc4f0efacd93d02d071c9000000000000000000000000c02aaa39b223fe8d0a0e5c4f27ead9083c756cc2000000000000000000000000a0b86991c6218b36c1d19d4a2e9eb0ce3606eb4800000000000000000000000000000000000000000000000000000000000001f4',
                            output:
                              '0x0000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d36',
                            error: '',
                            calls: null,
                          },
                          {
                            from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                            to: '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2',
                            type: 'CALL',
                            gas: '0x1bf9157a',
                            gasUsed: '0x5da6',
                            value: '0x71afd498d0000',
                            input: '0xd0e30db0',
                            output: '0x',
                            error: '',
                            calls: null,
                            logs: [
                              {
                                address:
                                  '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2',
                                topics: [
                                  '0xe1fffcc4923d04b559f4d29a8bfc6cda04eb5b0d3c460751c2402c5c5cc9109c',
                                  '0x00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4',
                                ],
                                data: '0x00000000000000000000000000000000000000000000000000071afd498d0000',
                              },
                            ],
                          },
                          {
                            from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                            to: '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2',
                            type: 'CALL',
                            gas: '0x1bf8b790',
                            gasUsed: '0x17ae',
                            value: '0x0',
                            input:
                              '0xa9059cbb0000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d3600000000000000000000000000000000000000000000000000071afd498d0000',
                            output:
                              '0x0000000000000000000000000000000000000000000000000000000000000001',
                            error: '',
                            calls: null,
                            logs: [
                              {
                                address:
                                  '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2',
                                topics: [
                                  '0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef',
                                  '0x00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4',
                                  '0x0000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d36',
                                ],
                                data: '0x00000000000000000000000000000000000000000000000000071afd498d0000',
                              },
                            ],
                          },
                        ],
                      },
                      {
                        from: '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        to: '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2',
                        type: 'STATICCALL',
                        gas: '0x1c6a48f7',
                        gasUsed: '0x216',
                        value: null,
                        input:
                          '0x70a082310000000000000000000000001ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        output:
                          '0x00000000000000000000000000000000000000000000000bd3875ff74244a952',
                        error: '',
                        calls: null,
                      },
                    ],
                    logs: [
                      {
                        address: '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36',
                        topics: [
                          '0x19b47279256b2a23a1665c810c8d55a1758940ee09377d4f8d26497a3577dc83',
                          '0x00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4',
                          '0x00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4',
                        ],
                        data: '0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffff98e52a00000000000000000000000000000000000000000000000000071afd498d0000000000000000000000000000000000000000432ff2e932894fe0874ac6a384b0000000000000000000000000000000000000000000000000015f0dcf64944869000000000000000000000000000000000000000000000000000000000002f9f600000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004f29944800',
                      },
                    ],
                  },
                  {
                    from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                    to: '0xdaecee3c08e953bd5f89a5cc90ac560413d709e3',
                    type: 'DELEGATECALL',
                    gas: '0x1cddaccb',
                    gasUsed: '0x6ec',
                    value: '0x71afd498d0000',
                    input:
                      '0x4e6c8ed800000000000000000000000041ff9aa7e16b8b1a8a8dc4f0efacd93d02d071c9000000000000000000000000a0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000000000000000000000006b175474e89094c44da98b954eedeac495271d0f0000000000000000000000000000000000000000000000000000000000000064',
                    output:
                      '0x000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                    error: '',
                    calls: null,
                  },
                  {
                    from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                    to: '0xd9e497bd8f491fe163b42a62c296fb54caea74b7',
                    type: 'CALL',
                    gas: '0x1cdd9474',
                    gasUsed: '0x1c824',
                    value: '0x0',
                    input: `0x128acb08000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000671ad6000000000000000000000000fffd8963efd1fc6a506488495d951d5263988d2500000000000000000000000000000000000000000000000000000000000000a000000000000000000000000000000000000000000000000000000000000000c00000000000000000000000000000000000000000000000000000000000000020000000000000000000000000000000000000000000000000000000000000004000000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4000000000000000000000000000000000000000000000000000000000000002ba0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000646b175474e89094c44da98b954eedeac495271d0f000000000000000000000000000000000000000000`,
                    output:
                      '0xffffffffffffffffffffffffffffffffffffffffffffffffa23cd4ca724703370000000000000000000000000000000000000000000000000000000000671ad6',
                    error: '',
                    calls: [
                      {
                        from: '0xd9e497bd8f491fe163b42a62c296fb54caea74b7',
                        to: '0x22832de1178e7f4a8d868411422c720220ddf3d2',
                        type: 'CALL',
                        gas: '0x1c69e198',
                        gasUsed: '0x4335',
                        value: '0x0',
                        input:
                          '0x214a6fe20000000000000000000000000000000000000000000000000000000065fb3f07',
                        output: '0x',
                        error: '',
                        calls: [
                          {
                            from: '0x22832de1178e7f4a8d868411422c720220ddf3d2',
                            to: '0x556b9306565093c855aea9ae92a594704c2cd59e',
                            type: 'STATICCALL',
                            gas: '0x1bf82eec',
                            gasUsed: '0x1540',
                            value: null,
                            input:
                              '0xa15ea89f000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                            output:
                              '0x0000000000000000000000000000000000000000001bc9cc2365913cdef3d2e10000000000000000000000000000000000000000000000000000000065fc2307',
                            error: '',
                            calls: null,
                          },
                        ],
                      },
                      {
                        from: '0xd9e497bd8f491fe163b42a62c296fb54caea74b7',
                        to: '0x6b175474e89094c44da98b954eedeac495271d0f',
                        type: 'CALL',
                        gas: '0x1c69330d',
                        gasUsed: '0x75de',
                        value: '0x0',
                        input: `0xa9059cbb000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}0000000000000000000000000000000000000000000000005dc32b358db8fcc9`,
                        output:
                          '0x0000000000000000000000000000000000000000000000000000000000000001',
                        error: '',
                        calls: null,
                        logs: [
                          {
                            address:
                              '0x6b175474e89094c44da98b954eedeac495271d0f',
                            topics: [
                              '0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef',
                              '0x000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                              `0x000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}`,
                            ],
                            data: '0x0000000000000000000000000000000000000000000000005dc32b358db8fcc9',
                          },
                        ],
                      },
                      {
                        from: '0xd9e497bd8f491fe163b42a62c296fb54caea74b7',
                        to: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                        type: 'STATICCALL',
                        gas: '0x1c68bba0',
                        gasUsed: '0xd0b',
                        value: null,
                        input:
                          '0x70a08231000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                        output:
                          '0x000000000000000000000000000000000000000000000000000000203cbae53b',
                        error: '',
                        calls: [
                          {
                            from: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                            to: '0x43506849d7c04f9138d1a2050bbf3a0c054402dd',
                            type: 'DELEGATECALL',
                            gas: '0x1bf715d8',
                            gasUsed: '0x9f9',
                            value: '0x0',
                            input:
                              '0x70a08231000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                            output:
                              '0x000000000000000000000000000000000000000000000000000000203cbae53b',
                            error: '',
                            calls: null,
                          },
                        ],
                      },
                      {
                        from: '0xd9e497bd8f491fe163b42a62c296fb54caea74b7',
                        to: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                        type: 'CALL',
                        gas: '0x1c68abb4',
                        gasUsed: '0x373e',
                        value: '0x0',
                        input:
                          '0x23a69e75ffffffffffffffffffffffffffffffffffffffffffffffffa23cd4ca724703370000000000000000000000000000000000000000000000000000000000671ad6000000000000000000000000000000000000000000000000000000000000006000000000000000000000000000000000000000000000000000000000000000c00000000000000000000000000000000000000000000000000000000000000020000000000000000000000000000000000000000000000000000000000000004000000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4000000000000000000000000000000000000000000000000000000000000002ba0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000646b175474e89094c44da98b954eedeac495271d0f000000000000000000000000000000000000000000',
                        output: '0x',
                        error: '',
                        calls: [
                          {
                            from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                            to: '0xdaecee3c08e953bd5f89a5cc90ac560413d709e3',
                            type: 'DELEGATECALL',
                            gas: '0x1bf7000e',
                            gasUsed: '0x7c8',
                            value: '0x0',
                            input:
                              '0x8bdb192500000000000000000000000041ff9aa7e16b8b1a8a8dc4f0efacd93d02d071c9000000000000000000000000a0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000000000000000000000006b175474e89094c44da98b954eedeac495271d0f0000000000000000000000000000000000000000000000000000000000000064',
                            output:
                              '0x000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                            error: '',
                            calls: null,
                          },
                          {
                            from: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
                            to: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                            type: 'CALL',
                            gas: '0x1bf6f3a5',
                            gasUsed: '0x203c',
                            value: '0x0',
                            input:
                              '0xa9059cbb000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b70000000000000000000000000000000000000000000000000000000000671ad6',
                            output:
                              '0x0000000000000000000000000000000000000000000000000000000000000001',
                            error: '',
                            calls: [
                              {
                                from: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                                to: '0x43506849d7c04f9138d1a2050bbf3a0c054402dd',
                                type: 'DELEGATECALL',
                                gas: '0x1b8714fa',
                                gasUsed: '0x1d27',
                                value: '0x0',
                                input:
                                  '0xa9059cbb000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b70000000000000000000000000000000000000000000000000000000000671ad6',
                                output:
                                  '0x0000000000000000000000000000000000000000000000000000000000000001',
                                error: '',
                                calls: null,
                                logs: [
                                  {
                                    address:
                                      '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                                    topics: [
                                      '0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef',
                                      '0x00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4',
                                      '0x000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                                    ],
                                    data: '0x0000000000000000000000000000000000000000000000000000000000671ad6',
                                  },
                                ],
                              },
                            ],
                          },
                        ],
                      },
                      {
                        from: '0xd9e497bd8f491fe163b42a62c296fb54caea74b7',
                        to: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                        type: 'STATICCALL',
                        gas: '0x1c6872db',
                        gasUsed: '0x53b',
                        value: null,
                        input:
                          '0x70a08231000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                        output:
                          '0x000000000000000000000000000000000000000000000000000000203d220011',
                        error: '',
                        calls: [
                          {
                            from: '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48',
                            to: '0x43506849d7c04f9138d1a2050bbf3a0c054402dd',
                            type: 'DELEGATECALL',
                            gas: '0x1bf6ce36',
                            gasUsed: '0x229',
                            value: '0x0',
                            input:
                              '0x70a08231000000000000000000000000d9e497bd8f491fe163b42a62c296fb54caea74b7',
                            output:
                              '0x000000000000000000000000000000000000000000000000000000203d220011',
                            error: '',
                            calls: null,
                          },
                        ],
                      },
                    ],
                    logs: [
                      {
                        address: '0xd9e497bd8f491fe163b42a62c296fb54caea74b7',
                        topics: [
                          '0x19b47279256b2a23a1665c810c8d55a1758940ee09377d4f8d26497a3577dc83',
                          '0x00000000000000000000000013f4ea83d0bd40e75c8222255bc855a974568dd4',
                          `0x000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}`,
                        ],
                        data: '0xffffffffffffffffffffffffffffffffffffffffffffffffa23cd4ca724703370000000000000000000000000000000000000000000000000000000000671ad60000000000000000000000000000000000000000000010c700afe76cb29923e80000000000000000000000000000000000000000000000037a74fbab94f3e0d1fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffbc89c000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000df',
                      },
                    ],
                  },
                ],
              },
            ],
          },
          feeEstimate: 11633709217510676,
          baseFeePerGas: 40482817574,
        },
      ],
      blockNumber: '0x1293669',
      id: '9e8034c8-b9b2-4aff-ba5b-13ddc0c2a7e6',
    },
    id: '0',
  },
};

// The core simulation code confirms the token balance changes buy running
// a simulation of the transaction, sandwiched between calls to token balances
// for the user's wallet. This is what the second request accomplishes.
export const BUY_ERC20_REQUEST_2_MOCK: MockRequestResponse = {
  request: {
    id: '1',
    jsonrpc: '2.0',
    method: 'infura_simulateTransactions',
    params: [
      {
        transactions: [
          {
            from: SENDER_ADDRESS_MOCK,
            to: '0x6b175474e89094c44da98b954eedeac495271d0f',
            data: `0x70a08231000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}`,
          },
          {
            chainId: '0x1',
            from: SENDER_ADDRESS_MOCK,
            to: '0x13f4ea83d0bd40e75c8222255bc855a974568dd4',
            value: '0x71afd498d0000',
            data: `0x5ae401dc0000000000000000000000000000000000000000000000000000000065fb43ab0000000000000000000000000000000000000000000000000000000000000040000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000124b858183f00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000080000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}00000000000000000000000000000000000000000000000000071afd498d00000000000000000000000000000000000000000000000000005d4bc0025786653d0000000000000000000000000000000000000000000000000000000000000042c02aaa39b223fe8d0a0e5c4f27ead9083c756cc20001f4a0b86991c6218b36c1d19d4a2e9eb0ce3606eb480000646b175474e89094c44da98b954eedeac495271d0f00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000`,
          },
          {
            from: SENDER_ADDRESS_MOCK,
            to: '0x6b175474e89094c44da98b954eedeac495271d0f',
            data: `0x70a08231000000000000000000000000${SENDER_ADDRESS_NO_0X_MOCK}`,
          },
        ],
      },
    ],
  },
  response: {
    jsonrpc: '2.0',
    result: {
      transactions: [
        {
          return:
            '0x0000000000000000000000000000000000000000000000000000000000000000',
          status: '0x1',
          gasUsed: '0x5de2',
          gasLimit: '0x5f34',
          fees: [
            {
              maxFeePerGas: '0xf19b9f48d',
              maxPriorityFeePerGas: '0x9febc9',
              balanceNeeded: '0x59d9d3b865ed8',
              currentBalance: '0x77f9fd8d99e7e0',
              error: '',
            },
          ],
          stateDiff: {
            post: {
              [SENDER_ADDRESS_MOCK]: {
                nonce: '0x4',
              },
            },
            pre: {
              [SENDER_ADDRESS_MOCK]: {
                balance: '0x77f9fd8d99e7e0',
                nonce: '0x3',
              },
            },
          },
          feeEstimate: 972988071597550,
          baseFeePerGas: 40482817574,
        },
        {
          return:
            '0x00000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000000005dc32b358db8fcc9',
          status: '0x1',
          gasUsed: '0x4628e',
          gasLimit: '0x53cc5',
          fees: [
            {
              maxFeePerGas: '0xf19b9f48d',
              maxPriorityFeePerGas: '0x9febc9',
              balanceNeeded: '0x56314571bd7946',
              currentBalance: '0x725c6052138908',
              error: '',
            },
          ],
          stateDiff: {
            post: {
              '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000b000b000302f9f6000000000000432ff2e932894fe0874ac6a384b0',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000000000441ce10709c0a0992095e8e5cd95ea470de',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000013a0b298f9b81d1e0000000000000000000000012aa0e421',
                },
              },
              '0x22832de1178e7f4a8d868411422c720220ddf3d2': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x00000000000000000000000000007708590f241159371f6dd86140347968995f',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fb3f0700000000000000037a742348f5c5c152',
                },
              },
              [SENDER_ADDRESS_MOCK]: {
                balance: '0x70df00440ce7e0',
                nonce: '0x5',
              },
              '0x6b175474e89094c44da98b954eedeac495271d0f': {
                storage: {
                  '0x930de7bb643d1647fc9c859fcc7a7c7bdd0a4a0c3a4bbd02f09ca0cc743567a9':
                    '0x00000000000000000000000000000000000000000000026a5b2c87ebb8bb8d1c',
                  '0xd5dc93f3552cd59356296ab764be72c9bd93d66c2b8576cd18a4677f92be097d':
                    '0x0000000000000000000000000000000000000000000000005dc32b358db8fcc9',
                },
              },
              '0x86e9bd5e42a9afde8d9c2594e84e49cc7718f381': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x000000000000000000000000000ef30a346933b4bd99d3bccde01f6d16e1452c',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fb3f070000000000000000015eea78185df6b5',
                },
              },
              '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48': {
                storage: {
                  '0x32b8c3e0163a3eceb0dd8f4a69ff6a8f786c1f0b2824e29def2a833031a4d374':
                    '0x00000000000000000000000000000000000000000000000000000077a52fa7b8',
                  '0x3dc5bf0351aa60e0f5e0c2094c7bce3615793f84804cfb4ae4169d81e5d802ee':
                    '0x000000000000000000000000000000000000000000000000000000203d220011',
                },
              },
              '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2': {
                balance: '0x27c965cfc53bf0f2c70b6',
                storage: {
                  '0x83f8975b25917b7df8ba465bc82d6b4472b7883d2ee49412f33c76db8bf49be0':
                    '0x00000000000000000000000000000000000000000000000bd3875ff74244a952',
                },
              },
              '0xd9e497bd8f491fe163b42a62c296fb54caea74b7': {
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000100010000fbc89c0000000000000000000010c700afe76cb29923e8',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x0000000000000000000000000000000000000000f1e2d8a8f477b67ef05de74c',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000000000000013ec29600000000000000011e28c9f4ab7ce8e4',
                },
              },
            },
            pre: {
              '0x1ac1a8feaaea1900c4166deeed0c11cc10669d36': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000b000b000302f9f6000000000000432fedbb64e891ccc2a450f04080',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000000000441ce10008cb545e85f10ea6254adf2328d',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000013a0b249d023d51e0000000000000000000000012aa0e421',
                },
              },
              '0x22832de1178e7f4a8d868411422c720220ddf3d2': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x000000000000000000000000000077084f83018a34e0b30b6c5cbbecc5beadbe',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fa28c700000000000000037a742348f5c5c152',
                },
              },
              [SENDER_ADDRESS_MOCK]: {
                balance: '0x77f9fd8d99e7e0',
                nonce: '0x4',
              },
              '0x6b175474e89094c44da98b954eedeac495271d0f': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x930de7bb643d1647fc9c859fcc7a7c7bdd0a4a0c3a4bbd02f09ca0cc743567a9':
                    '0x00000000000000000000000000000000000000000000026ab8efb321467489e5',
                },
              },
              '0x86e9bd5e42a9afde8d9c2594e84e49cc7718f381': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000001':
                    '0x000000000000000000000000000ef2efd2b96a4061ac9c50b8d33ded38fd6e81',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x00000000000000000000000165fb3d870000000000000000015eea78185df6b5',
                },
              },
              '0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x32b8c3e0163a3eceb0dd8f4a69ff6a8f786c1f0b2824e29def2a833031a4d374':
                    '0x00000000000000000000000000000000000000000000000000000077a596c28e',
                  '0x3dc5bf0351aa60e0f5e0c2094c7bce3615793f84804cfb4ae4169d81e5d802ee':
                    '0x000000000000000000000000000000000000000000000000000000203cbae53b',
                },
              },
              '0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2': {
                balance: '0x27c965cf538c1c59f70b6',
                nonce: '0x1',
                storage: {
                  '0x83f8975b25917b7df8ba465bc82d6b4472b7883d2ee49412f33c76db8bf49be0':
                    '0x00000000000000000000000000000000000000000000000bd38044f9f8b7a952',
                },
              },
              '0xd9e497bd8f491fe163b42a62c296fb54caea74b7': {
                balance: '0x0',
                nonce: '0x1',
                storage: {
                  '0x0000000000000000000000000000000000000000000000000000000000000000':
                    '0x000000000100010000fbc89c0000000000000000000010c7009243db83250922',
                  '0x0000000000000000000000000000000000000000000000000000000000000003':
                    '0x0000000000000000000000000000000000000000f1e2d826b87f89c78bd2b02f',
                  '0x0000000000000000000000000000000000000000000000000000000000000004':
                    '0x000000000000000000000000013ec1b700000000000000011e28c9f4ab7ce8e4',
                },
              },
            },
          },
          feeEstimate: 11633996591798050,
          baseFeePerGas: 40482817574,
        },
        {
          return:
            '0x0000000000000000000000000000000000000000000000005dc32b358db8fcc9',
          status: '0x1',
          gasUsed: '0x5de2',
          gasLimit: '0x5f34',
          fees: [
            {
              maxFeePerGas: '0xf19b9f48d',
              maxPriorityFeePerGas: '0x9febc9',
              balanceNeeded: '0x59d9d3b865ed8',
              currentBalance: '0x1c2b1ae0560fc2',
              error: '',
            },
          ],
          stateDiff: {
            post: {
              [SENDER_ADDRESS_MOCK]: {
                nonce: '0x6',
              },
            },
            pre: {
              [SENDER_ADDRESS_MOCK]: {
                balance: '0x70df00440ce7e0',
                nonce: '0x5',
              },
            },
          },
          feeEstimate: 972988071597550,
          baseFeePerGas: 40482817574,
        },
      ],
      blockNumber: '0x1293669',
      id: 'faaab4c5-edf5-4077-ac75-8d26278ca2c5',
    },
    id: '1',
  },
};
